/* package_cache.c, Copyright (c) 1995, Doug L. Hoffman
 * 
 * Program: package_cache.c
 * Author : Doug L. Hoffman, hoffman@tatooine.vnet.net
 * Purpose: package record cache routines
 * Created: Wed Nov  1 14:39:55 1995 by hoffman
 * Revised: Wed Nov  8 12:49:21 1995 by r.faith@ieee.org
 * 
 * Disclaimer:
 * 
 *   Copyright (c) 1995, Doug Hoffman
 *   This program comes with ABSOLUTELY NO WARRANTY.
 *
 *   This program is free software; you can redistribute it and/or modify it
 *   under the terms of the GNU General Public License as published by the
 *   Free Software Foundation; either version 2, or (at your option) any
 *   later version.
 * 
 *   This program is distributed in the hope that it will be useful, but
 *   WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   General Public License for more details.
 * 
 *   You should have received a copy of the GNU General Public License along
 *   with this program; if not, write to the Free Software Foundation, Inc.,
 *   675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * 
 * Abstract:
 *
 *   The file contains the programs and data structures needed to implement
 *   the pmdb package record cache feature.
 * 
 * Description:
 * 
 * Modification History:
 * 
 * $Log: package_cache.c,v $
 * Revision 1.2  1995/11/08 17:57:06  faith
 * Use PMDB_DISP_* consistently
 *
 * Revision 1.1  1995/11/06  18:35:22  hoffman
 * pm package cache routines added.
 *
 * 
 */

#include "pmdb.h"

extern int	 PmdbDeleteCount;

/* ----- local global variables.
 */

static PMDBPackageRec 	PackageCache[PMDB_PACKAGE_CACHE_SIZE];
static char		*EntryName[PMDB_PACKAGE_CACHE_SIZE];
static int		CacheCount;
static int		CacheNext;

/* =============== update_pcache
 *
 *  Function to find an entry in the package record cache.
 */

int find_pcache_entry( char *name)
{
   int	i;
   
   for (i=0; i<CacheCount; i++) {
      if (EntryName[i] != NULL) {
	 if (0 == strcmp( name, EntryName[i])) return i;
      }
   }

   return -1;
}

/* =============== write_pcache
 *
 *  Function to write out the contents of the package record cache.
 */

int flush_pcache()
{
   int		i;
   datum	data;
   datum	key;
   
   for (i=0; i<CacheCount; i++) {
      if (EntryName[i] != NULL) {
	 key.dsize = strlen( EntryName[i]);
	 key.dptr  = EntryName[i];
	 pack_package_record( &PackageCache[i], &data);
	 gdbm_delete( PMDBFile[PMDB_PACKAGES].gdbmFile, key);
	 PmdbDeleteCount += data.dsize;
	 if (0 != (gdbm_store( PMDBFile[PMDB_PACKAGES].gdbmFile,
			       key, data, GDBM_REPLACE))) {
	    return gdbm_errno;
	 }
	 free( data.dptr);	/* free the name string as well */
	 pmdb_free_package_rec( &PackageCache[i]);
	 free( EntryName[i]);
      }
   }

   return 0;
}

/* =============== init_pcache
 *
 *  Function to initialize the package record cache. This function is to
 *  be called from pmdb_open *ONLY*. It assumes that a flush has been
 *  performed or the cache has never been used before. No attempt is made to
 *  free memory. The size of the cache is returned.
 */

int init_pcache()
{
   int i;
   
   for (i=0; i<PMDB_PACKAGE_CACHE_SIZE; i++) 
	 EntryName[i] = NULL;
   
   CacheCount = 0;
   CacheNext  = 0;

   return PMDB_PACKAGE_CACHE_SIZE;
}


/* =============== update_pcache
 *
 *  Function to update a record in the package record cache. The value of
 *  fileFlag should be "I" for installed, or "O" for overwritten.
 */

int update_pcache( PMDBFileRec *record, char fileFlag)
{
   int		i, j;
   datum	data;
   datum	key;
   datum	pkey;

   /* ----- Find the record in the cache...
    *
    *  The package record in question is sought in the package cache.
    *  If not found a spill might be necessary.
    */
   
   pkey.dsize = strlen( record->field[PMDB_FR_PACKAGE]);
   pkey.dptr  = (char *)record->field[PMDB_FR_PACKAGE];
   
   if (-1 == (j=find_pcache_entry( pkey.dptr))) {
      if (CacheCount == PMDB_PACKAGE_CACHE_SIZE) {
	 /* ----- The cache is full
	  *
	  *  The package cache is full so we need to spill the entry
	  *  currently pointed to by CacheNext. The entries storage is
	  *  also freed.
	  */
	 j = CacheNext;
	 key.dsize = strlen( EntryName[j]);
	 key.dptr  = EntryName[j];
	 pack_package_record( &PackageCache[j], &data);
	 gdbm_delete( PMDBFile[PMDB_PACKAGES].gdbmFile, key);
	 PmdbDeleteCount += data.dsize;
	 if (0 != (gdbm_store( PMDBFile[PMDB_PACKAGES].gdbmFile,
			       key, data, GDBM_REPLACE))) {
	    return gdbm_errno;
	 }
	 free( data.dptr);	/* free the name string as well */
	 pmdb_free_package_rec( &PackageCache[j]);
      }
      else {
	 j = CacheNext;
	 CacheCount++;
      }
      if (++CacheNext == PMDB_PACKAGE_CACHE_SIZE) CacheNext = 0;
      
      /* ----- Read in the missing package record
       *
       *  We now have an empty cache record ready to receive the specified
       *  record. Read and unpack the package record named in pkey.
       */
	 
      data = gdbm_fetch( PMDBFile[PMDB_PACKAGES].gdbmFile, pkey);
      if (NULL == data.dptr) {
	 return gdbm_errno;
      }
      unpack_package_record( record->field[PMDB_FR_PACKAGE],
			     &data,
			     &PackageCache[j]);
      EntryName[j] = strdup( record->field[PMDB_FR_PACKAGE]);
      free( data.dptr);
   }

   /* ----- Got the cache entry, update the data.
    *
    *  Regardless of how we got here, j names the cache entry that contains
    *  the reccord in question. Next we update the file entry.
    */
   
   i = mark_file_in_package( record->field[PMDB_FR_KEY],
			     &PackageCache[j], fileFlag);
   if (i == -1) {
      add_file_to_package( record->field[PMDB_FR_KEY], &PackageCache[j]);
   }

   return 0;
}
