/* pmdb.c, Copyright (c) 1995, Doug Hoffman
 * 
 * Program: pmdb.c
 * Author : Doug Hoffman, hoffman@cybernetics.com
 * Purpose: c program pmdb.
 * Created: Fri Mar 24 14:36:34 1995 by hoffman
 * Revised: Wed Nov  1 13:48:43 1995 by hoffman
 * 
 * Disclaimer:
 * 
 *   Copyright (c) 1995, Doug Hoffman
 *   This program comes with ABSOLUTELY NO WARRANTY.
 * 
 *   This program is free software; you can redistribute it and/or modify it
 *   under the terms of the GNU General Public License as published by the
 *   Free Software Foundation; either version 2, or (at your option) any
 *   later version.
 * 
 *   This program is distributed in the hope that it will be useful, but
 *   WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   General Public License for more details.
 * 
 *   You should have received a copy of the GNU General Public License along
 *   with this program; if not, write to the Free Software Foundation, Inc.,
 *   675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Abstract:
 * 
 * Description:
 * 
 * Modification History:
 * 
 * $Log: pmdb.c,v $
 * Revision 1.6  1995/11/06 18:35:23  hoffman
 * pm package cache routines added.
 *
 * Revision 1.5  1995/05/14  07:38:47  faith
 * Finished adding const to all appropriate places.
 * Debugged free's of already freed space.
 * Debugged memory leaks.
 * Applied patches from Doug.
 *
 * Revision 1.4  1995/05/13  20:57:16  faith
 * Added const.  Added fields to records.  Added options to pmdb.
 *
 * Revision 1.3  1995/05/13  18:06:59  hoffman
 * PM database library for RedHat Software.
 *
 * Revision 1.2  1995/05/09  19:56:41  hoffman
 * First working version, can create database and read back version stamp.
 *
 * Revision 1.1  1995/05/04  17:21:17  hoffman
 * Initial revision
 *
 * 
 */

#include "pmdb.h"

#ifdef __GNU_LIBRARY__
#define GNU_GETOPT
#endif


/* ---------- Typedefs */

typedef enum {
   False = 0, True = 1
} Boolean;

/* ---------- Some global variables */
   
char  PgmName[]   = "pmdb";
char  RevStr[]    = "$Revision: 1.6 $";
char  ModStr[]    = "Modified: Wed Nov  1 13:48:43 1995 by hoffman";
char  Copyright[] = "Copyright (C) Doug L. Hoffman, 1995";

int	Verbose;
int	Debug;
Boolean	Option_D;
Boolean	Option_F;
Boolean	Option_f;
Boolean	Option_i;
Boolean	Option_l;
Boolean	Option_P;
Boolean	Option_p;
Boolean	Option_r;
Boolean	Option_v;


#ifdef GNU_GETOPT
struct option longopts[] =
{
 /* { name  has_arg  *flag  val } */
    {"debug",        0, 0, 'd'}, /* turn on program debugging */
    {"delete",       0, 0, 'D'}, /* delete */
    {"files",        0, 0, 'F'}, /* list all file entries */
    {"file",         1, 0, 'f'}, /* list a file entry */
    {"help",         0, 0, 'h'}, /* give help */
    {"init",         0, 0, 'i'}, /* initialize the database */
    {"list",         0, 0, 'l'}, /* long listing format */
    {"packages",     0, 0, 'P'}, /* list all package entries */
    {"package",      1, 0, 'p'}, /* list a package entry */
    {"reorganize",   0, 0, 'r'}, /* reorganize */
    {"version",      0, 0, 'v'}, /* verbose mode */
    {"pgm_version",  0, 0, 'V'}, /* display version number */
    {"database",     1, 0, 500},
    { 0, 0, 0, 0 }
};
#endif



/* ---------- Some (moderately) usefull local functions
 *
 *  These functions display usage, help, version, etc.
 */

static void usage(char *pgmName)
{
   fprintf(stderr,
           "Usage: %s [-DdFhilPrSVv] [-f file_name] [-p package_name]\n",
           pgmName);
}

/* ----- Help message */


static void help(char *pgmName)
{
#ifdef GNU_GETOPT
   static const char  *help_msg[] = {
      " -D              --delete           delete whatever",
      " -d              --debug            turn on debugging",
      " -F              --files            list all files",
      " -f file_name    --file             look up a file",
      " -h              --help             give this help",
      " -i              --init             initialize the database",
      " -l              --list             long listing format",
      " -P              --packages         list all packages",
      " -p package_name --package          look up a package",
      " -r              --reorganize       reorganize the database",
      " -S              --squawk           be verbose while running",
      " -V              --pgm_version      display program version number",
      " -v              --version          display database version number",
      "                 --database path    specify path to database",
      0};
#else
   static const char  *help_msg[] = {
      " -D              delete whatever",
      " -d              turn on debugging",
      " -F              list all files",
      " -f file_name    look up a file",
      " -h              give this help",
      " -i              initialize the database",
      " -l              long listing format",
      " -P              list all packages",
      " -p package_name look up a package",
      " -r              reorganize the database",
      " -S              be verbose while running",
      " -V              display program version number",
      " -v              display database version number",
      0};
#endif
   
   char **p = (char **)help_msg;

   usage( pgmName);
   while (*p) fprintf(stderr, "%s\n", *p++);
}


/* ----- Version message */


static void version( char *pgmName )
{
   char *rev;
   int  i;

   i = strlen( RevStr ) - 14;
   if (i > 0) {
      rev = RevStr + 10;
      rev[strlen(rev) - 1] = '\0';
   }
   else {
      rev = (char *) "1.0";
      i = 3;
   }
   
   fprintf(stderr, "%s %*s (%s)\n", pgmName, i, rev, ModStr);
   fprintf(stderr,"%s\n", Copyright);
}

/* =============== local functions ===================== */

void print_file_rec( PMDBFileRec *rec);
void print_package_rec( PMDBPackageRec *rec);
void print_package_files( PMDBPackageRec *rec);

void print_file_rec( PMDBFileRec *rec)
{
   int		i;

   for (i=0; i<PMDB_FR_ENTRIES; i++) {
      printf("%s ", rec->field[i]);
   }
   putchar('\n');
}

void print_package_rec( PMDBPackageRec *rec)
{
   int		i;

   for (i=0; i<PMDB_PR_ENTRIES; i++) {
      printf("%s ", rec->field[i]);
   }
   putchar('\n');
}

void print_package_files( PMDBPackageRec *rec)
{
   int		i;

   for (i=0; i<rec->count; i++) {
      printf("  %s (%s)\n", rec->file[i], rec->disp[i]);
   }
   putchar('\n');
}

/* =============== start of main routine =============== */

int main( int argc, char **argv)
{
   int          c;

   int		i;
   char		*s;

   PMDBPackageRec	package;
   PMDBFileRec		record;
   char			*fileName = NULL;
   const char           *path     = NULL;
   
   /* ----- Parse the command line options
    *
    *  The following is just the standard "getopt" type stuff.
    */

#ifdef GNU_GETOPT
   while ((c = getopt_long (argc, argv, "DdFf:hilPp:rSVv",
                            longopts, (int *)0)) != EOF) {
#else
   while ((c = getopt(argc, argv, "DdFf:hilPp:rSVv")) != -1) {
#endif
      
      switch (c) {

      case 500: {
	 path = strdup( optarg );
	 break;
      }
          
      case 'D': {               /* delete */
	 Option_D = True;
         break;
      }
	 
      case 'd': {               /* debugging */
/*       parse_debug( optarg ); */
         Debug++;
         break;
      }
	 
      case 'F': {               /* all files */
	 Option_F = True;
         break;
      }
	 
      case 'f': {               /* files */
	 Option_f = True;
	 fileName = optarg;
         break;
      }
	 
      case 'h': {               /* help */
         help( PgmName );
         exit(0);
         break;
      }
         
      case 'i': {		/* init */
         Option_i = True;
         break;
      }
         
      case 'l': {		/* long listing format */
         Option_l = True;
         break;
      }
	 
      case 'P': {               /* all packages */
	 Option_P = True;
         break;
      }
	 
      case 'p': {               /* package */
	 Option_p = True;
	 fileName = optarg;
         break;
      }
	
      case 'r': {               /* reorganize */
	 Option_r = True;
         break;
      }
	 
         
      case 'S': {
         Verbose++;		/* squawk - be verbose */
         break;
      }
         
      case 'V': {               /* pgm version   */
         version( argv[0]);
         break;
      }
         
      case 'v': {
         Option_v = True;		/* database version */
         break;
      }
         
      default: {
         usage( PgmName);
         exit(1);
      }}
   }
/* --------========== Perform Requested Operation  ==========---------- */

   if (Option_i) {
      if (Debug) printf("calling pmdb_init...\n");
      i = pmdb_init(path);
      if (Debug) printf("  result of pmdb_init is %d\n", i);
      
      if (Debug) printf("calling pmdb_open...\n");
      i = pmdb_open(path,GDBM_WRITER);
      if (Debug) printf("  result of pmdb_open is %d\n", i);
     
      if (Debug) printf("calling pmdb_close...\n");
      i = pmdb_close();
      if (Debug) printf("  result of pmdb_close is %d\n", i);
      
      if (Debug) printf("Fini.\n");
   }

   if (Option_r) {
      i = pmdb_open(path,GDBM_WRITER);
      i = pmdb_reorganize();
      i = pmdb_close();
   }

   
   if (Option_v) {
      i = pmdb_open(path,GDBM_READER);
      if (i) {
	 fprintf( stderr,"pmdb: opening database: %s\n",
		  strerror( i));
	 exit( i);
      }
	 
      s = pmdb_read_version();
      if (s == NULL ) {
	 if (errno != 0) {
	    fprintf( stderr,"pmdb: reading version stamp: %s\n",
		     strerror(errno));
	    exit( errno);
	 }
      }
      printf("%s\n", s);
      i = pmdb_close();
   }

   /* ---------- lookup a file entry
    *
    */

   if (Option_f) {
      i = pmdb_open(path,GDBM_READER);
      if (i) {
	 fprintf( stderr,"pmdb: opening database: %s\n",
		  strerror( i));
	 exit( i);
      }

      /* ----- see if a pathname or just a file name
       *
       *  If the fileName supplied has a "/" in it assume that it is a
       *  pathname, otherwise do a full file scan using the string as
       *  a filename.
       */
      
      if (NULL != index( fileName, '/')) {
	 /* -- see if a read or a delete */
	 if (Option_D) {
	    pmdb_delete_file( fileName);
	 }
	 else {
	    if (0 != (i = pmdb_read_file( fileName, &record))) {
	       if (errno != 0) {
		  fprintf( stderr,"pmdb: reading file record: %s\n",
			   strerror(errno));
		  exit( errno);
	       }
	       else printf("Not found\n");
	    }
	    else {
	       print_file_rec( &record);
	       pmdb_free_file_rec( &record);
	    }
	 }
      }
      else { 
	 if (0 != (i = pmdb_read_first_file( &record))) {
	    if (errno != 0) {
	       fprintf( stderr,"pmdb: reading first file record: %s\n",
			strerror(errno));
	       exit( errno);
	    }
	    else printf("Not found\n");
	 }
	 else {
	    if (match_file( fileName, record.field[PMDB_FR_KEY])) {
	       if (Option_D) {
		  printf("deleting %s\n", record.field[PMDB_FR_KEY]);
		  pmdb_delete_file( record.field[PMDB_FR_KEY]);
	       }
	       else {
		  print_file_rec( &record);
	       }
	       pmdb_free_file_rec( &record);
	    }

	    /* --- loop through the file */ 

	    while (0 == (i = pmdb_read_next_file( &record))) {
	       if (match_file( fileName, record.field[PMDB_FR_KEY])) {
		  if (Option_D) {
		     printf("deleting %s\n", record.field[PMDB_FR_KEY]);
		     pmdb_delete_file( record.field[PMDB_FR_KEY]);
		  }
		  else {
		     print_file_rec( &record);
		  }
		  pmdb_free_file_rec( &record);
	       }
	       if (i == -1) {
		  fprintf( stderr,"pmdb: reading next file record: %s\n",
			   strerror(errno));
		  exit( errno);
	       }
	    }
	 }
      }
      i = pmdb_close();
   }

   /* ---------- lookup a file entry
    *
    */

   else if (Option_F) {	 
      i = pmdb_open(path,GDBM_READER);
      if (i) {
	 fprintf( stderr,"pmdb: opening database: %s\n",
		  strerror( i));
	 exit( i);
      }	 
      if (0 != (i = pmdb_read_first_file( &record))) {
	 if (errno != 0) {
	    fprintf( stderr,"pmdb: reading first file record: %s\n",
		     strerror(errno));
	    exit( errno);
	 }
	 else printf("Not found\n");
      }
      else {
	 print_file_rec( &record);
	 pmdb_free_file_rec( &record);

	 /* --- loop through the file */

	 while (0 == (i = pmdb_read_next_file( &record))) {
	    print_file_rec( &record);
	    pmdb_free_file_rec( &record);
	 }
	 if (i == -1) {
	    fprintf( stderr,"pmdb: reading next file record: %s\n",
		     strerror(errno));
	    exit( errno);
	 }
      }	 
        i = pmdb_close();
    }

   /* ---------- lookup a package entry
    *
    */

   else if (Option_p) {
      if (Debug) printf("calling pmdb_open...\n");
      i = pmdb_open(path,GDBM_READER);
      if (i) {
	 fprintf( stderr,"pmdb: opening database: %s\n",
		  strerror( i));
	 exit( i);
      }
      if (Debug) printf("  result of pmdb_open is %d\n", i);

      /* ----- see if a pathname or just a file name
       *
       *  If the fileName supplied has a "/" in it assume that it is a
       *  pathname, otherwise do a full file scan using the string as
       *  a filename.
       */
      
      if (Debug) printf("calling pmdb_read_package <%s>...\n",
			fileName);
      if (0 != (i = pmdb_read_package( fileName, &package))) {
	 if (errno != 0) {
	    fprintf( stderr,"pmdb: reading package record: %s\n",
		     strerror(errno));
	    exit( errno);
	 }
	 else printf("Not found\n");
      }
      else {
	 print_package_rec( &package);
	 if (Option_l) print_package_files( &package);
	 pmdb_free_package_rec( &package);
      }	 
 
      if (Debug) printf("calling pmdb_close...\n");
      i = pmdb_close();
      if (Debug) printf("  result of pmdb_close is %d\n", i);  
   }

   /* ---------- lookup all package entry
    *
    */

   else if (Option_P) {
      if (Debug) printf("calling pmdb_open...\n");
      i = pmdb_open(path,GDBM_READER);
      if (i) {
	 fprintf( stderr,"pmdb: opening database: %s\n",
		  strerror( i));
	 exit( i);
      }	 
      if (Debug) printf("  result of pmdb_open is %d\n", i);
 
      if (Debug) printf("calling pmdb_read_first_package...\n");
      if (0 != (i = pmdb_read_first_package( &package))) {
	 if (errno != 0) {
	    fprintf( stderr,"pmdb: reading first package record: %s\n",
		     strerror(errno));
	    exit( errno);
	 }
	 else printf("Not found\n");
      }
      else {
	 print_package_rec( &package);
	 pmdb_free_package_rec( &package);
	 /* --- loop through the file */
	 if (Debug) printf("calling pmdb_read_next_package...\n");
	 while (0 == (i = pmdb_read_next_package( &package))) {
	    print_package_rec( &package);
	    if (Option_l) print_package_files( &package);
	    pmdb_free_package_rec( &package);
	 }
	 if (i == -1) {
	    fprintf( stderr,"pmdb: reading next package record: %s\n",
		     strerror(errno));
	    exit( errno);
	 }
      }	 
 
      if (Debug) printf("calling pmdb_close...\n");
      i = pmdb_close();
      if (Debug) printf("  result of pmdb_close is %d\n", i);
   }

   /* ===============  get out'a Dodge  =============== */

   if (path) free( (char *)path );
   exit(0);

} /* end main */

