/* config.c -- Configuration file support
 * Created: Wed May  3 13:11:37 1995 by r.faith@ieee.org
 * Revised: Tue Sep 19 09:54:34 1995 by r.faith@ieee.org
 * Copyright 1995 Rickard E. Faith (r.faith@ieee.org)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: config.c,v 1.12 1995/09/29 13:32:52 faith Exp $
 * 
 */

#define EXTERN /* */
#include "pmlib.h"

typedef enum { String, Flag } Types;

typedef struct Config {
   const char   *name;
   Types        type;
   const char   **var;
   const char   *def;
   int          *flag;
   int          flagDef;
} Config;

/* WARNING: this table is seached serially for exact matches.  If, at some
   later time, the ability to use only a prefix is provided, then the order
   of the table will become significant.  Further, there is an assumption
   that BUILDDIR will be an environment variable accessible to the user's
   script, so make sure that this is not changed. */

static Config config[] = {
   { "path",         String, &PmPath,        \
     ".:/sbin:/usr/sbin:/bin:/usr/bin:/usr/bin/X11" },
   { "distribution", String, &PmDistribution, "pm"                           },
   { "pmroot",       String, &PmRoot,         "/usr/src/@distribution@"      },
   { "notesdir",     String, &PmNotesDir,
     "@pmroot@/notes:@pmroot@/notes_nonfree:@pmroot@/notes_private" },
   { "sourcedir",    String, &PmSourceDir,    "$notes$sources$:/usr/src/gnu" },
   { "builddir",     String, &PmBuildDir,     "@pmroot@/build"               },
   { "buildsubdir",  String, &PmBuildSubDir,  "$NAME-$VERSION"               },
   { "srcdistdir",   String, &PmSrcDistDir,   "$notes$dist$:@pmroot@/dist"   },
   { "bindistdir",   String, &PmBinDistDir,   "$notes$dist$:@pmroot@/dist"   },
   { "docdir",       String, &PmDocDir,       "/usr/doc"                     },
   { "infodir",      String, &PmInfoDir,      "/usr/info"                    },
   { "usegroup",     Flag,   0, 0, &PmUseGroup,     0                        },
   { "forcegroup",   Flag,   0, 0, &PmForceGroup,   0                        },
   { "umask",        Flag,   0, 0, &PmUmask,        022                      },
   { "forceinstall", Flag,   0, 0, &PmForceInstall, 0                        },
   { "hash",         Flag,   0, 0, &PmHash,         0                        },
   { "quiet",        Flag,   0, 0, &PmQuiet,        0                        },
   { "pedantic",     Flag,   0, 0, &PmPedantic,     0                        },
   { "nodb",         Flag,   0, 0, &PmNoDatabase,   0                        },
   { "notesext",     String, &PmNotesExt,     ".Notes"                       },
   { "patchext",     String, &PmPatchExt,     ".Patch"                       },
   { "srcext",       String, &PmSrcExt,       PM_SOURCE_EXT                  },
   { "binext",       String, &PmBinExt,       PM_BINARY_EXT                  },
   { "tarroot",      String, &PmTarRoot,      "/"                            },
   { "instroot",     String, &PmInstRoot,     "/"                            },
   { "tmpdir",       String, &PmTmp,          "/tmp"                         },
   { "dbdir",        String, &PmDBDir,        "/var/lib/pm"                  },
   { "bourneshell",  String, &PmBourneShell,  "/bin/sh"                      },
   { 0, 0, 0, 0 }
};

static void load( const char *filename )
{
   FILE   *str;
   char   buffer[BUFSIZ];
   char   *value;
   Config *pt;
   char   *p;

   if (!(str = fopen( filename, "r" ))) return;
   
   while (fgets( buffer, BUFSIZ, str )) {
      if (buffer[0] == '#') continue;
      for (value = buffer;
	   *value && *value != ' ' && *value != '\t' && *value != '\n';
	   value++);
      *value = '\0';		/* buffer now points to name */
      ++value;

				/* remove heading spaces from value */
      while (*value == ' ' || *value == '\t' || *value == '\n') ++value;
      if (!*value) {
	 pm_warning( PMERR_INVCNFG,
		     "in %s: \"%s\" is not a valid line\n",
		     filename,
		     buffer );
	 continue;
      }

				/* Remove comment from end of value */
      for (p = value; *p && *p != '#'; ++p);
      *p = '\0';

				/* Remove trailing space from value */
      for (p = value; *p && *p != ' '&& *p != '\t' && *p != '\n'; ++p);
      *p = '\0';

      for (pt = config; pt->name; pt++)
	 if (!strcmp( buffer, pt->name )) {
	    if (pt->type == String) {
	       xfree( (void *)*pt->var );
	       *pt->var = xstrdup( value );
	    } else {
	       assert( pt->type == Flag );
	       *pt->flag = strtol( value, NULL, 0);
	    }
	    goto cont;
	 }
      pm_warning( PMERR_INVCNFG,
		  "in %s: \"%s\" is not a valid field\n",
		  filename,
		  buffer );
cont:
   }
   
   fclose( str );
}

static Config *find( const char *start, const char *end )
{
   Config *pt;
   int    length = end - start + 1 - 2;

   for (pt = config; pt->name; pt++)
      if (!strncmp( pt->name, start + 1, length )) return pt;
   return NULL;
}

static void expand_macros( void )
{
   int        flag;
   Config     *pt, *subst;
   char       *start, *end;
   char       *new;
   int        length;
   const char *s;
   char       *d;

   for (flag = 1; flag;) for (flag = 0, pt = config; pt->name; ++pt) {
      if (pt->type == String && (start = strchr( *pt->var, '@'))) {
	 end = strchr( start + 1, '@');
	 if (end) {
	    if (end == start + 1)
	       memmove( start, end, strlen( end ) ); /* @@ -> @ */
	    else if ((subst = find( start, end ))) {
	       ++flag;
	       length = strlen( *subst->var ) /* new */
			+ strlen( *pt->var )  /* old */
			- (end - start + 1)   /* what is taken away */
			+ 1;	              /* terminating null */
	       new = xmalloc( length );
	       
				/* copy first part */
	       for (s = *pt->var, d = new; s != start;) *d++ = *s++;
	       
				/* copy new thing */
	       if (end == start + 1) *d++ = '@';
	       else for (s = *subst->var; *s;) *d++ = *s++;
	       
				/* copy rest of old thing */
	       for (s = end + 1; *s;) *d++ = *s++;
	       
				/* terminating null */
	       *d = '\0';
	       
	       xfree( (void *)*pt->var );
	       *pt->var = new;
	    }
	 }
      }
   }
}

static void expand_tilde( void )
{
   Config        *pt;
   const char    *slash = NULL;
   struct passwd *pw;
   char          *username;
   char          *new;

   for (pt = config; pt->name; pt++) {
      const char *value;

      if (pt->type != String) continue;

      value = *pt->var;
      assert( value );
      
      if (*value != '~') continue;
      if (value[1] == '/' || value[1] == '\0') {
	 if ((username = getlogin())) pw = getpwnam( username );
	 else                         pw = getpwuid( getuid() );
	 if (!pw)
	    pm_fatal( PMERR_NOUSER, "\n" );
	 slash = value + 1;
      } else {
	 int length;

	 if (!(slash = strchr( value, '/' ))) slash = strchr( value, '\0' );

	 length = slash - value;
	 username = alloca( length + 1 );
	 strncpy( username, value, length );
	 username[ length ] = '\0';
	 pw = getpwnam( username + 1 );
	 if (!pw)
	    pm_fatal( PMERR_INVUSER, "%s\n", username + 1 );
      }
      new = xmalloc( strlen( pw->pw_dir ) + strlen( value ) + 1 );
      strcpy( new, pw->pw_dir );
      if (slash && *slash) {
	 strcat( new, "/" );
	 strcat( new, slash + 1 );
      }
      xfree( (void *)*pt->var );
      *pt->var = new;
   }
}

void pm_config( const char *specified )
{
   Config *pt;
   char   *p;

				/* Set global timestamp */
   time( &PmTimeStamp );
   PmTimeString = xstrdup( ctime( &PmTimeStamp ) );
   for (p = (char *)PmTimeString; *p && *p != '\n'; ++p); /* NLS-independent */
   *p = '\0';			/* Kill newline */
   
				/* Load defaults */
   for (pt = config; pt->name; ++pt) {
      switch (pt->type) {
      case String:
	 PRINTF(PM_CONFIG,("%s <- %s\n",pt->name,pt->def));
	 *pt->var = xstrdup( pt->def );
	 break;
      case Flag:
	 PRINTF(PM_CONFIG,("%s <- %d\n",pt->name,pt->flagDef));
	 *pt->flag = pt->flagDef;
	 break;
      default:
	 pm_fatal( PMERR_INTERNAL, "pm_config\n" );
      }
   }

				/* Override from config files */
   if (specified) {
      load( specified);
   } else {
      load( "/etc/PmConfig" );
      load( "~/.PmConfig" );
      load( "./.PmConfig" );
   }

				/* Macro expansion */
   expand_macros();
   expand_tilde();

   if (TEST(PM_CONFIG)) {
      for (pt = config; pt->name; pt++)
	 if (pt->type == String) printf( "%s = %s\n", pt->name, *pt->var );
	 else                    printf( "%s = %d\n", pt->name, *pt->flag );
   }

   pm_umask( PmUmask );
}

void pm_shutdown( void )
{
   Config *pt;

   for (pt = config; pt->name; pt++)
      if (pt->type == String)
	 xfree( (void *)*pt->var );
   if (PmTimeString) xfree( (char *)PmTimeString );
   PmTimeString = NULL;
}

void pm_config_env( FILE *str )
{
   Config *pt;
   char   buffer[BUFSIZ];

   for (pt = config; pt->name; pt++)
      if (pt->var) {
	 char *p = buffer;

	 strcpy( buffer, pt->name );
	 for (p = buffer; *p; *p++ = toupper( *p ));

	 if (!strcmp( pt->name, "BUILDDIR") && PmUseGroup && PmPkgGroup) {
				/* Kludge for groups */
	    const char *path = pm_file_path( 2, *pt->var, PmPkgGroup );
	    fprintf( str, "export %s=\"%s\"\n", buffer, path );
	 } else {
	    if (pt->type == String) {
	       fprintf( str, "export %s=\"%s\"\n", buffer, *pt->var );
	    } else {
	       assert( pt->type == Flag );
	       fprintf( str, "export %s=%d\n", buffer, *pt->flag );
	    }
	 }
      }
}
