/* db.c -- Interface to pmdb routines
 * Created: Sat May 13 10:13:27 1995 by r.faith@ieee.org
 * Revised: Thu Oct 24 10:46:10 1996 by faith@cs.unc.edu
 * Copyright 1995 Rickard E. Faith (r.faith@ieee.org)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: db.c,v 1.14 1996/11/11 19:28:28 faith Exp $
 * 
 */

#include "pmlib.h"
#include "pmdb.h"

static int dbOpen      = 0;
static int dbFirstOpen = 1;

static void print_file( const char *file, const char *disp, const char *msg )
{
   PMDBFileRec fr;

   if (msg || pmdb_read_file( file, &fr )) {
      printf( "%s %s %s\n", msg ?: "*", disp, file );
   } else {
      time_t mode  = strtol( fr.field[PMDB_FR_MODE], NULL, 8 );
      time_t mtime = strtol( fr.field[PMDB_FR_MTIME], NULL, 10 );
      time_t itime = strtol( fr.field[PMDB_FR_TIMEINSTALLED], NULL, 10 );

      printf( "%c%c%c%c%c%c%c%c%c%c %c%c %s %s\n",
	      fr.field[PMDB_FR_KIND][0],
	      (S_IRUSR & mode) ? 'r' : '-',
	      (S_IWUSR & mode) ? 'w' : '-',
	      ((S_ISUID & mode)
	       ? ((S_IXUSR & mode) ? 's' : 'S')
	       : ((S_IXUSR & mode) ? 'x' : '-')),
	      (S_IRGRP & mode) ? 'r' : '-',
	      (S_IWGRP & mode) ? 'w' : '-',
	      ((S_ISGID & mode)
	       ? ((S_IXGRP & mode) ? 's' : 'S')
	       : ((S_IXGRP & mode) ? 'x' : '-')),
	      (S_IROTH & mode) ? 'r' : '-',
	      (S_IWOTH & mode) ? 'w' : '-',
	      ((S_ISVTX & mode)
	       ? ((S_IXGRP & mode) ? 't' : 'T')
	       : ((S_IXGRP & mode) ? 'x' : '-')),
	      *disp,
	      fr.field[PMDB_FR_TYPE][0],
	      fr.field[PMDB_FR_PACKAGE],
	      file );
      printf( "%10s %24.24s %s %s",
	      fr.field[PMDB_FR_SIZE],
	      ctime( &mtime ),
	      fr.field[PMDB_FR_OWNER],
	      fr.field[PMDB_FR_GROUP] );
      if (fr.field[PMDB_FR_KIND][0] == PM_KIND_CHAR
	  || fr.field[PMDB_FR_KIND][0] == PM_KIND_BLOCK)
	 printf( " (%s,%s)",
		 fr.field[PMDB_FR_MAJOR], fr.field[PMDB_FR_MINOR] );
      printf( "\n" );

      printf( "           %24.24s (Ins) %s\n",
	      ctime( &itime ),
	      fr.field[PMDB_FR_CHECKSUM] );
      if (fr.field[PMDB_FR_LINK][0])
	 printf( "           Link to %s\n", fr.field[PMDB_FR_LINK] );

      pmdb_free_file_rec( &fr );
   }
}

static void print_package( PMDBPackageRec *pr,
			   int withHeader,
			   int withFiles,
			   int withLongFiles )
{
   PMDBAncilRec ar;
   int          i;
   time_t       itime = strtol( pr->field[PMDB_PR_TIMEINSTALLED], NULL, 10 );
   
   printf( "* %s installed %24.24s (%s) %s\n",
	   pr->field[PMDB_PR_KEY],
	   ctime( &itime ),
	   pr->field[PMDB_PR_GROUP],
	   pr->field[PMDB_PR_DISTRIBUTION] );
   if (withHeader) {
      if (!pmdb_read_ancil( pr->field[PMDB_PR_KEY], &ar )) {
	 for (i = 0; i < ar.count; i++) {
	    printf( "* %s %s\n", ar.type[i], ar.field[i] );
	 }
	 pmdb_free_ancil_rec( &ar );
      }
   }

   if (withFiles) {
      for (i = 0; i < pr->count; i++) {
	 print_file( pr->file[i], pr->disp[i], withLongFiles ? NULL : "*" );
      }
   }
}

static void pm_db_shutdown( void )
{
   if (dbOpen) {
      pm_warning( PMERR_DBSHUT, "\n" );
      pm_db_close();
   }
}

void pm_db_open( int writeFlag )
{
   int flags = writeFlag ? (GDBM_WRITER|GDBM_FAST) : GDBM_READER;

   if (PmNoDatabase) return;

   if (dbOpen) pm_fatal( PMERR_DBOPEN, "\n" );
   
   if (dbFirstOpen) atexit( pm_db_shutdown );
   dbFirstOpen = 0;

   PRINTF(PM_DB,("(Opening database for %s)\n",writeFlag?"write":"read"));
   
   if (pmdb_open( PmDBDir, flags )) {
      pm_file_hierarchy( PmDBDir, 1 );
      if (pmdb_open( PmDBDir, flags )) {
	 if (pmdb_init( PmDBDir ) || pmdb_open( PmDBDir, flags ))
	    pm_fatal( PMERR_NODB,
		      "%s, cannot %s\n",
		      PmDBDir,
		      writeFlag ? "write" : "read" );
      }
   }
   dbOpen = 1;
}

void pm_db_close( void )
{
   if (PmNoDatabase) return;

   if (!dbOpen) pm_fatal( PMERR_DBCLOSED, "\n" );
   
   PRINTF(PM_DB,("(Closing database)\n"));
   if (pmdb_close())
      pm_fatal( PMERR_DBCLOSE, "\n" );
   dbOpen = 0;
}

void pm_db_check_previous_version( void )
{
   PMDBPackageRec pr;
   int            retval;
   int            count = 0;

   if (PmNoDatabase) return;
   PRINTF(PM_DB,("(" __FUNCTION__ ")\n"));
   
   pm_db_open( 0 );
   
				/* Check previous packages */
   for (retval = pmdb_read_first_package( &pr );
	!retval;
	retval = pmdb_read_next_package( &pr )) {

      if (!strcmp( pr.field[PMDB_PR_NAME], PmPkgName )) {
	 if ((!PmPkgSubName && !pr.field[PMDB_PR_SUBNAME][0])
	     || (PmPkgSubName
		 && !strcmp( PmPkgSubName, pr.field[PMDB_PR_SUBNAME]))) {

	    print_package( &pr, TEST(PM_VERBOSE), 0, 0 );
	    ++count;
	 }
      }
      pmdb_free_package_rec( &pr );
   }

   pm_db_close();

   if (count) {
      printf( "* %d conflicting package%s installed,"
	      " use --force to overwrite\n",
	      count, count == 1 ? "" : "s" );
      pm_fatal( PMERR_INSTCONF, "\n" );
   }
}

int pm_db_check_previous_file( const char *file )
{
   PMDBFileRec fr;
   int         retval = 0;

   if (PmNoDatabase) return 0;
   PRINTF(PM_DB,("(" __FUNCTION__ "(%s))\n",file));
   
   pm_db_open( 0 );

   if (!pmdb_read_file( file, &fr )) {
      PMDBPackageRec pr;

      if (!pmdb_read_package( fr.field[PMDB_FR_PACKAGE], &pr )) {
				/* If !PmPedantic, skip conflicts within
                                   our own group.  E.g., emacs-19.28-el
                                   does not conflict with
                                   emacs-19.28-elc. */
	 if (!PmPedantic
	     && !strcmp( pr.field[PMDB_PR_NAME], PmPkgName )
	     && !strcmp( pr.field[PMDB_PR_VERSION], PmPkgVersion )
	     && ((!PmPkgRelease && !pr.field[PMDB_PR_RELEASE][0])
		 || (PmPkgRelease
		     && !strcmp( pr.field[PMDB_PR_RELEASE], PmPkgRelease )))) {
	    retval = 0;
	 } else {
	    int    i;
	    
				/* Find disposition. */
	    for (i = 0; i < pr.count; i++)
	       if (!strcmp( pr.file[i], file )) {
		  if (pr.disp[i][0] == PMDB_DISP_INSTALLED) {
		     retval = 1;

		     print_package( &pr, TEST(PM_VERBOSE), 0, 0 );
		     print_file( pr.file[i], pr.disp[i], " " );
		  }
		  break;
	       }
	 }

	 pmdb_free_package_rec( &pr );
      }
      pmdb_free_file_rec( &fr );
   }

   pm_db_close();
   
   return retval;
}

void pm_db_list_package_all( int longFlag )
{
   PMDBPackageRec pr;
   int            retval;
   int            count = 0;

   if (PmNoDatabase) return;
   
   pm_db_open( 0 );
   for (retval = pmdb_read_first_package( &pr );
	!retval;
	retval = pmdb_read_next_package( &pr )) {

      ++count;
      print_package( &pr, TEST(PM_VERBOSE), longFlag, TEST(PM_VERBOSE) );
      pmdb_free_package_rec( &pr );
   }

   printf( "* %d package%s installed\n", count, count == 1 ? "" : "s" );
   
   pm_db_close();
}

const char *pm_db_notesfile( const char *name )
{
   PMDBPackageRec pr;

   if (PmNoDatabase) return NULL;

   pm_db_open( 0 );
   if (!pmdb_read_package( name, &pr )) {
      pm_db_close();
      if (PmPkgName)         xfree( (char *)PmPkgName );
      if (PmPkgVersion)      xfree( (char *)PmPkgVersion );
      if (PmPkgRelease)      xfree( (char *)PmPkgRelease );
      if (PmPkgSubName)      xfree( (char *)PmPkgSubName );
      if (PmPkgGroup)        xfree( (char *)PmPkgGroup );
      if (PmPkgDistribution) xfree( (char *)PmPkgDistribution );
      if (PmPkgCopyright)    xfree( (char *)PmPkgCopyright );
      PmPkgName         = xstrdup( pr.field[PMDB_PR_NAME] );
      PmPkgVersion      = xstrdup( pr.field[PMDB_PR_VERSION] );
      PmPkgRelease      = xstrdup( pr.field[PMDB_PR_RELEASE] );
      PmPkgSubName      = xstrdup( pr.field[PMDB_PR_SUBNAME] );
      PmPkgGroup        = xstrdup( pr.field[PMDB_PR_GROUP] );
      PmPkgDistribution = xstrdup( pr.field[PMDB_PR_DISTRIBUTION] );

      return pm_gen_notespath( PmPkgSubName );
   }
   pm_db_close();
   return NULL;
}

void pm_db_list_package_byname( const char *name, int longFlag )
{
   PMDBPackageRec pr;
   int            retval;
   int            count = 0;

   if (PmNoDatabase) return;
   
   pm_db_open( 0 );
   for (retval = pmdb_read_first_package( &pr );
	!retval;
	retval = pmdb_read_next_package( &pr )) {

      if (!strcmp( pr.field[PMDB_PR_NAME], name )) {
	 ++count;
	 print_package( &pr, TEST(PM_VERBOSE), longFlag, TEST(PM_VERBOSE) );
      }
      pmdb_free_package_rec( &pr );
   }
   
   printf( "* %d package%s match%s\n",
	   count, count == 1 ? "" : "s", count == 1 ? "es" : "" );
   
   pm_db_close();
}

void pm_db_install_package( PmEntryList entries, List fileList, List dispList )
{
   PMDBPackageRec pr;
   PMDBAncilRec   ar;
   PmEntryList    p;
   int            i;
   char           itime[100];
   List           type;
   List           field;
   
   if (PmNoDatabase) return;
   PRINTF(PM_DB,("(" __FUNCTION__ ")\n"));
   
   pm_db_open( 1 );
   
   if (fileList->count != dispList->count)
      pm_fatal( PMERR_INTERNAL,
		"List counts not equal: %d, %d\n",
		fileList->count,
		dispList->count );

   sprintf( itime, "%lu", (long unsigned)PmTimeStamp );

   pr.count = fileList->count;
   pr.file  = fileList->lines;
   pr.disp  = dispList->lines;
   pr.field[PMDB_PR_KEY]           = pm_gen_canonical( PmPkgSubName );
   pr.field[PMDB_PR_NAME]          = PmPkgName         ?: "";
   pr.field[PMDB_PR_VERSION]       = PmPkgVersion      ?: "";
   pr.field[PMDB_PR_RELEASE]       = PmPkgRelease      ?: "";
   pr.field[PMDB_PR_SUBNAME]       = PmPkgSubName      ?: "";
   pr.field[PMDB_PR_GROUP]         = PmPkgGroup        ?: "";
   pr.field[PMDB_PR_DISTRIBUTION]  = PmPkgDistribution ?: "";
   pr.field[PMDB_PR_TIMEINSTALLED] = itime;

   PRINTF(PM_DB,("(" __FUNCTION__ ": calling pmdb_write_package)\n"));
   pmdb_write_package( &pr );
   PRINTF(PM_DB,("(" __FUNCTION__ ": pmdb_write_package done)\n"));

				/* Add ancillary data -- everything not in
                                   the PMDBPackageRec itself. */
   type  = pm_list_create();
   field = pm_list_create();

   pm_notes_build_ancillary( type, field );

   ar.key = pm_gen_canonical( PmPkgSubName );
   ar.count = type->count;
   ar.type  = type->lines;
   ar.field = field->lines;
   PRINTF(PM_DB,("(" __FUNCTION__ ": calling pmdb_write_ancil)\n"));
   pmdb_write_ancil( &ar );
   PRINTF(PM_DB,("(" __FUNCTION__ ": pmdb_write_ancil done)\n"));

   pm_list_free( field );
   pm_list_free( type );
   

				/* Iterate by hand because of the tree data
				   structure that must be in sync. */

   for (p = entries, i = 0; p && p->entry; p = p->next, i++) {
      PmEntry e = p->entry;
      
      assert( !strcmp( e->name, fileList->lines[i] ) );
      if (dispList->lines[i][0] == PMDB_DISP_INSTALLED) {
	 PMDBFileRec fr;
	 char        typeS[2];
	 char        kindS[2];
	 char        sizeS[100];
	 char        modeS[100];
	 char        mtimeS[100];
	 char        majorS[100];
	 char        minorS[100];

	 typeS[0] = e->type; typeS[1] = '\0';
	 kindS[0] = e->kind; kindS[1] = '\0';
	 
	 sprintf( sizeS,  "%lu", (unsigned long)e->size );
	 sprintf( modeS,  "0%o", e->mode );
	 sprintf( mtimeS, "%lu", (unsigned long)e->mtime );
	 sprintf( majorS, "%d",  major( e->device ) );
	 sprintf( minorS, "%d",  minor( e->device ) );
	 sprintf( itime,  "%lu", (long unsigned)PmTimeStamp );
	 
	 fr.field[PMDB_FR_KEY]           = e->name;
	 fr.field[PMDB_FR_TYPE]          = typeS;
	 fr.field[PMDB_FR_KIND]          = kindS;
	 fr.field[PMDB_FR_PACKAGE]       = pm_gen_canonical( PmPkgSubName );
	 fr.field[PMDB_FR_SIZE]          = sizeS;
	 fr.field[PMDB_FR_MODE]          = modeS;
	 fr.field[PMDB_FR_MTIME]         = mtimeS;
	 fr.field[PMDB_FR_MAJOR]         = majorS;
	 fr.field[PMDB_FR_MINOR]         = minorS;
	 fr.field[PMDB_FR_OWNER]         = e->user;
	 fr.field[PMDB_FR_GROUP]         = e->group;
	 fr.field[PMDB_FR_CHECKSUM]      = e->checksum ?: "";
	 fr.field[PMDB_FR_LINK]          = e->link ?: "";
	 fr.field[PMDB_FR_TIMEINSTALLED] = itime;
	 
	 PRINTF(PM_DB,("(" __FUNCTION__ ": pmdb_write_file(%s))\n",
		       e->name));
	 pmdb_write_file( &fr, 1 );
      }
   }
      
   pm_db_close();
}

const char *pm_db_get_checksum( const char *filename )
{
   PMDBFileRec fr;
   const char  *retval;
   int         needsClose = 0;

   PRINTF(PM_DB,("(" __FUNCTION__ "(%s))\n",filename));

   if (PmNoDatabase) return NULL;
   
   if (!dbOpen) {
      pm_db_open( 0 );
      ++needsClose;
   }
   
   if (pmdb_read_file( filename, &fr )) retval = xstrdup( "0" );
   else {
      retval = xstrdup( fr.field[PMDB_FR_CHECKSUM] );
      pmdb_free_file_rec( &fr );
   }
   if (needsClose) pm_db_close();
   
   return retval;
}

void pm_db_rename( const char *oldname, const char *newname )
{
   PMDBFileRec fr;
   const char  *oldkey;
   int         needsClose = 0;
   
   if (PmNoDatabase) return;
   
   if (!dbOpen) {
      pm_db_open( 1 );
      ++needsClose;
   }
   
   if (!pmdb_read_file( oldname, &fr )) {
      oldkey = fr.field[PMDB_FR_KEY];
      
      pmdb_delete_file( oldkey ); /* delete old */
      fr.field[PMDB_FR_KEY] = newname;
      pmdb_write_file( &fr, 0 ); /* add new */

      fr.field[PMDB_FR_KEY] = oldkey; /* put back before free */
      pmdb_free_file_rec( &fr );
   }
   
   if (needsClose) pm_db_close();
}

void pm_db_delete_file( const char *filename )
{
   int needsClose = 0;
   
   if (PmNoDatabase) return;
   
   if (!dbOpen) {
      pm_db_open( 1 );
      ++needsClose;
   }
   
   pmdb_delete_file( filename );
   
   if (needsClose) pm_db_close();
}

void pm_db_delete_package( const char *name )
{
   PMDBPackageRec pr;
   PMDBPackageRec newPr;
   int            i;
   List           fileList;
   List           dispList;
   struct l {
      PMDBFileRec *fr;
      struct l    *next;
   }              *frHead = NULL;

   static void preserve( const char *newName, PMDBFileRec *fr )
      {
	 char        installed[2] = { PMDB_DISP_INSTALLED, 0 };
	 struct l    *next = xmalloc( sizeof( struct l ) );
	 PMDBFileRec *new = xmalloc( sizeof( PMDBFileRec ) );
	 int         j;

	 for (j = 0; j < PMDB_FR_ENTRIES; j++)
	    new->field[j] = xstrdup( fr->field[j] );
	 if (strcmp( newName, fr->field[PMDB_FR_KEY] )) {
				/* Change name */
	    xfree( (void *)new->field[PMDB_FR_KEY] );
	    new->field[PMDB_FR_KEY] = xstrdup( newName );
	 }

	 next->fr   = new;
	 next->next = frHead;
	 frHead     = next;
	 
	 ++newPr.count;
	 pm_list_add( fileList, name );
	 pm_list_add( dispList, installed );
      }

   if (PmNoDatabase) return;

   pm_db_open( 1 );
   
   if (!pmdb_read_package( name, &pr )) {
      newPr.count = 0;
      newPr.file  = (fileList = pm_list_create())->lines;
      newPr.disp  = (dispList = pm_list_create())->lines;
      for (i = 0; i < PMDB_PR_ENTRIES; i++)
	 newPr.field[i] = xstrdup( pr.field[i] );
   
      for (i = 0; i < pr.count; i++) {
	 if (pr.disp[i][0] == PMDB_DISP_INSTALLED) {
	    PMDBFileRec fr;

	    if (!pmdb_read_file( pr.file[i], &fr )) {
	       switch (fr.field[PMDB_FR_TYPE][0]) {
	       case PM_TYPE_CONFIG: {
		  CheckSum   cksum        = pm_checksum_file( pr.file[i] );
		  const char *cksumString = pm_checksum_tostring( cksum );

		  if (!strcmp( cksumString, fr.field[PMDB_FR_CHECKSUM] )) {
		     if (pm_unlink( pr.file[i] )) {
			if (!PmQuiet)
			   printf( "* Cannot delete config file %s\n",
				   pr.file[i] );
			preserve( pr.file[i], &fr );
		     } else {
			if (!PmQuiet)
			   printf( "* Deleted config file %s\n", pr.file[i] );
		     }
		  } else {
		     char *buffer = alloca( strlen( pr.file[i] ) + 5 );

		     strcpy( buffer, pr.file[i] );
		     strcat( buffer, ".");
		     strcat( buffer, PM_OLD );
		     if (!PmQuiet) {
			printf( "* Renaming config file %s to %s\n",
				pr.file[i],
				buffer );
			printf( "*   (Config file %s removed from database)\n",
				pr.file[i] );
		     }
		     pm_rename( pr.file[i], buffer );
		     pmdb_delete_file( pr.file[i] );
		     preserve( buffer, &fr );
		  }

		  xfree( (char *)cksumString );
		  pm_checksum_free( cksum );
		  break;
	       }
	       case PM_TYPE_REGULAR:
		  if (pm_unlink( pr.file[i] )) {
		     if (!PmQuiet)
			printf( "* Cannot delete file %s\n", pr.file[i] );
		     preserve( pr.file[i], &fr );
		  } else {
		     if (!PmQuiet)
			printf( "* Deleted file %s\n", pr.file[i] );
		  }
		  break;
	       case PM_TYPE_DIRECTORY:
				/* Do directories later, after the files
                                   have been removed. */
		  break;
	       }
	       pmdb_free_file_rec( &fr );
	    }
	 }
      }
				/* Loop again to delete directories */
      for (i = 0; i < pr.count; i++) {
	 if (pr.disp[i][0] == PMDB_DISP_INSTALLED) {
	    PMDBFileRec fr;
	    
	    if (!pmdb_read_file( pr.file[i], &fr )) {
	       switch (fr.field[PMDB_FR_TYPE][0]) {
	       case PM_TYPE_DIRECTORY:
		  if (pm_rmdir( pr.file[i] )) {
		     if (!PmQuiet)
			printf( "* Cannot delete directory %s\n",
				pr.file[i] );
		     preserve( pr.file[i], &fr );
		  } else {
		     if (!PmQuiet)
			printf( "* Deleting directory %s\n", pr.file[i] );
		  }
		  break;
	       }
	       pmdb_free_file_rec( &fr );
	    }
	 }
      }
      if (!PmQuiet)
	 printf( "* Deleting %s package from database\n", name );
      if (newPr.count) {
	 PMDBAncilRec an;
	 struct l     *pt;

				/* Preserver ancillary information */
	 pmdb_read_ancil( name, &an );
				/* Delete package and all file records */
	 pmdb_delete_package( name );
				/* Write back package */
	 pmdb_write_package( &newPr );
				/* Write back preserved file records */
	 for (pt = frHead; pt; pt = pt->next)
	    pmdb_write_file( pt->fr, 1 );
				/* Write back ancillary information */
	 pmdb_write_ancil( &an );

				/* Free memory */
	 pmdb_free_ancil_rec( &an );
	 for (pt = frHead; pt; pt = pt->next) {
	    struct l *next = pt->next;
	    
	    pmdb_free_file_rec( pt->fr );
	    xfree( pt->fr );
	    xfree( pt );
	    pt = next;
	 }
	 
      } else {
	 pmdb_delete_package( name );
      }
      pmdb_free_package_rec( &pr );
   } else {
      pm_warning( PMERR_NOPACKAGE, "%s\n", name );
   }
   
   pm_db_close();
}

void pm_db_whence( const char *filename )
{
   char           cwd[MAXPATHLEN];
   PMDBFileRec    fr;
   PMDBPackageRec pr;

   if (PmNoDatabase) return;
   
   pm_db_open( 0 );
   
   if (*filename != '/') {
      const char *path;
				/* Relative path.  The database doesn't
                                   deal well with these, but we'll try to
                                   fix it up. */

      getcwd( cwd, MAXPATHLEN );
      
      if (*filename == '.') {
	 if (filename[1] == '/') path = pm_file_path( 2, cwd, filename + 2 );
	 else {
	    pm_warning( PMERR_RELATIVE, "%s\n", filename );
	    return;
	 }
      } else {
	 path = pm_file_path( 2, cwd, filename );
      }
      if (pmdb_read_file( path, &fr )) {
	 printf( "* Not in databse: %s\n", path );
	 xfree( (char *)path );
	 return;
      }
      xfree( (char *)path );
   } else {
      if (pmdb_read_file( filename, &fr )) {
	 printf( "* Not in databse: %s\n", filename );
	 return;
      }
   }

   printf( "\n* %s is from:\n", fr.field[PMDB_FR_KEY] );
   if (pmdb_read_package( fr.field[PMDB_FR_PACKAGE], &pr )) {
      printf( "* %s (cannot locate, database corrupt)\n",
	      fr.field[PMDB_FR_PACKAGE] );
   } else {
      print_package( &pr, TEST(PM_VERBOSE), 0, 0 );
   }

   if (fr.field[PMDB_FR_KIND][0] == PM_KIND_REGULAR) {
      CheckSum cksum          = pm_checksum_file( fr.field[PMDB_FR_KEY] );
      const char *cksumString = pm_checksum_tostring( cksum );

      if (strcmp( cksumString, fr.field[PMDB_FR_CHECKSUM] ))
	 printf( "   **** Checksum mismatch: %s\n"
		 "                           %s\n",
		 fr.field[PMDB_FR_CHECKSUM],
		 cksumString );
      else
	 printf( "   * Checksums match: %s\n", fr.field[PMDB_FR_CHECKSUM] );
      xfree( (char *)cksumString );
      pm_checksum_free( cksum );
   }

   pm_db_close();
}
