/* error.c -- Standard error messages
 * Created: Tue May  2 15:39:14 1995 by r.faith@ieee.org
 * Revised: Sun Aug 13 12:05:18 1995 by r.faith@ieee.org
 * Copyright 1995 Rickard E. Faith (r.faith@ieee.org)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * 
 * $Id: error.c,v 1.4 1995/08/13 16:58:44 faith Exp $
 * 
 */

#include "pmlib.h"
#include <stdarg.h>
#include <signal.h>

extern const char *PgmName;

static int print_error( FILE *str, unsigned int message )
{
   fprintf( str, "%s: ", PgmName );
   
   switch (message) {
   case PMERR_INVDBFLG:  fprintf( str, "Invalid debug flag: " );      return 0;
   case PMERR_INVCNFG:   fprintf( str, "Invalid config: " );          return 0;
   case PMERR_INVUSER:   fprintf( str, "Invalid username: " );        return 0;
   case PMERR_INVSETUP:  fprintf( str, "Invalid %%setup flag: " );    return 0;
   case PMERR_INVPATCH:  fprintf( str, "Invalid %%patch flag: " );    return 0;
   case PMERR_INVMACRO:  fprintf( str, "Invalid macro: " );           return 0;
   case PMERR_INTERNAL:  fprintf( str, "Internal error: " );          return 0;
   case PMERR_NOUSER:    fprintf( str, "Cannot find username" );      return 0;
   case PMERR_NOGROUP:   fprintf( str, "Cannot find group name" );    return 0;
   case PMERR_NOHOST:    fprintf( str, "Cannot find host name" );     return 0;
   case PMERR_NOFILE:    fprintf( str, "Cannot find file: " );        return 0;
   case PMERR_NOMATCH:   fprintf( str, "No match: " );                return 0;
   case PMERR_TOOLONG:   fprintf( str, "Name too long: " );           return 0;
   case PMERR_STAT:      fprintf( str, "Cannot stat file: " );        return 1;
   case PMERR_LSTAT:     fprintf( str, "Cannot lstat file: " );       return 1;
   case PMERR_EOF:       fprintf( str, "Unexpected end of file: " );  return 1;
   case PMERR_READOPEN:  fprintf( str, "Cannot read file: " );        return 1;
   case PMERR_WRITEOPEN: fprintf( str, "Cannot write file: " );       return 1;
   case PMERR_DIROPEN:   fprintf( str, "Cannot read directory: " );   return 1;
   case PMERR_CHILDREN:  fprintf( str, "Too many children exist: " ); return 0;
   case PMERR_CHILDERR:  fprintf( str, "Child failed: " );            return 0;
   case PMERR_SCRIPTERR: fprintf( str, "Script failed: " );           return 0;
   case PMERR_PIPE:      fprintf( str, "Pipe error" );                return 1;
   case PMERR_FORK:      fprintf( str, "Fork error" );                return 1;
   case PMERR_EXEC:      fprintf( str, "Exec error" );                return 1;
   case PMERR_NOCORE:    fprintf( str, "Out of memory: " );           return 0;
   case PMERR_NULLPT:    fprintf( str, "Freeing null pointer" );      return 0;
   case PMERR_SIG:       fprintf( str, "Signal handling problem" );   return 1;
   case PMERR_SIGNAL:    fprintf( str, "Received signal: " );         return 0;
   case PMERR_NONOTES:   fprintf( str, "No notes file: " );           return 0;
   case PMERR_NOTAR:     fprintf( str, "No distribution file: " );    return 0;
   case PMERR_NOPATCH:   fprintf( str, "No patch file: " );           return 0;
   case PMERR_NOSOURCE:  fprintf( str, "No source file: " );          return 0;
   case PMERR_AMBIG:     fprintf( str, "Ambiguous package name: " );  return 0;
   case PMERR_BUILD:     fprintf( str, "Build failed" );              return 0;
   case PMERR_PACK:      fprintf( str, "Packaging failed" );          return 0;
   case PMERR_INST:      fprintf( str, "Installation failed" );       return 0;
   case PMERR_FILESMAC:  fprintf( str, "Too many arguments: " );      return 0;
   case PMERR_TAR:       fprintf( str, "Error reading tar file: " );  return 0;
   case PMERR_UNTAR:     fprintf( str, "Error untar'ing file: " );    return 0;
   case PMERR_READLINK:  fprintf( str, "Cannot read link: " );        return 0;
   case PMERR_NOTES:     fprintf( str, "Notes error: " );             return 0;
   case PMERR_NODB:      fprintf( str, "Cannot open database: " );    return 1;
   case PMERR_DBCLOSE:   fprintf( str, "Cannot close database" );     return 1;
   case PMERR_INSTCONF:  fprintf( str, "Installation conflicts" );    return 0;
   case PMERR_NOPACKAGE: fprintf( str, "Cannot locate package: " );   return 0;
   case PMERR_RELATIVE:  fprintf( str, "Relative path: " );           return 0;
   case PMERR_DBOPEN:    fprintf( str, "Database already open" );     return 0;
   case PMERR_DBCLOSED:  fprintf( str, "Database already closed" );   return 0;
   case PMERR_DBSHUT:    fprintf( str, "Database open during exit" ); return 0;
   }
   fprintf( str, "Unknown error %d: ", message );
   return 0;
}

void pm_fatal( unsigned int message, const char *format, ... )
{
   va_list ap;
   int     old_errno = errno;
   int     errno_valid;

				/* Restore signal handlers */
   signal( SIGINT, SIG_DFL );
   signal( SIGQUIT, SIG_DFL );
   
   fflush( stdout );
   errno_valid = print_error( stderr, message );
   va_start( ap, format );
   vfprintf( stderr, format, ap );
   va_end( ap );
   if (errno_valid) {
      errno = old_errno;
      perror( PgmName );
   }
   fflush( stderr );
   fflush( stdout );
   exit( message );
}

void pm_warning( unsigned int message, const char *format, ... )
{
   va_list ap;
   int     old_errno = errno;
   int     errno_valid;

   fflush( stdout );
   errno_valid = print_error( stderr, message );
   va_start( ap, format );
   vfprintf( stderr, format, ap );
   va_end( ap );
   if (errno_valid)
      fprintf( stderr, "%s: %s\n", PgmName, strerror( old_errno ) );
   fflush( stderr );
   fflush( stdout );
}
