/* exec.c -- Exec compress and tar
 * Created: Sat May  6 15:55:29 1995 by r.faith@ieee.org
 * Revised: Fri Sep 15 00:41:16 1995 by r.faith@ieee.org
 * Copyright 1995 Rickard E. Faith (r.faith@ieee.org)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: exec.c,v 1.7 1995/09/15 05:26:28 faith Exp $
 *
 * Note: The functions in this file are somewhat like system(3).  POSIX.2
 * semantics for system(3) requires that the parent ignore SIGINT and
 * SIGQUIT while the child is running.
 *
 */

#include "pmlib.h"
#include <fcntl.h>
#include <signal.h>

#if HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif
#ifndef WEXITSTATUS
# define WEXITSTATUS(stat_val) ((unsigned)(stat_val) >> 8)
#endif
#ifndef WIFEXITED
# define WIFEXITED(stat_val) (((stat_val) & 255) == 0)
#endif

#define PM_MAX_CHILD_COUNT 3

static int              childCount;
static struct sigaction ignore, intr, quit;
static sigset_t         block, savemask;

void pm_child_clear( void )
{
   if (childCount)
      pm_fatal( PMERR_CHILDREN, "%d, expecting 0\n", childCount );

				/* ignore SIGINT and SIGQUIT */
   ignore.sa_handler = SIG_IGN;
   ignore.sa_flags   = 0;
   if (sigaction( SIGINT, &ignore, &intr )) pm_fatal( PMERR_SIG, "\n" );
   if (sigaction( SIGQUIT, &ignore, &quit )) {
      int save = errno;
      sigaction( SIGINT, &intr, NULL );	/* restore old handler */
      errno = save;
      pm_fatal( PMERR_SIG, "\n" );
   }
   
				/* block SIGCHLD */
   sigemptyset( &block );
   sigaddset( &block, SIGCHLD );
   if (sigprocmask( SIG_BLOCK, &block, &savemask )) {
      int save = errno;
      sigaction( SIGINT, &intr, NULL );	/* restore old handler */
      sigaction( SIGQUIT, &quit, NULL ); /* restore old handler */
      errno = save;
      pm_fatal( PMERR_SIG, "\n" );
   }
}

static void pm_child_log( int pid, int fd )
{
   PRINTF(PM_CHILD,("(Child = %d, fd = %d)\n",pid,fd));
   if (childCount >= PM_MAX_CHILD_COUNT)
      pm_fatal( PMERR_CHILDREN,
		"%d, expecting < %d\n", childCount, PM_MAX_CHILD_COUNT );
   ++childCount;
}

int pm_child_wait( void )
{
   int status;
   int exitStatus = 0;
   int pid;
   
   while (childCount) {
      pid = wait( &status );
      PRINTF(PM_CHILD,("(Child (%d) status = %d)\n",pid,status));
      --childCount;
      if (WIFEXITED( status ))
	 exitStatus |= WEXITSTATUS( status );
      
				/* SIGPIPE is ok here, since tar may
                                   shutdown early.  Anything else is a
                                   problem.  */
      if (WIFSIGNALED( status ) && WTERMSIG( status ) != SIGPIPE)
	 exitStatus |= 128 + WTERMSIG( status ); /* like bash :-) */
   }

				/* Restore signals */
   sigaction( SIGINT, &intr, NULL );
   sigaction( SIGQUIT, &quit, NULL );
   sigprocmask( SIG_SETMASK, &savemask, NULL );
   
   return exitStatus;
}

static int pm_popen( int in, const char *bin, const char *command, int errors )
{
   List list;
   int  pid;
   int  fd[2];
   char buffer[MAXPATHLEN];	/* Arbitrary command length */

   strcpy( buffer, bin );
   strcat( buffer, " " );
   strcat( buffer, command );
   PRINTF(PM_EXEC,("(Executing %s)\n",buffer));
   list = pm_argify( buffer );

   pm_list_add( list, NULL );	/* for execvp */
   fflush( stdout );

   if (pipe( fd ) < 0)
      pm_fatal( PMERR_PIPE, "\n" );
   
   if ((pid = fork()) < 0)
      pm_fatal( PMERR_FORK, "\n" );

   if (pid > 0) {		/* Parent */
      pm_child_log( pid, in );
      close( fd[1] );
      if (in >= 0) close( in );
   } else {			/* Child */
				/* Restore signals */
      sigaction( SIGINT, &intr, NULL );
      sigaction( SIGQUIT, &quit, NULL );
      sigprocmask( SIG_SETMASK, &savemask, NULL );
      
      close( fd[0] );
      dup2( fd[1], STDOUT_FILENO );
      close( fd[1] );

      if (in >= 0) {
	 dup2( in, STDIN_FILENO );
	 close( in );
      }

      if (errors < 0) {		/* dup stderr to stdout */
	 dup2( STDOUT_FILENO, STDERR_FILENO );
      } else if (!errors) {	/* suppress stderr */
	 int fdout = open( "/dev/null", O_WRONLY );

	 if (fdout >= 0) {
	    dup2( fdout, STDERR_FILENO );
	    close( fdout );
	 }
      }

      execvp( bin, list->lines );
      
				/* Exec failed, but that's ok, since we may
                                   have renamed gzip to gzip.old and/or tar
                                   to tar.old. */
      strcpy( buffer, bin );
      strcat( buffer, ".");
      strcat( buffer, PM_OLD );
      execvp( buffer, list->lines );
      
				/* Exec failed: we're totally hosed. */
      fprintf( stderr, "Cannot exec %s\n", bin );
      _exit( 127 );
   }

   pm_list_free( list );
   return fd[0];
}


int pm_gunzip_filter( int fd )
{
   return pm_popen( fd, PM_GZIP_PROG, PM_GUNZIP_FILTER, TEST(PM_VERBOSE) );
}

int pm_gzip_filter( int fd )
{
   return pm_popen( fd, PM_GZIP_PROG, PM_GZIP_FILTER, TEST(PM_VERBOSE) );
}

static int pm_untar_filter( int fd )
{
   if (TEST(PM_VERBOSE))
      return pm_popen( fd, PM_TAR_PROG, PM_UNTAR_FILTER_VERBOSE, -1 );
   else
      return pm_popen( fd, PM_TAR_PROG, PM_UNTAR_FILTER, -1 );
}

/* Tar a list of files in listname to the tar file in filename */

int pm_tar( const char *listname, const char *root, const char *filename )
{
   char *command = alloca( strlen(PM_TARLIST_FILTER) + strlen(listname) + 1 );
   FILE *str     = pm_file_create( filename );
   int  tar, gzip;
   char buffer[PM_BUFFERSIZE];
   int  count;
   int  retval;
   char oldwd[MAXPATHLEN];

   getcwd( oldwd, MAXPATHLEN );
   chdir( root );
   sprintf( command, PM_TARLIST_FILTER, listname );
   
   pm_child_clear();
   tar = pm_popen( -1, PM_TAR_PROG, command, !PmQuiet );
   gzip = pm_gzip_filter( tar );

   while ((count = read( gzip, buffer, PM_BUFFERSIZE )) > 0) {
      fwrite( buffer, count, 1, str );
   }
   fclose( str );

   chdir( oldwd );

   retval = pm_child_wait();
   close( gzip );
   close( tar );
   
   PRINTF(PM_EXEC,("(Exit status = 0x%02x)\n",retval));
   return retval;
}

/* Untar a file into .  The function gets passed each line from tar.  It
   should be able to parse 'tar x', 'tar xv', and 'tar xvv' for best
   results. */

int pm_untar( const char *filename, int file, int compressed,
	      void (*f)( const char *line ) )
{
   int  tar;
   int  gzip;
   char buffer[PM_BUFFERSIZE];
   int  retval;
   FILE *str;

   pm_child_clear();

   if (compressed) {
      gzip = pm_gunzip_filter( file );
      tar  = pm_untar_filter( gzip );
   } else {
      tar  = pm_untar_filter( file );
   }

   if (!(str = fdopen( tar, "r" )))
      pm_fatal( PMERR_READOPEN, "%s (fdopen)\n", filename );
   
				/* Make line-buffered for efficient fgets */
   setvbuf( str, NULL, _IOLBF, 0 );
   while (fgets( buffer, PM_BUFFERSIZE, str )) f( buffer );

   fclose( str );

   retval = pm_child_wait();

   PRINTF(PM_EXEC,("(Untar exit status = 0x%02x)\n",retval));
   return retval;
}

int pm_shell( const char *command )
{
   List list = pm_argify( command );
   int  pid;
   int  retval;

   PRINTF(PM_EXEC,("(Executing %s)\n",command));

   pm_list_add( list, NULL );	/* for execvp */

   pm_child_clear();

   if ((pid = fork()) < 0)
      pm_fatal( PMERR_FORK, "\n" );

   if (pid > 0) {		/* Parent */
      pm_child_log( pid, -1 );
   } else {			/* Child */
				/* Restore signals */
      sigaction( SIGINT, &intr, NULL );
      sigaction( SIGQUIT, &quit, NULL );
      sigprocmask( SIG_SETMASK, &savemask, NULL );

      if (PmQuiet) {
	 int fd = open( "/dev/null", O_WRONLY );

	 if (fd >= 0) {
	    dup2( fd, STDOUT_FILENO );
	    close( fd );
	 }
      }

      if (PmQuiet) {
	 int fd = open( "/dev/null", O_WRONLY );

	 if (fd >= 0) {
	    dup2( fd, STDERR_FILENO );
	    close( fd );
	 }
      }
      
      if (!TEST(PM_DEBUG))
	 execv( list->lines[0], list->lines );
      fprintf( stderr, "Cannot exec %s\n", list->lines[0] );
      _exit( 127 );
   }

   pm_list_free( list );
   retval = pm_child_wait();
   
   PRINTF(PM_EXEC,("(Shell exit status = 0x%02x)\n",retval));
   return retval;
}

int pm_chroot( const char *root, int (*f)( void * ), void *param )
{
   int pid;
   int retval;
   
   PRINTF(PM_EXEC,("(Chroot %s)\n",root));
   
   pm_child_clear();

   if ((pid = fork()) < 0)
      pm_fatal( PMERR_FORK, "\n" );

   if (pid > 0) {		/* Parent */
      pm_child_log( pid, -1 );
   } else {			/* Child */
				/* Restore signals */
      sigaction( SIGINT, &intr, NULL );
      sigaction( SIGQUIT, &quit, NULL );
      sigprocmask( SIG_SETMASK, &savemask, NULL );

      if (!getuid()) {
	 if ((retval = chroot( root ))) {
	    _exit( retval );
	 }
      } else {
	 if (strcmp( root, "/" ))
	    fprintf( stderr, "Must be root to chroot(%s)\n", root );
      }
      _exit( f( param ) );
   }

   retval = pm_child_wait();
   
   PRINTF(PM_EXEC,("(Chroot exit status = 0x%02x)\n",retval));
   return retval;
}
