/* list.c -- String list management
 * Created: Wed May  3 12:31:55 1995 by r.faith@ieee.org
 * Revised: Thu Nov 16 11:31:33 1995 by r.faith@ieee.org
 * Copyright 1995 Rickard E. Faith (r.faith@ieee.org)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: list.c,v 1.3 1995/12/19 03:11:23 faith Exp $
 * 
 */

#include "pmlib.h"

List pm_list_create( void )
{
   List list = xmalloc( sizeof( struct List ) );

   list->count   = 0;
   list->maximum = 0;
   list->lines   = NULL;

   return list;
}

List pm_list_add( List list, const char *string )
{
   if (++list->count > list->maximum) {
      if (list->lines) {
	 list->maximum += 10;
	 list->lines = xrealloc( list->lines,
				 list->maximum * sizeof( char ** ) );
      } else {
	 list->maximum = 10;
	 list->lines = xmalloc( list->maximum * sizeof( char ** ) );
      }
   }

   list->lines[ list->count - 1 ] = string ? xstrdup( string ) : NULL;
   
   return list;
}

void pm_list_free( List list )
{
   int i;

   for (i = 0; i < list->count; i++)
      if (list->lines[i]) xfree( list->lines[i] );
   if (list->count) xfree( list->lines );
   xfree( list );
   list = NULL;
}

/* Read a stream into a List.  NOTE: Assumes that the stream is a
   newline-terminated text file and that no line is more than PM_BUFFERSIZE
   bytes long.  These are reasonable assumptions for this application. */

List pm_list_stream( FILE *str )
{
   char buffer[PM_BUFFERSIZE];
   List list = pm_list_create();

   while (fgets( buffer, PM_BUFFERSIZE, str )) {
      int last;

      buffer[PM_BUFFERSIZE-1] = '\0';
      last = strlen( buffer ) - 1;
      if (buffer[last] == '\n') buffer[last] = '\0';
      pm_list_add( list, buffer );
   }

   return list;
}

List pm_list_file( const char *filename )
{
   FILE *str;
   List list;

   if (!filename || access( filename, R_OK)) return NULL;
   str = pm_file_read( filename );
   list = pm_list_stream( str );
   pm_file_close( str );
   return list;
}

void pm_list_iterate( List list, int (*f)( const char * ) )
{
   int i;
				/* Don't call with a NULL */
   for (i = 0; i < list->count; i++)
      if (list->lines[i] && f( list->lines[i] )) return;
}

void pm_list_write( FILE *str, List list )
{
   int i;

   for (i = 0; i < list->count; i++)
      fprintf( str, "%s\n", list->lines[i] );
}
