/* tar.c -- Tar file support
 * Created: Tue May  2 17:34:08 1995 by r.faith@ieee.org from
 *          GNU Tar (prep.ai.mit.edu:/pub/gnu/tar-1.11.2.tar.gz).
 * Revised: Fri May 12 20:37:16 1995 by r.faith@ieee.org
 * Copyright (C) 1988, 1992, 1993 Free Software Foundation
 * Changes Copyright 1995 Rickard E. Faith (r.faith@ieee.org)
 *
 * From the GNU Tar files used (list.c and create.c):
 *
 * GNU Tar is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * GNU Tar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNU Tar; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: tar.c,v 1.1 1995/05/13 19:07:13 faith Exp $
 * 
 */

#include "pmlib.h"
#include "tar.h"
#include <fcntl.h>

#define isodigit(c) (((c) >= '0') && ((c) <= '7'))

extern const char *PgmName;

/* Quick and dirty octal conversion.  Result is -1 if the field is invalid
  (all blank, or nonoctal).  */

static long from_oct( int digs, char *where )
{
  long value;

  while (isspace( *where )) {	/* Skip spaces */
     where++;
     if (--digs <= 0)
	return -1;		/* All blank field */
  }
  value = 0;
  while (digs > 0 && isodigit( *where )) { /* Scan til nonoctal */
     value = (value << 3) | (*where++ - '0');
     --digs;
  }

  if (digs > 0 && *where && !isspace( *where ))
     return -1;			/* Ended on non-space/nul */

  return value;
}

/* Read what should be a tar header.  Return 0 on success, 1 on bad
   checksum, 2 on a record of all zeros (EOF), and EOF on true EOF.  If the
   checksum is valid and h is non-NULL, then copy the header structure. */

static int read_tar_header( FILE *str, TarHeader header )
{
   union record *h = (union record *)header;
   char         *p;
   int          i;
   long         sum        = 0;
   long         signed_sum = 0;
   long         recsum;

   if (!h) h = alloca( sizeof( union record ) );

   if (!fread( h, RECORDSIZE, 1, str )) return EOF;

   recsum = from_oct (8, h->header.chksum);
   
   p = h->charptr;
   
   for (i = sizeof( union record ); --i >= 0; /* void */) {
      /*
       * We can't use unsigned char here because of old compilers,
       * e.g. V7.
       */
      signed_sum += *p;
      sum += 0xff & *p++;
   }

   /* Adjust checksum to count the "chksum" field as blanks. */
   for (i = sizeof (h->header.chksum); --i >= 0;) {
      sum -= 0xff & h->header.chksum[i];
      signed_sum -= (char)h->header.chksum[i];
   }
   sum += ' ' * sizeof( h->header.chksum );
   signed_sum += ' ' * sizeof( h->header.chksum );

   if (sum == 8 * ' ') {
      /*
       * This is a zeroed record...whole record is 0's except
       * for the 8 blanks we faked for the checksum field.
       */
      return 2;
   }
   
   if (sum != recsum && signed_sum != recsum) return 1;

#if 0
   printf( "Checksum = %ld (signed = %ld, unsigned = %ld)\n",
	   recsum, signed_sum, sum );
   printf( "Size = %ld\n", from_oct( 1 + 12, header.header.size ) );
#endif

   return 0;
}

/* Given the size derived from header, read a sufficient number of blocks
   so that another header is now available. */

static int skip_to_next_header( FILE *str, TarHeader header )
{
   union record  *h = (union record *)header;
   long unsigned size = from_oct( 1 + 12, h->header.size );
   long unsigned records = size / RECORDSIZE;
   char          tmp[RECORDSIZE];
   int           i;

   if (size % RECORDSIZE) ++records;

				/* Can't use seek, since str might be a
                                   pipe. */
   for (i = 0; i < records; i++)
      if (fread( &tmp, RECORDSIZE, 1, str ) != 1) return 1;
   return 0;
}

/* Allocate a TarHeader. */

TarHeader pm_tar_alloc_header( void )
{
   union record *h = xmalloc( sizeof( union record ) );

   memset( h, 0, sizeof( union record ) );
   return h;
}

/* Free a TarHeader. */

void pm_tar_free_header( TarHeader header )
{
   if (header) xfree( header );
}

/* Get the next tar header from the tar file into TarHeader.  Return 0 on
  success, 1 on bad checksum, 2 on a record of all zeros (EOF), and EOF on
  true EOF.  TarHeader is only updated if everything was successful. */

int pm_tar_next_header( FILE *str, TarHeader header )
{
   union record *h = (union record *)header;

   if (pm_tar_size( h )) if (skip_to_next_header( str, h )) return EOF;
   return read_tar_header( str, h );
}

/* Get the size, in bytes, of the file. */

long unsigned pm_tar_size( TarHeader header )
{
   union record *h = (union record *)header;
   
   return from_oct( 1 + 12, h->header.size );
}

/* Get the name of the file */

const char *pm_tar_filename( TarHeader header )
{
   union record *h = (union record *)header;

   return h->header.arch_name;
}

/* Given the header, and assuming the input stream is positioned for the
   next record, read the file into a newly allocated buffer.  Return a
   pointer to that buffer.  NOTE: it is the caller's responsibility to free
   this buffer!  */

char *pm_tar_read( FILE *str, TarHeader header )
{
   union record  *h = (union record *)header;
   long unsigned size = from_oct( 1 + 12, h->header.size );
   long unsigned records = size / RECORDSIZE;
   char          *buffer;
   int           count;

   if (size % RECORDSIZE) ++records;

   if (records) {
      buffer = xmalloc( records * RECORDSIZE );
      memset( buffer + RECORDSIZE * (records - 1 ), 0, RECORDSIZE );
      
      if ((count = fread( buffer, RECORDSIZE, records, str )) != records)
	 pm_fatal( PMERR_EOF, "%d of %d records read\n", count, records );

      return buffer;
   } else
      return NULL;
}

/* Given the header, and assuming the input stream is positioned for the
   next record, read the file into a List structure.  NOTE: Assumes that
   the stream (or the gunzip'd stream) is a newline-terminated text file
   and that no line is more than PM_BUFFERSIZE bytes long.  These are
   reasonable assumptions for this application (i.e., reading a notes
   file).  Returns NULL on error.  On success, returns a List structure
   that must be deallocated by the user. */

List pm_tar_listify( FILE *str, TarHeader header )
{
   union record  *h       = (union record *)header;
   long unsigned size     = from_oct( 1 + 12, h->header.size );
   char          *image   = pm_tar_read( str, header );
   char          *imagePt = image;
   int           compressed;
   FILE          *s       = NULL;
   char          *tmp;
   List          list;
   char          buffer[PM_BUFFERSIZE];
   long unsigned total;
   char          *p;
   int           c;

   if (!image) return NULL;

   compressed = pm_buffer_compressed( image );

   if (compressed) {	/* buffer is compressed.  sigh. */
      FILE *tmpStream;
      int  fd;
      int gzip;
				/* Write image to a file.  We can't just
                                   save the file position and back up 2-4
                                   bytes because str is a pipe.  If we
                                   don't write to a file, we have to manage
                                   both an input and an output stream to
                                   the gunzip filter process.  If we don't
                                   fork, this would be hell.  But dealing
                                   with forking and making children is
                                   probably less reliable and a whole lot
                                   more work than writing the temporary
                                   file. */
      tmp = tempnam( PmTmp, PgmName );
      tmpStream = pm_file_create( tmp );
      fwrite( image, size, 1, tmpStream );
      fclose( tmpStream );

      if ((fd = open( tmp, O_RDONLY )) < 0)
	 pm_fatal( PMERR_READOPEN, "%s\n", tmp );
				/* pm_child_clear() was called already!!! */
      gzip = pm_gunzip_filter( fd );
      
      if (!(s = fdopen( gzip, "r" )))
	 pm_fatal( PMERR_READOPEN, "%s (fdopen)\n", tmp );
   }

   list = pm_list_create();
   
   for (p = buffer, total = 0; compressed || total < size; ++total) {
      if (compressed) c = getc( s );
      else            c = *imagePt++;
      
				/* c == '\0' should never happen, but if we
                                   got a corrupt tar file, it is
                                   possible. */

      if (c == EOF) break;	/* probably from gzip */
      if (c == '\n' || c == '\0') {
	 *p = '\0';
	 pm_list_add( list, buffer );
	 p = buffer;
      } else
	 *p++ = c;
   }
   if (p != buffer) {
				/* This never happens, but if some luser
                                   removed the final \n from the Notes
                                   file, we'd lose a file entry.  Of
                                   course, lusers shouldn't be editing
                                   these things anyway. . . */
      *p = '\0';
      if (*buffer == '/') pm_list_add( list, buffer );
   }

   xfree( image );
   if (compressed) {
      fclose( s );
				/* pm_child_wait() will be called later!!! */
   }
   
   return list;
}


				/* Below is some debugging stuff that is no
                                   longer needed.  Leave it here just in
                                   case someone needs it later. */


/* Decode things from a file header record into a "struct stat".  Also set
   "*stdp" to !=0 or ==0 depending whether header record is "Unix Standard"
   tar format or regular old tar format.

   read_header() has already decoded the checksum.  */

static void decode_header ( union record *header, struct stat *st, int *stdp )
{
   memset( st, 0, sizeof( struct stat ) );
   st->st_mode = from_oct( 8, header->header.mode );
   st->st_mode &= 07777;
   st->st_mtime = from_oct( 1 + 12, header->header.mtime );
   st->st_size = from_oct( 1 + 12, header->header.size );

   if (!strcmp( header->header.magic, TMAGIC )) {
      /* Unix Standard tar archive */
      *stdp = 1;
      st->st_uid = from_oct( 8, header->header.uid );
      st->st_gid = from_oct( 8, header->header.gid );

#if 0
#if defined(S_IFBLK) || defined(S_IFCHR)
      switch (header->header.linkflag) {
      case LF_BLK:
      case LF_CHR:
	 st->st_rdev = makedev (from_oct (8, header->header.devmajor),
				from_oct (8, header->header.devminor));
      }
#endif
#endif
   } else {
				/* Old fashioned tar archive */
      *stdp = 0;
      st->st_uid = from_oct( 8, header->header.uid );
      st->st_gid = from_oct( 8, header->header.gid );
      st->st_rdev = 0;
   }
}

/* Print the header information in a human-readable format for debugging. */

void pm_tar_print_header( FILE *str, TarHeader header )
{
   union record *h = (union record *)header;
   struct stat  st;
   int          std;
   const char   *type;

   decode_header( h, &st, &std );

   switch (h->header.linkflag) {
				/* Standard fields */
   case LF_OLDNORMAL: type = "old normal";    break;
   case LF_NORMAL:    type = "normal";        break;
   case LF_LINK:      type = "link";          break;
   case LF_SYMLINK:   type = "symlink";       break;
   case LF_CHR:       type = "char special";  break;
   case LF_BLK:       type = "block special"; break;
   case LF_DIR:       type = "directory";     break;
   case LF_FIFO:      type = "fifo special";  break;
   case LF_CONTIG:    type = "contiguous";    break;

				/* GNU extensions */
   case LF_DUMPDIR:   type = "* dumpdir";     break;
   case LF_LONGLINK:  type = "* longlink";    break;
   case LF_LONGNAME:  type = "* longname";    break;
   case LF_MULTIVOL:  type = "* multivol";    break;
   case LF_NAMES:     type = "* names";       break;
   case LF_SPARSE:    type = "* sparse";      break;
   case LF_VOLHDR:    type = "* volhdr";      break;
   default:
      type = alloca( 100 * sizeof (char ) );
      sprintf( (char *)type, "***** Unknown: %c",  (int)h->header.linkflag );
      break;
   }

   fprintf( str,
	    "%d %s (%s): %ld %s",
	    std,
	    h->header.arch_name,
	    type,
	    st.st_size,
	    ctime( &st.st_mtime ) );
}
