/* pack.c -- 
 * Created: Sun May  7 19:46:40 1995 by r.faith@ieee.org
 * Revised: Thu Aug 31 22:01:12 1995 by r.faith@ieee.org
 * Copyright 1995 Rickard E. Faith (r.faith@ieee.org)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: pack.c,v 1.2 1995/09/12 02:24:18 faith Exp $
 * 
 */

#include "pm.h"

extern const char *PgmName;
extern const char *PgmPath;

int pack( const char *name )
{
   const char *notesFile = pm_resolve_notesfile( name );
   List       list       = pm_notes_read( notesFile );
   List       files;
   List       fileSections;
   char       *tmp       = tempnam( PmTmp, PgmName );
   int        exitStatus = 0;
   FILE       *str;
   const char *path;
   char       filename[MAXPATHLEN];
   int        i;

				/* Since we are passing a complete list of
                                   files to tar, the list cannot contain
                                   any non-empty directories -- otherwise
                                   the files in the directories will be
                                   tar'd twice.  */
   static int writeFile( const char *file )
      {
	 if (pm_file_nonempty_dir( file + 1 )) return 0;
	 
	 if (file[1] == '/')
	    fprintf( str, "%s\n", file + 2 );
	 else
	    fprintf( str, "%s\n", file + 1 );
	 return 0;
      }

   pm_notes_parse( list );
   pm_notes_check( 2 );		/* Magic number for builds/packs */
   fileSections = pm_notes_files_sections( list );

				/* Iterate by hand to avoid exercising
                                   trampoline bug in early versions of gcc
                                   (multiple, nested, nested functions may
                                   fail on some architectures). */
   
   for (i = 0; i < fileSections->count && !exitStatus; i++) {
      files = pm_notes_files( list, fileSections->lines[i] );
      pm_notes_copy( list, files, fileSections->lines[i] );

      str = pm_file_create( tmp );
      writeFile( pm_gen_notespath( fileSections->lines[i] ) );
      pm_list_iterate( files, writeFile );
      fclose( str );

      strcpy( filename, pm_gen_canonical( fileSections->lines[i] ) );
      if (PmBinExt) strcat( filename, PmBinExt );

      path = pm_create_distpath( PmBinDistDir, filename );
      if (!PmQuiet) printf( "Creating: %s\n", path );
      exitStatus = pm_tar( tmp, PmTarRoot, path );
      xfree( (char *)path );

      pm_unlink( tmp );
   
      pm_list_free( files );
   }

   pm_list_free( fileSections );
   xfree( tmp );
   pm_notes_free( list );
   xfree( (char *)notesFile );
   
   return exitStatus;
}

int pack_source( const char *name )
{
   const char *notesFile = pm_resolve_notesfile( name );
   List       list       = pm_notes_read( notesFile );
   char       *tmp       = tempnam( PmTmp, PgmName );
   const char *path;
   FILE       *str;
   int        i;
   char       filename[MAXPATHLEN];
   int        exitStatus;

   pm_notes_parse( list );
   pm_notes_check( 0 );

   str = pm_file_create( tmp );
   fprintf( str, "#!%s -e\n", PmBourneShell );
   fprintf( str, "# This is an automatically generated script\n" );
   fprintf( str, "# Generated by %s on %s\n", pm_version(), PmTimeString );
   fprintf( str, "# Source packaging script for %s\n",
	    pm_gen_canonical( NULL ) );

   if (TEST(PM_VERBOSE)) fprintf( str, "set -x\n" );

   fprintf( str, "cd %s\n", PmTmp );
   fprintf( str, "mkdir %s.dir\n", tmp );
   fprintf( str, "cd %s.dir\n", tmp );

   fprintf( str, "ln -s %s .\n", notesFile );
   
   for (i = 0; i < PM_MAXSOURCE; i++)
      if (PmPkgSource[i]) {
	 const char *sp = pm_resolve_sourcefile( PmPkgSource[i] );

	 fprintf( str, "ln -s %s .\n", sp );
	 xfree( (char *)sp );
      }

   for (i = 0; i < PM_MAXPATCH; i++)
      if (PmPkgPatch[i]) {
	 const char *sp = pm_resolve_patchfile( PmPkgPatch[i] );

	 fprintf( str, "ln -s %s .\n", sp );
	 xfree( (char *)sp );
      }

   strcpy( filename, pm_gen_canonical( NULL ) );
   if (PmSrcExt) strcat( filename, PmSrcExt );
   path = pm_create_distpath( PmSrcDistDir, filename );
   if (TEST(PM_VERBOSE))
      fprintf( str, PM_TAR_SOURCES_VERBOSE, path );
   else
      fprintf( str, PM_TAR_SOURCES, path );
   fprintf( str, "\n" );

   fprintf( str, "cd %s\n", PmTmp );
   fprintf( str, "rm -rf %s.dir\n", tmp );
   fprintf( str, "exit 0" );
   fclose( str );
   pm_chmod( tmp, 0700 );
   if (!PmQuiet) printf( "Creating: %s\n", path );
   exitStatus = pm_shell( tmp );
   pm_unlink( tmp );

   xfree( (char *)path );
   xfree( tmp );
   pm_notes_free( list );
   xfree( (char *)notesFile );
   return exitStatus;
}
