/* pm.c -- 
 * Created: Tue May  2 15:22:58 1995 by r.faith@ieee.org
 * Revised: Mon Nov 11 14:03:06 1996 by faith@cs.unc.edu
 * Copyright 1995, 1996 Rickard E. Faith (r.faith@ieee.org)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: pm.c,v 1.13 1996/11/11 19:10:27 faith Exp $
 * 
 */

#include "pm.h"
#include <signal.h>

extern const char  *PgmVersion;
const char         *PgmName      = "pm";
const char         *PgmCopyright =
  "Copyright 1995, 1996 Rickard E. Faith and Doug L. Hoffman\n"
  "  This program is free software; you can redistribute it and/or modify it\n"
  "  under the terms of the GNU General Public License as published by the\n"
  "  Free Software Foundation; either version 1, or (at your option) any\n"
  "  later version.";
const char  *PgmPath;

static int BuildOption;
static int InstallOption;
static int ListOption;
static int DeleteOption;
static int WhenceOption;

#define PM_LISTALL      0x00000001
#define PM_LISTLONG     0x00000002
#define PM_LISTPACKAGE  0x00000004
#define PM_DELETE       0x00000010
#define PM_WHENCE       0x00000100


static void sig_handler( int sig )
{
   pm_fatal( PMERR_SIGNAL, "%s\n", strsignal( sig ) );
}

static void banner( void )
{
   fprintf( stderr, "%s\n%s\n",
	    pm_version(),
	    PgmCopyright );
}

static void help( void ) /* __attribute((noreturn))__ */
{
   static const char  *help_msg[] = {
      " -h              --help               give this help",
      " -V              --version            display version",
      " -i              --install            install binary distribution",
      "                 --test               check but don't install",
      "                 --pedantic           increased conflict checking",
      "                 --force              force binary installation",
      "                 --noforce            don't force installation",
      " -u              --update             same as --install --force",
      " -B build_opt    --build build_opt    build from sources",
      "                 --clean              clean up after build",
      "                 --reprep             clean and re-patch after build",
      "                 --dist dist_name     specify distribution name",
      " -p              --package            package",
      " -P              --spackage           source-package",
      "                 --delete             delete package",
      "                 --config config_file specify configuration file",
      "                 --noconfig           don't read configuration files", 
      " -v              --verbose            verbose",
      "                 --quiet              quiet",
      "                 --all                list all packages",
      "                 --list name          list packages with name",
      "                 --long               list files with packages",
      "                 --whence file        list package containing file",
      "                 --nodb               don't update database",
      " -d debug_opt    --debug debug_opt    turn on debugging",
      " -D                                   same as -d debug",
      "                 --compileonly        skip prep",
      "                 --installonly        skip compile",
      "",
      "Build options are as follows:",
      "  prep (p)                 untar and patch sources",
      "  build (compile, c)       prep + compilation",
      "  install (i)              build + installation",
      "  package (a)              install + package",
      "  spackage (A)             package + source-package",
      0 };
   const char **p = help_msg;

   banner();
   while (*p) fprintf(stderr, "%s\n", *p++ );
   exit( PMERR_HELP );
}

static void decode_build_option( const char *option )
{
   struct options {
      const char *option;
      int        flag;
   } options[] = {
      { "prep",     PM_PREP },
      { "p",        PM_PREP },
      { "build",    PM_PREP | PM_BUILD },
      { "compile",  PM_PREP | PM_BUILD },
      { "c",        PM_PREP | PM_BUILD },
      { "install",  PM_PREP | PM_BUILD | PM_INST },
      { "i",        PM_PREP | PM_BUILD | PM_INST },
      { "pack",     PM_PREP | PM_BUILD | PM_INST | PM_PACK },
      { "package",  PM_PREP | PM_BUILD | PM_INST | PM_PACK },
      { "spack",    PM_PREP | PM_BUILD | PM_INST | PM_PACK | PM_PACKSOURCE },
      { "spackage", PM_PREP | PM_BUILD | PM_INST | PM_PACK | PM_PACKSOURCE },
      { "all",      PM_PREP | PM_BUILD | PM_INST | PM_PACK                 },
      { "a",        PM_PREP | PM_BUILD | PM_INST | PM_PACK                 },
      { "A",        PM_PREP | PM_BUILD | PM_INST | PM_PACK | PM_PACKSOURCE },
      { 0, 0 }
   };
   struct options *p = options;

   while (p->option) {
      if (!strcmp( option, p->option )) {
	 BuildOption |= p->flag;
	 return;
      }
      ++p;
   }
}

int main( int argc, char **argv )
{
   int           c;
   int           i;
   int           verbosity = 0;
   const char    *config   = NULL;
   struct option longopts[] = {
      { "verbose",      0, 0, 'v' },
      { "debug",        1, 0, 'd' },
      { "help",         0, 0, 'h' },
      { "version",      0, 0, 'V' },
      { "install",      0, 0, 'i' },
      { "update",       0, 0, 'u' },
      { "build",        1, 0, 'B' },
      { "clean",        0, 0, 500 },
      { "reprep",       0, 0, 501 },
      { "dist",         1, 0, 502 },
      { "package",      0, 0, 'p' },
      { "spackage",     0, 0, 'P' },
      { "usegroup",     0, 0, 503 },
      { "nogroup",      0, 0, 504 },
      { "umask",        1, 0, 505 },
      { "config",       1, 0, 506 },
      { "noconfig",     0, 0, 507 },
      { "dbgmask",      1, 0, 508 },
      { "force",        0, 0, 509 },
      { "noforce",      0, 0, 510 },
      { "hash",         0, 0, 511 },
      { "nohash",       0, 0, 512 },
      { "quiet",        0, 0, 513 },
      { "noquiet",      0, 0, 514 },
      { "all",          0, 0, 516 },
      { "long",         0, 0, 517 },
      { "list",         0, 0, 518 },
      { "test",         0, 0, 519 },
      { "pedantic",     0, 0, 520 },
      { "nopendantic",  0, 0, 521 },
      { "delete",       0, 0, 522 },
      { "whence",       0, 0, 523 },
      { "nodb",         0, 0, 524 },
      { "installonly",  0, 0, 525 },
      { "compileonly",  0, 0, 526 },
      { "install-only", 0, 0, 525 },
      { "compile-only", 0, 0, 526 },
      { 0,              0, 0,  0  }
   };

   if (signal( SIGINT, SIG_IGN ) != SIG_IGN)  signal( SIGINT, sig_handler );
   if (signal( SIGQUIT, SIG_IGN ) != SIG_IGN) signal( SIGQUIT, sig_handler );

   PgmPath = argv[0];
   
   while ((c = getopt_long( argc, argv,
			    "B:pPVhDd:vi", longopts, NULL )) != EOF)
      switch (c) {
	 case 506: config = optarg;                            break;
	 case 507: config = "/dev/null";                       break;
      }

				/* Other options can override config... */
   pm_config( config );

   optarg = NULL;
   optind = 0;
   while ((c = getopt_long( argc, argv,
			    "B:pPVhDd:viu", longopts, NULL )) != EOF)
         switch (c) {
	 case 'B': decode_build_option( optarg );              break;
	 case 'p': BuildOption |= PM_PACK;                     break;
	 case 'P': BuildOption |= PM_PACKSOURCE;               break;
	 case 500: BuildOption |= PM_CLEAN;                    break;
	 case 501: BuildOption |= PM_REPREP;                   break;
	 case 502: PmDistribution = strdup( optarg );          break;
	 case 503: PmUseGroup = 1;                             break;
	 case 504: PmUseGroup = 0;                             break;
	 case 505: PmUmask = strtol( optarg, NULL, 0 );        break;
	 case 506:                                             break;
	 case 507:                                             break;
	 case 508: pm_debug_flags = strtol( optarg, NULL, 0 ); break;
	 case 509: PmForceInstall = 1;                         break;
	 case 510: PmForceInstall = 0;                         break;
	 case 511: PmHash = 1;                                 break;
	 case 512: PmHash = 0;                                 break;
	 case 513: PmQuiet = 1;                                break;
	 case 514: PmQuiet = 0;                                break;
	 case 516: ListOption  |= PM_LISTALL;                  break;
	 case 517: ListOption  |= PM_LISTLONG;                 break;
	 case 518: ListOption  |= PM_LISTPACKAGE;              break;
	 case 519: PmTestFunction = 1;                         break;
	 case 520: PmPedantic = 1;                             break;
	 case 521: PmPedantic = 0;                             break;
	 case 522: DeleteOption |= PM_DELETE;                  break;
	 case 523: WhenceOption |= PM_WHENCE;                  break;
	 case 524: PmNoDatabase = 1;                           break;
	 case 525: BuildOption = PM_INST;                      break;
	 case 526: BuildOption = PM_BUILD;                     break;
	 case 'V': banner(); exit( PMERR_HELP );               break;
	 case '?':
         case 'h': help();                                     break;
	 case 'D': pm_debug( "debug" );                        break;
         case 'd': pm_debug( optarg );                         break;
	 case 'i': ++InstallOption;                            break;
	 case 'u': ++InstallOption; PmForceInstall = 1;        break;
	 case 'v':
	    switch (++verbosity) {
	    case 1: pm_debug( "verbose" ); break;
	    case 2: pm_debug( "search" );  break;
	    case 3: pm_debug( "exec" );    break;
	    }
	    break;
	 default: help();
         }

   if (BuildOption) {
      if (InstallOption || ListOption || DeleteOption || WhenceOption) help();

      if (optind >= argc) help();
      for (i = optind; i < argc; i++) {
	 if (!(BuildOption ^ PM_PACK)) {
	    if (pack( argv[i] )) pm_fatal( PMERR_PACK, "\n" );
	 } else if (!(BuildOption ^ PM_PACKSOURCE)) {
	    if (pack_source( argv[i] )) pm_fatal( PMERR_PACK, "\n" );
	 } else if (!(BuildOption ^ (PM_PACK | PM_PACKSOURCE))) {
	    if (pack( argv[i] )) pm_fatal( PMERR_PACK, "\n" );
	    if (pack_source( argv[i] )) pm_fatal( PMERR_PACK, "\n" );
	 } else {
	    if (build( argv[i], BuildOption, config ))
	       pm_fatal( PMERR_BUILD, "\n" );
	 }
      }
   } else if (InstallOption) {
      if (BuildOption || ListOption || DeleteOption || WhenceOption) help();

      if (optind >= argc) help();
      for (i = optind; i < argc; i++) {
	 if (install( argv[i] )) pm_fatal( PMERR_INST, "\n" );
      }
   } else if (ListOption) {
      int longFlag = (ListOption & PM_LISTLONG);
      
      if (BuildOption || InstallOption || DeleteOption || WhenceOption) help();
      if (ListOption & PM_LISTALL) pm_db_list_package_all( longFlag );
      else {
	 if (optind >= argc) help();
	 for (i = optind; i < argc; i++) {
	    pm_db_list_package_byname( argv[i], longFlag );
	 }
      }
   } else if (DeleteOption) {
      if (BuildOption || InstallOption || ListOption || WhenceOption) help();
      if (optind >= argc) help();
      for (i = optind; i < argc; i++) delete( argv[i] );
   } else if (WhenceOption) {
      if (BuildOption || InstallOption || ListOption || DeleteOption) help();
      if (optind >= argc) help();
      for (i = optind; i < argc; i++)
	 pm_db_whence( argv[i] );
   } else
      help();
   
   pm_shutdown();
   return 0;
}
