/*
 * INET		An implementation of the TCP/IP protocol suite for the LINUX
 *		operating system.  INET is implemented using the  BSD Socket
 *		interface as the means of communication with the user level.
 *
 *		ROUTE - implementation of the IP router.
 *
 * Version:	@(#)route.c	1.0.7	05/07/93
 *
 * Authors:	Ross Biro, <bir7@leland.Stanford.Edu>
 *		Fred N. van Kempen, <waltje@uWalt.NL.Mugnet.ORG>
 *
 *		This program is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 */
#include <asm/segment.h>
#include <asm/system.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/string.h>
#include <linux/socket.h>
#include <linux/sockios.h>
#include <linux/errno.h>
#include <linux/in.h>
#include "inet.h"
#include "timer.h"
#include "dev.h"
#include "ip.h"
#include "protocol.h"
#include "route.h"
#include "tcp.h"
#include "skbuff.h"
#include "sock.h"
#include "arp.h"
#include "icmp.h"


/*
 * This is an entry in the IP routing table.
 * It is isolated from the other routing table definitions
 * because it contains a lot of IP dependencies.  So, we
 * only define it in this file. -FvK
 */
struct rtable {
  struct rtable		*rt_next;
  unsigned long		rt_dst;
  unsigned long		rt_gateway;
  u_char		rt_flags;
  u_char		rt_metric;
  short			rt_refcnt;
  u_long		rt_use;
  struct device		*rt_dev;
};


static struct rtable *rt_base = NULL;


/* Dump the contents of a routing table entry. */
static void
rt_print(struct rtable *rt)
{
  if (rt == NULL || inet_debug != DBG_RT) return;

  printk("RT: %06lx NXT=%06lx FLAGS=0x%02lx\n",
		(long) rt, (long) rt->rt_next, rt->rt_flags);
  printk("    TARGET=%s ", in_ntoa(rt->rt_dst));
  printk("GW=%s ", in_ntoa(rt->rt_gateway));
  printk("    DEV=%s USE=%ld REF=%ld METRIC=%d\n",
	rt->rt_dev->name, rt->rt_use, rt->rt_refcnt, rt->rt_metric);
}


/* Remove a routing table entry. */
static void
rt_del(unsigned long dst)
{
  struct rtable *r, *x, *p;

  dprintf(DBG_RT, "RT: flushing for dst %s\n", in_ntoa(dst));
  if ((r = rt_base) == NULL) return;
  p = NULL;
  while(r != NULL) {
	if (r->rt_dst == dst) {
		if (p == NULL) rt_base = r->rt_next;
		  else p->rt_next = r->rt_next;
		x = r->rt_next;
		kfree_s(r, sizeof(struct rtable));
		r = x;
	} else {
		p = r;
		r = r->rt_next;
	}
  }
}


/* Remove all routing table entries for a device. */
void
rt_flush(struct device *dev)
{
  struct rtable *r, *x, *p;

  dprintf(DBG_RT, "RT: flushing for dev 0x%08lx (%s)\n", (long)dev, dev->name);
  if ((r = rt_base) == NULL) return;
  p = NULL;
  while(r != NULL) {
	if (r->rt_dev == dev) {
		if (p == NULL) rt_base = r->rt_next;
		  else p->rt_next = r->rt_next;
		x = r->rt_next;
		kfree_s(r, sizeof(struct rtable));
		r = x;
	} else {
		p = r;
		r = r->rt_next;
	}
  }
}


void
rt_add(short flags, short metric, unsigned long dst, unsigned long gw,
       struct device *dev)
{
  struct rtable *r, *r1;
  struct rtable *rt;
  int mask;

  /* Allocate an entry. */
  rt = kmalloc(sizeof(struct rtable), GFP_ATOMIC);
  if (rt == NULL) {
	dprintf(DBG_RT, "RT: no memory for new route!\n");
	return;
  }

  /* Fill in the fields. */
  memset(rt, 0, sizeof(struct rtable));
  rt->rt_flags = (flags | RTF_UP);
  if (gw != 0) rt->rt_flags |= RTF_GATEWAY;
  rt->rt_dev = dev;
  rt->rt_metric = metric;
  rt->rt_gateway = gw;

  /*
   * If this is coming from an ICMP redirect message, truncate
   * the TARGET if we are creating an entry for a NETWORK. Use
   * an Internet class C network mask.  Yuck :-(
   */
  if (flags & RTF_DYNAMIC) {
	if (flags & RTF_HOST) rt->rt_dst = dst;
	  else rt->rt_dst = (dst & htonl(IN_CLASSC_NET));
  } else rt->rt_dst = dst;

  rt_print(rt);

  if (rt_base == NULL) {
	rt->rt_next = NULL;
	rt_base = rt;
	return;
  }

  /*
   * What we have to do is loop though this until we have
   * found the first address which has the same generality
   * as the one in rt.  Then we can put rt in after it.
   */
  for (mask = 0xff000000; mask != 0xffffffff; mask = (mask >> 8) | mask) {
	if (mask & dst) {
		mask = mask << 8;
		break;
	}
  }
  dprintf(DBG_RT, "RT: mask = %X\n", mask);
  r1 = rt_base;

  /* See if we are getting a duplicate. */
  for (r = rt_base; r != NULL; r = r->rt_next) {
	if (r->rt_dst == dst) {
		if (r == rt_base) {
			rt->rt_next = r->rt_next;
			rt_base = rt;
		} else {
			rt->rt_next = r->rt_next;
			r1->rt_next = rt;
		}
		kfree_s(r, sizeof(struct rtable));
		return;
	}

	if (! (r->rt_dst & mask)) {
		dprintf(DBG_RT, "RT: adding before r=%X\n", r);
		rt_print(r);
		if (r == rt_base) {
			rt->rt_next = rt_base;
			rt_base = rt;
			return;
		}
		rt->rt_next = r;
		r1->rt_next = rt;
		return;
	}
	r1 = r;
  }
  dprintf(DBG_RT, "RT: adding after r1=%X\n", r1);
  rt_print(r1);

  /* Goes at the end. */
  rt->rt_next = NULL;
  r1->rt_next = rt;
}


static int
rt_new(struct rtentry *r)
{
  struct device *dev;

  if (r->rt_dst.sa_family != AF_INET) {
	printk("RT: We only know about AF_INET !\n");
	return(-EINVAL);
  }
  dev = dev_get(r->rt_dev);
  if (dev == NULL) {
	dprintf(DBG_RT, "RT: invalid interface name %s given!\n", r->rt_dev);
	return(-EINVAL);
  }
  rt_add(r->rt_flags, r->rt_metric,
	 (*(struct sockaddr_in *) &r->rt_dst).sin_addr.s_addr,
	 (*(struct sockaddr_in *) &r->rt_gateway).sin_addr.s_addr, dev);

  return(0);
}


static int
rt_kill(struct rtentry *r)
{
  struct sockaddr_in *trg;

  trg = (struct sockaddr_in *) &r->rt_dst;
  rt_del(trg->sin_addr.s_addr);

  return(0);
}


/* Called from the PROCfs module. */
int
rt_get_info(char *buffer)
{
  struct rtable *r;
#ifndef HUMAN_READABLE
  struct rtentry rt;
#endif
  char *pos;

  pos = buffer;

#ifdef HUMAN_READABLE
  pos += sprintf(pos, "Iface\tDestination\tGateway \tFlags\tRefCnt\tUse\tMetric\n");
#endif
  
  for (r = rt_base; r != NULL; r = r->rt_next) {
#ifdef HUMAN_READABLE
        pos += sprintf(pos, "%s\t%08X\t%08X\t%02X\t%d\t%d\t%d\n",
		r->rt_dev->name, r->rt_dst, r->rt_gateway, r->rt_flags, r->rt_refcnt, r->rt_use, r->rt_metric);
#else
	memset(&rt, 0, sizeof(struct rtentry));
	strcpy(rt.rt_dev, r->rt_dev->name);
	(*(struct sockaddr_in *)&rt.rt_dst).sin_addr.s_addr = r->rt_dst;
	(*(struct sockaddr_in *)&rt.rt_gateway).sin_addr.s_addr = r->rt_gateway;
	rt.rt_flags = r->rt_flags;
	rt.rt_metric = r->rt_metric;
	rt.rt_use = r->rt_use;
	rt.rt_refcnt = r->rt_refcnt;
	memcpy(pos, &rt, sizeof(struct rtentry));
	pos += sizeof(struct rtentry);
#endif
  }
  return(pos - buffer);
}


struct device *
ip_route(struct options *opt, unsigned long daddr , unsigned long *raddr)
{
  struct rtable *rt;

  for (rt = rt_base; rt != NULL; rt = rt->rt_next) {
	if (ip_addr_match(rt->rt_dst, daddr)) {
		dprintf(DBG_RT, "RT: %s via %s",
				in_ntoa(daddr), rt->rt_dev->name);
		dprintf(DBG_RT, " (%s)\n", in_ntoa(rt->rt_gateway));
		rt->rt_use++;
		*raddr = rt->rt_gateway;
		return(rt->rt_dev);
	}
  }
  return(NULL);
};


int
rt_ioctl(unsigned int cmd, void *arg)
{
  struct rtentry rt;
  int ret;

  switch(cmd) {
	case SIOCADDRT:
		dprintf(DBG_RT, "RT: ioctl(SIOCADDRT, 0x%08X)\n", arg);
		verify_area(VERIFY_WRITE, arg, sizeof(struct rtentry));
		memcpy_fromfs(&rt, arg, sizeof(struct rtentry));
		ret = rt_new(&rt);
		break;
	case SIOCDELRT:
		dprintf(DBG_RT, "RT: ioctl(SIOCDELRT, 0x%08X)\n", arg);
		verify_area(VERIFY_WRITE, arg, sizeof(struct rtentry));
		memcpy_fromfs(&rt, arg, sizeof(struct rtentry));
		ret = rt_kill(&rt);
		break;
	default:
		ret = -EINVAL;
  }

  return(ret);
}
