/*
 * INET		An implementation of the TCP/IP protocol suite for the LINUX
 *		operating system.  INET is implemented using the  BSD Socket
 *		interface as the means of communication with the user level.
 *
 *		TIMER - implementation of software timers.
 *
 * Version:	@(#)timer.c	1.0.3	05/07/93
 *
 * Authors:	Ross Biro, <bir7@leland.Stanford.Edu>
 *		Fred N. van Kempen, <waltje@uWalt.NL.Mugnet.ORG>
 *
 *		This program is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 */
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/socket.h>
#include <linux/in.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/timer.h>
#include <asm/system.h>
#include <linux/interrupt.h>
#include "inet.h"
#include "timer.h"
#include "dev.h"
#include "ip.h"
#include "protocol.h"
#include "tcp.h"
#include "skbuff.h"
#include "sock.h"
#include "arp.h"


#undef	TIMER_DEBUG
#ifdef	TIMER_DEBUG
#   define PRINTK(x)	printk x
#else
#   define PRINTK(x)	/**/
#endif


volatile struct timer *timer_base=NULL;
unsigned long seq_offset;


void
delete_timer(struct timer *t)
{
  struct timer *tm;

  PRINTK (("delete_timer (t=%X)\n",t));
  if (timer_base == NULL || t == NULL) return;
  cli();
  if (t == timer_base) {
	timer_base = t->next;
	if (timer_base != NULL) {
		timer_table[NET_TIMER].expires = timer_base->when;
		timer_active |= (1 << NET_TIMER);
	} else {
		timer_active &= ~(1 << NET_TIMER);
	}
	sti();
	return;
  }
  for (tm = (struct timer *)timer_base;
       tm->next != NULL ;
       tm = (struct timer *)tm->next) {
	if (tm->next == t) {
		tm->next = t->next;
		sti();
		return;
	}
  }
  sti();
}


void
reset_timer(struct timer *t)
{
  struct timer *tm;

  if (t == NULL) {
	printk("*** reset timer NULL timer\n");
	__asm__ ("\t int $3\n"::);
  }
  delete_timer(t);
  t->when = timer_seq + t->len;
  PRINTK (("reset_timer (t=%X) when = %d jiffies = %d\n", t, t->when, jiffies));

  /* First see if it goes at the beginning. */
  cli();
  if (timer_base == NULL) {
	t->next = NULL;
	timer_base = t;
	timer_table[NET_TIMER].expires = timer_base->when;
	timer_active |= (1 << NET_TIMER);
	sti();
	return;
  }
  if (t->when < timer_base->when) {
	t->next = timer_base;
	timer_base = t;
	timer_table[NET_TIMER].expires = timer_base->when;

  	timer_active |= 1 << NET_TIMER;
	sti();
	return;
  }
  for (tm = (struct timer *)timer_base; ; tm=(struct timer *)tm->next) {
	if (tm->next == NULL || t->when < tm->next->when) {
		t->next = tm->next;
		tm->next = t;
		sti();
		return;
	}
  }
}


/*
 * Now we will only be called whenever we need to do
 * something, but we must be sure to process all of the
 * sockets that need it.
 */
void
net_timer(void)
{
  struct sock *sk;

  while (timer_base != NULL && jiffies >= timer_base->when) {
	int why;

	sk = timer_base->sk;
	cli();
	if (sk->inuse) {
		sti();
		break;
	}
	sk->inuse = 1;
	sti();
	why = sk->timeout;

	PRINTK (("net_timer: found sk=%X why = %d\n",sk, why));
	if (sk->keepopen) {
		sk->time_wait.len = TCP_TIMEOUT_LEN;
		sk->timeout = TIME_KEEPOPEN;
		reset_timer((struct timer *)timer_base);
	} else {
		sk->timeout = 0;
		delete_timer((struct timer *)timer_base);
	}
	
	/* Always see if we need to send an ack. */
	if (sk->ack_backlog) {
		sk->prot->read_wakeup(sk);
		if (!sk->dead) wake_up(sk->sleep);
	}
	
	/* Now we need to figure out why the socket was on the timer. */
	switch (why) {
		case TIME_DONE:
			if (!sk->dead || sk->state != TCP_CLOSE) {
				printk("non dead socket in time_done\n");
				release_sock(sk);
				break;
			}
			destroy_sock(sk);
			break;
		case TIME_DESTROY:
			/*
			 * We've waited for a while for all the memory
			 * assosiated with the socket to be freed.  We
			 * need to print an error message.
			 */
			PRINTK (("possible memory leak.  sk = %X\n", sk));
			reset_timer((struct timer *)&sk->time_wait);
			sk->inuse = 0;
			break;
		case TIME_CLOSE:
			/* We've waited long enough, close the socket. */
			sk->state = TCP_CLOSE;
			delete_timer((struct timer *)&sk->time_wait);

			/*
			 * Kill the ARP entry in case the hardware
			 * has changed.
			 */
			arp_destroy(sk->daddr);
			if (!sk->dead) wake_up(sk->sleep);
			release_sock(sk);
			break;
		case TIME_WRITE: /* try to retransmit. */
			/*
			 * It could be we got here because we
			 * needed to send an ack.  So we need
			 * to check for that.
			 */
			if (sk->send_head != NULL) {
				if (before(jiffies, sk->send_head->when +
							sk->rtt)) {
					sk->time_wait.len = sk->rtt;
					sk->timeout = TIME_WRITE;
					reset_timer((struct timer *)&sk->time_wait);
					release_sock(sk);
					break;
				}
				PRINTK (("retransmitting.\n"));
				sk->prot->retransmit(sk, 0);

				if (sk->retransmits > TCP_RETR1) {
					PRINTK (("timer.c TIME_WRITE time-out 1\n"));
					arp_destroy(sk->daddr);
					ip_route_check(sk->daddr);
				}

				if (sk->retransmits > TCP_RETR2) {
					PRINTK (("timer.c TIME_WRITE time-out 2\n"));
					sk->err = ETIMEDOUT;
					if (sk->state == TCP_FIN_WAIT1 ||
					    sk->state == TCP_FIN_WAIT2 ||
					    sk->state == TCP_LAST_ACK) {
						sk->state = TCP_TIME_WAIT;
						sk->timeout = TIME_CLOSE;
						sk->time_wait.len =
							TCP_TIMEWAIT_LEN;
						reset_timer((struct timer *)
								&sk->time_wait);
						release_sock(sk);
						break;
					} else {
						sk->prot->close(sk,1);
						break;
					}
				}
				release_sock(sk);
				break;
			}
			release_sock(sk);
			break;
		case TIME_KEEPOPEN:
			/* Send something to keep the connection open. */
			if (sk->prot->write_wakeup != NULL)
					sk->prot->write_wakeup(sk);
			sk->retransmits ++;
			if (sk->shutdown == SHUTDOWN_MASK) {
				sk->prot->close(sk,1);
				sk->state = TCP_CLOSE;
			}

			if (sk->retransmits > TCP_RETR1) {
				PRINTK (("timer.c TIME_KEEPOPEN time-out 1\n"));
				arp_destroy(sk->daddr);
				ip_route_check(sk->daddr);
				release_sock(sk);
				break;
			}
			if (sk->retransmits > TCP_RETR2) {
				PRINTK (("timer.c TIME_KEEPOPEN time-out 2\n"));
				arp_destroy (sk->daddr);
				sk->err = ETIMEDOUT;
				if (sk->state == TCP_FIN_WAIT1 ||
				    sk->state == TCP_FIN_WAIT2) {
					sk->state = TCP_TIME_WAIT;
					if (!sk->dead) wake_up(sk->sleep);
					release_sock(sk);
				} else {
					sk->prot->close (sk, 1);
				}
				break;
			}
			release_sock(sk);
			break;
		default:
			release_sock(sk);
			break;
	}
  }

  /* Now we need to reset the timer. */
  cli();
  if (timer_base != NULL) {
	timer_table[NET_TIMER].expires = timer_base->when;
	timer_active |= 1 << NET_TIMER;
  }
  sti();
}
