/*
 * UNIX		An implementation of the AF_UNIX network domain for the
 *		LINUX operating system.  UNIX is implemented using the
 *		BSD Socket interface as the means of communication with
 *		the user level.
 *
 *		The functions in this file provide an interface between
 *		the PROC file system and the "unix" family of networking
 *		protocols. It is mainly used for debugging and statistics.
 *
 * Version:	@(#)proc.c	1.0.3	04/22/93
 *
 * Authors:	Ross Biro, <bir7@leland.Stanford.Edu>
 *		Fred N. van Kempen, <waltje@uWalt.NL.Mugnet.ORG>
 *		Gerald J. Heim, <heim@peanuts.informatik.uni-tuebingen.de>
 *
 *		This program is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 */
#include <linux/autoconf.h>
#include <linux/sched.h>
#include <linux/string.h>
#include <linux/socket.h>
#include <linux/net.h>
#include <linux/ddi.h>
#include <linux/un.h>
#include <sys/param.h>
#include "unix.h"


/* Called from PROCfs. */
int unix_get_info(char *buffer)
{
  char *pos;
  char *ss_state, *ss_type;
  int i;

  pos = buffer;
  pos += sprintf(pos, "Num  #Ref Proto\tFlags\tType\t\tState\t\tPath\n");

  for(i = 0; i < NSOCKETS; i++) {
	if (unix_datas[i].refcnt) {

		/*
		 * Maybe translations like these should be done by a
		 * user-program.  Translate type-enumeration to string.
		 */
		switch(unix_datas[i].socket->type) {
			case SOCK_STREAM:
				ss_type = "SOCK_STREAM";
				break;
			case SOCK_DGRAM:
				ss_type = "SOCK_DGRAM";
				break;
			case SOCK_RAW:
				ss_type = "SOCK_RAW\t";
				break;
			case SOCK_RDM:
				ss_type = "SOCK_RDM\t";
				break;
			case SOCK_SEQPACKET:
				ss_type = "SOCK_SEQPACKET";
				break;
			case SOCK_PACKET:
				ss_type = "SOCK_PACKET";
				break;
			default:
				printk("UNIX: netinfo: unknown socket type %d.\n",
					unix_datas[i].socket->type);
				ss_type = "UNKNOWN\t";
		}

		/* Turn the state-information in human-readable form. */
		switch(unix_datas[i].socket->state) {
			case SS_FREE:
				ss_state = "FREE\t";
				break;
			case SS_UNCONNECTED:
				/*
				 * Unconnected sockets may be listening
				 * for something.
				 */
				if (unix_datas[i].socket->flags &
					SO_ACCEPTCON) {
					ss_state = "LISTENING";
				} else {
					ss_state = "UNCONNECTED";
				}
				break;
			case SS_CONNECTING:
				ss_state = "CONNECTING";
				break;
			case SS_CONNECTED:
				ss_state = "CONNECTED";
				break;
			case SS_DISCONNECTING:
				ss_state = "DISCONNECTING";
				break;
			default:
				printk("UNIX: netinfo: unknown state.\n");
				ss_state = "UNKNOWN\t";
		}

		pos += sprintf(pos, " %d :  %d   %d\t%x\t%s\t%s", i,
			unix_datas[i].refcnt,
			unix_datas[i].protocol,
			unix_datas[i].socket->flags,
			ss_type,
			ss_state
		);

		/* If socket is bound to a filename, we'll print it. */
		if(unix_datas[i].sockaddr_len>0) {
			pos += sprintf(pos, "\t%s\n",
				unix_datas[i].sockaddr_un.sun_path);
		} else { /* just add a newline */
			*pos='\n';
			pos++;
			*pos='\0';
		}

		/*
		 * Check wether buffer _may_ overflow in the next loop.
		 * Since sockets may have very very long paths, we make
		 * MAXPATHLEN+100 the minimum space left for a new line.
		 */
		if (pos > buffer+PAGE_SIZE-100-MAXPATHLEN) {
			printk("UNIX: netinfo: oops, too many sockets.\n");
			return(pos - buffer);
		}
	}
  }
  return(pos - buffer);
}
