/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.services;

import java.io.IOException;
import org.ovirt.engine.sdk4.Request;
import org.ovirt.engine.sdk4.Response;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.types.User;

/**
 * A service to manage a user in the system.
 * Use this service to either get users details or remove users.
 * In order to add new users please use
 * <<services/users>>.
 */
public interface UserService extends Service {
    /**
     * Gets the system user information.
     * 
     * Usage:
     * 
     * ....
     * GET /ovirt-engine/api/users/1234
     * ....
     * 
     * Will return the user information:
     * 
     * [source,xml]
     * ----
     * <user href="/ovirt-engine/api/users/1234" id="1234">
     *   <name>admin</name>
     *   <link href="/ovirt-engine/api/users/1234/sshpublickeys" rel="sshpublickeys"/>
     *   <link href="/ovirt-engine/api/users/1234/roles" rel="roles"/>
     *   <link href="/ovirt-engine/api/users/1234/permissions" rel="permissions"/>
     *   <link href="/ovirt-engine/api/users/1234/tags" rel="tags"/>
     *   <department></department>
     *   <domain_entry_id>23456</domain_entry_id>
     *   <email>user1@domain.com</email>
     *   <last_name>Lastname</last_name>
     *   <namespace>*</namespace>
     *   <principal>user1</principal>
     *   <user_name>user1@domain-authz</user_name>
     *   <domain href="/ovirt-engine/api/domains/45678" id="45678">
     *     <name>domain-authz</name>
     *   </domain>
     * </user>
     * ----
     */
    public interface GetRequest extends Request<GetRequest, GetResponse> {
        /**
         * Indicates which inner links should be _followed_. The objects referenced by these links will be fetched as part
         * of the current request. See <<documents/003_common_concepts/follow, here>> for details.
         */
        GetRequest follow(String follow);
    }
    
    /**
     * Gets the system user information.
     * 
     * Usage:
     * 
     * ....
     * GET /ovirt-engine/api/users/1234
     * ....
     * 
     * Will return the user information:
     * 
     * [source,xml]
     * ----
     * <user href="/ovirt-engine/api/users/1234" id="1234">
     *   <name>admin</name>
     *   <link href="/ovirt-engine/api/users/1234/sshpublickeys" rel="sshpublickeys"/>
     *   <link href="/ovirt-engine/api/users/1234/roles" rel="roles"/>
     *   <link href="/ovirt-engine/api/users/1234/permissions" rel="permissions"/>
     *   <link href="/ovirt-engine/api/users/1234/tags" rel="tags"/>
     *   <department></department>
     *   <domain_entry_id>23456</domain_entry_id>
     *   <email>user1@domain.com</email>
     *   <last_name>Lastname</last_name>
     *   <namespace>*</namespace>
     *   <principal>user1</principal>
     *   <user_name>user1@domain-authz</user_name>
     *   <domain href="/ovirt-engine/api/domains/45678" id="45678">
     *     <name>domain-authz</name>
     *   </domain>
     * </user>
     * ----
     */
    public interface GetResponse extends Response {
        /**
         * The system user.
         */
        User user();
    }
    
    /**
     * Gets the system user information.
     * 
     * Usage:
     * 
     * ....
     * GET /ovirt-engine/api/users/1234
     * ....
     * 
     * Will return the user information:
     * 
     * [source,xml]
     * ----
     * <user href="/ovirt-engine/api/users/1234" id="1234">
     *   <name>admin</name>
     *   <link href="/ovirt-engine/api/users/1234/sshpublickeys" rel="sshpublickeys"/>
     *   <link href="/ovirt-engine/api/users/1234/roles" rel="roles"/>
     *   <link href="/ovirt-engine/api/users/1234/permissions" rel="permissions"/>
     *   <link href="/ovirt-engine/api/users/1234/tags" rel="tags"/>
     *   <department></department>
     *   <domain_entry_id>23456</domain_entry_id>
     *   <email>user1@domain.com</email>
     *   <last_name>Lastname</last_name>
     *   <namespace>*</namespace>
     *   <principal>user1</principal>
     *   <user_name>user1@domain-authz</user_name>
     *   <domain href="/ovirt-engine/api/domains/45678" id="45678">
     *     <name>domain-authz</name>
     *   </domain>
     * </user>
     * ----
     */
    GetRequest get();
    
    /**
     * Removes the system user.
     * 
     * Usage:
     * 
     * ....
     * DELETE /ovirt-engine/api/users/1234
     * ....
     */
    public interface RemoveRequest extends Request<RemoveRequest, RemoveResponse> {
        /**
         * Indicates if the remove should be performed asynchronously.
         */
        RemoveRequest async(Boolean async);
    }
    
    /**
     * Removes the system user.
     * 
     * Usage:
     * 
     * ....
     * DELETE /ovirt-engine/api/users/1234
     * ....
     */
    public interface RemoveResponse extends Response {
    }
    
    /**
     * Removes the system user.
     * 
     * Usage:
     * 
     * ....
     * DELETE /ovirt-engine/api/users/1234
     * ....
     */
    RemoveRequest remove();
    
    AssignedPermissionsService permissionsService();
    AssignedRolesService rolesService();
    SshPublicKeysService sshPublicKeysService();
    AssignedTagsService tagsService();
    /**
     * Service locator method, returns individual service on which the URI is dispatched.
     */
    Service service(String path);
}

