#
# otopi -- plugable installer
#


"""Answer file creation plugin."""

# This plugin handles answer file creation.
# It does not handle reading answer files or setting env keys to be added
# to the generated file - that's up to the enabled dialog plugin.


import gettext
import os


from otopi import common
from otopi import constants
from otopi import plugin
from otopi import util


def _(m):
    return gettext.dgettext(message=m, domain='otopi')


@util.export
class Plugin(plugin.PluginBase):
    """Answer file creation plugin.

    Environment:
        DialogEnv.ANSWER_FILE_CONTENT -- is populated by this plugin.
        DialogEnv.ANSWER_FILE -- If set, this plugin writes there the content.

    """

    def __init__(self, context):
        super(Plugin, self).__init__(context=context)

    @plugin.event(
        stage=plugin.Stages.STAGE_INIT,
    )
    def _init(self):
        self.environment.setdefault(
            constants.DialogEnv.ANSWER_FILE,
            None
        )

    @plugin.event(
        stage=plugin.Stages.STAGE_CLEANUP,
        name=constants.Stages.ANSWER_FILE_GENERATED,
    )
    def _generate_answer_file(self):
        content = (
            '# OTOPI answer file, generated by human dialog\n'
            '[environment:default]\n'
        )
        for key, value in self.environment.items():
            if key.startswith(
                constants.CoreEnv.QUESTION_PREFIX
            ):
                content = '{prev}{key}={type}:{value}\n'.format(
                    prev=content,
                    key=key,
                    type=common.typeName(value),
                    value=(
                        '\n '.join(value)
                        # We want the next lines to be
                        # indented, so that
                        # configparser will treat them
                        # as a single multi-line value.
                        # So we join with '\n '.
                        if isinstance(value, list)
                        else value
                    ),
                )
        self.environment[constants.DialogEnv.ANSWER_FILE_CONTENT] = content
        if self.environment[constants.DialogEnv.ANSWER_FILE]:
            answerfile = self.resolveFile(
                self.environment[constants.DialogEnv.ANSWER_FILE]
            )
            self.logger.info(
                _("Generating OTOPI answer file '{name}'").format(
                    name=answerfile,
                )
            )
            with open(answerfile, 'w') as f:
                os.fchmod(f.fileno(), 0o600)
                f.write(content)


# vim: expandtab tabstop=4 shiftwidth=4
