/************************** BEGIN fpe.h *****************************
 FAUST Architecture File
 Copyright (C) 2003-2022 GRAME, Centre National de Creation Musicale
 ---------------------------------------------------------------------
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU Lesser General Public License as published by
 the Free Software Foundation; either version 2.1 of the License, or
 (at your option) any later version.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 GNU Lesser General Public License for more details.
 
 You should have received a copy of the GNU Lesser General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 
 EXCEPTION : As a special exception, you may create a larger work
 that contains this FAUST architecture section and distribute
 that work under terms of your choice, so long as this FAUST
 architecture section is not modified.
 ********************************************************************/

#ifndef __fpe__
#define __fpe__

#include <cfenv>
#include <iostream>
#include <signal.h>

#ifndef HAVE_FEENABLEEXCEPT
#if defined(__APPLE__) && defined(__MACH__)

// Public domain polyfill for feenableexcept on OS X
// http://www-personal.umich.edu/~williams/archive/computation/fe-handling-example.c

inline int feenableexcept(unsigned int excepts)
{
    static fenv_t fenv;
    unsigned int new_excepts = excepts & FE_ALL_EXCEPT;
    // previous masks
    unsigned int old_excepts;
    
    if (fegetenv(&fenv)) {
        return -1;
    }
    old_excepts = fenv.__control & FE_ALL_EXCEPT;
    
    // unmask
    fenv.__control &= ~new_excepts;
    fenv.__mxcsr   &= ~(new_excepts << 7);
    
    return fesetenv(&fenv) ? -1 : old_excepts;
}

inline int fedisableexcept(unsigned int excepts)
{
    static fenv_t fenv;
    unsigned int new_excepts = excepts & FE_ALL_EXCEPT;
    // all previous masks
    unsigned int old_excepts;
    
    if (fegetenv(&fenv)) {
        return -1;
    }
    old_excepts = fenv.__control & FE_ALL_EXCEPT;
    
    // mask
    fenv.__control |= new_excepts;
    fenv.__mxcsr   |= new_excepts << 7;
    
    return fesetenv(&fenv) ? -1 : old_excepts;
}

#else
inline int feenableexcept(unsigned int excepts)
{
#pragma STDC FENV_ACCESS ON
    fexcept_t flags;
    /* Save current exception flags. */
    fegetexceptflag(&flags, FE_ALL_EXCEPT);
    
    feclearexcept(FE_ALL_EXCEPT);   /* clear all fp exception conditions */
    return fesetexceptflag(&flags, excepts) != 0 ? -1 : flags; /* set new flags */
    
}

inline int fedisableexcept(unsigned int excepts)
{
#pragma STDC FENV_ACCESS ON
    fexcept_t flags;
    /* Save current exception flags. */
    fegetexceptflag(&flags, FE_ALL_EXCEPT);
    
    feclearexcept(FE_ALL_EXCEPT);   /* clear all fp exception conditions */
    return fesetexceptflag(&flags, ~excepts) != 0 ? -1 : flags; /* set new flags */
}

#endif
#endif

// Install a signal handler for floating-point exceptions to turn them into C++ exceptions.
// On scope exit, restores the previous FPU environment and signal handler.
static void fPEHandler(int, siginfo_t* sip, ucontext_t*)
{
    int fe_code = sip->si_code;
    
    switch (fe_code) {
    #ifdef FPE_NOOP  // occurs in OS X
        case FPE_NOOP: throw std::runtime_error("FE_NOOP"); break;
    #endif
        case FPE_FLTDIV: throw std::runtime_error("FE_DIVBYZERO"); break;   // divideByZero
        case FPE_FLTINV: throw std::runtime_error("FE_INVALID"); break;     // invalid
        case FPE_FLTOVF: throw std::runtime_error("FE_OVERFLOW"); break;    // overflow
        case FPE_FLTUND: throw std::runtime_error("FE_UNDERFLOW"); break;   // underflow
        case FPE_FLTRES: throw std::runtime_error("FE_INEXACT"); break;     // inexact
        case FPE_FLTSUB: throw std::runtime_error("FE_INVALID"); break;     // invalid
        case FPE_INTDIV: throw std::runtime_error("INT_DIVBYZERO"); break;  // divideByZero
        case FPE_INTOVF: throw std::runtime_error("INT_OVERFLOW"); break;   // overflow
        default: throw std::runtime_error("FE_NOOP"); break;
    }
}

struct ScopedFPEHandler {
    // Installs the handler and enables FE_INVALID/FE_DIVBYZERO/FE_OVERFLOW
    ScopedFPEHandler()
    {
        feclearexcept(FE_ALL_EXCEPT);
        feenableexcept(FE_INVALID|FE_DIVBYZERO|FE_OVERFLOW);
        
        struct sigaction act;
        act.sa_sigaction = (void(*)(int, siginfo_t*, void*))fPEHandler;
        sigemptyset(&act.sa_mask);
        act.sa_flags = SA_SIGINFO;
        sigaction(SIGFPE, &act, &fOldAction);
    }
    
    ~ScopedFPEHandler()
    {
        fesetenv(&fOldEnv);
        sigaction(SIGFPE, &fOldAction, nullptr);
    }
    
    ScopedFPEHandler(const ScopedFPEHandler&) = delete;
    ScopedFPEHandler& operator=(const ScopedFPEHandler&) = delete;
    
private:
    fenv_t fOldEnv{};
    struct sigaction fOldAction{};
};

/* 
 
TRY_FPE/CATCH_FPE to be used by wrapping code that needs for be protected
 
TRY_FPE
compute(...);
CATCH_FPE
 
*/

#define TRY_FPE     \
ScopedFPEHandler fpe_scope;    \
try {               \

#define CATCH_FPE                   \
} catch (std::runtime_error e) {    \
    std::cerr << "Math exception : " << e.what() << std::endl;  \
    exit(-1);                       \
}                                   \

#endif

/**************************  END  fpe.h **************************/
