/*
 * Decompiled with CFR 0.152.
 */
package net.kano.joustsim.trust;

import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;
import net.kano.joscar.CopyOnWriteArrayList;
import net.kano.joscar.DefensiveTools;
import net.kano.joustsim.trust.CertificateHolder;
import net.kano.joustsim.trust.CertificateTrustListener;
import net.kano.joustsim.trust.CertificateTrustManager;
import net.kano.joustsim.trust.DefaultCertificateHolder;
import net.kano.joustsim.trust.SignerInfo;
import net.kano.joustsim.trust.SignerTrustManager;
import net.kano.joustsim.trust.TrustTools;
import net.kano.joustsim.trust.TrustedCertificateInfo;
import net.kano.joustsim.trust.TrustedCertificatesListener;

public class TrustedCertificatesTracker {
    private static final Logger LOGGER = Logger.getLogger(TrustedCertificatesTracker.class.getName());
    private final CertificateTrustManager certTrustMgr;
    private final SignerTrustManager signerTrustMgr;
    private Map<CertificateHolder, TrustedCertificateInfoImpl> trackedCerts = new HashMap<CertificateHolder, TrustedCertificateInfoImpl>();
    private Map<X509Certificate, SignerInfoImpl> signers = new HashMap<X509Certificate, SignerInfoImpl>();
    private CopyOnWriteArrayList<TrustedCertificatesListener> listeners = new CopyOnWriteArrayList();

    public TrustedCertificatesTracker(CertificateTrustManager certTrustMgr, SignerTrustManager signerTrustMgr) {
        this.certTrustMgr = certTrustMgr;
        this.signerTrustMgr = signerTrustMgr;
        if (certTrustMgr != null) {
            certTrustMgr.addTrustListener(new CertificateTrustListener(){

                public void trustAdded(CertificateTrustManager manager, X509Certificate cert) {
                    LOGGER.info("TrustedCertificatesTracker: now trusted: " + cert.getSubjectDN());
                    TrustedCertificatesTracker.this.certTrustAdded(cert);
                }

                public void trustRemoved(CertificateTrustManager manager, X509Certificate cert) {
                    TrustedCertificatesTracker.this.certTrustRemoved(cert);
                }
            });
        } else {
            LOGGER.fine("Warning: Trusted certificates tracker will not track explicitly trusted certificates since the certificate trust manager is null");
        }
        if (signerTrustMgr != null) {
            signerTrustMgr.addTrustListener(new CertificateTrustListener(){

                public void trustAdded(CertificateTrustManager manager, X509Certificate cert) {
                    TrustedCertificatesTracker.this.signerTrustAdded(cert);
                }

                public void trustRemoved(CertificateTrustManager manager, X509Certificate cert) {
                    TrustedCertificatesTracker.this.signerTrustRemoved(cert);
                }
            });
        } else {
            LOGGER.fine("Warning: Trusted certificates tracker will not track signer-trusted certificates since the signer trust manager is null");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void signerTrustAdded(X509Certificate signer) {
        ArrayList<TrustedCertificateInfoImpl> newTrusted = new ArrayList<TrustedCertificateInfoImpl>();
        TrustedCertificatesTracker trustedCertificatesTracker = this;
        synchronized (trustedCertificatesTracker) {
            for (TrustedCertificateInfoImpl info : this.trackedCerts.values()) {
                boolean wasTrusted = info.isSomehowTrusted();
                X509Certificate cert = info.getCertificate();
                if (!TrustTools.isSigned(signer, cert)) continue;
                this.registerSignerSignee(info, signer);
                if (wasTrusted || !info.isSomehowTrusted()) continue;
                newTrusted.add(info);
            }
        }
        for (TrustedCertificateInfoImpl aNewTrusted : newTrusted) {
            this.fireNowTrustedEvent(aNewTrusted);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void signerTrustRemoved(X509Certificate cert) {
        ArrayList<TrustedCertificateInfoImpl> noLongerTrusted = new ArrayList<TrustedCertificateInfoImpl>();
        TrustedCertificatesTracker trustedCertificatesTracker = this;
        synchronized (trustedCertificatesTracker) {
            SignerInfo signerInfo = this.signers.get(cert);
            if (signerInfo == null) {
                return;
            }
            List<TrustedCertificateInfo> signedCerts = signerInfo.getSignedCerts();
            for (TrustedCertificateInfo signedCert1 : signedCerts) {
                TrustedCertificateInfoImpl signedCert = (TrustedCertificateInfoImpl)signedCert1;
                boolean wasTrusted = signedCert.isSomehowTrusted();
                signedCert.removeSigner(signerInfo);
                if (!wasTrusted || signedCert.isSomehowTrusted()) continue;
                noLongerTrusted.add(signedCert);
            }
            this.signers.remove(cert);
        }
        for (TrustedCertificateInfoImpl aNoLongerTrusted : noLongerTrusted) {
            this.fireNoLongerTrustedEvent(aNoLongerTrusted);
        }
    }

    private void fireNowTrustedEvent(TrustedCertificateInfo info) {
        assert (!Thread.holdsLock(this));
        for (TrustedCertificatesListener listener : this.listeners) {
            listener.certificateTrusted(this, info);
        }
    }

    private void fireNoLongerTrustedEvent(TrustedCertificateInfo info) {
        assert (!Thread.holdsLock(this));
        for (TrustedCertificatesListener listener : this.listeners) {
            listener.certificateNoLongerTrusted(this, info);
        }
    }

    private synchronized void registerSignerSignee(TrustedCertificateInfoImpl info, X509Certificate signer) {
        info.addSigner(this.registerSigner(signer, info));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void certTrustAdded(X509Certificate cert) {
        boolean newTrusted;
        TrustedCertificateInfoImpl info;
        TrustedCertificatesTracker trustedCertificatesTracker = this;
        synchronized (trustedCertificatesTracker) {
            DefaultCertificateHolder holder = new DefaultCertificateHolder(cert);
            info = this.trackedCerts.get(holder);
            if (info == null) {
                return;
            }
            boolean wasTrusted = info.isSomehowTrusted();
            info.setExplicitlyTrusted(true);
            newTrusted = !wasTrusted && info.isSomehowTrusted();
        }
        if (newTrusted) {
            LOGGER.info("TrustedCertificatesTracker: cert is now trusted, firing events: " + cert.getSubjectDN());
            this.fireNowTrustedEvent(info);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void certTrustRemoved(X509Certificate cert) {
        boolean noLongerTrusted;
        TrustedCertificateInfoImpl info;
        TrustedCertificatesTracker trustedCertificatesTracker = this;
        synchronized (trustedCertificatesTracker) {
            DefaultCertificateHolder holder = new DefaultCertificateHolder(cert);
            info = this.trackedCerts.get(holder);
            if (info == null) {
                return;
            }
            boolean wasTrusted = info.isSomehowTrusted();
            info.setExplicitlyTrusted(false);
            noLongerTrusted = wasTrusted && !info.isSomehowTrusted();
        }
        if (noLongerTrusted) {
            this.fireNoLongerTrustedEvent(info);
        }
    }

    public void addTrustedCertsListener(TrustedCertificatesListener l) {
        this.listeners.addIfAbsent(l);
    }

    public void removeTrustedCertsListener(TrustedCertificatesListener l) {
        this.listeners.remove(l);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean addTrackedCertificate(X509Certificate cert) {
        boolean newTrusted;
        TrustedCertificateInfoImpl info;
        DefensiveTools.checkNull(cert, "cert");
        TrustedCertificatesTracker trustedCertificatesTracker = this;
        synchronized (trustedCertificatesTracker) {
            DefaultCertificateHolder holder = new DefaultCertificateHolder(cert);
            if (this.trackedCerts.containsKey(holder)) {
                return false;
            }
            info = new TrustedCertificateInfoImpl(cert);
            CertificateTrustManager certTrustMgr = this.certTrustMgr;
            boolean explicitlyTrusted = certTrustMgr != null ? certTrustMgr.isTrusted(cert) : false;
            info.setExplicitlyTrusted(explicitlyTrusted);
            SignerTrustManager signerTrustMgr = this.signerTrustMgr;
            if (signerTrustMgr != null) {
                List<X509Certificate> ts = signerTrustMgr.getTrustedSigners(cert);
                for (X509Certificate signer : ts) {
                    if (signer == null) continue;
                    this.registerSignerSignee(info, signer);
                }
            }
            this.trackedCerts.put(holder, info);
            newTrusted = info.isSomehowTrusted();
        }
        if (newTrusted) {
            LOGGER.info("TrustedCertificatesTracker: cert " + cert.getSubjectDN() + " is now trusted");
            this.fireNowTrustedEvent(info);
        }
        return true;
    }

    private synchronized SignerInfo registerSigner(X509Certificate signer, TrustedCertificateInfoImpl signee) {
        DefensiveTools.checkNull(signer, "signer");
        DefensiveTools.checkNull(signee, "signee");
        SignerInfoImpl info = this.getSignerInfoInstance(signer);
        info.addSignee(signee);
        return info;
    }

    private SignerInfoImpl getSignerInfoInstance(X509Certificate signer) {
        assert (Thread.holdsLock(this));
        SignerInfoImpl info = this.signers.get(signer);
        if (info == null) {
            info = new SignerInfoImpl(signer);
            this.signers.put(signer, info);
        }
        return info;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean removeTrackedCertificate(X509Certificate cert) {
        boolean noLongerTrusted;
        TrustedCertificateInfo info;
        DefensiveTools.checkNull(cert, "cert");
        TrustedCertificatesTracker trustedCertificatesTracker = this;
        synchronized (trustedCertificatesTracker) {
            info = this.trackedCerts.remove(new DefaultCertificateHolder(cert));
            if (info == null) {
                return false;
            }
            noLongerTrusted = info.isSomehowTrusted();
            List<SignerInfo> signers = info.getSigners();
            for (SignerInfo signer1 : signers) {
                SignerInfoImpl signer = (SignerInfoImpl)signer1;
                signer.removeSignee(info);
            }
        }
        if (noLongerTrusted) {
            this.fireNoLongerTrustedEvent(info);
        }
        return true;
    }

    public synchronized boolean isTrusted(X509Certificate cert) {
        DefaultCertificateHolder holder = new DefaultCertificateHolder(cert);
        TrustedCertificateInfo info = this.trackedCerts.get(holder);
        if (info == null) {
            return false;
        }
        return info.isSomehowTrusted();
    }

    /*
     * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
     */
    private static final class SignerInfoImpl
    extends DefaultCertificateHolder
    implements SignerInfo {
        private final Set<TrustedCertificateInfo> signees = new HashSet<TrustedCertificateInfo>(10);

        private SignerInfoImpl(X509Certificate cert) {
            super(cert);
        }

        private synchronized void addSignee(TrustedCertificateInfoImpl signee) {
            DefensiveTools.checkNull(signee, "signee");
            assert (TrustTools.isSigned(this.getCertificate(), signee.getCertificate()));
            this.signees.add(signee);
        }

        @Override
        public synchronized List<TrustedCertificateInfo> getSignedCerts() {
            return DefensiveTools.getUnmodifiableCopy(this.signees);
        }

        private synchronized void removeSignee(TrustedCertificateInfo signee) {
            DefensiveTools.checkNull(signee, "signee");
            this.signees.remove(signee);
        }
    }

    /*
     * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
     */
    public static final class TrustedCertificateInfoImpl
    extends DefaultCertificateHolder
    implements TrustedCertificateInfo {
        private boolean explicitlyTrusted = false;
        private final Set<SignerInfo> signers = new HashSet<SignerInfo>(5);

        private TrustedCertificateInfoImpl(X509Certificate cert) {
            super(cert);
        }

        @Override
        public synchronized boolean isSomehowTrusted() {
            return this.explicitlyTrusted || !this.signers.isEmpty();
        }

        @Override
        public synchronized boolean isExplicitlyTrusted() {
            return this.explicitlyTrusted;
        }

        private synchronized void setExplicitlyTrusted(boolean trusted) {
            this.explicitlyTrusted = trusted;
        }

        private synchronized boolean addSigner(SignerInfo signer) {
            DefensiveTools.checkNull(signer, "signer");
            return this.signers.add(signer);
        }

        @Override
        public synchronized List<SignerInfo> getSigners() {
            return DefensiveTools.getUnmodifiableCopy(this.signers);
        }

        @Override
        public synchronized boolean isSignedBy(SignerInfo signer) {
            DefensiveTools.checkNull(signer, "signer");
            return this.signers.contains(signer);
        }

        private synchronized boolean removeSigner(SignerInfo signer) {
            DefensiveTools.checkNull(signer, "signer");
            return this.signers.remove(signer);
        }
    }
}

