/*
 * Decompiled with CFR 0.152.
 */
package com.ircclouds.irc.api.negotiators;

import com.ircclouds.irc.api.CapabilityNegotiator;
import com.ircclouds.irc.api.IRCApi;
import com.ircclouds.irc.api.commands.CapCmd;
import com.ircclouds.irc.api.commands.CapEndCmd;
import com.ircclouds.irc.api.commands.CapLsCmd;
import com.ircclouds.irc.api.commands.ICommand;
import com.ircclouds.irc.api.domain.messages.ServerNumericMessage;
import com.ircclouds.irc.api.domain.messages.interfaces.IMessage;
import com.ircclouds.irc.api.listeners.IMessageListener;
import com.ircclouds.irc.api.negotiators.api.Relay;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CompositeNegotiator
implements CapabilityNegotiator,
IMessageListener {
    private static final Logger LOG = LoggerFactory.getLogger(CompositeNegotiator.class);
    private static final Pattern CAPABILITY_LS = Pattern.compile("\\sCAP\\s+([^\\s]+)\\s+LS\\s+:([\\w-_]+(?:\\s+[\\w-_]+)*)\\s*$", 0);
    private static final Pattern CAPABILITY_ACK = Pattern.compile("\\sCAP\\s+([^\\s]+)\\s+ACK\\s+:([\\w-_]+(?:\\s+[\\w-_]+)*)\\s*$", 0);
    private static final Pattern CAPABILITY_NAK = Pattern.compile("\\sCAP\\s+([^\\s]+)\\s+NAK");
    private final List<Capability> capabilities;
    private final Host host;
    private final LinkedList<Capability> requested = new LinkedList();
    private final LinkedList<Capability> acknowledged = new LinkedList();
    private final LinkedList<Capability> conversed = new LinkedList();
    private IRCApi irc;
    private Relay relay;
    private boolean negotiationInProgress;

    public CompositeNegotiator(List<Capability> capabilities, Host host) {
        if (capabilities == null) {
            throw new NullPointerException("capabilities");
        }
        this.capabilities = Collections.unmodifiableList(CompositeNegotiator.verify(capabilities));
        this.host = host;
        this.negotiationInProgress = false;
    }

    private static List<Capability> verify(List<Capability> caps) {
        for (Capability cap : caps) {
            if (cap.getId() != null && !cap.getId().isEmpty()) continue;
            throw new IllegalArgumentException("capability " + cap.getId() + " cannot have null or empty id");
        }
        return caps;
    }

    @Override
    public CapCmd initiate(final IRCApi irc) {
        if (irc == null) {
            throw new IllegalArgumentException("irc instance is required");
        }
        this.irc = irc;
        this.relay = new Relay(){

            @Override
            public void send(String msg) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("CAPABILITY: " + msg);
                }
                irc.rawMessage(msg);
            }
        };
        this.requested.clear();
        this.acknowledged.clear();
        this.conversed.clear();
        this.negotiationInProgress = true;
        return new CapLsCmd();
    }

    @Override
    public void onMessage(IMessage msg) {
        ServerNumericMessage numeric;
        if (!this.negotiationInProgress) {
            return;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("SERVER: {}", (Object)msg.asRaw());
        }
        if (msg instanceof ServerNumericMessage && (numeric = (ServerNumericMessage)msg).getNumericCode() == 1) {
            LOG.warn("Detected IRC server numeric message 001. Apparently CAP NEG phase has ended. Ending negotiation process.");
            this.endNegotiation();
            return;
        }
        String raw = msg.asRaw();
        Matcher capLs = CAPABILITY_LS.matcher(raw);
        Matcher capAck = CAPABILITY_ACK.matcher(raw);
        Matcher capNak = CAPABILITY_NAK.matcher(raw);
        try {
            if (capLs.find()) {
                LinkedList<Cap> responseCaps = CompositeNegotiator.parseResponseCaps(capLs.group(2));
                List<Capability> reject = this.unsupportedCapabilities(responseCaps);
                this.feedbackRejection(reject);
                List<Capability> request = this.requestedCapabilities(reject);
                if (request.isEmpty()) {
                    this.endNegotiation();
                } else {
                    this.requested.addAll(request);
                    this.sendCapabilityRequest();
                }
                return;
            }
            if (capAck.find()) {
                LinkedList<Cap> responseCaps = CompositeNegotiator.parseResponseCaps(capAck.group(2));
                List<Capability> confirms = this.acknowledgeCapabilities(responseCaps);
                if (!confirms.isEmpty()) {
                    this.sendCapabilityConfirmation(confirms);
                    this.acknowledged.addAll(confirms);
                    this.feedbackAcknowledgements(confirms);
                }
                msg = null;
            } else if (capNak.find()) {
                LOG.error("Capability request NOT Acknowledged: {} (this may be due to inconsistent server responses)", (Object)raw);
                this.endNegotiation();
                return;
            }
            if (!this.requested.isEmpty()) {
                LOG.info("Not all requested capabilities have been acknowledged yet. Awaiting further CAP ACK messages from server for remaining capabilities. ({})", (Object)this.repr(this.requested));
                return;
            }
            Capability cap = this.conversingCapability();
            if (cap != null) {
                do {
                    boolean continu;
                    if (msg == null) {
                        LOG.debug("Starting conversation of capability: {}", (Object)cap.getId());
                        continu = cap.converse(this.relay, null);
                    } else {
                        LOG.debug("Continuing conversation of capability: {}", (Object)cap.getId());
                        continu = cap.converse(this.relay, msg.asRaw());
                    }
                    if (continu) {
                        LOG.debug("Conversation will continue, waiting for message from server.");
                        break;
                    }
                    this.conversed.add(cap);
                    LOG.debug("Finished conversation of capability: {}", (Object)cap.getId());
                    msg = null;
                } while ((cap = this.conversingCapability()) != null);
                if (cap == null) {
                    this.endNegotiation();
                }
            }
        }
        catch (RuntimeException e) {
            LOG.error("Error occurred during CAP negotiation. Prematurely ending CAP negotiation phase and continuing IRC registration as is.", (Throwable)e);
            this.endNegotiation();
        }
    }

    static LinkedList<Cap> parseResponseCaps(String responseText) {
        LinkedList<Cap> caps = new LinkedList<Cap>();
        for (String capdesc : responseText.split("\\s+")) {
            if (capdesc.isEmpty()) continue;
            caps.add(new Cap(capdesc));
        }
        return caps;
    }

    private List<Capability> unsupportedCapabilities(LinkedList<Cap> capLs) {
        LinkedList<Capability> found = new LinkedList<Capability>();
        block0: for (Capability request : this.capabilities) {
            for (Cap available : capLs) {
                if (!request.getId().equals(available.id)) continue;
                if (request.enable() != available.isEnabled() && available.isMandatory()) continue block0;
                found.add(request);
                continue block0;
            }
        }
        LinkedList<Capability> unsupported = new LinkedList<Capability>(this.capabilities);
        unsupported.removeAll(found);
        if (LOG.isTraceEnabled()) {
            LOG.trace("Supported capabilities: {}", (Object)this.repr(found));
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Unsupported capabilities: {}", (Object)this.repr(unsupported));
        }
        return unsupported;
    }

    private List<Capability> requestedCapabilities(List<Capability> unsupported) {
        LinkedList<Capability> requests = new LinkedList<Capability>(this.capabilities);
        requests.removeAll(unsupported);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Requesting capabilities: {}", (Object)this.repr(requests));
        }
        return requests;
    }

    private void sendCapabilityRequest() {
        StringBuilder requestLine = new StringBuilder("CAP REQ :");
        for (Capability cap : this.requested) {
            requestLine.append(this.repr(cap)).append(' ');
        }
        if (requestLine.length() > 510) {
            LOG.warn("Capability request message is larger than supported by IRC. Some capabilities may not have been requested correctly. Multiple messages containing capability requests are not supported yet.");
        }
        CompositeNegotiator.send(this.irc, requestLine.toString());
    }

    private List<Capability> acknowledgeCapabilities(LinkedList<Cap> capAck) {
        LinkedList<Capability> acks = new LinkedList<Capability>();
        LinkedList<Capability> needsConfirmation = new LinkedList<Capability>();
        Iterator requestIt = this.requested.iterator();
        block0: while (requestIt.hasNext()) {
            Capability request = (Capability)requestIt.next();
            for (Cap cap : capAck) {
                if (!request.getId().equals(cap.id)) continue;
                if (request.enable() == cap.enabled) {
                    if (cap.requiresAck) {
                        needsConfirmation.add(request);
                    } else {
                        acks.add(request);
                    }
                } else {
                    LOG.warn("Inverse of requested capability was acknowledged by IRC server: {} ({})", (Object)cap.id, (Object)cap.enabled);
                    this.feedbackRejection(Collections.singletonList(request));
                }
                requestIt.remove();
                continue block0;
            }
        }
        this.acknowledged.addAll(acks);
        this.feedbackAcknowledgements(acks);
        return needsConfirmation;
    }

    private void sendCapabilityConfirmation(List<Capability> confirms) {
        StringBuilder confirmation = new StringBuilder("CAP ACK :");
        for (Capability cap : confirms) {
            confirmation.append('~').append(this.repr(cap)).append(' ');
        }
        CompositeNegotiator.send(this.irc, confirmation.toString());
    }

    private void feedbackAcknowledgements(List<Capability> caps) {
        if (this.host == null) {
            return;
        }
        for (Capability cap : caps) {
            try {
                this.host.acknowledge(cap);
            }
            catch (RuntimeException e) {
                LOG.warn("BUG: host threw a runtime exception while processing acknowledgement.", (Throwable)e);
            }
        }
    }

    private void feedbackRejection(List<Capability> caps) {
        if (this.host == null) {
            return;
        }
        for (Capability cap : caps) {
            try {
                this.host.reject(cap);
            }
            catch (RuntimeException e) {
                LOG.warn("BUG: host threw a runtime exception while processing rejection.", (Throwable)e);
            }
        }
    }

    private Capability conversingCapability() {
        LinkedList<Capability> caps = new LinkedList<Capability>(this.acknowledged);
        caps.removeAll(this.conversed);
        if (caps.isEmpty()) {
            return null;
        }
        return caps.getFirst();
    }

    private void endNegotiation() {
        this.negotiationInProgress = false;
        CompositeNegotiator.send(this.irc, new CapEndCmd());
        LOG.debug("Capability negotiation phase finished. CAP END sent.");
    }

    private String repr(Capability cap) {
        if (cap.enable()) {
            return cap.getId();
        }
        return "-" + cap.getId();
    }

    private String repr(List<Capability> caps) {
        StringBuilder line = new StringBuilder("{");
        for (Capability cap : caps) {
            line.append(this.repr(cap)).append(", ");
        }
        line.append("}");
        return line.toString();
    }

    private static void send(IRCApi irc, ICommand cmd) {
        CompositeNegotiator.send(irc, cmd.asString());
    }

    private static void send(IRCApi irc, String msg) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("NEGOTIATOR: " + msg);
        }
        irc.rawMessage(msg);
    }

    public static interface Capability {
        public String getId();

        public boolean enable();

        public boolean converse(Relay var1, String var2);
    }

    public static interface Host {
        public void acknowledge(Capability var1);

        public void reject(Capability var1);
    }

    static class Cap {
        private final boolean enabled;
        private final boolean requiresAck;
        private final boolean mandatory;
        private final String id;

        private Cap(String response) {
            int start = 0;
            boolean setEnabled = true;
            boolean setRequiresAck = false;
            boolean setMandatory = false;
            block5: for (int i = 0; i < response.length(); ++i) {
                switch (response.charAt(i)) {
                    case '-': {
                        setEnabled = false;
                        continue block5;
                    }
                    case '~': {
                        setRequiresAck = true;
                        continue block5;
                    }
                    case '=': {
                        setMandatory = true;
                        continue block5;
                    }
                    default: {
                        start = i;
                        break block5;
                    }
                }
            }
            this.id = response.substring(start);
            this.enabled = setEnabled;
            this.requiresAck = setRequiresAck;
            this.mandatory = setMandatory;
        }

        String getId() {
            return this.id;
        }

        boolean isEnabled() {
            return this.enabled;
        }

        boolean isRquiresAck() {
            return this.requiresAck;
        }

        boolean isMandatory() {
            return this.mandatory;
        }
    }
}

