/* Code to zoom into a series of images */
#include <stdio.h>
#include <setjmp.h>
#include <stdlib.h>
#include <aalib.h>
#include <math.h>
#include "jpeglib.h"

#define MIN(x,y) ((x) < (y) ? (x) : (y))
#define MAX(x,y) ((x) > (y) ? (x) : (y))

#define inline

struct hotspot
{
	unsigned int x;
	unsigned int y;
};

struct image
{
	struct image *next;
	struct image *prev;

	unsigned int width;
	unsigned int height;

	unsigned char *data;

	unsigned int num_hotspots;
	unsigned int selected_hotspot;
	struct hotspot hotspots[0];
};

static inline unsigned char
gray(unsigned r, unsigned g, unsigned b)
{
#if 1
	return sqrt(r*r + g*g + b*b);
#else
	return r + g + b / 3;
#endif
}

static unsigned char *
load_jpeg(const char *filename, unsigned int *width, unsigned int *height)
{
	struct jpeg_decompress_struct cinfo;
	FILE * infile;
	unsigned char *buffer;
	struct jpeg_error_mgr jerr;


	if ((infile = fopen(filename, "r")) == NULL) {
		fprintf(stderr, "Can't open %s\n", filename);
		return NULL;
	}

	cinfo.err = jpeg_std_error(&jerr);
	jpeg_create_decompress(&cinfo);
	jpeg_stdio_src(&cinfo, infile);
	jpeg_read_header(&cinfo, TRUE);

	jpeg_start_decompress(&cinfo);

	*width = cinfo.output_width;
	*height = cinfo.output_height;
	buffer = malloc(*width * *height);
	if (buffer) {
		char line[*width * 3];
		void *lines = line;
		unsigned char *bptr = buffer;

		while (cinfo.output_scanline < cinfo.output_height) {
			unsigned int i;
			jpeg_read_scanlines(&cinfo, &lines, 1);

			for (i = 0; i < *width; i++)
				*(bptr++) = gray(line[i*3],
						 line[i*3 + 1], line[i*3 + 2]);
		}
	}
	jpeg_finish_decompress(&cinfo);
	jpeg_destroy_decompress(&cinfo);

	fclose(infile);
	return buffer;
}

/* Figure lower x given the zoom factor and center spot */
static inline int
lower_x(const struct image *image, unsigned int spot, float zoom,
	unsigned int xscreensize)
{
	return - (image->hotspots[spot].x * zoom) + xscreensize/2;
}

static inline int
upper_x(const struct image *image, unsigned int spot, float zoom,
	unsigned int xscreensize)
{
	return (image->width - image->hotspots[spot].x) * zoom + xscreensize/2;
}

static inline int
lower_y(const struct image *image, unsigned int spot, float zoom,
	unsigned int yscreensize)
{
	return - (image->hotspots[spot].y * zoom) + yscreensize/2;
}

static inline int
upper_y(const struct image *image, unsigned int spot, float zoom,
	unsigned int yscreensize)
{
	return (image->height - image->hotspots[spot].y) * zoom +yscreensize/2;
}

static inline void
draw_image(const struct image *image,
	   int x1, int y1, int x2, int y2,
	   aa_context *context)
{
	/* Step through the screen positions */
	int x, y;
	float xstep, ystep;

	xstep = ((float)image->width)/(x2 - x1);
	ystep = ((float)image->height)/(y2 - y1);

	for (x = MAX(0, x1); x < MIN(x2, aa_imgwidth(context)); x++) {
		for (y = MAX(0, y1); y < MIN(y2, aa_imgheight(context)); y++) {
			aa_putpixel(context, x, y,
				    image->data[(unsigned)((x - x1)*xstep)
					       + (unsigned)((y - y1)*ystep)
					       * image->width]);
		}
	}
}

/* Draw an image, centered about the hotspot, at the given zoom
   factor.

   If fullscreen, returns 0 if it can't cover full screen.  Otherwise
   0 means it's microscopic.
  */
static int
draw(const struct image *image,
     float zoom, 
     aa_context *context,
     int fullscreen)
{
	int x1, y1, x2, y2;

	x1 = lower_x(image, image->selected_hotspot, zoom,
		     aa_imgwidth(context));
	x2 = upper_x(image, image->selected_hotspot, zoom,
		     aa_imgwidth(context));

	y1 = lower_y(image, image->selected_hotspot, zoom,
		     aa_imgheight(context));
	y2 = upper_y(image, image->selected_hotspot, zoom,
		     aa_imgheight(context));

	if (fullscreen) {
		if (x1 > 0
		    || y1 > 0
		    || x2 < aa_imgwidth(context)
		    || y2 < aa_imgheight(context))
		return 0;
	} else {
		if (x2 - x1 < 2)
			return 0;
	}

	draw_image(image, x1, y1, x2, y2, context);
	return 1;
}

#define ZOOM_DIFFERENCE 200

static void
zoom(struct image *image_ring, unsigned int maxdepth, aa_context *context)
{
	float base_zoom = 100.0;
	unsigned int i, depth = 1;
	int key = 0;
	aa_renderparams *params = aa_getrenderparams();
	float speed = 0.95;

	aa_autoinitkbd(context, 0);

	do {
		float zoom;
		struct image *img;

		/* Largest one. */
		if (!draw(image_ring, base_zoom, context, 1)) {
			if (depth < maxdepth) {
				image_ring = image_ring->prev;
				depth++;
				printf("Introducing new image %p (depth %u)\n",
				       image_ring, depth);
				base_zoom *= ZOOM_DIFFERENCE;
				image_ring->selected_hotspot = 
					(random() >> 1) 
					% image_ring->num_hotspots;
			}
			draw(image_ring, base_zoom, context, 0);
		}

		/* Rest */
		zoom = base_zoom / ZOOM_DIFFERENCE;
		for (i = 0, img = image_ring->next;
		     i < depth;
		     i++, zoom /= ZOOM_DIFFERENCE, img = img->next) {
			if (!draw(img, zoom, context, 0)) {
				depth = i;
				printf("Tossing image %p (depth %u)\n",
				       img, depth);
				break;
			}
		}

		aa_render(context, params,
			  0, 0,
			  aa_scrwidth(context),
			  aa_scrheight(context));
		aa_flush(context);
		base_zoom *= speed;

		key = aa_getkey(context, 0);
		switch (key) {
		case AA_UP:
			speed *= 0.99;
			break;

		case AA_DOWN:
			speed /= 0.99;
			if (speed > 0.9999) speed = 0.9999;
			break;

		case '+':
		case '=':
			params->gamma *= 0.9;
			break;

		case '-':
			params->gamma /= 0.9;
			break;

		case ' ':
			if (params->dither == AA_NONE)
				params->dither = AA_ERRORDISTRIB;
			else if (params->dither == AA_ERRORDISTRIB)
				params->dither = AA_FLOYD_S;
			else if (params->dither == AA_FLOYD_S)
				params->dither = AA_NONE;
			break;

		case 'r':
			params->inversion = !params->inversion;
			break;
		}
	} while (key != 'q');
}

#define MAX_HOTSPOTS 20

static struct image *
load_hotspots(struct image *image_ring,
	      const char *name,
	      void *buffer, unsigned int width, unsigned int height)
{
	char hotspotfile[strlen(name) + sizeof(".spots")];
	struct image *ret = malloc(sizeof(struct image) 
				   + MAX_HOTSPOTS * sizeof(struct hotspot));
	FILE *f;

	sprintf(hotspotfile, "%s.spots", name);
	f = fopen(hotspotfile, "r");
	if (!f) {
		/* Can't open: make one hotspot in the middle. */
		ret->num_hotspots = 1;
		ret->hotspots[0].x = width/2;
		ret->hotspots[0].y = height/2;
	} else {
		unsigned int i;

		for (i = 0; i < MAX_HOTSPOTS; i++) {
			if (fscanf(f, "%u,%u\n",
				   &ret->hotspots[i].x,
				   &ret->hotspots[i].y) != 2)
				break;
		}
		ret->num_hotspots = i;
		if (i == 0) {
			fprintf(stderr, "No hotspots found in `%s'\n",
				hotspotfile);
			exit(1);
		}
		else printf("Read %u hotspots for %s\n", i, name);
		fclose(f);
	}

	ret->width = width;
	ret->height = height;
	ret->data = buffer;

	if (!image_ring)
		ret->next = ret->prev = ret;
	else {
		ret->next = image_ring;
		ret->prev = ret->next->prev;
		ret->next->prev = ret;
		ret->prev->next = ret;
	}

	return ret;
}

int main(int argc, char *argv[])
{
     aa_context *context;
     struct image *image_ring = NULL;
     unsigned int i;

     if (!aa_parseoptions(NULL, NULL, &argc, argv)) {
	     printf("Usage: %s [options]\n"
		    "Options:\n"
		    "%s", argv[0], aa_help);
	     exit(1);
     }
     if (argc < 2) {
	     fprintf(stderr, "Need JPEGs as args.  Other options: %s\n",
		     aa_help);
	     exit(1);
     }

     for (i = 1; i < argc; i++) {
	     unsigned char *buffer;
	     unsigned int width, height;

	     buffer = load_jpeg(argv[i], &width, &height);
	     if (!buffer) {
		     fprintf(stderr, "Error loading `%s'\n", argv[i]);
		     exit(1);
	     }

	     image_ring = load_hotspots(image_ring, argv[i],
					buffer, width, height);
	     printf("image_ring = %p\n", image_ring);
     }

     context = aa_autoinit(&aa_defparams);
     if (context == NULL) {
	     fprintf(stderr,"Cannot initialize AA-lib. Sorry\n");
	     exit(1);
     }

     zoom(image_ring, argc-1, context);

     aa_close(context);
     return 0;
}
	
