#include "net.h"
#include <errno.h>
#include <string.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <netdb.h>
#include <sys/select.h>
#include "scan.h"
#include <stdlib.h>
#include <sys/time.h> 
extern int errno;
static fd_set wfd;
static int maxfd=0;
static int timeout_sec=2;

char *inet_htoa(long int ip){
        struct in_addr adr;
        adr.s_addr=(unsigned long int)htonl(ip);
        return inet_ntoa(adr);
}

static char *_net_namesp(int port,int c){
	struct servent *info = getservbyport(htons(port), NULL);
	if(info==NULL)
		return NULL;
	if(c==0)
		return info->s_name;
	else 
		return info->s_proto;
}

char *net_portname(int port){
	return _net_namesp(port,0);
}

char *net_portprota(int port){
	return _net_namesp(port,1);	
}

void reset_list(void){
	maxfd=0;
	FD_ZERO(&wfd);
}

void add_to_list(int sock){
	FD_SET(sock,&wfd);
	maxfd=(maxfd>sock?maxfd:sock);
}

int is_in_list(int sock){
	return FD_ISSET(sock,&wfd);
}

int get_timeout(void){
	return timeout_sec;
}

void set_timeout(int sec){
	timeout_sec=sec;
}

#define MAXOUT 0
int listen_list(void){
	int ret;
	static int outs=0;
	struct timeval timeout;
	timeout.tv_sec=get_timeout();
	timeout.tv_usec=0;
	ret=select(maxfd + 1, NULL, &wfd, NULL,&timeout);
	if(ret==0){
		outs++;
		fprintf(stderr,"Timeout: %d\n",outs);
		if(outs>MAXOUT){
			fprintf(stderr,"Reach max number of timeouts\n");
			return 2;
		}	
	} else 
		outs=0;
	if(ret==-1){
		switch(errno){
		default:
		case ENOMEM:
		case EINVAL:
		case EBADF:
			perror("select:");
			return -1;
		case EINTR:
			return 1;
		}
	}	
	return 0;
}

static void connetc_error(int er,int *status){
	errno=er;
	switch(er){
	case ECONNREFUSED:
		*status=ST_NOCONNECT;
		if(verbose_level()>3)
			perror("connect_to:");
		break;
	case EINPROGRESS:
	case EALREADY:
		*status=ST_WAIT;
		if(verbose_level()>=3)
			perror("connect_to:");
		break;
	case ETIMEDOUT:
	case ENETUNREACH:
	case EHOSTUNREACH:
	case EHOSTDOWN:
		*status=ST_FAIL;
		if(verbose_level()>=3)
			perror("connect_to:");
		break;
	case EAGAIN:
	case EMFILE:
	case ENFILE:
		*status=ST_NORESOURCE;
		if(verbose_level()>=3)
			perror("connect_to:");
		break;
	case EBADF:
	case EFAULT:
	case ENOTSOCK:
	case EISCONN:
	case EADDRINUSE:
	case EAFNOSUPPORT:
	case EACCES:
	case EPERM:
		*status=ST_ERROR;
		perror("connect_to");
		break;
	default:
		perror("connect_to");
		exit(-1);
	}	
}

void get_con_status(int sock,int *status){
	int rg;
	socklen_t len=sizeof(rg);
	if(getsockopt(sock,SOL_SOCKET, SO_ERROR,&rg, &len)==-1){
		*status=ST_ERROR;
		return;
	}
	if(rg==0){
		*status=ST_CONNECT;
	} else 
		connetc_error(rg,status);
	if(*status!=ST_WAIT)
		close_socket(sock);
}	

int connect_to(long int ip, int port, int *status){
	struct sockaddr_in serv_addr;
	int res;

	memset(&serv_addr,0, sizeof(struct sockaddr_in));
	res=socket(PF_INET, SOCK_STREAM, 0);
	if(res==-1){
		switch(errno){
		case ENOBUFS:
		case ENOMEM:
		case ENFILE:
		case EMFILE:
			*status=ST_NORESOURCE;
			if(verbose_level()>=2)
				perror("connect_to:");
			break;
		default :
		case EINVAL:
		case EACCES:
		case EPROTONOSUPPORT:
		case EAFNOSUPPORT:
			*status=ST_ERROR;
			perror("connect_to:");
		}
		return -1;
	}
	if(fcntl(res,F_SETFL,O_NONBLOCK)==-1){
		close_socket(res);
		*status=ST_ERROR;
		perror("connect_to (fcntl): ");
		return -1;
	}
	serv_addr.sin_family = AF_INET;
	serv_addr.sin_addr.s_addr= htonl(ip);
	serv_addr.sin_port = htons(port);
	if(verbose_level()>=3) 
		printf("connect_to: ip %s, port %d\n",inet_htoa(ip),port);
	if(connect(res,(struct sockaddr *) &serv_addr, sizeof(struct sockaddr_in))==-1){
		connetc_error(errno,status);
		if(*status==ST_WAIT)
			return res;
	} else 
		*status=ST_CONNECT;
	close_socket(res);
	return 0;
}
