/*
 * PSIP - A lightweight GTK GUI for pjsip
 * (C) James Budiono 2011, 2015
 * License: GNU GPL Version 3 or later, please see attached gpl-3.0.txt 
 * or http://www.gnu.org/copyleft/gpl.html
 * 
 * All the config-related functions are in this file.
 */

#include <stdlib.h> 
#include <string.h>
#include <stdio.h>

#include <unistd.h>
#include <sys/types.h>
#include <pwd.h>

#include "cJSON.h"
#include "psip.h"

gchararray config_file;
gchararray activity_log_file;
account_details *accounts;
account_details *active_account;

/* =================== Utility functions, internal ========================= */

/* === get location of config file === */
gchararray get_config_file() {
	struct passwd *p;
	p = getpwuid(getuid());
	if (p) {
		config_file = g_strdup_printf("%s" G_DIR_SEPARATOR_S ".%s", p->pw_dir, CONFIG_FILE);
		activity_log_file = g_strdup_printf("%s" G_DIR_SEPARATOR_S "%s", p->pw_dir, ACTIVITY_LOG_FILE);
	}
	else {
		config_file = CONFIG_FILE;
		activity_log_file = ACTIVITY_LOG_FILE;
	}
	g_print("Config file location: %s\n",config_file);	
	return config_file;
}

/* Parse text to JSON, then render back to text, and print! */
void doit(char *text)
{
	char *out;cJSON *json;
	
	json=cJSON_Parse(text);
	if (!json) {printf("Error before: [%s]\n",cJSON_GetErrorPtr());}
	else
	{
		out=cJSON_Print(json);
		cJSON_Delete(json);
		printf("%s\n",out);
		free(out);
	}
}

/* === load buddies - separate it from main loop so that we can reuse for loading default buddies === */
void load_buddies(cJSON *buddy_categories) {
	GtkTreeIter iter_root, iter;

	int i, no_of_categories = cJSON_GetArraySize (buddy_categories);
	for (i=0; i<no_of_categories; i++) {
		
		cJSON *category = cJSON_GetArrayItem (buddy_categories, i);
		if (!category) return;
		
		cJSON *nick;
		nick = cJSON_GetObjectItem (category, "category");
		if (nick) {
			gtk_tree_store_append (psip_state->buddytree, &iter_root, NULL);
			gtk_tree_store_set (psip_state->buddytree, &iter_root, BUDDY_NICK, nick->valuestring, BUDDY_ICON, "gtk-no", GTK_END_OF_LIST);
		}
		
		cJSON *buddies = cJSON_GetObjectItem (category, "buddies");
		if (!buddies) return;
		
		int j, no_of_buddies = cJSON_GetArraySize (buddies);
		for (j=0; j<no_of_buddies; j++) {
			cJSON *buddy = cJSON_GetArrayItem (buddies, j);
			if (buddy) {
				cJSON *nick = cJSON_GetObjectItem(buddy, "nick");
				cJSON *address = cJSON_GetObjectItem(buddy, "address");
				if (nick && address) {
					gtk_tree_store_append (psip_state->buddytree, &iter, &iter_root);
					gtk_tree_store_set (psip_state->buddytree, &iter, BUDDY_ICON, "gtk-no", BUDDY_ID, UNREGISTERED_BUDDY, BUDDY_STATUS, FALSE, BUDDY_STATUS_TEXT, "Offline", 
										BUDDY_NICK, nick->valuestring, BUDDY_ADDRESS, address->valuestring, GTK_END_OF_LIST);	
				}
			}
		}
	}
}

/* === default settings === */		
void load_default_buddies() {
	cJSON *buddy_categories = cJSON_Parse (DEFAULT_BUDDIES);
	if (buddy_categories) {
		load_buddies (buddy_categories);
		cJSON_Delete (buddy_categories);
	}	
}

void config_load_default() {
	gtk_entry_set_text (psip_state->help_command_field, DEFAULT_HELP_COMMAND);	
	gtk_entry_set_text (psip_state->ring_command_field, DEFAULT_RING_COMMAND);	
	gtk_entry_set_text (psip_state->call_timeout_field, g_strdup_printf("%d", DEFAULT_INCOMING_TIMEOUT));
	gtk_button_set_label (GTK_BUTTON(psip_state->audio_outputs[0]), "dmix");
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(psip_state->audio_outputs[0]), TRUE);
	gtk_entry_set_text (psip_state->max_calls_field, g_strdup_printf("%d", DEFAULT_MAX_CALLS));	
	gtk_combo_box_set_active (psip_state->sampling_rate_field, DEFAULT_SAMPLING_CHOICE);		
#ifdef 	DEFAULT_RINGTONE_FILE
	psip_state->ring_filename = g_strdup(DEFAULT_RINGTONE_FILE);
	gtk_file_chooser_set_filename (GTK_FILE_CHOOSER (psip_state->ring_wav_file_button), psip_state->ring_filename);
#endif
	load_default_buddies();
	add_account(); // must have at least one account
	load_account_details_to_preferences_dialog(active_account);	
	populate_account_selector();	
}

/* === load a single account details from JSON (use for loading) === */
account_details *load_account_details(cJSON *parent) {
	account_details *account;
	account = g_malloc0(sizeof(account_details));
	cJSON *object=NULL;	

	object = cJSON_GetObjectItem(parent, "name"); 
	if (object) account->name = g_strdup(object->valuestring);
	
	object = cJSON_GetObjectItem(parent, "sip-url"); 
	if (object) account->sip_url = g_strdup(object->valuestring);

	object = cJSON_GetObjectItem(parent, "registrar");
	if (object) account->registrar = g_strdup(object->valuestring);
	
	object = cJSON_GetObjectItem(parent, "realm");
	if (object) account->realm = g_strdup(object->valuestring);
	
	object = cJSON_GetObjectItem(parent, "user");
	if (object) account->user = g_strdup(object->valuestring);
	
	object = cJSON_GetObjectItem(parent, "password");
	if (object) account->password = g_strdup(object->valuestring);

	object = cJSON_GetObjectItem(parent, "use-srtp");
	if (object) account->enable_srtp = object->valueint;

	object = cJSON_GetObjectItem(parent, "proxy");
	if (object) account->proxy = g_strdup(object->valuestring);
	
	object = cJSON_GetObjectItem(parent, "active");
	if (object) account->active = object->valueint;

	// remember active account - if there are multiple, the last one is used
	if (account->active) active_account = account;
		
	return account;
}

/* === load all accounts from JSON config file === */
void load_accounts(cJSON *account_arrays) {
	account_details **acctp = &accounts;
	int i, no_of_items = cJSON_GetArraySize (account_arrays);
	for (i=0; i<no_of_items; i++) {
		*acctp = load_account_details(cJSON_GetArrayItem (account_arrays, i));
		acctp = &(*acctp)->next;
	}
	if (!active_account) active_account = accounts;
	if (active_account) active_account->active = TRUE;
}

/* === save all accounts back to the JSON config file === */
void save_accounts(cJSON *accounts_array) {
	account_details *acct = accounts;
	cJSON *account;
	
	while (acct) {
		//g_print("%s\n", acct->sip_url);
		cJSON_AddItemToArray (accounts_array, account = cJSON_CreateObject());
		cJSON_AddStringToObject (account, "name", acct->name);
		cJSON_AddStringToObject (account, "sip-url", acct->sip_url);
		cJSON_AddStringToObject (account, "registrar", acct->registrar);
		cJSON_AddStringToObject (account, "realm", acct->realm);
		cJSON_AddStringToObject (account, "user", acct->user);
		cJSON_AddStringToObject (account, "password", acct->password);
		cJSON_AddNumberToObject (account, "use-srtp", acct->enable_srtp);
		cJSON_AddStringToObject (account, "proxy", acct->proxy);
		cJSON_AddNumberToObject (account, "active", acct->active);
		acct = acct->next;
	}
}


/* =================== Public functions ===================== */

/* === remove an account from account list === */
void remove_account(account_details *acct) {
	if (!acct) return; 
	
	// update head as needed
	if (acct == accounts) {
		if (!accounts->next) return; // don't delete the only account
		accounts = accounts->next;
	} else {
		// find the parent of this account and update its next field
		account_details *p = accounts;
		while(p) {
			if (p->next == acct) {
				p->next = acct->next;
				break;
			}
			p=p->next;
		}
	}
	
	// update active account
	if (acct == active_account) { 
		active_account = active_account->next;
		if (!active_account) active_account = accounts;
		if (active_account) active_account->active = TRUE;
	}
	
	// free it
	if (acct->name) g_free(acct->name);
	if (acct->sip_url) g_free(acct->sip_url);
	if (acct->registrar) g_free(acct->registrar);
	if (acct->realm) g_free(acct->realm);
	if (acct->user) g_free(acct->user);
	if (acct->password) g_free(acct->password);
	if (acct->proxy) g_free(acct->proxy);
	g_free(acct);
}

/* === create a new empty account and insert it to the end of account lists ===*/
account_details *add_account() {
	account_details *p = accounts;
	account_details *acct;

	acct = g_malloc0 (sizeof(account_details));
	acct->name = g_strdup("New account");
	acct->sip_url = g_strdup("");
	acct->registrar = g_strdup("");
	acct->realm = g_strdup("");
	acct->user = g_strdup("");
	acct->password = g_strdup("");
	acct->proxy = g_strdup("");
	
	while (p && p->next) p=p->next;
	if (p) {
		p->next = acct;
	} else {
		accounts = acct;
	}
	if (!active_account) active_account = acct;
	return acct;
}

/* === update add an existing account ===*/
void update_account_details(account_details *acct, 
	     gchararray name, gchararray sip_url, gchararray registrar,
	     gchararray realm, gchararray user, gchararray password,
	     gchararray proxy, int enable_srtp, int active)
{
	if (!acct) return;
	g_free(acct->name); if (name) acct->name=g_strdup(name); else name=g_strdup("");
	g_free(acct->sip_url); if (sip_url) acct->sip_url=g_strdup(sip_url); else sip_url=g_strdup("");
	g_free(acct->registrar); if (registrar) acct->registrar=g_strdup(registrar); else registrar=g_strdup("");
	g_free(acct->realm); if (realm) acct->realm=g_strdup(realm); else realm=g_strdup("");
	g_free(acct->user); if (user) acct->user=g_strdup(user); else user=g_strdup("");
	g_free(acct->password); if (password) acct->password=g_strdup(password); else password=g_strdup("");
	g_free(acct->proxy); if (proxy) acct->proxy=g_strdup(proxy); else proxy=g_strdup("");
	acct->enable_srtp = enable_srtp; acct->active = active;
	
	account_details *p;
	if (active && acct != active_account) {
		active_account = acct;
		// de-activate others
		p = accounts;
		while (p) {
			if (p!=acct) p->active = FALSE;
			p=p->next;
		}
	} else if (!active && acct == active_account) {
		// activate the first account
		active_account = accounts;
		if (active_account) active_account->active = TRUE;
	}
}


/* === string must be g_strdup-ed, it will be freed here with g_free === */
void log_activity (gchararray text) {
	if (psip_state->activity_log_file) {
		gchararray timestamp = get_time_stamp();
		gchararray s = g_strdup_printf ("%s %s\n", timestamp, text);
		fputs (s, psip_state->activity_log_file);
		g_free (timestamp);
	}
	g_free (text);
}

/* === load config from json file === */
void load_config() {
	get_config_file();
	
	//make sure these objects are not destroyed until we say so		
	g_object_ref(psip_state->buddytree); 
	
	//always-default stuff
	gtk_entry_set_text (psip_state->online_status_field, DEFAULT_ONLINE_STATUS);
	gtk_range_set_value (GTK_RANGE(psip_state->quality_scaler), DEFAULT_QUALITY);
	
	FILE *f=fopen(config_file,"rb");
	if (!f) {
		config_load_default();
		return;
	}
	fseek(f,0,SEEK_END);long len=ftell(f);fseek(f,0,SEEK_SET);
	char *data=malloc(len+1);fread(data,1,len,f);fclose(f);
	
	//doit(data);
	cJSON *root = cJSON_Parse (data);
	if (!root) {
		free(data);
		config_load_default();
		return;		
	}

	// load_accounts
	cJSON *accounts_array = cJSON_GetObjectItem (root, "accounts");		
	if (accounts_array) {
		load_accounts(accounts_array);		
	} else {
		// if there is no account, create one (must one with one account at least)
		add_account();
	}
	load_account_details_to_preferences_dialog(active_account);	
	populate_account_selector();

	// load settings
	cJSON *settings = cJSON_GetObjectItem (root, "settings");
	if (settings) {
		cJSON *object = cJSON_GetObjectItem(settings, "ring-command");
		if (object) gtk_entry_set_text (psip_state->ring_command_field, object->valuestring);
		
		object = cJSON_GetObjectItem(settings, "loglevel");
		if (object) gtk_range_set_value (GTK_RANGE(psip_state->loglevel_scaler), object->valuedouble);
		
		object = cJSON_GetObjectItem(settings, "help-command");
		if (object) gtk_entry_set_text (psip_state->help_command_field, object->valuestring);

		object = cJSON_GetObjectItem(settings, "call-timeout");
		if (object) gtk_entry_set_text (psip_state->call_timeout_field, object->valuestring);

		object = cJSON_GetObjectItem(settings, "minimise-tray");
		if (object) gtk_toggle_button_set_active (psip_state->minimise_tray_field, object->valueint);

		object = cJSON_GetObjectItem(settings, "auto-register");
		if (object) gtk_toggle_button_set_active (psip_state->auto_register_field, object->valueint);

		object = cJSON_GetObjectItem(settings, "audio-in");
		if (object) gtk_button_set_label (GTK_BUTTON(psip_state->audio_inputs[0]), object->valuestring);

		object = cJSON_GetObjectItem(settings, "audio-out");
		if (object) gtk_button_set_label (GTK_BUTTON(psip_state->audio_outputs[0]), object->valuestring);

		object = cJSON_GetObjectItem(settings, "ringtone-file");
		if (object) {
			gtk_file_chooser_set_filename (GTK_FILE_CHOOSER (psip_state->ring_wav_file_button), object->valuestring);
			psip_state->ring_filename = g_strdup (object->valuestring);
		}

		object = cJSON_GetObjectItem(settings, "im-beep");
		if (object) gtk_toggle_button_set_active (psip_state->beep_on_im_field, object->valueint);

		object = cJSON_GetObjectItem(settings, "im-command");
		if (object) gtk_entry_set_text (psip_state->im_command_field, object->valuestring);

		object = cJSON_GetObjectItem(settings, "activity-log");
		if (object) gtk_toggle_button_set_active (psip_state->activity_log_field, object->valueint);
		
		object = cJSON_GetObjectItem(settings, "sampling-rate");
		if (object) gtk_combo_box_set_active (psip_state->sampling_rate_field, object->valueint);	
		
		object = cJSON_GetObjectItem(settings, "quality");
		if (object) gtk_range_set_value (GTK_RANGE(psip_state->quality_scaler), object->valuedouble);
		

		//network settings
		object = cJSON_GetObjectItem(settings, "max-calls");
		if (object) gtk_entry_set_text (psip_state->max_calls_field, object->valuestring);	
		
		object = cJSON_GetObjectItem(settings, "sip-port");
		if (object) gtk_entry_set_text (psip_state->sip_port_field, object->valuestring);	
			
		object = cJSON_GetObjectItem(settings, "public-ip");
		if (object) gtk_entry_set_text (psip_state->public_ip_field, object->valuestring);	

		object = cJSON_GetObjectItem(settings, "stun-server");
		if (object) gtk_entry_set_text (psip_state->stun_server_field, object->valuestring);	

		object = cJSON_GetObjectItem(settings, "disable-vad");
		if (object) gtk_toggle_button_set_active (psip_state->disable_vad_field, object->valueint);

		object = cJSON_GetObjectItem(settings, "enable-ice");
		if (object) gtk_toggle_button_set_active (psip_state->enable_ice_field, object->valueint);

		object = cJSON_GetObjectItem(settings, "turn-server");
		if (object) gtk_entry_set_text (psip_state->turn_server_field, object->valuestring);	

		object = cJSON_GetObjectItem(settings, "turn-user");
		if (object) gtk_entry_set_text (psip_state->turn_user_field, object->valuestring);	

		object = cJSON_GetObjectItem(settings, "turn-password");
		if (object) gtk_entry_set_text (psip_state->turn_password_field, object->valuestring);	

		object = cJSON_GetObjectItem(settings, "turn-realm");
		if (object) gtk_entry_set_text (psip_state->turn_realm_field, object->valuestring);		

		object = cJSON_GetObjectItem(settings, "turn-use-tcp");
		if (object) gtk_toggle_button_set_active (psip_state->turn_use_tcp_field, object->valueint);
		
		object = cJSON_GetObjectItem(settings, "disable-tcp");
		if (object) gtk_toggle_button_set_active (psip_state->disable_tcp_field, object->valueint);

		object = cJSON_GetObjectItem(settings, "disable-optional-srtp");
		if (object) gtk_toggle_button_set_active (psip_state->disable_optional_srtp_field, object->valueint);
	}

	//load buddies
	cJSON *buddy_categories = cJSON_GetObjectItem (root, "buddy-categories");
	if (buddy_categories) load_buddies (buddy_categories);
		
	// cleanup
	cJSON_Delete(root);	
	free(data);	
		
	//start activity log, if requested
	if (gtk_toggle_button_get_active(psip_state->activity_log_field)) {
		psip_state->activity_log_file = fopen (activity_log_file, "a");
		g_print ("Activity log file location: %s\n", activity_log_file);
	}
	log_activity (g_strdup ("Activity Log opened") );
	return;
}

/* === save current config to json file === */
void save_config() {
	log_activity (g_strdup ("Activity Log closed") );
	if (psip_state->activity_log_file) {
		FILE *ff = 	psip_state->activity_log_file;
		psip_state->activity_log_file = NULL;
		fclose (ff);
	}
	
	FILE *f=fopen(config_file,"wb");
	if (!f) return;

	//create root objects
	cJSON *root, *accounts_array, *settings, *buddy_categories;
	
	root = cJSON_CreateObject();
	cJSON_AddItemToObject (root, "accounts", accounts_array = cJSON_CreateArray());
	cJSON_AddItemToObject (root, "settings", settings = cJSON_CreateObject());
	cJSON_AddItemToObject (root, "buddy-categories", buddy_categories = cJSON_CreateArray());
	
	// save accounts
	save_accounts (accounts_array);

	//save settings
	cJSON_AddStringToObject (settings, "ring-command", gtk_entry_get_text(psip_state->ring_command_field));
	cJSON_AddNumberToObject (settings, "loglevel", gtk_range_get_value(GTK_RANGE(psip_state->loglevel_scaler)));
	cJSON_AddStringToObject (settings, "help-command", gtk_entry_get_text(psip_state->help_command_field));
	cJSON_AddStringToObject (settings, "call-timeout", gtk_entry_get_text(psip_state->call_timeout_field));
	cJSON_AddNumberToObject (settings, "minimise-tray", gtk_toggle_button_get_active(psip_state->minimise_tray_field));
	cJSON_AddNumberToObject (settings, "auto-register", gtk_toggle_button_get_active(psip_state->auto_register_field));
	if (psip_state->ring_filename) cJSON_AddStringToObject (settings, "ringtone-file", psip_state->ring_filename);
	cJSON_AddNumberToObject (settings, "im-beep", gtk_toggle_button_get_active(psip_state->beep_on_im_field));
	cJSON_AddStringToObject (settings, "im-command", gtk_entry_get_text(psip_state->im_command_field));	
	cJSON_AddNumberToObject (settings, "activity-log", gtk_toggle_button_get_active(psip_state->activity_log_field));
	cJSON_AddNumberToObject (settings, "sampling-rate", gtk_combo_box_get_active (psip_state->sampling_rate_field));
	cJSON_AddNumberToObject (settings, "quality", gtk_range_get_value(GTK_RANGE(psip_state->quality_scaler)));	

	//network settings
	cJSON_AddStringToObject (settings, "max-calls", gtk_entry_get_text(psip_state->max_calls_field));
	cJSON_AddStringToObject (settings, "sip-port", gtk_entry_get_text(psip_state->sip_port_field));		
	cJSON_AddStringToObject (settings, "public-ip", gtk_entry_get_text(psip_state->public_ip_field));
	cJSON_AddStringToObject (settings, "stun-server", gtk_entry_get_text(psip_state->stun_server_field));
	cJSON_AddNumberToObject (settings, "disable-vad", gtk_toggle_button_get_active(psip_state->disable_vad_field));
	cJSON_AddNumberToObject (settings, "enable-ice", gtk_toggle_button_get_active(psip_state->enable_ice_field));
	cJSON_AddStringToObject (settings, "turn-server", gtk_entry_get_text(psip_state->turn_server_field));
	cJSON_AddStringToObject (settings, "turn-user", gtk_entry_get_text(psip_state->turn_user_field));
	cJSON_AddStringToObject (settings, "turn-password", gtk_entry_get_text(psip_state->turn_password_field));
	cJSON_AddStringToObject (settings, "turn-realm", gtk_entry_get_text(psip_state->turn_realm_field));	
	cJSON_AddNumberToObject (settings, "turn-use-tcp", gtk_toggle_button_get_active(psip_state->turn_use_tcp_field));	
	cJSON_AddNumberToObject (settings, "disable-tcp", gtk_toggle_button_get_active(psip_state->disable_tcp_field));	
	cJSON_AddNumberToObject (settings, "disable-optional-srtp", gtk_toggle_button_get_active(psip_state->disable_optional_srtp_field));	
	
	//audio settings
	audio_settings as;
	get_audio_preference_settings (&as);
	if (as.input) cJSON_AddStringToObject (settings, "audio-in", as.input);
	if (as.output) cJSON_AddStringToObject (settings, "audio-out", as.output);	

	//save buddies - this is a bit complicated
	GtkTreeIter iter_root, iter;
	gchararray nick, address;
	cJSON *buddy, *buddies, *category;

	gboolean ok = gtk_tree_model_get_iter_first (GTK_TREE_MODEL (psip_state->buddytree), &iter_root);
	while (ok) {
		gtk_tree_model_get (GTK_TREE_MODEL (psip_state->buddytree), &iter_root, BUDDY_NICK, &nick, GTK_END_OF_LIST);
		cJSON_AddItemToArray (buddy_categories, category = cJSON_CreateObject());
		cJSON_AddStringToObject (category, "category", nick);
		cJSON_AddItemToObject (category, "buddies", buddies = cJSON_CreateArray());
		g_free(nick);
		
		gboolean ok2 = gtk_tree_model_iter_children (GTK_TREE_MODEL (psip_state->buddytree), &iter, &iter_root);
		while (ok2) {
			gtk_tree_model_get (GTK_TREE_MODEL (psip_state->buddytree), &iter, BUDDY_NICK, &nick, BUDDY_ADDRESS, &address, GTK_END_OF_LIST); 	
			cJSON_AddItemToArray (buddies, buddy = cJSON_CreateObject() );
			cJSON_AddStringToObject (buddy,"nick", nick);
			cJSON_AddStringToObject (buddy,"address", address);
			g_free(nick);
			g_free(address);

			ok2 = gtk_tree_model_iter_next (GTK_TREE_MODEL (psip_state->buddytree), &iter);
		}			
		ok = gtk_tree_model_iter_next (GTK_TREE_MODEL (psip_state->buddytree), &iter_root);
	}
	g_object_unref (psip_state->buddytree);
	
	//save file
	char *out = cJSON_Print (root);
	//g_print(out);
	fwrite (out,1,strlen(out),f);
	
	free(out);
	cJSON_Delete (root);
	fclose(f);
}

