/*
 * Copyright (C) 2001, 2002, 2003 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of
 * the License at http://www.redhat.com/licenses/ccmpl.html
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 */

package com.arsdigita.forum;

import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.RequestLocal;
import com.arsdigita.domain.DataObjectNotFoundException;
import com.arsdigita.kernel.permissions.PrivilegeDescriptor;
import com.arsdigita.messaging.MessageThread;
import com.arsdigita.util.UncheckedWrapperException;
import com.arsdigita.kernel.Kernel;
import com.arsdigita.kernel.Party;
import com.arsdigita.util.Assert;

import java.math.BigDecimal;

import org.apache.log4j.Logger;

/**
 * The entry point into all the global state parameters that the Forum
 * application expects to have available to it when running, e.g. the current
 * forum, thread, etc.
 *
 * <p>This is a request local object.</p>
 *
 * @author Vadim Nasardinov (vadimn@redhat.com)
 * @see com.arsdigita.kernel.KernelContext
 */
public final class ForumContext {

    private static final Logger s_log = Logger.getLogger(ForumContext.class);

    private static final RequestLocal s_context = new RequestLocal() {
            public Object initialValue(PageState state) {
                return new ForumContext();
            }
        };

    private BigDecimal m_threadID;
    private BigDecimal m_categorySelection;
    private MessageThread m_thread;
    private Forum m_forum;

    private boolean m_canEdit;
    private boolean m_canAdminister;
    private boolean m_canModerate;

    ForumContext() {
        m_forum = (Forum)Kernel.getContext().getResource();
        Assert.exists(m_forum, Forum.class);
        
        Party party = Kernel.getContext().getParty();
        
        m_canEdit = m_forum.canEdit(party);
        m_canModerate = m_forum.canModerate(party);
        m_canAdminister = m_forum.canAdminister(party);
    }

    public static ForumContext getContext(PageState state) {
        return (ForumContext)s_context.get(state);
    }

    static void setContext(PageState state, ForumContext context) {
        s_context.set(state, context);
    }

    /**
     * Retrieves the thread ID.
     */
    public BigDecimal getThreadID() {
        return m_threadID;
    }

    /**
     * Sets the current thread ID.
     */
    public void setThreadID(BigDecimal threadID) {
        if ( m_threadID == threadID) {
            return;
        }
        m_threadID = threadID;
        try {
            MessageThread thread = new MessageThread(threadID);
            // FIXME: (Seb 20040521)
            // Although 'Public' doesn't seem to have problem with it,
            // the assertion fails for normal registered users ??!?!?
            // Commenting it out until we find better solution.
            // thread.assertPrivilege(PrivilegeDescriptor.READ);
            m_thread =  thread;
        } catch (DataObjectNotFoundException ex) {
            throw new UncheckedWrapperException(
                "Couldn't find a MessageThread for " + threadID, ex);
        }
    }

    /**
     * Retrieves the current message thread.
     **/
    public MessageThread getMessageThread() {
        return m_thread;
    }

    /**
     * Retrieves the current Forum.
     **/
    public Forum getForum() {
        return m_forum;
    }
    
    
    public boolean canEdit() {
        return m_canEdit;
    }
    
    public boolean canAdminister() {
        return m_canAdminister;
    }
    
    public boolean canModerate() {
        return m_canModerate;
    }
    

    public BigDecimal getCategorySelection() {
        return m_categorySelection;
    }

    public void setCategorySelection(BigDecimal categorySelection) {
        m_categorySelection = categorySelection;
    }
}
