/*
 * Copyright (C) 2001, 2002, 2003 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of
 * the License at http://www.redhat.com/licenses/ccmpl.html
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 */

package com.arsdigita.forum;

import com.arsdigita.domain.DataObjectNotFoundException;
import com.arsdigita.web.URL;
import com.arsdigita.web.ParameterMap;
import com.arsdigita.messaging.ThreadedMessage;
import com.arsdigita.notification.BaseSubscription;
import com.arsdigita.persistence.DataObject;
import com.arsdigita.persistence.OID;
import com.arsdigita.util.Assert;
import com.arsdigita.util.StringUtils;
import org.apache.log4j.Logger;

/**
 * The abstract Subscription class provides the ability for Users
 * to sign up for email notifications.  Subclasses will specify
 * the object to which the notifications apply.
 *
 * The default implementation provides instant notifications.  Subclasses
 * should override sendNotification() to alter this behavior.
 *
 * @author Kevin Scaldeferri (kevin@arsdigita.com)
 */
public abstract class Subscription extends BaseSubscription {
    public static final String versionId =
        "$Id: //apps/forum/dev/src/com/arsdigita/forum/Subscription.java#8 $" +
        "$Author: sskracic $" +
        "$DateTime: 2004/06/04 17:59:27 $";

    private static final Logger s_log = Logger.getLogger(Subscription.class);

    public Subscription(String objectType) {
        super(objectType);
    }

    public Subscription(DataObject dataObj) {
        super(dataObj);
    }

    public Subscription(OID oid) throws DataObjectNotFoundException {
        super(oid);
    }

    /**
     * Returns a header for forum alerts with the following standard
     * information:
     *
     * <pre>
     * Forum    : Name
     * Subject  : Subject
     * com.arsdigita.messaging.ThreadedMessageed by: User
     * </pre>
     *
     * @return a header to insert at the top of the alert.
     */
    public String getHeader(ThreadedMessage msg) {
        Assert.truth(msg instanceof Post,
                     "Parameter msg should be a Post");

        Post post = (Post)msg;

        String author = post.getFrom().getName();
        if (author == null) {
            author = "Unknown";
        }

        StringBuffer sb = new StringBuffer();
        sb.append("Forum    : ");
        sb.append(post.getForum().getDisplayName()).append("\n");
        sb.append("Subject  : ");
        sb.append(post.getSubject()).append("\n");
        sb.append("Posted by: ");
        sb.append(author).append("\n\n");

        return sb.toString();
    }

    /**
     * Returns the signature to be appended to the alert.  The default
     * implementation returns a separator and a generic messages.
     */
    public String getSignature(ThreadedMessage post) {
        return SEPARATOR + ALERT_BLURB;
    }

    /**
     * @return an appropriate message to direct people back to the
     * forum, for inclusion in the signature of an alert, or an empty
     * string if the URL cannot be determined.
     */
    protected static String getReturnURLMessage(Post post) {
        final ParameterMap params = new ParameterMap();
        params.setParameter("thread", post.getThread().getID());

        final URL url = URL.there(post.getForum(), "/thread.jsp", params);

        StringBuffer sb = new StringBuffer();
        sb.append("To reply to this message, go to:\n");

        sb.append(url.getURL());

        return sb.toString();
    }

    /*
     * @return an appropriate separator for the body and signature of a post.
     */
    private static String getSeparator() {
        return "\n\n" + StringUtils.repeat('-', 20) + "\n\n";
    }
}
