/*
 * Copyright (C) 2001, 2002, 2003 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of
 * the License at http://www.redhat.com/licenses/ccmpl.html
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 */

package com.arsdigita.forum;

import com.arsdigita.domain.DataObjectNotFoundException;
import com.arsdigita.domain.DomainCollection;
import com.arsdigita.domain.DomainObject;
import com.arsdigita.kernel.Party;
import com.arsdigita.kernel.permissions.PermissionService;
import com.arsdigita.kernel.permissions.PrivilegeDescriptor;
import com.arsdigita.messaging.MessageThread;
import com.arsdigita.notification.Notification;
import com.arsdigita.persistence.DataCollection;
import com.arsdigita.persistence.DataObject;
import com.arsdigita.persistence.DataQuery;
import com.arsdigita.persistence.DataQueryDataCollectionAdapter;
import com.arsdigita.persistence.OID;
import com.arsdigita.persistence.SessionManager;

import java.math.BigDecimal;

/**
 * <font color="red">Experimental</font>
 * Class for managing subscriptions to individual threads in a Forum.
 *
 * @author Kevin Scaldeferri (kevin@arsdigita.com)
 */
public class ThreadSubscription extends Subscription {

    public static final String BASE_DATA_OBJECT_TYPE =
        "com.arsdigita.forum.ThreadSubscription";

    private static final String THREAD = "thread";

    private MessageThread m_thread = null;

    public ThreadSubscription() {
        super(BASE_DATA_OBJECT_TYPE);
    }

    public ThreadSubscription(String objectType) {
        super(objectType);
    }

    public ThreadSubscription(DataObject dataObj) {
        super(dataObj);
    }

    public ThreadSubscription(OID oid) throws DataObjectNotFoundException {
        super(oid);
    }

    public ThreadSubscription(BigDecimal id)
        throws DataObjectNotFoundException {
        super(new OID(BASE_DATA_OBJECT_TYPE, id));
    }

    protected String getBaseDataObjectType() {
        return BASE_DATA_OBJECT_TYPE;
    }

    /**
     * @deprecated the return type of this method will soon become
     *             MessageThread
     */
    public Post getThread() {
        return (Post)getThreadReal().getRootMessage();
    }

    public MessageThread getThreadReal() {
        if (m_thread == null) {
            DataObject threadData = (DataObject) get(THREAD);
            if (threadData != null) {
                m_thread = new MessageThread(threadData);
            }
        }
        return m_thread;
    }

    public void doWriteCheck() {
        getThread().assertPrivilege(PrivilegeDescriptor.READ);
    }

    /**
     * @deprecated
     */
    public void setThread(Post post) {
        setThread(post.getThread());
    }

    public void setThread(MessageThread thread) {
        m_thread = thread;
        setAssociation(THREAD, thread);
    }

    protected void afterSave() {
        PermissionService.setContext(this, getThreadReal());
        PermissionService.setContext(getGroup(), getThreadReal());
        super.afterSave();
    }

    /**
     *  Delete all notifications sent with the sender being the group
     * associated with this subscription.
     */
    protected void beforeDelete() {
        DataCollection notifications = SessionManager.getSession()
            .retrieve(Notification.BASE_DATA_OBJECT_TYPE);
        notifications.addEqualsFilter("partyTo", getGroup().getID());
        while (notifications.next()) {
            Notification no = new Notification(notifications.getDataObject().getOID());
            no.setMessageDelete(Boolean.FALSE);
            no.delete();
        }
        super.beforeDelete();
    }

    /**
     * Retrieves the subscription associated with a thread.  Note:
     * post must be the root of the thread.
     *
     * @deprecated
     */
    public static ThreadSubscription getThreadSubscription(Post post) {
        return ThreadSubscription.getThreadSubscription(post.getThread());
    }

    public static ThreadSubscription getThreadSubscription(
                                                           MessageThread thread) {

        DataCollection subs = SessionManager.getSession().
            retrieve(BASE_DATA_OBJECT_TYPE);

        subs.addEqualsFilter("thread.id", thread.getID());

        ThreadSubscription sub = null;
        if (subs.next()) {
            sub = new ThreadSubscription(subs.getDataObject());
        }

        subs.close();
        return sub;
    }

    public static DomainCollection getSubsForUser(Party party) {
        DataQuery subs = SessionManager.getSession()
            .retrieveQuery("com.arsdigita.forum.getUserThreadSubscriptions");

        subs.setParameter("userID", party.getID());

        return new DomainCollection(new DataQueryDataCollectionAdapter(subs, "subscription")) {
                public DomainObject getDomainObject() {
                    return new ThreadSubscription(m_dataCollection.getDataObject());
                }
            };
    }

    /**
     * Returns a signature with information about replying to the
     * message
     */
    public String getSignature(Post post) {

        return SEPARATOR
            + ALERT_BLURB
            + "You are receiving this email because you subscribed to "
            + "alerts on this thread.\n\n"
            + REPLY_BLURB
            + getReturnURLMessage(post);
    }
}
