/*
 * Copyright (C) 2001, 2002, 2003 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of
 * the License at http://www.redhat.com/licenses/ccmpl.html
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 */

package com.arsdigita.forum.ui;

import com.arsdigita.bebop.Container;
import com.arsdigita.bebop.FormData;
import com.arsdigita.bebop.FormProcessException;
import com.arsdigita.bebop.FormStep;
import com.arsdigita.bebop.Label;
import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.SimpleContainer;
import com.arsdigita.bebop.Wizard;
import com.arsdigita.bebop.event.FormCancelListener;
import com.arsdigita.bebop.event.FormInitListener;
import com.arsdigita.bebop.event.FormProcessListener;
import com.arsdigita.bebop.event.FormSectionEvent;
import com.arsdigita.bebop.event.PrintEvent;
import com.arsdigita.bebop.event.PrintListener;
import com.arsdigita.bebop.form.TextArea;
import com.arsdigita.bebop.form.TextField;
import com.arsdigita.bebop.parameters.NotEmptyValidationListener;
import com.arsdigita.bebop.parameters.ParameterModel;
import com.arsdigita.bebop.parameters.StringLengthValidationListener;
import com.arsdigita.bebop.parameters.StringParameter;
import com.arsdigita.forum.Post;
import com.arsdigita.kernel.Kernel;
import com.arsdigita.kernel.security.UserContext;
import com.arsdigita.toolbox.ui.TextTypeWidget;
import com.arsdigita.xml.Element;
import com.arsdigita.util.HtmlToText;
import com.arsdigita.util.MessageType;
import com.arsdigita.util.UncheckedWrapperException;
import com.arsdigita.web.RedirectSignal;

import java.io.IOException;
import org.apache.log4j.Logger;

/**
 * Class PostForm
 *
 * @author Jon Orris (jorris@arsdigita.com)
 *
 * @version $Revision #1 $DateTime: 2003/09/15 13:58:12 $
 */
public abstract class PostForm extends Wizard implements Constants {

    public static final String versionId =
        "$Id: //apps/forum/dev/src/com/arsdigita/forum/ui/PostForm.java#8 $" +
        "$Author: dan $" +
        "$DateTime: 2003/09/15 13:58:12 $";

    private static final Logger s_log = Logger.getLogger(PostForm.class);

    private TextField m_subject;
    private TextArea m_body;
    private TextTypeWidget m_bodyType;

    public PostForm(String name) {
        super(name, new SimpleContainer());        
    }

    protected void setupComponent() {
        add(dataEntryStep());
        add(confirmStep());

        addInitListener(new PostInitListener());
        addProcessListener(new PostProcessListener());
        addCancelListener(new PostCancelListener());
    }

    protected void setSubject(PageState state,
                              String text) {
        m_subject.setValue(state, text);
    }

    protected Container dataEntryStep() {
        FormStep initial = new FormStep(
            "initial",
            new SimpleContainer("forum:postForm", FORUM_XML_NS));

        m_subject = new TextField(new StringParameter("subject"));
        m_subject.addValidationListener(new NotEmptyValidationListener());
        m_subject.addValidationListener(new StringLengthValidationListener(250));
        m_subject.setSize(60);
        initial.add(m_subject);

        m_body = new TextArea(new StringParameter("message"),
                              8, 60, TextArea.SOFT);
        m_body.addValidationListener(new NotEmptyValidationListener());
        m_body.addValidationListener(new StringLengthValidationListener(4000));
        initial.add(m_body);

        m_bodyType = new TextTypeWidget(new StringParameter("bodyType"),
                                        MessageType.TEXT_PLAIN);
        initial.add(m_bodyType);

        return initial;
    }

    protected Container confirmStep() {
        SimpleContainer postContainer = new SimpleContainer
            ("forum:postConfirm", FORUM_XML_NS) {
                
                public void generateXML(PageState state,
                                        Element parent) {
                    Element content = generateParent(parent);
                    
                    Element subject = content.newChildElement("subject");
                    subject.setText((String)m_subject.getValue(state));

                    Element body = content.newChildElement("body");
                    body.setText(HtmlToText.generateHTMLText(
                                     (String)m_body.getValue(state),
                                     (String)m_bodyType.getValue(state)));
                }
            };
        
        return postContainer;
    }

    protected abstract Post getPost(PageState state,
                                    boolean create);

    protected void initWidgets(PageState state,
                               Post post) {
        if (post != null) {
            m_subject.setValue(state, post.getSubject());
            m_body.setValue(state, post.getBody());
            m_bodyType.setValue(state, post.getBodyType());
        }
    }
    
    protected void processWidgets(PageState state,
                                  Post post) {
        post.setSubject((String)m_subject.getValue(state));
        post.setBody((String)m_body.getValue(state),
                     (String)m_bodyType.getValue(state));
    }

    private class PostInitListener implements FormInitListener {        
        public void init(FormSectionEvent e) {
            PageState state = e.getPageState();
            
            if ( Kernel.getContext().getParty() == null ) {
                UserContext.redirectToLoginPage(state.getRequest());
            }
            
            initWidgets(state,
                        getPost(state, false));
        }
    }

    private class PostProcessListener implements FormProcessListener {   
        public void process(FormSectionEvent e)
            throws FormProcessException {

            PageState state = e.getPageState();
            
            Post post = getPost(state, true);
            processWidgets(state, post);
            
            post.sendNotifications();
            post.sendModeratorAlerts();
        
            post.save();

            fireCompletionEvent(state);
        }
    }

    private class PostCancelListener implements FormCancelListener { 
        public void cancel(FormSectionEvent e) 
            throws FormProcessException {

            PageState state = e.getPageState();
            fireCompletionEvent(state);
        }
    }
}
