/*
 * Copyright (C) 2001, 2002, 2003 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of
 * the License at http://www.redhat.com/licenses/ccmpl.html
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 */

package com.arsdigita.forum;

import com.arsdigita.categorization.Category;
import com.arsdigita.categorization.CategoryCollection;
import com.arsdigita.domain.DataObjectNotFoundException;
import com.arsdigita.kernel.User;
import com.arsdigita.persistence.DataCollection;
import com.arsdigita.persistence.PersistenceException;
import com.arsdigita.persistence.Session;
import com.arsdigita.persistence.SessionManager;
import com.arsdigita.tools.junit.framework.BaseTestCase;

/**
 * Test cases for bboard posts.
 *
 * @author Kevin Scaldeferri (kevin@arsdigita.com)
 * @author Ron Henderson (ron@arsdigita.com)
 */

public class PostTest extends BaseTestCase {

    public static final String versionId = "$Id: //apps/forum/dev/test/src/com/arsdigita/forum/PostTest.java#5 $ by $Author: vadim $, $DateTime: 2004/01/09 16:26:51 $";

    protected static final String SUBJECT = "This is the subject";
    protected static final String BODY    = "<p>This is the <b>body</b></p>";
    protected static final String TYPE    = "text/html";

    private User  m_user  = null;
    private Post  m_msg   = null;
    private Forum m_forum = null;

    public PostTest(String name) {
        super(name);
    }

    /**
     * Create a default forum with a single post.
     */

    public void setUp() {

        m_forum = Forum.create("bboard_message_test", "name", null);
        m_forum.setDescription("description");
        m_forum.save();

        m_user = BboardSuite.createUser();

        m_msg = Post.create(m_forum);
        m_msg.setSubject(SUBJECT);
        m_msg.setBody(BODY,TYPE);
        m_msg.setFrom(m_user);
        m_msg.save();
    }

    /**
     * Create a simple message using the various set methods.
     */


    public void testMessageCreate() {
        assertEquals(SUBJECT,m_msg.getSubject());
        assertEquals(BODY,m_msg.getBody());
        assertEquals(TYPE,m_msg.getBodyType());
        assertEquals(m_forum.getID(),m_msg.getRefersTo());
    }

    /**
     * Create a simple message and save it to the database.
     */

    public void testMessageRetrieve() {
        // Retrieve a copy of the message and verify its contents
        Post msg2 = null;
        try {
            msg2 = new Post(m_msg.getOID());
        } catch (DataObjectNotFoundException e) {
            fail("Couldn't retrieve bboard message after we saved it");
        }

        assertEquals(m_msg.getSubject(), msg2.getSubject());
        assertEquals(m_msg.getBody(), msg2.getBody());
        assertEquals(m_msg.getBodyType(), msg2.getBodyType());

        assertEquals(m_msg.getRefersTo(), msg2.getRefersTo());
    }

    /* Tests basic retrieve all. Adds to rows and retrieves
     * them in reverse order. This also checks setting, storing
     * and getting the refersTo attribute.
     */
    public void testCreateThread004() {
        Post msg2 = Post.create(m_forum);
        msg2.setSubject(SUBJECT);
        msg2.setBody(BODY,TYPE);
        msg2.setFrom(m_user);
        msg2.save();


        Session session = SessionManager.getSession();
        DataCollection msgThread =
            session.retrieve("com.arsdigita.messaging.ThreadedMessage");
        msgThread.addEqualsFilter("objectID", m_forum.getID());
        msgThread.addOrder("id desc");

        int x = 1;

        while (msgThread.next()) {
            if (x==1) {
                assertEquals(msgThread.get("id"), msg2.getID());
                assertEquals(msgThread.get("objectID"), m_msg.getRefersTo());
            } else if (x==2) {
                assertEquals(msgThread.get("id"), m_msg.getID());
            }

            x++;
        }

    }

    /**
     * Create a post and categorize it.
     */

    public void testCategorizedPost() {

        Category root = m_forum.getRootCategory();
        ForumTest.buildSubCategories(root);

        // Retrieve a child category to use for the post


        CategoryCollection categories = root.getChildren();
        assertTrue("has at least one category", categories.next());
        Category postCategory = categories.getCategory();

        // Create a new post and categorize it

        Post post = Post.create(m_forum);
        post.setSubject(SUBJECT);
        post.setBody(BODY,TYPE);
        post.setFrom(m_user);

        // Verify that the post cannot be categorized until we save it.

        try {
            post.mapCategory(postCategory);
            fail("Categorized an unsaved post");
        } catch (PersistenceException ex) {
            // correct
        }

        post.save();
        post.mapCategory(postCategory);

        // Verify that the category now contains one child object.
        assertEquals(1, postCategory.getNumberOfChildObjects());

        assertEquals(postCategory, post.getCategory());
    }


    public void testReplyTo() {
        Post reply = (Post)m_msg.replyTo();
        reply.setSubject(SUBJECT);
        reply.setBody(BODY, TYPE);
        reply.setFrom(BboardSuite.createUser());
        reply.save();
    }

    public void testSendNotifications() {
        m_msg.sendNotifications();
    }

    public void testSubscriptionCreation() {
        m_msg.createThreadSubscription();
        assertNotNull(ThreadSubscription.getThreadSubscription(
                                                               m_msg.getThread()));
    }
    /**
     * Main method required to make this test runnable.
     */

    public static void main(String args[]) {
        junit.textui.TestRunner.run(PostTest.class);
    }

}
