/***************************************************************************
 *   Copyright (C) 2006 by Stephan Binner <binner@kde.org>                 *
 *   Copyright (c) 2006 Debajyoti Bera <dbera.web@gmail.com>               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.          *
 ***************************************************************************/

#include "kickoff-beagle-plugin.h"

#include <qregexp.h>
#include <qtimer.h>

#include <kapplication.h>
#include <kdesktopfile.h>
#include <kgenericfactory.h>
#include <kservice.h>

QString dc_identifier = "dc:identifier";
QString dc_title = "dc:title";
QString parent_dc_title = "parent:dc:title";
QString exactfilename = "beagle:ExactFilename";
QString fixme_name = "fixme:Name";
QString beagle_filename = "beagle:Filename";
QString fixme_attachment_title = "fixme:attachment_title";
QString fixme_hasattachments = "fixme:hasAttachments";
QString parent_prefix = "parent:";
QString fixme_folder = "fixme:folder";
QString fixme_categories = "fixme:Categories";
QString fixme_comment = "fixme:Comment";
QString fixme_width = "fixme:width";
QString fixme_height = "fixme:height";
QString fixme_from_address = "fixme:from_address";
QString fixme_artist = "fixme:artist";
QString fixme_album = "fixme:album";
QString dc_source = "dc:source";
QString dc_publisher = "dc:publisher";
QString digikam_tag = "digikam:Tag";
QString fixme_speakingto = "fixme:speakingto";
QString fixme_starttime = "fixme:starttime";
QString comma_string = ",";

static CATEGORY getHitCategory (Hit *hit)
{
    QString hittype = hit->getType();
    QString hitsource = hit->getSource();

    // if hit source is None, dont handle it. Might be anthrax-envelope :)
    if (hitsource.isNull())
        return OTHER;

    if (hitsource == "documentation")
        return DOCS;

    if (hittype == "IMLog")
        return CHATS;

    // sure shots
    if (hittype == "FeedItem")
        return FEEDS;
    if (hittype == "WebHistory")
        return WEBHIST;
    if (hittype == "MailMessage")
        return MAILS;
    if (hittype == "Note")
        return NOTES;

    // check for applications
    if (hittype == "File" && (*hit) ["beagle:FilenameExtension"] == ".desktop")
        return APPS;

    // check for music
    QString hitmimetype = hit->getMimeType();
    if (hitsource == "Amarok"
        || hitmimetype.startsWith ("audio")
        || hitmimetype == "application/ogg")
        return MUSIC; // not an exhaustive search

    // check for images from files
    if (hitsource == "Files" && hitmimetype.startsWith ("image"))
        return PICS;

    if (hitsource == "Files" && hitmimetype.startsWith ("video"))
        return VIDEOS;

    if (hitsource == "Files")
        return FILES;

    return OTHER;
}

K_EXPORT_COMPONENT_FACTORY( kickoffsearch_beagle,
                            KGenericFactory<KickoffBeaglePlugin>( "kickoffsearch_beagle" ) )

KickoffBeaglePlugin::KickoffBeaglePlugin(QObject *parent, const char* name, const QStringList&)
            : KickoffSearch::Plugin(parent, name ), genericTitle( true )
{
    g_type_init ();
    current_beagle_client = NULL;
}

bool KickoffBeaglePlugin::daemonRunning()
{
    return beagle_util_daemon_is_running();
}

void KickoffBeaglePlugin::query(QString term, bool _genericTitle)
{
    genericTitle = _genericTitle;
    current_query_str = term;

    // Beagle search
    if (current_beagle_client != NULL) {
	kdDebug () << "Previous client w/id " << current_beagle_client->id << " still running ... ignoring it." << endl;
	current_beagle_client->stopClient ();
    }
    current_beagle_client_id = KApplication::random ();
    kdDebug () << "Creating client with id:" << current_beagle_client_id << endl;

    BeagleClient *beagle_client = beagle_client_new (NULL);
    if (beagle_client == NULL) {
        kdDebug() << "beagle service not running ..." << endl;
        return;
    }

    QStringList sources, types;
    BeagleQuery *beagle_query = BeagleUtil::createQueryFromString (term, sources, types, 99); // maximum 99 results, if this doesnt work, blame the stars

    current_beagle_client = new BeagleSearchClient (
                                current_beagle_client_id,
                                this,
                                beagle_client,
                                beagle_query,
                                false);
    current_beagle_client->start();
//    kdDebug () << "Query dispatched at " << time (NULL) << endl;
}

void KickoffBeaglePlugin::cleanClientList ()
{
    toclean_list_mutex.lock ();
    BeagleSearchClient *old_client = toclean_client_list.take (0);
    if (old_client != NULL) { // failsafe
	kdDebug () << "Cleanup old client " << old_client->id << endl;
	delete old_client;
    }
    toclean_list_mutex.unlock ();
}

void KickoffBeaglePlugin::customEvent (QCustomEvent *e)
{
    if (e->type () == RESULTFOUND) {
//        kdDebug () << "Quick query thread at " << time (NULL) << " with current_id=" << current_beagle_client_id <<  " finished ..." << endl;
        BeagleSearchResult *result = (BeagleSearchResult *) e->data ();
        if (current_beagle_client_id != result->client_id) {
            kdDebug () << "Stale result from " << result->client_id << endl;
	    delete result;
	    // FIXME: Should I also free e ?
        } else {
            kdDebug () << "Good results ...total=" << result->total << endl;
            showResults (result);
        }
        //KPassivePopup::message( "This is the message", this );
    } else if (e->type () == SEARCHOVER) {
        BeagleSearchClient *client = (BeagleSearchClient *) e->data ();
	if (client == NULL) {
//	    kdDebug () << "Query finished event at " << time (NULL) << " but client is already deleted" << endl;
	    return;
	}
//        kdDebug () << "Query finished event at " << time (NULL) << " for id=" << client->id << endl;
	if (current_beagle_client_id == client->id) {
	    kickoffSearchInterface()->searchOver();
 	    current_beagle_client = NULL; // important !
	}
    } else if (e->type () == KILLME) {
        BeagleSearchClient *client = (BeagleSearchClient *) e->data ();
	if (client->finished ())
	    delete client;
	else {
	    // add client to cleanup list
	    toclean_list_mutex.lock ();
	    toclean_client_list.append (client);
	    kdDebug () << "Scheduling client to be deleted in 500ms" << endl;
	    toclean_list_mutex.unlock ();
	    QTimer::singleShot (500, this, SLOT (cleanClientList ()));
	}
    }
}

// this method decides what to display in the result list
HitMenuItem *KickoffBeaglePlugin::hitToHitMenuItem (int category, Hit *hit)
{
    QString title, info, mimetype, icon;
    int score = 0;
    KURL uri;

#if 0
    kdDebug() << "*** " << hit->getUri() << endl;
    QDict<QStringList> all = hit->getAllProperties();
    QDictIterator<QStringList> it( all );
    for( ; it.current(); ++it )
        kdDebug() << it.currentKey() << ": " << *(it.current()) << endl;
#endif

    switch (category) {
	case FILES:
	    {
		uri = hit->getUri ();
		QString uristr = uri.path ();
	    	title = (*hit) [exactfilename];
	    	int last_slash = uristr.findRev ('/', -1);
                info = i18n("Folder: %1").arg(last_slash == 0 ? "/" 
                        : uristr.section ('/', -2, -2));
	    }
	    break;
	case MAILS:
	    {
		QString prefix = QString::null;
		bool is_attachment = ((*hit) [parent_prefix + fixme_hasattachments] == "true");
		bool has_parent = (! hit->getParentUri ().isEmpty ());
		bool parent_mbox_file = false;
		if (has_parent)
		    parent_mbox_file = ((*hit) [parent_prefix + fixme_folder] == QString::null);

		// Logic:
		// If has_parent == false, everything is normal
		// If has_parent == true, parent_mbox_file == false, everything is normal, use uri
		// FIXME: If has_parent == true, parent_mbox_file == true, ???
		// If has_parent == true, is_attachment == true, hit is attach and access with prefix "parent:", use parenturi
		// Else, not attachment (multipart), access with prefix "parent:", use parenturi

		if (has_parent && !parent_mbox_file) {
		    uri = hit->getParentUri ();
		    prefix = parent_prefix;
		    if (is_attachment)
			title = (*hit) [fixme_attachment_title];
		    if (title.isEmpty ())
			title = (*hit) [prefix + dc_title];
		    if (title.isEmpty ())
			title = i18n("No subject");
		    if (is_attachment)
			title = title.prepend (i18n("(Attachment) "));
		    info = (i18n("From %1").arg((*hit) [prefix + fixme_from_address]));
		} else {
		    uri = hit->getUri ();
		    title = (*hit) [dc_title];
		    info = (i18n("From %1").arg((*hit) [fixme_from_address]));
		}
	    }
	    mimetype = "message/rfc822"; // to handle attachment results
	    break;
 	case MUSIC:
	    uri = hit->getUri ();
	    title = (*hit) [exactfilename];
	    {
		QString artist = (*hit) [fixme_artist];
		QString album = (*hit) [fixme_album];
		if (! artist.isEmpty ())
		    info = (i18n("By %1").arg(artist));
		else if (! album.isEmpty ())
		    info = (i18n("From Album %1").arg(album));
		else {
		    QString uristr = uri.path ();
		    int last_slash = uristr.findRev ('/', -1);
                    info = i18n("Folder: %1")
                        .arg(last_slash == 0 ? "/" : uristr.section ('/', -2, -2));
		}
	    }
	    break;
 	case VIDEOS:
	    uri = hit->getUri ();
	    title = (*hit) [exactfilename];
	    {
		QString uristr = uri.path ();
		int last_slash = uristr.findRev ('/', -1);
                info = i18n("Folder: %1").arg(last_slash == 0 ? "/" : uristr.section ('/', -2, -2));
	    }
	    break;
	case WEBHIST:
	    uri = hit->getUri ();
	    title = (*hit) [dc_title];
	    title = title.replace(QRegExp("\n")," ");
	    mimetype = "text/html";
	    if (title.isEmpty () || title.stripWhiteSpace ().isEmpty ()) {
		title = uri.prettyURL ();
	    } else {
		info = uri.host () + uri.path ();
	    }
	    break;
	case FEEDS:
	    {
		uri = KURL ((*hit) [dc_identifier]);
	    	title = (*hit) [dc_title];
	    	mimetype = "text/html";
	    	QString publisher = (*hit) [dc_publisher];
	    	QString source = (*hit) [dc_source];
	    	if (! publisher.isEmpty ())
	    	    info = publisher;
	    	else if (! source.isEmpty ())
	    	    info = source;
	    }
	    break;
	case PICS:
	    {
		uri = hit->getUri ();
		title = (*hit) [exactfilename];
		QString width = (*hit) [fixme_width];
		QString height = (*hit) [fixme_height];
		if (width.isEmpty () || height.isEmpty ()) {
		    QString uristr = uri.path ();
		    int last_slash = uristr.findRev ('/', -1);
                    info = i18n("Folder: %1")
                        .arg(last_slash == 0 ? "/" : uristr.section ('/', -2, -2));
		    break;
		}
		info = (QString (" (%1x%2)").arg (width).arg (height));
		const QStringList *tags = hit->getProperties (digikam_tag);
		if (tags == NULL)
		    break;
		QString tags_string = tags->join (comma_string);
		info += (" " + tags_string);
	    }
	    break;
	case APPS:
	    {
		uri = hit->getUri ();
	    	title = (*hit) [dc_title];
		KDesktopFile desktopfile(uri.path(),true);
		if (genericTitle && !desktopfile.readGenericName().isEmpty()) {
		  title = desktopfile.readGenericName();
		  info = desktopfile.readName();
		}
		else {
		  title = desktopfile.readName();
		  info = desktopfile.readGenericName();
		}
		icon = desktopfile.readIcon();
                QString input = current_query_str.lower();
		QString command = desktopfile.readEntry("Exec");
		if (command==input)
                  score = 100;
                else if (command.find(input)==0)
                  score = 50;
                else if (command.find(input)!=-1)
                  score = 10;
		else if (title==input)
                  score = 100;
                else if (title.find(input)==0)
                  score = 50;
                else if (title.find(input)!=-1)
                  score = 10;
	    	break;
	    }
	    break;
	case NOTES:
            {
	        uri = hit->getUri ();
	        title = (*hit) [dc_title];
	        title = i18n("Title: %1").arg(title.isEmpty() ? i18n("Untitled") : title);

	        if (hit->getSource()=="KNotes")
                   icon="knotes";
                else
                   icon="contents2";
            }
	    break;
	case CHATS:
            {
	        uri = hit->getUri ();
	        title = (*hit) [fixme_speakingto];
	        title = i18n("Conversation With %1").arg(title.isEmpty() ? i18n("Unknown Person") : title);
	        QDateTime datetime;
	        datetime = datetimeFromString((*hit) [fixme_starttime]);
                info=i18n("Date: %1").arg(KGlobal::locale()->formatDateTime(datetime,false));
	        if (hit->getMimeType()=="beagle/x-kopete-log")
                   icon="kopete";
                else
                   icon="gaim";
            }
	    break;
	case DOCS:
	    {
		uri = hit->getUri ();
		title = (*hit) [dc_title];
		if (title.isEmpty () || title.stripWhiteSpace ().isEmpty ())
		    title = uri.prettyURL ();
		else {
			QString uristr = uri.path ();
			int last_slash = uristr.findRev ('/', -1);
                        info = i18n("Folder: %1").arg(last_slash == 0 ? "/" : uristr.section ('/',
                                    -2, -2));
		}
	    }
	    break;
	default:
	    return NULL;
    }
    if (mimetype.isEmpty ())
	mimetype = hit->getMimeType ();
    return new HitMenuItem (title, info, uri, mimetype, 0, category, icon, score);
}

void KickoffBeaglePlugin::showResults(BeagleSearchResult *result)
{
    if (result->total == 0 ) {
	// Dont report error from here ...
        kdDebug() << "No matches found" << endl;
	delete result;
	return;
    }

    const QPtrList<Hit> *hits = result->getHits();
    if (hits == NULL) {
        kdDebug () << "Hmm... null" << endl;
	delete result;
        return;
    }
    kickoffSearchInterface()->initCategoryTitlesUpdate();

    QPtrListIterator<Hit> it (*hits);
    Hit *hit;
    for (; (hit = it.current ()) != NULL; ++it) {
	CATEGORY category = getHitCategory (hit);

	// if category is not handled, continue
	if (category == OTHER)
          continue;

        if ( category == APPS ) {
            // we need to check if this is useful
            KService cs( hit->getUri().path() );
            if ( cs.noDisplay() )
                continue;
        }

        if (!kickoffSearchInterface()->anotherHitMenuItemAllowed(category))
          continue;

        HitMenuItem *hit_item = hitToHitMenuItem (category, hit);

        if (!hit_item)
	   continue;

        kickoffSearchInterface()->addHitMenuItem(hit_item);
    }

    kickoffSearchInterface()->updateCategoryTitles();

    delete result;
}

QDateTime KickoffBeaglePlugin::datetimeFromString( const QString& s)
{
      int year( s.mid( 0, 4 ).toInt() );
      int month( s.mid( 4, 2 ).toInt() );
      int day( s.mid( 6, 2 ).toInt() );
      int hour( s.mid( 8, 2 ).toInt() );
      int min( s.mid( 10, 2 ).toInt() );
      int sec( s.mid( 12, 2 ).toInt() );
      return QDateTime(QDate(year,month,day),QTime(hour,min,sec));
}

#include "kickoff-beagle-plugin.moc"
