/*****************************************************************

Copyright (c) 1996-2000 the kicker authors. See file AUTHORS.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************/

#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>
#include <dmctl.h>

#include <qimage.h>
#include <qpainter.h>
#include <qstyle.h>
#include <qwidgetstack.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qvbox.h>
#include <qheader.h>
#include <qdrawutil.h>
#include <qdragobject.h>
#include <qcursor.h>
#include <qpaintdevicemetrics.h>
#include <qbuffer.h>
#include <qtooltip.h>
#include <qstylesheet.h>
#include <qiconview.h>

#include <dcopclient.h>
#include <kapplication.h>
#include <kaboutkde.h>
#include <kpixmapeffect.h>
#include <kaction.h>
#include <kbookmarkmenu.h>
#include <kconfig.h>
#include <kdebug.h>
#include <kglobal.h>
#include <kglobalsettings.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kstandarddirs.h>
#include <kcombobox.h>
#include <kwin.h>
#include <kdebug.h>
#include <kmimetype.h>
#include <kmultipledrag.h>

#include "client_mnu.h"
#include "container_base.h"
#include "global.h"
#include "kbutton.h"
#include "kicker.h"
#include "kickerSettings.h"
#include "konqbookmarkmanager.h"
#include "menuinfo.h"
#include "menumanager.h"
#include "popupmenutitle.h"
#include "quickbrowser_mnu.h"
#include "recentapps.h"

#include "k_mnu.h"
#include "k_new_mnu.h"
#include "itemview.h"

// --------------------------------------------------------------------------

KMenuItem::~KMenuItem()
{
    ItemView *listview = dynamic_cast<ItemView*>( listView() );
    if ( listview && listview->m_lastOne == this) {
      listview->m_lastOne = 0;
      listview->m_old_contentY = -1;
    }
}

static double pointSize( double pixelSize, QPaintDevice *w )
{
    return pixelSize * 72. / QPaintDevice::x11AppDpiY( w->x11Screen () );
}

static int pixelSize( double pixelSize, QPaintDevice *w )
{
    return qRound( pixelSize * QPaintDevice::x11AppDpiY( w->x11Screen () ) / 72. );
}

void KMenuItem::init()
{
    setMultiLinesEnabled(true);
    m_s = 0;
    m_path = QString::null;
    m_icon = QString::null;
    m_menuPath = QString::null;
    setDragEnabled(true);
    m_has_children = false;
    m_old_width = -1;
    right_triangle.load( locate( "appdata", "pics/right_triangle.png" ) );
}

void KMenuItem::setTitle(const QString& txt)
{
    m_title = txt;
    setText( 0, txt );
    setup();
}

void KMenuItem::setToolTip(const QString& txt)
{
    m_tooltip = txt;
}

void KMenuItem::setDescription(const QString& txt)
{
    m_description = txt;
    setup();
}

void KMenuItem::setIcon(const QString& icon, int size)
{
    m_icon = icon;
    QListViewItem::setPixmap(0, KGlobal::iconLoader()->loadIcon(icon, KIcon::Panel, size ));
}

void KMenuItem::setHasChildren( bool flag )
{
    m_has_children = flag;
    repaint();
}

void KMenuItem::setup()
{
    // if someone configured a larger generalFont than 10pt, he might have a _real_ problem with 7pt
    // the 7pt could be read out of konquerorrc I guess
    float min_font_size = 7. * QMAX(1., KGlobalSettings::generalFont().pointSizeFloat() / 10.);

    const int expected_height = 38;
    description_font_size = QMAX( pointSize( expected_height * .3, listView() ), min_font_size );
    title_font_size = QMAX( pointSize( expected_height * .25, listView() ), min_font_size + 1 );

    //kdDebug() << description_font_size << " " << title_font_size << " " << pointSize( expected_height * .25, listView() ) << endl;
    QListViewItem::setup();
    setHeight( (int)QMAX( expected_height, pixelSize( title_font_size + description_font_size * 2.3, listView())));
}

int KMenuItem::width ( const QFontMetrics & , const QListView * lv, int  )
{
    return lv->viewport()->width();
}

void KMenuItem::paintCell(QPainter* p, const QColorGroup & cg, int column, int width, int align)
{
    ItemView *listview = static_cast<ItemView*>( listView() );
    int bottom = listView()->itemRect( this ).bottom();
    int diff = bottom - listView()->viewport()->height();

    KPixmap pm;
    pm.resize( width, height() );
    QPainter pp( &pm );
    paintCellInter( &pp, cg, column, width, align );
    pp.end();

    if ( diff > 0 && diff <= height() ) // cut off
    {
        pm.resize( width, height() - diff );
        KPixmapEffect::blend( pm, float( diff ) / height(),
                              cg.color( QColorGroup::Background ),
                              KPixmapEffect::VerticalGradient );
        p->drawPixmap( 0, 0, pm );
        if ( listview->m_lastOne != this )
        {
            listview->m_lastOne = this;
            listview->m_old_contentY = -1;
        }
    }
    else
    {
        p->drawPixmap( 0, 0, pm );
        if ( this == listview->m_lastOne ) {
            if ( bottom < 0 )
                listview->m_lastOne = static_cast<KMenuItem*>( itemAbove() );
            else
                listview->m_lastOne = static_cast<KMenuItem*>( itemBelow() );
            listview->m_old_contentY = -1;
            repaint();
        }
    }
}

void KMenuItem::makeGradient( KPixmap &off, const QColor &c )
{
    KPixmap blend;
    blend.resize( off.width() / 3, off.height() );
    bitBlt( &blend, 0, 0, &off, off.width() - blend.width(), 0, blend.width(), blend.height() );
    KPixmapEffect::blend( blend, 0.2, c, KPixmapEffect::HorizontalGradient );
    QPainter p( &off );
    p.drawPixmap( off.width() - blend.width(), 0, blend );
    p.end();
}

void KMenuItem::paintCellInter(QPainter* p, const QColorGroup & cg, int column, int width, int align)
{
    const BackgroundMode bgmode = listView()->viewport()->backgroundMode();
    const QColorGroup::ColorRole crole = QPalette::backgroundRoleFromMode( bgmode );
    QColor backg = cg.color( crole );

    if ( isSelected() )
        backg = cg.color( QColorGroup::Highlight );
    p->fillRect( 0, 0, width, height(), backg );

    QFontMetrics fm( p->fontMetrics() );

    int pixsize = 32;
    if ( height() < 36 )
        pixsize = 16;
    const int left_margin = 30;
    const int margin = 3;

//    p->drawText( 2, 2, left_margin - 2, height(), align, QString::number( childCount () ) );

    const QPixmap * pix = pixmap( column );

    if ( pix )
    {
        QPixmap pix32 = *pix;

        if ( pix->width() > pixsize )
        {
            QImage i = pix->convertToImage().smoothScale( pixsize, pixsize );
            pix32.convertFromImage( i );
        }
        p->drawPixmap( (pixsize - pix32.width()) / 2 + left_margin,
                       ( height() - pix32.height() ) / 2, pix32 );
    }

    if ( m_title.isEmpty() )
        return;

    int r = left_margin + pixsize + margin * 2;

    QFont f1 = p->font();
    f1.setPointSizeFloat( title_font_size );
    f1.setWeight( QFont::Normal ); // QFont::DemiBold == 63

    QFont f2 = p->font();
    f2.setPointSizeFloat( description_font_size );
    f2.setWeight( QFont::Light );

    int f1h = QFontMetrics( f1 ).height();
    int f2h = QFontMetrics( f2 ).height();

    const int text_margin = 2;
    int spacing = ( height() - f1h - f2h - text_margin ) / 2;
    if ( m_description.isEmpty() )
        spacing = ( height() - f1h ) / 2;

    int right_triangle_size = pixelSize( 7, listView() );

    int right_margin = listView()->verticalScrollBar()->width();
    if ( m_has_children )
        right_margin += right_triangle_size * 2;

    KPixmap off;
    QPainter pp;

    off.resize( width-text_margin-r-right_margin, height() );
    pp.begin( &off );
    pp.fillRect( 0, 0, off.width(), off.height(), backg );

    if (isSelected())
       pp.setPen( cg.color( QColorGroup::HighlightedText ) );
    else
       pp.setPen( cg.color( QColorGroup::Text ) );

    pp.setFont( f1 );
    pp.drawText( 0, 0, off.width(), off.height(), align, m_title );
    pp.end();
    if ( QFontMetrics( f1 ).width( m_title ) > off.width() )
    {
        makeGradient( off, backg );
        if ( !m_description.isEmpty() )
            setToolTip( m_title + "<br><br>" + m_description );
        else
            setToolTip( m_title );
    }
    p->drawPixmap( r, spacing, off );

    if ( !m_description.isEmpty() )
    {
        pp.begin( &off );
        pp.fillRect( 0, 0, off.width(), off.height(), backg );

        QColor myColor = cg.color( QColorGroup::Text ).light( 200 );
        if ( qGray( myColor.rgb() ) == 0 )
            myColor = QColor( 100, 100, 110 );
        pp.setPen( myColor );
        pp.setPen( isSelected() ? cg.color( QColorGroup::Mid ) : myColor );
        pp.setFont( f2 );
        pp.drawText( 0, 0, off.width(), off.height(), align, m_description );
        pp.end();
        if ( QFontMetrics( f2 ).width( m_description ) > off.width() )
        {
            makeGradient( off, backg );
            setToolTip( m_title + "<br><br>" + m_description );
        }
        p->drawPixmap( r, spacing + text_margin + f1h, off );
    }

    if ( m_has_children )
    {
        QImage i = right_triangle.convertToImage().smoothScale( right_triangle_size,
                                                                right_triangle_size );
        QPixmap tri;
        tri.convertFromImage( i );

        p->drawPixmap( listView()->width() -  right_margin, ( height() - f1h ) / 2, tri );
    }

    if ( m_old_width != width )
    {
        // the listview caches paint events
        m_old_width = width;
        repaint();
    }
}

// --------------------------------------------------------------------------

KMenuItemSeparator::KMenuItemSeparator(int nId, QListView* parent)
    : KMenuItem(nId, parent), lv(parent), cached_width( 0 )
{
    setEnabled(false);
    left_margin = 15;
}

void KMenuItemSeparator::setup()
{
    KMenuItem::setup();

    QFont f = QFont();
    QFontMetrics fm(f);
    f.setPointSize( 8 );
    if ( itemAbove() && !text( 0 ).isEmpty() )
        setHeight( (int)QMAX( 34., fm.height() * 1.4) );
    else
        setHeight( (int)QMAX( 26., fm.height() * 1.4 ) );
}

void KMenuItemSeparator::setLink( const QString &text, const QString &url )
{
    m_link_text = text;
    m_link_url = url;
    m_link_rect = QRect();
}

bool KMenuItemSeparator::hitsLink( const QPoint &pos )
{
    return m_link_rect.contains( pos );
}

void KMenuItemSeparator::preparePixmap( int width )
{
    if ( cached_width != width )
    {
        pixmap.load( locate("data", "kicker/pics/menu_separator.png" ) );
        QImage i = pixmap.convertToImage().smoothScale( width - 15 - left_margin, pixmap.height() );
        pixmap.convertFromImage( i );
        cached_width = width;
    }
}

void KMenuItemSeparator::paintCell(QPainter* p, const QColorGroup & cg, int column, int width, int align)
{
    preparePixmap(width);

    const int h = height();

    if (text(0).isEmpty()) {
      KMenuItem::paintCell(p, cg, column, width, align);
      p->drawPixmap( 15 , h/2, pixmap );
    }
    else {
      const BackgroundMode bgmode = lv->viewport()->backgroundMode();
      const QColorGroup::ColorRole crole = QPalette::backgroundRoleFromMode( bgmode );
      p->fillRect( 0, 0, width, h, cg.brush( crole ) );

      int margin = 0;
      if ( itemAbove() ) {
          p->drawPixmap( 15 , h/4, pixmap );
          margin = h / 4;
      }
      QFont f = listView()->font();
      f.setWeight( QFont::Normal );
      f.setPointSize( 8 );
      p->setFont( f );
      QColor myColor = cg.color( QColorGroup::Text ).light( 200 );
      if ( qGray( myColor.rgb() ) == 0 )
          myColor = QColor( 100, 100, 110 );
      p->setPen( myColor );
      int twidth = p->fontMetrics().width(text(0));
      int lwidth = 0;
      int swidth = 0;
      int fwidth = 0;

      if ( !m_link_text.isEmpty() )
      {
          swidth = p->fontMetrics().width( " (" );
          lwidth = p->fontMetrics().width(m_link_text );
          fwidth = p->fontMetrics().width( ")" );
      }
      int pos = int(lv->width() * 0.9 - twidth - swidth - lwidth - fwidth);
      p->drawText( pos, margin + 5,
                   width, h - ( margin +5 ), AlignTop, text(0) );
      if ( !m_link_text.isEmpty() )
      {
          pos += twidth;
          p->drawText( pos, margin + 5,
                       width, h - ( margin +5 ), AlignTop, " (" );
          pos += swidth;
          p->setPen( cg.color( QColorGroup::Link ) );
          f.setUnderline( true );
          p->setFont( f );
          p->drawText( pos, margin + 5,
                       width, h - ( margin +5 ), AlignTop, m_link_text );
          m_link_rect = QRect( pos, margin + 5, lwidth, p->fontMetrics().height() );
          pos += lwidth;
          f.setUnderline( false );
          p->setFont( f );
          p->drawText( pos, margin + 5,
                       width, h - ( margin +5 ), AlignTop, ")" );
      }
    }
}

KMenuItemHeader::KMenuItemHeader(int nId, const QString& relPath, QListView* parent)
    : KMenuItemSeparator(nId, parent)
{
    setEnabled( false );
    QString path;
    if (relPath.startsWith( "new/" /*"kicker:/new/"*/ )) {
       paths.append( "kicker:/goup/" );
       texts.append( i18n("New Applications") );
       icons.append( "clock" );
    }
    else if (relPath == "kicker:/restart/") {
       texts.append( i18n("Restart Computer") );
    }
    else if (relPath == "kicker:/switchuser/") {
       texts.append( i18n("Switch User") );
    }
    else {
      KServiceGroup::Ptr subMenuRoot = KServiceGroup::group(relPath);
      QStringList items = QStringList::split( '/', relPath );
      for ( QStringList::ConstIterator it = items.begin(); it != items.end(); ++it )
      {
        path += *it + "/";
        paths.append( "kicker:/goup/" + path );
        KServiceGroup::Ptr subMenuRoot = KServiceGroup::group(path);
        QString groupCaption = subMenuRoot->caption();
        texts.append( groupCaption );
        icons.append( subMenuRoot->icon() );
      }
    }

    setPath( "kicker:/goup/" + path ); // the last wins for now
    left_margin = 10;
}

void KMenuItemHeader::setup()
{
    KMenuItem::setup();

    QFontMetrics fm( listView()->font() );
    setHeight( QMAX( int( texts.count() * fm.height() + ( texts.count() + 1 ) * 2 + 10 ), height()) );
    // nada
}

void KMenuItemHeader::paintCell(QPainter* p, const QColorGroup & cg, int , int width, int align )
{
    preparePixmap(width);

    const BackgroundMode bgmode = listView()->viewport()->backgroundMode();
    const QColorGroup::ColorRole crole = QPalette::backgroundRoleFromMode( bgmode );

    QBrush br = cg.brush( crole );
    if ( isSelected() ) {
        br = cg.brush( QColorGroup::Highlight );
        p->fillRect( 0, 0, width, height() - 3, br );
    } else {
        p->fillRect( 0, 0, width, height(), br );
    }

    QFontMetrics fm( p->fontMetrics() );
    const int left_margin = 10;

    const int margin = 3;

    int r = left_margin + margin * 2;

    const int min_font_size = 7;
    int title_font_size = qRound( pixelSize( QMAX( pointSize( 12, listView() ), min_font_size + 1 ), listView() ) );

    QFont f1 = p->font();
    f1.setPixelSize( title_font_size );
    p->setFont( f1 );
    int f1h = QFontMetrics( f1 ).height();

    p->setPen( cg.color( QColorGroup::Text ) );

    const int text_margin = 2;
    int spacing = ( height() - texts.count() * f1h - QMAX( texts.count() - 1, 0 ) * text_margin ) / 2;

    for ( uint i = 0; i < texts.count(); ++i )
    {
        if (i==texts.count()-1) {
            f1.setWeight( QFont::DemiBold );
            p->setFont( f1 );
            f1h = QFontMetrics( f1 ).height();
        }

        p->drawText( r, spacing, width-text_margin-r, height(), align, texts[i] );
        spacing += text_margin + f1h;
        r += title_font_size;
    }

    p->drawPixmap( left_margin , height() - 2, pixmap );
}

KMenuSpacer::KMenuSpacer(int nId, QListView* parent)
    : KMenuItem(nId, parent)
{
    setEnabled(false);
}

void KMenuSpacer::setup()
{
    // nada
}

void KMenuSpacer::paintCell(QPainter* p, const QColorGroup & cg, int , int width, int )
{
    const BackgroundMode bgmode = listView()->viewport()->backgroundMode();
    const QColorGroup::ColorRole crole = QPalette::backgroundRoleFromMode( bgmode );
    QBrush br = cg.brush( crole );

    p->fillRect( 0, 0, width, height(), br );
}

void KMenuSpacer::setHeight( int i )
{
    KMenuItem::setHeight( i );
}

class ItemViewTip : public QToolTip
{
public:
    ItemViewTip( QWidget *parent, QListView *lv );

    void maybeTip( const QPoint &pos );

private:
    QListView *view;

};

ItemViewTip::ItemViewTip( QWidget *parent, QListView *lv )
    : QToolTip( parent ), view( lv )
{
}

void ItemViewTip::maybeTip( const QPoint &pos )
{
    KMenuItem *item = dynamic_cast<KMenuItem*>( view->itemAt( pos ) );
    QPoint contentsPos = view->viewportToContents( pos );
    if ( !item )
        return;

    if ( item->toolTip().isNull() )
        return;

    QRect r = view->itemRect( item );
    int headerPos = view->header()->sectionPos( 0 );
    r.setLeft( headerPos );
    r.setRight( headerPos + view->header()->sectionSize( 0 ) );
    tip( r, item->toolTip() );
}

// --------------------------------------------------------------------------

ItemView::ItemView(QWidget* parent, const char* name)
    : KListView(parent, name), m_spacer( 0 ),
      m_mouseMoveSelects(true), m_iconSize(32)
{
    setHScrollBarMode( QScrollView::AlwaysOff );
    setFrameStyle( QFrame::NoFrame );
    setSelectionMode(QListView::Single);
    addColumn("");
    header()->setStretchEnabled(1, 0);
    //setColumnWidthMode(0, QListView::Maximum);
    header()->hide();
    setMouseTracking(true);
    setItemMargin(0);
    setSorting(-1);
    setTreeStepSize(38);
    setFocusPolicy(QWidget::NoFocus);

    m_lastOne = 0;
    m_old_contentY = -1;

    connect(this, SIGNAL(mouseButtonClicked( int, QListViewItem*, const QPoint &, int )),
                  SLOT(slotItemClicked(int, QListViewItem*, const QPoint &, int)));

    connect(this, SIGNAL(returnPressed(QListViewItem*)), SLOT(slotItemClicked(QListViewItem*)));
    connect(this, SIGNAL(spacePressed(QListViewItem*)), SLOT(slotItemClicked(QListViewItem*)));

    new ItemViewTip( viewport(), this );
}

KMenuItemHeader *ItemView::insertHeader(int id, const QString &relpath)
{
    KMenuItemHeader *newItem = new KMenuItemHeader(id, relpath, this );
    moveItemToIndex(newItem, 1);
    setBackPath( "kicker:/goup/" + relpath ); // the last wins for now

    return newItem;
}

KMenuItem* ItemView::findItem(int nId)
{
    for (QListViewItemIterator it(this); it.current(); ++it)
    {
	if(static_cast<KMenuItem*>(it.current())->id() == nId)
	    return static_cast<KMenuItem*>(it.current());
    }

    return 0L;
}

bool ItemView::focusNextPrevChild(bool /*next*/)
{
    return false;
}

KMenuItem* ItemView::itemAtIndex(int nIndex)
{
    if(nIndex <= 0)
	return 0L;

    if(nIndex >= childCount())
      return static_cast<KMenuItem*>(lastItem());

    int i = 1;
    QListViewItemIterator it(this);
    for (;it.current(); ++i, ++it) {
	if(i == nIndex)
	    return static_cast<KMenuItem*>(it.current());
    }

    return static_cast<KMenuItem*>(lastItem());
}

KMenuItem* ItemView::insertItem( const QString& icon, const QString& text, const QString& description, const
                                 QString& path, int nId, int nIndex, KMenuItem *parent)
{
    KMenuItem* newItem = findItem(nId);

    if(!newItem && parent)
        newItem = new KMenuItem(nId, parent );
    else if ( !newItem )
	newItem = new KMenuItem(nId, this );

    newItem->setIcon(icon, m_iconSize);
    newItem->setTitle(text);
    newItem->setDescription(description);
    newItem->setPath(path);

    if (nIndex==-1)
      nIndex=childCount();

    moveItemToIndex(newItem, nIndex);

    return newItem;
}

KMenuItem* ItemView::insertItem( const QString& icon, const QString& text, const QString& description,
                                 int nId, int nIndex, KMenuItem *parent)
{
   return insertItem( icon, text, description, QString::null, nId, nIndex, parent);
}

int ItemView::setItemEnabled(int id, bool enabled)
{
    KMenuItem* item = findItem(id);

    if(item)
	item->setEnabled(enabled);

    return 0;
}

KMenuItemSeparator *ItemView::insertSeparator(int nId, const QString& text, int nIndex)
{
    KMenuItemSeparator *newItem = new KMenuItemSeparator(nId, this);

    newItem->setText(0, text);

    if (nIndex==-1)
      nIndex=childCount();

    moveItemToIndex(newItem, nIndex);
    return newItem;
}

void ItemView::moveItemToIndex(KMenuItem* item, int nIndex)
{

    if (nIndex <= 0) {
          takeItem(item);
          KListView::insertItem(item);
    }
    else {
        item->moveItem(itemAtIndex(nIndex));
    }
}

void ItemView::slotMoveContent()
{
    if ( !m_spacer )
        return;

    int item_height = 0;
    QListViewItemIterator it( this );
    while ( it.current() ) {
        if ( !dynamic_cast<KMenuSpacer*>( it.current() ) && !it.current()->parent() && it.current()->isVisible() )  {
            it.current()->invalidateHeight();
            item_height += it.current()->totalHeight();
        }
        ++it;
    }

    if ( height() > item_height )
        m_spacer->setHeight( height() - item_height );
    else
        m_spacer->setHeight( 0 );
}

KMenuItem *ItemView::insertMenuItem(KService::Ptr& s, int nId, int nIndex, KMenuItem* parentItem,
                                    const QString& aliasname, const QString & label, const QString & categoryIcon )
{
    if (!s)
	return 0;

    QString serviceName = aliasname.isEmpty() ? s->name() : aliasname;

    kdDebug() << "insertMenuItem " << nId << " " << nIndex << " " << s->name() << endl;
    KMenuItem* newItem = 0; //findItem(nId);
    if(!newItem)
	newItem = parentItem ? new KMenuItem(nId, parentItem) : new KMenuItem(nId, this);

    newItem->setIcon(s->icon()=="unknown" ? categoryIcon : s->icon(), m_iconSize);
    if ((KickerSettings::DescriptionAndName || KickerSettings::menuEntryFormat()
            == KickerSettings::DescriptionOnly) && !s->genericName().isEmpty()) {
      newItem->setTitle(s->genericName());
      newItem->setDescription(label.isEmpty() ? serviceName : label);
    }
    else {
      newItem->setTitle(label.isEmpty() ? serviceName : label);
      newItem->setDescription(s->genericName());
    }
    newItem->setService(s);

    if (nIndex==-2)
      return newItem;

    if (nIndex==-1)
      nIndex=childCount();

    moveItemToIndex(newItem, nIndex);

    return newItem;
}

KMenuItem* ItemView::insertDocumentItem(const QString& s, int nId, int nIndex, const QStringList* /*suppressGenericNames*/,
                                        const QString& /*aliasname*/)
{
    KMenuItem* newItem = findItem(nId);

    if(!newItem)
	newItem = new KMenuItem(nId, this);

    KMimeType::Ptr mt = KMimeType::findByURL( s );
    newItem->setIcon(KMimeType::iconForURL( s ), m_iconSize);
    newItem->setTitle(s);
    newItem->setDescription(mt->comment());
    newItem->setPath(s);

    if (nIndex==-1)
      nIndex=childCount();

    moveItemToIndex(newItem, nIndex);

    return newItem;
}

KMenuItem* ItemView::insertRecentlyItem(const QString& s, int nId, int nIndex)
{
    KDesktopFile f(s, true /* read only */);

    KMenuItem* newItem = findItem(nId);

    if(!newItem)
	newItem = new KMenuItem(nId, this);

    newItem->setIcon(f.readIcon(), m_iconSize);

    // work around upstream fixed bug
    QString name=f.readName();
    if (name.isEmpty())
      name=f.readURL();

    newItem->setTitle(name);

    QString comment = f.readComment();
    if (comment.isEmpty()) {
      KURL url(f.readURL());
      if (!url.host().isEmpty())
        comment = i18n("Host: %1").arg(url.host());
    }

    newItem->setDescription(comment);
    newItem->setPath(s);

    if (nIndex==-1)
      nIndex=childCount();

    moveItemToIndex(newItem, nIndex);

    return newItem;
}

int ItemView::insertItem(PopupMenuTitle*, int, int)
{
    return 0;
}

KMenuItem* ItemView::insertSubItem(const QString& icon, const QString& caption, const QString& description, const QString& path, KMenuItem* parentItem)
{
#warning FIXME
    KMenuItem* newItem = parentItem ? new KMenuItem(-1, parentItem) : new KMenuItem(-1, this);
    newItem->setTitle(caption);
    newItem->setDescription(description);
    newItem->setIcon(icon, m_iconSize);
    newItem->setPath(path);

    return newItem;
}



void ItemView::slotItemClicked(int button, QListViewItem * item, const QPoint & /*pos*/, int /*c*/ )
{
    if (button==1)
      slotItemClicked(item);
}

void ItemView::slotItemClicked(QListViewItem* item)
{
    KMenuItem* kitem = dynamic_cast<KMenuItem*>(item);
    if ( !kitem )
        return;

    if(kitem->service()) {
        emit startService(kitem->service());
    }
    else if(!kitem->path().isEmpty()) {
        emit startURL(kitem->path());
    }
}

void ItemView::contentsMousePressEvent ( QMouseEvent * e )
{
    KListView::contentsMousePressEvent( e );

    QPoint vp = contentsToViewport(e->pos());
    KMenuItemSeparator *si = dynamic_cast<KMenuItemSeparator*>( itemAt( vp ) );
    if ( si )
    {
        if ( si->hitsLink( vp - itemRect(si).topLeft() ) )
            emit startURL( si->linkUrl() );
    }
}

void ItemView::contentsMouseMoveEvent(QMouseEvent *e)
{
    KListView::contentsMouseMoveEvent(e);

    QPoint vp = contentsToViewport(e->pos());
    QListViewItem * i = itemAt( vp );

    if (m_mouseMoveSelects) {
      if(i && i->isEnabled() && !i->isSelected() &&
         // FIXME: This is wrong if you drag over the items.
         (e->state() & (LeftButton|MidButton|RightButton)) == 0)
          KListView::setSelected(i, true);
      else if (!i && selectedItem())
          KListView::setSelected(selectedItem(), false);
    }

    bool link_cursor = false;
    KMenuItemSeparator *si = dynamic_cast<KMenuItemSeparator*>( i );
    if ( si )
    {
        link_cursor = si->hitsLink( vp - itemRect(si).topLeft() );
    }

    if ( link_cursor )
        setCursor( Qt::PointingHandCursor );
    else
        unsetCursor();

}

void ItemView::leaveEvent(QEvent* e)
{
    KListView::leaveEvent(e);

    clearSelection();
}

void ItemView::resizeEvent ( QResizeEvent * e )
{
    KListView::resizeEvent( e );
//    if ( m_lastOne )
//        int diff = itemRect( m_lastOne ).bottom() - viewport()->height();
}

void ItemView::viewportPaintEvent ( QPaintEvent * pe )
{
    //kdDebug() << "viewportPaintEvent " << pe->rect() << " " << contentsY () << " " << m_old_contentY << endl;
    KListView::viewportPaintEvent( pe );

    if ( m_lastOne && m_old_contentY != contentsY() ) {
        m_old_contentY = contentsY();
        m_lastOne->repaint();
    }
}

void ItemView::clear()
{
    KListView::clear();
    m_lastOne = 0;
    m_old_contentY = -1;
    m_back_url = QString::null;
}

void ItemView::contentsWheelEvent(QWheelEvent *e)
{
    KListView::contentsWheelEvent(e);

    QPoint vp = contentsToViewport(e->pos());
    QListViewItem * i = itemAt( vp );

    if(i && i->isEnabled() && !i->isSelected() &&
       // FIXME: This is wrong if you drag over the items.
       (e->state() & (LeftButton|MidButton|RightButton)) == 0)
        KListView::setSelected(i, true);
    else if (!i && selectedItem())
        KListView::setSelected(selectedItem(), false);
}

QDragObject * ItemView::dragObject()
{
    KMultipleDrag* o = 0;
    QListViewItem *item = itemAt( viewport()->mapFromGlobal(QCursor::pos()) );
    if ( item ) {
      KMenuItem* kitem = static_cast<KMenuItem*>(item);

      if (dynamic_cast<KMenuItemHeader*>(item))
        return 0;

      o = new KMultipleDrag(viewport());
      QPixmap pix = KGlobal::iconLoader()->loadIcon( kitem->icon(), KIcon::Panel, m_iconSize);
      QPixmap add = KGlobal::iconLoader()->loadIcon( "add", KIcon::Small );

      QPainter p( &pix );
      p.drawPixmap(pix.height()-add.height(), pix.width()-add.width(), add);
      p.end();

      QBitmap mask;

      if (pix.mask())
          mask = *pix.mask();
      else {
	  mask.resize(pix.size());
	  mask.fill(Qt::color1);
      }

      bitBlt( &mask, pix.width()-add.width(), pix.height()-add.height(), add.mask(), 0, 0, add.width(), add.height(), OrROP );
      pix.setMask( mask );
      o->setPixmap(pix);

      if(kitem->service()) {
        // If the path to the desktop file is relative, try to get the full
        // path from KStdDirs.
        QString path = kitem->service()->desktopEntryPath();
        path = locate("apps", path);
        o->addDragObject(new KURLDrag(KURL::List(KURL(path)), 0));
      }
      else if (kitem->path().startsWith("kicker:/new") || kitem->path().startsWith("system:/")) {
        delete o;
        return 0;
      }
      else if (kitem->hasChildren()) {
         o->addDragObject(new KURLDrag(KURL::List(KURL("programs:/"+kitem->menuPath())), 0));
         return o;
      }
      else if(!kitem->path().isEmpty() && !kitem->path().startsWith("kicker:/") && !kitem->path().startsWith("kaddressbook --uid")) {
         QString uri = kitem->path();

         if (uri.startsWith(locateLocal("data", QString::fromLatin1("RecentDocuments/")))) {
             KDesktopFile df(uri,true);
             uri=df.readURL();
         }

         o->addDragObject(new KURLDrag(KURL::List(KURL(uri)), 0));
      }

      o->addDragObject(new KMenuItemDrag(*kitem,this));
    }
    return o;
}

int ItemView::goodHeight()
{
    int item_height = 0;
    QListViewItemIterator it( this );
    while ( it.current() ) {
        if ( !dynamic_cast<KMenuSpacer*>( it.current() ) && !it.current()->parent() && it.current()->isVisible() )  {
            item_height += it.current()->height();
        }
        ++it;
    }

    return item_height;
}


KMenuItemDrag::KMenuItemDrag(KMenuItem& item, QWidget *dragSource)
    : QDragObject(dragSource, 0)
{
    QBuffer buff(a);
    buff.open(IO_WriteOnly);
    QDataStream s(&buff);

    s << item.id() << (item.service() ? item.service()->storageId() : QString::null)
      << item.title() << item.description() << item.icon() << item.path();
}

KMenuItemDrag::~KMenuItemDrag()
{
}

const char * KMenuItemDrag::format(int i) const
{
    if (i == 0)
        return "application/kmenuitem";

    return 0;
}

QByteArray KMenuItemDrag::encodedData(const char* mimeType) const
{
    if (QString("application/kmenuitem") == mimeType)
        return a;

    return QByteArray();
}

bool KMenuItemDrag::canDecode(const QMimeSource * e)
{
    if (e->provides( "application/kmenuitem" ) )
        return true;

    return false;
}

bool ItemView::acceptDrag (QDropEvent* event) const
{
    if ( !acceptDrops() )
        return false;

    if (KMenuItemDrag::canDecode(event))
        return true;

    if (QTextDrag::canDecode(event)) {
        QString text;
        QTextDrag::decode(event,text);
        return !text.startsWith("programs:/");
    }

    return itemsMovable();
}

bool KMenuItemDrag::decode(const QMimeSource* e, KMenuItemInfo& item)
{
    QByteArray a = e->encodedData("application/kmenuitem");

    if (a.isEmpty()) {
        QStringList l;
        bool ret = QUriDrag::decodeToUnicodeUris( e, l );
        if ( ret )
        {
            for ( QStringList::ConstIterator it = l.begin(); it != l.end(); ++it )
            {
                QString url = *it;
                kdDebug () << "Url " << url << endl;
                item.m_path = KURL( url ).path();
                if ( KDesktopFile::isDesktopFile( item.m_path ) )
                {
                    KDesktopFile df( item.m_path, true );
                    item.m_description = df.readGenericName();
                    item.m_icon = df.readIcon();
                    item.m_title = df.readName();
                }
                else
                {
                    item.m_title = item.m_path;
                    item.m_icon = KMimeType::iconForURL( url );
                    item.m_title = item.m_path.section( '/', -1, -1 );
                    int last_slash = url.findRev ('/', -1);
                    if (last_slash == 0)
                        item.m_description = i18n("Directory: /)");
                    else
                        item.m_description = i18n("Directory: ") + url.section ('/', -2, -2);
                }

                return true;
            }
        }
        return false;
    }

    QBuffer buff(a);
    buff.open(IO_ReadOnly);
    QDataStream s(&buff);

    KMenuItemInfo i;
    QString storageId;
    s >> i.m_id >> storageId >> i.m_title >> i.m_description >> i.m_icon >> i.m_path;

    i.m_s = storageId.isEmpty() ? 0 : KService::serviceByStorageId(storageId);
    item = i;

    return true;
}

FavoritesItemView::FavoritesItemView(QWidget* parent, const char* name)
    : ItemView(parent, name)
{
}

bool FavoritesItemView::acceptDrag (QDropEvent* event) const
{
    if (event->source()==this->viewport())
        return true;

    if (KMenuItemDrag::canDecode(event)) {
        KMenuItemInfo item;
        KMenuItemDrag::decode(event,item);
        QStringList favs = KickerSettings::favorites();

        if (item.m_s)
            return favs.find(item.m_s->storageId())==favs.end();
        else {
            QStringList::Iterator it;

            QString uri = item.m_path;

            if (uri.startsWith(locateLocal("data", QString::fromLatin1("RecentDocuments/")))) {
               KDesktopFile df(uri,true);
               uri=df.readURL();
            }

            for (it = favs.begin(); it != favs.end(); ++it) {
                if ((*it)[0]=='/') {
                    KDesktopFile df((*it),true);
                    if (df.readURL().replace("file://",QString::null)==uri)
                        break;
                }
            }
            return it==favs.end();
        }
    }

    if (QTextDrag::canDecode(event)) {
        QString text;
        QTextDrag::decode(event,text);
        QStringList favs = KickerSettings::favorites();

        if (text.endsWith(".desktop")) {
            KService::Ptr p = KService::serviceByDesktopPath(text.replace("file://",QString::null));
            return (p && favs.find(p->storageId())==favs.end());
        }
        else {
            QStringList::Iterator it;
            for (it = favs.begin(); it != favs.end(); ++it) {
                if ((*it)[0]=='/') {
                    KDesktopFile df((*it),true);
                    if (df.readURL().replace("file://",QString::null)==text)
                        break;
                }
            }
            return it==favs.end();
        }
    }

    return itemsMovable();
}

#include "itemview.moc"

// vim:cindent:sw=4:
