/***************************************************************************
 *   Copyright (C) 1998-2013 by authors (see AUTHORS.txt)                  *
 *                                                                         *
 *   This file is part of LuxRender.                                       *
 *                                                                         *
 *   Lux Renderer is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 3 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   Lux Renderer is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>. *
 *                                                                         *
 *   This project is based on PBRT ; see http://www.pbrt.org               *
 *   Lux Renderer website : http://www.luxrender.net                       *
 ***************************************************************************/

#ifndef _LUXRAYS_RAY_H
#define _LUXRAYS_RAY_H

#include "luxrays/core/epsilon.h"
#include "luxrays/core/geometry/vector.h"
#include "luxrays/core/geometry/point.h"

#include <boost/math/special_functions/sign.hpp>

namespace luxrays {

// OpenCL data types
namespace ocl {
#include "luxrays/core/geometry/ray_types.cl"
}

class  Ray {
public:
	// Ray Public Methods
	Ray() : maxt(std::numeric_limits<float>::infinity()), time(0.f) {
		mint = MachineEpsilon::E(1.f);
	}

	Ray(const Point &origin, const Vector &direction) : o(origin),
		d(direction), maxt(std::numeric_limits<float>::infinity()),
		time(0.f) {
		mint = MachineEpsilon::E(origin);
	}

	Ray(const Point &origin, const Vector &direction,
		float start, float end = std::numeric_limits<float>::infinity(),
		float t = 0.f)
		: o(origin), d(direction), mint(start), maxt(end), time(t) { }

	Point operator()(float t) const { return o + d * t; }
	void GetDirectionSigns(int signs[3]) const {
		signs[0] = boost::math::signbit(d.x);
		signs[1] = boost::math::signbit(d.y);
		signs[2] = boost::math::signbit(d.z);
	}

	// Ray Public Data
	Point o;
	Vector d;
	mutable float mint, maxt;
	float time;
	float pad[3]; // Add padding to avoid size discrepancies with OpenCL
};

inline std::ostream &operator<<(std::ostream &os, const Ray &r) {
	os << "Ray[" << r.o << ", " << r.d << ", " << r.mint << "," << r.maxt << "]";
	return os;
}

class RayHit {
public:
	float t;
	float b1, b2; // Barycentric coordinates of the hit point
	unsigned int meshIndex, triangleIndex;

	void SetMiss() { meshIndex = 0xffffffffu; };
	bool Miss() const { return (meshIndex == 0xffffffffu); };
};

}

#endif	/* _LUXRAYS_RAY_H */
