/***************************************************************************
 *   Copyright (C) 1998-2013 by authors (see AUTHORS.txt)                  *
 *                                                                         *
 *   This file is part of LuxRays.                                         *
 *                                                                         *
 *   LuxRays is free software; you can redistribute it and/or modify       *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 3 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   LuxRays is distributed in the hope that it will be useful,            *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>. *
 *                                                                         *
 *   LuxRays website: http://www.luxrender.net                             *
 ***************************************************************************/

#include "slg/sampler/sampler.h"

using namespace luxrays;
using namespace slg;

/*
 * Sobol sequence direction vectors.
 *
 * This file contains code to create direction vectors for generating sobol
 * sequences in high dimensions. It is adapted from code on this webpage:
 *
 * http://web.maths.unsw.edu.au/~fkuo/sobol/
 *
 * From these papers:
 * 
 * S. Joe and F. Y. Kuo, Remark on Algorithm 659: Implementing Sobol's quasirandom
 * sequence generator, ACM Trans. Math. Softw. 29, 49-57 (2003)
 *
 * S. Joe and F. Y. Kuo, Constructing Sobol sequences with better two-dimensional
 * projections, SIAM J. Sci. Comput. 30, 2635-2654 (2008)
 *
 */

/* Copyright (c) 2008, Frances Y. Kuo and Stephen Joe
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 * 
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 * 
 *     * Neither the names of the copyright holders nor the names of the
 *       University of New South Wales and the University of Waikato
 *       and its contributors may be used to endorse or promote products derived
 *       from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#define SOBOL_MAX_NUMBER 32

typedef struct {
	u_int d, s, a;
	u_int m[SOBOL_MAX_NUMBER];
} SobolDirectionNumbers;

static SobolDirectionNumbers SOBOL_NUMBERS[SOBOL_MAX_DIMENSIONS - 1] = {
	{2, 1, 0, {1}},
	{3, 2, 1, {1, 3}},
	{4, 3, 1, {1, 3, 1}},
	{5, 3, 2, {1, 1, 1}},
	{6, 4, 1, {1, 1, 3, 3}},
	{7, 4, 4, {1, 3, 5, 13}},
	{8, 5, 2, {1, 1, 5, 5, 17}},
	{9, 5, 4, {1, 1, 5, 5, 5}},
	{10, 5, 7, {1, 1, 7, 11, 19}},
	{11, 5, 11, {1, 1, 5, 1, 1}},
	{12, 5, 13, {1, 1, 1, 3, 11}},
	{13, 5, 14, {1, 3, 5, 5, 31}},
	{14, 6, 1, {1, 3, 3, 9, 7, 49}},
	{15, 6, 13, {1, 1, 1, 15, 21, 21}},
	{16, 6, 16, {1, 3, 1, 13, 27, 49}},
	{17, 6, 19, {1, 1, 1, 15, 7, 5}},
	{18, 6, 22, {1, 3, 1, 15, 13, 25}},
	{19, 6, 25, {1, 1, 5, 5, 19, 61}},
	{20, 7, 1, {1, 3, 7, 11, 23, 15, 103}},
	{21, 7, 4, {1, 3, 7, 13, 13, 15, 69}},
	{22, 7, 7, {1, 1, 3, 13, 7, 35, 63}},
	{23, 7, 8, {1, 3, 5, 9, 1, 25, 53}},
	{24, 7, 14, {1, 3, 1, 13, 9, 35, 107}},
	{25, 7, 19, {1, 3, 1, 5, 27, 61, 31}},
	{26, 7, 21, {1, 1, 5, 11, 19, 41, 61}},
	{27, 7, 28, {1, 3, 5, 3, 3, 13, 69}},
	{28, 7, 31, {1, 1, 7, 13, 1, 19, 1}},
	{29, 7, 32, {1, 3, 7, 5, 13, 19, 59}},
	{30, 7, 37, {1, 1, 3, 9, 25, 29, 41}},
	{31, 7, 41, {1, 3, 5, 13, 23, 1, 55}},
	{32, 7, 42, {1, 3, 7, 3, 13, 59, 17}},
	{33, 7, 50, {1, 3, 1, 3, 5, 53, 69}},
	{34, 7, 55, {1, 1, 5, 5, 23, 33, 13}},
	{35, 7, 56, {1, 1, 7, 7, 1, 61, 123}},
	{36, 7, 59, {1, 1, 7, 9, 13, 61, 49}},
	{37, 7, 62, {1, 3, 3, 5, 3, 55, 33}},
	{38, 8, 14, {1, 3, 1, 15, 31, 13, 49, 245}},
	{39, 8, 21, {1, 3, 5, 15, 31, 59, 63, 97}},
	{40, 8, 22, {1, 3, 1, 11, 11, 11, 77, 249}},
	{41, 8, 38, {1, 3, 1, 11, 27, 43, 71, 9}},
	{42, 8, 47, {1, 1, 7, 15, 21, 11, 81, 45}},
	{43, 8, 49, {1, 3, 7, 3, 25, 31, 65, 79}},
	{44, 8, 50, {1, 3, 1, 1, 19, 11, 3, 205}},
	{45, 8, 52, {1, 1, 5, 9, 19, 21, 29, 157}},
	{46, 8, 56, {1, 3, 7, 11, 1, 33, 89, 185}},
	{47, 8, 67, {1, 3, 3, 3, 15, 9, 79, 71}},
	{48, 8, 70, {1, 3, 7, 11, 15, 39, 119, 27}},
	{49, 8, 84, {1, 1, 3, 1, 11, 31, 97, 225}},
	{50, 8, 97, {1, 1, 1, 3, 23, 43, 57, 177}},
	{51, 8, 103, {1, 3, 7, 7, 17, 17, 37, 71}},
	{52, 8, 115, {1, 3, 1, 5, 27, 63, 123, 213}},
	{53, 8, 122, {1, 1, 3, 5, 11, 43, 53, 133}},
	{54, 9, 8, {1, 3, 5, 5, 29, 17, 47, 173, 479}},
	{55, 9, 13, {1, 3, 3, 11, 3, 1, 109, 9, 69}},
	{56, 9, 16, {1, 1, 1, 5, 17, 39, 23, 5, 343}},
	{57, 9, 22, {1, 3, 1, 5, 25, 15, 31, 103, 499}},
	{58, 9, 25, {1, 1, 1, 11, 11, 17, 63, 105, 183}},
	{59, 9, 44, {1, 1, 5, 11, 9, 29, 97, 231, 363}},
	{60, 9, 47, {1, 1, 5, 15, 19, 45, 41, 7, 383}},
	{61, 9, 52, {1, 3, 7, 7, 31, 19, 83, 137, 221}},
	{62, 9, 55, {1, 1, 1, 3, 23, 15, 111, 223, 83}},
	{63, 9, 59, {1, 1, 5, 13, 31, 15, 55, 25, 161}},
	{64, 9, 62, {1, 1, 3, 13, 25, 47, 39, 87, 257}},
	{65, 9, 67, {1, 1, 1, 11, 21, 53, 125, 249, 293}},
	{66, 9, 74, {1, 1, 7, 11, 11, 7, 57, 79, 323}},
	{67, 9, 81, {1, 1, 5, 5, 17, 13, 81, 3, 131}},
	{68, 9, 82, {1, 1, 7, 13, 23, 7, 65, 251, 475}},
	{69, 9, 87, {1, 3, 5, 1, 9, 43, 3, 149, 11}},
	{70, 9, 91, {1, 1, 3, 13, 31, 13, 13, 255, 487}},
	{71, 9, 94, {1, 3, 3, 1, 5, 63, 89, 91, 127}},
	{72, 9, 103, {1, 1, 3, 3, 1, 19, 123, 127, 237}},
	{73, 9, 104, {1, 1, 5, 7, 23, 31, 37, 243, 289}},
	{74, 9, 109, {1, 1, 5, 11, 17, 53, 117, 183, 491}},
	{75, 9, 122, {1, 1, 1, 5, 1, 13, 13, 209, 345}},
	{76, 9, 124, {1, 1, 3, 15, 1, 57, 115, 7, 33}},
	{77, 9, 137, {1, 3, 1, 11, 7, 43, 81, 207, 175}},
	{78, 9, 138, {1, 3, 1, 1, 15, 27, 63, 255, 49}},
	{79, 9, 143, {1, 3, 5, 3, 27, 61, 105, 171, 305}},
	{80, 9, 145, {1, 1, 5, 3, 1, 3, 57, 249, 149}},
	{81, 9, 152, {1, 1, 3, 5, 5, 57, 15, 13, 159}},
	{82, 9, 157, {1, 1, 1, 11, 7, 11, 105, 141, 225}},
	{83, 9, 167, {1, 3, 3, 5, 27, 59, 121, 101, 271}},
	{84, 9, 173, {1, 3, 5, 9, 11, 49, 51, 59, 115}},
	{85, 9, 176, {1, 1, 7, 1, 23, 45, 125, 71, 419}},
	{86, 9, 181, {1, 1, 3, 5, 23, 5, 105, 109, 75}},
	{87, 9, 182, {1, 1, 7, 15, 7, 11, 67, 121, 453}},
	{88, 9, 185, {1, 3, 7, 3, 9, 13, 31, 27, 449}},
	{89, 9, 191, {1, 3, 1, 15, 19, 39, 39, 89, 15}},
	{90, 9, 194, {1, 1, 1, 1, 1, 33, 73, 145, 379}},
	{91, 9, 199, {1, 3, 1, 15, 15, 43, 29, 13, 483}},
	{92, 9, 218, {1, 1, 7, 3, 19, 27, 85, 131, 431}},
	{93, 9, 220, {1, 3, 3, 3, 5, 35, 23, 195, 349}},
	{94, 9, 227, {1, 3, 3, 7, 9, 27, 39, 59, 297}},
	{95, 9, 229, {1, 1, 3, 9, 11, 17, 13, 241, 157}},
	{96, 9, 230, {1, 3, 7, 15, 25, 57, 33, 189, 213}},
	{97, 9, 234, {1, 1, 7, 1, 9, 55, 73, 83, 217}},
	{98, 9, 236, {1, 3, 3, 13, 19, 27, 23, 113, 249}},
	{99, 9, 241, {1, 3, 5, 3, 23, 43, 3, 253, 479}},
	{100, 9, 244, {1, 1, 5, 5, 11, 5, 45, 117, 217}},
	{101, 9, 253, {1, 3, 3, 7, 29, 37, 33, 123, 147}},
	{102, 10, 4, {1, 3, 1, 15, 5, 5, 37, 227, 223, 459}},
	{103, 10, 13, {1, 1, 7, 5, 5, 39, 63, 255, 135, 487}},
	{104, 10, 19, {1, 3, 1, 7, 9, 7, 87, 249, 217, 599}},
	{105, 10, 22, {1, 1, 3, 13, 9, 47, 7, 225, 363, 247}},
	{106, 10, 50, {1, 3, 7, 13, 19, 13, 9, 67, 9, 737}},
	{107, 10, 55, {1, 3, 5, 5, 19, 59, 7, 41, 319, 677}},
	{108, 10, 64, {1, 1, 5, 3, 31, 63, 15, 43, 207, 789}},
	{109, 10, 69, {1, 1, 7, 9, 13, 39, 3, 47, 497, 169}},
	{110, 10, 98, {1, 3, 1, 7, 21, 17, 97, 19, 415, 905}},
	{111, 10, 107, {1, 3, 7, 1, 3, 31, 71, 111, 165, 127}},
	{112, 10, 115, {1, 1, 5, 11, 1, 61, 83, 119, 203, 847}},
	{113, 10, 121, {1, 3, 3, 13, 9, 61, 19, 97, 47, 35}},
	{114, 10, 127, {1, 1, 7, 7, 15, 29, 63, 95, 417, 469}},
	{115, 10, 134, {1, 3, 1, 9, 25, 9, 71, 57, 213, 385}},
	{116, 10, 140, {1, 3, 5, 13, 31, 47, 101, 57, 39, 341}},
	{117, 10, 145, {1, 1, 3, 3, 31, 57, 125, 173, 365, 551}},
	{118, 10, 152, {1, 3, 7, 1, 13, 57, 67, 157, 451, 707}},
	{119, 10, 158, {1, 1, 1, 7, 21, 13, 105, 89, 429, 965}},
	{120, 10, 161, {1, 1, 5, 9, 17, 51, 45, 119, 157, 141}},
	{121, 10, 171, {1, 3, 7, 7, 13, 45, 91, 9, 129, 741}},
	{122, 10, 181, {1, 3, 7, 1, 23, 57, 67, 141, 151, 571}},
	{123, 10, 194, {1, 1, 3, 11, 17, 47, 93, 107, 375, 157}},
	{124, 10, 199, {1, 3, 3, 5, 11, 21, 43, 51, 169, 915}},
	{125, 10, 203, {1, 1, 5, 3, 15, 55, 101, 67, 455, 625}},
	{126, 10, 208, {1, 3, 5, 9, 1, 23, 29, 47, 345, 595}},
	{127, 10, 227, {1, 3, 7, 7, 5, 49, 29, 155, 323, 589}},
	{128, 10, 242, {1, 3, 3, 7, 5, 41, 127, 61, 261, 717}},
	{129, 10, 251, {1, 3, 7, 7, 17, 23, 117, 67, 129, 1009}},
	{130, 10, 253, {1, 1, 3, 13, 11, 39, 21, 207, 123, 305}},
	{131, 10, 265, {1, 1, 3, 9, 29, 3, 95, 47, 231, 73}},
	{132, 10, 266, {1, 3, 1, 9, 1, 29, 117, 21, 441, 259}},
	{133, 10, 274, {1, 3, 1, 13, 21, 39, 125, 211, 439, 723}},
	{134, 10, 283, {1, 1, 7, 3, 17, 63, 115, 89, 49, 773}},
	{135, 10, 289, {1, 3, 7, 13, 11, 33, 101, 107, 63, 73}},
	{136, 10, 295, {1, 1, 5, 5, 13, 57, 63, 135, 437, 177}},
	{137, 10, 301, {1, 1, 3, 7, 27, 63, 93, 47, 417, 483}},
	{138, 10, 316, {1, 1, 3, 1, 23, 29, 1, 191, 49, 23}},
	{139, 10, 319, {1, 1, 3, 15, 25, 55, 9, 101, 219, 607}},
	{140, 10, 324, {1, 3, 1, 7, 7, 19, 51, 251, 393, 307}},
	{141, 10, 346, {1, 3, 3, 3, 25, 55, 17, 75, 337, 3}},
	{142, 10, 352, {1, 1, 1, 13, 25, 17, 65, 45, 479, 413}},
	{143, 10, 361, {1, 1, 7, 7, 27, 49, 99, 161, 213, 727}},
	{144, 10, 367, {1, 3, 5, 1, 23, 5, 43, 41, 251, 857}},
	{145, 10, 382, {1, 3, 3, 7, 11, 61, 39, 87, 383, 835}},
	{146, 10, 395, {1, 1, 3, 15, 13, 7, 29, 7, 505, 923}},
	{147, 10, 398, {1, 3, 7, 1, 5, 31, 47, 157, 445, 501}},
	{148, 10, 400, {1, 1, 3, 7, 1, 43, 9, 147, 115, 605}},
	{149, 10, 412, {1, 3, 3, 13, 5, 1, 119, 211, 455, 1001}},
	{150, 10, 419, {1, 1, 3, 5, 13, 19, 3, 243, 75, 843}},
	{151, 10, 422, {1, 3, 7, 7, 1, 19, 91, 249, 357, 589}},
	{152, 10, 426, {1, 1, 1, 9, 1, 25, 109, 197, 279, 411}},
	{153, 10, 428, {1, 3, 1, 15, 23, 57, 59, 135, 191, 75}},
	{154, 10, 433, {1, 1, 5, 15, 29, 21, 39, 253, 383, 349}},
	{155, 10, 446, {1, 3, 3, 5, 19, 45, 61, 151, 199, 981}},
	{156, 10, 454, {1, 3, 5, 13, 9, 61, 107, 141, 141, 1}},
	{157, 10, 457, {1, 3, 1, 11, 27, 25, 85, 105, 309, 979}},
	{158, 10, 472, {1, 3, 3, 11, 19, 7, 115, 223, 349, 43}},
	{159, 10, 493, {1, 1, 7, 9, 21, 39, 123, 21, 275, 927}},
	{160, 10, 505, {1, 1, 7, 13, 15, 41, 47, 243, 303, 437}},
	{161, 10, 508, {1, 1, 1, 7, 7, 3, 15, 99, 409, 719}},
	{162, 11, 2, {1, 3, 3, 15, 27, 49, 113, 123, 113, 67, 469}},
	{163, 11, 11, {1, 3, 7, 11, 3, 23, 87, 169, 119, 483, 199}},
	{164, 11, 21, {1, 1, 5, 15, 7, 17, 109, 229, 179, 213, 741}},
	{165, 11, 22, {1, 1, 5, 13, 11, 17, 25, 135, 403, 557, 1433}},
	{166, 11, 35, {1, 3, 1, 1, 1, 61, 67, 215, 189, 945, 1243}},
	{167, 11, 49, {1, 1, 7, 13, 17, 33, 9, 221, 429, 217, 1679}},
	{168, 11, 50, {1, 1, 3, 11, 27, 3, 15, 93, 93, 865, 1049}},
	{169, 11, 56, {1, 3, 7, 7, 25, 41, 121, 35, 373, 379, 1547}},
	{170, 11, 61, {1, 3, 3, 9, 11, 35, 45, 205, 241, 9, 59}},
	{171, 11, 70, {1, 3, 1, 7, 3, 51, 7, 177, 53, 975, 89}},
	{172, 11, 74, {1, 1, 3, 5, 27, 1, 113, 231, 299, 759, 861}},
	{173, 11, 79, {1, 3, 3, 15, 25, 29, 5, 255, 139, 891, 2031}},
	{174, 11, 84, {1, 3, 1, 1, 13, 9, 109, 193, 419, 95, 17}},
	{175, 11, 88, {1, 1, 7, 9, 3, 7, 29, 41, 135, 839, 867}},
	{176, 11, 103, {1, 1, 7, 9, 25, 49, 123, 217, 113, 909, 215}},
	{177, 11, 104, {1, 1, 7, 3, 23, 15, 43, 133, 217, 327, 901}},
	{178, 11, 112, {1, 1, 3, 3, 13, 53, 63, 123, 477, 711, 1387}},
	{179, 11, 115, {1, 1, 3, 15, 7, 29, 75, 119, 181, 957, 247}},
	{180, 11, 117, {1, 1, 1, 11, 27, 25, 109, 151, 267, 99, 1461}},
	{181, 11, 122, {1, 3, 7, 15, 5, 5, 53, 145, 11, 725, 1501}},
	{182, 11, 134, {1, 3, 7, 1, 9, 43, 71, 229, 157, 607, 1835}},
	{183, 11, 137, {1, 3, 3, 13, 25, 1, 5, 27, 471, 349, 127}},
	{184, 11, 146, {1, 1, 1, 1, 23, 37, 9, 221, 269, 897, 1685}},
	{185, 11, 148, {1, 1, 3, 3, 31, 29, 51, 19, 311, 553, 1969}},
	{186, 11, 157, {1, 3, 7, 5, 5, 55, 17, 39, 475, 671, 1529}},
	{187, 11, 158, {1, 1, 7, 1, 1, 35, 47, 27, 437, 395, 1635}},
	{188, 11, 162, {1, 1, 7, 3, 13, 23, 43, 135, 327, 139, 389}},
	{189, 11, 164, {1, 3, 7, 3, 9, 25, 91, 25, 429, 219, 513}},
	{190, 11, 168, {1, 1, 3, 5, 13, 29, 119, 201, 277, 157, 2043}},
	{191, 11, 173, {1, 3, 5, 3, 29, 57, 13, 17, 167, 739, 1031}},
	{192, 11, 185, {1, 3, 3, 5, 29, 21, 95, 27, 255, 679, 1531}},
	{193, 11, 186, {1, 3, 7, 15, 9, 5, 21, 71, 61, 961, 1201}},
	{194, 11, 191, {1, 3, 5, 13, 15, 57, 33, 93, 459, 867, 223}},
	{195, 11, 193, {1, 1, 1, 15, 17, 43, 127, 191, 67, 177, 1073}},
	{196, 11, 199, {1, 1, 1, 15, 23, 7, 21, 199, 75, 293, 1611}},
	{197, 11, 213, {1, 3, 7, 13, 15, 39, 21, 149, 65, 741, 319}},
	{198, 11, 214, {1, 3, 7, 11, 23, 13, 101, 89, 277, 519, 711}},
	{199, 11, 220, {1, 3, 7, 15, 19, 27, 85, 203, 441, 97, 1895}},
	{200, 11, 227, {1, 3, 1, 3, 29, 25, 21, 155, 11, 191, 197}},
	{201, 11, 236, {1, 1, 7, 5, 27, 11, 81, 101, 457, 675, 1687}},
	{202, 11, 242, {1, 3, 1, 5, 25, 5, 65, 193, 41, 567, 781}},
	{203, 11, 251, {1, 3, 1, 5, 11, 15, 113, 77, 411, 695, 1111}},
	{204, 11, 256, {1, 1, 3, 9, 11, 53, 119, 171, 55, 297, 509}},
	{205, 11, 259, {1, 1, 1, 1, 11, 39, 113, 139, 165, 347, 595}},
	{206, 11, 265, {1, 3, 7, 11, 9, 17, 101, 13, 81, 325, 1733}},
	{207, 11, 266, {1, 3, 1, 1, 21, 43, 115, 9, 113, 907, 645}},
	{208, 11, 276, {1, 1, 7, 3, 9, 25, 117, 197, 159, 471, 475}},
	{209, 11, 292, {1, 3, 1, 9, 11, 21, 57, 207, 485, 613, 1661}},
	{210, 11, 304, {1, 1, 7, 7, 27, 55, 49, 223, 89, 85, 1523}},
	{211, 11, 310, {1, 1, 5, 3, 19, 41, 45, 51, 447, 299, 1355}},
	{212, 11, 316, {1, 3, 1, 13, 1, 33, 117, 143, 313, 187, 1073}},
	{213, 11, 319, {1, 1, 7, 7, 5, 11, 65, 97, 377, 377, 1501}},
	{214, 11, 322, {1, 3, 1, 1, 21, 35, 95, 65, 99, 23, 1239}},
	{215, 11, 328, {1, 1, 5, 9, 3, 37, 95, 167, 115, 425, 867}},
	{216, 11, 334, {1, 3, 3, 13, 1, 37, 27, 189, 81, 679, 773}},
	{217, 11, 339, {1, 1, 3, 11, 1, 61, 99, 233, 429, 969, 49}},
	{218, 11, 341, {1, 1, 1, 7, 25, 63, 99, 165, 245, 793, 1143}},
	{219, 11, 345, {1, 1, 5, 11, 11, 43, 55, 65, 71, 283, 273}},
	{220, 11, 346, {1, 1, 5, 5, 9, 3, 101, 251, 355, 379, 1611}},
	{221, 11, 362, {1, 1, 1, 15, 21, 63, 85, 99, 49, 749, 1335}},
	{222, 11, 367, {1, 1, 5, 13, 27, 9, 121, 43, 255, 715, 289}},
	{223, 11, 372, {1, 3, 1, 5, 27, 19, 17, 223, 77, 571, 1415}},
	{224, 11, 375, {1, 1, 5, 3, 13, 59, 125, 251, 195, 551, 1737}},
	{225, 11, 376, {1, 3, 3, 15, 13, 27, 49, 105, 389, 971, 755}},
	{226, 11, 381, {1, 3, 5, 15, 23, 43, 35, 107, 447, 763, 253}},
	{227, 11, 385, {1, 3, 5, 11, 21, 3, 17, 39, 497, 407, 611}},
	{228, 11, 388, {1, 1, 7, 13, 15, 31, 113, 17, 23, 507, 1995}},
	{229, 11, 392, {1, 1, 7, 15, 3, 15, 31, 153, 423, 79, 503}},
	{230, 11, 409, {1, 1, 7, 9, 19, 25, 23, 171, 505, 923, 1989}},
	{231, 11, 415, {1, 1, 5, 9, 21, 27, 121, 223, 133, 87, 697}},
	{232, 11, 416, {1, 1, 5, 5, 9, 19, 107, 99, 319, 765, 1461}},
	{233, 11, 421, {1, 1, 3, 3, 19, 25, 3, 101, 171, 729, 187}},
	{234, 11, 428, {1, 1, 3, 1, 13, 23, 85, 93, 291, 209, 37}},
	{235, 11, 431, {1, 1, 1, 15, 25, 25, 77, 253, 333, 947, 1073}},
	{236, 11, 434, {1, 1, 3, 9, 17, 29, 55, 47, 255, 305, 2037}},
	{237, 11, 439, {1, 3, 3, 9, 29, 63, 9, 103, 489, 939, 1523}},
	{238, 11, 446, {1, 3, 7, 15, 7, 31, 89, 175, 369, 339, 595}},
	{239, 11, 451, {1, 3, 7, 13, 25, 5, 71, 207, 251, 367, 665}},
	{240, 11, 453, {1, 3, 3, 3, 21, 25, 75, 35, 31, 321, 1603}},
	{241, 11, 457, {1, 1, 1, 9, 11, 1, 65, 5, 11, 329, 535}},
	{242, 11, 458, {1, 1, 5, 3, 19, 13, 17, 43, 379, 485, 383}},
	{243, 11, 471, {1, 3, 5, 13, 13, 9, 85, 147, 489, 787, 1133}},
	{244, 11, 475, {1, 3, 1, 1, 5, 51, 37, 129, 195, 297, 1783}},
	{245, 11, 478, {1, 1, 3, 15, 19, 57, 59, 181, 455, 697, 2033}},
	{246, 11, 484, {1, 3, 7, 1, 27, 9, 65, 145, 325, 189, 201}},
	{247, 11, 493, {1, 3, 1, 15, 31, 23, 19, 5, 485, 581, 539}},
	{248, 11, 494, {1, 1, 7, 13, 11, 15, 65, 83, 185, 847, 831}},
	{249, 11, 499, {1, 3, 5, 7, 7, 55, 73, 15, 303, 511, 1905}},
	{250, 11, 502, {1, 3, 5, 9, 7, 21, 45, 15, 397, 385, 597}},
	{251, 11, 517, {1, 3, 7, 3, 23, 13, 73, 221, 511, 883, 1265}},
	{252, 11, 518, {1, 1, 3, 11, 1, 51, 73, 185, 33, 975, 1441}},
	{253, 11, 524, {1, 3, 3, 9, 19, 59, 21, 39, 339, 37, 143}},
	{254, 11, 527, {1, 1, 7, 1, 31, 33, 19, 167, 117, 635, 639}},
	{255, 11, 555, {1, 1, 1, 3, 5, 13, 59, 83, 355, 349, 1967}},
	{256, 11, 560, {1, 1, 1, 5, 19, 3, 53, 133, 97, 863, 983}},
	{257, 11, 565, {1, 3, 1, 13, 9, 41, 91, 105, 173, 97, 625}},
	{258, 11, 569, {1, 1, 5, 3, 7, 49, 115, 133, 71, 231, 1063}},
	{259, 11, 578, {1, 1, 7, 5, 17, 43, 47, 45, 497, 547, 757}},
	{260, 11, 580, {1, 3, 5, 15, 21, 61, 123, 191, 249, 31, 631}},
	{261, 11, 587, {1, 3, 7, 9, 17, 7, 11, 185, 127, 169, 1951}},
	{262, 11, 589, {1, 1, 5, 13, 11, 11, 9, 49, 29, 125, 791}},
	{263, 11, 590, {1, 1, 1, 15, 31, 41, 13, 167, 273, 429, 57}},
	{264, 11, 601, {1, 3, 5, 3, 27, 7, 35, 209, 65, 265, 1393}},
	{265, 11, 607, {1, 3, 1, 13, 31, 19, 53, 143, 135, 9, 1021}},
	{266, 11, 611, {1, 1, 7, 13, 31, 5, 115, 153, 143, 957, 623}},
	{267, 11, 614, {1, 1, 5, 11, 25, 19, 29, 31, 297, 943, 443}},
	{268, 11, 617, {1, 3, 3, 5, 21, 11, 127, 81, 479, 25, 699}},
	{269, 11, 618, {1, 1, 3, 11, 25, 31, 97, 19, 195, 781, 705}},
	{270, 11, 625, {1, 1, 5, 5, 31, 11, 75, 207, 197, 885, 2037}},
	{271, 11, 628, {1, 1, 1, 11, 9, 23, 29, 231, 307, 17, 1497}},
	{272, 11, 635, {1, 1, 5, 11, 11, 43, 111, 233, 307, 523, 1259}},
	{273, 11, 641, {1, 1, 7, 5, 1, 21, 107, 229, 343, 933, 217}},
	{274, 11, 647, {1, 1, 1, 11, 3, 21, 125, 131, 405, 599, 1469}},
	{275, 11, 654, {1, 3, 5, 5, 9, 39, 33, 81, 389, 151, 811}},
	{276, 11, 659, {1, 1, 7, 7, 7, 1, 59, 223, 265, 529, 2021}},
	{277, 11, 662, {1, 3, 1, 3, 9, 23, 85, 181, 47, 265, 49}},
	{278, 11, 672, {1, 3, 5, 11, 19, 23, 9, 7, 157, 299, 1983}},
	{279, 11, 675, {1, 3, 1, 5, 15, 5, 21, 105, 29, 339, 1041}},
	{280, 11, 682, {1, 1, 1, 1, 5, 33, 65, 85, 111, 705, 479}},
	{281, 11, 684, {1, 1, 1, 7, 9, 35, 77, 87, 151, 321, 101}},
	{282, 11, 689, {1, 1, 5, 7, 17, 1, 51, 197, 175, 811, 1229}},
	{283, 11, 695, {1, 3, 3, 15, 23, 37, 85, 185, 239, 543, 731}},
	{284, 11, 696, {1, 3, 1, 7, 7, 55, 111, 109, 289, 439, 243}},
	{285, 11, 713, {1, 1, 7, 11, 17, 53, 35, 217, 259, 853, 1667}},
	{286, 11, 719, {1, 3, 1, 9, 1, 63, 87, 17, 73, 565, 1091}},
	{287, 11, 724, {1, 1, 3, 3, 11, 41, 1, 57, 295, 263, 1029}},
	{288, 11, 733, {1, 1, 5, 1, 27, 45, 109, 161, 411, 421, 1395}},
	{289, 11, 734, {1, 3, 5, 11, 25, 35, 47, 191, 339, 417, 1727}},
	{290, 11, 740, {1, 1, 5, 15, 21, 1, 93, 251, 351, 217, 1767}},
	{291, 11, 747, {1, 3, 3, 11, 3, 7, 75, 155, 313, 211, 491}},
	{292, 11, 749, {1, 3, 3, 5, 11, 9, 101, 161, 453, 913, 1067}},
	{293, 11, 752, {1, 1, 3, 1, 15, 45, 127, 141, 163, 727, 1597}},
	{294, 11, 755, {1, 3, 3, 7, 1, 33, 63, 73, 73, 341, 1691}},
	{295, 11, 762, {1, 3, 5, 13, 15, 39, 53, 235, 77, 99, 949}},
	{296, 11, 770, {1, 1, 5, 13, 31, 17, 97, 13, 215, 301, 1927}},
	{297, 11, 782, {1, 1, 7, 1, 1, 37, 91, 93, 441, 251, 1131}},
	{298, 11, 784, {1, 3, 7, 9, 25, 5, 105, 69, 81, 943, 1459}},
	{299, 11, 787, {1, 3, 7, 11, 31, 43, 13, 209, 27, 1017, 501}},
	{300, 11, 789, {1, 1, 7, 15, 1, 33, 31, 233, 161, 507, 387}},
	{301, 11, 793, {1, 3, 3, 5, 5, 53, 33, 177, 503, 627, 1927}},
	{302, 11, 796, {1, 1, 7, 11, 7, 61, 119, 31, 457, 229, 1875}},
	{303, 11, 803, {1, 1, 5, 15, 19, 5, 53, 201, 157, 885, 1057}},
	{304, 11, 805, {1, 3, 7, 9, 1, 35, 51, 113, 249, 425, 1009}},
	{305, 11, 810, {1, 3, 5, 7, 21, 53, 37, 155, 119, 345, 631}},
	{306, 11, 815, {1, 3, 5, 7, 15, 31, 109, 69, 503, 595, 1879}},
	{307, 11, 824, {1, 3, 3, 1, 25, 35, 65, 131, 403, 705, 503}},
	{308, 11, 829, {1, 3, 7, 7, 19, 33, 11, 153, 45, 633, 499}},
	{309, 11, 830, {1, 3, 3, 5, 11, 3, 29, 93, 487, 33, 703}},
	{310, 11, 832, {1, 1, 3, 15, 21, 53, 107, 179, 387, 927, 1757}},
	{311, 11, 841, {1, 1, 3, 7, 21, 45, 51, 147, 175, 317, 361}},
	{312, 11, 847, {1, 1, 1, 7, 7, 13, 15, 243, 269, 795, 1965}},
	{313, 11, 849, {1, 1, 3, 5, 19, 33, 57, 115, 443, 537, 627}},
	{314, 11, 861, {1, 3, 3, 9, 3, 39, 25, 61, 185, 717, 1049}},
	{315, 11, 871, {1, 3, 7, 3, 7, 37, 107, 153, 7, 269, 1581}},
	{316, 11, 878, {1, 1, 7, 3, 7, 41, 91, 41, 145, 489, 1245}},
	{317, 11, 889, {1, 1, 5, 9, 7, 7, 105, 81, 403, 407, 283}},
	{318, 11, 892, {1, 1, 7, 9, 27, 55, 29, 77, 193, 963, 949}},
	{319, 11, 901, {1, 1, 5, 3, 25, 51, 107, 63, 403, 917, 815}},
	{320, 11, 908, {1, 1, 7, 3, 7, 61, 19, 51, 457, 599, 535}},
	{321, 11, 920, {1, 3, 7, 1, 23, 51, 105, 153, 239, 215, 1847}},
	{322, 11, 923, {1, 1, 3, 5, 27, 23, 79, 49, 495, 45, 1935}},
	{323, 11, 942, {1, 1, 1, 11, 11, 47, 55, 133, 495, 999, 1461}},
	{324, 11, 949, {1, 1, 3, 15, 27, 51, 93, 17, 355, 763, 1675}},
	{325, 11, 950, {1, 3, 1, 3, 1, 3, 79, 119, 499, 17, 995}},
	{326, 11, 954, {1, 1, 1, 1, 15, 43, 45, 17, 167, 973, 799}},
	{327, 11, 961, {1, 1, 1, 3, 27, 49, 89, 29, 483, 913, 2023}},
	{328, 11, 968, {1, 1, 3, 3, 5, 11, 75, 7, 41, 851, 611}},
	{329, 11, 971, {1, 3, 1, 3, 7, 57, 39, 123, 257, 283, 507}},
	{330, 11, 973, {1, 3, 3, 11, 27, 23, 113, 229, 187, 299, 133}},
	{331, 11, 979, {1, 1, 3, 13, 9, 63, 101, 77, 451, 169, 337}},
	{332, 11, 982, {1, 3, 7, 3, 3, 59, 45, 195, 229, 415, 409}},
	{333, 11, 986, {1, 3, 5, 3, 11, 19, 71, 93, 43, 857, 369}},
	{334, 11, 998, {1, 3, 7, 9, 19, 33, 115, 19, 241, 703, 247}},
	{335, 11, 1001, {1, 3, 5, 11, 5, 35, 21, 155, 463, 1005, 1073}},
	{336, 11, 1010, {1, 3, 7, 3, 25, 15, 109, 83, 93, 69, 1189}},
	{337, 11, 1012, {1, 3, 5, 7, 5, 21, 93, 133, 135, 167, 903}},
	{338, 12, 41, {1, 1, 7, 7, 3, 59, 121, 161, 285, 815, 1769, 3705}},
	{339, 12, 52, {1, 3, 1, 1, 3, 47, 103, 171, 381, 609, 185, 373}},
	{340, 12, 61, {1, 3, 3, 15, 23, 33, 107, 131, 441, 445, 689, 2059}},
	{341, 12, 62, {1, 3, 3, 11, 7, 53, 101, 167, 435, 803, 1255, 3781}},
	{342, 12, 76, {1, 1, 5, 11, 15, 59, 41, 19, 135, 835, 1263, 505}},
	{343, 12, 104, {1, 1, 7, 11, 21, 49, 23, 219, 127, 961, 1065, 385}},
	{344, 12, 117, {1, 3, 5, 15, 7, 47, 117, 217, 45, 731, 1639, 733}},
	{345, 12, 131, {1, 1, 7, 11, 27, 57, 91, 87, 81, 35, 1269, 1007}},
	{346, 12, 143, {1, 1, 3, 11, 15, 37, 53, 219, 193, 937, 1899, 3733}},
	{347, 12, 145, {1, 3, 5, 3, 13, 11, 27, 19, 199, 393, 965, 2195}},
	{348, 12, 157, {1, 3, 1, 3, 5, 1, 37, 173, 413, 1023, 553, 409}},
	{349, 12, 167, {1, 3, 1, 7, 15, 29, 123, 95, 255, 373, 1799, 3841}},
	{350, 12, 171, {1, 3, 5, 13, 21, 57, 51, 17, 511, 195, 1157, 1831}},
	{351, 12, 176, {1, 1, 1, 15, 29, 19, 7, 73, 295, 519, 587, 3523}},
	{352, 12, 181, {1, 1, 5, 13, 13, 35, 115, 191, 123, 535, 717, 1661}},
	{353, 12, 194, {1, 3, 3, 5, 23, 21, 47, 251, 379, 921, 1119, 297}},
	{354, 12, 217, {1, 3, 3, 9, 29, 53, 121, 201, 135, 193, 523, 2943}},
	{355, 12, 236, {1, 1, 1, 7, 29, 45, 125, 9, 99, 867, 425, 601}},
	{356, 12, 239, {1, 3, 1, 9, 13, 15, 67, 181, 109, 293, 1305, 3079}},
	{357, 12, 262, {1, 3, 3, 9, 5, 35, 15, 209, 305, 87, 767, 2795}},
	{358, 12, 283, {1, 3, 3, 11, 27, 57, 113, 123, 179, 643, 149, 523}},
	{359, 12, 286, {1, 1, 3, 15, 11, 17, 67, 223, 63, 657, 335, 3309}},
	{360, 12, 307, {1, 1, 1, 9, 25, 29, 109, 159, 39, 513, 571, 1761}},
	{361, 12, 313, {1, 1, 3, 1, 5, 63, 75, 19, 455, 601, 123, 691}},
	{362, 12, 319, {1, 1, 1, 3, 21, 5, 45, 169, 377, 513, 1951, 2565}},
	{363, 12, 348, {1, 1, 3, 11, 3, 33, 119, 69, 253, 907, 805, 1449}},
	{364, 12, 352, {1, 1, 5, 13, 31, 15, 17, 7, 499, 61, 687, 1867}},
	{365, 12, 357, {1, 3, 7, 11, 17, 33, 73, 77, 299, 243, 641, 2345}},
	{366, 12, 391, {1, 1, 7, 11, 9, 35, 31, 235, 359, 647, 379, 1161}},
	{367, 12, 398, {1, 3, 3, 15, 31, 25, 5, 67, 33, 45, 437, 4067}},
	{368, 12, 400, {1, 1, 3, 11, 7, 17, 37, 87, 333, 253, 1517, 2921}},
	{369, 12, 412, {1, 1, 7, 15, 7, 15, 107, 189, 153, 769, 1521, 3427}},
	{370, 12, 415, {1, 3, 5, 13, 5, 61, 113, 37, 293, 393, 113, 43}},
	{371, 12, 422, {1, 1, 1, 15, 29, 43, 107, 31, 167, 147, 301, 1021}},
	{372, 12, 440, {1, 1, 1, 13, 3, 1, 35, 93, 195, 181, 2027, 1491}},
	{373, 12, 460, {1, 3, 3, 3, 13, 33, 77, 199, 153, 221, 1699, 3671}},
	{374, 12, 465, {1, 3, 5, 13, 7, 49, 123, 155, 495, 681, 819, 809}},
	{375, 12, 468, {1, 3, 5, 15, 27, 61, 117, 189, 183, 887, 617, 4053}},
	{376, 12, 515, {1, 1, 1, 7, 31, 59, 125, 235, 389, 369, 447, 1039}},
	{377, 12, 536, {1, 3, 5, 1, 5, 39, 115, 89, 249, 377, 431, 3747}},
	{378, 12, 539, {1, 1, 1, 5, 7, 47, 59, 157, 77, 445, 699, 3439}},
	{379, 12, 551, {1, 1, 3, 5, 11, 21, 19, 75, 11, 599, 1575, 735}},
	{380, 12, 558, {1, 3, 5, 3, 19, 13, 41, 69, 199, 143, 1761, 3215}},
	{381, 12, 563, {1, 3, 5, 7, 19, 43, 25, 41, 41, 11, 1647, 2783}},
	{382, 12, 570, {1, 3, 1, 9, 19, 45, 111, 97, 405, 399, 457, 3219}},
	{383, 12, 595, {1, 1, 3, 1, 23, 15, 65, 121, 59, 985, 829, 2259}},
	{384, 12, 598, {1, 1, 3, 7, 17, 13, 107, 229, 75, 551, 1299, 2363}},
	{385, 12, 617, {1, 1, 5, 5, 21, 57, 23, 199, 509, 139, 2007, 3875}},
	{386, 12, 647, {1, 3, 1, 11, 19, 53, 15, 229, 215, 741, 695, 823}},
	{387, 12, 654, {1, 3, 7, 1, 29, 3, 17, 163, 417, 559, 549, 319}},
	{388, 12, 678, {1, 3, 1, 13, 17, 9, 47, 133, 365, 7, 1937, 1071}},
	{389, 12, 713, {1, 3, 5, 7, 19, 37, 55, 163, 301, 249, 689, 2327}},
	{390, 12, 738, {1, 3, 5, 13, 11, 23, 61, 205, 257, 377, 615, 1457}},
	{391, 12, 747, {1, 3, 5, 1, 23, 37, 13, 75, 331, 495, 579, 3367}},
	{392, 12, 750, {1, 1, 1, 9, 1, 23, 49, 129, 475, 543, 883, 2531}},
	{393, 12, 757, {1, 3, 1, 5, 23, 59, 51, 35, 343, 695, 219, 369}},
	{394, 12, 772, {1, 3, 3, 1, 27, 17, 63, 97, 71, 507, 1929, 613}},
	{395, 12, 803, {1, 1, 5, 1, 21, 31, 11, 109, 247, 409, 1817, 2173}},
	{396, 12, 810, {1, 1, 3, 15, 23, 9, 7, 209, 301, 23, 147, 1691}},
	{397, 12, 812, {1, 1, 7, 5, 5, 19, 37, 229, 249, 277, 1115, 2309}},
	{398, 12, 850, {1, 1, 1, 5, 5, 63, 5, 249, 285, 431, 343, 2467}},
	{399, 12, 862, {1, 1, 1, 11, 7, 45, 35, 75, 505, 537, 29, 2919}},
	{400, 12, 906, {1, 3, 5, 15, 11, 39, 15, 63, 263, 9, 199, 445}},
	{401, 12, 908, {1, 3, 3, 3, 27, 63, 53, 171, 227, 63, 1049, 827}},
	{402, 12, 929, {1, 1, 3, 13, 7, 11, 115, 183, 179, 937, 1785, 381}},
	{403, 12, 930, {1, 3, 1, 11, 13, 15, 107, 81, 53, 295, 1785, 3757}},
	{404, 12, 954, {1, 3, 3, 13, 11, 5, 109, 243, 3, 505, 323, 1373}},
	{405, 12, 964, {1, 3, 3, 11, 21, 51, 17, 177, 381, 937, 1263, 3889}},
	{406, 12, 982, {1, 3, 5, 9, 27, 25, 85, 193, 143, 573, 1189, 2995}},
	{407, 12, 985, {1, 3, 5, 11, 13, 9, 81, 21, 159, 953, 91, 1751}},
	{408, 12, 991, {1, 1, 3, 3, 27, 61, 11, 253, 391, 333, 1105, 635}},
	{409, 12, 992, {1, 3, 3, 15, 9, 57, 95, 81, 419, 735, 251, 1141}},
	{410, 12, 1067, {1, 1, 5, 9, 31, 39, 59, 13, 319, 807, 1241, 2433}},
	{411, 12, 1070, {1, 3, 3, 5, 27, 13, 107, 141, 423, 937, 2027, 3233}},
	{412, 12, 1096, {1, 3, 3, 9, 9, 25, 125, 23, 443, 835, 1245, 847}},
	{413, 12, 1099, {1, 1, 7, 15, 17, 17, 83, 107, 411, 285, 847, 1571}},
	{414, 12, 1116, {1, 1, 3, 13, 29, 61, 37, 81, 349, 727, 1453, 1957}},
	{415, 12, 1143, {1, 3, 7, 11, 31, 13, 59, 77, 273, 591, 1265, 1533}},
	{416, 12, 1165, {1, 1, 7, 7, 13, 17, 25, 25, 187, 329, 347, 1473}},
	{417, 12, 1178, {1, 3, 7, 7, 5, 51, 37, 99, 221, 153, 503, 2583}},
	{418, 12, 1184, {1, 3, 1, 13, 19, 27, 11, 69, 181, 479, 1183, 3229}},
	{419, 12, 1202, {1, 3, 3, 13, 23, 21, 103, 147, 323, 909, 947, 315}},
	{420, 12, 1213, {1, 3, 1, 3, 23, 1, 31, 59, 93, 513, 45, 2271}},
	{421, 12, 1221, {1, 3, 5, 1, 7, 43, 109, 59, 231, 41, 1515, 2385}},
	{422, 12, 1240, {1, 3, 1, 5, 31, 57, 49, 223, 283, 1013, 11, 701}},
	{423, 12, 1246, {1, 1, 5, 1, 19, 53, 55, 31, 31, 299, 495, 693}},
	{424, 12, 1252, {1, 3, 3, 9, 5, 33, 77, 253, 427, 791, 731, 1019}},
	{425, 12, 1255, {1, 3, 7, 11, 1, 9, 119, 203, 53, 877, 1707, 3499}},
	{426, 12, 1267, {1, 1, 3, 7, 13, 39, 55, 159, 423, 113, 1653, 3455}},
	{427, 12, 1293, {1, 1, 3, 5, 21, 47, 51, 59, 55, 411, 931, 251}},
	{428, 12, 1301, {1, 3, 7, 3, 31, 25, 81, 115, 405, 239, 741, 455}},
	{429, 12, 1305, {1, 1, 5, 1, 31, 3, 101, 83, 479, 491, 1779, 2225}},
	{430, 12, 1332, {1, 3, 3, 3, 9, 37, 107, 161, 203, 503, 767, 3435}},
	{431, 12, 1349, {1, 3, 7, 9, 1, 27, 61, 119, 233, 39, 1375, 4089}},
	{432, 12, 1384, {1, 1, 5, 9, 1, 31, 45, 51, 369, 587, 383, 2813}},
	{433, 12, 1392, {1, 3, 7, 5, 31, 7, 49, 119, 487, 591, 1627, 53}},
	{434, 12, 1402, {1, 1, 7, 1, 9, 47, 1, 223, 369, 711, 1603, 1917}},
	{435, 12, 1413, {1, 3, 5, 3, 21, 37, 111, 17, 483, 739, 1193, 2775}},
	{436, 12, 1417, {1, 3, 3, 7, 17, 11, 51, 117, 455, 191, 1493, 3821}},
	{437, 12, 1423, {1, 1, 5, 9, 23, 39, 99, 181, 343, 485, 99, 1931}},
	{438, 12, 1451, {1, 3, 1, 7, 29, 49, 31, 71, 489, 527, 1763, 2909}},
	{439, 12, 1480, {1, 1, 5, 11, 5, 5, 73, 189, 321, 57, 1191, 3685}},
	{440, 12, 1491, {1, 1, 5, 15, 13, 45, 125, 207, 371, 415, 315, 983}},
	{441, 12, 1503, {1, 3, 3, 5, 25, 59, 33, 31, 239, 919, 1859, 2709}},
	{442, 12, 1504, {1, 3, 5, 13, 27, 61, 23, 115, 61, 413, 1275, 3559}},
	{443, 12, 1513, {1, 3, 7, 15, 5, 59, 101, 81, 47, 967, 809, 3189}},
	{444, 12, 1538, {1, 1, 5, 11, 31, 15, 39, 25, 173, 505, 809, 2677}},
	{445, 12, 1544, {1, 1, 5, 9, 19, 13, 95, 89, 511, 127, 1395, 2935}},
	{446, 12, 1547, {1, 1, 5, 5, 31, 45, 9, 57, 91, 303, 1295, 3215}},
	{447, 12, 1555, {1, 3, 3, 3, 19, 15, 113, 187, 217, 489, 1285, 1803}},
	{448, 12, 1574, {1, 1, 3, 1, 13, 29, 57, 139, 255, 197, 537, 2183}},
	{449, 12, 1603, {1, 3, 1, 15, 11, 7, 53, 255, 467, 9, 757, 3167}},
	{450, 12, 1615, {1, 3, 3, 15, 21, 13, 9, 189, 359, 323, 49, 333}},
	{451, 12, 1618, {1, 3, 7, 11, 7, 37, 21, 119, 401, 157, 1659, 1069}},
	{452, 12, 1629, {1, 1, 5, 7, 17, 33, 115, 229, 149, 151, 2027, 279}},
	{453, 12, 1634, {1, 1, 5, 15, 5, 49, 77, 155, 383, 385, 1985, 945}},
	{454, 12, 1636, {1, 3, 7, 3, 7, 55, 85, 41, 357, 527, 1715, 1619}},
	{455, 12, 1639, {1, 1, 3, 1, 21, 45, 115, 21, 199, 967, 1581, 3807}},
	{456, 12, 1657, {1, 1, 3, 7, 21, 39, 117, 191, 169, 73, 413, 3417}},
	{457, 12, 1667, {1, 1, 1, 13, 1, 31, 57, 195, 231, 321, 367, 1027}},
	{458, 12, 1681, {1, 3, 7, 3, 11, 29, 47, 161, 71, 419, 1721, 437}},
	{459, 12, 1697, {1, 1, 7, 3, 11, 9, 43, 65, 157, 1, 1851, 823}},
	{460, 12, 1704, {1, 1, 1, 5, 21, 15, 31, 101, 293, 299, 127, 1321}},
	{461, 12, 1709, {1, 1, 7, 1, 27, 1, 11, 229, 241, 705, 43, 1475}},
	{462, 12, 1722, {1, 3, 7, 1, 5, 15, 73, 183, 193, 55, 1345, 49}},
	{463, 12, 1730, {1, 3, 3, 3, 19, 3, 55, 21, 169, 663, 1675, 137}},
	{464, 12, 1732, {1, 1, 1, 13, 7, 21, 69, 67, 373, 965, 1273, 2279}},
	{465, 12, 1802, {1, 1, 7, 7, 21, 23, 17, 43, 341, 845, 465, 3355}},
	{466, 12, 1804, {1, 3, 5, 5, 25, 5, 81, 101, 233, 139, 359, 2057}},
	{467, 12, 1815, {1, 1, 3, 11, 15, 39, 55, 3, 471, 765, 1143, 3941}},
	{468, 12, 1826, {1, 1, 7, 15, 9, 57, 81, 79, 215, 433, 333, 3855}},
	{469, 12, 1832, {1, 1, 5, 5, 19, 45, 83, 31, 209, 363, 701, 1303}},
	{470, 12, 1843, {1, 3, 7, 5, 1, 13, 55, 163, 435, 807, 287, 2031}},
	{471, 12, 1849, {1, 3, 3, 7, 3, 3, 17, 197, 39, 169, 489, 1769}},
	{472, 12, 1863, {1, 1, 3, 5, 29, 43, 87, 161, 289, 339, 1233, 2353}},
	{473, 12, 1905, {1, 3, 3, 9, 21, 9, 77, 1, 453, 167, 1643, 2227}},
	{474, 12, 1928, {1, 1, 7, 1, 15, 7, 67, 33, 193, 241, 1031, 2339}},
	{475, 12, 1933, {1, 3, 1, 11, 1, 63, 45, 65, 265, 661, 849, 1979}},
	{476, 12, 1939, {1, 3, 1, 13, 19, 49, 3, 11, 159, 213, 659, 2839}},
	{477, 12, 1976, {1, 3, 5, 11, 9, 29, 27, 227, 253, 449, 1403, 3427}},
	{478, 12, 1996, {1, 1, 3, 1, 7, 3, 77, 143, 277, 779, 1499, 475}},
	{479, 12, 2013, {1, 1, 1, 5, 11, 23, 87, 131, 393, 849, 193, 3189}},
	{480, 12, 2014, {1, 3, 5, 11, 3, 3, 89, 9, 449, 243, 1501, 1739}},
	{481, 12, 2020, {1, 3, 1, 9, 29, 29, 113, 15, 65, 611, 135, 3687}},
	{482, 13, 13, {1, 1, 1, 9, 21, 19, 39, 151, 395, 501, 1339, 959, 2725}},
	{483, 13, 19, {1, 3, 7, 1, 7, 35, 45, 33, 119, 225, 1631, 1695, 1459}},
	{484, 13, 26, {1, 1, 1, 3, 25, 55, 37, 79, 167, 907, 1075, 271, 4059}},
	{485, 13, 41, {1, 3, 5, 13, 5, 13, 53, 165, 437, 67, 1705, 3177, 8095}},
	{486, 13, 50, {1, 3, 3, 13, 27, 57, 95, 55, 443, 245, 1945, 1725, 1929}},
	{487, 13, 55, {1, 3, 1, 9, 5, 33, 109, 35, 99, 827, 341, 2401, 2411}},
	{488, 13, 69, {1, 1, 5, 9, 7, 33, 43, 39, 87, 799, 635, 3481, 7159}},
	{489, 13, 70, {1, 3, 1, 1, 31, 15, 45, 27, 337, 113, 987, 2065, 2529}},
	{490, 13, 79, {1, 1, 5, 9, 5, 15, 105, 123, 479, 289, 1609, 2177, 4629}},
	{491, 13, 82, {1, 3, 5, 11, 31, 47, 97, 87, 385, 195, 1041, 651, 3271}},
	{492, 13, 87, {1, 1, 3, 7, 17, 3, 101, 55, 87, 629, 1687, 1387, 2745}},
	{493, 13, 93, {1, 3, 5, 5, 7, 21, 9, 237, 313, 549, 1107, 117, 6183}},
	{494, 13, 94, {1, 1, 3, 9, 9, 5, 55, 201, 487, 851, 1103, 2993, 4055}},
	{495, 13, 97, {1, 1, 5, 9, 31, 19, 59, 7, 363, 381, 1167, 2057, 5715}},
	{496, 13, 100, {1, 3, 3, 15, 23, 63, 19, 227, 387, 827, 487, 1049, 7471}},
	{497, 13, 112, {1, 3, 1, 5, 23, 25, 61, 245, 363, 863, 963, 3583, 6475}},
	{498, 13, 121, {1, 1, 5, 1, 5, 27, 81, 85, 275, 49, 235, 3291, 1195}},
	{499, 13, 134, {1, 1, 5, 7, 23, 53, 85, 107, 511, 779, 1265, 1093, 7859}},
	{500, 13, 138, {1, 3, 3, 1, 9, 21, 75, 219, 59, 485, 1739, 3845, 1109}},
	{501, 13, 148, {1, 3, 5, 1, 13, 41, 19, 143, 293, 391, 2023, 1791, 4399}},
	{502, 13, 151, {1, 3, 7, 15, 21, 13, 21, 195, 215, 413, 523, 2099, 2341}},
	{503, 13, 157, {1, 1, 1, 3, 29, 51, 47, 57, 135, 575, 943, 1673, 541}},
	{504, 13, 161, {1, 3, 5, 1, 9, 13, 113, 175, 447, 115, 657, 4077, 5973}},
	{505, 13, 179, {1, 1, 1, 11, 17, 41, 37, 95, 297, 579, 911, 2207, 2387}},
	{506, 13, 181, {1, 3, 5, 3, 23, 11, 23, 231, 93, 667, 711, 1563, 7961}},
	{507, 13, 188, {1, 1, 7, 3, 17, 59, 13, 181, 141, 991, 1817, 457, 1711}},
	{508, 13, 196, {1, 3, 3, 5, 31, 59, 81, 205, 245, 537, 1049, 997, 1815}},
	{509, 13, 203, {1, 3, 7, 5, 17, 13, 9, 79, 17, 185, 5, 2211, 6263}},
	{510, 13, 206, {1, 3, 7, 13, 7, 53, 61, 145, 13, 285, 1203, 947, 2933}},
	{511, 13, 223, {1, 1, 7, 3, 31, 19, 69, 217, 47, 441, 1893, 673, 4451}},
	{512, 13, 224, {1, 1, 1, 1, 25, 9, 23, 225, 385, 629, 603, 3747, 4241}},
	{513, 13, 227, {1, 3, 1, 9, 5, 37, 31, 237, 431, 79, 1521, 459, 2523}},
	{514, 13, 230, {1, 3, 7, 3, 9, 43, 105, 179, 5, 225, 799, 1777, 4893}},
	{515, 13, 239, {1, 1, 3, 1, 29, 45, 29, 159, 267, 247, 455, 847, 3909}},
	{516, 13, 241, {1, 1, 3, 7, 25, 21, 121, 57, 467, 275, 719, 1521, 7319}},
	{517, 13, 248, {1, 3, 1, 3, 11, 35, 119, 123, 81, 979, 1187, 3623, 4293}},
	{518, 13, 253, {1, 1, 1, 7, 15, 25, 121, 235, 25, 487, 873, 1787, 1977}},
	{519, 13, 268, {1, 1, 1, 11, 3, 7, 17, 135, 345, 353, 383, 4011, 2573}},
	{520, 13, 274, {1, 3, 7, 15, 27, 13, 97, 123, 65, 675, 951, 1285, 6559}},
	{521, 13, 283, {1, 3, 7, 3, 7, 1, 71, 19, 325, 765, 337, 1197, 2697}},
	{522, 13, 286, {1, 3, 5, 1, 31, 37, 11, 71, 169, 283, 83, 3801, 7083}},
	{523, 13, 289, {1, 1, 3, 15, 17, 29, 83, 65, 275, 679, 1749, 4007, 7749}},
	{524, 13, 301, {1, 1, 3, 1, 21, 11, 41, 95, 237, 361, 1819, 2783, 2383}},
	{525, 13, 302, {1, 3, 7, 11, 29, 57, 111, 187, 465, 145, 605, 1987, 8109}},
	{526, 13, 316, {1, 1, 3, 3, 19, 15, 55, 83, 357, 1001, 643, 1517, 6529}},
	{527, 13, 319, {1, 3, 1, 5, 29, 35, 73, 23, 77, 619, 1523, 1725, 8145}},
	{528, 13, 324, {1, 1, 5, 5, 19, 23, 7, 197, 449, 337, 717, 2921, 315}},
	{529, 13, 331, {1, 3, 5, 9, 7, 63, 117, 97, 97, 813, 1925, 2817, 1579}},
	{530, 13, 333, {1, 1, 1, 11, 31, 7, 25, 235, 231, 133, 1007, 1371, 1553}},
	{531, 13, 345, {1, 1, 7, 5, 19, 7, 47, 171, 267, 243, 1331, 567, 6033}},
	{532, 13, 351, {1, 1, 5, 1, 7, 49, 55, 89, 109, 735, 1455, 3193, 6239}},
	{533, 13, 358, {1, 1, 1, 7, 1, 61, 9, 103, 3, 929, 1481, 2927, 2957}},
	{534, 13, 375, {1, 1, 5, 13, 17, 21, 75, 49, 255, 1019, 1161, 2133, 1177}},
	{535, 13, 379, {1, 3, 1, 3, 13, 15, 41, 247, 211, 409, 1163, 523, 2635}},
	{536, 13, 381, {1, 3, 7, 7, 21, 59, 91, 149, 479, 391, 681, 2311, 6249}},
	{537, 13, 386, {1, 1, 5, 11, 27, 53, 21, 211, 197, 815, 719, 1605, 255}},
	{538, 13, 403, {1, 1, 3, 3, 9, 33, 59, 3, 323, 1, 101, 1135, 8105}},
	{539, 13, 405, {1, 3, 3, 1, 29, 5, 17, 141, 51, 991, 841, 327, 3859}},
	{540, 13, 419, {1, 3, 1, 5, 11, 19, 23, 89, 175, 173, 165, 2881, 1881}},
	{541, 13, 426, {1, 1, 1, 15, 13, 51, 87, 39, 495, 611, 1341, 1531, 7029}},
	{542, 13, 428, {1, 1, 3, 11, 13, 55, 75, 185, 57, 61, 1917, 2051, 5965}},
	{543, 13, 439, {1, 1, 5, 5, 7, 53, 11, 217, 213, 933, 921, 3607, 5175}},
	{544, 13, 440, {1, 3, 3, 5, 17, 53, 103, 251, 369, 781, 1319, 3717, 4439}},
	{545, 13, 446, {1, 3, 5, 13, 1, 39, 25, 235, 321, 773, 251, 3111, 6397}},
	{546, 13, 451, {1, 1, 7, 3, 31, 5, 25, 29, 325, 385, 1313, 127, 4705}},
	{547, 13, 454, {1, 1, 5, 15, 15, 27, 15, 85, 239, 243, 1633, 3473, 2621}},
	{548, 13, 458, {1, 3, 3, 3, 9, 19, 113, 13, 137, 165, 25, 2957, 7549}},
	{549, 13, 465, {1, 3, 1, 3, 11, 21, 3, 97, 417, 183, 1205, 1437, 247}},
	{550, 13, 468, {1, 1, 7, 3, 17, 21, 125, 55, 67, 387, 385, 2323, 887}},
	{551, 13, 472, {1, 3, 5, 5, 29, 11, 103, 223, 233, 641, 133, 415, 1297}},
	{552, 13, 475, {1, 3, 3, 11, 1, 9, 5, 189, 235, 1007, 1363, 3985, 889}},
	{553, 13, 477, {1, 3, 7, 9, 23, 19, 19, 183, 269, 403, 1643, 3559, 5189}},
	{554, 13, 496, {1, 3, 7, 3, 29, 45, 17, 69, 475, 149, 1291, 2689, 7625}},
	{555, 13, 502, {1, 3, 7, 3, 27, 37, 41, 73, 253, 1001, 431, 1111, 7887}},
	{556, 13, 508, {1, 1, 7, 5, 3, 7, 87, 143, 289, 495, 631, 3011, 6151}},
	{557, 13, 517, {1, 1, 1, 13, 5, 45, 17, 167, 23, 975, 801, 1975, 6833}},
	{558, 13, 521, {1, 3, 1, 11, 7, 21, 39, 23, 213, 429, 1301, 2059, 197}},
	{559, 13, 527, {1, 3, 3, 15, 3, 57, 121, 133, 29, 711, 1961, 2497, 189}},
	{560, 13, 530, {1, 1, 3, 5, 11, 55, 115, 137, 233, 673, 985, 2849, 5911}},
	{561, 13, 532, {1, 1, 7, 15, 29, 45, 1, 241, 329, 323, 925, 2821, 3331}},
	{562, 13, 542, {1, 1, 5, 7, 13, 31, 81, 105, 199, 145, 195, 1365, 5119}},
	{563, 13, 552, {1, 3, 7, 11, 3, 55, 11, 31, 117, 343, 1265, 1837, 2451}},
	{564, 13, 555, {1, 1, 3, 7, 29, 57, 61, 179, 429, 591, 177, 1945, 2159}},
	{565, 13, 560, {1, 3, 5, 11, 23, 49, 101, 137, 339, 323, 1035, 1749, 7737}},
	{566, 13, 566, {1, 3, 1, 13, 21, 35, 55, 79, 19, 269, 1055, 2651, 7083}},
	{567, 13, 575, {1, 3, 3, 11, 9, 9, 95, 167, 437, 361, 1185, 4083, 603}},
	{568, 13, 577, {1, 1, 1, 7, 31, 61, 77, 65, 489, 657, 691, 2423, 4147}},
	{569, 13, 589, {1, 3, 5, 7, 21, 37, 87, 191, 311, 453, 2013, 829, 2619}},
	{570, 13, 590, {1, 1, 5, 9, 17, 47, 35, 101, 5, 813, 1157, 1279, 7365}},
	{571, 13, 602, {1, 1, 5, 3, 11, 35, 113, 199, 369, 721, 901, 1471, 7801}},
	{572, 13, 607, {1, 3, 1, 5, 9, 61, 83, 157, 391, 739, 1957, 2123, 4341}},
	{573, 13, 608, {1, 3, 5, 11, 19, 19, 111, 225, 383, 219, 997, 717, 7505}},
	{574, 13, 611, {1, 3, 1, 11, 13, 63, 35, 127, 209, 831, 501, 3017, 3507}},
	{575, 13, 613, {1, 3, 7, 9, 29, 7, 11, 163, 81, 563, 1445, 3215, 6377}},
	{576, 13, 625, {1, 3, 7, 11, 25, 3, 39, 195, 491, 45, 839, 4021, 4899}},
	{577, 13, 644, {1, 3, 7, 15, 13, 5, 67, 143, 117, 505, 1281, 3679, 5695}},
	{578, 13, 651, {1, 3, 7, 9, 9, 19, 21, 221, 147, 763, 683, 2211, 589}},
	{579, 13, 654, {1, 1, 3, 5, 21, 47, 53, 109, 299, 807, 1153, 1209, 7961}},
	{580, 13, 656, {1, 3, 7, 11, 9, 31, 45, 43, 505, 647, 1127, 2681, 4917}},
	{581, 13, 662, {1, 1, 5, 15, 31, 41, 63, 113, 399, 727, 673, 2587, 5259}},
	{582, 13, 668, {1, 1, 1, 13, 17, 53, 35, 99, 57, 243, 1447, 1919, 2831}},
	{583, 13, 681, {1, 3, 7, 11, 23, 51, 13, 9, 49, 449, 997, 3073, 4407}},
	{584, 13, 682, {1, 3, 5, 7, 23, 33, 89, 41, 415, 53, 697, 1113, 1489}},
	{585, 13, 689, {1, 1, 3, 7, 1, 13, 29, 13, 255, 749, 77, 3463, 1761}},
	{586, 13, 696, {1, 3, 3, 7, 13, 15, 93, 191, 309, 869, 739, 1041, 3053}},
	{587, 13, 699, {1, 3, 5, 13, 5, 19, 109, 211, 347, 839, 893, 2947, 7735}},
	{588, 13, 707, {1, 3, 1, 13, 27, 3, 119, 157, 485, 99, 1703, 3895, 573}},
	{589, 13, 709, {1, 3, 7, 11, 1, 23, 123, 105, 31, 359, 275, 1775, 3685}},
	{590, 13, 714, {1, 3, 3, 5, 27, 11, 125, 3, 413, 199, 2043, 2895, 2945}},
	{591, 13, 716, {1, 3, 3, 3, 15, 49, 121, 159, 233, 543, 193, 4007, 321}},
	{592, 13, 719, {1, 1, 3, 5, 9, 47, 87, 1, 51, 1011, 1595, 2239, 6467}},
	{593, 13, 727, {1, 3, 7, 9, 1, 33, 87, 137, 469, 749, 1413, 805, 6817}},
	{594, 13, 734, {1, 3, 1, 13, 19, 45, 95, 227, 29, 677, 1275, 3395, 4451}},
	{595, 13, 738, {1, 1, 7, 5, 7, 63, 33, 71, 443, 561, 1311, 3069, 6943}},
	{596, 13, 743, {1, 1, 1, 13, 9, 37, 23, 69, 13, 415, 1479, 1197, 861}},
	{597, 13, 747, {1, 3, 3, 13, 27, 21, 13, 233, 105, 777, 345, 2443, 1105}},
	{598, 13, 757, {1, 1, 7, 11, 23, 13, 21, 147, 221, 549, 73, 2729, 6279}},
	{599, 13, 769, {1, 1, 7, 7, 25, 27, 15, 45, 227, 39, 75, 1191, 3563}},
	{600, 13, 770, {1, 1, 5, 7, 13, 49, 99, 167, 227, 13, 353, 1047, 8075}},
	{601, 13, 776, {1, 1, 3, 13, 31, 9, 27, 7, 461, 737, 1559, 3243, 53}},
	{602, 13, 790, {1, 3, 1, 1, 21, 41, 97, 165, 171, 821, 587, 2137, 2293}},
	{603, 13, 799, {1, 3, 1, 11, 17, 41, 29, 187, 87, 599, 1467, 1395, 5931}},
	{604, 13, 805, {1, 1, 1, 9, 9, 49, 89, 205, 409, 453, 61, 1923, 1257}},
	{605, 13, 809, {1, 3, 7, 3, 9, 43, 89, 143, 431, 83, 1243, 1795, 3599}},
	{606, 13, 812, {1, 3, 5, 13, 3, 25, 59, 219, 43, 223, 797, 2651, 6015}},
	{607, 13, 820, {1, 1, 5, 15, 7, 55, 65, 207, 213, 311, 1287, 1269, 6467}},
	{608, 13, 827, {1, 3, 7, 11, 21, 57, 31, 183, 351, 857, 911, 1683, 7155}},
	{609, 13, 829, {1, 3, 5, 11, 27, 1, 21, 47, 387, 383, 1593, 115, 3805}},
	{610, 13, 835, {1, 3, 1, 1, 13, 23, 87, 173, 181, 619, 1653, 3931, 6073}},
	{611, 13, 841, {1, 1, 7, 5, 17, 43, 37, 61, 307, 621, 1785, 55, 115}},
	{612, 13, 844, {1, 3, 7, 15, 25, 61, 123, 15, 237, 671, 1473, 467, 1907}},
	{613, 13, 856, {1, 1, 7, 5, 29, 57, 75, 237, 85, 699, 159, 3577, 4771}},
	{614, 13, 859, {1, 1, 1, 11, 25, 19, 51, 1, 147, 31, 895, 2617, 625}},
	{615, 13, 862, {1, 3, 7, 5, 29, 15, 115, 175, 395, 391, 1141, 1827, 1181}},
	{616, 13, 865, {1, 3, 5, 7, 17, 7, 11, 193, 89, 243, 561, 3787, 4551}},
	{617, 13, 885, {1, 3, 1, 11, 7, 57, 7, 125, 403, 947, 1261, 409, 8083}},
	{618, 13, 890, {1, 1, 5, 13, 21, 63, 115, 233, 231, 921, 1747, 3635, 2519}},
	{619, 13, 905, {1, 1, 5, 11, 3, 27, 15, 91, 505, 591, 1451, 3881, 2997}},
	{620, 13, 916, {1, 1, 3, 11, 21, 9, 109, 153, 317, 533, 593, 3967, 2797}},
	{621, 13, 925, {1, 3, 3, 13, 9, 57, 121, 245, 219, 867, 967, 791, 7095}},
	{622, 13, 935, {1, 1, 1, 9, 29, 21, 99, 35, 375, 959, 329, 4087, 7171}},
	{623, 13, 939, {1, 1, 1, 9, 11, 17, 17, 97, 89, 135, 631, 3809, 3253}},
	{624, 13, 942, {1, 1, 1, 15, 21, 51, 91, 249, 459, 801, 757, 2353, 2033}},
	{625, 13, 949, {1, 3, 5, 9, 23, 29, 77, 53, 399, 767, 1817, 2171, 1629}},
	{626, 13, 953, {1, 1, 3, 5, 29, 5, 43, 121, 17, 859, 1479, 3785, 6641}},
	{627, 13, 956, {1, 1, 3, 7, 7, 61, 45, 109, 371, 833, 91, 153, 4553}},
	{628, 13, 961, {1, 1, 3, 11, 7, 55, 81, 123, 389, 139, 1933, 891, 1789}},
	{629, 13, 968, {1, 3, 7, 15, 25, 17, 93, 165, 503, 717, 1553, 1475, 1627}},
	{630, 13, 976, {1, 1, 1, 13, 13, 63, 13, 225, 357, 571, 33, 4073, 3795}},
	{631, 13, 988, {1, 1, 3, 11, 1, 31, 107, 145, 407, 961, 501, 2987, 103}},
	{632, 13, 995, {1, 1, 7, 1, 23, 63, 49, 193, 173, 281, 25, 2465, 5927}},
	{633, 13, 997, {1, 1, 7, 1, 1, 1, 85, 77, 273, 693, 349, 1239, 4503}},
	{634, 13, 1007, {1, 1, 5, 11, 7, 61, 9, 121, 25, 357, 1443, 405, 7827}},
	{635, 13, 1015, {1, 1, 7, 13, 11, 53, 11, 207, 145, 211, 1703, 1081, 2117}},
	{636, 13, 1016, {1, 1, 3, 11, 27, 23, 19, 9, 297, 279, 1481, 2273, 6387}},
	{637, 13, 1027, {1, 3, 3, 5, 15, 45, 3, 41, 305, 87, 1815, 3461, 5349}},
	{638, 13, 1036, {1, 3, 3, 13, 9, 37, 79, 125, 259, 561, 1087, 4091, 793}},
	{639, 13, 1039, {1, 3, 5, 7, 31, 55, 7, 145, 347, 929, 589, 2783, 5905}},
	{640, 13, 1041, {1, 1, 7, 15, 3, 25, 1, 181, 13, 243, 653, 2235, 7445}},
	{641, 13, 1048, {1, 3, 5, 5, 17, 53, 65, 7, 33, 583, 1363, 1313, 2319}},
	{642, 13, 1053, {1, 3, 3, 7, 27, 47, 97, 201, 187, 321, 63, 1515, 7917}},
	{643, 13, 1054, {1, 1, 3, 5, 23, 9, 3, 165, 61, 19, 1789, 3783, 3037}},
	{644, 13, 1058, {1, 3, 1, 13, 15, 43, 125, 191, 67, 273, 1551, 2227, 5253}},
	{645, 13, 1075, {1, 1, 1, 13, 25, 53, 107, 33, 299, 249, 1475, 2233, 907}},
	{646, 13, 1082, {1, 3, 5, 1, 23, 37, 85, 17, 207, 643, 665, 2933, 5199}},
	{647, 13, 1090, {1, 1, 7, 7, 25, 57, 59, 41, 15, 751, 751, 1749, 7053}},
	{648, 13, 1109, {1, 3, 3, 1, 13, 25, 127, 93, 281, 613, 875, 2223, 6345}},
	{649, 13, 1110, {1, 1, 5, 3, 29, 55, 79, 249, 43, 317, 533, 995, 1991}},
	{650, 13, 1119, {1, 3, 3, 15, 17, 49, 79, 31, 193, 233, 1437, 2615, 819}},
	{651, 13, 1126, {1, 1, 5, 15, 25, 3, 123, 145, 377, 9, 455, 1191, 3953}},
	{652, 13, 1130, {1, 3, 5, 3, 15, 19, 41, 231, 81, 393, 3, 19, 2409}},
	{653, 13, 1135, {1, 1, 3, 1, 27, 43, 113, 179, 7, 853, 947, 2731, 297}},
	{654, 13, 1137, {1, 1, 1, 11, 29, 39, 53, 191, 443, 689, 529, 3329, 7431}},
	{655, 13, 1140, {1, 3, 7, 5, 3, 29, 19, 67, 441, 113, 949, 2769, 4169}},
	{656, 13, 1149, {1, 3, 5, 11, 11, 55, 85, 169, 215, 815, 803, 2345, 3967}},
	{657, 13, 1156, {1, 1, 7, 9, 5, 45, 111, 5, 419, 375, 303, 1725, 4489}},
	{658, 13, 1159, {1, 3, 5, 15, 29, 43, 79, 19, 23, 417, 381, 541, 4923}},
	{659, 13, 1160, {1, 1, 3, 15, 3, 31, 117, 39, 117, 305, 1227, 1223, 143}},
	{660, 13, 1165, {1, 1, 5, 9, 5, 47, 87, 239, 181, 353, 1561, 3313, 1921}},
	{661, 13, 1173, {1, 3, 3, 1, 3, 15, 53, 221, 441, 987, 1997, 2529, 8059}},
	{662, 13, 1178, {1, 1, 7, 11, 15, 57, 111, 139, 137, 883, 1881, 2823, 5661}},
	{663, 13, 1183, {1, 3, 5, 5, 21, 11, 5, 13, 27, 973, 587, 1331, 1373}},
	{664, 13, 1184, {1, 1, 7, 11, 29, 51, 93, 29, 217, 221, 55, 2477, 1979}},
	{665, 13, 1189, {1, 3, 3, 13, 3, 11, 49, 75, 379, 371, 1441, 793, 7633}},
	{666, 13, 1194, {1, 1, 1, 13, 19, 45, 89, 249, 91, 649, 1695, 915, 5619}},
	{667, 13, 1211, {1, 3, 1, 7, 7, 29, 1, 77, 313, 895, 519, 771, 295}},
	{668, 13, 1214, {1, 3, 1, 15, 5, 3, 1, 57, 331, 109, 485, 2853, 6831}},
	{669, 13, 1216, {1, 1, 1, 15, 17, 3, 35, 99, 245, 971, 839, 2509, 2803}},
	{670, 13, 1225, {1, 3, 3, 3, 9, 37, 57, 251, 325, 317, 529, 1313, 6379}},
	{671, 13, 1231, {1, 1, 1, 15, 25, 59, 1, 119, 95, 15, 795, 2375, 6463}},
	{672, 13, 1239, {1, 3, 1, 5, 1, 49, 117, 21, 47, 179, 863, 85, 1669}},
	{673, 13, 1243, {1, 3, 7, 3, 9, 37, 19, 221, 455, 973, 571, 1427, 817}},
	{674, 13, 1246, {1, 1, 1, 15, 17, 9, 67, 213, 127, 887, 1299, 2913, 7451}},
	{675, 13, 1249, {1, 3, 1, 13, 27, 27, 41, 43, 171, 623, 691, 391, 4885}},
	{676, 13, 1259, {1, 3, 1, 13, 17, 17, 123, 239, 143, 227, 1151, 519, 6543}},
	{677, 13, 1273, {1, 3, 7, 5, 7, 63, 97, 39, 101, 555, 1057, 381, 7891}},
	{678, 13, 1274, {1, 3, 5, 1, 3, 27, 85, 129, 161, 875, 1945, 3541, 695}},
	{679, 13, 1281, {1, 3, 3, 5, 21, 59, 25, 183, 35, 25, 987, 1459, 181}},
	{680, 13, 1287, {1, 3, 5, 13, 1, 15, 127, 237, 349, 337, 1491, 2383, 7811}},
	{681, 13, 1294, {1, 3, 5, 5, 31, 5, 109, 51, 409, 733, 1395, 3207, 6049}},
	{682, 13, 1296, {1, 1, 5, 7, 13, 35, 113, 25, 263, 389, 299, 2521, 1783}},
	{683, 13, 1305, {1, 3, 7, 11, 15, 47, 97, 73, 55, 75, 113, 2695, 1023}},
	{684, 13, 1306, {1, 3, 1, 1, 3, 13, 69, 211, 289, 483, 1335, 787, 677}},
	{685, 13, 1318, {1, 1, 3, 3, 17, 7, 37, 77, 505, 137, 1113, 345, 2975}},
	{686, 13, 1332, {1, 1, 1, 13, 3, 11, 95, 199, 453, 109, 479, 3725, 239}},
	{687, 13, 1335, {1, 1, 7, 15, 19, 53, 3, 145, 359, 863, 347, 3833, 3043}},
	{688, 13, 1336, {1, 1, 7, 15, 25, 63, 127, 129, 125, 195, 155, 2211, 8153}},
	{689, 13, 1341, {1, 1, 7, 13, 9, 49, 121, 115, 73, 119, 1851, 727, 47}},
	{690, 13, 1342, {1, 3, 3, 13, 13, 11, 71, 7, 45, 591, 133, 2407, 5563}},
	{691, 13, 1362, {1, 1, 1, 13, 23, 29, 87, 89, 501, 71, 1759, 1119, 687}},
	{692, 13, 1364, {1, 1, 7, 7, 13, 7, 13, 183, 53, 951, 1877, 3991, 6771}},
	{693, 13, 1368, {1, 3, 7, 11, 7, 1, 27, 47, 61, 21, 919, 961, 1091}},
	{694, 13, 1378, {1, 3, 5, 5, 1, 27, 1, 5, 63, 157, 1297, 1049, 5893}},
	{695, 13, 1387, {1, 3, 7, 9, 19, 33, 17, 133, 425, 797, 1721, 153, 119}},
	{696, 13, 1389, {1, 3, 3, 7, 13, 37, 1, 215, 509, 1003, 61, 2353, 7511}},
	{697, 13, 1397, {1, 1, 7, 1, 29, 19, 31, 79, 199, 555, 1209, 1603, 6089}},
	{698, 13, 1401, {1, 3, 1, 1, 5, 31, 111, 127, 333, 429, 1863, 3925, 5411}},
	{699, 13, 1408, {1, 1, 7, 5, 5, 5, 123, 191, 47, 993, 269, 4051, 2111}},
	{700, 13, 1418, {1, 1, 5, 15, 1, 9, 87, 5, 47, 463, 865, 1813, 7357}},
	{701, 13, 1425, {1, 3, 1, 3, 23, 63, 123, 83, 511, 777, 63, 1285, 4537}},
	{702, 13, 1426, {1, 3, 3, 7, 27, 25, 31, 65, 441, 529, 1815, 1893, 323}},
	{703, 13, 1431, {1, 3, 7, 5, 11, 19, 7, 5, 397, 811, 755, 2883, 4217}},
	{704, 13, 1435, {1, 3, 1, 13, 9, 21, 13, 7, 271, 539, 1769, 3243, 5325}},
	{705, 13, 1441, {1, 1, 7, 1, 31, 13, 47, 131, 181, 457, 1559, 2663, 6653}},
	{706, 13, 1444, {1, 3, 3, 7, 29, 55, 25, 203, 419, 91, 437, 1159, 5691}},
	{707, 13, 1462, {1, 1, 3, 13, 29, 19, 71, 217, 337, 329, 501, 939, 2205}},
	{708, 13, 1471, {1, 1, 3, 1, 1, 27, 17, 201, 97, 285, 1269, 4043, 2207}},
	{709, 13, 1474, {1, 1, 1, 1, 3, 41, 13, 199, 141, 129, 1515, 3129, 5969}},
	{710, 13, 1483, {1, 3, 3, 9, 3, 17, 119, 41, 271, 933, 877, 701, 2197}},
	{711, 13, 1485, {1, 1, 1, 7, 15, 47, 3, 195, 115, 821, 725, 843, 6071}},
	{712, 13, 1494, {1, 3, 5, 15, 17, 33, 85, 65, 297, 571, 1123, 2743, 5727}},
	{713, 13, 1497, {1, 1, 5, 11, 27, 15, 37, 235, 415, 293, 1439, 2739, 4171}},
	{714, 13, 1516, {1, 3, 7, 7, 1, 55, 71, 35, 307, 11, 401, 1881, 933}},
	{715, 13, 1522, {1, 3, 1, 11, 21, 37, 3, 177, 119, 339, 559, 3991, 3437}},
	{716, 13, 1534, {1, 3, 3, 9, 17, 17, 97, 119, 301, 169, 157, 3267, 2261}},
	{717, 13, 1543, {1, 3, 3, 9, 29, 3, 111, 101, 355, 869, 375, 2609, 7377}},
	{718, 13, 1552, {1, 3, 5, 9, 7, 21, 123, 99, 343, 693, 1927, 1605, 4923}},
	{719, 13, 1557, {1, 1, 3, 5, 13, 31, 99, 17, 75, 385, 1539, 1553, 7077}},
	{720, 13, 1558, {1, 3, 3, 5, 31, 35, 107, 11, 407, 1019, 1317, 3593, 7203}},
	{721, 13, 1567, {1, 3, 3, 13, 17, 33, 99, 245, 401, 957, 157, 1949, 1571}},
	{722, 13, 1568, {1, 3, 1, 11, 27, 15, 11, 109, 429, 307, 1911, 2701, 861}},
	{723, 13, 1574, {1, 1, 5, 13, 13, 35, 55, 255, 311, 957, 1803, 2673, 5195}},
	{724, 13, 1592, {1, 1, 1, 11, 19, 3, 89, 37, 211, 783, 1355, 3567, 7135}},
	{725, 13, 1605, {1, 1, 5, 5, 21, 49, 79, 17, 509, 331, 183, 3831, 855}},
	{726, 13, 1606, {1, 3, 7, 5, 29, 19, 85, 109, 105, 523, 845, 3385, 7477}},
	{727, 13, 1610, {1, 1, 1, 7, 25, 17, 125, 131, 53, 757, 253, 2989, 2939}},
	{728, 13, 1617, {1, 3, 3, 9, 19, 23, 105, 39, 351, 677, 211, 401, 8103}},
	{729, 13, 1623, {1, 3, 5, 1, 5, 11, 17, 3, 405, 469, 1569, 2865, 3133}},
	{730, 13, 1630, {1, 1, 3, 13, 15, 5, 117, 179, 139, 145, 477, 1137, 2537}},
	{731, 13, 1634, {1, 1, 7, 9, 5, 21, 9, 93, 211, 963, 1207, 3343, 4911}},
	{732, 13, 1640, {1, 1, 1, 9, 13, 43, 17, 53, 81, 793, 1571, 2523, 3683}},
	{733, 13, 1643, {1, 3, 3, 13, 25, 21, 5, 59, 489, 987, 1941, 171, 6009}},
	{734, 13, 1648, {1, 3, 3, 7, 1, 39, 89, 171, 403, 467, 1767, 3423, 2791}},
	{735, 13, 1651, {1, 1, 3, 9, 19, 49, 91, 125, 163, 1013, 89, 2849, 6785}},
	{736, 13, 1653, {1, 1, 5, 9, 9, 11, 15, 241, 43, 297, 1719, 1541, 1821}},
	{737, 13, 1670, {1, 3, 7, 15, 29, 23, 103, 239, 191, 33, 1043, 3649, 6579}},
	{738, 13, 1676, {1, 3, 3, 9, 21, 51, 123, 55, 223, 645, 1463, 4021, 5891}},
	{739, 13, 1684, {1, 1, 5, 7, 3, 41, 27, 235, 391, 303, 2021, 3187, 7607}},
	{740, 13, 1687, {1, 1, 1, 9, 5, 49, 49, 29, 377, 251, 1887, 1017, 1301}},
	{741, 13, 1691, {1, 1, 3, 3, 13, 41, 27, 47, 223, 23, 517, 3227, 6731}},
	{742, 13, 1693, {1, 1, 7, 1, 31, 25, 47, 9, 511, 623, 2047, 1263, 1511}},
	{743, 13, 1698, {1, 1, 3, 15, 15, 23, 53, 1, 261, 595, 85, 241, 7047}},
	{744, 13, 1709, {1, 3, 3, 11, 17, 5, 81, 73, 149, 781, 2035, 3163, 4247}},
	{745, 13, 1715, {1, 3, 7, 7, 29, 59, 49, 79, 397, 901, 1105, 2191, 6277}},
	{746, 13, 1722, {1, 3, 3, 11, 13, 27, 25, 173, 107, 73, 1265, 585, 5251}},
	{747, 13, 1732, {1, 1, 7, 15, 29, 23, 73, 229, 235, 887, 1469, 4073, 2591}},
	{748, 13, 1735, {1, 1, 3, 9, 17, 15, 83, 173, 207, 879, 1701, 1509, 11}},
	{749, 13, 1747, {1, 1, 3, 5, 5, 37, 65, 161, 39, 421, 1153, 2007, 5355}},
	{750, 13, 1749, {1, 1, 7, 11, 23, 37, 5, 11, 9, 499, 17, 157, 5747}},
	{751, 13, 1754, {1, 3, 7, 13, 25, 9, 49, 7, 39, 945, 1349, 1759, 1441}},
	{752, 13, 1777, {1, 1, 5, 3, 21, 15, 113, 81, 265, 837, 333, 3625, 6133}},
	{753, 13, 1784, {1, 3, 1, 11, 13, 27, 73, 109, 297, 327, 299, 3253, 6957}},
	{754, 13, 1790, {1, 1, 3, 13, 19, 39, 123, 73, 65, 5, 1061, 2187, 5055}},
	{755, 13, 1795, {1, 1, 3, 1, 11, 31, 21, 115, 453, 857, 711, 495, 549}},
	{756, 13, 1801, {1, 3, 7, 7, 15, 29, 79, 103, 47, 713, 1735, 3121, 6321}},
	{757, 13, 1802, {1, 1, 5, 5, 29, 9, 97, 33, 471, 705, 329, 1501, 1349}},
	{758, 13, 1812, {1, 3, 3, 1, 21, 9, 111, 209, 71, 47, 491, 2143, 1797}},
	{759, 13, 1828, {1, 3, 3, 3, 11, 39, 21, 135, 445, 259, 607, 3811, 5449}},
	{760, 13, 1831, {1, 1, 7, 9, 11, 25, 113, 251, 395, 317, 317, 91, 1979}},
	{761, 13, 1837, {1, 3, 1, 9, 3, 21, 103, 133, 389, 943, 1235, 1749, 7063}},
	{762, 13, 1838, {1, 1, 3, 7, 1, 11, 5, 15, 497, 477, 479, 3079, 6969}},
	{763, 13, 1840, {1, 1, 3, 3, 15, 39, 105, 131, 475, 465, 181, 865, 3813}},
	{764, 13, 1845, {1, 1, 7, 9, 19, 63, 123, 131, 415, 525, 457, 2471, 3135}},
	{765, 13, 1863, {1, 3, 7, 15, 25, 35, 123, 45, 341, 805, 485, 4049, 7065}},
	{766, 13, 1864, {1, 1, 1, 5, 29, 9, 47, 227, 51, 867, 1873, 1593, 2271}},
	{767, 13, 1867, {1, 1, 7, 15, 31, 9, 71, 117, 285, 711, 837, 1435, 6275}},
	{768, 13, 1870, {1, 3, 1, 1, 5, 19, 79, 25, 301, 415, 1871, 645, 3251}},
	{769, 13, 1877, {1, 3, 1, 3, 17, 51, 99, 185, 447, 43, 523, 219, 429}},
	{770, 13, 1881, {1, 3, 1, 13, 29, 13, 51, 93, 7, 995, 757, 3017, 6865}},
	{771, 13, 1884, {1, 1, 3, 15, 7, 25, 75, 17, 155, 981, 1231, 1229, 1995}},
	{772, 13, 1903, {1, 3, 5, 3, 27, 45, 71, 73, 225, 763, 377, 1139, 2863}},
	{773, 13, 1917, {1, 1, 3, 1, 1, 39, 69, 113, 29, 371, 1051, 793, 3749}},
	{774, 13, 1918, {1, 1, 3, 13, 23, 61, 27, 183, 307, 431, 1345, 2757, 4031}},
	{775, 13, 1922, {1, 3, 7, 5, 5, 59, 117, 197, 303, 721, 877, 723, 1601}},
	{776, 13, 1924, {1, 3, 5, 1, 27, 33, 99, 237, 485, 711, 665, 3077, 5105}},
	{777, 13, 1928, {1, 1, 3, 1, 13, 9, 103, 201, 23, 951, 2029, 165, 2093}},
	{778, 13, 1931, {1, 3, 5, 13, 5, 29, 55, 85, 221, 677, 611, 3613, 4567}},
	{779, 13, 1951, {1, 1, 1, 1, 7, 61, 9, 233, 261, 561, 953, 4023, 2443}},
	{780, 13, 1952, {1, 3, 3, 13, 1, 17, 103, 71, 223, 213, 833, 1747, 6999}},
	{781, 13, 1957, {1, 3, 5, 15, 25, 53, 57, 187, 25, 695, 1207, 4089, 2877}},
	{782, 13, 1958, {1, 1, 7, 1, 7, 31, 87, 129, 493, 519, 1555, 1155, 4637}},
	{783, 13, 1964, {1, 1, 1, 15, 21, 17, 23, 29, 19, 255, 927, 1791, 3093}},
	{784, 13, 1967, {1, 1, 3, 9, 17, 33, 95, 129, 175, 461, 287, 2633, 2325}},
	{785, 13, 1970, {1, 3, 5, 7, 23, 19, 63, 209, 249, 583, 1373, 2039, 2225}},
	{786, 13, 1972, {1, 3, 3, 5, 5, 19, 79, 241, 459, 355, 1455, 3313, 3639}},
	{787, 13, 1994, {1, 1, 7, 9, 21, 41, 97, 119, 129, 769, 1541, 3495, 7741}},
	{788, 13, 2002, {1, 1, 7, 11, 9, 29, 35, 255, 141, 937, 1763, 41, 1393}},
	{789, 13, 2007, {1, 3, 7, 1, 13, 51, 61, 157, 177, 847, 1829, 3539, 285}},
	{790, 13, 2008, {1, 1, 1, 15, 21, 13, 9, 55, 397, 19, 1495, 1255, 7235}},
	{791, 13, 2023, {1, 1, 7, 7, 25, 37, 53, 237, 319, 197, 269, 1205, 1485}},
	{792, 13, 2030, {1, 1, 5, 15, 23, 17, 35, 247, 323, 807, 233, 3681, 4407}},
	{793, 13, 2035, {1, 1, 3, 7, 9, 59, 85, 105, 493, 763, 1639, 391, 1451}},
	{794, 13, 2038, {1, 3, 3, 9, 15, 33, 5, 253, 129, 625, 1527, 2793, 6057}},
	{795, 13, 2042, {1, 3, 1, 1, 7, 47, 21, 161, 235, 83, 397, 3563, 5953}},
	{796, 13, 2047, {1, 3, 7, 11, 3, 41, 25, 117, 375, 779, 1297, 3715, 8117}},
	{797, 13, 2051, {1, 1, 3, 7, 31, 19, 103, 173, 475, 189, 2035, 2921, 1107}},
	{798, 13, 2058, {1, 1, 7, 3, 25, 7, 93, 255, 307, 113, 1893, 2233, 6919}},
	{799, 13, 2060, {1, 3, 5, 15, 9, 57, 79, 143, 165, 5, 1389, 193, 693}},
	{800, 13, 2071, {1, 3, 5, 1, 29, 45, 91, 49, 189, 461, 439, 1283, 7835}},
	{801, 13, 2084, {1, 1, 3, 13, 11, 61, 41, 231, 373, 695, 395, 915, 5393}},
	{802, 13, 2087, {1, 3, 7, 11, 5, 51, 67, 53, 483, 95, 1943, 247, 5653}},
	{803, 13, 2099, {1, 3, 7, 5, 5, 57, 45, 235, 137, 793, 1069, 1661, 1557}},
	{804, 13, 2108, {1, 3, 5, 3, 25, 55, 103, 177, 81, 861, 1151, 143, 7655}},
	{805, 13, 2111, {1, 1, 3, 1, 21, 41, 67, 131, 253, 431, 1269, 3181, 3429}},
	{806, 13, 2120, {1, 3, 1, 1, 21, 7, 77, 221, 257, 663, 71, 2949, 2481}},
	{807, 13, 2128, {1, 3, 5, 3, 3, 23, 45, 107, 299, 739, 1013, 3, 3165}},
	{808, 13, 2138, {1, 1, 5, 1, 3, 37, 109, 37, 243, 983, 1221, 1691, 3869}},
	{809, 13, 2143, {1, 1, 5, 5, 31, 7, 5, 193, 397, 867, 1495, 3435, 7441}},
	{810, 13, 2144, {1, 1, 1, 1, 17, 59, 97, 233, 389, 597, 1013, 1631, 483}},
	{811, 13, 2153, {1, 1, 1, 11, 7, 41, 107, 53, 111, 125, 1513, 1921, 7647}},
	{812, 13, 2156, {1, 3, 3, 3, 31, 29, 117, 3, 365, 971, 1139, 2123, 5913}},
	{813, 13, 2162, {1, 1, 1, 13, 23, 3, 1, 167, 475, 639, 1811, 3841, 3081}},
	{814, 13, 2167, {1, 1, 5, 3, 5, 47, 65, 123, 275, 783, 95, 119, 7591}},
	{815, 13, 2178, {1, 3, 1, 15, 13, 33, 93, 237, 467, 431, 705, 4013, 4035}},
	{816, 13, 2183, {1, 3, 5, 1, 19, 7, 101, 231, 155, 737, 1381, 3343, 2051}},
	{817, 13, 2202, {1, 1, 5, 9, 15, 49, 45, 163, 433, 765, 2031, 201, 2589}},
	{818, 13, 2211, {1, 3, 7, 9, 19, 41, 31, 89, 93, 623, 105, 745, 4409}},
	{819, 13, 2214, {1, 1, 5, 1, 11, 45, 127, 85, 389, 439, 829, 477, 7965}},
	{820, 13, 2223, {1, 3, 3, 15, 13, 41, 1, 207, 435, 585, 311, 1725, 2737}},
	{821, 13, 2225, {1, 3, 3, 3, 13, 49, 21, 31, 197, 799, 1411, 2959, 7133}},
	{822, 13, 2232, {1, 3, 1, 3, 7, 43, 9, 141, 133, 579, 1059, 93, 957}},
	{823, 13, 2237, {1, 3, 7, 1, 15, 51, 23, 213, 381, 851, 699, 2261, 3419}},
	{824, 13, 2257, {1, 3, 5, 9, 25, 35, 67, 141, 35, 409, 1423, 365, 1645}},
	{825, 13, 2260, {1, 3, 3, 11, 15, 33, 27, 181, 93, 87, 1761, 3511, 1353}},
	{826, 13, 2267, {1, 3, 5, 3, 25, 63, 111, 137, 321, 819, 705, 1547, 7271}},
	{827, 13, 2274, {1, 3, 1, 1, 5, 57, 99, 59, 411, 757, 1371, 3953, 3695}},
	{828, 13, 2276, {1, 3, 5, 11, 11, 21, 25, 147, 239, 455, 709, 953, 7175}},
	{829, 13, 2285, {1, 3, 3, 15, 5, 53, 91, 205, 341, 63, 723, 1565, 7135}},
	{830, 13, 2288, {1, 1, 7, 15, 11, 21, 99, 79, 63, 593, 2007, 3629, 5271}},
	{831, 13, 2293, {1, 3, 3, 1, 9, 21, 45, 175, 453, 435, 1855, 2649, 6959}},
	{832, 13, 2294, {1, 1, 3, 15, 15, 33, 121, 121, 251, 431, 1127, 3305, 4199}},
	{833, 13, 2297, {1, 1, 1, 9, 31, 15, 71, 29, 345, 391, 1159, 2809, 345}},
	{834, 13, 2303, {1, 3, 7, 1, 23, 29, 95, 151, 327, 727, 647, 1623, 2971}},
	{835, 13, 2308, {1, 1, 7, 7, 9, 29, 79, 91, 127, 909, 1293, 1315, 5315}},
	{836, 13, 2311, {1, 1, 5, 11, 13, 37, 89, 73, 149, 477, 1909, 3343, 525}},
	{837, 13, 2318, {1, 3, 5, 7, 5, 59, 55, 255, 223, 459, 2027, 237, 4205}},
	{838, 13, 2323, {1, 1, 1, 7, 27, 11, 95, 65, 325, 835, 907, 3801, 3787}},
	{839, 13, 2332, {1, 1, 1, 11, 27, 33, 99, 175, 51, 913, 331, 1851, 4133}},
	{840, 13, 2341, {1, 3, 5, 5, 13, 37, 31, 99, 273, 409, 1827, 3845, 5491}},
	{841, 13, 2345, {1, 1, 3, 7, 23, 19, 107, 85, 283, 523, 509, 451, 421}},
	{842, 13, 2348, {1, 3, 5, 7, 13, 9, 51, 81, 87, 619, 61, 2803, 5271}},
	{843, 13, 2354, {1, 1, 1, 15, 9, 45, 35, 219, 401, 271, 953, 649, 6847}},
	{844, 13, 2368, {1, 1, 7, 11, 9, 45, 17, 219, 169, 837, 1483, 1605, 2901}},
	{845, 13, 2377, {1, 1, 7, 7, 21, 43, 37, 33, 291, 359, 71, 2899, 7037}},
	{846, 13, 2380, {1, 3, 3, 13, 31, 53, 37, 15, 149, 949, 551, 3445, 5455}},
	{847, 13, 2383, {1, 3, 1, 5, 19, 45, 81, 223, 193, 439, 2047, 3879, 789}},
	{848, 13, 2388, {1, 1, 7, 3, 11, 63, 35, 61, 255, 563, 459, 2991, 3359}},
	{849, 13, 2395, {1, 1, 5, 9, 13, 49, 47, 185, 239, 221, 1533, 3635, 2045}},
	{850, 13, 2397, {1, 3, 7, 3, 25, 37, 127, 223, 51, 357, 483, 3837, 6873}},
	{851, 13, 2401, {1, 1, 7, 9, 31, 37, 113, 31, 387, 833, 1243, 1543, 5535}},
	{852, 13, 2411, {1, 3, 1, 9, 23, 59, 119, 221, 73, 185, 2007, 2885, 2563}},
	{853, 13, 2413, {1, 1, 1, 13, 7, 33, 53, 179, 67, 185, 1541, 1807, 4659}},
	{854, 13, 2419, {1, 3, 1, 11, 31, 37, 23, 215, 269, 357, 207, 645, 4219}},
	{855, 13, 2435, {1, 3, 3, 13, 19, 27, 107, 55, 91, 71, 1695, 1815, 89}},
	{856, 13, 2442, {1, 1, 3, 15, 3, 19, 35, 247, 49, 529, 1523, 3317, 6151}},
	{857, 13, 2455, {1, 1, 7, 7, 23, 25, 107, 139, 483, 503, 1277, 243, 7879}},
	{858, 13, 2472, {1, 3, 3, 13, 3, 15, 11, 197, 135, 839, 985, 275, 5527}},
	{859, 13, 2478, {1, 3, 5, 3, 25, 47, 95, 21, 113, 307, 1001, 3065, 295}},
	{860, 13, 2490, {1, 1, 3, 9, 19, 19, 99, 213, 363, 449, 735, 2851, 2521}},
	{861, 13, 2507, {1, 1, 3, 9, 5, 49, 63, 61, 157, 857, 497, 2801, 6987}},
	{862, 13, 2509, {1, 1, 1, 9, 1, 41, 109, 119, 499, 939, 867, 3675, 8023}},
	{863, 13, 2517, {1, 3, 1, 1, 13, 33, 109, 123, 289, 3, 1271, 2773, 4265}},
	{864, 13, 2524, {1, 3, 1, 11, 9, 57, 83, 221, 95, 43, 1189, 457, 7133}},
	{865, 13, 2528, {1, 1, 7, 3, 11, 49, 33, 219, 229, 289, 685, 3359, 4495}},
	{866, 13, 2531, {1, 3, 1, 3, 19, 43, 67, 193, 41, 771, 407, 81, 3891}},
	{867, 13, 2538, {1, 1, 7, 11, 5, 29, 51, 175, 297, 539, 1, 2245, 6439}},
	{868, 13, 2545, {1, 3, 7, 15, 21, 33, 117, 183, 511, 489, 1283, 3281, 5979}},
	{869, 13, 2546, {1, 3, 7, 5, 9, 3, 125, 147, 359, 549, 369, 3049, 2405}},
	{870, 13, 2555, {1, 3, 5, 7, 19, 5, 65, 97, 483, 377, 1523, 1457, 2995}},
	{871, 13, 2557, {1, 1, 5, 1, 11, 21, 41, 113, 277, 131, 1475, 1043, 2367}},
	{872, 13, 2564, {1, 3, 3, 1, 15, 17, 101, 69, 443, 865, 817, 1421, 5231}},
	{873, 13, 2573, {1, 1, 3, 3, 3, 55, 95, 99, 75, 195, 1929, 3931, 5855}},
	{874, 13, 2579, {1, 3, 1, 3, 19, 23, 93, 213, 241, 551, 1307, 585, 7729}},
	{875, 13, 2592, {1, 3, 1, 11, 23, 15, 53, 249, 467, 519, 95, 741, 409}},
	{876, 13, 2598, {1, 1, 1, 15, 29, 37, 43, 203, 233, 877, 77, 1933, 2729}},
	{877, 13, 2607, {1, 3, 7, 11, 27, 39, 43, 161, 255, 15, 1463, 833, 495}},
	{878, 13, 2612, {1, 1, 7, 11, 3, 53, 81, 67, 375, 823, 1903, 3061, 395}},
	{879, 13, 2619, {1, 1, 1, 1, 15, 37, 93, 233, 247, 501, 1321, 3275, 5409}},
	{880, 13, 2621, {1, 3, 3, 7, 7, 11, 5, 105, 139, 983, 1239, 531, 3881}},
	{881, 13, 2627, {1, 1, 5, 3, 19, 49, 107, 227, 361, 101, 355, 2649, 7383}},
	{882, 13, 2633, {1, 1, 7, 5, 25, 41, 101, 121, 209, 293, 1937, 2259, 5557}},
	{883, 13, 2636, {1, 1, 3, 7, 7, 1, 9, 13, 463, 1019, 995, 3159, 107}},
	{884, 13, 2642, {1, 3, 5, 11, 5, 35, 127, 97, 261, 789, 807, 807, 6257}},
	{885, 13, 2654, {1, 1, 7, 5, 11, 13, 45, 91, 417, 101, 1973, 3645, 2107}},
	{886, 13, 2660, {1, 1, 3, 7, 5, 63, 57, 49, 203, 157, 115, 1393, 8117}},
	{887, 13, 2669, {1, 3, 5, 5, 3, 43, 15, 155, 127, 489, 1165, 3701, 4867}},
	{888, 13, 2675, {1, 1, 7, 7, 29, 29, 69, 215, 415, 367, 371, 1901, 6075}},
	{889, 13, 2684, {1, 1, 1, 3, 11, 33, 89, 149, 433, 705, 1437, 1597, 505}},
	{890, 13, 2694, {1, 3, 5, 1, 13, 37, 19, 119, 5, 581, 2037, 1633, 2099}},
	{891, 13, 2703, {1, 3, 7, 13, 5, 49, 103, 245, 215, 515, 133, 2007, 1933}},
	{892, 13, 2706, {1, 3, 1, 9, 1, 3, 25, 197, 253, 387, 1683, 2267, 221}},
	{893, 13, 2712, {1, 3, 5, 15, 21, 9, 73, 201, 405, 999, 437, 3877, 6045}},
	{894, 13, 2715, {1, 1, 3, 1, 31, 55, 25, 83, 421, 395, 1807, 2129, 7797}},
	{895, 13, 2722, {1, 1, 3, 1, 23, 21, 121, 183, 125, 347, 143, 3685, 4317}},
	{896, 13, 2727, {1, 3, 3, 3, 17, 45, 17, 223, 267, 795, 1815, 1309, 155}},
	{897, 13, 2734, {1, 1, 1, 15, 17, 59, 5, 133, 15, 715, 1503, 153, 2887}},
	{898, 13, 2742, {1, 1, 1, 1, 27, 13, 119, 77, 243, 995, 1851, 3719, 4695}},
	{899, 13, 2745, {1, 3, 1, 5, 31, 49, 43, 165, 49, 609, 1265, 1141, 505}},
	{900, 13, 2751, {1, 1, 7, 13, 11, 63, 21, 253, 229, 585, 1543, 3719, 4141}},
	{901, 13, 2766, {1, 3, 7, 11, 23, 27, 17, 131, 295, 895, 1493, 1411, 3247}},
	{902, 13, 2768, {1, 1, 5, 9, 29, 7, 97, 15, 113, 445, 859, 1483, 1121}},
	{903, 13, 2780, {1, 3, 1, 9, 13, 49, 99, 107, 323, 201, 681, 3071, 5281}},
	{904, 13, 2790, {1, 1, 1, 15, 9, 19, 61, 161, 7, 87, 587, 2199, 2811}},
	{905, 13, 2794, {1, 3, 3, 15, 15, 19, 95, 45, 299, 829, 981, 3479, 487}},
	{906, 13, 2796, {1, 1, 1, 9, 3, 37, 7, 19, 227, 13, 397, 513, 1257}},
	{907, 13, 2801, {1, 1, 5, 15, 15, 13, 17, 111, 135, 929, 1145, 811, 1801}},
	{908, 13, 2804, {1, 3, 1, 3, 27, 57, 31, 19, 279, 103, 693, 631, 3409}},
	{909, 13, 2807, {1, 1, 1, 1, 15, 13, 67, 83, 23, 799, 1735, 2063, 3363}},
	{910, 13, 2816, {1, 3, 3, 7, 3, 1, 61, 31, 41, 533, 2025, 4067, 6963}},
	{911, 13, 2821, {1, 1, 5, 7, 17, 27, 81, 79, 107, 205, 29, 97, 4883}},
	{912, 13, 2831, {1, 1, 1, 5, 19, 49, 91, 201, 283, 949, 651, 3819, 5073}},
	{913, 13, 2834, {1, 1, 7, 9, 11, 13, 73, 197, 37, 219, 1931, 3369, 6017}},
	{914, 13, 2839, {1, 1, 7, 15, 11, 7, 75, 205, 7, 819, 399, 661, 6487}},
	{915, 13, 2845, {1, 3, 3, 3, 27, 37, 95, 41, 307, 165, 1077, 3485, 563}},
	{916, 13, 2852, {1, 3, 5, 3, 21, 49, 57, 179, 109, 627, 1789, 431, 2941}},
	{917, 13, 2856, {1, 1, 7, 5, 11, 19, 43, 137, 149, 679, 1543, 245, 1381}},
	{918, 13, 2861, {1, 3, 5, 5, 15, 3, 69, 81, 135, 159, 1363, 3401, 6355}},
	{919, 13, 2873, {1, 3, 5, 1, 9, 61, 49, 53, 319, 25, 1647, 1297, 615}},
	{920, 13, 2874, {1, 3, 5, 11, 31, 43, 9, 101, 71, 919, 335, 3147, 5823}},
	{921, 13, 2888, {1, 3, 1, 1, 15, 5, 29, 109, 511, 945, 867, 3677, 6915}},
	{922, 13, 2893, {1, 3, 3, 15, 17, 49, 91, 111, 215, 29, 1879, 97, 2505}},
	{923, 13, 2894, {1, 3, 1, 13, 19, 61, 11, 111, 163, 777, 533, 1113, 5339}},
	{924, 13, 2902, {1, 1, 7, 9, 17, 55, 117, 91, 455, 289, 557, 913, 4455}},
	{925, 13, 2917, {1, 3, 1, 7, 25, 19, 123, 37, 1, 277, 717, 2965, 4469}},
	{926, 13, 2921, {1, 3, 7, 3, 19, 23, 87, 235, 209, 457, 2041, 2893, 1805}},
	{927, 13, 2922, {1, 3, 3, 5, 5, 43, 23, 61, 351, 791, 59, 2009, 2909}},
	{928, 13, 2929, {1, 1, 3, 7, 5, 1, 27, 231, 385, 257, 1261, 2701, 1807}},
	{929, 13, 2935, {1, 3, 1, 1, 27, 19, 87, 253, 131, 685, 1743, 3983, 2651}},
	{930, 13, 2946, {1, 3, 7, 11, 21, 17, 11, 81, 191, 641, 1821, 3005, 7251}},
	{931, 13, 2951, {1, 3, 3, 5, 15, 31, 41, 213, 55, 931, 1953, 49, 6037}},
	{932, 13, 2957, {1, 1, 7, 15, 7, 27, 65, 223, 113, 79, 1875, 911, 5445}},
	{933, 13, 2960, {1, 3, 7, 7, 23, 55, 51, 167, 495, 25, 1585, 3447, 799}},
	{934, 13, 2966, {1, 1, 3, 7, 27, 15, 95, 193, 337, 415, 975, 3085, 967}},
	{935, 13, 2972, {1, 1, 7, 15, 19, 7, 93, 41, 433, 551, 401, 3169, 3971}},
	{936, 13, 2976, {1, 1, 7, 11, 13, 15, 53, 69, 433, 59, 1117, 3359, 6231}},
	{937, 13, 2979, {1, 1, 7, 3, 23, 5, 115, 201, 225, 109, 1903, 3897, 6265}},
	{938, 13, 2985, {1, 1, 1, 11, 17, 1, 39, 143, 361, 659, 1105, 23, 4923}},
	{939, 13, 3000, {1, 1, 1, 9, 27, 57, 85, 227, 261, 119, 1881, 3965, 6999}},
	{940, 13, 3003, {1, 3, 7, 7, 15, 7, 107, 17, 315, 49, 1591, 905, 7789}},
	{941, 13, 3013, {1, 3, 1, 7, 29, 3, 47, 237, 157, 769, 839, 3199, 3195}},
	{942, 13, 3018, {1, 1, 3, 15, 25, 39, 63, 15, 111, 857, 881, 1505, 7671}},
	{943, 13, 3020, {1, 1, 7, 1, 3, 35, 41, 215, 99, 895, 1025, 1483, 4707}},
	{944, 13, 3025, {1, 3, 5, 1, 1, 31, 25, 247, 113, 841, 397, 1825, 6969}},
	{945, 13, 3042, {1, 1, 3, 5, 19, 41, 49, 243, 225, 973, 241, 175, 1041}},
	{946, 13, 3047, {1, 1, 1, 7, 15, 15, 105, 141, 83, 75, 1675, 3523, 5219}},
	{947, 13, 3048, {1, 1, 7, 5, 13, 27, 47, 199, 445, 841, 959, 1157, 2209}},
	{948, 13, 3051, {1, 3, 5, 15, 23, 31, 31, 81, 85, 33, 785, 2639, 7799}},
	{949, 13, 3054, {1, 1, 5, 13, 21, 3, 47, 99, 235, 943, 1731, 2467, 7891}},
	{950, 13, 3056, {1, 1, 1, 3, 17, 53, 85, 219, 73, 131, 1339, 875, 1191}},
	{951, 13, 3065, {1, 1, 5, 7, 17, 63, 113, 7, 185, 557, 749, 3563, 4973}},
	{952, 13, 3073, {1, 3, 3, 15, 15, 21, 43, 111, 155, 689, 345, 423, 3597}},
	{953, 13, 3074, {1, 1, 5, 1, 15, 29, 93, 5, 361, 713, 695, 3937, 425}},
	{954, 13, 3083, {1, 3, 7, 7, 13, 41, 115, 175, 315, 937, 123, 2841, 4457}},
	{955, 13, 3086, {1, 1, 3, 11, 25, 5, 103, 53, 423, 811, 657, 399, 7257}},
	{956, 13, 3091, {1, 1, 1, 1, 1, 13, 101, 211, 383, 325, 97, 1703, 4429}},
	{957, 13, 3097, {1, 3, 7, 9, 31, 45, 83, 157, 509, 701, 841, 1105, 3643}},
	{958, 13, 3109, {1, 1, 1, 7, 1, 9, 69, 17, 129, 281, 1161, 2945, 7693}},
	{959, 13, 3116, {1, 3, 7, 1, 11, 29, 51, 143, 77, 433, 1723, 2317, 5641}},
	{960, 13, 3124, {1, 1, 1, 1, 21, 43, 13, 67, 177, 505, 1629, 1267, 4885}},
	{961, 13, 3128, {1, 1, 3, 11, 27, 63, 111, 47, 233, 781, 453, 1679, 3209}},
	{962, 13, 3153, {1, 1, 3, 13, 29, 27, 119, 141, 493, 971, 461, 1159, 633}},
	{963, 13, 3160, {1, 1, 3, 15, 23, 5, 79, 215, 163, 149, 1805, 2399, 61}},
	{964, 13, 3165, {1, 3, 5, 13, 19, 5, 1, 39, 409, 561, 709, 829, 1357}},
	{965, 13, 3172, {1, 3, 3, 13, 19, 43, 9, 177, 449, 447, 73, 2107, 5669}},
	{966, 13, 3175, {1, 3, 5, 1, 23, 13, 63, 109, 203, 593, 829, 4017, 6881}},
	{967, 13, 3184, {1, 1, 5, 7, 3, 9, 53, 175, 391, 169, 1283, 3793, 4451}},
	{968, 13, 3193, {1, 1, 5, 7, 29, 43, 9, 5, 209, 77, 927, 2941, 8145}},
	{969, 13, 3196, {1, 3, 5, 15, 17, 49, 5, 143, 131, 771, 1685, 925, 2175}},
	{970, 13, 3200, {1, 1, 3, 11, 27, 27, 27, 159, 161, 1015, 1587, 4049, 1983}},
	{971, 13, 3203, {1, 3, 1, 3, 23, 57, 119, 67, 481, 577, 389, 3319, 5325}},
	{972, 13, 3205, {1, 3, 5, 1, 19, 39, 87, 61, 329, 657, 1773, 31, 1707}},
	{973, 13, 3209, {1, 1, 3, 1, 5, 25, 15, 241, 131, 815, 1751, 3029, 8039}},
	{974, 13, 3224, {1, 3, 3, 13, 27, 13, 77, 87, 437, 57, 621, 1031, 7891}},
	{975, 13, 3239, {1, 3, 1, 13, 23, 51, 117, 37, 331, 745, 605, 3179, 4713}},
	{976, 13, 3251, {1, 1, 5, 5, 19, 17, 99, 167, 87, 721, 737, 789, 2165}},
	{977, 13, 3254, {1, 3, 5, 13, 1, 51, 119, 211, 165, 299, 1327, 3053, 3343}},
	{978, 13, 3265, {1, 1, 5, 15, 29, 45, 17, 129, 67, 345, 1553, 2705, 7369}},
	{979, 13, 3266, {1, 1, 1, 9, 23, 7, 13, 209, 7, 407, 317, 3077, 7287}},
	{980, 13, 3275, {1, 1, 1, 5, 9, 59, 89, 3, 487, 451, 505, 2499, 7563}},
	{981, 13, 3280, {1, 3, 1, 7, 21, 1, 21, 203, 101, 417, 1389, 2751, 1397}},
	{982, 13, 3283, {1, 3, 7, 13, 7, 31, 3, 247, 349, 485, 1259, 549, 6321}},
	{983, 13, 3286, {1, 1, 7, 7, 27, 33, 107, 197, 293, 729, 1753, 2571, 103}},
	{984, 13, 3301, {1, 3, 5, 9, 25, 35, 5, 253, 137, 213, 2041, 3387, 1809}},
	{985, 13, 3302, {1, 1, 7, 13, 15, 35, 67, 83, 295, 175, 839, 2831, 839}},
	{986, 13, 3305, {1, 3, 3, 11, 3, 17, 55, 141, 247, 991, 117, 3799, 1221}},
	{987, 13, 3319, {1, 1, 5, 1, 11, 37, 87, 233, 457, 653, 899, 2933, 3105}},
	{988, 13, 3323, {1, 1, 3, 15, 3, 31, 67, 167, 437, 9, 651, 1109, 1139}},
	{989, 13, 3326, {1, 1, 3, 1, 7, 63, 67, 17, 11, 883, 1855, 1941, 4751}},
	{990, 13, 3331, {1, 3, 7, 9, 19, 33, 113, 117, 495, 39, 1795, 2561, 5519}},
	{991, 13, 3348, {1, 1, 7, 5, 1, 3, 103, 37, 201, 223, 1101, 877, 6483}},
	{992, 13, 3351, {1, 1, 5, 9, 29, 49, 51, 33, 439, 917, 861, 1321, 2135}},
	{993, 13, 3358, {1, 1, 3, 3, 1, 5, 17, 93, 217, 619, 613, 1357, 6095}},
	{994, 13, 3368, {1, 3, 1, 11, 3, 21, 5, 41, 15, 175, 843, 2937, 6849}},
	{995, 13, 3374, {1, 3, 3, 7, 9, 57, 55, 127, 79, 287, 445, 2205, 7989}},
	{996, 13, 3376, {1, 1, 7, 13, 23, 17, 93, 129, 157, 135, 1747, 1813, 4183}},
	{997, 13, 3379, {1, 1, 1, 5, 31, 59, 99, 33, 425, 329, 887, 367, 1761}},
	{998, 13, 3385, {1, 1, 7, 9, 17, 53, 77, 139, 435, 387, 49, 3649, 1773}},
	{999, 13, 3386, {1, 3, 3, 15, 21, 57, 45, 161, 331, 719, 273, 3479, 4173}},
	{1000, 13, 3396, {1, 1, 3, 9, 3, 3, 105, 201, 373, 877, 919, 1263, 6649}},
	{1001, 13, 3420, {1, 3, 1, 15, 13, 43, 13, 99, 73, 163, 353, 3569, 5601}},
	{1002, 13, 3423, {1, 3, 7, 3, 5, 9, 69, 177, 449, 47, 781, 1125, 4245}},
	{1003, 13, 3430, {1, 1, 1, 5, 3, 45, 1, 123, 409, 903, 205, 2057, 7637}},
	{1004, 13, 3433, {1, 3, 5, 9, 19, 47, 87, 135, 481, 799, 101, 3409, 2241}},
	{1005, 13, 3434, {1, 3, 1, 13, 3, 25, 15, 27, 181, 967, 669, 2577, 7249}},
	{1006, 13, 3439, {1, 1, 7, 3, 31, 5, 103, 53, 1, 911, 1209, 3697, 6685}},
	{1007, 13, 3442, {1, 1, 3, 1, 5, 5, 49, 135, 281, 747, 761, 2973, 7963}},
	{1008, 13, 3444, {1, 3, 3, 5, 19, 61, 125, 199, 299, 515, 1365, 369, 7027}},
	{1009, 13, 3453, {1, 3, 1, 7, 5, 41, 63, 229, 283, 571, 147, 447, 657}},
	{1010, 13, 3464, {1, 3, 1, 11, 5, 15, 55, 7, 259, 61, 27, 1429, 5631}},
	{1011, 13, 3477, {1, 1, 5, 1, 3, 53, 51, 253, 155, 553, 1293, 3735, 6567}},
	{1012, 13, 3478, {1, 3, 5, 9, 5, 41, 21, 159, 101, 785, 1981, 3799, 7693}},
	{1013, 13, 3482, {1, 3, 7, 7, 9, 3, 95, 105, 129, 213, 1215, 1027, 5699}},
	{1014, 13, 3487, {1, 1, 3, 3, 29, 13, 9, 253, 449, 321, 341, 2879, 171}},
	{1015, 13, 3497, {1, 3, 7, 11, 21, 11, 75, 35, 43, 965, 675, 2217, 7175}},
	{1016, 13, 3500, {1, 1, 5, 15, 31, 5, 29, 137, 311, 751, 47, 1367, 5921}},
	{1017, 13, 3505, {1, 1, 3, 15, 17, 1, 45, 69, 55, 649, 835, 569, 7615}},
	{1018, 13, 3506, {1, 3, 1, 13, 31, 7, 23, 15, 391, 145, 1845, 1825, 1403}},
	{1019, 13, 3511, {1, 1, 3, 15, 5, 9, 79, 77, 105, 399, 1933, 2503, 4781}},
	{1020, 13, 3512, {1, 3, 1, 3, 17, 47, 19, 13, 107, 475, 759, 2933, 3761}},
	{1021, 13, 3515, {1, 1, 7, 11, 3, 7, 121, 209, 397, 877, 293, 847, 7039}},
	{1022, 13, 3525, {1, 1, 1, 15, 29, 45, 5, 109, 335, 461, 143, 931, 4045}},
	{1023, 13, 3532, {1, 3, 1, 7, 11, 57, 73, 89, 201, 173, 803, 3953, 5205}},
	{1024, 13, 3538, {1, 1, 5, 11, 11, 33, 37, 29, 263, 1019, 657, 1453, 7807}},
	{1025, 13, 3540, {1, 3, 3, 13, 31, 25, 37, 47, 261, 607, 1703, 2603, 417}},
	{1026, 13, 3547, {1, 1, 1, 1, 31, 61, 45, 115, 275, 239, 1989, 1897, 4329}},
	{1027, 13, 3549, {1, 3, 5, 3, 31, 3, 11, 173, 335, 579, 1193, 2219, 7875}},
	{1028, 13, 3560, {1, 1, 7, 9, 29, 45, 13, 67, 399, 177, 1293, 3865, 2225}},
	{1029, 13, 3571, {1, 1, 7, 11, 11, 51, 121, 227, 469, 905, 929, 2635, 4165}},
	{1030, 13, 3577, {1, 3, 7, 9, 13, 39, 55, 167, 23, 147, 1603, 2083, 4645}},
	{1031, 13, 3583, {1, 1, 3, 15, 27, 53, 11, 155, 157, 629, 259, 3009, 4605}},
	{1032, 13, 3590, {1, 3, 1, 7, 15, 47, 51, 1, 259, 603, 887, 2833, 6581}},
	{1033, 13, 3593, {1, 3, 5, 3, 1, 47, 91, 43, 361, 571, 29, 1453, 4269}},
	{1034, 13, 3594, {1, 1, 3, 9, 11, 51, 55, 23, 415, 277, 1423, 3475, 1527}},
	{1035, 13, 3599, {1, 1, 3, 11, 29, 49, 101, 75, 299, 709, 805, 4037, 4389}},
	{1036, 13, 3601, {1, 1, 7, 3, 23, 1, 37, 51, 379, 771, 1301, 3717, 6673}},
	{1037, 13, 3602, {1, 1, 5, 3, 23, 11, 125, 177, 375, 665, 951, 1577, 2603}},
	{1038, 13, 3613, {1, 1, 1, 1, 1, 5, 71, 255, 21, 459, 467, 2083, 5415}},
	{1039, 13, 3623, {1, 1, 5, 13, 23, 29, 109, 157, 363, 971, 549, 647, 1177}},
	{1040, 13, 3630, {1, 1, 3, 9, 7, 15, 101, 3, 365, 213, 745, 1155, 6099}},
	{1041, 13, 3638, {1, 3, 5, 15, 15, 19, 47, 179, 303, 521, 1279, 219, 2415}},
	{1042, 13, 3649, {1, 3, 3, 13, 27, 11, 83, 165, 369, 989, 261, 3933, 4809}},
	{1043, 13, 3655, {1, 1, 3, 11, 31, 59, 1, 185, 53, 703, 1471, 2935, 1107}},
	{1044, 13, 3662, {1, 3, 3, 7, 25, 3, 81, 27, 93, 521, 433, 2859, 5861}},
	{1045, 13, 3667, {1, 3, 3, 11, 29, 15, 49, 167, 315, 927, 543, 3473, 4307}},
	{1046, 13, 3669, {1, 3, 1, 3, 29, 33, 53, 15, 183, 691, 703, 1311, 3393}},
	{1047, 13, 3676, {1, 3, 5, 13, 23, 49, 3, 11, 1, 357, 1407, 415, 7211}},
	{1048, 13, 3683, {1, 3, 7, 15, 1, 25, 91, 113, 323, 371, 189, 925, 1181}},
	{1049, 13, 3700, {1, 3, 3, 3, 17, 59, 119, 199, 115, 223, 877, 2193, 193}},
	{1050, 13, 3709, {1, 1, 1, 5, 5, 35, 31, 59, 437, 411, 37, 2405, 3797}},
	{1051, 13, 3710, {1, 3, 1, 13, 9, 37, 1, 241, 59, 157, 1785, 1223, 563}},
	{1052, 13, 3713, {1, 3, 5, 13, 3, 21, 25, 95, 15, 745, 85, 701, 5361}},
	{1053, 13, 3723, {1, 3, 7, 1, 31, 33, 111, 195, 35, 913, 2013, 2951, 6611}},
	{1054, 13, 3725, {1, 3, 5, 1, 19, 3, 75, 119, 111, 409, 951, 1457, 4957}},
	{1055, 13, 3728, {1, 3, 1, 15, 19, 59, 3, 155, 237, 657, 1967, 3323, 6235}},
	{1056, 13, 3734, {1, 1, 5, 1, 3, 19, 45, 105, 377, 881, 167, 2255, 4483}},
	{1057, 13, 3737, {1, 1, 7, 7, 13, 13, 99, 89, 201, 279, 161, 2483, 6001}},
	{1058, 13, 3738, {1, 1, 7, 3, 13, 17, 97, 129, 137, 377, 1519, 183, 3725}},
	{1059, 13, 3744, {1, 1, 7, 9, 9, 5, 45, 135, 115, 181, 1685, 3505, 4387}},
	{1060, 13, 3750, {1, 1, 1, 1, 19, 35, 69, 113, 305, 419, 949, 2969, 247}},
	{1061, 13, 3762, {1, 1, 5, 13, 23, 61, 13, 139, 501, 811, 67, 1501, 6493}},
	{1062, 13, 3764, {1, 1, 3, 13, 15, 41, 27, 217, 293, 13, 145, 2631, 6991}},
	{1063, 13, 3774, {1, 3, 3, 13, 15, 37, 71, 123, 285, 49, 627, 1283, 5993}},
	{1064, 13, 3776, {1, 3, 3, 11, 9, 25, 11, 1, 203, 353, 1347, 1999, 2799}},
	{1065, 13, 3786, {1, 3, 5, 1, 7, 49, 101, 231, 499, 63, 1977, 2207, 7829}},
	{1066, 13, 3800, {1, 1, 7, 1, 17, 15, 115, 139, 381, 943, 623, 4037, 2971}},
	{1067, 13, 3803, {1, 1, 3, 5, 13, 55, 23, 87, 139, 795, 1669, 1375, 1185}},
	{1068, 13, 3809, {1, 3, 3, 5, 5, 45, 97, 253, 241, 333, 645, 555, 7867}},
	{1069, 13, 3816, {1, 3, 5, 1, 1, 1, 89, 27, 407, 509, 1433, 609, 2355}},
	{1070, 13, 3821, {1, 3, 7, 1, 27, 29, 5, 157, 495, 811, 1293, 1143, 827}},
	{1071, 13, 3827, {1, 1, 3, 3, 25, 49, 127, 111, 191, 3, 845, 1383, 2521}},
	{1072, 13, 3829, {1, 1, 5, 7, 5, 51, 101, 155, 237, 461, 831, 3091, 3851}},
	{1073, 13, 3836, {1, 3, 7, 1, 29, 35, 105, 91, 285, 705, 131, 395, 6011}},
	{1074, 13, 3842, {1, 3, 5, 3, 13, 21, 83, 173, 221, 827, 1775, 1931, 6727}},
	{1075, 13, 3844, {1, 1, 3, 5, 3, 25, 95, 115, 205, 569, 1447, 933, 6425}},
	{1076, 13, 3847, {1, 1, 7, 9, 31, 3, 17, 175, 145, 447, 1321, 1069, 6527}},
	{1077, 13, 3853, {1, 1, 3, 3, 23, 1, 79, 51, 421, 419, 873, 3939, 1801}},
	{1078, 13, 3861, {1, 1, 5, 1, 3, 39, 15, 85, 169, 669, 919, 397, 5579}},
	{1079, 13, 3871, {1, 3, 5, 1, 21, 61, 87, 217, 251, 619, 1091, 4009, 229}},
	{1080, 13, 3872, {1, 1, 1, 11, 23, 55, 85, 121, 363, 867, 315, 447, 3373}},
	{1081, 13, 3881, {1, 3, 3, 13, 29, 19, 89, 85, 137, 469, 1873, 2765, 3975}},
	{1082, 13, 3890, {1, 3, 7, 13, 19, 63, 61, 77, 67, 361, 11, 1787, 4703}},
	{1083, 13, 3892, {1, 1, 3, 11, 7, 15, 127, 105, 179, 857, 1671, 3647, 3389}},
	{1084, 13, 3909, {1, 1, 1, 7, 19, 21, 99, 161, 499, 519, 1287, 2973, 479}},
	{1085, 13, 3921, {1, 1, 3, 13, 29, 51, 95, 251, 55, 519, 1955, 2881, 5951}},
	{1086, 13, 3934, {1, 1, 3, 11, 23, 63, 121, 237, 175, 311, 701, 1539, 2383}},
	{1087, 13, 3938, {1, 1, 7, 5, 5, 45, 73, 97, 5, 153, 715, 2037, 3353}},
	{1088, 13, 3947, {1, 1, 1, 3, 13, 7, 67, 173, 425, 843, 1497, 2729, 5193}},
	{1089, 13, 3950, {1, 1, 7, 1, 23, 3, 119, 11, 77, 141, 1905, 2269, 4269}},
	{1090, 13, 3952, {1, 1, 7, 15, 1, 23, 79, 251, 439, 603, 405, 2449, 6383}},
	{1091, 13, 3964, {1, 3, 7, 11, 29, 27, 47, 255, 47, 661, 1967, 1007, 3689}},
	{1092, 13, 3974, {1, 3, 7, 5, 19, 39, 35, 115, 417, 373, 291, 329, 603}},
	{1093, 13, 3980, {1, 3, 1, 9, 11, 33, 27, 193, 207, 423, 1311, 1369, 7307}},
	{1094, 13, 3983, {1, 1, 3, 11, 9, 29, 83, 17, 497, 493, 329, 3141, 5935}},
	{1095, 13, 3986, {1, 3, 1, 5, 31, 51, 29, 171, 51, 493, 1621, 3501, 4091}},
	{1096, 13, 3995, {1, 1, 5, 9, 21, 43, 105, 207, 245, 363, 1191, 699, 1139}},
	{1097, 13, 3998, {1, 1, 3, 11, 19, 5, 81, 119, 247, 169, 1337, 45, 6565}},
	{1098, 13, 4001, {1, 3, 1, 11, 3, 51, 3, 101, 159, 11, 253, 299, 5043}},
	{1099, 13, 4002, {1, 3, 1, 5, 11, 53, 85, 39, 57, 645, 2007, 1039, 3627}},
	{1100, 13, 4004, {1, 3, 5, 3, 17, 61, 97, 165, 415, 357, 283, 601, 5505}},
	{1101, 13, 4008, {1, 3, 7, 3, 9, 51, 49, 85, 3, 227, 137, 309, 243}},
	{1102, 13, 4011, {1, 1, 5, 3, 11, 59, 11, 131, 409, 703, 455, 123, 6727}},
	{1103, 13, 4016, {1, 3, 7, 9, 25, 49, 21, 171, 287, 379, 667, 313, 713}},
	{1104, 13, 4033, {1, 1, 3, 9, 7, 35, 47, 3, 367, 581, 1627, 1665, 3905}},
	{1105, 13, 4036, {1, 3, 1, 1, 29, 57, 35, 55, 255, 653, 823, 2197, 6179}},
	{1106, 13, 4040, {1, 3, 7, 15, 17, 15, 117, 83, 359, 163, 115, 2999, 5373}},
	{1107, 13, 4053, {1, 1, 5, 3, 21, 61, 35, 97, 71, 687, 207, 2917, 1049}},
	{1108, 13, 4058, {1, 1, 1, 15, 13, 15, 125, 81, 263, 661, 417, 3243, 1669}},
	{1109, 13, 4081, {1, 1, 7, 3, 3, 19, 111, 193, 443, 339, 659, 1211, 1557}},
	{1110, 13, 4091, {1, 3, 1, 3, 27, 3, 3, 173, 391, 213, 803, 3281, 3207}},
	{1111, 13, 4094, {1, 1, 5, 15, 19, 1, 7, 211, 157, 603, 403, 1387, 1583}},
	{1112, 14, 21, {1, 3, 5, 13, 17, 53, 125, 13, 339, 723, 521, 413, 5801, 10451}},
	{1113, 14, 28, {1, 1, 3, 13, 29, 9, 99, 77, 141, 609, 1533, 983, 2039, 51}},
	{1114, 14, 41, {1, 1, 3, 11, 21, 55, 5, 51, 423, 309, 525, 3715, 3025, 15055}},
	{1115, 14, 47, {1, 1, 3, 7, 9, 21, 77, 171, 239, 341, 1653, 1093, 2273, 10723}},
	{1116, 14, 61, {1, 1, 1, 15, 31, 15, 23, 35, 317, 869, 1917, 1415, 4313, 3687}},
	{1117, 14, 84, {1, 1, 1, 5, 21, 25, 99, 167, 439, 453, 473, 431, 6665, 4989}},
	{1118, 14, 87, {1, 1, 7, 9, 31, 47, 81, 83, 345, 43, 1363, 1885, 3155, 3185}},
	{1119, 14, 93, {1, 3, 7, 1, 31, 17, 61, 185, 341, 129, 547, 717, 2071, 9991}},
	{1120, 14, 94, {1, 3, 1, 13, 23, 61, 77, 217, 455, 77, 1263, 1601, 3501, 14953}},
	{1121, 14, 103, {1, 1, 7, 7, 19, 19, 1, 229, 431, 943, 1069, 1949, 1289, 15729}},
	{1122, 14, 117, {1, 1, 3, 5, 1, 35, 97, 251, 487, 459, 1265, 1739, 165, 10365}},
	{1123, 14, 121, {1, 3, 5, 3, 11, 25, 79, 175, 383, 545, 187, 197, 4329, 3363}},
	{1124, 14, 134, {1, 1, 3, 3, 29, 9, 63, 55, 175, 277, 431, 2549, 2629, 6409}},
	{1125, 14, 137, {1, 1, 3, 15, 17, 21, 79, 139, 99, 135, 1763, 1805, 3471, 5439}},
	{1126, 14, 157, {1, 1, 3, 9, 9, 15, 35, 119, 289, 835, 769, 3843, 4119, 4421}},
	{1127, 14, 161, {1, 1, 1, 5, 19, 19, 67, 199, 307, 815, 1367, 1231, 3927, 6593}},
	{1128, 14, 205, {1, 1, 3, 1, 29, 51, 121, 209, 431, 47, 1115, 907, 2535, 9755}},
	{1129, 14, 206, {1, 1, 3, 5, 17, 1, 5, 119, 121, 223, 1719, 1291, 3947, 15891}},
	{1130, 14, 211, {1, 3, 1, 15, 29, 25, 3, 131, 373, 307, 645, 3513, 1289, 1987}},
	{1131, 14, 214, {1, 3, 3, 11, 29, 45, 105, 179, 331, 465, 891, 1315, 403, 3057}},
	{1132, 14, 218, {1, 1, 5, 13, 17, 59, 77, 127, 485, 855, 1147, 3093, 891, 9869}},
	{1133, 14, 234, {1, 1, 1, 7, 23, 27, 31, 203, 285, 463, 827, 685, 1349, 15051}},
	{1134, 14, 236, {1, 1, 1, 5, 29, 5, 107, 195, 31, 425, 19, 2865, 3869, 11153}},
	{1135, 14, 248, {1, 1, 7, 5, 7, 47, 1, 73, 307, 347, 393, 2205, 7709, 15121}},
	{1136, 14, 262, {1, 1, 1, 13, 15, 61, 25, 131, 113, 369, 1995, 2527, 4475, 1745}},
	{1137, 14, 299, {1, 1, 1, 1, 31, 63, 21, 253, 307, 281, 859, 3319, 6721, 2891}},
	{1138, 14, 304, {1, 1, 3, 11, 1, 17, 5, 183, 301, 979, 651, 1685, 6343, 10067}},
	{1139, 14, 319, {1, 1, 5, 15, 23, 45, 99, 145, 263, 507, 1381, 3425, 2215, 1815}},
	{1140, 14, 322, {1, 3, 1, 5, 11, 63, 85, 203, 411, 881, 1369, 1237, 4657, 6541}},
	{1141, 14, 334, {1, 3, 3, 13, 17, 53, 121, 201, 269, 983, 215, 3187, 7121, 6111}},
	{1142, 14, 355, {1, 3, 5, 15, 15, 5, 13, 143, 3, 313, 1677, 1093, 3295, 3387}},
	{1143, 14, 357, {1, 1, 3, 13, 3, 23, 73, 17, 257, 965, 239, 1271, 2803, 7327}},
	{1144, 14, 358, {1, 3, 5, 13, 9, 57, 115, 37, 41, 467, 135, 1403, 3811, 4741}},
	{1145, 14, 369, {1, 3, 7, 15, 9, 33, 39, 203, 351, 367, 1355, 1403, 3685, 4757}},
	{1146, 14, 372, {1, 3, 5, 11, 31, 3, 113, 123, 203, 421, 1821, 3151, 2375, 4419}},
	{1147, 14, 375, {1, 1, 1, 7, 21, 63, 99, 23, 133, 79, 991, 1755, 4989, 4709}},
	{1148, 14, 388, {1, 3, 5, 1, 25, 63, 113, 239, 49, 443, 173, 1261, 3201, 10599}},
	{1149, 14, 400, {1, 3, 3, 13, 3, 25, 101, 169, 23, 585, 327, 1327, 111, 10059}},
	{1150, 14, 415, {1, 3, 3, 5, 19, 1, 33, 89, 437, 213, 1933, 1741, 2603, 5625}},
	{1151, 14, 446, {1, 3, 1, 3, 15, 15, 25, 139, 73, 335, 237, 2461, 3101, 14951}},
	{1152, 14, 451, {1, 3, 5, 1, 31, 15, 31, 187, 177, 659, 1339, 3767, 4975, 7123}},
	{1153, 14, 458, {1, 3, 1, 3, 25, 19, 47, 89, 107, 107, 649, 683, 3123, 11085}},
	{1154, 14, 471, {1, 3, 7, 9, 15, 21, 101, 25, 11, 625, 1555, 675, 3893, 5805}},
	{1155, 14, 484, {1, 1, 1, 5, 7, 49, 123, 21, 439, 369, 207, 535, 4619, 14665}},
	{1156, 14, 501, {1, 1, 5, 7, 1, 25, 103, 185, 99, 239, 1093, 1561, 6177, 4039}},
	{1157, 14, 502, {1, 3, 7, 5, 29, 21, 43, 103, 343, 973, 1561, 2975, 7467, 7947}},
	{1158, 14, 517, {1, 1, 7, 9, 19, 3, 13, 23, 461, 813, 1191, 985, 559, 3317}},
	{1159, 14, 545, {1, 3, 5, 5, 27, 31, 79, 15, 365, 901, 1949, 117, 3619, 13311}},
	{1160, 14, 569, {1, 3, 5, 7, 5, 33, 67, 199, 425, 189, 1691, 3099, 815, 1677}},
	{1161, 14, 617, {1, 1, 7, 11, 13, 29, 73, 137, 265, 601, 445, 3893, 2511, 8047}},
	{1162, 14, 618, {1, 1, 3, 1, 13, 5, 57, 101, 357, 391, 335, 601, 1359, 1065}},
	{1163, 14, 623, {1, 1, 1, 1, 25, 57, 27, 115, 31, 873, 611, 2125, 447, 13585}},
	{1164, 14, 625, {1, 3, 3, 13, 27, 17, 73, 11, 359, 33, 1153, 271, 4537, 15141}},
	{1165, 14, 637, {1, 3, 7, 3, 11, 63, 103, 61, 59, 629, 1629, 3279, 3919, 3177}},
	{1166, 14, 661, {1, 1, 5, 15, 3, 63, 85, 193, 381, 165, 175, 3247, 2501, 4209}},
	{1167, 14, 668, {1, 1, 5, 15, 1, 33, 59, 219, 487, 193, 1557, 703, 2907, 7953}},
	{1168, 14, 684, {1, 1, 7, 3, 9, 3, 105, 95, 389, 991, 21, 3841, 6983, 285}},
	{1169, 14, 695, {1, 1, 1, 1, 1, 31, 25, 137, 117, 67, 1283, 1963, 6591, 15541}},
	{1170, 14, 716, {1, 3, 5, 11, 7, 15, 127, 89, 453, 777, 1827, 2311, 7661, 11833}},
	{1171, 14, 719, {1, 1, 7, 13, 19, 29, 79, 165, 223, 453, 2039, 3961, 6467, 5481}},
	{1172, 14, 722, {1, 3, 3, 7, 17, 41, 43, 157, 323, 3, 1001, 2109, 4513, 12127}},
	{1173, 14, 731, {1, 1, 5, 9, 31, 57, 3, 217, 113, 271, 1663, 1367, 6949, 8165}},
	{1174, 14, 738, {1, 1, 7, 15, 27, 35, 81, 235, 61, 205, 525, 311, 6357, 2527}},
	{1175, 14, 747, {1, 3, 1, 9, 19, 29, 71, 207, 321, 1011, 1615, 1333, 3459, 6681}},
	{1176, 14, 755, {1, 3, 7, 7, 3, 57, 41, 19, 25, 397, 565, 1837, 7625, 11813}},
	{1177, 14, 761, {1, 3, 3, 1, 27, 47, 31, 79, 441, 961, 1255, 423, 2405, 913}},
	{1178, 14, 767, {1, 3, 3, 13, 3, 29, 69, 227, 85, 201, 395, 3199, 3869, 13099}},
	{1179, 14, 775, {1, 3, 3, 7, 29, 61, 99, 7, 27, 227, 945, 873, 475, 4363}},
	{1180, 14, 782, {1, 3, 5, 13, 19, 21, 57, 149, 217, 443, 565, 453, 5487, 10981}},
	{1181, 14, 787, {1, 3, 3, 1, 9, 27, 47, 191, 35, 395, 1429, 4079, 6871, 8013}},
	{1182, 14, 794, {1, 3, 5, 15, 5, 43, 9, 79, 279, 563, 1125, 985, 8117, 4099}},
	{1183, 14, 803, {1, 3, 5, 1, 13, 41, 21, 117, 287, 667, 701, 1483, 8167, 13283}},
	{1184, 14, 812, {1, 3, 1, 3, 15, 15, 59, 5, 383, 509, 1657, 3977, 7697, 10941}},
	{1185, 14, 817, {1, 3, 1, 1, 17, 29, 19, 23, 377, 45, 981, 1631, 3557, 6749}},
	{1186, 14, 824, {1, 3, 3, 9, 9, 51, 9, 193, 345, 361, 1679, 3333, 713, 5387}},
	{1187, 14, 829, {1, 3, 5, 5, 17, 45, 97, 17, 385, 349, 105, 2245, 7295, 14393}},
	{1188, 14, 850, {1, 3, 7, 3, 19, 51, 35, 99, 79, 301, 1563, 399, 5879, 14675}},
	{1189, 14, 866, {1, 1, 7, 15, 13, 53, 55, 203, 417, 161, 2033, 1845, 6763, 3585}},
	{1190, 14, 871, {1, 1, 3, 3, 7, 23, 11, 43, 241, 309, 1453, 3147, 2619, 3163}},
	{1191, 14, 877, {1, 1, 1, 11, 17, 1, 17, 137, 443, 465, 993, 3217, 7879, 14607}},
	{1192, 14, 920, {1, 1, 7, 13, 29, 49, 71, 217, 291, 793, 135, 21, 2503, 11091}},
	{1193, 14, 935, {1, 3, 1, 11, 31, 51, 121, 227, 377, 157, 1457, 1317, 5625, 6217}},
	{1194, 14, 959, {1, 1, 3, 7, 23, 61, 47, 93, 79, 617, 1805, 2403, 5513, 16335}},
	{1195, 14, 979, {1, 3, 5, 11, 23, 25, 41, 11, 495, 587, 1223, 3107, 1469, 15223}},
	{1196, 14, 992, {1, 3, 7, 7, 9, 1, 1, 49, 23, 723, 1761, 3717, 7375, 10875}},
	{1197, 14, 1010, {1, 3, 3, 11, 25, 37, 57, 63, 309, 603, 183, 285, 1663, 5627}},
	{1198, 14, 1012, {1, 3, 7, 11, 19, 25, 25, 201, 391, 257, 529, 1645, 1, 15111}},
	{1199, 14, 1015, {1, 3, 3, 9, 11, 43, 91, 65, 5, 959, 301, 1015, 6343, 3453}},
	{1200, 14, 1033, {1, 3, 3, 11, 17, 17, 103, 37, 77, 973, 575, 439, 49, 3639}},
	{1201, 14, 1036, {1, 1, 5, 7, 1, 15, 107, 237, 231, 967, 923, 1101, 6715, 1713}},
	{1202, 14, 1053, {1, 3, 1, 15, 9, 33, 29, 211, 245, 601, 1783, 887, 1209, 11785}},
	{1203, 14, 1057, {1, 3, 3, 7, 21, 43, 27, 89, 27, 141, 865, 367, 1379, 4063}},
	{1204, 14, 1069, {1, 3, 7, 7, 15, 17, 15, 15, 131, 649, 1955, 3289, 3983, 10689}},
	{1205, 14, 1072, {1, 3, 1, 5, 17, 7, 125, 69, 359, 981, 1345, 933, 5281, 7113}},
	{1206, 14, 1075, {1, 1, 5, 9, 17, 7, 41, 207, 497, 1015, 493, 891, 3563, 3541}},
	{1207, 14, 1087, {1, 3, 5, 11, 27, 3, 47, 31, 303, 1007, 2047, 2203, 6257, 8369}},
	{1208, 14, 1089, {1, 1, 1, 15, 25, 15, 89, 51, 217, 357, 1133, 1917, 213, 3365}},
	{1209, 14, 1137, {1, 1, 5, 13, 29, 23, 123, 207, 429, 805, 819, 2357, 6313, 11019}},
	{1210, 14, 1166, {1, 1, 3, 7, 19, 15, 41, 73, 279, 11, 1089, 3107, 7737, 15953}},
	{1211, 14, 1174, {1, 3, 5, 7, 7, 15, 41, 73, 493, 457, 1731, 1139, 2513, 12373}},
	{1212, 14, 1180, {1, 3, 5, 9, 17, 5, 55, 155, 173, 1005, 529, 3175, 7667, 4747}},
	{1213, 14, 1204, {1, 1, 7, 7, 5, 21, 105, 31, 205, 847, 1033, 3167, 2347, 8499}},
	{1214, 14, 1211, {1, 3, 5, 3, 11, 17, 59, 189, 179, 1007, 33, 3287, 4813, 8177}},
	{1215, 14, 1219, {1, 3, 3, 13, 27, 47, 47, 171, 413, 875, 1081, 1259, 7139, 8645}},
	{1216, 14, 1236, {1, 3, 5, 7, 25, 21, 51, 29, 361, 293, 51, 1119, 1453, 5283}},
	{1217, 14, 1255, {1, 3, 7, 7, 29, 55, 103, 199, 511, 341, 1957, 3987, 2855, 1279}},
	{1218, 14, 1264, {1, 1, 1, 9, 23, 51, 61, 63, 391, 37, 55, 3771, 6517, 15913}},
	{1219, 14, 1306, {1, 1, 1, 9, 3, 19, 13, 147, 453, 855, 1321, 189, 5043, 11215}},
	{1220, 14, 1330, {1, 3, 3, 13, 23, 3, 87, 155, 401, 981, 607, 3413, 995, 6473}},
	{1221, 14, 1341, {1, 3, 1, 9, 29, 47, 95, 123, 421, 353, 1867, 2609, 2569, 14083}},
	{1222, 14, 1344, {1, 1, 5, 13, 25, 39, 29, 111, 125, 545, 1493, 2371, 6361, 6307}},
	{1223, 14, 1347, {1, 3, 3, 11, 13, 31, 87, 75, 27, 393, 921, 3655, 3343, 16349}},
	{1224, 14, 1349, {1, 1, 5, 9, 19, 19, 7, 129, 223, 715, 433, 1627, 4463, 2951}},
	{1225, 14, 1361, {1, 1, 7, 1, 31, 13, 49, 33, 89, 43, 1529, 725, 3809, 3427}},
	{1226, 14, 1380, {1, 1, 7, 3, 1, 27, 45, 9, 309, 875, 659, 2661, 553, 7069}},
	{1227, 14, 1390, {1, 1, 7, 15, 13, 37, 61, 19, 125, 683, 1227, 2255, 1455, 9339}},
	{1228, 14, 1404, {1, 3, 5, 7, 19, 7, 71, 21, 465, 645, 1885, 873, 7405, 1913}},
	{1229, 14, 1435, {1, 3, 1, 11, 11, 35, 79, 61, 79, 57, 1603, 3719, 6323, 16371}},
	{1230, 14, 1444, {1, 1, 7, 1, 29, 57, 85, 21, 205, 37, 2045, 683, 4901, 8223}},
	{1231, 14, 1453, {1, 1, 5, 13, 31, 31, 65, 131, 259, 535, 967, 3943, 2605, 2089}},
	{1232, 14, 1461, {1, 1, 7, 9, 27, 61, 39, 243, 207, 41, 1909, 3279, 1331, 4635}},
	{1233, 14, 1462, {1, 3, 3, 5, 11, 63, 105, 19, 169, 95, 773, 3175, 1869, 1797}},
	{1234, 14, 1465, {1, 3, 3, 15, 13, 33, 107, 197, 153, 795, 1477, 105, 4965, 991}},
	{1235, 14, 1468, {1, 3, 7, 11, 11, 37, 23, 149, 197, 3, 1035, 3857, 553, 1059}},
	{1236, 14, 1474, {1, 3, 1, 3, 17, 29, 89, 189, 193, 59, 1477, 3517, 2565, 7739}},
	{1237, 14, 1483, {1, 1, 1, 9, 23, 3, 25, 163, 469, 305, 1791, 3393, 6141, 8119}},
	{1238, 14, 1488, {1, 3, 5, 7, 7, 41, 19, 101, 179, 487, 1071, 2761, 8043, 5103}},
	{1239, 14, 1493, {1, 1, 7, 9, 1, 21, 101, 103, 349, 85, 1841, 1033, 4473, 3563}},
	{1240, 14, 1500, {1, 1, 3, 13, 23, 61, 39, 27, 479, 13, 45, 1371, 7897, 10637}},
	{1241, 14, 1509, {1, 1, 5, 9, 17, 61, 71, 55, 355, 99, 1695, 3053, 839, 959}},
	{1242, 14, 1510, {1, 1, 3, 1, 7, 27, 87, 221, 327, 241, 461, 3177, 5933, 8299}},
	{1243, 14, 1514, {1, 3, 7, 9, 5, 41, 111, 245, 447, 263, 1363, 1767, 6331, 3355}},
	{1244, 14, 1519, {1, 3, 3, 13, 15, 11, 15, 169, 429, 149, 1965, 2477, 7733, 2499}},
	{1245, 14, 1528, {1, 1, 5, 15, 15, 47, 25, 33, 469, 701, 773, 2747, 1533, 14633}},
	{1246, 14, 1533, {1, 3, 1, 5, 19, 57, 37, 75, 423, 11, 685, 2487, 1779, 8797}},
	{1247, 14, 1540, {1, 3, 1, 5, 19, 41, 67, 99, 333, 991, 953, 3221, 939, 4197}},
	{1248, 14, 1550, {1, 3, 1, 15, 11, 39, 25, 1, 159, 679, 465, 1611, 5799, 2537}},
	{1249, 14, 1567, {1, 1, 5, 11, 5, 37, 37, 7, 101, 703, 235, 23, 2209, 12799}},
	{1250, 14, 1571, {1, 1, 7, 3, 11, 23, 71, 215, 45, 269, 1539, 3625, 5773, 6889}},
	{1251, 14, 1573, {1, 3, 5, 15, 27, 33, 105, 109, 205, 653, 821, 435, 1087, 2495}},
	{1252, 14, 1578, {1, 1, 3, 5, 11, 39, 53, 213, 41, 385, 1425, 25, 5553, 12523}},
	{1253, 14, 1598, {1, 3, 5, 15, 29, 49, 13, 253, 505, 407, 985, 2569, 6727, 4761}},
	{1254, 14, 1606, {1, 1, 1, 3, 29, 17, 69, 47, 25, 819, 1145, 2479, 1183, 3343}},
	{1255, 14, 1618, {1, 3, 1, 15, 25, 61, 43, 55, 279, 579, 361, 355, 6101, 3143}},
	{1256, 14, 1630, {1, 3, 5, 11, 3, 59, 125, 101, 451, 495, 1711, 3443, 3625, 15579}},
	{1257, 14, 1634, {1, 3, 1, 11, 25, 61, 49, 219, 23, 795, 481, 3609, 3691, 15419}},
	{1258, 14, 1640, {1, 3, 7, 5, 9, 59, 49, 233, 345, 143, 181, 3587, 3041, 1219}},
	{1259, 14, 1643, {1, 3, 7, 13, 9, 31, 39, 137, 261, 919, 1367, 3145, 4659, 5875}},
	{1260, 14, 1654, {1, 1, 3, 3, 27, 43, 95, 65, 301, 915, 31, 451, 7743, 7277}},
	{1261, 14, 1679, {1, 3, 1, 5, 23, 37, 53, 31, 203, 453, 71, 1585, 6011, 16369}},
	{1262, 14, 1688, {1, 1, 5, 1, 15, 47, 91, 227, 297, 45, 1415, 3647, 7811, 14015}},
	{1263, 14, 1698, {1, 1, 1, 1, 29, 27, 93, 121, 169, 69, 1361, 2907, 1867, 7017}},
	{1264, 14, 1703, {1, 3, 1, 7, 23, 53, 77, 41, 25, 873, 1333, 3889, 3239, 1771}},
	{1265, 14, 1704, {1, 1, 1, 7, 31, 27, 87, 81, 167, 343, 1981, 2499, 7749, 15747}},
	{1266, 14, 1722, {1, 3, 5, 13, 1, 17, 97, 37, 81, 645, 1167, 3547, 7769, 10731}},
	{1267, 14, 1735, {1, 1, 7, 5, 9, 17, 31, 55, 151, 463, 1041, 2303, 4015, 3737}},
	{1268, 14, 1750, {1, 1, 3, 11, 31, 9, 81, 213, 95, 215, 2031, 2129, 4299, 3021}},
	{1269, 14, 1753, {1, 1, 1, 3, 25, 25, 115, 229, 101, 441, 783, 1729, 7905, 2375}},
	{1270, 14, 1760, {1, 1, 5, 9, 3, 19, 73, 35, 379, 493, 1333, 1647, 13, 197}},
	{1271, 14, 1789, {1, 1, 7, 9, 3, 55, 99, 43, 281, 9, 73, 2477, 8183, 11055}},
	{1272, 14, 1792, {1, 3, 7, 13, 25, 19, 27, 195, 469, 175, 355, 1861, 7255, 15377}},
	{1273, 14, 1802, {1, 1, 3, 11, 15, 19, 115, 31, 413, 835, 697, 879, 6515, 13465}},
	{1274, 14, 1819, {1, 3, 3, 15, 3, 61, 105, 201, 151, 739, 49, 3963, 2573, 3303}},
	{1275, 14, 1825, {1, 3, 5, 7, 23, 5, 11, 215, 19, 591, 509, 2887, 1631, 4391}},
	{1276, 14, 1828, {1, 3, 3, 3, 25, 1, 109, 5, 363, 545, 1745, 503, 827, 4677}},
	{1277, 14, 1832, {1, 1, 3, 15, 1, 45, 121, 141, 497, 745, 1825, 2041, 2561, 8153}},
	{1278, 14, 1845, {1, 3, 1, 11, 29, 7, 71, 241, 7, 39, 1379, 2479, 7483, 7195}},
	{1279, 14, 1846, {1, 1, 7, 11, 3, 27, 39, 97, 339, 217, 1409, 1569, 4761, 1567}},
	{1280, 14, 1857, {1, 1, 5, 15, 11, 53, 87, 213, 297, 923, 393, 717, 3297, 16123}},
	{1281, 14, 1869, {1, 1, 1, 11, 27, 41, 121, 49, 225, 379, 1305, 319, 2461, 5445}},
	{1282, 14, 1897, {1, 1, 5, 5, 25, 3, 121, 23, 47, 843, 1679, 1427, 6393, 4199}},
	{1283, 14, 1906, {1, 1, 5, 13, 17, 3, 17, 25, 161, 487, 121, 361, 1375, 10745}},
	{1284, 14, 1908, {1, 1, 7, 3, 3, 37, 7, 245, 107, 107, 745, 2415, 2131, 11419}},
	{1285, 14, 1911, {1, 1, 5, 3, 3, 23, 67, 91, 281, 387, 465, 905, 883, 9775}},
	{1286, 14, 1934, {1, 3, 7, 15, 25, 55, 123, 49, 23, 983, 1903, 2589, 2073, 7823}},
	{1287, 14, 1962, {1, 1, 5, 11, 25, 17, 63, 229, 267, 175, 1759, 1947, 479, 11089}},
	{1288, 14, 1967, {1, 3, 7, 3, 11, 37, 83, 95, 415, 1003, 1175, 2361, 2117, 9809}},
	{1289, 14, 1972, {1, 3, 1, 9, 5, 39, 51, 129, 249, 161, 1981, 2755, 8057, 13641}},
	{1290, 14, 1975, {1, 1, 7, 1, 15, 47, 9, 197, 199, 549, 1091, 2853, 2331, 4535}},
	{1291, 14, 1999, {1, 3, 3, 13, 15, 21, 23, 111, 463, 719, 1667, 377, 5039, 10723}},
	{1292, 14, 2004, {1, 1, 3, 7, 23, 47, 39, 47, 307, 949, 1651, 2525, 5835, 1425}},
	{1293, 14, 2011, {1, 3, 3, 9, 23, 47, 111, 39, 251, 1001, 179, 3985, 535, 15435}},
	{1294, 14, 2013, {1, 1, 3, 13, 5, 45, 51, 123, 205, 651, 1583, 1691, 1631, 11975}},
	{1295, 14, 2037, {1, 1, 7, 9, 1, 29, 59, 27, 389, 497, 1459, 1633, 521, 14037}},
	{1296, 14, 2051, {1, 1, 3, 3, 3, 23, 35, 247, 371, 729, 931, 681, 1777, 8353}},
	{1297, 14, 2063, {1, 3, 3, 1, 19, 15, 17, 191, 495, 643, 319, 37, 5691, 7049}},
	{1298, 14, 2066, {1, 3, 5, 11, 5, 31, 123, 243, 335, 573, 113, 209, 4825, 7783}},
	{1299, 14, 2094, {1, 3, 7, 7, 29, 19, 25, 191, 89, 515, 55, 3013, 4523, 12913}},
	{1300, 14, 2128, {1, 1, 3, 3, 15, 3, 35, 37, 339, 7, 697, 359, 4553, 1431}},
	{1301, 14, 2154, {1, 3, 1, 1, 9, 15, 33, 77, 161, 13, 255, 1187, 6587, 11715}},
	{1302, 14, 2164, {1, 3, 7, 7, 25, 57, 61, 171, 231, 43, 1219, 903, 5623, 4781}},
	{1303, 14, 2198, {1, 1, 5, 15, 29, 47, 117, 23, 213, 907, 1423, 369, 4529, 9651}},
	{1304, 14, 2217, {1, 1, 5, 7, 15, 55, 105, 249, 401, 37, 1885, 3779, 3441, 9057}},
	{1305, 14, 2220, {1, 1, 5, 3, 3, 27, 49, 89, 335, 561, 1235, 3251, 2731, 12711}},
	{1306, 14, 2223, {1, 1, 1, 15, 29, 49, 37, 173, 25, 743, 1321, 821, 5899, 9213}},
	{1307, 14, 2238, {1, 1, 7, 3, 1, 41, 61, 209, 275, 925, 521, 3029, 1569, 9277}},
	{1308, 14, 2245, {1, 3, 5, 13, 17, 1, 11, 171, 441, 119, 1589, 299, 157, 11439}},
	{1309, 14, 2252, {1, 1, 5, 9, 13, 33, 27, 77, 363, 939, 1103, 2135, 1759, 5429}},
	{1310, 14, 2258, {1, 3, 7, 1, 17, 39, 49, 201, 49, 803, 2003, 1193, 7415, 13847}},
	{1311, 14, 2264, {1, 1, 5, 5, 17, 49, 39, 19, 311, 801, 1441, 3263, 7973, 14181}},
	{1312, 14, 2280, {1, 1, 3, 9, 9, 27, 59, 89, 81, 473, 1369, 3121, 7929, 10905}},
	{1313, 14, 2285, {1, 3, 3, 5, 17, 35, 35, 239, 379, 431, 501, 3561, 2059, 9679}},
	{1314, 14, 2293, {1, 3, 5, 15, 25, 29, 113, 179, 269, 891, 301, 2017, 7513, 9379}},
	{1315, 14, 2294, {1, 3, 1, 11, 17, 35, 49, 149, 135, 661, 1691, 3169, 3765, 9003}},
	{1316, 14, 2298, {1, 3, 7, 15, 5, 21, 53, 241, 475, 271, 683, 2351, 2181, 6333}},
	{1317, 14, 2303, {1, 1, 7, 13, 25, 33, 71, 153, 221, 507, 2017, 2401, 7545, 8489}},
	{1318, 14, 2306, {1, 1, 7, 5, 1, 49, 87, 1, 179, 331, 1597, 3713, 809, 11109}},
	{1319, 14, 2311, {1, 3, 1, 5, 5, 61, 93, 39, 479, 977, 1099, 1291, 7049, 2797}},
	{1320, 14, 2326, {1, 3, 1, 13, 19, 41, 57, 77, 5, 117, 125, 115, 3969, 1345}},
	{1321, 14, 2354, {1, 1, 1, 9, 15, 9, 57, 7, 219, 41, 767, 23, 5771, 14175}},
	{1322, 14, 2373, {1, 3, 7, 9, 17, 61, 1, 59, 227, 349, 63, 189, 3871, 7919}},
	{1323, 14, 2380, {1, 3, 5, 5, 9, 29, 33, 203, 413, 701, 1129, 2103, 1889, 8377}},
	{1324, 14, 2385, {1, 1, 3, 1, 9, 17, 69, 115, 123, 1001, 1, 2893, 3957, 8593}},
	{1325, 14, 2392, {1, 1, 3, 1, 31, 41, 83, 91, 113, 195, 1121, 2665, 6815, 1189}},
	{1326, 14, 2401, {1, 1, 1, 13, 3, 59, 13, 123, 95, 103, 1689, 2809, 5049, 4055}},
	{1327, 14, 2402, {1, 1, 1, 15, 21, 41, 11, 167, 375, 589, 207, 1631, 1597, 8091}},
	{1328, 14, 2408, {1, 3, 5, 5, 1, 33, 57, 89, 157, 921, 1353, 2777, 461, 14567}},
	{1329, 14, 2419, {1, 1, 5, 1, 25, 5, 51, 247, 1, 577, 463, 3741, 303, 16059}},
	{1330, 14, 2450, {1, 1, 7, 5, 13, 7, 17, 87, 51, 987, 835, 93, 5203, 3973}},
	{1331, 14, 2452, {1, 1, 7, 7, 3, 27, 7, 1, 135, 171, 231, 3349, 4459, 2925}},
	{1332, 14, 2477, {1, 1, 5, 5, 9, 51, 71, 153, 115, 315, 265, 2207, 4127, 12631}},
	{1333, 14, 2509, {1, 1, 3, 15, 23, 59, 35, 121, 425, 921, 1255, 2123, 5811, 15937}},
	{1334, 14, 2510, {1, 3, 7, 7, 11, 21, 45, 57, 269, 395, 555, 783, 6677, 2889}},
	{1335, 14, 2515, {1, 3, 5, 7, 31, 19, 73, 35, 465, 349, 1429, 863, 4707, 6121}},
	{1336, 14, 2524, {1, 3, 3, 9, 25, 27, 119, 159, 195, 949, 19, 73, 4511, 15711}},
	{1337, 14, 2527, {1, 3, 3, 7, 9, 59, 47, 57, 91, 749, 1579, 1297, 2445, 5167}},
	{1338, 14, 2531, {1, 3, 3, 3, 31, 57, 19, 203, 61, 927, 1477, 2863, 1305, 11673}},
	{1339, 14, 2552, {1, 3, 7, 11, 29, 13, 3, 111, 351, 79, 1863, 2213, 3273, 7049}},
	{1340, 14, 2561, {1, 3, 3, 9, 7, 23, 47, 237, 121, 877, 441, 119, 2723, 3989}},
	{1341, 14, 2567, {1, 3, 3, 11, 17, 23, 63, 177, 231, 363, 1451, 33, 2169, 7251}},
	{1342, 14, 2571, {1, 1, 5, 11, 31, 41, 93, 229, 39, 1009, 1061, 433, 2393, 15401}},
	{1343, 14, 2586, {1, 1, 5, 15, 31, 37, 25, 135, 135, 897, 33, 3713, 7663, 8079}},
	{1344, 14, 2588, {1, 1, 5, 7, 17, 49, 43, 89, 411, 731, 1431, 3893, 1635, 7063}},
	{1345, 14, 2595, {1, 1, 1, 13, 29, 27, 5, 77, 283, 913, 789, 817, 3309, 475}},
	{1346, 14, 2607, {1, 1, 3, 1, 19, 21, 67, 77, 423, 551, 5, 1057, 5469, 7859}},
	{1347, 14, 2621, {1, 1, 5, 1, 1, 21, 99, 237, 215, 759, 1505, 1983, 1517, 8923}},
	{1348, 14, 2630, {1, 3, 5, 7, 19, 61, 73, 215, 165, 127, 205, 259, 7755, 15395}},
	{1349, 14, 2639, {1, 1, 5, 9, 15, 23, 17, 111, 471, 751, 1923, 775, 6901, 13095}},
	{1350, 14, 2653, {1, 1, 7, 1, 25, 5, 63, 141, 461, 687, 1589, 1559, 7719, 11349}},
	{1351, 14, 2670, {1, 1, 1, 3, 11, 63, 11, 27, 253, 439, 297, 1315, 829, 3765}},
	{1352, 14, 2672, {1, 3, 1, 1, 9, 47, 127, 179, 173, 809, 241, 35, 7355, 5049}},
	{1353, 14, 2700, {1, 3, 1, 11, 19, 63, 93, 1, 205, 977, 303, 3409, 6529, 10927}},
	{1354, 14, 2711, {1, 3, 7, 9, 31, 63, 41, 79, 477, 91, 1801, 3487, 6885, 13341}},
	{1355, 14, 2715, {1, 1, 3, 7, 15, 59, 9, 101, 459, 247, 549, 2855, 5765, 7785}},
	{1356, 14, 2748, {1, 1, 7, 3, 13, 59, 71, 123, 93, 517, 1453, 2389, 4429, 5053}},
	{1357, 14, 2751, {1, 1, 5, 3, 19, 21, 77, 53, 81, 879, 1653, 1637, 3667, 2623}},
	{1358, 14, 2753, {1, 1, 1, 15, 17, 57, 65, 53, 407, 765, 417, 497, 5009, 2175}},
	{1359, 14, 2754, {1, 3, 3, 7, 31, 13, 5, 203, 263, 17, 119, 1607, 6773, 11195}},
	{1360, 14, 2760, {1, 3, 3, 13, 19, 13, 13, 147, 93, 735, 689, 781, 655, 6853}},
	{1361, 14, 2774, {1, 1, 1, 1, 1, 25, 63, 159, 493, 987, 71, 1249, 5859, 11717}},
	{1362, 14, 2784, {1, 1, 1, 15, 13, 23, 61, 61, 5, 947, 1853, 3331, 467, 8081}},
	{1363, 14, 2793, {1, 1, 3, 9, 19, 61, 65, 189, 95, 309, 283, 1725, 5683, 15463}},
	{1364, 14, 2804, {1, 1, 7, 5, 9, 33, 35, 75, 475, 831, 1445, 1485, 5047, 9631}},
	{1365, 14, 2811, {1, 1, 3, 15, 11, 23, 59, 87, 433, 221, 685, 3113, 4095, 13819}},
	{1366, 14, 2822, {1, 1, 7, 15, 25, 29, 67, 17, 349, 353, 1321, 563, 57, 533}},
	{1367, 14, 2826, {1, 3, 3, 3, 5, 43, 109, 217, 15, 185, 1895, 1015, 1831, 10623}},
	{1368, 14, 2836, {1, 1, 7, 1, 1, 47, 81, 185, 59, 691, 191, 3709, 1535, 13347}},
	{1369, 14, 2839, {1, 1, 5, 1, 23, 57, 83, 217, 457, 771, 1877, 2789, 8143, 4797}},
	{1370, 14, 2840, {1, 1, 3, 7, 23, 35, 79, 49, 227, 205, 1523, 3873, 4843, 10505}},
	{1371, 14, 2893, {1, 1, 1, 1, 17, 43, 121, 95, 205, 35, 189, 2061, 1693, 13273}},
	{1372, 14, 2901, {1, 1, 1, 15, 31, 49, 83, 249, 433, 497, 1949, 1845, 5215, 5971}},
	{1373, 14, 2902, {1, 3, 1, 1, 21, 53, 73, 211, 265, 929, 923, 279, 3621, 9469}},
	{1374, 14, 2905, {1, 3, 7, 7, 1, 57, 13, 45, 467, 705, 371, 1345, 1647, 3411}},
	{1375, 14, 2912, {1, 3, 1, 11, 27, 29, 117, 163, 143, 669, 489, 3913, 7891, 9031}},
	{1376, 14, 2915, {1, 3, 7, 15, 27, 15, 77, 217, 107, 839, 1517, 1543, 357, 10365}},
	{1377, 14, 2918, {1, 1, 1, 5, 31, 17, 107, 245, 345, 939, 1453, 3645, 6865, 16173}},
	{1378, 14, 2939, {1, 3, 5, 5, 9, 61, 43, 97, 453, 917, 945, 2143, 5473, 5611}},
	{1379, 14, 2965, {1, 1, 5, 11, 3, 33, 71, 97, 137, 549, 1605, 3839, 4883, 2677}},
	{1380, 14, 2966, {1, 3, 1, 11, 29, 23, 85, 47, 225, 633, 1613, 1297, 1415, 15813}},
	{1381, 14, 2975, {1, 1, 3, 3, 9, 19, 57, 107, 79, 449, 1951, 753, 6317, 10377}},
	{1382, 14, 2988, {1, 1, 1, 5, 21, 3, 39, 187, 299, 517, 1313, 741, 7259, 4197}},
	{1383, 14, 2993, {1, 1, 5, 13, 1, 39, 39, 41, 381, 123, 1257, 3185, 493, 3723}},
	{1384, 14, 3006, {1, 3, 7, 7, 3, 37, 15, 161, 129, 169, 555, 3605, 4287, 15831}},
	{1385, 14, 3017, {1, 3, 7, 15, 15, 23, 81, 163, 257, 791, 505, 1903, 2703, 11919}},
	{1386, 14, 3031, {1, 3, 7, 7, 27, 63, 17, 147, 111, 851, 1533, 1365, 5359, 3315}},
	{1387, 14, 3038, {1, 3, 7, 1, 15, 5, 61, 143, 385, 261, 1019, 1705, 1737, 14485}},
	{1388, 14, 3041, {1, 3, 5, 5, 25, 17, 49, 229, 431, 567, 1613, 3431, 2139, 2981}},
	{1389, 14, 3042, {1, 3, 5, 11, 17, 57, 71, 241, 31, 1007, 1695, 2965, 149, 14125}},
	{1390, 14, 3051, {1, 1, 3, 11, 7, 49, 39, 101, 5, 501, 1491, 3061, 225, 12255}},
	{1391, 14, 3073, {1, 3, 5, 7, 17, 35, 37, 97, 415, 15, 1349, 997, 2949, 4511}},
	{1392, 14, 3088, {1, 3, 1, 5, 25, 35, 99, 183, 161, 59, 1363, 515, 3767, 3641}},
	{1393, 14, 3097, {1, 1, 7, 15, 7, 15, 127, 137, 281, 67, 139, 2315, 3517, 13371}},
	{1394, 14, 3098, {1, 1, 5, 15, 23, 49, 19, 79, 425, 805, 1035, 429, 7707, 14195}},
	{1395, 14, 3103, {1, 3, 5, 3, 21, 25, 123, 11, 425, 475, 961, 2995, 7405, 5449}},
	{1396, 14, 3104, {1, 1, 7, 1, 21, 1, 75, 231, 451, 595, 719, 2369, 5907, 1227}},
	{1397, 14, 3146, {1, 1, 1, 9, 21, 57, 45, 255, 19, 79, 481, 3363, 3451, 8399}},
	{1398, 14, 3148, {1, 1, 7, 13, 31, 49, 95, 69, 483, 427, 37, 4047, 7057, 9111}},
	{1399, 14, 3153, {1, 3, 3, 11, 3, 61, 87, 79, 499, 91, 771, 1987, 2017, 3381}},
	{1400, 14, 3159, {1, 3, 1, 7, 5, 57, 1, 121, 155, 225, 501, 477, 6555, 9863}},
	{1401, 14, 3182, {1, 3, 7, 11, 27, 49, 83, 213, 61, 283, 1599, 3205, 2525, 8553}},
	{1402, 14, 3187, {1, 1, 1, 9, 9, 49, 3, 51, 141, 33, 301, 2167, 587, 15067}},
	{1403, 14, 3189, {1, 1, 1, 11, 7, 55, 99, 81, 191, 553, 953, 3753, 6731, 1093}},
	{1404, 14, 3199, {1, 1, 3, 3, 11, 59, 57, 235, 297, 197, 853, 1411, 3799, 7527}},
	{1405, 14, 3239, {1, 3, 5, 3, 7, 7, 5, 201, 393, 95, 91, 3273, 6285, 10661}},
	{1406, 14, 3263, {1, 1, 5, 7, 17, 57, 87, 3, 413, 915, 659, 369, 3593, 14429}},
	{1407, 14, 3271, {1, 3, 7, 1, 31, 31, 45, 115, 417, 427, 745, 4087, 953, 1119}},
	{1408, 14, 3275, {1, 3, 7, 3, 29, 43, 45, 221, 41, 641, 451, 173, 2999, 12103}},
	{1409, 14, 3278, {1, 1, 3, 11, 25, 57, 117, 201, 135, 787, 1525, 3879, 3247, 8907}},
	{1410, 14, 3280, {1, 1, 7, 11, 3, 35, 69, 157, 331, 615, 573, 2169, 3575, 289}},
	{1411, 14, 3283, {1, 3, 3, 13, 15, 51, 67, 127, 265, 495, 103, 3145, 2685, 15919}},
	{1412, 14, 3290, {1, 3, 5, 11, 31, 27, 65, 57, 153, 465, 1163, 467, 4103, 4713}},
	{1413, 14, 3311, {1, 3, 7, 3, 23, 31, 9, 51, 239, 417, 1597, 229, 2865, 15199}},
	{1414, 14, 3316, {1, 3, 5, 3, 11, 45, 123, 217, 31, 765, 1009, 2001, 3645, 9407}},
	{1415, 14, 3343, {1, 3, 3, 9, 5, 23, 117, 83, 237, 1017, 251, 1187, 2631, 5151}},
	{1416, 14, 3346, {1, 1, 1, 7, 23, 55, 97, 141, 501, 305, 467, 4061, 2369, 15973}},
	{1417, 14, 3357, {1, 1, 7, 5, 31, 51, 125, 191, 219, 495, 37, 3337, 813, 241}},
	{1418, 14, 3358, {1, 3, 1, 1, 11, 39, 93, 109, 285, 147, 1297, 737, 4051, 7223}},
	{1419, 14, 3361, {1, 3, 1, 15, 13, 17, 57, 171, 463, 163, 609, 1681, 7583, 9231}},
	{1420, 14, 3362, {1, 3, 1, 1, 23, 5, 51, 5, 205, 415, 419, 989, 4239, 10943}},
	{1421, 14, 3364, {1, 1, 3, 15, 3, 13, 65, 145, 387, 59, 395, 1067, 4143, 5649}},
	{1422, 14, 3386, {1, 3, 1, 13, 9, 59, 121, 127, 95, 71, 1541, 1423, 1753, 8041}},
	{1423, 14, 3418, {1, 1, 3, 15, 7, 5, 69, 167, 181, 991, 1189, 4017, 5935, 6669}},
	{1424, 14, 3424, {1, 3, 5, 7, 23, 41, 53, 21, 47, 261, 1231, 2011, 133, 2247}},
	{1425, 14, 3433, {1, 1, 1, 5, 17, 47, 77, 19, 331, 609, 1893, 3965, 3123, 9093}},
	{1426, 14, 3434, {1, 3, 1, 3, 9, 39, 103, 231, 249, 75, 373, 107, 1823, 10801}},
	{1427, 14, 3436, {1, 3, 3, 7, 1, 51, 35, 111, 137, 879, 1221, 225, 4285, 2287}},
	{1428, 14, 3463, {1, 1, 7, 9, 23, 17, 75, 245, 409, 163, 395, 3731, 7111, 6845}},
	{1429, 14, 3467, {1, 1, 3, 13, 29, 47, 75, 153, 497, 621, 1691, 3187, 2125, 10533}},
	{1430, 14, 3477, {1, 1, 7, 7, 9, 7, 55, 159, 255, 417, 1335, 643, 3843, 3733}},
	{1431, 14, 3484, {1, 3, 3, 1, 21, 41, 7, 21, 5, 679, 1655, 95, 5699, 5785}},
	{1432, 14, 3505, {1, 1, 1, 13, 19, 7, 85, 7, 195, 357, 1097, 2893, 2913, 9635}},
	{1433, 14, 3508, {1, 1, 5, 9, 25, 33, 41, 155, 39, 655, 1993, 3117, 3639, 7977}},
	{1434, 14, 3515, {1, 1, 1, 13, 3, 63, 121, 247, 151, 673, 609, 285, 2299, 7663}},
	{1435, 14, 3532, {1, 3, 7, 11, 17, 13, 49, 253, 245, 21, 273, 993, 911, 863}},
	{1436, 14, 3553, {1, 1, 5, 5, 23, 1, 121, 95, 225, 9, 1237, 1183, 6461, 559}},
	{1437, 14, 3554, {1, 3, 7, 13, 3, 7, 121, 151, 233, 561, 281, 3583, 897, 1767}},
	{1438, 14, 3568, {1, 1, 7, 7, 9, 47, 107, 41, 25, 569, 1697, 2299, 6231, 12209}},
	{1439, 14, 3573, {1, 3, 7, 7, 27, 43, 59, 37, 31, 51, 503, 149, 4043, 11847}},
	{1440, 14, 3587, {1, 3, 3, 11, 5, 1, 119, 181, 47, 641, 685, 4017, 637, 16251}},
	{1441, 14, 3589, {1, 3, 3, 7, 11, 1, 101, 7, 239, 747, 307, 1721, 5979, 4367}},
	{1442, 14, 3596, {1, 3, 5, 7, 1, 63, 19, 151, 469, 333, 1587, 2453, 897, 4711}},
	{1443, 14, 3608, {1, 3, 1, 5, 11, 61, 21, 253, 91, 993, 1347, 1993, 5607, 13031}},
	{1444, 14, 3620, {1, 3, 5, 5, 1, 39, 65, 71, 189, 389, 1437, 1055, 6439, 3989}},
	{1445, 14, 3630, {1, 1, 3, 3, 19, 15, 93, 3, 339, 165, 1675, 3953, 2145, 12113}},
	{1446, 14, 3644, {1, 1, 3, 13, 13, 45, 5, 175, 211, 993, 705, 2761, 3023, 13633}},
	{1447, 14, 3649, {1, 1, 3, 1, 19, 39, 121, 29, 287, 87, 281, 3491, 7107, 13007}},
	{1448, 14, 3664, {1, 1, 7, 1, 29, 49, 103, 187, 39, 923, 51, 1533, 3249, 4399}},
	{1449, 14, 3679, {1, 1, 5, 5, 5, 43, 25, 107, 453, 955, 115, 57, 4589, 14573}},
	{1450, 14, 3680, {1, 1, 3, 5, 21, 45, 103, 99, 183, 987, 1207, 1697, 8033, 13703}},
	{1451, 14, 3685, {1, 1, 5, 7, 11, 23, 9, 17, 261, 749, 1957, 935, 6283, 8625}},
	{1452, 14, 3686, {1, 1, 1, 9, 9, 51, 69, 225, 265, 323, 1161, 2993, 7305, 2249}},
	{1453, 14, 3698, {1, 3, 1, 9, 23, 19, 57, 205, 503, 489, 1499, 3277, 817, 11931}},
	{1454, 14, 3714, {1, 3, 3, 5, 1, 7, 49, 1, 313, 123, 643, 2027, 1469, 3585}},
	{1455, 14, 3726, {1, 3, 7, 11, 27, 47, 95, 111, 27, 213, 465, 3693, 3661, 7531}},
	{1456, 14, 3737, {1, 1, 7, 9, 3, 37, 115, 189, 31, 613, 1393, 1229, 4767, 12425}},
	{1457, 14, 3767, {1, 1, 3, 3, 25, 17, 99, 47, 161, 931, 959, 1293, 7095, 8325}},
	{1458, 14, 3782, {1, 1, 1, 7, 23, 9, 11, 51, 205, 419, 479, 1497, 2493, 13921}},
	{1459, 14, 3786, {1, 3, 1, 9, 17, 29, 51, 79, 159, 435, 477, 413, 3815, 5589}},
	{1460, 14, 3793, {1, 3, 7, 5, 7, 23, 99, 43, 169, 665, 403, 1163, 4337, 1335}},
	{1461, 14, 3796, {1, 3, 1, 5, 25, 27, 125, 249, 421, 267, 1259, 4089, 59, 9377}},
	{1462, 14, 3805, {1, 3, 3, 1, 27, 37, 91, 17, 123, 597, 1749, 3449, 6503, 11043}},
	{1463, 14, 3815, {1, 3, 7, 7, 23, 41, 19, 245, 109, 569, 547, 1917, 7943, 2697}},
	{1464, 14, 3841, {1, 3, 7, 7, 9, 1, 123, 105, 329, 435, 2013, 2745, 347, 11045}},
	{1465, 14, 3847, {1, 1, 1, 13, 29, 53, 51, 67, 105, 89, 1887, 3543, 963, 8159}},
	{1466, 14, 3853, {1, 1, 5, 3, 5, 27, 41, 67, 67, 883, 973, 1785, 901, 14969}},
	{1467, 14, 3862, {1, 3, 3, 13, 17, 11, 117, 115, 163, 939, 79, 641, 4365, 2267}},
	{1468, 14, 3875, {1, 1, 3, 3, 9, 5, 41, 123, 149, 9, 1533, 3939, 5995, 12701}},
	{1469, 14, 3902, {1, 1, 1, 15, 31, 1, 101, 229, 191, 965, 61, 2671, 4177, 15779}},
	{1470, 14, 3904, {1, 1, 5, 15, 1, 25, 49, 185, 33, 697, 1043, 2639, 7819, 3171}},
	{1471, 14, 3916, {1, 3, 5, 13, 19, 9, 111, 49, 47, 847, 1865, 717, 5287, 13417}},
	{1472, 14, 3947, {1, 3, 7, 11, 5, 61, 63, 111, 171, 735, 2003, 73, 5701, 647}},
	{1473, 14, 3949, {1, 3, 1, 11, 1, 49, 121, 79, 431, 671, 1241, 1161, 2057, 263}},
	{1474, 14, 3955, {1, 3, 3, 1, 1, 23, 75, 15, 117, 641, 313, 1525, 2041, 1409}},
	{1475, 14, 3962, {1, 3, 5, 11, 15, 57, 13, 67, 139, 131, 1339, 2419, 7945, 11877}},
	{1476, 14, 3971, {1, 3, 1, 1, 19, 39, 97, 83, 297, 595, 1611, 5, 4753, 3435}},
	{1477, 14, 3980, {1, 3, 1, 9, 7, 49, 125, 101, 383, 717, 63, 2295, 3873, 13461}},
	{1478, 14, 3985, {1, 1, 3, 3, 15, 29, 89, 77, 269, 689, 229, 1207, 7311, 8663}},
	{1479, 14, 3998, {1, 1, 1, 1, 1, 61, 25, 255, 203, 233, 271, 987, 2277, 8735}},
	{1480, 14, 4001, {1, 1, 5, 7, 21, 27, 63, 79, 337, 133, 1453, 3633, 6157, 15875}},
	{1481, 14, 4002, {1, 3, 1, 7, 7, 55, 31, 81, 203, 709, 1743, 1677, 4247, 11411}},
	{1482, 14, 4016, {1, 1, 3, 3, 29, 51, 37, 17, 487, 325, 1393, 1433, 3467, 2851}},
	{1483, 14, 4021, {1, 1, 7, 9, 3, 41, 99, 177, 241, 869, 739, 2729, 4585, 14801}},
	{1484, 14, 4026, {1, 1, 3, 1, 9, 43, 97, 65, 99, 295, 1693, 2083, 3241, 4073}},
	{1485, 14, 4043, {1, 1, 1, 9, 5, 39, 67, 119, 235, 543, 795, 2773, 3195, 6273}},
	{1486, 14, 4079, {1, 1, 5, 5, 21, 41, 89, 1, 85, 81, 57, 2951, 1531, 10101}},
	{1487, 14, 4102, {1, 1, 1, 7, 3, 35, 127, 69, 39, 265, 1643, 2973, 267, 12365}},
	{1488, 14, 4106, {1, 3, 1, 1, 21, 57, 99, 205, 119, 477, 1771, 1989, 2761, 12573}},
	{1489, 14, 4119, {1, 1, 3, 13, 1, 59, 93, 125, 279, 935, 1877, 2061, 4845, 7835}},
	{1490, 14, 4126, {1, 1, 7, 9, 7, 45, 69, 99, 273, 35, 1579, 2137, 7175, 6999}},
	{1491, 14, 4147, {1, 1, 7, 7, 29, 21, 127, 91, 9, 383, 787, 1783, 601, 5047}},
	{1492, 14, 4149, {1, 1, 7, 13, 7, 29, 35, 219, 43, 581, 2043, 2211, 6169, 12173}},
	{1493, 14, 4164, {1, 3, 5, 13, 29, 29, 39, 63, 411, 645, 415, 2383, 1989, 11411}},
	{1494, 14, 4174, {1, 1, 7, 9, 15, 9, 87, 95, 321, 709, 543, 3831, 2453, 4167}},
	{1495, 14, 4181, {1, 3, 1, 5, 31, 25, 5, 85, 239, 487, 1613, 3937, 4661, 3535}},
	{1496, 14, 4185, {1, 3, 5, 11, 27, 41, 3, 201, 39, 91, 1997, 237, 5639, 14703}},
	{1497, 14, 4188, {1, 1, 7, 3, 27, 49, 87, 71, 473, 247, 1007, 47, 475, 5413}},
	{1498, 14, 4202, {1, 3, 7, 15, 9, 57, 81, 149, 287, 333, 1911, 3417, 1081, 8995}},
	{1499, 14, 4228, {1, 1, 5, 1, 3, 63, 43, 151, 97, 431, 961, 1019, 5153, 2407}},
	{1500, 14, 4232, {1, 1, 5, 5, 27, 21, 127, 161, 507, 311, 129, 3489, 1133, 3241}},
	{1501, 14, 4246, {1, 3, 7, 15, 21, 33, 117, 83, 497, 667, 1399, 931, 1695, 8171}},
	{1502, 14, 4252, {1, 1, 1, 13, 3, 39, 53, 27, 193, 993, 671, 1871, 7579, 11457}},
	{1503, 14, 4256, {1, 1, 5, 11, 7, 39, 81, 107, 195, 387, 849, 395, 1317, 6487}},
	{1504, 14, 4286, {1, 3, 3, 3, 3, 15, 45, 127, 279, 111, 331, 357, 4637, 4697}},
	{1505, 14, 4303, {1, 1, 3, 9, 21, 49, 47, 97, 61, 101, 181, 1867, 1201, 14099}},
	{1506, 14, 4306, {1, 1, 5, 11, 25, 19, 51, 51, 101, 451, 545, 101, 7497, 9141}},
	{1507, 14, 4311, {1, 1, 1, 3, 13, 53, 119, 81, 377, 245, 765, 251, 3757, 16045}},
	{1508, 14, 4317, {1, 1, 1, 3, 5, 61, 65, 37, 331, 925, 1439, 3219, 2843, 11397}},
	{1509, 14, 4342, {1, 3, 5, 9, 23, 31, 95, 155, 83, 641, 1129, 135, 477, 1623}},
	{1510, 14, 4346, {1, 1, 3, 9, 9, 61, 93, 11, 331, 585, 799, 1417, 1533, 463}},
	{1511, 14, 4377, {1, 1, 7, 7, 21, 51, 61, 29, 467, 935, 11, 3357, 1087, 12337}},
	{1512, 14, 4401, {1, 3, 3, 11, 1, 39, 103, 153, 351, 893, 1823, 835, 2149, 4203}},
	{1513, 14, 4407, {1, 1, 1, 9, 31, 13, 61, 235, 369, 359, 835, 2067, 2697, 15289}},
	{1514, 14, 4414, {1, 1, 7, 1, 15, 1, 107, 27, 201, 451, 1521, 313, 3195, 3847}},
	{1515, 14, 4422, {1, 1, 5, 13, 1, 27, 63, 137, 355, 489, 2039, 1015, 2519, 13797}},
	{1516, 14, 4431, {1, 1, 7, 9, 29, 33, 23, 197, 49, 555, 1087, 3447, 7299, 15513}},
	{1517, 14, 4434, {1, 3, 5, 11, 7, 37, 55, 63, 443, 573, 1715, 631, 3405, 6155}},
	{1518, 14, 4436, {1, 3, 3, 3, 31, 35, 51, 167, 225, 617, 2007, 2555, 6819, 12709}},
	{1519, 14, 4443, {1, 1, 1, 13, 15, 5, 73, 85, 109, 43, 1067, 3941, 1125, 10269}},
	{1520, 14, 4459, {1, 1, 7, 11, 17, 3, 127, 145, 279, 19, 1007, 3287, 4751, 12507}},
	{1521, 14, 4461, {1, 3, 7, 3, 19, 1, 117, 111, 193, 435, 47, 1801, 529, 8547}},
	{1522, 14, 4462, {1, 3, 3, 13, 1, 19, 101, 19, 469, 187, 207, 1243, 8153, 3273}},
	{1523, 14, 4473, {1, 3, 1, 5, 11, 51, 69, 189, 453, 775, 241, 3331, 4067, 14759}},
	{1524, 14, 4497, {1, 1, 1, 1, 23, 55, 113, 133, 497, 731, 391, 2777, 3529, 955}},
	{1525, 14, 4504, {1, 3, 1, 11, 5, 49, 59, 35, 261, 949, 325, 3595, 7433, 11099}},
	{1526, 14, 4507, {1, 3, 5, 9, 13, 37, 103, 219, 329, 865, 1787, 2497, 7249, 9877}},
	{1527, 14, 4525, {1, 3, 7, 9, 11, 33, 19, 255, 191, 935, 1115, 1901, 1577, 9623}},
	{1528, 14, 4534, {1, 1, 5, 7, 29, 23, 77, 43, 283, 143, 1211, 73, 2835, 10235}},
	{1529, 14, 4538, {1, 1, 7, 3, 3, 27, 35, 173, 453, 425, 1225, 3023, 2159, 8433}},
	{1530, 14, 4548, {1, 1, 1, 5, 27, 21, 35, 25, 71, 145, 1545, 523, 4527, 7655}},
	{1531, 14, 4552, {1, 1, 5, 3, 13, 49, 61, 157, 113, 775, 763, 1785, 225, 11851}},
	{1532, 14, 4560, {1, 1, 3, 1, 31, 57, 97, 229, 291, 777, 213, 4067, 921, 8203}},
	{1533, 14, 4575, {1, 1, 5, 1, 25, 13, 125, 123, 263, 207, 119, 3111, 3841, 843}},
	{1534, 14, 4599, {1, 1, 7, 7, 25, 57, 81, 129, 31, 133, 1869, 2949, 5563, 14965}},
	{1535, 14, 4612, {1, 3, 3, 7, 3, 51, 33, 127, 281, 425, 1253, 405, 7941, 8799}},
	{1536, 14, 4619, {1, 1, 3, 9, 3, 63, 93, 173, 255, 609, 49, 111, 7785, 15865}},
	{1537, 14, 4640, {1, 1, 1, 3, 17, 59, 113, 55, 155, 789, 1335, 177, 3071, 1851}},
	{1538, 14, 4643, {1, 3, 7, 15, 15, 23, 35, 35, 131, 623, 47, 437, 1337, 9891}},
	{1539, 14, 4677, {1, 3, 7, 5, 29, 57, 39, 31, 111, 271, 59, 1473, 949, 3899}},
	{1540, 14, 4687, {1, 1, 3, 11, 17, 19, 41, 229, 259, 691, 1455, 3023, 7455, 9711}},
	{1541, 14, 4723, {1, 3, 5, 11, 29, 13, 9, 165, 499, 355, 1415, 1395, 7595, 15571}},
	{1542, 14, 4730, {1, 3, 1, 9, 5, 5, 25, 247, 185, 241, 1325, 3133, 7471, 2649}},
	{1543, 14, 4736, {1, 3, 3, 11, 17, 29, 57, 61, 51, 203, 993, 1837, 3785, 15163}},
	{1544, 14, 4741, {1, 1, 7, 7, 21, 57, 79, 165, 277, 133, 93, 1055, 7169, 15685}},
	{1545, 14, 4763, {1, 1, 5, 3, 5, 17, 25, 177, 95, 323, 367, 1359, 4915, 6409}},
	{1546, 14, 4765, {1, 1, 1, 1, 11, 25, 115, 45, 373, 221, 1483, 591, 6561, 4527}},
	{1547, 14, 4770, {1, 3, 5, 3, 5, 23, 69, 77, 313, 473, 1037, 4045, 3969, 5445}},
	{1548, 14, 4781, {1, 3, 1, 5, 1, 15, 73, 83, 439, 463, 203, 361, 6835, 1061}},
	{1549, 14, 4808, {1, 1, 3, 11, 21, 5, 89, 233, 405, 253, 773, 3901, 6085, 5677}},
	{1550, 14, 4822, {1, 1, 3, 9, 15, 53, 71, 29, 101, 599, 1073, 705, 4507, 12779}},
	{1551, 14, 4828, {1, 1, 3, 1, 3, 9, 27, 97, 207, 859, 417, 735, 2179, 5071}},
	{1552, 14, 4831, {1, 1, 1, 3, 13, 63, 65, 125, 195, 611, 649, 2221, 3143, 143}},
	{1553, 14, 4842, {1, 3, 3, 15, 17, 57, 99, 119, 243, 407, 1229, 813, 5245, 1893}},
	{1554, 14, 4855, {1, 1, 1, 5, 27, 27, 49, 13, 313, 287, 473, 2629, 3509, 11371}},
	{1555, 14, 4859, {1, 1, 7, 7, 23, 3, 75, 59, 245, 689, 1215, 2375, 3325, 1593}},
	{1556, 14, 4867, {1, 3, 1, 5, 21, 51, 43, 107, 91, 611, 1405, 677, 2087, 9565}},
	{1557, 14, 4870, {1, 3, 7, 11, 9, 27, 81, 101, 449, 201, 1507, 2217, 6767, 8059}},
	{1558, 14, 4881, {1, 1, 3, 9, 13, 41, 21, 195, 421, 315, 347, 2621, 2359, 9247}},
	{1559, 14, 4893, {1, 1, 5, 7, 31, 45, 77, 229, 455, 575, 1087, 1147, 2273, 13773}},
	{1560, 14, 4910, {1, 1, 1, 1, 9, 5, 87, 19, 207, 545, 1435, 495, 1299, 4947}},
	{1561, 14, 4917, {1, 1, 3, 3, 15, 9, 63, 67, 219, 735, 1911, 2361, 6503, 11977}},
	{1562, 14, 4929, {1, 3, 1, 9, 31, 27, 103, 153, 81, 939, 461, 2753, 697, 537}},
	{1563, 14, 4939, {1, 3, 3, 9, 21, 53, 49, 211, 415, 817, 321, 3775, 2921, 9473}},
	{1564, 14, 4947, {1, 1, 7, 3, 23, 55, 15, 51, 435, 1013, 73, 3967, 4575, 13099}},
	{1565, 14, 4949, {1, 1, 3, 7, 5, 27, 43, 225, 267, 21, 1261, 603, 6913, 4421}},
	{1566, 14, 4954, {1, 1, 7, 13, 25, 31, 101, 109, 237, 91, 1587, 1987, 2795, 6991}},
	{1567, 14, 4972, {1, 1, 3, 13, 23, 51, 91, 89, 287, 39, 1513, 463, 6135, 10469}},
	{1568, 14, 4975, {1, 3, 3, 1, 9, 43, 125, 157, 369, 495, 1849, 785, 6357, 6557}},
	{1569, 14, 5000, {1, 3, 1, 13, 5, 25, 107, 139, 367, 239, 1671, 1239, 7027, 5291}},
	{1570, 14, 5005, {1, 3, 5, 13, 11, 13, 35, 177, 45, 939, 251, 59, 333, 13105}},
	{1571, 14, 5029, {1, 3, 5, 7, 29, 57, 109, 227, 435, 739, 423, 1941, 3345, 12731}},
	{1572, 14, 5039, {1, 3, 3, 9, 23, 51, 19, 207, 69, 99, 955, 519, 7305, 2415}},
	{1573, 14, 5044, {1, 1, 5, 13, 17, 1, 67, 201, 61, 403, 1059, 2915, 2419, 12773}},
	{1574, 14, 5051, {1, 3, 1, 11, 17, 19, 25, 27, 207, 299, 143, 1955, 5669, 2301}},
	{1575, 14, 5056, {1, 1, 5, 3, 25, 57, 45, 255, 489, 1011, 1699, 2637, 5279, 12211}},
	{1576, 14, 5073, {1, 3, 3, 15, 7, 47, 113, 33, 511, 907, 1815, 1741, 2091, 13857}},
	{1577, 14, 5096, {1, 3, 3, 5, 5, 27, 95, 3, 353, 253, 947, 393, 1815, 14551}},
	{1578, 14, 5128, {1, 1, 5, 11, 29, 19, 63, 117, 293, 861, 2039, 9, 5999, 6909}},
	{1579, 14, 5134, {1, 3, 7, 3, 15, 63, 107, 173, 509, 817, 99, 2825, 131, 7917}},
	{1580, 14, 5161, {1, 3, 1, 1, 29, 49, 33, 153, 119, 777, 1315, 3581, 5675, 4043}},
	{1581, 14, 5179, {1, 3, 5, 15, 13, 11, 17, 147, 327, 305, 367, 3237, 5423, 13757}},
	{1582, 14, 5193, {1, 1, 5, 13, 1, 39, 35, 29, 25, 751, 1365, 2243, 8181, 7063}},
	{1583, 14, 5199, {1, 3, 7, 11, 25, 53, 11, 111, 289, 755, 1201, 691, 3679, 3725}},
	{1584, 14, 5202, {1, 1, 1, 11, 11, 37, 33, 211, 395, 691, 1817, 861, 6485, 12077}},
	{1585, 14, 5204, {1, 3, 3, 11, 21, 3, 111, 171, 305, 561, 1501, 2011, 7841, 10931}},
	{1586, 14, 5218, {1, 3, 7, 9, 9, 59, 109, 113, 31, 915, 103, 1861, 2779, 10619}},
	{1587, 14, 5247, {1, 1, 1, 1, 7, 25, 61, 97, 103, 723, 1145, 3105, 371, 339}},
	{1588, 14, 5260, {1, 1, 7, 13, 17, 9, 113, 51, 233, 209, 1117, 211, 6969, 2347}},
	{1589, 14, 5271, {1, 1, 5, 9, 25, 43, 21, 217, 327, 735, 197, 1063, 799, 801}},
	{1590, 14, 5301, {1, 1, 7, 13, 9, 13, 73, 33, 415, 923, 863, 1999, 5383, 8119}},
	{1591, 14, 5305, {1, 3, 1, 5, 7, 33, 51, 185, 289, 967, 1277, 1011, 767, 15505}},
	{1592, 14, 5319, {1, 3, 3, 13, 21, 11, 105, 235, 343, 1021, 2009, 2251, 3865, 6923}},
	{1593, 14, 5326, {1, 3, 5, 9, 29, 11, 33, 17, 149, 155, 1739, 3039, 7015, 2401}},
	{1594, 14, 5328, {1, 3, 7, 7, 17, 13, 89, 177, 297, 267, 545, 3861, 329, 13267}},
	{1595, 14, 5333, {1, 3, 5, 15, 27, 33, 1, 231, 181, 557, 447, 379, 7845, 1295}},
	{1596, 14, 5364, {1, 1, 5, 13, 3, 63, 59, 33, 263, 877, 1867, 1383, 641, 7139}},
	{1597, 14, 5376, {1, 3, 7, 5, 13, 51, 9, 113, 223, 605, 1189, 4063, 6925, 9563}},
	{1598, 14, 5399, {1, 1, 1, 13, 5, 35, 83, 107, 295, 231, 265, 5, 4087, 6407}},
	{1599, 14, 5416, {1, 1, 5, 1, 7, 25, 95, 137, 97, 987, 1753, 2781, 1369, 6903}},
	{1600, 14, 5421, {1, 1, 5, 13, 19, 61, 77, 229, 193, 165, 811, 249, 79, 10719}},
	{1601, 14, 5427, {1, 3, 7, 7, 27, 9, 119, 193, 459, 43, 1989, 2959, 3595, 6341}},
	{1602, 14, 5429, {1, 1, 5, 11, 5, 43, 35, 33, 25, 581, 897, 351, 4201, 3971}},
	{1603, 14, 5430, {1, 1, 7, 11, 21, 29, 53, 45, 359, 197, 313, 3825, 6717, 4077}},
	{1604, 14, 5434, {1, 1, 1, 15, 3, 45, 99, 133, 357, 315, 1159, 241, 2463, 11253}},
	{1605, 14, 5441, {1, 1, 7, 11, 9, 33, 111, 85, 443, 601, 447, 337, 6471, 7029}},
	{1606, 14, 5451, {1, 3, 7, 9, 13, 33, 25, 31, 9, 729, 1763, 4077, 7575, 7877}},
	{1607, 14, 5465, {1, 3, 5, 13, 13, 37, 29, 103, 53, 229, 591, 1073, 1323, 14405}},
	{1608, 14, 5466, {1, 1, 5, 1, 17, 33, 15, 183, 473, 297, 2003, 93, 4955, 1787}},
	{1609, 14, 5471, {1, 1, 5, 13, 5, 29, 113, 161, 267, 451, 1193, 149, 273, 11809}},
	{1610, 14, 5477, {1, 1, 1, 9, 17, 39, 47, 233, 165, 373, 955, 2891, 7523, 7235}},
	{1611, 14, 5492, {1, 1, 1, 3, 7, 21, 115, 205, 153, 449, 339, 2073, 1077, 5749}},
	{1612, 14, 5495, {1, 1, 7, 13, 9, 39, 117, 187, 37, 753, 227, 3519, 7391, 5751}},
	{1613, 14, 5505, {1, 1, 1, 9, 5, 19, 41, 161, 141, 195, 1719, 3321, 5, 12877}},
	{1614, 14, 5515, {1, 3, 7, 11, 21, 13, 83, 55, 299, 75, 1905, 3765, 4685, 12297}},
	{1615, 14, 5525, {1, 1, 7, 3, 3, 23, 111, 243, 187, 297, 1061, 2515, 977, 9555}},
	{1616, 14, 5529, {1, 3, 7, 3, 29, 11, 103, 177, 225, 875, 1649, 1401, 6383, 8309}},
	{1617, 14, 5532, {1, 3, 5, 3, 3, 41, 71, 3, 373, 757, 701, 2825, 1521, 13217}},
	{1618, 14, 5539, {1, 1, 5, 3, 11, 5, 103, 227, 209, 723, 1543, 3895, 6345, 7901}},
	{1619, 14, 5541, {1, 1, 5, 1, 9, 51, 77, 67, 359, 937, 557, 993, 3871, 3577}},
	{1620, 14, 5556, {1, 3, 7, 1, 1, 15, 121, 239, 29, 113, 1123, 3877, 6941, 14129}},
	{1621, 14, 5566, {1, 1, 5, 1, 27, 61, 83, 113, 185, 601, 947, 3933, 381, 13869}},
	{1622, 14, 5568, {1, 1, 5, 3, 5, 37, 97, 31, 81, 367, 747, 1811, 5313, 14151}},
	{1623, 14, 5574, {1, 3, 5, 9, 27, 61, 87, 31, 185, 521, 837, 959, 5001, 3957}},
	{1624, 14, 5595, {1, 3, 5, 3, 11, 61, 37, 19, 107, 749, 1345, 3829, 6701, 4315}},
	{1625, 14, 5602, {1, 3, 1, 15, 13, 45, 101, 113, 243, 963, 1861, 3283, 1419, 12131}},
	{1626, 14, 5611, {1, 1, 7, 1, 11, 63, 17, 117, 271, 819, 677, 669, 1991, 12511}},
	{1627, 14, 5616, {1, 1, 1, 13, 13, 33, 41, 73, 187, 537, 993, 3147, 1013, 16063}},
	{1628, 14, 5622, {1, 3, 1, 1, 25, 21, 107, 81, 117, 917, 113, 349, 4475, 9149}},
	{1629, 14, 5628, {1, 1, 1, 11, 21, 21, 29, 251, 125, 681, 141, 2893, 5843, 14359}},
	{1630, 14, 5655, {1, 3, 3, 1, 5, 41, 85, 163, 387, 29, 1593, 221, 2769, 10809}},
	{1631, 14, 5662, {1, 3, 5, 11, 1, 17, 69, 127, 273, 449, 1855, 2971, 7031, 10583}},
	{1632, 14, 5675, {1, 1, 5, 7, 1, 61, 9, 211, 123, 563, 111, 1883, 5801, 2191}},
	{1633, 14, 5689, {1, 1, 3, 11, 11, 51, 1, 81, 405, 803, 2017, 161, 5429, 731}},
	{1634, 14, 5722, {1, 1, 7, 9, 15, 55, 65, 51, 459, 485, 1539, 3135, 2929, 7867}},
	{1635, 14, 5724, {1, 1, 7, 11, 3, 45, 15, 7, 331, 417, 1813, 4009, 1341, 10965}},
	{1636, 14, 5728, {1, 1, 1, 5, 9, 29, 89, 121, 277, 509, 1989, 1293, 4787, 16097}},
	{1637, 14, 5731, {1, 1, 3, 9, 17, 45, 97, 197, 339, 943, 1377, 2947, 5833, 7}},
	{1638, 14, 5746, {1, 1, 7, 9, 15, 61, 75, 233, 401, 705, 825, 2521, 3787, 14387}},
	{1639, 14, 5764, {1, 1, 7, 15, 25, 57, 3, 43, 361, 459, 1551, 1859, 6787, 2293}},
	{1640, 14, 5771, {1, 3, 3, 11, 11, 35, 91, 65, 43, 509, 1829, 1149, 4801, 4109}},
	{1641, 14, 5781, {1, 3, 5, 9, 15, 3, 81, 109, 231, 481, 417, 2505, 315, 6693}},
	{1642, 14, 5801, {1, 1, 3, 9, 3, 7, 107, 221, 297, 543, 149, 579, 927, 79}},
	{1643, 14, 5809, {1, 3, 1, 11, 17, 3, 81, 137, 157, 587, 741, 1277, 2631, 3953}},
	{1644, 14, 5810, {1, 1, 7, 5, 13, 43, 117, 19, 495, 185, 1105, 605, 5249, 11099}},
	{1645, 14, 5812, {1, 1, 7, 9, 23, 55, 91, 213, 21, 779, 857, 2047, 7813, 10053}},
	{1646, 14, 5841, {1, 1, 1, 1, 27, 7, 39, 181, 63, 519, 1073, 3147, 4111, 363}},
	{1647, 14, 5848, {1, 3, 7, 9, 15, 61, 7, 139, 495, 805, 1545, 3789, 2411, 3989}},
	{1648, 14, 5853, {1, 1, 3, 1, 25, 11, 23, 241, 167, 607, 479, 153, 7787, 13929}},
	{1649, 14, 5854, {1, 3, 5, 15, 29, 35, 45, 71, 457, 297, 883, 3021, 5361, 15427}},
	{1650, 14, 5858, {1, 3, 1, 7, 29, 27, 93, 241, 427, 89, 1185, 37, 3863, 14095}},
	{1651, 14, 5892, {1, 3, 1, 5, 5, 45, 51, 15, 235, 889, 1649, 2331, 2713, 10943}},
	{1652, 14, 5907, {1, 1, 3, 11, 11, 15, 71, 85, 135, 163, 139, 1147, 1043, 3195}},
	{1653, 14, 5910, {1, 3, 5, 13, 3, 43, 71, 131, 473, 933, 569, 2491, 7751, 1865}},
	{1654, 14, 5913, {1, 1, 7, 9, 21, 37, 105, 227, 329, 509, 1319, 307, 1557, 14625}},
	{1655, 14, 5920, {1, 1, 3, 13, 15, 1, 25, 93, 335, 953, 769, 4039, 369, 10727}},
	{1656, 14, 5929, {1, 3, 7, 5, 17, 21, 59, 89, 437, 679, 437, 1543, 7663, 5005}},
	{1657, 14, 5949, {1, 1, 7, 15, 27, 49, 125, 13, 397, 877, 1087, 2191, 4711, 9065}},
	{1658, 14, 5952, {1, 1, 7, 5, 15, 47, 115, 125, 187, 31, 1003, 2575, 5397, 3883}},
	{1659, 14, 5955, {1, 1, 7, 11, 15, 1, 127, 207, 383, 707, 183, 1053, 3123, 14071}},
	{1660, 14, 5962, {1, 3, 3, 1, 31, 53, 15, 19, 477, 245, 777, 1613, 5813, 7443}},
	{1661, 14, 5975, {1, 3, 1, 11, 23, 59, 65, 23, 493, 157, 1389, 2833, 4535, 3907}},
	{1662, 14, 5985, {1, 1, 7, 1, 19, 7, 51, 135, 327, 441, 1841, 3091, 3451, 14381}},
	{1663, 14, 5997, {1, 1, 7, 7, 3, 37, 29, 249, 437, 319, 1693, 945, 7639, 5923}},
	{1664, 14, 5998, {1, 3, 7, 15, 7, 61, 81, 127, 383, 99, 23, 3833, 3973, 7651}},
	{1665, 14, 6012, {1, 3, 1, 7, 7, 21, 119, 185, 243, 619, 1363, 2033, 4835, 5089}},
	{1666, 14, 6016, {1, 3, 1, 1, 3, 27, 63, 145, 271, 735, 695, 3981, 3049, 5433}},
	{1667, 14, 6026, {1, 3, 3, 1, 3, 29, 79, 211, 279, 819, 501, 3665, 1455, 10455}},
	{1668, 14, 6036, {1, 1, 3, 3, 31, 61, 113, 5, 411, 91, 489, 3257, 5939, 6715}},
	{1669, 14, 6040, {1, 1, 5, 1, 23, 11, 103, 89, 377, 441, 43, 967, 3383, 8717}},
	{1670, 14, 6045, {1, 1, 5, 13, 29, 39, 97, 189, 197, 621, 1755, 333, 6783, 9711}},
	{1671, 14, 6055, {1, 1, 5, 13, 27, 17, 97, 197, 351, 799, 335, 765, 5329, 12549}},
	{1672, 14, 6059, {1, 1, 5, 11, 29, 17, 9, 211, 127, 633, 1187, 3965, 4145, 12667}},
	{1673, 14, 6088, {1, 1, 7, 5, 27, 29, 65, 115, 287, 325, 461, 5, 899, 2027}},
	{1674, 14, 6115, {1, 1, 1, 5, 27, 17, 31, 13, 231, 627, 1163, 649, 1693, 9975}},
	{1675, 14, 6124, {1, 3, 1, 15, 7, 49, 113, 123, 427, 603, 347, 2785, 7129, 4645}},
	{1676, 14, 6127, {1, 1, 3, 7, 1, 33, 113, 105, 411, 939, 205, 3965, 4361, 4649}},
	{1677, 14, 6132, {1, 1, 1, 1, 5, 21, 35, 159, 275, 929, 1193, 3205, 4787, 3515}},
	{1678, 14, 6146, {1, 1, 1, 5, 1, 21, 29, 191, 275, 233, 1239, 515, 4349, 14989}},
	{1679, 14, 6158, {1, 1, 5, 11, 27, 43, 111, 83, 153, 577, 1537, 149, 231, 839}},
	{1680, 14, 6169, {1, 3, 5, 13, 21, 19, 57, 69, 87, 163, 271, 3535, 1057, 8517}},
	{1681, 14, 6206, {1, 3, 3, 13, 17, 49, 65, 45, 457, 241, 391, 2033, 2507, 7771}},
	{1682, 14, 6228, {1, 1, 5, 7, 11, 19, 79, 133, 341, 761, 27, 3905, 4137, 14363}},
	{1683, 14, 6237, {1, 3, 3, 13, 19, 1, 11, 139, 249, 245, 1393, 2151, 2857, 1665}},
	{1684, 14, 6244, {1, 1, 3, 15, 11, 7, 127, 47, 385, 1007, 713, 2235, 5489, 8755}},
	{1685, 14, 6247, {1, 3, 5, 13, 19, 21, 21, 167, 405, 655, 1653, 889, 7367, 4177}},
	{1686, 14, 6256, {1, 1, 5, 3, 19, 63, 99, 39, 89, 415, 951, 2863, 6569, 3797}},
	{1687, 14, 6281, {1, 1, 1, 13, 31, 29, 119, 35, 311, 839, 1749, 941, 7487, 2385}},
	{1688, 14, 6282, {1, 3, 7, 3, 17, 3, 97, 143, 465, 345, 1457, 2201, 5329, 359}},
	{1689, 14, 6284, {1, 3, 7, 11, 1, 15, 3, 115, 335, 567, 1749, 1811, 3491, 15939}},
	{1690, 14, 6296, {1, 1, 3, 13, 3, 21, 7, 141, 149, 571, 1877, 473, 2143, 9569}},
	{1691, 14, 6299, {1, 3, 3, 11, 23, 61, 47, 179, 297, 453, 181, 3405, 2981, 13409}},
	{1692, 14, 6302, {1, 3, 1, 13, 1, 43, 5, 201, 371, 1003, 367, 2709, 7675, 14973}},
	{1693, 14, 6325, {1, 3, 3, 15, 29, 17, 19, 241, 495, 317, 1135, 2227, 6457, 4783}},
	{1694, 14, 6349, {1, 3, 3, 7, 29, 9, 57, 95, 261, 531, 1717, 3389, 7991, 3793}},
	{1695, 14, 6352, {1, 1, 1, 5, 31, 43, 73, 119, 499, 589, 1529, 3337, 4097, 15641}},
	{1696, 14, 6362, {1, 1, 7, 9, 29, 43, 127, 91, 243, 979, 1325, 2835, 2787, 9445}},
	{1697, 14, 6383, {1, 1, 7, 5, 9, 3, 115, 199, 219, 901, 747, 1077, 3197, 2443}},
	{1698, 14, 6386, {1, 3, 5, 1, 3, 43, 7, 117, 297, 313, 1043, 1579, 5099, 13289}},
	{1699, 14, 6395, {1, 1, 7, 11, 29, 33, 15, 121, 131, 579, 317, 1871, 1121, 11653}},
	{1700, 14, 6397, {1, 1, 5, 9, 25, 25, 43, 89, 355, 1011, 1385, 2901, 6387, 1653}},
	{1701, 14, 6415, {1, 1, 1, 9, 5, 47, 61, 165, 85, 757, 1397, 1177, 1657, 4899}},
	{1702, 14, 6424, {1, 1, 3, 9, 11, 49, 15, 139, 261, 613, 931, 1299, 2777, 2835}},
	{1703, 14, 6429, {1, 1, 1, 5, 3, 55, 83, 227, 125, 581, 1607, 1171, 6681, 14463}},
	{1704, 14, 6439, {1, 3, 5, 13, 5, 55, 3, 247, 493, 155, 1073, 3743, 5719, 4019}},
	{1705, 14, 6451, {1, 1, 7, 1, 11, 23, 13, 75, 399, 847, 499, 1643, 6977, 3699}},
	{1706, 14, 6489, {1, 3, 1, 9, 11, 41, 47, 131, 313, 627, 481, 2469, 3281, 979}},
	{1707, 14, 6496, {1, 3, 5, 13, 29, 3, 65, 101, 11, 29, 1807, 153, 1487, 16109}},
	{1708, 14, 6502, {1, 1, 5, 9, 13, 31, 83, 195, 351, 355, 467, 3871, 3085, 4441}},
	{1709, 14, 6511, {1, 3, 5, 3, 19, 21, 111, 179, 143, 361, 1619, 1547, 3409, 6905}},
	{1710, 14, 6514, {1, 1, 5, 9, 31, 1, 93, 199, 491, 135, 1627, 2559, 1389, 14561}},
	{1711, 14, 6520, {1, 3, 3, 9, 25, 53, 3, 105, 39, 445, 259, 1045, 1129, 9153}},
	{1712, 14, 6523, {1, 1, 5, 9, 19, 63, 71, 9, 73, 435, 1377, 4015, 1821, 6453}},
	{1713, 14, 6529, {1, 3, 7, 13, 19, 13, 37, 247, 391, 23, 1491, 1257, 6395, 237}},
	{1714, 14, 6532, {1, 1, 3, 3, 19, 55, 109, 23, 227, 747, 729, 2221, 727, 2209}},
	{1715, 14, 6547, {1, 1, 5, 11, 25, 21, 75, 37, 219, 355, 1005, 1895, 7039, 5225}},
	{1716, 14, 6549, {1, 3, 5, 13, 11, 43, 9, 67, 87, 797, 1077, 245, 4521, 11845}},
	{1717, 14, 6598, {1, 3, 5, 3, 15, 29, 127, 237, 277, 373, 1859, 3083, 587, 1123}},
	{1718, 14, 6601, {1, 1, 7, 15, 13, 7, 103, 53, 13, 965, 1497, 775, 3439, 1501}},
	{1719, 14, 6610, {1, 3, 3, 15, 17, 13, 97, 169, 67, 953, 189, 2739, 1459, 10543}},
	{1720, 14, 6622, {1, 1, 5, 1, 17, 39, 15, 127, 327, 989, 1471, 3235, 2801, 15311}},
	{1721, 14, 6632, {1, 1, 1, 15, 5, 37, 55, 155, 47, 463, 1851, 3467, 2765, 9359}},
	{1722, 14, 6655, {1, 3, 3, 15, 1, 13, 93, 239, 291, 115, 365, 61, 395, 15853}},
	{1723, 14, 6665, {1, 1, 5, 1, 19, 27, 61, 95, 105, 369, 1557, 961, 6917, 3621}},
	{1724, 14, 6666, {1, 3, 3, 9, 7, 35, 115, 53, 111, 345, 1145, 1687, 3401, 12107}},
	{1725, 14, 6695, {1, 1, 1, 5, 7, 31, 63, 19, 373, 79, 1369, 3037, 2835, 4439}},
	{1726, 14, 6701, {1, 3, 7, 9, 11, 17, 29, 33, 331, 447, 1981, 3333, 6535, 6557}},
	{1727, 14, 6709, {1, 3, 3, 5, 11, 41, 29, 43, 365, 279, 1919, 945, 179, 1987}},
	{1728, 14, 6710, {1, 3, 1, 13, 7, 7, 25, 33, 103, 367, 1267, 763, 5691, 8643}},
	{1729, 14, 6741, {1, 3, 1, 5, 11, 15, 3, 213, 511, 211, 1069, 4047, 3335, 12729}},
	{1730, 14, 6745, {1, 1, 3, 1, 5, 11, 27, 201, 361, 537, 679, 3709, 293, 2997}},
	{1731, 14, 6758, {1, 1, 3, 1, 25, 15, 19, 185, 369, 577, 1625, 655, 2363, 3861}},
	{1732, 14, 6767, {1, 1, 5, 5, 1, 47, 61, 45, 411, 597, 955, 1007, 3775, 5809}},
	{1733, 14, 6772, {1, 1, 5, 3, 27, 51, 101, 167, 429, 333, 1703, 3541, 2947, 3695}},
	{1734, 14, 6782, {1, 3, 5, 5, 1, 53, 17, 63, 141, 215, 1223, 3129, 635, 15919}},
	{1735, 14, 6797, {1, 3, 3, 1, 23, 31, 25, 11, 195, 241, 995, 3941, 573, 13855}},
	{1736, 14, 6800, {1, 3, 3, 7, 17, 13, 71, 203, 465, 479, 1857, 1493, 8067, 7113}},
	{1737, 14, 6843, {1, 1, 5, 3, 11, 57, 9, 59, 225, 691, 425, 2423, 6031, 6631}},
	{1738, 14, 6845, {1, 3, 7, 1, 29, 57, 103, 123, 401, 807, 471, 2759, 5113, 15937}},
	{1739, 14, 6860, {1, 3, 1, 1, 3, 1, 67, 123, 157, 655, 519, 323, 1853, 15041}},
	{1740, 14, 6865, {1, 1, 7, 5, 11, 11, 105, 135, 247, 689, 1141, 2347, 7113, 9347}},
	{1741, 14, 6878, {1, 1, 3, 11, 15, 37, 87, 3, 209, 575, 1521, 3863, 3893, 211}},
	{1742, 14, 6887, {1, 3, 1, 3, 29, 55, 115, 31, 19, 195, 985, 3275, 363, 9801}},
	{1743, 14, 6888, {1, 1, 3, 9, 13, 31, 57, 251, 201, 275, 1751, 389, 1463, 13159}},
	{1744, 14, 6901, {1, 3, 5, 15, 19, 51, 127, 255, 397, 243, 29, 3007, 7845, 4687}},
	{1745, 14, 6906, {1, 1, 7, 15, 9, 37, 39, 217, 509, 137, 1123, 3361, 6323, 5323}},
	{1746, 14, 6940, {1, 3, 7, 5, 25, 3, 93, 203, 345, 581, 261, 2811, 4829, 6977}},
	{1747, 14, 6947, {1, 1, 7, 1, 15, 41, 51, 227, 447, 893, 1209, 3865, 5229, 4277}},
	{1748, 14, 6953, {1, 1, 1, 5, 31, 19, 23, 195, 359, 853, 595, 337, 2503, 16371}},
	{1749, 14, 6954, {1, 3, 7, 5, 5, 13, 89, 157, 351, 777, 151, 3565, 4219, 7423}},
	{1750, 14, 6959, {1, 1, 1, 5, 7, 1, 9, 89, 175, 909, 1523, 2295, 7949, 6739}},
	{1751, 14, 6961, {1, 3, 5, 15, 27, 17, 11, 235, 19, 105, 457, 465, 3819, 11335}},
	{1752, 14, 6964, {1, 3, 1, 13, 3, 41, 85, 221, 451, 613, 543, 2265, 6831, 1725}},
	{1753, 14, 6991, {1, 1, 7, 7, 3, 29, 9, 197, 455, 665, 343, 1811, 5395, 393}},
	{1754, 14, 6993, {1, 1, 3, 13, 29, 55, 71, 95, 475, 615, 2029, 123, 413, 16127}},
	{1755, 14, 6999, {1, 1, 5, 9, 15, 61, 9, 51, 105, 271, 511, 2801, 693, 11839}},
	{1756, 14, 7016, {1, 1, 7, 13, 29, 9, 105, 59, 377, 635, 717, 4033, 6963, 10541}},
	{1757, 14, 7029, {1, 1, 1, 13, 7, 13, 59, 17, 335, 355, 77, 3665, 7003, 9521}},
	{1758, 14, 7036, {1, 3, 1, 1, 23, 43, 51, 209, 151, 365, 1021, 2859, 3937, 2899}},
	{1759, 14, 7045, {1, 1, 3, 3, 31, 41, 111, 107, 171, 433, 1233, 505, 2971, 6927}},
	{1760, 14, 7076, {1, 3, 7, 13, 17, 25, 127, 195, 257, 551, 1867, 2145, 3695, 14567}},
	{1761, 14, 7083, {1, 1, 5, 13, 13, 45, 39, 195, 55, 991, 1981, 1043, 5875, 581}},
	{1762, 14, 7094, {1, 3, 3, 11, 25, 31, 91, 153, 415, 449, 1301, 563, 7755, 10671}},
	{1763, 14, 7097, {1, 1, 3, 5, 31, 63, 1, 157, 229, 949, 971, 137, 6589, 8387}},
	{1764, 14, 7123, {1, 3, 7, 15, 25, 7, 89, 133, 73, 497, 1361, 613, 455, 1005}},
	{1765, 14, 7130, {1, 3, 3, 1, 13, 5, 119, 93, 175, 511, 1923, 763, 7573, 7545}},
	{1766, 14, 7139, {1, 1, 3, 15, 27, 59, 49, 205, 497, 485, 117, 2523, 4495, 15153}},
	{1767, 14, 7145, {1, 3, 7, 9, 15, 47, 111, 31, 363, 11, 475, 2931, 6813, 1259}},
	{1768, 14, 7146, {1, 1, 5, 5, 1, 35, 95, 225, 17, 991, 809, 2601, 6455, 13803}},
	{1769, 14, 7178, {1, 1, 5, 5, 15, 1, 1, 171, 433, 887, 1813, 3431, 2471, 7803}},
	{1770, 14, 7186, {1, 3, 3, 15, 1, 15, 43, 179, 15, 949, 1881, 1027, 6989, 8955}},
	{1771, 14, 7192, {1, 3, 7, 13, 1, 3, 49, 183, 373, 175, 1733, 913, 929, 1065}},
	{1772, 14, 7198, {1, 3, 5, 7, 15, 51, 107, 115, 323, 357, 167, 2069, 7541, 9601}},
	{1773, 14, 7222, {1, 1, 3, 5, 5, 21, 31, 107, 21, 299, 1937, 43, 3673, 8155}},
	{1774, 14, 7269, {1, 3, 5, 11, 9, 55, 35, 113, 29, 99, 161, 1607, 8141, 4951}},
	{1775, 14, 7270, {1, 3, 7, 15, 25, 7, 113, 179, 213, 19, 1717, 1027, 2021, 11263}},
	{1776, 14, 7276, {1, 1, 5, 1, 31, 33, 85, 111, 67, 95, 2013, 2217, 871, 5329}},
	{1777, 14, 7287, {1, 1, 1, 7, 7, 63, 67, 145, 495, 419, 1945, 3437, 6255, 151}},
	{1778, 14, 7307, {1, 3, 5, 7, 17, 37, 97, 187, 215, 399, 1603, 2195, 5923, 769}},
	{1779, 14, 7315, {1, 1, 3, 9, 25, 1, 119, 193, 385, 861, 2005, 2769, 675, 767}},
	{1780, 14, 7334, {1, 3, 1, 15, 19, 7, 5, 227, 173, 383, 289, 461, 579, 3689}},
	{1781, 14, 7340, {1, 3, 1, 11, 1, 37, 93, 239, 465, 891, 1479, 921, 4439, 15265}},
	{1782, 14, 7351, {1, 1, 1, 13, 27, 61, 99, 69, 279, 655, 1853, 1593, 6319, 9003}},
	{1783, 14, 7352, {1, 1, 1, 11, 5, 7, 19, 7, 387, 303, 321, 931, 5809, 16029}},
	{1784, 14, 7357, {1, 1, 1, 15, 21, 55, 43, 107, 217, 687, 19, 3225, 3419, 9991}},
	{1785, 14, 7360, {1, 1, 7, 5, 7, 55, 79, 41, 317, 357, 859, 1205, 191, 9395}},
	{1786, 14, 7363, {1, 1, 3, 11, 3, 43, 7, 133, 115, 995, 1205, 1055, 4153, 10481}},
	{1787, 14, 7384, {1, 1, 7, 11, 31, 57, 53, 9, 459, 223, 1969, 3513, 7033, 8505}},
	{1788, 14, 7396, {1, 1, 3, 7, 17, 11, 115, 255, 281, 97, 1685, 2039, 2845, 11637}},
	{1789, 14, 7403, {1, 3, 7, 1, 23, 41, 69, 199, 53, 105, 657, 1453, 4429, 1101}},
	{1790, 14, 7406, {1, 3, 1, 5, 11, 33, 91, 131, 191, 73, 823, 117, 1053, 127}},
	{1791, 14, 7425, {1, 3, 7, 11, 7, 3, 21, 65, 187, 103, 1393, 1797, 6673, 1409}},
	{1792, 14, 7437, {1, 3, 7, 1, 31, 25, 25, 161, 299, 275, 417, 2267, 6861, 1255}},
	{1793, 14, 7445, {1, 3, 5, 13, 5, 11, 61, 155, 115, 1001, 747, 889, 3235, 5709}},
	{1794, 14, 7450, {1, 3, 7, 7, 7, 1, 97, 177, 507, 273, 1781, 3455, 5123, 15607}},
	{1795, 14, 7455, {1, 1, 7, 5, 1, 7, 59, 49, 147, 343, 97, 3517, 5611, 8705}},
	{1796, 14, 7461, {1, 1, 5, 13, 21, 29, 13, 21, 503, 515, 1217, 3905, 5513, 15849}},
	{1797, 14, 7466, {1, 3, 1, 9, 9, 39, 65, 111, 385, 757, 583, 2225, 2039, 2817}},
	{1798, 14, 7488, {1, 3, 3, 15, 23, 17, 63, 169, 503, 949, 849, 461, 6799, 669}},
	{1799, 14, 7494, {1, 1, 1, 3, 1, 41, 63, 159, 251, 457, 521, 1653, 623, 3287}},
	{1800, 14, 7515, {1, 1, 7, 3, 9, 1, 41, 37, 441, 921, 1415, 2955, 5841, 1451}},
	{1801, 14, 7517, {1, 1, 5, 11, 23, 29, 89, 185, 413, 357, 1131, 2369, 3835, 6233}},
	{1802, 14, 7521, {1, 1, 5, 15, 27, 35, 17, 73, 315, 911, 1761, 797, 5349, 3219}},
	{1803, 14, 7536, {1, 3, 7, 11, 21, 9, 119, 233, 249, 901, 189, 3625, 2691, 16201}},
	{1804, 14, 7546, {1, 3, 3, 13, 29, 61, 105, 145, 187, 79, 609, 321, 4289, 3933}},
	{1805, 14, 7569, {1, 3, 1, 15, 19, 63, 13, 185, 115, 219, 1021, 1205, 4273, 11521}},
	{1806, 14, 7591, {1, 1, 3, 3, 23, 31, 93, 153, 87, 947, 1039, 469, 4047, 8869}},
	{1807, 14, 7592, {1, 1, 1, 1, 9, 1, 85, 3, 15, 995, 455, 2769, 6781, 16203}},
	{1808, 14, 7598, {1, 1, 3, 3, 13, 7, 55, 215, 185, 367, 765, 441, 4497, 1521}},
	{1809, 14, 7612, {1, 1, 1, 5, 1, 31, 13, 95, 417, 735, 975, 3407, 4871, 16133}},
	{1810, 14, 7623, {1, 1, 3, 3, 5, 43, 111, 107, 419, 515, 1075, 3597, 1187, 4143}},
	{1811, 14, 7632, {1, 1, 3, 13, 31, 51, 83, 163, 489, 887, 863, 599, 9, 13861}},
	{1812, 14, 7637, {1, 3, 3, 3, 19, 27, 91, 115, 103, 969, 593, 3667, 1867, 15433}},
	{1813, 14, 7644, {1, 3, 3, 13, 7, 25, 47, 141, 57, 553, 1785, 1709, 7453, 2209}},
	{1814, 14, 7657, {1, 3, 1, 13, 11, 13, 71, 219, 5, 451, 2043, 1605, 6439, 12203}},
	{1815, 14, 7665, {1, 3, 1, 13, 5, 57, 61, 223, 401, 413, 321, 1365, 619, 12477}},
	{1816, 14, 7672, {1, 3, 1, 5, 25, 57, 89, 211, 195, 455, 1165, 3979, 6313, 5751}},
	{1817, 14, 7682, {1, 1, 1, 9, 31, 23, 71, 145, 89, 285, 1593, 1171, 5685, 15459}},
	{1818, 14, 7699, {1, 3, 7, 7, 9, 41, 65, 251, 65, 137, 1577, 3027, 5555, 2865}},
	{1819, 14, 7702, {1, 1, 5, 13, 27, 5, 125, 21, 171, 647, 983, 2921, 6623, 5695}},
	{1820, 14, 7724, {1, 1, 1, 13, 15, 9, 117, 197, 123, 953, 1191, 3657, 5757, 15957}},
	{1821, 14, 7749, {1, 1, 3, 7, 29, 13, 5, 175, 395, 127, 679, 255, 6055, 7639}},
	{1822, 14, 7753, {1, 3, 7, 15, 15, 51, 77, 147, 319, 147, 1775, 3983, 3175, 5723}},
	{1823, 14, 7754, {1, 3, 3, 3, 7, 11, 119, 41, 43, 153, 975, 679, 3081, 10359}},
	{1824, 14, 7761, {1, 1, 5, 13, 3, 7, 65, 67, 63, 399, 1561, 2789, 2083, 12289}},
	{1825, 14, 7771, {1, 1, 7, 3, 19, 53, 103, 67, 35, 865, 161, 93, 2533, 3851}},
	{1826, 14, 7777, {1, 1, 1, 11, 31, 9, 29, 189, 199, 817, 1571, 395, 345, 3777}},
	{1827, 14, 7784, {1, 3, 5, 11, 31, 3, 9, 67, 277, 735, 181, 2777, 3009, 7233}},
	{1828, 14, 7804, {1, 1, 3, 3, 17, 7, 17, 3, 375, 933, 237, 3919, 5409, 3355}},
	{1829, 14, 7807, {1, 3, 3, 5, 9, 27, 19, 77, 221, 3, 1965, 309, 3001, 15977}},
	{1830, 14, 7808, {1, 1, 5, 1, 3, 33, 35, 133, 37, 709, 627, 1705, 2525, 4307}},
	{1831, 14, 7818, {1, 1, 7, 3, 25, 21, 105, 55, 375, 681, 881, 1299, 5879, 459}},
	{1832, 14, 7835, {1, 3, 7, 1, 13, 7, 113, 103, 313, 515, 1041, 3683, 4619, 5093}},
	{1833, 14, 7842, {1, 1, 3, 7, 19, 43, 83, 37, 39, 133, 1759, 1171, 1521, 13717}},
	{1834, 14, 7865, {1, 1, 7, 13, 7, 35, 15, 155, 293, 1001, 157, 3883, 405, 1797}},
	{1835, 14, 7868, {1, 1, 3, 3, 13, 19, 125, 49, 333, 387, 339, 1815, 4503, 7359}},
	{1836, 14, 7880, {1, 1, 3, 13, 19, 19, 105, 225, 151, 27, 1251, 885, 4815, 7863}},
	{1837, 14, 7883, {1, 1, 1, 5, 7, 59, 17, 145, 77, 117, 1355, 1429, 2301, 16177}},
	{1838, 14, 7891, {1, 3, 3, 13, 5, 31, 119, 167, 459, 727, 1799, 2537, 695, 13637}},
	{1839, 14, 7897, {1, 3, 3, 3, 27, 51, 107, 85, 267, 57, 1279, 823, 6247, 3603}},
	{1840, 14, 7907, {1, 1, 7, 15, 29, 17, 67, 197, 215, 465, 109, 3461, 5269, 15287}},
	{1841, 14, 7910, {1, 1, 3, 5, 11, 15, 123, 53, 293, 797, 1105, 1777, 6509, 217}},
	{1842, 14, 7924, {1, 3, 3, 13, 3, 5, 109, 53, 203, 693, 871, 135, 369, 11149}},
	{1843, 14, 7933, {1, 3, 5, 15, 17, 43, 81, 235, 119, 817, 1777, 261, 8049, 4251}},
	{1844, 14, 7934, {1, 1, 3, 7, 7, 13, 87, 99, 481, 931, 1507, 651, 5267, 8281}},
	{1845, 14, 7942, {1, 3, 1, 13, 27, 43, 77, 225, 341, 163, 933, 429, 4943, 7781}},
	{1846, 14, 7948, {1, 1, 7, 1, 1, 49, 85, 211, 449, 479, 1395, 787, 5653, 14891}},
	{1847, 14, 7959, {1, 1, 5, 9, 25, 13, 49, 85, 125, 85, 1281, 3365, 4305, 11791}},
	{1848, 14, 7984, {1, 3, 1, 13, 3, 31, 117, 39, 43, 151, 663, 669, 1571, 5207}},
	{1849, 14, 7994, {1, 3, 7, 15, 17, 7, 79, 163, 37, 841, 1799, 1787, 4501, 3785}},
	{1850, 14, 7999, {1, 1, 3, 9, 1, 23, 67, 191, 449, 931, 1521, 2705, 887, 7037}},
	{1851, 14, 8014, {1, 1, 1, 1, 5, 13, 55, 161, 419, 577, 1703, 2589, 2651, 2873}},
	{1852, 14, 8021, {1, 3, 3, 3, 5, 19, 37, 169, 69, 1003, 1755, 3101, 1469, 8583}},
	{1853, 14, 8041, {1, 1, 1, 1, 11, 33, 105, 79, 283, 91, 299, 835, 3193, 5593}},
	{1854, 14, 8049, {1, 3, 3, 13, 25, 21, 81, 213, 465, 475, 331, 457, 61, 9511}},
	{1855, 14, 8050, {1, 1, 3, 11, 1, 11, 77, 95, 455, 949, 1999, 1833, 1275, 5631}},
	{1856, 14, 8068, {1, 1, 1, 1, 15, 25, 51, 137, 275, 451, 1179, 3595, 5177, 7105}},
	{1857, 14, 8080, {1, 3, 3, 3, 3, 59, 79, 143, 393, 583, 349, 3039, 7079, 14245}},
	{1858, 14, 8095, {1, 1, 7, 9, 21, 11, 123, 105, 53, 297, 803, 4025, 5421, 14527}},
	{1859, 14, 8102, {1, 3, 7, 11, 21, 15, 103, 109, 311, 321, 1217, 2777, 5457, 1823}},
	{1860, 14, 8106, {1, 3, 5, 11, 19, 31, 79, 89, 295, 413, 817, 499, 3699, 14411}},
	{1861, 14, 8120, {1, 1, 1, 5, 11, 3, 81, 13, 315, 841, 1543, 411, 6883, 6347}},
	{1862, 14, 8133, {1, 3, 3, 11, 23, 43, 23, 131, 17, 517, 995, 2687, 7443, 15085}},
	{1863, 14, 8134, {1, 1, 1, 1, 11, 57, 73, 9, 123, 905, 1763, 1789, 3701, 7131}},
	{1864, 14, 8143, {1, 1, 3, 5, 9, 53, 99, 229, 43, 207, 625, 1583, 6727, 15249}},
	{1865, 14, 8162, {1, 1, 7, 7, 17, 39, 91, 1, 297, 711, 225, 513, 7391, 291}},
	{1866, 14, 8168, {1, 1, 7, 11, 7, 55, 111, 129, 423, 521, 1807, 3015, 1449, 12321}},
	{1867, 14, 8179, {1, 3, 7, 3, 13, 9, 125, 187, 11, 485, 647, 275, 3495, 11989}},
	{1868, 15, 1, {1, 1, 3, 11, 11, 25, 49, 33, 361, 105, 271, 3841, 4837, 2437, 30181}},
	{1869, 15, 8, {1, 3, 5, 1, 27, 15, 119, 35, 159, 273, 1489, 3157, 5433, 3337, 26859}},
	{1870, 15, 11, {1, 3, 5, 13, 23, 31, 97, 145, 41, 605, 1455, 59, 5389, 5527, 14447}},
	{1871, 15, 22, {1, 1, 7, 9, 7, 41, 61, 193, 353, 879, 1805, 581, 5447, 11177, 7331}},
	{1872, 15, 26, {1, 1, 7, 11, 29, 19, 55, 207, 361, 759, 63, 2255, 2119, 14671, 21783}},
	{1873, 15, 47, {1, 3, 1, 13, 17, 7, 73, 179, 103, 23, 917, 1205, 4925, 1691, 5419}},
	{1874, 15, 59, {1, 3, 5, 3, 15, 3, 9, 109, 227, 861, 867, 3529, 1535, 489, 22873}},
	{1875, 15, 64, {1, 3, 3, 9, 15, 15, 95, 193, 385, 997, 1525, 1865, 1425, 4079, 14771}},
	{1876, 15, 67, {1, 1, 3, 5, 5, 29, 49, 171, 171, 623, 1167, 3743, 1809, 12009, 7043}},
	{1877, 15, 73, {1, 3, 7, 5, 23, 11, 87, 183, 299, 555, 1857, 489, 3505, 9161, 28763}},
	{1878, 15, 82, {1, 3, 5, 9, 19, 21, 85, 127, 337, 439, 1183, 1891, 1877, 4373, 10451}},
	{1879, 15, 97, {1, 3, 7, 13, 27, 17, 29, 83, 463, 385, 1167, 3453, 4523, 4759, 9321}},
	{1880, 15, 103, {1, 1, 3, 7, 21, 59, 65, 83, 177, 763, 317, 2913, 7527, 5967, 17167}},
	{1881, 15, 110, {1, 1, 7, 15, 13, 27, 49, 35, 253, 101, 1699, 355, 2181, 10859, 24221}},
	{1882, 15, 115, {1, 1, 5, 1, 17, 17, 81, 91, 349, 655, 1373, 2225, 945, 899, 31801}},
	{1883, 15, 122, {1, 3, 7, 11, 5, 1, 81, 53, 215, 587, 167, 4045, 5671, 5597, 13529}},
	{1884, 15, 128, {1, 3, 5, 15, 1, 9, 59, 235, 315, 195, 909, 2237, 505, 10415, 28145}},
	{1885, 15, 138, {1, 1, 1, 3, 9, 31, 41, 43, 275, 921, 25, 671, 5737, 11241, 4193}},
	{1886, 15, 146, {1, 3, 3, 13, 29, 13, 95, 213, 317, 995, 1489, 3779, 3043, 8569, 28823}},
	{1887, 15, 171, {1, 1, 7, 5, 9, 49, 125, 241, 87, 153, 1673, 3849, 7253, 1715, 11627}},
	{1888, 15, 174, {1, 1, 3, 9, 27, 27, 19, 223, 63, 463, 1095, 1395, 6643, 11589, 2145}},
	{1889, 15, 176, {1, 1, 3, 15, 21, 17, 45, 23, 357, 11, 1307, 1791, 2481, 2123, 24341}},
	{1890, 15, 182, {1, 3, 5, 15, 31, 53, 117, 51, 433, 193, 1239, 3329, 2403, 12745, 32219}},
	{1891, 15, 194, {1, 1, 5, 9, 7, 27, 9, 115, 417, 579, 83, 173, 4717, 15665, 27463}},
	{1892, 15, 208, {1, 3, 5, 7, 9, 9, 31, 35, 249, 567, 331, 905, 5101, 14817, 14255}},
	{1893, 15, 211, {1, 3, 7, 3, 1, 61, 29, 129, 119, 421, 1597, 2987, 3041, 7629, 23451}},
	{1894, 15, 220, {1, 1, 7, 9, 13, 1, 99, 105, 107, 509, 989, 2259, 1009, 6827, 8903}},
	{1895, 15, 229, {1, 3, 5, 15, 11, 29, 85, 29, 265, 105, 2035, 3349, 3543, 13903, 10213}},
	{1896, 15, 230, {1, 3, 1, 1, 25, 19, 53, 139, 467, 485, 491, 3067, 7353, 13861, 25819}},
	{1897, 15, 239, {1, 1, 5, 3, 3, 43, 41, 185, 45, 463, 351, 2823, 2519, 6705, 11395}},
	{1898, 15, 254, {1, 3, 7, 13, 11, 15, 87, 221, 427, 673, 1631, 599, 3259, 10691, 31283}},
	{1899, 15, 265, {1, 3, 5, 11, 9, 9, 15, 49, 275, 335, 1613, 3587, 5309, 14849, 26475}},
	{1900, 15, 285, {1, 3, 7, 9, 29, 13, 79, 225, 381, 781, 1411, 2761, 7157, 14983, 19717}},
	{1901, 15, 290, {1, 1, 7, 11, 29, 25, 117, 183, 101, 651, 653, 3157, 445, 14389, 23293}},
	{1902, 15, 319, {1, 1, 1, 3, 5, 33, 73, 155, 473, 387, 591, 2045, 5965, 16299, 31499}},
	{1903, 15, 324, {1, 3, 1, 7, 11, 33, 29, 21, 491, 937, 729, 4075, 975, 2461, 18991}},
	{1904, 15, 327, {1, 3, 7, 15, 29, 39, 105, 111, 173, 943, 69, 295, 8175, 13037, 26131}},
	{1905, 15, 333, {1, 1, 5, 15, 7, 5, 97, 147, 105, 887, 443, 2595, 5889, 10753, 1619}},
	{1906, 15, 357, {1, 3, 3, 15, 11, 45, 87, 207, 353, 909, 1847, 323, 2283, 12885, 16415}},
	{1907, 15, 364, {1, 1, 5, 3, 19, 33, 43, 79, 115, 653, 359, 2873, 4609, 12439, 6339}},
	{1908, 15, 395, {1, 3, 7, 9, 17, 61, 49, 227, 291, 69, 1753, 3899, 483, 3187, 29041}},
	{1909, 15, 397, {1, 3, 5, 3, 25, 35, 61, 211, 393, 199, 691, 1779, 6295, 13371, 15817}},
	{1910, 15, 405, {1, 3, 7, 5, 7, 23, 37, 91, 245, 915, 579, 867, 6193, 1063, 17363}},
	{1911, 15, 409, {1, 3, 7, 7, 23, 51, 41, 63, 375, 3, 159, 1889, 4419, 1687, 17977}},
	{1912, 15, 419, {1, 1, 1, 7, 13, 11, 53, 43, 317, 325, 1749, 2423, 4123, 8595, 20773}},
	{1913, 15, 422, {1, 1, 7, 7, 9, 9, 61, 113, 437, 213, 1407, 645, 4345, 807, 30411}},
	{1914, 15, 431, {1, 3, 3, 11, 17, 39, 17, 113, 391, 385, 581, 2023, 7449, 10153, 22033}},
	{1915, 15, 433, {1, 1, 3, 5, 29, 31, 101, 215, 379, 377, 1113, 2855, 7147, 14377, 25515}},
	{1916, 15, 436, {1, 3, 5, 5, 13, 3, 121, 125, 227, 969, 11, 1115, 5657, 9209, 6117}},
	{1917, 15, 440, {1, 3, 7, 15, 29, 17, 33, 123, 317, 301, 749, 1365, 5619, 605, 1613}},
	{1918, 15, 453, {1, 3, 1, 15, 7, 53, 125, 249, 219, 655, 105, 2825, 1649, 12783, 19777}},
	{1919, 15, 460, {1, 1, 7, 1, 25, 53, 19, 53, 157, 373, 1855, 495, 5065, 9465, 2313}},
	{1920, 15, 471, {1, 3, 5, 13, 3, 57, 57, 161, 431, 415, 1859, 1033, 6349, 1577, 31579}},
	{1921, 15, 478, {1, 1, 7, 5, 23, 63, 29, 221, 13, 965, 1997, 2265, 1583, 10491, 9551}},
	{1922, 15, 482, {1, 1, 3, 13, 31, 25, 23, 61, 285, 5, 2005, 879, 795, 13299, 19685}},
	{1923, 15, 488, {1, 1, 7, 1, 21, 45, 121, 89, 263, 543, 1333, 2711, 219, 10823, 26139}},
	{1924, 15, 524, {1, 1, 3, 3, 27, 13, 19, 117, 161, 457, 1541, 295, 4953, 12125, 14503}},
	{1925, 15, 529, {1, 3, 5, 3, 7, 63, 13, 247, 439, 681, 977, 2537, 6923, 10323, 7349}},
	{1926, 15, 535, {1, 3, 5, 9, 3, 51, 81, 251, 349, 983, 581, 2515, 2281, 2849, 31915}},
	{1927, 15, 536, {1, 3, 5, 3, 11, 63, 47, 137, 303, 627, 91, 2269, 7097, 2145, 31059}},
	{1928, 15, 539, {1, 1, 3, 15, 13, 17, 53, 27, 133, 13, 117, 1837, 4103, 5843, 29153}},
	{1929, 15, 563, {1, 1, 5, 13, 21, 33, 37, 253, 465, 209, 309, 49, 3209, 15677, 14569}},
	{1930, 15, 566, {1, 1, 7, 15, 13, 21, 33, 203, 499, 141, 1155, 3893, 1663, 2115, 27459}},
	{1931, 15, 572, {1, 3, 5, 11, 21, 9, 39, 157, 257, 273, 1257, 1831, 515, 7969, 20133}},
	{1932, 15, 577, {1, 1, 3, 13, 19, 29, 15, 189, 103, 219, 1395, 517, 7425, 6585, 15865}},
	{1933, 15, 587, {1, 1, 5, 11, 21, 31, 49, 151, 39, 537, 1783, 3449, 6915, 223, 11587}},
	{1934, 15, 592, {1, 3, 3, 11, 7, 63, 69, 31, 27, 911, 1903, 2821, 7977, 12949, 32257}},
	{1935, 15, 602, {1, 1, 7, 9, 25, 45, 23, 233, 511, 595, 1383, 1721, 6789, 12055, 21179}},
	{1936, 15, 623, {1, 1, 7, 13, 1, 27, 123, 49, 439, 683, 501, 641, 1947, 6111, 25423}},
	{1937, 15, 635, {1, 3, 3, 5, 1, 23, 57, 241, 243, 593, 2039, 1617, 2209, 5171, 9675}},
	{1938, 15, 638, {1, 1, 1, 7, 5, 19, 83, 55, 481, 125, 177, 1021, 1139, 11403, 23099}},
	{1939, 15, 654, {1, 1, 3, 5, 29, 39, 33, 217, 461, 907, 733, 3795, 4811, 12939, 27715}},
	{1940, 15, 656, {1, 3, 7, 3, 7, 11, 39, 165, 495, 147, 999, 1827, 817, 603, 9293}},
	{1941, 15, 659, {1, 3, 7, 15, 25, 53, 35, 15, 431, 733, 1213, 2907, 8087, 3939, 27363}},
	{1942, 15, 665, {1, 3, 7, 13, 13, 9, 33, 27, 485, 183, 455, 3341, 2555, 4985, 8793}},
	{1943, 15, 675, {1, 1, 1, 15, 25, 47, 75, 21, 205, 15, 1639, 3067, 1295, 11693, 16903}},
	{1944, 15, 677, {1, 1, 1, 15, 3, 31, 93, 57, 43, 185, 251, 1899, 7885, 10829, 3609}},
	{1945, 15, 687, {1, 1, 3, 1, 29, 9, 69, 223, 221, 537, 365, 3411, 5771, 15279, 5309}},
	{1946, 15, 696, {1, 1, 7, 5, 1, 5, 125, 243, 213, 1003, 1571, 3355, 3981, 8781, 25993}},
	{1947, 15, 701, {1, 1, 1, 13, 7, 19, 53, 243, 301, 75, 1183, 2723, 6687, 13, 16581}},
	{1948, 15, 704, {1, 3, 1, 13, 17, 51, 91, 239, 437, 191, 1065, 2495, 5755, 3405, 8299}},
	{1949, 15, 710, {1, 1, 5, 5, 11, 59, 21, 169, 299, 123, 1845, 2199, 2157, 14461, 10327}},
	{1950, 15, 721, {1, 3, 7, 7, 19, 47, 51, 179, 41, 19, 1347, 2325, 8063, 5993, 15653}},
	{1951, 15, 728, {1, 1, 1, 9, 25, 27, 7, 133, 223, 533, 719, 353, 7093, 8285, 10375}},
	{1952, 15, 738, {1, 3, 5, 15, 31, 5, 67, 39, 441, 495, 977, 3699, 1435, 11385, 14567}},
	{1953, 15, 740, {1, 1, 3, 15, 15, 39, 25, 33, 91, 523, 249, 4035, 769, 5181, 9691}},
	{1954, 15, 749, {1, 1, 3, 3, 3, 57, 83, 187, 423, 165, 161, 3453, 2241, 981, 8429}},
	{1955, 15, 758, {1, 1, 7, 15, 1, 17, 57, 189, 283, 11, 823, 3505, 7025, 11879, 15441}},
	{1956, 15, 761, {1, 1, 3, 11, 1, 41, 7, 255, 385, 339, 607, 1405, 1473, 13697, 9491}},
	{1957, 15, 772, {1, 1, 7, 15, 5, 9, 91, 99, 211, 233, 51, 2663, 1165, 9283, 18495}},
	{1958, 15, 776, {1, 1, 3, 7, 21, 37, 13, 91, 39, 27, 1021, 2813, 5937, 6645, 3403}},
	{1959, 15, 782, {1, 3, 1, 1, 29, 29, 5, 69, 399, 665, 1407, 3921, 2653, 11753, 18925}},
	{1960, 15, 789, {1, 3, 7, 15, 13, 41, 39, 1, 437, 549, 161, 2315, 5631, 8335, 22661}},
	{1961, 15, 810, {1, 1, 3, 1, 7, 17, 115, 61, 69, 955, 475, 3763, 8035, 927, 17893}},
	{1962, 15, 812, {1, 3, 1, 13, 21, 59, 81, 145, 463, 145, 1941, 2777, 7453, 14229, 11281}},
	{1963, 15, 818, {1, 1, 1, 15, 15, 11, 27, 165, 461, 395, 1645, 3611, 7463, 12379, 26787}},
	{1964, 15, 830, {1, 1, 7, 9, 29, 19, 27, 123, 21, 149, 1643, 4001, 7207, 6769, 4647}},
	{1965, 15, 832, {1, 1, 1, 11, 13, 9, 103, 139, 185, 587, 591, 1113, 2223, 11667, 32671}},
	{1966, 15, 852, {1, 3, 1, 1, 31, 13, 19, 93, 229, 125, 1471, 2369, 3055, 10277, 28563}},
	{1967, 15, 855, {1, 3, 7, 5, 7, 53, 99, 175, 161, 851, 617, 4027, 2357, 11199, 1931}},
	{1968, 15, 859, {1, 3, 5, 11, 3, 31, 111, 179, 237, 845, 539, 1057, 259, 3417, 26637}},
	{1969, 15, 865, {1, 1, 5, 3, 21, 49, 125, 119, 463, 403, 737, 1811, 3941, 13015, 29081}},
	{1970, 15, 877, {1, 3, 5, 13, 5, 29, 69, 251, 313, 357, 663, 1097, 3307, 12845, 28495}},
	{1971, 15, 895, {1, 3, 3, 5, 29, 17, 89, 15, 411, 409, 2013, 757, 4085, 12521, 11131}},
	{1972, 15, 901, {1, 1, 1, 15, 7, 51, 3, 193, 493, 133, 381, 2027, 227, 6635, 12931}},
	{1973, 15, 902, {1, 1, 1, 15, 7, 23, 99, 203, 323, 1007, 1465, 2887, 2215, 1787, 22069}},
	{1974, 15, 906, {1, 1, 5, 9, 29, 59, 77, 151, 509, 313, 415, 3977, 5431, 8019, 8571}},
	{1975, 15, 916, {1, 3, 1, 15, 19, 13, 57, 217, 87, 119, 25, 1149, 5667, 3765, 6959}},
	{1976, 15, 920, {1, 3, 7, 13, 19, 31, 119, 3, 457, 117, 905, 361, 1483, 12405, 27005}},
	{1977, 15, 949, {1, 3, 5, 11, 15, 35, 61, 77, 119, 51, 1753, 2765, 1091, 10573, 23595}},
	{1978, 15, 962, {1, 3, 3, 7, 1, 35, 17, 93, 197, 511, 1253, 3031, 2739, 15127, 15147}},
	{1979, 15, 964, {1, 3, 3, 1, 11, 55, 55, 107, 161, 75, 129, 2195, 2023, 4877, 25797}},
	{1980, 15, 967, {1, 3, 5, 7, 23, 19, 113, 167, 167, 271, 1303, 125, 5057, 1323, 5165}},
	{1981, 15, 981, {1, 1, 5, 3, 21, 31, 11, 119, 215, 483, 1535, 407, 6485, 15401, 30297}},
	{1982, 15, 982, {1, 3, 5, 9, 21, 5, 77, 95, 443, 247, 913, 605, 365, 7465, 19707}},
	{1983, 15, 985, {1, 3, 1, 7, 17, 59, 9, 35, 391, 767, 1493, 475, 4725, 7529, 31579}},
	{1984, 15, 991, {1, 3, 3, 7, 31, 21, 61, 31, 421, 179, 273, 771, 5745, 10575, 32765}},
	{1985, 15, 1007, {1, 3, 5, 15, 27, 13, 125, 55, 423, 1021, 497, 3521, 6903, 15111, 8285}},
	{1986, 15, 1016, {1, 1, 5, 9, 13, 31, 105, 93, 421, 709, 643, 1079, 1533, 9149, 10799}},
	{1987, 15, 1024, {1, 3, 1, 11, 19, 29, 53, 199, 319, 247, 655, 3039, 6411, 12267, 14245}},
	{1988, 15, 1051, {1, 3, 1, 11, 9, 57, 5, 91, 469, 149, 259, 329, 5433, 6941, 15093}},
	{1989, 15, 1060, {1, 3, 1, 5, 5, 51, 59, 25, 455, 367, 1623, 441, 3155, 11695, 20767}},
	{1990, 15, 1070, {1, 3, 7, 7, 11, 49, 113, 95, 91, 389, 605, 1973, 2051, 2315, 22229}},
	{1991, 15, 1072, {1, 3, 5, 3, 19, 11, 99, 135, 433, 781, 1473, 885, 1105, 3573, 3739}},
	{1992, 15, 1084, {1, 3, 1, 11, 3, 25, 9, 227, 433, 723, 317, 139, 6627, 8067, 28439}},
	{1993, 15, 1089, {1, 1, 1, 9, 9, 9, 5, 63, 241, 215, 1991, 2949, 3943, 775, 31511}},
	{1994, 15, 1095, {1, 1, 3, 7, 17, 49, 35, 167, 131, 107, 1295, 2465, 4577, 11147, 29833}},
	{1995, 15, 1114, {1, 1, 5, 1, 5, 25, 119, 129, 391, 743, 1069, 2957, 349, 6891, 13635}},
	{1996, 15, 1123, {1, 3, 1, 7, 9, 31, 63, 253, 215, 51, 1347, 2361, 3125, 13049, 28461}},
	{1997, 15, 1132, {1, 1, 7, 9, 3, 31, 21, 163, 255, 47, 259, 535, 5461, 3349, 30649}},
	{1998, 15, 1154, {1, 3, 3, 13, 17, 33, 87, 47, 243, 709, 929, 3943, 3107, 3421, 13721}},
	{1999, 15, 1156, {1, 3, 5, 11, 25, 61, 61, 173, 397, 735, 2005, 3355, 8121, 11593, 27697}},
	{2000, 15, 1163, {1, 3, 5, 15, 17, 43, 63, 231, 275, 311, 1277, 2669, 7307, 2099, 9755}},
	{2001, 15, 1171, {1, 3, 5, 3, 25, 43, 71, 191, 9, 121, 1873, 3747, 7491, 14055, 24293}},
	{2002, 15, 1202, {1, 3, 5, 13, 17, 35, 113, 113, 385, 941, 39, 2705, 1225, 5167, 1373}},
	{2003, 15, 1228, {1, 3, 5, 5, 7, 35, 19, 105, 487, 71, 139, 627, 4187, 3183, 713}},
	{2004, 15, 1239, {1, 1, 5, 13, 29, 29, 103, 5, 157, 869, 1675, 423, 6689, 10697, 5303}},
	{2005, 15, 1255, {1, 1, 5, 1, 29, 31, 61, 111, 473, 963, 685, 1483, 2383, 8109, 8495}},
	{2006, 15, 1256, {1, 1, 5, 3, 19, 13, 95, 113, 217, 59, 1353, 1647, 3617, 3271, 2321}},
	{2007, 15, 1262, {1, 3, 5, 7, 25, 35, 59, 131, 309, 445, 415, 93, 1453, 8789, 30201}},
	{2008, 15, 1270, {1, 1, 5, 1, 5, 43, 71, 241, 123, 189, 831, 3469, 8093, 6187, 32721}},
	{2009, 15, 1279, {1, 3, 7, 5, 25, 31, 123, 171, 319, 379, 889, 2365, 4881, 12225, 16609}},
	{2010, 15, 1308, {1, 3, 1, 11, 27, 43, 121, 63, 291, 591, 811, 1995, 4777, 2083, 31385}},
	{2011, 15, 1322, {1, 1, 5, 11, 27, 53, 85, 187, 461, 823, 703, 399, 6925, 11517, 28697}},
	{2012, 15, 1329, {1, 1, 3, 5, 13, 11, 33, 121, 93, 717, 1275, 3877, 4247, 5845, 26909}},
	{2013, 15, 1330, {1, 3, 1, 9, 7, 5, 47, 199, 367, 561, 185, 2855, 5997, 2699, 7581}},
	{2014, 15, 1336, {1, 1, 5, 9, 23, 11, 71, 201, 61, 729, 1011, 3529, 663, 1413, 25675}},
	{2015, 15, 1341, {1, 3, 7, 13, 27, 21, 11, 127, 281, 487, 1217, 3129, 5541, 3129, 17783}},
	{2016, 15, 1347, {1, 1, 5, 9, 1, 29, 85, 193, 213, 743, 1473, 611, 391, 9405, 21137}},
	{2017, 15, 1349, {1, 3, 3, 3, 31, 63, 37, 147, 39, 351, 79, 3069, 2441, 8901, 8777}},
	{2018, 15, 1359, {1, 1, 7, 7, 25, 49, 55, 47, 441, 343, 1267, 1123, 5917, 14395, 10579}},
	{2019, 15, 1367, {1, 1, 7, 1, 13, 55, 55, 123, 103, 773, 125, 2145, 4743, 13347, 2589}},
	{2020, 15, 1368, {1, 3, 7, 3, 9, 33, 25, 183, 469, 213, 291, 75, 6725, 6847, 26745}},
	{2021, 15, 1390, {1, 3, 3, 7, 15, 43, 7, 79, 171, 21, 1767, 2537, 4285, 12007, 24039}},
	{2022, 15, 1413, {1, 3, 7, 13, 9, 61, 125, 23, 227, 879, 215, 1635, 2835, 883, 15939}},
	{2023, 15, 1414, {1, 1, 5, 13, 25, 45, 63, 43, 183, 829, 149, 989, 987, 3819, 12181}},
	{2024, 15, 1437, {1, 1, 3, 7, 19, 27, 35, 83, 135, 459, 785, 131, 2655, 3329, 3009}},
	{2025, 15, 1441, {1, 1, 7, 5, 11, 41, 9, 219, 475, 985, 1329, 3787, 1975, 4679, 8627}},
	{2026, 15, 1462, {1, 1, 7, 3, 1, 17, 91, 155, 3, 763, 1879, 233, 215, 2955, 25993}},
	{2027, 15, 1465, {1, 1, 1, 11, 25, 11, 23, 227, 453, 775, 1935, 3833, 4583, 269, 705}},
	{2028, 15, 1480, {1, 3, 3, 11, 7, 25, 105, 21, 449, 555, 1275, 3475, 5503, 15617, 813}},
	{2029, 15, 1486, {1, 3, 7, 13, 31, 37, 25, 255, 233, 663, 1155, 1563, 4775, 7449, 29949}},
	{2030, 15, 1504, {1, 1, 3, 1, 23, 51, 51, 137, 63, 809, 349, 2789, 6953, 10605, 18959}},
	{2031, 15, 1509, {1, 3, 3, 13, 21, 45, 15, 161, 393, 229, 437, 2967, 4019, 3893, 21305}},
	{2032, 15, 1514, {1, 1, 3, 7, 5, 11, 15, 211, 287, 131, 1847, 2569, 7881, 15669, 31037}},
	{2033, 15, 1522, {1, 3, 3, 15, 27, 19, 85, 251, 221, 639, 665, 3729, 5771, 7873, 28005}},
	{2034, 15, 1528, {1, 3, 7, 15, 15, 47, 93, 215, 343, 85, 1401, 1375, 2949, 13661, 25453}},
	{2035, 15, 1552, {1, 1, 1, 9, 7, 51, 53, 217, 471, 389, 551, 1141, 1767, 2237, 17797}},
	{2036, 15, 1555, {1, 1, 7, 9, 3, 29, 65, 29, 223, 591, 1719, 1049, 7643, 3853, 29867}},
	{2037, 15, 1571, {1, 1, 1, 11, 13, 41, 85, 29, 451, 387, 1783, 3733, 8033, 4711, 31643}},
	{2038, 15, 1578, {1, 3, 1, 11, 11, 57, 75, 153, 7, 373, 2011, 271, 469, 3267, 18969}},
	{2039, 15, 1585, {1, 1, 5, 3, 19, 43, 7, 243, 385, 293, 923, 843, 4895, 469, 8421}},
	{2040, 15, 1588, {1, 3, 1, 15, 29, 47, 17, 125, 471, 927, 349, 3859, 3059, 11483, 14791}},
	{2041, 15, 1603, {1, 3, 1, 11, 17, 17, 111, 109, 9, 213, 1313, 3903, 4411, 4329, 28277}},
	{2042, 15, 1609, {1, 3, 3, 15, 1, 55, 47, 69, 143, 789, 1149, 3833, 5053, 6949, 10569}},
	{2043, 15, 1617, {1, 3, 5, 7, 11, 15, 79, 83, 123, 937, 1115, 2775, 3041, 11869, 21167}},
	{2044, 15, 1620, {1, 3, 7, 13, 9, 47, 45, 221, 139, 923, 1661, 1379, 2485, 7233, 6035}},
	{2045, 15, 1629, {1, 1, 3, 3, 11, 55, 77, 3, 87, 693, 1991, 1145, 2783, 16207, 24569}},
	{2046, 15, 1636, {1, 1, 5, 11, 3, 35, 91, 9, 391, 927, 101, 1839, 3755, 10345, 16907}},
	{2047, 15, 1648, {1, 3, 5, 3, 5, 49, 79, 91, 205, 443, 1369, 197, 2537, 11219, 17765}},
	{2048, 15, 1667, {1, 1, 3, 15, 9, 7, 25, 25, 357, 247, 477, 421, 7679, 5987, 30079}},
	{2049, 15, 1669, {1, 1, 5, 3, 29, 5, 89, 117, 481, 491, 371, 389, 7101, 2253, 23617}},
	{2050, 15, 1682, {1, 1, 5, 13, 29, 59, 17, 181, 511, 291, 1991, 3499, 8177, 5559, 30045}},
	{2051, 15, 1697, {1, 3, 3, 11, 23, 31, 117, 217, 241, 115, 749, 945, 1897, 12253, 8473}},
	{2052, 15, 1704, {1, 1, 7, 15, 25, 47, 31, 1, 165, 311, 635, 3629, 1593, 8305, 30033}},
	{2053, 15, 1709, {1, 3, 5, 9, 3, 17, 101, 237, 379, 503, 49, 929, 1687, 3865, 26723}},
	{2054, 15, 1727, {1, 3, 5, 5, 15, 41, 1, 239, 53, 215, 1733, 827, 579, 4089, 6579}},
	{2055, 15, 1730, {1, 3, 1, 15, 15, 21, 35, 21, 403, 257, 1475, 2403, 4705, 11553, 203}},
	{2056, 15, 1732, {1, 3, 5, 11, 9, 53, 113, 9, 447, 511, 543, 3141, 7389, 11249, 431}},
	{2057, 15, 1741, {1, 3, 5, 9, 9, 11, 55, 93, 325, 411, 305, 2573, 6871, 12339, 6435}},
	{2058, 15, 1744, {1, 3, 3, 7, 31, 27, 21, 113, 99, 853, 365, 589, 3731, 10875, 12767}},
	{2059, 15, 1759, {1, 3, 1, 7, 15, 27, 31, 17, 275, 93, 1161, 2619, 1329, 7307, 587}},
	{2060, 15, 1765, {1, 3, 5, 9, 17, 47, 49, 237, 27, 193, 1237, 591, 5151, 5521, 31583}},
	{2061, 15, 1766, {1, 3, 5, 3, 13, 1, 27, 87, 43, 977, 305, 3293, 2475, 14571, 18321}},
	{2062, 15, 1778, {1, 1, 5, 7, 15, 13, 101, 1, 291, 807, 1711, 2277, 5573, 11051, 13133}},
	{2063, 15, 1780, {1, 3, 3, 1, 9, 3, 65, 81, 415, 733, 1527, 2747, 6069, 159, 7095}},
	{2064, 15, 1783, {1, 3, 3, 15, 27, 1, 71, 49, 231, 851, 2039, 613, 1899, 2537, 14511}},
	{2065, 15, 1797, {1, 1, 1, 11, 3, 41, 55, 23, 247, 1011, 581, 2363, 2745, 1337, 20931}},
	{2066, 15, 1807, {1, 1, 3, 11, 17, 61, 67, 255, 143, 357, 945, 3407, 5817, 4155, 23851}},
	{2067, 15, 1821, {1, 3, 5, 3, 23, 1, 75, 247, 265, 413, 1899, 2565, 6629, 15655, 16117}},
	{2068, 15, 1832, {1, 1, 1, 9, 11, 49, 11, 189, 223, 177, 1457, 1931, 163, 15905, 17297}},
	{2069, 15, 1835, {1, 3, 7, 13, 17, 1, 111, 189, 343, 961, 427, 2507, 2393, 8653, 6353}},
	{2070, 15, 1849, {1, 3, 7, 13, 23, 61, 59, 51, 313, 963, 791, 3681, 5637, 3965, 9263}},
	{2071, 15, 1850, {1, 3, 7, 7, 21, 53, 127, 141, 499, 859, 337, 2835, 3195, 4351, 32369}},
	{2072, 15, 1863, {1, 1, 7, 5, 1, 5, 53, 63, 497, 535, 35, 305, 4395, 9757, 13193}},
	{2073, 15, 1867, {1, 1, 5, 13, 13, 31, 59, 229, 211, 745, 1453, 3677, 3005, 7703, 23907}},
	{2074, 15, 1869, {1, 3, 5, 5, 7, 63, 17, 197, 493, 861, 499, 3015, 6349, 1815, 7437}},
	{2075, 15, 1872, {1, 1, 1, 13, 13, 37, 29, 189, 253, 1017, 321, 3145, 407, 7547, 17099}},
	{2076, 15, 1887, {1, 3, 3, 3, 23, 53, 69, 77, 175, 17, 1831, 841, 3851, 1295, 32107}},
	{2077, 15, 1888, {1, 3, 7, 13, 13, 39, 107, 237, 389, 729, 635, 3717, 3041, 3169, 14987}},
	{2078, 15, 1897, {1, 1, 3, 1, 25, 7, 69, 35, 495, 49, 659, 2783, 6051, 13875, 23927}},
	{2079, 15, 1906, {1, 3, 7, 5, 5, 25, 49, 7, 193, 493, 93, 657, 1515, 13975, 14155}},
	{2080, 15, 1917, {1, 3, 1, 1, 11, 15, 113, 45, 21, 595, 731, 3397, 4117, 9711, 16625}},
	{2081, 15, 1927, {1, 3, 3, 9, 19, 19, 59, 7, 105, 579, 599, 2859, 97, 14717, 15361}},
	{2082, 15, 1939, {1, 1, 1, 5, 27, 49, 113, 5, 367, 563, 1397, 2805, 3021, 3111, 20671}},
	{2083, 15, 1941, {1, 3, 3, 15, 27, 51, 99, 167, 109, 365, 1959, 1523, 6959, 14405, 18191}},
	{2084, 15, 1948, {1, 3, 1, 5, 21, 51, 125, 67, 123, 45, 1657, 51, 4825, 14081, 31049}},
	{2085, 15, 1970, {1, 1, 5, 7, 21, 59, 21, 249, 77, 793, 1687, 2561, 2241, 4321, 7477}},
	{2086, 15, 1979, {1, 1, 1, 7, 15, 35, 71, 29, 267, 611, 1813, 1823, 7039, 3299, 9919}},
	{2087, 15, 1982, {1, 3, 7, 11, 21, 59, 109, 213, 371, 785, 659, 1687, 4827, 6017, 19619}},
	{2088, 15, 2002, {1, 1, 3, 11, 27, 17, 1, 55, 367, 939, 333, 127, 5105, 2405, 28139}},
	{2089, 15, 2020, {1, 1, 7, 13, 5, 35, 59, 133, 509, 573, 625, 3857, 7935, 5279, 3727}},
	{2090, 15, 2024, {1, 1, 1, 7, 11, 47, 127, 157, 19, 403, 151, 1143, 7407, 8985, 32521}},
	{2091, 15, 2032, {1, 3, 1, 1, 5, 13, 105, 123, 63, 139, 1569, 1983, 563, 7175, 27705}},
	{2092, 15, 2053, {1, 1, 3, 13, 9, 35, 105, 227, 145, 21, 1369, 57, 393, 2921, 18511}},
	{2093, 15, 2060, {1, 3, 1, 7, 17, 61, 99, 187, 261, 281, 437, 2219, 5999, 1857, 18001}},
	{2094, 15, 2063, {1, 3, 3, 5, 1, 59, 67, 45, 451, 439, 2005, 3607, 3, 7167, 14227}},
	{2095, 15, 2066, {1, 3, 3, 3, 29, 19, 25, 251, 275, 733, 1749, 4021, 871, 3227, 13701}},
	{2096, 15, 2075, {1, 3, 3, 13, 27, 53, 57, 243, 491, 521, 1921, 1037, 5013, 5703, 15261}},
	{2097, 15, 2078, {1, 3, 1, 11, 13, 57, 1, 15, 123, 533, 785, 335, 1423, 14269, 3483}},
	{2098, 15, 2081, {1, 3, 7, 13, 15, 55, 5, 139, 385, 47, 1981, 1291, 7397, 12925, 29445}},
	{2099, 15, 2091, {1, 1, 7, 1, 23, 23, 59, 93, 117, 57, 63, 3047, 4849, 11637, 25311}},
	{2100, 15, 2096, {1, 1, 7, 13, 19, 37, 25, 203, 477, 447, 1345, 3485, 2099, 13347, 11621}},
	{2101, 15, 2102, {1, 1, 7, 3, 11, 23, 81, 17, 41, 735, 1149, 3253, 7665, 8291, 22293}},
	{2102, 15, 2106, {1, 1, 5, 3, 15, 9, 57, 167, 463, 493, 747, 1947, 6471, 1111, 31619}},
	{2103, 15, 2116, {1, 1, 5, 15, 7, 15, 107, 205, 325, 167, 1749, 927, 3589, 6127, 7617}},
	{2104, 15, 2120, {1, 1, 1, 13, 21, 25, 83, 147, 411, 399, 1423, 2279, 3661, 7591, 17429}},
	{2105, 15, 2125, {1, 1, 1, 9, 5, 17, 69, 205, 243, 647, 473, 1717, 1977, 10725, 2913}},
	{2106, 15, 2134, {1, 1, 3, 5, 5, 37, 103, 15, 485, 641, 1761, 3755, 6997, 10985, 11773}},
	{2107, 15, 2178, {1, 1, 5, 13, 9, 51, 87, 195, 97, 807, 1801, 961, 6341, 4307, 29105}},
	{2108, 15, 2180, {1, 3, 1, 13, 9, 35, 83, 61, 387, 817, 951, 3993, 7831, 8479, 23941}},
	{2109, 15, 2187, {1, 1, 7, 11, 19, 47, 75, 37, 91, 337, 953, 1169, 163, 2259, 24713}},
	{2110, 15, 2189, {1, 1, 1, 11, 13, 15, 83, 171, 159, 87, 619, 2973, 2653, 13725, 12499}},
	{2111, 15, 2190, {1, 3, 5, 3, 5, 63, 119, 25, 343, 269, 553, 2183, 959, 3825, 22189}},
	{2112, 15, 2208, {1, 1, 5, 15, 5, 37, 89, 109, 497, 1013, 265, 669, 1859, 2647, 3445}},
	{2113, 15, 2214, {1, 3, 3, 9, 21, 21, 15, 245, 107, 649, 367, 1601, 7279, 15783, 4943}},
	{2114, 15, 2237, {1, 3, 3, 15, 5, 41, 125, 113, 159, 161, 1191, 3491, 3531, 55, 20857}},
	{2115, 15, 2252, {1, 3, 5, 9, 21, 57, 21, 195, 99, 193, 1915, 2923, 6349, 15085, 24929}},
	{2116, 15, 2257, {1, 1, 1, 11, 31, 11, 73, 141, 361, 621, 1021, 2067, 5115, 12665, 26845}},
	{2117, 15, 2260, {1, 1, 1, 3, 29, 11, 43, 61, 209, 923, 1753, 1937, 843, 205, 8367}},
	{2118, 15, 2264, {1, 1, 1, 5, 15, 33, 119, 209, 215, 973, 1775, 815, 6693, 7957, 14517}},
	{2119, 15, 2270, {1, 1, 1, 5, 17, 57, 27, 147, 489, 59, 1439, 2279, 445, 11791, 19739}},
	{2120, 15, 2279, {1, 3, 1, 7, 11, 55, 1, 83, 305, 17, 1909, 405, 2325, 5293, 28559}},
	{2121, 15, 2288, {1, 3, 3, 7, 11, 27, 103, 157, 455, 1005, 2033, 3145, 1919, 15723, 25197}},
	{2122, 15, 2305, {1, 1, 5, 11, 15, 51, 37, 131, 503, 1007, 1795, 2421, 1335, 7413, 21741}},
	{2123, 15, 2312, {1, 1, 3, 1, 23, 63, 69, 83, 419, 283, 583, 123, 7725, 2243, 8403}},
	{2124, 15, 2317, {1, 1, 5, 5, 27, 45, 109, 17, 299, 65, 351, 947, 1165, 10723, 2053}},
	{2125, 15, 2323, {1, 1, 3, 3, 23, 61, 115, 253, 1, 931, 1481, 3187, 441, 14735, 27207}},
	{2126, 15, 2329, {1, 1, 5, 3, 25, 11, 83, 141, 359, 343, 901, 1629, 731, 12841, 14357}},
	{2127, 15, 2335, {1, 1, 3, 9, 7, 45, 97, 3, 299, 217, 663, 1527, 6379, 4527, 26147}},
	{2128, 15, 2342, {1, 1, 7, 9, 11, 53, 9, 203, 337, 713, 1517, 719, 4587, 11443, 26905}},
	{2129, 15, 2345, {1, 1, 7, 9, 11, 41, 125, 213, 237, 377, 361, 3231, 4223, 3263, 12655}},
	{2130, 15, 2365, {1, 3, 7, 7, 7, 33, 99, 19, 117, 273, 985, 107, 3831, 10135, 19423}},
	{2131, 15, 2371, {1, 1, 5, 15, 25, 41, 13, 125, 449, 169, 1149, 4021, 5663, 3077, 19163}},
	{2132, 15, 2378, {1, 3, 5, 9, 25, 57, 47, 103, 269, 51, 1805, 2503, 6687, 8065, 12045}},
	{2133, 15, 2385, {1, 3, 5, 7, 3, 35, 87, 225, 189, 229, 931, 3293, 1347, 1427, 3269}},
	{2134, 15, 2395, {1, 1, 1, 3, 5, 31, 61, 19, 247, 9, 1667, 343, 559, 2703, 3763}},
	{2135, 15, 2404, {1, 3, 5, 15, 31, 19, 57, 187, 109, 121, 1287, 2269, 659, 16235, 1273}},
	{2136, 15, 2414, {1, 1, 1, 3, 5, 47, 59, 243, 255, 97, 1959, 1723, 1347, 3019, 26989}},
	{2137, 15, 2426, {1, 3, 3, 15, 29, 35, 75, 67, 497, 731, 193, 3307, 3579, 12005, 7209}},
	{2138, 15, 2428, {1, 1, 5, 9, 13, 35, 79, 213, 51, 983, 1927, 1793, 5037, 5463, 965}},
	{2139, 15, 2441, {1, 1, 7, 11, 5, 41, 7, 83, 15, 411, 1775, 3515, 6755, 3249, 16425}},
	{2140, 15, 2456, {1, 3, 5, 1, 19, 61, 3, 19, 395, 819, 1331, 179, 5225, 5333, 3601}},
	{2141, 15, 2466, {1, 1, 3, 9, 7, 5, 87, 15, 387, 609, 1465, 277, 987, 8377, 903}},
	{2142, 15, 2468, {1, 1, 1, 3, 15, 11, 123, 107, 355, 333, 285, 1801, 6989, 1549, 25791}},
	{2143, 15, 2475, {1, 1, 7, 13, 27, 13, 73, 111, 481, 227, 1091, 365, 5713, 5087, 27217}},
	{2144, 15, 2489, {1, 3, 3, 15, 1, 55, 95, 213, 377, 405, 139, 1867, 2175, 4217, 28813}},
	{2145, 15, 2495, {1, 3, 5, 11, 21, 43, 109, 155, 181, 901, 1951, 507, 4389, 10815, 3141}},
	{2146, 15, 2497, {1, 1, 1, 15, 17, 11, 43, 215, 501, 19, 259, 3479, 6381, 6927, 31247}},
	{2147, 15, 2510, {1, 3, 5, 15, 19, 61, 75, 41, 391, 95, 865, 1441, 7993, 13979, 24663}},
	{2148, 15, 2512, {1, 3, 1, 3, 21, 15, 115, 213, 1, 645, 777, 1517, 2543, 11223, 3633}},
	{2149, 15, 2522, {1, 3, 5, 3, 9, 57, 39, 211, 407, 65, 1795, 2805, 2799, 8691, 1987}},
	{2150, 15, 2533, {1, 1, 3, 13, 17, 55, 47, 113, 29, 139, 1301, 3303, 1129, 13947, 29821}},
	{2151, 15, 2543, {1, 1, 3, 13, 5, 35, 97, 151, 477, 409, 1397, 3399, 4421, 15929, 6163}},
	{2152, 15, 2551, {1, 3, 1, 9, 21, 51, 99, 133, 149, 763, 623, 173, 4311, 11081, 1095}},
	{2153, 15, 2552, {1, 3, 7, 15, 13, 3, 99, 3, 195, 907, 1335, 1355, 7977, 5773, 32383}},
	{2154, 15, 2557, {1, 1, 3, 9, 17, 43, 43, 217, 475, 917, 1373, 1677, 4871, 9619, 16657}},
	{2155, 15, 2567, {1, 3, 3, 7, 31, 31, 55, 11, 73, 693, 25, 417, 1195, 6225, 32279}},
	{2156, 15, 2581, {1, 3, 5, 9, 21, 57, 127, 149, 79, 379, 1609, 2543, 6473, 16033, 27191}},
	{2157, 15, 2586, {1, 1, 5, 1, 13, 9, 81, 153, 297, 789, 1749, 2819, 3961, 11231, 24927}},
	{2158, 15, 2597, {1, 3, 5, 3, 23, 61, 45, 43, 43, 133, 1481, 1543, 2991, 13739, 10287}},
	{2159, 15, 2601, {1, 1, 3, 9, 25, 43, 31, 177, 337, 193, 1083, 1, 991, 9725, 8379}},
	{2160, 15, 2622, {1, 3, 5, 11, 13, 33, 65, 83, 421, 149, 409, 2443, 7423, 8847, 29599}},
	{2161, 15, 2633, {1, 1, 5, 11, 11, 1, 23, 225, 77, 585, 1505, 2525, 739, 10915, 25733}},
	{2162, 15, 2636, {1, 3, 7, 13, 7, 55, 3, 223, 415, 521, 1865, 2349, 5663, 7455, 16569}},
	{2163, 15, 2642, {1, 1, 7, 13, 1, 45, 121, 49, 463, 99, 1061, 2559, 5087, 13389, 11035}},
	{2164, 15, 2644, {1, 3, 7, 11, 31, 51, 35, 235, 385, 1023, 1771, 2013, 5437, 4877, 22119}},
	{2165, 15, 2653, {1, 3, 3, 11, 21, 3, 11, 119, 81, 737, 1093, 2377, 4055, 1121, 15767}},
	{2166, 15, 2667, {1, 1, 5, 13, 9, 3, 83, 217, 387, 249, 1047, 1861, 4103, 15367, 24545}},
	{2167, 15, 2669, {1, 3, 3, 1, 5, 37, 43, 183, 383, 463, 937, 1165, 1481, 959, 17047}},
	{2168, 15, 2672, {1, 1, 3, 5, 7, 43, 127, 243, 81, 1021, 165, 753, 4711, 12965, 22049}},
	{2169, 15, 2675, {1, 1, 5, 5, 3, 61, 65, 53, 425, 89, 5, 1467, 1395, 9579, 8961}},
	{2170, 15, 2682, {1, 3, 7, 13, 11, 35, 123, 21, 83, 689, 667, 1203, 5959, 15697, 26885}},
	{2171, 15, 2687, {1, 1, 5, 13, 9, 49, 41, 101, 291, 339, 1067, 657, 4453, 1137, 21131}},
	{2172, 15, 2691, {1, 3, 3, 3, 17, 61, 11, 213, 27, 805, 1691, 1057, 6011, 11941, 18883}},
	{2173, 15, 2698, {1, 3, 1, 7, 3, 51, 5, 63, 121, 3, 245, 2631, 3737, 16121, 26803}},
	{2174, 15, 2708, {1, 3, 1, 1, 23, 51, 79, 19, 161, 107, 609, 3489, 3389, 4035, 2427}},
	{2175, 15, 2712, {1, 3, 1, 1, 17, 11, 101, 101, 373, 63, 1641, 285, 1333, 165, 14025}},
	{2176, 15, 2718, {1, 1, 1, 5, 1, 51, 83, 137, 45, 1019, 821, 867, 6055, 10443, 9857}},
	{2177, 15, 2722, {1, 3, 1, 5, 17, 23, 25, 181, 429, 495, 317, 3219, 5963, 13945, 9969}},
	{2178, 15, 2736, {1, 3, 7, 3, 3, 15, 123, 191, 369, 177, 1697, 2113, 3889, 5201, 21839}},
	{2179, 15, 2741, {1, 3, 1, 11, 21, 39, 51, 139, 271, 605, 1007, 3513, 3365, 3781, 6799}},
	{2180, 15, 2756, {1, 1, 7, 5, 13, 19, 47, 165, 249, 405, 255, 1295, 4513, 14395, 5587}},
	{2181, 15, 2765, {1, 1, 3, 7, 5, 17, 99, 1, 393, 31, 621, 797, 6113, 16003, 32043}},
	{2182, 15, 2774, {1, 3, 5, 13, 11, 21, 65, 81, 147, 443, 775, 3671, 7029, 11749, 3339}},
	{2183, 15, 2799, {1, 3, 7, 1, 23, 33, 99, 177, 161, 577, 1729, 617, 3465, 11787, 17577}},
	{2184, 15, 2804, {1, 1, 5, 7, 15, 15, 53, 193, 97, 255, 1223, 545, 5153, 873, 24525}},
	{2185, 15, 2825, {1, 3, 5, 1, 7, 57, 47, 121, 383, 835, 1709, 2363, 4731, 12163, 7001}},
	{2186, 15, 2826, {1, 3, 3, 11, 19, 33, 63, 99, 387, 95, 783, 1009, 6373, 4021, 7685}},
	{2187, 15, 2840, {1, 1, 1, 15, 25, 33, 73, 135, 335, 785, 935, 1927, 5847, 10501, 7719}},
	{2188, 15, 2843, {1, 1, 5, 3, 27, 45, 71, 215, 489, 157, 1189, 2577, 6901, 10219, 3025}},
	{2189, 15, 2846, {1, 1, 7, 7, 21, 3, 97, 225, 101, 159, 293, 2789, 7955, 14829, 1209}},
	{2190, 15, 2849, {1, 3, 1, 5, 23, 41, 83, 63, 361, 195, 1707, 2081, 5363, 6327, 179}},
	{2191, 15, 2867, {1, 1, 3, 1, 21, 51, 59, 67, 175, 363, 825, 2971, 3321, 8837, 11805}},
	{2192, 15, 2876, {1, 3, 7, 1, 19, 3, 15, 21, 429, 675, 1589, 2615, 2575, 1537, 7139}},
	{2193, 15, 2891, {1, 3, 3, 5, 21, 29, 17, 115, 345, 397, 523, 1699, 7043, 11173, 3023}},
	{2194, 15, 2902, {1, 1, 5, 7, 19, 63, 99, 175, 91, 433, 153, 3749, 517, 13667, 7423}},
	{2195, 15, 2912, {1, 3, 7, 3, 25, 23, 53, 149, 65, 551, 1231, 365, 6637, 15137, 16319}},
	{2196, 15, 2917, {1, 3, 7, 13, 5, 45, 11, 151, 323, 31, 1749, 409, 6753, 10503, 14991}},
	{2197, 15, 2927, {1, 3, 7, 3, 5, 21, 29, 117, 321, 341, 1811, 3619, 4337, 12255, 8629}},
	{2198, 15, 2941, {1, 3, 7, 3, 7, 3, 5, 221, 407, 671, 1763, 3669, 2353, 8175, 23489}},
	{2199, 15, 2965, {1, 1, 3, 7, 11, 55, 53, 185, 247, 35, 1823, 513, 1379, 11827, 20069}},
	{2200, 15, 2970, {1, 3, 3, 5, 29, 51, 73, 191, 185, 961, 881, 2019, 5651, 1019, 15587}},
	{2201, 15, 2982, {1, 3, 7, 13, 7, 55, 59, 5, 417, 829, 453, 2339, 587, 13283, 797}},
	{2202, 15, 2993, {1, 3, 7, 3, 11, 41, 75, 85, 65, 149, 1583, 529, 2707, 11479, 7109}},
	{2203, 15, 3018, {1, 3, 7, 9, 13, 57, 37, 243, 91, 613, 665, 171, 1631, 13737, 2377}},
	{2204, 15, 3023, {1, 1, 3, 7, 5, 43, 97, 53, 477, 793, 999, 3647, 2555, 7371, 19295}},
	{2205, 15, 3025, {1, 1, 7, 1, 1, 9, 99, 253, 317, 817, 1559, 2081, 2529, 14611, 15997}},
	{2206, 15, 3026, {1, 3, 3, 1, 5, 41, 57, 121, 387, 441, 709, 1511, 7045, 8409, 13297}},
	{2207, 15, 3028, {1, 1, 1, 13, 29, 57, 63, 183, 327, 473, 1943, 213, 3973, 16289, 2739}},
	{2208, 15, 3032, {1, 3, 7, 9, 25, 15, 75, 185, 335, 881, 1041, 3339, 4471, 6823, 21121}},
	{2209, 15, 3053, {1, 3, 3, 13, 23, 3, 57, 117, 511, 927, 771, 3229, 949, 15487, 11963}},
	{2210, 15, 3054, {1, 1, 3, 7, 27, 19, 55, 207, 331, 705, 1945, 797, 7125, 10493, 16585}},
	{2211, 15, 3065, {1, 3, 1, 1, 29, 7, 91, 93, 459, 93, 1501, 1927, 6415, 16255, 9823}},
	{2212, 15, 3071, {1, 1, 5, 5, 31, 11, 97, 179, 505, 807, 877, 4003, 4377, 8851, 4239}},
	{2213, 15, 3076, {1, 1, 3, 5, 11, 25, 17, 131, 23, 95, 311, 1429, 2029, 13091, 23739}},
	{2214, 15, 3088, {1, 1, 3, 11, 13, 27, 33, 127, 481, 117, 1127, 1619, 6493, 8507, 6615}},
	{2215, 15, 3107, {1, 3, 1, 13, 19, 27, 89, 101, 27, 235, 1579, 1701, 4421, 16037, 16239}},
	{2216, 15, 3146, {1, 3, 1, 15, 1, 15, 3, 117, 317, 475, 1691, 2423, 5519, 1703, 2969}},
	{2217, 15, 3148, {1, 1, 3, 1, 13, 15, 19, 37, 237, 467, 1321, 453, 2169, 13313, 31499}},
	{2218, 15, 3159, {1, 1, 3, 15, 29, 55, 31, 199, 85, 285, 967, 367, 3941, 151, 20587}},
	{2219, 15, 3165, {1, 3, 7, 15, 7, 13, 31, 35, 117, 543, 1179, 3441, 3039, 11225, 30229}},
	{2220, 15, 3170, {1, 1, 3, 15, 3, 43, 1, 63, 353, 395, 1775, 3493, 5175, 13193, 25343}},
	{2221, 15, 3179, {1, 3, 3, 15, 17, 25, 57, 205, 411, 83, 1877, 2093, 5599, 12115, 8751}},
	{2222, 15, 3182, {1, 1, 1, 11, 15, 9, 115, 99, 85, 887, 987, 4015, 7077, 3739, 21505}},
	{2223, 15, 3205, {1, 3, 1, 11, 25, 39, 127, 37, 329, 273, 1531, 3211, 7115, 15501, 26575}},
	{2224, 15, 3212, {1, 1, 5, 13, 15, 3, 3, 101, 431, 645, 493, 723, 8083, 1423, 14879}},
	{2225, 15, 3218, {1, 3, 3, 5, 31, 35, 37, 131, 259, 849, 325, 3403, 3627, 3295, 30885}},
	{2226, 15, 3220, {1, 3, 7, 1, 9, 3, 31, 201, 379, 907, 1005, 3333, 7457, 2533, 30357}},
	{2227, 15, 3223, {1, 3, 1, 9, 7, 7, 95, 103, 121, 157, 895, 2683, 5839, 12403, 14327}},
	{2228, 15, 3227, {1, 3, 7, 3, 13, 5, 55, 233, 3, 855, 859, 1115, 3883, 8041, 3353}},
	{2229, 15, 3233, {1, 1, 5, 9, 3, 55, 99, 79, 263, 831, 1579, 205, 5673, 1999, 14879}},
	{2230, 15, 3234, {1, 3, 1, 5, 17, 25, 85, 19, 189, 141, 877, 667, 4461, 11915, 23247}},
	{2231, 15, 3254, {1, 1, 5, 5, 1, 35, 15, 219, 469, 725, 1793, 3683, 3661, 15627, 30197}},
	{2232, 15, 3263, {1, 1, 7, 5, 27, 3, 41, 153, 431, 487, 759, 1345, 6735, 9937, 26277}},
	{2233, 15, 3268, {1, 1, 1, 11, 11, 13, 41, 121, 265, 465, 1447, 5, 3407, 1907, 10037}},
	{2234, 15, 3272, {1, 3, 5, 9, 15, 63, 5, 7, 407, 83, 365, 3687, 7721, 6973, 16967}},
	{2235, 15, 3277, {1, 1, 7, 7, 5, 41, 75, 155, 417, 565, 1199, 1111, 2823, 10703, 22561}},
	{2236, 15, 3292, {1, 3, 7, 5, 7, 43, 39, 185, 105, 327, 1977, 1137, 3261, 10583, 11661}},
	{2237, 15, 3295, {1, 3, 7, 7, 19, 19, 103, 137, 169, 273, 1357, 3413, 7647, 10531, 32489}},
	{2238, 15, 3296, {1, 1, 3, 13, 13, 3, 81, 23, 161, 295, 735, 2031, 1027, 15513, 20165}},
	{2239, 15, 3301, {1, 1, 5, 1, 15, 1, 91, 35, 375, 207, 1417, 1115, 2237, 11749, 8509}},
	{2240, 15, 3306, {1, 3, 7, 3, 25, 51, 49, 219, 195, 417, 1523, 3953, 5739, 7499, 27071}},
	{2241, 15, 3313, {1, 1, 3, 11, 23, 29, 19, 81, 421, 633, 513, 547, 7545, 29, 11909}},
	{2242, 15, 3346, {1, 1, 1, 7, 13, 61, 33, 243, 221, 231, 111, 879, 2861, 1795, 27531}},
	{2243, 15, 3367, {1, 3, 7, 3, 19, 21, 1, 141, 159, 605, 969, 3013, 6583, 2447, 19919}},
	{2244, 15, 3371, {1, 3, 7, 3, 31, 9, 91, 83, 29, 873, 929, 43, 2253, 12539, 23951}},
	{2245, 15, 3373, {1, 1, 5, 3, 31, 15, 87, 105, 319, 973, 1489, 3417, 3377, 15749, 2357}},
	{2246, 15, 3374, {1, 1, 3, 15, 7, 23, 3, 81, 383, 419, 713, 997, 6873, 593, 285}},
	{2247, 15, 3376, {1, 3, 3, 1, 29, 13, 29, 101, 441, 693, 2039, 2951, 5921, 12129, 12053}},
	{2248, 15, 3382, {1, 1, 3, 15, 9, 29, 97, 117, 421, 433, 1017, 125, 3607, 9415, 6843}},
	{2249, 15, 3388, {1, 3, 5, 9, 11, 13, 75, 155, 413, 75, 109, 1599, 6161, 16115, 12621}},
	{2250, 15, 3391, {1, 3, 3, 3, 11, 13, 49, 225, 401, 599, 1815, 1643, 7853, 13305, 25195}},
	{2251, 15, 3403, {1, 3, 7, 5, 15, 11, 27, 95, 387, 931, 549, 2179, 3397, 15883, 16563}},
	{2252, 15, 3406, {1, 1, 7, 3, 9, 39, 121, 5, 453, 27, 1747, 657, 2593, 1289, 12577}},
	{2253, 15, 3413, {1, 3, 7, 5, 25, 25, 109, 49, 185, 985, 631, 803, 3865, 8955, 17901}},
	{2254, 15, 3420, {1, 1, 3, 13, 3, 59, 47, 49, 139, 275, 1471, 2995, 5593, 14011, 18741}},
	{2255, 15, 3427, {1, 1, 5, 15, 29, 11, 97, 225, 245, 291, 1873, 2365, 767, 3419, 14943}},
	{2256, 15, 3441, {1, 3, 3, 5, 15, 17, 19, 209, 359, 891, 1375, 2003, 7247, 5299, 28841}},
	{2257, 15, 3453, {1, 3, 7, 7, 9, 55, 105, 35, 77, 47, 1023, 13, 2901, 847, 10265}},
	{2258, 15, 3464, {1, 3, 7, 7, 7, 5, 65, 233, 141, 277, 1333, 2357, 443, 7257, 21979}},
	{2259, 15, 3469, {1, 3, 5, 11, 13, 63, 41, 87, 193, 737, 1085, 2317, 7869, 10149, 12163}},
	{2260, 15, 3481, {1, 3, 1, 1, 7, 57, 75, 235, 461, 857, 155, 2679, 5925, 2565, 10881}},
	{2261, 15, 3488, {1, 1, 7, 15, 13, 41, 63, 135, 433, 387, 1943, 2249, 5469, 11679, 28661}},
	{2262, 15, 3497, {1, 3, 3, 13, 5, 3, 103, 161, 367, 649, 789, 1179, 4163, 5699, 16787}},
	{2263, 15, 3503, {1, 3, 7, 7, 31, 13, 45, 141, 113, 769, 1035, 457, 6709, 14989, 27311}},
	{2264, 15, 3511, {1, 1, 3, 1, 1, 43, 119, 145, 111, 593, 1139, 417, 637, 4437, 17285}},
	{2265, 15, 3515, {1, 3, 5, 9, 9, 33, 19, 99, 201, 685, 1793, 2621, 6857, 8769, 5623}},
	{2266, 15, 3525, {1, 3, 5, 5, 23, 43, 27, 189, 325, 415, 215, 1253, 3599, 1215, 10093}},
	{2267, 15, 3529, {1, 1, 3, 13, 11, 35, 113, 173, 503, 19, 1459, 503, 5363, 3967, 13945}},
	{2268, 15, 3547, {1, 1, 5, 11, 31, 49, 13, 173, 199, 623, 1231, 2495, 6581, 7957, 25321}},
	{2269, 15, 3550, {1, 3, 1, 9, 23, 3, 79, 149, 505, 937, 1839, 3701, 1673, 8589, 8031}},
	{2270, 15, 3573, {1, 3, 3, 5, 21, 27, 107, 11, 505, 407, 177, 3593, 4729, 12773, 11685}},
	{2271, 15, 3583, {1, 3, 1, 11, 29, 49, 79, 53, 61, 895, 2035, 563, 5613, 6065, 6207}},
	{2272, 15, 3594, {1, 1, 3, 7, 1, 53, 3, 215, 99, 865, 1749, 3533, 4305, 1243, 28463}},
	{2273, 15, 3607, {1, 1, 1, 13, 31, 59, 115, 53, 403, 909, 847, 103, 4967, 10623, 30073}},
	{2274, 15, 3613, {1, 1, 7, 5, 27, 1, 119, 83, 457, 81, 395, 811, 6221, 14337, 541}},
	{2275, 15, 3624, {1, 1, 5, 5, 5, 53, 83, 117, 269, 327, 875, 101, 3343, 715, 26339}},
	{2276, 15, 3630, {1, 1, 1, 11, 31, 39, 121, 147, 305, 383, 1211, 1897, 7647, 11687, 18907}},
	{2277, 15, 3635, {1, 3, 3, 15, 23, 53, 17, 85, 395, 503, 61, 1745, 4713, 4641, 13787}},
	{2278, 15, 3642, {1, 1, 7, 7, 27, 1, 105, 29, 287, 37, 959, 975, 4427, 4705, 10175}},
	{2279, 15, 3644, {1, 3, 3, 5, 7, 63, 57, 199, 27, 107, 1095, 3923, 6969, 713, 11619}},
	{2280, 15, 3650, {1, 3, 5, 1, 5, 49, 85, 45, 449, 45, 49, 3419, 1109, 455, 15917}},
	{2281, 15, 3679, {1, 1, 1, 5, 13, 15, 39, 27, 467, 85, 1537, 3055, 1977, 8829, 25231}},
	{2282, 15, 3690, {1, 1, 1, 15, 1, 47, 23, 121, 147, 547, 1865, 1491, 779, 3515, 12667}},
	{2283, 15, 3698, {1, 3, 3, 1, 19, 5, 77, 101, 1, 721, 1149, 2967, 4925, 11889, 16655}},
	{2284, 15, 3704, {1, 1, 1, 7, 1, 35, 95, 239, 127, 855, 1031, 455, 7631, 6039, 21983}},
	{2285, 15, 3707, {1, 3, 7, 9, 23, 43, 75, 105, 335, 223, 1825, 3217, 413, 7473, 30005}},
	{2286, 15, 3713, {1, 1, 5, 15, 29, 9, 43, 145, 223, 523, 511, 323, 5955, 11141, 22533}},
	{2287, 15, 3754, {1, 1, 3, 1, 13, 61, 93, 133, 461, 233, 383, 693, 7347, 3165, 27493}},
	{2288, 15, 3756, {1, 3, 7, 1, 13, 45, 113, 207, 53, 1007, 815, 1145, 2937, 289, 22195}},
	{2289, 15, 3761, {1, 3, 5, 5, 19, 17, 113, 89, 19, 1023, 1625, 3277, 697, 5187, 15433}},
	{2290, 15, 3776, {1, 1, 3, 13, 21, 15, 15, 197, 409, 391, 1993, 2475, 3189, 4431, 29585}},
	{2291, 15, 3781, {1, 1, 5, 5, 31, 7, 111, 231, 187, 543, 45, 3863, 3811, 4573, 4437}},
	{2292, 15, 3788, {1, 3, 3, 7, 19, 7, 123, 23, 79, 513, 189, 3663, 1291, 13257, 8949}},
	{2293, 15, 3791, {1, 1, 5, 13, 3, 53, 109, 133, 157, 223, 651, 3059, 6055, 14455, 26903}},
	{2294, 15, 3794, {1, 1, 7, 1, 23, 63, 59, 229, 17, 199, 643, 637, 7631, 13647, 7399}},
	{2295, 15, 3806, {1, 1, 1, 3, 1, 51, 119, 67, 335, 543, 913, 3565, 4795, 13405, 7463}},
	{2296, 15, 3841, {1, 1, 5, 3, 31, 5, 91, 97, 23, 223, 837, 1353, 1929, 12043, 10039}},
	{2297, 15, 3848, {1, 3, 5, 7, 19, 3, 79, 171, 301, 687, 1545, 355, 4709, 12965, 16797}},
	{2298, 15, 3851, {1, 3, 5, 11, 11, 49, 111, 123, 251, 569, 1605, 401, 5439, 13519, 8847}},
	{2299, 15, 3856, {1, 3, 1, 3, 3, 53, 7, 55, 369, 633, 181, 4037, 2993, 15815, 8661}},
	{2300, 15, 3868, {1, 1, 1, 13, 31, 29, 75, 167, 279, 597, 539, 1791, 8013, 4387, 9717}},
	{2301, 15, 3875, {1, 1, 5, 7, 17, 15, 99, 183, 211, 49, 225, 3143, 4537, 13141, 23375}},
	{2302, 15, 3882, {1, 1, 3, 5, 3, 59, 25, 149, 467, 69, 1939, 1007, 2765, 4693, 29815}},
	{2303, 15, 3884, {1, 3, 1, 3, 17, 33, 119, 189, 447, 251, 879, 177, 5395, 13487, 9587}},
	{2304, 15, 3889, {1, 3, 3, 7, 15, 31, 115, 3, 21, 817, 475, 1849, 6041, 12541, 18701}},
	{2305, 15, 3892, {1, 1, 5, 13, 31, 33, 7, 115, 361, 587, 1919, 1007, 3537, 7493, 19357}},
	{2306, 15, 3919, {1, 3, 7, 13, 23, 35, 15, 111, 123, 633, 805, 1983, 2109, 14477, 4985}},
	{2307, 15, 3921, {1, 3, 3, 11, 25, 13, 11, 205, 97, 893, 927, 1291, 4007, 13593, 29693}},
	{2308, 15, 3958, {1, 3, 5, 15, 9, 13, 121, 89, 215, 823, 1389, 1581, 8095, 4707, 16061}},
	{2309, 15, 3961, {1, 3, 1, 3, 23, 39, 83, 23, 47, 941, 1419, 2389, 5699, 7519, 5829}},
	{2310, 15, 3973, {1, 3, 1, 9, 23, 43, 79, 237, 93, 203, 695, 225, 5645, 3591, 16775}},
	{2311, 15, 3977, {1, 3, 5, 3, 15, 19, 89, 129, 375, 125, 225, 1323, 2267, 11607, 17937}},
	{2312, 15, 3985, {1, 3, 3, 1, 31, 37, 93, 133, 377, 959, 707, 621, 7179, 15493, 30287}},
	{2313, 15, 3991, {1, 3, 7, 13, 5, 13, 15, 1, 37, 525, 1641, 2829, 6139, 4069, 19187}},
	{2314, 15, 4004, {1, 3, 3, 9, 17, 3, 67, 97, 375, 845, 403, 973, 3919, 2275, 31627}},
	{2315, 15, 4007, {1, 1, 3, 3, 25, 7, 91, 67, 271, 465, 481, 3477, 5229, 241, 8411}},
	{2316, 15, 4019, {1, 1, 1, 11, 1, 41, 109, 115, 75, 787, 309, 2887, 179, 9073, 13895}},
	{2317, 15, 4045, {1, 3, 3, 15, 11, 31, 113, 91, 303, 907, 1933, 2167, 7799, 11821, 20659}},
	{2318, 15, 4054, {1, 3, 1, 15, 27, 17, 21, 41, 99, 137, 1397, 929, 5819, 11977, 6201}},
	{2319, 15, 4057, {1, 1, 7, 13, 21, 29, 47, 239, 287, 305, 899, 2711, 1723, 3315, 199}},
	{2320, 15, 4058, {1, 1, 1, 3, 31, 21, 101, 149, 107, 761, 1197, 1703, 4803, 8411, 10649}},
	{2321, 15, 4070, {1, 1, 5, 15, 23, 45, 109, 221, 85, 619, 169, 1013, 3305, 9451, 26189}},
	{2322, 15, 4101, {1, 3, 5, 13, 7, 57, 19, 153, 231, 627, 565, 1595, 6309, 5037, 25505}},
	{2323, 15, 4113, {1, 1, 7, 7, 1, 45, 43, 79, 271, 59, 219, 2255, 1785, 7919, 24061}},
	{2324, 15, 4114, {1, 3, 7, 5, 31, 57, 57, 231, 33, 227, 531, 679, 1141, 85, 19777}},
	{2325, 15, 4119, {1, 1, 3, 15, 11, 59, 59, 169, 459, 693, 907, 1191, 3783, 12809, 6263}},
	{2326, 15, 4129, {1, 1, 7, 13, 19, 21, 105, 65, 267, 141, 1547, 781, 7295, 13565, 17775}},
	{2327, 15, 4141, {1, 3, 3, 5, 31, 63, 97, 155, 477, 661, 329, 797, 2539, 4061, 10537}},
	{2328, 15, 4142, {1, 3, 3, 7, 11, 17, 119, 89, 71, 103, 1043, 413, 6035, 12829, 11559}},
	{2329, 15, 4147, {1, 3, 1, 9, 5, 19, 53, 185, 103, 629, 2015, 1257, 5163, 10581, 13449}},
	{2330, 15, 4149, {1, 1, 1, 5, 23, 35, 25, 129, 179, 959, 677, 2249, 6315, 12151, 3459}},
	{2331, 15, 4150, {1, 1, 1, 1, 9, 47, 93, 45, 35, 45, 265, 2065, 6225, 25, 27135}},
	{2332, 15, 4164, {1, 3, 1, 11, 21, 53, 127, 163, 311, 667, 597, 1561, 4515, 23, 9551}},
	{2333, 15, 4168, {1, 1, 3, 3, 7, 47, 105, 211, 241, 95, 389, 899, 6001, 8129, 19889}},
	{2334, 15, 4186, {1, 1, 3, 15, 29, 45, 9, 27, 483, 799, 269, 1811, 4493, 7109, 22149}},
	{2335, 15, 4198, {1, 1, 3, 3, 29, 5, 57, 205, 187, 615, 1677, 3987, 4577, 8799, 16311}},
	{2336, 15, 4207, {1, 1, 5, 3, 15, 5, 91, 101, 319, 445, 1261, 2039, 4071, 8249, 11611}},
	{2337, 15, 4221, {1, 3, 7, 11, 19, 17, 1, 185, 153, 579, 1001, 2031, 2295, 16335, 24771}},
	{2338, 15, 4225, {1, 3, 3, 15, 13, 45, 93, 185, 319, 667, 1085, 93, 577, 11551, 11355}},
	{2339, 15, 4231, {1, 1, 7, 13, 3, 61, 45, 191, 51, 981, 1151, 2715, 2503, 4147, 4587}},
	{2340, 15, 4238, {1, 1, 3, 3, 27, 17, 71, 141, 57, 981, 1033, 333, 4639, 15885, 1039}},
	{2341, 15, 4243, {1, 3, 3, 15, 21, 55, 33, 123, 357, 893, 829, 4045, 5027, 11727, 13357}},
	{2342, 15, 4249, {1, 1, 1, 9, 31, 47, 27, 223, 311, 205, 179, 3411, 4019, 10997, 28115}},
	{2343, 15, 4250, {1, 3, 5, 1, 3, 39, 15, 7, 501, 641, 735, 295, 2005, 12641, 19779}},
	{2344, 15, 4252, {1, 3, 3, 1, 15, 1, 75, 243, 329, 267, 1323, 2285, 5389, 11881, 15737}},
	{2345, 15, 4259, {1, 1, 3, 3, 13, 17, 101, 99, 209, 939, 1147, 3221, 5159, 3435, 183}},
	{2346, 15, 4279, {1, 1, 1, 1, 27, 43, 29, 179, 179, 659, 807, 313, 4165, 963, 11317}},
	{2347, 15, 4285, {1, 1, 3, 13, 9, 51, 125, 245, 381, 555, 1383, 3887, 2045, 12829, 12029}},
	{2348, 15, 4288, {1, 1, 1, 9, 29, 39, 55, 127, 235, 617, 1553, 3133, 7735, 14725, 16733}},
	{2349, 15, 4303, {1, 1, 3, 5, 15, 9, 47, 217, 89, 987, 1083, 1045, 4745, 12915, 13719}},
	{2350, 15, 4312, {1, 3, 3, 7, 23, 3, 35, 79, 45, 435, 1549, 2645, 2831, 10359, 10041}},
	{2351, 15, 4322, {1, 1, 7, 15, 31, 61, 25, 223, 511, 319, 487, 1677, 739, 7097, 18417}},
	{2352, 15, 4327, {1, 1, 7, 5, 19, 21, 123, 237, 299, 367, 1341, 1449, 2949, 8629, 11051}},
	{2353, 15, 4336, {1, 3, 7, 7, 31, 53, 125, 33, 257, 719, 1297, 895, 5095, 10237, 12309}},
	{2354, 15, 4359, {1, 3, 1, 5, 31, 59, 73, 211, 97, 209, 1289, 4033, 6143, 14275, 7997}},
	{2355, 15, 4384, {1, 1, 5, 7, 31, 5, 75, 105, 389, 985, 9, 4033, 1185, 7821, 19083}},
	{2356, 15, 4387, {1, 1, 1, 15, 11, 39, 73, 253, 275, 813, 25, 3441, 2493, 5873, 3739}},
	{2357, 15, 4401, {1, 3, 7, 1, 31, 19, 119, 5, 109, 397, 1329, 3347, 5941, 12449, 2533}},
	{2358, 15, 4407, {1, 1, 1, 1, 5, 59, 61, 175, 435, 985, 65, 3781, 5425, 15073, 16361}},
	{2359, 15, 4428, {1, 3, 5, 7, 31, 13, 53, 87, 69, 305, 1455, 273, 2197, 4277, 24423}},
	{2360, 15, 4436, {1, 3, 3, 15, 13, 13, 91, 171, 71, 583, 15, 3599, 6801, 10041, 26097}},
	{2361, 15, 4450, {1, 3, 3, 5, 5, 13, 91, 225, 63, 69, 1795, 341, 461, 5015, 9471}},
	{2362, 15, 4452, {1, 3, 7, 5, 21, 55, 109, 39, 459, 925, 229, 2855, 5807, 2117, 31739}},
	{2363, 15, 4459, {1, 1, 3, 3, 1, 5, 17, 177, 401, 727, 1555, 3097, 1243, 5933, 14579}},
	{2364, 15, 4461, {1, 1, 7, 3, 19, 19, 37, 87, 105, 73, 197, 4067, 6237, 10553, 9207}},
	{2365, 15, 4470, {1, 1, 3, 15, 1, 55, 119, 115, 441, 3, 1003, 1631, 197, 12929, 25385}},
	{2366, 15, 4483, {1, 3, 7, 11, 31, 1, 119, 49, 467, 647, 685, 2771, 3689, 11049, 26787}},
	{2367, 15, 4485, {1, 1, 1, 11, 19, 19, 21, 73, 459, 935, 615, 371, 1099, 14407, 10375}},
	{2368, 15, 4486, {1, 3, 5, 13, 15, 3, 107, 179, 259, 677, 1101, 315, 7673, 14639, 11241}},
	{2369, 15, 4492, {1, 1, 7, 9, 15, 21, 93, 25, 349, 23, 1087, 27, 5691, 12997, 29301}},
	{2370, 15, 4497, {1, 3, 3, 5, 7, 43, 1, 195, 69, 753, 1315, 2629, 3259, 5595, 19439}},
	{2371, 15, 4514, {1, 3, 5, 5, 31, 9, 75, 217, 217, 197, 1925, 2033, 3585, 15219, 20251}},
	{2372, 15, 4533, {1, 1, 5, 11, 17, 31, 3, 209, 315, 49, 949, 2267, 4611, 4375, 16431}},
	{2373, 15, 4537, {1, 1, 7, 9, 17, 35, 13, 115, 119, 553, 1527, 2857, 3599, 391, 25101}},
	{2374, 15, 4546, {1, 3, 3, 15, 13, 59, 17, 177, 301, 719, 909, 1663, 5033, 1129, 529}},
	{2375, 15, 4551, {1, 1, 7, 5, 15, 13, 99, 157, 379, 975, 1019, 2251, 3807, 10621, 351}},
	{2376, 15, 4555, {1, 3, 3, 13, 5, 57, 5, 31, 361, 981, 883, 3723, 2259, 5151, 11783}},
	{2377, 15, 4560, {1, 1, 1, 13, 1, 43, 125, 19, 77, 509, 1817, 3795, 1863, 8401, 27253}},
	{2378, 15, 4569, {1, 1, 5, 7, 19, 41, 21, 151, 89, 189, 769, 1937, 4497, 13607, 24691}},
	{2379, 15, 4576, {1, 1, 1, 9, 21, 9, 1, 195, 31, 907, 1719, 1549, 809, 13629, 16597}},
	{2380, 15, 4582, {1, 1, 1, 3, 21, 61, 103, 219, 311, 849, 523, 21, 4533, 6367, 3935}},
	{2381, 15, 4586, {1, 1, 7, 9, 7, 33, 77, 19, 489, 933, 1729, 1813, 6741, 10701, 7}},
	{2382, 15, 4609, {1, 1, 1, 5, 23, 53, 43, 63, 453, 209, 1313, 2847, 2641, 13783, 14983}},
	{2383, 15, 4610, {1, 3, 7, 7, 15, 45, 83, 241, 509, 659, 213, 221, 5205, 6019, 18945}},
	{2384, 15, 4612, {1, 1, 5, 9, 25, 43, 37, 9, 191, 505, 765, 295, 953, 1045, 11203}},
	{2385, 15, 4649, {1, 3, 7, 11, 5, 49, 45, 177, 379, 695, 355, 1711, 7747, 497, 7597}},
	{2386, 15, 4652, {1, 1, 5, 13, 23, 47, 101, 145, 301, 207, 195, 2225, 8093, 15345, 14097}},
	{2387, 15, 4672, {1, 3, 7, 13, 9, 9, 55, 223, 343, 921, 1825, 3281, 2627, 855, 27651}},
	{2388, 15, 4677, {1, 1, 7, 1, 21, 1, 67, 149, 433, 111, 577, 3675, 495, 9043, 23613}},
	{2389, 15, 4684, {1, 3, 1, 13, 9, 39, 37, 73, 117, 559, 1131, 2511, 7599, 8393, 24747}},
	{2390, 15, 4690, {1, 3, 3, 7, 11, 15, 85, 229, 7, 21, 1649, 739, 375, 13991, 27053}},
	{2391, 15, 4695, {1, 1, 5, 5, 15, 41, 49, 117, 173, 825, 1343, 377, 1789, 12519, 30667}},
	{2392, 15, 4696, {1, 1, 7, 15, 9, 11, 97, 99, 347, 729, 9, 1703, 1177, 5189, 9061}},
	{2393, 15, 4702, {1, 1, 5, 11, 15, 25, 99, 63, 89, 675, 561, 215, 8111, 3955, 24635}},
	{2394, 15, 4705, {1, 1, 1, 1, 7, 53, 99, 193, 233, 731, 733, 1883, 7783, 14413, 14003}},
	{2395, 15, 4717, {1, 3, 5, 7, 31, 23, 45, 153, 337, 293, 443, 2301, 5135, 7455, 13123}},
	{2396, 15, 4726, {1, 3, 1, 3, 23, 53, 23, 165, 53, 875, 1543, 1035, 4247, 5101, 28445}},
	{2397, 15, 4736, {1, 1, 1, 15, 13, 41, 77, 93, 205, 743, 1101, 1413, 2371, 7183, 12337}},
	{2398, 15, 4753, {1, 1, 3, 15, 17, 63, 25, 101, 147, 149, 1207, 3525, 2661, 9539, 11145}},
	{2399, 15, 4754, {1, 3, 1, 9, 17, 5, 3, 35, 389, 909, 1017, 2803, 5243, 13025, 8851}},
	{2400, 15, 4756, {1, 1, 7, 15, 19, 27, 69, 91, 71, 547, 1421, 831, 6969, 5517, 28233}},
	{2401, 15, 4775, {1, 1, 3, 3, 17, 45, 55, 63, 263, 819, 1211, 2739, 655, 13269, 22281}},
	{2402, 15, 4801, {1, 3, 1, 5, 23, 13, 81, 251, 83, 551, 491, 1029, 3561, 357, 23393}},
	{2403, 15, 4819, {1, 3, 1, 13, 25, 27, 93, 143, 407, 403, 1395, 1733, 3187, 1917, 31453}},
	{2404, 15, 4828, {1, 1, 7, 13, 3, 21, 85, 113, 483, 461, 1343, 561, 2081, 10857, 24253}},
	{2405, 15, 4838, {1, 1, 1, 1, 11, 11, 53, 135, 25, 163, 1729, 617, 1533, 10881, 16041}},
	{2406, 15, 4852, {1, 1, 5, 1, 3, 49, 125, 139, 77, 891, 815, 3431, 4875, 12513, 4595}},
	{2407, 15, 4856, {1, 1, 1, 1, 27, 63, 111, 109, 421, 425, 345, 1613, 5447, 1357, 32413}},
	{2408, 15, 4873, {1, 3, 5, 3, 17, 5, 37, 171, 259, 281, 1003, 2901, 3241, 15557, 21415}},
	{2409, 15, 4887, {1, 1, 5, 11, 15, 55, 75, 199, 493, 215, 1625, 2345, 7873, 2325, 11003}},
	{2410, 15, 4891, {1, 3, 7, 1, 21, 33, 23, 5, 495, 941, 1185, 475, 5799, 15161, 10677}},
	{2411, 15, 4904, {1, 1, 5, 9, 31, 37, 37, 29, 217, 389, 297, 3097, 7319, 2601, 15307}},
	{2412, 15, 4912, {1, 3, 7, 5, 7, 45, 111, 167, 297, 275, 1669, 2489, 1511, 15753, 1289}},
	{2413, 15, 4921, {1, 3, 1, 7, 3, 45, 19, 11, 189, 199, 1227, 2647, 1897, 9077, 17189}},
	{2414, 15, 4936, {1, 1, 1, 13, 15, 39, 19, 179, 147, 341, 283, 3029, 7599, 8937, 18761}},
	{2415, 15, 4941, {1, 3, 3, 9, 3, 11, 41, 255, 365, 835, 921, 389, 919, 15223, 14541}},
	{2416, 15, 4942, {1, 1, 3, 3, 5, 37, 29, 203, 313, 271, 1207, 487, 3711, 3811, 26757}},
	{2417, 15, 4963, {1, 3, 7, 9, 19, 53, 49, 139, 351, 537, 1681, 1595, 5399, 13839, 28685}},
	{2418, 15, 4984, {1, 3, 1, 1, 15, 35, 21, 37, 247, 891, 1855, 1243, 3137, 10381, 30379}},
	{2419, 15, 4990, {1, 3, 7, 5, 9, 47, 91, 25, 479, 337, 781, 3545, 1045, 9491, 22853}},
	{2420, 15, 5005, {1, 1, 5, 15, 19, 31, 81, 5, 117, 923, 565, 2443, 7383, 1795, 11685}},
	{2421, 15, 5013, {1, 3, 3, 5, 17, 15, 21, 245, 489, 889, 2047, 2737, 7445, 14785, 13401}},
	{2422, 15, 5020, {1, 1, 1, 15, 19, 45, 67, 117, 299, 607, 953, 743, 6863, 12123, 6701}},
	{2423, 15, 5039, {1, 1, 3, 1, 1, 43, 19, 129, 345, 861, 209, 2387, 7205, 7131, 8235}},
	{2424, 15, 5048, {1, 3, 5, 1, 1, 13, 75, 99, 333, 157, 23, 1217, 1857, 15479, 16031}},
	{2425, 15, 5062, {1, 3, 3, 11, 7, 61, 119, 89, 491, 401, 227, 1739, 3807, 16003, 2875}},
	{2426, 15, 5080, {1, 3, 7, 15, 13, 55, 3, 159, 405, 593, 975, 361, 2563, 6061, 28087}},
	{2427, 15, 5085, {1, 1, 3, 13, 19, 5, 5, 9, 119, 41, 33, 1111, 4443, 4663, 28841}},
	{2428, 15, 5086, {1, 1, 7, 7, 25, 59, 125, 255, 49, 947, 1673, 2947, 6369, 2267, 8813}},
	{2429, 15, 5095, {1, 1, 5, 15, 25, 25, 111, 193, 217, 193, 821, 2779, 69, 2957, 27043}},
	{2430, 15, 5096, {1, 3, 5, 7, 21, 19, 51, 157, 203, 487, 1745, 1875, 911, 14071, 7557}},
	{2431, 15, 5102, {1, 1, 5, 9, 3, 15, 55, 73, 313, 245, 1061, 1929, 3035, 607, 11563}},
	{2432, 15, 5107, {1, 1, 5, 7, 3, 57, 105, 121, 461, 43, 803, 1801, 4059, 2157, 17547}},
	{2433, 15, 5141, {1, 3, 7, 7, 19, 11, 1, 121, 499, 841, 601, 3515, 2969, 13697, 8917}},
	{2434, 15, 5145, {1, 3, 3, 3, 13, 35, 113, 231, 391, 689, 697, 2871, 7387, 715, 27005}},
	{2435, 15, 5148, {1, 1, 1, 13, 19, 5, 17, 43, 175, 291, 987, 1917, 7635, 15655, 10689}},
	{2436, 15, 5157, {1, 1, 7, 15, 19, 37, 121, 243, 125, 623, 1231, 29, 2325, 5147, 21435}},
	{2437, 15, 5158, {1, 3, 5, 15, 25, 27, 57, 187, 77, 401, 1489, 2977, 5415, 3381, 2551}},
	{2438, 15, 5162, {1, 1, 1, 7, 1, 1, 85, 27, 115, 559, 9, 2365, 711, 5733, 2819}},
	{2439, 15, 5172, {1, 3, 1, 15, 9, 29, 61, 113, 169, 349, 591, 1061, 6041, 7613, 23691}},
	{2440, 15, 5182, {1, 1, 5, 1, 13, 45, 49, 227, 345, 563, 87, 3597, 3961, 7205, 8441}},
	{2441, 15, 5184, {1, 1, 1, 5, 3, 21, 121, 183, 463, 83, 1365, 539, 1485, 10063, 24867}},
	{2442, 15, 5193, {1, 3, 5, 5, 3, 61, 101, 237, 41, 147, 1907, 3049, 7583, 8283, 6099}},
	{2443, 15, 5199, {1, 3, 1, 15, 31, 57, 19, 155, 445, 805, 1793, 207, 1975, 3357, 14281}},
	{2444, 15, 5201, {1, 1, 7, 13, 9, 39, 27, 73, 165, 345, 543, 4095, 133, 10469, 11573}},
	{2445, 15, 5204, {1, 1, 7, 15, 17, 57, 99, 81, 359, 367, 1057, 1173, 4225, 15127, 2615}},
	{2446, 15, 5211, {1, 3, 5, 3, 31, 23, 113, 111, 495, 947, 1625, 1195, 2053, 1509, 1347}},
	{2447, 15, 5223, {1, 1, 5, 5, 9, 47, 25, 63, 455, 107, 771, 3815, 3827, 16287, 11615}},
	{2448, 15, 5230, {1, 1, 7, 9, 17, 61, 51, 215, 63, 123, 1253, 3927, 721, 9647, 3283}},
	{2449, 15, 5232, {1, 1, 5, 15, 11, 17, 83, 255, 473, 107, 681, 763, 7855, 8043, 31503}},
	{2450, 15, 5253, {1, 3, 1, 7, 7, 31, 37, 5, 253, 155, 2017, 609, 1421, 14927, 25241}},
	{2451, 15, 5257, {1, 3, 3, 13, 31, 25, 21, 241, 431, 193, 681, 2265, 5091, 11479, 21443}},
	{2452, 15, 5260, {1, 3, 5, 5, 15, 9, 49, 255, 157, 995, 631, 1995, 3605, 9085, 24245}},
	{2453, 15, 5284, {1, 3, 3, 7, 19, 31, 85, 153, 493, 951, 451, 1587, 6609, 3681, 13205}},
	{2454, 15, 5306, {1, 1, 5, 1, 17, 41, 107, 231, 307, 361, 575, 3239, 3443, 16159, 20625}},
	{2455, 15, 5331, {1, 1, 7, 9, 31, 49, 93, 79, 181, 117, 1241, 3645, 4901, 12599, 13247}},
	{2456, 15, 5334, {1, 3, 3, 9, 7, 31, 127, 201, 11, 199, 1851, 23, 5667, 8159, 20951}},
	{2457, 15, 5364, {1, 3, 3, 7, 3, 37, 29, 189, 65, 461, 769, 321, 6577, 16223, 16865}},
	{2458, 15, 5367, {1, 1, 5, 11, 1, 13, 91, 167, 33, 111, 1445, 1047, 2479, 12623, 22893}},
	{2459, 15, 5371, {1, 1, 3, 1, 3, 1, 47, 185, 329, 903, 1651, 3005, 907, 1255, 8303}},
	{2460, 15, 5382, {1, 3, 5, 13, 19, 31, 5, 233, 265, 769, 1303, 2503, 2229, 14019, 20257}},
	{2461, 15, 5386, {1, 3, 7, 3, 27, 11, 67, 195, 5, 661, 125, 3761, 7211, 16043, 7267}},
	{2462, 15, 5399, {1, 1, 1, 3, 27, 13, 115, 25, 473, 417, 1751, 2223, 2099, 5913, 14273}},
	{2463, 15, 5400, {1, 3, 7, 15, 13, 53, 99, 115, 225, 737, 1621, 539, 4131, 471, 31865}},
	{2464, 15, 5409, {1, 1, 5, 5, 25, 19, 39, 207, 153, 569, 1755, 2477, 3065, 7383, 29919}},
	{2465, 15, 5415, {1, 3, 5, 11, 13, 59, 33, 3, 435, 273, 701, 3819, 7291, 11803, 26111}},
	{2466, 15, 5416, {1, 1, 3, 9, 29, 19, 71, 59, 93, 1019, 887, 83, 4675, 7541, 26821}},
	{2467, 15, 5424, {1, 3, 1, 3, 21, 53, 71, 73, 43, 321, 1581, 1399, 4043, 12995, 16825}},
	{2468, 15, 5436, {1, 3, 7, 15, 3, 13, 37, 11, 93, 873, 1193, 3481, 451, 15869, 17879}},
	{2469, 15, 5454, {1, 3, 1, 11, 31, 19, 101, 57, 129, 753, 853, 463, 6757, 11083, 8667}},
	{2470, 15, 5462, {1, 3, 5, 15, 25, 41, 25, 197, 235, 609, 905, 993, 3233, 1935, 24661}},
	{2471, 15, 5468, {1, 3, 1, 5, 21, 7, 53, 107, 473, 77, 1135, 1045, 4933, 5615, 15931}},
	{2472, 15, 5481, {1, 3, 7, 11, 3, 9, 105, 183, 151, 527, 425, 975, 4073, 913, 2793}},
	{2473, 15, 5505, {1, 1, 7, 13, 19, 61, 81, 9, 413, 851, 1723, 1113, 1453, 8635, 3353}},
	{2474, 15, 5511, {1, 3, 7, 15, 19, 53, 83, 31, 441, 343, 575, 935, 4543, 1303, 12567}},
	{2475, 15, 5518, {1, 1, 1, 5, 29, 19, 119, 75, 3, 591, 845, 649, 1717, 13695, 26905}},
	{2476, 15, 5530, {1, 1, 7, 9, 5, 53, 127, 191, 15, 773, 1433, 2899, 21, 4977, 17839}},
	{2477, 15, 5532, {1, 1, 5, 9, 21, 9, 99, 115, 397, 99, 725, 3835, 973, 1219, 21159}},
	{2478, 15, 5539, {1, 3, 5, 3, 7, 39, 29, 93, 303, 913, 981, 3549, 5225, 10907, 393}},
	{2479, 15, 5553, {1, 3, 3, 11, 9, 25, 105, 101, 1, 867, 389, 2241, 773, 14123, 10015}},
	{2480, 15, 5573, {1, 1, 5, 1, 1, 37, 117, 213, 391, 779, 1851, 1485, 1277, 5607, 819}},
	{2481, 15, 5580, {1, 3, 7, 1, 3, 5, 43, 47, 483, 367, 749, 1693, 4961, 15257, 3775}},
	{2482, 15, 5597, {1, 3, 3, 1, 27, 11, 21, 83, 437, 379, 1041, 393, 5611, 2421, 31739}},
	{2483, 15, 5602, {1, 3, 5, 7, 19, 1, 79, 63, 53, 201, 1159, 2501, 6327, 11317, 9537}},
	{2484, 15, 5608, {1, 3, 5, 13, 9, 37, 61, 217, 427, 913, 1311, 3503, 5473, 10583, 19723}},
	{2485, 15, 5611, {1, 1, 3, 9, 11, 29, 121, 175, 141, 515, 925, 837, 6011, 10419, 32157}},
	{2486, 15, 5613, {1, 3, 5, 9, 27, 57, 97, 175, 365, 367, 1737, 3845, 1257, 12243, 2201}},
	{2487, 15, 5625, {1, 3, 3, 9, 23, 1, 53, 123, 127, 333, 1335, 707, 5747, 6541, 9809}},
	{2488, 15, 5632, {1, 3, 1, 9, 17, 37, 101, 41, 91, 61, 433, 979, 4345, 12351, 10829}},
	{2489, 15, 5635, {1, 3, 3, 13, 3, 21, 15, 49, 257, 99, 1793, 2987, 5233, 11625, 28069}},
	{2490, 15, 5638, {1, 1, 7, 11, 21, 13, 89, 11, 135, 153, 783, 2893, 6815, 12007, 15605}},
	{2491, 15, 5652, {1, 3, 7, 13, 5, 61, 73, 5, 269, 699, 925, 2925, 5919, 5841, 24875}},
	{2492, 15, 5659, {1, 3, 5, 5, 25, 45, 43, 93, 15, 927, 1253, 319, 1173, 14559, 20221}},
	{2493, 15, 5677, {1, 1, 3, 3, 27, 45, 9, 103, 447, 627, 1239, 3869, 2169, 49, 17917}},
	{2494, 15, 5686, {1, 3, 7, 7, 11, 9, 1, 1, 1, 527, 825, 3295, 623, 2095, 10537}},
	{2495, 15, 5689, {1, 3, 3, 11, 21, 11, 59, 165, 33, 743, 1461, 1535, 6393, 1301, 17823}},
	{2496, 15, 5698, {1, 1, 7, 3, 19, 43, 47, 245, 469, 551, 1447, 1963, 169, 1481, 31925}},
	{2497, 15, 5703, {1, 1, 3, 1, 11, 21, 51, 7, 251, 199, 1153, 767, 6417, 3417, 30171}},
	{2498, 15, 5707, {1, 3, 7, 1, 31, 5, 41, 103, 447, 263, 211, 2029, 8021, 4705, 10579}},
	{2499, 15, 5731, {1, 1, 3, 5, 17, 25, 55, 75, 393, 107, 2017, 2389, 1685, 14021, 9161}},
	{2500, 15, 5738, {1, 1, 1, 9, 13, 1, 75, 237, 205, 461, 689, 2531, 2839, 13925, 23351}},
	{2501, 15, 5743, {1, 3, 7, 1, 23, 39, 33, 189, 157, 571, 239, 1053, 1559, 1685, 23059}},
	{2502, 15, 5748, {1, 3, 3, 3, 27, 61, 71, 121, 49, 157, 1341, 1707, 2417, 11689, 26507}},
	{2503, 15, 5758, {1, 3, 7, 7, 19, 63, 47, 53, 95, 791, 1467, 1273, 2045, 755, 8555}},
	{2504, 15, 5762, {1, 1, 3, 15, 27, 33, 21, 253, 317, 153, 1509, 1765, 3809, 601, 5907}},
	{2505, 15, 5768, {1, 3, 5, 15, 11, 17, 97, 91, 165, 199, 1751, 2135, 1315, 3077, 29995}},
	{2506, 15, 5773, {1, 3, 1, 5, 3, 33, 93, 49, 39, 743, 341, 2549, 7603, 3369, 30889}},
	{2507, 15, 5776, {1, 1, 3, 13, 3, 5, 87, 63, 293, 785, 1591, 675, 3915, 2209, 18201}},
	{2508, 15, 5815, {1, 3, 3, 11, 3, 15, 69, 231, 241, 127, 429, 2201, 8173, 12549, 25745}},
	{2509, 15, 5841, {1, 1, 5, 11, 15, 39, 3, 29, 125, 685, 643, 1385, 829, 7347, 28793}},
	{2510, 15, 5847, {1, 1, 7, 15, 27, 15, 59, 237, 299, 773, 1097, 3875, 6503, 7129, 28495}},
	{2511, 15, 5860, {1, 3, 5, 13, 9, 17, 31, 227, 69, 443, 1633, 525, 1659, 14681, 15209}},
	{2512, 15, 5870, {1, 3, 5, 5, 13, 51, 69, 173, 111, 433, 279, 2145, 2091, 9741, 24881}},
	{2513, 15, 5875, {1, 3, 1, 7, 7, 35, 55, 51, 357, 99, 1789, 333, 2073, 10151, 14527}},
	{2514, 15, 5877, {1, 3, 3, 7, 13, 41, 101, 87, 425, 701, 1143, 2733, 6473, 8667, 17419}},
	{2515, 15, 5884, {1, 1, 5, 5, 25, 29, 63, 31, 385, 537, 563, 607, 6723, 9251, 6531}},
	{2516, 15, 5892, {1, 3, 5, 5, 9, 63, 111, 131, 239, 723, 705, 2805, 6579, 12691, 17521}},
	{2517, 15, 5902, {1, 3, 1, 7, 31, 55, 101, 225, 477, 271, 611, 3179, 7859, 9835, 2165}},
	{2518, 15, 5910, {1, 1, 3, 3, 5, 15, 81, 127, 391, 333, 419, 1091, 5997, 12315, 31521}},
	{2519, 15, 5916, {1, 3, 5, 15, 23, 7, 35, 109, 181, 295, 825, 419, 969, 15753, 9365}},
	{2520, 15, 5919, {1, 3, 5, 5, 25, 23, 69, 177, 325, 359, 1577, 619, 6233, 11753, 8103}},
	{2521, 15, 5935, {1, 3, 5, 11, 31, 13, 79, 61, 241, 1011, 1961, 949, 6211, 497, 7099}},
	{2522, 15, 5937, {1, 3, 5, 3, 25, 19, 67, 235, 337, 1015, 1485, 355, 3653, 12735, 14503}},
	{2523, 15, 5944, {1, 3, 5, 7, 31, 23, 35, 231, 147, 15, 263, 1995, 431, 5941, 18931}},
	{2524, 15, 5947, {1, 3, 3, 7, 1, 35, 37, 7, 85, 443, 715, 743, 2189, 12537, 17427}},
	{2525, 15, 5958, {1, 1, 3, 1, 7, 41, 1, 209, 121, 929, 661, 3999, 955, 5123, 31115}},
	{2526, 15, 5962, {1, 1, 3, 5, 11, 43, 127, 125, 107, 293, 273, 2071, 3003, 11631, 7769}},
	{2527, 15, 5969, {1, 1, 1, 13, 13, 29, 39, 217, 111, 779, 1287, 1675, 4201, 4869, 20403}},
	{2528, 15, 5981, {1, 1, 3, 15, 25, 53, 25, 135, 389, 925, 1971, 663, 7545, 2673, 7725}},
	{2529, 15, 5995, {1, 1, 5, 13, 3, 59, 97, 91, 357, 45, 947, 3031, 8095, 6269, 13975}},
	{2530, 15, 5998, {1, 1, 5, 15, 25, 31, 1, 171, 375, 939, 507, 3591, 1089, 13605, 2813}},
	{2531, 15, 6003, {1, 1, 3, 7, 25, 21, 41, 131, 147, 737, 9, 1603, 1859, 11573, 28397}},
	{2532, 15, 6010, {1, 3, 3, 9, 21, 9, 59, 27, 169, 875, 711, 1389, 2899, 7937, 4173}},
	{2533, 15, 6016, {1, 1, 5, 9, 13, 29, 71, 39, 51, 337, 1067, 2661, 1203, 5967, 19249}},
	{2534, 15, 6025, {1, 3, 7, 1, 17, 21, 43, 79, 181, 741, 1901, 3445, 7171, 2109, 1589}},
	{2535, 15, 6031, {1, 1, 3, 9, 23, 37, 105, 51, 227, 775, 1265, 2987, 2197, 13903, 28891}},
	{2536, 15, 6036, {1, 1, 1, 13, 23, 47, 111, 41, 93, 261, 75, 2155, 4301, 11517, 16101}},
	{2537, 15, 6039, {1, 1, 3, 3, 27, 27, 123, 125, 501, 775, 413, 1065, 7607, 15065, 26013}},
	{2538, 15, 6045, {1, 3, 7, 3, 27, 11, 59, 87, 207, 743, 1765, 2969, 913, 8101, 11583}},
	{2539, 15, 6049, {1, 3, 3, 1, 23, 7, 113, 17, 285, 993, 695, 2399, 5019, 4779, 28917}},
	{2540, 15, 6052, {1, 3, 1, 5, 11, 51, 49, 139, 213, 435, 1475, 2209, 6695, 12981, 9851}},
	{2541, 15, 6067, {1, 3, 5, 7, 1, 63, 31, 151, 173, 767, 1453, 1497, 6911, 9597, 25551}},
	{2542, 15, 6074, {1, 1, 7, 7, 21, 53, 39, 159, 389, 231, 309, 359, 7701, 14819, 5175}},
	{2543, 15, 6087, {1, 1, 1, 1, 11, 47, 83, 29, 247, 89, 369, 2727, 3103, 14421, 17369}},
	{2544, 15, 6101, {1, 3, 1, 5, 25, 25, 111, 245, 239, 755, 113, 1765, 3583, 917, 403}},
	{2545, 15, 6121, {1, 3, 3, 3, 5, 59, 85, 151, 463, 591, 743, 3767, 121, 2927, 11031}},
	{2546, 15, 6129, {1, 3, 5, 9, 11, 39, 77, 161, 275, 233, 1991, 2683, 6545, 2423, 32113}},
	{2547, 15, 6142, {1, 3, 5, 11, 5, 57, 13, 229, 329, 757, 1863, 3959, 4243, 7265, 15599}},
	{2548, 15, 6151, {1, 1, 1, 1, 1, 23, 19, 67, 453, 593, 2011, 1813, 4695, 8903, 9623}},
	{2549, 15, 6157, {1, 3, 3, 7, 1, 29, 103, 255, 493, 647, 1709, 4065, 4199, 949, 28829}},
	{2550, 15, 6166, {1, 1, 7, 9, 3, 55, 53, 33, 5, 223, 423, 3347, 7647, 7211, 25157}},
	{2551, 15, 6170, {1, 3, 5, 13, 3, 43, 79, 255, 471, 573, 1007, 2119, 6731, 10047, 23179}},
	{2552, 15, 6175, {1, 1, 1, 3, 7, 39, 55, 61, 53, 377, 435, 401, 3307, 12621, 14299}},
	{2553, 15, 6186, {1, 3, 3, 7, 21, 31, 67, 17, 243, 425, 747, 2995, 1389, 2557, 18415}},
	{2554, 15, 6203, {1, 3, 1, 3, 3, 39, 75, 11, 447, 249, 1135, 1011, 1657, 10767, 19501}},
	{2555, 15, 6217, {1, 3, 1, 11, 17, 51, 117, 129, 17, 143, 785, 103, 5049, 14703, 28479}},
	{2556, 15, 6231, {1, 3, 7, 5, 13, 17, 75, 255, 75, 661, 1175, 477, 1811, 1479, 15783}},
	{2557, 15, 6241, {1, 3, 7, 9, 11, 57, 101, 77, 431, 247, 997, 3657, 5117, 6815, 3841}},
	{2558, 15, 6242, {1, 1, 5, 1, 17, 21, 101, 183, 209, 69, 299, 1585, 6381, 12983, 10053}},
	{2559, 15, 6248, {1, 1, 7, 3, 5, 13, 21, 63, 83, 857, 749, 1251, 5363, 9629, 16563}},
	{2560, 15, 6256, {1, 3, 3, 9, 3, 59, 9, 45, 55, 489, 137, 2423, 2661, 12111, 4375}},
	{2561, 15, 6265, {1, 1, 5, 9, 23, 9, 41, 177, 447, 671, 1631, 3115, 4215, 14435, 8743}},
	{2562, 15, 6275, {1, 3, 7, 11, 19, 23, 15, 221, 413, 783, 1247, 2343, 4397, 3145, 32043}},
	{2563, 15, 6277, {1, 3, 3, 1, 31, 55, 31, 87, 333, 849, 1777, 343, 5199, 1507, 11621}},
	{2564, 15, 6302, {1, 3, 7, 3, 17, 57, 63, 63, 111, 977, 631, 3019, 2953, 14273, 29209}},
	{2565, 15, 6315, {1, 3, 1, 13, 9, 39, 87, 15, 397, 185, 701, 1487, 3807, 13727, 19883}},
	{2566, 15, 6318, {1, 3, 7, 1, 17, 57, 57, 157, 119, 181, 899, 353, 3603, 15041, 7421}},
	{2567, 15, 6330, {1, 1, 7, 3, 29, 13, 29, 191, 105, 373, 961, 1991, 5531, 6793, 29497}},
	{2568, 15, 6343, {1, 3, 3, 11, 7, 61, 65, 39, 215, 187, 191, 1651, 2481, 3951, 24965}},
	{2569, 15, 6347, {1, 1, 7, 5, 25, 11, 105, 23, 257, 771, 1359, 2837, 7821, 12223, 28033}},
	{2570, 15, 6350, {1, 3, 5, 11, 3, 3, 23, 139, 407, 885, 1679, 2979, 8149, 14281, 12487}},
	{2571, 15, 6352, {1, 3, 7, 3, 21, 45, 13, 85, 249, 1015, 2023, 1429, 965, 7091, 31721}},
	{2572, 15, 6371, {1, 1, 1, 13, 19, 5, 119, 47, 91, 285, 211, 2607, 4287, 9197, 455}},
	{2573, 15, 6383, {1, 3, 1, 1, 9, 59, 25, 137, 121, 287, 577, 3325, 2365, 8823, 5033}},
	{2574, 15, 6386, {1, 3, 3, 13, 25, 63, 99, 43, 15, 855, 245, 3189, 59, 5181, 21299}},
	{2575, 15, 6405, {1, 3, 5, 11, 7, 9, 41, 157, 359, 773, 1347, 2049, 4589, 13731, 32133}},
	{2576, 15, 6409, {1, 1, 7, 11, 31, 37, 83, 105, 183, 375, 79, 1821, 1989, 15199, 22207}},
	{2577, 15, 6410, {1, 1, 5, 3, 23, 37, 127, 9, 467, 651, 993, 69, 6943, 4093, 20871}},
	{2578, 15, 6433, {1, 1, 3, 15, 31, 49, 123, 149, 211, 371, 1825, 3011, 485, 1251, 17343}},
	{2579, 15, 6436, {1, 1, 1, 15, 11, 33, 127, 251, 89, 317, 1869, 219, 2275, 14201, 27063}},
	{2580, 15, 6439, {1, 1, 5, 5, 19, 5, 81, 35, 233, 95, 9, 863, 725, 11095, 16217}},
	{2581, 15, 6463, {1, 1, 1, 15, 23, 47, 51, 43, 169, 637, 865, 57, 1509, 1683, 7587}},
	{2582, 15, 6468, {1, 3, 1, 3, 7, 7, 117, 187, 273, 303, 717, 3091, 2083, 3315, 647}},
	{2583, 15, 6477, {1, 1, 5, 15, 13, 27, 23, 227, 145, 547, 1783, 987, 6895, 7135, 11023}},
	{2584, 15, 6496, {1, 1, 5, 11, 21, 39, 57, 203, 477, 17, 985, 1729, 4297, 7483, 13263}},
	{2585, 15, 6511, {1, 3, 7, 9, 3, 49, 71, 45, 143, 967, 39, 583, 2123, 5165, 17437}},
	{2586, 15, 6516, {1, 1, 1, 9, 21, 51, 71, 163, 441, 709, 397, 445, 6167, 7753, 11513}},
	{2587, 15, 6519, {1, 1, 7, 7, 27, 35, 5, 181, 449, 53, 621, 3401, 5263, 4557, 9141}},
	{2588, 15, 6523, {1, 1, 5, 7, 7, 37, 83, 111, 485, 881, 465, 3371, 5603, 371, 29393}},
	{2589, 15, 6530, {1, 3, 1, 15, 7, 47, 41, 245, 377, 823, 309, 3929, 2159, 13917, 13365}},
	{2590, 15, 6539, {1, 3, 7, 7, 7, 29, 25, 141, 19, 611, 79, 2689, 109, 12321, 8345}},
	{2591, 15, 6547, {1, 1, 1, 13, 3, 53, 113, 151, 381, 791, 137, 3185, 3567, 211, 597}},
	{2592, 15, 6589, {1, 1, 3, 9, 7, 53, 87, 89, 491, 861, 467, 3763, 2025, 4187, 9637}},
	{2593, 15, 6592, {1, 1, 7, 1, 27, 33, 71, 41, 63, 1011, 741, 1135, 175, 3739, 21493}},
	{2594, 15, 6601, {1, 3, 3, 5, 9, 19, 55, 175, 325, 55, 1193, 1423, 2049, 9633, 17515}},
	{2595, 15, 6610, {1, 1, 3, 1, 27, 55, 69, 103, 401, 707, 825, 399, 6799, 13199, 6295}},
	{2596, 15, 6616, {1, 3, 7, 3, 19, 63, 25, 151, 17, 159, 1673, 615, 6317, 13261, 26267}},
	{2597, 15, 6619, {1, 3, 7, 9, 27, 1, 77, 129, 423, 647, 707, 2579, 3525, 6723, 31615}},
	{2598, 15, 6626, {1, 3, 3, 7, 7, 31, 35, 241, 309, 369, 895, 3683, 4795, 11319, 451}},
	{2599, 15, 6635, {1, 3, 5, 7, 17, 7, 117, 141, 267, 713, 569, 1915, 4369, 7793, 30853}},
	{2600, 15, 6637, {1, 3, 7, 1, 29, 61, 81, 73, 413, 13, 1977, 3229, 5853, 8451, 15539}},
	{2601, 15, 6638, {1, 3, 7, 1, 5, 45, 109, 21, 431, 487, 2019, 2647, 927, 16015, 10711}},
	{2602, 15, 6652, {1, 3, 1, 3, 11, 19, 37, 183, 451, 377, 269, 3993, 3229, 4899, 26561}},
	{2603, 15, 6656, {1, 3, 1, 11, 5, 19, 121, 55, 57, 117, 687, 83, 3047, 1367, 17595}},
	{2604, 15, 6662, {1, 3, 1, 7, 17, 31, 41, 219, 239, 963, 199, 2895, 5599, 7639, 17201}},
	{2605, 15, 6689, {1, 3, 3, 5, 27, 53, 71, 183, 509, 771, 1809, 1539, 2229, 4893, 17115}},
	{2606, 15, 6699, {1, 1, 3, 9, 9, 9, 13, 49, 265, 643, 1929, 859, 497, 9797, 27771}},
	{2607, 15, 6710, {1, 3, 7, 11, 19, 39, 115, 139, 207, 903, 963, 1849, 4403, 6229, 10021}},
	{2608, 15, 6714, {1, 3, 7, 13, 3, 57, 99, 223, 503, 423, 1755, 807, 1885, 213, 18723}},
	{2609, 15, 6719, {1, 3, 7, 15, 11, 15, 111, 193, 243, 599, 593, 3385, 5393, 15073, 17777}},
	{2610, 15, 6739, {1, 1, 5, 3, 19, 63, 121, 207, 99, 435, 1961, 2747, 6405, 3971, 23481}},
	{2611, 15, 6751, {1, 3, 5, 13, 9, 29, 79, 131, 415, 49, 229, 1003, 3263, 12975, 15987}},
	{2612, 15, 6775, {1, 1, 3, 7, 1, 41, 127, 155, 29, 73, 963, 659, 2741, 3465, 2595}},
	{2613, 15, 6779, {1, 1, 3, 5, 23, 23, 93, 233, 113, 521, 427, 1557, 6917, 12953, 22441}},
	{2614, 15, 6788, {1, 1, 5, 13, 5, 25, 85, 191, 387, 69, 955, 243, 4473, 9813, 21711}},
	{2615, 15, 6798, {1, 3, 3, 7, 1, 53, 95, 65, 231, 995, 539, 2103, 5513, 14087, 28655}},
	{2616, 15, 6815, {1, 3, 5, 3, 17, 13, 19, 227, 197, 91, 1437, 1121, 3307, 6903, 3297}},
	{2617, 15, 6819, {1, 1, 5, 11, 31, 29, 109, 171, 257, 783, 861, 9, 4895, 1859, 10909}},
	{2618, 15, 6825, {1, 1, 7, 13, 5, 47, 61, 5, 363, 351, 1525, 823, 2883, 12435, 17629}},
	{2619, 15, 6826, {1, 1, 5, 11, 9, 3, 69, 159, 371, 477, 1223, 1973, 2757, 413, 31223}},
	{2620, 15, 6836, {1, 1, 3, 5, 23, 45, 43, 195, 423, 829, 1673, 1563, 6633, 14775, 21097}},
	{2621, 15, 6843, {1, 1, 3, 3, 13, 9, 107, 209, 49, 609, 1047, 3691, 7483, 4269, 7557}},
	{2622, 15, 6845, {1, 1, 3, 15, 3, 43, 73, 161, 53, 813, 325, 3439, 7009, 8691, 11711}},
	{2623, 15, 6858, {1, 1, 3, 3, 23, 45, 99, 61, 407, 15, 1515, 1557, 953, 8567, 13729}},
	{2624, 15, 6868, {1, 1, 5, 9, 31, 35, 117, 57, 227, 923, 1373, 1811, 3405, 11979, 10149}},
	{2625, 15, 6877, {1, 1, 3, 9, 15, 53, 105, 209, 153, 67, 1477, 667, 3077, 4911, 3871}},
	{2626, 15, 6881, {1, 1, 3, 3, 21, 53, 93, 101, 183, 1023, 3, 3041, 5815, 9043, 5801}},
	{2627, 15, 6891, {1, 3, 3, 5, 17, 49, 127, 161, 321, 869, 1369, 923, 3235, 711, 30007}},
	{2628, 15, 6896, {1, 1, 3, 3, 15, 17, 97, 229, 389, 159, 1075, 2001, 7905, 15191, 14693}},
	{2629, 15, 6899, {1, 1, 5, 11, 5, 5, 121, 173, 95, 173, 1883, 3915, 1439, 9981, 24375}},
	{2630, 15, 6901, {1, 3, 3, 1, 31, 53, 29, 189, 37, 623, 217, 949, 3959, 7189, 25427}},
	{2631, 15, 6908, {1, 3, 5, 9, 21, 45, 101, 23, 355, 729, 797, 2317, 2931, 7433, 29175}},
	{2632, 15, 6914, {1, 3, 7, 1, 1, 63, 63, 155, 237, 865, 1169, 43, 7335, 6445, 7979}},
	{2633, 15, 6916, {1, 3, 7, 7, 11, 51, 37, 199, 503, 991, 319, 3013, 7885, 12837, 32419}},
	{2634, 15, 6923, {1, 3, 7, 7, 27, 31, 101, 243, 37, 811, 1909, 109, 6455, 7903, 11821}},
	{2635, 15, 6925, {1, 1, 3, 13, 23, 21, 89, 99, 243, 605, 1017, 1871, 1101, 12825, 8227}},
	{2636, 15, 6928, {1, 3, 3, 13, 19, 3, 51, 59, 501, 605, 385, 2189, 3229, 7981, 31407}},
	{2637, 15, 6931, {1, 1, 1, 1, 25, 11, 127, 215, 295, 237, 1245, 3657, 7803, 3897, 655}},
	{2638, 15, 6934, {1, 1, 7, 7, 5, 9, 63, 129, 143, 417, 795, 3409, 2847, 5887, 3093}},
	{2639, 15, 6937, {1, 3, 3, 13, 7, 57, 67, 57, 5, 847, 1185, 3349, 4841, 11457, 8857}},
	{2640, 15, 6938, {1, 1, 3, 3, 9, 53, 51, 43, 85, 437, 13, 2543, 3651, 15493, 767}},
	{2641, 15, 6949, {1, 1, 7, 9, 1, 49, 97, 115, 133, 1011, 1399, 2653, 7765, 13999, 12097}},
	{2642, 15, 6956, {1, 1, 5, 1, 3, 27, 123, 107, 389, 401, 1759, 1333, 1371, 5277, 14865}},
	{2643, 15, 6973, {1, 1, 5, 1, 13, 23, 3, 123, 137, 821, 399, 1671, 3095, 3121, 31387}},
	{2644, 15, 6976, {1, 1, 5, 3, 7, 35, 57, 237, 509, 753, 1783, 2815, 6495, 13283, 7091}},
	{2645, 15, 6981, {1, 1, 7, 11, 5, 37, 77, 109, 7, 969, 1087, 3705, 1695, 14223, 28959}},
	{2646, 15, 6988, {1, 3, 1, 11, 25, 5, 25, 163, 179, 185, 671, 1031, 4537, 11601, 9323}},
	{2647, 15, 6999, {1, 1, 3, 7, 17, 25, 49, 221, 183, 619, 1953, 343, 4523, 14883, 6833}},
	{2648, 15, 7016, {1, 3, 7, 5, 27, 19, 59, 153, 11, 807, 513, 3019, 6875, 5307, 8405}},
	{2649, 15, 7027, {1, 1, 1, 13, 25, 41, 21, 109, 321, 135, 497, 1235, 5177, 5167, 18609}},
	{2650, 15, 7029, {1, 1, 7, 5, 21, 53, 25, 197, 411, 503, 1009, 1921, 4305, 2633, 31415}},
	{2651, 15, 7055, {1, 3, 5, 1, 25, 45, 27, 227, 271, 903, 639, 3805, 657, 8683, 29585}},
	{2652, 15, 7058, {1, 1, 5, 3, 9, 49, 37, 35, 351, 491, 851, 2983, 31, 5619, 6919}},
	{2653, 15, 7074, {1, 1, 5, 3, 11, 49, 33, 153, 393, 1017, 1561, 2795, 4435, 12589, 22349}},
	{2654, 15, 7083, {1, 1, 1, 15, 17, 29, 49, 245, 217, 359, 1133, 393, 3317, 415, 16407}},
	{2655, 15, 7093, {1, 1, 3, 5, 3, 9, 95, 63, 319, 319, 1009, 19, 6453, 16279, 6975}},
	{2656, 15, 7100, {1, 1, 5, 9, 3, 25, 67, 95, 369, 237, 285, 2409, 671, 5143, 121}},
	{2657, 15, 7105, {1, 1, 3, 1, 9, 49, 35, 87, 317, 185, 445, 2263, 7923, 10183, 26615}},
	{2658, 15, 7112, {1, 3, 3, 11, 9, 59, 29, 135, 129, 477, 353, 3571, 1057, 16329, 23523}},
	{2659, 15, 7118, {1, 1, 1, 15, 13, 11, 19, 5, 133, 827, 1799, 1893, 1939, 1101, 12147}},
	{2660, 15, 7120, {1, 1, 3, 3, 15, 49, 33, 185, 511, 1013, 41, 3499, 6235, 7643, 16725}},
	{2661, 15, 7129, {1, 1, 5, 11, 27, 45, 89, 157, 63, 137, 2047, 1275, 4995, 625, 6111}},
	{2662, 15, 7166, {1, 3, 7, 11, 3, 1, 121, 1, 341, 33, 1895, 3033, 3929, 10257, 21037}},
	{2663, 15, 7207, {1, 3, 3, 11, 7, 11, 117, 5, 115, 287, 335, 3415, 5397, 15065, 19121}},
	{2664, 15, 7216, {1, 3, 3, 13, 21, 25, 15, 125, 277, 125, 801, 3761, 2623, 11333, 16867}},
	{2665, 15, 7226, {1, 3, 5, 11, 19, 33, 21, 71, 499, 747, 1515, 185, 1759, 14623, 895}},
	{2666, 15, 7234, {1, 3, 7, 1, 29, 35, 9, 203, 277, 299, 1509, 2017, 2897, 14175, 1643}},
	{2667, 15, 7236, {1, 3, 5, 11, 7, 47, 111, 197, 459, 941, 1619, 2119, 2191, 11049, 6811}},
	{2668, 15, 7246, {1, 1, 5, 9, 7, 43, 103, 115, 87, 269, 1235, 77, 5887, 1611, 29041}},
	{2669, 15, 7248, {1, 1, 5, 7, 1, 61, 83, 225, 179, 81, 1145, 2403, 1485, 8967, 20607}},
	{2670, 15, 7254, {1, 3, 3, 1, 25, 47, 27, 157, 359, 803, 1683, 1995, 6445, 13113, 17899}},
	{2671, 15, 7263, {1, 3, 1, 7, 21, 37, 43, 119, 245, 49, 1581, 2275, 3311, 4087, 29765}},
	{2672, 15, 7273, {1, 1, 3, 13, 5, 33, 49, 191, 455, 105, 665, 3855, 3207, 2671, 32203}},
	{2673, 15, 7274, {1, 3, 1, 1, 25, 63, 19, 217, 17, 353, 947, 1951, 4097, 9041, 11921}},
	{2674, 15, 7293, {1, 3, 1, 7, 21, 31, 113, 97, 347, 993, 1799, 3831, 3711, 6193, 1235}},
	{2675, 15, 7297, {1, 1, 1, 5, 3, 63, 11, 203, 425, 445, 1361, 531, 1265, 1755, 11685}},
	{2676, 15, 7310, {1, 3, 1, 7, 13, 29, 23, 85, 57, 467, 1835, 133, 7961, 4175, 2445}},
	{2677, 15, 7315, {1, 1, 1, 15, 23, 27, 37, 5, 123, 913, 1293, 1633, 3113, 5413, 26407}},
	{2678, 15, 7317, {1, 1, 5, 13, 27, 1, 121, 151, 303, 931, 375, 3679, 1863, 12301, 30907}},
	{2679, 15, 7331, {1, 3, 1, 9, 31, 9, 49, 203, 177, 937, 1503, 933, 5867, 12533, 13621}},
	{2680, 15, 7338, {1, 3, 3, 15, 1, 41, 23, 191, 191, 931, 837, 3553, 2611, 4735, 18105}},
	{2681, 15, 7340, {1, 1, 5, 7, 27, 49, 51, 111, 435, 195, 1229, 711, 7145, 14571, 31707}},
	{2682, 15, 7346, {1, 1, 7, 7, 3, 41, 59, 203, 291, 903, 1727, 2757, 1463, 6287, 31535}},
	{2683, 15, 7355, {1, 1, 7, 13, 23, 5, 75, 3, 207, 525, 411, 2133, 2231, 477, 7155}},
	{2684, 15, 7366, {1, 3, 5, 7, 13, 19, 111, 225, 489, 83, 1177, 4077, 4617, 14413, 7133}},
	{2685, 15, 7383, {1, 3, 1, 7, 9, 59, 3, 113, 379, 803, 1289, 3347, 4127, 6669, 14867}},
	{2686, 15, 7389, {1, 3, 7, 3, 31, 37, 87, 79, 399, 749, 995, 1611, 3137, 12543, 31955}},
	{2687, 15, 7393, {1, 1, 5, 7, 21, 59, 49, 45, 511, 639, 1033, 2169, 3265, 15001, 10745}},
	{2688, 15, 7396, {1, 1, 5, 1, 25, 19, 23, 203, 11, 883, 1031, 4087, 5059, 11321, 21675}},
	{2689, 15, 7400, {1, 3, 7, 5, 11, 27, 33, 205, 163, 289, 501, 3505, 1515, 1895, 15889}},
	{2690, 15, 7414, {1, 3, 1, 1, 23, 7, 39, 239, 29, 119, 1499, 2071, 6495, 12107, 5339}},
	{2691, 15, 7417, {1, 3, 1, 1, 23, 29, 55, 181, 327, 905, 427, 1033, 427, 3687, 5367}},
	{2692, 15, 7426, {1, 3, 3, 7, 21, 27, 115, 127, 393, 855, 1291, 2121, 381, 9995, 29757}},
	{2693, 15, 7432, {1, 3, 5, 1, 25, 13, 15, 183, 269, 1005, 1531, 3451, 3975, 9479, 23695}},
	{2694, 15, 7452, {1, 3, 7, 7, 19, 31, 111, 97, 33, 179, 1343, 2069, 977, 5043, 9129}},
	{2695, 15, 7468, {1, 3, 1, 5, 17, 57, 99, 129, 379, 829, 837, 1845, 3613, 7351, 19291}},
	{2696, 15, 7488, {1, 3, 3, 5, 31, 23, 119, 229, 135, 389, 9, 705, 6697, 15441, 5303}},
	{2697, 15, 7491, {1, 1, 1, 11, 25, 31, 105, 95, 5, 931, 789, 375, 7543, 9957, 28627}},
	{2698, 15, 7494, {1, 1, 7, 15, 21, 17, 19, 103, 389, 545, 1725, 2867, 4251, 3829, 6907}},
	{2699, 15, 7497, {1, 3, 7, 7, 15, 37, 97, 65, 337, 409, 1649, 2869, 7929, 8905, 21989}},
	{2700, 15, 7515, {1, 3, 5, 3, 11, 15, 69, 29, 353, 207, 233, 411, 2047, 10303, 31655}},
	{2701, 15, 7531, {1, 3, 3, 7, 27, 43, 125, 107, 69, 981, 215, 1955, 3589, 597, 12703}},
	{2702, 15, 7552, {1, 1, 7, 9, 25, 13, 109, 73, 227, 663, 1115, 285, 471, 3359, 15787}},
	{2703, 15, 7562, {1, 3, 7, 5, 1, 45, 7, 79, 441, 149, 701, 1457, 6595, 14829, 20865}},
	{2704, 15, 7564, {1, 3, 7, 15, 15, 47, 83, 239, 295, 23, 1085, 813, 1209, 3573, 2855}},
	{2705, 15, 7569, {1, 1, 3, 15, 13, 7, 59, 67, 255, 537, 1841, 3857, 6821, 15175, 13997}},
	{2706, 15, 7582, {1, 3, 1, 1, 9, 57, 59, 21, 21, 41, 1693, 2805, 7953, 1389, 14105}},
	{2707, 15, 7585, {1, 3, 5, 15, 19, 49, 107, 117, 99, 607, 145, 53, 1863, 9383, 12029}},
	{2708, 15, 7588, {1, 3, 3, 13, 1, 39, 5, 141, 503, 265, 281, 1785, 2673, 6597, 6333}},
	{2709, 15, 7592, {1, 1, 5, 3, 3, 19, 3, 181, 169, 269, 955, 2399, 3157, 11053, 8563}},
	{2710, 15, 7597, {1, 3, 3, 13, 11, 1, 95, 43, 179, 507, 443, 209, 3239, 14239, 21829}},
	{2711, 15, 7603, {1, 1, 7, 9, 3, 17, 99, 179, 445, 479, 1897, 1507, 5753, 4757, 2135}},
	{2712, 15, 7610, {1, 3, 3, 1, 9, 51, 29, 13, 295, 291, 927, 85, 5707, 7447, 32319}},
	{2713, 15, 7624, {1, 1, 1, 3, 13, 11, 21, 157, 213, 327, 1071, 591, 2639, 15405, 6617}},
	{2714, 15, 7642, {1, 3, 5, 1, 7, 25, 55, 47, 495, 681, 727, 2707, 2955, 705, 7489}},
	{2715, 15, 7647, {1, 1, 3, 9, 17, 3, 73, 67, 465, 367, 1473, 3195, 7825, 5299, 1817}},
	{2716, 15, 7653, {1, 1, 1, 1, 19, 31, 77, 253, 71, 599, 1601, 871, 2243, 6699, 13013}},
	{2717, 15, 7654, {1, 1, 7, 9, 21, 1, 71, 115, 5, 65, 767, 925, 7901, 10761, 19431}},
	{2718, 15, 7666, {1, 3, 1, 7, 23, 31, 31, 15, 105, 391, 585, 2995, 2635, 10607, 24951}},
	{2719, 15, 7668, {1, 3, 3, 1, 19, 25, 71, 211, 41, 197, 787, 225, 6781, 813, 10117}},
	{2720, 15, 7684, {1, 3, 3, 3, 17, 29, 3, 153, 231, 643, 1151, 447, 3699, 9625, 26677}},
	{2721, 15, 7705, {1, 1, 5, 9, 1, 25, 71, 21, 395, 297, 557, 3841, 233, 1877, 4569}},
	{2722, 15, 7732, {1, 1, 3, 13, 1, 45, 115, 61, 5, 937, 173, 2109, 2927, 9599, 9155}},
	{2723, 15, 7741, {1, 1, 3, 3, 15, 21, 61, 121, 253, 285, 1083, 3545, 5537, 6773, 2629}},
	{2724, 15, 7749, {1, 3, 3, 15, 13, 63, 33, 77, 49, 849, 1795, 2771, 5481, 9833, 603}},
	{2725, 15, 7750, {1, 1, 7, 5, 1, 39, 113, 237, 225, 1005, 1687, 2297, 3213, 2605, 14669}},
	{2726, 15, 7759, {1, 1, 3, 1, 11, 1, 39, 23, 67, 441, 1235, 2545, 3139, 15901, 29243}},
	{2727, 15, 7764, {1, 3, 1, 3, 15, 49, 39, 57, 311, 345, 525, 223, 4923, 6311, 25275}},
	{2728, 15, 7777, {1, 1, 5, 7, 9, 13, 69, 11, 349, 423, 1773, 1055, 1001, 9359, 17025}},
	{2729, 15, 7790, {1, 1, 1, 13, 15, 63, 89, 207, 335, 591, 1223, 2701, 55, 12471, 13127}},
	{2730, 15, 7817, {1, 1, 3, 5, 15, 19, 83, 67, 407, 113, 1961, 779, 5803, 12417, 21751}},
	{2731, 15, 7826, {1, 3, 3, 1, 21, 53, 81, 95, 405, 427, 1047, 2443, 4153, 5843, 22511}},
	{2732, 15, 7831, {1, 1, 7, 7, 7, 25, 115, 155, 453, 537, 741, 2379, 2343, 16035, 19587}},
	{2733, 15, 7859, {1, 3, 3, 11, 27, 21, 111, 121, 503, 437, 803, 3399, 5303, 10163, 18199}},
	{2734, 15, 7871, {1, 1, 5, 13, 19, 27, 7, 81, 259, 545, 965, 743, 4533, 8813, 21253}},
	{2735, 15, 7873, {1, 1, 5, 5, 1, 59, 37, 11, 105, 343, 75, 1319, 6317, 9593, 1699}},
	{2736, 15, 7876, {1, 3, 1, 9, 13, 9, 115, 131, 387, 1023, 253, 693, 5191, 12777, 10565}},
	{2737, 15, 7900, {1, 3, 1, 15, 7, 35, 111, 195, 287, 305, 533, 1901, 3363, 10085, 30791}},
	{2738, 15, 7904, {1, 1, 3, 9, 27, 51, 21, 77, 413, 925, 717, 791, 4147, 585, 5649}},
	{2739, 15, 7913, {1, 3, 3, 5, 25, 59, 79, 249, 185, 567, 71, 1997, 7373, 2327, 18637}},
	{2740, 15, 7916, {1, 3, 3, 11, 15, 21, 97, 99, 391, 57, 1973, 29, 7451, 2529, 25737}},
	{2741, 15, 7922, {1, 3, 7, 5, 7, 59, 93, 5, 287, 469, 1639, 3637, 5465, 14431, 32265}},
	{2742, 15, 7946, {1, 1, 3, 11, 3, 1, 71, 75, 427, 299, 811, 3697, 3529, 5433, 26957}},
	{2743, 15, 7953, {1, 3, 1, 9, 19, 59, 37, 255, 165, 1005, 19, 2851, 4309, 455, 9485}},
	{2744, 15, 7956, {1, 1, 1, 5, 1, 55, 15, 233, 133, 47, 1831, 713, 2601, 1017, 3201}},
	{2745, 15, 7963, {1, 1, 5, 5, 21, 55, 127, 69, 377, 41, 25, 2295, 7595, 4733, 11615}},
	{2746, 15, 7979, {1, 1, 5, 3, 23, 5, 7, 181, 161, 775, 1095, 2271, 6637, 14489, 6873}},
	{2747, 15, 7981, {1, 3, 5, 9, 9, 15, 5, 133, 357, 21, 127, 2685, 6299, 4363, 17573}},
	{2748, 15, 7984, {1, 3, 3, 9, 13, 39, 51, 223, 201, 401, 1839, 2461, 7633, 6039, 10445}},
	{2749, 15, 7989, {1, 1, 5, 1, 9, 21, 19, 249, 227, 359, 255, 2895, 4117, 2073, 27687}},
	{2750, 15, 7999, {1, 1, 5, 15, 5, 61, 113, 161, 95, 3, 877, 2775, 293, 6655, 4023}},
	{2751, 15, 8001, {1, 3, 7, 1, 7, 55, 73, 39, 295, 403, 985, 2315, 1667, 13525, 1453}},
	{2752, 15, 8021, {1, 1, 5, 1, 27, 1, 85, 195, 11, 713, 1841, 3895, 3131, 2193, 17607}},
	{2753, 15, 8056, {1, 3, 5, 13, 25, 1, 119, 97, 239, 167, 1393, 1753, 6989, 12155, 12509}},
	{2754, 15, 8080, {1, 1, 7, 15, 31, 21, 41, 255, 425, 445, 165, 2097, 5627, 4971, 13207}},
	{2755, 15, 8083, {1, 1, 1, 15, 13, 33, 81, 105, 453, 197, 13, 1547, 7381, 8709, 15103}},
	{2756, 15, 8089, {1, 1, 3, 11, 11, 33, 107, 123, 483, 367, 121, 995, 1911, 8205, 22577}},
	{2757, 15, 8090, {1, 1, 1, 9, 9, 43, 71, 49, 273, 431, 1705, 3313, 4259, 16291, 14345}},
	{2758, 15, 8114, {1, 1, 1, 7, 3, 1, 43, 213, 97, 547, 1559, 1149, 2791, 3751, 887}},
	{2759, 15, 8128, {1, 1, 3, 15, 25, 47, 49, 251, 425, 35, 295, 3767, 6305, 9633, 5045}},
	{2760, 15, 8133, {1, 3, 3, 1, 5, 55, 91, 245, 27, 981, 331, 555, 6553, 11017, 15289}},
	{2761, 15, 8145, {1, 1, 3, 7, 1, 23, 23, 155, 223, 565, 1005, 3211, 3847, 7479, 3643}},
	{2762, 15, 8155, {1, 1, 5, 1, 17, 7, 47, 95, 35, 779, 1685, 2099, 7505, 15425, 18089}},
	{2763, 15, 8161, {1, 3, 3, 7, 3, 63, 83, 151, 211, 147, 611, 1171, 1681, 7687, 13423}},
	{2764, 15, 8182, {1, 3, 3, 1, 3, 27, 107, 117, 497, 537, 195, 3075, 2753, 1665, 19399}},
	{2765, 15, 8186, {1, 1, 1, 7, 23, 5, 103, 209, 117, 845, 1243, 1283, 4253, 9723, 20937}},
	{2766, 15, 8191, {1, 3, 1, 1, 5, 49, 7, 13, 419, 125, 287, 1599, 8161, 1275, 24661}},
	{2767, 15, 8192, {1, 3, 3, 3, 13, 63, 23, 183, 39, 979, 1301, 2349, 905, 15805, 30151}},
	{2768, 15, 8195, {1, 1, 3, 9, 17, 11, 97, 189, 189, 511, 1779, 2077, 6891, 11623, 23949}},
	{2769, 15, 8201, {1, 1, 7, 11, 13, 45, 15, 37, 11, 853, 915, 1569, 6103, 10633, 3137}},
	{2770, 15, 8207, {1, 3, 3, 5, 15, 61, 91, 255, 131, 821, 1755, 1501, 2663, 1747, 941}},
	{2771, 15, 8210, {1, 1, 3, 7, 19, 19, 65, 95, 499, 239, 2023, 3185, 4649, 3861, 3767}},
	{2772, 15, 8228, {1, 3, 5, 15, 15, 63, 55, 93, 127, 303, 171, 1763, 4991, 9479, 9917}},
	{2773, 15, 8249, {1, 3, 7, 5, 31, 53, 111, 35, 433, 163, 1903, 3991, 3585, 643, 21941}},
	{2774, 15, 8252, {1, 3, 1, 9, 27, 39, 67, 89, 487, 349, 587, 1723, 4311, 11321, 25785}},
	{2775, 15, 8258, {1, 3, 5, 7, 1, 63, 23, 237, 507, 689, 1341, 441, 1721, 843, 20335}},
	{2776, 15, 8267, {1, 1, 3, 3, 31, 63, 83, 103, 25, 799, 1379, 1817, 3809, 12285, 16673}},
	{2777, 15, 8270, {1, 1, 5, 3, 25, 29, 99, 193, 21, 549, 33, 3109, 4135, 10071, 32355}},
	{2778, 15, 8275, {1, 3, 1, 7, 13, 27, 83, 189, 121, 167, 379, 1503, 7955, 13189, 313}},
	{2779, 15, 8284, {1, 3, 5, 15, 25, 19, 83, 87, 257, 237, 709, 1169, 1561, 7117, 4785}},
	{2780, 15, 8293, {1, 1, 1, 7, 9, 55, 21, 5, 439, 367, 403, 2311, 6243, 8349, 13127}},
	{2781, 15, 8298, {1, 3, 7, 3, 5, 35, 51, 67, 453, 767, 29, 3293, 6665, 11459, 2799}},
	{2782, 15, 8305, {1, 3, 3, 3, 5, 19, 59, 7, 367, 683, 783, 1317, 7119, 6129, 19525}},
	{2783, 15, 8317, {1, 1, 5, 5, 5, 19, 61, 67, 381, 291, 875, 2179, 2481, 9325, 11253}},
	{2784, 15, 8328, {1, 3, 5, 5, 7, 47, 107, 9, 141, 667, 1989, 821, 3909, 1733, 10187}},
	{2785, 15, 8336, {1, 1, 7, 7, 31, 61, 1, 71, 477, 689, 1539, 3617, 8105, 6535, 3293}},
	{2786, 15, 8345, {1, 1, 5, 5, 23, 9, 103, 197, 241, 249, 297, 3607, 6217, 1673, 30103}},
	{2787, 15, 8351, {1, 3, 1, 5, 23, 15, 115, 105, 365, 51, 825, 2687, 359, 16325, 15083}},
	{2788, 15, 8367, {1, 1, 3, 11, 29, 45, 65, 251, 169, 189, 1243, 2345, 1345, 14471, 25631}},
	{2789, 15, 8379, {1, 1, 5, 9, 7, 63, 81, 167, 309, 539, 1169, 3949, 4193, 12047, 1491}},
	{2790, 15, 8381, {1, 3, 1, 9, 29, 33, 89, 167, 67, 73, 1885, 477, 5745, 13365, 6819}},
	{2791, 15, 8382, {1, 3, 7, 9, 9, 49, 95, 13, 157, 997, 1725, 935, 7543, 6349, 18277}},
	{2792, 15, 8393, {1, 1, 5, 5, 11, 59, 97, 17, 303, 469, 93, 2761, 7395, 9021, 24299}},
	{2793, 15, 8402, {1, 1, 7, 3, 27, 63, 71, 99, 407, 139, 711, 2589, 4715, 5405, 3277}},
	{2794, 15, 8414, {1, 3, 7, 3, 11, 15, 49, 57, 271, 493, 1165, 2839, 8191, 2609, 14759}},
	{2795, 15, 8417, {1, 1, 1, 7, 21, 15, 71, 245, 413, 473, 1321, 1165, 1027, 6983, 12867}},
	{2796, 15, 8420, {1, 1, 5, 3, 15, 21, 19, 197, 401, 627, 2047, 2761, 5807, 5751, 28025}},
	{2797, 15, 8429, {1, 1, 3, 3, 5, 57, 19, 209, 341, 165, 489, 455, 231, 14385, 12457}},
	{2798, 15, 8435, {1, 3, 3, 11, 13, 63, 79, 129, 17, 315, 1881, 1069, 177, 12013, 29567}},
	{2799, 15, 8438, {1, 1, 3, 7, 31, 29, 51, 235, 475, 375, 617, 437, 6379, 8505, 23079}},
	{2800, 15, 8450, {1, 1, 3, 7, 27, 3, 3, 137, 203, 959, 363, 371, 2899, 13491, 22979}},
	{2801, 15, 8452, {1, 3, 3, 3, 9, 1, 57, 7, 363, 537, 713, 2417, 509, 7747, 22135}},
	{2802, 15, 8459, {1, 3, 3, 3, 13, 21, 79, 121, 487, 921, 113, 281, 2853, 14855, 19747}},
	{2803, 15, 8470, {1, 1, 1, 11, 3, 53, 89, 123, 307, 585, 567, 1925, 505, 15935, 20419}},
	{2804, 15, 8486, {1, 1, 3, 3, 15, 45, 77, 197, 499, 683, 1405, 3573, 981, 14135, 19763}},
	{2805, 15, 8490, {1, 1, 1, 11, 27, 31, 61, 191, 29, 601, 373, 2011, 6193, 3599, 4387}},
	{2806, 15, 8500, {1, 3, 5, 9, 7, 13, 1, 193, 469, 603, 1315, 3329, 3761, 8355, 10425}},
	{2807, 15, 8524, {1, 1, 3, 9, 29, 61, 103, 17, 117, 251, 2029, 2963, 3763, 16117, 6627}},
	{2808, 15, 8536, {1, 3, 1, 3, 7, 51, 91, 145, 497, 657, 871, 3707, 5905, 10449, 14901}},
	{2809, 15, 8552, {1, 1, 3, 1, 3, 53, 23, 149, 461, 333, 1809, 1315, 1815, 8223, 13297}},
	{2810, 15, 8558, {1, 1, 1, 7, 15, 31, 3, 47, 443, 829, 1305, 893, 4191, 9681, 32661}},
	{2811, 15, 8570, {1, 3, 1, 3, 27, 43, 51, 221, 295, 825, 649, 2953, 6203, 8237, 20253}},
	{2812, 15, 8576, {1, 3, 1, 3, 9, 35, 41, 195, 249, 225, 387, 3789, 1499, 2559, 28413}},
	{2813, 15, 8582, {1, 1, 5, 15, 19, 29, 13, 115, 333, 787, 787, 723, 2987, 6227, 10865}},
	{2814, 15, 8594, {1, 3, 5, 13, 5, 59, 5, 251, 79, 387, 11, 3167, 6619, 13317, 18979}},
	{2815, 15, 8606, {1, 1, 7, 11, 31, 51, 43, 1, 189, 519, 1945, 2129, 4365, 14059, 3139}},
	{2816, 15, 8619, {1, 1, 7, 5, 31, 9, 43, 19, 151, 533, 1061, 3849, 6871, 6941, 14935}},
	{2817, 15, 8621, {1, 3, 7, 5, 19, 57, 7, 129, 25, 353, 17, 1739, 6513, 399, 28835}},
	{2818, 15, 8624, {1, 3, 5, 15, 25, 15, 37, 125, 39, 239, 271, 65, 2189, 10449, 11815}},
	{2819, 15, 8633, {1, 3, 7, 15, 19, 57, 47, 245, 509, 945, 385, 3987, 3585, 14711, 9655}},
	{2820, 15, 8641, {1, 1, 3, 13, 21, 31, 13, 81, 9, 489, 1321, 63, 1363, 2219, 19541}},
	{2821, 15, 8653, {1, 1, 5, 7, 3, 57, 25, 147, 23, 553, 889, 307, 6429, 15807, 12861}},
	{2822, 15, 8654, {1, 1, 3, 15, 29, 21, 99, 237, 151, 881, 675, 3625, 1159, 11759, 21347}},
	{2823, 15, 8662, {1, 1, 7, 1, 9, 13, 111, 239, 235, 609, 1569, 3271, 2837, 13807, 7301}},
	{2824, 15, 8675, {1, 3, 1, 15, 7, 59, 27, 81, 129, 9, 647, 3595, 1877, 1067, 1859}},
	{2825, 15, 8689, {1, 3, 7, 1, 3, 25, 119, 57, 145, 441, 1045, 789, 215, 1265, 9369}},
	{2826, 15, 8695, {1, 3, 7, 3, 17, 25, 87, 211, 441, 229, 223, 2795, 7241, 7007, 20575}},
	{2827, 15, 8702, {1, 1, 3, 1, 13, 1, 55, 227, 389, 141, 1097, 2487, 7603, 4161, 5025}},
	{2828, 15, 8706, {1, 1, 3, 5, 15, 29, 29, 145, 233, 209, 891, 89, 8097, 2897, 26685}},
	{2829, 15, 8720, {1, 1, 3, 1, 29, 53, 19, 95, 161, 359, 435, 3313, 4955, 7965, 21015}},
	{2830, 15, 8729, {1, 3, 5, 9, 19, 3, 109, 77, 29, 937, 1663, 125, 2453, 1069, 20639}},
	{2831, 15, 8739, {1, 3, 7, 13, 5, 23, 43, 231, 347, 591, 1963, 2491, 4045, 16029, 8149}},
	{2832, 15, 8741, {1, 1, 5, 1, 13, 3, 75, 211, 419, 929, 901, 3453, 8121, 799, 8897}},
	{2833, 15, 8751, {1, 1, 7, 15, 11, 11, 123, 111, 309, 415, 1071, 975, 2009, 12945, 19617}},
	{2834, 15, 8759, {1, 1, 1, 7, 31, 35, 81, 255, 89, 643, 451, 513, 497, 11751, 24215}},
	{2835, 15, 8766, {1, 3, 5, 5, 25, 17, 5, 165, 139, 929, 1927, 1353, 7427, 9719, 17087}},
	{2836, 15, 8777, {1, 3, 5, 1, 21, 55, 79, 85, 333, 847, 1305, 851, 5057, 8361, 18269}},
	{2837, 15, 8783, {1, 3, 7, 15, 27, 17, 55, 125, 395, 223, 271, 781, 1639, 10569, 11143}},
	{2838, 15, 8786, {1, 1, 7, 9, 7, 33, 127, 85, 209, 339, 483, 241, 2523, 14951, 6855}},
	{2839, 15, 8795, {1, 1, 3, 9, 5, 19, 9, 183, 435, 343, 1105, 3139, 7617, 1311, 267}},
	{2840, 15, 8802, {1, 1, 5, 1, 15, 53, 11, 63, 113, 241, 855, 3123, 4777, 3495, 23345}},
	{2841, 15, 8814, {1, 3, 1, 5, 19, 29, 119, 205, 167, 683, 289, 1629, 4977, 8981, 6867}},
	{2842, 15, 8821, {1, 3, 1, 1, 31, 63, 95, 159, 267, 231, 863, 3385, 5315, 7267, 13757}},
	{2843, 15, 8828, {1, 3, 5, 11, 19, 21, 53, 41, 125, 179, 533, 1279, 3759, 7073, 13905}},
	{2844, 15, 8831, {1, 3, 5, 9, 17, 7, 27, 67, 97, 809, 1423, 2743, 2859, 16121, 329}},
	{2845, 15, 8837, {1, 3, 1, 15, 1, 41, 59, 155, 509, 51, 1827, 3739, 3879, 13369, 30821}},
	{2846, 15, 8842, {1, 3, 3, 7, 21, 31, 7, 13, 347, 919, 1225, 497, 5051, 3769, 20211}},
	{2847, 15, 8855, {1, 3, 7, 13, 31, 9, 127, 195, 123, 387, 3, 3593, 6623, 9827, 29319}},
	{2848, 15, 8856, {1, 1, 3, 9, 7, 27, 95, 211, 287, 189, 1683, 1999, 7641, 14983, 4699}},
	{2849, 15, 8868, {1, 1, 5, 3, 7, 21, 29, 189, 101, 423, 885, 3275, 6569, 11023, 22265}},
	{2850, 15, 8877, {1, 3, 5, 3, 9, 33, 79, 75, 327, 975, 287, 3025, 2157, 7301, 24447}},
	{2851, 15, 8890, {1, 3, 3, 15, 31, 27, 63, 1, 71, 119, 1151, 517, 6131, 11055, 179}},
	{2852, 15, 8892, {1, 3, 7, 11, 23, 15, 101, 247, 349, 735, 673, 997, 6451, 229, 32103}},
	{2853, 15, 8900, {1, 3, 5, 15, 7, 1, 51, 135, 207, 741, 1831, 1235, 4747, 11915, 22009}},
	{2854, 15, 8909, {1, 3, 1, 13, 9, 31, 19, 221, 465, 681, 627, 2595, 5617, 14201, 30355}},
	{2855, 15, 8912, {1, 1, 3, 1, 13, 49, 55, 155, 11, 885, 1275, 3591, 2217, 6659, 30885}},
	{2856, 15, 8921, {1, 1, 7, 11, 27, 57, 93, 95, 243, 63, 1405, 2049, 7689, 15943, 18503}},
	{2857, 15, 8922, {1, 1, 7, 7, 5, 11, 47, 189, 467, 631, 1665, 2717, 4285, 2087, 1435}},
	{2858, 15, 8927, {1, 1, 3, 11, 7, 27, 127, 3, 231, 757, 435, 2545, 3537, 9127, 19915}},
	{2859, 15, 8943, {1, 1, 5, 13, 5, 29, 85, 127, 339, 875, 497, 1573, 6553, 11983, 18029}},
	{2860, 15, 8948, {1, 3, 1, 1, 21, 3, 15, 91, 231, 683, 1529, 2651, 4147, 13437, 23861}},
	{2861, 15, 8951, {1, 3, 1, 7, 27, 17, 19, 179, 243, 223, 1037, 1501, 5935, 2259, 25185}},
	{2862, 15, 8958, {1, 1, 3, 15, 11, 19, 127, 27, 483, 219, 583, 2555, 531, 3451, 17875}},
	{2863, 15, 8984, {1, 1, 1, 13, 31, 39, 89, 149, 363, 741, 1355, 4067, 3171, 6783, 1799}},
	{2864, 15, 8994, {1, 1, 3, 11, 25, 51, 45, 235, 379, 123, 1701, 725, 1991, 7471, 9833}},
	{2865, 15, 9000, {1, 1, 5, 13, 15, 47, 13, 201, 263, 57, 375, 2963, 7475, 15929, 13775}},
	{2866, 15, 9013, {1, 1, 3, 1, 29, 29, 11, 161, 345, 253, 97, 255, 7267, 2379, 3933}},
	{2867, 15, 9018, {1, 3, 1, 15, 3, 47, 11, 69, 347, 747, 795, 2401, 3367, 2383, 6125}},
	{2868, 15, 9020, {1, 1, 7, 3, 1, 49, 101, 47, 71, 761, 1503, 2619, 191, 8895, 873}},
	{2869, 15, 9031, {1, 3, 3, 5, 25, 41, 93, 85, 427, 109, 1675, 2409, 4317, 9233, 30283}},
	{2870, 15, 9035, {1, 1, 3, 9, 11, 3, 67, 159, 425, 751, 887, 1415, 403, 15977, 10739}},
	{2871, 15, 9045, {1, 1, 5, 13, 9, 1, 9, 103, 481, 601, 931, 1957, 5763, 7095, 27141}},
	{2872, 15, 9052, {1, 1, 3, 15, 29, 13, 43, 33, 297, 269, 1041, 1411, 3461, 12043, 10045}},
	{2873, 15, 9056, {1, 3, 5, 3, 3, 3, 5, 7, 185, 753, 133, 1561, 5595, 13777, 25795}},
	{2874, 15, 9059, {1, 3, 5, 5, 1, 19, 29, 145, 163, 149, 619, 2603, 7757, 10035, 10189}},
	{2875, 15, 9066, {1, 3, 7, 15, 27, 15, 111, 173, 135, 117, 157, 2601, 7919, 12111, 22795}},
	{2876, 15, 9076, {1, 3, 1, 1, 29, 27, 65, 31, 101, 715, 289, 3643, 2335, 6789, 23397}},
	{2877, 15, 9089, {1, 3, 1, 3, 11, 45, 71, 109, 321, 423, 1695, 169, 3075, 12423, 11391}},
	{2878, 15, 9129, {1, 1, 3, 9, 13, 51, 35, 121, 203, 279, 433, 2725, 7951, 2105, 27333}},
	{2879, 15, 9132, {1, 1, 1, 15, 23, 31, 25, 105, 501, 441, 1511, 3133, 2811, 10595, 21779}},
	{2880, 15, 9147, {1, 1, 5, 13, 7, 1, 97, 193, 121, 993, 1347, 1903, 1883, 6583, 24535}},
	{2881, 15, 9164, {1, 1, 7, 9, 7, 29, 17, 41, 101, 447, 1289, 387, 1891, 2723, 26091}},
	{2882, 15, 9167, {1, 1, 3, 3, 3, 53, 81, 81, 177, 165, 195, 3413, 8177, 3817, 8453}},
	{2883, 15, 9185, {1, 3, 7, 15, 15, 31, 23, 31, 337, 439, 1773, 63, 5351, 5491, 1767}},
	{2884, 15, 9195, {1, 3, 1, 11, 5, 15, 23, 75, 437, 553, 429, 2705, 3625, 13851, 19865}},
	{2885, 15, 9197, {1, 3, 3, 9, 13, 15, 33, 235, 215, 415, 1737, 1409, 2101, 14623, 14717}},
	{2886, 15, 9210, {1, 3, 7, 7, 13, 51, 101, 217, 175, 813, 1639, 4009, 1625, 4991, 17525}},
	{2887, 15, 9217, {1, 1, 5, 13, 23, 33, 29, 175, 39, 673, 557, 3239, 5129, 11049, 27227}},
	{2888, 15, 9229, {1, 3, 7, 13, 1, 37, 33, 139, 493, 891, 1883, 2525, 5741, 15795, 5875}},
	{2889, 15, 9248, {1, 3, 1, 15, 15, 27, 127, 111, 147, 363, 725, 3077, 4341, 9131, 24635}},
	{2890, 15, 9254, {1, 1, 7, 3, 17, 25, 59, 135, 177, 635, 73, 3455, 3083, 6009, 13033}},
	{2891, 15, 9263, {1, 1, 1, 5, 15, 53, 93, 161, 215, 459, 1087, 179, 2235, 8885, 15309}},
	{2892, 15, 9266, {1, 1, 7, 13, 7, 17, 75, 173, 449, 855, 103, 2739, 3421, 11811, 18805}},
	{2893, 15, 9268, {1, 1, 7, 9, 5, 11, 53, 75, 247, 249, 1201, 953, 2455, 4589, 6027}},
	{2894, 15, 9290, {1, 1, 5, 13, 27, 51, 119, 39, 137, 11, 1435, 3773, 3889, 6081, 11829}},
	{2895, 15, 9310, {1, 1, 5, 5, 5, 35, 1, 197, 501, 185, 1039, 1563, 6421, 14373, 25655}},
	{2896, 15, 9316, {1, 1, 3, 13, 31, 55, 115, 183, 483, 655, 1351, 3203, 725, 3299, 22579}},
	{2897, 15, 9338, {1, 3, 5, 11, 31, 31, 83, 59, 395, 21, 1881, 2821, 2251, 11781, 26265}},
	{2898, 15, 9340, {1, 3, 7, 13, 21, 19, 103, 21, 403, 443, 1951, 55, 985, 15983, 15087}},
	{2899, 15, 9343, {1, 1, 5, 15, 29, 11, 51, 53, 255, 183, 1475, 1491, 259, 387, 10303}},
	{2900, 15, 9344, {1, 3, 5, 7, 21, 37, 45, 39, 479, 637, 1325, 3753, 3319, 7403, 31759}},
	{2901, 15, 9350, {1, 1, 3, 5, 7, 43, 89, 53, 269, 187, 995, 141, 119, 8139, 29699}},
	{2902, 15, 9354, {1, 1, 1, 5, 1, 53, 3, 23, 379, 223, 1889, 4035, 1437, 12425, 9051}},
	{2903, 15, 9359, {1, 3, 1, 13, 3, 31, 61, 43, 249, 449, 901, 1921, 3495, 8599, 5263}},
	{2904, 15, 9361, {1, 1, 3, 5, 3, 25, 35, 133, 25, 597, 915, 3663, 5147, 11831, 24269}},
	{2905, 15, 9364, {1, 1, 1, 9, 21, 27, 93, 93, 217, 299, 1881, 3647, 4825, 7989, 24121}},
	{2906, 15, 9368, {1, 3, 1, 15, 5, 15, 49, 129, 315, 631, 2037, 1567, 4043, 15589, 30905}},
	{2907, 15, 9371, {1, 3, 3, 7, 25, 5, 123, 51, 47, 471, 1563, 3947, 7975, 3681, 9611}},
	{2908, 15, 9373, {1, 3, 7, 15, 1, 17, 73, 245, 465, 95, 95, 1159, 1319, 4675, 8841}},
	{2909, 15, 9389, {1, 1, 3, 15, 5, 51, 35, 71, 423, 651, 753, 173, 2131, 15799, 29601}},
	{2910, 15, 9390, {1, 1, 1, 1, 3, 53, 83, 187, 445, 827, 1549, 979, 5363, 1701, 2149}},
	{2911, 15, 9409, {1, 1, 7, 9, 3, 15, 65, 161, 37, 233, 771, 3749, 727, 6857, 17175}},
	{2912, 15, 9443, {1, 1, 7, 7, 27, 29, 107, 247, 249, 353, 773, 3677, 7273, 5419, 29397}},
	{2913, 15, 9445, {1, 3, 3, 7, 31, 49, 87, 159, 145, 497, 1715, 2115, 5035, 6431, 7245}},
	{2914, 15, 9446, {1, 3, 3, 5, 7, 31, 51, 117, 101, 617, 557, 2551, 6589, 13295, 31975}},
	{2915, 15, 9452, {1, 1, 3, 3, 15, 27, 125, 163, 169, 893, 1771, 25, 5787, 10267, 10297}},
	{2916, 15, 9490, {1, 1, 1, 5, 9, 47, 85, 65, 289, 783, 1105, 4035, 4111, 2589, 24575}},
	{2917, 15, 9492, {1, 3, 3, 13, 23, 33, 7, 49, 301, 531, 1713, 2755, 5543, 8153, 24099}},
	{2918, 15, 9495, {1, 1, 5, 9, 7, 39, 101, 67, 417, 923, 757, 1537, 5553, 12233, 20881}},
	{2919, 15, 9508, {1, 1, 5, 1, 19, 7, 25, 123, 125, 183, 573, 3317, 6867, 871, 17631}},
	{2920, 15, 9523, {1, 1, 3, 15, 19, 13, 117, 41, 129, 715, 1525, 2257, 2179, 10807, 23271}},
	{2921, 15, 9543, {1, 3, 1, 5, 25, 53, 19, 169, 289, 569, 1135, 1967, 7001, 15883, 15113}},
	{2922, 15, 9558, {1, 3, 7, 15, 7, 37, 127, 147, 415, 313, 1541, 1889, 3763, 16199, 12681}},
	{2923, 15, 9567, {1, 1, 3, 9, 1, 35, 95, 137, 237, 951, 899, 3177, 6073, 10655, 31687}},
	{2924, 15, 9580, {1, 1, 5, 5, 29, 57, 45, 253, 297, 529, 1553, 467, 8035, 15675, 21691}},
	{2925, 15, 9585, {1, 3, 7, 15, 25, 41, 59, 81, 87, 985, 1001, 2369, 661, 7551, 11829}},
	{2926, 15, 9591, {1, 1, 7, 9, 27, 21, 7, 233, 309, 67, 701, 2737, 4261, 2467, 15691}},
	{2927, 15, 9611, {1, 3, 7, 1, 19, 55, 47, 155, 333, 101, 517, 1991, 4619, 10435, 27241}},
	{2928, 15, 9613, {1, 1, 7, 3, 23, 35, 7, 125, 157, 537, 933, 3281, 4975, 8969, 27581}},
	{2929, 15, 9614, {1, 1, 3, 7, 19, 53, 81, 103, 461, 435, 777, 335, 5261, 12249, 9695}},
	{2930, 15, 9621, {1, 3, 1, 7, 19, 9, 75, 245, 355, 37, 1855, 1339, 3107, 7251, 16543}},
	{2931, 15, 9631, {1, 1, 1, 3, 5, 35, 39, 223, 113, 423, 1423, 713, 6113, 349, 24147}},
	{2932, 15, 9642, {1, 3, 1, 1, 15, 31, 11, 75, 499, 345, 1253, 2629, 2551, 7483, 25395}},
	{2933, 15, 9656, {1, 1, 3, 11, 25, 25, 3, 211, 185, 45, 1865, 1805, 3303, 11091, 529}},
	{2934, 15, 9661, {1, 3, 1, 1, 9, 21, 7, 165, 107, 641, 1083, 2805, 2099, 5855, 18477}},
	{2935, 15, 9667, {1, 3, 5, 3, 9, 21, 77, 103, 505, 277, 335, 797, 3869, 2957, 1979}},
	{2936, 15, 9694, {1, 3, 5, 15, 31, 23, 77, 247, 303, 891, 1261, 3233, 3495, 13111, 13185}},
	{2937, 15, 9715, {1, 3, 5, 11, 11, 35, 49, 229, 149, 931, 881, 775, 2949, 3141, 29157}},
	{2938, 15, 9722, {1, 1, 3, 5, 19, 57, 23, 95, 347, 221, 195, 3561, 1481, 2063, 3979}},
	{2939, 15, 9738, {1, 3, 5, 3, 13, 1, 23, 173, 431, 29, 421, 3235, 2751, 4447, 28283}},
	{2940, 15, 9745, {1, 1, 5, 13, 23, 3, 1, 9, 327, 855, 1251, 2997, 6129, 4223, 11555}},
	{2941, 15, 9758, {1, 3, 7, 13, 29, 21, 37, 229, 217, 353, 1239, 3955, 491, 12183, 14777}},
	{2942, 15, 9764, {1, 1, 5, 5, 1, 33, 103, 187, 183, 939, 1873, 2633, 6143, 15405, 17353}},
	{2943, 15, 9782, {1, 1, 1, 9, 21, 27, 71, 129, 499, 279, 1181, 4053, 2485, 1961, 30603}},
	{2944, 15, 9791, {1, 1, 3, 15, 21, 37, 45, 201, 221, 187, 727, 1241, 6171, 1383, 22277}},
	{2945, 15, 9793, {1, 3, 7, 5, 21, 17, 67, 177, 323, 601, 633, 865, 6131, 10329, 8689}},
	{2946, 15, 9794, {1, 3, 5, 9, 15, 45, 71, 43, 359, 651, 103, 403, 3249, 11769, 6567}},
	{2947, 15, 9805, {1, 3, 3, 13, 3, 23, 101, 145, 367, 999, 1489, 3673, 2959, 10855, 16029}},
	{2948, 15, 9808, {1, 3, 7, 3, 13, 43, 123, 87, 55, 1015, 141, 2917, 6567, 16025, 25555}},
	{2949, 15, 9811, {1, 3, 1, 3, 17, 7, 21, 161, 41, 889, 1315, 1897, 639, 15451, 3049}},
	{2950, 15, 9817, {1, 3, 5, 15, 27, 33, 55, 17, 81, 431, 325, 909, 3547, 10121, 17815}},
	{2951, 15, 9824, {1, 1, 3, 1, 15, 37, 43, 137, 203, 191, 1129, 1585, 435, 3177, 769}},
	{2952, 15, 9836, {1, 3, 7, 11, 21, 23, 125, 41, 17, 951, 465, 3691, 3465, 13247, 13779}},
	{2953, 15, 9851, {1, 3, 3, 1, 31, 23, 43, 101, 405, 739, 1061, 2955, 5643, 16137, 8763}},
	{2954, 15, 9853, {1, 1, 5, 1, 19, 33, 99, 109, 203, 65, 395, 2775, 1373, 2557, 5875}},
	{2955, 15, 9854, {1, 3, 3, 3, 27, 51, 79, 63, 331, 365, 1071, 1661, 4549, 8561, 1719}},
	{2956, 15, 9877, {1, 3, 3, 9, 3, 17, 53, 161, 141, 489, 1325, 1709, 1381, 5093, 171}},
	{2957, 15, 9881, {1, 1, 7, 15, 9, 3, 95, 237, 197, 949, 7, 1837, 729, 10111, 6637}},
	{2958, 15, 9923, {1, 1, 3, 3, 19, 31, 57, 173, 483, 861, 1001, 1919, 3389, 11777, 20693}},
	{2959, 15, 9935, {1, 3, 1, 9, 27, 13, 113, 177, 75, 925, 949, 119, 4759, 7775, 23033}},
	{2960, 15, 9937, {1, 1, 7, 15, 23, 15, 65, 61, 137, 653, 1843, 323, 379, 15157, 29885}},
	{2961, 15, 9954, {1, 3, 3, 7, 29, 3, 11, 205, 347, 745, 1477, 3929, 5749, 4735, 29435}},
	{2962, 15, 9959, {1, 3, 5, 9, 1, 11, 111, 15, 7, 69, 45, 3607, 1099, 9203, 21301}},
	{2963, 15, 9963, {1, 3, 3, 3, 23, 3, 83, 173, 73, 485, 681, 1867, 3839, 11823, 13339}},
	{2964, 15, 9968, {1, 1, 3, 11, 31, 43, 107, 127, 465, 389, 1595, 427, 1571, 5885, 29569}},
	{2965, 15, 9973, {1, 1, 7, 9, 27, 25, 117, 27, 287, 391, 279, 3247, 35, 12973, 5483}},
	{2966, 15, 9974, {1, 3, 7, 11, 19, 55, 45, 127, 245, 945, 305, 3907, 2455, 3163, 31}},
	{2967, 15, 9980, {1, 1, 7, 11, 15, 17, 65, 15, 37, 207, 1447, 3027, 2281, 6557, 16717}},
	{2968, 15, 9983, {1, 1, 1, 13, 5, 27, 33, 213, 29, 603, 1171, 3235, 2255, 2017, 30999}},
	{2969, 15, 9985, {1, 3, 1, 5, 11, 1, 73, 233, 69, 125, 397, 297, 3337, 6191, 31055}},
	{2970, 15, 10003, {1, 1, 1, 15, 1, 1, 65, 145, 201, 917, 1891, 2999, 4069, 10413, 15819}},
	{2971, 15, 10010, {1, 3, 5, 13, 15, 51, 115, 167, 311, 375, 1069, 2595, 3337, 753, 11903}},
	{2972, 15, 10034, {1, 1, 3, 1, 1, 23, 69, 125, 147, 915, 1945, 411, 979, 13863, 30443}},
	{2973, 15, 10040, {1, 3, 1, 11, 5, 1, 93, 23, 135, 93, 1689, 23, 3519, 4491, 24673}},
	{2974, 15, 10063, {1, 1, 7, 3, 11, 59, 93, 153, 487, 475, 1191, 1455, 5963, 8259, 18811}},
	{2975, 15, 10077, {1, 1, 3, 1, 13, 15, 55, 71, 433, 33, 491, 1835, 5695, 10509, 347}},
	{2976, 15, 10081, {1, 1, 1, 15, 19, 1, 23, 47, 235, 101, 1057, 901, 5477, 7079, 30885}},
	{2977, 15, 10082, {1, 1, 5, 13, 11, 43, 119, 77, 441, 121, 783, 827, 1757, 12751, 31593}},
	{2978, 15, 10084, {1, 3, 7, 11, 19, 17, 37, 225, 329, 231, 515, 1541, 7371, 6355, 10905}},
	{2979, 15, 10088, {1, 1, 5, 13, 7, 11, 35, 215, 345, 577, 147, 2803, 3291, 4631, 5329}},
	{2980, 15, 10091, {1, 1, 3, 9, 21, 55, 113, 251, 25, 221, 1445, 3385, 1589, 4109, 29897}},
	{2981, 15, 10105, {1, 1, 5, 7, 9, 45, 5, 33, 331, 285, 1101, 3131, 2713, 5653, 3823}},
	{2982, 15, 10111, {1, 3, 7, 7, 5, 39, 43, 167, 481, 629, 777, 1827, 4653, 403, 4781}},
	{2983, 15, 10118, {1, 3, 3, 7, 31, 33, 31, 159, 313, 673, 1425, 663, 5819, 1297, 26627}},
	{2984, 15, 10127, {1, 3, 3, 1, 19, 61, 117, 93, 373, 491, 1031, 757, 4185, 771, 7265}},
	{2985, 15, 10135, {1, 1, 7, 9, 3, 45, 65, 223, 437, 41, 1139, 2733, 5963, 2709, 25429}},
	{2986, 15, 10169, {1, 3, 5, 11, 21, 27, 31, 127, 255, 761, 1865, 1319, 6583, 9235, 10717}},
	{2987, 15, 10172, {1, 1, 1, 5, 21, 1, 63, 43, 413, 557, 567, 2893, 8017, 2307, 29525}},
	{2988, 15, 10183, {1, 1, 7, 3, 31, 1, 15, 235, 215, 395, 1971, 469, 5275, 431, 5349}},
	{2989, 15, 10190, {1, 1, 1, 13, 25, 59, 71, 245, 389, 279, 1293, 89, 6551, 10285, 14495}},
	{2990, 15, 10192, {1, 1, 5, 5, 9, 63, 17, 229, 425, 939, 877, 3689, 7229, 6707, 30771}},
	{2991, 15, 10211, {1, 3, 7, 7, 11, 29, 43, 41, 25, 237, 1585, 3735, 2617, 7541, 26243}},
	{2992, 15, 10218, {1, 1, 7, 9, 21, 5, 69, 231, 361, 39, 1695, 3043, 2973, 5487, 12857}},
	{2993, 15, 10228, {1, 1, 5, 3, 17, 63, 91, 217, 407, 133, 1373, 4021, 1737, 10043, 4561}},
	{2994, 15, 10235, {1, 3, 7, 9, 31, 13, 101, 231, 175, 457, 89, 2167, 2725, 8505, 375}},
	{2995, 15, 10242, {1, 1, 3, 15, 31, 11, 27, 211, 347, 291, 1881, 3091, 3307, 5117, 13341}},
	{2996, 15, 10244, {1, 3, 5, 5, 13, 25, 5, 197, 237, 135, 635, 1175, 5085, 14737, 10807}},
	{2997, 15, 10271, {1, 3, 3, 9, 7, 63, 107, 127, 147, 477, 1813, 2619, 8089, 2651, 26549}},
	{2998, 15, 10278, {1, 1, 5, 11, 15, 45, 27, 133, 45, 621, 707, 2679, 5929, 19, 9599}},
	{2999, 15, 10296, {1, 3, 7, 9, 21, 37, 41, 255, 69, 1009, 1999, 367, 6177, 10017, 3549}},
	{3000, 15, 10299, {1, 1, 1, 15, 19, 55, 73, 189, 423, 983, 1811, 2551, 4765, 12077, 18205}},
	{3001, 15, 10307, {1, 1, 5, 7, 17, 13, 25, 225, 463, 471, 631, 1811, 5797, 3235, 32253}},
	{3002, 15, 10309, {1, 3, 7, 1, 29, 7, 123, 187, 331, 735, 1757, 1115, 2077, 15725, 2183}},
	{3003, 15, 10310, {1, 3, 7, 9, 17, 61, 111, 93, 21, 1003, 1905, 3719, 2111, 11845, 6427}},
	{3004, 15, 10314, {1, 3, 7, 7, 17, 21, 51, 59, 115, 723, 2039, 2833, 5969, 5737, 18407}},
	{3005, 15, 10316, {1, 3, 3, 13, 9, 47, 95, 233, 13, 281, 1049, 619, 405, 16205, 20097}},
	{3006, 15, 10321, {1, 3, 7, 13, 9, 41, 11, 171, 453, 713, 587, 1669, 2489, 10277, 18599}},
	{3007, 15, 10328, {1, 3, 3, 13, 21, 41, 123, 173, 511, 399, 859, 1515, 5773, 12535, 26289}},
	{3008, 15, 10338, {1, 1, 7, 15, 11, 3, 113, 111, 73, 7, 1191, 2573, 7713, 465, 27615}},
	{3009, 15, 10343, {1, 1, 7, 15, 5, 5, 39, 11, 489, 13, 1041, 1639, 7879, 11899, 6899}},
	{3010, 15, 10344, {1, 1, 5, 9, 27, 31, 15, 237, 401, 795, 1675, 2361, 7333, 12507, 14627}},
	{3011, 15, 10347, {1, 3, 1, 7, 21, 53, 31, 81, 189, 683, 1283, 419, 7585, 9207, 15053}},
	{3012, 15, 10352, {1, 3, 5, 11, 21, 1, 49, 251, 403, 489, 1235, 429, 4855, 4081, 17575}},
	{3013, 15, 10364, {1, 3, 1, 15, 29, 33, 77, 53, 105, 731, 749, 2677, 3967, 7967, 18723}},
	{3014, 15, 10373, {1, 3, 3, 11, 9, 47, 11, 95, 137, 923, 599, 1585, 1969, 9625, 19171}},
	{3015, 15, 10386, {1, 1, 1, 5, 7, 7, 85, 49, 339, 883, 261, 2125, 3831, 9797, 16395}},
	{3016, 15, 10391, {1, 3, 3, 3, 5, 9, 33, 99, 75, 889, 101, 2099, 6635, 11511, 21573}},
	{3017, 15, 10398, {1, 1, 5, 11, 1, 11, 79, 49, 7, 131, 471, 1235, 3287, 14777, 12053}},
	{3018, 15, 10408, {1, 1, 5, 15, 9, 9, 83, 15, 21, 899, 1785, 2349, 3471, 6723, 1405}},
	{3019, 15, 10413, {1, 3, 5, 11, 1, 7, 121, 223, 509, 859, 1037, 491, 5529, 481, 17029}},
	{3020, 15, 10422, {1, 1, 7, 5, 17, 35, 91, 171, 113, 65, 423, 2371, 5105, 12827, 31087}},
	{3021, 15, 10445, {1, 1, 3, 3, 21, 47, 55, 11, 159, 51, 263, 2113, 661, 9147, 28929}},
	{3022, 15, 10460, {1, 1, 1, 9, 19, 7, 43, 223, 207, 787, 543, 2141, 4247, 7369, 29031}},
	{3023, 15, 10463, {1, 1, 7, 11, 11, 51, 121, 9, 211, 905, 687, 889, 1827, 13395, 3507}},
	{3024, 15, 10464, {1, 3, 1, 7, 15, 23, 5, 139, 469, 569, 33, 3477, 5391, 13665, 17011}},
	{3025, 15, 10474, {1, 1, 1, 15, 29, 29, 29, 201, 63, 1019, 97, 1671, 9, 4617, 19833}},
	{3026, 15, 10476, {1, 1, 5, 15, 25, 5, 67, 225, 189, 919, 1471, 1451, 5017, 16189, 31555}},
	{3027, 15, 10487, {1, 3, 5, 5, 15, 51, 89, 221, 149, 863, 43, 2381, 1767, 8037, 5319}},
	{3028, 15, 10494, {1, 3, 3, 1, 15, 17, 5, 77, 69, 27, 1883, 63, 5987, 1497, 3723}},
	{3029, 15, 10499, {1, 3, 7, 11, 7, 5, 113, 229, 123, 709, 1531, 641, 6655, 14923, 22947}},
	{3030, 15, 10506, {1, 3, 1, 13, 21, 15, 45, 175, 81, 499, 1113, 587, 7573, 11689, 15651}},
	{3031, 15, 10513, {1, 3, 1, 1, 29, 43, 101, 37, 131, 757, 465, 743, 2737, 8063, 23967}},
	{3032, 15, 10516, {1, 1, 7, 13, 9, 21, 39, 177, 51, 691, 2047, 1519, 6137, 5271, 8703}},
	{3033, 15, 10523, {1, 1, 3, 3, 5, 55, 63, 21, 3, 317, 461, 527, 2673, 16211, 6721}},
	{3034, 15, 10539, {1, 3, 5, 5, 5, 47, 7, 241, 387, 589, 323, 203, 7241, 14761, 13287}},
	{3035, 15, 10549, {1, 3, 5, 3, 23, 63, 55, 61, 231, 1023, 1315, 1181, 243, 7389, 25639}},
	{3036, 15, 10550, {1, 1, 7, 13, 31, 43, 41, 81, 127, 887, 1513, 4055, 1361, 2443, 6963}},
	{3037, 15, 10567, {1, 1, 1, 5, 7, 43, 43, 33, 323, 911, 1373, 3053, 6503, 513, 6457}},
	{3038, 15, 10576, {1, 1, 7, 11, 25, 61, 21, 149, 205, 349, 1433, 1587, 149, 7275, 5465}},
	{3039, 15, 10625, {1, 3, 5, 5, 11, 9, 31, 217, 119, 511, 209, 3325, 2023, 2877, 463}},
	{3040, 15, 10635, {1, 3, 5, 15, 21, 47, 89, 41, 347, 849, 1375, 3311, 807, 11443, 27643}},
	{3041, 15, 10643, {1, 1, 5, 7, 29, 43, 123, 191, 321, 373, 447, 2145, 1221, 2071, 12689}},
	{3042, 15, 10656, {1, 3, 5, 15, 1, 21, 43, 141, 461, 779, 1109, 2915, 909, 8585, 19859}},
	{3043, 15, 10671, {1, 3, 3, 11, 5, 17, 57, 13, 393, 661, 1761, 2455, 43, 8593, 20505}},
	{3044, 15, 10676, {1, 3, 5, 1, 31, 47, 65, 249, 77, 513, 851, 2381, 3447, 693, 7729}},
	{3045, 15, 10683, {1, 3, 5, 15, 31, 19, 83, 47, 369, 697, 1815, 819, 7573, 9245, 8013}},
	{3046, 15, 10685, {1, 3, 5, 5, 11, 25, 27, 151, 107, 339, 299, 3869, 3393, 5661, 2947}},
	{3047, 15, 10688, {1, 1, 3, 1, 1, 59, 85, 57, 175, 465, 239, 3115, 7157, 7035, 11463}},
	{3048, 15, 10697, {1, 1, 7, 5, 31, 41, 53, 149, 121, 743, 189, 159, 5289, 2945, 1179}},
	{3049, 15, 10700, {1, 3, 3, 15, 23, 51, 83, 25, 159, 163, 61, 713, 4529, 5253, 1603}},
	{3050, 15, 10712, {1, 3, 5, 11, 7, 29, 15, 177, 507, 695, 1305, 1863, 7525, 3063, 27433}},
	{3051, 15, 10724, {1, 1, 3, 11, 5, 41, 115, 227, 409, 951, 591, 4003, 7717, 4369, 15637}},
	{3052, 15, 10728, {1, 1, 7, 11, 23, 55, 71, 135, 51, 411, 2003, 2375, 6823, 1711, 4443}},
	{3053, 15, 10734, {1, 3, 1, 3, 31, 47, 31, 233, 243, 3, 313, 1649, 6955, 13679, 32327}},
	{3054, 15, 10739, {1, 1, 3, 11, 29, 9, 1, 79, 247, 677, 685, 3107, 5987, 9675, 29523}},
	{3055, 15, 10762, {1, 1, 1, 7, 25, 31, 39, 241, 483, 839, 1143, 437, 2317, 2437, 173}},
	{3056, 15, 10772, {1, 1, 5, 1, 17, 19, 83, 57, 39, 479, 715, 1911, 1091, 10937, 22145}},
	{3057, 15, 10781, {1, 1, 7, 1, 27, 45, 35, 55, 477, 719, 217, 3349, 7717, 6853, 9699}},
	{3058, 15, 10800, {1, 3, 1, 11, 9, 39, 25, 223, 303, 713, 151, 2611, 4629, 5855, 31729}},
	{3059, 15, 10805, {1, 1, 1, 11, 13, 35, 53, 39, 167, 779, 1673, 1221, 6281, 15113, 32027}},
	{3060, 15, 10827, {1, 1, 5, 9, 19, 63, 89, 113, 199, 107, 1015, 835, 2879, 9499, 25597}},
	{3061, 15, 10830, {1, 1, 7, 3, 19, 37, 15, 23, 449, 641, 1811, 3407, 6775, 6283, 31157}},
	{3062, 15, 10837, {1, 1, 3, 1, 19, 15, 31, 99, 511, 897, 1693, 2093, 955, 15897, 26693}},
	{3063, 15, 10841, {1, 1, 5, 1, 5, 15, 47, 19, 441, 541, 1621, 3877, 6407, 15991, 1931}},
	{3064, 15, 10847, {1, 3, 5, 9, 21, 61, 15, 77, 265, 351, 879, 3835, 6555, 2349, 23235}},
	{3065, 15, 10848, {1, 1, 5, 11, 25, 37, 29, 181, 341, 641, 1213, 1319, 6359, 6231, 32573}},
	{3066, 15, 10857, {1, 1, 1, 7, 1, 37, 87, 123, 33, 913, 111, 2613, 4895, 12595, 26633}},
	{3067, 15, 10866, {1, 3, 5, 3, 27, 11, 45, 89, 183, 241, 1355, 783, 3343, 239, 8643}},
	{3068, 15, 10868, {1, 3, 7, 7, 9, 35, 67, 187, 233, 577, 1445, 3063, 6039, 16233, 1453}},
	{3069, 15, 10872, {1, 1, 3, 13, 27, 11, 23, 15, 95, 63, 1931, 911, 8149, 6833, 3051}},
	{3070, 15, 10887, {1, 3, 3, 5, 29, 49, 125, 117, 47, 143, 423, 3215, 3605, 3677, 17155}},
	{3071, 15, 10899, {1, 3, 1, 1, 31, 1, 123, 195, 83, 893, 1947, 339, 2927, 7183, 15443}},
	{3072, 15, 10901, {1, 1, 7, 13, 31, 15, 91, 207, 39, 275, 439, 2617, 3093, 11041, 24997}},
	{3073, 15, 10915, {1, 1, 5, 3, 3, 41, 13, 67, 361, 497, 25, 3807, 3551, 9681, 21043}},
	{3074, 15, 10924, {1, 3, 3, 3, 11, 27, 103, 59, 427, 327, 1705, 29, 8127, 1641, 20847}},
	{3075, 15, 10929, {1, 3, 7, 5, 3, 37, 81, 137, 225, 101, 187, 3067, 2491, 12687, 16227}},
	{3076, 15, 10942, {1, 3, 5, 15, 15, 33, 69, 223, 225, 771, 1917, 2293, 2889, 12083, 23971}},
	{3077, 15, 10971, {1, 1, 3, 5, 11, 9, 121, 81, 203, 425, 1189, 2011, 3041, 3247, 739}},
	{3078, 15, 10992, {1, 3, 1, 1, 13, 9, 39, 169, 437, 571, 1481, 3355, 3895, 8975, 31031}},
	{3079, 15, 10995, {1, 3, 1, 11, 1, 43, 35, 35, 293, 11, 657, 1415, 5021, 14463, 17945}},
	{3080, 15, 11002, {1, 1, 5, 5, 13, 47, 91, 15, 159, 23, 971, 3575, 757, 13477, 31757}},
	{3081, 15, 11010, {1, 1, 7, 1, 5, 63, 69, 27, 71, 129, 123, 3767, 89, 7865, 1189}},
	{3082, 15, 11027, {1, 3, 3, 5, 23, 1, 83, 3, 487, 491, 217, 2583, 3889, 15009, 9227}},
	{3083, 15, 11029, {1, 3, 5, 15, 25, 1, 73, 107, 245, 191, 1449, 571, 1403, 6953, 17457}},
	{3084, 15, 11045, {1, 3, 3, 3, 27, 19, 25, 105, 207, 857, 1161, 3657, 2107, 7955, 517}},
	{3085, 15, 11057, {1, 3, 3, 9, 21, 29, 5, 103, 219, 35, 3, 1635, 4815, 15797, 29839}},
	{3086, 15, 11070, {1, 1, 7, 7, 3, 63, 75, 77, 13, 57, 603, 2333, 7761, 14397, 10875}},
	{3087, 15, 11092, {1, 3, 7, 13, 3, 11, 5, 255, 1, 947, 1695, 1927, 7447, 7407, 20797}},
	{3088, 15, 11099, {1, 1, 5, 1, 1, 21, 105, 73, 429, 973, 1801, 3943, 6161, 1309, 3359}},
	{3089, 15, 11106, {1, 1, 3, 15, 27, 9, 9, 129, 117, 545, 9, 1983, 6351, 10925, 27337}},
	{3090, 15, 11115, {1, 3, 3, 5, 5, 5, 13, 155, 503, 875, 1243, 2259, 3445, 11953, 6517}},
	{3091, 15, 11120, {1, 1, 7, 3, 29, 21, 121, 147, 413, 423, 1887, 2429, 2765, 16335, 3071}},
	{3092, 15, 11126, {1, 1, 7, 9, 5, 53, 41, 137, 463, 583, 1627, 1731, 6675, 3703, 8177}},
	{3093, 15, 11153, {1, 3, 5, 11, 31, 29, 67, 159, 425, 25, 1457, 139, 5019, 701, 7357}},
	{3094, 15, 11190, {1, 3, 1, 5, 25, 15, 123, 123, 245, 859, 249, 2175, 2137, 5765, 4873}},
	{3095, 15, 11199, {1, 1, 3, 5, 23, 1, 111, 111, 111, 469, 1473, 1777, 3579, 13503, 2569}},
	{3096, 15, 11222, {1, 1, 7, 3, 17, 23, 51, 23, 499, 135, 713, 3317, 807, 9589, 11349}},
	{3097, 15, 11225, {1, 1, 1, 15, 9, 51, 75, 159, 359, 773, 1521, 2913, 5901, 3047, 14649}},
	{3098, 15, 11226, {1, 1, 3, 1, 13, 61, 117, 195, 49, 267, 57, 1769, 3621, 9415, 29443}},
	{3099, 15, 11231, {1, 3, 7, 11, 3, 25, 33, 31, 315, 191, 359, 3399, 2481, 13831, 20205}},
	{3100, 15, 11244, {1, 3, 3, 5, 31, 43, 35, 125, 291, 51, 1469, 3857, 1707, 2641, 32137}},
	{3101, 15, 11259, {1, 3, 5, 1, 25, 11, 113, 137, 211, 159, 1667, 939, 6439, 5337, 32059}},
	{3102, 15, 11261, {1, 3, 3, 11, 31, 61, 99, 49, 383, 343, 395, 51, 6931, 16039, 5901}},
	{3103, 15, 11270, {1, 1, 3, 5, 9, 63, 63, 49, 405, 915, 1505, 2141, 6749, 7799, 17313}},
	{3104, 15, 11273, {1, 3, 7, 11, 15, 11, 49, 161, 155, 869, 121, 301, 6561, 4279, 15233}},
	{3105, 15, 11300, {1, 1, 5, 13, 19, 13, 103, 59, 503, 293, 701, 2527, 5327, 13927, 5025}},
	{3106, 15, 11307, {1, 1, 7, 1, 1, 37, 55, 155, 485, 399, 855, 2677, 5927, 9657, 2795}},
	{3107, 15, 11318, {1, 1, 1, 5, 19, 15, 121, 69, 385, 75, 1567, 2649, 5601, 12981, 15903}},
	{3108, 15, 11332, {1, 1, 1, 11, 19, 21, 45, 59, 505, 737, 15, 1383, 1177, 8375, 15557}},
	{3109, 15, 11335, {1, 1, 7, 13, 29, 19, 123, 127, 469, 773, 733, 3289, 8087, 5803, 27897}},
	{3110, 15, 11341, {1, 3, 3, 11, 19, 55, 101, 67, 485, 939, 607, 1521, 6161, 12235, 16499}},
	{3111, 15, 11347, {1, 3, 5, 13, 29, 31, 31, 9, 453, 151, 1055, 3873, 405, 12877, 29829}},
	{3112, 15, 11354, {1, 3, 5, 1, 17, 1, 17, 131, 107, 1003, 1749, 1849, 6207, 2153, 21275}},
	{3113, 15, 11360, {1, 3, 7, 15, 7, 25, 51, 143, 51, 517, 1841, 1771, 5389, 4633, 11123}},
	{3114, 15, 11369, {1, 3, 7, 11, 23, 7, 89, 95, 403, 361, 835, 585, 2783, 8091, 5141}},
	{3115, 15, 11372, {1, 3, 1, 9, 1, 53, 115, 11, 493, 587, 305, 3605, 1711, 4169, 20013}},
	{3116, 15, 11378, {1, 3, 7, 3, 17, 59, 55, 251, 49, 759, 1227, 3685, 7765, 1445, 20385}},
	{3117, 15, 11396, {1, 1, 5, 7, 29, 55, 19, 157, 129, 927, 893, 1235, 1955, 8153, 2865}},
	{3118, 15, 11405, {1, 3, 1, 15, 21, 35, 81, 53, 175, 939, 1635, 3597, 747, 14011, 12867}},
	{3119, 15, 11417, {1, 3, 7, 1, 27, 61, 91, 73, 405, 677, 603, 2715, 7099, 941, 24523}},
	{3120, 15, 11424, {1, 3, 5, 9, 13, 45, 35, 167, 57, 483, 735, 2777, 7847, 6257, 13109}},
	{3121, 15, 11427, {1, 3, 5, 7, 1, 3, 97, 13, 159, 533, 1791, 1061, 981, 10795, 26165}},
	{3122, 15, 11430, {1, 1, 5, 13, 27, 5, 125, 25, 251, 221, 1909, 197, 6987, 11537, 15287}},
	{3123, 15, 11439, {1, 3, 5, 5, 27, 15, 1, 131, 375, 923, 81, 3153, 6071, 2515, 23729}},
	{3124, 15, 11442, {1, 3, 3, 9, 9, 23, 71, 13, 465, 261, 937, 1549, 5993, 11325, 15065}},
	{3125, 15, 11448, {1, 3, 1, 3, 7, 63, 17, 129, 435, 23, 215, 2251, 1561, 9703, 26483}},
	{3126, 15, 11461, {1, 1, 3, 1, 5, 53, 77, 109, 9, 605, 371, 2081, 6023, 7145, 15837}},
	{3127, 15, 11468, {1, 3, 7, 11, 27, 39, 115, 47, 259, 337, 1857, 3465, 1549, 7747, 8525}},
	{3128, 15, 11471, {1, 3, 7, 7, 29, 29, 75, 77, 29, 661, 899, 3137, 2661, 15271, 28093}},
	{3129, 15, 11473, {1, 1, 1, 3, 3, 3, 11, 219, 39, 757, 1465, 249, 7445, 7013, 15187}},
	{3130, 15, 11476, {1, 3, 7, 15, 15, 1, 39, 245, 427, 1003, 1493, 1913, 6435, 14787, 13481}},
	{3131, 15, 11480, {1, 1, 7, 3, 3, 37, 5, 97, 343, 833, 1379, 1551, 5403, 1843, 5877}},
	{3132, 15, 11489, {1, 3, 1, 1, 3, 17, 17, 163, 339, 691, 1707, 1845, 5941, 4259, 24531}},
	{3133, 15, 11499, {1, 1, 1, 1, 27, 21, 85, 221, 71, 949, 1753, 391, 6349, 15901, 20811}},
	{3134, 15, 11516, {1, 1, 1, 5, 31, 19, 45, 99, 469, 783, 1747, 3807, 5889, 9485, 13715}},
	{3135, 15, 11522, {1, 3, 1, 9, 23, 21, 39, 25, 421, 713, 461, 2857, 5023, 5341, 6409}},
	{3136, 15, 11531, {1, 3, 7, 5, 25, 19, 59, 147, 387, 857, 375, 3103, 1261, 13697, 25675}},
	{3137, 15, 11539, {1, 3, 5, 5, 31, 21, 49, 251, 463, 441, 473, 3487, 3915, 11151, 17721}},
	{3138, 15, 11546, {1, 1, 3, 9, 15, 47, 81, 219, 143, 141, 81, 1705, 5847, 3437, 30521}},
	{3139, 15, 11551, {1, 1, 7, 3, 25, 19, 97, 41, 77, 105, 1337, 695, 7589, 8587, 7509}},
	{3140, 15, 11564, {1, 1, 5, 13, 3, 11, 61, 19, 139, 667, 963, 1567, 5715, 7079, 15967}},
	{3141, 15, 11582, {1, 1, 5, 5, 5, 29, 67, 57, 477, 173, 1163, 727, 823, 15635, 17705}},
	{3142, 15, 11589, {1, 3, 7, 11, 13, 39, 57, 193, 73, 617, 535, 1623, 4581, 4451, 2589}},
	{3143, 15, 11593, {1, 1, 5, 5, 9, 27, 75, 127, 325, 413, 1669, 1749, 8045, 16199, 12237}},
	{3144, 15, 11601, {1, 1, 3, 1, 17, 23, 27, 189, 319, 953, 347, 909, 4401, 12791, 25077}},
	{3145, 15, 11608, {1, 1, 3, 3, 17, 51, 37, 79, 301, 607, 885, 1169, 3275, 3327, 20013}},
	{3146, 15, 11617, {1, 3, 5, 3, 21, 9, 99, 213, 387, 889, 575, 3591, 5377, 2981, 23989}},
	{3147, 15, 11630, {1, 3, 3, 13, 11, 7, 23, 255, 279, 853, 453, 2377, 8123, 15393, 9669}},
	{3148, 15, 11663, {1, 3, 1, 7, 11, 9, 109, 35, 405, 449, 1967, 2943, 3485, 5031, 14273}},
	{3149, 15, 11666, {1, 3, 3, 1, 25, 27, 43, 115, 435, 675, 1937, 1477, 4831, 9417, 7017}},
	{3150, 15, 11668, {1, 1, 7, 13, 19, 45, 83, 241, 487, 215, 1453, 209, 4061, 1765, 15623}},
	{3151, 15, 11677, {1, 1, 7, 7, 21, 31, 95, 9, 287, 1005, 1933, 3405, 6913, 7733, 18975}},
	{3152, 15, 11682, {1, 1, 1, 11, 13, 11, 25, 39, 283, 57, 255, 2809, 5767, 6127, 6705}},
	{3153, 15, 11687, {1, 3, 1, 11, 1, 51, 73, 181, 261, 215, 385, 2777, 5169, 12431, 23563}},
	{3154, 15, 11696, {1, 3, 3, 9, 9, 39, 123, 197, 501, 679, 109, 3369, 4817, 8855, 7997}},
	{3155, 15, 11713, {1, 1, 5, 1, 29, 61, 15, 183, 453, 999, 1211, 3217, 8035, 5153, 19975}},
	{3156, 15, 11728, {1, 3, 7, 11, 11, 21, 51, 45, 379, 793, 289, 309, 1229, 7159, 581}},
	{3157, 15, 11747, {1, 1, 3, 9, 17, 11, 75, 67, 289, 191, 1083, 2949, 6063, 6611, 21595}},
	{3158, 15, 11750, {1, 3, 7, 3, 27, 45, 59, 193, 485, 277, 27, 1219, 2389, 15875, 6273}},
	{3159, 15, 11754, {1, 1, 5, 3, 31, 29, 65, 197, 115, 163, 9, 783, 5573, 2833, 12603}},
	{3160, 15, 11759, {1, 1, 3, 7, 5, 53, 115, 181, 175, 749, 1335, 1151, 2131, 12467, 15811}},
	{3161, 15, 11761, {1, 1, 1, 9, 27, 39, 11, 1, 443, 677, 777, 1857, 7459, 3177, 3875}},
	{3162, 15, 11764, {1, 1, 7, 7, 17, 3, 23, 161, 105, 603, 1991, 3845, 465, 11467, 2077}},
	{3163, 15, 11767, {1, 1, 3, 13, 5, 23, 39, 35, 399, 795, 265, 207, 1823, 15069, 31839}},
	{3164, 15, 11797, {1, 1, 1, 1, 29, 61, 89, 193, 41, 99, 315, 1021, 6109, 12507, 19973}},
	{3165, 15, 11804, {1, 1, 5, 3, 13, 57, 119, 251, 215, 695, 1521, 4081, 2481, 657, 855}},
	{3166, 15, 11808, {1, 1, 7, 3, 25, 5, 3, 133, 111, 385, 773, 1027, 4327, 3031, 3537}},
	{3167, 15, 11831, {1, 3, 7, 5, 5, 27, 43, 117, 479, 83, 1421, 2791, 6551, 6231, 10353}},
	{3168, 15, 11832, {1, 1, 1, 13, 3, 29, 35, 71, 85, 821, 1671, 3057, 797, 13683, 7025}},
	{3169, 15, 11849, {1, 3, 7, 1, 1, 47, 117, 233, 141, 993, 1381, 2551, 1031, 11765, 18429}},
	{3170, 15, 11855, {1, 3, 1, 3, 13, 3, 77, 29, 35, 807, 1109, 695, 5605, 5477, 449}},
	{3171, 15, 11863, {1, 1, 1, 15, 21, 37, 117, 105, 273, 311, 1287, 1415, 1221, 1847, 19487}},
	{3172, 15, 11880, {1, 3, 1, 11, 21, 61, 107, 225, 335, 501, 1995, 2399, 5475, 12613, 18877}},
	{3173, 15, 11883, {1, 3, 3, 1, 31, 41, 27, 205, 103, 837, 639, 2007, 2759, 12471, 1457}},
	{3174, 15, 11885, {1, 1, 7, 13, 29, 39, 71, 245, 105, 235, 1277, 1515, 6129, 15947, 26653}},
	{3175, 15, 11898, {1, 1, 7, 5, 7, 13, 87, 251, 315, 1017, 587, 2917, 5911, 2919, 29715}},
	{3176, 15, 11916, {1, 1, 1, 3, 7, 19, 81, 243, 177, 917, 2023, 2365, 7465, 4901, 29841}},
	{3177, 15, 11924, {1, 3, 5, 15, 1, 31, 15, 147, 285, 1003, 1757, 47, 6925, 1197, 19633}},
	{3178, 15, 11928, {1, 1, 5, 7, 27, 25, 47, 209, 85, 403, 1399, 2331, 3663, 595, 13407}},
	{3179, 15, 11947, {1, 3, 5, 9, 7, 25, 7, 139, 389, 817, 1153, 1421, 5735, 9577, 10269}},
	{3180, 15, 11955, {1, 1, 1, 9, 5, 61, 49, 117, 389, 541, 433, 1405, 2575, 223, 7265}},
	{3181, 15, 11961, {1, 1, 5, 7, 15, 1, 81, 207, 435, 843, 835, 3797, 7637, 5333, 31115}},
	{3182, 15, 11962, {1, 3, 7, 11, 13, 3, 47, 249, 301, 715, 2015, 3049, 8155, 10989, 26051}},
	{3183, 15, 11982, {1, 1, 7, 7, 3, 33, 119, 113, 381, 575, 367, 41, 3317, 11727, 4351}},
	{3184, 15, 11990, {1, 3, 3, 13, 9, 3, 51, 37, 173, 137, 533, 1827, 631, 10047, 6267}},
	{3185, 15, 12010, {1, 3, 3, 11, 17, 39, 61, 245, 13, 139, 1281, 1319, 1233, 13629, 32269}},
	{3186, 15, 12018, {1, 1, 1, 7, 15, 17, 91, 109, 163, 609, 11, 3251, 7653, 14035, 31755}},
	{3187, 15, 12027, {1, 3, 3, 15, 13, 21, 55, 231, 385, 133, 1833, 2637, 6935, 14303, 26745}},
	{3188, 15, 12029, {1, 1, 1, 7, 17, 41, 125, 141, 89, 823, 1411, 3637, 6211, 13323, 6111}},
	{3189, 15, 12035, {1, 1, 1, 11, 1, 21, 9, 43, 97, 685, 1223, 1491, 121, 1793, 2397}},
	{3190, 15, 12055, {1, 3, 5, 5, 17, 17, 5, 223, 129, 865, 1839, 1137, 6391, 4377, 9233}},
	{3191, 15, 12062, {1, 3, 7, 15, 21, 55, 5, 77, 341, 637, 1853, 1435, 1195, 9283, 21257}},
	{3192, 15, 12068, {1, 3, 5, 1, 9, 49, 43, 211, 127, 655, 1397, 1235, 5279, 2351, 24229}},
	{3193, 15, 12071, {1, 3, 5, 3, 25, 29, 13, 229, 25, 675, 837, 2753, 2125, 9863, 11293}},
	{3194, 15, 12072, {1, 3, 5, 7, 23, 43, 127, 1, 163, 237, 337, 3019, 7747, 16227, 2881}},
	{3195, 15, 12086, {1, 3, 5, 5, 25, 9, 43, 171, 421, 521, 1885, 337, 7873, 6347, 13181}},
	{3196, 15, 12097, {1, 3, 5, 5, 7, 47, 107, 173, 163, 191, 625, 3389, 2833, 7945, 24787}},
	{3197, 15, 12098, {1, 1, 7, 3, 27, 57, 27, 209, 253, 815, 301, 1633, 3945, 5051, 28851}},
	{3198, 15, 12100, {1, 3, 7, 9, 9, 51, 103, 213, 437, 189, 1857, 1331, 5551, 10641, 27405}},
	{3199, 15, 12112, {1, 1, 5, 5, 15, 1, 25, 105, 117, 347, 161, 3369, 3589, 12903, 23559}},
	{3200, 15, 12118, {1, 1, 1, 5, 3, 61, 93, 51, 81, 281, 1383, 745, 4137, 2005, 3635}},
	{3201, 15, 12133, {1, 3, 7, 5, 13, 57, 111, 211, 303, 477, 359, 4019, 6779, 5129, 22035}},
	{3202, 15, 12134, {1, 1, 1, 7, 17, 29, 113, 113, 201, 531, 749, 739, 2879, 3315, 18733}},
	{3203, 15, 12137, {1, 3, 7, 13, 21, 55, 21, 183, 359, 75, 377, 2211, 4281, 14317, 28307}},
	{3204, 15, 12161, {1, 3, 7, 1, 21, 1, 49, 213, 317, 75, 113, 1741, 7963, 12785, 11571}},
	{3205, 15, 12162, {1, 3, 7, 9, 11, 31, 29, 101, 261, 141, 715, 2727, 8187, 2075, 32433}},
	{3206, 15, 12171, {1, 3, 7, 3, 23, 9, 17, 143, 385, 211, 593, 241, 6567, 10777, 6677}},
	{3207, 15, 12174, {1, 1, 3, 15, 3, 17, 117, 99, 91, 793, 989, 2421, 5643, 16103, 9759}},
	{3208, 15, 12185, {1, 3, 7, 11, 23, 43, 107, 35, 421, 431, 743, 853, 7939, 12169, 4199}},
	{3209, 15, 12204, {1, 1, 1, 11, 21, 53, 17, 203, 123, 395, 59, 929, 255, 7585, 10945}},
	{3210, 15, 12212, {1, 3, 3, 15, 17, 57, 57, 133, 67, 71, 1685, 903, 4079, 15423, 26495}},
	{3211, 15, 12215, {1, 1, 1, 15, 3, 47, 95, 39, 405, 407, 1557, 3001, 6225, 15187, 5663}},
	{3212, 15, 12216, {1, 3, 5, 5, 13, 47, 33, 61, 375, 1023, 1981, 2773, 2375, 11321, 17731}},
	{3213, 15, 12253, {1, 3, 5, 9, 17, 59, 117, 95, 493, 149, 1269, 2865, 369, 2109, 24601}},
	{3214, 15, 12260, {1, 3, 5, 13, 17, 63, 67, 247, 95, 721, 67, 305, 6179, 15399, 32559}},
	{3215, 15, 12277, {1, 1, 5, 1, 3, 21, 41, 15, 453, 475, 2017, 3193, 5903, 897, 4237}},
	{3216, 15, 12289, {1, 1, 5, 3, 15, 41, 1, 141, 441, 575, 155, 3791, 7711, 11231, 24611}},
	{3217, 15, 12295, {1, 3, 7, 1, 17, 53, 27, 169, 31, 437, 963, 1793, 7777, 1917, 29311}},
	{3218, 15, 12314, {1, 3, 3, 13, 9, 27, 77, 87, 329, 885, 749, 1713, 6013, 6921, 629}},
	{3219, 15, 12323, {1, 3, 5, 13, 3, 7, 53, 27, 353, 267, 925, 2141, 439, 15175, 30851}},
	{3220, 15, 12325, {1, 3, 3, 13, 17, 57, 35, 101, 265, 901, 1825, 2159, 6149, 5967, 24023}},
	{3221, 15, 12335, {1, 1, 5, 11, 13, 51, 99, 111, 193, 415, 1541, 2929, 5045, 3147, 12587}},
	{3222, 15, 12349, {1, 3, 7, 11, 15, 9, 33, 17, 511, 815, 299, 1077, 6171, 10451, 15039}},
	{3223, 15, 12358, {1, 1, 1, 15, 25, 63, 51, 137, 449, 951, 1051, 1101, 4725, 2463, 7355}},
	{3224, 15, 12372, {1, 1, 1, 7, 27, 63, 29, 179, 317, 521, 1459, 827, 6599, 13459, 15439}},
	{3225, 15, 12376, {1, 3, 3, 15, 17, 31, 37, 191, 229, 245, 181, 941, 5761, 1849, 31599}},
	{3226, 15, 12379, {1, 1, 1, 9, 27, 45, 67, 239, 481, 615, 1667, 3751, 8141, 10013, 2125}},
	{3227, 15, 12386, {1, 1, 1, 1, 13, 51, 117, 135, 73, 151, 1291, 2541, 1411, 3767, 26949}},
	{3228, 15, 12395, {1, 3, 1, 9, 7, 11, 21, 187, 243, 857, 1951, 865, 7273, 2041, 8155}},
	{3229, 15, 12416, {1, 1, 3, 3, 19, 33, 89, 115, 455, 137, 707, 1867, 4221, 2433, 9119}},
	{3230, 15, 12421, {1, 1, 3, 11, 5, 3, 121, 1, 71, 951, 603, 3873, 723, 3285, 19289}},
	{3231, 15, 12440, {1, 3, 7, 15, 21, 1, 117, 17, 455, 519, 731, 3003, 5741, 9557, 29163}},
	{3232, 15, 12452, {1, 1, 3, 13, 25, 5, 43, 147, 209, 895, 255, 1231, 241, 487, 15593}},
	{3233, 15, 12455, {1, 1, 3, 13, 7, 1, 89, 187, 217, 927, 2029, 3521, 2777, 8103, 22819}},
	{3234, 15, 12456, {1, 1, 7, 11, 7, 33, 3, 73, 5, 489, 227, 2259, 7031, 6425, 26135}},
	{3235, 15, 12462, {1, 3, 3, 7, 31, 19, 97, 201, 455, 819, 945, 2771, 8083, 8711, 2835}},
	{3236, 15, 12467, {1, 1, 1, 5, 15, 45, 43, 157, 245, 967, 877, 2289, 4499, 9891, 18827}},
	{3237, 15, 12479, {1, 3, 1, 7, 21, 59, 123, 63, 231, 485, 1781, 1211, 4597, 5269, 1607}},
	{3238, 15, 12505, {1, 1, 1, 13, 23, 39, 105, 55, 249, 991, 1625, 3089, 3825, 4275, 29139}},
	{3239, 15, 12521, {1, 3, 3, 1, 29, 29, 55, 169, 13, 895, 1355, 1101, 6063, 12935, 23215}},
	{3240, 15, 12535, {1, 1, 5, 5, 31, 49, 99, 137, 209, 1017, 1179, 3931, 637, 14131, 19285}},
	{3241, 15, 12547, {1, 1, 1, 1, 3, 11, 119, 11, 215, 337, 243, 3883, 3807, 7335, 11901}},
	{3242, 15, 12556, {1, 3, 7, 3, 7, 27, 71, 225, 219, 367, 1213, 2739, 1185, 10175, 21313}},
	{3243, 15, 12561, {1, 3, 7, 13, 7, 49, 23, 223, 61, 1011, 797, 1335, 6711, 5961, 5605}},
	{3244, 15, 12568, {1, 3, 3, 11, 19, 37, 1, 149, 39, 661, 929, 2125, 2299, 5181, 28083}},
	{3245, 15, 12578, {1, 3, 3, 13, 13, 9, 67, 21, 463, 279, 529, 523, 6705, 11011, 31695}},
	{3246, 15, 12583, {1, 3, 1, 5, 13, 1, 123, 3, 291, 625, 1949, 2713, 5917, 10343, 13627}},
	{3247, 15, 12595, {1, 1, 3, 9, 27, 41, 3, 207, 103, 265, 811, 549, 6109, 313, 8889}},
	{3248, 15, 12604, {1, 3, 3, 13, 23, 43, 99, 33, 279, 463, 955, 793, 4113, 10615, 16957}},
	{3249, 15, 12610, {1, 1, 5, 7, 11, 49, 79, 45, 17, 937, 359, 1037, 1099, 3003, 31561}},
	{3250, 15, 12621, {1, 1, 1, 7, 3, 45, 111, 35, 109, 983, 53, 4057, 7349, 3599, 2209}},
	{3251, 15, 12622, {1, 3, 7, 11, 9, 43, 27, 9, 85, 529, 1497, 347, 759, 12449, 11373}},
	{3252, 15, 12624, {1, 1, 3, 9, 17, 1, 49, 31, 367, 813, 1385, 2025, 773, 4679, 4543}},
	{3253, 15, 12629, {1, 1, 5, 15, 15, 9, 43, 97, 239, 995, 1037, 841, 4167, 12113, 23765}},
	{3254, 15, 12630, {1, 3, 5, 9, 29, 53, 123, 49, 221, 113, 1157, 73, 6087, 1363, 11029}},
	{3255, 15, 12639, {1, 3, 1, 13, 3, 15, 69, 199, 279, 919, 5, 161, 4817, 15031, 121}},
	{3256, 15, 12640, {1, 3, 1, 9, 3, 31, 117, 77, 393, 241, 645, 3181, 1067, 15879, 2037}},
	{3257, 15, 12650, {1, 3, 3, 15, 3, 63, 57, 33, 117, 789, 941, 1301, 5865, 12693, 3523}},
	{3258, 15, 12679, {1, 1, 5, 13, 3, 61, 51, 151, 175, 305, 95, 1557, 6567, 7841, 13903}},
	{3259, 15, 12680, {1, 3, 3, 5, 15, 25, 127, 79, 245, 767, 645, 3933, 1357, 12579, 4067}},
	{3260, 15, 12698, {1, 3, 5, 11, 21, 31, 13, 251, 127, 231, 1795, 2627, 1191, 3363, 23543}},
	{3261, 15, 12716, {1, 1, 3, 5, 7, 49, 121, 57, 131, 481, 1879, 525, 5225, 337, 1957}},
	{3262, 15, 12721, {1, 1, 5, 13, 9, 55, 27, 37, 211, 125, 119, 3373, 251, 12357, 13975}},
	{3263, 15, 12722, {1, 3, 3, 15, 1, 51, 91, 119, 233, 993, 203, 1635, 1167, 6327, 29119}},
	{3264, 15, 12731, {1, 1, 7, 1, 13, 5, 23, 253, 121, 989, 1105, 3321, 3221, 6073, 21185}},
	{3265, 15, 12742, {1, 1, 3, 15, 13, 49, 121, 247, 247, 133, 485, 1067, 7875, 411, 7647}},
	{3266, 15, 12745, {1, 3, 7, 13, 31, 37, 127, 241, 145, 133, 53, 267, 2029, 3703, 16123}},
	{3267, 15, 12751, {1, 3, 1, 15, 15, 9, 15, 89, 35, 367, 401, 61, 1953, 7873, 17861}},
	{3268, 15, 12759, {1, 1, 1, 1, 1, 41, 71, 249, 213, 779, 1385, 1767, 999, 15151, 16647}},
	{3269, 15, 12763, {1, 3, 7, 13, 31, 23, 123, 235, 343, 949, 309, 3777, 3587, 853, 19779}},
	{3270, 15, 12769, {1, 1, 3, 13, 29, 35, 5, 37, 63, 757, 303, 1579, 3443, 243, 11873}},
	{3271, 15, 12781, {1, 3, 1, 9, 19, 49, 81, 53, 11, 901, 1857, 147, 3103, 14019, 21}},
	{3272, 15, 12793, {1, 3, 7, 13, 3, 39, 99, 99, 45, 91, 1567, 551, 3129, 4809, 29057}},
	{3273, 15, 12799, {1, 3, 7, 3, 3, 27, 17, 231, 377, 381, 1479, 2525, 2595, 2799, 25737}},
	{3274, 15, 12815, {1, 3, 5, 15, 15, 25, 103, 215, 301, 59, 1417, 981, 7579, 12731, 22329}},
	{3275, 15, 12824, {1, 1, 1, 13, 5, 31, 61, 31, 349, 925, 1301, 685, 435, 11567, 10715}},
	{3276, 15, 12836, {1, 1, 7, 9, 19, 57, 109, 1, 37, 377, 1015, 2273, 6741, 3191, 15949}},
	{3277, 15, 12845, {1, 3, 3, 13, 3, 23, 103, 127, 11, 59, 1847, 1175, 425, 3423, 20643}},
	{3278, 15, 12853, {1, 3, 3, 7, 3, 11, 105, 141, 55, 217, 1427, 477, 667, 9403, 11905}},
	{3279, 15, 12854, {1, 3, 3, 5, 3, 27, 11, 187, 495, 907, 1925, 445, 6639, 8159, 15225}},
	{3280, 15, 12857, {1, 3, 1, 5, 27, 31, 77, 213, 73, 343, 1123, 3609, 2431, 15329, 32165}},
	{3281, 15, 12866, {1, 1, 7, 5, 1, 11, 105, 139, 485, 1007, 709, 3509, 5231, 11717, 31433}},
	{3282, 15, 12872, {1, 1, 3, 15, 23, 49, 95, 169, 399, 1019, 19, 2013, 5311, 7951, 22609}},
	{3283, 15, 12875, {1, 3, 1, 7, 13, 3, 29, 203, 209, 701, 1791, 2615, 5351, 4237, 12565}},
	{3284, 15, 12878, {1, 3, 1, 15, 27, 11, 91, 31, 205, 205, 1683, 901, 5129, 6049, 11865}},
	{3285, 15, 12880, {1, 1, 7, 7, 27, 59, 21, 3, 209, 79, 769, 4013, 2041, 2645, 11561}},
	{3286, 15, 12885, {1, 3, 7, 11, 5, 45, 39, 243, 185, 871, 795, 1845, 8043, 6285, 20991}},
	{3287, 15, 12901, {1, 1, 5, 7, 13, 7, 15, 165, 349, 179, 789, 1269, 3787, 5429, 26567}},
	{3288, 15, 12902, {1, 3, 3, 13, 31, 23, 75, 41, 177, 735, 1889, 4039, 3519, 15003, 965}},
	{3289, 15, 12920, {1, 3, 1, 7, 15, 57, 15, 139, 27, 469, 1003, 691, 7893, 9643, 30983}},
	{3290, 15, 12926, {1, 3, 1, 13, 23, 27, 3, 237, 233, 875, 99, 883, 6167, 5463, 6245}},
	{3291, 15, 12929, {1, 1, 5, 13, 25, 57, 79, 51, 147, 619, 1147, 279, 6583, 1939, 477}},
	{3292, 15, 12939, {1, 3, 5, 5, 31, 61, 125, 163, 213, 699, 583, 3865, 615, 9707, 11651}},
	{3293, 15, 12941, {1, 1, 5, 1, 5, 21, 93, 239, 31, 641, 777, 27, 5247, 8993, 21053}},
	{3294, 15, 12950, {1, 3, 7, 9, 1, 13, 61, 57, 503, 453, 83, 3271, 2845, 1121, 18639}},
	{3295, 15, 12953, {1, 1, 7, 5, 29, 53, 13, 219, 379, 441, 821, 3179, 4877, 2535, 7557}},
	{3296, 15, 12992, {1, 1, 7, 13, 9, 53, 17, 183, 265, 393, 587, 2753, 6453, 7135, 24737}},
	{3297, 15, 13002, {1, 1, 1, 13, 11, 23, 73, 109, 393, 1013, 559, 755, 7291, 6631, 26509}},
	{3298, 15, 13010, {1, 3, 1, 5, 5, 15, 107, 103, 355, 307, 1559, 837, 5413, 5285, 17489}},
	{3299, 15, 13058, {1, 1, 5, 7, 17, 21, 21, 23, 109, 709, 1947, 3585, 3629, 4669, 949}},
	{3300, 15, 13072, {1, 3, 7, 1, 9, 33, 85, 147, 467, 259, 1913, 199, 7399, 9551, 22387}},
	{3301, 15, 13084, {1, 3, 5, 11, 15, 53, 23, 41, 249, 515, 1161, 2467, 1299, 7449, 2613}},
	{3302, 15, 13087, {1, 1, 5, 5, 5, 29, 91, 139, 487, 545, 321, 3035, 4545, 6747, 21673}},
	{3303, 15, 13091, {1, 1, 3, 13, 23, 49, 95, 103, 25, 119, 469, 2515, 2551, 841, 25089}},
	{3304, 15, 13097, {1, 1, 5, 7, 11, 31, 31, 197, 245, 715, 257, 4043, 8099, 11531, 5617}},
	{3305, 15, 13108, {1, 1, 3, 3, 19, 7, 9, 179, 103, 995, 191, 179, 3843, 5215, 27639}},
	{3306, 15, 13123, {1, 3, 1, 7, 23, 59, 25, 65, 399, 211, 1453, 3511, 7203, 16015, 32197}},
	{3307, 15, 13149, {1, 3, 3, 5, 9, 35, 109, 67, 197, 449, 643, 519, 5751, 15551, 11331}},
	{3308, 15, 13150, {1, 3, 5, 3, 1, 17, 53, 201, 265, 351, 467, 911, 1117, 7183, 20371}},
	{3309, 15, 13163, {1, 1, 7, 7, 27, 17, 93, 81, 227, 619, 1191, 735, 815, 12615, 2719}},
	{3310, 15, 13166, {1, 3, 1, 15, 19, 3, 83, 75, 343, 297, 1019, 3469, 4383, 13299, 29755}},
	{3311, 15, 13178, {1, 1, 5, 3, 13, 55, 119, 169, 85, 595, 299, 2469, 5625, 2877, 16117}},
	{3312, 15, 13180, {1, 1, 3, 5, 15, 17, 61, 161, 47, 393, 143, 867, 5517, 9495, 12795}},
	{3313, 15, 13184, {1, 3, 5, 1, 27, 31, 113, 125, 251, 687, 969, 1473, 2245, 6355, 13655}},
	{3314, 15, 13204, {1, 1, 1, 5, 5, 37, 29, 133, 443, 899, 277, 2353, 7223, 4459, 19159}},
	{3315, 15, 13238, {1, 1, 3, 9, 19, 27, 53, 145, 195, 167, 2045, 447, 1803, 1895, 8431}},
	{3316, 15, 13242, {1, 1, 3, 9, 5, 27, 91, 147, 233, 451, 475, 27, 4629, 16181, 16437}},
	{3317, 15, 13249, {1, 3, 5, 3, 29, 17, 53, 167, 433, 689, 1131, 2985, 1553, 11697, 6993}},
	{3318, 15, 13255, {1, 3, 3, 13, 21, 43, 69, 229, 399, 525, 179, 237, 7017, 5703, 17653}},
	{3319, 15, 13269, {1, 1, 3, 15, 13, 39, 75, 163, 229, 875, 197, 477, 3667, 15501, 15801}},
	{3320, 15, 13270, {1, 1, 7, 15, 15, 51, 81, 187, 487, 673, 865, 1915, 1009, 5935, 8097}},
	{3321, 15, 13274, {1, 3, 5, 5, 7, 3, 63, 77, 495, 815, 391, 2321, 1007, 15661, 30715}},
	{3322, 15, 13285, {1, 1, 7, 3, 17, 25, 83, 173, 173, 911, 1373, 2957, 4549, 15977, 17695}},
	{3323, 15, 13289, {1, 1, 7, 13, 13, 23, 77, 147, 497, 1003, 1687, 1795, 1393, 1881, 8479}},
	{3324, 15, 13298, {1, 3, 7, 11, 27, 43, 97, 25, 61, 457, 203, 2573, 5943, 15021, 4003}},
	{3325, 15, 13307, {1, 3, 3, 13, 9, 37, 37, 25, 219, 889, 1535, 2791, 4531, 13679, 12663}},
	{3326, 15, 13312, {1, 1, 3, 1, 17, 7, 51, 123, 89, 887, 1467, 1645, 3767, 6383, 30837}},
	{3327, 15, 13335, {1, 3, 3, 1, 21, 47, 5, 151, 83, 257, 569, 2711, 637, 12569, 16893}},
	{3328, 15, 13345, {1, 3, 7, 1, 31, 37, 73, 3, 115, 919, 1817, 2483, 4811, 15245, 4375}},
	{3329, 15, 13357, {1, 1, 1, 5, 1, 39, 39, 231, 9, 733, 455, 3383, 4777, 7235, 12631}},
	{3330, 15, 13366, {1, 1, 7, 9, 13, 25, 55, 25, 73, 59, 1699, 929, 755, 1279, 5583}},
	{3331, 15, 13372, {1, 3, 5, 3, 9, 49, 79, 55, 479, 179, 1159, 4079, 3503, 11603, 12361}},
	{3332, 15, 13380, {1, 1, 5, 9, 21, 45, 31, 163, 377, 817, 219, 147, 2581, 12769, 30783}},
	{3333, 15, 13384, {1, 3, 1, 7, 15, 27, 39, 189, 493, 259, 1663, 1213, 961, 11089, 16079}},
	{3334, 15, 13395, {1, 1, 5, 9, 5, 41, 13, 153, 313, 337, 1027, 1267, 4249, 13071, 27043}},
	{3335, 15, 13407, {1, 3, 7, 3, 13, 11, 23, 255, 51, 527, 317, 3217, 5037, 12723, 17411}},
	{3336, 15, 13408, {1, 1, 5, 1, 25, 57, 83, 97, 233, 513, 1283, 2675, 4111, 4111, 32141}},
	{3337, 15, 13413, {1, 3, 3, 15, 25, 33, 103, 81, 155, 189, 139, 1179, 2691, 15119, 13959}},
	{3338, 15, 13414, {1, 3, 3, 1, 25, 55, 67, 19, 19, 9, 437, 579, 4273, 10733, 7125}},
	{3339, 15, 13417, {1, 1, 1, 7, 23, 41, 47, 5, 435, 749, 595, 199, 3941, 7199, 4795}},
	{3340, 15, 13437, {1, 3, 1, 15, 5, 49, 35, 9, 199, 703, 1769, 3269, 5689, 13063, 22771}},
	{3341, 15, 13441, {1, 1, 5, 5, 21, 55, 125, 55, 63, 149, 1167, 3577, 1051, 3921, 20545}},
	{3342, 15, 13447, {1, 3, 7, 13, 29, 53, 107, 193, 163, 339, 1335, 1573, 5809, 5681, 29487}},
	{3343, 15, 13456, {1, 1, 1, 9, 17, 9, 91, 177, 211, 453, 1807, 1881, 6051, 225, 6021}},
	{3344, 15, 13459, {1, 1, 1, 13, 15, 1, 29, 43, 181, 105, 1945, 2313, 6429, 2901, 6221}},
	{3345, 15, 13461, {1, 3, 5, 11, 29, 55, 115, 115, 187, 1013, 697, 1885, 121, 12387, 32443}},
	{3346, 15, 13466, {1, 1, 1, 7, 19, 51, 21, 107, 55, 125, 1655, 2281, 3293, 15749, 27521}},
	{3347, 15, 13484, {1, 1, 7, 9, 19, 9, 81, 93, 139, 401, 193, 73, 5159, 9323, 6019}},
	{3348, 15, 13487, {1, 1, 7, 9, 15, 51, 115, 69, 247, 599, 1163, 2251, 1211, 8827, 15581}},
	{3349, 15, 13489, {1, 1, 7, 9, 5, 39, 75, 185, 321, 911, 849, 843, 6791, 10407, 10513}},
	{3350, 15, 13492, {1, 1, 5, 5, 15, 9, 21, 47, 459, 681, 2001, 1545, 5939, 7073, 29043}},
	{3351, 15, 13496, {1, 3, 1, 11, 13, 25, 53, 97, 487, 797, 567, 3757, 5597, 6313, 18531}},
	{3352, 15, 13510, {1, 1, 3, 3, 29, 55, 11, 219, 325, 591, 2015, 383, 2595, 11855, 22501}},
	{3353, 15, 13531, {1, 1, 1, 5, 15, 57, 33, 125, 323, 749, 1843, 4019, 2075, 6673, 6957}},
	{3354, 15, 13538, {1, 1, 5, 7, 19, 7, 47, 239, 51, 107, 1081, 467, 5493, 7617, 10355}},
	{3355, 15, 13543, {1, 3, 1, 1, 11, 3, 67, 199, 175, 421, 935, 309, 4449, 6363, 9183}},
	{3356, 15, 13547, {1, 1, 1, 7, 9, 33, 3, 219, 481, 513, 417, 1267, 2863, 765, 18431}},
	{3357, 15, 13572, {1, 3, 1, 1, 19, 1, 89, 109, 415, 105, 487, 3241, 7465, 9233, 16307}},
	{3358, 15, 13581, {1, 1, 3, 13, 9, 43, 25, 231, 383, 789, 1855, 691, 3465, 2387, 11715}},
	{3359, 15, 13590, {1, 3, 3, 3, 13, 39, 63, 107, 33, 265, 437, 117, 3179, 5543, 28179}},
	{3360, 15, 13605, {1, 3, 3, 13, 21, 5, 31, 111, 321, 425, 253, 3501, 3209, 15429, 18383}},
	{3361, 15, 13612, {1, 3, 5, 9, 1, 27, 117, 187, 433, 459, 1999, 1069, 4857, 8591, 26343}},
	{3362, 15, 13624, {1, 1, 7, 3, 15, 43, 11, 193, 391, 341, 1203, 1259, 7265, 1783, 13161}},
	{3363, 15, 13630, {1, 1, 7, 1, 5, 15, 45, 143, 193, 985, 1105, 3483, 2421, 9687, 22347}},
	{3364, 15, 13632, {1, 3, 7, 13, 21, 17, 79, 231, 487, 663, 1101, 1025, 5779, 14681, 29181}},
	{3365, 15, 13638, {1, 1, 3, 9, 15, 19, 55, 219, 27, 963, 1513, 1017, 3907, 12279, 32655}},
	{3366, 15, 13661, {1, 3, 7, 3, 31, 27, 17, 1, 51, 861, 529, 1225, 6395, 15787, 5231}},
	{3367, 15, 13665, {1, 3, 3, 11, 27, 7, 101, 143, 21, 191, 1437, 2393, 4097, 14319, 6977}},
	{3368, 15, 13668, {1, 3, 3, 3, 25, 35, 105, 141, 433, 269, 1469, 2939, 5387, 7373, 7863}},
	{3369, 15, 13686, {1, 3, 7, 5, 5, 21, 23, 11, 217, 357, 1847, 101, 1161, 5297, 14879}},
	{3370, 15, 13699, {1, 3, 1, 3, 25, 23, 81, 217, 505, 161, 1667, 1343, 1751, 2463, 26431}},
	{3371, 15, 13701, {1, 1, 3, 1, 17, 51, 125, 205, 385, 351, 731, 2911, 2749, 2689, 27031}},
	{3372, 15, 13708, {1, 1, 5, 5, 5, 17, 31, 171, 477, 671, 167, 1797, 8047, 10123, 4325}},
	{3373, 15, 13716, {1, 1, 7, 1, 11, 23, 123, 161, 99, 1007, 765, 1965, 5395, 16193, 17751}},
	{3374, 15, 13725, {1, 3, 1, 9, 13, 11, 111, 217, 31, 753, 377, 2267, 7893, 7195, 24999}},
	{3375, 15, 13730, {1, 3, 1, 9, 21, 53, 127, 121, 151, 395, 1447, 1411, 5179, 12043, 27607}},
	{3376, 15, 13742, {1, 1, 5, 3, 11, 37, 97, 139, 113, 835, 229, 3741, 827, 5527, 5779}},
	{3377, 15, 13747, {1, 1, 7, 7, 27, 55, 11, 55, 429, 269, 1179, 233, 1053, 10225, 16703}},
	{3378, 15, 13749, {1, 1, 1, 3, 15, 9, 67, 119, 95, 753, 511, 2507, 3953, 6403, 27635}},
	{3379, 15, 13753, {1, 3, 3, 7, 27, 57, 25, 27, 249, 515, 193, 4043, 2017, 751, 10949}},
	{3380, 15, 13754, {1, 3, 1, 9, 31, 57, 67, 21, 177, 573, 1835, 2015, 6201, 2383, 31087}},
	{3381, 15, 13771, {1, 1, 5, 1, 19, 3, 89, 243, 69, 387, 1905, 3465, 2775, 7713, 30081}},
	{3382, 15, 13773, {1, 1, 3, 3, 9, 59, 15, 89, 85, 605, 881, 263, 2551, 797, 16541}},
	{3383, 15, 13782, {1, 3, 7, 11, 25, 41, 59, 139, 405, 571, 1147, 2963, 4175, 12901, 6309}},
	{3384, 15, 13785, {1, 3, 1, 5, 29, 29, 11, 243, 183, 281, 807, 1, 7079, 10079, 13865}},
	{3385, 15, 13798, {1, 3, 7, 5, 5, 1, 89, 55, 423, 157, 675, 1849, 241, 6467, 15459}},
	{3386, 15, 13802, {1, 1, 7, 11, 15, 63, 89, 109, 501, 549, 317, 3043, 1151, 3895, 19851}},
	{3387, 15, 13809, {1, 3, 1, 15, 7, 23, 97, 97, 225, 737, 1117, 3325, 209, 14169, 10813}},
	{3388, 15, 13828, {1, 3, 7, 13, 13, 39, 91, 153, 395, 879, 1041, 3753, 5577, 1985, 25247}},
	{3389, 15, 13832, {1, 1, 1, 3, 17, 15, 113, 143, 101, 901, 1119, 1819, 3577, 3441, 31511}},
	{3390, 15, 13840, {1, 3, 1, 11, 15, 27, 21, 37, 287, 121, 451, 1353, 2173, 299, 18791}},
	{3391, 15, 13850, {1, 3, 3, 5, 23, 1, 49, 145, 315, 769, 99, 1385, 5961, 9121, 1465}},
	{3392, 15, 13861, {1, 3, 3, 13, 21, 39, 39, 233, 271, 113, 615, 2257, 3765, 5921, 313}},
	{3393, 15, 13874, {1, 3, 7, 7, 25, 45, 11, 237, 83, 203, 929, 1819, 2679, 11583, 30091}},
	{3394, 15, 13876, {1, 1, 1, 7, 21, 63, 85, 251, 133, 991, 1515, 2547, 6051, 7279, 3569}},
	{3395, 15, 13886, {1, 3, 7, 15, 11, 19, 87, 17, 313, 283, 1021, 2743, 4855, 13741, 17955}},
	{3396, 15, 13897, {1, 1, 7, 13, 29, 13, 61, 93, 81, 91, 995, 907, 4847, 2599, 20041}},
	{3397, 15, 13900, {1, 1, 3, 11, 13, 45, 103, 33, 243, 109, 2029, 121, 231, 16179, 13741}},
	{3398, 15, 13915, {1, 3, 5, 9, 9, 5, 73, 225, 199, 723, 611, 1909, 2345, 10257, 9909}},
	{3399, 15, 13927, {1, 1, 3, 11, 7, 5, 33, 89, 431, 899, 803, 3173, 6131, 16097, 20561}},
	{3400, 15, 13951, {1, 3, 3, 7, 7, 47, 23, 47, 411, 69, 239, 661, 5591, 10457, 24245}},
	{3401, 15, 13955, {1, 1, 5, 15, 25, 35, 87, 23, 115, 939, 1579, 119, 4001, 13791, 9729}},
	{3402, 15, 13962, {1, 3, 5, 11, 25, 45, 29, 195, 369, 237, 735, 155, 123, 4415, 32255}},
	{3403, 15, 13969, {1, 3, 3, 9, 13, 53, 15, 77, 313, 75, 529, 925, 5679, 14585, 19889}},
	{3404, 15, 13979, {1, 1, 7, 15, 15, 27, 105, 13, 31, 669, 563, 1809, 4321, 7797, 4177}},
	{3405, 15, 13988, {1, 1, 5, 9, 3, 29, 111, 177, 33, 235, 1951, 1561, 141, 4803, 16327}},
	{3406, 15, 13998, {1, 1, 1, 7, 9, 41, 1, 149, 95, 933, 115, 1619, 771, 8189, 8781}},
	{3407, 15, 14000, {1, 1, 5, 3, 13, 41, 33, 159, 355, 159, 1243, 1439, 6571, 14397, 31321}},
	{3408, 15, 14005, {1, 1, 7, 11, 9, 15, 91, 145, 457, 255, 1449, 611, 1449, 2521, 28949}},
	{3409, 15, 14027, {1, 3, 7, 5, 27, 57, 35, 99, 447, 287, 743, 1163, 4379, 7361, 3831}},
	{3410, 15, 14037, {1, 3, 3, 7, 15, 53, 41, 83, 133, 571, 1739, 531, 2921, 11527, 21941}},
	{3411, 15, 14051, {1, 1, 1, 13, 9, 27, 39, 113, 429, 447, 595, 3171, 5245, 4095, 14847}},
	{3412, 15, 14054, {1, 1, 3, 7, 19, 19, 21, 101, 489, 1011, 265, 3899, 3225, 11701, 5193}},
	{3413, 15, 14060, {1, 3, 7, 3, 15, 25, 103, 213, 441, 215, 1483, 263, 3561, 7915, 7969}},
	{3414, 15, 14063, {1, 3, 3, 3, 11, 47, 97, 29, 489, 867, 1347, 2155, 4871, 8001, 18305}},
	{3415, 15, 14071, {1, 3, 1, 9, 25, 15, 61, 17, 343, 775, 1765, 3803, 4577, 8437, 12605}},
	{3416, 15, 14078, {1, 1, 5, 3, 11, 39, 69, 23, 23, 65, 1967, 2429, 1703, 6671, 14981}},
	{3417, 15, 14080, {1, 1, 5, 15, 23, 59, 125, 51, 225, 439, 2019, 2589, 7781, 13111, 2911}},
	{3418, 15, 14085, {1, 1, 1, 3, 1, 31, 37, 245, 203, 305, 821, 367, 5211, 9791, 21777}},
	{3419, 15, 14086, {1, 1, 5, 9, 9, 31, 97, 25, 271, 83, 343, 2461, 1805, 14383, 10059}},
	{3420, 15, 14095, {1, 1, 5, 13, 15, 33, 127, 109, 137, 963, 961, 1647, 7881, 8133, 22359}},
	{3421, 15, 14138, {1, 1, 3, 7, 25, 31, 123, 241, 283, 1, 1781, 23, 971, 6485, 127}},
	{3422, 15, 14145, {1, 1, 5, 15, 15, 27, 25, 145, 395, 679, 979, 571, 1585, 14787, 7465}},
	{3423, 15, 14158, {1, 1, 5, 7, 13, 11, 7, 131, 511, 597, 379, 1513, 6267, 16039, 1503}},
	{3424, 15, 14166, {1, 1, 1, 13, 15, 49, 73, 217, 353, 577, 1913, 1127, 961, 11557, 24993}},
	{3425, 15, 14179, {1, 3, 3, 9, 7, 3, 105, 141, 377, 687, 1917, 485, 983, 11149, 23303}},
	{3426, 15, 14181, {1, 1, 3, 15, 11, 7, 117, 179, 505, 67, 1817, 913, 5757, 1981, 1637}},
	{3427, 15, 14188, {1, 1, 1, 7, 5, 29, 3, 43, 223, 295, 1895, 3425, 5355, 5155, 17197}},
	{3428, 15, 14193, {1, 1, 7, 9, 21, 59, 121, 245, 73, 233, 1527, 869, 4145, 7995, 6473}},
	{3429, 15, 14200, {1, 1, 5, 13, 17, 21, 89, 179, 495, 669, 453, 2603, 5969, 6161, 4743}},
	{3430, 15, 14203, {1, 1, 7, 11, 25, 21, 103, 131, 391, 249, 1633, 2603, 2207, 8987, 15487}},
	{3431, 15, 14215, {1, 3, 7, 9, 13, 45, 99, 251, 115, 597, 1505, 2421, 1231, 10015, 24295}},
	{3432, 15, 14224, {1, 1, 5, 5, 31, 49, 17, 67, 463, 813, 1491, 3309, 7881, 8109, 7289}},
	{3433, 15, 14230, {1, 3, 1, 15, 23, 35, 123, 21, 169, 499, 95, 603, 1829, 7865, 26313}},
	{3434, 15, 14233, {1, 1, 7, 1, 9, 29, 45, 65, 95, 97, 673, 3673, 2969, 2317, 22209}},
	{3435, 15, 14236, {1, 1, 3, 7, 29, 33, 121, 17, 331, 487, 1901, 1951, 5383, 9375, 4029}},
	{3436, 15, 14246, {1, 3, 7, 9, 25, 43, 91, 147, 141, 401, 1647, 2697, 4645, 7179, 31857}},
	{3437, 15, 14267, {1, 3, 5, 11, 9, 31, 127, 105, 39, 883, 1635, 919, 5069, 2875, 24519}},
	{3438, 15, 14282, {1, 1, 5, 9, 1, 63, 73, 135, 95, 503, 385, 3903, 545, 12635, 27569}},
	{3439, 15, 14287, {1, 1, 3, 11, 27, 31, 47, 173, 55, 339, 1255, 1947, 793, 14133, 13963}},
	{3440, 15, 14301, {1, 1, 3, 15, 17, 33, 113, 249, 401, 743, 1307, 3123, 627, 1253, 13285}},
	{3441, 15, 14323, {1, 1, 3, 1, 9, 7, 39, 65, 281, 107, 833, 193, 2987, 12267, 31335}},
	{3442, 15, 14325, {1, 1, 7, 3, 15, 21, 99, 211, 39, 179, 587, 1169, 6455, 8225, 2049}},
	{3443, 15, 14329, {1, 3, 5, 13, 5, 13, 123, 1, 223, 273, 731, 2967, 4793, 4229, 26031}},
	{3444, 15, 14339, {1, 1, 1, 1, 3, 17, 7, 23, 225, 757, 743, 1257, 2047, 12509, 25467}},
	{3445, 15, 14342, {1, 1, 7, 15, 29, 3, 15, 113, 227, 675, 1295, 2777, 2921, 5485, 2577}},
	{3446, 15, 14351, {1, 3, 7, 13, 19, 21, 85, 129, 45, 599, 317, 1513, 4953, 10383, 25253}},
	{3447, 15, 14356, {1, 1, 7, 11, 13, 47, 127, 67, 219, 131, 905, 2005, 851, 15243, 5777}},
	{3448, 15, 14359, {1, 1, 5, 3, 23, 57, 57, 189, 153, 37, 955, 2049, 1295, 15119, 27213}},
	{3449, 15, 14370, {1, 3, 7, 11, 13, 61, 3, 241, 269, 789, 1595, 2369, 4843, 11347, 21543}},
	{3450, 15, 14402, {1, 1, 5, 5, 25, 21, 19, 237, 3, 605, 1343, 3965, 3511, 7889, 27759}},
	{3451, 15, 14411, {1, 3, 1, 15, 21, 15, 123, 5, 345, 945, 283, 1313, 335, 2085, 19505}},
	{3452, 15, 14421, {1, 1, 3, 3, 5, 21, 123, 89, 67, 11, 1247, 1155, 287, 13455, 5693}},
	{3453, 15, 14431, {1, 3, 3, 13, 1, 53, 101, 27, 387, 379, 19, 751, 2445, 11737, 975}},
	{3454, 15, 14435, {1, 3, 3, 3, 9, 29, 81, 117, 443, 145, 1619, 1813, 8125, 5829, 28617}},
	{3455, 15, 14442, {1, 1, 5, 15, 27, 15, 83, 83, 61, 715, 1655, 1631, 3457, 2727, 2163}},
	{3456, 15, 14447, {1, 3, 1, 5, 11, 11, 121, 7, 135, 883, 927, 1817, 6839, 12361, 24119}},
	{3457, 15, 14456, {1, 3, 7, 11, 23, 59, 39, 165, 109, 355, 1303, 381, 5697, 275, 3771}},
	{3458, 15, 14459, {1, 3, 5, 11, 11, 5, 81, 157, 55, 435, 613, 127, 4087, 3791, 21627}},
	{3459, 15, 14472, {1, 3, 7, 15, 13, 37, 83, 195, 207, 771, 51, 3685, 6389, 1229, 11101}},
	{3460, 15, 14477, {1, 3, 7, 13, 31, 3, 9, 13, 487, 95, 77, 809, 5809, 12887, 29933}},
	{3461, 15, 14490, {1, 1, 3, 7, 25, 9, 13, 29, 353, 659, 1785, 3825, 3729, 13109, 12973}},
	{3462, 15, 14496, {1, 1, 1, 5, 21, 3, 97, 1, 245, 917, 29, 1429, 8141, 7569, 32493}},
	{3463, 15, 14501, {1, 3, 1, 9, 19, 13, 13, 109, 377, 1007, 1737, 1939, 1419, 1145, 5065}},
	{3464, 15, 14505, {1, 1, 7, 9, 27, 57, 53, 69, 423, 43, 1629, 1003, 1473, 10809, 5659}},
	{3465, 15, 14513, {1, 1, 1, 9, 1, 45, 11, 231, 351, 155, 663, 2783, 3491, 5725, 25207}},
	{3466, 15, 14520, {1, 1, 1, 3, 15, 25, 77, 89, 231, 813, 657, 2603, 4885, 1383, 14499}},
	{3467, 15, 14534, {1, 3, 5, 5, 9, 21, 101, 181, 449, 491, 737, 803, 659, 11771, 545}},
	{3468, 15, 14562, {1, 3, 7, 9, 7, 19, 27, 199, 265, 329, 1031, 1235, 3191, 10071, 16281}},
	{3469, 15, 14576, {1, 1, 7, 11, 27, 55, 3, 127, 503, 1003, 1041, 1953, 5835, 4851, 13485}},
	{3470, 15, 14579, {1, 1, 7, 15, 5, 45, 97, 61, 221, 497, 1949, 3163, 4707, 8441, 1437}},
	{3471, 15, 14585, {1, 3, 5, 1, 3, 35, 107, 9, 473, 971, 227, 2225, 3999, 3095, 18879}},
	{3472, 15, 14586, {1, 1, 1, 9, 21, 59, 21, 1, 41, 435, 575, 491, 1839, 1095, 9727}},
	{3473, 15, 14606, {1, 3, 5, 9, 13, 29, 123, 251, 465, 701, 1105, 829, 573, 11503, 11861}},
	{3474, 15, 14627, {1, 3, 3, 13, 27, 59, 29, 111, 225, 973, 561, 1481, 835, 9261, 13831}},
	{3475, 15, 14630, {1, 1, 1, 7, 17, 3, 97, 211, 333, 315, 571, 3523, 7305, 6461, 20139}},
	{3476, 15, 14634, {1, 3, 7, 11, 31, 21, 105, 247, 113, 863, 1767, 381, 4623, 8935, 7911}},
	{3477, 15, 14636, {1, 1, 5, 7, 29, 45, 17, 155, 69, 17, 655, 1983, 6385, 6177, 7961}},
	{3478, 15, 14647, {1, 3, 3, 15, 31, 15, 63, 81, 309, 115, 393, 3445, 689, 13963, 18887}},
	{3479, 15, 14653, {1, 1, 5, 1, 19, 39, 127, 61, 357, 53, 195, 2745, 7853, 5753, 3669}},
	{3480, 15, 14659, {1, 3, 7, 7, 17, 51, 57, 145, 451, 365, 1517, 909, 4265, 10737, 9579}},
	{3481, 15, 14671, {1, 1, 3, 13, 3, 37, 121, 103, 257, 47, 1685, 2951, 5753, 15379, 8899}},
	{3482, 15, 14674, {1, 1, 5, 7, 31, 63, 61, 197, 97, 773, 133, 1517, 3093, 14879, 22941}},
	{3483, 15, 14701, {1, 1, 5, 1, 3, 9, 27, 53, 97, 663, 1915, 409, 471, 1391, 24853}},
	{3484, 15, 14716, {1, 1, 1, 7, 21, 53, 69, 5, 187, 571, 2023, 997, 323, 12059, 7071}},
	{3485, 15, 14719, {1, 3, 3, 1, 7, 59, 55, 157, 101, 123, 1301, 3709, 4673, 3897, 28791}},
	{3486, 15, 14720, {1, 3, 7, 5, 5, 23, 39, 139, 365, 415, 1481, 3415, 6323, 11109, 5719}},
	{3487, 15, 14725, {1, 3, 5, 3, 5, 11, 23, 143, 243, 229, 183, 3367, 3187, 8151, 28351}},
	{3488, 15, 14730, {1, 3, 7, 9, 5, 37, 29, 23, 437, 827, 985, 2879, 7611, 1391, 19087}},
	{3489, 15, 14743, {1, 3, 3, 5, 7, 9, 5, 143, 217, 757, 1697, 2459, 453, 8679, 4513}},
	{3490, 15, 14747, {1, 3, 5, 5, 11, 33, 3, 143, 293, 921, 185, 2461, 5547, 12247, 28591}},
	{3491, 15, 14786, {1, 3, 7, 5, 3, 53, 43, 179, 235, 417, 1307, 1367, 3695, 12809, 1807}},
	{3492, 15, 14788, {1, 3, 1, 11, 15, 43, 115, 229, 157, 25, 687, 3347, 271, 5777, 8557}},
	{3493, 15, 14792, {1, 3, 7, 5, 27, 37, 55, 135, 209, 47, 1603, 957, 5785, 11141, 10407}},
	{3494, 15, 14795, {1, 1, 1, 15, 17, 17, 103, 29, 489, 493, 119, 1707, 3463, 1815, 32055}},
	{3495, 15, 14809, {1, 3, 7, 11, 17, 13, 115, 145, 77, 515, 1911, 477, 5997, 8731, 3143}},
	{3496, 15, 14831, {1, 3, 1, 13, 31, 41, 73, 91, 231, 1, 455, 2023, 4691, 3613, 16329}},
	{3497, 15, 14834, {1, 1, 5, 15, 15, 39, 17, 117, 131, 657, 1939, 2245, 2575, 195, 25209}},
	{3498, 15, 14850, {1, 3, 7, 15, 5, 51, 69, 141, 499, 931, 1165, 2119, 1703, 10867, 28443}},
	{3499, 15, 14855, {1, 1, 1, 15, 13, 45, 45, 103, 115, 177, 651, 2545, 1417, 5349, 3385}},
	{3500, 15, 14859, {1, 3, 3, 1, 1, 41, 117, 15, 225, 861, 843, 2775, 4543, 6275, 14671}},
	{3501, 15, 14864, {1, 3, 5, 15, 5, 35, 87, 193, 341, 55, 1131, 945, 6865, 11271, 18705}},
	{3502, 15, 14876, {1, 3, 5, 9, 13, 35, 71, 197, 79, 351, 3, 3939, 1105, 12455, 28921}},
	{3503, 15, 14889, {1, 3, 1, 13, 9, 23, 89, 165, 59, 257, 1369, 161, 6255, 2997, 19175}},
	{3504, 15, 14890, {1, 3, 5, 3, 5, 41, 107, 231, 111, 207, 1865, 2079, 5891, 2487, 5863}},
	{3505, 15, 14898, {1, 3, 7, 15, 3, 3, 105, 235, 263, 991, 367, 1885, 1769, 7805, 11909}},
	{3506, 15, 14909, {1, 3, 3, 5, 15, 59, 67, 247, 77, 367, 1641, 1959, 1921, 5939, 17355}},
	{3507, 15, 14917, {1, 1, 7, 1, 3, 53, 37, 5, 221, 779, 1353, 1633, 2769, 6355, 8505}},
	{3508, 15, 14924, {1, 1, 7, 13, 11, 13, 73, 227, 115, 523, 355, 3127, 7545, 8409, 22335}},
	{3509, 15, 14929, {1, 1, 5, 11, 21, 15, 91, 115, 427, 683, 461, 2433, 6313, 4595, 24401}},
	{3510, 15, 14942, {1, 3, 7, 5, 29, 21, 57, 215, 423, 717, 1455, 705, 6835, 4503, 26077}},
	{3511, 15, 14951, {1, 1, 1, 15, 3, 33, 25, 227, 381, 477, 1023, 2751, 2229, 631, 16903}},
	{3512, 15, 14969, {1, 3, 1, 11, 9, 17, 59, 73, 53, 671, 251, 1729, 7593, 12473, 22533}},
	{3513, 15, 14970, {1, 3, 3, 1, 3, 35, 37, 173, 459, 143, 135, 3871, 2689, 8007, 4379}},
	{3514, 15, 14972, {1, 3, 5, 9, 23, 19, 43, 45, 493, 509, 1851, 1615, 5675, 13793, 6973}},
	{3515, 15, 14982, {1, 3, 3, 15, 5, 17, 77, 85, 451, 753, 579, 1057, 4851, 6017, 4195}},
	{3516, 15, 14988, {1, 3, 3, 5, 31, 29, 81, 159, 103, 391, 15, 899, 4623, 5957, 31961}},
	{3517, 15, 14994, {1, 1, 1, 7, 17, 57, 81, 17, 177, 633, 49, 2793, 5229, 5995, 9491}},
	{3518, 15, 15005, {1, 1, 7, 15, 17, 19, 65, 57, 189, 239, 1229, 929, 2681, 12845, 29311}},
	{3519, 15, 15016, {1, 3, 1, 11, 13, 47, 61, 203, 383, 875, 943, 139, 4217, 8279, 1047}},
	{3520, 15, 15024, {1, 3, 7, 13, 23, 7, 1, 69, 47, 537, 1325, 3101, 685, 14057, 19953}},
	{3521, 15, 15030, {1, 3, 3, 1, 1, 7, 39, 77, 47, 755, 527, 2985, 5433, 15095, 27741}},
	{3522, 15, 15048, {1, 1, 7, 5, 23, 57, 79, 155, 81, 937, 1071, 3929, 1655, 3831, 17351}},
	{3523, 15, 15054, {1, 3, 7, 1, 3, 41, 13, 235, 207, 487, 1883, 2247, 1231, 2751, 15615}},
	{3524, 15, 15066, {1, 1, 7, 1, 21, 57, 95, 191, 119, 483, 283, 2221, 5665, 14819, 26097}},
	{3525, 15, 15071, {1, 3, 1, 1, 9, 59, 27, 51, 393, 31, 925, 715, 7705, 14885, 28767}},
	{3526, 15, 15072, {1, 1, 3, 3, 3, 61, 109, 131, 113, 249, 1331, 2521, 2973, 6375, 20093}},
	{3527, 15, 15075, {1, 3, 7, 9, 31, 37, 125, 245, 237, 245, 111, 379, 7495, 15531, 2325}},
	{3528, 15, 15119, {1, 3, 7, 13, 21, 21, 57, 21, 449, 969, 417, 2999, 509, 639, 7797}},
	{3529, 15, 15121, {1, 3, 7, 7, 7, 29, 11, 175, 55, 705, 891, 863, 3021, 10071, 10267}},
	{3530, 15, 15133, {1, 1, 3, 13, 19, 17, 127, 57, 449, 579, 337, 899, 1235, 11269, 4245}},
	{3531, 15, 15138, {1, 1, 1, 11, 29, 61, 35, 75, 249, 683, 287, 45, 3277, 7521, 2073}},
	{3532, 15, 15143, {1, 3, 5, 5, 15, 25, 77, 63, 63, 801, 1387, 1533, 2185, 10899, 28381}},
	{3533, 15, 15170, {1, 3, 1, 1, 21, 49, 3, 249, 419, 575, 87, 3749, 2523, 16125, 9483}},
	{3534, 15, 15194, {1, 1, 1, 11, 21, 43, 85, 211, 449, 439, 1495, 1841, 4765, 15253, 1467}},
	{3535, 15, 15212, {1, 3, 3, 15, 3, 37, 31, 243, 187, 995, 1103, 2723, 1523, 15967, 28649}},
	{3536, 15, 15223, {1, 1, 5, 11, 9, 11, 17, 87, 335, 125, 1079, 1657, 1237, 8059, 29833}},
	{3537, 15, 15229, {1, 3, 1, 3, 3, 41, 35, 37, 33, 61, 505, 3203, 5, 101, 8571}},
	{3538, 15, 15254, {1, 1, 3, 11, 9, 11, 85, 235, 261, 473, 109, 2127, 5745, 6389, 7431}},
	{3539, 15, 15263, {1, 1, 5, 15, 3, 55, 77, 97, 17, 193, 1267, 3063, 6531, 9797, 8639}},
	{3540, 15, 15270, {1, 1, 5, 5, 25, 41, 79, 83, 485, 697, 149, 1023, 89, 6115, 15227}},
	{3541, 15, 15273, {1, 1, 3, 15, 1, 9, 73, 251, 33, 599, 1017, 353, 4305, 16033, 29663}},
	{3542, 15, 15287, {1, 3, 7, 15, 3, 1, 89, 39, 125, 337, 1445, 3131, 3685, 9849, 25829}},
	{3543, 15, 15299, {1, 3, 7, 3, 19, 1, 63, 179, 349, 135, 185, 2977, 2527, 15087, 18133}},
	{3544, 15, 15301, {1, 1, 3, 3, 23, 7, 91, 221, 325, 723, 345, 81, 8077, 5501, 8453}},
	{3545, 15, 15306, {1, 1, 3, 9, 7, 3, 13, 173, 479, 161, 1989, 3255, 2069, 6717, 559}},
	{3546, 15, 15313, {1, 3, 3, 5, 9, 61, 93, 203, 277, 367, 1141, 981, 4745, 12625, 21003}},
	{3547, 15, 15320, {1, 3, 5, 5, 27, 17, 5, 211, 403, 701, 5, 3091, 4611, 5615, 23667}},
	{3548, 15, 15323, {1, 1, 3, 1, 21, 61, 125, 77, 57, 463, 1499, 791, 2087, 2805, 18829}},
	{3549, 15, 15329, {1, 3, 5, 3, 11, 41, 125, 231, 119, 837, 831, 1331, 7439, 2381, 3759}},
	{3550, 15, 15332, {1, 3, 1, 11, 19, 59, 117, 107, 443, 699, 315, 1491, 2581, 15871, 17159}},
	{3551, 15, 15341, {1, 3, 5, 11, 5, 9, 121, 35, 209, 877, 527, 3493, 4657, 16093, 17589}},
	{3552, 15, 15359, {1, 1, 7, 15, 9, 43, 119, 29, 381, 479, 1443, 3171, 5053, 9625, 21161}},
	{3553, 15, 15361, {1, 1, 3, 5, 15, 21, 31, 223, 83, 399, 1529, 3605, 6343, 10469, 10099}},
	{3554, 15, 15364, {1, 1, 3, 5, 5, 45, 23, 123, 353, 971, 85, 3069, 3245, 6569, 13241}},
	{3555, 15, 15367, {1, 1, 1, 3, 25, 49, 5, 77, 491, 881, 993, 1195, 7677, 5709, 10807}},
	{3556, 15, 15379, {1, 3, 3, 3, 5, 49, 127, 255, 183, 583, 1599, 987, 7281, 7149, 28507}},
	{3557, 15, 15391, {1, 1, 5, 1, 13, 55, 55, 157, 197, 25, 1971, 3161, 3903, 8919, 13563}},
	{3558, 15, 15415, {1, 3, 7, 9, 3, 37, 79, 193, 25, 103, 843, 2651, 6341, 2653, 24337}},
	{3559, 15, 15416, {1, 1, 7, 3, 25, 49, 99, 139, 45, 211, 2033, 2331, 7037, 7177, 1755}},
	{3560, 15, 15419, {1, 3, 7, 3, 5, 19, 127, 135, 403, 221, 141, 1065, 3935, 2745, 25979}},
	{3561, 15, 15433, {1, 1, 3, 3, 31, 23, 111, 37, 261, 7, 835, 2379, 7927, 8181, 23751}},
	{3562, 15, 15469, {1, 3, 7, 15, 1, 39, 79, 3, 103, 427, 1917, 809, 5039, 689, 1939}},
	{3563, 15, 15478, {1, 1, 1, 15, 29, 37, 39, 243, 149, 353, 763, 3405, 5751, 9441, 6653}},
	{3564, 15, 15481, {1, 3, 3, 11, 1, 57, 125, 151, 445, 423, 841, 2265, 5017, 15863, 13057}},
	{3565, 15, 15482, {1, 3, 5, 13, 11, 49, 61, 159, 211, 917, 561, 1903, 3985, 11117, 28969}},
	{3566, 15, 15498, {1, 3, 5, 13, 29, 5, 35, 51, 91, 291, 9, 3713, 3341, 4551, 12085}},
	{3567, 15, 15505, {1, 3, 3, 1, 1, 39, 111, 141, 319, 179, 1709, 1605, 5063, 13279, 10003}},
	{3568, 15, 15517, {1, 1, 3, 9, 7, 59, 91, 41, 343, 475, 1669, 2311, 5141, 12661, 25847}},
	{3569, 15, 15518, {1, 3, 5, 9, 9, 11, 49, 221, 1, 243, 791, 229, 503, 373, 19189}},
	{3570, 15, 15527, {1, 1, 5, 11, 17, 13, 45, 57, 215, 491, 1601, 2183, 3713, 429, 22007}},
	{3571, 15, 15528, {1, 1, 3, 11, 31, 61, 23, 237, 261, 955, 1085, 1541, 2601, 909, 7749}},
	{3572, 15, 15545, {1, 1, 3, 9, 13, 11, 121, 173, 177, 551, 1757, 2745, 2265, 4611, 743}},
	{3573, 15, 15548, {1, 1, 3, 15, 23, 43, 107, 239, 463, 369, 1857, 1073, 1247, 1029, 22557}},
	{3574, 15, 15554, {1, 1, 3, 11, 23, 35, 89, 93, 41, 941, 1141, 2339, 1423, 8007, 28685}},
	{3575, 15, 15565, {1, 3, 5, 13, 29, 7, 79, 15, 59, 145, 1237, 2215, 1257, 12621, 31101}},
	{3576, 15, 15577, {1, 1, 3, 7, 13, 55, 57, 229, 205, 1009, 341, 3901, 5189, 957, 32587}},
	{3577, 15, 15580, {1, 3, 7, 11, 1, 1, 41, 7, 365, 407, 1609, 1423, 6483, 5171, 32519}},
	{3578, 15, 15587, {1, 3, 7, 3, 17, 31, 125, 27, 125, 335, 1395, 2639, 329, 2549, 14449}},
	{3579, 15, 15601, {1, 3, 3, 7, 19, 45, 11, 73, 123, 179, 1685, 3385, 2379, 3387, 16793}},
	{3580, 15, 15604, {1, 3, 7, 5, 31, 25, 47, 153, 121, 453, 935, 3953, 2081, 12145, 24979}},
	{3581, 15, 15611, {1, 1, 7, 13, 25, 11, 65, 3, 277, 237, 1129, 1801, 4165, 9065, 18747}},
	{3582, 15, 15616, {1, 1, 7, 7, 13, 5, 37, 253, 507, 645, 1355, 3401, 6707, 6329, 11237}},
	{3583, 15, 15619, {1, 1, 3, 15, 17, 49, 3, 233, 407, 451, 69, 3859, 3171, 12303, 21031}},
	{3584, 15, 15625, {1, 1, 3, 3, 9, 53, 119, 117, 401, 903, 1449, 3639, 4083, 2095, 22085}},
	{3585, 15, 15633, {1, 3, 7, 15, 5, 61, 117, 193, 137, 431, 195, 4019, 3047, 5049, 14281}},
	{3586, 15, 15674, {1, 1, 1, 15, 17, 19, 29, 83, 449, 257, 1105, 1949, 1749, 3459, 6343}},
	{3587, 15, 15681, {1, 1, 1, 15, 23, 39, 61, 219, 109, 365, 863, 1813, 6673, 15999, 5101}},
	{3588, 15, 15691, {1, 1, 5, 5, 13, 11, 37, 151, 365, 719, 1233, 2425, 1285, 1721, 1205}},
	{3589, 15, 15693, {1, 3, 3, 3, 7, 53, 109, 153, 45, 425, 1741, 1229, 4405, 8071, 25155}},
	{3590, 15, 15696, {1, 3, 1, 1, 1, 13, 39, 49, 413, 77, 1367, 2553, 5563, 7659, 3467}},
	{3591, 15, 15712, {1, 1, 5, 9, 3, 49, 23, 11, 445, 121, 1505, 877, 4137, 1809, 2429}},
	{3592, 15, 15717, {1, 1, 1, 11, 21, 13, 93, 33, 493, 805, 775, 2939, 2961, 13625, 31879}},
	{3593, 15, 15724, {1, 1, 7, 5, 1, 59, 63, 131, 373, 23, 337, 2107, 5315, 4889, 22851}},
	{3594, 15, 15727, {1, 1, 3, 13, 21, 47, 15, 131, 353, 793, 1891, 1757, 5793, 1147, 23697}},
	{3595, 15, 15730, {1, 3, 5, 13, 7, 59, 25, 135, 259, 109, 1835, 429, 8153, 7355, 145}},
	{3596, 15, 15746, {1, 3, 3, 13, 9, 47, 121, 89, 89, 635, 1079, 2353, 4803, 11369, 12653}},
	{3597, 15, 15751, {1, 3, 5, 9, 23, 39, 49, 231, 105, 603, 613, 2021, 6073, 11819, 10595}},
	{3598, 15, 15760, {1, 3, 7, 7, 7, 19, 19, 155, 347, 387, 1459, 3793, 619, 14437, 2455}},
	{3599, 15, 15770, {1, 1, 1, 15, 21, 35, 19, 185, 483, 425, 479, 3429, 5403, 10791, 14219}},
	{3600, 15, 15782, {1, 1, 3, 11, 5, 51, 105, 63, 493, 677, 1457, 2865, 5619, 9321, 19583}},
	{3601, 15, 15791, {1, 1, 3, 3, 23, 1, 77, 177, 263, 289, 1567, 3837, 5359, 3269, 16023}},
	{3602, 15, 15796, {1, 1, 7, 3, 13, 61, 79, 77, 51, 953, 1417, 795, 4467, 2981, 25131}},
	{3603, 15, 15808, {1, 1, 5, 13, 23, 13, 29, 185, 337, 7, 149, 3609, 8119, 9545, 16579}},
	{3604, 15, 15814, {1, 3, 1, 5, 23, 9, 123, 15, 99, 55, 1021, 3709, 1521, 15189, 22193}},
	{3605, 15, 15825, {1, 3, 7, 9, 13, 41, 39, 45, 49, 181, 1587, 3213, 1037, 14775, 3333}},
	{3606, 15, 15828, {1, 1, 1, 7, 29, 55, 59, 31, 411, 601, 191, 283, 3211, 7951, 7919}},
	{3607, 15, 15835, {1, 1, 7, 7, 21, 47, 7, 193, 343, 831, 1267, 3289, 1015, 13093, 2717}},
	{3608, 15, 15844, {1, 3, 7, 1, 17, 9, 97, 19, 279, 827, 1699, 3573, 3137, 3535, 17791}},
	{3609, 15, 15847, {1, 1, 5, 11, 27, 15, 103, 135, 35, 625, 1575, 97, 7013, 13353, 19333}},
	{3610, 15, 15853, {1, 3, 3, 7, 17, 13, 49, 135, 435, 743, 1799, 2655, 4839, 2893, 31153}},
	{3611, 15, 15856, {1, 1, 5, 1, 3, 41, 1, 195, 53, 803, 1575, 2939, 3873, 10495, 5211}},
	{3612, 15, 15877, {1, 3, 1, 15, 19, 19, 37, 59, 355, 483, 685, 3899, 4645, 15127, 3479}},
	{3613, 15, 15878, {1, 1, 5, 3, 25, 9, 9, 229, 101, 631, 1165, 4091, 3723, 10655, 9463}},
	{3614, 15, 15887, {1, 3, 5, 15, 5, 13, 91, 61, 19, 469, 1675, 3331, 3121, 3435, 4111}},
	{3615, 15, 15908, {1, 1, 7, 1, 31, 61, 23, 83, 165, 551, 1097, 3825, 5385, 4723, 3635}},
	{3616, 15, 15917, {1, 3, 7, 15, 9, 31, 11, 121, 503, 855, 561, 1647, 1229, 1147, 15997}},
	{3617, 15, 15923, {1, 3, 7, 13, 21, 47, 41, 195, 197, 719, 1263, 3609, 7515, 2659, 30713}},
	{3618, 15, 15930, {1, 1, 1, 7, 31, 61, 101, 101, 479, 571, 605, 301, 6633, 15587, 23665}},
	{3619, 15, 15937, {1, 3, 7, 3, 25, 39, 35, 225, 135, 463, 53, 709, 5129, 4135, 10421}},
	{3620, 15, 15958, {1, 1, 5, 13, 19, 55, 107, 15, 163, 287, 673, 899, 5197, 4619, 3465}},
	{3621, 15, 15977, {1, 3, 3, 5, 21, 49, 15, 105, 283, 877, 1875, 1079, 3431, 13053, 26599}},
	{3622, 15, 15991, {1, 1, 7, 1, 1, 1, 95, 113, 119, 575, 1159, 2325, 6895, 12177, 4369}},
	{3623, 15, 16007, {1, 1, 1, 11, 25, 25, 83, 207, 301, 729, 1947, 2321, 3621, 15707, 11303}},
	{3624, 15, 16011, {1, 1, 5, 5, 7, 63, 83, 105, 211, 175, 1817, 2883, 5385, 7437, 24865}},
	{3625, 15, 16014, {1, 3, 7, 5, 23, 39, 19, 211, 151, 295, 573, 223, 5065, 6345, 23187}},
	{3626, 15, 16021, {1, 1, 7, 11, 15, 31, 89, 123, 57, 695, 685, 1799, 659, 9929, 22933}},
	{3627, 15, 16022, {1, 1, 7, 7, 19, 17, 27, 137, 117, 141, 1481, 869, 7061, 3073, 19671}},
	{3628, 15, 16028, {1, 3, 3, 11, 9, 19, 123, 93, 39, 517, 883, 3769, 2267, 8089, 6617}},
	{3629, 15, 16035, {1, 3, 1, 7, 9, 61, 51, 241, 319, 853, 1239, 899, 105, 1677, 29351}},
	{3630, 15, 16041, {1, 1, 7, 15, 13, 59, 85, 175, 223, 87, 905, 3175, 3405, 3489, 18475}},
	{3631, 15, 16056, {1, 1, 1, 15, 1, 55, 79, 97, 315, 605, 851, 4015, 3689, 9371, 31523}},
	{3632, 15, 16069, {1, 1, 5, 15, 1, 39, 91, 27, 211, 881, 1375, 2307, 5791, 10185, 23093}},
	{3633, 15, 16076, {1, 3, 1, 5, 3, 17, 59, 219, 105, 623, 21, 2843, 3427, 4799, 3793}},
	{3634, 15, 16081, {1, 3, 3, 7, 21, 55, 17, 29, 397, 93, 1981, 4047, 935, 5971, 14589}},
	{3635, 15, 16087, {1, 1, 3, 9, 5, 57, 63, 27, 373, 815, 167, 205, 367, 4945, 30041}},
	{3636, 15, 16088, {1, 1, 5, 9, 7, 3, 69, 35, 197, 309, 1729, 3735, 1523, 10427, 26253}},
	{3637, 15, 16110, {1, 1, 3, 7, 7, 49, 35, 189, 297, 311, 2025, 305, 3863, 14393, 2533}},
	{3638, 15, 16112, {1, 3, 3, 9, 17, 31, 5, 17, 167, 601, 909, 3149, 2533, 12123, 25325}},
	{3639, 15, 16117, {1, 3, 5, 3, 11, 41, 69, 199, 79, 611, 133, 3519, 5955, 4609, 27403}},
	{3640, 15, 16150, {1, 3, 3, 13, 3, 17, 53, 165, 361, 797, 1447, 869, 6707, 6541, 32249}},
	{3641, 15, 16153, {1, 3, 1, 1, 29, 47, 17, 45, 473, 199, 1595, 3095, 3635, 6965, 21859}},
	{3642, 15, 16160, {1, 1, 3, 9, 1, 15, 59, 163, 91, 811, 1087, 1707, 6743, 12643, 29901}},
	{3643, 15, 16166, {1, 1, 1, 3, 19, 21, 7, 209, 121, 821, 709, 1085, 5333, 7689, 28355}},
	{3644, 15, 16172, {1, 3, 1, 15, 5, 27, 115, 31, 37, 79, 1347, 155, 3709, 13251, 32151}},
	{3645, 15, 16190, {1, 3, 7, 15, 27, 27, 127, 231, 137, 205, 1665, 1461, 299, 2797, 879}},
	{3646, 15, 16195, {1, 1, 1, 7, 13, 3, 127, 13, 253, 481, 1435, 1895, 2665, 7611, 17761}},
	{3647, 15, 16204, {1, 1, 3, 7, 7, 21, 71, 247, 301, 183, 1785, 331, 4835, 2251, 4493}},
	{3648, 15, 16216, {1, 3, 7, 9, 9, 1, 77, 169, 103, 647, 1959, 1847, 5803, 3421, 15915}},
	{3649, 15, 16222, {1, 3, 1, 7, 19, 17, 81, 45, 263, 549, 1607, 2177, 1117, 14427, 16451}},
	{3650, 15, 16228, {1, 1, 7, 15, 27, 25, 27, 27, 33, 813, 1667, 253, 2749, 927, 29707}},
	{3651, 15, 16245, {1, 1, 7, 3, 17, 29, 13, 67, 417, 303, 19, 3809, 7225, 12775, 3933}},
	{3652, 15, 16255, {1, 1, 1, 11, 13, 41, 77, 217, 281, 659, 1099, 3047, 1619, 525, 4313}},
	{3653, 15, 16265, {1, 3, 3, 9, 23, 47, 5, 33, 219, 531, 77, 2307, 1893, 8335, 8281}},
	{3654, 15, 16273, {1, 3, 7, 3, 3, 35, 27, 249, 159, 495, 431, 3001, 1475, 11505, 15693}},
	{3655, 15, 16276, {1, 1, 5, 9, 21, 49, 43, 159, 465, 959, 179, 993, 121, 11569, 21027}},
	{3656, 15, 16283, {1, 3, 1, 5, 1, 61, 9, 221, 231, 55, 191, 2829, 3331, 8911, 15109}},
	{3657, 15, 16295, {1, 1, 7, 1, 7, 35, 67, 97, 159, 191, 935, 3151, 6397, 10751, 1835}},
	{3658, 15, 16304, {1, 1, 1, 7, 15, 39, 127, 163, 437, 333, 829, 753, 8151, 13239, 523}},
	{3659, 15, 16313, {1, 1, 3, 13, 9, 25, 73, 155, 445, 239, 2035, 15, 5243, 15531, 1733}},
	{3660, 15, 16319, {1, 3, 7, 15, 5, 25, 3, 55, 117, 57, 783, 1509, 7043, 13159, 8557}},
	{3661, 15, 16328, {1, 3, 5, 1, 21, 55, 89, 119, 199, 79, 161, 1597, 3263, 3335, 5757}},
	{3662, 15, 16345, {1, 3, 7, 5, 27, 23, 85, 113, 111, 211, 389, 1513, 2759, 7945, 931}},
	{3663, 15, 16355, {1, 1, 1, 7, 1, 5, 17, 177, 357, 619, 5, 2583, 621, 2973, 28845}},
	{3664, 15, 16364, {1, 3, 7, 13, 11, 21, 47, 99, 421, 279, 1541, 1305, 4571, 6127, 20735}},
	{3665, 15, 16372, {1, 3, 5, 5, 23, 43, 19, 137, 425, 409, 1625, 2671, 4385, 3197, 25753}},
	{3666, 15, 16375, {1, 1, 7, 5, 27, 17, 57, 15, 383, 181, 951, 2115, 5237, 1495, 9671}},
	{3667, 15, 16382, {1, 3, 3, 11, 9, 1, 53, 127, 375, 499, 1487, 121, 1465, 3175, 24337}},
	{3668, 16, 22, {1, 3, 7, 11, 29, 35, 67, 129, 221, 439, 1159, 3501, 7741, 8885, 11381, 20707}},
	{3669, 16, 28, {1, 3, 5, 11, 29, 59, 23, 117, 343, 637, 1825, 1687, 2823, 11641, 3311, 23603}},
	{3670, 16, 31, {1, 1, 5, 11, 1, 35, 103, 155, 233, 575, 1761, 503, 4175, 6105, 29737, 32681}},
	{3671, 16, 41, {1, 3, 3, 1, 5, 63, 27, 71, 245, 433, 1779, 2475, 5479, 4705, 10795, 34247}},
	{3672, 16, 94, {1, 3, 5, 7, 29, 45, 117, 5, 393, 849, 843, 3131, 6995, 9979, 28907, 30115}},
	{3673, 16, 107, {1, 3, 5, 9, 27, 29, 69, 5, 395, 561, 1531, 409, 2779, 8785, 16405, 27315}},
	{3674, 16, 151, {1, 3, 1, 9, 15, 29, 85, 3, 331, 19, 1941, 567, 6957, 747, 1627, 11347}},
	{3675, 16, 158, {1, 1, 3, 9, 27, 45, 47, 127, 133, 921, 1817, 2231, 6333, 14371, 12799, 9831}},
	{3676, 16, 167, {1, 1, 5, 15, 31, 7, 125, 13, 455, 159, 331, 3629, 4705, 11261, 3657, 36307}},
	{3677, 16, 174, {1, 1, 5, 9, 11, 53, 51, 35, 87, 885, 1975, 3899, 1013, 7667, 32385, 33635}},
	{3678, 16, 203, {1, 1, 1, 3, 7, 45, 107, 177, 193, 765, 731, 139, 5563, 623, 16485, 54999}},
	{3679, 16, 208, {1, 1, 5, 9, 17, 53, 117, 69, 385, 587, 1483, 149, 2769, 3013, 18183, 10173}},
	{3680, 16, 214, {1, 1, 5, 11, 5, 3, 25, 153, 351, 749, 801, 3077, 3209, 11189, 25241, 14115}},
	{3681, 16, 223, {1, 1, 7, 9, 1, 47, 41, 247, 135, 163, 899, 1517, 5647, 10595, 32531, 12497}},
	{3682, 16, 227, {1, 3, 5, 11, 5, 61, 111, 215, 251, 279, 825, 2155, 3527, 173, 10973, 59257}},
	{3683, 16, 266, {1, 3, 5, 11, 25, 15, 71, 83, 135, 231, 1415, 3761, 7513, 8337, 28979, 43615}},
	{3684, 16, 268, {1, 3, 5, 13, 19, 5, 55, 165, 141, 119, 1891, 2255, 4735, 16217, 26195, 50527}},
	{3685, 16, 274, {1, 1, 7, 15, 23, 59, 59, 191, 1, 855, 453, 2619, 5013, 14749, 24335, 44339}},
	{3686, 16, 279, {1, 1, 1, 13, 15, 41, 51, 147, 229, 495, 1191, 867, 1525, 581, 29713, 26391}},
	{3687, 16, 302, {1, 1, 1, 9, 29, 5, 59, 127, 105, 417, 301, 2249, 6335, 3513, 17373, 52977}},
	{3688, 16, 310, {1, 1, 3, 7, 21, 27, 109, 143, 63, 347, 1429, 2889, 2597, 10243, 9913, 22687}},
	{3689, 16, 322, {1, 3, 5, 5, 7, 3, 125, 147, 313, 351, 1163, 415, 5615, 5571, 7089, 55621}},
	{3690, 16, 328, {1, 3, 3, 3, 31, 43, 101, 93, 9, 671, 135, 333, 2169, 11169, 7403, 50707}},
	{3691, 16, 336, {1, 1, 7, 13, 15, 33, 125, 155, 227, 827, 1047, 2441, 3007, 10881, 19969, 63805}},
	{3692, 16, 370, {1, 3, 3, 5, 31, 33, 29, 249, 159, 797, 1475, 841, 6933, 6417, 25629, 61865}},
	{3693, 16, 398, {1, 3, 3, 15, 11, 55, 11, 117, 149, 911, 1589, 3133, 6477, 6123, 10471, 41099}},
	{3694, 16, 421, {1, 3, 3, 9, 27, 37, 1, 119, 509, 969, 831, 3771, 2093, 13621, 31737, 43269}},
	{3695, 16, 436, {1, 1, 1, 1, 9, 23, 119, 109, 487, 753, 1673, 2163, 3349, 4741, 29971, 3407}},
	{3696, 16, 440, {1, 3, 3, 7, 25, 7, 67, 9, 461, 631, 651, 2271, 5663, 2621, 3953, 20975}},
	{3697, 16, 451, {1, 1, 5, 11, 13, 31, 29, 255, 371, 517, 845, 3649, 1187, 10061, 22887, 58417}},
	{3698, 16, 454, {1, 3, 5, 13, 29, 1, 11, 137, 151, 249, 167, 1243, 997, 11023, 11875, 42315}},
	{3699, 16, 463, {1, 1, 5, 5, 5, 55, 103, 71, 255, 1023, 209, 1005, 2147, 11527, 17863, 6661}},
	{3700, 16, 465, {1, 1, 3, 3, 31, 39, 7, 151, 353, 775, 1313, 1257, 4197, 2625, 9571, 27269}},
	{3701, 16, 494, {1, 1, 1, 3, 7, 17, 3, 127, 501, 503, 1879, 2329, 3049, 10603, 2111, 33189}},
	{3702, 16, 508, {1, 3, 3, 7, 13, 59, 93, 13, 375, 483, 1991, 2257, 3003, 1699, 4339, 51827}},
	{3703, 16, 532, {1, 3, 7, 15, 27, 41, 59, 225, 405, 807, 1545, 2581, 1173, 14137, 3413, 39299}},
	{3704, 16, 555, {1, 1, 1, 3, 9, 23, 37, 123, 465, 1023, 1065, 1455, 5107, 3839, 20451, 11461}},
	{3705, 16, 563, {1, 1, 1, 11, 19, 55, 91, 121, 317, 199, 215, 3031, 7223, 11891, 21463, 64921}},
	{3706, 16, 577, {1, 3, 7, 11, 19, 5, 5, 115, 399, 219, 71, 1465, 281, 14451, 26807, 42541}},
	{3707, 16, 580, {1, 3, 5, 13, 3, 33, 75, 35, 19, 559, 761, 947, 7479, 15325, 31453, 20561}},
	{3708, 16, 584, {1, 3, 3, 13, 23, 47, 99, 73, 331, 353, 401, 1737, 6235, 13781, 5547, 56443}},
	{3709, 16, 607, {1, 3, 3, 13, 21, 37, 41, 205, 87, 399, 51, 3175, 7403, 12875, 21129, 7079}},
	{3710, 16, 608, {1, 3, 5, 11, 15, 47, 33, 39, 465, 871, 277, 2351, 695, 1953, 24293, 20595}},
	{3711, 16, 665, {1, 1, 7, 11, 13, 15, 115, 59, 469, 715, 191, 1927, 905, 13463, 29005, 46789}},
	{3712, 16, 675, {1, 3, 5, 9, 13, 55, 79, 17, 265, 887, 905, 3985, 6907, 3379, 20055, 58569}},
	{3713, 16, 692, {1, 1, 7, 11, 21, 29, 23, 109, 17, 427, 1623, 2219, 3857, 3709, 25033, 63823}},
	{3714, 16, 707, {1, 3, 5, 15, 19, 27, 113, 15, 25, 63, 1885, 2693, 5301, 9385, 14137, 26097}},
	{3715, 16, 737, {1, 3, 3, 11, 17, 5, 73, 143, 79, 957, 461, 1709, 4909, 2285, 18113, 8401}},
	{3716, 16, 750, {1, 1, 3, 7, 9, 9, 101, 127, 137, 755, 1359, 1965, 83, 13335, 27763, 7941}},
	{3717, 16, 757, {1, 1, 1, 3, 13, 61, 95, 61, 295, 615, 555, 2163, 8155, 14043, 21465, 46741}},
	{3718, 16, 800, {1, 1, 1, 13, 29, 19, 111, 17, 373, 153, 1703, 2199, 7209, 15845, 1879, 7493}},
	{3719, 16, 805, {1, 3, 1, 13, 21, 51, 49, 51, 255, 151, 207, 1915, 7629, 2705, 8739, 7467}},
	{3720, 16, 809, {1, 3, 7, 5, 21, 21, 23, 193, 467, 739, 519, 2315, 2953, 10633, 9163, 6007}},
	{3721, 16, 837, {1, 3, 1, 5, 23, 19, 23, 247, 93, 297, 1089, 2349, 4683, 13609, 7615, 18647}},
	{3722, 16, 865, {1, 1, 3, 3, 21, 39, 19, 71, 93, 1, 133, 3531, 7503, 2819, 24211, 1739}},
	{3723, 16, 949, {1, 3, 5, 13, 9, 43, 31, 111, 493, 739, 705, 2715, 3613, 11877, 27945, 46053}},
	{3724, 16, 950, {1, 1, 7, 13, 27, 59, 103, 129, 53, 531, 1379, 1441, 5341, 14937, 5079, 39881}},
	{3725, 16, 956, {1, 1, 3, 3, 11, 63, 91, 95, 433, 393, 715, 809, 591, 4141, 17417, 54107}},
	{3726, 16, 961, {1, 3, 5, 1, 7, 25, 25, 175, 205, 803, 183, 1441, 1279, 2753, 20001, 56677}},
	{3727, 16, 1016, {1, 1, 5, 3, 13, 23, 77, 25, 133, 137, 1907, 1313, 2463, 14339, 13, 57757}},
	{3728, 16, 1030, {1, 1, 5, 9, 23, 35, 1, 119, 111, 61, 403, 1815, 1985, 5651, 10883, 55943}},
	{3729, 16, 1072, {1, 3, 1, 7, 21, 43, 115, 7, 107, 719, 759, 1521, 467, 8735, 29785, 63821}},
	{3730, 16, 1119, {1, 1, 3, 13, 19, 17, 51, 141, 399, 569, 703, 2221, 2809, 13355, 1907, 15837}},
	{3731, 16, 1130, {1, 1, 5, 15, 15, 53, 57, 31, 481, 69, 1439, 4049, 6727, 11307, 20683, 63517}},
	{3732, 16, 1135, {1, 1, 1, 3, 13, 27, 9, 255, 363, 131, 1745, 2489, 6451, 6585, 12873, 35405}},
	{3733, 16, 1137, {1, 3, 5, 1, 17, 31, 113, 135, 449, 915, 1017, 2317, 6821, 5483, 30707, 45279}},
	{3734, 16, 1144, {1, 3, 5, 1, 13, 47, 25, 53, 413, 545, 1777, 3049, 7527, 9689, 25935, 9919}},
	{3735, 16, 1149, {1, 3, 7, 11, 17, 39, 13, 131, 295, 517, 1755, 2977, 6267, 12351, 8957, 17765}},
	{3736, 16, 1180, {1, 1, 7, 5, 27, 57, 47, 21, 125, 429, 1169, 1717, 5455, 16359, 29065, 6671}},
	{3737, 16, 1214, {1, 1, 5, 5, 21, 15, 79, 241, 83, 515, 859, 2351, 3125, 7465, 30475, 19759}},
	{3738, 16, 1221, {1, 3, 1, 9, 11, 5, 81, 11, 7, 221, 141, 3329, 3435, 323, 18999, 54735}},
	{3739, 16, 1234, {1, 1, 1, 15, 7, 57, 87, 251, 63, 561, 929, 1367, 2511, 14527, 9335, 38775}},
	{3740, 16, 1239, {1, 3, 3, 9, 23, 37, 59, 105, 179, 515, 235, 2445, 433, 13039, 27005, 48829}},
	{3741, 16, 1249, {1, 1, 1, 1, 23, 37, 103, 31, 89, 921, 1687, 831, 387, 10237, 1241, 19295}},
	{3742, 16, 1250, {1, 3, 3, 7, 25, 23, 57, 251, 309, 579, 603, 807, 7383, 8579, 4025, 16757}},
	{3743, 16, 1267, {1, 1, 3, 15, 23, 59, 29, 33, 467, 641, 1271, 2915, 2549, 14767, 26557, 43483}},
	{3744, 16, 1273, {1, 1, 7, 13, 1, 57, 23, 129, 321, 75, 189, 4087, 5011, 4355, 25759, 37153}},
	{3745, 16, 1342, {1, 1, 5, 1, 21, 57, 25, 183, 37, 669, 259, 1381, 877, 10245, 16643, 61035}},
	{3746, 16, 1344, {1, 1, 7, 5, 11, 11, 85, 141, 393, 957, 1745, 2243, 1681, 5583, 16527, 12017}},
	{3747, 16, 1373, {1, 1, 5, 15, 23, 31, 5, 169, 287, 527, 1831, 2937, 7533, 9739, 24305, 2239}},
	{3748, 16, 1378, {1, 1, 7, 1, 7, 13, 3, 243, 189, 309, 607, 3659, 6369, 7649, 24255, 55373}},
	{3749, 16, 1408, {1, 1, 1, 3, 3, 59, 103, 209, 287, 913, 1223, 1063, 7715, 6073, 26697, 25671}},
	{3750, 16, 1417, {1, 3, 7, 5, 19, 19, 117, 191, 275, 637, 991, 2199, 2921, 10553, 21211, 25981}},
	{3751, 16, 1418, {1, 3, 3, 5, 29, 59, 17, 13, 127, 57, 1405, 3181, 2237, 1795, 21419, 43421}},
	{3752, 16, 1448, {1, 1, 1, 15, 25, 41, 11, 117, 463, 425, 305, 1441, 4307, 7967, 17529, 4043}},
	{3753, 16, 1454, {1, 3, 5, 5, 19, 53, 69, 73, 453, 611, 1583, 1721, 6303, 10561, 18527, 48973}},
	{3754, 16, 1510, {1, 1, 7, 11, 15, 61, 87, 69, 463, 771, 819, 469, 8165, 8897, 29657, 55161}},
	{3755, 16, 1513, {1, 1, 5, 1, 15, 25, 23, 47, 287, 457, 1219, 473, 4127, 3349, 9425, 41541}},
	{3756, 16, 1522, {1, 3, 7, 5, 17, 17, 33, 161, 239, 231, 241, 1297, 4879, 12761, 20939, 65261}},
	{3757, 16, 1543, {1, 3, 3, 9, 19, 53, 95, 89, 117, 333, 1815, 2217, 7779, 8213, 4667, 58395}},
	{3758, 16, 1550, {1, 3, 3, 9, 17, 7, 41, 99, 371, 797, 729, 2851, 2003, 4463, 20793, 54315}},
	{3759, 16, 1552, {1, 3, 5, 5, 23, 39, 19, 235, 163, 365, 141, 791, 455, 2761, 9115, 53351}},
	{3760, 16, 1588, {1, 3, 3, 3, 9, 27, 29, 139, 165, 867, 2023, 1333, 3771, 10451, 9141, 41177}},
	{3761, 16, 1592, {1, 1, 3, 7, 3, 11, 125, 157, 355, 519, 187, 3381, 1151, 1629, 25247, 42797}},
	{3762, 16, 1597, {1, 3, 3, 3, 21, 25, 37, 155, 257, 311, 961, 1945, 1383, 5679, 7857, 7183}},
	{3763, 16, 1606, {1, 3, 3, 5, 29, 11, 49, 125, 171, 605, 1923, 2781, 2555, 5063, 5075, 43301}},
	{3764, 16, 1610, {1, 3, 5, 9, 27, 1, 27, 149, 253, 205, 1299, 2901, 2891, 975, 7641, 8115}},
	{3765, 16, 1617, {1, 3, 5, 3, 31, 7, 49, 215, 81, 791, 1485, 837, 5051, 1947, 7521, 25723}},
	{3766, 16, 1623, {1, 3, 5, 7, 23, 25, 69, 13, 3, 859, 441, 3577, 1687, 6559, 8687, 46757}},
	{3767, 16, 1657, {1, 1, 1, 9, 1, 59, 3, 31, 251, 187, 617, 2607, 4635, 6121, 8565, 8871}},
	{3768, 16, 1697, {1, 3, 3, 9, 29, 37, 127, 87, 153, 633, 1691, 2729, 3167, 3219, 21237, 25573}},
	{3769, 16, 1729, {1, 1, 5, 13, 19, 63, 93, 235, 299, 621, 405, 663, 6639, 12265, 9303, 42719}},
	{3770, 16, 1735, {1, 1, 3, 9, 25, 11, 9, 231, 101, 335, 1793, 1497, 7069, 4171, 30199, 63}},
	{3771, 16, 1769, {1, 1, 1, 1, 5, 19, 17, 217, 165, 413, 925, 1409, 6559, 14537, 22057, 44331}},
	{3772, 16, 1778, {1, 1, 3, 7, 11, 51, 45, 217, 57, 795, 951, 2933, 6705, 137, 30525, 9679}},
	{3773, 16, 1826, {1, 1, 3, 15, 27, 47, 35, 125, 363, 619, 1027, 2861, 3923, 10459, 16789, 27277}},
	{3774, 16, 1858, {1, 1, 7, 7, 13, 37, 33, 29, 385, 851, 143, 119, 7345, 4251, 25121, 31609}},
	{3775, 16, 1870, {1, 3, 1, 1, 17, 25, 119, 7, 365, 397, 601, 2087, 6903, 15345, 14671, 37889}},
	{3776, 16, 1875, {1, 3, 1, 13, 19, 51, 41, 139, 133, 723, 25, 2621, 1257, 7037, 9527, 50037}},
	{3777, 16, 1922, {1, 1, 5, 11, 5, 59, 119, 75, 397, 545, 1095, 585, 3271, 1049, 123, 33029}},
	{3778, 16, 1924, {1, 1, 7, 11, 9, 27, 21, 197, 177, 31, 453, 2457, 2733, 7787, 1923, 24639}},
	{3779, 16, 1933, {1, 1, 7, 13, 29, 13, 91, 91, 243, 279, 601, 1699, 7169, 4727, 7815, 29099}},
	{3780, 16, 1972, {1, 3, 7, 5, 1, 35, 27, 235, 163, 913, 1479, 769, 7179, 1983, 25977, 55373}},
	{3781, 16, 1979, {1, 3, 5, 11, 9, 33, 99, 141, 301, 109, 1785, 129, 1707, 5181, 4797, 9979}},
	{3782, 16, 1987, {1, 1, 1, 13, 3, 47, 89, 43, 293, 87, 1689, 3885, 7747, 5607, 477, 31887}},
	{3783, 16, 1994, {1, 1, 5, 1, 9, 21, 73, 37, 45, 621, 1855, 3691, 4899, 2191, 13459, 23543}},
	{3784, 16, 2008, {1, 1, 1, 1, 7, 39, 61, 125, 341, 905, 213, 1755, 241, 13407, 8791, 10165}},
	{3785, 16, 2023, {1, 1, 1, 1, 19, 31, 79, 19, 55, 875, 1017, 1787, 4879, 533, 15029, 52295}},
	{3786, 16, 2029, {1, 3, 1, 1, 9, 59, 113, 71, 113, 649, 561, 71, 5253, 783, 7389, 19361}},
	{3787, 16, 2053, {1, 1, 1, 11, 5, 39, 61, 225, 291, 907, 795, 1099, 597, 11829, 15137, 42865}},
	{3788, 16, 2081, {1, 3, 1, 5, 25, 11, 71, 155, 271, 309, 1981, 1253, 463, 1133, 20833, 48625}},
	{3789, 16, 2087, {1, 3, 5, 9, 7, 41, 87, 241, 457, 899, 1493, 3675, 3025, 10607, 22569, 52813}},
	{3790, 16, 2094, {1, 3, 7, 13, 7, 37, 37, 103, 281, 915, 1259, 4049, 559, 173, 4123, 63767}},
	{3791, 16, 2111, {1, 3, 7, 15, 13, 57, 9, 51, 39, 549, 1431, 2887, 1081, 4643, 16331, 14221}},
	{3792, 16, 2113, {1, 3, 5, 7, 13, 1, 101, 125, 25, 713, 1423, 513, 3323, 9951, 7163, 20969}},
	{3793, 16, 2114, {1, 1, 7, 15, 11, 25, 25, 3, 47, 531, 1529, 471, 6191, 10051, 29671, 49085}},
	{3794, 16, 2123, {1, 1, 3, 5, 23, 51, 117, 141, 55, 275, 761, 1923, 6267, 2291, 3701, 26615}},
	{3795, 16, 2190, {1, 1, 7, 9, 15, 19, 111, 65, 137, 373, 1753, 3591, 1137, 11639, 28591, 27265}},
	{3796, 16, 2231, {1, 3, 1, 15, 29, 5, 67, 13, 425, 961, 453, 2481, 1407, 3479, 23303, 30407}},
	{3797, 16, 2276, {1, 1, 5, 3, 19, 39, 39, 123, 351, 77, 1339, 1765, 3767, 1907, 13627, 23877}},
	{3798, 16, 2285, {1, 3, 5, 9, 23, 7, 103, 177, 221, 197, 561, 2121, 7231, 12053, 30127, 29849}},
	{3799, 16, 2297, {1, 1, 5, 7, 15, 1, 3, 123, 197, 493, 171, 2425, 3865, 4061, 31883, 2491}},
	{3800, 16, 2336, {1, 1, 3, 13, 29, 33, 99, 67, 327, 969, 1793, 1871, 1839, 13059, 7605, 16797}},
	{3801, 16, 2345, {1, 3, 5, 11, 25, 53, 25, 93, 303, 623, 1889, 1471, 1213, 14459, 8527, 25095}},
	{3802, 16, 2353, {1, 1, 1, 13, 15, 3, 115, 3, 289, 743, 1855, 359, 2375, 13765, 19711, 40765}},
	{3803, 16, 2363, {1, 1, 7, 11, 27, 51, 85, 163, 219, 871, 637, 2011, 5981, 587, 17521, 17333}},
	{3804, 16, 2368, {1, 3, 5, 1, 21, 59, 49, 39, 305, 513, 2017, 285, 5817, 13123, 27765, 46741}},
	{3805, 16, 2373, {1, 3, 3, 7, 21, 39, 71, 163, 423, 845, 783, 397, 7319, 10677, 13407, 47471}},
	{3806, 16, 2391, {1, 3, 7, 5, 21, 59, 99, 179, 473, 687, 1393, 723, 2245, 2933, 25943, 7769}},
	{3807, 16, 2402, {1, 1, 5, 9, 5, 45, 71, 189, 165, 555, 643, 2289, 3133, 12319, 22209, 1533}},
	{3808, 16, 2413, {1, 1, 3, 9, 7, 43, 1, 155, 323, 169, 339, 2561, 4049, 4953, 5289, 8783}},
	{3809, 16, 2422, {1, 3, 1, 11, 15, 5, 25, 201, 267, 891, 561, 501, 575, 15147, 1743, 45237}},
	{3810, 16, 2425, {1, 3, 5, 13, 25, 27, 105, 205, 165, 795, 975, 943, 7413, 10299, 14839, 54895}},
	{3811, 16, 2461, {1, 1, 5, 1, 17, 43, 69, 103, 449, 917, 103, 945, 513, 709, 11647, 28065}},
	{3812, 16, 2462, {1, 1, 3, 15, 23, 51, 23, 7, 159, 743, 177, 3457, 415, 1775, 25353, 36385}},
	{3813, 16, 2490, {1, 3, 5, 13, 9, 63, 121, 19, 165, 449, 1523, 1959, 6901, 12281, 29149, 45999}},
	{3814, 16, 2492, {1, 3, 7, 11, 17, 19, 9, 155, 373, 753, 1313, 2205, 3571, 16317, 16151, 15325}},
	{3815, 16, 2510, {1, 3, 3, 7, 15, 43, 65, 183, 407, 123, 1151, 375, 3461, 6673, 12985, 21005}},
	{3816, 16, 2564, {1, 3, 7, 7, 9, 1, 87, 247, 489, 123, 1677, 1947, 7961, 13497, 27919, 28993}},
	{3817, 16, 2573, {1, 3, 3, 7, 19, 21, 95, 227, 217, 133, 69, 1535, 699, 3521, 29255, 34733}},
	{3818, 16, 2598, {1, 3, 5, 3, 7, 57, 45, 251, 407, 81, 1259, 2425, 2217, 13097, 12773, 14643}},
	{3819, 16, 2627, {1, 1, 1, 11, 23, 37, 13, 229, 467, 591, 1521, 469, 3763, 2289, 14233, 24053}},
	{3820, 16, 2633, {1, 3, 5, 1, 27, 53, 105, 5, 85, 765, 1973, 2597, 5725, 1063, 18145, 961}},
	{3821, 16, 2647, {1, 3, 7, 1, 21, 47, 115, 95, 403, 3, 1593, 3379, 7371, 15553, 12503, 57979}},
	{3822, 16, 2660, {1, 1, 3, 1, 1, 35, 121, 29, 379, 245, 919, 2673, 3503, 14197, 31193, 8355}},
	{3823, 16, 2664, {1, 3, 5, 11, 19, 49, 97, 7, 195, 1013, 1671, 3415, 2009, 13389, 4837, 27453}},
	{3824, 16, 2678, {1, 1, 5, 13, 9, 15, 115, 97, 463, 449, 303, 2681, 1215, 12559, 15685, 21321}},
	{3825, 16, 2684, {1, 3, 5, 13, 23, 5, 113, 193, 419, 301, 1121, 317, 5503, 4683, 25519, 65}},
	{3826, 16, 2691, {1, 3, 3, 7, 15, 29, 45, 97, 323, 475, 143, 1173, 4033, 8939, 31849, 3575}},
	{3827, 16, 2759, {1, 1, 7, 7, 21, 1, 101, 143, 197, 409, 855, 1753, 5211, 3763, 11139, 37309}},
	{3828, 16, 2768, {1, 1, 3, 13, 25, 33, 55, 45, 381, 349, 991, 535, 4823, 3701, 31629, 48037}},
	{3829, 16, 2773, {1, 3, 1, 11, 17, 51, 27, 57, 409, 551, 949, 365, 8093, 10831, 19697, 39437}},
	{3830, 16, 2794, {1, 3, 5, 3, 31, 33, 81, 49, 91, 865, 469, 2115, 377, 8237, 31907, 38239}},
	{3831, 16, 2813, {1, 1, 3, 7, 29, 59, 57, 17, 121, 889, 1557, 1797, 5001, 14209, 21355, 59739}},
	{3832, 16, 2831, {1, 1, 5, 9, 11, 45, 89, 87, 397, 785, 525, 1593, 5251, 12449, 23579, 54265}},
	{3833, 16, 2843, {1, 3, 5, 11, 5, 31, 19, 47, 207, 331, 91, 1691, 5171, 53, 15945, 33349}},
	{3834, 16, 2846, {1, 1, 1, 15, 11, 41, 91, 177, 505, 871, 815, 3673, 5631, 9915, 1133, 37861}},
	{3835, 16, 2849, {1, 3, 5, 5, 25, 63, 53, 231, 55, 51, 481, 303, 1859, 11973, 28557, 22045}},
	{3836, 16, 2856, {1, 1, 5, 3, 27, 11, 37, 91, 363, 411, 1131, 3369, 377, 6585, 7353, 42949}},
	{3837, 16, 2893, {1, 3, 1, 9, 31, 63, 83, 23, 405, 941, 119, 1471, 2509, 15507, 29239, 49613}},
	{3838, 16, 2901, {1, 1, 5, 1, 11, 63, 117, 237, 407, 231, 1425, 71, 8005, 4023, 9029, 59819}},
	{3839, 16, 2924, {1, 1, 5, 7, 1, 9, 43, 87, 351, 63, 1075, 3381, 5447, 2437, 24983, 26905}},
	{3840, 16, 2942, {1, 3, 7, 5, 5, 35, 33, 89, 251, 819, 1735, 2625, 6363, 6837, 27603, 26669}},
	{3841, 16, 2975, {1, 3, 7, 13, 29, 63, 51, 245, 371, 791, 907, 3499, 3033, 8443, 20023, 1805}},
	{3842, 16, 2979, {1, 1, 5, 7, 13, 15, 109, 197, 451, 709, 929, 3193, 5727, 11185, 29479, 1671}},
	{3843, 16, 2985, {1, 1, 7, 13, 19, 23, 97, 9, 359, 635, 777, 39, 893, 2531, 13563, 19295}},
	{3844, 16, 3020, {1, 1, 5, 1, 31, 63, 55, 7, 157, 877, 991, 1317, 1595, 2019, 21435, 52255}},
	{3845, 16, 3025, {1, 1, 5, 3, 19, 37, 23, 13, 335, 431, 483, 615, 2431, 505, 26245, 63323}},
	{3846, 16, 3028, {1, 3, 7, 5, 5, 9, 37, 65, 303, 423, 1907, 2661, 7213, 2975, 29045, 16243}},
	{3847, 16, 3051, {1, 3, 1, 5, 13, 37, 115, 217, 227, 159, 707, 1387, 943, 4935, 5503, 35171}},
	{3848, 16, 3127, {1, 3, 7, 9, 19, 15, 87, 233, 453, 159, 169, 1077, 2129, 413, 19773, 629}},
	{3849, 16, 3142, {1, 1, 5, 15, 29, 39, 37, 243, 233, 365, 1843, 2219, 1255, 15287, 603, 13511}},
	{3850, 16, 3145, {1, 1, 3, 3, 31, 53, 33, 125, 497, 597, 127, 1829, 3905, 2611, 4263, 40971}},
	{3851, 16, 3156, {1, 3, 5, 9, 11, 47, 71, 215, 383, 321, 1445, 135, 5953, 8791, 22073, 16537}},
	{3852, 16, 3165, {1, 3, 3, 13, 15, 7, 7, 133, 401, 459, 1117, 3165, 4105, 11943, 22431, 56821}},
	{3853, 16, 3196, {1, 1, 7, 9, 31, 39, 19, 7, 19, 401, 79, 3641, 6815, 1489, 7537, 49467}},
	{3854, 16, 3199, {1, 3, 7, 7, 17, 11, 91, 205, 251, 321, 515, 3521, 311, 3169, 271, 34749}},
	{3855, 16, 3217, {1, 3, 3, 7, 29, 15, 5, 153, 83, 603, 1373, 997, 4939, 9811, 243, 5375}},
	{3856, 16, 3218, {1, 1, 3, 11, 21, 47, 25, 221, 237, 177, 535, 2819, 6213, 7877, 26795, 36609}},
	{3857, 16, 3253, {1, 3, 7, 3, 31, 1, 69, 73, 47, 653, 139, 1649, 7183, 1293, 26507, 38415}},
	{3858, 16, 3258, {1, 1, 1, 13, 17, 41, 23, 73, 115, 509, 787, 3733, 1871, 171, 29967, 39941}},
	{3859, 16, 3260, {1, 3, 5, 1, 9, 7, 61, 23, 105, 381, 1421, 2887, 3717, 643, 26375, 57991}},
	{3860, 16, 3289, {1, 3, 3, 3, 19, 3, 101, 117, 393, 83, 1255, 3331, 6481, 8661, 20855, 28875}},
	{3861, 16, 3314, {1, 3, 5, 11, 21, 13, 111, 193, 51, 899, 159, 1989, 7931, 10511, 3933, 447}},
	{3862, 16, 3326, {1, 1, 5, 15, 23, 35, 49, 139, 397, 145, 597, 1847, 7077, 715, 20227, 42183}},
	{3863, 16, 3331, {1, 3, 3, 3, 17, 3, 87, 233, 35, 317, 337, 237, 6901, 3439, 20033, 10307}},
	{3864, 16, 3371, {1, 3, 5, 3, 11, 35, 13, 171, 7, 963, 1443, 1501, 7617, 963, 25453, 62589}},
	{3865, 16, 3381, {1, 1, 1, 5, 11, 9, 39, 175, 409, 411, 1407, 2743, 4255, 989, 15823, 1707}},
	{3866, 16, 3396, {1, 1, 7, 13, 27, 55, 63, 239, 355, 417, 2007, 2299, 2921, 1637, 10687, 60615}},
	{3867, 16, 3441, {1, 1, 7, 9, 5, 61, 57, 73, 263, 307, 2003, 1763, 639, 5885, 14709, 16985}},
	{3868, 16, 3442, {1, 1, 3, 3, 21, 55, 19, 249, 509, 533, 1361, 1397, 2777, 15523, 4389, 13339}},
	{3869, 16, 3460, {1, 3, 5, 15, 9, 3, 91, 237, 451, 299, 1541, 4083, 879, 7859, 21585, 14833}},
	{3870, 16, 3477, {1, 1, 7, 3, 31, 47, 49, 231, 123, 391, 1633, 2567, 5577, 1631, 27951, 22913}},
	{3871, 16, 3491, {1, 3, 7, 13, 11, 13, 1, 111, 183, 87, 839, 1915, 5523, 3677, 13065, 38225}},
	{3872, 16, 3493, {1, 1, 3, 7, 15, 15, 63, 241, 167, 345, 653, 701, 4725, 12911, 11545, 24475}},
	{3873, 16, 3543, {1, 1, 3, 7, 25, 15, 49, 235, 331, 639, 965, 1117, 7147, 3789, 3309, 20255}},
	{3874, 16, 3549, {1, 3, 5, 7, 7, 63, 93, 241, 253, 31, 951, 3723, 3359, 7303, 191, 36427}},
	{3875, 16, 3550, {1, 3, 7, 9, 9, 59, 5, 107, 181, 413, 1269, 3121, 1929, 11921, 8931, 47459}},
	{3876, 16, 3553, {1, 3, 1, 15, 25, 27, 13, 47, 295, 111, 1287, 2551, 4887, 4145, 17063, 42037}},
	{3877, 16, 3563, {1, 3, 3, 13, 17, 17, 21, 17, 491, 845, 1463, 1305, 1375, 16149, 19331, 25043}},
	{3878, 16, 3568, {1, 3, 5, 1, 27, 5, 93, 139, 283, 711, 1141, 1743, 5001, 8851, 19351, 12275}},
	{3879, 16, 3604, {1, 1, 1, 1, 23, 25, 51, 63, 429, 735, 201, 3785, 6677, 16375, 19681, 17857}},
	{3880, 16, 3632, {1, 3, 3, 3, 9, 63, 71, 147, 463, 465, 1163, 1045, 6967, 12537, 31853, 38391}},
	{3881, 16, 3650, {1, 3, 7, 1, 5, 51, 79, 239, 389, 3, 601, 3787, 7635, 16295, 1681, 63971}},
	{3882, 16, 3662, {1, 3, 1, 3, 5, 31, 103, 89, 321, 971, 783, 3685, 1155, 10353, 2167, 35423}},
	{3883, 16, 3674, {1, 1, 5, 15, 25, 19, 93, 59, 361, 217, 1141, 597, 5877, 15961, 1593, 22925}},
	{3884, 16, 3685, {1, 3, 1, 9, 25, 59, 69, 89, 477, 89, 487, 237, 5625, 9579, 30421, 21883}},
	{3885, 16, 3686, {1, 1, 3, 7, 1, 5, 13, 225, 9, 981, 1081, 1407, 6855, 15215, 21713, 62313}},
	{3886, 16, 3700, {1, 1, 7, 15, 11, 13, 119, 109, 151, 245, 1195, 3741, 755, 8047, 15431, 21001}},
	{3887, 16, 3703, {1, 3, 7, 3, 17, 47, 107, 137, 99, 255, 1597, 3281, 5779, 13487, 15061, 19199}},
	{3888, 16, 3704, {1, 1, 3, 3, 9, 39, 77, 227, 511, 839, 1375, 3887, 25, 14763, 13259, 217}},
	{3889, 16, 3723, {1, 3, 5, 7, 17, 3, 87, 61, 439, 287, 709, 4085, 4251, 8945, 28203, 24011}},
	{3890, 16, 3743, {1, 3, 1, 1, 29, 25, 49, 101, 209, 359, 285, 1593, 4161, 2943, 23225, 6381}},
	{3891, 16, 3753, {1, 1, 3, 13, 1, 45, 87, 7, 491, 399, 905, 1403, 4791, 7419, 14355, 47767}},
	{3892, 16, 3756, {1, 1, 7, 15, 13, 25, 111, 197, 297, 301, 499, 4007, 2235, 7681, 4641, 32447}},
	{3893, 16, 3759, {1, 1, 3, 3, 27, 41, 97, 83, 405, 353, 1609, 201, 1503, 10673, 29377, 20445}},
	{3894, 16, 3762, {1, 1, 7, 3, 9, 47, 65, 191, 207, 545, 377, 3011, 7361, 3467, 14073, 46769}},
	{3895, 16, 3771, {1, 1, 7, 5, 7, 39, 9, 91, 187, 949, 1829, 161, 3689, 4145, 32675, 23263}},
	{3896, 16, 3776, {1, 1, 5, 9, 29, 9, 83, 113, 77, 673, 613, 3645, 6671, 8583, 27701, 18615}},
	{3897, 16, 3779, {1, 3, 5, 9, 29, 13, 127, 247, 285, 845, 463, 539, 4441, 1867, 12469, 16213}},
	{3898, 16, 3839, {1, 3, 7, 15, 1, 29, 47, 157, 239, 595, 563, 1103, 3431, 2849, 28125, 19969}},
	{3899, 16, 3856, {1, 1, 1, 15, 25, 13, 1, 131, 57, 257, 2021, 169, 7603, 10721, 21675, 63171}},
	{3900, 16, 3871, {1, 3, 5, 3, 5, 19, 31, 57, 275, 381, 775, 681, 1145, 12237, 5141, 29375}},
	{3901, 16, 3887, {1, 3, 5, 13, 27, 13, 47, 201, 267, 581, 1563, 3845, 951, 7209, 27253, 19755}},
	{3902, 16, 3896, {1, 3, 5, 15, 19, 35, 57, 17, 61, 273, 967, 3029, 1747, 1753, 31321, 23711}},
	{3903, 16, 3901, {1, 1, 1, 5, 13, 13, 7, 177, 335, 393, 1401, 1411, 4703, 8259, 1281, 39835}},
	{3904, 16, 3916, {1, 1, 3, 15, 25, 27, 27, 121, 183, 105, 663, 1375, 6987, 7151, 13763, 39323}},
	{3905, 16, 3919, {1, 3, 7, 5, 15, 1, 81, 129, 455, 163, 675, 81, 3735, 14409, 7269, 16425}},
	{3906, 16, 3937, {1, 3, 3, 11, 13, 7, 79, 157, 165, 663, 229, 3539, 1837, 6485, 30729, 42157}},
	{3907, 16, 3943, {1, 1, 5, 15, 9, 9, 9, 47, 133, 863, 43, 1461, 511, 13991, 24781, 19221}},
	{3908, 16, 3955, {1, 3, 1, 7, 31, 33, 103, 13, 159, 689, 1353, 4025, 6051, 7683, 1741, 30047}},
	{3909, 16, 3961, {1, 1, 3, 11, 5, 45, 71, 219, 475, 585, 1207, 3163, 4661, 4713, 12729, 30445}},
	{3910, 16, 3988, {1, 3, 7, 5, 5, 53, 101, 227, 129, 521, 91, 1129, 4683, 11235, 24697, 45055}},
	{3911, 16, 3997, {1, 1, 3, 13, 1, 43, 7, 1, 73, 857, 1713, 185, 1685, 2369, 24187, 40419}},
	{3912, 16, 4011, {1, 1, 7, 7, 21, 7, 13, 177, 503, 1003, 1091, 2411, 1433, 9063, 13901, 3329}},
	{3913, 16, 4026, {1, 1, 7, 1, 7, 41, 99, 203, 325, 249, 1763, 545, 2981, 14125, 7815, 11385}},
	{3914, 16, 4033, {1, 3, 7, 11, 3, 11, 95, 137, 325, 701, 1177, 1631, 4483, 2955, 30229, 25577}},
	{3915, 16, 4045, {1, 1, 7, 7, 17, 45, 77, 103, 143, 97, 1963, 3635, 1539, 10491, 23483, 22767}},
	{3916, 16, 4060, {1, 1, 7, 15, 7, 5, 81, 63, 243, 55, 39, 207, 2315, 8285, 8155, 11631}},
	{3917, 16, 4063, {1, 3, 5, 15, 23, 19, 115, 9, 125, 851, 161, 3767, 3513, 1855, 11139, 1719}},
	{3918, 16, 4064, {1, 3, 7, 11, 11, 23, 15, 13, 235, 5, 1039, 1425, 6485, 5539, 8967, 64809}},
	{3919, 16, 4126, {1, 3, 5, 7, 19, 11, 83, 135, 45, 905, 1081, 1857, 3185, 13555, 21365, 38143}},
	{3920, 16, 4136, {1, 1, 5, 1, 25, 27, 119, 109, 167, 847, 1539, 2653, 797, 11185, 23501, 22389}},
	{3921, 16, 4167, {1, 1, 7, 7, 11, 3, 51, 97, 277, 557, 207, 3645, 825, 8521, 26653, 60071}},
	{3922, 16, 4173, {1, 3, 3, 15, 17, 35, 57, 7, 267, 549, 97, 243, 1137, 10311, 6737, 19077}},
	{3923, 16, 4188, {1, 1, 1, 15, 23, 33, 27, 203, 415, 1023, 1145, 1881, 7715, 4413, 3727, 5185}},
	{3924, 16, 4195, {1, 1, 3, 3, 13, 47, 63, 13, 75, 505, 595, 2911, 4029, 14187, 23151, 42877}},
	{3925, 16, 4226, {1, 1, 5, 15, 23, 5, 11, 65, 147, 675, 1961, 2177, 727, 15077, 23759, 10195}},
	{3926, 16, 4291, {1, 3, 5, 9, 9, 39, 69, 229, 341, 627, 1331, 3139, 3921, 9219, 14887, 4659}},
	{3927, 16, 4298, {1, 1, 7, 3, 1, 35, 49, 71, 165, 83, 719, 2771, 6475, 7821, 16709, 4449}},
	{3928, 16, 4308, {1, 3, 5, 5, 23, 15, 3, 57, 465, 77, 121, 3767, 6841, 13601, 12035, 14075}},
	{3929, 16, 4312, {1, 1, 7, 3, 3, 23, 45, 131, 287, 941, 713, 415, 6865, 14209, 29555, 55493}},
	{3930, 16, 4336, {1, 3, 5, 11, 29, 35, 55, 75, 225, 779, 569, 1795, 1377, 12765, 19081, 47287}},
	{3931, 16, 4371, {1, 3, 7, 3, 31, 47, 127, 89, 157, 737, 1395, 3615, 7923, 14731, 15797, 40061}},
	{3932, 16, 4378, {1, 1, 1, 11, 21, 37, 21, 59, 9, 141, 193, 3095, 3435, 12371, 26931, 61861}},
	{3933, 16, 4384, {1, 1, 3, 7, 13, 51, 15, 153, 77, 1013, 651, 3949, 6229, 14297, 1039, 46139}},
	{3934, 16, 4393, {1, 3, 3, 13, 7, 43, 95, 61, 217, 3, 549, 739, 123, 3661, 15375, 13919}},
	{3935, 16, 4421, {1, 3, 5, 9, 13, 37, 101, 89, 55, 413, 1089, 775, 7575, 13063, 31393, 29583}},
	{3936, 16, 4425, {1, 1, 3, 9, 25, 63, 119, 143, 499, 145, 603, 2067, 4713, 13457, 14053, 117}},
	{3937, 16, 4439, {1, 1, 5, 9, 7, 23, 57, 253, 115, 591, 2003, 63, 7615, 11493, 28519, 47087}},
	{3938, 16, 4440, {1, 1, 7, 3, 7, 53, 121, 33, 233, 645, 1093, 1697, 7213, 2603, 10743, 51303}},
	{3939, 16, 4500, {1, 3, 5, 7, 13, 31, 17, 125, 93, 969, 159, 1529, 7165, 7371, 8707, 56953}},
	{3940, 16, 4514, {1, 3, 3, 1, 13, 9, 91, 25, 171, 843, 1635, 2043, 1043, 15893, 11409, 53689}},
	{3941, 16, 4523, {1, 3, 5, 7, 13, 19, 89, 97, 203, 923, 1109, 2061, 463, 11703, 8925, 56015}},
	{3942, 16, 4534, {1, 3, 5, 11, 5, 21, 79, 237, 195, 649, 717, 211, 919, 12855, 3045, 39659}},
	{3943, 16, 4593, {1, 1, 1, 15, 13, 19, 21, 69, 393, 257, 1263, 309, 3209, 8403, 24467, 6467}},
	{3944, 16, 4615, {1, 1, 1, 11, 7, 27, 59, 117, 379, 353, 943, 2513, 3869, 4567, 12989, 13139}},
	{3945, 16, 4630, {1, 1, 1, 3, 13, 43, 11, 15, 149, 237, 1555, 71, 2357, 15773, 21419, 40571}},
	{3946, 16, 4636, {1, 3, 1, 9, 19, 23, 59, 215, 15, 921, 1729, 249, 3785, 7171, 1233, 3449}},
	{3947, 16, 4645, {1, 1, 1, 7, 7, 37, 63, 205, 75, 599, 951, 2513, 3347, 2497, 8019, 5433}},
	{3948, 16, 4684, {1, 3, 3, 15, 27, 17, 25, 201, 23, 699, 1525, 465, 1115, 12299, 14747, 40363}},
	{3949, 16, 4687, {1, 1, 1, 3, 29, 59, 115, 233, 107, 815, 291, 3821, 7325, 7381, 21445, 33917}},
	{3950, 16, 4723, {1, 3, 1, 11, 11, 33, 107, 171, 421, 893, 587, 3373, 4101, 3885, 25383, 12035}},
	{3951, 16, 4735, {1, 3, 3, 7, 5, 23, 43, 51, 357, 77, 1327, 2995, 1321, 1571, 26419, 23603}},
	{3952, 16, 4746, {1, 3, 7, 9, 27, 57, 101, 51, 215, 215, 469, 303, 723, 2903, 30569, 42631}},
	{3953, 16, 4779, {1, 3, 3, 13, 1, 7, 63, 205, 143, 321, 1439, 253, 2667, 1271, 11761, 55631}},
	{3954, 16, 4782, {1, 1, 7, 9, 3, 7, 7, 15, 503, 875, 1619, 1715, 5047, 5665, 5503, 17745}},
	{3955, 16, 4793, {1, 1, 7, 15, 19, 49, 65, 31, 245, 371, 377, 2963, 6185, 5519, 10743, 33231}},
	{3956, 16, 4796, {1, 1, 7, 3, 25, 27, 115, 51, 299, 451, 285, 1709, 6153, 14881, 17861, 22071}},
	{3957, 16, 4813, {1, 3, 1, 5, 21, 21, 127, 185, 325, 995, 213, 3279, 4769, 15943, 2589, 29567}},
	{3958, 16, 4850, {1, 3, 7, 5, 21, 9, 63, 59, 159, 743, 663, 2965, 97, 8993, 25633, 29033}},
	{3959, 16, 4867, {1, 3, 7, 13, 3, 35, 59, 101, 21, 659, 1531, 3995, 795, 2143, 21749, 52715}},
	{3960, 16, 4874, {1, 3, 3, 15, 27, 29, 95, 1, 501, 425, 417, 2351, 7877, 4127, 3633, 23347}},
	{3961, 16, 4881, {1, 3, 5, 7, 7, 49, 55, 19, 329, 467, 425, 1609, 6987, 16123, 26879, 42883}},
	{3962, 16, 4894, {1, 1, 1, 15, 17, 21, 13, 13, 85, 7, 677, 3739, 5491, 6299, 29957, 55765}},
	{3963, 16, 4904, {1, 1, 1, 7, 31, 21, 1, 5, 193, 659, 979, 3409, 3151, 6615, 7445, 8151}},
	{3964, 16, 4927, {1, 3, 1, 1, 11, 61, 27, 205, 263, 805, 955, 3469, 1233, 1609, 15329, 13353}},
	{3965, 16, 4929, {1, 3, 3, 9, 3, 29, 59, 75, 149, 557, 663, 3887, 3369, 3397, 10611, 9511}},
	{3966, 16, 4989, {1, 1, 7, 13, 29, 21, 101, 139, 99, 411, 569, 2343, 6901, 1685, 20599, 49543}},
	{3967, 16, 5000, {1, 3, 3, 15, 11, 3, 87, 89, 5, 293, 291, 1405, 1489, 9877, 32505, 32263}},
	{3968, 16, 5020, {1, 1, 5, 5, 19, 45, 89, 5, 381, 253, 1339, 707, 4645, 14177, 29441, 8965}},
	{3969, 16, 5036, {1, 3, 7, 15, 27, 45, 25, 177, 81, 229, 1339, 2143, 6547, 6841, 23449, 14813}},
	{3970, 16, 5041, {1, 1, 1, 3, 27, 23, 81, 157, 53, 513, 1435, 277, 2353, 3545, 21461, 51479}},
	{3971, 16, 5059, {1, 3, 1, 3, 3, 17, 75, 139, 283, 881, 1157, 2081, 937, 14549, 10215, 13053}},
	{3972, 16, 5074, {1, 1, 7, 9, 25, 27, 27, 133, 21, 559, 225, 613, 6931, 11785, 23413, 35757}},
	{3973, 16, 5090, {1, 1, 3, 13, 19, 9, 65, 49, 453, 779, 621, 1151, 1807, 13269, 6515, 17113}},
	{3974, 16, 5110, {1, 1, 1, 13, 21, 49, 39, 79, 119, 401, 903, 493, 3779, 7389, 29425, 28091}},
	{3975, 16, 5134, {1, 3, 1, 3, 23, 57, 59, 213, 463, 839, 1201, 1951, 5197, 13035, 29657, 35517}},
	{3976, 16, 5152, {1, 3, 7, 7, 3, 49, 29, 181, 367, 101, 1277, 3329, 3563, 10373, 29757, 62555}},
	{3977, 16, 5176, {1, 3, 1, 7, 31, 31, 117, 213, 373, 57, 1095, 2733, 3431, 3915, 7665, 44459}},
	{3978, 16, 5181, {1, 1, 7, 5, 9, 25, 47, 117, 355, 495, 1367, 2579, 5617, 787, 27655, 18885}},
	{3979, 16, 5204, {1, 1, 1, 3, 9, 39, 113, 87, 107, 477, 891, 2273, 4239, 7521, 147, 1737}},
	{3980, 16, 5218, {1, 1, 1, 3, 13, 61, 81, 225, 113, 441, 889, 1915, 3897, 15179, 4053, 5925}},
	{3981, 16, 5242, {1, 1, 5, 3, 15, 47, 59, 187, 475, 197, 1381, 33, 4605, 1487, 14359, 33769}},
	{3982, 16, 5253, {1, 3, 7, 15, 23, 45, 53, 215, 129, 465, 795, 53, 7077, 9517, 2663, 55397}},
	{3983, 16, 5260, {1, 1, 7, 13, 25, 49, 105, 255, 245, 153, 1093, 2123, 2823, 5125, 17483, 49003}},
	{3984, 16, 5281, {1, 1, 1, 13, 31, 5, 7, 243, 255, 231, 1663, 1007, 7573, 405, 29183, 11367}},
	{3985, 16, 5282, {1, 1, 5, 13, 15, 15, 115, 91, 63, 1013, 1713, 1945, 6397, 14213, 24417, 40807}},
	{3986, 16, 5313, {1, 1, 5, 3, 19, 49, 91, 25, 43, 601, 25, 2405, 1973, 629, 497, 12625}},
	{3987, 16, 5316, {1, 1, 3, 5, 13, 45, 11, 81, 251, 545, 1155, 1409, 7187, 847, 2835, 32909}},
	{3988, 16, 5326, {1, 3, 1, 13, 27, 57, 1, 103, 465, 809, 1727, 3721, 3347, 3791, 17247, 8377}},
	{3989, 16, 5340, {1, 3, 3, 15, 25, 31, 91, 91, 119, 205, 1431, 703, 5327, 7323, 30415, 61955}},
	{3990, 16, 5347, {1, 3, 5, 11, 19, 39, 79, 243, 109, 463, 1869, 2133, 4139, 10461, 14793, 24025}},
	{3991, 16, 5354, {1, 3, 5, 7, 23, 41, 5, 7, 249, 3, 1743, 489, 4921, 397, 30955, 22207}},
	{3992, 16, 5368, {1, 3, 5, 15, 3, 7, 115, 19, 217, 231, 1183, 3723, 5055, 12967, 7855, 5067}},
	{3993, 16, 5394, {1, 3, 3, 3, 11, 31, 113, 41, 429, 797, 557, 1199, 1819, 1933, 9917, 32229}},
	{3994, 16, 5396, {1, 1, 5, 3, 13, 63, 31, 183, 465, 915, 723, 3227, 4125, 2813, 26313, 34287}},
	{3995, 16, 5400, {1, 1, 7, 5, 31, 55, 117, 243, 37, 885, 85, 1067, 3895, 15655, 28527, 32109}},
	{3996, 16, 5405, {1, 3, 7, 15, 17, 43, 43, 173, 119, 187, 1161, 599, 4595, 1681, 11981, 681}},
	{3997, 16, 5439, {1, 1, 7, 7, 29, 47, 25, 93, 411, 103, 783, 1029, 1927, 3569, 27647, 8281}},
	{3998, 16, 5442, {1, 3, 3, 9, 19, 57, 31, 183, 141, 889, 157, 2267, 5701, 6273, 25739, 34039}},
	{3999, 16, 5459, {1, 3, 5, 1, 29, 35, 105, 165, 505, 299, 1149, 2397, 2013, 11591, 15917, 4791}},
	{4000, 16, 5478, {1, 3, 3, 9, 7, 35, 47, 77, 69, 335, 585, 1131, 531, 8597, 307, 55985}},
	{4001, 16, 5484, {1, 3, 7, 1, 29, 9, 25, 155, 149, 845, 567, 3735, 3501, 9365, 12025, 19131}},
	{4002, 16, 5508, {1, 3, 5, 3, 11, 31, 25, 9, 411, 519, 1611, 1441, 1487, 10049, 14373, 24605}},
	{4003, 16, 5523, {1, 3, 3, 5, 3, 7, 101, 107, 339, 155, 1843, 2529, 443, 8177, 28655, 8151}},
	{4004, 16, 5545, {1, 1, 7, 5, 29, 37, 73, 131, 125, 451, 947, 797, 5053, 10155, 30801, 27235}},
	{4005, 16, 5565, {1, 1, 7, 13, 19, 47, 101, 45, 495, 457, 1293, 1971, 5495, 12737, 17687, 26123}},
	{4006, 16, 5566, {1, 1, 7, 7, 11, 11, 75, 177, 251, 553, 1883, 3379, 1429, 12227, 10301, 16467}},
	{4007, 16, 5580, {1, 3, 3, 9, 3, 61, 95, 35, 97, 551, 233, 2045, 4873, 9109, 10019, 64523}},
	{4008, 16, 5608, {1, 3, 1, 5, 11, 3, 15, 177, 301, 573, 2029, 191, 5551, 12083, 27287, 57235}},
	{4009, 16, 5613, {1, 3, 5, 1, 21, 9, 121, 169, 347, 187, 57, 3163, 5609, 1921, 17581, 28351}},
	{4010, 16, 5647, {1, 3, 3, 1, 31, 51, 15, 45, 429, 245, 573, 1595, 5343, 7519, 17009, 1299}},
	{4011, 16, 5661, {1, 1, 7, 3, 13, 47, 109, 65, 389, 867, 963, 145, 1089, 9749, 19625, 43121}},
	{4012, 16, 5671, {1, 3, 1, 7, 21, 61, 77, 67, 143, 579, 625, 2007, 6343, 4259, 21233, 237}},
	{4013, 16, 5678, {1, 3, 5, 9, 27, 15, 107, 91, 399, 895, 645, 2301, 439, 6789, 18299, 47285}},
	{4014, 16, 5680, {1, 3, 3, 5, 13, 11, 43, 199, 505, 409, 25, 2057, 479, 6031, 9561, 51613}},
	{4015, 16, 5685, {1, 1, 7, 13, 15, 55, 105, 53, 171, 925, 1849, 2881, 6951, 13069, 865, 41019}},
	{4016, 16, 5689, {1, 3, 1, 9, 17, 31, 45, 23, 411, 185, 189, 2123, 2583, 12713, 12681, 2231}},
	{4017, 16, 5692, {1, 3, 7, 9, 3, 63, 11, 253, 177, 127, 545, 3293, 4449, 15995, 10223, 33529}},
	{4018, 16, 5724, {1, 1, 5, 11, 13, 7, 53, 161, 421, 551, 697, 627, 3879, 1639, 24419, 3337}},
	{4019, 16, 5745, {1, 1, 7, 7, 27, 7, 37, 205, 429, 407, 1133, 3563, 2921, 6173, 11173, 3009}},
	{4020, 16, 5755, {1, 3, 3, 15, 31, 39, 117, 81, 337, 729, 567, 2299, 1481, 3189, 1795, 40299}},
	{4021, 16, 5757, {1, 3, 5, 15, 15, 47, 91, 127, 275, 55, 951, 3423, 2879, 6115, 1549, 2287}},
	{4022, 16, 5786, {1, 3, 3, 11, 17, 3, 127, 207, 141, 889, 185, 1095, 4567, 1371, 30545, 54445}},
	{4023, 16, 5792, {1, 1, 7, 3, 25, 11, 11, 139, 43, 1, 1977, 397, 5775, 6913, 13249, 46767}},
	{4024, 16, 5810, {1, 1, 7, 7, 27, 13, 31, 251, 191, 1015, 161, 3719, 5321, 13013, 25187, 51881}},
	{4025, 16, 5824, {1, 1, 1, 1, 3, 3, 13, 19, 423, 349, 1955, 2103, 6395, 3315, 23809, 25925}},
	{4026, 16, 5869, {1, 3, 5, 13, 3, 59, 41, 167, 423, 93, 1299, 2623, 5829, 8537, 8701, 43757}},
	{4027, 16, 5872, {1, 3, 5, 11, 9, 19, 127, 119, 329, 819, 7, 3809, 5305, 3643, 27369, 61827}},
	{4028, 16, 5895, {1, 3, 1, 15, 25, 43, 55, 159, 205, 911, 983, 2825, 3751, 7845, 12023, 18431}},
	{4029, 16, 5923, {1, 3, 3, 13, 11, 1, 65, 133, 479, 181, 679, 981, 3317, 6241, 11693, 9619}},
	{4030, 16, 5925, {1, 3, 3, 3, 21, 25, 117, 183, 127, 73, 703, 1469, 257, 11229, 10167, 50847}},
	{4031, 16, 5926, {1, 1, 5, 13, 5, 5, 113, 15, 231, 269, 989, 465, 3267, 15239, 29503, 42855}},
	{4032, 16, 5944, {1, 3, 3, 15, 9, 63, 79, 27, 21, 709, 1969, 3761, 1015, 13619, 4205, 40591}},
	{4033, 16, 5986, {1, 1, 7, 11, 29, 3, 5, 45, 107, 131, 1287, 3551, 849, 2003, 27451, 47103}},
	{4034, 16, 6012, {1, 3, 5, 11, 3, 47, 59, 53, 369, 249, 915, 2475, 7539, 763, 7063, 63971}},
	{4035, 16, 6015, {1, 1, 5, 1, 7, 53, 45, 127, 321, 341, 635, 2277, 1383, 10951, 29055, 45087}},
	{4036, 16, 6046, {1, 3, 7, 3, 5, 1, 119, 79, 487, 93, 25, 491, 4085, 6403, 27779, 8753}},
	{4037, 16, 6049, {1, 1, 1, 3, 9, 59, 49, 141, 323, 703, 1175, 423, 4323, 10083, 4289, 28931}},
	{4038, 16, 6061, {1, 3, 3, 15, 31, 31, 73, 15, 187, 653, 91, 1707, 1431, 9861, 19071, 8137}},
	{4039, 16, 6067, {1, 1, 1, 5, 27, 63, 93, 1, 329, 353, 863, 473, 7681, 10653, 15819, 8495}},
	{4040, 16, 6099, {1, 1, 1, 5, 25, 57, 119, 167, 219, 319, 231, 1065, 6217, 5131, 1513, 49281}},
	{4041, 16, 6121, {1, 3, 7, 3, 17, 3, 113, 91, 201, 179, 1907, 3423, 821, 12927, 24827, 49403}},
	{4042, 16, 6155, {1, 1, 5, 7, 19, 63, 75, 151, 387, 489, 777, 2049, 1151, 1351, 25687, 4143}},
	{4043, 16, 6163, {1, 3, 5, 7, 9, 37, 9, 3, 87, 385, 1667, 2139, 7527, 16133, 30023, 28783}},
	{4044, 16, 6203, {1, 1, 5, 9, 11, 55, 95, 73, 413, 867, 589, 2901, 3021, 413, 5955, 38921}},
	{4045, 16, 6208, {1, 3, 5, 15, 1, 17, 17, 7, 485, 323, 519, 2325, 2255, 4211, 20661, 28931}},
	{4046, 16, 6231, {1, 1, 5, 13, 21, 19, 85, 189, 167, 645, 1475, 3095, 7123, 3351, 7961, 20967}},
	{4047, 16, 6241, {1, 1, 7, 13, 3, 47, 13, 213, 237, 291, 285, 1465, 1765, 12361, 32651, 54205}},
	{4048, 16, 6254, {1, 3, 7, 13, 13, 27, 71, 35, 67, 117, 647, 2359, 3295, 8445, 24761, 29379}},
	{4049, 16, 6262, {1, 1, 1, 3, 3, 19, 23, 37, 5, 1019, 5, 1605, 2291, 14015, 9311, 39751}},
	{4050, 16, 6266, {1, 3, 3, 3, 31, 1, 65, 159, 221, 675, 1061, 971, 2333, 8265, 14361, 3263}},
	{4051, 16, 6275, {1, 1, 3, 7, 3, 5, 81, 17, 101, 991, 753, 2883, 4977, 4409, 1757, 26803}},
	{4052, 16, 6278, {1, 1, 5, 9, 13, 25, 45, 43, 199, 967, 829, 713, 4547, 7223, 6497, 53895}},
	{4053, 16, 6292, {1, 1, 7, 5, 23, 15, 89, 179, 509, 147, 315, 133, 111, 15577, 23427, 5229}},
	{4054, 16, 6329, {1, 3, 3, 7, 27, 7, 113, 65, 315, 135, 1309, 1179, 5755, 7513, 6815, 5137}},
	{4055, 16, 6355, {1, 1, 3, 7, 1, 13, 29, 155, 477, 721, 71, 865, 3897, 3331, 30641, 65471}},
	{4056, 16, 6357, {1, 1, 7, 3, 29, 45, 83, 3, 461, 109, 1545, 1365, 6633, 16137, 23859, 5995}},
	{4057, 16, 6374, {1, 3, 1, 1, 3, 33, 77, 83, 459, 187, 879, 3731, 6939, 6707, 23409, 24245}},
	{4058, 16, 6380, {1, 3, 5, 5, 13, 43, 127, 41, 29, 735, 1391, 2947, 4873, 4627, 15, 41719}},
	{4059, 16, 6423, {1, 3, 1, 3, 17, 17, 51, 93, 189, 227, 449, 2809, 825, 2009, 9563, 41435}},
	{4060, 16, 6427, {1, 3, 3, 11, 25, 47, 113, 173, 141, 919, 677, 117, 5293, 815, 23749, 19789}},
	{4061, 16, 6430, {1, 1, 1, 13, 15, 61, 121, 223, 53, 389, 489, 1527, 4771, 8975, 8005, 14275}},
	{4062, 16, 6436, {1, 1, 3, 15, 31, 57, 111, 145, 279, 991, 489, 3239, 7647, 473, 31279, 33447}},
	{4063, 16, 6443, {1, 1, 7, 5, 31, 13, 75, 185, 395, 611, 609, 159, 7931, 9887, 4115, 53121}},
	{4064, 16, 6445, {1, 3, 5, 5, 13, 39, 103, 237, 77, 913, 511, 1583, 6763, 14523, 4247, 63403}},
	{4065, 16, 6458, {1, 1, 1, 15, 11, 5, 43, 43, 297, 827, 747, 5, 3785, 15021, 11291, 36743}},
	{4066, 16, 6478, {1, 1, 7, 9, 9, 53, 113, 95, 403, 53, 1335, 4033, 8147, 11963, 6523, 23675}},
	{4067, 16, 6490, {1, 1, 5, 9, 27, 29, 69, 79, 327, 409, 1147, 1579, 2625, 12227, 30933, 9057}},
	{4068, 16, 6508, {1, 1, 1, 7, 1, 33, 29, 173, 5, 517, 437, 2035, 57, 12825, 22095, 65519}},
	{4069, 16, 6519, {1, 1, 3, 7, 27, 29, 53, 79, 429, 707, 589, 2605, 339, 7039, 19319, 17649}},
	{4070, 16, 6520, {1, 3, 3, 11, 9, 57, 43, 117, 39, 193, 1427, 2553, 6877, 7653, 29041, 44865}},
	{4071, 16, 6530, {1, 3, 3, 7, 23, 45, 29, 151, 265, 739, 365, 3565, 6447, 9761, 24021, 679}},
	{4072, 16, 6541, {1, 3, 5, 1, 1, 43, 73, 55, 131, 175, 959, 659, 7315, 15145, 18301, 14865}},
	{4073, 16, 6556, {1, 1, 3, 5, 15, 15, 91, 113, 359, 241, 1627, 1069, 1761, 211, 32671, 58833}},
	{4074, 16, 6607, {1, 3, 3, 7, 29, 27, 79, 53, 409, 81, 693, 3137, 7385, 11007, 28459, 28621}},
	{4075, 16, 6612, {1, 1, 7, 5, 29, 7, 67, 195, 77, 773, 1361, 2153, 4459, 7301, 5129, 17797}},
	{4076, 16, 6626, {1, 3, 3, 7, 25, 27, 91, 223, 115, 91, 1645, 2167, 1955, 9601, 22127, 13055}},
	{4077, 16, 6632, {1, 3, 7, 3, 27, 53, 67, 249, 51, 151, 663, 3231, 895, 6777, 3037, 56755}},
	{4078, 16, 6649, {1, 1, 5, 1, 25, 63, 71, 179, 375, 301, 1127, 2125, 783, 14481, 7293, 47883}},
	{4079, 16, 6666, {1, 1, 3, 9, 25, 3, 39, 69, 1, 85, 1271, 1571, 1953, 5077, 20369, 44827}},
	{4080, 16, 6674, {1, 3, 1, 13, 11, 61, 77, 59, 127, 475, 1609, 3553, 2553, 15589, 9351, 59787}},
	{4081, 16, 6733, {1, 3, 1, 5, 21, 7, 61, 27, 199, 653, 1243, 2481, 5369, 12903, 30229, 39453}},
	{4082, 16, 6782, {1, 3, 7, 3, 13, 15, 107, 153, 501, 573, 863, 3179, 6019, 15177, 16075, 43767}},
	{4083, 16, 6786, {1, 1, 7, 1, 23, 55, 17, 35, 5, 137, 1707, 1377, 6857, 15361, 27299, 61871}},
	{4084, 16, 6798, {1, 3, 5, 7, 27, 17, 87, 213, 95, 125, 1239, 3923, 4193, 11049, 12783, 45017}},
	{4085, 16, 6821, {1, 1, 5, 15, 9, 55, 11, 217, 7, 249, 369, 205, 4251, 13785, 24781, 48929}},
	{4086, 16, 6840, {1, 3, 7, 1, 15, 35, 33, 107, 503, 563, 1591, 3487, 7495, 1767, 24791, 31281}},
	{4087, 16, 6846, {1, 3, 1, 11, 3, 15, 47, 193, 289, 253, 909, 1105, 5119, 1415, 7737, 4341}},
	{4088, 16, 6865, {1, 1, 1, 3, 23, 33, 53, 187, 469, 573, 835, 2049, 791, 1177, 31051, 30955}},
	{4089, 16, 6884, {1, 3, 3, 11, 15, 51, 77, 143, 369, 991, 1103, 1293, 6019, 6361, 26301, 20741}},
	{4090, 16, 6891, {1, 1, 1, 5, 17, 19, 85, 135, 113, 593, 579, 1063, 7173, 2491, 9355, 19223}},
	{4091, 16, 6925, {1, 1, 5, 15, 25, 51, 107, 193, 31, 1, 1693, 125, 6223, 14619, 22683, 26321}},
	{4092, 16, 6938, {1, 1, 7, 1, 17, 45, 87, 39, 87, 499, 1185, 2763, 3989, 2863, 24555, 33817}},
	{4093, 16, 6967, {1, 3, 1, 11, 31, 5, 121, 231, 185, 793, 255, 2785, 5261, 3687, 21711, 3941}},
	{4094, 16, 6988, {1, 3, 7, 15, 5, 59, 73, 227, 365, 937, 893, 2155, 4385, 14345, 6813, 28461}},
	{4095, 16, 6996, {1, 1, 5, 7, 7, 23, 7, 239, 431, 45, 1015, 1663, 1893, 5035, 24075, 2119}},
	{4096, 16, 7009, {1, 3, 5, 1, 3, 15, 63, 103, 119, 801, 1681, 3463, 6083, 6453, 11379, 8205}},
	{4097, 16, 7016, {1, 3, 7, 9, 21, 61, 9, 9, 433, 541, 603, 3905, 3787, 10187, 3643, 21319}},
	{4098, 16, 7030, {1, 3, 5, 3, 11, 1, 101, 243, 363, 559, 561, 1163, 455, 4657, 1147, 39961}},
	{4099, 16, 7043, {1, 3, 5, 13, 17, 37, 57, 47, 483, 571, 1579, 1137, 8125, 12271, 23279, 1615}},
	{4100, 16, 7045, {1, 3, 5, 1, 21, 5, 13, 155, 75, 705, 389, 2855, 6345, 2279, 12627, 49451}},
	{4101, 16, 7052, {1, 1, 3, 9, 15, 51, 73, 99, 445, 227, 1705, 2175, 8111, 9381, 31555, 48491}},
	{4102, 16, 7073, {1, 3, 3, 5, 9, 63, 107, 51, 461, 979, 1033, 421, 4807, 11707, 13261, 26015}},
	{4103, 16, 7142, {1, 1, 5, 3, 13, 53, 117, 249, 57, 851, 1391, 3245, 35, 16043, 24399, 63667}},
	{4104, 16, 7153, {1, 3, 1, 11, 23, 33, 57, 125, 385, 495, 871, 199, 4269, 2269, 22897, 23597}},
	{4105, 16, 7168, {1, 3, 5, 15, 29, 11, 77, 21, 479, 369, 723, 3721, 1121, 9463, 19775, 54525}},
	{4106, 16, 7174, {1, 3, 5, 7, 7, 45, 29, 153, 395, 223, 1917, 3713, 5087, 10827, 1383, 36823}},
	{4107, 16, 7183, {1, 3, 1, 3, 31, 19, 111, 55, 275, 923, 917, 2925, 673, 6579, 18783, 5137}},
	{4108, 16, 7195, {1, 3, 1, 15, 25, 31, 59, 255, 31, 697, 1751, 381, 299, 295, 14037, 40953}},
	{4109, 16, 7204, {1, 3, 1, 7, 15, 23, 69, 219, 351, 183, 1083, 2227, 6249, 9385, 13167, 2901}},
	{4110, 16, 7214, {1, 3, 7, 1, 5, 61, 117, 13, 67, 521, 41, 2929, 3937, 1215, 25593, 32627}},
	{4111, 16, 7222, {1, 3, 5, 1, 9, 47, 63, 39, 371, 657, 491, 2243, 4049, 10517, 12409, 40597}},
	{4112, 16, 7267, {1, 3, 7, 15, 17, 3, 77, 13, 275, 225, 487, 2055, 1293, 15927, 31773, 18275}},
	{4113, 16, 7269, {1, 1, 5, 13, 11, 57, 113, 27, 191, 363, 1341, 3487, 8031, 13801, 7563, 40675}},
	{4114, 16, 7279, {1, 1, 3, 3, 27, 31, 103, 143, 271, 305, 2033, 3623, 4219, 9307, 7501, 8959}},
	{4115, 16, 7293, {1, 1, 1, 13, 1, 3, 27, 97, 475, 143, 333, 2997, 1807, 4231, 27437, 59717}},
	{4116, 16, 7312, {1, 3, 7, 5, 5, 3, 69, 233, 309, 511, 1429, 1887, 2745, 4969, 17595, 5451}},
	{4117, 16, 7327, {1, 1, 7, 3, 23, 17, 115, 89, 265, 467, 257, 2027, 5331, 1195, 4451, 8621}},
	{4118, 16, 7334, {1, 1, 7, 13, 29, 35, 117, 155, 99, 327, 853, 3595, 2997, 10745, 21619, 26549}},
	{4119, 16, 7337, {1, 3, 3, 13, 1, 13, 75, 151, 67, 271, 1609, 1117, 4293, 4645, 12005, 55983}},
	{4120, 16, 7343, {1, 1, 1, 13, 1, 61, 101, 63, 161, 261, 1759, 567, 665, 2339, 9157, 55603}},
	{4121, 16, 7346, {1, 1, 7, 11, 25, 19, 71, 27, 255, 435, 227, 4087, 4309, 14903, 14513, 30207}},
	{4122, 16, 7372, {1, 3, 3, 3, 11, 41, 1, 67, 383, 403, 45, 1521, 1535, 3353, 27361, 7549}},
	{4123, 16, 7387, {1, 1, 1, 1, 13, 51, 31, 137, 147, 907, 19, 3639, 3739, 877, 15005, 60357}},
	{4124, 16, 7390, {1, 1, 3, 11, 11, 23, 29, 173, 105, 873, 1727, 2761, 2015, 7491, 17491, 41065}},
	{4125, 16, 7396, {1, 1, 5, 3, 31, 1, 119, 53, 11, 731, 393, 4031, 4421, 15715, 6431, 18089}},
	{4126, 16, 7423, {1, 1, 3, 5, 15, 37, 55, 147, 307, 521, 711, 3085, 5989, 8081, 23351, 35259}},
	{4127, 16, 7428, {1, 3, 5, 13, 21, 19, 47, 107, 447, 713, 1655, 2809, 4741, 2105, 9255, 103}},
	{4128, 16, 7437, {1, 3, 1, 3, 17, 47, 63, 125, 343, 763, 1777, 607, 5625, 9517, 7221, 27257}},
	{4129, 16, 7466, {1, 1, 7, 5, 31, 13, 67, 255, 41, 947, 99, 721, 7367, 11427, 1357, 12383}},
	{4130, 16, 7474, {1, 1, 7, 3, 23, 27, 73, 185, 189, 545, 1877, 3169, 5419, 15873, 29059, 23983}},
	{4131, 16, 7476, {1, 1, 3, 1, 5, 13, 81, 45, 79, 717, 819, 3539, 7561, 7319, 5113, 27273}},
	{4132, 16, 7483, {1, 3, 7, 9, 21, 25, 71, 247, 41, 583, 771, 3745, 1883, 5717, 755, 14549}},
	{4133, 16, 7518, {1, 1, 3, 7, 23, 25, 87, 143, 499, 515, 1753, 1229, 173, 10629, 30579, 29643}},
	{4134, 16, 7527, {1, 3, 1, 13, 29, 33, 31, 69, 503, 117, 1717, 101, 7647, 1567, 28677, 3079}},
	{4135, 16, 7545, {1, 3, 1, 15, 29, 45, 57, 81, 171, 813, 505, 3647, 3913, 5557, 2477, 42429}},
	{4136, 16, 7572, {1, 1, 5, 13, 21, 13, 81, 5, 471, 221, 1563, 1741, 7269, 16327, 22687, 5291}},
	{4137, 16, 7586, {1, 3, 5, 3, 23, 41, 107, 61, 95, 79, 467, 1533, 739, 6791, 26911, 20309}},
	{4138, 16, 7597, {1, 3, 7, 7, 3, 53, 71, 163, 459, 975, 687, 1115, 5241, 299, 26361, 38583}},
	{4139, 16, 7630, {1, 3, 1, 9, 3, 63, 7, 133, 421, 325, 823, 1175, 6201, 5707, 31539, 34645}},
	{4140, 16, 7653, {1, 3, 7, 5, 27, 27, 107, 239, 247, 257, 1367, 3685, 7839, 11693, 3237, 13085}},
	{4141, 16, 7657, {1, 1, 1, 3, 27, 41, 51, 69, 271, 809, 1453, 519, 1301, 2367, 637, 5267}},
	{4142, 16, 7671, {1, 3, 7, 13, 19, 17, 3, 69, 369, 447, 1685, 4075, 6143, 11387, 5411, 29825}},
	{4143, 16, 7672, {1, 1, 3, 1, 25, 61, 79, 163, 1, 905, 1969, 2735, 7709, 16121, 20441, 4543}},
	{4144, 16, 7715, {1, 3, 7, 5, 15, 29, 7, 245, 343, 803, 1719, 3993, 983, 2925, 10393, 6053}},
	{4145, 16, 7717, {1, 3, 1, 7, 17, 55, 63, 29, 441, 387, 885, 3269, 1977, 1927, 3657, 47043}},
	{4146, 16, 7732, {1, 1, 3, 1, 17, 59, 51, 9, 173, 327, 1185, 3241, 3785, 10907, 19429, 22209}},
	{4147, 16, 7735, {1, 1, 3, 13, 21, 57, 125, 5, 359, 437, 1231, 2441, 5813, 9991, 283, 52555}},
	{4148, 16, 7753, {1, 3, 1, 7, 15, 19, 39, 125, 405, 381, 1757, 4075, 5565, 2065, 295, 8867}},
	{4149, 16, 7811, {1, 3, 3, 11, 7, 33, 95, 19, 253, 141, 1093, 1787, 7495, 5229, 15923, 44157}},
	{4150, 16, 7817, {1, 3, 7, 15, 1, 49, 69, 163, 85, 345, 901, 2329, 8003, 9915, 2209, 33979}},
	{4151, 16, 7825, {1, 1, 1, 9, 23, 51, 125, 163, 257, 681, 565, 945, 6375, 8679, 5985, 28919}},
	{4152, 16, 7832, {1, 3, 5, 7, 11, 23, 123, 231, 377, 121, 1519, 2061, 2957, 14537, 17625, 37773}},
	{4153, 16, 7838, {1, 3, 5, 1, 17, 43, 89, 119, 455, 279, 1857, 3405, 5225, 13035, 6055, 30861}},
	{4154, 16, 7841, {1, 3, 7, 15, 31, 63, 25, 225, 3, 527, 355, 1435, 5763, 15203, 26579, 45659}},
	{4155, 16, 7844, {1, 1, 1, 3, 27, 43, 71, 193, 135, 5, 683, 925, 7023, 7711, 2807, 56003}},
	{4156, 16, 7859, {1, 1, 1, 11, 3, 3, 109, 29, 109, 683, 419, 3295, 1961, 5953, 8887, 1523}},
	{4157, 16, 7861, {1, 3, 3, 11, 17, 39, 19, 225, 103, 249, 81, 3875, 4515, 3969, 24745, 60031}},
	{4158, 16, 7873, {1, 1, 3, 3, 3, 23, 15, 149, 305, 399, 1347, 1001, 597, 10003, 22123, 29919}},
	{4159, 16, 7880, {1, 3, 5, 1, 23, 35, 123, 7, 283, 283, 759, 3061, 2011, 7771, 32201, 40667}},
	{4160, 16, 7897, {1, 3, 7, 15, 23, 5, 81, 51, 357, 79, 133, 285, 425, 7743, 13499, 18983}},
	{4161, 16, 7904, {1, 3, 3, 5, 17, 37, 75, 221, 473, 111, 335, 683, 7353, 2283, 13457, 61171}},
	{4162, 16, 7910, {1, 3, 1, 7, 13, 45, 13, 223, 149, 209, 727, 3553, 2573, 177, 855, 44139}},
	{4163, 16, 7960, {1, 1, 3, 15, 23, 5, 103, 139, 17, 29, 1961, 3021, 5745, 12963, 30669, 44171}},
	{4164, 16, 7969, {1, 3, 1, 1, 3, 33, 67, 203, 29, 785, 71, 1693, 4487, 10221, 24523, 51223}},
	{4165, 16, 7970, {1, 1, 5, 7, 7, 27, 17, 183, 229, 669, 1675, 3751, 3233, 10677, 7509, 52313}},
	{4166, 16, 7976, {1, 1, 5, 5, 25, 35, 21, 163, 483, 399, 195, 3465, 6349, 545, 18861, 31759}},
	{4167, 16, 7979, {1, 3, 1, 5, 15, 39, 13, 157, 71, 841, 447, 3625, 53, 12745, 2719, 27617}},
	{4168, 16, 8007, {1, 1, 5, 5, 3, 45, 113, 121, 263, 887, 709, 2189, 3811, 1409, 10451, 48777}},
	{4169, 16, 8041, {1, 1, 5, 15, 9, 41, 31, 95, 377, 215, 437, 3633, 433, 11935, 15283, 55451}},
	{4170, 16, 8047, {1, 1, 7, 7, 13, 23, 79, 3, 451, 409, 1103, 1771, 553, 3175, 28703, 49357}},
	{4171, 16, 8052, {1, 3, 1, 1, 13, 33, 95, 133, 419, 851, 955, 3985, 5869, 14219, 253, 46883}},
	{4172, 16, 8061, {1, 3, 3, 3, 23, 55, 91, 207, 281, 355, 361, 261, 6837, 4401, 25455, 25313}},
	{4173, 16, 8078, {1, 3, 5, 9, 27, 9, 107, 51, 69, 555, 835, 3541, 1827, 5737, 31225, 55619}},
	{4174, 16, 8128, {1, 1, 1, 11, 27, 51, 79, 85, 447, 447, 9, 2803, 377, 4347, 2183, 61257}},
	{4175, 16, 8146, {1, 1, 1, 3, 23, 21, 51, 217, 297, 135, 573, 689, 4947, 14143, 26247, 43061}},
	{4176, 16, 8162, {1, 3, 1, 7, 15, 13, 27, 151, 463, 707, 43, 3641, 4999, 3783, 9083, 22085}},
	{4177, 16, 8250, {1, 3, 3, 5, 3, 1, 15, 119, 343, 605, 105, 2939, 2259, 889, 21759, 34073}},
	{4178, 16, 8270, {1, 1, 1, 7, 3, 63, 103, 1, 235, 317, 263, 2701, 7331, 15921, 17295, 613}},
	{4179, 16, 8294, {1, 1, 7, 3, 19, 3, 5, 17, 105, 491, 755, 233, 5397, 12365, 16325, 59377}},
	{4180, 16, 8324, {1, 3, 3, 15, 15, 27, 37, 151, 481, 949, 1473, 233, 1925, 15471, 24957, 3241}},
	{4181, 16, 8351, {1, 1, 7, 5, 9, 61, 49, 91, 169, 179, 701, 3957, 473, 15087, 6109, 25083}},
	{4182, 16, 8357, {1, 3, 3, 11, 27, 43, 5, 33, 69, 705, 733, 2675, 2677, 4235, 11109, 15557}},
	{4183, 16, 8361, {1, 3, 1, 3, 17, 19, 101, 119, 289, 341, 1395, 563, 6859, 10359, 10077, 26905}},
	{4184, 16, 8364, {1, 1, 1, 15, 21, 47, 41, 145, 439, 583, 1755, 1977, 5235, 15961, 21315, 60577}},
	{4185, 16, 8393, {1, 1, 5, 3, 9, 59, 71, 143, 27, 1007, 313, 1567, 1685, 11063, 28889, 44253}},
	{4186, 16, 8396, {1, 1, 5, 5, 29, 29, 43, 127, 13, 585, 1245, 187, 2697, 8791, 19561, 6463}},
	{4187, 16, 8407, {1, 1, 3, 11, 29, 39, 127, 75, 23, 521, 421, 3115, 139, 5429, 23341, 58035}},
	{4188, 16, 8413, {1, 1, 3, 1, 27, 35, 27, 9, 185, 653, 887, 2715, 3775, 1753, 22105, 62095}},
	{4189, 16, 8414, {1, 1, 5, 5, 5, 63, 23, 31, 317, 1001, 1751, 1185, 7933, 525, 30501, 15565}},
	{4190, 16, 8432, {1, 1, 1, 5, 9, 27, 79, 91, 453, 995, 1041, 3213, 8027, 5855, 7435, 64079}},
	{4191, 16, 8435, {1, 1, 3, 11, 1, 51, 27, 195, 139, 41, 1891, 1437, 1033, 11671, 3235, 35083}},
	{4192, 16, 8441, {1, 3, 1, 3, 11, 25, 33, 249, 373, 497, 1631, 293, 3657, 10741, 15831, 59545}},
	{4193, 16, 8447, {1, 1, 1, 1, 15, 63, 13, 165, 113, 559, 1615, 3579, 1993, 1907, 22335, 47791}},
	{4194, 16, 8449, {1, 1, 3, 15, 13, 49, 63, 235, 31, 811, 1729, 221, 5143, 11547, 30029, 52003}},
	{4195, 16, 8456, {1, 1, 5, 13, 29, 61, 25, 221, 421, 221, 583, 373, 2341, 7493, 13981, 54141}},
	{4196, 16, 8485, {1, 1, 5, 11, 21, 49, 71, 249, 237, 369, 1273, 1067, 4411, 409, 7219, 52933}},
	{4197, 16, 8504, {1, 3, 1, 1, 13, 23, 53, 15, 137, 553, 401, 2247, 5591, 14021, 445, 20433}},
	{4198, 16, 8512, {1, 1, 7, 7, 7, 23, 19, 189, 119, 643, 847, 2123, 5343, 11839, 4575, 60461}},
	{4199, 16, 8532, {1, 1, 5, 5, 11, 19, 111, 219, 185, 499, 595, 723, 3519, 10891, 27603, 29261}},
	{4200, 16, 8551, {1, 3, 3, 1, 9, 13, 95, 227, 459, 133, 1535, 3481, 7187, 14797, 16511, 6737}},
	{4201, 16, 8560, {1, 1, 7, 7, 19, 57, 117, 7, 455, 941, 455, 797, 6313, 10071, 18651, 25013}},
	{4202, 16, 8566, {1, 3, 7, 3, 25, 25, 79, 19, 383, 971, 1625, 2803, 2461, 633, 32317, 48407}},
	{4203, 16, 8581, {1, 1, 7, 7, 25, 43, 93, 135, 155, 685, 2001, 3007, 7315, 15555, 30401, 36291}},
	{4204, 16, 8609, {1, 1, 1, 5, 13, 33, 123, 221, 341, 105, 1075, 3125, 5323, 14293, 29875, 52215}},
	{4205, 16, 8639, {1, 1, 3, 9, 29, 51, 25, 59, 171, 563, 1695, 2845, 6067, 10671, 2909, 33977}},
	{4206, 16, 8641, {1, 3, 3, 7, 25, 21, 39, 65, 485, 949, 1773, 2775, 6019, 14587, 6715, 54793}},
	{4207, 16, 8671, {1, 1, 7, 11, 17, 57, 125, 17, 111, 167, 289, 3939, 7357, 2289, 1717, 45225}},
	{4208, 16, 8695, {1, 1, 7, 7, 21, 55, 3, 139, 471, 747, 1437, 1353, 7657, 13133, 14193, 38191}},
	{4209, 16, 8701, {1, 3, 5, 7, 25, 57, 55, 17, 315, 159, 437, 933, 7493, 6025, 2775, 24287}},
	{4210, 16, 8711, {1, 1, 7, 1, 15, 45, 67, 191, 355, 681, 1763, 1237, 105, 1425, 26089, 42879}},
	{4211, 16, 8739, {1, 1, 5, 13, 13, 53, 25, 127, 103, 155, 935, 2561, 475, 4341, 30541, 43835}},
	{4212, 16, 8763, {1, 1, 5, 15, 27, 59, 99, 173, 189, 41, 105, 3287, 4071, 15005, 18301, 34487}},
	{4213, 16, 8778, {1, 1, 5, 11, 21, 9, 57, 115, 495, 561, 579, 397, 3049, 2007, 5041, 37345}},
	{4214, 16, 8783, {1, 1, 5, 11, 15, 11, 101, 241, 69, 483, 1007, 4069, 5221, 5323, 20177, 24397}},
	{4215, 16, 8785, {1, 1, 1, 7, 29, 15, 119, 161, 21, 517, 847, 2217, 4487, 4817, 24053, 21683}},
	{4216, 16, 8797, {1, 3, 1, 3, 3, 51, 85, 63, 345, 799, 1699, 3961, 7109, 3931, 28173, 46851}},
	{4217, 16, 8802, {1, 1, 5, 7, 15, 25, 97, 139, 331, 969, 1129, 2451, 3107, 12235, 12949, 29837}},
	{4218, 16, 8816, {1, 3, 7, 1, 19, 21, 51, 155, 295, 565, 29, 2107, 341, 14611, 15281, 50727}},
	{4219, 16, 8828, {1, 3, 1, 11, 3, 37, 13, 217, 429, 217, 301, 1217, 5655, 13845, 32465, 23559}},
	{4220, 16, 8837, {1, 3, 3, 9, 9, 57, 79, 231, 433, 703, 699, 3813, 7035, 5885, 19185, 52551}},
	{4221, 16, 8852, {1, 1, 1, 5, 19, 17, 31, 209, 49, 515, 279, 909, 5881, 2673, 23635, 23101}},
	{4222, 16, 8859, {1, 1, 5, 7, 3, 3, 119, 139, 245, 775, 1009, 1157, 1405, 9737, 17671, 62981}},
	{4223, 16, 8889, {1, 3, 7, 11, 17, 21, 105, 21, 67, 871, 233, 3607, 571, 9141, 9751, 28093}},
	{4224, 16, 8900, {1, 1, 3, 13, 5, 53, 91, 181, 143, 375, 1113, 705, 837, 10505, 11459, 57753}},
	{4225, 16, 8903, {1, 3, 5, 11, 9, 19, 107, 229, 305, 107, 1027, 691, 4677, 8987, 7931, 951}},
	{4226, 16, 8909, {1, 1, 7, 9, 9, 17, 39, 195, 103, 315, 517, 123, 7167, 7039, 3571, 40469}},
	{4227, 16, 8910, {1, 1, 1, 5, 1, 21, 121, 53, 427, 111, 717, 1065, 2843, 5873, 28411, 42443}},
	{4228, 16, 8922, {1, 1, 3, 11, 27, 7, 37, 255, 429, 363, 997, 2429, 6871, 1271, 29375, 62897}},
	{4229, 16, 8924, {1, 3, 3, 13, 23, 23, 123, 119, 213, 51, 771, 1603, 1621, 1497, 23667, 13443}},
	{4230, 16, 8955, {1, 1, 3, 13, 17, 21, 81, 17, 211, 815, 1751, 3875, 4001, 3927, 6185, 28753}},
	{4231, 16, 8958, {1, 3, 1, 5, 13, 41, 23, 187, 475, 353, 1307, 543, 5077, 3459, 20553, 29119}},
	{4232, 16, 8980, {1, 1, 1, 7, 1, 39, 3, 247, 375, 101, 81, 1515, 1079, 15307, 18865, 63115}},
	{4233, 16, 8994, {1, 1, 5, 9, 23, 7, 61, 45, 379, 553, 435, 1805, 4147, 2289, 22081, 40041}},
	{4234, 16, 9006, {1, 1, 7, 3, 17, 13, 107, 169, 119, 981, 1825, 3329, 7779, 12245, 28367, 6749}},
	{4235, 16, 9017, {1, 3, 3, 13, 29, 13, 93, 155, 331, 507, 1073, 279, 6615, 14077, 3005, 10171}},
	{4236, 16, 9032, {1, 1, 5, 7, 29, 23, 81, 181, 321, 921, 1531, 2607, 7291, 1255, 29889, 30095}},
	{4237, 16, 9040, {1, 1, 1, 5, 7, 1, 9, 231, 203, 559, 243, 3999, 3649, 7939, 14729, 34771}},
	{4238, 16, 9061, {1, 3, 7, 3, 17, 29, 79, 251, 305, 641, 1125, 1155, 7139, 6721, 43, 34927}},
	{4239, 16, 9066, {1, 1, 5, 13, 21, 39, 55, 103, 143, 487, 849, 1111, 1105, 8483, 5417, 10521}},
	{4240, 16, 9071, {1, 1, 5, 5, 19, 5, 111, 49, 95, 917, 843, 2539, 6831, 9019, 16045, 59363}},
	{4241, 16, 9076, {1, 3, 3, 11, 7, 45, 87, 51, 49, 615, 603, 1623, 5351, 11939, 21945, 40539}},
	{4242, 16, 9086, {1, 1, 5, 9, 9, 33, 113, 37, 163, 853, 1313, 4021, 965, 11465, 8573, 24425}},
	{4243, 16, 9104, {1, 3, 3, 9, 17, 19, 121, 95, 93, 441, 1951, 3335, 6279, 16087, 14763, 60771}},
	{4244, 16, 9150, {1, 3, 3, 9, 13, 15, 19, 129, 257, 641, 533, 1667, 5959, 2259, 10439, 29745}},
	{4245, 16, 9161, {1, 1, 7, 7, 11, 31, 45, 247, 233, 101, 899, 2033, 7803, 11423, 30645, 7723}},
	{4246, 16, 9164, {1, 3, 5, 11, 23, 3, 69, 79, 319, 125, 1463, 2047, 7311, 5819, 445, 13725}},
	{4247, 16, 9185, {1, 1, 1, 3, 7, 43, 83, 89, 467, 709, 1993, 3773, 7805, 305, 15701, 51101}},
	{4248, 16, 9188, {1, 1, 7, 5, 19, 53, 77, 237, 27, 853, 1003, 2041, 5739, 4663, 9783, 23761}},
	{4249, 16, 9212, {1, 1, 3, 7, 19, 31, 71, 33, 479, 693, 1503, 9, 2779, 1481, 9413, 36227}},
	{4250, 16, 9230, {1, 3, 1, 11, 9, 23, 1, 99, 247, 33, 1987, 1577, 8029, 7785, 29947, 38751}},
	{4251, 16, 9242, {1, 1, 1, 3, 15, 57, 23, 53, 431, 553, 1433, 2447, 1871, 10701, 30961, 12281}},
	{4252, 16, 9247, {1, 3, 5, 9, 11, 49, 123, 91, 87, 155, 301, 3339, 6183, 15895, 17309, 45927}},
	{4253, 16, 9260, {1, 1, 1, 9, 9, 41, 79, 123, 261, 547, 1931, 2553, 7405, 14431, 24079, 20769}},
	{4254, 16, 9280, {1, 3, 1, 3, 31, 17, 17, 137, 419, 591, 1693, 3977, 861, 16025, 189, 60995}},
	{4255, 16, 9300, {1, 3, 7, 11, 19, 47, 107, 243, 87, 135, 507, 189, 1397, 3841, 22999, 50781}},
	{4256, 16, 9313, {1, 3, 5, 5, 15, 11, 19, 239, 133, 295, 673, 2389, 4753, 4363, 21669, 25579}},
	{4257, 16, 9325, {1, 3, 5, 7, 19, 43, 55, 129, 239, 89, 1731, 1381, 5483, 11773, 9201, 17745}},
	{4258, 16, 9343, {1, 3, 1, 13, 3, 15, 77, 131, 417, 845, 1953, 2871, 1789, 10343, 11363, 20699}},
	{4259, 16, 9349, {1, 3, 7, 1, 9, 43, 55, 223, 239, 317, 1951, 2725, 209, 3853, 2201, 6839}},
	{4260, 16, 9354, {1, 3, 1, 3, 7, 35, 29, 21, 149, 779, 467, 65, 811, 4859, 14021, 38429}},
	{4261, 16, 9367, {1, 3, 7, 1, 19, 31, 97, 9, 11, 415, 689, 1513, 2475, 5039, 5669, 65103}},
	{4262, 16, 9368, {1, 3, 3, 3, 11, 25, 37, 247, 189, 911, 429, 2395, 3653, 79, 28115, 23513}},
	{4263, 16, 9455, {1, 1, 5, 5, 5, 27, 25, 45, 291, 455, 741, 2259, 8131, 11779, 14693, 58729}},
	{4264, 16, 9458, {1, 3, 3, 7, 21, 33, 67, 49, 153, 491, 1811, 1955, 925, 15555, 13801, 48717}},
	{4265, 16, 9469, {1, 3, 1, 3, 11, 53, 105, 129, 457, 225, 497, 1123, 973, 2901, 26655, 3643}},
	{4266, 16, 9481, {1, 1, 7, 13, 29, 49, 71, 37, 321, 865, 735, 357, 7629, 6011, 28221, 39041}},
	{4267, 16, 9482, {1, 3, 5, 3, 19, 59, 65, 199, 69, 391, 1735, 3075, 287, 16213, 3211, 22425}},
	{4268, 16, 9495, {1, 1, 1, 5, 15, 61, 67, 255, 5, 689, 759, 155, 7245, 5881, 21685, 3863}},
	{4269, 16, 9526, {1, 1, 3, 11, 23, 63, 69, 241, 359, 735, 371, 603, 2707, 15833, 31795, 14901}},
	{4270, 16, 9530, {1, 1, 1, 7, 19, 33, 83, 19, 13, 667, 317, 3891, 5497, 8213, 4913, 22387}},
	{4271, 16, 9558, {1, 3, 5, 9, 13, 21, 11, 187, 249, 647, 349, 605, 2199, 5033, 29773, 48953}},
	{4272, 16, 9562, {1, 3, 3, 11, 3, 3, 93, 235, 441, 933, 383, 2007, 4015, 4175, 3937, 20623}},
	{4273, 16, 9573, {1, 3, 7, 13, 3, 61, 87, 219, 263, 651, 1343, 2709, 31, 16249, 4749, 28909}},
	{4274, 16, 9583, {1, 3, 1, 1, 17, 19, 101, 107, 499, 127, 13, 2123, 5597, 3751, 14527, 12009}},
	{4275, 16, 9595, {1, 3, 5, 13, 27, 57, 1, 195, 107, 947, 1475, 2831, 6449, 16117, 20555, 61513}},
	{4276, 16, 9597, {1, 3, 1, 9, 9, 47, 89, 187, 401, 299, 637, 197, 1235, 12655, 25025, 24443}},
	{4277, 16, 9616, {1, 1, 3, 5, 9, 57, 33, 41, 451, 983, 391, 2707, 705, 13913, 28843, 34091}},
	{4278, 16, 9638, {1, 3, 5, 3, 29, 19, 61, 31, 349, 253, 1267, 3345, 2151, 11309, 19623, 62407}},
	{4279, 16, 9649, {1, 1, 1, 3, 11, 37, 31, 59, 1, 253, 103, 2811, 1871, 4239, 26311, 32507}},
	{4280, 16, 9662, {1, 1, 5, 7, 7, 7, 69, 63, 281, 901, 1785, 2131, 4265, 253, 13997, 30177}},
	{4281, 16, 9691, {1, 3, 1, 11, 3, 27, 111, 67, 411, 751, 241, 293, 6271, 4187, 22119, 63737}},
	{4282, 16, 9700, {1, 3, 5, 5, 27, 19, 45, 203, 81, 59, 1839, 935, 4847, 1869, 12037, 30971}},
	{4283, 16, 9703, {1, 1, 3, 9, 19, 25, 9, 27, 373, 875, 1735, 689, 2957, 7873, 29771, 4093}},
	{4284, 16, 9710, {1, 1, 7, 11, 13, 39, 11, 129, 53, 433, 1731, 899, 5855, 10221, 24165, 50205}},
	{4285, 16, 9721, {1, 3, 1, 15, 25, 31, 85, 49, 325, 299, 183, 287, 2425, 15353, 25999, 59129}},
	{4286, 16, 9724, {1, 1, 5, 5, 17, 25, 23, 5, 287, 677, 591, 981, 429, 15297, 14573, 61095}},
	{4287, 16, 9727, {1, 1, 5, 15, 5, 15, 67, 195, 209, 341, 1621, 1379, 3031, 5469, 31563, 49291}},
	{4288, 16, 9743, {1, 1, 1, 1, 25, 33, 61, 201, 15, 125, 969, 1965, 2021, 10253, 23801, 28165}},
	{4289, 16, 9779, {1, 1, 5, 5, 13, 17, 5, 245, 11, 133, 287, 1929, 4331, 15919, 29663, 10243}},
	{4290, 16, 9785, {1, 1, 7, 9, 19, 33, 39, 191, 489, 631, 69, 1883, 2183, 14993, 32715, 62217}},
	{4291, 16, 9811, {1, 3, 3, 13, 23, 61, 103, 193, 501, 431, 437, 417, 6557, 11701, 27577, 42943}},
	{4292, 16, 9820, {1, 3, 3, 9, 9, 27, 69, 247, 469, 841, 733, 813, 2673, 7145, 5385, 44917}},
	{4293, 16, 9827, {1, 1, 7, 9, 5, 13, 19, 71, 323, 923, 1885, 3031, 4507, 13787, 14149, 1483}},
	{4294, 16, 9851, {1, 3, 1, 5, 1, 15, 89, 229, 301, 733, 1343, 2415, 6463, 1875, 9293, 6037}},
	{4295, 16, 9854, {1, 3, 1, 5, 29, 57, 67, 121, 311, 441, 1079, 1963, 7137, 6745, 9893, 22811}},
	{4296, 16, 9863, {1, 1, 7, 7, 25, 13, 27, 61, 331, 361, 481, 3783, 3061, 7827, 18885, 27583}},
	{4297, 16, 9884, {1, 3, 1, 5, 17, 47, 19, 83, 309, 65, 1573, 3437, 5623, 12691, 21075, 27789}},
	{4298, 16, 9894, {1, 1, 7, 9, 13, 51, 7, 209, 131, 111, 1143, 53, 7277, 9297, 20869, 33121}},
	{4299, 16, 9903, {1, 1, 3, 9, 13, 17, 57, 89, 239, 281, 775, 333, 5631, 2805, 10195, 9665}},
	{4300, 16, 9908, {1, 1, 3, 7, 19, 39, 71, 79, 63, 551, 103, 3169, 2761, 13929, 20751, 18951}},
	{4301, 16, 9912, {1, 1, 7, 11, 5, 23, 7, 249, 447, 483, 433, 3117, 1421, 14991, 5397, 19813}},
	{4302, 16, 9925, {1, 3, 1, 13, 3, 9, 109, 241, 181, 33, 853, 3939, 3751, 2151, 28375, 64443}},
	{4303, 16, 9926, {1, 1, 7, 7, 3, 33, 65, 211, 251, 631, 1819, 3913, 3353, 12975, 19117, 51515}},
	{4304, 16, 9971, {1, 1, 1, 13, 3, 21, 9, 203, 223, 229, 1399, 117, 6297, 11535, 16383, 12541}},
	{4305, 16, 9973, {1, 1, 5, 7, 25, 9, 53, 27, 497, 103, 1915, 2179, 3679, 11375, 18907, 63385}},
	{4306, 16, 9977, {1, 3, 1, 5, 1, 53, 91, 169, 87, 387, 377, 1121, 7241, 5133, 1949, 13433}},
	{4307, 16, 10021, {1, 1, 1, 3, 27, 35, 61, 189, 241, 445, 287, 325, 127, 2363, 30663, 43557}},
	{4308, 16, 10039, {1, 3, 1, 3, 17, 47, 59, 237, 213, 979, 807, 85, 4621, 9915, 13631, 55657}},
	{4309, 16, 10048, {1, 3, 5, 7, 27, 5, 101, 89, 495, 675, 1181, 2295, 1913, 3731, 32639, 58297}},
	{4310, 16, 10051, {1, 3, 3, 11, 5, 41, 49, 229, 93, 659, 927, 3425, 4083, 11859, 10603, 20631}},
	{4311, 16, 10065, {1, 3, 5, 11, 31, 51, 67, 69, 253, 497, 1665, 1985, 5439, 15999, 4175, 62175}},
	{4312, 16, 10071, {1, 1, 7, 11, 1, 21, 95, 19, 205, 513, 1329, 1821, 1251, 2381, 32623, 23923}},
	{4313, 16, 10072, {1, 1, 5, 13, 3, 1, 29, 175, 315, 865, 599, 1695, 1391, 2313, 31035, 19159}},
	{4314, 16, 10101, {1, 3, 3, 1, 3, 45, 109, 93, 481, 285, 869, 3441, 3715, 1355, 9581, 50173}},
	{4315, 16, 10106, {1, 3, 7, 7, 15, 35, 107, 107, 315, 213, 281, 2073, 4689, 5877, 31, 40967}},
	{4316, 16, 10130, {1, 1, 3, 11, 11, 3, 73, 41, 79, 37, 481, 1993, 931, 7677, 11321, 45735}},
	{4317, 16, 10136, {1, 1, 7, 1, 15, 21, 65, 237, 263, 849, 863, 4039, 3171, 13381, 30373, 51639}},
	{4318, 16, 10148, {1, 1, 1, 3, 21, 57, 113, 3, 487, 41, 1277, 633, 2839, 4977, 14537, 31749}},
	{4319, 16, 10155, {1, 1, 5, 1, 1, 55, 71, 181, 147, 895, 1839, 2157, 3187, 6403, 30367, 48915}},
	{4320, 16, 10157, {1, 1, 5, 3, 9, 17, 19, 127, 115, 875, 831, 2439, 7475, 1921, 18657, 27709}},
	{4321, 16, 10160, {1, 3, 3, 13, 29, 11, 35, 81, 291, 483, 625, 3957, 6017, 12543, 18773, 2471}},
	{4322, 16, 10166, {1, 3, 3, 13, 11, 39, 7, 85, 493, 209, 819, 3277, 4275, 8997, 22943, 33199}},
	{4323, 16, 10183, {1, 1, 1, 7, 11, 25, 1, 57, 505, 135, 1713, 3051, 5893, 10711, 10681, 12235}},
	{4324, 16, 10192, {1, 3, 5, 11, 23, 33, 13, 107, 289, 251, 235, 1747, 4097, 12237, 17559, 5063}},
	{4325, 16, 10225, {1, 3, 3, 9, 19, 17, 23, 45, 297, 147, 1301, 2057, 7871, 9971, 1965, 62449}},
	{4326, 16, 10241, {1, 1, 7, 3, 17, 21, 19, 203, 289, 897, 1967, 3519, 3261, 8199, 24181, 23273}},
	{4327, 16, 10247, {1, 1, 7, 11, 1, 17, 25, 63, 509, 969, 47, 1329, 701, 5227, 419, 14839}},
	{4328, 16, 10284, {1, 3, 5, 11, 1, 41, 81, 157, 395, 97, 1919, 3043, 6015, 15, 23733, 55485}},
	{4329, 16, 10304, {1, 1, 3, 11, 17, 37, 17, 181, 179, 297, 1007, 1559, 6275, 11645, 7535, 28941}},
	{4330, 16, 10322, {1, 3, 7, 15, 5, 47, 31, 237, 215, 563, 207, 1867, 6635, 6857, 11837, 22865}},
	{4331, 16, 10331, {1, 3, 1, 7, 7, 39, 51, 179, 57, 987, 893, 2715, 1045, 5799, 19805, 54275}},
	{4332, 16, 10333, {1, 1, 7, 15, 25, 9, 127, 243, 323, 1013, 929, 2891, 7549, 1071, 17663, 15247}},
	{4333, 16, 10340, {1, 1, 1, 5, 25, 23, 101, 9, 371, 89, 1749, 3559, 8071, 13887, 14807, 42825}},
	{4334, 16, 10347, {1, 3, 3, 11, 21, 41, 3, 77, 3, 709, 1745, 3615, 4141, 5275, 28329, 59739}},
	{4335, 16, 10357, {1, 1, 7, 13, 1, 11, 73, 183, 363, 241, 863, 3983, 3235, 293, 649, 441}},
	{4336, 16, 10371, {1, 1, 5, 3, 13, 27, 13, 191, 57, 639, 1803, 2353, 3143, 12763, 5771, 36155}},
	{4337, 16, 10386, {1, 1, 5, 3, 1, 53, 85, 45, 283, 823, 1213, 3261, 4599, 13267, 4613, 12657}},
	{4338, 16, 10404, {1, 3, 5, 15, 27, 49, 59, 123, 357, 527, 337, 2751, 3999, 8525, 12501, 40621}},
	{4339, 16, 10414, {1, 1, 1, 7, 1, 55, 85, 17, 447, 599, 1315, 2313, 1649, 907, 25647, 3251}},
	{4340, 16, 10422, {1, 3, 5, 13, 9, 1, 37, 121, 143, 1, 631, 2273, 1673, 3649, 27533, 28731}},
	{4341, 16, 10448, {1, 1, 7, 13, 9, 31, 57, 249, 397, 815, 501, 895, 1217, 11387, 8635, 65193}},
	{4342, 16, 10451, {1, 1, 5, 5, 9, 35, 95, 193, 133, 513, 1929, 3841, 3063, 2383, 24413, 51185}},
	{4343, 16, 10473, {1, 1, 1, 13, 3, 49, 45, 191, 15, 181, 1819, 3741, 1227, 12775, 9461, 44951}},
	{4344, 16, 10479, {1, 1, 1, 1, 3, 45, 121, 19, 269, 829, 517, 3913, 183, 11019, 24969, 21973}},
	{4345, 16, 10501, {1, 1, 5, 11, 31, 39, 125, 189, 401, 57, 1557, 1727, 1415, 4025, 30353, 36589}},
	{4346, 16, 10530, {1, 1, 3, 9, 3, 55, 125, 187, 409, 499, 1853, 2781, 4323, 16159, 16345, 34659}},
	{4347, 16, 10542, {1, 3, 5, 11, 31, 5, 125, 137, 197, 475, 2003, 617, 1289, 8365, 28981, 57537}},
	{4348, 16, 10544, {1, 1, 1, 5, 19, 29, 83, 127, 311, 177, 1635, 2187, 5377, 12841, 7591, 6095}},
	{4349, 16, 10571, {1, 3, 5, 5, 21, 39, 65, 197, 115, 411, 1457, 3011, 7021, 14119, 61, 21107}},
	{4350, 16, 10628, {1, 3, 3, 9, 19, 57, 99, 23, 451, 507, 973, 415, 7123, 11367, 29767, 23763}},
	{4351, 16, 10643, {1, 1, 5, 7, 29, 23, 47, 11, 267, 873, 591, 373, 7097, 3783, 23289, 5547}},
	{4352, 16, 10673, {1, 1, 5, 15, 7, 7, 7, 91, 389, 841, 1995, 459, 7013, 3109, 23615, 21519}},
	{4353, 16, 10683, {1, 3, 1, 1, 13, 25, 87, 235, 193, 201, 713, 1633, 3117, 13609, 17211, 573}},
	{4354, 16, 10736, {1, 1, 1, 9, 27, 53, 105, 39, 217, 781, 997, 661, 6243, 6427, 17739, 62239}},
	{4355, 16, 10795, {1, 1, 7, 3, 3, 49, 75, 125, 239, 195, 215, 2983, 1185, 4743, 12069, 55509}},
	{4356, 16, 10797, {1, 1, 5, 15, 31, 11, 9, 91, 253, 361, 571, 1589, 2425, 4279, 3765, 46519}},
	{4357, 16, 10815, {1, 1, 3, 3, 21, 49, 49, 213, 399, 253, 1565, 2447, 453, 7957, 24799, 58503}},
	{4358, 16, 10817, {1, 1, 7, 1, 5, 59, 81, 97, 209, 477, 17, 3085, 3655, 599, 24011, 14981}},
	{4359, 16, 10842, {1, 3, 3, 13, 19, 49, 7, 35, 111, 169, 629, 1587, 5345, 13699, 21187, 30199}},
	{4360, 16, 10844, {1, 1, 3, 13, 19, 59, 73, 127, 475, 509, 9, 2661, 711, 15835, 31429, 33885}},
	{4361, 16, 10863, {1, 3, 5, 3, 31, 15, 43, 185, 29, 897, 1041, 1057, 6285, 13925, 4023, 25741}},
	{4362, 16, 10899, {1, 3, 5, 11, 1, 33, 63, 155, 175, 501, 1147, 3395, 3285, 16237, 22315, 50945}},
	{4363, 16, 10902, {1, 3, 3, 3, 5, 13, 77, 227, 85, 139, 139, 1, 7147, 2023, 32705, 38753}},
	{4364, 16, 10917, {1, 3, 5, 11, 31, 59, 35, 179, 489, 537, 1537, 2877, 4937, 10825, 2445, 34907}},
	{4365, 16, 10953, {1, 3, 7, 11, 17, 17, 95, 223, 165, 925, 829, 3971, 1, 7393, 8825, 25705}},
	{4366, 16, 10967, {1, 1, 1, 1, 25, 17, 25, 143, 385, 907, 1381, 1823, 3819, 8475, 5321, 12037}},
	{4367, 16, 11002, {1, 1, 5, 11, 7, 47, 97, 85, 105, 23, 263, 1329, 1905, 12121, 29635, 41249}},
	{4368, 16, 11024, {1, 1, 7, 11, 1, 51, 13, 13, 5, 143, 83, 3831, 959, 6083, 16997, 59887}},
	{4369, 16, 11029, {1, 3, 3, 13, 25, 9, 31, 5, 215, 791, 767, 1733, 2715, 14283, 25795, 54249}},
	{4370, 16, 11030, {1, 3, 7, 5, 11, 19, 125, 81, 71, 131, 1869, 1111, 6763, 5275, 23095, 1139}},
	{4371, 16, 11043, {1, 3, 3, 9, 25, 43, 119, 49, 133, 217, 521, 1367, 5867, 6829, 29871, 60383}},
	{4372, 16, 11087, {1, 1, 7, 9, 5, 11, 59, 157, 279, 301, 481, 3273, 7943, 3273, 27783, 17271}},
	{4373, 16, 11106, {1, 3, 3, 13, 11, 57, 13, 5, 435, 169, 541, 517, 2359, 9121, 27911, 15679}},
	{4374, 16, 11130, {1, 1, 3, 9, 9, 55, 65, 113, 21, 807, 373, 2825, 1527, 15565, 8339, 7227}},
	{4375, 16, 11156, {1, 3, 5, 9, 1, 1, 49, 255, 477, 821, 1647, 713, 6841, 3187, 22649, 51469}},
	{4376, 16, 11176, {1, 3, 3, 11, 13, 43, 63, 139, 71, 809, 993, 2429, 833, 6545, 10987, 39567}},
	{4377, 16, 11221, {1, 1, 1, 9, 19, 23, 47, 199, 191, 827, 391, 837, 1209, 2493, 24071, 46589}},
	{4378, 16, 11267, {1, 1, 5, 13, 25, 51, 39, 43, 103, 899, 1729, 2389, 2965, 189, 3063, 50609}},
	{4379, 16, 11282, {1, 1, 3, 1, 5, 29, 105, 201, 503, 199, 507, 205, 2389, 695, 15233, 50353}},
	{4380, 16, 11294, {1, 3, 1, 7, 19, 53, 45, 21, 89, 545, 1885, 765, 6673, 13485, 9987, 2609}},
	{4381, 16, 11332, {1, 3, 7, 13, 17, 7, 59, 23, 159, 309, 1407, 2483, 1807, 15733, 5603, 52989}},
	{4382, 16, 11353, {1, 1, 1, 11, 13, 19, 123, 185, 413, 745, 361, 2391, 6697, 2281, 11999, 13175}},
	{4383, 16, 11369, {1, 3, 5, 5, 11, 49, 123, 173, 325, 667, 895, 1067, 8121, 10577, 30561, 17391}},
	{4384, 16, 11372, {1, 1, 5, 5, 23, 21, 77, 223, 281, 161, 141, 345, 3879, 11393, 26907, 53805}},
	{4385, 16, 11375, {1, 3, 3, 5, 3, 47, 17, 109, 185, 139, 957, 1417, 4553, 6101, 29537, 34821}},
	{4386, 16, 11413, {1, 1, 5, 3, 29, 49, 89, 61, 45, 593, 269, 1483, 2971, 991, 21239, 29301}},
	{4387, 16, 11429, {1, 1, 3, 13, 29, 45, 33, 253, 291, 783, 737, 2363, 2651, 8627, 21785, 58419}},
	{4388, 16, 11430, {1, 3, 7, 15, 29, 15, 81, 185, 363, 681, 1737, 3789, 4365, 3295, 23205, 4457}},
	{4389, 16, 11444, {1, 3, 3, 11, 15, 39, 67, 167, 197, 357, 871, 2201, 5377, 6299, 20873, 59283}},
	{4390, 16, 11462, {1, 3, 3, 5, 9, 15, 127, 49, 21, 719, 357, 425, 5507, 9639, 275, 47503}},
	{4391, 16, 11465, {1, 1, 7, 1, 13, 63, 111, 121, 21, 481, 247, 3147, 5783, 8947, 20809, 42039}},
	{4392, 16, 11471, {1, 1, 3, 3, 31, 33, 9, 69, 187, 517, 2029, 2205, 7661, 4757, 27525, 9665}},
	{4393, 16, 11473, {1, 1, 1, 13, 7, 41, 103, 161, 233, 221, 693, 213, 4609, 7771, 28703, 17407}},
	{4394, 16, 11495, {1, 3, 7, 15, 31, 47, 27, 111, 479, 1003, 1687, 347, 2179, 11861, 8169, 31941}},
	{4395, 16, 11499, {1, 1, 3, 5, 5, 63, 25, 125, 199, 147, 589, 3565, 3449, 8331, 8923, 31207}},
	{4396, 16, 11519, {1, 1, 3, 11, 19, 25, 77, 99, 299, 19, 1641, 2193, 4299, 3635, 20621, 267}},
	{4397, 16, 11562, {1, 3, 7, 11, 9, 59, 7, 167, 1, 775, 29, 1935, 3723, 11835, 2887, 65285}},
	{4398, 16, 11576, {1, 1, 7, 13, 5, 47, 101, 155, 235, 93, 465, 3581, 1837, 7675, 10789, 45167}},
	{4399, 16, 11582, {1, 1, 3, 5, 9, 59, 121, 109, 59, 821, 1625, 343, 1591, 3875, 13729, 56381}},
	{4400, 16, 11596, {1, 1, 1, 9, 27, 53, 93, 215, 133, 561, 39, 2591, 1041, 11913, 24493, 37921}},
	{4401, 16, 11602, {1, 1, 1, 7, 23, 7, 81, 107, 219, 943, 563, 1083, 5803, 5687, 32559, 62727}},
	{4402, 16, 11611, {1, 3, 7, 7, 21, 53, 3, 5, 231, 601, 1561, 103, 3837, 2675, 5263, 23375}},
	{4403, 16, 11627, {1, 1, 3, 13, 15, 27, 89, 7, 251, 887, 951, 3001, 5687, 4921, 5091, 59337}},
	{4404, 16, 11682, {1, 3, 7, 15, 25, 53, 19, 155, 185, 503, 547, 1917, 7633, 15167, 14177, 46761}},
	{4405, 16, 11687, {1, 3, 5, 15, 21, 49, 13, 163, 471, 281, 1141, 3013, 6847, 9261, 15955, 9397}},
	{4406, 16, 11691, {1, 3, 3, 3, 1, 21, 19, 239, 479, 609, 65, 2735, 5337, 6293, 19351, 34135}},
	{4407, 16, 11733, {1, 1, 7, 1, 9, 1, 119, 23, 411, 535, 101, 1597, 2379, 2421, 31471, 36473}},
	{4408, 16, 11747, {1, 3, 1, 11, 31, 63, 17, 225, 45, 409, 59, 3943, 8043, 11759, 10667, 58793}},
	{4409, 16, 11759, {1, 1, 3, 3, 9, 49, 61, 239, 245, 765, 1945, 3567, 5355, 14799, 7141, 59511}},
	{4410, 16, 11778, {1, 3, 7, 9, 3, 33, 103, 99, 35, 799, 1347, 2253, 8189, 14177, 13479, 13749}},
	{4411, 16, 11852, {1, 3, 1, 15, 15, 51, 7, 179, 471, 265, 1571, 2983, 701, 15133, 7885, 29977}},
	{4412, 16, 11857, {1, 1, 5, 9, 15, 37, 49, 213, 113, 729, 1115, 2727, 2635, 8433, 11145, 46779}},
	{4413, 16, 11858, {1, 3, 5, 11, 7, 3, 115, 151, 133, 723, 7, 4063, 5807, 9845, 25829, 29315}},
	{4414, 16, 11893, {1, 3, 5, 9, 25, 55, 17, 135, 145, 379, 1603, 3459, 5773, 6545, 17509, 25847}},
	{4415, 16, 11907, {1, 1, 7, 11, 1, 61, 113, 147, 489, 775, 1347, 2199, 299, 9589, 19931, 1335}},
	{4416, 16, 11928, {1, 3, 1, 3, 1, 7, 27, 243, 355, 425, 1215, 3723, 3489, 9285, 12739, 24797}},
	{4417, 16, 11931, {1, 3, 5, 11, 15, 25, 57, 221, 247, 647, 259, 1665, 7055, 2835, 16411, 42999}},
	{4418, 16, 11933, {1, 1, 3, 7, 9, 25, 61, 233, 73, 235, 1539, 1865, 5671, 1329, 5767, 43039}},
	{4419, 16, 11967, {1, 1, 7, 9, 21, 11, 123, 7, 41, 407, 1485, 1723, 585, 10597, 16215, 63403}},
	{4420, 16, 11976, {1, 1, 5, 7, 27, 9, 123, 101, 273, 673, 1141, 3841, 4041, 13169, 8221, 12915}},
	{4421, 16, 11989, {1, 3, 1, 13, 3, 17, 105, 17, 237, 321, 855, 2821, 2467, 3503, 15187, 1689}},
	{4422, 16, 12003, {1, 1, 5, 5, 19, 23, 9, 205, 87, 153, 1543, 1193, 6619, 6845, 8459, 37533}},
	{4423, 16, 12024, {1, 1, 7, 15, 13, 29, 79, 9, 203, 211, 239, 2349, 3447, 7797, 19311, 58071}},
	{4424, 16, 12030, {1, 3, 5, 11, 5, 49, 71, 151, 333, 895, 1095, 2471, 2477, 14493, 16711, 14393}},
	{4425, 16, 12037, {1, 1, 5, 13, 17, 19, 25, 149, 111, 631, 763, 2535, 3631, 1809, 8163, 18037}},
	{4426, 16, 12044, {1, 3, 3, 13, 23, 61, 25, 179, 351, 247, 1021, 2413, 2585, 3731, 5435, 52723}},
	{4427, 16, 12052, {1, 1, 5, 11, 1, 39, 65, 59, 21, 927, 1989, 2823, 4857, 15521, 30495, 16067}},
	{4428, 16, 12059, {1, 3, 3, 7, 17, 5, 17, 125, 379, 875, 1565, 2435, 933, 6809, 20129, 26339}},
	{4429, 16, 12075, {1, 1, 7, 5, 3, 57, 51, 213, 455, 663, 2007, 3995, 4889, 9527, 23507, 3261}},
	{4430, 16, 12083, {1, 3, 7, 5, 1, 29, 85, 151, 165, 123, 1425, 2851, 4427, 7683, 21085, 28925}},
	{4431, 16, 12117, {1, 1, 5, 3, 11, 33, 127, 3, 41, 591, 1539, 3823, 125, 421, 9051, 55025}},
	{4432, 16, 12138, {1, 3, 1, 5, 7, 9, 69, 35, 59, 477, 1207, 1245, 6423, 11329, 26535, 37621}},
	{4433, 16, 12146, {1, 3, 1, 1, 15, 35, 17, 123, 303, 193, 1489, 2587, 1883, 4063, 1921, 60413}},
	{4434, 16, 12202, {1, 1, 5, 1, 7, 61, 39, 247, 139, 1015, 757, 823, 4757, 9307, 32287, 37241}},
	{4435, 16, 12230, {1, 1, 7, 15, 3, 5, 85, 93, 457, 999, 1331, 919, 5271, 11687, 24233, 38803}},
	{4436, 16, 12254, {1, 3, 3, 9, 5, 43, 37, 13, 505, 603, 1857, 2675, 2017, 9481, 10873, 54755}},
	{4437, 16, 12304, {1, 1, 5, 15, 13, 3, 7, 239, 471, 835, 553, 413, 4029, 8613, 15533, 58987}},
	{4438, 16, 12316, {1, 3, 1, 5, 19, 27, 21, 43, 57, 755, 1245, 2805, 3799, 2013, 21145, 10317}},
	{4439, 16, 12337, {1, 3, 5, 13, 9, 23, 35, 5, 315, 169, 399, 2641, 1525, 10561, 11917, 33009}},
	{4440, 16, 12347, {1, 3, 5, 7, 23, 53, 101, 105, 451, 207, 1271, 3067, 6725, 15525, 7951, 1283}},
	{4441, 16, 12367, {1, 1, 5, 5, 27, 21, 77, 143, 213, 443, 149, 2667, 5269, 10359, 25287, 5843}},
	{4442, 16, 12398, {1, 3, 5, 5, 25, 27, 109, 157, 459, 767, 765, 2667, 1833, 3781, 9077, 64321}},
	{4443, 16, 12421, {1, 3, 3, 13, 31, 25, 97, 237, 279, 431, 1713, 809, 1989, 10431, 5867, 42197}},
	{4444, 16, 12428, {1, 3, 7, 3, 9, 5, 5, 191, 73, 521, 787, 903, 3073, 2067, 24741, 57029}},
	{4445, 16, 12446, {1, 3, 3, 1, 3, 41, 125, 53, 125, 781, 865, 3677, 6279, 357, 10667, 1127}},
	{4446, 16, 12449, {1, 1, 5, 11, 25, 19, 99, 121, 359, 73, 607, 2149, 5739, 15669, 29457, 57549}},
	{4447, 16, 12467, {1, 1, 5, 3, 23, 5, 35, 55, 369, 239, 329, 3057, 3757, 12523, 5017, 52185}},
	{4448, 16, 12479, {1, 3, 3, 13, 17, 61, 69, 165, 267, 323, 2007, 2025, 4423, 15975, 31897, 37013}},
	{4449, 16, 12502, {1, 3, 7, 13, 19, 19, 87, 111, 389, 611, 1523, 963, 4671, 12569, 21839, 10919}},
	{4450, 16, 12521, {1, 1, 1, 3, 1, 27, 13, 227, 29, 457, 221, 127, 5335, 16247, 19559, 19185}},
	{4451, 16, 12553, {1, 3, 5, 7, 29, 21, 23, 157, 197, 87, 1591, 1829, 407, 15885, 14933, 1997}},
	{4452, 16, 12568, {1, 1, 1, 9, 3, 35, 43, 187, 195, 325, 197, 2905, 7323, 1563, 7659, 45185}},
	{4453, 16, 12573, {1, 1, 1, 15, 3, 23, 105, 33, 73, 495, 1409, 2583, 1099, 1041, 28955, 60293}},
	{4454, 16, 12592, {1, 3, 7, 13, 25, 19, 99, 137, 139, 719, 529, 1147, 5813, 11551, 30183, 14593}},
	{4455, 16, 12597, {1, 3, 3, 5, 17, 25, 73, 193, 309, 887, 1655, 1641, 2091, 12087, 21881, 1145}},
	{4456, 16, 12601, {1, 3, 1, 1, 27, 41, 13, 151, 83, 645, 327, 1795, 2717, 12433, 22751, 9823}},
	{4457, 16, 12615, {1, 1, 3, 7, 1, 43, 77, 229, 59, 499, 1883, 135, 3461, 9821, 219, 6111}},
	{4458, 16, 12619, {1, 3, 3, 3, 23, 7, 17, 67, 361, 565, 1621, 3253, 7973, 281, 3209, 48215}},
	{4459, 16, 12694, {1, 1, 3, 7, 31, 15, 97, 141, 197, 883, 1689, 269, 7487, 10403, 18903, 58147}},
	{4460, 16, 12697, {1, 3, 3, 3, 23, 9, 87, 125, 359, 527, 1251, 637, 1145, 12721, 14693, 6021}},
	{4461, 16, 12698, {1, 1, 3, 5, 13, 43, 105, 173, 205, 859, 1237, 1227, 1409, 15513, 25317, 30745}},
	{4462, 16, 12713, {1, 3, 3, 15, 31, 43, 125, 149, 145, 109, 975, 1167, 7629, 8373, 5923, 64117}},
	{4463, 16, 12722, {1, 3, 1, 15, 3, 33, 89, 173, 379, 615, 1401, 1567, 4453, 7461, 32555, 64369}},
	{4464, 16, 12733, {1, 3, 7, 11, 27, 23, 45, 7, 15, 21, 1663, 3327, 5611, 8535, 27669, 25525}},
	{4465, 16, 12736, {1, 1, 3, 15, 15, 61, 127, 145, 151, 41, 629, 767, 5801, 3395, 1157, 30033}},
	{4466, 16, 12790, {1, 1, 1, 5, 9, 63, 73, 9, 299, 237, 369, 1295, 4869, 6821, 19961, 32129}},
	{4467, 16, 12794, {1, 1, 5, 13, 19, 7, 119, 35, 319, 405, 1255, 1299, 4311, 14999, 24567, 4001}},
	{4468, 16, 12803, {1, 1, 1, 13, 9, 39, 13, 207, 355, 691, 37, 3137, 6073, 16179, 28661, 41}},
	{4469, 16, 12815, {1, 1, 3, 7, 21, 3, 123, 27, 187, 183, 769, 2367, 2957, 7065, 17855, 60805}},
	{4470, 16, 12829, {1, 1, 1, 1, 19, 31, 71, 85, 303, 617, 1007, 283, 8087, 11079, 11463, 65295}},
	{4471, 16, 12833, {1, 1, 3, 13, 25, 63, 61, 187, 401, 465, 1485, 801, 3649, 7763, 8495, 54479}},
	{4472, 16, 12840, {1, 3, 7, 3, 13, 51, 41, 119, 311, 699, 1113, 3631, 1981, 3259, 25871, 45659}},
	{4473, 16, 12875, {1, 3, 7, 13, 31, 27, 57, 181, 325, 107, 1745, 635, 3941, 3305, 14563, 29855}},
	{4474, 16, 12877, {1, 3, 7, 15, 5, 55, 5, 147, 365, 485, 1841, 3673, 6513, 11121, 5725, 18027}},
	{4475, 16, 12916, {1, 3, 5, 11, 13, 45, 35, 79, 109, 683, 1171, 3015, 2163, 4823, 4365, 42931}},
	{4476, 16, 12923, {1, 1, 7, 7, 13, 45, 57, 39, 297, 985, 1559, 487, 5071, 2657, 9401, 41889}},
	{4477, 16, 12935, {1, 1, 5, 9, 29, 33, 79, 237, 509, 537, 549, 3657, 7141, 15189, 30853, 36089}},
	{4478, 16, 12949, {1, 3, 5, 7, 31, 15, 75, 73, 237, 273, 865, 743, 2607, 7611, 18441, 12703}},
	{4479, 16, 12954, {1, 1, 1, 9, 27, 9, 35, 137, 265, 181, 1341, 1945, 5615, 161, 18369, 4791}},
	{4480, 16, 12956, {1, 3, 7, 11, 27, 29, 29, 43, 489, 177, 1489, 2927, 623, 14571, 22447, 46905}},
	{4481, 16, 12959, {1, 3, 3, 3, 19, 41, 107, 53, 239, 263, 1433, 1655, 7991, 7405, 2735, 25519}},
	{4482, 16, 12978, {1, 3, 5, 7, 19, 37, 73, 243, 215, 445, 1781, 3223, 187, 8443, 18185, 45093}},
	{4483, 16, 13001, {1, 1, 3, 13, 9, 57, 111, 111, 221, 505, 1261, 3045, 1655, 16247, 21033, 17993}},
	{4484, 16, 13010, {1, 1, 7, 5, 7, 55, 77, 5, 131, 969, 1481, 2883, 2645, 3003, 5601, 37063}},
	{4485, 16, 13043, {1, 1, 5, 15, 29, 55, 39, 197, 349, 29, 341, 67, 1977, 425, 4063, 42705}},
	{4486, 16, 13049, {1, 1, 7, 13, 1, 57, 81, 81, 129, 681, 1407, 2465, 3627, 2325, 31649, 18449}},
	{4487, 16, 13058, {1, 3, 5, 5, 23, 59, 35, 217, 393, 155, 1315, 105, 2285, 5167, 27997, 55193}},
	{4488, 16, 13112, {1, 1, 7, 3, 11, 59, 53, 179, 319, 819, 947, 3881, 765, 4219, 16405, 48055}},
	{4489, 16, 13140, {1, 1, 3, 9, 23, 9, 67, 67, 137, 523, 1585, 2441, 167, 5217, 12031, 14297}},
	{4490, 16, 13149, {1, 1, 5, 13, 31, 63, 121, 91, 439, 917, 203, 1519, 4363, 2391, 25755, 26677}},
	{4491, 16, 13163, {1, 1, 3, 5, 25, 31, 11, 95, 339, 817, 35, 3923, 7365, 10537, 5521, 54579}},
	{4492, 16, 13187, {1, 3, 7, 1, 3, 33, 47, 13, 139, 445, 1357, 3907, 7495, 8789, 26589, 43479}},
	{4493, 16, 13196, {1, 1, 1, 11, 5, 45, 61, 13, 167, 287, 931, 881, 5713, 12937, 12951, 21597}},
	{4494, 16, 13237, {1, 3, 5, 1, 29, 23, 7, 117, 503, 897, 733, 1113, 7205, 11507, 561, 34011}},
	{4495, 16, 13244, {1, 3, 5, 7, 3, 51, 21, 147, 35, 259, 689, 3801, 2481, 9673, 4065, 595}},
	{4496, 16, 13264, {1, 3, 3, 9, 9, 29, 5, 191, 393, 979, 1627, 937, 75, 2339, 24007, 30555}},
	{4497, 16, 13279, {1, 1, 5, 7, 9, 35, 111, 23, 113, 563, 1689, 1575, 6127, 9919, 2555, 52261}},
	{4498, 16, 13292, {1, 3, 1, 5, 31, 21, 117, 159, 473, 279, 1281, 311, 159, 3343, 27761, 59989}},
	{4499, 16, 13295, {1, 1, 5, 1, 23, 31, 67, 241, 401, 69, 933, 777, 267, 12411, 23767, 9047}},
	{4500, 16, 13307, {1, 1, 5, 1, 15, 49, 99, 15, 267, 913, 1581, 3713, 651, 14275, 10103, 57619}},
	{4501, 16, 13312, {1, 3, 5, 9, 19, 23, 95, 5, 449, 153, 1195, 1315, 2347, 12683, 10865, 50703}},
	{4502, 16, 13317, {1, 1, 1, 13, 17, 17, 115, 31, 135, 725, 795, 1695, 6199, 4179, 5223, 48457}},
	{4503, 16, 13327, {1, 3, 5, 15, 31, 15, 3, 247, 385, 269, 1665, 581, 2809, 6333, 7459, 14815}},
	{4504, 16, 13348, {1, 3, 7, 5, 15, 35, 117, 85, 11, 621, 1523, 981, 511, 14113, 4673, 22683}},
	{4505, 16, 13390, {1, 1, 7, 1, 27, 61, 119, 249, 431, 147, 173, 423, 1353, 4747, 12761, 32947}},
	{4506, 16, 13413, {1, 3, 7, 1, 23, 39, 15, 153, 219, 359, 1233, 169, 2817, 11043, 12435, 30135}},
	{4507, 16, 13417, {1, 1, 5, 1, 1, 55, 39, 1, 151, 865, 125, 2351, 6315, 1165, 20163, 43991}},
	{4508, 16, 13418, {1, 1, 3, 9, 25, 41, 115, 221, 129, 265, 1887, 4057, 7523, 13591, 5735, 59645}},
	{4509, 16, 13451, {1, 1, 5, 5, 19, 15, 9, 77, 511, 627, 153, 2015, 247, 15949, 9715, 45411}},
	{4510, 16, 13475, {1, 1, 7, 7, 17, 17, 107, 183, 39, 647, 1339, 3915, 4937, 537, 27011, 58937}},
	{4511, 16, 13482, {1, 3, 3, 13, 3, 3, 69, 201, 431, 65, 683, 121, 7017, 2791, 16713, 62555}},
	{4512, 16, 13510, {1, 3, 7, 3, 7, 41, 117, 237, 23, 757, 545, 3899, 1837, 5555, 7891, 29151}},
	{4513, 16, 13527, {1, 1, 1, 3, 27, 15, 39, 195, 353, 299, 1431, 2279, 1795, 13773, 24915, 49701}},
	{4514, 16, 13543, {1, 1, 5, 5, 7, 7, 125, 5, 401, 523, 1967, 2471, 7279, 7559, 12025, 60599}},
	{4515, 16, 13547, {1, 1, 1, 13, 13, 59, 13, 249, 465, 847, 1483, 975, 7729, 2773, 15745, 51237}},
	{4516, 16, 13627, {1, 1, 7, 9, 31, 41, 115, 141, 247, 355, 1109, 3239, 6495, 4515, 30145, 47705}},
	{4517, 16, 13650, {1, 1, 3, 13, 29, 41, 69, 179, 45, 271, 1909, 3095, 7199, 14049, 9903, 33383}},
	{4518, 16, 13683, {1, 1, 3, 13, 7, 45, 105, 105, 243, 121, 67, 3169, 237, 137, 4175, 54325}},
	{4519, 16, 13696, {1, 3, 3, 11, 19, 51, 93, 155, 79, 579, 1621, 1251, 7639, 15875, 25815, 56063}},
	{4520, 16, 13702, {1, 3, 3, 9, 27, 27, 77, 45, 217, 965, 1045, 875, 4515, 11261, 27859, 757}},
	{4521, 16, 13713, {1, 1, 3, 11, 17, 7, 81, 37, 299, 765, 977, 3371, 3163, 13267, 18345, 9239}},
	{4522, 16, 13739, {1, 1, 1, 3, 15, 23, 115, 11, 183, 511, 557, 3253, 153, 8465, 22659, 42143}},
	{4523, 16, 13749, {1, 1, 5, 13, 17, 61, 127, 219, 225, 981, 615, 731, 4069, 12637, 11601, 38257}},
	{4524, 16, 13767, {1, 1, 5, 3, 29, 3, 73, 79, 393, 779, 823, 2473, 3811, 4417, 9399, 50011}},
	{4525, 16, 13774, {1, 3, 3, 9, 21, 35, 61, 99, 115, 657, 629, 1129, 2355, 12665, 459, 40831}},
	{4526, 16, 13781, {1, 3, 1, 7, 25, 61, 53, 249, 15, 649, 665, 595, 1441, 8035, 5381, 7147}},
	{4527, 16, 13795, {1, 3, 1, 7, 19, 9, 27, 207, 205, 461, 1069, 4039, 6549, 2333, 29, 50067}},
	{4528, 16, 13821, {1, 1, 5, 3, 15, 7, 115, 205, 71, 73, 53, 71, 6049, 15293, 17041, 20313}},
	{4529, 16, 13825, {1, 1, 7, 7, 9, 7, 119, 99, 357, 729, 2041, 3355, 5333, 1263, 30521, 64867}},
	{4530, 16, 13838, {1, 1, 1, 7, 31, 63, 63, 159, 281, 295, 913, 2161, 8033, 13789, 17711, 14915}},
	{4531, 16, 13852, {1, 1, 7, 9, 29, 55, 69, 129, 453, 361, 1883, 17, 1765, 111, 10311, 55019}},
	{4532, 16, 13879, {1, 1, 5, 9, 21, 15, 31, 57, 291, 915, 945, 1775, 5905, 9073, 3271, 15571}},
	{4533, 16, 13888, {1, 1, 1, 7, 21, 11, 1, 9, 167, 143, 1535, 1267, 6675, 13037, 19513, 52027}},
	{4534, 16, 13897, {1, 3, 3, 7, 31, 45, 57, 105, 63, 121, 631, 679, 3873, 16333, 32069, 64725}},
	{4535, 16, 13906, {1, 1, 1, 9, 23, 41, 29, 207, 489, 319, 905, 3147, 4195, 2697, 5281, 1771}},
	{4536, 16, 13939, {1, 1, 1, 9, 25, 33, 57, 201, 405, 111, 407, 3489, 449, 8601, 1273, 42105}},
	{4537, 16, 13962, {1, 1, 1, 3, 19, 45, 123, 159, 237, 173, 781, 787, 7537, 15453, 25567, 53729}},
	{4538, 16, 13981, {1, 1, 7, 3, 29, 9, 89, 207, 345, 685, 1701, 2859, 8065, 9289, 2459, 28367}},
	{4539, 16, 13985, {1, 3, 1, 3, 31, 41, 55, 241, 81, 1001, 595, 1725, 853, 11427, 20617, 1717}},
	{4540, 16, 14020, {1, 1, 3, 3, 9, 45, 121, 69, 177, 1017, 211, 2753, 6923, 1387, 32063, 45337}},
	{4541, 16, 14030, {1, 1, 5, 15, 21, 23, 95, 61, 485, 403, 619, 3035, 4881, 13423, 17815, 35221}},
	{4542, 16, 14041, {1, 1, 3, 3, 3, 59, 23, 69, 77, 309, 227, 2877, 3739, 3539, 20083, 23415}},
	{4543, 16, 14047, {1, 3, 7, 3, 17, 43, 95, 239, 223, 353, 1237, 3239, 1369, 7245, 32401, 63889}},
	{4544, 16, 14048, {1, 1, 1, 5, 25, 63, 123, 3, 291, 819, 793, 241, 5619, 8871, 18341, 2757}},
	{4545, 16, 14066, {1, 3, 7, 15, 3, 21, 17, 97, 395, 333, 909, 3783, 3635, 751, 24227, 44281}},
	{4546, 16, 14092, {1, 3, 7, 13, 29, 49, 123, 195, 191, 159, 211, 1887, 3047, 4871, 2607, 32425}},
	{4547, 16, 14120, {1, 1, 7, 7, 15, 57, 91, 255, 267, 897, 441, 1581, 953, 6951, 17275, 50229}},
	{4548, 16, 14126, {1, 3, 7, 15, 1, 35, 91, 219, 7, 117, 119, 2687, 1215, 9517, 10849, 28347}},
	{4549, 16, 14131, {1, 1, 1, 1, 21, 55, 67, 221, 503, 883, 1037, 2965, 1485, 8557, 28091, 25459}},
	{4550, 16, 14143, {1, 3, 5, 9, 19, 3, 73, 123, 95, 307, 1339, 3669, 5077, 12049, 523, 21457}},
	{4551, 16, 14146, {1, 3, 1, 13, 3, 1, 111, 97, 371, 697, 1989, 3715, 7875, 6841, 7009, 17809}},
	{4552, 16, 14152, {1, 1, 1, 9, 25, 21, 19, 43, 329, 531, 491, 1147, 1469, 12841, 29651, 29517}},
	{4553, 16, 14155, {1, 1, 5, 1, 15, 63, 101, 197, 477, 245, 341, 61, 3803, 10001, 5519, 19083}},
	{4554, 16, 14157, {1, 3, 7, 5, 13, 43, 7, 143, 291, 531, 1727, 871, 7091, 5737, 24285, 51017}},
	{4555, 16, 14188, {1, 3, 5, 1, 17, 13, 15, 85, 361, 153, 989, 1367, 4705, 3599, 4441, 52471}},
	{4556, 16, 14206, {1, 1, 7, 13, 21, 43, 111, 29, 299, 439, 1929, 283, 5901, 14113, 3929, 55843}},
	{4557, 16, 14243, {1, 3, 3, 9, 31, 59, 63, 43, 91, 171, 733, 1359, 425, 8505, 19777, 54723}},
	{4558, 16, 14278, {1, 1, 5, 7, 31, 1, 97, 253, 331, 307, 1749, 2115, 2535, 9945, 11013, 14231}},
	{4559, 16, 14308, {1, 1, 5, 11, 13, 7, 109, 237, 301, 383, 683, 1603, 6393, 2437, 12101, 1767}},
	{4560, 16, 14317, {1, 1, 3, 11, 15, 61, 119, 199, 109, 265, 1431, 1729, 3409, 10129, 16945, 34681}},
	{4561, 16, 14335, {1, 3, 7, 9, 13, 59, 121, 73, 29, 513, 279, 457, 7985, 15199, 10185, 33621}},
	{4562, 16, 14354, {1, 3, 7, 7, 23, 17, 27, 65, 387, 487, 1803, 2789, 461, 11201, 7001, 40229}},
	{4563, 16, 14356, {1, 1, 3, 15, 9, 13, 55, 127, 33, 513, 1055, 643, 505, 3005, 6121, 64147}},
	{4564, 16, 14379, {1, 3, 5, 15, 5, 11, 77, 233, 175, 691, 171, 2491, 6915, 14195, 7845, 36499}},
	{4565, 16, 14381, {1, 1, 5, 11, 19, 45, 103, 207, 99, 645, 1739, 1517, 5907, 16035, 14559, 44007}},
	{4566, 16, 14384, {1, 3, 7, 15, 21, 27, 53, 107, 89, 291, 983, 3527, 1025, 2985, 13747, 32715}},
	{4567, 16, 14389, {1, 1, 3, 15, 23, 17, 27, 209, 77, 959, 813, 3597, 5809, 693, 10325, 16855}},
	{4568, 16, 14390, {1, 1, 7, 11, 23, 53, 123, 99, 211, 935, 1371, 1657, 4699, 2683, 27933, 21451}},
	{4569, 16, 14414, {1, 3, 3, 3, 17, 21, 93, 201, 423, 351, 1639, 227, 5719, 13111, 5993, 44615}},
	{4570, 16, 14425, {1, 1, 7, 3, 13, 49, 59, 255, 213, 147, 1441, 3593, 6419, 15657, 1947, 62713}},
	{4571, 16, 14462, {1, 3, 1, 7, 7, 41, 79, 135, 275, 585, 925, 3139, 4351, 1827, 23533, 63031}},
	{4572, 16, 14472, {1, 1, 7, 3, 11, 1, 13, 149, 29, 897, 1043, 2393, 3931, 6741, 19973, 46303}},
	{4573, 16, 14508, {1, 1, 1, 13, 13, 57, 9, 253, 149, 67, 1531, 4049, 3013, 13947, 3371, 35317}},
	{4574, 16, 14511, {1, 3, 1, 1, 15, 19, 11, 125, 179, 383, 1273, 1551, 6441, 6579, 19659, 31005}},
	{4575, 16, 14519, {1, 1, 3, 15, 29, 37, 11, 199, 273, 663, 549, 3167, 2049, 8815, 30719, 47905}},
	{4576, 16, 14528, {1, 1, 1, 15, 7, 9, 113, 231, 155, 27, 419, 1303, 4493, 5633, 5743, 51335}},
	{4577, 16, 14561, {1, 3, 5, 13, 21, 35, 7, 63, 391, 637, 2011, 841, 5933, 10563, 7593, 34767}},
	{4578, 16, 14574, {1, 3, 1, 15, 19, 13, 89, 127, 507, 715, 1305, 2989, 7551, 1953, 26101, 54913}},
	{4579, 16, 14588, {1, 1, 5, 1, 1, 33, 101, 211, 173, 761, 177, 2721, 6949, 15801, 6639, 21405}},
	{4580, 16, 14594, {1, 3, 1, 13, 15, 23, 113, 177, 43, 743, 57, 3875, 7833, 13619, 17637, 5547}},
	{4581, 16, 14606, {1, 1, 3, 13, 21, 7, 123, 83, 391, 731, 597, 2595, 1949, 14619, 17141, 60595}},
	{4582, 16, 14614, {1, 3, 7, 13, 31, 55, 15, 43, 17, 855, 233, 1411, 1063, 12977, 22159, 59185}},
	{4583, 16, 14639, {1, 3, 7, 7, 17, 53, 67, 37, 245, 941, 1213, 1965, 6635, 10189, 12979, 63503}},
	{4584, 16, 14641, {1, 1, 5, 15, 31, 23, 15, 175, 177, 643, 1705, 3541, 2009, 12005, 27281, 16821}},
	{4585, 16, 14648, {1, 3, 1, 13, 27, 37, 1, 171, 255, 445, 171, 3555, 8169, 399, 20975, 36195}},
	{4586, 16, 14668, {1, 3, 7, 11, 13, 15, 123, 65, 173, 317, 1991, 2093, 8073, 12831, 15455, 30175}},
	{4587, 16, 14673, {1, 3, 1, 7, 5, 53, 59, 219, 407, 501, 875, 2627, 1335, 14387, 25523, 28337}},
	{4588, 16, 14679, {1, 3, 3, 13, 13, 41, 93, 125, 41, 461, 887, 1085, 3393, 11945, 16329, 43531}},
	{4589, 16, 14695, {1, 3, 1, 11, 21, 39, 1, 185, 429, 285, 443, 1165, 451, 10903, 31511, 50555}},
	{4590, 16, 14702, {1, 1, 7, 5, 11, 25, 61, 171, 493, 733, 215, 1871, 7783, 14113, 2061, 58961}},
	{4591, 16, 14704, {1, 1, 7, 7, 27, 23, 53, 45, 131, 301, 275, 3855, 4883, 6303, 25269, 12697}},
	{4592, 16, 14740, {1, 3, 5, 7, 11, 15, 71, 101, 377, 803, 1369, 3741, 633, 10571, 30659, 31101}},
	{4593, 16, 14754, {1, 1, 5, 15, 19, 53, 3, 153, 411, 685, 1405, 109, 5755, 13311, 3713, 24579}},
	{4594, 16, 14774, {1, 1, 3, 3, 27, 7, 89, 39, 5, 853, 1757, 2927, 2889, 9735, 17959, 39301}},
	{4595, 16, 14792, {1, 3, 1, 3, 13, 41, 57, 71, 187, 285, 825, 1807, 7261, 2645, 21861, 1839}},
	{4596, 16, 14797, {1, 3, 3, 5, 15, 21, 23, 7, 341, 981, 891, 721, 7221, 3137, 28725, 54993}},
	{4597, 16, 14806, {1, 1, 5, 3, 3, 61, 59, 97, 205, 359, 185, 3361, 7637, 15473, 6351, 62097}},
	{4598, 16, 14812, {1, 1, 1, 9, 13, 11, 123, 71, 199, 251, 535, 371, 1605, 12107, 13833, 2019}},
	{4599, 16, 14856, {1, 1, 7, 13, 27, 1, 43, 73, 409, 601, 1481, 649, 3293, 12257, 23377, 17225}},
	{4600, 16, 14876, {1, 1, 7, 11, 15, 55, 99, 45, 261, 461, 1507, 3575, 4425, 9895, 20191, 61863}},
	{4601, 16, 14900, {1, 3, 7, 1, 3, 7, 19, 85, 139, 691, 1685, 137, 7547, 16143, 14193, 52479}},
	{4602, 16, 14910, {1, 3, 5, 9, 17, 61, 7, 189, 31, 641, 1381, 3999, 4909, 8463, 31761, 54493}},
	{4603, 16, 14912, {1, 1, 5, 15, 17, 15, 69, 111, 55, 1011, 1859, 2643, 6043, 5125, 15875, 611}},
	{4604, 16, 14915, {1, 1, 3, 5, 3, 33, 73, 227, 327, 369, 189, 1841, 5625, 1179, 18651, 34951}},
	{4605, 16, 14922, {1, 3, 7, 13, 1, 17, 109, 149, 89, 889, 799, 3423, 2599, 14525, 12763, 23855}},
	{4606, 16, 14939, {1, 1, 3, 15, 5, 63, 87, 7, 63, 171, 1215, 557, 3009, 16305, 23517, 40101}},
	{4607, 16, 14946, {1, 1, 3, 3, 29, 31, 79, 183, 401, 813, 41, 1111, 5669, 15697, 7813, 10215}},
	{4608, 16, 14955, {1, 1, 5, 7, 9, 25, 25, 57, 343, 375, 535, 3405, 1909, 3201, 2417, 52285}},
	{4609, 16, 14966, {1, 1, 5, 9, 25, 19, 33, 183, 29, 991, 1045, 2249, 2933, 12525, 13943, 10423}},
	{4610, 16, 14976, {1, 3, 1, 7, 3, 45, 49, 37, 429, 67, 821, 1289, 7311, 16165, 25861, 57715}},
	{4611, 16, 14986, {1, 1, 7, 3, 19, 3, 33, 153, 505, 683, 611, 1691, 6421, 15517, 19161, 49013}},
	{4612, 16, 14993, {1, 3, 7, 7, 21, 21, 85, 55, 293, 199, 1671, 1881, 7147, 8241, 16173, 51873}},
	{4613, 16, 15012, {1, 3, 1, 15, 3, 61, 97, 191, 435, 511, 1599, 2705, 1897, 2607, 1801, 48583}},
	{4614, 16, 15041, {1, 1, 5, 3, 9, 23, 23, 185, 401, 947, 33, 385, 7491, 14129, 14561, 13759}},
	{4615, 16, 15053, {1, 3, 5, 15, 19, 21, 37, 233, 149, 673, 29, 1315, 3487, 6705, 28283, 43135}},
	{4616, 16, 15056, {1, 1, 1, 11, 9, 35, 101, 255, 345, 829, 689, 2747, 2145, 2101, 24863, 35529}},
	{4617, 16, 15059, {1, 3, 7, 9, 5, 5, 53, 247, 157, 729, 1621, 129, 2485, 5371, 11115, 47771}},
	{4618, 16, 15110, {1, 1, 3, 9, 29, 29, 13, 229, 87, 281, 1119, 1085, 4423, 1667, 27067, 50397}},
	{4619, 16, 15116, {1, 1, 3, 7, 11, 29, 77, 85, 121, 495, 501, 3209, 3531, 2307, 11367, 34135}},
	{4620, 16, 15133, {1, 1, 7, 9, 3, 37, 33, 209, 493, 869, 367, 3221, 1643, 3353, 22851, 4313}},
	{4621, 16, 15134, {1, 1, 1, 7, 15, 53, 27, 17, 29, 345, 821, 1831, 1963, 10089, 5101, 32689}},
	{4622, 16, 15137, {1, 1, 3, 9, 9, 61, 31, 215, 497, 591, 1301, 157, 3329, 13877, 9017, 34673}},
	{4623, 16, 15147, {1, 1, 5, 1, 29, 49, 93, 139, 279, 167, 143, 279, 6145, 6247, 519, 8869}},
	{4624, 16, 15182, {1, 3, 3, 1, 25, 41, 81, 219, 505, 335, 1941, 2963, 413, 775, 4181, 55269}},
	{4625, 16, 15203, {1, 1, 1, 11, 27, 23, 91, 9, 497, 811, 1469, 1999, 5377, 2943, 17635, 11151}},
	{4626, 16, 15215, {1, 1, 5, 15, 17, 23, 15, 235, 271, 749, 1873, 3805, 5405, 7421, 24339, 14351}},
	{4627, 16, 15245, {1, 3, 7, 1, 5, 61, 81, 9, 269, 43, 1391, 2711, 6973, 11299, 2263, 8715}},
	{4628, 16, 15246, {1, 1, 5, 13, 11, 1, 69, 235, 25, 227, 63, 2591, 913, 12555, 6263, 38981}},
	{4629, 16, 15264, {1, 3, 1, 7, 17, 7, 97, 251, 149, 959, 1895, 1179, 4031, 15975, 20313, 64067}},
	{4630, 16, 15269, {1, 3, 7, 15, 3, 17, 85, 229, 149, 925, 585, 3755, 2359, 13131, 12665, 28861}},
	{4631, 16, 15296, {1, 3, 3, 9, 5, 31, 107, 93, 347, 851, 1249, 2161, 6095, 8315, 20259, 39527}},
	{4632, 16, 15314, {1, 3, 7, 7, 21, 63, 85, 241, 501, 627, 1211, 1713, 6907, 4229, 7557, 55719}},
	{4633, 16, 15323, {1, 1, 1, 13, 19, 43, 21, 177, 13, 353, 679, 511, 5565, 993, 25345, 25087}},
	{4634, 16, 15364, {1, 3, 3, 15, 21, 15, 87, 83, 381, 547, 1429, 2417, 2425, 2097, 20889, 12353}},
	{4635, 16, 15386, {1, 3, 1, 11, 23, 21, 69, 147, 427, 271, 1829, 525, 2951, 10773, 32425, 17685}},
	{4636, 16, 15391, {1, 3, 1, 7, 15, 55, 21, 131, 195, 927, 635, 3505, 3817, 14883, 1149, 10079}},
	{4637, 16, 15436, {1, 3, 3, 9, 23, 15, 45, 147, 249, 87, 377, 1551, 4343, 15373, 2895, 44973}},
	{4638, 16, 15460, {1, 3, 1, 7, 31, 63, 67, 107, 109, 1019, 815, 231, 8135, 559, 8175, 21735}},
	{4639, 16, 15464, {1, 1, 5, 7, 7, 63, 103, 133, 167, 883, 1647, 2827, 6015, 8541, 16963, 37129}},
	{4640, 16, 15469, {1, 3, 5, 9, 27, 25, 59, 147, 29, 943, 865, 1233, 2165, 15259, 2235, 25831}},
	{4641, 16, 15470, {1, 1, 5, 7, 25, 5, 67, 89, 493, 111, 359, 1115, 7963, 6545, 7749, 29179}},
	{4642, 16, 15477, {1, 3, 7, 5, 19, 17, 89, 195, 337, 115, 1417, 3837, 4761, 1959, 16205, 61597}},
	{4643, 16, 15488, {1, 1, 5, 11, 25, 43, 3, 111, 491, 897, 1541, 909, 4751, 739, 7827, 64485}},
	{4644, 16, 15494, {1, 1, 5, 15, 19, 61, 39, 111, 451, 419, 1657, 2427, 4589, 5577, 23967, 19259}},
	{4645, 16, 15548, {1, 3, 3, 1, 31, 15, 7, 131, 329, 847, 537, 1775, 3833, 5683, 17267, 16389}},
	{4646, 16, 15551, {1, 1, 7, 1, 9, 29, 13, 25, 409, 513, 1695, 2175, 5099, 727, 5723, 43547}},
	{4647, 16, 15560, {1, 1, 5, 7, 13, 11, 29, 123, 127, 193, 1647, 3157, 2149, 16209, 19909, 14473}},
	{4648, 16, 15563, {1, 1, 1, 15, 15, 37, 125, 157, 487, 143, 1891, 2895, 4397, 10685, 1463, 55027}},
	{4649, 16, 15604, {1, 3, 7, 1, 1, 15, 115, 105, 479, 529, 1095, 2687, 4483, 15027, 15487, 7113}},
	{4650, 16, 15607, {1, 1, 3, 9, 23, 63, 113, 211, 155, 931, 175, 3037, 2359, 10413, 24561, 21099}},
	{4651, 16, 15616, {1, 3, 3, 11, 5, 15, 13, 37, 257, 447, 203, 545, 2467, 9979, 17543, 62703}},
	{4652, 16, 15631, {1, 1, 3, 7, 17, 31, 83, 91, 79, 265, 1415, 2347, 5337, 7615, 27739, 33841}},
	{4653, 16, 15699, {1, 3, 5, 7, 17, 63, 37, 153, 347, 909, 1945, 7, 2071, 15195, 32083, 26713}},
	{4654, 16, 15701, {1, 1, 3, 11, 19, 51, 69, 21, 323, 635, 443, 1685, 6275, 13787, 20921, 45553}},
	{4655, 16, 15708, {1, 3, 3, 7, 15, 35, 67, 247, 257, 429, 2029, 523, 3219, 3893, 26677, 53273}},
	{4656, 16, 15739, {1, 1, 7, 9, 9, 3, 119, 121, 445, 149, 1539, 1887, 2995, 14867, 809, 48065}},
	{4657, 16, 15746, {1, 3, 5, 13, 15, 27, 75, 9, 217, 35, 1363, 2383, 4357, 1153, 20565, 62277}},
	{4658, 16, 15772, {1, 1, 7, 1, 21, 1, 11, 53, 331, 765, 407, 453, 2725, 11199, 645, 14915}},
	{4659, 16, 15793, {1, 1, 5, 1, 29, 11, 5, 159, 395, 53, 323, 1347, 5529, 8525, 24003, 20535}},
	{4660, 16, 15832, {1, 3, 3, 15, 9, 19, 87, 181, 391, 639, 703, 611, 997, 359, 2471, 58465}},
	{4661, 16, 15837, {1, 3, 5, 9, 27, 9, 117, 47, 223, 509, 1537, 1235, 3885, 6767, 17131, 63421}},
	{4662, 16, 15866, {1, 1, 5, 1, 15, 15, 113, 67, 477, 597, 1795, 3065, 4565, 3609, 16419, 19667}},
	{4663, 16, 15899, {1, 1, 7, 11, 1, 63, 33, 211, 271, 773, 499, 2309, 1303, 14015, 30377, 53195}},
	{4664, 16, 15911, {1, 1, 7, 11, 5, 23, 17, 183, 321, 315, 203, 3371, 907, 9331, 21031, 33765}},
	{4665, 16, 15918, {1, 3, 3, 7, 7, 53, 111, 69, 441, 283, 195, 2415, 7293, 7659, 2731, 5417}},
	{4666, 16, 15952, {1, 3, 5, 15, 3, 61, 5, 241, 427, 463, 1729, 653, 7671, 10979, 7247, 36931}},
	{4667, 16, 15962, {1, 3, 1, 9, 3, 5, 105, 117, 465, 853, 2005, 3925, 4419, 4441, 3701, 50747}},
	{4668, 16, 15967, {1, 1, 3, 7, 1, 3, 3, 149, 65, 405, 299, 99, 481, 14323, 11565, 6227}},
	{4669, 16, 15973, {1, 3, 7, 5, 29, 3, 19, 3, 253, 895, 879, 2435, 2151, 10673, 11013, 43055}},
	{4670, 16, 15977, {1, 3, 1, 11, 15, 57, 127, 197, 319, 913, 1039, 811, 7767, 5791, 31725, 8733}},
	{4671, 16, 16016, {1, 1, 7, 3, 13, 25, 25, 81, 229, 185, 39, 2789, 579, 4973, 28617, 60871}},
	{4672, 16, 16035, {1, 1, 7, 3, 25, 17, 41, 7, 103, 269, 55, 2651, 7579, 10935, 8917, 14323}},
	{4673, 16, 16044, {1, 3, 7, 7, 13, 7, 99, 205, 293, 877, 1893, 3013, 2389, 6021, 2645, 18175}},
	{4674, 16, 16067, {1, 1, 3, 7, 9, 39, 59, 187, 191, 761, 339, 3381, 2921, 5197, 16963, 43019}},
	{4675, 16, 16082, {1, 3, 3, 13, 7, 23, 41, 203, 311, 981, 323, 1675, 6689, 579, 3885, 64475}},
	{4676, 16, 16084, {1, 3, 5, 15, 21, 39, 35, 193, 167, 1009, 493, 829, 6571, 1299, 13061, 1163}},
	{4677, 16, 16098, {1, 1, 3, 5, 3, 19, 123, 123, 111, 599, 193, 3419, 7173, 5595, 12449, 52247}},
	{4678, 16, 16107, {1, 3, 5, 11, 9, 25, 65, 49, 239, 953, 481, 3455, 4335, 305, 22469, 11949}},
	{4679, 16, 16142, {1, 1, 3, 7, 15, 1, 13, 77, 147, 49, 1445, 931, 3405, 15951, 15215, 26451}},
	{4680, 16, 16149, {1, 3, 1, 1, 21, 53, 17, 7, 247, 243, 805, 795, 489, 13351, 13493, 30937}},
	{4681, 16, 16165, {1, 3, 7, 5, 5, 13, 39, 115, 397, 757, 423, 2559, 1677, 9449, 24563, 869}},
	{4682, 16, 16172, {1, 1, 3, 11, 23, 9, 27, 233, 165, 853, 1721, 599, 551, 11657, 27623, 40119}},
	{4683, 16, 16178, {1, 1, 3, 1, 3, 47, 75, 207, 113, 417, 1317, 2215, 2395, 1841, 23125, 50401}},
	{4684, 16, 16197, {1, 3, 3, 1, 13, 55, 103, 55, 351, 785, 1665, 3603, 7705, 4811, 21129, 38115}},
	{4685, 16, 16201, {1, 1, 1, 5, 5, 49, 93, 189, 317, 701, 1545, 1017, 4133, 7655, 19827, 52155}},
	{4686, 16, 16215, {1, 3, 3, 13, 17, 37, 7, 249, 139, 529, 235, 3801, 7871, 459, 15127, 13231}},
	{4687, 16, 16221, {1, 3, 7, 5, 7, 63, 99, 241, 131, 455, 1287, 3539, 8029, 12661, 23313, 54029}},
	{4688, 16, 16226, {1, 3, 1, 3, 29, 63, 51, 227, 497, 685, 1351, 449, 7851, 10815, 17379, 62097}},
	{4689, 16, 16232, {1, 3, 1, 11, 25, 61, 73, 29, 467, 533, 855, 853, 5557, 10953, 18307, 27135}},
	{4690, 16, 16246, {1, 1, 7, 3, 13, 49, 63, 171, 177, 283, 1365, 3087, 5445, 15109, 12523, 25193}},
	{4691, 16, 16261, {1, 3, 5, 15, 9, 39, 95, 81, 417, 199, 1653, 3673, 2663, 8101, 12199, 22759}},
	{4692, 16, 16279, {1, 1, 3, 9, 29, 15, 29, 215, 21, 721, 245, 1197, 7251, 5721, 6735, 7751}},
	{4693, 16, 16280, {1, 3, 5, 5, 21, 7, 81, 61, 157, 707, 819, 1689, 4203, 5559, 25483, 43325}},
	{4694, 16, 16290, {1, 1, 7, 13, 15, 51, 47, 197, 269, 921, 353, 2865, 6227, 537, 20015, 53823}},
	{4695, 16, 16314, {1, 1, 3, 5, 13, 25, 91, 221, 111, 587, 1119, 2343, 4651, 4641, 15915, 36323}},
	{4696, 16, 16345, {1, 1, 7, 11, 1, 45, 7, 215, 483, 545, 731, 3041, 3121, 8681, 20651, 4069}},
	{4697, 16, 16355, {1, 3, 7, 13, 13, 27, 109, 65, 103, 805, 299, 2069, 6017, 14565, 20505, 16161}},
	{4698, 16, 16361, {1, 1, 7, 5, 11, 33, 105, 213, 237, 583, 1033, 2333, 845, 6493, 505, 2563}},
	{4699, 16, 16393, {1, 1, 5, 7, 3, 5, 11, 173, 373, 341, 269, 177, 3175, 9413, 601, 13591}},
	{4700, 16, 16394, {1, 1, 5, 13, 7, 57, 61, 187, 121, 405, 665, 111, 7535, 3355, 14051, 511}},
	{4701, 16, 16438, {1, 1, 1, 3, 3, 29, 15, 253, 227, 123, 333, 1343, 7313, 1955, 17741, 30831}},
	{4702, 16, 16450, {1, 3, 5, 1, 5, 47, 65, 183, 199, 839, 925, 2711, 4609, 201, 15177, 29817}},
	{4703, 16, 16507, {1, 3, 7, 9, 21, 63, 5, 163, 265, 581, 1795, 3937, 4641, 2073, 32225, 60831}},
	{4704, 16, 16523, {1, 1, 1, 5, 7, 47, 125, 203, 511, 841, 1937, 3431, 1495, 12827, 5893, 19265}},
	{4705, 16, 16533, {1, 1, 5, 5, 9, 49, 17, 247, 391, 241, 3, 2253, 6319, 89, 4449, 6371}},
	{4706, 16, 16603, {1, 3, 1, 1, 31, 7, 51, 61, 461, 391, 273, 1609, 5825, 16029, 3851, 39213}},
	{4707, 16, 16648, {1, 3, 3, 7, 29, 21, 65, 75, 317, 925, 1319, 3827, 965, 5685, 17007, 64365}},
	{4708, 16, 16653, {1, 3, 1, 5, 23, 23, 109, 59, 31, 659, 635, 2209, 857, 9847, 2507, 18325}},
	{4709, 16, 16672, {1, 1, 1, 1, 17, 51, 53, 77, 461, 147, 229, 2821, 2405, 1259, 1121, 17429}},
	{4710, 16, 16682, {1, 3, 5, 3, 31, 3, 57, 157, 321, 731, 1609, 2139, 899, 12599, 19803, 51083}},
	{4711, 16, 16709, {1, 1, 3, 11, 27, 43, 73, 209, 431, 587, 1247, 2803, 3593, 1351, 18701, 33423}},
	{4712, 16, 16713, {1, 3, 5, 13, 27, 19, 67, 245, 339, 879, 2035, 1801, 5845, 3883, 22057, 15771}},
	{4713, 16, 16719, {1, 1, 3, 11, 11, 55, 93, 51, 57, 127, 1325, 3975, 3989, 2347, 18831, 2979}},
	{4714, 16, 16733, {1, 1, 1, 13, 17, 1, 17, 103, 303, 777, 1973, 2943, 7439, 8953, 27227, 10229}},
	{4715, 16, 16740, {1, 3, 3, 15, 1, 41, 53, 219, 415, 399, 995, 205, 7719, 10937, 31879, 755}},
	{4716, 16, 16761, {1, 3, 7, 9, 13, 7, 99, 93, 419, 1019, 1605, 161, 3831, 9147, 7877, 1333}},
	{4717, 16, 16767, {1, 3, 7, 15, 5, 51, 37, 115, 259, 549, 353, 2067, 7657, 1283, 20333, 2325}},
	{4718, 16, 16771, {1, 1, 3, 3, 23, 33, 63, 233, 363, 719, 1099, 471, 3079, 10577, 19063, 31535}},
	{4719, 16, 16788, {1, 3, 7, 15, 23, 19, 109, 105, 497, 881, 1055, 537, 4607, 239, 22785, 60201}},
	{4720, 16, 16811, {1, 3, 3, 5, 19, 11, 1, 207, 163, 437, 713, 667, 1427, 7505, 28055, 43101}},
	{4721, 16, 16814, {1, 3, 5, 5, 25, 45, 75, 9, 109, 545, 573, 2685, 1013, 2973, 18619, 55945}},
	{4722, 16, 16816, {1, 1, 1, 3, 27, 27, 39, 33, 285, 453, 613, 2707, 2155, 4355, 21105, 7969}},
	{4723, 16, 16828, {1, 3, 3, 9, 1, 31, 71, 101, 491, 409, 65, 1479, 5743, 525, 2863, 53657}},
	{4724, 16, 16834, {1, 1, 3, 1, 17, 63, 55, 11, 125, 447, 275, 2243, 6827, 5753, 32401, 13819}},
	{4725, 16, 16863, {1, 1, 3, 9, 21, 47, 5, 127, 285, 471, 1681, 945, 6141, 5651, 10273, 30811}},
	{4726, 16, 16864, {1, 3, 3, 1, 13, 53, 91, 3, 255, 429, 107, 2937, 2971, 10251, 15009, 37477}},
	{4727, 16, 16879, {1, 1, 7, 13, 21, 63, 73, 3, 63, 491, 101, 1981, 7457, 879, 6243, 22275}},
	{4728, 16, 16888, {1, 3, 1, 1, 11, 43, 121, 101, 293, 639, 645, 2723, 2075, 3629, 22105, 18199}},
	{4729, 16, 16904, {1, 1, 3, 1, 31, 9, 69, 97, 511, 663, 1147, 1237, 389, 255, 8661, 38533}},
	{4730, 16, 16909, {1, 3, 3, 7, 3, 13, 23, 71, 197, 439, 2003, 1771, 8073, 1549, 29089, 5409}},
	{4731, 16, 16921, {1, 3, 1, 1, 9, 23, 1, 221, 159, 699, 593, 3385, 3869, 10105, 22097, 34753}},
	{4732, 16, 16934, {1, 1, 7, 1, 29, 47, 41, 137, 333, 357, 325, 3151, 6641, 3823, 8763, 28327}},
	{4733, 16, 16951, {1, 3, 1, 7, 19, 19, 39, 225, 477, 619, 583, 229, 6177, 9615, 1203, 13711}},
	{4734, 16, 16983, {1, 1, 3, 13, 9, 41, 127, 147, 13, 301, 861, 2019, 3517, 1147, 21587, 42387}},
	{4735, 16, 16999, {1, 1, 5, 11, 9, 63, 11, 121, 251, 199, 483, 2287, 4667, 3887, 10611, 6019}},
	{4736, 16, 17000, {1, 1, 3, 13, 23, 19, 89, 73, 355, 399, 749, 687, 5707, 11443, 817, 38967}},
	{4737, 16, 17006, {1, 3, 5, 9, 3, 23, 115, 207, 373, 541, 73, 1285, 7245, 12505, 5787, 61207}},
	{4738, 16, 17020, {1, 3, 5, 15, 27, 37, 115, 203, 195, 793, 1577, 1283, 7299, 4025, 5319, 5375}},
	{4739, 16, 17030, {1, 1, 7, 15, 25, 19, 61, 11, 215, 771, 1057, 451, 1965, 13693, 25617, 42657}},
	{4740, 16, 17033, {1, 3, 3, 7, 1, 19, 23, 217, 175, 901, 2009, 2989, 5111, 5027, 4805, 37843}},
	{4741, 16, 17044, {1, 3, 1, 11, 11, 37, 3, 131, 459, 769, 201, 3979, 3009, 8691, 27005, 32175}},
	{4742, 16, 17051, {1, 3, 5, 7, 27, 27, 117, 23, 403, 1003, 1501, 785, 6313, 10187, 5085, 30751}},
	{4743, 16, 17072, {1, 1, 7, 3, 11, 41, 73, 151, 19, 657, 131, 1901, 3879, 14995, 24085, 56621}},
	{4744, 16, 17078, {1, 3, 7, 15, 23, 3, 61, 199, 67, 483, 1961, 3583, 5937, 5749, 16685, 11831}},
	{4745, 16, 17084, {1, 1, 3, 15, 25, 15, 97, 9, 299, 641, 883, 2901, 123, 1523, 7055, 15609}},
	{4746, 16, 17087, {1, 3, 5, 5, 31, 55, 19, 45, 239, 543, 2005, 1041, 1711, 11059, 19927, 17475}},
	{4747, 16, 17090, {1, 1, 3, 9, 5, 59, 105, 209, 323, 613, 1963, 2227, 2947, 11761, 21375, 13265}},
	{4748, 16, 17123, {1, 3, 3, 15, 1, 5, 117, 37, 93, 243, 305, 2299, 5163, 9205, 28761, 35987}},
	{4749, 16, 17132, {1, 1, 1, 5, 5, 29, 13, 147, 457, 187, 1729, 1547, 7745, 13155, 7833, 57449}},
	{4750, 16, 17140, {1, 3, 3, 13, 1, 51, 49, 253, 23, 389, 1611, 3045, 5909, 3947, 25105, 3327}},
	{4751, 16, 17149, {1, 3, 1, 11, 15, 47, 19, 15, 231, 57, 763, 1879, 1765, 14861, 22893, 19437}},
	{4752, 16, 17157, {1, 1, 3, 15, 1, 19, 85, 65, 139, 629, 361, 3513, 3807, 799, 8349, 29247}},
	{4753, 16, 17164, {1, 3, 3, 13, 9, 11, 65, 201, 471, 89, 355, 121, 3947, 10775, 3599, 6041}},
	{4754, 16, 17170, {1, 3, 7, 3, 5, 53, 33, 167, 431, 129, 1449, 3263, 7691, 12569, 7551, 41101}},
	{4755, 16, 17179, {1, 1, 3, 15, 9, 41, 5, 239, 361, 773, 955, 3663, 6051, 12889, 5841, 59615}},
	{4756, 16, 17237, {1, 1, 7, 5, 5, 33, 97, 9, 495, 845, 1953, 3533, 5715, 15055, 25211, 9351}},
	{4757, 16, 17248, {1, 3, 1, 11, 25, 37, 83, 153, 289, 739, 353, 1121, 7641, 2081, 28439, 38085}},
	{4758, 16, 17260, {1, 3, 1, 1, 27, 9, 63, 135, 395, 641, 1759, 3727, 4371, 5193, 2783, 12389}},
	{4759, 16, 17272, {1, 3, 3, 15, 3, 9, 5, 153, 111, 675, 1957, 3817, 4269, 10787, 3413, 34199}},
	{4760, 16, 17275, {1, 3, 5, 9, 23, 23, 97, 137, 255, 249, 333, 2329, 1055, 13769, 13109, 33443}},
	{4761, 16, 17287, {1, 1, 1, 15, 7, 37, 99, 219, 483, 755, 263, 3523, 6179, 4817, 29873, 12771}},
	{4762, 16, 17294, {1, 1, 3, 5, 23, 7, 77, 97, 105, 631, 175, 1911, 7271, 1009, 24081, 61207}},
	{4763, 16, 17296, {1, 3, 5, 3, 1, 31, 71, 91, 265, 669, 1839, 3989, 8127, 15001, 1419, 8895}},
	{4764, 16, 17305, {1, 3, 1, 13, 27, 51, 93, 155, 49, 991, 653, 203, 3863, 5363, 31969, 36083}},
	{4765, 16, 17312, {1, 3, 3, 7, 31, 27, 21, 73, 21, 675, 407, 1215, 2963, 6799, 15259, 13125}},
	{4766, 16, 17321, {1, 3, 5, 13, 5, 53, 19, 215, 243, 487, 689, 2519, 6393, 3987, 30847, 37919}},
	{4767, 16, 17367, {1, 3, 3, 7, 5, 31, 115, 231, 255, 955, 2023, 1487, 6575, 9873, 22585, 29951}},
	{4768, 16, 17368, {1, 3, 5, 11, 11, 57, 109, 203, 417, 29, 1311, 893, 1047, 2413, 9305, 38219}},
	{4769, 16, 17378, {1, 3, 1, 7, 23, 51, 113, 3, 105, 915, 1145, 3431, 7331, 3323, 31669, 21485}},
	{4770, 16, 17433, {1, 1, 7, 13, 9, 29, 119, 205, 403, 1023, 257, 863, 2983, 1895, 16539, 23385}},
	{4771, 16, 17455, {1, 1, 7, 13, 27, 21, 47, 139, 341, 509, 1107, 2197, 3649, 14301, 30789, 48783}},
	{4772, 16, 17457, {1, 3, 3, 7, 25, 19, 99, 11, 309, 919, 1809, 3015, 1587, 3779, 1289, 30207}},
	{4773, 16, 17508, {1, 3, 5, 11, 9, 43, 57, 171, 9, 151, 173, 2301, 7723, 2083, 319, 52883}},
	{4774, 16, 17559, {1, 1, 3, 1, 3, 13, 63, 11, 231, 117, 1257, 237, 581, 13871, 15501, 8741}},
	{4775, 16, 17560, {1, 3, 5, 9, 13, 63, 55, 155, 291, 721, 123, 929, 3351, 11651, 12513, 1779}},
	{4776, 16, 17582, {1, 3, 7, 3, 31, 5, 61, 81, 465, 639, 1363, 3157, 2401, 9513, 32559, 34477}},
	{4777, 16, 17596, {1, 3, 1, 15, 27, 25, 3, 117, 277, 13, 707, 3825, 7287, 10181, 30127, 57247}},
	{4778, 16, 17599, {1, 1, 7, 11, 21, 21, 53, 17, 407, 851, 1191, 285, 6815, 1595, 25507, 8099}},
	{4779, 16, 17613, {1, 3, 5, 9, 9, 61, 83, 61, 65, 671, 63, 311, 6897, 15327, 29809, 4899}},
	{4780, 16, 17619, {1, 1, 7, 1, 21, 45, 99, 235, 477, 461, 1119, 4087, 2057, 14861, 31969, 24357}},
	{4781, 16, 17622, {1, 1, 7, 9, 31, 9, 65, 123, 281, 273, 1059, 1625, 6265, 9635, 11563, 45053}},
	{4782, 16, 17655, {1, 3, 3, 7, 1, 41, 15, 23, 43, 727, 1271, 1741, 765, 13265, 4145, 1335}},
	{4783, 16, 17661, {1, 1, 3, 7, 17, 55, 107, 231, 263, 197, 659, 3621, 2789, 5171, 28635, 13595}},
	{4784, 16, 17698, {1, 1, 5, 1, 27, 23, 13, 83, 431, 507, 1571, 1573, 1733, 12171, 8181, 30843}},
	{4785, 16, 17712, {1, 3, 7, 11, 1, 53, 107, 39, 497, 579, 453, 1339, 1415, 10317, 2741, 34599}},
	{4786, 16, 17715, {1, 3, 3, 5, 31, 41, 49, 41, 33, 665, 1783, 87, 317, 6603, 22603, 22879}},
	{4787, 16, 17721, {1, 3, 1, 15, 5, 47, 41, 87, 231, 853, 1615, 2299, 4643, 9249, 15641, 14323}},
	{4788, 16, 17722, {1, 3, 7, 9, 5, 45, 55, 153, 31, 247, 67, 2335, 6057, 4379, 27579, 38437}},
	{4789, 16, 17742, {1, 1, 5, 7, 9, 3, 73, 81, 479, 909, 1097, 3945, 4485, 7815, 22855, 20825}},
	{4790, 16, 17778, {1, 3, 1, 15, 19, 43, 97, 57, 339, 167, 135, 1777, 7681, 9715, 13863, 6347}},
	{4791, 16, 17818, {1, 1, 1, 1, 5, 53, 33, 123, 449, 165, 1283, 2977, 5919, 12929, 32073, 61851}},
	{4792, 16, 17836, {1, 1, 5, 15, 27, 27, 19, 157, 267, 651, 1319, 3841, 7739, 9947, 16801, 41325}},
	{4793, 16, 17841, {1, 3, 7, 9, 19, 7, 83, 95, 401, 293, 437, 1983, 119, 7553, 11097, 11925}},
	{4794, 16, 17856, {1, 1, 3, 5, 21, 1, 53, 201, 385, 843, 1801, 99, 2697, 9659, 28789, 31417}},
	{4795, 16, 17883, {1, 1, 5, 5, 29, 57, 103, 89, 77, 597, 1849, 3433, 4267, 11167, 3841, 44023}},
	{4796, 16, 17896, {1, 1, 7, 1, 21, 47, 113, 253, 249, 431, 1899, 2859, 7345, 5725, 14805, 19999}},
	{4797, 16, 17901, {1, 3, 3, 5, 1, 11, 77, 213, 359, 665, 1855, 2743, 2407, 14677, 17957, 63257}},
	{4798, 16, 17926, {1, 3, 7, 13, 23, 29, 127, 183, 275, 849, 1005, 3159, 3867, 13029, 7527, 13035}},
	{4799, 16, 17937, {1, 1, 1, 15, 29, 47, 81, 225, 77, 879, 1279, 1929, 1457, 2025, 32229, 2847}},
	{4800, 16, 17992, {1, 1, 1, 3, 29, 45, 37, 189, 217, 53, 281, 1785, 1929, 763, 5875, 34303}},
	{4801, 16, 17995, {1, 3, 1, 9, 21, 61, 21, 149, 215, 13, 1221, 5, 7153, 14089, 3119, 33115}},
	{4802, 16, 17998, {1, 3, 7, 11, 7, 57, 89, 185, 485, 649, 1765, 747, 2983, 6791, 25015, 13627}},
	{4803, 16, 18021, {1, 1, 1, 9, 11, 53, 77, 203, 425, 237, 487, 2317, 1047, 8277, 23405, 30445}},
	{4804, 16, 18039, {1, 1, 3, 5, 7, 29, 39, 195, 109, 381, 655, 931, 4469, 16215, 10627, 64171}},
	{4805, 16, 18067, {1, 3, 1, 3, 5, 9, 89, 131, 509, 275, 489, 3161, 3701, 11951, 6579, 42839}},
	{4806, 16, 18122, {1, 3, 7, 13, 15, 37, 65, 91, 305, 433, 1815, 169, 3117, 47, 30331, 34863}},
	{4807, 16, 18129, {1, 3, 3, 9, 21, 3, 21, 113, 25, 833, 1579, 4021, 3481, 55, 20833, 46277}},
	{4808, 16, 18130, {1, 1, 1, 5, 19, 37, 61, 229, 61, 363, 817, 1235, 6235, 7261, 29917, 43057}},
	{4809, 16, 18142, {1, 3, 1, 9, 7, 59, 119, 189, 341, 945, 633, 3683, 2589, 15453, 4989, 40055}},
	{4810, 16, 18148, {1, 1, 1, 5, 25, 63, 61, 73, 207, 205, 1011, 2857, 8137, 2121, 26731, 46011}},
	{4811, 16, 18163, {1, 3, 7, 11, 13, 59, 107, 57, 49, 555, 441, 1771, 4939, 12107, 8263, 16243}},
	{4812, 16, 18192, {1, 3, 5, 13, 15, 49, 89, 217, 83, 225, 2001, 2727, 4651, 619, 16473, 47525}},
	{4813, 16, 18211, {1, 3, 5, 9, 5, 63, 115, 91, 337, 757, 703, 559, 1683, 14875, 30769, 30331}},
	{4814, 16, 18228, {1, 3, 1, 15, 3, 3, 119, 79, 487, 519, 523, 1177, 7105, 12343, 24671, 31711}},
	{4815, 16, 18264, {1, 1, 7, 15, 25, 63, 87, 23, 59, 277, 849, 953, 4567, 11309, 26181, 749}},
	{4816, 16, 18347, {1, 3, 7, 15, 5, 33, 21, 127, 3, 239, 839, 997, 7253, 8183, 19779, 4185}},
	{4817, 16, 18372, {1, 1, 5, 15, 25, 37, 99, 51, 465, 137, 1339, 541, 4319, 9883, 17425, 30743}},
	{4818, 16, 18409, {1, 3, 3, 5, 13, 7, 3, 249, 365, 749, 527, 3675, 3005, 12905, 9621, 899}},
	{4819, 16, 18412, {1, 3, 3, 7, 29, 31, 69, 21, 365, 1021, 1329, 2623, 3549, 5491, 21293, 63771}},
	{4820, 16, 18418, {1, 1, 5, 9, 5, 35, 53, 247, 193, 17, 227, 381, 5233, 821, 3991, 4439}},
	{4821, 16, 18423, {1, 1, 7, 15, 5, 59, 27, 167, 489, 335, 1565, 2999, 1777, 5009, 519, 57967}},
	{4822, 16, 18433, {1, 1, 1, 11, 25, 47, 23, 155, 419, 863, 1905, 355, 1089, 5871, 10149, 53341}},
	{4823, 16, 18439, {1, 1, 7, 7, 29, 55, 127, 83, 33, 309, 2017, 1021, 5987, 1101, 13657, 60587}},
	{4824, 16, 18445, {1, 1, 1, 7, 3, 1, 9, 75, 257, 407, 659, 529, 2087, 14759, 14483, 36425}},
	{4825, 16, 18451, {1, 3, 7, 3, 11, 29, 113, 255, 301, 799, 1171, 3721, 135, 3467, 7109, 50339}},
	{4826, 16, 18467, {1, 1, 1, 7, 21, 25, 15, 31, 61, 491, 57, 189, 2033, 4363, 31295, 16313}},
	{4827, 16, 18502, {1, 1, 5, 1, 5, 17, 33, 77, 483, 469, 355, 2245, 4165, 459, 30411, 29507}},
	{4828, 16, 18514, {1, 1, 3, 13, 1, 27, 29, 85, 491, 787, 1157, 1299, 4743, 14795, 32587, 12807}},
	{4829, 16, 18547, {1, 3, 3, 1, 23, 45, 35, 129, 3, 55, 969, 2387, 3929, 10397, 19879, 2723}},
	{4830, 16, 18575, {1, 1, 1, 7, 19, 3, 9, 23, 497, 347, 2039, 913, 5925, 7965, 5789, 40949}},
	{4831, 16, 18593, {1, 1, 7, 1, 29, 61, 89, 3, 133, 647, 1585, 2661, 1875, 1859, 3937, 12707}},
	{4832, 16, 18613, {1, 3, 3, 7, 25, 11, 41, 149, 427, 463, 901, 2433, 2617, 13511, 3443, 39867}},
	{4833, 16, 18620, {1, 1, 1, 5, 27, 33, 103, 251, 201, 1023, 1979, 3745, 6365, 14945, 22153, 55535}},
	{4834, 16, 18637, {1, 3, 1, 15, 23, 25, 57, 215, 111, 181, 385, 1123, 3095, 7085, 31863, 37393}},
	{4835, 16, 18640, {1, 3, 5, 13, 17, 35, 27, 159, 255, 241, 413, 1823, 5329, 1825, 28489, 58763}},
	{4836, 16, 18712, {1, 3, 1, 15, 3, 33, 97, 27, 409, 889, 409, 2315, 4743, 14827, 3037, 57149}},
	{4837, 16, 18728, {1, 1, 3, 5, 19, 63, 93, 51, 233, 715, 1571, 1101, 2751, 14805, 25683, 13323}},
	{4838, 16, 18742, {1, 3, 7, 3, 21, 15, 117, 179, 263, 229, 199, 2597, 3999, 3037, 3801, 4775}},
	{4839, 16, 18748, {1, 3, 3, 15, 1, 15, 49, 91, 383, 21, 1955, 773, 1213, 5971, 19525, 8699}},
	{4840, 16, 18753, {1, 3, 1, 15, 29, 49, 11, 101, 261, 761, 709, 3811, 4055, 15675, 32305, 15173}},
	{4841, 16, 18756, {1, 1, 1, 3, 9, 41, 127, 23, 413, 461, 237, 1595, 2257, 2971, 31845, 61485}},
	{4842, 16, 18771, {1, 1, 1, 11, 23, 13, 63, 21, 23, 209, 1317, 1071, 3619, 7275, 9343, 21455}},
	{4843, 16, 18814, {1, 1, 5, 9, 31, 35, 41, 249, 477, 569, 1175, 1571, 4679, 10337, 3507, 23415}},
	{4844, 16, 18818, {1, 3, 5, 11, 29, 3, 117, 65, 301, 913, 1709, 1765, 1801, 15513, 31495, 38131}},
	{4845, 16, 18827, {1, 3, 5, 11, 27, 3, 71, 195, 81, 313, 505, 3941, 3797, 2031, 24151, 65085}},
	{4846, 16, 18835, {1, 1, 1, 5, 13, 17, 59, 151, 191, 489, 1267, 3207, 4495, 15145, 12789, 46313}},
	{4847, 16, 18842, {1, 3, 1, 7, 29, 9, 25, 31, 33, 527, 1939, 4089, 333, 757, 8895, 25331}},
	{4848, 16, 18854, {1, 1, 1, 1, 9, 27, 11, 205, 211, 141, 1171, 1881, 4029, 10587, 30103, 39661}},
	{4849, 16, 18858, {1, 1, 3, 3, 23, 3, 23, 175, 355, 753, 183, 1331, 6403, 3369, 29891, 11109}},
	{4850, 16, 18895, {1, 3, 7, 3, 17, 25, 95, 145, 135, 525, 1073, 1841, 3951, 2027, 23053, 19699}},
	{4851, 16, 18914, {1, 1, 5, 3, 11, 43, 117, 67, 299, 885, 1095, 2777, 8185, 14331, 29543, 655}},
	{4852, 16, 18925, {1, 3, 7, 7, 3, 59, 127, 147, 323, 713, 99, 1179, 6395, 1821, 12673, 35587}},
	{4853, 16, 18933, {1, 3, 5, 3, 7, 11, 33, 87, 99, 967, 1443, 1585, 6215, 15125, 30747, 21513}},
	{4854, 16, 18937, {1, 3, 7, 11, 23, 5, 91, 171, 229, 601, 833, 3157, 1691, 15081, 20607, 7643}},
	{4855, 16, 18944, {1, 1, 3, 1, 5, 1, 39, 59, 157, 7, 601, 2079, 3045, 3693, 26511, 13245}},
	{4856, 16, 18973, {1, 3, 5, 9, 9, 27, 83, 135, 185, 379, 2027, 1407, 7409, 7363, 26471, 35907}},
	{4857, 16, 19001, {1, 3, 7, 15, 29, 49, 1, 69, 383, 915, 183, 3809, 4511, 8751, 4715, 7033}},
	{4858, 16, 19012, {1, 3, 3, 3, 1, 17, 71, 233, 243, 933, 1165, 3089, 3565, 6521, 13203, 13065}},
	{4859, 16, 19016, {1, 1, 5, 9, 9, 55, 53, 129, 331, 943, 587, 2573, 2247, 15101, 4987, 62983}},
	{4860, 16, 19027, {1, 3, 1, 13, 11, 43, 45, 127, 129, 857, 669, 321, 3915, 4477, 26973, 19911}},
	{4861, 16, 19040, {1, 3, 1, 13, 15, 3, 83, 23, 13, 441, 953, 2373, 3539, 4895, 26327, 61961}},
	{4862, 16, 19074, {1, 1, 5, 13, 23, 11, 125, 83, 339, 901, 1809, 2691, 3789, 15007, 10569, 65399}},
	{4863, 16, 19085, {1, 3, 1, 1, 17, 27, 105, 199, 435, 245, 1227, 3029, 3911, 1021, 2931, 24957}},
	{4864, 16, 19093, {1, 3, 1, 11, 17, 5, 123, 39, 413, 627, 1149, 3925, 6635, 8959, 31387, 65047}},
	{4865, 16, 19100, {1, 3, 5, 1, 23, 41, 93, 217, 21, 115, 1311, 3901, 2559, 2925, 30755, 7575}},
	{4866, 16, 19107, {1, 1, 3, 9, 13, 11, 63, 171, 135, 983, 1679, 395, 7349, 5153, 26405, 40589}},
	{4867, 16, 19128, {1, 3, 7, 13, 27, 47, 53, 169, 85, 871, 1087, 619, 7399, 9719, 6349, 59211}},
	{4868, 16, 19141, {1, 3, 3, 15, 31, 45, 3, 33, 11, 879, 929, 1977, 1939, 1049, 16159, 41515}},
	{4869, 16, 19142, {1, 3, 5, 11, 9, 27, 13, 23, 127, 747, 1121, 2497, 8141, 8601, 12431, 3243}},
	{4870, 16, 19156, {1, 3, 7, 15, 23, 43, 23, 225, 283, 13, 1837, 2089, 6435, 10121, 22307, 58767}},
	{4871, 16, 19169, {1, 1, 5, 11, 17, 3, 41, 221, 143, 669, 261, 1367, 7813, 15265, 32751, 62007}},
	{4872, 16, 19176, {1, 1, 1, 1, 5, 45, 41, 161, 327, 185, 1403, 485, 2831, 10025, 12555, 47661}},
	{4873, 16, 19222, {1, 3, 7, 1, 31, 55, 87, 83, 439, 929, 653, 4095, 5443, 7361, 27801, 12979}},
	{4874, 16, 19226, {1, 3, 1, 7, 1, 57, 11, 145, 55, 269, 711, 1889, 8023, 7171, 3247, 35691}},
	{4875, 16, 19247, {1, 1, 1, 3, 15, 1, 15, 131, 479, 797, 815, 2343, 1603, 10775, 21341, 20825}},
	{4876, 16, 19259, {1, 3, 5, 9, 3, 27, 31, 117, 441, 177, 215, 3991, 3197, 8397, 19195, 3883}},
	{4877, 16, 19262, {1, 1, 5, 13, 1, 19, 13, 27, 157, 1001, 43, 251, 7997, 7495, 16515, 44287}},
	{4878, 16, 19291, {1, 1, 3, 5, 17, 57, 117, 53, 413, 905, 551, 149, 7531, 14885, 32493, 34961}},
	{4879, 16, 19309, {1, 3, 3, 7, 27, 1, 7, 13, 259, 21, 189, 451, 6171, 3603, 12053, 45619}},
	{4880, 16, 19324, {1, 1, 7, 11, 5, 41, 59, 119, 419, 93, 1399, 629, 1269, 3789, 17035, 61583}},
	{4881, 16, 19334, {1, 3, 5, 11, 1, 11, 59, 83, 473, 273, 839, 3111, 3081, 10159, 6143, 16297}},
	{4882, 16, 19338, {1, 1, 5, 15, 25, 15, 17, 63, 275, 927, 189, 89, 6595, 4399, 27201, 57205}},
	{4883, 16, 19343, {1, 1, 7, 3, 31, 17, 63, 203, 321, 655, 1751, 2985, 3291, 11567, 15135, 49747}},
	{4884, 16, 19376, {1, 3, 5, 13, 27, 25, 89, 39, 299, 833, 1269, 271, 6481, 3045, 7203, 20279}},
	{4885, 16, 19408, {1, 3, 1, 1, 29, 13, 13, 37, 33, 563, 1825, 3257, 3153, 963, 25801, 15861}},
	{4886, 16, 19413, {1, 3, 5, 11, 15, 7, 49, 117, 479, 221, 579, 2995, 1673, 14927, 2869, 35547}},
	{4887, 16, 19420, {1, 3, 1, 11, 31, 11, 77, 161, 183, 187, 1967, 3037, 4441, 10547, 1443, 8619}},
	{4888, 16, 19441, {1, 1, 3, 11, 27, 41, 83, 179, 293, 421, 1395, 1673, 6375, 9801, 14265, 18117}},
	{4889, 16, 19444, {1, 1, 3, 7, 9, 19, 55, 235, 499, 637, 1121, 3583, 8007, 3749, 20903, 6179}},
	{4890, 16, 19454, {1, 3, 7, 13, 9, 55, 125, 77, 395, 9, 2005, 2247, 1609, 6805, 13099, 26367}},
	{4891, 16, 19461, {1, 3, 5, 13, 9, 41, 49, 133, 443, 995, 209, 341, 8013, 11037, 29663, 21161}},
	{4892, 16, 19473, {1, 3, 1, 1, 1, 47, 45, 243, 161, 433, 129, 39, 4885, 8777, 6395, 16953}},
	{4893, 16, 19479, {1, 3, 3, 15, 11, 13, 41, 113, 279, 657, 763, 2411, 7115, 463, 10759, 50493}},
	{4894, 16, 19489, {1, 1, 5, 5, 31, 5, 25, 181, 385, 445, 625, 313, 4983, 11763, 6065, 63835}},
	{4895, 16, 19591, {1, 3, 3, 15, 13, 25, 103, 5, 205, 223, 1327, 73, 677, 6751, 2071, 24963}},
	{4896, 16, 19605, {1, 1, 7, 15, 21, 61, 21, 11, 47, 775, 113, 991, 1943, 1459, 18049, 45025}},
	{4897, 16, 19616, {1, 3, 3, 1, 11, 43, 27, 89, 49, 711, 173, 181, 1261, 2751, 21321, 5467}},
	{4898, 16, 19619, {1, 3, 3, 7, 17, 7, 57, 61, 87, 973, 985, 1849, 559, 7319, 11457, 46071}},
	{4899, 16, 19653, {1, 1, 1, 1, 9, 37, 99, 157, 423, 189, 1355, 3983, 6357, 10825, 26517, 45815}},
	{4900, 16, 19654, {1, 1, 3, 11, 23, 33, 57, 55, 59, 831, 339, 725, 359, 14859, 17523, 36149}},
	{4901, 16, 19681, {1, 1, 5, 5, 27, 29, 47, 147, 153, 801, 1737, 3617, 5447, 8011, 30631, 7921}},
	{4902, 16, 19711, {1, 1, 5, 1, 11, 43, 35, 105, 69, 453, 1023, 875, 6755, 6015, 12449, 50235}},
	{4903, 16, 19719, {1, 3, 1, 5, 29, 31, 43, 89, 369, 891, 1447, 353, 8103, 2555, 1197, 64005}},
	{4904, 16, 19726, {1, 3, 3, 9, 21, 33, 117, 205, 473, 289, 1699, 2361, 7083, 13001, 24127, 48611}},
	{4905, 16, 19738, {1, 3, 3, 15, 3, 23, 79, 139, 475, 511, 181, 1331, 6137, 2653, 14071, 16947}},
	{4906, 16, 19767, {1, 3, 3, 1, 11, 47, 51, 217, 305, 599, 1609, 237, 4537, 5377, 717, 13269}},
	{4907, 16, 19819, {1, 1, 7, 3, 19, 31, 1, 173, 509, 817, 785, 1223, 5585, 8911, 643, 44575}},
	{4908, 16, 19864, {1, 1, 3, 11, 5, 11, 31, 129, 269, 369, 1833, 2885, 441, 11517, 2323, 26735}},
	{4909, 16, 19867, {1, 1, 5, 9, 7, 51, 31, 21, 5, 157, 541, 2939, 4569, 1679, 17467, 27995}},
	{4910, 16, 19885, {1, 1, 7, 3, 21, 33, 85, 213, 41, 851, 1947, 3205, 5065, 6079, 30789, 63677}},
	{4911, 16, 19894, {1, 1, 5, 3, 9, 53, 3, 179, 157, 407, 537, 1193, 4645, 13791, 28153, 11349}},
	{4912, 16, 19900, {1, 1, 7, 13, 25, 61, 9, 151, 263, 143, 1583, 2859, 6617, 8861, 4163, 40695}},
	{4913, 16, 19903, {1, 1, 1, 1, 7, 25, 19, 207, 335, 1019, 1919, 3893, 831, 12421, 4667, 38967}},
	{4914, 16, 19941, {1, 3, 5, 3, 5, 51, 81, 9, 425, 333, 451, 2569, 2771, 12145, 26065, 14385}},
	{4915, 16, 19951, {1, 1, 3, 7, 3, 49, 17, 147, 327, 331, 1197, 7, 3703, 15247, 9723, 52819}},
	{4916, 16, 19959, {1, 3, 3, 7, 27, 21, 117, 229, 255, 603, 195, 1049, 6243, 13593, 14553, 8267}},
	{4917, 16, 19966, {1, 1, 5, 15, 9, 53, 1, 187, 79, 151, 321, 1883, 6105, 13879, 8201, 53213}},
	{4918, 16, 20009, {1, 1, 1, 7, 21, 27, 103, 147, 351, 901, 1927, 2145, 2685, 453, 15173, 7371}},
	{4919, 16, 20018, {1, 1, 3, 5, 21, 27, 125, 77, 395, 27, 827, 3617, 6033, 1511, 29461, 18907}},
	{4920, 16, 20020, {1, 3, 5, 3, 3, 27, 75, 129, 441, 831, 1213, 2499, 5829, 12181, 7991, 58167}},
	{4921, 16, 20038, {1, 1, 1, 9, 3, 33, 85, 135, 45, 405, 1731, 551, 1251, 7895, 3975, 41621}},
	{4922, 16, 20049, {1, 3, 5, 7, 19, 19, 25, 7, 477, 569, 1089, 2705, 6157, 10279, 14029, 36229}},
	{4923, 16, 20066, {1, 3, 7, 3, 5, 19, 99, 137, 67, 361, 2021, 2891, 1957, 14961, 22673, 45707}},
	{4924, 16, 20108, {1, 3, 7, 1, 21, 11, 81, 225, 151, 235, 1761, 3875, 7427, 11213, 27023, 17945}},
	{4925, 16, 20130, {1, 1, 7, 1, 3, 1, 3, 123, 39, 769, 1467, 1907, 7833, 2099, 19, 54653}},
	{4926, 16, 20132, {1, 1, 1, 3, 25, 35, 33, 111, 407, 497, 1527, 3999, 3083, 6221, 8319, 56959}},
	{4927, 16, 20167, {1, 1, 3, 15, 21, 49, 113, 11, 191, 801, 1835, 3413, 3379, 13129, 3655, 23885}},
	{4928, 16, 20219, {1, 3, 1, 5, 21, 57, 87, 133, 409, 325, 569, 2099, 8143, 315, 23287, 21905}},
	{4929, 16, 20227, {1, 1, 5, 5, 21, 43, 25, 169, 265, 123, 81, 2683, 6137, 7341, 16383, 26435}},
	{4930, 16, 20263, {1, 3, 1, 5, 23, 17, 125, 173, 3, 829, 693, 751, 8021, 3701, 32643, 35405}},
	{4931, 16, 20267, {1, 1, 3, 1, 31, 13, 1, 195, 435, 487, 961, 1681, 1233, 6001, 3113, 29515}},
	{4932, 16, 20272, {1, 1, 7, 5, 9, 41, 81, 137, 257, 337, 1837, 145, 4191, 6313, 9991, 25541}},
	{4933, 16, 20289, {1, 1, 5, 13, 29, 13, 1, 117, 501, 991, 571, 793, 1433, 6005, 19, 61135}},
	{4934, 16, 20296, {1, 1, 7, 1, 9, 43, 65, 69, 297, 331, 1777, 1843, 6477, 13943, 1301, 51001}},
	{4935, 16, 20307, {1, 1, 1, 3, 7, 35, 23, 211, 33, 649, 255, 1831, 635, 9965, 16679, 14531}},
	{4936, 16, 20316, {1, 1, 1, 13, 23, 57, 113, 247, 321, 401, 1761, 4001, 1823, 14457, 5251, 4965}},
	{4937, 16, 20323, {1, 1, 5, 5, 31, 5, 53, 103, 297, 575, 1577, 2217, 977, 14415, 16953, 14793}},
	{4938, 16, 20335, {1, 1, 5, 7, 9, 19, 25, 29, 121, 563, 1707, 901, 6167, 10799, 11897, 31187}},
	{4939, 16, 20344, {1, 1, 5, 9, 17, 39, 89, 29, 251, 259, 411, 819, 6037, 4601, 11481, 46141}},
	{4940, 16, 20354, {1, 1, 1, 15, 23, 9, 65, 95, 189, 93, 1485, 2417, 5183, 5513, 26623, 42637}},
	{4941, 16, 20360, {1, 1, 5, 5, 3, 3, 113, 161, 463, 225, 1089, 2023, 2341, 14931, 28097, 56365}},
	{4942, 16, 20368, {1, 1, 5, 9, 9, 3, 109, 141, 27, 473, 107, 4023, 3279, 7595, 13289, 35649}},
	{4943, 16, 20390, {1, 1, 5, 3, 9, 37, 73, 153, 487, 57, 2035, 3583, 239, 2183, 10903, 171}},
	{4944, 16, 20402, {1, 3, 3, 15, 23, 39, 87, 217, 451, 597, 1817, 2883, 145, 11341, 16015, 16765}},
	{4945, 16, 20413, {1, 1, 7, 5, 19, 61, 45, 37, 473, 883, 277, 2801, 13, 7021, 3851, 53223}},
	{4946, 16, 20425, {1, 3, 3, 9, 1, 35, 97, 237, 279, 541, 1911, 661, 7603, 8183, 22071, 37317}},
	{4947, 16, 20428, {1, 3, 3, 11, 11, 63, 101, 71, 227, 259, 1545, 2779, 3859, 4859, 18957, 31749}},
	{4948, 16, 20434, {1, 3, 3, 1, 27, 29, 91, 215, 381, 607, 1701, 1709, 247, 12403, 29943, 59533}},
	{4949, 16, 20443, {1, 1, 7, 1, 11, 23, 47, 141, 37, 881, 1443, 3921, 3281, 7417, 1549, 50653}},
	{4950, 16, 20488, {1, 1, 7, 11, 23, 61, 17, 39, 373, 871, 1107, 1875, 1419, 3981, 1333, 11485}},
	{4951, 16, 20502, {1, 1, 7, 11, 21, 51, 127, 145, 129, 425, 1263, 1989, 699, 7317, 24827, 15049}},
	{4952, 16, 20505, {1, 1, 1, 11, 9, 59, 59, 67, 329, 841, 905, 467, 1905, 895, 29711, 25585}},
	{4953, 16, 20535, {1, 1, 1, 15, 3, 39, 11, 205, 297, 383, 445, 2139, 2935, 2399, 22237, 20355}},
	{4954, 16, 20541, {1, 3, 7, 7, 17, 9, 17, 205, 369, 1019, 1703, 755, 5507, 14749, 16461, 14519}},
	{4955, 16, 20554, {1, 3, 7, 3, 5, 31, 97, 35, 43, 249, 773, 4033, 6085, 1241, 24743, 22415}},
	{4956, 16, 20577, {1, 3, 7, 3, 17, 11, 45, 203, 251, 669, 1463, 1897, 7913, 2315, 30307, 15431}},
	{4957, 16, 20583, {1, 1, 5, 5, 7, 53, 83, 13, 1, 841, 423, 1059, 3951, 14209, 11113, 13931}},
	{4958, 16, 20602, {1, 3, 3, 5, 5, 15, 11, 71, 237, 553, 829, 3653, 4991, 1003, 8353, 52173}},
	{4959, 16, 20611, {1, 3, 3, 9, 27, 39, 83, 137, 315, 883, 1155, 3541, 3815, 10633, 13277, 20269}},
	{4960, 16, 20614, {1, 3, 3, 15, 13, 55, 43, 19, 345, 351, 1117, 1747, 1949, 3195, 12241, 52441}},
	{4961, 16, 20626, {1, 1, 3, 5, 1, 11, 113, 117, 37, 103, 1681, 3269, 1659, 14779, 30479, 31123}},
	{4962, 16, 20628, {1, 3, 7, 13, 1, 63, 9, 63, 65, 737, 785, 1713, 8123, 10053, 29871, 17647}},
	{4963, 16, 20635, {1, 1, 3, 5, 17, 45, 71, 37, 283, 145, 1927, 75, 7355, 4681, 2777, 31465}},
	{4964, 16, 20642, {1, 1, 3, 7, 21, 19, 113, 89, 67, 751, 99, 421, 201, 6345, 9473, 39849}},
	{4965, 16, 20674, {1, 1, 5, 11, 31, 57, 75, 79, 393, 745, 1435, 3039, 1175, 983, 923, 42867}},
	{4966, 16, 20716, {1, 1, 5, 9, 31, 47, 31, 61, 85, 651, 1733, 3973, 1979, 7223, 13817, 30593}},
	{4967, 16, 20734, {1, 1, 1, 11, 31, 21, 23, 177, 401, 55, 537, 3775, 3241, 15157, 11849, 15629}},
	{4968, 16, 20765, {1, 1, 1, 13, 31, 53, 79, 57, 35, 617, 61, 89, 6917, 6045, 23879, 45485}},
	{4969, 16, 20801, {1, 3, 7, 7, 3, 43, 57, 243, 107, 321, 273, 2171, 2069, 6171, 29181, 8281}},
	{4970, 16, 20804, {1, 1, 1, 11, 3, 27, 51, 57, 81, 795, 1673, 2601, 7335, 16243, 863, 20663}},
	{4971, 16, 20808, {1, 1, 5, 9, 7, 19, 31, 87, 509, 899, 1133, 1609, 2163, 7595, 10523, 43181}},
	{4972, 16, 20831, {1, 1, 1, 7, 21, 53, 103, 43, 507, 317, 685, 1329, 7057, 7275, 2277, 28271}},
	{4973, 16, 20832, {1, 1, 7, 7, 3, 35, 81, 81, 261, 587, 1571, 2771, 4653, 6517, 25101, 56753}},
	{4974, 16, 20862, {1, 3, 1, 11, 17, 61, 29, 137, 7, 929, 393, 2513, 2423, 5457, 6285, 12525}},
	{4975, 16, 20877, {1, 3, 1, 9, 25, 63, 17, 45, 439, 591, 273, 877, 7741, 8381, 32277, 24635}},
	{4976, 16, 20880, {1, 3, 1, 5, 19, 11, 17, 175, 297, 77, 961, 3331, 5193, 14347, 12713, 32067}},
	{4977, 16, 20885, {1, 1, 5, 11, 3, 17, 13, 21, 219, 653, 1279, 1657, 7659, 14459, 27661, 38093}},
	{4978, 16, 20889, {1, 3, 7, 7, 29, 17, 67, 35, 451, 91, 919, 3163, 7459, 14971, 4317, 42503}},
	{4979, 16, 20905, {1, 3, 3, 15, 7, 61, 69, 211, 349, 97, 911, 503, 3903, 12327, 11049, 29387}},
	{4980, 16, 20914, {1, 1, 7, 3, 5, 7, 63, 237, 387, 931, 693, 1143, 6545, 8529, 25217, 53087}},
	{4981, 16, 20967, {1, 1, 5, 7, 1, 13, 21, 169, 259, 289, 437, 649, 4905, 15261, 29997, 34043}},
	{4982, 16, 21028, {1, 1, 1, 9, 25, 13, 19, 229, 29, 213, 1941, 1501, 3463, 15761, 15635, 39687}},
	{4983, 16, 21031, {1, 1, 5, 7, 13, 29, 101, 57, 483, 913, 1025, 2139, 4327, 7847, 12455, 41797}},
	{4984, 16, 21043, {1, 1, 3, 11, 17, 27, 97, 79, 411, 9, 1797, 3721, 5291, 859, 8889, 6151}},
	{4985, 16, 21052, {1, 1, 1, 5, 17, 61, 45, 187, 157, 301, 1017, 1507, 6031, 4271, 32593, 23739}},
	{4986, 16, 21058, {1, 1, 3, 11, 31, 39, 7, 169, 15, 799, 1585, 2055, 4683, 13247, 23743, 50399}},
	{4987, 16, 21087, {1, 1, 1, 9, 25, 37, 15, 39, 339, 383, 1153, 1211, 5745, 15249, 26021, 39871}},
	{4988, 16, 21088, {1, 1, 3, 13, 17, 51, 27, 137, 231, 877, 309, 3633, 2575, 12259, 2743, 14781}},
	{4989, 16, 21093, {1, 1, 5, 7, 5, 33, 95, 19, 37, 829, 1489, 3525, 3887, 8277, 21867, 3581}},
	{4990, 16, 21106, {1, 1, 1, 15, 11, 33, 99, 213, 365, 549, 1603, 777, 3787, 12209, 14999, 50607}},
	{4991, 16, 21108, {1, 3, 1, 9, 23, 25, 57, 147, 73, 285, 1229, 1763, 4579, 14771, 4003, 38197}},
	{4992, 16, 21118, {1, 1, 5, 1, 15, 55, 25, 209, 135, 895, 311, 139, 2445, 6903, 12129, 27907}},
	{4993, 16, 21122, {1, 1, 5, 7, 23, 29, 33, 135, 325, 517, 2021, 1721, 4235, 2363, 12905, 18811}},
	{4994, 16, 21131, {1, 1, 1, 9, 3, 19, 69, 29, 157, 787, 1969, 3711, 8179, 5691, 4059, 25541}},
	{4995, 16, 21139, {1, 1, 5, 15, 1, 61, 11, 195, 317, 13, 923, 2149, 4001, 12843, 27109, 30625}},
	{4996, 16, 21141, {1, 3, 1, 7, 3, 13, 45, 187, 445, 859, 53, 3227, 4381, 8273, 32699, 48719}},
	{4997, 16, 21146, {1, 1, 7, 7, 21, 19, 47, 101, 119, 311, 577, 3309, 4585, 12109, 15153, 22915}},
	{4998, 16, 21162, {1, 3, 5, 15, 21, 39, 15, 211, 349, 237, 1873, 3333, 7837, 12811, 14321, 20227}},
	{4999, 16, 21164, {1, 1, 5, 5, 19, 47, 15, 239, 23, 763, 537, 1477, 2231, 10885, 19487, 47385}},
	{5000, 16, 21184, {1, 1, 3, 1, 19, 37, 67, 85, 11, 817, 869, 2249, 4111, 12411, 10405, 20055}},
	{5001, 16, 21208, {1, 1, 3, 3, 1, 41, 85, 137, 91, 369, 1863, 759, 303, 15859, 8063, 12811}},
	{5002, 16, 21211, {1, 3, 1, 11, 23, 1, 11, 219, 201, 573, 1573, 619, 2959, 6485, 7483, 46099}},
	{5003, 16, 21213, {1, 3, 3, 9, 13, 9, 9, 255, 47, 375, 409, 1435, 1665, 14967, 3247, 18193}},
	{5004, 16, 21230, {1, 1, 1, 5, 9, 61, 121, 173, 51, 415, 1621, 3947, 1379, 847, 23187, 39259}},
	{5005, 16, 21270, {1, 1, 1, 7, 3, 19, 95, 59, 187, 453, 1533, 445, 2699, 4817, 25983, 50925}},
	{5006, 16, 21276, {1, 3, 5, 13, 25, 25, 33, 5, 497, 1, 535, 1653, 6541, 10939, 17721, 43829}},
	{5007, 16, 21285, {1, 3, 7, 11, 9, 59, 115, 127, 85, 505, 541, 3243, 5853, 12673, 25275, 39577}},
	{5008, 16, 21297, {1, 3, 7, 1, 17, 25, 83, 127, 225, 295, 1823, 2051, 847, 4249, 13763, 5723}},
	{5009, 16, 21304, {1, 1, 1, 5, 3, 63, 39, 131, 191, 983, 119, 3287, 3335, 7969, 31347, 39439}},
	{5010, 16, 21310, {1, 3, 7, 9, 19, 31, 19, 91, 35, 677, 1229, 1371, 6497, 3315, 15239, 54235}},
	{5011, 16, 21330, {1, 1, 1, 15, 3, 49, 113, 199, 135, 35, 709, 385, 7923, 3711, 18351, 12711}},
	{5012, 16, 21339, {1, 1, 3, 15, 31, 13, 41, 1, 183, 95, 1625, 1675, 7881, 6607, 4165, 27151}},
	{5013, 16, 21346, {1, 3, 3, 15, 21, 57, 81, 49, 5, 297, 131, 883, 1113, 2497, 32129, 39139}},
	{5014, 16, 21391, {1, 3, 5, 7, 29, 47, 101, 173, 299, 879, 143, 3341, 3929, 6797, 8753, 47963}},
	{5015, 16, 21427, {1, 3, 3, 13, 11, 39, 27, 187, 27, 763, 1515, 1903, 5897, 10061, 14595, 12713}},
	{5016, 16, 21451, {1, 3, 5, 11, 27, 35, 37, 213, 45, 867, 1591, 3083, 8123, 5045, 31703, 61487}},
	{5017, 16, 21465, {1, 1, 3, 5, 3, 31, 23, 89, 369, 371, 1165, 3673, 6821, 333, 10881, 4153}},
	{5018, 16, 21468, {1, 1, 7, 13, 1, 33, 49, 195, 223, 197, 1799, 2427, 6171, 493, 13503, 23619}},
	{5019, 16, 21471, {1, 1, 3, 9, 5, 59, 105, 215, 449, 527, 1661, 2643, 309, 11239, 11633, 63459}},
	{5020, 16, 21533, {1, 1, 3, 11, 13, 11, 15, 99, 409, 807, 1911, 883, 1323, 9037, 6401, 545}},
	{5021, 16, 21610, {1, 1, 5, 7, 7, 7, 1, 167, 353, 923, 1403, 3109, 4981, 3877, 25451, 4667}},
	{5022, 16, 21615, {1, 1, 5, 11, 11, 25, 121, 153, 111, 785, 1301, 1497, 6267, 14919, 24125, 52029}},
	{5023, 16, 21630, {1, 3, 3, 5, 29, 55, 63, 177, 305, 41, 111, 1065, 1127, 113, 2815, 12979}},
	{5024, 16, 21633, {1, 3, 5, 7, 23, 39, 17, 179, 267, 917, 511, 3923, 915, 14173, 10689, 50749}},
	{5025, 16, 21657, {1, 1, 5, 3, 9, 45, 15, 157, 495, 625, 407, 2769, 3267, 7593, 17957, 54597}},
	{5026, 16, 21658, {1, 3, 3, 11, 21, 13, 5, 207, 107, 965, 1803, 1541, 3487, 3483, 109, 26923}},
	{5027, 16, 21669, {1, 1, 5, 11, 25, 49, 99, 135, 109, 371, 1307, 1815, 1095, 2329, 27101, 52269}},
	{5028, 16, 21670, {1, 3, 5, 5, 13, 9, 109, 79, 151, 47, 311, 3873, 3645, 3773, 1083, 31599}},
	{5029, 16, 21673, {1, 3, 5, 15, 1, 9, 87, 21, 145, 583, 159, 2435, 587, 10123, 24803, 19993}},
	{5030, 16, 21701, {1, 3, 1, 1, 23, 11, 5, 45, 373, 1011, 1353, 277, 7051, 3845, 12391, 25313}},
	{5031, 16, 21719, {1, 1, 1, 9, 13, 13, 109, 251, 97, 483, 1723, 2555, 813, 9345, 11351, 44705}},
	{5032, 16, 21720, {1, 3, 5, 7, 21, 49, 63, 13, 119, 813, 1559, 983, 499, 15159, 24163, 59903}},
	{5033, 16, 21747, {1, 1, 3, 5, 27, 33, 27, 165, 207, 693, 1401, 1357, 7637, 337, 10163, 43273}},
	{5034, 16, 21819, {1, 1, 5, 13, 29, 7, 71, 187, 1, 655, 1829, 2645, 6969, 5435, 28415, 33199}},
	{5035, 16, 21839, {1, 1, 1, 13, 17, 21, 13, 141, 41, 267, 1165, 1893, 3455, 10737, 16693, 33065}},
	{5036, 16, 21854, {1, 1, 5, 1, 7, 27, 7, 67, 107, 11, 763, 2529, 3023, 15745, 17023, 51911}},
	{5037, 16, 21857, {1, 3, 3, 3, 7, 57, 123, 249, 309, 511, 1655, 1379, 725, 7325, 20261, 65039}},
	{5038, 16, 21864, {1, 1, 5, 11, 3, 27, 23, 27, 285, 771, 2017, 1727, 4847, 2665, 30655, 47625}},
	{5039, 16, 21882, {1, 3, 7, 7, 17, 3, 93, 133, 427, 1021, 1135, 341, 6711, 11713, 24157, 1561}},
	{5040, 16, 21900, {1, 1, 3, 7, 15, 55, 11, 247, 65, 115, 1967, 535, 841, 15131, 28381, 31337}},
	{5041, 16, 21903, {1, 3, 1, 7, 9, 45, 73, 151, 127, 125, 767, 2003, 6893, 3875, 451, 30275}},
	{5042, 16, 21928, {1, 1, 7, 3, 5, 55, 127, 123, 163, 763, 1165, 1637, 6267, 7215, 23403, 20961}},
	{5043, 16, 21931, {1, 1, 1, 13, 1, 21, 65, 141, 369, 413, 1675, 27, 7357, 6929, 18083, 829}},
	{5044, 16, 21946, {1, 3, 5, 13, 1, 17, 97, 107, 249, 931, 47, 3537, 2245, 9827, 13673, 23217}},
	{5045, 16, 21971, {1, 1, 1, 11, 13, 19, 43, 31, 51, 727, 1001, 927, 771, 11853, 5761, 60537}},
	{5046, 16, 21974, {1, 3, 1, 7, 23, 27, 115, 5, 201, 431, 1021, 585, 6069, 12511, 6129, 2105}},
	{5047, 16, 21978, {1, 1, 3, 11, 3, 25, 75, 129, 389, 131, 223, 2263, 5377, 5561, 15633, 39527}},
	{5048, 16, 21993, {1, 3, 3, 1, 27, 43, 101, 55, 319, 549, 1971, 2255, 353, 93, 25661, 59077}},
	{5049, 16, 21994, {1, 1, 5, 11, 29, 57, 27, 135, 341, 913, 1637, 1781, 457, 11293, 1013, 53863}},
	{5050, 16, 22030, {1, 1, 1, 11, 3, 51, 79, 251, 443, 651, 393, 3635, 7397, 5443, 4225, 991}},
	{5051, 16, 22035, {1, 3, 7, 13, 31, 9, 3, 109, 427, 735, 891, 2789, 2169, 6459, 355, 43177}},
	{5052, 16, 22063, {1, 3, 3, 3, 13, 7, 93, 195, 293, 37, 75, 2467, 933, 8017, 9925, 61397}},
	{5053, 16, 22068, {1, 1, 5, 7, 25, 15, 69, 199, 161, 769, 387, 1491, 4553, 4173, 25631, 37089}},
	{5054, 16, 22086, {1, 3, 1, 3, 7, 59, 53, 93, 103, 413, 531, 887, 6149, 2901, 22611, 27135}},
	{5055, 16, 22104, {1, 1, 1, 13, 31, 39, 71, 215, 385, 821, 1603, 3043, 4967, 10953, 11543, 64433}},
	{5056, 16, 22119, {1, 3, 1, 7, 7, 63, 5, 143, 1, 339, 1165, 3809, 4257, 12879, 21581, 21307}},
	{5057, 16, 22153, {1, 1, 1, 15, 1, 35, 67, 227, 277, 879, 513, 3423, 6153, 11573, 12809, 34335}},
	{5058, 16, 22168, {1, 3, 7, 9, 9, 39, 47, 17, 101, 179, 631, 1307, 481, 871, 16807, 39811}},
	{5059, 16, 22183, {1, 3, 1, 1, 13, 25, 53, 179, 285, 267, 407, 3781, 3267, 3545, 525, 15733}},
	{5060, 16, 22189, {1, 1, 1, 13, 11, 35, 45, 181, 319, 767, 283, 3021, 5405, 403, 3587, 7291}},
	{5061, 16, 22204, {1, 1, 7, 3, 5, 9, 67, 129, 101, 117, 267, 1925, 1209, 13095, 7123, 62941}},
	{5062, 16, 22230, {1, 1, 7, 3, 5, 63, 109, 199, 95, 421, 193, 3519, 6551, 955, 1679, 16627}},
	{5063, 16, 22240, {1, 1, 5, 13, 1, 47, 71, 157, 343, 653, 981, 1335, 3737, 7185, 28861, 22883}},
	{5064, 16, 22246, {1, 1, 3, 15, 7, 63, 7, 81, 481, 5, 1159, 1361, 4167, 2575, 7437, 16917}},
	{5065, 16, 22269, {1, 3, 7, 1, 27, 17, 61, 193, 317, 841, 1149, 955, 5161, 4275, 1955, 15665}},
	{5066, 16, 22282, {1, 1, 1, 7, 19, 63, 77, 57, 367, 237, 579, 3701, 5763, 4951, 24151, 45215}},
	{5067, 16, 22302, {1, 1, 5, 11, 29, 7, 119, 155, 431, 999, 757, 2433, 5811, 3709, 29573, 23639}},
	{5068, 16, 22330, {1, 3, 3, 3, 15, 35, 125, 13, 275, 507, 1719, 1537, 2349, 12909, 8107, 9845}},
	{5069, 16, 22347, {1, 3, 1, 13, 27, 27, 11, 69, 15, 1017, 207, 625, 809, 2921, 8939, 30293}},
	{5070, 16, 22349, {1, 1, 7, 11, 31, 51, 113, 193, 69, 845, 73, 919, 3523, 3987, 23665, 36527}},
	{5071, 16, 22383, {1, 3, 7, 11, 21, 31, 103, 29, 5, 81, 1081, 3847, 4697, 8857, 30769, 40053}},
	{5072, 16, 22386, {1, 1, 1, 1, 5, 11, 5, 169, 13, 899, 769, 3823, 5405, 5991, 3821, 27767}},
	{5073, 16, 22432, {1, 1, 3, 15, 1, 35, 9, 83, 23, 701, 1807, 1681, 4009, 127, 31751, 38059}},
	{5074, 16, 22450, {1, 3, 3, 7, 9, 61, 73, 111, 193, 607, 2001, 413, 3751, 16337, 16597, 23959}},
	{5075, 16, 22473, {1, 1, 7, 7, 21, 29, 53, 253, 187, 507, 1191, 3521, 463, 2167, 23785, 19867}},
	{5076, 16, 22487, {1, 3, 5, 3, 19, 43, 101, 93, 257, 61, 1589, 3883, 1975, 7283, 5253, 23257}},
	{5077, 16, 22527, {1, 3, 1, 9, 1, 63, 25, 101, 377, 571, 1701, 3385, 243, 12051, 32619, 10459}},
	{5078, 16, 22537, {1, 1, 1, 5, 17, 11, 37, 197, 205, 879, 625, 959, 7389, 7857, 20615, 20405}},
	{5079, 16, 22557, {1, 3, 5, 5, 27, 41, 9, 109, 197, 623, 1045, 63, 7977, 11355, 28613, 5131}},
	{5080, 16, 22561, {1, 1, 5, 11, 5, 29, 27, 85, 131, 247, 433, 3981, 2677, 15415, 869, 6045}},
	{5081, 16, 22568, {1, 3, 1, 13, 9, 49, 25, 79, 135, 719, 93, 631, 2149, 5929, 29833, 38673}},
	{5082, 16, 22573, {1, 3, 7, 11, 15, 13, 37, 233, 227, 205, 1579, 65, 1429, 13499, 26355, 63821}},
	{5083, 16, 22591, {1, 1, 5, 11, 21, 19, 7, 183, 409, 275, 899, 3665, 6207, 849, 24339, 1617}},
	{5084, 16, 22593, {1, 3, 1, 3, 21, 61, 23, 125, 463, 489, 1265, 2975, 3907, 11881, 7533, 43331}},
	{5085, 16, 22605, {1, 3, 1, 15, 15, 51, 83, 31, 175, 47, 1791, 3651, 6735, 5013, 723, 24141}},
	{5086, 16, 22620, {1, 3, 1, 9, 17, 41, 57, 43, 469, 911, 1251, 2105, 3133, 3437, 10097, 26687}},
	{5087, 16, 22627, {1, 1, 3, 3, 9, 9, 125, 201, 141, 943, 1509, 1239, 6575, 8707, 363, 23309}},
	{5088, 16, 22636, {1, 1, 5, 3, 19, 37, 43, 141, 413, 149, 1449, 1003, 4473, 12395, 4101, 61201}},
	{5089, 16, 22647, {1, 1, 5, 11, 17, 37, 41, 33, 57, 627, 325, 1895, 1773, 1339, 24859, 587}},
	{5090, 16, 22697, {1, 1, 1, 3, 5, 49, 127, 109, 361, 853, 1437, 3451, 4031, 5379, 27463, 47425}},
	{5091, 16, 22715, {1, 3, 5, 7, 9, 57, 71, 219, 347, 791, 797, 73, 6241, 12717, 24429, 40977}},
	{5092, 16, 22725, {1, 1, 5, 9, 27, 43, 43, 227, 433, 413, 1109, 2589, 4535, 8947, 8121, 43479}},
	{5093, 16, 22732, {1, 3, 7, 1, 9, 21, 81, 23, 157, 313, 197, 2845, 8059, 15957, 28657, 28093}},
	{5094, 16, 22749, {1, 3, 1, 11, 15, 17, 115, 27, 421, 743, 1885, 2089, 5011, 7137, 7395, 36853}},
	{5095, 16, 22766, {1, 1, 5, 15, 5, 53, 69, 255, 63, 29, 1011, 3201, 1467, 15441, 26255, 62895}},
	{5096, 16, 22768, {1, 3, 1, 11, 3, 47, 35, 195, 149, 849, 1317, 439, 3539, 845, 15295, 42183}},
	{5097, 16, 22771, {1, 1, 5, 15, 15, 37, 67, 105, 495, 985, 1777, 3137, 8039, 11795, 29771, 35045}},
	{5098, 16, 22788, {1, 1, 3, 1, 25, 17, 67, 227, 229, 419, 1319, 3325, 1293, 8585, 28425, 34013}},
	{5099, 16, 22797, {1, 1, 5, 1, 27, 51, 71, 197, 375, 407, 259, 4005, 3309, 5475, 13421, 60065}},
	{5100, 16, 22822, {1, 3, 1, 5, 11, 17, 89, 45, 311, 425, 1629, 773, 7267, 8699, 27547, 37081}},
	{5101, 16, 22828, {1, 3, 1, 7, 9, 25, 101, 105, 489, 217, 103, 1959, 4049, 5793, 31201, 11947}},
	{5102, 16, 22851, {1, 1, 5, 5, 19, 3, 63, 55, 431, 49, 273, 3253, 5357, 13801, 9735, 21883}},
	{5103, 16, 22888, {1, 1, 1, 11, 13, 3, 75, 201, 477, 201, 1875, 657, 6217, 8651, 2207, 16421}},
	{5104, 16, 22893, {1, 1, 5, 13, 5, 31, 75, 113, 25, 171, 1147, 3089, 7095, 4405, 26155, 42323}},
	{5105, 16, 22901, {1, 3, 5, 5, 5, 49, 99, 171, 445, 1023, 1793, 3159, 5809, 12509, 31723, 60411}},
	{5106, 16, 22902, {1, 3, 7, 3, 23, 51, 111, 27, 103, 159, 433, 293, 1741, 3599, 4067, 40667}},
	{5107, 16, 22921, {1, 3, 3, 13, 11, 9, 11, 21, 453, 35, 1649, 1261, 3539, 14081, 5581, 57105}},
	{5108, 16, 22929, {1, 3, 3, 13, 7, 9, 113, 87, 391, 647, 223, 1345, 4481, 9855, 20129, 10807}},
	{5109, 16, 22946, {1, 3, 7, 15, 3, 61, 15, 117, 97, 495, 985, 819, 181, 1363, 13111, 35857}},
	{5110, 16, 22948, {1, 3, 1, 9, 3, 27, 125, 151, 217, 961, 707, 2647, 5307, 621, 12581, 13941}},
	{5111, 16, 22951, {1, 3, 1, 11, 17, 37, 35, 211, 179, 29, 627, 3623, 6429, 16237, 24699, 14385}},
	{5112, 16, 22958, {1, 3, 3, 9, 3, 57, 35, 3, 85, 1017, 1739, 2241, 7297, 15637, 27085, 41237}},
	{5113, 16, 22975, {1, 1, 3, 7, 7, 13, 5, 85, 505, 51, 409, 867, 677, 12451, 13633, 47883}},
	{5114, 16, 22983, {1, 3, 5, 13, 5, 51, 37, 79, 237, 133, 1331, 3263, 349, 4971, 16067, 62485}},
	{5115, 16, 22990, {1, 1, 7, 11, 29, 41, 101, 219, 391, 1023, 1681, 3163, 4071, 14665, 11041, 14523}},
	{5116, 16, 23032, {1, 1, 3, 3, 13, 55, 37, 119, 471, 665, 1315, 3071, 5993, 12005, 13549, 63425}},
	{5117, 16, 23047, {1, 3, 5, 7, 5, 25, 59, 71, 77, 841, 91, 1841, 6997, 1139, 11843, 52209}},
	{5118, 16, 23053, {1, 3, 7, 15, 17, 25, 85, 173, 373, 459, 1713, 1055, 5337, 9921, 15213, 44235}},
	{5119, 16, 23054, {1, 1, 1, 15, 7, 11, 89, 237, 131, 565, 745, 457, 4447, 5581, 11053, 43819}},
	{5120, 16, 23082, {1, 3, 5, 1, 29, 21, 11, 7, 125, 851, 2023, 3321, 1885, 67, 8809, 43291}},
	{5121, 16, 23095, {1, 3, 5, 11, 11, 43, 41, 97, 353, 813, 85, 2453, 769, 11709, 4697, 2849}},
	{5122, 16, 23116, {1, 1, 5, 5, 21, 29, 87, 179, 157, 981, 129, 2139, 6841, 5479, 27111, 20749}},
	{5123, 16, 23197, {1, 1, 3, 9, 31, 59, 61, 15, 423, 33, 467, 1817, 6535, 7341, 31061, 20937}},
	{5124, 16, 23221, {1, 1, 7, 3, 1, 21, 127, 135, 321, 699, 727, 3079, 753, 3971, 5611, 28345}},
	{5125, 16, 23257, {1, 3, 7, 11, 27, 3, 39, 63, 389, 433, 43, 1443, 6241, 10769, 20485, 58823}},
	{5126, 16, 23260, {1, 1, 1, 11, 3, 13, 5, 57, 503, 707, 677, 3355, 6691, 8841, 20041, 11867}},
	{5127, 16, 23263, {1, 1, 3, 11, 31, 39, 107, 221, 81, 125, 1439, 2429, 2109, 3931, 31007, 10915}},
	{5128, 16, 23267, {1, 3, 3, 9, 17, 53, 13, 121, 127, 15, 1825, 1909, 5951, 13503, 31565, 56163}},
	{5129, 16, 23282, {1, 1, 1, 1, 19, 55, 3, 153, 385, 277, 1297, 3655, 6027, 3057, 11341, 46989}},
	{5130, 16, 23284, {1, 1, 5, 9, 3, 55, 37, 223, 353, 141, 1917, 3827, 2255, 7617, 10971, 10641}},
	{5131, 16, 23314, {1, 3, 7, 9, 29, 41, 71, 19, 137, 243, 2047, 395, 6981, 15887, 9479, 60199}},
	{5132, 16, 23326, {1, 1, 1, 9, 17, 43, 51, 191, 405, 727, 485, 987, 1855, 15801, 22529, 10165}},
	{5133, 16, 23335, {1, 3, 1, 7, 27, 31, 69, 255, 153, 793, 1353, 1981, 83, 11387, 6747, 23379}},
	{5134, 16, 23368, {1, 1, 5, 5, 31, 49, 83, 157, 63, 647, 1367, 3995, 5899, 8429, 18317, 3471}},
	{5135, 16, 23373, {1, 3, 5, 13, 19, 15, 99, 13, 143, 887, 529, 2855, 573, 9799, 13585, 59263}},
	{5136, 16, 23401, {1, 3, 5, 3, 13, 47, 103, 87, 11, 381, 397, 899, 71, 15539, 13005, 38297}},
	{5137, 16, 23415, {1, 1, 1, 3, 1, 53, 45, 141, 1, 941, 261, 3291, 5177, 9843, 6309, 62799}},
	{5138, 16, 23432, {1, 1, 5, 9, 29, 31, 107, 57, 135, 229, 1147, 247, 265, 12757, 21365, 7219}},
	{5139, 16, 23476, {1, 1, 1, 3, 1, 49, 55, 247, 495, 449, 141, 1349, 7393, 2589, 23587, 1097}},
	{5140, 16, 23479, {1, 3, 5, 1, 9, 11, 73, 153, 89, 575, 1805, 137, 435, 3687, 32169, 24275}},
	{5141, 16, 23497, {1, 1, 7, 15, 25, 61, 51, 21, 109, 139, 611, 3907, 6721, 5081, 6665, 51463}},
	{5142, 16, 23505, {1, 1, 1, 9, 13, 23, 59, 203, 33, 1013, 1533, 291, 6171, 15463, 8581, 9497}},
	{5143, 16, 23508, {1, 3, 3, 9, 7, 25, 51, 189, 49, 265, 1163, 1141, 2467, 7839, 7083, 65527}},
	{5144, 16, 23531, {1, 1, 3, 9, 19, 33, 77, 9, 181, 919, 623, 1521, 7853, 2199, 24115, 60607}},
	{5145, 16, 23536, {1, 1, 3, 11, 9, 11, 27, 57, 355, 313, 151, 3391, 4869, 12541, 30031, 29455}},
	{5146, 16, 23542, {1, 3, 5, 9, 17, 13, 91, 123, 235, 841, 1113, 1451, 501, 3863, 32483, 9445}},
	{5147, 16, 23565, {1, 3, 7, 3, 13, 25, 87, 243, 449, 293, 1279, 3571, 2693, 13459, 5895, 38127}},
	{5148, 16, 23573, {1, 1, 3, 15, 27, 61, 7, 57, 255, 971, 131, 2585, 2187, 7191, 17779, 34587}},
	{5149, 16, 23590, {1, 3, 3, 7, 23, 29, 55, 41, 463, 873, 1781, 2851, 4731, 9819, 25587, 32199}},
	{5150, 16, 23593, {1, 1, 1, 9, 29, 39, 25, 143, 171, 141, 223, 467, 4417, 9575, 23159, 33819}},
	{5151, 16, 23604, {1, 1, 5, 13, 19, 61, 19, 75, 25, 361, 585, 1627, 2231, 1831, 24885, 37917}},
	{5152, 16, 23621, {1, 3, 7, 7, 23, 19, 59, 55, 323, 55, 143, 131, 27, 15363, 26423, 43963}},
	{5153, 16, 23622, {1, 1, 5, 5, 25, 9, 33, 17, 427, 481, 315, 3999, 4757, 4545, 7695, 56733}},
	{5154, 16, 23636, {1, 3, 5, 13, 5, 59, 45, 117, 115, 263, 1441, 3307, 1085, 3875, 25869, 19725}},
	{5155, 16, 23662, {1, 3, 3, 15, 13, 39, 31, 243, 293, 345, 343, 1911, 6123, 12577, 32081, 59993}},
	{5156, 16, 23667, {1, 3, 5, 13, 17, 37, 105, 201, 205, 929, 435, 1467, 8063, 6963, 13709, 53275}},
	{5157, 16, 23703, {1, 3, 7, 15, 31, 3, 65, 221, 191, 413, 427, 2579, 377, 2793, 26943, 61299}},
	{5158, 16, 23725, {1, 3, 5, 3, 11, 61, 75, 107, 53, 689, 1845, 859, 333, 889, 27607, 48795}},
	{5159, 16, 23734, {1, 1, 5, 7, 1, 11, 37, 181, 323, 187, 1511, 25, 5517, 11957, 7093, 429}},
	{5160, 16, 23738, {1, 3, 3, 1, 25, 31, 125, 139, 433, 583, 683, 587, 5389, 1225, 26047, 18717}},
	{5161, 16, 23752, {1, 3, 1, 15, 23, 33, 23, 147, 279, 513, 157, 4023, 2669, 7543, 2111, 9191}},
	{5162, 16, 23800, {1, 3, 1, 1, 5, 39, 55, 127, 257, 649, 347, 3001, 2215, 15579, 29665, 10337}},
	{5163, 16, 23803, {1, 1, 1, 15, 19, 55, 105, 183, 505, 1003, 1311, 2253, 1861, 3561, 19581, 46183}},
	{5164, 16, 23818, {1, 3, 1, 9, 23, 5, 127, 215, 195, 817, 719, 1285, 919, 8627, 20427, 2723}},
	{5165, 16, 23832, {1, 1, 1, 5, 15, 31, 43, 131, 377, 57, 1531, 915, 2871, 1805, 19765, 60529}},
	{5166, 16, 23842, {1, 3, 3, 3, 15, 1, 93, 55, 477, 221, 643, 1319, 959, 475, 14015, 48823}},
	{5167, 16, 23851, {1, 3, 3, 7, 19, 13, 13, 191, 421, 751, 1103, 2129, 1973, 14935, 26485, 41269}},
	{5168, 16, 23873, {1, 1, 5, 13, 17, 43, 81, 83, 67, 643, 1799, 1157, 4365, 2815, 29871, 5351}},
	{5169, 16, 23883, {1, 3, 1, 9, 21, 31, 87, 177, 25, 403, 1357, 4047, 959, 5133, 7307, 4333}},
	{5170, 16, 23888, {1, 1, 7, 7, 29, 39, 27, 139, 159, 529, 1323, 3823, 4569, 12711, 30263, 10961}},
	{5171, 16, 23910, {1, 3, 1, 13, 27, 15, 59, 1, 107, 723, 497, 43, 143, 16119, 29177, 5653}},
	{5172, 16, 23934, {1, 1, 5, 9, 15, 41, 23, 109, 101, 639, 277, 1687, 1311, 1995, 5403, 6867}},
	{5173, 16, 23938, {1, 1, 5, 3, 13, 1, 95, 177, 379, 545, 789, 3479, 4135, 445, 5869, 38923}},
	{5174, 16, 23949, {1, 1, 3, 9, 21, 31, 23, 65, 209, 383, 271, 367, 6605, 1169, 27267, 9331}},
	{5175, 16, 23955, {1, 1, 1, 1, 27, 39, 121, 29, 155, 465, 947, 2675, 6753, 11647, 29781, 30251}},
	{5176, 16, 23967, {1, 3, 1, 5, 7, 45, 7, 21, 223, 363, 1021, 751, 2257, 14423, 19629, 64867}},
	{5177, 16, 23973, {1, 3, 1, 9, 31, 53, 13, 99, 49, 389, 867, 327, 4145, 3265, 15423, 14985}},
	{5178, 16, 23991, {1, 1, 1, 15, 11, 11, 27, 41, 333, 161, 1343, 2023, 3789, 13563, 16957, 26849}},
	{5179, 16, 24024, {1, 3, 7, 1, 7, 51, 7, 163, 239, 393, 231, 3985, 309, 875, 837, 24791}},
	{5180, 16, 24030, {1, 1, 1, 7, 1, 43, 105, 7, 351, 755, 1781, 1925, 4961, 2961, 13427, 44317}},
	{5181, 16, 24039, {1, 3, 1, 9, 17, 39, 81, 75, 201, 931, 1547, 1857, 7251, 11687, 14437, 12603}},
	{5182, 16, 24067, {1, 3, 3, 15, 15, 23, 95, 7, 193, 9, 1749, 809, 5083, 14645, 24893, 35979}},
	{5183, 16, 24069, {1, 1, 7, 1, 9, 9, 127, 149, 433, 985, 1347, 3379, 2881, 681, 20777, 30703}},
	{5184, 16, 24084, {1, 3, 1, 11, 1, 27, 121, 111, 251, 45, 1341, 1709, 3733, 11297, 29063, 57119}},
	{5185, 16, 24112, {1, 3, 3, 1, 19, 3, 77, 127, 187, 831, 1125, 3119, 4665, 9857, 5301, 36755}},
	{5186, 16, 24115, {1, 3, 3, 3, 29, 29, 61, 225, 257, 635, 665, 1279, 3019, 2401, 8253, 40251}},
	{5187, 16, 24184, {1, 1, 7, 9, 29, 43, 47, 95, 221, 823, 257, 1485, 5183, 225, 27675, 60479}},
	{5188, 16, 24187, {1, 1, 5, 3, 15, 49, 25, 69, 101, 393, 901, 305, 4917, 13479, 30209, 9199}},
	{5189, 16, 24262, {1, 1, 3, 15, 1, 9, 47, 243, 403, 341, 143, 1365, 1817, 6017, 3853, 58625}},
	{5190, 16, 24276, {1, 3, 3, 11, 9, 49, 93, 149, 39, 177, 1863, 1027, 659, 9253, 2131, 26943}},
	{5191, 16, 24279, {1, 3, 1, 3, 25, 1, 43, 255, 217, 353, 957, 39, 4607, 15761, 24291, 33619}},
	{5192, 16, 24313, {1, 1, 1, 7, 29, 51, 71, 237, 1, 703, 19, 213, 6429, 11783, 22049, 30597}},
	{5193, 16, 24331, {1, 3, 1, 7, 31, 63, 105, 203, 473, 731, 257, 91, 5749, 4099, 30125, 40171}},
	{5194, 16, 24336, {1, 3, 7, 9, 7, 29, 119, 181, 225, 743, 1031, 55, 3997, 16221, 11663, 14847}},
	{5195, 16, 24348, {1, 3, 3, 11, 5, 21, 43, 17, 473, 981, 125, 2077, 6141, 4757, 7585, 29207}},
	{5196, 16, 24367, {1, 1, 7, 1, 27, 61, 27, 45, 267, 483, 119, 767, 957, 3411, 2653, 53967}},
	{5197, 16, 24372, {1, 1, 1, 3, 9, 41, 43, 17, 485, 253, 825, 3605, 5919, 9285, 1815, 6095}},
	{5198, 16, 24402, {1, 1, 1, 11, 7, 5, 53, 107, 309, 609, 1389, 2035, 861, 15565, 9375, 42363}},
	{5199, 16, 24420, {1, 3, 5, 3, 27, 57, 7, 177, 183, 227, 865, 183, 2903, 6325, 4393, 5257}},
	{5200, 16, 24444, {1, 3, 1, 5, 21, 29, 79, 107, 365, 427, 813, 3563, 7713, 3865, 4289, 28555}},
	{5201, 16, 24465, {1, 3, 5, 7, 11, 27, 1, 197, 425, 769, 1737, 3627, 1273, 4469, 11967, 823}},
	{5202, 16, 24466, {1, 1, 1, 13, 3, 31, 127, 159, 471, 367, 2047, 949, 1591, 12429, 21497, 27153}},
	{5203, 16, 24471, {1, 3, 1, 3, 3, 31, 31, 87, 243, 413, 1777, 1361, 4575, 1147, 17451, 33985}},
	{5204, 16, 24475, {1, 3, 3, 5, 13, 47, 45, 3, 165, 329, 743, 397, 2479, 4999, 12921, 26689}},
	{5205, 16, 24481, {1, 1, 5, 7, 17, 59, 25, 117, 217, 601, 235, 2691, 5569, 7853, 31063, 28281}},
	{5206, 16, 24488, {1, 3, 1, 11, 11, 39, 71, 77, 481, 39, 363, 1921, 3263, 12849, 13325, 41803}},
	{5207, 16, 24501, {1, 3, 7, 5, 19, 1, 59, 197, 239, 787, 1657, 1449, 4245, 1317, 19609, 53583}},
	{5208, 16, 24514, {1, 1, 7, 1, 13, 33, 81, 53, 223, 323, 477, 2421, 4045, 1855, 5823, 9661}},
	{5209, 16, 24531, {1, 3, 1, 7, 1, 3, 121, 19, 329, 569, 481, 3443, 499, 12407, 19625, 4627}},
	{5210, 16, 24534, {1, 1, 1, 7, 3, 33, 91, 241, 69, 581, 1635, 1025, 4677, 14651, 5229, 19555}},
	{5211, 16, 24559, {1, 3, 1, 15, 9, 11, 99, 47, 489, 755, 601, 1221, 4251, 4377, 20567, 33839}},
	{5212, 16, 24602, {1, 3, 3, 5, 21, 21, 127, 151, 499, 971, 1627, 609, 2365, 3183, 7413, 697}},
	{5213, 16, 24614, {1, 3, 7, 13, 5, 35, 61, 121, 51, 297, 29, 1825, 495, 1299, 12741, 3253}},
	{5214, 16, 24637, {1, 1, 1, 1, 13, 15, 49, 205, 235, 9, 849, 1101, 4533, 10221, 32419, 50151}},
	{5215, 16, 24664, {1, 1, 3, 13, 29, 31, 57, 77, 217, 195, 1951, 189, 981, 6169, 22677, 64415}},
	{5216, 16, 24676, {1, 3, 1, 5, 15, 37, 25, 233, 43, 843, 1205, 153, 6339, 3767, 16725, 32699}},
	{5217, 16, 24679, {1, 3, 3, 1, 19, 37, 11, 217, 461, 897, 1181, 2815, 295, 15153, 25351, 57211}},
	{5218, 16, 24697, {1, 3, 5, 11, 15, 9, 5, 179, 353, 769, 1457, 2919, 1201, 14871, 29549, 52265}},
	{5219, 16, 24709, {1, 1, 3, 3, 5, 51, 127, 221, 329, 543, 1537, 2701, 2709, 9311, 2715, 42669}},
	{5220, 16, 24714, {1, 3, 5, 15, 5, 41, 79, 233, 445, 265, 2001, 935, 3133, 3977, 3601, 21389}},
	{5221, 16, 24716, {1, 3, 5, 11, 3, 7, 115, 45, 311, 827, 1897, 3399, 4251, 5341, 22621, 25519}},
	{5222, 16, 24731, {1, 3, 7, 1, 11, 57, 117, 103, 401, 505, 1683, 2161, 4363, 11189, 20263, 13065}},
	{5223, 16, 24744, {1, 1, 1, 7, 23, 29, 31, 77, 63, 179, 195, 2747, 579, 11701, 5101, 11497}},
	{5224, 16, 24762, {1, 3, 7, 3, 9, 33, 81, 165, 433, 545, 1893, 3709, 3813, 6615, 1485, 6395}},
	{5225, 16, 24764, {1, 3, 1, 15, 9, 5, 27, 157, 389, 683, 1919, 509, 455, 3865, 2303, 6993}},
	{5226, 16, 24769, {1, 3, 3, 9, 5, 5, 3, 5, 299, 59, 539, 1199, 2443, 10821, 3359, 44345}},
	{5227, 16, 24806, {1, 3, 5, 9, 21, 37, 87, 35, 501, 943, 1313, 3929, 351, 9851, 22971, 35659}},
	{5228, 16, 24812, {1, 3, 7, 11, 11, 33, 77, 175, 411, 315, 1797, 2731, 4611, 1809, 22027, 50595}},
	{5229, 16, 24872, {1, 3, 7, 13, 15, 11, 13, 189, 209, 1015, 1869, 1593, 6887, 8571, 7217, 2641}},
	{5230, 16, 24878, {1, 1, 3, 13, 29, 15, 119, 127, 329, 275, 865, 1693, 225, 15735, 11071, 37127}},
	{5231, 16, 24883, {1, 3, 7, 13, 9, 31, 85, 25, 281, 401, 1923, 2391, 3875, 2079, 2055, 53275}},
	{5232, 16, 24909, {1, 3, 1, 5, 23, 57, 79, 127, 209, 901, 315, 1165, 3393, 15095, 1981, 34993}},
	{5233, 16, 24943, {1, 3, 7, 7, 25, 13, 15, 223, 157, 335, 1061, 395, 6895, 6283, 18375, 4887}},
	{5234, 16, 24946, {1, 3, 7, 13, 9, 15, 99, 201, 105, 643, 117, 3027, 641, 13353, 4343, 11875}},
	{5235, 16, 24964, {1, 1, 3, 11, 5, 51, 5, 77, 281, 207, 1201, 1187, 8107, 6625, 7517, 34377}},
	{5236, 16, 24992, {1, 1, 1, 5, 29, 61, 3, 181, 297, 151, 565, 2713, 6611, 8665, 32425, 50399}},
	{5237, 16, 25016, {1, 3, 1, 5, 1, 61, 41, 245, 95, 647, 49, 2195, 5927, 15531, 28547, 51075}},
	{5238, 16, 25022, {1, 3, 3, 15, 15, 63, 123, 63, 77, 813, 423, 715, 91, 3793, 20901, 54927}},
	{5239, 16, 25024, {1, 3, 7, 9, 15, 35, 31, 161, 127, 13, 1667, 1225, 5279, 15487, 18769, 24887}},
	{5240, 16, 25039, {1, 1, 3, 5, 27, 25, 61, 21, 447, 175, 1419, 2691, 1993, 13059, 30809, 38325}},
	{5241, 16, 25051, {1, 3, 1, 3, 15, 21, 15, 193, 233, 435, 1993, 4003, 4581, 13837, 16535, 43781}},
	{5242, 16, 25060, {1, 1, 1, 5, 1, 11, 21, 253, 59, 59, 497, 77, 2165, 8197, 5933, 25743}},
	{5243, 16, 25072, {1, 1, 3, 9, 25, 61, 29, 217, 95, 269, 799, 409, 801, 421, 19065, 53443}},
	{5244, 16, 25097, {1, 1, 7, 1, 1, 41, 71, 59, 191, 867, 223, 1467, 6679, 16031, 4451, 15313}},
	{5245, 16, 25122, {1, 1, 1, 11, 13, 27, 7, 241, 167, 969, 1267, 2953, 3769, 2415, 30065, 39483}},
	{5246, 16, 25148, {1, 1, 1, 3, 25, 61, 103, 23, 53, 799, 989, 3859, 7299, 13613, 12007, 37535}},
	{5247, 16, 25194, {1, 1, 7, 1, 29, 19, 121, 57, 355, 663, 643, 3723, 1775, 10363, 1389, 16039}},
	{5248, 16, 25201, {1, 3, 7, 3, 21, 55, 51, 67, 363, 843, 1187, 1983, 7757, 5413, 26575, 53329}},
	{5249, 16, 25204, {1, 3, 1, 3, 31, 55, 73, 55, 75, 533, 197, 1463, 2933, 6033, 24397, 41579}},
	{5250, 16, 25238, {1, 3, 1, 11, 9, 15, 107, 141, 473, 825, 901, 937, 7433, 13119, 20047, 6695}},
	{5251, 16, 25241, {1, 3, 5, 7, 19, 27, 3, 149, 507, 137, 1025, 1841, 33, 3113, 15101, 28187}},
	{5252, 16, 25248, {1, 3, 5, 7, 31, 27, 53, 45, 177, 129, 1241, 1525, 991, 4141, 17681, 39435}},
	{5253, 16, 25257, {1, 1, 1, 15, 31, 57, 29, 137, 395, 563, 101, 3367, 1277, 5431, 1169, 44321}},
	{5254, 16, 25275, {1, 3, 5, 7, 21, 15, 123, 181, 113, 313, 1763, 1429, 2985, 715, 26129, 549}},
	{5255, 16, 25278, {1, 3, 3, 1, 15, 27, 27, 139, 507, 79, 1999, 2505, 485, 7011, 13369, 7159}},
	{5256, 16, 25304, {1, 3, 3, 11, 27, 53, 109, 179, 399, 657, 697, 421, 5467, 4273, 7837, 11631}},
	{5257, 16, 25307, {1, 1, 1, 15, 1, 57, 91, 199, 443, 569, 1945, 2531, 6349, 4851, 3931, 20537}},
	{5258, 16, 25320, {1, 1, 3, 13, 3, 3, 107, 237, 261, 377, 135, 2809, 7239, 1613, 24035, 26053}},
	{5259, 16, 25334, {1, 3, 3, 5, 3, 59, 65, 197, 411, 363, 1729, 967, 3963, 4535, 111, 7273}},
	{5260, 16, 25348, {1, 1, 7, 3, 13, 39, 105, 235, 203, 1015, 1031, 3127, 1209, 10817, 22177, 44117}},
	{5261, 16, 25357, {1, 3, 3, 13, 19, 21, 123, 31, 59, 185, 1007, 1115, 1965, 13087, 18489, 34063}},
	{5262, 16, 25372, {1, 1, 7, 5, 27, 7, 33, 159, 245, 57, 2003, 3229, 2095, 4939, 31355, 23121}},
	{5263, 16, 25394, {1, 3, 3, 9, 9, 41, 49, 203, 397, 915, 821, 3685, 2269, 11159, 25441, 46377}},
	{5264, 16, 25454, {1, 3, 7, 5, 29, 33, 29, 227, 361, 961, 1905, 1149, 2799, 8115, 28235, 25685}},
	{5265, 16, 25465, {1, 3, 1, 1, 19, 13, 73, 103, 11, 183, 853, 2425, 3809, 2391, 18615, 32735}},
	{5266, 16, 25472, {1, 1, 3, 3, 21, 57, 47, 57, 157, 43, 1085, 3319, 461, 11499, 6809, 10435}},
	{5267, 16, 25492, {1, 1, 5, 5, 17, 21, 55, 17, 421, 865, 159, 1643, 4523, 1485, 11937, 8287}},
	{5268, 16, 25505, {1, 1, 3, 11, 7, 43, 39, 37, 187, 797, 1273, 1227, 2683, 1249, 3375, 44499}},
	{5269, 16, 25517, {1, 1, 5, 11, 17, 35, 27, 73, 97, 59, 921, 2171, 7577, 2847, 93, 35911}},
	{5270, 16, 25530, {1, 1, 5, 1, 5, 17, 117, 189, 357, 581, 1945, 2141, 1679, 12019, 11249, 6809}},
	{5271, 16, 25558, {1, 1, 5, 7, 15, 53, 9, 191, 153, 257, 533, 493, 2389, 4657, 20757, 57275}},
	{5272, 16, 25562, {1, 1, 1, 11, 13, 35, 85, 37, 501, 525, 543, 4057, 2001, 6183, 949, 18465}},
	{5273, 16, 25598, {1, 1, 1, 3, 15, 7, 39, 169, 359, 671, 731, 1523, 211, 1233, 29515, 57787}},
	{5274, 16, 25609, {1, 1, 3, 7, 27, 7, 41, 15, 71, 733, 1919, 401, 1915, 4815, 10571, 839}},
	{5275, 16, 25612, {1, 1, 7, 13, 27, 61, 5, 25, 293, 779, 477, 1537, 6695, 7435, 1281, 64369}},
	{5276, 16, 25627, {1, 1, 7, 7, 19, 11, 101, 45, 449, 883, 1181, 3521, 6019, 16305, 23429, 43789}},
	{5277, 16, 25651, {1, 1, 7, 5, 1, 49, 121, 89, 275, 367, 461, 1717, 2733, 4403, 9123, 35217}},
	{5278, 16, 25653, {1, 1, 7, 1, 1, 37, 41, 221, 281, 531, 357, 3783, 3561, 8135, 18405, 56045}},
	{5279, 16, 25668, {1, 3, 5, 7, 29, 9, 127, 37, 13, 519, 1059, 3973, 7253, 15159, 19337, 57103}},
	{5280, 16, 25711, {1, 3, 3, 15, 3, 41, 91, 7, 63, 747, 1649, 3367, 5945, 3603, 28465, 511}},
	{5281, 16, 25765, {1, 3, 3, 15, 27, 19, 67, 179, 505, 131, 149, 1753, 3603, 1135, 15811, 5305}},
	{5282, 16, 25770, {1, 1, 1, 5, 5, 63, 71, 235, 151, 651, 1383, 249, 3223, 14559, 26809, 4551}},
	{5283, 16, 25775, {1, 3, 3, 9, 29, 41, 67, 111, 175, 515, 1123, 1707, 6653, 8233, 22775, 61029}},
	{5284, 16, 25777, {1, 3, 3, 9, 23, 1, 75, 145, 159, 785, 537, 1995, 2241, 8641, 30709, 41373}},
	{5285, 16, 25783, {1, 1, 5, 9, 21, 1, 87, 233, 401, 643, 197, 3437, 8163, 6363, 6537, 17283}},
	{5286, 16, 25801, {1, 3, 5, 3, 23, 19, 55, 243, 405, 369, 1221, 1959, 5455, 11729, 26117, 9097}},
	{5287, 16, 25802, {1, 1, 3, 13, 3, 57, 71, 235, 125, 263, 873, 1079, 2627, 1343, 1979, 49519}},
	{5288, 16, 25812, {1, 3, 1, 11, 21, 15, 27, 7, 425, 935, 305, 2593, 4437, 9517, 26207, 4229}},
	{5289, 16, 25821, {1, 1, 3, 13, 11, 53, 1, 149, 97, 939, 147, 3365, 5023, 607, 2083, 8715}},
	{5290, 16, 25897, {1, 1, 5, 3, 13, 13, 113, 51, 263, 837, 1145, 3621, 5697, 2867, 7975, 22839}},
	{5291, 16, 25906, {1, 1, 3, 15, 31, 9, 91, 231, 399, 295, 1935, 4021, 7669, 3867, 28015, 9865}},
	{5292, 16, 25929, {1, 1, 1, 1, 13, 59, 51, 35, 407, 733, 1629, 2429, 291, 11923, 32129, 28847}},
	{5293, 16, 25940, {1, 3, 3, 11, 25, 21, 13, 117, 31, 547, 327, 2801, 2247, 4051, 27523, 4257}},
	{5294, 16, 25950, {1, 1, 7, 7, 15, 33, 15, 17, 255, 363, 1013, 1463, 7537, 14093, 21883, 35389}},
	{5295, 16, 25968, {1, 1, 5, 9, 11, 61, 7, 161, 121, 413, 515, 413, 4439, 15405, 30265, 23939}},
	{5296, 16, 25971, {1, 3, 7, 7, 11, 15, 5, 181, 315, 231, 1567, 2985, 1653, 12251, 269, 37317}},
	{5297, 16, 25977, {1, 3, 1, 11, 3, 15, 91, 45, 489, 571, 11, 1239, 7705, 4303, 12535, 21359}},
	{5298, 16, 25994, {1, 1, 7, 15, 29, 43, 81, 63, 483, 851, 389, 1719, 6111, 15293, 2513, 44397}},
	{5299, 16, 25996, {1, 1, 5, 15, 25, 33, 97, 131, 183, 269, 1903, 2733, 7197, 4507, 24471, 36871}},
	{5300, 16, 25999, {1, 3, 3, 13, 17, 33, 73, 83, 247, 207, 79, 1139, 581, 12147, 3539, 45423}},
	{5301, 16, 26001, {1, 1, 1, 15, 29, 49, 79, 151, 267, 393, 1995, 105, 2873, 3981, 19147, 53987}},
	{5302, 16, 26030, {1, 1, 5, 1, 31, 25, 39, 203, 459, 181, 661, 717, 6235, 13413, 1197, 40665}},
	{5303, 16, 26069, {1, 1, 5, 9, 19, 33, 65, 239, 463, 133, 461, 3601, 7755, 1771, 20683, 7417}},
	{5304, 16, 26100, {1, 1, 1, 1, 25, 19, 25, 155, 431, 33, 341, 959, 5679, 1205, 2599, 37499}},
	{5305, 16, 26109, {1, 1, 3, 5, 25, 5, 83, 87, 91, 991, 1833, 4063, 147, 14497, 25725, 4617}},
	{5306, 16, 26131, {1, 3, 5, 7, 31, 7, 73, 51, 339, 313, 1593, 2089, 7387, 15759, 3249, 7953}},
	{5307, 16, 26144, {1, 3, 7, 1, 27, 49, 35, 11, 21, 45, 1689, 1665, 4591, 3713, 12781, 4805}},
	{5308, 16, 26149, {1, 1, 3, 9, 29, 51, 73, 51, 303, 179, 67, 3917, 7615, 6131, 26225, 55991}},
	{5309, 16, 26162, {1, 3, 7, 11, 9, 63, 29, 47, 153, 883, 1859, 1913, 3563, 11451, 6333, 51367}},
	{5310, 16, 26167, {1, 3, 1, 3, 5, 25, 69, 87, 389, 613, 989, 3557, 1339, 12503, 14505, 63209}},
	{5311, 16, 26173, {1, 1, 3, 1, 5, 13, 37, 163, 499, 163, 2025, 1467, 5059, 8479, 2889, 62957}},
	{5312, 16, 26179, {1, 1, 7, 9, 23, 31, 109, 49, 73, 197, 337, 2763, 4789, 8983, 9691, 32155}},
	{5313, 16, 26193, {1, 3, 1, 3, 31, 25, 121, 91, 371, 339, 833, 2217, 4997, 9425, 10685, 60037}},
	{5314, 16, 26230, {1, 1, 7, 11, 31, 3, 105, 125, 255, 175, 803, 1787, 6231, 4441, 5031, 29737}},
	{5315, 16, 26234, {1, 1, 1, 11, 21, 63, 75, 209, 393, 437, 495, 2397, 4759, 15703, 29869, 62685}},
	{5316, 16, 26246, {1, 1, 7, 7, 25, 33, 117, 7, 293, 623, 2001, 2409, 2487, 14803, 3329, 38277}},
	{5317, 16, 26267, {1, 1, 7, 9, 31, 51, 75, 151, 487, 85, 639, 3529, 4491, 1957, 22099, 20263}},
	{5318, 16, 26283, {1, 1, 7, 5, 3, 29, 11, 1, 255, 555, 1269, 779, 1525, 7689, 25847, 39347}},
	{5319, 16, 26300, {1, 1, 7, 7, 19, 21, 9, 123, 3, 943, 1627, 2979, 919, 4565, 31435, 59789}},
	{5320, 16, 26341, {1, 3, 7, 5, 29, 13, 57, 221, 9, 893, 1685, 1879, 4575, 7369, 26191, 6067}},
	{5321, 16, 26356, {1, 3, 7, 9, 13, 11, 9, 27, 313, 751, 1377, 1121, 3799, 1673, 16305, 30665}},
	{5322, 16, 26377, {1, 3, 3, 13, 23, 17, 59, 47, 499, 525, 681, 3195, 1611, 7003, 7325, 53019}},
	{5323, 16, 26397, {1, 3, 1, 7, 23, 51, 59, 127, 67, 263, 1305, 2479, 637, 9441, 6329, 12857}},
	{5324, 16, 26404, {1, 1, 7, 7, 9, 3, 51, 193, 205, 503, 19, 2513, 7489, 9241, 15371, 20875}},
	{5325, 16, 26411, {1, 3, 3, 1, 1, 57, 17, 181, 23, 549, 769, 2325, 3669, 7017, 25601, 64479}},
	{5326, 16, 26422, {1, 3, 1, 15, 5, 55, 53, 13, 327, 447, 1031, 1599, 3639, 15305, 1387, 16035}},
	{5327, 16, 26451, {1, 3, 7, 15, 9, 41, 53, 113, 97, 99, 1377, 4047, 3713, 8891, 5601, 5853}},
	{5328, 16, 26454, {1, 1, 7, 9, 9, 7, 29, 249, 411, 315, 181, 2153, 6135, 6947, 27595, 15553}},
	{5329, 16, 26463, {1, 1, 7, 11, 3, 57, 35, 55, 165, 313, 577, 3049, 4259, 4231, 7225, 58973}},
	{5330, 16, 26488, {1, 1, 1, 1, 15, 43, 53, 143, 157, 843, 465, 3897, 4797, 12305, 28807, 46381}},
	{5331, 16, 26498, {1, 3, 7, 9, 17, 3, 99, 61, 475, 507, 831, 2207, 367, 27, 23205, 2303}},
	{5332, 16, 26509, {1, 1, 3, 11, 27, 29, 99, 237, 43, 955, 361, 3231, 1863, 7973, 8969, 58663}},
	{5333, 16, 26512, {1, 3, 1, 7, 15, 15, 11, 251, 135, 261, 675, 3723, 7675, 7993, 1725, 41149}},
	{5334, 16, 26517, {1, 3, 3, 9, 29, 11, 105, 37, 151, 215, 1911, 4051, 5427, 11019, 9073, 33129}},
	{5335, 16, 26534, {1, 3, 3, 1, 19, 7, 103, 81, 371, 253, 1043, 1231, 6497, 10377, 2349, 29047}},
	{5336, 16, 26545, {1, 3, 7, 9, 15, 11, 85, 61, 507, 629, 811, 3883, 1435, 13035, 31913, 2153}},
	{5337, 16, 26546, {1, 1, 5, 11, 13, 7, 63, 147, 117, 223, 1217, 3317, 3275, 6851, 2917, 55901}},
	{5338, 16, 26636, {1, 3, 3, 9, 7, 21, 1, 63, 117, 297, 405, 797, 337, 10173, 8327, 29157}},
	{5339, 16, 26749, {1, 1, 7, 11, 31, 63, 97, 191, 259, 421, 1829, 2117, 4203, 11919, 18001, 55791}},
	{5340, 16, 26753, {1, 3, 7, 9, 21, 13, 125, 247, 133, 611, 463, 227, 7815, 8877, 17961, 3641}},
	{5341, 16, 26759, {1, 1, 7, 9, 27, 21, 97, 165, 371, 715, 491, 3929, 3067, 12501, 5511, 18217}},
	{5342, 16, 26774, {1, 3, 3, 15, 27, 17, 81, 161, 263, 273, 135, 1159, 7535, 4581, 16065, 11493}},
	{5343, 16, 26789, {1, 3, 3, 7, 11, 59, 111, 47, 381, 413, 243, 2173, 4957, 2451, 15669, 22071}},
	{5344, 16, 26808, {1, 3, 7, 5, 3, 31, 65, 131, 111, 141, 1891, 2983, 3331, 769, 24075, 40865}},
	{5345, 16, 26825, {1, 3, 7, 11, 11, 63, 7, 213, 333, 111, 1235, 961, 3749, 9123, 5067, 9657}},
	{5346, 16, 26831, {1, 3, 3, 1, 9, 33, 1, 225, 37, 951, 1995, 3215, 3117, 3723, 15013, 64525}},
	{5347, 16, 26859, {1, 3, 3, 13, 29, 19, 103, 65, 67, 423, 1679, 3791, 5551, 11711, 195, 52291}},
	{5348, 16, 26888, {1, 3, 1, 15, 31, 7, 65, 249, 489, 287, 1385, 1075, 1357, 13593, 20853, 46221}},
	{5349, 16, 26918, {1, 1, 1, 13, 23, 45, 29, 175, 147, 101, 1007, 1867, 5387, 12683, 29609, 55861}},
	{5350, 16, 26929, {1, 3, 5, 13, 21, 31, 85, 187, 183, 179, 1337, 481, 71, 6117, 2177, 27915}},
	{5351, 16, 26950, {1, 3, 5, 1, 15, 5, 11, 141, 205, 177, 891, 3591, 4371, 889, 12957, 61083}},
	{5352, 16, 26954, {1, 3, 7, 7, 11, 39, 81, 241, 13, 757, 521, 3029, 2345, 12385, 20683, 64053}},
	{5353, 16, 26973, {1, 1, 5, 13, 7, 3, 77, 211, 215, 97, 1409, 1021, 1267, 4785, 27231, 2877}},
	{5354, 16, 26997, {1, 3, 5, 3, 11, 57, 93, 39, 415, 179, 1033, 3267, 5383, 10451, 27117, 10711}},
	{5355, 16, 26998, {1, 1, 1, 1, 3, 45, 93, 179, 453, 995, 1423, 3849, 4381, 15789, 20789, 18775}},
	{5356, 16, 27008, {1, 3, 1, 3, 13, 25, 33, 165, 351, 887, 1109, 195, 8131, 3061, 16743, 22997}},
	{5357, 16, 27038, {1, 3, 1, 5, 23, 35, 93, 155, 333, 603, 1571, 229, 2979, 6295, 20793, 40901}},
	{5358, 16, 27109, {1, 3, 3, 11, 29, 57, 101, 61, 487, 719, 1009, 1933, 7815, 15329, 12489, 26105}},
	{5359, 16, 27127, {1, 3, 3, 9, 23, 59, 73, 13, 141, 815, 1819, 3557, 2555, 5901, 23039, 62321}},
	{5360, 16, 27150, {1, 1, 3, 5, 19, 49, 27, 139, 35, 995, 565, 323, 6439, 15679, 27017, 30889}},
	{5361, 16, 27168, {1, 3, 7, 3, 1, 3, 27, 153, 235, 59, 989, 2763, 4197, 3931, 31227, 27129}},
	{5362, 16, 27178, {1, 3, 1, 15, 23, 45, 71, 205, 465, 451, 347, 1909, 3287, 8363, 9081, 35641}},
	{5363, 16, 27212, {1, 1, 5, 1, 25, 29, 17, 201, 463, 903, 1729, 3435, 2483, 10835, 14315, 52505}},
	{5364, 16, 27224, {1, 1, 1, 15, 13, 23, 3, 175, 273, 305, 1945, 3319, 7777, 9411, 4209, 4047}},
	{5365, 16, 27229, {1, 1, 5, 15, 25, 5, 71, 35, 307, 89, 301, 3465, 1497, 13467, 21911, 13611}},
	{5366, 16, 27246, {1, 3, 1, 7, 11, 7, 33, 241, 349, 751, 633, 3281, 6733, 13833, 23605, 34307}},
	{5367, 16, 27251, {1, 1, 1, 15, 17, 27, 45, 13, 259, 843, 1207, 1735, 4063, 6259, 1751, 45107}},
	{5368, 16, 27257, {1, 1, 5, 15, 29, 51, 73, 95, 5, 31, 933, 423, 5505, 2193, 14919, 2715}},
	{5369, 16, 27258, {1, 3, 1, 3, 23, 5, 29, 7, 271, 485, 827, 1159, 77, 16337, 27933, 18741}},
	{5370, 16, 27260, {1, 3, 7, 9, 23, 33, 47, 191, 59, 301, 1277, 3745, 7837, 799, 2861, 25853}},
	{5371, 16, 27287, {1, 3, 7, 13, 13, 39, 33, 91, 279, 855, 1917, 3601, 3971, 6193, 16951, 6115}},
	{5372, 16, 27300, {1, 1, 3, 13, 15, 59, 89, 239, 313, 545, 431, 3823, 5741, 14981, 2647, 42813}},
	{5373, 16, 27315, {1, 1, 1, 3, 17, 21, 45, 37, 343, 737, 1795, 2659, 2897, 7683, 15191, 1393}},
	{5374, 16, 27329, {1, 1, 3, 3, 19, 55, 27, 201, 459, 953, 1531, 671, 5667, 11695, 149, 14605}},
	{5375, 16, 27332, {1, 3, 7, 13, 9, 63, 67, 229, 69, 819, 859, 2035, 5725, 13403, 24197, 2599}},
	{5376, 16, 27349, {1, 1, 7, 7, 1, 59, 45, 105, 327, 779, 59, 791, 7593, 8189, 28161, 13339}},
	{5377, 16, 27350, {1, 3, 3, 15, 25, 55, 125, 189, 327, 733, 115, 3541, 5227, 12143, 32719, 15785}},
	{5378, 16, 27354, {1, 3, 3, 7, 19, 51, 35, 63, 507, 89, 1441, 2369, 4927, 7953, 10193, 8331}},
	{5379, 16, 27359, {1, 1, 5, 5, 21, 1, 41, 49, 217, 1001, 1649, 2789, 5267, 1525, 3811, 40785}},
	{5380, 16, 27377, {1, 1, 7, 15, 31, 21, 33, 183, 425, 393, 367, 3253, 3047, 465, 28229, 44743}},
	{5381, 16, 27378, {1, 3, 7, 5, 1, 23, 11, 71, 269, 707, 5, 2931, 1959, 15089, 9299, 43927}},
	{5382, 16, 27383, {1, 3, 5, 15, 21, 51, 31, 15, 49, 481, 297, 3871, 751, 10661, 26401, 62923}},
	{5383, 16, 27384, {1, 3, 1, 7, 17, 27, 35, 255, 205, 865, 1659, 1921, 5457, 11633, 2825, 13549}},
	{5384, 16, 27387, {1, 1, 5, 15, 17, 35, 83, 237, 437, 7, 2001, 2225, 2601, 10841, 7953, 20651}},
	{5385, 16, 27392, {1, 1, 1, 3, 3, 37, 43, 135, 451, 203, 1319, 261, 3889, 14489, 9635, 52545}},
	{5386, 16, 27402, {1, 3, 3, 13, 15, 41, 95, 207, 43, 997, 207, 3459, 5995, 5187, 15851, 28551}},
	{5387, 16, 27438, {1, 1, 1, 5, 23, 57, 49, 101, 303, 921, 745, 1407, 7071, 2765, 18703, 32671}},
	{5388, 16, 27481, {1, 1, 7, 13, 9, 59, 65, 157, 209, 295, 107, 3175, 3401, 1197, 1875, 9033}},
	{5389, 16, 27482, {1, 1, 3, 3, 17, 9, 101, 75, 177, 905, 1013, 397, 3421, 6475, 15897, 11269}},
	{5390, 16, 27494, {1, 3, 1, 5, 29, 53, 105, 83, 383, 137, 1169, 1245, 6973, 8701, 317, 10073}},
	{5391, 16, 27531, {1, 1, 1, 3, 15, 55, 69, 219, 507, 707, 945, 397, 779, 4157, 10333, 7869}},
	{5392, 16, 27542, {1, 3, 1, 3, 9, 21, 125, 153, 107, 969, 1979, 651, 1199, 11419, 17043, 32269}},
	{5393, 16, 27546, {1, 1, 1, 7, 1, 29, 71, 127, 209, 853, 1515, 1169, 5055, 9981, 30291, 29569}},
	{5394, 16, 27564, {1, 3, 1, 11, 1, 1, 109, 251, 329, 633, 2021, 1237, 2147, 11471, 26025, 19649}},
	{5395, 16, 27567, {1, 1, 5, 1, 5, 7, 77, 175, 251, 143, 711, 1241, 2133, 9993, 9203, 24949}},
	{5396, 16, 27582, {1, 3, 5, 11, 19, 11, 101, 83, 91, 595, 753, 2389, 1887, 11569, 29759, 55785}},
	{5397, 16, 27608, {1, 1, 1, 3, 29, 47, 49, 249, 495, 451, 1887, 2547, 543, 2755, 17207, 24379}},
	{5398, 16, 27611, {1, 3, 7, 7, 19, 15, 95, 143, 109, 221, 2041, 3593, 4571, 14547, 14217, 16711}},
	{5399, 16, 27624, {1, 3, 5, 13, 27, 55, 31, 209, 39, 989, 1435, 1665, 7265, 14127, 13517, 37647}},
	{5400, 16, 27629, {1, 1, 3, 7, 1, 49, 63, 71, 249, 371, 435, 3591, 2677, 143, 28897, 38981}},
	{5401, 16, 27655, {1, 1, 7, 7, 21, 9, 53, 221, 23, 515, 1971, 3759, 3511, 10207, 12929, 42021}},
	{5402, 16, 27667, {1, 3, 1, 13, 25, 21, 3, 85, 421, 19, 663, 3219, 3541, 13021, 5765, 39623}},
	{5403, 16, 27676, {1, 3, 1, 7, 11, 5, 125, 169, 293, 715, 1111, 2965, 4815, 6047, 27207, 23093}},
	{5404, 16, 27710, {1, 1, 5, 13, 11, 15, 37, 201, 457, 551, 821, 25, 435, 14307, 25855, 1811}},
	{5405, 16, 27724, {1, 3, 3, 9, 27, 15, 1, 253, 217, 549, 1013, 2277, 4171, 3813, 19857, 8641}},
	{5406, 16, 27745, {1, 3, 5, 5, 29, 37, 71, 49, 163, 949, 425, 2459, 945, 13125, 13981, 21669}},
	{5407, 16, 27752, {1, 3, 3, 9, 17, 23, 53, 235, 83, 887, 439, 1939, 7601, 15275, 15739, 17623}},
	{5408, 16, 27770, {1, 3, 5, 13, 7, 51, 73, 67, 167, 849, 2021, 2977, 6591, 3721, 5827, 40897}},
	{5409, 16, 27779, {1, 1, 5, 11, 27, 19, 81, 181, 383, 23, 1061, 3327, 1671, 7113, 7435, 17591}},
	{5410, 16, 27781, {1, 3, 3, 7, 25, 33, 73, 23, 103, 821, 917, 1425, 4739, 7227, 12365, 29509}},
	{5411, 16, 27791, {1, 1, 1, 7, 3, 37, 81, 231, 135, 663, 1983, 399, 6881, 14991, 4957, 20913}},
	{5412, 16, 27809, {1, 3, 7, 15, 25, 41, 65, 215, 301, 471, 1669, 65, 227, 9307, 29867, 9503}},
	{5413, 16, 27810, {1, 1, 7, 3, 25, 47, 31, 63, 53, 995, 33, 1297, 3423, 12301, 16255, 14467}},
	{5414, 16, 27815, {1, 3, 1, 1, 31, 25, 79, 131, 353, 169, 1425, 2257, 2631, 1559, 793, 48383}},
	{5415, 16, 27827, {1, 3, 3, 5, 31, 9, 93, 35, 503, 243, 595, 2939, 771, 7333, 13429, 59457}},
	{5416, 16, 27834, {1, 3, 1, 7, 5, 51, 21, 237, 453, 743, 775, 2207, 453, 12077, 12283, 9735}},
	{5417, 16, 27865, {1, 3, 1, 15, 5, 47, 59, 239, 87, 97, 885, 3191, 2547, 13227, 7433, 4989}},
	{5418, 16, 27899, {1, 3, 5, 15, 21, 33, 41, 155, 509, 317, 517, 1101, 133, 1897, 8235, 57673}},
	{5419, 16, 27901, {1, 1, 5, 15, 7, 9, 59, 155, 415, 831, 1173, 1263, 5451, 7181, 7233, 51483}},
	{5420, 16, 27914, {1, 1, 7, 3, 31, 43, 71, 39, 155, 529, 895, 827, 3203, 4625, 32185, 53507}},
	{5421, 16, 27950, {1, 3, 1, 11, 15, 17, 85, 141, 277, 439, 1775, 4015, 4457, 281, 22455, 47591}},
	{5422, 16, 27994, {1, 3, 5, 11, 25, 41, 93, 39, 51, 655, 1347, 3109, 2479, 9057, 18939, 26217}},
	{5423, 16, 28005, {1, 3, 3, 11, 31, 41, 7, 189, 241, 443, 65, 1723, 4817, 13405, 9641, 63965}},
	{5424, 16, 28009, {1, 1, 5, 3, 19, 29, 111, 11, 399, 277, 425, 1331, 5365, 14521, 16449, 29411}},
	{5425, 16, 28033, {1, 1, 3, 9, 25, 53, 91, 175, 481, 307, 1025, 71, 7425, 10667, 4053, 25605}},
	{5426, 16, 28039, {1, 3, 7, 7, 3, 13, 75, 175, 467, 363, 1889, 1759, 1155, 5511, 13047, 39637}},
	{5427, 16, 28060, {1, 3, 7, 9, 5, 21, 65, 43, 223, 97, 835, 2253, 3313, 9817, 23015, 55365}},
	{5428, 16, 28067, {1, 1, 1, 13, 9, 63, 95, 61, 417, 713, 1469, 1815, 795, 13609, 1567, 33535}},
	{5429, 16, 28069, {1, 3, 7, 1, 25, 45, 41, 27, 53, 407, 1633, 1317, 6267, 3293, 8899, 45235}},
	{5430, 16, 28099, {1, 3, 5, 11, 23, 47, 91, 211, 41, 775, 1301, 1407, 7931, 4491, 7579, 62321}},
	{5431, 16, 28113, {1, 1, 1, 7, 23, 15, 57, 31, 437, 293, 1999, 2589, 5453, 2519, 15533, 26949}},
	{5432, 16, 28114, {1, 3, 1, 9, 1, 27, 41, 165, 129, 297, 1887, 1171, 201, 5817, 24503, 38911}},
	{5433, 16, 28139, {1, 3, 1, 7, 1, 11, 63, 225, 191, 623, 1281, 3275, 167, 14697, 4905, 47289}},
	{5434, 16, 28142, {1, 3, 7, 7, 15, 47, 87, 177, 303, 391, 355, 3997, 7557, 6201, 20531, 22483}},
	{5435, 16, 28153, {1, 3, 3, 15, 17, 31, 111, 87, 61, 477, 1581, 3787, 5919, 10511, 2607, 62951}},
	{5436, 16, 28166, {1, 3, 3, 9, 29, 19, 63, 27, 205, 915, 1131, 3821, 673, 2875, 12703, 14367}},
	{5437, 16, 28172, {1, 3, 7, 1, 21, 19, 25, 97, 281, 635, 629, 181, 5207, 11133, 3687, 3489}},
	{5438, 16, 28183, {1, 3, 3, 9, 5, 63, 3, 21, 385, 713, 1805, 3583, 2807, 15455, 13057, 39771}},
	{5439, 16, 28194, {1, 3, 5, 9, 3, 59, 1, 253, 123, 405, 575, 3911, 4609, 11869, 23593, 947}},
	{5440, 16, 28232, {1, 1, 7, 5, 21, 27, 101, 221, 413, 153, 1647, 3637, 803, 5697, 20761, 61137}},
	{5441, 16, 28245, {1, 3, 7, 13, 31, 35, 111, 253, 187, 499, 465, 157, 5551, 10417, 323, 34913}},
	{5442, 16, 28246, {1, 1, 1, 7, 11, 41, 29, 65, 393, 69, 1373, 2291, 7807, 13159, 13735, 31001}},
	{5443, 16, 28252, {1, 3, 7, 13, 31, 49, 1, 35, 377, 11, 427, 2803, 1725, 9165, 22633, 63985}},
	{5444, 16, 28265, {1, 3, 7, 13, 3, 41, 27, 43, 269, 599, 1035, 3681, 309, 6011, 1065, 27901}},
	{5445, 16, 28301, {1, 3, 5, 13, 1, 19, 105, 143, 425, 883, 1669, 155, 189, 8573, 10759, 25507}},
	{5446, 16, 28323, {1, 3, 5, 1, 15, 37, 115, 9, 149, 79, 1733, 1045, 1849, 3289, 13957, 63171}},
	{5447, 16, 28344, {1, 1, 3, 9, 17, 27, 49, 201, 155, 901, 47, 1585, 4419, 8117, 25425, 14699}},
	{5448, 16, 28362, {1, 1, 7, 13, 19, 55, 19, 21, 427, 77, 1295, 1471, 6271, 7985, 19337, 12701}},
	{5449, 16, 28400, {1, 1, 1, 1, 11, 49, 101, 53, 175, 157, 839, 2713, 6149, 6391, 8089, 27739}},
	{5450, 16, 28417, {1, 3, 1, 1, 5, 21, 121, 199, 107, 221, 993, 1737, 409, 2545, 9287, 54605}},
	{5451, 16, 28454, {1, 1, 1, 3, 25, 25, 51, 121, 371, 861, 967, 3257, 6221, 11701, 27897, 42509}},
	{5452, 16, 28466, {1, 1, 1, 3, 17, 25, 101, 191, 313, 817, 815, 1855, 7999, 12649, 23385, 26081}},
	{5453, 16, 28468, {1, 1, 5, 1, 25, 55, 51, 237, 63, 943, 455, 619, 2381, 9773, 14575, 34205}},
	{5454, 16, 28477, {1, 3, 3, 3, 13, 49, 101, 37, 457, 727, 1009, 2389, 4841, 16303, 9599, 17773}},
	{5455, 16, 28498, {1, 1, 7, 9, 19, 59, 111, 205, 19, 229, 1755, 1169, 7767, 13335, 19669, 33269}},
	{5456, 16, 28510, {1, 3, 1, 15, 29, 1, 51, 167, 7, 415, 1295, 3165, 1241, 12859, 5531, 20083}},
	{5457, 16, 28513, {1, 1, 3, 7, 7, 51, 31, 221, 57, 643, 1461, 3951, 6237, 5757, 1907, 40471}},
	{5458, 16, 28571, {1, 3, 3, 5, 23, 39, 49, 177, 183, 117, 1379, 3803, 771, 12723, 22291, 32667}},
	{5459, 16, 28573, {1, 1, 3, 13, 27, 17, 39, 27, 313, 141, 1421, 2967, 2213, 1915, 23219, 15113}},
	{5460, 16, 28578, {1, 1, 1, 11, 5, 55, 51, 55, 389, 895, 57, 1447, 1497, 2799, 19585, 11587}},
	{5461, 16, 28587, {1, 1, 5, 13, 11, 55, 91, 77, 69, 131, 93, 1383, 3321, 10487, 15087, 8539}},
	{5462, 16, 28601, {1, 1, 3, 9, 23, 49, 107, 131, 363, 733, 1189, 3575, 7815, 10071, 20291, 7533}},
	{5463, 16, 28646, {1, 1, 3, 15, 31, 31, 73, 15, 199, 17, 761, 3271, 1419, 12985, 32717, 37317}},
	{5464, 16, 28663, {1, 3, 1, 13, 23, 9, 3, 59, 109, 729, 1321, 4023, 7041, 14445, 22205, 8993}},
	{5465, 16, 28681, {1, 1, 3, 15, 19, 43, 99, 59, 491, 479, 715, 2235, 7493, 889, 31465, 1375}},
	{5466, 16, 28682, {1, 3, 3, 15, 9, 47, 35, 115, 227, 615, 605, 1143, 5923, 10939, 9497, 24911}},
	{5467, 16, 28699, {1, 1, 3, 15, 23, 53, 111, 87, 423, 497, 85, 3525, 7341, 8885, 21543, 30083}},
	{5468, 16, 28706, {1, 1, 5, 3, 21, 5, 117, 157, 407, 743, 715, 1883, 4425, 10187, 6395, 43673}},
	{5469, 16, 28708, {1, 3, 3, 3, 31, 39, 119, 77, 269, 891, 1391, 3085, 2881, 10639, 3391, 44911}},
	{5470, 16, 28717, {1, 3, 7, 5, 7, 5, 115, 91, 5, 107, 1401, 1409, 1811, 737, 5399, 9119}},
	{5471, 16, 28720, {1, 1, 5, 9, 17, 45, 107, 15, 397, 299, 1219, 1675, 963, 10111, 31679, 13809}},
	{5472, 16, 28735, {1, 1, 3, 7, 29, 17, 43, 95, 261, 601, 1091, 3633, 1357, 13461, 16583, 12183}},
	{5473, 16, 28761, {1, 1, 5, 1, 19, 55, 5, 195, 187, 427, 421, 1717, 4223, 2943, 23147, 32985}},
	{5474, 16, 28783, {1, 3, 1, 3, 3, 23, 69, 95, 347, 273, 1223, 3061, 1587, 4707, 32415, 53991}},
	{5475, 16, 28788, {1, 1, 7, 13, 15, 13, 29, 151, 325, 949, 2029, 813, 5339, 14165, 1159, 56917}},
	{5476, 16, 28811, {1, 1, 1, 13, 9, 33, 67, 109, 215, 313, 1407, 3543, 2403, 5051, 20367, 13527}},
	{5477, 16, 28825, {1, 3, 1, 9, 5, 1, 9, 195, 497, 495, 1107, 745, 1647, 10637, 1933, 44965}},
	{5478, 16, 28838, {1, 1, 3, 9, 13, 19, 49, 183, 497, 519, 1433, 519, 4317, 2359, 10349, 63789}},
	{5479, 16, 28850, {1, 3, 5, 9, 29, 45, 55, 163, 189, 533, 275, 237, 5453, 8895, 6377, 14117}},
	{5480, 16, 28891, {1, 3, 7, 5, 25, 3, 111, 241, 139, 383, 689, 3481, 2557, 11163, 5275, 14671}},
	{5481, 16, 28897, {1, 3, 3, 9, 3, 5, 5, 141, 507, 331, 645, 1957, 5857, 2083, 24717, 11131}},
	{5482, 16, 28932, {1, 1, 5, 1, 11, 49, 113, 45, 491, 945, 1467, 3485, 6369, 15983, 14489, 12679}},
	{5483, 16, 28975, {1, 3, 7, 9, 11, 41, 77, 127, 147, 635, 1757, 587, 7423, 4233, 14875, 30531}},
	{5484, 16, 28998, {1, 3, 3, 9, 17, 29, 21, 249, 155, 441, 1443, 2093, 1967, 2117, 5815, 3857}},
	{5485, 16, 29052, {1, 3, 5, 3, 11, 55, 75, 157, 105, 507, 309, 3737, 2645, 7547, 29373, 62775}},
	{5486, 16, 29090, {1, 1, 3, 3, 11, 29, 49, 241, 21, 653, 1273, 715, 8123, 14241, 25257, 1681}},
	{5487, 16, 29096, {1, 1, 7, 5, 11, 31, 33, 215, 243, 369, 247, 3365, 4065, 9389, 32457, 58393}},
	{5488, 16, 29104, {1, 3, 5, 3, 31, 55, 51, 201, 439, 835, 1805, 25, 7987, 10611, 26893, 43663}},
	{5489, 16, 29113, {1, 1, 5, 9, 27, 51, 29, 31, 17, 163, 71, 603, 3083, 12439, 11043, 6471}},
	{5490, 16, 29133, {1, 1, 5, 7, 13, 1, 91, 109, 213, 721, 1345, 3037, 3047, 5209, 15559, 17467}},
	{5491, 16, 29142, {1, 1, 3, 9, 19, 37, 93, 185, 107, 859, 501, 3843, 1631, 4389, 2215, 52225}},
	{5492, 16, 29170, {1, 1, 3, 3, 25, 5, 119, 17, 33, 841, 997, 439, 6135, 7405, 8445, 40087}},
	{5493, 16, 29192, {1, 1, 7, 15, 19, 17, 101, 43, 423, 647, 29, 1143, 3259, 7807, 15929, 809}},
	{5494, 16, 29221, {1, 1, 7, 13, 21, 57, 83, 101, 183, 309, 171, 3173, 7919, 7263, 29403, 11055}},
	{5495, 16, 29236, {1, 1, 1, 13, 3, 1, 57, 15, 435, 713, 459, 847, 3115, 191, 19809, 43037}},
	{5496, 16, 29246, {1, 1, 7, 7, 17, 45, 91, 117, 157, 647, 121, 4091, 3611, 14169, 19883, 9069}},
	{5497, 16, 29293, {1, 1, 7, 7, 1, 47, 21, 253, 419, 157, 549, 2105, 4475, 3127, 3939, 5809}},
	{5498, 16, 29305, {1, 1, 5, 7, 15, 7, 71, 195, 87, 757, 77, 1391, 151, 12995, 26403, 17789}},
	{5499, 16, 29312, {1, 1, 1, 15, 15, 3, 79, 43, 475, 263, 1195, 2385, 5955, 7039, 15625, 19263}},
	{5500, 16, 29339, {1, 1, 5, 13, 13, 29, 5, 29, 489, 929, 2027, 2771, 6899, 14173, 13747, 1019}},
	{5501, 16, 29365, {1, 1, 7, 1, 5, 45, 37, 85, 221, 871, 627, 3445, 4853, 4243, 21651, 30201}},
	{5502, 16, 29389, {1, 3, 7, 11, 9, 49, 73, 245, 161, 321, 579, 2641, 6653, 5513, 11555, 53091}},
	{5503, 16, 29402, {1, 1, 7, 7, 25, 63, 101, 179, 497, 113, 9, 549, 5341, 6097, 13305, 52421}},
	{5504, 16, 29423, {1, 3, 3, 7, 21, 7, 89, 79, 137, 651, 189, 3025, 1403, 4559, 32611, 1857}},
	{5505, 16, 29443, {1, 3, 1, 13, 27, 55, 61, 135, 81, 195, 799, 3477, 4873, 2691, 29769, 59033}},
	{5506, 16, 29445, {1, 3, 3, 15, 29, 11, 7, 11, 151, 649, 1511, 2327, 6921, 12911, 3571, 35039}},
	{5507, 16, 29463, {1, 1, 5, 11, 25, 19, 49, 133, 455, 373, 1827, 3619, 2127, 3365, 11327, 52785}},
	{5508, 16, 29473, {1, 3, 5, 1, 9, 19, 107, 171, 205, 93, 1557, 2693, 4297, 4415, 20407, 19221}},
	{5509, 16, 29493, {1, 3, 3, 11, 15, 45, 37, 143, 61, 759, 2047, 2465, 3923, 9477, 30831, 46377}},
	{5510, 16, 29506, {1, 3, 7, 11, 17, 51, 117, 129, 77, 579, 1167, 1575, 1967, 10099, 22137, 31431}},
	{5511, 16, 29518, {1, 1, 5, 13, 31, 61, 67, 37, 49, 283, 235, 783, 7353, 5149, 12245, 18725}},
	{5512, 16, 29532, {1, 1, 5, 3, 17, 33, 35, 83, 359, 253, 1911, 913, 6481, 4635, 24223, 19693}},
	{5513, 16, 29560, {1, 1, 1, 13, 19, 15, 81, 131, 417, 969, 1911, 2829, 3097, 5333, 11175, 52269}},
	{5514, 16, 29590, {1, 3, 7, 15, 5, 39, 19, 205, 329, 83, 1473, 3259, 6409, 12297, 30557, 40917}},
	{5515, 16, 29594, {1, 3, 1, 15, 17, 33, 123, 185, 501, 299, 621, 929, 5797, 10539, 12321, 61043}},
	{5516, 16, 29637, {1, 3, 3, 1, 7, 51, 119, 19, 17, 203, 373, 2145, 2367, 9965, 28071, 50083}},
	{5517, 16, 29647, {1, 1, 1, 5, 1, 35, 43, 243, 91, 793, 1299, 2705, 7987, 1291, 10147, 17863}},
	{5518, 16, 29650, {1, 3, 5, 15, 27, 13, 99, 33, 179, 479, 897, 1113, 1639, 12321, 23987, 36219}},
	{5519, 16, 29655, {1, 1, 5, 9, 29, 41, 85, 9, 389, 583, 293, 1727, 2575, 13767, 15443, 40027}},
	{5520, 16, 29661, {1, 1, 7, 11, 29, 33, 93, 115, 51, 747, 1569, 3557, 869, 1991, 29877, 44131}},
	{5521, 16, 29680, {1, 3, 7, 7, 29, 11, 33, 137, 411, 689, 1815, 1789, 6557, 5973, 19445, 49449}},
	{5522, 16, 29721, {1, 1, 5, 1, 17, 3, 77, 55, 351, 325, 983, 3935, 819, 14127, 18893, 62553}},
	{5523, 16, 29751, {1, 3, 3, 1, 15, 33, 25, 159, 135, 385, 837, 3615, 1649, 1687, 3421, 47579}},
	{5524, 16, 29755, {1, 3, 1, 7, 17, 25, 125, 169, 469, 919, 1789, 863, 2827, 949, 21347, 10517}},
	{5525, 16, 29760, {1, 3, 1, 11, 27, 19, 45, 255, 175, 483, 1073, 3779, 611, 2809, 179, 19767}},
	{5526, 16, 29772, {1, 1, 3, 1, 21, 61, 47, 171, 179, 85, 61, 1209, 4005, 11439, 8477, 27229}},
	{5527, 16, 29778, {1, 1, 1, 1, 3, 1, 43, 159, 261, 411, 1449, 1621, 3681, 3465, 24029, 3493}},
	{5528, 16, 29799, {1, 3, 1, 11, 5, 13, 9, 23, 369, 769, 363, 3329, 409, 13151, 30269, 9621}},
	{5529, 16, 29824, {1, 1, 5, 1, 13, 39, 121, 39, 295, 981, 1151, 4039, 8179, 5007, 25527, 1249}},
	{5530, 16, 29841, {1, 3, 7, 5, 17, 21, 47, 233, 211, 349, 643, 109, 7553, 11453, 30967, 30959}},
	{5531, 16, 29842, {1, 1, 5, 3, 31, 39, 105, 137, 487, 855, 107, 1567, 2385, 2889, 25777, 33709}},
	{5532, 16, 29853, {1, 1, 1, 9, 1, 7, 9, 69, 465, 965, 355, 299, 3327, 14997, 14599, 2241}},
	{5533, 16, 29867, {1, 3, 5, 11, 5, 39, 69, 203, 367, 611, 199, 3931, 5039, 8683, 8675, 49151}},
	{5534, 16, 29949, {1, 1, 7, 13, 31, 35, 101, 213, 273, 827, 203, 2773, 4131, 1397, 15311, 62903}},
	{5535, 16, 29950, {1, 3, 7, 9, 23, 41, 33, 213, 411, 965, 563, 3035, 247, 15019, 20429, 61081}},
	{5536, 16, 29964, {1, 1, 5, 5, 5, 1, 1, 203, 27, 199, 67, 1301, 7831, 12839, 2777, 6325}},
	{5537, 16, 29967, {1, 3, 1, 11, 27, 3, 11, 173, 9, 121, 1701, 2741, 29, 16319, 15849, 11989}},
	{5538, 16, 29985, {1, 1, 5, 13, 17, 49, 125, 153, 261, 603, 1617, 3967, 6083, 7745, 19683, 49885}},
	{5539, 16, 29992, {1, 3, 3, 7, 23, 13, 39, 169, 135, 361, 579, 1443, 7615, 2389, 5669, 651}},
	{5540, 16, 30000, {1, 3, 5, 9, 31, 19, 81, 83, 483, 93, 1895, 2285, 7771, 8281, 8353, 39677}},
	{5541, 16, 30020, {1, 1, 7, 7, 23, 51, 127, 25, 101, 611, 1095, 3013, 2685, 8153, 22629, 53355}},
	{5542, 16, 30024, {1, 1, 1, 11, 11, 37, 35, 127, 317, 877, 1591, 401, 4121, 9945, 12121, 28257}},
	{5543, 16, 30030, {1, 3, 5, 11, 23, 9, 43, 135, 37, 405, 2009, 2903, 3065, 6591, 8473, 58231}},
	{5544, 16, 30066, {1, 1, 3, 11, 21, 45, 21, 205, 425, 891, 357, 2609, 495, 7541, 2161, 37853}},
	{5545, 16, 30071, {1, 3, 1, 1, 25, 9, 113, 243, 317, 491, 997, 2023, 5869, 13643, 11483, 6733}},
	{5546, 16, 30078, {1, 3, 1, 15, 13, 3, 75, 25, 409, 421, 1817, 857, 4575, 12559, 1211, 62177}},
	{5547, 16, 30082, {1, 1, 3, 7, 17, 35, 115, 195, 217, 223, 1195, 749, 5619, 7265, 7369, 46907}},
	{5548, 16, 30096, {1, 1, 1, 13, 5, 57, 117, 161, 121, 533, 987, 3959, 5047, 15213, 15811, 41841}},
	{5549, 16, 30101, {1, 3, 7, 1, 19, 55, 97, 191, 217, 75, 1881, 3351, 3737, 12179, 22875, 28767}},
	{5550, 16, 30102, {1, 3, 1, 9, 15, 41, 9, 97, 491, 31, 1191, 963, 875, 8259, 2723, 9503}},
	{5551, 16, 30122, {1, 3, 7, 9, 3, 17, 21, 71, 1, 523, 2031, 3469, 3181, 8707, 6093, 8837}},
	{5552, 16, 30141, {1, 3, 5, 3, 5, 1, 11, 91, 33, 37, 643, 85, 4325, 4293, 8351, 28311}},
	{5553, 16, 30159, {1, 3, 7, 5, 15, 45, 47, 183, 391, 113, 493, 3607, 2541, 13521, 31613, 36049}},
	{5554, 16, 30168, {1, 1, 3, 9, 15, 33, 115, 69, 289, 217, 1875, 1339, 4995, 9073, 6909, 15977}},
	{5555, 16, 30177, {1, 1, 7, 3, 9, 29, 39, 219, 119, 369, 893, 1293, 4511, 15703, 11093, 30259}},
	{5556, 16, 30183, {1, 1, 5, 13, 19, 9, 17, 75, 149, 415, 35, 97, 563, 1689, 18311, 54291}},
	{5557, 16, 30197, {1, 1, 7, 3, 17, 15, 71, 29, 25, 883, 1801, 1675, 5585, 9413, 3813, 26673}},
	{5558, 16, 30213, {1, 1, 3, 15, 5, 13, 31, 41, 311, 411, 573, 281, 8075, 7163, 11817, 29121}},
	{5559, 16, 30231, {1, 1, 7, 9, 7, 57, 15, 141, 337, 123, 269, 3737, 6455, 2539, 13655, 59809}},
	{5560, 16, 30232, {1, 3, 1, 15, 15, 23, 111, 51, 429, 483, 1567, 1317, 8057, 1609, 30181, 35687}},
	{5561, 16, 30241, {1, 3, 7, 9, 25, 43, 67, 13, 319, 587, 1827, 443, 2031, 8563, 16173, 58667}},
	{5562, 16, 30253, {1, 3, 5, 13, 11, 63, 89, 105, 377, 257, 7, 4077, 5091, 5125, 25, 39033}},
	{5563, 16, 30259, {1, 3, 3, 1, 9, 29, 7, 87, 239, 469, 1851, 1711, 5797, 7137, 11405, 20175}},
	{5564, 16, 30262, {1, 3, 3, 1, 13, 17, 101, 209, 301, 95, 1181, 3091, 4451, 1241, 17057, 335}},
	{5565, 16, 30268, {1, 1, 1, 9, 31, 7, 81, 161, 391, 677, 115, 141, 5375, 7279, 1887, 1645}},
	{5566, 16, 30297, {1, 1, 1, 11, 27, 61, 3, 195, 189, 409, 1747, 331, 2931, 9535, 1369, 47233}},
	{5567, 16, 30316, {1, 3, 5, 15, 7, 15, 105, 255, 491, 689, 97, 1131, 3459, 7147, 27541, 62307}},
	{5568, 16, 30322, {1, 3, 5, 9, 5, 23, 1, 209, 233, 717, 1919, 1835, 5073, 10403, 28979, 1945}},
	{5569, 16, 30344, {1, 1, 3, 9, 3, 35, 107, 209, 255, 447, 227, 273, 443, 9367, 24105, 34095}},
	{5570, 16, 30350, {1, 1, 3, 11, 3, 33, 5, 165, 83, 787, 1555, 31, 4351, 16301, 27453, 56739}},
	{5571, 16, 30355, {1, 1, 5, 5, 29, 9, 127, 253, 281, 487, 441, 1129, 2811, 9113, 28855, 57117}},
	{5572, 16, 30429, {1, 1, 1, 13, 13, 1, 17, 143, 121, 917, 1571, 3777, 2297, 3691, 3001, 42327}},
	{5573, 16, 30445, {1, 1, 5, 1, 25, 7, 41, 245, 241, 929, 1203, 3755, 7113, 9333, 22549, 12253}},
	{5574, 16, 30477, {1, 3, 1, 11, 1, 13, 69, 73, 285, 975, 1331, 3411, 7777, 3489, 2763, 44297}},
	{5575, 16, 30513, {1, 3, 5, 11, 3, 37, 21, 105, 153, 307, 989, 627, 3127, 6169, 10573, 22139}},
	{5576, 16, 30520, {1, 3, 5, 15, 11, 11, 39, 21, 355, 437, 547, 2283, 6443, 5561, 6367, 53899}},
	{5577, 16, 30540, {1, 1, 1, 9, 25, 51, 97, 175, 131, 207, 1367, 2561, 7455, 8289, 5877, 4383}},
	{5578, 16, 30551, {1, 3, 7, 1, 29, 17, 7, 1, 43, 831, 591, 2145, 975, 909, 23107, 43987}},
	{5579, 16, 30573, {1, 3, 5, 13, 5, 47, 65, 65, 439, 807, 271, 1615, 1873, 10905, 30537, 3337}},
	{5580, 16, 30609, {1, 1, 1, 13, 29, 1, 53, 5, 307, 347, 1059, 545, 1129, 11883, 5969, 50433}},
	{5581, 16, 30622, {1, 1, 3, 5, 31, 29, 63, 201, 255, 803, 677, 1499, 1691, 14037, 18251, 6881}},
	{5582, 16, 30635, {1, 3, 1, 5, 5, 13, 13, 121, 505, 855, 467, 2803, 3297, 4689, 18443, 60757}},
	{5583, 16, 30658, {1, 1, 5, 13, 11, 19, 11, 201, 101, 431, 693, 1267, 6909, 7759, 2265, 6125}},
	{5584, 16, 30667, {1, 1, 7, 13, 9, 3, 37, 209, 269, 27, 1041, 2587, 4667, 11077, 27009, 27967}},
	{5585, 16, 30681, {1, 1, 5, 5, 1, 5, 127, 179, 463, 949, 1525, 231, 1201, 3283, 9929, 46677}},
	{5586, 16, 30684, {1, 3, 1, 15, 9, 11, 89, 129, 331, 833, 1415, 229, 2059, 13145, 30525, 33773}},
	{5587, 16, 30703, {1, 1, 7, 15, 7, 43, 95, 177, 313, 989, 483, 825, 1885, 4535, 8213, 39835}},
	{5588, 16, 30705, {1, 1, 7, 3, 19, 27, 45, 163, 17, 523, 1565, 3753, 7433, 14117, 8499, 40177}},
	{5589, 16, 30715, {1, 3, 3, 15, 23, 45, 95, 31, 55, 469, 383, 237, 6287, 5561, 20901, 48259}},
	{5590, 16, 30742, {1, 1, 3, 1, 23, 61, 101, 185, 35, 553, 463, 1169, 2875, 12491, 14327, 47999}},
	{5591, 16, 30748, {1, 3, 3, 13, 23, 29, 77, 21, 19, 3, 769, 1943, 2081, 9135, 29767, 11367}},
	{5592, 16, 30758, {1, 1, 5, 15, 5, 11, 59, 163, 355, 993, 375, 3181, 2675, 8515, 17007, 38467}},
	{5593, 16, 30767, {1, 1, 5, 13, 19, 5, 107, 83, 123, 843, 413, 2137, 7531, 3833, 6149, 55925}},
	{5594, 16, 30770, {1, 3, 1, 13, 23, 9, 41, 145, 265, 591, 1899, 3145, 5255, 13653, 12245, 25367}},
	{5595, 16, 30772, {1, 1, 3, 15, 1, 45, 119, 79, 121, 137, 1945, 2041, 2409, 1377, 29501, 29885}},
	{5596, 16, 30807, {1, 1, 7, 11, 27, 57, 75, 183, 341, 237, 1909, 2785, 5973, 9965, 21729, 45089}},
	{5597, 16, 30814, {1, 3, 5, 7, 21, 1, 41, 189, 131, 1021, 1375, 1463, 5985, 12499, 4115, 9131}},
	{5598, 16, 30824, {1, 3, 7, 15, 21, 19, 59, 171, 339, 841, 1725, 2909, 6437, 2499, 17191, 43275}},
	{5599, 16, 30857, {1, 3, 1, 1, 15, 55, 31, 199, 351, 183, 1819, 1873, 7877, 12407, 7881, 1663}},
	{5600, 16, 30875, {1, 1, 3, 15, 1, 61, 111, 61, 115, 243, 1281, 3195, 1229, 10973, 189, 36049}},
	{5601, 16, 30931, {1, 1, 3, 15, 13, 13, 3, 49, 61, 839, 1615, 1853, 3619, 7805, 25441, 8789}},
	{5602, 16, 30933, {1, 3, 1, 9, 27, 43, 7, 193, 397, 869, 1079, 1785, 6535, 1801, 29363, 59269}},
	{5603, 16, 30949, {1, 3, 5, 5, 31, 57, 37, 53, 41, 871, 809, 1235, 1011, 12979, 8749, 52151}},
	{5604, 16, 30953, {1, 1, 7, 13, 25, 59, 69, 117, 463, 587, 513, 297, 6991, 5905, 25737, 37249}},
	{5605, 16, 30982, {1, 1, 5, 1, 27, 19, 121, 97, 349, 793, 1971, 3057, 4781, 15841, 22625, 58637}},
	{5606, 16, 31010, {1, 1, 5, 5, 25, 31, 11, 133, 411, 239, 1071, 3473, 1733, 7175, 31841, 46665}},
	{5607, 16, 31012, {1, 3, 3, 13, 19, 25, 99, 175, 271, 175, 1755, 3597, 4615, 15207, 25573, 16089}},
	{5608, 16, 31039, {1, 1, 7, 11, 17, 19, 119, 91, 505, 791, 55, 2979, 7463, 10147, 23647, 33283}},
	{5609, 16, 31041, {1, 3, 1, 1, 21, 11, 43, 173, 239, 839, 1533, 1559, 549, 15621, 22133, 46387}},
	{5610, 16, 31061, {1, 1, 3, 13, 31, 15, 73, 15, 209, 267, 701, 2899, 1163, 10093, 7727, 44211}},
	{5611, 16, 31082, {1, 3, 1, 11, 29, 21, 5, 39, 421, 375, 411, 3693, 3901, 8507, 10883, 16189}},
	{5612, 16, 31087, {1, 3, 1, 7, 13, 13, 73, 167, 149, 677, 1435, 621, 2511, 13813, 13129, 55327}},
	{5613, 16, 31096, {1, 3, 5, 15, 7, 59, 83, 221, 77, 357, 281, 2689, 5629, 5837, 1701, 30811}},
	{5614, 16, 31115, {1, 3, 3, 11, 17, 1, 43, 95, 473, 981, 1487, 1337, 905, 3307, 22357, 181}},
	{5615, 16, 31130, {1, 1, 3, 7, 1, 27, 9, 3, 489, 1, 1265, 2463, 539, 12769, 825, 6149}},
	{5616, 16, 31168, {1, 3, 3, 3, 11, 27, 81, 237, 411, 241, 1613, 931, 6397, 4325, 29651, 49003}},
	{5617, 16, 31171, {1, 3, 3, 13, 1, 19, 55, 73, 47, 203, 1661, 1245, 6847, 2457, 25427, 33069}},
	{5618, 16, 31177, {1, 3, 7, 3, 7, 47, 11, 165, 391, 457, 301, 1213, 1913, 14531, 7847, 14811}},
	{5619, 16, 31180, {1, 3, 1, 9, 1, 9, 57, 203, 15, 733, 1131, 2751, 5869, 3165, 21497, 28881}},
	{5620, 16, 31191, {1, 3, 1, 5, 9, 7, 29, 85, 71, 571, 469, 2395, 2819, 8443, 2281, 50489}},
	{5621, 16, 31207, {1, 3, 5, 11, 13, 63, 47, 47, 349, 21, 861, 2217, 2945, 6967, 6605, 16459}},
	{5622, 16, 31247, {1, 1, 7, 5, 13, 3, 41, 53, 409, 289, 1225, 2965, 5283, 1785, 14443, 51755}},
	{5623, 16, 31249, {1, 3, 7, 13, 19, 1, 29, 191, 119, 37, 697, 1909, 481, 14157, 13425, 60581}},
	{5624, 16, 31285, {1, 3, 1, 13, 1, 15, 105, 79, 505, 681, 1741, 3683, 5775, 7479, 11387, 1321}},
	{5625, 16, 31303, {1, 1, 1, 11, 9, 35, 77, 73, 351, 217, 2029, 2845, 5143, 5677, 15465, 33123}},
	{5626, 16, 31310, {1, 1, 3, 3, 19, 49, 63, 109, 335, 743, 741, 1673, 3311, 3139, 25197, 13793}},
	{5627, 16, 31337, {1, 3, 1, 3, 29, 63, 79, 1, 493, 13, 1487, 4015, 6983, 1433, 26023, 55591}},
	{5628, 16, 31352, {1, 3, 3, 11, 1, 25, 57, 207, 309, 201, 1513, 1749, 3785, 9217, 11531, 40597}},
	{5629, 16, 31357, {1, 3, 7, 13, 3, 23, 69, 253, 311, 773, 807, 1063, 745, 4843, 25221, 55885}},
	{5630, 16, 31374, {1, 1, 3, 11, 29, 47, 67, 183, 11, 259, 5, 1935, 2295, 8105, 19139, 11707}},
	{5631, 16, 31379, {1, 1, 3, 3, 23, 3, 53, 165, 255, 501, 1547, 3649, 5631, 13307, 8869, 5595}},
	{5632, 16, 31388, {1, 1, 3, 5, 7, 29, 37, 223, 289, 925, 959, 309, 1479, 3141, 18661, 52123}},
	{5633, 16, 31410, {1, 3, 1, 1, 7, 59, 101, 219, 91, 793, 1103, 1485, 7547, 12889, 19097, 15613}},
	{5634, 16, 31416, {1, 1, 5, 15, 1, 17, 79, 83, 131, 683, 1611, 1635, 5405, 9621, 29489, 4801}},
	{5635, 16, 31467, {1, 1, 5, 7, 31, 63, 59, 125, 401, 261, 1445, 33, 187, 12913, 8639, 48413}},
	{5636, 16, 31495, {1, 3, 3, 13, 27, 37, 27, 99, 379, 851, 1311, 4051, 5483, 13935, 29679, 30905}},
	{5637, 16, 31504, {1, 1, 3, 1, 7, 57, 79, 23, 97, 561, 1083, 2327, 1545, 5387, 12119, 29717}},
	{5638, 16, 31507, {1, 1, 7, 7, 9, 41, 63, 165, 315, 247, 89, 2055, 7399, 1399, 2057, 39851}},
	{5639, 16, 31509, {1, 1, 1, 15, 9, 23, 7, 15, 457, 669, 661, 3269, 915, 3475, 15845, 59769}},
	{5640, 16, 31514, {1, 3, 7, 15, 17, 53, 83, 5, 457, 103, 1297, 2413, 1095, 7711, 27935, 56357}},
	{5641, 16, 31516, {1, 1, 3, 5, 17, 3, 81, 23, 165, 341, 781, 3583, 1751, 6763, 13937, 35331}},
	{5642, 16, 31530, {1, 1, 5, 11, 31, 21, 7, 63, 369, 867, 573, 45, 2781, 4507, 21553, 51933}},
	{5643, 16, 31555, {1, 1, 5, 15, 1, 37, 85, 133, 489, 733, 1471, 2089, 979, 7723, 7339, 59595}},
	{5644, 16, 31567, {1, 1, 1, 1, 7, 3, 3, 77, 137, 1009, 481, 1343, 397, 15865, 21701, 37509}},
	{5645, 16, 31570, {1, 3, 7, 5, 17, 57, 19, 245, 249, 289, 1847, 3057, 4905, 5905, 32459, 41305}},
	{5646, 16, 31586, {1, 1, 5, 1, 23, 23, 1, 177, 115, 337, 983, 421, 3135, 6319, 27109, 59641}},
	{5647, 16, 31598, {1, 3, 1, 5, 25, 1, 63, 73, 61, 967, 1567, 2645, 7347, 11877, 28777, 38507}},
	{5648, 16, 31605, {1, 1, 3, 9, 5, 41, 39, 101, 339, 337, 1079, 3861, 5049, 5601, 14377, 34093}},
	{5649, 16, 31609, {1, 3, 7, 7, 3, 47, 95, 157, 167, 1011, 1117, 3669, 7993, 11735, 8505, 64713}},
	{5650, 16, 31612, {1, 3, 1, 9, 3, 33, 11, 33, 65, 329, 401, 2659, 2851, 3903, 29791, 41613}},
	{5651, 16, 31626, {1, 1, 1, 15, 15, 17, 9, 69, 359, 41, 1475, 1919, 5829, 2189, 21295, 33255}},
	{5652, 16, 31634, {1, 3, 1, 3, 9, 23, 73, 247, 399, 775, 419, 3033, 865, 12595, 16345, 15079}},
	{5653, 16, 31655, {1, 3, 1, 5, 1, 17, 33, 23, 419, 585, 673, 929, 6955, 10247, 12647, 29107}},
	{5654, 16, 31681, {1, 3, 3, 13, 9, 33, 11, 13, 127, 529, 1219, 2401, 6459, 14745, 5123, 53023}},
	{5655, 16, 31705, {1, 3, 5, 11, 23, 11, 5, 19, 281, 121, 1671, 2171, 4545, 10691, 24875, 28849}},
	{5656, 16, 31706, {1, 3, 1, 3, 13, 25, 85, 131, 127, 977, 1599, 3319, 3107, 3185, 4879, 3455}},
	{5657, 16, 31718, {1, 1, 5, 1, 3, 13, 77, 15, 133, 185, 1319, 727, 2181, 12175, 28017, 28023}},
	{5658, 16, 31735, {1, 3, 7, 5, 29, 51, 113, 203, 331, 847, 1, 3445, 3669, 7711, 13647, 58651}},
	{5659, 16, 31741, {1, 3, 1, 3, 31, 27, 35, 199, 491, 839, 1275, 3385, 4743, 821, 26259, 11345}},
	{5660, 16, 31744, {1, 1, 7, 9, 21, 47, 9, 67, 119, 985, 127, 1987, 5451, 6403, 26183, 8349}},
	{5661, 16, 31762, {1, 3, 5, 1, 19, 3, 91, 217, 301, 595, 1789, 735, 4993, 229, 18033, 59625}},
	{5662, 16, 31774, {1, 3, 3, 3, 11, 25, 103, 211, 117, 9, 773, 1521, 2265, 8277, 23179, 22433}},
	{5663, 16, 31864, {1, 1, 7, 9, 3, 27, 63, 255, 175, 699, 293, 2409, 3155, 285, 8663, 53503}},
	{5664, 16, 31874, {1, 1, 5, 7, 27, 23, 63, 213, 323, 697, 1541, 3497, 2985, 12389, 11155, 26217}},
	{5665, 16, 31900, {1, 3, 1, 3, 31, 7, 47, 207, 185, 873, 1063, 1055, 205, 12469, 23505, 56245}},
	{5666, 16, 31910, {1, 3, 7, 13, 31, 17, 47, 95, 91, 483, 1997, 3273, 445, 2601, 15219, 10997}},
	{5667, 16, 31928, {1, 3, 3, 5, 29, 45, 29, 83, 457, 823, 1395, 1411, 1879, 9409, 11609, 32001}},
	{5668, 16, 31965, {1, 3, 5, 11, 21, 11, 43, 73, 159, 137, 29, 1957, 815, 5077, 16127, 42199}},
	{5669, 16, 31976, {1, 3, 5, 13, 9, 59, 47, 215, 293, 807, 309, 1951, 2285, 9287, 1019, 49501}},
	{5670, 16, 32016, {1, 1, 7, 13, 31, 7, 95, 189, 233, 363, 1039, 1675, 1715, 9049, 8537, 31051}},
	{5671, 16, 32032, {1, 3, 7, 9, 23, 35, 125, 251, 107, 401, 1113, 3585, 6331, 2363, 27889, 28877}},
	{5672, 16, 32037, {1, 1, 7, 13, 9, 1, 13, 69, 257, 369, 547, 1595, 1823, 9553, 25653, 31181}},
	{5673, 16, 32062, {1, 1, 7, 11, 9, 43, 3, 93, 69, 1019, 1935, 3297, 47, 7101, 1037, 63473}},
	{5674, 16, 32069, {1, 1, 7, 5, 21, 9, 97, 105, 405, 893, 1673, 3783, 2965, 7329, 4549, 25433}},
	{5675, 16, 32115, {1, 1, 5, 13, 5, 17, 31, 123, 415, 173, 1333, 2245, 1557, 16011, 28321, 4039}},
	{5676, 16, 32128, {1, 1, 5, 9, 15, 3, 27, 79, 511, 39, 945, 49, 3231, 9199, 21327, 11183}},
	{5677, 16, 32171, {1, 3, 3, 9, 3, 15, 115, 141, 387, 341, 953, 399, 6109, 12037, 21079, 26745}},
	{5678, 16, 32173, {1, 3, 3, 1, 5, 5, 31, 195, 477, 755, 687, 3811, 805, 679, 20687, 46299}},
	{5679, 16, 32182, {1, 1, 7, 15, 1, 31, 67, 159, 205, 141, 1667, 3077, 451, 13161, 16211, 6887}},
	{5680, 16, 32191, {1, 3, 3, 1, 7, 43, 87, 5, 49, 205, 231, 3957, 2947, 13199, 15743, 4681}},
	{5681, 16, 32193, {1, 3, 3, 15, 25, 37, 95, 11, 439, 553, 59, 1241, 7407, 13467, 22403, 44441}},
	{5682, 16, 32194, {1, 1, 1, 3, 21, 3, 127, 239, 491, 139, 1411, 3417, 4247, 6247, 13809, 31609}},
	{5683, 16, 32229, {1, 1, 5, 1, 9, 13, 5, 155, 109, 593, 119, 4091, 1911, 8301, 4239, 50081}},
	{5684, 16, 32230, {1, 3, 5, 13, 27, 3, 99, 225, 253, 169, 801, 3741, 1905, 12073, 31831, 17997}},
	{5685, 16, 32248, {1, 3, 7, 15, 9, 23, 93, 171, 453, 983, 1657, 1133, 6381, 5229, 32303, 17439}},
	{5686, 16, 32263, {1, 1, 7, 11, 7, 5, 125, 141, 63, 763, 1293, 1007, 4579, 1479, 11977, 59261}},
	{5687, 16, 32264, {1, 3, 1, 7, 1, 15, 49, 41, 367, 639, 1933, 401, 2335, 2441, 13653, 55555}},
	{5688, 16, 32269, {1, 3, 1, 7, 15, 23, 5, 213, 45, 721, 543, 2133, 4525, 9719, 28053, 54075}},
	{5689, 16, 32298, {1, 3, 7, 3, 11, 7, 23, 35, 169, 829, 1957, 2423, 3583, 4951, 28957, 29753}},
	{5690, 16, 32335, {1, 1, 3, 3, 1, 5, 19, 235, 175, 969, 229, 2335, 7215, 10195, 7487, 64191}},
	{5691, 16, 32340, {1, 1, 7, 3, 27, 1, 73, 49, 445, 863, 69, 3555, 993, 9553, 31941, 29901}},
	{5692, 16, 32356, {1, 3, 5, 11, 9, 25, 59, 177, 23, 997, 1041, 1135, 3879, 767, 2263, 51267}},
	{5693, 16, 32374, {1, 1, 7, 3, 1, 63, 49, 51, 237, 569, 1293, 1143, 3125, 16315, 17009, 24821}},
	{5694, 16, 32390, {1, 3, 3, 15, 11, 17, 121, 25, 349, 833, 557, 1975, 5405, 15189, 31243, 53541}},
	{5695, 16, 32401, {1, 3, 7, 9, 11, 15, 39, 15, 75, 87, 55, 2069, 3291, 507, 16925, 57751}},
	{5696, 16, 32414, {1, 1, 3, 15, 1, 21, 61, 139, 357, 931, 647, 947, 2291, 15557, 6739, 5881}},
	{5697, 16, 32417, {1, 3, 1, 9, 1, 47, 73, 59, 115, 497, 733, 1777, 905, 16181, 4351, 7345}},
	{5698, 16, 32442, {1, 3, 3, 7, 5, 21, 67, 113, 71, 743, 757, 1851, 7899, 10315, 15437, 61803}},
	{5699, 16, 32450, {1, 3, 7, 1, 9, 23, 77, 131, 395, 767, 1229, 2629, 5731, 11907, 32217, 18473}},
	{5700, 16, 32461, {1, 3, 5, 15, 1, 23, 123, 207, 291, 565, 1211, 501, 2111, 11381, 5171, 54841}},
	{5701, 16, 32473, {1, 1, 1, 15, 21, 13, 3, 175, 405, 109, 1353, 2495, 7619, 14971, 28179, 34737}},
	{5702, 16, 32479, {1, 3, 5, 3, 17, 25, 53, 71, 229, 729, 1953, 3119, 7747, 1551, 23417, 35563}},
	{5703, 16, 32530, {1, 1, 7, 7, 11, 31, 81, 43, 149, 537, 1253, 2759, 431, 4813, 8375, 46329}},
	{5704, 16, 32536, {1, 1, 1, 5, 11, 27, 61, 199, 239, 889, 723, 2353, 5663, 7385, 28165, 14675}},
	{5705, 16, 32548, {1, 3, 1, 7, 3, 3, 83, 247, 247, 57, 579, 1163, 2615, 4051, 2809, 46413}},
	{5706, 16, 32577, {1, 1, 3, 11, 13, 47, 11, 235, 475, 35, 843, 2329, 3519, 8899, 14533, 24889}},
	{5707, 16, 32628, {1, 3, 1, 1, 7, 31, 15, 101, 327, 499, 471, 1001, 339, 11863, 24787, 47191}},
	{5708, 16, 32642, {1, 1, 7, 1, 3, 55, 93, 43, 11, 65, 289, 1249, 5325, 13867, 29841, 34333}},
	{5709, 16, 32665, {1, 3, 3, 1, 25, 61, 87, 113, 115, 265, 1007, 1129, 7633, 6109, 5733, 22649}},
	{5710, 16, 32666, {1, 3, 1, 11, 31, 59, 127, 83, 33, 419, 1037, 3777, 6383, 2711, 2113, 17233}},
	{5711, 16, 32668, {1, 1, 5, 13, 11, 17, 73, 41, 257, 223, 359, 3821, 4617, 1943, 11331, 40153}},
	{5712, 16, 32696, {1, 1, 1, 1, 9, 25, 43, 179, 17, 1021, 1323, 761, 5861, 11547, 26017, 5165}},
	{5713, 16, 32722, {1, 3, 5, 3, 29, 21, 53, 111, 213, 717, 1101, 3215, 3021, 16343, 23387, 33439}},
	{5714, 16, 32757, {1, 3, 5, 13, 29, 11, 21, 89, 107, 111, 1121, 2785, 3493, 9873, 13, 40863}},
	{5715, 16, 32758, {1, 1, 5, 13, 15, 15, 111, 219, 59, 43, 333, 3581, 1311, 2799, 23987, 21637}},
	{5716, 17, 4, {1, 3, 1, 11, 21, 57, 115, 247, 499, 525, 1629, 3679, 2109, 6607, 27435, 1745, 71201}},
	{5717, 17, 7, {1, 3, 3, 3, 31, 17, 113, 165, 189, 361, 103, 1775, 3001, 3865, 30591, 2873, 17129}},
	{5718, 17, 16, {1, 1, 5, 5, 15, 47, 47, 85, 247, 471, 713, 3571, 2407, 9811, 8187, 32133, 8541}},
	{5719, 17, 22, {1, 3, 3, 1, 15, 1, 59, 151, 469, 351, 671, 2925, 7207, 5061, 28691, 4363, 50767}},
	{5720, 17, 25, {1, 1, 5, 7, 11, 35, 67, 45, 193, 3, 627, 3333, 6497, 12307, 28807, 13997, 108645}},
	{5721, 17, 31, {1, 3, 1, 1, 17, 63, 125, 185, 485, 759, 717, 1993, 6707, 3993, 2181, 8173, 18057}},
	{5722, 17, 32, {1, 1, 3, 13, 7, 15, 113, 207, 103, 191, 1895, 2595, 3873, 12021, 19259, 12553, 119119}},
	{5723, 17, 42, {1, 3, 7, 1, 17, 11, 101, 209, 315, 9, 901, 2303, 7623, 7459, 26391, 45143, 5753}},
	{5724, 17, 52, {1, 1, 5, 15, 1, 5, 71, 155, 167, 89, 145, 3483, 2385, 15205, 9193, 20637, 58473}},
	{5725, 17, 61, {1, 1, 5, 7, 25, 55, 57, 51, 333, 299, 1721, 1667, 6513, 10191, 29405, 21923, 76593}},
	{5726, 17, 70, {1, 1, 5, 1, 7, 37, 107, 91, 241, 137, 627, 2749, 5573, 11243, 26197, 4545, 105599}},
	{5727, 17, 76, {1, 3, 1, 5, 25, 37, 73, 61, 57, 249, 1953, 1385, 6479, 3701, 10693, 617, 62535}},
	{5728, 17, 81, {1, 1, 1, 15, 5, 63, 41, 151, 395, 681, 227, 3027, 8123, 15091, 15475, 35671, 21129}},
	{5729, 17, 87, {1, 3, 5, 11, 29, 21, 15, 233, 103, 463, 1829, 2257, 1717, 2249, 9599, 5097, 55705}},
	{5730, 17, 93, {1, 3, 5, 1, 29, 3, 35, 151, 193, 105, 1107, 827, 7169, 1843, 15225, 29025, 43165}},
	{5731, 17, 98, {1, 1, 7, 15, 17, 51, 93, 199, 205, 41, 113, 1081, 1571, 11471, 11057, 16149, 66905}},
	{5732, 17, 122, {1, 1, 3, 11, 5, 25, 107, 195, 51, 675, 1683, 3739, 1653, 611, 23249, 53157, 127785}},
	{5733, 17, 133, {1, 1, 7, 5, 7, 3, 25, 145, 453, 735, 441, 77, 8171, 9281, 22749, 36973, 106237}},
	{5734, 17, 134, {1, 1, 3, 13, 13, 5, 95, 33, 223, 369, 453, 2031, 3531, 6931, 8977, 54109, 115487}},
	{5735, 17, 140, {1, 1, 7, 7, 1, 61, 33, 183, 245, 623, 529, 1831, 1867, 2845, 8311, 10143, 67897}},
	{5736, 17, 146, {1, 3, 7, 11, 27, 23, 93, 9, 61, 451, 67, 1695, 4227, 2415, 19249, 44765, 24611}},
	{5737, 17, 158, {1, 3, 3, 11, 29, 57, 65, 117, 349, 149, 363, 1095, 4989, 3071, 17519, 18079, 7277}},
	{5738, 17, 171, {1, 3, 5, 9, 1, 7, 59, 87, 307, 111, 1291, 789, 7361, 6477, 11229, 36785, 33303}},
	{5739, 17, 176, {1, 3, 5, 1, 19, 47, 53, 81, 127, 849, 1479, 1459, 1889, 15087, 22115, 20587, 121005}},
	{5740, 17, 179, {1, 1, 7, 15, 31, 31, 71, 55, 253, 927, 277, 2087, 1313, 3721, 22729, 34709, 9821}},
	{5741, 17, 182, {1, 3, 5, 13, 13, 63, 73, 41, 165, 315, 1907, 2005, 691, 725, 22685, 8673, 76011}},
	{5742, 17, 191, {1, 1, 5, 9, 23, 61, 47, 167, 279, 683, 683, 1261, 4037, 15251, 9421, 45359, 38001}},
	{5743, 17, 193, {1, 1, 7, 3, 17, 33, 69, 139, 235, 709, 1475, 2483, 7559, 8581, 23965, 31153, 5097}},
	{5744, 17, 224, {1, 1, 7, 15, 23, 61, 43, 5, 433, 531, 761, 2749, 2881, 5225, 13491, 16479, 50203}},
	{5745, 17, 227, {1, 1, 3, 9, 29, 7, 9, 23, 339, 315, 1723, 779, 2983, 6571, 16025, 63055, 111103}},
	{5746, 17, 229, {1, 1, 7, 13, 23, 55, 71, 121, 297, 193, 41, 3165, 4419, 5853, 28127, 56151, 16597}},
	{5747, 17, 236, {1, 1, 5, 7, 7, 23, 93, 11, 261, 297, 1769, 1239, 2579, 531, 4423, 7891, 21729}},
	{5748, 17, 248, {1, 3, 5, 1, 13, 35, 83, 85, 125, 887, 161, 3311, 7261, 9557, 28975, 28643, 21479}},
	{5749, 17, 262, {1, 3, 5, 3, 27, 5, 47, 175, 287, 867, 141, 3079, 7583, 4997, 18271, 24097, 96319}},
	{5750, 17, 273, {1, 3, 5, 1, 21, 51, 47, 67, 211, 281, 1861, 1169, 6403, 4229, 3995, 9921, 41515}},
	{5751, 17, 276, {1, 3, 3, 11, 23, 23, 81, 55, 441, 211, 169, 3197, 7213, 7205, 15, 11771, 129091}},
	{5752, 17, 280, {1, 3, 7, 3, 23, 39, 23, 163, 253, 1005, 1775, 3393, 7659, 8065, 30021, 61065, 35171}},
	{5753, 17, 283, {1, 3, 1, 1, 29, 29, 39, 143, 191, 711, 1077, 13, 4137, 15425, 11139, 1269, 71915}},
	{5754, 17, 290, {1, 3, 3, 5, 11, 41, 101, 127, 301, 335, 45, 2065, 5835, 7801, 2639, 5735, 63445}},
	{5755, 17, 309, {1, 3, 5, 9, 3, 39, 51, 53, 489, 663, 951, 3931, 3075, 753, 22179, 20573, 10775}},
	{5756, 17, 316, {1, 3, 3, 15, 13, 31, 1, 237, 79, 587, 395, 591, 607, 13105, 21301, 26829, 112181}},
	{5757, 17, 319, {1, 1, 7, 7, 5, 55, 31, 117, 247, 229, 247, 307, 3821, 6483, 31317, 22975, 40535}},
	{5758, 17, 321, {1, 3, 7, 15, 15, 59, 101, 17, 437, 373, 1727, 471, 2783, 7825, 24555, 58765, 5097}},
	{5759, 17, 328, {1, 1, 3, 9, 31, 27, 71, 147, 71, 871, 793, 2363, 3213, 13383, 29801, 53187, 70021}},
	{5760, 17, 346, {1, 3, 1, 1, 19, 47, 121, 61, 303, 565, 1371, 3703, 2201, 6835, 26041, 56039, 80227}},
	{5761, 17, 355, {1, 1, 5, 5, 3, 45, 91, 61, 257, 947, 1449, 4031, 4925, 8627, 11909, 9529, 3429}},
	{5762, 17, 367, {1, 1, 1, 7, 9, 63, 69, 233, 141, 361, 1443, 2157, 2877, 643, 2779, 8109, 126911}},
	{5763, 17, 369, {1, 1, 5, 1, 5, 3, 67, 157, 21, 1, 361, 35, 1475, 12877, 22169, 6653, 85005}},
	{5764, 17, 372, {1, 1, 7, 9, 25, 1, 7, 175, 47, 963, 405, 3955, 3905, 8429, 8483, 62037, 11323}},
	{5765, 17, 382, {1, 1, 5, 11, 29, 23, 77, 211, 319, 745, 1935, 2429, 1687, 2173, 1571, 19457, 117777}},
	{5766, 17, 388, {1, 1, 7, 5, 15, 57, 121, 189, 303, 79, 527, 1801, 71, 9857, 14197, 59007, 75341}},
	{5767, 17, 392, {1, 3, 3, 5, 25, 3, 19, 141, 155, 157, 287, 769, 5789, 8443, 31823, 1019, 79111}},
	{5768, 17, 395, {1, 1, 5, 11, 27, 27, 117, 141, 355, 1023, 869, 995, 6311, 6573, 11721, 1565, 35517}},
	{5769, 17, 397, {1, 1, 1, 9, 1, 33, 107, 51, 41, 889, 1191, 1055, 503, 14779, 6641, 58117, 74157}},
	{5770, 17, 403, {1, 1, 7, 5, 13, 39, 39, 33, 293, 75, 963, 3379, 1847, 12371, 9005, 38107, 69753}},
	{5771, 17, 409, {1, 1, 5, 5, 7, 37, 19, 241, 427, 635, 1711, 3835, 773, 10525, 17207, 1675, 127255}},
	{5772, 17, 410, {1, 1, 3, 7, 17, 19, 11, 113, 191, 947, 1133, 3173, 213, 10125, 1373, 56797, 111011}},
	{5773, 17, 425, {1, 3, 1, 1, 29, 45, 65, 237, 223, 695, 697, 3197, 6887, 8079, 22099, 12079, 54847}},
	{5774, 17, 443, {1, 3, 3, 7, 5, 47, 19, 215, 341, 863, 1879, 571, 7113, 2465, 23407, 52555, 44375}},
	{5775, 17, 472, {1, 3, 5, 11, 25, 31, 109, 73, 429, 553, 1905, 1753, 6733, 4433, 13785, 32041, 27115}},
	{5776, 17, 475, {1, 1, 1, 3, 27, 5, 97, 47, 343, 977, 1241, 721, 3355, 3559, 28349, 56389, 63103}},
	{5777, 17, 481, {1, 3, 3, 9, 21, 53, 57, 211, 73, 155, 1855, 715, 3179, 5963, 10061, 35141, 63131}},
	{5778, 17, 488, {1, 3, 1, 15, 21, 25, 51, 73, 31, 25, 1385, 637, 6585, 49, 2105, 6829, 9353}},
	{5779, 17, 493, {1, 1, 7, 5, 11, 55, 31, 69, 145, 637, 1131, 2175, 3547, 13031, 2131, 12361, 74737}},
	{5780, 17, 501, {1, 3, 3, 5, 31, 7, 119, 119, 309, 925, 895, 3813, 1131, 4765, 17865, 48707, 113577}},
	{5781, 17, 515, {1, 3, 3, 9, 13, 33, 127, 177, 323, 727, 1881, 775, 7329, 11881, 28309, 987, 116093}},
	{5782, 17, 522, {1, 1, 3, 5, 31, 55, 39, 41, 511, 157, 1655, 2991, 3633, 8521, 27049, 18771, 54015}},
	{5783, 17, 524, {1, 3, 5, 13, 11, 45, 113, 185, 375, 661, 1331, 4013, 5521, 1037, 23365, 30239, 76957}},
	{5784, 17, 527, {1, 3, 3, 7, 19, 7, 23, 17, 435, 913, 1985, 353, 6049, 7549, 3371, 60867, 41099}},
	{5785, 17, 535, {1, 3, 3, 15, 17, 9, 53, 127, 149, 849, 1181, 2237, 1345, 539, 19715, 26277, 125445}},
	{5786, 17, 542, {1, 1, 1, 3, 1, 9, 67, 79, 79, 795, 1793, 3167, 5917, 5323, 22043, 22007, 3917}},
	{5787, 17, 545, {1, 3, 5, 9, 15, 19, 59, 37, 141, 145, 413, 1095, 7709, 669, 27061, 40171, 101499}},
	{5788, 17, 555, {1, 3, 1, 1, 9, 49, 109, 7, 119, 861, 875, 1049, 4125, 6113, 15699, 6105, 48799}},
	{5789, 17, 558, {1, 1, 3, 9, 11, 29, 43, 175, 371, 357, 1181, 3933, 43, 4559, 10333, 23603, 83095}},
	{5790, 17, 560, {1, 3, 3, 9, 9, 7, 57, 61, 409, 143, 591, 761, 4107, 8117, 1051, 4471, 91771}},
	{5791, 17, 563, {1, 1, 3, 11, 3, 53, 119, 21, 213, 219, 51, 3491, 7143, 937, 24693, 3211, 99463}},
	{5792, 17, 570, {1, 1, 3, 3, 1, 47, 53, 153, 211, 523, 1637, 3351, 3753, 12489, 31825, 27613, 96431}},
	{5793, 17, 578, {1, 1, 5, 15, 23, 57, 81, 231, 147, 9, 1043, 3157, 1463, 4835, 22435, 57407, 59615}},
	{5794, 17, 583, {1, 3, 3, 13, 15, 63, 111, 5, 449, 957, 1175, 2887, 7741, 8975, 28775, 4067, 69283}},
	{5795, 17, 590, {1, 3, 1, 1, 5, 61, 109, 211, 349, 179, 951, 153, 3147, 7555, 27037, 59829, 16077}},
	{5796, 17, 597, {1, 3, 3, 7, 15, 33, 53, 61, 309, 991, 227, 3437, 3983, 14559, 13065, 46387, 49105}},
	{5797, 17, 604, {1, 3, 5, 3, 25, 23, 97, 139, 315, 601, 1179, 1083, 6799, 1813, 15511, 60433, 65641}},
	{5798, 17, 608, {1, 1, 7, 1, 11, 43, 87, 87, 173, 161, 91, 3011, 1869, 2313, 13691, 3509, 39433}},
	{5799, 17, 614, {1, 3, 5, 7, 15, 5, 39, 251, 269, 819, 815, 2283, 5635, 6953, 27017, 65143, 45281}},
	{5800, 17, 635, {1, 3, 7, 9, 1, 37, 9, 57, 467, 37, 1743, 4031, 3751, 8105, 23789, 46847, 21911}},
	{5801, 17, 637, {1, 1, 7, 1, 23, 47, 63, 99, 59, 951, 1837, 2829, 161, 857, 4045, 9945, 53487}},
	{5802, 17, 653, {1, 3, 7, 7, 11, 47, 43, 99, 279, 945, 1189, 2091, 4597, 183, 15527, 7151, 112403}},
	{5803, 17, 654, {1, 3, 3, 15, 9, 53, 63, 135, 119, 95, 131, 2461, 157, 10631, 20847, 51699, 58865}},
	{5804, 17, 659, {1, 1, 3, 1, 25, 3, 115, 29, 303, 361, 1529, 3993, 5899, 11501, 4463, 47121, 75333}},
	{5805, 17, 666, {1, 3, 1, 15, 9, 39, 31, 199, 305, 279, 15, 611, 561, 6593, 3189, 1863, 61875}},
	{5806, 17, 671, {1, 3, 5, 15, 5, 49, 87, 17, 87, 5, 1179, 1351, 7647, 7529, 15901, 30351, 31959}},
	{5807, 17, 689, {1, 3, 3, 9, 31, 57, 127, 239, 349, 773, 547, 2649, 1309, 8071, 10741, 57645, 14423}},
	{5808, 17, 690, {1, 1, 5, 9, 5, 15, 59, 185, 315, 411, 1425, 3905, 853, 12393, 21, 15195, 114291}},
	{5809, 17, 695, {1, 3, 1, 5, 29, 47, 19, 203, 319, 673, 1169, 2413, 5295, 6251, 19883, 2725, 28937}},
	{5810, 17, 713, {1, 3, 1, 5, 21, 55, 19, 185, 103, 827, 117, 341, 3315, 5625, 345, 63845, 49081}},
	{5811, 17, 722, {1, 1, 7, 9, 27, 51, 105, 15, 243, 735, 1221, 1641, 293, 14423, 5363, 60873, 66223}},
	{5812, 17, 733, {1, 1, 5, 1, 19, 5, 109, 131, 131, 67, 231, 2907, 4389, 5079, 20503, 59045, 33625}},
	{5813, 17, 758, {1, 3, 1, 5, 5, 15, 79, 67, 287, 225, 519, 1543, 2389, 671, 7767, 62625, 61639}},
	{5814, 17, 770, {1, 1, 1, 9, 25, 35, 83, 15, 291, 207, 1757, 3691, 5669, 11255, 27939, 57813, 46251}},
	{5815, 17, 782, {1, 3, 1, 1, 29, 3, 83, 109, 323, 179, 1855, 3205, 7665, 16201, 13863, 16347, 98977}},
	{5816, 17, 784, {1, 3, 1, 13, 17, 1, 101, 183, 153, 985, 125, 999, 855, 15897, 19491, 8953, 23277}},
	{5817, 17, 793, {1, 1, 7, 11, 9, 33, 45, 229, 411, 155, 537, 3037, 1785, 11719, 8589, 16617, 47339}},
	{5818, 17, 803, {1, 1, 5, 5, 9, 11, 7, 163, 305, 621, 1647, 2609, 7901, 14421, 23447, 1205, 52681}},
	{5819, 17, 805, {1, 3, 3, 1, 7, 29, 39, 227, 419, 561, 129, 3299, 3123, 4243, 18689, 12335, 71783}},
	{5820, 17, 812, {1, 3, 1, 9, 11, 61, 65, 207, 123, 763, 485, 1943, 3617, 415, 22397, 58597, 128017}},
	{5821, 17, 838, {1, 1, 5, 13, 25, 43, 115, 73, 269, 137, 1765, 705, 1705, 16137, 22751, 60021, 4333}},
	{5822, 17, 849, {1, 1, 5, 13, 3, 57, 9, 141, 75, 695, 597, 3435, 1085, 4905, 19625, 16061, 12111}},
	{5823, 17, 875, {1, 1, 5, 9, 29, 13, 119, 251, 353, 421, 1955, 3503, 2605, 2587, 12503, 46419, 128815}},
	{5824, 17, 877, {1, 3, 5, 7, 7, 29, 67, 25, 37, 327, 1607, 1899, 1691, 5801, 17441, 9755, 24993}},
	{5825, 17, 880, {1, 1, 3, 11, 17, 29, 121, 201, 371, 597, 213, 2361, 6615, 169, 24801, 56175, 129241}},
	{5826, 17, 892, {1, 3, 5, 1, 31, 63, 85, 77, 151, 599, 103, 677, 4431, 12897, 6373, 40349, 100819}},
	{5827, 17, 895, {1, 3, 5, 9, 25, 9, 119, 219, 379, 939, 1907, 945, 5819, 7433, 32519, 56493, 50441}},
	{5828, 17, 899, {1, 1, 3, 9, 13, 1, 63, 189, 135, 839, 1821, 2247, 2547, 965, 6847, 63335, 32921}},
	{5829, 17, 919, {1, 3, 5, 13, 21, 25, 111, 37, 319, 469, 1999, 1637, 8167, 2641, 24615, 63713, 115923}},
	{5830, 17, 920, {1, 3, 5, 9, 9, 27, 1, 63, 275, 223, 1675, 3833, 7377, 9755, 6279, 37161, 108805}},
	{5831, 17, 932, {1, 3, 3, 13, 29, 23, 21, 73, 401, 863, 701, 2527, 4557, 5549, 22493, 6651, 39167}},
	{5832, 17, 935, {1, 1, 3, 15, 25, 21, 97, 25, 83, 925, 2029, 3789, 3241, 7617, 13699, 31123, 124619}},
	{5833, 17, 936, {1, 3, 7, 5, 23, 7, 95, 227, 123, 215, 359, 2099, 4505, 8477, 32665, 18211, 99679}},
	{5834, 17, 941, {1, 3, 1, 9, 11, 57, 75, 17, 105, 175, 831, 1033, 5425, 8419, 16163, 23901, 33889}},
	{5835, 17, 950, {1, 1, 7, 1, 17, 49, 71, 23, 129, 413, 333, 2547, 4627, 14961, 16745, 53649, 73059}},
	{5836, 17, 961, {1, 3, 5, 3, 13, 33, 121, 147, 443, 187, 1949, 319, 8141, 14359, 11203, 53569, 70415}},
	{5837, 17, 962, {1, 3, 1, 11, 15, 1, 23, 29, 509, 985, 1217, 3755, 385, 3697, 24631, 37619, 62435}},
	{5838, 17, 971, {1, 3, 3, 3, 17, 11, 107, 37, 227, 913, 259, 2799, 3249, 2347, 9703, 52741, 101187}},
	{5839, 17, 982, {1, 1, 5, 13, 25, 25, 47, 77, 405, 415, 1947, 1675, 5079, 1333, 10059, 32033, 88975}},
	{5840, 17, 986, {1, 3, 5, 9, 27, 7, 19, 241, 445, 205, 333, 285, 7997, 6339, 29643, 10229, 29965}},
	{5841, 17, 1012, {1, 3, 5, 11, 17, 9, 91, 223, 173, 1013, 779, 3967, 781, 5471, 4309, 24795, 99203}},
	{5842, 17, 1021, {1, 1, 1, 3, 19, 53, 7, 159, 351, 515, 223, 3375, 1, 4985, 16729, 43333, 85917}},
	{5843, 17, 1024, {1, 3, 3, 1, 19, 35, 95, 69, 19, 157, 1177, 579, 7109, 3499, 3219, 26641, 49491}},
	{5844, 17, 1029, {1, 3, 3, 5, 25, 21, 125, 5, 39, 857, 615, 2925, 2005, 5503, 25523, 36711, 30939}},
	{5845, 17, 1030, {1, 3, 1, 5, 11, 33, 29, 5, 425, 125, 939, 1641, 321, 1023, 12551, 4587, 116617}},
	{5846, 17, 1051, {1, 3, 3, 13, 9, 59, 93, 137, 103, 517, 1555, 13, 7965, 13629, 14339, 37425, 65891}},
	{5847, 17, 1054, {1, 3, 7, 1, 31, 31, 87, 237, 365, 951, 267, 2019, 5085, 6133, 29371, 50319, 94313}},
	{5848, 17, 1064, {1, 3, 5, 7, 17, 19, 23, 225, 501, 189, 1291, 603, 6873, 8633, 11425, 30565, 26355}},
	{5849, 17, 1067, {1, 3, 7, 11, 23, 17, 91, 111, 415, 225, 1287, 2081, 4683, 12069, 3627, 32281, 17995}},
	{5850, 17, 1082, {1, 1, 5, 15, 25, 59, 75, 203, 179, 405, 1711, 3147, 7483, 5583, 3729, 11765, 61019}},
	{5851, 17, 1096, {1, 3, 3, 9, 3, 43, 65, 7, 269, 33, 829, 1789, 967, 13119, 26329, 16937, 18533}},
	{5852, 17, 1116, {1, 1, 3, 15, 11, 39, 73, 11, 31, 143, 1913, 1227, 1363, 11831, 28687, 50489, 106373}},
	{5853, 17, 1119, {1, 1, 3, 3, 25, 19, 15, 11, 349, 1011, 421, 3193, 3665, 6149, 20729, 6997, 51437}},
	{5854, 17, 1129, {1, 3, 5, 9, 13, 63, 73, 55, 417, 223, 1753, 2913, 4809, 3947, 10769, 5751, 93867}},
	{5855, 17, 1130, {1, 3, 7, 13, 31, 39, 39, 133, 483, 839, 1137, 3303, 7285, 4309, 24079, 60529, 103337}},
	{5856, 17, 1132, {1, 1, 3, 7, 1, 55, 3, 253, 435, 589, 1949, 1461, 513, 381, 29455, 4263, 16831}},
	{5857, 17, 1137, {1, 1, 1, 15, 25, 19, 77, 101, 299, 187, 1021, 1533, 8021, 4165, 2277, 18927, 110439}},
	{5858, 17, 1147, {1, 1, 1, 11, 9, 35, 71, 159, 409, 527, 15, 4073, 5749, 8563, 2503, 53015, 111581}},
	{5859, 17, 1150, {1, 1, 7, 5, 21, 47, 113, 23, 477, 559, 543, 409, 4701, 11479, 30761, 8373, 87777}},
	{5860, 17, 1154, {1, 3, 5, 13, 9, 27, 25, 137, 81, 37, 799, 857, 3539, 4471, 15753, 59015, 48589}},
	{5861, 17, 1165, {1, 1, 3, 7, 11, 57, 103, 83, 209, 71, 193, 3251, 4839, 13959, 32009, 6471, 23631}},
	{5862, 17, 1166, {1, 1, 7, 11, 25, 33, 85, 31, 371, 253, 1667, 1627, 6159, 10039, 15177, 52121, 39475}},
	{5863, 17, 1174, {1, 1, 5, 9, 13, 55, 37, 13, 95, 113, 1895, 1525, 1907, 6361, 5863, 27767, 108143}},
	{5864, 17, 1177, {1, 1, 3, 13, 21, 5, 53, 39, 485, 171, 1355, 2117, 3127, 6467, 31697, 45343, 111477}},
	{5865, 17, 1184, {1, 1, 7, 15, 13, 57, 11, 231, 329, 703, 1823, 2983, 215, 2835, 19719, 56637, 126169}},
	{5866, 17, 1194, {1, 3, 5, 15, 13, 51, 13, 173, 301, 867, 127, 2391, 2795, 4945, 13293, 49947, 10765}},
	{5867, 17, 1204, {1, 3, 3, 9, 23, 5, 29, 165, 467, 599, 1181, 3213, 4069, 5473, 8937, 51495, 42611}},
	{5868, 17, 1208, {1, 1, 7, 15, 5, 5, 31, 125, 397, 519, 1465, 115, 7877, 7025, 14213, 50343, 85827}},
	{5869, 17, 1213, {1, 3, 7, 3, 25, 59, 95, 103, 101, 347, 95, 3, 1251, 15109, 12615, 7511, 56789}},
	{5870, 17, 1219, {1, 3, 5, 9, 13, 59, 71, 19, 107, 73, 345, 3177, 6519, 2407, 18033, 31075, 113185}},
	{5871, 17, 1233, {1, 1, 1, 3, 27, 37, 5, 219, 169, 149, 355, 549, 1811, 11351, 22627, 53931, 88619}},
	{5872, 17, 1264, {1, 3, 1, 3, 27, 7, 9, 97, 399, 947, 1393, 3917, 5439, 15845, 19465, 30123, 69099}},
	{5873, 17, 1267, {1, 1, 7, 9, 13, 25, 107, 45, 111, 409, 967, 3359, 2499, 1703, 20763, 45187, 16265}},
	{5874, 17, 1281, {1, 1, 1, 13, 5, 49, 43, 249, 49, 947, 597, 1773, 2387, 2693, 15297, 57969, 53385}},
	{5875, 17, 1312, {1, 1, 7, 15, 27, 25, 27, 121, 421, 781, 143, 817, 7335, 14211, 139, 55601, 56671}},
	{5876, 17, 1321, {1, 3, 1, 5, 29, 47, 77, 23, 413, 931, 785, 1221, 769, 13131, 26955, 56441, 85745}},
	{5877, 17, 1330, {1, 1, 1, 11, 27, 3, 53, 21, 467, 43, 1533, 1053, 691, 6369, 8325, 51087, 71261}},
	{5878, 17, 1332, {1, 1, 3, 15, 7, 9, 43, 225, 293, 143, 1049, 3095, 6119, 3165, 9913, 26023, 62657}},
	{5879, 17, 1335, {1, 3, 7, 9, 11, 39, 99, 193, 217, 941, 259, 3811, 6757, 281, 10377, 46961, 48949}},
	{5880, 17, 1341, {1, 1, 1, 1, 25, 1, 99, 61, 495, 861, 2013, 487, 2821, 12921, 30111, 27213, 97363}},
	{5881, 17, 1356, {1, 1, 5, 9, 23, 33, 103, 237, 161, 721, 2021, 159, 995, 475, 20615, 30961, 31767}},
	{5882, 17, 1371, {1, 3, 1, 1, 5, 59, 63, 139, 451, 789, 1285, 655, 5501, 273, 21061, 35937, 20811}},
	{5883, 17, 1377, {1, 3, 3, 9, 9, 15, 121, 233, 287, 929, 1605, 1243, 417, 1695, 29903, 28699, 85981}},
	{5884, 17, 1380, {1, 3, 3, 5, 7, 25, 27, 253, 469, 255, 285, 2467, 4897, 4079, 29759, 50351, 76451}},
	{5885, 17, 1384, {1, 1, 3, 3, 5, 33, 29, 209, 291, 967, 1429, 1953, 5957, 14065, 8875, 32675, 4629}},
	{5886, 17, 1395, {1, 3, 5, 9, 7, 31, 97, 21, 177, 485, 1115, 4051, 6683, 7761, 30181, 37531, 51789}},
	{5887, 17, 1397, {1, 1, 7, 3, 25, 51, 23, 183, 57, 699, 1245, 2519, 2783, 4457, 6381, 43199, 40071}},
	{5888, 17, 1411, {1, 3, 5, 5, 19, 55, 45, 101, 299, 461, 1009, 319, 7335, 7769, 5479, 61113, 7937}},
	{5889, 17, 1414, {1, 1, 7, 3, 29, 21, 55, 55, 437, 771, 363, 683, 4299, 15569, 13813, 40663, 86285}},
	{5890, 17, 1426, {1, 1, 1, 13, 31, 35, 93, 175, 451, 387, 1145, 3367, 3833, 13495, 11019, 48925, 85721}},
	{5891, 17, 1432, {1, 1, 7, 15, 31, 21, 55, 205, 117, 895, 535, 2627, 1473, 10779, 24493, 42999, 130805}},
	{5892, 17, 1435, {1, 1, 3, 13, 27, 11, 45, 37, 193, 237, 1505, 1405, 3613, 9565, 3037, 53643, 85211}},
	{5893, 17, 1437, {1, 1, 3, 13, 9, 17, 19, 27, 117, 503, 65, 1033, 7891, 4005, 9229, 20999, 96601}},
	{5894, 17, 1442, {1, 3, 3, 5, 17, 3, 71, 79, 145, 985, 935, 3997, 6239, 12511, 13895, 65031, 126383}},
	{5895, 17, 1454, {1, 1, 5, 1, 23, 55, 3, 105, 71, 243, 1479, 111, 7103, 10753, 26193, 35833, 14583}},
	{5896, 17, 1468, {1, 3, 3, 3, 15, 3, 73, 125, 267, 29, 1775, 1437, 8091, 10891, 25731, 54381, 12821}},
	{5897, 17, 1473, {1, 1, 1, 3, 23, 15, 67, 123, 401, 347, 807, 1097, 31, 11209, 8727, 58149, 129099}},
	{5898, 17, 1488, {1, 3, 3, 7, 7, 61, 49, 129, 423, 535, 135, 3587, 233, 4509, 23209, 59203, 41297}},
	{5899, 17, 1491, {1, 3, 1, 7, 5, 29, 65, 31, 335, 855, 835, 1421, 3081, 14219, 16321, 48269, 41603}},
	{5900, 17, 1509, {1, 1, 1, 13, 3, 21, 5, 117, 163, 603, 1519, 3789, 7873, 10981, 4615, 9165, 83929}},
	{5901, 17, 1524, {1, 3, 5, 11, 15, 21, 75, 151, 193, 757, 647, 1603, 333, 10515, 22771, 55459, 3315}},
	{5902, 17, 1533, {1, 1, 7, 1, 27, 3, 63, 197, 271, 175, 1599, 2119, 1031, 8671, 10893, 35641, 94535}},
	{5903, 17, 1555, {1, 1, 1, 15, 1, 59, 93, 17, 5, 213, 1663, 941, 435, 8107, 1963, 34951, 106181}},
	{5904, 17, 1567, {1, 1, 5, 11, 13, 35, 111, 97, 267, 737, 2023, 1301, 7407, 11249, 31785, 31933, 20673}},
	{5905, 17, 1571, {1, 3, 3, 15, 5, 15, 29, 63, 189, 687, 27, 2005, 7129, 11377, 23175, 42389, 30933}},
	{5906, 17, 1586, {1, 1, 1, 9, 13, 63, 7, 155, 67, 291, 1419, 755, 2623, 4749, 22971, 7545, 55711}},
	{5907, 17, 1592, {1, 3, 7, 7, 23, 29, 83, 151, 213, 201, 157, 3051, 6553, 6401, 15931, 47941, 22869}},
	{5908, 17, 1595, {1, 3, 5, 5, 7, 45, 33, 155, 225, 25, 49, 2419, 4241, 6835, 11401, 50725, 118343}},
	{5909, 17, 1600, {1, 1, 3, 13, 31, 27, 37, 41, 19, 375, 1771, 1789, 2313, 2577, 12615, 22715, 22179}},
	{5910, 17, 1606, {1, 3, 1, 11, 17, 53, 55, 229, 235, 837, 143, 3583, 2789, 5471, 6515, 44565, 8619}},
	{5911, 17, 1627, {1, 1, 5, 15, 5, 17, 23, 95, 217, 551, 353, 27, 3973, 2547, 27903, 50611, 72277}},
	{5912, 17, 1648, {1, 1, 3, 7, 5, 13, 41, 111, 157, 215, 1327, 3073, 1871, 11875, 24239, 40527, 97637}},
	{5913, 17, 1651, {1, 3, 1, 1, 29, 63, 111, 187, 369, 395, 1197, 3229, 4353, 14715, 29671, 50503, 89321}},
	{5914, 17, 1654, {1, 3, 1, 1, 5, 63, 11, 39, 171, 209, 463, 3421, 3451, 4453, 14397, 2219, 98261}},
	{5915, 17, 1667, {1, 3, 3, 5, 1, 1, 13, 101, 67, 815, 1521, 1543, 7221, 7337, 10765, 30029, 47881}},
	{5916, 17, 1669, {1, 1, 5, 7, 9, 9, 33, 197, 439, 893, 961, 11, 4319, 14265, 24839, 33581, 35531}},
	{5917, 17, 1674, {1, 3, 3, 15, 29, 35, 43, 229, 313, 369, 955, 1069, 2939, 12623, 20373, 1533, 9105}},
	{5918, 17, 1687, {1, 3, 1, 7, 21, 7, 127, 243, 103, 353, 859, 3789, 4369, 12063, 22369, 14531, 94289}},
	{5919, 17, 1698, {1, 3, 5, 15, 1, 27, 65, 127, 229, 99, 627, 2693, 7173, 7305, 29971, 7097, 10113}},
	{5920, 17, 1710, {1, 1, 5, 15, 3, 47, 61, 29, 155, 725, 1727, 2667, 7003, 16277, 21983, 21365, 129365}},
	{5921, 17, 1717, {1, 1, 5, 7, 27, 61, 115, 133, 137, 661, 1201, 2151, 367, 3567, 12885, 62143, 53955}},
	{5922, 17, 1722, {1, 1, 1, 11, 9, 41, 113, 103, 469, 687, 1541, 3679, 6833, 10493, 32747, 39909, 121445}},
	{5923, 17, 1735, {1, 1, 7, 5, 5, 5, 91, 91, 5, 405, 529, 3999, 6783, 2387, 16621, 12919, 8659}},
	{5924, 17, 1741, {1, 1, 7, 13, 21, 47, 125, 155, 83, 913, 1833, 4027, 6657, 7031, 31231, 58201, 88943}},
	{5925, 17, 1749, {1, 3, 7, 3, 17, 55, 25, 29, 181, 205, 1173, 1081, 6475, 5037, 18461, 22487, 114131}},
	{5926, 17, 1750, {1, 1, 7, 7, 25, 63, 101, 103, 171, 191, 1863, 3441, 2515, 14179, 30123, 19145, 31669}},
	{5927, 17, 1769, {1, 3, 7, 11, 29, 49, 73, 163, 415, 821, 1809, 723, 7049, 14565, 4829, 19395, 61131}},
	{5928, 17, 1775, {1, 1, 7, 9, 5, 25, 103, 167, 381, 757, 813, 471, 3021, 6619, 20929, 38133, 129505}},
	{5929, 17, 1777, {1, 1, 5, 13, 25, 61, 59, 199, 257, 999, 169, 3289, 7181, 2049, 2185, 39045, 102703}},
	{5930, 17, 1778, {1, 1, 3, 1, 21, 1, 111, 125, 289, 33, 701, 3491, 5569, 8055, 23149, 26793, 102563}},
	{5931, 17, 1792, {1, 1, 7, 3, 25, 15, 105, 235, 307, 201, 1947, 699, 2519, 10615, 29345, 17061, 112949}},
	{5932, 17, 1797, {1, 3, 3, 15, 19, 1, 93, 173, 399, 13, 269, 1189, 523, 5145, 32731, 54087, 94123}},
	{5933, 17, 1802, {1, 3, 1, 15, 9, 41, 59, 79, 217, 833, 1993, 2429, 3599, 6919, 30911, 12615, 67947}},
	{5934, 17, 1822, {1, 3, 3, 13, 31, 9, 95, 37, 343, 955, 1363, 3851, 4091, 13165, 15241, 14853, 35747}},
	{5935, 17, 1825, {1, 1, 3, 5, 27, 39, 37, 217, 385, 473, 1997, 2247, 7353, 1503, 9003, 15055, 27289}},
	{5936, 17, 1831, {1, 3, 7, 11, 1, 13, 21, 243, 375, 91, 1295, 1661, 203, 15251, 15355, 16065, 24183}},
	{5937, 17, 1838, {1, 3, 1, 13, 11, 45, 85, 5, 275, 741, 1395, 4011, 7987, 16087, 24113, 50555, 128147}},
	{5938, 17, 1852, {1, 1, 1, 7, 3, 11, 13, 189, 55, 151, 395, 657, 807, 11973, 26297, 13043, 109641}},
	{5939, 17, 1855, {1, 1, 7, 13, 31, 19, 33, 235, 491, 647, 1115, 2299, 6381, 7525, 2237, 36197, 126457}},
	{5940, 17, 1860, {1, 3, 5, 1, 21, 15, 53, 231, 77, 347, 969, 141, 4501, 9429, 1815, 50887, 74581}},
	{5941, 17, 1867, {1, 1, 1, 9, 29, 43, 47, 103, 327, 131, 927, 441, 7517, 7277, 21065, 409, 50351}},
	{5942, 17, 1869, {1, 1, 5, 1, 11, 13, 103, 157, 239, 69, 1347, 477, 5017, 9723, 28133, 65135, 12359}},
	{5943, 17, 1875, {1, 1, 1, 13, 17, 63, 117, 189, 323, 565, 927, 1727, 5337, 13243, 5739, 31241, 14209}},
	{5944, 17, 1882, {1, 1, 3, 9, 29, 9, 103, 61, 467, 217, 1367, 2405, 5355, 5743, 31469, 30149, 98775}},
	{5945, 17, 1903, {1, 1, 1, 15, 23, 23, 17, 229, 103, 583, 179, 115, 7081, 9437, 32623, 62639, 72391}},
	{5946, 17, 1908, {1, 1, 5, 11, 11, 39, 97, 209, 115, 107, 593, 2347, 1445, 6179, 32011, 8435, 65847}},
	{5947, 17, 1917, {1, 3, 7, 3, 29, 27, 55, 111, 27, 731, 995, 1871, 5017, 1485, 11313, 2559, 6561}},
	{5948, 17, 1927, {1, 3, 1, 3, 27, 9, 103, 247, 83, 197, 517, 1629, 2189, 7255, 183, 35111, 15077}},
	{5949, 17, 1941, {1, 3, 7, 5, 31, 37, 87, 223, 343, 331, 1361, 3371, 2007, 13235, 10897, 63839, 109837}},
	{5950, 17, 1945, {1, 1, 7, 11, 17, 5, 41, 197, 489, 625, 1595, 2663, 5941, 14029, 30999, 16781, 116001}},
	{5951, 17, 1948, {1, 3, 3, 7, 19, 19, 61, 175, 125, 609, 1391, 147, 3001, 4189, 10133, 24031, 46219}},
	{5952, 17, 1962, {1, 1, 3, 13, 13, 57, 117, 181, 299, 939, 583, 3151, 829, 6561, 30449, 12211, 107879}},
	{5953, 17, 1975, {1, 1, 5, 11, 23, 45, 87, 115, 259, 613, 1001, 171, 57, 13789, 22173, 56837, 26263}},
	{5954, 17, 1976, {1, 1, 3, 3, 7, 43, 45, 131, 87, 251, 1411, 2737, 2739, 4595, 12561, 12043, 82885}},
	{5955, 17, 1987, {1, 3, 3, 7, 19, 39, 87, 223, 461, 37, 283, 3937, 6193, 10887, 11509, 41131, 38359}},
	{5956, 17, 1993, {1, 3, 1, 11, 11, 37, 25, 133, 105, 1013, 925, 3301, 239, 16295, 4831, 8649, 125767}},
	{5957, 17, 2004, {1, 3, 3, 11, 25, 11, 41, 155, 1, 717, 1587, 635, 279, 1803, 14817, 28669, 88835}},
	{5958, 17, 2020, {1, 3, 3, 11, 29, 17, 39, 51, 13, 871, 1197, 2561, 6671, 8465, 22709, 15933, 15923}},
	{5959, 17, 2029, {1, 3, 7, 1, 13, 17, 57, 43, 267, 261, 901, 241, 3767, 15053, 11017, 36321, 72497}},
	{5960, 17, 2030, {1, 3, 1, 15, 23, 13, 17, 63, 171, 919, 1387, 2673, 7605, 8523, 14807, 21187, 56057}},
	{5961, 17, 2038, {1, 3, 7, 15, 23, 41, 85, 95, 53, 629, 1877, 3167, 2411, 9619, 24621, 31213, 30069}},
	{5962, 17, 2041, {1, 1, 5, 3, 3, 25, 99, 39, 321, 549, 599, 1279, 2401, 2335, 8227, 59429, 94549}},
	{5963, 17, 2048, {1, 3, 3, 11, 9, 21, 29, 55, 477, 19, 1275, 29, 2253, 11421, 30401, 57059, 93219}},
	{5964, 17, 2054, {1, 1, 7, 1, 27, 13, 117, 249, 463, 769, 281, 515, 7467, 11507, 1621, 39765, 31109}},
	{5965, 17, 2057, {1, 3, 5, 7, 19, 7, 77, 107, 23, 895, 1013, 2701, 3805, 7327, 27247, 6119, 102395}},
	{5966, 17, 2058, {1, 1, 3, 13, 21, 49, 99, 15, 163, 641, 1703, 3061, 163, 4265, 32571, 13957, 75005}},
	{5967, 17, 2068, {1, 1, 5, 11, 27, 17, 87, 169, 427, 959, 361, 1023, 5727, 16279, 1099, 39081, 67215}},
	{5968, 17, 2072, {1, 3, 3, 9, 23, 13, 1, 91, 173, 325, 1881, 1385, 8023, 935, 9221, 19673, 36949}},
	{5969, 17, 2075, {1, 3, 1, 7, 7, 25, 119, 189, 107, 249, 811, 973, 6499, 101, 11281, 55227, 32361}},
	{5970, 17, 2077, {1, 1, 5, 13, 19, 37, 117, 95, 463, 587, 1419, 445, 4019, 7257, 29757, 50773, 52247}},
	{5971, 17, 2082, {1, 1, 1, 1, 17, 57, 81, 57, 43, 789, 1035, 625, 1707, 9683, 3681, 12411, 110623}},
	{5972, 17, 2084, {1, 1, 7, 5, 7, 57, 49, 91, 459, 513, 1869, 3377, 139, 10037, 24091, 54247, 41279}},
	{5973, 17, 2087, {1, 3, 3, 9, 9, 33, 29, 51, 355, 415, 1907, 809, 6543, 349, 18507, 12919, 41667}},
	{5974, 17, 2101, {1, 1, 5, 11, 3, 17, 73, 201, 121, 909, 1623, 799, 3271, 9051, 5717, 15169, 127861}},
	{5975, 17, 2111, {1, 1, 7, 7, 23, 31, 1, 155, 475, 87, 2001, 2459, 1285, 5931, 6803, 56757, 71671}},
	{5976, 17, 2113, {1, 1, 5, 13, 5, 1, 21, 109, 263, 841, 723, 1539, 7529, 433, 23721, 33195, 57001}},
	{5977, 17, 2119, {1, 3, 3, 13, 29, 55, 105, 231, 405, 265, 671, 351, 4693, 9033, 21963, 52073, 125131}},
	{5978, 17, 2147, {1, 3, 1, 13, 25, 51, 55, 227, 245, 983, 251, 2553, 2017, 1381, 31461, 3953, 75775}},
	{5979, 17, 2154, {1, 1, 1, 11, 31, 11, 91, 91, 287, 749, 1019, 4055, 3237, 6965, 14765, 1663, 82987}},
	{5980, 17, 2161, {1, 1, 7, 3, 11, 15, 67, 161, 79, 729, 1115, 3713, 2715, 9361, 9365, 26093, 63409}},
	{5981, 17, 2164, {1, 3, 1, 7, 1, 51, 125, 15, 457, 433, 405, 2329, 157, 4817, 25867, 38177, 45319}},
	{5982, 17, 2177, {1, 3, 7, 9, 25, 57, 5, 233, 481, 781, 1313, 3179, 7219, 8717, 14825, 16079, 127149}},
	{5983, 17, 2178, {1, 1, 7, 15, 27, 51, 5, 65, 77, 313, 1751, 1489, 4307, 10541, 11345, 52577, 18143}},
	{5984, 17, 2184, {1, 1, 1, 15, 21, 5, 113, 71, 411, 327, 1681, 1023, 5661, 15815, 5387, 10351, 21121}},
	{5985, 17, 2198, {1, 1, 5, 5, 29, 55, 25, 255, 69, 879, 501, 1915, 3731, 633, 12197, 5249, 31129}},
	{5986, 17, 2201, {1, 3, 5, 7, 3, 23, 107, 163, 485, 853, 359, 3069, 4353, 371, 6027, 53239, 105541}},
	{5987, 17, 2213, {1, 3, 5, 15, 7, 41, 9, 47, 33, 327, 621, 147, 577, 29, 14623, 3403, 9791}},
	{5988, 17, 2217, {1, 3, 3, 15, 29, 47, 41, 149, 477, 127, 573, 877, 3101, 5963, 28457, 14231, 67425}},
	{5989, 17, 2228, {1, 1, 1, 15, 31, 7, 55, 191, 101, 259, 1071, 219, 2233, 3583, 21969, 32745, 80529}},
	{5990, 17, 2240, {1, 3, 7, 13, 17, 53, 115, 69, 241, 71, 1475, 191, 509, 3721, 15537, 53773, 18005}},
	{5991, 17, 2245, {1, 1, 3, 9, 5, 57, 13, 95, 103, 871, 2043, 2239, 7833, 10727, 6513, 55273, 3781}},
	{5992, 17, 2250, {1, 1, 5, 5, 9, 11, 55, 151, 239, 537, 135, 2779, 7393, 15393, 11097, 58593, 100745}},
	{5993, 17, 2263, {1, 1, 1, 9, 15, 39, 29, 105, 441, 181, 1113, 2125, 8145, 11045, 6589, 33603, 83377}},
	{5994, 17, 2267, {1, 3, 1, 1, 11, 63, 69, 153, 225, 845, 675, 407, 4691, 13383, 27359, 38881, 5509}},
	{5995, 17, 2285, {1, 3, 7, 11, 23, 31, 69, 3, 41, 57, 683, 887, 6861, 12161, 14537, 27293, 113001}},
	{5996, 17, 2286, {1, 1, 1, 11, 5, 1, 101, 175, 437, 3, 1477, 1005, 6607, 7429, 7213, 4025, 66479}},
	{5997, 17, 2291, {1, 1, 7, 5, 19, 7, 99, 131, 273, 977, 1717, 3831, 175, 5673, 12577, 36787, 30945}},
	{5998, 17, 2298, {1, 3, 1, 1, 15, 37, 105, 195, 61, 869, 255, 2625, 7401, 9361, 13217, 52811, 130811}},
	{5999, 17, 2306, {1, 3, 5, 3, 29, 27, 105, 23, 511, 813, 1311, 2859, 1647, 1949, 1329, 27589, 125209}},
	{6000, 17, 2325, {1, 3, 3, 1, 21, 11, 119, 247, 123, 401, 409, 1845, 2133, 10793, 221, 43217, 14069}},
	{6001, 17, 2329, {1, 1, 5, 1, 29, 21, 51, 73, 501, 861, 725, 249, 4249, 8029, 15767, 11985, 18637}},
	{6002, 17, 2332, {1, 1, 5, 11, 19, 39, 97, 65, 13, 283, 489, 2307, 5239, 4161, 18639, 60035, 22405}},
	{6003, 17, 2335, {1, 3, 5, 1, 3, 7, 109, 27, 429, 663, 1569, 3001, 3453, 8627, 9719, 23941, 110451}},
	{6004, 17, 2339, {1, 3, 7, 5, 17, 13, 125, 209, 347, 95, 1937, 1419, 5661, 7171, 20607, 9777, 68343}},
	{6005, 17, 2346, {1, 1, 1, 1, 7, 41, 43, 229, 57, 49, 1863, 2819, 3735, 915, 1571, 11603, 116275}},
	{6006, 17, 2351, {1, 1, 7, 9, 21, 27, 5, 199, 181, 521, 303, 1097, 5427, 8899, 30325, 55457, 16189}},
	{6007, 17, 2353, {1, 3, 3, 7, 19, 41, 3, 205, 279, 223, 971, 633, 2617, 13191, 10193, 23375, 62563}},
	{6008, 17, 2363, {1, 3, 3, 13, 23, 59, 85, 25, 253, 405, 65, 1625, 4401, 4679, 14381, 57833, 30001}},
	{6009, 17, 2378, {1, 3, 3, 3, 13, 35, 11, 157, 123, 397, 119, 2513, 1919, 14583, 5469, 11463, 94711}},
	{6010, 17, 2383, {1, 1, 1, 7, 17, 37, 83, 211, 451, 939, 449, 13, 6671, 1457, 19855, 15053, 52327}},
	{6011, 17, 2391, {1, 1, 5, 3, 9, 57, 39, 183, 331, 451, 1391, 1865, 7801, 14293, 29069, 705, 109497}},
	{6012, 17, 2401, {1, 3, 7, 7, 23, 21, 85, 81, 255, 9, 1685, 2879, 6327, 12675, 31657, 38877, 74131}},
	{6013, 17, 2408, {1, 1, 5, 9, 25, 19, 41, 195, 31, 555, 927, 1445, 593, 11067, 10819, 17205, 82037}},
	{6014, 17, 2414, {1, 3, 1, 13, 1, 35, 29, 71, 323, 705, 53, 3885, 6223, 1319, 30853, 59935, 35949}},
	{6015, 17, 2419, {1, 1, 7, 3, 27, 63, 67, 31, 149, 61, 1611, 77, 4271, 3161, 12493, 38341, 53837}},
	{6016, 17, 2428, {1, 1, 1, 15, 27, 53, 31, 249, 429, 925, 1485, 1855, 4421, 5703, 10097, 14827, 36685}},
	{6017, 17, 2441, {1, 3, 7, 13, 7, 63, 53, 9, 317, 485, 1679, 3631, 3745, 5643, 21615, 45129, 48027}},
	{6018, 17, 2444, {1, 1, 1, 1, 17, 43, 19, 163, 441, 847, 937, 959, 6649, 13071, 1065, 55193, 129509}},
	{6019, 17, 2461, {1, 1, 1, 11, 29, 47, 9, 215, 397, 637, 961, 3139, 2007, 12603, 27657, 22825, 72873}},
	{6020, 17, 2480, {1, 3, 3, 15, 7, 45, 55, 163, 259, 899, 951, 3245, 4191, 15813, 20195, 8361, 54025}},
	{6021, 17, 2483, {1, 1, 5, 11, 3, 17, 13, 223, 289, 255, 875, 2937, 1593, 9729, 21569, 63199, 83875}},
	{6022, 17, 2486, {1, 1, 1, 15, 19, 31, 17, 129, 267, 9, 2015, 3233, 6799, 12891, 18473, 37865, 19547}},
	{6023, 17, 2489, {1, 1, 5, 5, 5, 29, 81, 37, 357, 539, 1525, 2839, 8041, 5569, 4423, 8907, 35461}},
	{6024, 17, 2490, {1, 1, 5, 5, 29, 11, 85, 61, 333, 521, 1111, 3627, 325, 9805, 17889, 25655, 39537}},
	{6025, 17, 2518, {1, 3, 5, 11, 11, 53, 81, 25, 79, 253, 1963, 287, 7487, 15045, 21431, 35417, 102391}},
	{6026, 17, 2527, {1, 1, 1, 5, 11, 33, 45, 45, 425, 773, 1817, 4077, 1471, 11655, 683, 7115, 92651}},
	{6027, 17, 2540, {1, 1, 3, 3, 21, 13, 101, 215, 311, 853, 41, 1007, 5511, 2581, 25565, 13155, 117225}},
	{6028, 17, 2558, {1, 1, 3, 11, 19, 9, 125, 59, 273, 691, 499, 1547, 567, 10259, 21963, 48725, 3601}},
	{6029, 17, 2567, {1, 1, 3, 7, 27, 31, 39, 125, 317, 625, 1329, 3947, 3943, 6889, 2811, 34055, 1449}},
	{6030, 17, 2571, {1, 1, 1, 3, 29, 45, 73, 239, 319, 611, 647, 1839, 5277, 7807, 3107, 14683, 20203}},
	{6031, 17, 2574, {1, 3, 3, 3, 5, 5, 107, 139, 103, 809, 1343, 4041, 3273, 1789, 16205, 47873, 27803}},
	{6032, 17, 2579, {1, 3, 1, 9, 21, 23, 13, 131, 105, 741, 1773, 981, 5633, 14609, 12281, 50343, 14317}},
	{6033, 17, 2585, {1, 1, 1, 5, 11, 5, 125, 171, 109, 555, 159, 905, 691, 12401, 22817, 41411, 70113}},
	{6034, 17, 2615, {1, 3, 3, 9, 31, 37, 109, 231, 59, 615, 799, 319, 2459, 4521, 8525, 4827, 22969}},
	{6035, 17, 2639, {1, 3, 1, 5, 11, 7, 49, 237, 345, 473, 981, 2073, 6525, 8805, 13403, 3659, 69897}},
	{6036, 17, 2641, {1, 3, 1, 5, 9, 37, 13, 203, 141, 573, 745, 2613, 5589, 607, 24483, 38427, 95775}},
	{6037, 17, 2644, {1, 1, 3, 1, 23, 61, 75, 57, 299, 191, 805, 2993, 5175, 12037, 13649, 58831, 48791}},
	{6038, 17, 2663, {1, 3, 7, 13, 31, 57, 13, 219, 185, 717, 1607, 3785, 4719, 11583, 29285, 48207, 92021}},
	{6039, 17, 2667, {1, 3, 7, 15, 23, 35, 23, 69, 411, 773, 1549, 1087, 1685, 15703, 27193, 62675, 43505}},
	{6040, 17, 2669, {1, 1, 5, 3, 25, 19, 97, 75, 493, 549, 1655, 2881, 4989, 2765, 4797, 43143, 113955}},
	{6041, 17, 2672, {1, 1, 5, 7, 21, 5, 65, 37, 383, 133, 1907, 3747, 1525, 5803, 19977, 50551, 23157}},
	{6042, 17, 2687, {1, 1, 1, 11, 15, 61, 59, 109, 489, 901, 1787, 1611, 6101, 10653, 3071, 35643, 56227}},
	{6043, 17, 2700, {1, 3, 1, 5, 15, 25, 121, 111, 25, 251, 1467, 1795, 1631, 13753, 32391, 14831, 90739}},
	{6044, 17, 2705, {1, 1, 1, 13, 23, 55, 119, 147, 45, 871, 1389, 1929, 1023, 16131, 10041, 40055, 23337}},
	{6045, 17, 2724, {1, 3, 1, 15, 27, 33, 23, 41, 463, 603, 1633, 3445, 2007, 5999, 11175, 18343, 13159}},
	{6046, 17, 2728, {1, 3, 1, 9, 17, 15, 107, 63, 493, 411, 293, 3669, 6143, 3057, 8253, 25491, 58907}},
	{6047, 17, 2733, {1, 3, 5, 11, 1, 43, 5, 117, 127, 813, 1881, 3711, 2567, 7819, 5809, 64471, 104221}},
	{6048, 17, 2741, {1, 3, 5, 9, 25, 27, 49, 93, 77, 705, 1773, 1745, 4605, 16137, 14621, 62893, 81637}},
	{6049, 17, 2748, {1, 3, 1, 15, 9, 29, 41, 101, 291, 763, 1475, 3185, 3661, 10351, 26645, 50375, 59373}},
	{6050, 17, 2751, {1, 1, 5, 15, 9, 31, 107, 159, 125, 471, 1023, 2361, 4805, 8073, 21563, 14903, 77801}},
	{6051, 17, 2756, {1, 3, 7, 1, 27, 17, 75, 129, 71, 697, 551, 1969, 6597, 13821, 2605, 61783, 74791}},
	{6052, 17, 2771, {1, 1, 7, 15, 17, 27, 49, 47, 59, 47, 1671, 2535, 1299, 2387, 24349, 23661, 91123}},
	{6053, 17, 2774, {1, 1, 5, 15, 21, 61, 45, 37, 415, 189, 143, 351, 1815, 3479, 2399, 56753, 123893}},
	{6054, 17, 2793, {1, 1, 3, 7, 7, 19, 93, 249, 335, 305, 1437, 1329, 2693, 13201, 9589, 61513, 115995}},
	{6055, 17, 2796, {1, 1, 1, 11, 21, 57, 33, 205, 235, 253, 751, 259, 6029, 9811, 10231, 36899, 78035}},
	{6056, 17, 2804, {1, 1, 1, 11, 13, 25, 115, 195, 111, 913, 1851, 3283, 6083, 11717, 2773, 40727, 493}},
	{6057, 17, 2814, {1, 3, 3, 9, 9, 17, 83, 137, 465, 671, 1277, 325, 2767, 12413, 21977, 47525, 23041}},
	{6058, 17, 2822, {1, 1, 1, 11, 15, 47, 65, 219, 271, 197, 297, 3195, 1325, 9991, 26385, 46055, 43151}},
	{6059, 17, 2845, {1, 1, 1, 13, 31, 21, 39, 89, 127, 629, 367, 2935, 6259, 6627, 15691, 55781, 97251}},
	{6060, 17, 2846, {1, 1, 7, 13, 11, 45, 65, 75, 211, 785, 1221, 2087, 7751, 15619, 25489, 28195, 69007}},
	{6061, 17, 2850, {1, 3, 5, 15, 27, 37, 75, 111, 487, 219, 233, 583, 6433, 15105, 355, 28331, 21105}},
	{6062, 17, 2855, {1, 3, 3, 15, 31, 53, 33, 95, 27, 197, 1727, 1467, 7115, 15479, 26873, 31075, 12793}},
	{6063, 17, 2856, {1, 3, 7, 1, 19, 3, 19, 105, 225, 599, 737, 107, 7951, 10193, 31699, 59207, 85619}},
	{6064, 17, 2867, {1, 3, 1, 3, 7, 17, 73, 191, 247, 421, 537, 1473, 189, 4219, 29993, 25491, 21189}},
	{6065, 17, 2891, {1, 3, 7, 7, 13, 21, 33, 95, 147, 699, 943, 2275, 4093, 6067, 9063, 25503, 111085}},
	{6066, 17, 2894, {1, 1, 7, 9, 13, 47, 123, 121, 347, 467, 225, 957, 2329, 14075, 29843, 61753, 97179}},
	{6067, 17, 2902, {1, 3, 3, 7, 17, 55, 37, 167, 215, 819, 163, 1747, 4485, 15991, 28011, 36351, 106495}},
	{6068, 17, 2908, {1, 1, 3, 9, 25, 5, 83, 199, 209, 395, 1757, 1967, 5739, 2573, 13989, 32145, 4847}},
	{6069, 17, 2951, {1, 3, 3, 13, 11, 21, 25, 223, 239, 569, 1877, 299, 8089, 3697, 801, 64775, 26827}},
	{6070, 17, 2970, {1, 3, 5, 7, 17, 9, 127, 9, 65, 919, 1073, 2661, 1337, 10065, 30099, 30929, 90067}},
	{6071, 17, 2972, {1, 3, 1, 13, 25, 41, 35, 251, 279, 351, 111, 3917, 2815, 7989, 9895, 54859, 126355}},
	{6072, 17, 2975, {1, 1, 3, 7, 17, 61, 13, 73, 335, 831, 703, 37, 2765, 13169, 12513, 56301, 13907}},
	{6073, 17, 2976, {1, 1, 5, 13, 11, 15, 33, 45, 505, 127, 1723, 17, 4927, 11453, 28859, 9671, 80041}},
	{6074, 17, 2981, {1, 3, 1, 5, 9, 1, 25, 147, 281, 601, 243, 2687, 5533, 6725, 11075, 34807, 24619}},
	{6075, 17, 2986, {1, 1, 3, 1, 7, 21, 71, 31, 485, 561, 1361, 1237, 8171, 15885, 7941, 4583, 32851}},
	{6076, 17, 2999, {1, 3, 7, 1, 5, 35, 95, 155, 283, 959, 577, 1343, 4269, 13481, 30819, 40273, 8711}},
	{6077, 17, 3000, {1, 3, 7, 3, 1, 53, 77, 45, 215, 537, 1045, 77, 2791, 3553, 13273, 23819, 62263}},
	{6078, 17, 3006, {1, 3, 1, 15, 29, 59, 7, 145, 85, 3, 251, 2691, 7547, 11241, 32295, 24645, 75739}},
	{6079, 17, 3014, {1, 1, 5, 9, 19, 9, 39, 163, 303, 233, 2039, 2027, 7169, 2773, 28649, 38317, 66761}},
	{6080, 17, 3028, {1, 3, 7, 5, 21, 27, 93, 227, 131, 1019, 1619, 1497, 4043, 1131, 25761, 20173, 99957}},
	{6081, 17, 3031, {1, 3, 7, 5, 19, 33, 15, 173, 435, 399, 531, 2001, 3221, 12627, 10153, 24421, 61805}},
	{6082, 17, 3035, {1, 3, 1, 9, 11, 3, 69, 105, 289, 183, 1103, 831, 2297, 1613, 18801, 54395, 54243}},
	{6083, 17, 3037, {1, 3, 3, 9, 3, 53, 113, 183, 79, 355, 1629, 1061, 3713, 4563, 14365, 43529, 56073}},
	{6084, 17, 3053, {1, 3, 7, 11, 31, 39, 107, 139, 187, 873, 225, 33, 4943, 15837, 225, 6407, 85967}},
	{6085, 17, 3059, {1, 3, 1, 11, 17, 47, 93, 233, 119, 699, 1429, 2845, 2061, 8887, 20665, 45497, 33107}},
	{6086, 17, 3065, {1, 3, 5, 1, 25, 11, 55, 75, 91, 1009, 1887, 3167, 515, 15929, 11659, 57953, 63401}},
	{6087, 17, 3080, {1, 1, 3, 15, 27, 59, 103, 53, 353, 553, 2021, 1543, 2785, 9373, 14609, 21213, 19911}},
	{6088, 17, 3091, {1, 3, 7, 9, 3, 1, 101, 133, 437, 773, 1399, 1067, 7419, 1793, 16589, 3483, 42065}},
	{6089, 17, 3094, {1, 3, 7, 1, 25, 57, 127, 113, 65, 577, 1865, 1527, 6485, 11273, 15803, 39625, 75219}},
	{6090, 17, 3109, {1, 3, 5, 9, 7, 63, 29, 89, 155, 45, 1029, 2407, 6783, 4749, 4849, 26639, 54059}},
	{6091, 17, 3110, {1, 3, 7, 9, 25, 13, 113, 41, 267, 767, 1071, 1689, 269, 14437, 21255, 39473, 65771}},
	{6092, 17, 3113, {1, 3, 1, 15, 5, 3, 77, 43, 391, 763, 59, 1027, 6263, 3715, 31061, 43311, 130725}},
	{6093, 17, 3116, {1, 3, 7, 7, 21, 51, 127, 71, 229, 171, 397, 1099, 871, 2717, 1643, 17363, 125979}},
	{6094, 17, 3136, {1, 1, 5, 15, 25, 11, 11, 113, 203, 795, 1703, 3901, 1113, 12819, 25345, 46691, 112313}},
	{6095, 17, 3139, {1, 3, 7, 5, 1, 59, 91, 81, 325, 483, 595, 1491, 7455, 6699, 199, 35597, 59851}},
	{6096, 17, 3141, {1, 3, 5, 1, 3, 33, 43, 195, 201, 575, 1395, 1305, 7001, 2023, 22419, 15233, 120355}},
	{6097, 17, 3154, {1, 1, 3, 3, 15, 37, 81, 59, 87, 675, 199, 3231, 4473, 5023, 16753, 51475, 102113}},
	{6098, 17, 3160, {1, 1, 7, 9, 13, 39, 65, 9, 51, 565, 1171, 119, 7875, 12149, 6565, 56849, 123235}},
	{6099, 17, 3169, {1, 3, 3, 7, 15, 45, 53, 93, 111, 533, 1849, 643, 2265, 10241, 24741, 11559, 74333}},
	{6100, 17, 3182, {1, 3, 1, 1, 11, 61, 75, 51, 5, 199, 535, 279, 5821, 6005, 2907, 32521, 74121}},
	{6101, 17, 3187, {1, 1, 3, 15, 3, 21, 29, 193, 71, 993, 1719, 1865, 6135, 7683, 12171, 29275, 14539}},
	{6102, 17, 3189, {1, 1, 1, 7, 7, 13, 1, 61, 315, 431, 1145, 2067, 5745, 1641, 1047, 55111, 129477}},
	{6103, 17, 3190, {1, 1, 5, 1, 21, 43, 115, 193, 153, 573, 1181, 3947, 7809, 11317, 30649, 56891, 47741}},
	{6104, 17, 3203, {1, 1, 5, 7, 19, 15, 61, 239, 109, 683, 395, 2869, 3103, 1531, 12019, 45159, 37525}},
	{6105, 17, 3217, {1, 1, 5, 7, 29, 55, 45, 7, 353, 659, 591, 3371, 5777, 8475, 2743, 47483, 11983}},
	{6106, 17, 3229, {1, 3, 1, 3, 13, 17, 39, 195, 43, 5, 1749, 2559, 5843, 8719, 21421, 58511, 105637}},
	{6107, 17, 3236, {1, 3, 5, 5, 5, 21, 29, 63, 387, 301, 567, 3325, 2109, 403, 23053, 24851, 14493}},
	{6108, 17, 3248, {1, 1, 3, 3, 17, 57, 107, 131, 85, 855, 1101, 3199, 7159, 14739, 4197, 27943, 113009}},
	{6109, 17, 3257, {1, 1, 3, 11, 1, 61, 31, 79, 33, 123, 1509, 507, 6679, 2279, 8465, 37279, 17553}},
	{6110, 17, 3278, {1, 3, 1, 15, 7, 33, 11, 71, 217, 609, 1661, 3437, 5497, 13365, 6247, 649, 26407}},
	{6111, 17, 3283, {1, 1, 3, 1, 19, 45, 49, 125, 5, 455, 1669, 4083, 253, 10101, 27327, 16401, 120399}},
	{6112, 17, 3289, {1, 3, 1, 1, 27, 19, 117, 137, 261, 341, 1697, 457, 7553, 12169, 30049, 49281, 36937}},
	{6113, 17, 3292, {1, 1, 1, 3, 9, 49, 33, 13, 461, 545, 1537, 2623, 883, 10921, 5583, 58997, 114183}},
	{6114, 17, 3302, {1, 1, 7, 9, 29, 53, 29, 165, 205, 989, 1347, 2343, 7505, 7609, 18503, 51677, 105993}},
	{6115, 17, 3316, {1, 1, 1, 13, 1, 29, 59, 121, 297, 659, 1965, 1765, 5255, 10971, 32613, 18763, 41983}},
	{6116, 17, 3328, {1, 3, 7, 11, 21, 41, 19, 47, 125, 485, 475, 2745, 4075, 8101, 31227, 4679, 115473}},
	{6117, 17, 3333, {1, 3, 3, 7, 21, 23, 55, 65, 223, 1001, 317, 1459, 183, 5139, 26553, 41471, 116373}},
	{6118, 17, 3337, {1, 1, 7, 3, 1, 9, 29, 139, 343, 913, 1993, 3139, 3791, 5869, 6057, 23863, 35737}},
	{6119, 17, 3340, {1, 3, 3, 3, 7, 21, 77, 197, 239, 467, 35, 591, 1061, 3417, 31811, 38825, 124981}},
	{6120, 17, 3368, {1, 3, 3, 1, 21, 29, 5, 213, 417, 111, 1681, 1409, 2899, 16233, 1053, 51235, 87767}},
	{6121, 17, 3371, {1, 1, 5, 3, 13, 47, 61, 203, 223, 73, 1947, 3613, 5885, 13567, 7593, 34329, 68597}},
	{6122, 17, 3376, {1, 3, 1, 1, 17, 9, 11, 187, 361, 973, 781, 1835, 1539, 12917, 21725, 48279, 115037}},
	{6123, 17, 3385, {1, 3, 1, 1, 9, 25, 117, 157, 433, 395, 403, 2183, 3327, 5427, 7505, 2673, 77137}},
	{6124, 17, 3386, {1, 1, 7, 15, 31, 15, 27, 155, 441, 837, 1877, 3829, 5139, 16331, 31183, 15803, 95699}},
	{6125, 17, 3393, {1, 1, 7, 15, 5, 51, 77, 179, 289, 727, 1763, 2529, 6715, 3967, 29267, 27293, 67953}},
	{6126, 17, 3399, {1, 3, 7, 13, 7, 3, 3, 17, 311, 547, 1465, 1413, 3937, 2725, 24523, 12321, 109763}},
	{6127, 17, 3405, {1, 3, 5, 15, 9, 5, 87, 135, 281, 97, 2021, 1903, 8007, 10321, 27989, 18993, 110407}},
	{6128, 17, 3414, {1, 1, 1, 13, 25, 61, 89, 107, 233, 823, 1375, 3531, 1757, 1577, 29457, 1461, 17217}},
	{6129, 17, 3433, {1, 1, 1, 13, 17, 17, 27, 193, 485, 759, 145, 3943, 4183, 14119, 11217, 3793, 1935}},
	{6130, 17, 3436, {1, 1, 1, 3, 13, 31, 101, 227, 311, 363, 1925, 1525, 5275, 2385, 15093, 48769, 121189}},
	{6131, 17, 3448, {1, 1, 5, 13, 11, 61, 89, 141, 117, 229, 417, 3935, 7249, 13869, 30591, 62763, 67521}},
	{6132, 17, 3467, {1, 1, 3, 15, 7, 59, 105, 239, 453, 221, 1101, 395, 2031, 8941, 23155, 7077, 125593}},
	{6133, 17, 3469, {1, 1, 1, 11, 7, 55, 99, 31, 305, 371, 1035, 577, 4473, 577, 371, 46093, 69157}},
	{6134, 17, 3472, {1, 3, 1, 9, 9, 33, 35, 245, 95, 47, 1623, 2965, 6849, 7269, 5321, 31641, 73321}},
	{6135, 17, 3477, {1, 1, 1, 15, 21, 61, 65, 65, 159, 151, 625, 2281, 2993, 1311, 29757, 24703, 71029}},
	{6136, 17, 3484, {1, 3, 5, 15, 29, 59, 29, 69, 351, 901, 631, 3501, 7031, 703, 20805, 36437, 94931}},
	{6137, 17, 3494, {1, 3, 7, 1, 21, 11, 19, 125, 237, 807, 1651, 2389, 7347, 11759, 27151, 38669, 965}},
	{6138, 17, 3505, {1, 1, 5, 1, 15, 41, 1, 105, 89, 127, 895, 29, 2339, 15951, 18633, 2781, 67269}},
	{6139, 17, 3515, {1, 1, 5, 15, 25, 7, 3, 33, 375, 447, 203, 2579, 6145, 14015, 9939, 52777, 123181}},
	{6140, 17, 3523, {1, 3, 1, 15, 29, 7, 7, 27, 451, 869, 107, 2457, 5557, 11601, 28957, 36181, 41419}},
	{6141, 17, 3530, {1, 1, 1, 7, 1, 57, 33, 213, 329, 763, 815, 169, 623, 155, 20529, 20603, 73311}},
	{6142, 17, 3543, {1, 3, 5, 7, 25, 21, 7, 217, 159, 89, 1373, 1735, 705, 4093, 13083, 3855, 55875}},
	{6143, 17, 3559, {1, 3, 1, 1, 29, 33, 105, 127, 95, 543, 235, 67, 691, 5015, 22139, 18251, 89945}},
	{6144, 17, 3568, {1, 1, 3, 11, 27, 53, 105, 83, 337, 331, 1571, 1145, 745, 1845, 17881, 17697, 88139}},
	{6145, 17, 3577, {1, 3, 7, 15, 19, 37, 119, 35, 35, 463, 1925, 1665, 673, 12193, 12137, 62371, 10957}},
	{6146, 17, 3578, {1, 3, 3, 3, 19, 21, 113, 29, 459, 467, 623, 2661, 857, 16265, 27509, 46555, 18867}},
	{6147, 17, 3594, {1, 3, 7, 5, 17, 49, 123, 41, 85, 673, 41, 1871, 7649, 8687, 28269, 64423, 93675}},
	{6148, 17, 3601, {1, 3, 3, 3, 7, 23, 101, 171, 181, 527, 65, 2387, 6629, 6089, 17387, 46551, 36143}},
	{6149, 17, 3607, {1, 1, 5, 1, 13, 51, 21, 251, 139, 429, 1993, 3767, 1089, 5459, 19407, 41747, 41033}},
	{6150, 17, 3608, {1, 1, 1, 11, 15, 9, 81, 91, 73, 969, 1513, 2067, 7959, 2605, 26641, 37631, 124571}},
	{6151, 17, 3620, {1, 1, 3, 15, 29, 15, 5, 57, 247, 901, 527, 3325, 5859, 11299, 9871, 63947, 125247}},
	{6152, 17, 3629, {1, 3, 1, 5, 1, 35, 75, 21, 307, 43, 1111, 3299, 1647, 3585, 31045, 18217, 95169}},
	{6153, 17, 3644, {1, 3, 1, 7, 23, 35, 11, 103, 3, 461, 1915, 4019, 453, 13111, 26941, 43091, 22917}},
	{6154, 17, 3656, {1, 1, 5, 5, 1, 61, 121, 167, 475, 5, 1749, 887, 2237, 5055, 7077, 29453, 17691}},
	{6155, 17, 3664, {1, 3, 3, 15, 15, 15, 9, 15, 171, 787, 1965, 577, 4507, 7325, 20901, 8557, 111909}},
	{6156, 17, 3670, {1, 3, 5, 1, 27, 15, 123, 141, 63, 55, 599, 4095, 1245, 13919, 27485, 49977, 74551}},
	{6157, 17, 3680, {1, 3, 5, 9, 21, 61, 79, 119, 7, 573, 1923, 2775, 3127, 12689, 12135, 53429, 130163}},
	{6158, 17, 3685, {1, 3, 3, 13, 27, 41, 67, 249, 447, 277, 311, 775, 8187, 10161, 12953, 22885, 121247}},
	{6159, 17, 3686, {1, 3, 5, 9, 21, 55, 115, 65, 45, 395, 481, 2063, 6493, 4199, 19219, 27119, 62255}},
	{6160, 17, 3695, {1, 1, 3, 13, 7, 41, 3, 127, 383, 923, 1725, 1033, 7731, 11971, 3089, 46459, 98369}},
	{6161, 17, 3698, {1, 1, 3, 11, 13, 39, 39, 149, 309, 311, 1491, 807, 2109, 363, 14637, 65429, 124731}},
	{6162, 17, 3703, {1, 1, 7, 13, 13, 35, 67, 81, 493, 859, 1177, 237, 4605, 15319, 16669, 16661, 21385}},
	{6163, 17, 3710, {1, 1, 3, 7, 7, 39, 57, 103, 239, 753, 221, 1611, 1557, 13317, 27453, 10245, 33839}},
	{6164, 17, 3714, {1, 1, 5, 13, 27, 53, 97, 41, 123, 253, 535, 1839, 5827, 7587, 1261, 20313, 65961}},
	{6165, 17, 3726, {1, 1, 7, 1, 11, 47, 93, 135, 223, 591, 1087, 3329, 3293, 14207, 6187, 54789, 23781}},
	{6166, 17, 3731, {1, 3, 7, 7, 25, 21, 97, 105, 269, 515, 1805, 3711, 3295, 7307, 21065, 65205, 116969}},
	{6167, 17, 3733, {1, 3, 1, 11, 25, 37, 21, 89, 109, 581, 1055, 2393, 1291, 1115, 25545, 36383, 93605}},
	{6168, 17, 3737, {1, 3, 7, 1, 27, 13, 113, 11, 395, 473, 943, 4045, 5507, 15051, 25203, 2971, 31961}},
	{6169, 17, 3756, {1, 1, 5, 5, 27, 35, 57, 219, 67, 949, 659, 203, 5235, 6509, 13731, 61533, 54963}},
	{6170, 17, 3759, {1, 3, 1, 1, 15, 39, 85, 13, 347, 99, 25, 3595, 3081, 13617, 14373, 58909, 102181}},
	{6171, 17, 3767, {1, 1, 7, 13, 3, 25, 97, 91, 287, 389, 665, 2981, 2301, 12625, 4495, 57489, 68677}},
	{6172, 17, 3776, {1, 1, 5, 1, 15, 57, 77, 55, 299, 713, 1457, 3699, 2807, 5549, 467, 47367, 8163}},
	{6173, 17, 3785, {1, 1, 7, 3, 23, 45, 91, 251, 501, 193, 1121, 2359, 4781, 12797, 13713, 55171, 927}},
	{6174, 17, 3793, {1, 3, 3, 7, 7, 31, 87, 163, 249, 163, 937, 1293, 4827, 10299, 31935, 58787, 80589}},
	{6175, 17, 3812, {1, 3, 1, 9, 7, 1, 73, 65, 475, 791, 1429, 3319, 7149, 433, 10373, 44061, 121195}},
	{6176, 17, 3815, {1, 1, 5, 9, 9, 61, 27, 249, 435, 437, 1329, 2163, 5859, 13663, 623, 55569, 94283}},
	{6177, 17, 3824, {1, 3, 7, 11, 1, 29, 117, 195, 399, 999, 1705, 1325, 6043, 9823, 27335, 30377, 16627}},
	{6178, 17, 3844, {1, 1, 1, 15, 5, 11, 63, 185, 15, 741, 1061, 2961, 3455, 5, 26587, 54081, 18107}},
	{6179, 17, 3859, {1, 1, 5, 7, 29, 57, 17, 203, 501, 177, 49, 2773, 8069, 12513, 14437, 64489, 58661}},
	{6180, 17, 3866, {1, 3, 3, 9, 11, 23, 121, 3, 415, 447, 1773, 135, 5901, 4951, 2683, 437, 126251}},
	{6181, 17, 3872, {1, 3, 3, 1, 7, 23, 17, 23, 115, 591, 1075, 3133, 49, 15183, 10615, 37857, 122609}},
	{6182, 17, 3884, {1, 1, 3, 3, 13, 49, 63, 37, 275, 763, 1135, 2913, 1563, 11037, 6693, 18799, 32089}},
	{6183, 17, 3889, {1, 3, 5, 11, 7, 29, 59, 45, 227, 941, 1947, 2733, 797, 10485, 7071, 14741, 11451}},
	{6184, 17, 3899, {1, 1, 1, 9, 21, 19, 77, 97, 75, 991, 187, 1003, 5619, 11013, 3931, 19907, 79723}},
	{6185, 17, 3902, {1, 1, 7, 13, 1, 57, 61, 177, 443, 227, 1347, 2665, 2011, 12329, 14137, 37795, 63331}},
	{6186, 17, 3909, {1, 3, 3, 9, 31, 59, 87, 93, 485, 635, 901, 1845, 6153, 10797, 1289, 8989, 41717}},
	{6187, 17, 3913, {1, 1, 1, 1, 3, 7, 85, 17, 67, 309, 1891, 435, 303, 8011, 32127, 54309, 21457}},
	{6188, 17, 3933, {1, 3, 7, 1, 29, 27, 41, 239, 293, 717, 1331, 917, 6145, 7131, 28199, 35093, 103683}},
	{6189, 17, 3938, {1, 3, 7, 3, 21, 63, 65, 233, 257, 789, 1095, 505, 4557, 16259, 7397, 24815, 89529}},
	{6190, 17, 3949, {1, 3, 3, 11, 29, 41, 55, 17, 335, 715, 779, 2121, 6393, 8887, 32753, 45647, 82665}},
	{6191, 17, 3952, {1, 1, 1, 11, 27, 47, 71, 13, 141, 283, 967, 3359, 4309, 6661, 20481, 23175, 50835}},
	{6192, 17, 3980, {1, 3, 3, 7, 3, 25, 19, 241, 409, 573, 1565, 3355, 1307, 12205, 18017, 8271, 117007}},
	{6193, 17, 3991, {1, 3, 3, 9, 21, 39, 21, 253, 439, 963, 341, 3637, 2275, 1845, 11015, 481, 83369}},
	{6194, 17, 3992, {1, 3, 7, 9, 31, 29, 29, 163, 111, 983, 571, 713, 2621, 11569, 13341, 28341, 130381}},
	{6195, 17, 4002, {1, 3, 7, 7, 11, 35, 89, 49, 81, 115, 113, 1857, 3527, 14819, 6909, 14659, 23557}},
	{6196, 17, 4008, {1, 3, 3, 15, 29, 41, 85, 241, 317, 737, 213, 1667, 5789, 16321, 13991, 36165, 124151}},
	{6197, 17, 4011, {1, 3, 1, 3, 31, 1, 75, 99, 495, 241, 1499, 1535, 2033, 2135, 6699, 58893, 37031}},
	{6198, 17, 4016, {1, 1, 7, 9, 25, 15, 101, 23, 477, 563, 1691, 2655, 2321, 2323, 4255, 22055, 99661}},
	{6199, 17, 4034, {1, 3, 7, 5, 7, 7, 49, 221, 51, 83, 279, 2205, 2939, 2119, 14073, 32839, 108075}},
	{6200, 17, 4036, {1, 3, 5, 11, 17, 39, 3, 127, 87, 501, 799, 401, 4439, 9895, 13017, 64975, 67177}},
	{6201, 17, 4063, {1, 3, 3, 9, 17, 41, 59, 95, 283, 309, 83, 1293, 6385, 5783, 30115, 33997, 12531}},
	{6202, 17, 4067, {1, 3, 5, 3, 7, 31, 69, 171, 225, 409, 1237, 3343, 835, 8039, 16723, 37203, 129047}},
	{6203, 17, 4073, {1, 3, 3, 15, 17, 23, 107, 1, 105, 135, 1245, 993, 4101, 7325, 7425, 17379, 98121}},
	{6204, 17, 4082, {1, 1, 7, 9, 27, 5, 67, 111, 75, 531, 243, 2239, 2527, 4513, 27059, 40533, 88169}},
	{6205, 17, 4091, {1, 3, 5, 7, 21, 63, 57, 15, 75, 679, 1729, 1845, 6259, 8531, 18691, 49321, 101599}},
	{6206, 17, 4093, {1, 1, 5, 9, 3, 35, 7, 201, 351, 885, 669, 2339, 5009, 279, 26469, 54597, 67933}},
	{6207, 17, 4101, {1, 3, 5, 13, 27, 5, 85, 161, 141, 733, 1017, 2021, 6951, 15595, 21817, 17243, 88607}},
	{6208, 17, 4113, {1, 3, 5, 1, 11, 31, 117, 97, 175, 629, 995, 1207, 2941, 5825, 5319, 48191, 9505}},
	{6209, 17, 4120, {1, 3, 3, 7, 25, 39, 45, 79, 21, 607, 1593, 1749, 7951, 10425, 17491, 16617, 56903}},
	{6210, 17, 4125, {1, 1, 1, 5, 15, 41, 107, 115, 79, 693, 919, 3513, 6793, 6541, 5545, 58583, 27963}},
	{6211, 17, 4126, {1, 3, 7, 11, 21, 19, 123, 1, 441, 531, 359, 2117, 2465, 11389, 13489, 32755, 4577}},
	{6212, 17, 4139, {1, 1, 5, 13, 7, 7, 7, 127, 201, 377, 1423, 269, 2611, 3339, 19153, 25659, 33069}},
	{6213, 17, 4142, {1, 3, 7, 1, 13, 35, 45, 5, 313, 739, 1779, 2983, 1815, 8817, 14239, 3921, 57975}},
	{6214, 17, 4144, {1, 3, 1, 11, 9, 39, 33, 111, 39, 255, 159, 2345, 2193, 11475, 12841, 47579, 90309}},
	{6215, 17, 4147, {1, 1, 1, 3, 27, 49, 85, 157, 243, 247, 1473, 323, 4631, 1787, 15193, 5533, 104999}},
	{6216, 17, 4153, {1, 1, 7, 9, 11, 29, 23, 219, 57, 339, 1797, 409, 6025, 10569, 27409, 15147, 130281}},
	{6217, 17, 4154, {1, 1, 7, 1, 31, 31, 113, 229, 63, 877, 319, 2655, 3335, 7743, 19593, 10089, 28215}},
	{6218, 17, 4164, {1, 1, 3, 11, 23, 3, 71, 235, 329, 751, 159, 2579, 5363, 12681, 20233, 53855, 16407}},
	{6219, 17, 4174, {1, 1, 5, 1, 7, 61, 21, 235, 379, 849, 61, 2969, 6399, 2655, 21635, 16955, 58675}},
	{6220, 17, 4182, {1, 3, 7, 7, 29, 15, 5, 11, 143, 699, 1875, 2115, 6633, 6195, 5829, 53633, 111221}},
	{6221, 17, 4185, {1, 3, 7, 11, 19, 41, 17, 219, 483, 829, 1233, 3183, 6283, 2363, 25245, 63075, 82733}},
	{6222, 17, 4188, {1, 3, 7, 13, 21, 17, 1, 207, 443, 575, 521, 2585, 6875, 14871, 14739, 10211, 127435}},
	{6223, 17, 4191, {1, 3, 7, 7, 15, 39, 99, 197, 219, 259, 1723, 3737, 6625, 849, 887, 41293, 53825}},
	{6224, 17, 4195, {1, 3, 3, 3, 5, 3, 75, 155, 189, 935, 85, 2273, 1375, 4217, 10709, 58047, 81689}},
	{6225, 17, 4219, {1, 3, 5, 5, 27, 27, 107, 229, 179, 887, 91, 421, 7313, 6495, 451, 43859, 40033}},
	{6226, 17, 4225, {1, 3, 5, 11, 25, 49, 121, 73, 169, 311, 1387, 1037, 6519, 9317, 26975, 50627, 46805}},
	{6227, 17, 4228, {1, 1, 5, 11, 17, 21, 19, 125, 387, 697, 1017, 1759, 7295, 9869, 28241, 9367, 119255}},
	{6228, 17, 4232, {1, 1, 7, 5, 29, 27, 87, 187, 95, 625, 933, 1751, 5253, 313, 30841, 16349, 67347}},
	{6229, 17, 4246, {1, 1, 3, 3, 15, 51, 23, 101, 183, 267, 243, 711, 983, 12461, 17801, 1429, 47273}},
	{6230, 17, 4255, {1, 1, 1, 3, 17, 3, 73, 67, 49, 449, 879, 2559, 401, 11983, 13697, 12023, 78855}},
	{6231, 17, 4274, {1, 3, 7, 15, 25, 25, 43, 81, 141, 161, 595, 621, 1165, 10869, 22875, 6741, 90017}},
	{6232, 17, 4283, {1, 3, 5, 11, 13, 57, 53, 219, 145, 937, 769, 1961, 4725, 3335, 12623, 8335, 46305}},
	{6233, 17, 4286, {1, 1, 3, 5, 7, 39, 19, 101, 313, 583, 483, 2515, 125, 5211, 2559, 11937, 126717}},
	{6234, 17, 4306, {1, 3, 1, 7, 7, 1, 117, 49, 231, 133, 381, 697, 927, 8263, 26529, 64881, 25059}},
	{6235, 17, 4311, {1, 1, 1, 15, 11, 25, 77, 149, 233, 215, 1239, 3045, 99, 11183, 30279, 32271, 100943}},
	{6236, 17, 4317, {1, 1, 5, 7, 31, 25, 1, 51, 221, 607, 1733, 2145, 6765, 7011, 16927, 29257, 2445}},
	{6237, 17, 4321, {1, 3, 5, 1, 19, 23, 123, 93, 381, 295, 765, 2335, 8025, 14003, 4801, 54243, 57297}},
	{6238, 17, 4324, {1, 1, 7, 9, 9, 31, 63, 191, 495, 527, 251, 2119, 1663, 209, 7445, 1441, 4075}},
	{6239, 17, 4331, {1, 3, 5, 5, 13, 17, 97, 79, 369, 55, 677, 2031, 7315, 4769, 31659, 21975, 22061}},
	{6240, 17, 4333, {1, 3, 3, 7, 3, 63, 121, 243, 39, 917, 1917, 297, 7241, 1565, 31675, 14443, 67239}},
	{6241, 17, 4359, {1, 3, 7, 1, 13, 25, 51, 65, 145, 475, 1853, 4023, 5121, 14411, 15993, 42165, 13615}},
	{6242, 17, 4360, {1, 3, 3, 1, 3, 51, 75, 29, 169, 311, 1309, 2929, 7669, 1507, 14605, 32667, 103861}},
	{6243, 17, 4368, {1, 3, 7, 1, 23, 37, 89, 211, 137, 495, 1469, 3425, 1167, 12429, 27301, 46857, 83007}},
	{6244, 17, 4373, {1, 3, 7, 7, 27, 37, 33, 129, 73, 23, 761, 119, 6217, 4749, 20835, 47477, 33665}},
	{6245, 17, 4389, {1, 1, 3, 5, 29, 35, 79, 21, 183, 933, 43, 3149, 5273, 12159, 20695, 5387, 23569}},
	{6246, 17, 4394, {1, 1, 5, 5, 3, 11, 57, 205, 349, 657, 1509, 3693, 5495, 11865, 13861, 62215, 94141}},
	{6247, 17, 4413, {1, 3, 1, 7, 17, 43, 117, 119, 75, 849, 1247, 643, 2691, 2289, 9759, 18683, 68649}},
	{6248, 17, 4422, {1, 1, 1, 15, 5, 55, 89, 177, 427, 701, 735, 2993, 5293, 15395, 567, 5501, 102393}},
	{6249, 17, 4431, {1, 3, 3, 15, 5, 37, 73, 111, 9, 141, 407, 1579, 6691, 11843, 6377, 64181, 97347}},
	{6250, 17, 4436, {1, 1, 5, 1, 9, 17, 71, 127, 285, 929, 1243, 2605, 359, 14589, 32603, 39879, 115901}},
	{6251, 17, 4440, {1, 3, 7, 15, 3, 27, 91, 121, 47, 631, 1589, 385, 5997, 14077, 21285, 33895, 36985}},
	{6252, 17, 4445, {1, 3, 3, 9, 1, 47, 89, 79, 213, 27, 547, 1703, 4035, 13205, 4341, 21895, 34247}},
	{6253, 17, 4452, {1, 3, 5, 7, 9, 9, 47, 89, 231, 857, 297, 2949, 2715, 1275, 14427, 20227, 21569}},
	{6254, 17, 4462, {1, 3, 1, 3, 15, 57, 61, 183, 377, 477, 1135, 1729, 2863, 8607, 29241, 34983, 84443}},
	{6255, 17, 4469, {1, 1, 7, 7, 5, 53, 91, 149, 71, 41, 1025, 3945, 3989, 15853, 20903, 26943, 99841}},
	{6256, 17, 4470, {1, 3, 3, 3, 29, 21, 59, 217, 483, 257, 331, 657, 2935, 945, 9821, 42501, 98087}},
	{6257, 17, 4473, {1, 3, 5, 3, 17, 39, 123, 103, 109, 957, 853, 3821, 555, 10869, 27673, 38315, 83105}},
	{6258, 17, 4479, {1, 3, 1, 3, 27, 7, 97, 57, 429, 53, 1791, 1405, 4113, 8435, 12845, 21567, 91559}},
	{6259, 17, 4480, {1, 3, 3, 1, 17, 61, 125, 77, 225, 395, 945, 3213, 1363, 15947, 27049, 4389, 64037}},
	{6260, 17, 4483, {1, 1, 1, 3, 15, 51, 15, 189, 449, 989, 939, 985, 6929, 13779, 25011, 22277, 72543}},
	{6261, 17, 4489, {1, 3, 3, 1, 25, 53, 5, 219, 195, 703, 163, 1405, 821, 6797, 14329, 1675, 96653}},
	{6262, 17, 4503, {1, 1, 7, 13, 7, 1, 45, 135, 369, 125, 711, 2509, 131, 13663, 29769, 19497, 116779}},
	{6263, 17, 4519, {1, 1, 7, 15, 23, 25, 7, 225, 435, 835, 1981, 2537, 5727, 15961, 30089, 58905, 100339}},
	{6264, 17, 4520, {1, 3, 7, 3, 19, 9, 79, 63, 371, 419, 1357, 3649, 7987, 14541, 6631, 50555, 84217}},
	{6265, 17, 4525, {1, 3, 3, 9, 7, 61, 11, 157, 99, 95, 945, 2803, 1703, 117, 12891, 21817, 84259}},
	{6266, 17, 4526, {1, 3, 7, 7, 25, 37, 111, 99, 65, 599, 1313, 2557, 5489, 3625, 7429, 19309, 78111}},
	{6267, 17, 4533, {1, 3, 1, 1, 19, 15, 85, 253, 347, 315, 1349, 983, 2507, 4155, 15311, 43535, 101409}},
	{6268, 17, 4552, {1, 3, 3, 3, 1, 55, 3, 57, 375, 107, 177, 1673, 6871, 7137, 10297, 65363, 42293}},
	{6269, 17, 4581, {1, 1, 1, 3, 9, 5, 83, 45, 139, 893, 63, 2859, 6333, 15591, 18491, 26387, 25573}},
	{6270, 17, 4585, {1, 1, 7, 15, 1, 39, 113, 127, 503, 617, 1367, 1855, 185, 4233, 5787, 8265, 42097}},
	{6271, 17, 4591, {1, 1, 3, 11, 11, 41, 119, 165, 331, 625, 81, 2495, 7247, 9139, 15269, 31447, 128425}},
	{6272, 17, 4594, {1, 1, 5, 5, 17, 35, 39, 1, 91, 563, 1841, 2975, 1233, 3837, 22145, 36719, 104503}},
	{6273, 17, 4596, {1, 1, 7, 3, 23, 35, 77, 69, 271, 487, 921, 2597, 8011, 13037, 6001, 20519, 32673}},
	{6274, 17, 4599, {1, 1, 1, 1, 29, 17, 11, 145, 473, 877, 813, 727, 6805, 3563, 13371, 22169, 17239}},
	{6275, 17, 4612, {1, 1, 1, 13, 17, 13, 1, 125, 313, 423, 1079, 2401, 2325, 2219, 24071, 25613, 34163}},
	{6276, 17, 4621, {1, 1, 5, 7, 29, 33, 53, 215, 11, 555, 555, 1965, 3643, 5433, 12923, 59655, 25339}},
	{6277, 17, 4630, {1, 3, 3, 3, 23, 37, 119, 117, 459, 359, 1849, 1019, 433, 15391, 5625, 52649, 81313}},
	{6278, 17, 4636, {1, 3, 3, 1, 21, 31, 121, 161, 113, 667, 863, 105, 3805, 14459, 28235, 24543, 89755}},
	{6279, 17, 4640, {1, 1, 5, 15, 17, 37, 15, 111, 511, 477, 611, 955, 2591, 16137, 14179, 30995, 129575}},
	{6280, 17, 4649, {1, 3, 3, 3, 21, 49, 25, 37, 287, 263, 851, 1015, 8133, 9429, 10959, 64483, 82533}},
	{6281, 17, 4650, {1, 1, 5, 1, 25, 19, 49, 159, 155, 443, 975, 1413, 321, 7871, 22935, 57303, 124027}},
	{6282, 17, 4660, {1, 3, 1, 1, 19, 45, 47, 89, 409, 509, 1249, 2445, 2053, 3781, 7517, 61869, 125137}},
	{6283, 17, 4677, {1, 1, 5, 13, 27, 57, 45, 43, 361, 329, 1321, 771, 4665, 12245, 18993, 15121, 127485}},
	{6284, 17, 4687, {1, 3, 3, 7, 3, 41, 127, 75, 485, 821, 497, 2649, 6423, 12419, 31421, 9441, 63645}},
	{6285, 17, 4696, {1, 1, 3, 5, 19, 61, 91, 35, 311, 287, 449, 3955, 5805, 5631, 25613, 55409, 104545}},
	{6286, 17, 4701, {1, 3, 7, 11, 27, 19, 27, 53, 19, 35, 1687, 3923, 3379, 10435, 15053, 12343, 89077}},
	{6287, 17, 4705, {1, 3, 5, 13, 31, 41, 15, 239, 349, 533, 1771, 737, 6503, 14355, 18781, 27805, 79049}},
	{6288, 17, 4706, {1, 3, 1, 3, 13, 11, 69, 227, 169, 873, 533, 2217, 1047, 12415, 12271, 22447, 14163}},
	{6289, 17, 4711, {1, 1, 3, 9, 7, 31, 23, 155, 133, 305, 1569, 521, 201, 10339, 16999, 29163, 32817}},
	{6290, 17, 4720, {1, 1, 1, 5, 31, 57, 43, 223, 121, 803, 357, 1855, 4321, 10245, 25725, 2543, 47395}},
	{6291, 17, 4723, {1, 3, 5, 9, 3, 5, 47, 189, 217, 899, 1455, 691, 1277, 7861, 3627, 14895, 41109}},
	{6292, 17, 4732, {1, 3, 7, 3, 29, 9, 37, 63, 453, 709, 921, 771, 8069, 239, 22639, 59937, 10635}},
	{6293, 17, 4736, {1, 3, 7, 1, 11, 51, 79, 131, 225, 757, 549, 1605, 3921, 1849, 16307, 29809, 120597}},
	{6294, 17, 4742, {1, 3, 7, 7, 1, 45, 33, 185, 23, 881, 1941, 4093, 4741, 11633, 2059, 32007, 11103}},
	{6295, 17, 4748, {1, 3, 5, 11, 17, 21, 43, 205, 363, 559, 697, 4057, 631, 6697, 883, 61705, 102791}},
	{6296, 17, 4754, {1, 1, 7, 9, 29, 35, 109, 85, 373, 321, 415, 2969, 6163, 6999, 9999, 36435, 125267}},
	{6297, 17, 4759, {1, 1, 7, 11, 25, 9, 113, 91, 337, 889, 947, 2093, 5289, 1367, 13297, 36155, 21825}},
	{6298, 17, 4769, {1, 1, 3, 9, 17, 25, 35, 79, 275, 687, 335, 1181, 7327, 3729, 1561, 27441, 114355}},
	{6299, 17, 4787, {1, 3, 3, 11, 25, 41, 27, 89, 115, 361, 871, 1497, 5735, 6365, 1737, 14277, 63847}},
	{6300, 17, 4807, {1, 3, 7, 7, 1, 63, 31, 73, 289, 67, 277, 1821, 4883, 10795, 11755, 15471, 105871}},
	{6301, 17, 4814, {1, 3, 7, 9, 23, 17, 37, 179, 409, 957, 373, 2393, 2363, 6735, 28737, 41927, 115735}},
	{6302, 17, 4837, {1, 1, 3, 9, 15, 43, 111, 61, 455, 181, 1643, 3063, 4311, 13705, 29993, 21731, 25243}},
	{6303, 17, 4867, {1, 1, 1, 15, 13, 13, 69, 187, 91, 395, 209, 3477, 4649, 7727, 30557, 14719, 1953}},
	{6304, 17, 4873, {1, 1, 1, 15, 9, 39, 119, 193, 459, 135, 567, 25, 4583, 8401, 22161, 14771, 74165}},
	{6305, 17, 4879, {1, 1, 3, 7, 5, 39, 77, 149, 293, 585, 1245, 3615, 357, 11613, 13865, 40227, 41023}},
	{6306, 17, 4884, {1, 1, 7, 9, 9, 37, 5, 177, 121, 181, 771, 733, 7683, 4855, 13629, 8349, 46137}},
	{6307, 17, 4898, {1, 1, 3, 13, 3, 37, 73, 69, 281, 109, 563, 1427, 5127, 8957, 16749, 41489, 49531}},
	{6308, 17, 4907, {1, 1, 7, 11, 29, 63, 79, 127, 95, 809, 1175, 1567, 6353, 7505, 26551, 5073, 53733}},
	{6309, 17, 4910, {1, 1, 1, 5, 25, 41, 59, 103, 59, 365, 1111, 3909, 3749, 14889, 3639, 10435, 45407}},
	{6310, 17, 4918, {1, 1, 1, 5, 3, 61, 93, 199, 97, 779, 67, 241, 6197, 6785, 16869, 7573, 46745}},
	{6311, 17, 4924, {1, 1, 5, 9, 27, 29, 21, 69, 165, 661, 1245, 1265, 2979, 9685, 17781, 23329, 48029}},
	{6312, 17, 4953, {1, 1, 1, 7, 7, 23, 39, 197, 169, 561, 499, 2197, 4371, 157, 6837, 44635, 94861}},
	{6313, 17, 4956, {1, 1, 5, 13, 7, 5, 9, 207, 321, 243, 899, 2967, 3553, 15413, 8961, 55039, 6459}},
	{6314, 17, 4965, {1, 3, 5, 3, 13, 25, 33, 145, 45, 979, 33, 2211, 7003, 11147, 11327, 55151, 30697}},
	{6315, 17, 4966, {1, 1, 3, 13, 7, 51, 25, 229, 231, 115, 1815, 3867, 1533, 15259, 8067, 64803, 87535}},
	{6316, 17, 4970, {1, 1, 3, 3, 21, 51, 101, 49, 227, 393, 1659, 955, 545, 7395, 31563, 5499, 130541}},
	{6317, 17, 4972, {1, 3, 1, 1, 21, 41, 57, 161, 269, 35, 893, 1817, 857, 7027, 973, 12529, 46659}},
	{6318, 17, 4983, {1, 1, 3, 7, 17, 35, 23, 29, 335, 725, 453, 1051, 6019, 7595, 29451, 1853, 116615}},
	{6319, 17, 4989, {1, 3, 3, 1, 3, 55, 73, 187, 213, 329, 997, 703, 5829, 7903, 1081, 33359, 119123}},
	{6320, 17, 4994, {1, 3, 3, 15, 29, 55, 15, 17, 245, 117, 1735, 767, 4457, 8803, 17621, 26925, 72487}},
	{6321, 17, 5000, {1, 3, 5, 3, 25, 7, 119, 139, 159, 199, 317, 3875, 8115, 7581, 29239, 50225, 48459}},
	{6322, 17, 5005, {1, 3, 7, 11, 11, 41, 107, 225, 395, 545, 259, 2379, 6709, 11669, 14545, 43663, 69979}},
	{6323, 17, 5014, {1, 3, 5, 13, 23, 45, 73, 137, 447, 305, 117, 2659, 7989, 233, 31991, 60495, 571}},
	{6324, 17, 5018, {1, 3, 7, 9, 1, 37, 31, 1, 433, 701, 159, 3811, 4529, 6697, 7121, 31107, 61555}},
	{6325, 17, 5023, {1, 3, 5, 5, 13, 21, 81, 63, 95, 741, 1189, 1567, 1223, 12371, 28435, 10537, 53785}},
	{6326, 17, 5039, {1, 1, 1, 11, 17, 31, 67, 121, 281, 593, 561, 1759, 387, 9639, 28595, 22473, 4935}},
	{6327, 17, 5053, {1, 3, 7, 3, 5, 43, 59, 151, 351, 263, 297, 423, 1681, 3785, 15171, 7145, 57531}},
	{6328, 17, 5054, {1, 3, 7, 15, 9, 35, 105, 189, 261, 175, 1669, 1289, 5401, 12801, 19585, 48169, 93195}},
	{6329, 17, 5061, {1, 1, 7, 1, 31, 41, 23, 237, 151, 549, 1079, 2933, 5509, 15593, 1791, 15757, 44607}},
	{6330, 17, 5065, {1, 1, 1, 3, 29, 1, 59, 115, 13, 999, 1179, 3561, 2749, 10059, 12861, 6797, 11793}},
	{6331, 17, 5080, {1, 3, 3, 7, 11, 5, 23, 217, 101, 775, 1497, 4047, 2427, 5117, 9683, 28895, 27557}},
	{6332, 17, 5083, {1, 3, 7, 5, 31, 55, 99, 65, 55, 587, 1271, 2277, 7947, 12995, 13149, 4463, 37625}},
	{6333, 17, 5107, {1, 1, 7, 11, 3, 63, 23, 191, 125, 365, 1153, 2657, 6763, 4557, 21643, 26885, 36753}},
	{6334, 17, 5119, {1, 1, 1, 15, 25, 15, 111, 135, 507, 745, 1947, 2545, 4329, 14325, 8187, 52021, 63401}},
	{6335, 17, 5146, {1, 1, 3, 3, 27, 25, 19, 211, 393, 467, 1015, 2495, 7135, 495, 10385, 26961, 49325}},
	{6336, 17, 5151, {1, 1, 3, 5, 15, 35, 3, 203, 337, 337, 703, 1989, 6869, 6055, 21095, 4749, 125669}},
	{6337, 17, 5152, {1, 1, 5, 1, 31, 39, 57, 101, 419, 717, 1489, 199, 5729, 3003, 2607, 64593, 11515}},
	{6338, 17, 5155, {1, 3, 7, 13, 15, 3, 33, 61, 17, 433, 1097, 957, 5351, 3043, 3679, 44881, 126909}},
	{6339, 17, 5169, {1, 1, 3, 11, 5, 1, 121, 175, 119, 367, 399, 2527, 2157, 2667, 31069, 24797, 119621}},
	{6340, 17, 5170, {1, 3, 1, 7, 27, 47, 115, 229, 455, 775, 73, 837, 1181, 3457, 4057, 33907, 67151}},
	{6341, 17, 5176, {1, 3, 3, 1, 7, 51, 71, 177, 463, 921, 393, 3137, 1225, 5709, 303, 20597, 77581}},
	{6342, 17, 5179, {1, 3, 5, 3, 31, 1, 93, 53, 177, 433, 1471, 2191, 4471, 9211, 19397, 57727, 60367}},
	{6343, 17, 5182, {1, 1, 3, 11, 29, 55, 121, 89, 67, 869, 1631, 2657, 7357, 7159, 22449, 16357, 20077}},
	{6344, 17, 5189, {1, 3, 7, 15, 11, 39, 127, 63, 211, 359, 971, 1221, 1909, 9963, 7827, 60923, 98495}},
	{6345, 17, 5193, {1, 1, 7, 9, 23, 47, 47, 85, 307, 471, 1287, 3825, 5451, 15151, 15647, 63043, 92443}},
	{6346, 17, 5196, {1, 3, 7, 5, 19, 11, 11, 27, 307, 695, 99, 1037, 1997, 13673, 591, 8183, 82197}},
	{6347, 17, 5204, {1, 3, 5, 5, 3, 53, 109, 227, 503, 855, 1269, 3903, 5049, 10647, 21751, 58707, 78311}},
	{6348, 17, 5207, {1, 1, 3, 11, 31, 3, 51, 211, 285, 919, 487, 3393, 3463, 2271, 8053, 56791, 33763}},
	{6349, 17, 5211, {1, 3, 3, 5, 21, 15, 5, 5, 327, 809, 915, 1365, 7323, 4247, 31603, 26261, 80389}},
	{6350, 17, 5220, {1, 3, 7, 7, 15, 33, 31, 221, 291, 815, 1307, 929, 3249, 14573, 13613, 59509, 59741}},
	{6351, 17, 5258, {1, 3, 7, 15, 19, 41, 61, 27, 353, 965, 1901, 87, 2669, 12757, 29723, 47165, 16521}},
	{6352, 17, 5265, {1, 3, 5, 3, 11, 43, 97, 215, 361, 901, 1425, 4063, 5327, 14119, 457, 43145, 107401}},
	{6353, 17, 5271, {1, 1, 3, 15, 19, 37, 101, 69, 131, 927, 897, 477, 7641, 4299, 21213, 26017, 123801}},
	{6354, 17, 5277, {1, 3, 7, 7, 19, 5, 11, 51, 277, 985, 1071, 3437, 6595, 9547, 11855, 64249, 30957}},
	{6355, 17, 5278, {1, 1, 7, 9, 21, 41, 89, 113, 61, 235, 685, 1419, 7619, 9863, 21221, 28685, 53409}},
	{6356, 17, 5282, {1, 1, 1, 1, 27, 1, 19, 3, 473, 827, 269, 1659, 2621, 12347, 13359, 64687, 99293}},
	{6357, 17, 5296, {1, 3, 7, 7, 29, 37, 61, 49, 215, 883, 625, 2671, 3743, 4517, 2075, 64865, 58611}},
	{6358, 17, 5299, {1, 3, 3, 7, 15, 11, 35, 37, 255, 781, 613, 3587, 7643, 13081, 32467, 14427, 15235}},
	{6359, 17, 5319, {1, 1, 1, 11, 31, 47, 107, 65, 489, 377, 425, 3453, 2901, 9999, 7687, 13311, 103947}},
	{6360, 17, 5328, {1, 3, 3, 7, 9, 17, 7, 107, 33, 545, 407, 3335, 7563, 14315, 32725, 8483, 69093}},
	{6361, 17, 5343, {1, 1, 1, 5, 17, 9, 87, 229, 417, 769, 423, 569, 7073, 8705, 24487, 63743, 69807}},
	{6362, 17, 5353, {1, 3, 1, 9, 1, 29, 75, 25, 483, 259, 1941, 1533, 8147, 14127, 24087, 37475, 130961}},
	{6363, 17, 5364, {1, 3, 3, 11, 15, 15, 51, 45, 215, 283, 1687, 185, 4521, 12205, 13041, 33283, 77007}},
	{6364, 17, 5368, {1, 1, 3, 3, 5, 47, 107, 67, 325, 87, 1831, 2845, 1645, 1741, 10811, 8983, 58515}},
	{6365, 17, 5379, {1, 3, 1, 13, 19, 17, 1, 151, 411, 915, 1739, 3781, 4939, 15767, 25897, 7205, 17285}},
	{6366, 17, 5381, {1, 3, 5, 15, 19, 1, 125, 33, 321, 325, 639, 4013, 967, 4347, 19743, 13445, 61229}},
	{6367, 17, 5399, {1, 3, 3, 13, 13, 37, 71, 85, 51, 775, 973, 739, 4341, 15707, 12221, 24321, 48073}},
	{6368, 17, 5415, {1, 1, 7, 13, 15, 13, 9, 211, 331, 429, 1323, 3027, 1091, 13311, 289, 57789, 93261}},
	{6369, 17, 5422, {1, 1, 1, 1, 27, 7, 13, 27, 67, 573, 455, 2353, 113, 11831, 9069, 4503, 89291}},
	{6370, 17, 5441, {1, 1, 1, 7, 21, 63, 47, 39, 419, 991, 1623, 11, 3153, 12633, 9425, 65087, 44935}},
	{6371, 17, 5451, {1, 3, 1, 7, 23, 11, 15, 11, 99, 543, 1739, 3955, 5883, 12469, 7529, 14177, 1945}},
	{6372, 17, 5456, {1, 3, 1, 3, 5, 17, 31, 251, 387, 311, 725, 3827, 6835, 5065, 3141, 43441, 87955}},
	{6373, 17, 5462, {1, 1, 1, 11, 25, 7, 75, 135, 67, 589, 889, 3429, 155, 9081, 28653, 8059, 57251}},
	{6374, 17, 5490, {1, 3, 5, 15, 21, 15, 103, 149, 311, 407, 1391, 717, 1765, 14887, 14381, 37483, 29587}},
	{6375, 17, 5495, {1, 3, 5, 5, 19, 31, 93, 5, 507, 193, 1735, 3841, 7895, 9853, 10317, 14867, 49529}},
	{6376, 17, 5501, {1, 3, 7, 7, 19, 3, 99, 201, 479, 313, 693, 3435, 5453, 1157, 23127, 49005, 20167}},
	{6377, 17, 5502, {1, 3, 7, 9, 15, 21, 123, 41, 19, 281, 1837, 2589, 1003, 1993, 18345, 10039, 89325}},
	{6378, 17, 5505, {1, 3, 5, 1, 19, 21, 77, 151, 145, 951, 2017, 609, 5847, 4475, 12439, 6357, 108277}},
	{6379, 17, 5512, {1, 1, 1, 9, 17, 21, 91, 91, 111, 951, 497, 1759, 503, 12787, 25117, 24323, 96447}},
	{6380, 17, 5523, {1, 1, 3, 11, 13, 9, 73, 205, 329, 243, 1187, 829, 2821, 5563, 14391, 771, 116441}},
	{6381, 17, 5529, {1, 1, 1, 1, 11, 57, 39, 221, 41, 521, 1541, 3515, 2367, 4179, 21039, 52943, 11627}},
	{6382, 17, 5548, {1, 3, 3, 3, 23, 13, 103, 125, 67, 217, 863, 3755, 213, 12657, 31399, 3771, 54107}},
	{6383, 17, 5551, {1, 3, 3, 7, 3, 9, 107, 217, 497, 935, 519, 3041, 323, 14895, 5695, 28789, 36085}},
	{6384, 17, 5553, {1, 1, 5, 11, 23, 33, 81, 23, 167, 3, 1683, 2279, 5365, 847, 14717, 9689, 64481}},
	{6385, 17, 5565, {1, 3, 1, 7, 1, 15, 107, 93, 429, 363, 1745, 1459, 5879, 8351, 17527, 44001, 70293}},
	{6386, 17, 5568, {1, 3, 3, 9, 27, 55, 125, 211, 141, 827, 1239, 663, 4803, 11067, 32039, 28091, 56421}},
	{6387, 17, 5577, {1, 3, 5, 5, 7, 13, 125, 231, 427, 483, 967, 549, 3105, 13919, 3017, 39207, 23253}},
	{6388, 17, 5578, {1, 3, 7, 3, 21, 29, 79, 67, 39, 451, 157, 337, 3585, 3621, 9545, 31205, 63201}},
	{6389, 17, 5583, {1, 3, 1, 1, 29, 25, 77, 57, 167, 899, 95, 2487, 3743, 5381, 3637, 56289, 39453}},
	{6390, 17, 5585, {1, 1, 1, 9, 29, 19, 41, 97, 75, 199, 1709, 483, 4099, 3113, 10953, 20659, 109273}},
	{6391, 17, 5588, {1, 3, 5, 15, 13, 9, 83, 43, 111, 789, 965, 4061, 1239, 14577, 10113, 26359, 52609}},
	{6392, 17, 5613, {1, 3, 5, 5, 11, 39, 113, 31, 457, 119, 725, 831, 4143, 5675, 27431, 12431, 94977}},
	{6393, 17, 5614, {1, 1, 3, 3, 25, 17, 93, 253, 307, 625, 143, 1061, 4415, 3563, 3313, 53527, 29537}},
	{6394, 17, 5616, {1, 3, 5, 3, 29, 41, 43, 109, 147, 919, 1675, 465, 6101, 12251, 28915, 15397, 85233}},
	{6395, 17, 5622, {1, 1, 1, 1, 31, 25, 59, 187, 439, 561, 559, 413, 1917, 9319, 27475, 49715, 32953}},
	{6396, 17, 5631, {1, 1, 7, 13, 23, 31, 95, 231, 141, 207, 1373, 2173, 2905, 169, 23825, 55071, 6147}},
	{6397, 17, 5637, {1, 1, 7, 13, 15, 39, 43, 117, 321, 297, 661, 2941, 7359, 11675, 15483, 24093, 7269}},
	{6398, 17, 5638, {1, 3, 3, 13, 9, 59, 51, 49, 81, 563, 745, 1843, 295, 4689, 19847, 42137, 63197}},
	{6399, 17, 5668, {1, 3, 1, 9, 5, 33, 21, 199, 509, 927, 1777, 1349, 3593, 1065, 24943, 55667, 73539}},
	{6400, 17, 5675, {1, 3, 1, 11, 17, 15, 91, 21, 59, 587, 1207, 543, 6669, 10861, 24755, 1789, 91249}},
	{6401, 17, 5683, {1, 3, 7, 15, 13, 47, 57, 147, 381, 1021, 921, 1347, 3847, 5969, 9075, 39081, 127241}},
	{6402, 17, 5695, {1, 3, 3, 15, 19, 15, 1, 97, 203, 409, 1745, 1217, 2199, 7945, 24361, 41771, 123127}},
	{6403, 17, 5703, {1, 3, 3, 5, 17, 17, 43, 255, 179, 717, 1993, 645, 6527, 1533, 32719, 27481, 122425}},
	{6404, 17, 5710, {1, 3, 5, 9, 13, 59, 15, 157, 373, 937, 27, 3325, 2297, 89, 10861, 48615, 16083}},
	{6405, 17, 5715, {1, 3, 1, 3, 19, 27, 109, 243, 189, 17, 99, 1879, 695, 11329, 12467, 6053, 41749}},
	{6406, 17, 5727, {1, 1, 5, 5, 23, 41, 103, 69, 171, 917, 1303, 2101, 617, 10017, 26525, 11009, 66137}},
	{6407, 17, 5738, {1, 1, 1, 9, 21, 45, 47, 171, 455, 257, 411, 4021, 6995, 12881, 4793, 51193, 60775}},
	{6408, 17, 5752, {1, 3, 7, 5, 25, 31, 89, 53, 321, 593, 1795, 2435, 3833, 2767, 17241, 63373, 25457}},
	{6409, 17, 5767, {1, 3, 1, 1, 3, 45, 19, 255, 179, 991, 1407, 3683, 1435, 6803, 12215, 12835, 2005}},
	{6410, 17, 5773, {1, 3, 7, 3, 17, 5, 117, 251, 71, 983, 1391, 3499, 5119, 7257, 7325, 16565, 6321}},
	{6411, 17, 5776, {1, 3, 5, 7, 5, 49, 47, 201, 297, 485, 1879, 2205, 4903, 13619, 22537, 5479, 121625}},
	{6412, 17, 5781, {1, 1, 3, 5, 27, 27, 87, 61, 145, 943, 343, 1639, 6307, 4549, 20765, 33479, 113697}},
	{6413, 17, 5791, {1, 1, 3, 9, 17, 5, 101, 129, 305, 653, 1901, 3901, 6361, 2369, 7449, 55259, 75215}},
	{6414, 17, 5792, {1, 1, 7, 5, 31, 45, 117, 55, 335, 827, 1309, 2603, 2111, 11005, 14747, 56999, 97373}},
	{6415, 17, 5795, {1, 1, 7, 11, 29, 29, 81, 175, 169, 453, 293, 2589, 1057, 15795, 32397, 65433, 79455}},
	{6416, 17, 5798, {1, 1, 1, 5, 11, 7, 13, 249, 29, 407, 1289, 2385, 8113, 15327, 4029, 32005, 105901}},
	{6417, 17, 5801, {1, 1, 5, 5, 7, 61, 103, 141, 109, 391, 631, 821, 1479, 14771, 25057, 1415, 8081}},
	{6418, 17, 5810, {1, 3, 1, 1, 9, 37, 17, 231, 501, 745, 1695, 45, 7797, 2945, 5529, 34747, 39069}},
	{6419, 17, 5812, {1, 1, 7, 9, 21, 59, 103, 103, 33, 875, 723, 3477, 4729, 7311, 29979, 60901, 72187}},
	{6420, 17, 5836, {1, 3, 3, 3, 15, 63, 93, 237, 203, 635, 1189, 2035, 6499, 9943, 9133, 62977, 29657}},
	{6421, 17, 5839, {1, 1, 1, 9, 3, 11, 63, 207, 95, 563, 775, 3009, 7125, 13141, 4489, 16343, 120951}},
	{6422, 17, 5841, {1, 1, 3, 1, 21, 57, 15, 217, 185, 305, 463, 1597, 6529, 4989, 14011, 11265, 131031}},
	{6423, 17, 5867, {1, 3, 5, 15, 17, 61, 35, 127, 411, 579, 1349, 615, 3293, 8475, 9773, 30635, 117639}},
	{6424, 17, 5870, {1, 1, 7, 9, 11, 3, 55, 105, 305, 223, 1899, 2217, 1261, 9831, 23693, 3013, 30489}},
	{6425, 17, 5877, {1, 3, 7, 15, 15, 29, 1, 99, 67, 293, 499, 1941, 5303, 1329, 24547, 14065, 7927}},
	{6426, 17, 5881, {1, 1, 5, 11, 17, 55, 71, 49, 499, 435, 985, 2803, 6139, 1503, 24167, 47181, 102529}},
	{6427, 17, 5899, {1, 3, 5, 1, 19, 53, 71, 17, 63, 469, 1871, 2051, 357, 11661, 5689, 36373, 13379}},
	{6428, 17, 5914, {1, 1, 5, 1, 27, 47, 23, 247, 59, 381, 1895, 2453, 3665, 5487, 24081, 50501, 91659}},
	{6429, 17, 5925, {1, 1, 5, 7, 29, 19, 3, 33, 83, 301, 133, 3603, 5133, 16171, 22905, 36271, 10405}},
	{6430, 17, 5929, {1, 3, 7, 9, 11, 23, 57, 87, 9, 731, 631, 3703, 2593, 12851, 7115, 8801, 108919}},
	{6431, 17, 5943, {1, 3, 3, 3, 23, 35, 33, 99, 343, 837, 231, 3921, 6975, 15093, 15049, 64623, 123523}},
	{6432, 17, 5949, {1, 1, 7, 11, 15, 61, 113, 103, 501, 57, 1345, 3155, 2965, 4433, 10605, 43765, 42169}},
	{6433, 17, 5962, {1, 1, 7, 13, 7, 53, 91, 121, 229, 127, 103, 833, 7829, 1571, 10847, 20861, 101155}},
	{6434, 17, 5969, {1, 3, 7, 1, 9, 25, 71, 103, 37, 473, 1133, 1129, 1651, 6965, 6937, 16597, 20439}},
	{6435, 17, 5976, {1, 1, 5, 9, 1, 9, 47, 131, 285, 967, 1869, 1075, 8127, 135, 15575, 38569, 123729}},
	{6436, 17, 5988, {1, 1, 7, 9, 5, 31, 33, 227, 347, 41, 2025, 3755, 857, 7805, 13121, 38307, 125825}},
	{6437, 17, 5997, {1, 3, 5, 7, 11, 11, 19, 55, 23, 627, 1477, 3093, 2779, 7653, 7165, 23053, 76123}},
	{6438, 17, 6006, {1, 1, 3, 1, 3, 47, 83, 89, 177, 381, 1247, 141, 7051, 6443, 27369, 34323, 43063}},
	{6439, 17, 6010, {1, 1, 7, 7, 13, 15, 55, 223, 351, 525, 1051, 3009, 5443, 11499, 8335, 37949, 69149}},
	{6440, 17, 6016, {1, 1, 1, 3, 13, 61, 89, 33, 129, 921, 1905, 201, 3141, 5531, 135, 34103, 56883}},
	{6441, 17, 6022, {1, 1, 5, 13, 17, 27, 13, 163, 169, 471, 1263, 1421, 7015, 7927, 21027, 58001, 26739}},
	{6442, 17, 6026, {1, 1, 1, 15, 19, 49, 109, 207, 245, 49, 1271, 3635, 2561, 5091, 24415, 59195, 67701}},
	{6443, 17, 6031, {1, 3, 5, 7, 27, 57, 99, 155, 461, 595, 1859, 1727, 857, 4993, 31733, 42141, 10035}},
	{6444, 17, 6040, {1, 1, 1, 15, 11, 11, 85, 9, 251, 375, 155, 379, 7501, 12559, 32583, 36317, 4675}},
	{6445, 17, 6043, {1, 1, 5, 13, 19, 57, 81, 69, 201, 293, 593, 3169, 4519, 9057, 16685, 12847, 123797}},
	{6446, 17, 6050, {1, 3, 1, 5, 5, 1, 19, 243, 345, 661, 561, 3549, 2541, 5887, 25879, 41467, 72799}},
	{6447, 17, 6059, {1, 1, 5, 13, 15, 51, 67, 61, 79, 89, 447, 1471, 4915, 10637, 10901, 48157, 103545}},
	{6448, 17, 6079, {1, 3, 5, 13, 31, 25, 73, 129, 435, 659, 1851, 3595, 753, 7717, 10927, 30115, 109221}},
	{6449, 17, 6099, {1, 1, 1, 3, 25, 3, 121, 43, 349, 205, 1209, 2671, 6445, 8755, 7171, 58631, 74319}},
	{6450, 17, 6101, {1, 1, 3, 1, 11, 15, 83, 37, 483, 65, 759, 1835, 3883, 1693, 30051, 61077, 1187}},
	{6451, 17, 6105, {1, 3, 7, 15, 29, 23, 85, 77, 139, 903, 1821, 943, 6453, 1523, 18539, 49039, 110787}},
	{6452, 17, 6108, {1, 1, 7, 15, 15, 17, 69, 253, 507, 921, 523, 79, 747, 4011, 25795, 42029, 88309}},
	{6453, 17, 6124, {1, 1, 7, 3, 25, 47, 119, 83, 313, 45, 985, 145, 205, 3407, 9013, 64517, 115811}},
	{6454, 17, 6132, {1, 1, 7, 1, 29, 21, 9, 123, 97, 545, 1987, 2979, 6901, 12667, 23325, 63635, 70593}},
	{6455, 17, 6145, {1, 3, 7, 3, 23, 45, 81, 255, 41, 29, 1493, 4065, 3201, 10479, 17193, 39999, 55493}},
	{6456, 17, 6146, {1, 3, 1, 3, 9, 43, 43, 135, 235, 603, 481, 3139, 2729, 14759, 7269, 7995, 110351}},
	{6457, 17, 6151, {1, 3, 1, 11, 17, 35, 113, 93, 417, 967, 755, 659, 3115, 16163, 22997, 38205, 126961}},
	{6458, 17, 6152, {1, 1, 7, 11, 29, 57, 81, 235, 93, 869, 475, 825, 6269, 15819, 14977, 53057, 116021}},
	{6459, 17, 6158, {1, 1, 7, 13, 5, 61, 5, 241, 245, 673, 1651, 3367, 2355, 713, 20107, 30133, 735}},
	{6460, 17, 6160, {1, 1, 5, 9, 21, 3, 121, 241, 129, 703, 1435, 1943, 5087, 13123, 30023, 58287, 50377}},
	{6461, 17, 6163, {1, 1, 1, 15, 23, 27, 67, 197, 123, 629, 169, 3303, 1679, 11051, 16875, 28055, 12379}},
	{6462, 17, 6165, {1, 1, 3, 3, 7, 63, 97, 43, 89, 739, 779, 2893, 7763, 6351, 26135, 44647, 127987}},
	{6463, 17, 6170, {1, 3, 3, 9, 31, 59, 95, 131, 131, 321, 1125, 127, 4865, 145, 26237, 47871, 114549}},
	{6464, 17, 6182, {1, 3, 3, 13, 21, 3, 33, 17, 445, 693, 1599, 2517, 1679, 2237, 15053, 30983, 106755}},
	{6465, 17, 6196, {1, 1, 5, 13, 31, 37, 49, 67, 403, 27, 575, 1795, 3385, 1067, 585, 60277, 123189}},
	{6466, 17, 6199, {1, 3, 1, 15, 13, 35, 23, 247, 493, 305, 363, 451, 4011, 3679, 18281, 31751, 127933}},
	{6467, 17, 6200, {1, 1, 7, 5, 21, 45, 123, 253, 469, 267, 985, 2349, 3427, 7653, 25685, 13747, 531}},
	{6468, 17, 6205, {1, 1, 5, 11, 7, 59, 105, 209, 27, 847, 593, 3775, 6165, 1655, 29867, 28465, 92193}},
	{6469, 17, 6226, {1, 3, 1, 11, 7, 25, 101, 81, 233, 311, 9, 2735, 3951, 485, 10105, 24489, 649}},
	{6470, 17, 6228, {1, 3, 1, 7, 27, 5, 115, 243, 295, 659, 215, 1787, 5131, 2513, 29201, 21195, 103383}},
	{6471, 17, 6237, {1, 3, 5, 13, 29, 21, 7, 57, 345, 467, 1297, 207, 5115, 335, 6153, 32959, 125697}},
	{6472, 17, 6247, {1, 1, 1, 9, 3, 63, 63, 5, 373, 123, 1265, 2365, 1623, 1561, 14805, 17487, 104787}},
	{6473, 17, 6251, {1, 3, 1, 5, 15, 13, 55, 69, 251, 341, 463, 2611, 4793, 12157, 4669, 11613, 128705}},
	{6474, 17, 6253, {1, 3, 7, 13, 19, 7, 93, 149, 453, 693, 1731, 861, 6971, 943, 18891, 56547, 34411}},
	{6475, 17, 6256, {1, 1, 7, 1, 27, 49, 27, 9, 281, 121, 581, 393, 2583, 1159, 26989, 39955, 100765}},
	{6476, 17, 6268, {1, 1, 3, 9, 3, 43, 97, 207, 311, 617, 1987, 2559, 2101, 15791, 30085, 40713, 41909}},
	{6477, 17, 6272, {1, 3, 1, 3, 15, 19, 53, 183, 375, 867, 397, 3203, 4207, 5381, 25065, 60357, 88739}},
	{6478, 17, 6275, {1, 3, 3, 3, 27, 51, 85, 231, 19, 559, 567, 4049, 4875, 14201, 11623, 39763, 57339}},
	{6479, 17, 6281, {1, 1, 5, 1, 19, 7, 81, 249, 41, 789, 985, 3725, 4053, 4255, 9861, 1609, 29511}},
	{6480, 17, 6289, {1, 3, 5, 5, 21, 13, 49, 41, 367, 283, 1161, 2753, 4733, 3691, 27931, 53055, 83625}},
	{6481, 17, 6335, {1, 3, 5, 11, 29, 47, 95, 51, 265, 85, 385, 833, 7957, 14985, 7017, 41937, 41377}},
	{6482, 17, 6338, {1, 1, 7, 5, 1, 23, 17, 191, 185, 323, 515, 3183, 7685, 7361, 21143, 5227, 110297}},
	{6483, 17, 6355, {1, 3, 3, 7, 11, 39, 31, 97, 237, 497, 1649, 3529, 6153, 5055, 29021, 35125, 121581}},
	{6484, 17, 6362, {1, 3, 5, 3, 17, 47, 105, 75, 55, 343, 595, 2447, 5575, 10673, 32015, 37541, 127867}},
	{6485, 17, 6373, {1, 3, 1, 7, 19, 39, 31, 135, 167, 979, 219, 1353, 489, 9667, 27107, 55565, 72291}},
	{6486, 17, 6386, {1, 1, 3, 13, 31, 49, 87, 93, 235, 577, 1551, 2663, 387, 1129, 26683, 31285, 15913}},
	{6487, 17, 6388, {1, 3, 3, 7, 15, 29, 61, 33, 115, 511, 1781, 2029, 4265, 6745, 1467, 34415, 40907}},
	{6488, 17, 6391, {1, 1, 7, 5, 1, 55, 13, 129, 167, 937, 79, 2047, 3589, 1979, 4153, 15229, 85745}},
	{6489, 17, 6397, {1, 1, 7, 15, 15, 25, 89, 129, 31, 435, 1359, 49, 2659, 2829, 8741, 25215, 4239}},
	{6490, 17, 6405, {1, 3, 5, 3, 11, 39, 95, 239, 187, 615, 1481, 3509, 1133, 13497, 24833, 59635, 45695}},
	{6491, 17, 6406, {1, 1, 5, 3, 19, 17, 17, 235, 315, 943, 883, 1381, 7129, 15709, 9847, 41183, 116071}},
	{6492, 17, 6410, {1, 1, 1, 3, 9, 63, 109, 209, 309, 1015, 1391, 2617, 1481, 6483, 4151, 28063, 49887}},
	{6493, 17, 6417, {1, 1, 5, 13, 23, 37, 31, 89, 501, 461, 41, 931, 7863, 15499, 25635, 16995, 41651}},
	{6494, 17, 6443, {1, 1, 1, 9, 29, 29, 125, 161, 219, 439, 1465, 1615, 7483, 7497, 1121, 49693, 30269}},
	{6495, 17, 6457, {1, 3, 1, 5, 7, 43, 27, 161, 431, 375, 419, 2995, 527, 8207, 747, 18491, 15351}},
	{6496, 17, 6468, {1, 1, 3, 13, 25, 21, 67, 177, 9, 453, 1171, 65, 2845, 16147, 12699, 30905, 122255}},
	{6497, 17, 6475, {1, 3, 1, 5, 29, 47, 77, 251, 473, 385, 947, 3239, 5375, 13617, 10639, 36005, 95821}},
	{6498, 17, 6486, {1, 3, 1, 15, 13, 1, 75, 223, 509, 19, 175, 1541, 637, 5711, 1097, 44901, 35277}},
	{6499, 17, 6489, {1, 3, 3, 7, 3, 27, 17, 151, 39, 583, 391, 2739, 7339, 2051, 17005, 49573, 85969}},
	{6500, 17, 6495, {1, 3, 1, 11, 3, 25, 119, 125, 17, 629, 201, 2347, 2923, 1273, 14871, 58299, 97667}},
	{6501, 17, 6499, {1, 1, 7, 1, 31, 39, 11, 121, 339, 667, 1863, 3479, 1895, 11319, 5683, 64969, 9261}},
	{6502, 17, 6505, {1, 1, 5, 9, 27, 61, 101, 221, 221, 583, 287, 707, 5931, 4225, 29537, 46097, 114361}},
	{6503, 17, 6511, {1, 1, 1, 9, 23, 47, 1, 35, 85, 1021, 151, 3153, 3867, 971, 31573, 4745, 107639}},
	{6504, 17, 6520, {1, 1, 7, 13, 15, 15, 63, 37, 291, 907, 411, 1571, 6415, 7443, 26635, 27945, 130909}},
	{6505, 17, 6529, {1, 3, 1, 9, 21, 13, 77, 147, 485, 107, 235, 481, 2389, 957, 11493, 53033, 46373}},
	{6506, 17, 6542, {1, 3, 5, 7, 3, 55, 125, 237, 205, 411, 1911, 4053, 5983, 15489, 29333, 44727, 62167}},
	{6507, 17, 6547, {1, 1, 3, 3, 17, 3, 59, 239, 209, 495, 447, 3427, 3425, 2347, 10057, 26147, 52243}},
	{6508, 17, 6550, {1, 1, 3, 1, 11, 31, 3, 139, 441, 997, 295, 1267, 2181, 6047, 32419, 62657, 24921}},
	{6509, 17, 6554, {1, 3, 7, 15, 5, 3, 11, 9, 211, 701, 1987, 2611, 6195, 14379, 22919, 15785, 52149}},
	{6510, 17, 6556, {1, 1, 7, 9, 7, 27, 35, 253, 343, 679, 103, 1217, 3983, 8677, 17671, 41347, 89355}},
	{6511, 17, 6560, {1, 1, 1, 5, 7, 55, 111, 115, 231, 999, 773, 2111, 3617, 2469, 16967, 60735, 24557}},
	{6512, 17, 6569, {1, 3, 5, 1, 29, 5, 77, 217, 131, 307, 473, 3595, 2713, 6503, 18459, 57245, 91897}},
	{6513, 17, 6572, {1, 3, 5, 13, 9, 33, 93, 31, 59, 343, 1337, 1971, 7593, 15629, 22693, 19885, 4139}},
	{6514, 17, 6590, {1, 3, 3, 3, 21, 33, 115, 205, 373, 587, 739, 669, 8065, 5339, 16507, 29455, 15863}},
	{6515, 17, 6592, {1, 3, 5, 11, 9, 43, 45, 41, 91, 87, 19, 1523, 5059, 9403, 6739, 36893, 6395}},
	{6516, 17, 6601, {1, 1, 5, 15, 19, 43, 81, 3, 401, 621, 1839, 1443, 179, 8085, 27021, 7757, 95011}},
	{6517, 17, 6610, {1, 3, 5, 15, 19, 21, 45, 167, 77, 977, 309, 431, 3437, 8327, 12895, 50521, 68473}},
	{6518, 17, 6632, {1, 3, 3, 15, 7, 21, 49, 169, 327, 271, 7, 785, 1767, 14747, 7083, 65223, 24213}},
	{6519, 17, 6635, {1, 1, 5, 9, 9, 51, 101, 197, 507, 839, 1413, 3131, 331, 15725, 32293, 60433, 86759}},
	{6520, 17, 6640, {1, 1, 7, 1, 17, 39, 127, 201, 341, 607, 1565, 1615, 1367, 16043, 28265, 29139, 63813}},
	{6521, 17, 6643, {1, 3, 5, 7, 9, 1, 107, 73, 121, 649, 1385, 3203, 2897, 8479, 28519, 34041, 1359}},
	{6522, 17, 6649, {1, 1, 7, 7, 21, 55, 19, 13, 415, 647, 2015, 107, 4167, 5033, 16849, 41407, 94387}},
	{6523, 17, 6659, {1, 3, 5, 13, 31, 27, 107, 95, 425, 679, 55, 3521, 6737, 11459, 19995, 64189, 44323}},
	{6524, 17, 6662, {1, 1, 3, 9, 17, 47, 29, 167, 17, 63, 5, 2505, 6483, 14089, 7127, 7907, 68555}},
	{6525, 17, 6666, {1, 1, 5, 3, 29, 3, 87, 107, 227, 893, 1821, 341, 5481, 13317, 10637, 8611, 28625}},
	{6526, 17, 6690, {1, 1, 1, 13, 11, 19, 59, 157, 397, 103, 1821, 3913, 3083, 6053, 1015, 25475, 94813}},
	{6527, 17, 6692, {1, 3, 1, 3, 15, 45, 1, 209, 335, 1015, 539, 2959, 1711, 2567, 30169, 147, 25383}},
	{6528, 17, 6704, {1, 3, 7, 1, 17, 5, 99, 121, 91, 531, 865, 1667, 5615, 4729, 7473, 21445, 37925}},
	{6529, 17, 6713, {1, 1, 7, 13, 3, 51, 27, 115, 439, 761, 1121, 1503, 3047, 2127, 29253, 48147, 10813}},
	{6530, 17, 6728, {1, 3, 7, 15, 1, 51, 33, 161, 509, 159, 1705, 3365, 7953, 14027, 3873, 29609, 33101}},
	{6531, 17, 6731, {1, 1, 5, 15, 15, 53, 119, 115, 433, 75, 497, 1259, 1681, 7715, 24767, 34647, 82007}},
	{6532, 17, 6734, {1, 1, 5, 3, 27, 63, 41, 181, 393, 439, 95, 2765, 7617, 817, 1311, 18595, 16921}},
	{6533, 17, 6746, {1, 3, 1, 15, 31, 7, 57, 89, 371, 745, 475, 3211, 6893, 10681, 18547, 28373, 127787}},
	{6534, 17, 6755, {1, 3, 5, 13, 5, 55, 45, 167, 147, 833, 765, 1153, 4037, 8503, 10751, 49541, 77489}},
	{6535, 17, 6757, {1, 3, 1, 11, 11, 7, 45, 167, 431, 759, 1035, 1367, 1649, 11711, 4915, 58915, 72479}},
	{6536, 17, 6764, {1, 1, 5, 1, 11, 3, 15, 135, 427, 637, 879, 1667, 6139, 14759, 25665, 13083, 67961}},
	{6537, 17, 6772, {1, 3, 3, 9, 1, 3, 73, 167, 269, 51, 1481, 3659, 7863, 7187, 3951, 10711, 5909}},
	{6538, 17, 6792, {1, 1, 3, 3, 9, 53, 101, 209, 109, 691, 1641, 919, 1083, 6247, 23041, 44681, 130105}},
	{6539, 17, 6797, {1, 3, 7, 5, 21, 55, 127, 9, 437, 225, 1599, 2575, 5407, 8099, 20009, 40339, 110581}},
	{6540, 17, 6821, {1, 3, 3, 13, 7, 41, 15, 137, 363, 337, 995, 1215, 3651, 11011, 27209, 53927, 78065}},
	{6541, 17, 6822, {1, 1, 1, 7, 11, 17, 27, 9, 481, 79, 905, 1297, 811, 10221, 463, 12979, 114731}},
	{6542, 17, 6831, {1, 1, 3, 13, 7, 59, 105, 79, 253, 699, 139, 3823, 4939, 12955, 32069, 7255, 18159}},
	{6543, 17, 6834, {1, 3, 5, 7, 29, 7, 79, 79, 147, 921, 425, 1423, 5967, 6397, 17393, 30009, 84075}},
	{6544, 17, 6851, {1, 3, 7, 13, 23, 45, 51, 141, 237, 443, 1101, 309, 4533, 7479, 22415, 31517, 120407}},
	{6545, 17, 6858, {1, 1, 5, 13, 3, 19, 97, 185, 59, 179, 1343, 2537, 3165, 16295, 25005, 49769, 78007}},
	{6546, 17, 6860, {1, 3, 7, 15, 11, 53, 127, 195, 309, 121, 1741, 1415, 225, 15645, 16365, 38729, 70061}},
	{6547, 17, 6871, {1, 3, 7, 11, 29, 35, 47, 109, 179, 3, 849, 2305, 3035, 15289, 31569, 28851, 90057}},
	{6548, 17, 6875, {1, 1, 7, 1, 13, 27, 93, 119, 439, 45, 623, 1263, 6595, 6669, 12981, 64721, 130109}},
	{6549, 17, 6884, {1, 1, 7, 13, 5, 43, 43, 99, 395, 417, 795, 3991, 5601, 13115, 12803, 52247, 39245}},
	{6550, 17, 6888, {1, 3, 3, 3, 15, 61, 85, 91, 407, 391, 359, 3885, 1925, 4873, 169, 41727, 129471}},
	{6551, 17, 6894, {1, 3, 3, 9, 11, 47, 3, 33, 355, 853, 1329, 1347, 1995, 8197, 10015, 787, 66773}},
	{6552, 17, 6919, {1, 3, 3, 13, 31, 31, 49, 195, 55, 185, 1743, 3523, 1781, 8469, 7623, 55933, 74953}},
	{6553, 17, 6940, {1, 3, 5, 15, 29, 31, 5, 45, 149, 71, 2033, 3171, 4601, 9941, 15005, 55709, 74403}},
	{6554, 17, 6950, {1, 3, 5, 3, 1, 27, 105, 7, 139, 805, 1877, 915, 1843, 11897, 29485, 19275, 44711}},
	{6555, 17, 6959, {1, 1, 5, 7, 25, 57, 111, 57, 401, 935, 1685, 2985, 2015, 13501, 14581, 53579, 117011}},
	{6556, 17, 6968, {1, 1, 5, 11, 13, 47, 63, 137, 145, 77, 1727, 2629, 7377, 6311, 537, 13703, 129503}},
	{6557, 17, 6981, {1, 1, 7, 9, 5, 49, 67, 51, 163, 989, 845, 7, 2141, 14467, 3197, 57581, 121087}},
	{6558, 17, 6988, {1, 1, 5, 3, 31, 49, 57, 103, 171, 491, 1109, 1255, 4353, 11927, 29525, 16685, 48469}},
	{6559, 17, 6996, {1, 1, 1, 3, 7, 29, 17, 111, 339, 747, 763, 179, 7747, 2483, 18415, 45301, 25155}},
	{6560, 17, 6999, {1, 1, 7, 7, 1, 41, 71, 109, 401, 815, 1311, 3933, 1349, 13327, 20847, 44391, 49721}},
	{6561, 17, 7015, {1, 1, 1, 15, 27, 57, 39, 129, 391, 701, 619, 3925, 701, 403, 11821, 30517, 22035}},
	{6562, 17, 7019, {1, 1, 5, 11, 21, 49, 109, 101, 497, 417, 73, 2727, 2899, 2777, 22161, 35561, 70211}},
	{6563, 17, 7022, {1, 1, 3, 3, 15, 43, 1, 159, 41, 833, 55, 2415, 5009, 9663, 31295, 29397, 3187}},
	{6564, 17, 7040, {1, 1, 3, 7, 27, 5, 113, 187, 453, 753, 1649, 1605, 2405, 11791, 4239, 20915, 54033}},
	{6565, 17, 7045, {1, 3, 1, 11, 1, 57, 49, 229, 283, 113, 345, 785, 8009, 11977, 30169, 63787, 32011}},
	{6566, 17, 7049, {1, 1, 7, 3, 5, 59, 57, 91, 327, 685, 219, 1949, 3095, 8389, 2035, 11903, 73461}},
	{6567, 17, 7055, {1, 1, 3, 3, 19, 59, 19, 37, 453, 1, 1811, 3263, 1807, 16147, 24861, 14003, 31747}},
	{6568, 17, 7073, {1, 1, 3, 11, 1, 53, 93, 203, 429, 629, 1931, 1487, 3301, 8805, 4901, 2459, 98555}},
	{6569, 17, 7076, {1, 1, 7, 5, 21, 5, 37, 135, 159, 749, 1589, 2631, 8145, 7279, 28397, 47113, 82309}},
	{6570, 17, 7085, {1, 1, 5, 15, 25, 61, 19, 51, 217, 495, 109, 1179, 2743, 12107, 12509, 13003, 94375}},
	{6571, 17, 7091, {1, 3, 3, 15, 11, 7, 67, 165, 57, 925, 427, 2549, 7189, 5917, 13113, 30933, 62703}},
	{6572, 17, 7103, {1, 1, 5, 5, 9, 5, 43, 5, 485, 159, 757, 3979, 4963, 3389, 29731, 48477, 113429}},
	{6573, 17, 7112, {1, 3, 5, 1, 5, 5, 81, 163, 493, 357, 2005, 1093, 5951, 1045, 10569, 40321, 56881}},
	{6574, 17, 7117, {1, 3, 1, 5, 7, 29, 11, 7, 7, 13, 1641, 1031, 4025, 16337, 24333, 9589, 37779}},
	{6575, 17, 7118, {1, 3, 5, 11, 15, 3, 69, 19, 141, 79, 749, 391, 4505, 6939, 3079, 3647, 22363}},
	{6576, 17, 7123, {1, 3, 3, 3, 29, 3, 7, 189, 183, 513, 1225, 239, 4203, 9197, 23507, 33089, 124433}},
	{6577, 17, 7126, {1, 3, 3, 13, 27, 37, 81, 221, 287, 891, 1197, 3501, 539, 2053, 20509, 48635, 50269}},
	{6578, 17, 7154, {1, 1, 5, 7, 13, 3, 35, 79, 3, 885, 343, 3527, 1043, 7197, 6973, 8515, 39315}},
	{6579, 17, 7180, {1, 3, 3, 9, 21, 53, 79, 225, 229, 759, 457, 293, 953, 12857, 20483, 3677, 93839}},
	{6580, 17, 7192, {1, 3, 5, 3, 5, 17, 45, 107, 153, 279, 761, 1923, 7013, 2989, 10137, 19107, 126897}},
	{6581, 17, 7195, {1, 3, 1, 3, 23, 53, 91, 1, 133, 729, 13, 2017, 6933, 7405, 1255, 49509, 105571}},
	{6582, 17, 7207, {1, 3, 5, 1, 9, 45, 35, 153, 209, 289, 1779, 2557, 315, 981, 15347, 30391, 16027}},
	{6583, 17, 7208, {1, 3, 3, 5, 17, 3, 51, 105, 263, 959, 1255, 1177, 8143, 10541, 7687, 38731, 93561}},
	{6584, 17, 7214, {1, 1, 1, 13, 19, 1, 15, 135, 447, 847, 663, 3893, 3539, 6833, 13265, 62923, 8375}},
	{6585, 17, 7222, {1, 3, 1, 15, 31, 11, 105, 1, 91, 523, 1583, 3493, 2665, 117, 10757, 29845, 127201}},
	{6586, 17, 7234, {1, 1, 1, 3, 29, 49, 9, 103, 309, 605, 1751, 1981, 833, 3653, 14001, 16545, 58513}},
	{6587, 17, 7254, {1, 1, 5, 9, 1, 19, 117, 71, 237, 765, 249, 1983, 2289, 6019, 26505, 31427, 64333}},
	{6588, 17, 7258, {1, 1, 3, 11, 15, 31, 5, 207, 347, 143, 11, 1987, 3569, 2051, 31051, 22193, 93289}},
	{6589, 17, 7264, {1, 1, 3, 5, 13, 15, 5, 73, 457, 611, 673, 2675, 8071, 13245, 19443, 14399, 99599}},
	{6590, 17, 7279, {1, 1, 1, 9, 11, 5, 103, 231, 31, 457, 1031, 2257, 3159, 8323, 31585, 26163, 45159}},
	{6591, 17, 7282, {1, 3, 1, 11, 29, 51, 29, 7, 89, 331, 783, 951, 6353, 15421, 12801, 8337, 119171}},
	{6592, 17, 7293, {1, 1, 3, 13, 23, 57, 63, 43, 505, 1, 657, 4005, 6327, 7545, 15455, 27097, 53649}},
	{6593, 17, 7297, {1, 1, 1, 5, 31, 7, 51, 107, 175, 461, 1893, 305, 157, 4819, 18549, 33087, 9499}},
	{6594, 17, 7322, {1, 3, 1, 3, 19, 45, 37, 9, 459, 143, 1327, 3611, 1899, 15109, 30151, 17911, 13233}},
	{6595, 17, 7324, {1, 1, 5, 15, 19, 49, 11, 227, 375, 661, 665, 259, 3659, 13723, 28239, 48159, 59209}},
	{6596, 17, 7351, {1, 3, 7, 7, 17, 49, 77, 161, 505, 713, 1521, 935, 3629, 5033, 26717, 47199, 3693}},
	{6597, 17, 7363, {1, 3, 5, 9, 17, 61, 1, 201, 259, 179, 1637, 2485, 4995, 2813, 19923, 43739, 32183}},
	{6598, 17, 7380, {1, 1, 3, 5, 1, 23, 125, 61, 225, 703, 2011, 1013, 6651, 14029, 27375, 23301, 80269}},
	{6599, 17, 7384, {1, 1, 3, 9, 11, 57, 37, 49, 321, 443, 1055, 1989, 4755, 8467, 22001, 18647, 112617}},
	{6600, 17, 7389, {1, 3, 1, 5, 5, 39, 21, 139, 101, 583, 1881, 2599, 4185, 15679, 22215, 19093, 76737}},
	{6601, 17, 7396, {1, 3, 1, 11, 31, 51, 85, 91, 159, 421, 2005, 1075, 7757, 12653, 25489, 3545, 62961}},
	{6602, 17, 7413, {1, 1, 1, 15, 27, 57, 75, 151, 357, 571, 395, 299, 5607, 12865, 2149, 21059, 120753}},
	{6603, 17, 7417, {1, 1, 1, 3, 15, 57, 63, 171, 265, 709, 1089, 677, 7243, 10207, 9789, 38431, 130415}},
	{6604, 17, 7431, {1, 3, 7, 5, 21, 9, 73, 149, 197, 773, 773, 3931, 4135, 5671, 2733, 57173, 90693}},
	{6605, 17, 7443, {1, 1, 5, 1, 23, 1, 47, 201, 33, 167, 1643, 4009, 2687, 5725, 28733, 27859, 55163}},
	{6606, 17, 7445, {1, 1, 5, 15, 25, 11, 57, 139, 471, 625, 1067, 3647, 6213, 15605, 23537, 5005, 32593}},
	{6607, 17, 7450, {1, 3, 1, 11, 17, 11, 25, 163, 199, 21, 1775, 3721, 2845, 15769, 2381, 27643, 19909}},
	{6608, 17, 7456, {1, 3, 5, 5, 21, 41, 23, 125, 401, 483, 535, 925, 7065, 1727, 3761, 8485, 3519}},
	{6609, 17, 7466, {1, 1, 3, 15, 27, 31, 11, 7, 93, 237, 611, 3635, 4747, 9751, 20607, 20473, 73935}},
	{6610, 17, 7468, {1, 1, 7, 3, 15, 19, 69, 169, 387, 291, 1981, 635, 3387, 15817, 20357, 47537, 107311}},
	{6611, 17, 7474, {1, 3, 7, 15, 13, 59, 31, 235, 399, 343, 1265, 2975, 6839, 13335, 5397, 58915, 31313}},
	{6612, 17, 7479, {1, 1, 7, 1, 3, 35, 81, 243, 387, 421, 1533, 799, 2615, 13219, 9041, 26967, 22677}},
	{6613, 17, 7486, {1, 1, 7, 15, 17, 41, 89, 115, 67, 569, 1647, 1831, 5533, 4629, 1413, 20037, 97343}},
	{6614, 17, 7497, {1, 1, 5, 1, 23, 41, 11, 149, 319, 377, 439, 1237, 4819, 14303, 14657, 61711, 129235}},
	{6615, 17, 7508, {1, 3, 3, 7, 9, 11, 79, 219, 249, 607, 1453, 2931, 3407, 13725, 28289, 42869, 96759}},
	{6616, 17, 7515, {1, 1, 5, 11, 7, 9, 101, 51, 11, 893, 697, 1221, 4237, 1873, 11191, 25517, 119861}},
	{6617, 17, 7533, {1, 1, 3, 11, 23, 23, 19, 245, 485, 317, 1945, 2339, 193, 9389, 30709, 33927, 95089}},
	{6618, 17, 7542, {1, 1, 3, 1, 27, 55, 5, 81, 63, 185, 223, 3639, 6093, 10053, 1793, 11885, 29307}},
	{6619, 17, 7546, {1, 1, 7, 13, 15, 41, 33, 133, 467, 457, 213, 3687, 1313, 2555, 19487, 44257, 108667}},
	{6620, 17, 7551, {1, 1, 3, 5, 31, 51, 53, 115, 449, 273, 1043, 2743, 1759, 2013, 28171, 57091, 76837}},
	{6621, 17, 7569, {1, 1, 5, 15, 21, 43, 11, 215, 151, 253, 913, 1889, 2799, 13787, 3869, 54413, 50991}},
	{6622, 17, 7572, {1, 1, 3, 13, 29, 19, 81, 123, 461, 203, 81, 555, 6601, 15689, 12637, 41467, 105343}},
	{6623, 17, 7595, {1, 1, 3, 13, 7, 21, 75, 111, 47, 481, 1519, 3299, 6199, 3501, 31323, 29215, 45607}},
	{6624, 17, 7603, {1, 3, 1, 3, 17, 51, 45, 223, 321, 233, 267, 3333, 3803, 3099, 4601, 29061, 88441}},
	{6625, 17, 7605, {1, 1, 5, 13, 23, 27, 7, 57, 273, 893, 773, 239, 6357, 15875, 5497, 21775, 108291}},
	{6626, 17, 7629, {1, 3, 1, 15, 25, 17, 11, 229, 175, 909, 691, 3507, 5247, 2933, 1741, 35059, 62841}},
	{6627, 17, 7632, {1, 3, 5, 1, 29, 7, 11, 69, 345, 87, 99, 3243, 5669, 11053, 1185, 6979, 117069}},
	{6628, 17, 7638, {1, 3, 5, 11, 13, 33, 23, 183, 89, 475, 643, 2773, 7899, 15219, 133, 5073, 129355}},
	{6629, 17, 7648, {1, 3, 7, 9, 23, 17, 31, 53, 455, 193, 1695, 2557, 1645, 12675, 27857, 50447, 121335}},
	{6630, 17, 7654, {1, 1, 3, 11, 15, 19, 41, 57, 305, 235, 1131, 1165, 1857, 13667, 19285, 29755, 118885}},
	{6631, 17, 7663, {1, 3, 7, 3, 9, 43, 107, 3, 275, 673, 677, 3769, 3097, 5497, 24911, 4617, 80505}},
	{6632, 17, 7675, {1, 1, 7, 9, 21, 39, 107, 155, 471, 753, 579, 2929, 4951, 4245, 25035, 41795, 86955}},
	{6633, 17, 7693, {1, 3, 1, 7, 31, 51, 27, 165, 147, 751, 709, 399, 45, 947, 9893, 32721, 122127}},
	{6634, 17, 7705, {1, 3, 3, 1, 31, 31, 73, 59, 351, 293, 845, 3139, 4177, 3537, 9465, 20689, 65837}},
	{6635, 17, 7717, {1, 3, 5, 9, 27, 29, 13, 115, 417, 435, 465, 1291, 5225, 11687, 29207, 39895, 55443}},
	{6636, 17, 7724, {1, 3, 3, 15, 29, 49, 111, 179, 221, 565, 787, 1811, 4055, 7863, 27273, 32975, 26985}},
	{6637, 17, 7727, {1, 1, 1, 7, 15, 49, 121, 145, 277, 27, 149, 965, 4903, 3497, 32333, 37217, 105073}},
	{6638, 17, 7735, {1, 1, 7, 1, 23, 29, 31, 77, 353, 349, 755, 2081, 4291, 567, 641, 41751, 20397}},
	{6639, 17, 7761, {1, 1, 5, 3, 25, 31, 97, 3, 405, 607, 965, 2981, 3217, 14695, 25977, 22457, 113539}},
	{6640, 17, 7767, {1, 3, 3, 15, 25, 3, 91, 125, 269, 825, 1163, 2181, 4247, 6813, 4699, 35091, 87771}},
	{6641, 17, 7783, {1, 1, 5, 9, 25, 23, 113, 145, 71, 31, 1115, 3729, 6793, 11869, 26509, 18779, 99499}},
	{6642, 17, 7784, {1, 1, 1, 9, 31, 51, 77, 217, 247, 599, 1541, 3217, 1383, 5203, 27971, 23647, 71823}},
	{6643, 17, 7798, {1, 1, 5, 7, 17, 35, 113, 73, 475, 511, 35, 1961, 5311, 2257, 1935, 58963, 94349}},
	{6644, 17, 7802, {1, 3, 1, 7, 27, 31, 67, 253, 95, 883, 1213, 855, 3429, 15049, 26715, 56099, 101797}},
	{6645, 17, 7811, {1, 1, 3, 5, 9, 9, 61, 57, 511, 537, 1803, 949, 1327, 3921, 11297, 13807, 64629}},
	{6646, 17, 7817, {1, 1, 5, 1, 31, 57, 105, 161, 309, 283, 1291, 2737, 7141, 7497, 25893, 14453, 35375}},
	{6647, 17, 7823, {1, 1, 3, 1, 21, 3, 77, 37, 13, 211, 1863, 1895, 8035, 5801, 25981, 12317, 48375}},
	{6648, 17, 7832, {1, 3, 7, 7, 25, 45, 13, 77, 185, 553, 1501, 1349, 5951, 15581, 32657, 18467, 128363}},
	{6649, 17, 7837, {1, 3, 5, 9, 23, 63, 105, 239, 213, 935, 1331, 3653, 2775, 6591, 6067, 34597, 19217}},
	{6650, 17, 7842, {1, 3, 7, 13, 15, 19, 79, 91, 391, 637, 1685, 2263, 3507, 2025, 2111, 15875, 14831}},
	{6651, 17, 7853, {1, 3, 3, 5, 7, 29, 81, 69, 511, 399, 343, 737, 2833, 1021, 10471, 18689, 36181}},
	{6652, 17, 7854, {1, 1, 5, 11, 21, 17, 39, 137, 145, 857, 583, 789, 8057, 15995, 32113, 64163, 37153}},
	{6653, 17, 7856, {1, 3, 3, 11, 9, 61, 87, 131, 487, 667, 1295, 493, 4629, 7719, 18157, 49715, 2051}},
	{6654, 17, 7861, {1, 3, 5, 9, 19, 5, 85, 3, 491, 353, 571, 2829, 4411, 343, 24781, 62325, 123959}},
	{6655, 17, 7862, {1, 1, 7, 13, 13, 39, 11, 31, 413, 285, 27, 2433, 3307, 6165, 26565, 40065, 102655}},
	{6656, 17, 7873, {1, 1, 5, 11, 25, 45, 7, 97, 9, 973, 1833, 2537, 1457, 7389, 24087, 38061, 122805}},
	{6657, 17, 7874, {1, 3, 5, 3, 21, 63, 77, 21, 249, 525, 1145, 1421, 8011, 3357, 15051, 30293, 127017}},
	{6658, 17, 7886, {1, 1, 5, 3, 13, 53, 81, 185, 303, 307, 1579, 841, 2277, 607, 10899, 34209, 215}},
	{6659, 17, 7914, {1, 3, 3, 13, 17, 1, 125, 145, 205, 763, 127, 1865, 4129, 849, 27247, 29845, 36515}},
	{6660, 17, 7927, {1, 3, 7, 13, 5, 59, 19, 71, 227, 111, 365, 1309, 5857, 6035, 32379, 11303, 127329}},
	{6661, 17, 7936, {1, 1, 1, 1, 19, 61, 79, 253, 459, 23, 725, 3615, 4583, 429, 13215, 31879, 4523}},
	{6662, 17, 7951, {1, 1, 1, 7, 19, 13, 53, 41, 243, 107, 1767, 983, 3483, 2249, 2209, 58895, 14805}},
	{6663, 17, 7963, {1, 1, 1, 9, 5, 63, 31, 85, 119, 307, 633, 3295, 841, 3495, 22965, 57587, 108271}},
	{6664, 17, 7966, {1, 3, 5, 9, 17, 13, 57, 49, 97, 613, 405, 2637, 3229, 14253, 4663, 61345, 33415}},
	{6665, 17, 7976, {1, 3, 1, 1, 17, 37, 63, 3, 5, 375, 1073, 3971, 665, 4445, 153, 20437, 38513}},
	{6666, 17, 7993, {1, 3, 3, 15, 5, 9, 77, 161, 409, 461, 443, 567, 5581, 8623, 27735, 9041, 5517}},
	{6667, 17, 8001, {1, 3, 5, 13, 13, 5, 19, 53, 263, 155, 557, 3973, 6841, 4829, 30751, 30025, 121973}},
	{6668, 17, 8004, {1, 3, 7, 9, 27, 37, 49, 243, 107, 1013, 1743, 1509, 4465, 15415, 4741, 41409, 72695}},
	{6669, 17, 8013, {1, 1, 3, 5, 11, 49, 39, 45, 21, 463, 875, 3681, 1901, 15325, 24553, 51369, 82227}},
	{6670, 17, 8021, {1, 1, 3, 15, 11, 35, 21, 91, 383, 149, 1815, 911, 4633, 12027, 12413, 22307, 120049}},
	{6671, 17, 8026, {1, 3, 5, 7, 7, 3, 15, 83, 477, 687, 145, 1705, 6893, 5233, 20171, 43337, 72603}},
	{6672, 17, 8028, {1, 1, 3, 9, 25, 35, 19, 173, 67, 5, 561, 2139, 4557, 4911, 26273, 38409, 22801}},
	{6673, 17, 8031, {1, 1, 3, 13, 15, 39, 85, 91, 91, 187, 1851, 1181, 4049, 16353, 26525, 43703, 19415}},
	{6674, 17, 8035, {1, 3, 1, 9, 13, 41, 77, 179, 415, 705, 693, 3017, 5847, 16191, 11435, 28979, 51839}},
	{6675, 17, 8042, {1, 1, 3, 5, 23, 15, 3, 159, 269, 67, 625, 4043, 4701, 1599, 6467, 10949, 80073}},
	{6676, 17, 8071, {1, 3, 3, 15, 7, 43, 81, 157, 393, 321, 1875, 2801, 4359, 11703, 1063, 64015, 109997}},
	{6677, 17, 8085, {1, 1, 7, 3, 25, 21, 37, 123, 133, 691, 973, 3115, 2291, 10519, 13339, 22751, 85445}},
	{6678, 17, 8092, {1, 3, 1, 1, 21, 21, 9, 23, 431, 679, 1873, 289, 4503, 3939, 14417, 36081, 18709}},
	{6679, 17, 8102, {1, 3, 5, 5, 1, 53, 109, 133, 33, 279, 727, 2233, 3065, 8557, 7487, 25797, 109177}},
	{6680, 17, 8105, {1, 1, 7, 7, 1, 9, 47, 127, 179, 757, 1985, 547, 169, 13393, 22669, 58795, 92897}},
	{6681, 17, 8114, {1, 3, 5, 11, 17, 21, 95, 219, 263, 579, 1493, 3283, 5461, 1235, 1749, 33325, 36033}},
	{6682, 17, 8123, {1, 1, 3, 11, 21, 49, 45, 143, 511, 983, 1933, 965, 7905, 1925, 27857, 40723, 68251}},
	{6683, 17, 8131, {1, 3, 7, 3, 27, 9, 73, 7, 441, 877, 107, 1599, 4795, 7251, 6819, 7671, 21137}},
	{6684, 17, 8140, {1, 1, 3, 3, 21, 25, 49, 43, 247, 949, 627, 2859, 2507, 4787, 11269, 53221, 126387}},
	{6685, 17, 8145, {1, 1, 5, 3, 5, 53, 127, 65, 353, 521, 1701, 2981, 3201, 611, 13475, 58015, 2605}},
	{6686, 17, 8157, {1, 1, 5, 13, 9, 39, 55, 103, 53, 281, 705, 2433, 6179, 3381, 31973, 30267, 91307}},
	{6687, 17, 8158, {1, 1, 7, 13, 31, 23, 29, 161, 347, 449, 123, 3427, 5731, 12691, 15175, 20487, 74695}},
	{6688, 17, 8185, {1, 3, 3, 15, 13, 19, 83, 137, 437, 317, 921, 913, 7979, 6665, 5313, 1435, 60271}},
	{6689, 17, 8186, {1, 3, 5, 7, 19, 23, 31, 131, 421, 95, 1999, 897, 4839, 1815, 12387, 45009, 2609}},
	{6690, 17, 8188, {1, 1, 1, 7, 3, 53, 121, 33, 47, 283, 813, 3841, 4449, 2543, 15211, 59285, 42551}},
	{6691, 17, 8192, {1, 3, 1, 13, 9, 43, 37, 167, 93, 417, 213, 2721, 3395, 2089, 13743, 32925, 91147}},
	{6692, 17, 8212, {1, 3, 7, 5, 31, 25, 97, 25, 19, 11, 543, 1889, 455, 5765, 9517, 56963, 131069}},
	{6693, 17, 8219, {1, 3, 1, 7, 3, 7, 87, 61, 209, 39, 1303, 1637, 6687, 8001, 5003, 47911, 110657}},
	{6694, 17, 8221, {1, 1, 5, 3, 11, 25, 99, 77, 379, 843, 1423, 2933, 7535, 4181, 32223, 49327, 48041}},
	{6695, 17, 8235, {1, 3, 3, 13, 9, 7, 85, 59, 47, 777, 401, 2449, 2795, 11289, 25023, 7725, 73887}},
	{6696, 17, 8237, {1, 1, 3, 5, 11, 51, 93, 57, 369, 871, 1175, 2705, 1253, 5169, 24691, 14243, 119667}},
	{6697, 17, 8249, {1, 3, 1, 3, 5, 7, 33, 171, 359, 115, 1909, 2003, 1413, 13829, 3471, 36185, 118399}},
	{6698, 17, 8260, {1, 1, 1, 11, 5, 49, 97, 145, 415, 731, 671, 2309, 7211, 11359, 22757, 15415, 70951}},
	{6699, 17, 8264, {1, 1, 3, 5, 7, 51, 61, 101, 375, 575, 1321, 2835, 7569, 9599, 4707, 7655, 53417}},
	{6700, 17, 8270, {1, 3, 1, 15, 9, 63, 25, 117, 203, 5, 1345, 2571, 5273, 2059, 4689, 27237, 23199}},
	{6701, 17, 8282, {1, 1, 3, 15, 15, 23, 69, 49, 349, 995, 5, 1565, 903, 10165, 9565, 6343, 16653}},
	{6702, 17, 8291, {1, 1, 3, 9, 21, 15, 69, 9, 463, 69, 1447, 2347, 5125, 7479, 18257, 14405, 51321}},
	{6703, 17, 8293, {1, 1, 7, 11, 23, 57, 57, 179, 17, 129, 999, 777, 6281, 1693, 31885, 31085, 29237}},
	{6704, 17, 8297, {1, 3, 5, 1, 25, 55, 15, 21, 199, 271, 1645, 1719, 2023, 10049, 15215, 11959, 44875}},
	{6705, 17, 8298, {1, 3, 1, 3, 29, 43, 83, 11, 281, 27, 429, 685, 7189, 9151, 8665, 9553, 115293}},
	{6706, 17, 8305, {1, 3, 1, 7, 17, 43, 125, 11, 189, 803, 713, 683, 7285, 4455, 18195, 45333, 32281}},
	{6707, 17, 8306, {1, 3, 3, 3, 11, 55, 21, 59, 173, 283, 709, 1561, 5391, 5097, 24725, 19217, 13769}},
	{6708, 17, 8311, {1, 3, 5, 13, 7, 29, 117, 207, 415, 525, 567, 1741, 3553, 6729, 433, 17619, 45971}},
	{6709, 17, 8318, {1, 1, 7, 7, 3, 23, 43, 43, 213, 823, 609, 1037, 3797, 4733, 30717, 61067, 89581}},
	{6710, 17, 8327, {1, 3, 5, 7, 11, 7, 7, 241, 379, 217, 739, 2815, 2549, 14297, 10283, 1509, 80613}},
	{6711, 17, 8345, {1, 1, 1, 1, 17, 45, 53, 229, 193, 893, 1881, 227, 6751, 7135, 20823, 36939, 27667}},
	{6712, 17, 8379, {1, 3, 3, 1, 15, 39, 27, 217, 101, 949, 1963, 2213, 2357, 4129, 11925, 841, 59259}},
	{6713, 17, 8390, {1, 1, 3, 3, 5, 53, 59, 255, 421, 1009, 683, 2171, 6691, 12489, 20865, 29363, 70611}},
	{6714, 17, 8394, {1, 1, 7, 15, 7, 31, 105, 141, 153, 401, 549, 3045, 5443, 11147, 18159, 24283, 21859}},
	{6715, 17, 8414, {1, 3, 7, 1, 11, 17, 17, 231, 175, 603, 1915, 111, 3203, 10627, 9687, 47235, 87057}},
	{6716, 17, 8417, {1, 1, 1, 11, 19, 21, 115, 41, 45, 727, 1523, 739, 3025, 10321, 27353, 63139, 16051}},
	{6717, 17, 8432, {1, 3, 7, 11, 13, 9, 33, 121, 237, 565, 2043, 2131, 3079, 12575, 2187, 14427, 85939}},
	{6718, 17, 8437, {1, 3, 1, 15, 21, 19, 91, 227, 485, 49, 101, 15, 1903, 4039, 23819, 40001, 66405}},
	{6719, 17, 8441, {1, 3, 1, 5, 15, 25, 65, 25, 393, 287, 1435, 1851, 6437, 5983, 13769, 37847, 120907}},
	{6720, 17, 8449, {1, 3, 7, 15, 15, 21, 97, 37, 359, 155, 807, 1421, 517, 13135, 2955, 56979, 52299}},
	{6721, 17, 8456, {1, 1, 5, 1, 27, 53, 79, 27, 467, 605, 267, 1193, 31, 6177, 12369, 32621, 38319}},
	{6722, 17, 8473, {1, 1, 1, 11, 27, 15, 15, 231, 205, 677, 331, 133, 3313, 7193, 8059, 36449, 21671}},
	{6723, 17, 8489, {1, 3, 3, 11, 19, 57, 113, 83, 399, 801, 1843, 2119, 2779, 14061, 30901, 28745, 120903}},
	{6724, 17, 8495, {1, 1, 1, 11, 5, 27, 121, 247, 467, 251, 1487, 251, 897, 3171, 28383, 22473, 1709}},
	{6725, 17, 8522, {1, 1, 1, 15, 7, 59, 123, 165, 123, 373, 167, 1323, 5239, 9027, 13791, 55593, 78785}},
	{6726, 17, 8524, {1, 3, 1, 11, 31, 11, 81, 229, 123, 183, 461, 1751, 5713, 2615, 27795, 1657, 39253}},
	{6727, 17, 8529, {1, 1, 7, 1, 21, 45, 107, 3, 283, 149, 549, 3731, 6435, 3595, 32753, 16079, 84257}},
	{6728, 17, 8545, {1, 3, 3, 15, 19, 9, 81, 37, 51, 341, 909, 985, 1503, 12787, 16129, 37789, 113515}},
	{6729, 17, 8557, {1, 3, 5, 13, 3, 33, 127, 219, 369, 341, 1191, 1305, 567, 2339, 31221, 49435, 114927}},
	{6730, 17, 8565, {1, 1, 7, 15, 29, 47, 103, 107, 257, 15, 2029, 2133, 2129, 11235, 29553, 49139, 33809}},
	{6731, 17, 8572, {1, 3, 3, 13, 23, 33, 105, 43, 155, 815, 1087, 2261, 2781, 3461, 7371, 4479, 123093}},
	{6732, 17, 8576, {1, 1, 1, 13, 17, 7, 89, 107, 143, 349, 637, 3651, 4153, 12131, 28393, 45781, 84133}},
	{6733, 17, 8582, {1, 3, 5, 11, 31, 47, 105, 101, 267, 403, 1853, 3977, 3277, 1737, 15503, 47365, 14361}},
	{6734, 17, 8594, {1, 1, 1, 13, 1, 63, 125, 107, 123, 183, 1027, 3491, 3597, 15949, 5779, 34665, 81257}},
	{6735, 17, 8629, {1, 3, 1, 9, 13, 5, 125, 41, 389, 73, 1487, 1983, 957, 12645, 13983, 7675, 72711}},
	{6736, 17, 8636, {1, 3, 7, 5, 17, 5, 25, 63, 211, 591, 261, 2345, 3883, 4403, 773, 43963, 93509}},
	{6737, 17, 8668, {1, 3, 3, 1, 11, 35, 15, 251, 225, 643, 537, 3769, 7593, 6113, 1377, 52185, 81459}},
	{6738, 17, 8678, {1, 3, 5, 15, 27, 27, 51, 35, 389, 853, 1437, 2803, 5739, 1887, 15099, 3299, 111827}},
	{6739, 17, 8701, {1, 1, 3, 15, 25, 63, 31, 201, 79, 131, 31, 3929, 4195, 13045, 8681, 48121, 110723}},
	{6740, 17, 8702, {1, 1, 5, 7, 11, 43, 101, 57, 69, 271, 189, 3087, 4893, 11365, 6945, 14285, 41961}},
	{6741, 17, 8708, {1, 1, 7, 9, 21, 61, 41, 123, 25, 947, 1619, 2895, 7879, 12397, 17405, 48139, 71519}},
	{6742, 17, 8712, {1, 3, 1, 15, 1, 27, 113, 225, 441, 855, 541, 357, 3111, 4867, 20571, 30627, 70123}},
	{6743, 17, 8745, {1, 3, 5, 3, 5, 33, 103, 1, 21, 93, 383, 407, 5145, 7857, 20289, 51943, 16223}},
	{6744, 17, 8754, {1, 1, 7, 15, 1, 13, 41, 215, 463, 417, 513, 3417, 1755, 16165, 7271, 3101, 54353}},
	{6745, 17, 8759, {1, 3, 3, 13, 19, 29, 5, 205, 245, 927, 1249, 773, 3653, 9959, 357, 40863, 37289}},
	{6746, 17, 8763, {1, 3, 3, 7, 3, 5, 85, 241, 29, 627, 1963, 3133, 1369, 503, 11449, 4699, 2573}},
	{6747, 17, 8766, {1, 1, 7, 15, 3, 35, 47, 157, 413, 437, 1627, 3953, 947, 12721, 22209, 34303, 81237}},
	{6748, 17, 8780, {1, 1, 5, 5, 1, 45, 47, 245, 253, 349, 1853, 3481, 6105, 7267, 3159, 38833, 117889}},
	{6749, 17, 8783, {1, 3, 7, 15, 23, 43, 25, 181, 121, 681, 479, 1239, 6155, 3317, 9419, 28717, 44643}},
	{6750, 17, 8786, {1, 3, 3, 15, 31, 43, 111, 99, 405, 991, 301, 1689, 7107, 16131, 16703, 24059, 40345}},
	{6751, 17, 8798, {1, 1, 3, 9, 25, 5, 107, 91, 117, 351, 1595, 163, 3007, 13743, 24535, 38671, 29745}},
	{6752, 17, 8804, {1, 3, 3, 5, 27, 47, 15, 195, 119, 919, 665, 1903, 1981, 7753, 21709, 33699, 15963}},
	{6753, 17, 8819, {1, 3, 1, 11, 23, 23, 75, 115, 477, 105, 541, 1111, 209, 13939, 17129, 7565, 75415}},
	{6754, 17, 8826, {1, 1, 1, 11, 7, 61, 123, 201, 305, 713, 779, 2059, 4899, 13733, 20529, 15617, 39833}},
	{6755, 17, 8835, {1, 1, 7, 11, 21, 7, 63, 113, 213, 871, 375, 29, 1925, 15237, 7091, 12229, 8457}},
	{6756, 17, 8838, {1, 1, 1, 7, 19, 57, 83, 91, 297, 255, 1993, 63, 5337, 4569, 21243, 40867, 46969}},
	{6757, 17, 8856, {1, 1, 3, 7, 13, 63, 91, 191, 281, 259, 1367, 3505, 5885, 10557, 12423, 56303, 14731}},
	{6758, 17, 8862, {1, 1, 5, 15, 27, 15, 29, 67, 115, 287, 253, 1497, 3739, 2183, 14427, 44931, 11547}},
	{6759, 17, 8871, {1, 3, 1, 9, 25, 61, 25, 113, 137, 819, 781, 3741, 2457, 7817, 31209, 20707, 93007}},
	{6760, 17, 8875, {1, 1, 7, 3, 5, 13, 23, 3, 365, 77, 1117, 3061, 4707, 3013, 27899, 10887, 78677}},
	{6761, 17, 8890, {1, 3, 1, 15, 1, 39, 85, 107, 483, 83, 603, 3121, 1995, 5241, 32319, 9515, 94551}},
	{6762, 17, 8892, {1, 1, 7, 3, 27, 13, 105, 41, 285, 237, 1589, 517, 2009, 10833, 1459, 26217, 50759}},
	{6763, 17, 8898, {1, 1, 3, 11, 27, 1, 127, 83, 355, 107, 1003, 657, 4997, 4123, 13151, 56601, 122307}},
	{6764, 17, 8927, {1, 1, 1, 7, 13, 17, 93, 75, 481, 473, 131, 1359, 4859, 1319, 23919, 50079, 128849}},
	{6765, 17, 8928, {1, 1, 3, 7, 9, 33, 111, 229, 11, 283, 1089, 3049, 1635, 959, 19109, 62821, 105391}},
	{6766, 17, 8945, {1, 3, 1, 3, 9, 47, 49, 169, 343, 929, 1379, 1985, 5867, 6053, 12179, 39727, 116053}},
	{6767, 17, 8952, {1, 3, 3, 15, 27, 39, 61, 113, 439, 719, 1313, 3701, 4793, 10275, 2943, 32405, 95457}},
	{6768, 17, 8955, {1, 1, 1, 1, 27, 49, 121, 171, 319, 365, 1593, 1655, 63, 6257, 18097, 35285, 112245}},
	{6769, 17, 8965, {1, 3, 1, 1, 19, 33, 89, 235, 281, 519, 1867, 525, 4475, 12059, 26611, 14789, 59541}},
	{6770, 17, 8972, {1, 3, 1, 15, 1, 51, 65, 71, 131, 599, 117, 2459, 7421, 7157, 24393, 48139, 53701}},
	{6771, 17, 8977, {1, 1, 7, 7, 1, 41, 57, 191, 207, 329, 43, 1235, 5671, 12243, 22549, 40751, 104513}},
	{6772, 17, 8990, {1, 3, 5, 13, 15, 21, 55, 187, 283, 209, 1511, 1329, 6665, 15953, 4521, 16879, 57625}},
	{6773, 17, 8996, {1, 1, 5, 3, 3, 53, 75, 123, 291, 663, 1893, 3669, 4903, 8575, 27971, 46977, 56357}},
	{6774, 17, 9025, {1, 3, 1, 5, 27, 41, 19, 199, 489, 197, 439, 3299, 6315, 6957, 15809, 35297, 5559}},
	{6775, 17, 9037, {1, 3, 5, 1, 3, 25, 109, 191, 33, 543, 125, 2309, 429, 14059, 3149, 45747, 47357}},
	{6776, 17, 9040, {1, 1, 3, 11, 15, 61, 109, 103, 305, 1, 1479, 2781, 6521, 8921, 23681, 9583, 87257}},
	{6777, 17, 9049, {1, 1, 7, 15, 5, 19, 121, 139, 177, 967, 1363, 705, 211, 11877, 22457, 34563, 7801}},
	{6778, 17, 9062, {1, 1, 7, 13, 9, 21, 103, 95, 483, 567, 5, 2095, 4659, 2447, 23521, 27273, 85867}},
	{6779, 17, 9068, {1, 3, 5, 15, 23, 55, 13, 237, 275, 113, 1431, 2931, 5165, 5317, 5625, 51865, 42177}},
	{6780, 17, 9076, {1, 3, 3, 7, 1, 23, 15, 171, 303, 43, 1137, 1255, 3843, 9049, 1799, 7075, 2115}},
	{6781, 17, 9079, {1, 1, 7, 5, 23, 53, 75, 129, 1, 511, 793, 265, 6535, 9641, 25173, 9449, 46949}},
	{6782, 17, 9099, {1, 3, 3, 1, 19, 39, 51, 173, 5, 281, 2047, 4065, 3225, 14587, 16947, 1459, 87227}},
	{6783, 17, 9107, {1, 3, 7, 13, 13, 53, 39, 115, 403, 37, 1533, 2727, 2229, 8291, 18687, 59553, 37629}},
	{6784, 17, 9114, {1, 3, 1, 9, 3, 55, 63, 191, 147, 321, 1287, 2419, 6881, 2249, 11141, 54839, 50263}},
	{6785, 17, 9123, {1, 1, 5, 3, 9, 61, 85, 139, 1, 409, 633, 53, 163, 14677, 13043, 12253, 106939}},
	{6786, 17, 9126, {1, 1, 7, 3, 19, 3, 7, 165, 497, 621, 1563, 1267, 8113, 2383, 17205, 13337, 102547}},
	{6787, 17, 9137, {1, 3, 3, 13, 15, 29, 23, 31, 481, 535, 471, 2125, 331, 9421, 29799, 27097, 5307}},
	{6788, 17, 9149, {1, 1, 1, 1, 31, 45, 47, 139, 235, 509, 889, 685, 1855, 13599, 24431, 62105, 109509}},
	{6789, 17, 9150, {1, 3, 1, 7, 3, 13, 25, 197, 111, 45, 1815, 1031, 4803, 349, 32369, 40837, 111529}},
	{6790, 17, 9155, {1, 1, 7, 1, 27, 9, 3, 73, 403, 321, 967, 2713, 6953, 16123, 8611, 48651, 120635}},
	{6791, 17, 9161, {1, 3, 5, 3, 3, 25, 69, 231, 249, 393, 1141, 1721, 7071, 3711, 15627, 21815, 104735}},
	{6792, 17, 9162, {1, 3, 1, 11, 19, 63, 77, 5, 55, 481, 1021, 119, 3941, 1227, 10997, 29513, 18923}},
	{6793, 17, 9167, {1, 3, 7, 5, 1, 11, 13, 99, 365, 797, 1993, 699, 3091, 11401, 3659, 15339, 90395}},
	{6794, 17, 9172, {1, 3, 5, 7, 31, 43, 55, 143, 273, 379, 1189, 1689, 4811, 5159, 3281, 63819, 57065}},
	{6795, 17, 9186, {1, 1, 1, 13, 9, 25, 9, 3, 461, 281, 959, 2439, 3187, 4837, 13857, 20221, 29733}},
	{6796, 17, 9188, {1, 1, 7, 11, 31, 17, 13, 101, 81, 921, 1329, 2421, 2747, 9435, 23313, 7093, 7547}},
	{6797, 17, 9191, {1, 1, 3, 3, 9, 51, 67, 95, 511, 1011, 1519, 4089, 5001, 1351, 15367, 50665, 92111}},
	{6798, 17, 9198, {1, 1, 5, 13, 27, 43, 115, 77, 439, 589, 31, 915, 7027, 697, 25143, 1443, 59093}},
	{6799, 17, 9200, {1, 1, 7, 3, 17, 5, 107, 117, 133, 649, 1309, 2979, 969, 9789, 12597, 24507, 106825}},
	{6800, 17, 9205, {1, 1, 7, 13, 1, 27, 97, 35, 431, 183, 199, 2619, 515, 89, 20281, 30291, 97977}},
	{6801, 17, 9206, {1, 1, 7, 1, 31, 9, 35, 11, 359, 21, 1875, 3227, 1307, 15691, 17343, 21163, 84671}},
	{6802, 17, 9215, {1, 3, 1, 11, 29, 21, 47, 137, 441, 841, 1641, 3283, 1371, 8835, 16287, 45009, 13779}},
	{6803, 17, 9227, {1, 1, 3, 9, 23, 53, 1, 99, 473, 649, 447, 2589, 5667, 15579, 6497, 44321, 46993}},
	{6804, 17, 9232, {1, 1, 7, 9, 31, 63, 95, 81, 197, 373, 1027, 3959, 7189, 13369, 17287, 53643, 12673}},
	{6805, 17, 9241, {1, 3, 1, 5, 25, 61, 79, 183, 489, 725, 1077, 1147, 113, 7357, 27505, 529, 61855}},
	{6806, 17, 9244, {1, 1, 7, 11, 19, 35, 73, 223, 125, 765, 1303, 2887, 7861, 14839, 9537, 27027, 94327}},
	{6807, 17, 9251, {1, 3, 1, 3, 17, 35, 63, 233, 317, 133, 1837, 3339, 4351, 10071, 5005, 13245, 34327}},
	{6808, 17, 9254, {1, 3, 1, 3, 17, 13, 59, 113, 247, 1015, 1831, 3391, 6337, 6853, 7145, 64309, 40109}},
	{6809, 17, 9275, {1, 3, 5, 13, 15, 23, 65, 203, 241, 545, 1521, 1253, 3171, 7777, 21145, 565, 87813}},
	{6810, 17, 9283, {1, 1, 5, 15, 31, 9, 9, 145, 409, 155, 409, 2935, 5817, 11427, 32617, 38167, 69465}},
	{6811, 17, 9285, {1, 1, 5, 11, 19, 31, 43, 85, 97, 931, 687, 1501, 3991, 2215, 11091, 64735, 56999}},
	{6812, 17, 9303, {1, 1, 1, 3, 7, 11, 101, 21, 345, 829, 531, 1475, 6617, 1187, 26885, 32135, 9733}},
	{6813, 17, 9304, {1, 3, 5, 11, 7, 49, 79, 197, 57, 15, 1845, 1485, 6167, 10887, 17083, 59367, 7411}},
	{6814, 17, 9313, {1, 3, 7, 5, 9, 33, 7, 91, 311, 847, 1435, 3573, 3693, 5369, 26817, 30105, 115337}},
	{6815, 17, 9314, {1, 3, 1, 9, 25, 43, 65, 69, 225, 337, 575, 1979, 5555, 8499, 8127, 33035, 52549}},
	{6816, 17, 9320, {1, 1, 3, 11, 17, 29, 71, 99, 379, 145, 1067, 2561, 7635, 5647, 32491, 56621, 93603}},
	{6817, 17, 9328, {1, 1, 5, 13, 25, 43, 75, 237, 407, 393, 1219, 3651, 7719, 11685, 26123, 62767, 1043}},
	{6818, 17, 9333, {1, 1, 7, 15, 13, 59, 9, 163, 273, 225, 873, 3201, 633, 6121, 18777, 58763, 77731}},
	{6819, 17, 9337, {1, 3, 7, 7, 3, 7, 99, 155, 279, 991, 799, 753, 7205, 9567, 23643, 38263, 19083}},
	{6820, 17, 9338, {1, 3, 7, 11, 11, 29, 65, 3, 207, 575, 253, 2407, 7935, 11323, 23239, 1923, 47737}},
	{6821, 17, 9340, {1, 1, 5, 9, 25, 47, 1, 25, 397, 1009, 193, 4031, 3023, 2029, 10561, 32363, 104405}},
	{6822, 17, 9353, {1, 3, 7, 9, 19, 55, 63, 179, 385, 97, 461, 3393, 8137, 8929, 17621, 9611, 58925}},
	{6823, 17, 9356, {1, 1, 1, 7, 1, 17, 127, 45, 157, 529, 809, 3545, 5173, 5083, 13325, 52295, 91261}},
	{6824, 17, 9364, {1, 1, 7, 9, 25, 49, 99, 79, 157, 535, 1569, 2195, 1725, 1187, 18423, 47957, 10043}},
	{6825, 17, 9373, {1, 1, 3, 7, 3, 31, 83, 45, 199, 665, 1261, 3497, 7885, 5761, 17187, 12041, 12867}},
	{6826, 17, 9374, {1, 3, 1, 7, 3, 55, 73, 215, 41, 1011, 1883, 1051, 7293, 1881, 27435, 29459, 130933}},
	{6827, 17, 9378, {1, 1, 3, 9, 21, 31, 113, 209, 35, 771, 365, 3151, 787, 3845, 26555, 13823, 36951}},
	{6828, 17, 9380, {1, 3, 7, 15, 13, 21, 119, 91, 15, 251, 1337, 2715, 1665, 3451, 8309, 11033, 127159}},
	{6829, 17, 9389, {1, 3, 1, 9, 9, 63, 5, 145, 357, 9, 859, 1565, 1141, 14689, 25121, 41337, 83357}},
	{6830, 17, 9395, {1, 1, 7, 11, 13, 63, 57, 151, 33, 595, 2025, 571, 4713, 11019, 26771, 16221, 92439}},
	{6831, 17, 9412, {1, 3, 3, 15, 29, 49, 93, 131, 167, 835, 33, 263, 93, 8475, 16139, 61237, 95081}},
	{6832, 17, 9422, {1, 1, 7, 13, 1, 57, 43, 91, 485, 841, 1415, 3083, 2665, 8521, 9825, 59955, 21763}},
	{6833, 17, 9439, {1, 1, 1, 1, 29, 47, 63, 107, 439, 847, 537, 2011, 7571, 3699, 23961, 54887, 92681}},
	{6834, 17, 9450, {1, 3, 7, 5, 27, 41, 105, 161, 95, 821, 451, 2627, 4687, 1899, 18851, 35167, 6869}},
	{6835, 17, 9452, {1, 1, 1, 11, 7, 7, 13, 163, 399, 471, 1587, 2561, 1241, 5365, 27189, 49883, 68101}},
	{6836, 17, 9482, {1, 3, 7, 9, 19, 5, 119, 251, 151, 359, 235, 2387, 3919, 7135, 17591, 1053, 6265}},
	{6837, 17, 9487, {1, 1, 5, 9, 13, 25, 43, 23, 453, 693, 517, 1235, 1045, 4299, 27877, 3733, 72269}},
	{6838, 17, 9489, {1, 1, 7, 1, 27, 43, 103, 249, 487, 67, 855, 3239, 2157, 8121, 4701, 37803, 49971}},
	{6839, 17, 9499, {1, 1, 3, 13, 1, 37, 125, 115, 365, 57, 1419, 4085, 7039, 10079, 14991, 48861, 61979}},
	{6840, 17, 9501, {1, 1, 5, 5, 3, 35, 109, 19, 219, 653, 1219, 1625, 6847, 11271, 4525, 56341, 57801}},
	{6841, 17, 9508, {1, 3, 7, 5, 31, 19, 37, 73, 185, 13, 1723, 1139, 5919, 11717, 27161, 13635, 51765}},
	{6842, 17, 9515, {1, 1, 1, 1, 19, 61, 53, 111, 215, 189, 1199, 591, 943, 2111, 17171, 15621, 128459}},
	{6843, 17, 9518, {1, 1, 7, 9, 17, 61, 101, 159, 85, 537, 15, 1427, 6139, 4091, 32639, 28655, 115385}},
	{6844, 17, 9520, {1, 1, 7, 5, 23, 31, 125, 7, 151, 967, 1079, 4059, 3287, 11673, 19307, 49469, 65981}},
	{6845, 17, 9526, {1, 3, 3, 1, 29, 59, 95, 119, 31, 427, 1653, 721, 5509, 6385, 17043, 45133, 74155}},
	{6846, 17, 9537, {1, 1, 3, 9, 13, 61, 35, 189, 1, 559, 119, 3719, 4137, 1369, 19147, 10923, 43909}},
	{6847, 17, 9552, {1, 3, 3, 13, 1, 41, 31, 185, 451, 379, 29, 153, 4121, 13153, 4171, 36993, 109241}},
	{6848, 17, 9571, {1, 1, 1, 9, 15, 41, 99, 17, 21, 93, 649, 2765, 6955, 10843, 12547, 64989, 63713}},
	{6849, 17, 9588, {1, 1, 7, 5, 5, 5, 73, 187, 473, 235, 1907, 409, 7335, 4429, 7493, 20703, 14505}},
	{6850, 17, 9613, {1, 1, 3, 11, 27, 59, 17, 103, 337, 117, 1241, 951, 3701, 10407, 16741, 46531, 56485}},
	{6851, 17, 9619, {1, 1, 3, 15, 11, 51, 111, 189, 137, 939, 97, 1563, 851, 13949, 1375, 41463, 61445}},
	{6852, 17, 9622, {1, 1, 7, 9, 19, 39, 117, 173, 165, 547, 483, 361, 6819, 15093, 13631, 29785, 29593}},
	{6853, 17, 9637, {1, 3, 3, 5, 15, 39, 41, 249, 455, 79, 233, 3133, 405, 9487, 23161, 32751, 117743}},
	{6854, 17, 9652, {1, 1, 5, 15, 7, 63, 7, 57, 127, 349, 1913, 1145, 3371, 3733, 30971, 35717, 60935}},
	{6855, 17, 9655, {1, 1, 7, 11, 7, 57, 49, 63, 51, 233, 855, 2125, 6961, 15011, 28503, 40549, 47175}},
	{6856, 17, 9661, {1, 3, 7, 1, 25, 49, 35, 39, 237, 545, 1637, 1401, 3279, 10499, 14463, 34973, 29485}},
	{6857, 17, 9664, {1, 3, 3, 13, 7, 13, 79, 141, 55, 277, 843, 3087, 2339, 6855, 10635, 13021, 11273}},
	{6858, 17, 9669, {1, 3, 1, 1, 11, 39, 51, 255, 119, 691, 559, 3287, 5485, 791, 19283, 51027, 8061}},
	{6859, 17, 9681, {1, 3, 7, 7, 3, 59, 119, 241, 185, 81, 1843, 2313, 7471, 15689, 2271, 59781, 107439}},
	{6860, 17, 9682, {1, 3, 3, 3, 17, 63, 93, 217, 329, 39, 583, 3031, 4315, 4623, 12557, 42063, 11877}},
	{6861, 17, 9688, {1, 1, 1, 1, 15, 57, 37, 233, 387, 639, 37, 425, 637, 1577, 16449, 33665, 80417}},
	{6862, 17, 9697, {1, 1, 1, 15, 25, 1, 67, 159, 423, 961, 959, 417, 5657, 8417, 8127, 29251, 105893}},
	{6863, 17, 9700, {1, 3, 5, 15, 31, 9, 87, 217, 259, 771, 1663, 2899, 1531, 7849, 1961, 61487, 55399}},
	{6864, 17, 9715, {1, 1, 3, 9, 21, 13, 39, 107, 89, 811, 449, 2569, 4617, 8977, 1649, 37721, 48943}},
	{6865, 17, 9722, {1, 3, 7, 15, 15, 59, 63, 195, 287, 677, 269, 1715, 3545, 3269, 5231, 46433, 25921}},
	{6866, 17, 9727, {1, 1, 5, 7, 19, 27, 57, 221, 243, 47, 1791, 2309, 2751, 4403, 7083, 34223, 64905}},
	{6867, 17, 9734, {1, 1, 1, 15, 1, 63, 119, 155, 383, 649, 429, 3857, 7309, 9823, 9539, 8933, 128573}},
	{6868, 17, 9740, {1, 3, 7, 11, 17, 19, 99, 19, 321, 415, 1501, 2123, 6119, 9705, 11397, 39521, 34327}},
	{6869, 17, 9743, {1, 1, 5, 15, 29, 37, 9, 95, 417, 19, 1637, 2949, 4961, 10743, 9619, 16045, 48083}},
	{6870, 17, 9745, {1, 1, 1, 11, 21, 17, 57, 23, 247, 201, 1781, 779, 2207, 2511, 4829, 13847, 77593}},
	{6871, 17, 9757, {1, 3, 1, 13, 7, 1, 95, 87, 223, 73, 1129, 383, 1355, 4965, 29645, 63465, 76281}},
	{6872, 17, 9761, {1, 3, 3, 13, 3, 47, 33, 123, 155, 621, 1019, 1817, 4083, 4723, 24701, 47503, 18007}},
	{6873, 17, 9762, {1, 1, 7, 15, 13, 41, 73, 93, 379, 923, 1183, 2475, 5901, 10599, 10053, 9941, 112107}},
	{6874, 17, 9767, {1, 1, 3, 3, 13, 35, 59, 231, 45, 1011, 1101, 2467, 2703, 10305, 12575, 7587, 25737}},
	{6875, 17, 9768, {1, 3, 7, 1, 21, 31, 9, 55, 373, 779, 397, 1551, 5139, 16339, 1769, 10413, 74059}},
	{6876, 17, 9774, {1, 1, 7, 15, 7, 3, 67, 179, 411, 217, 1219, 13, 1577, 13463, 12263, 41465, 83001}},
	{6877, 17, 9786, {1, 3, 7, 1, 21, 53, 7, 187, 395, 777, 391, 737, 47, 12681, 16749, 26507, 49415}},
	{6878, 17, 9796, {1, 1, 5, 7, 5, 57, 93, 53, 419, 731, 825, 487, 45, 9199, 20947, 56067, 45343}},
	{6879, 17, 9820, {1, 3, 3, 9, 31, 41, 35, 133, 63, 293, 1503, 51, 3111, 15711, 15051, 1965, 64951}},
	{6880, 17, 9823, {1, 1, 5, 9, 9, 47, 53, 229, 405, 621, 1795, 1923, 6609, 6983, 1695, 18021, 71893}},
	{6881, 17, 9839, {1, 1, 5, 9, 23, 13, 107, 13, 149, 759, 1113, 1329, 1747, 14159, 16705, 61841, 82955}},
	{6882, 17, 9844, {1, 3, 3, 9, 25, 49, 31, 145, 481, 609, 1847, 1485, 6345, 7859, 21231, 37303, 69975}},
	{6883, 17, 9851, {1, 3, 1, 15, 13, 49, 59, 221, 27, 517, 431, 3961, 6401, 8483, 10161, 37453, 128237}},
	{6884, 17, 9853, {1, 1, 3, 1, 3, 55, 37, 111, 263, 735, 655, 2831, 2219, 9449, 8413, 49585, 91355}},
	{6885, 17, 9863, {1, 3, 7, 1, 31, 33, 7, 55, 261, 977, 1215, 1967, 7297, 14815, 27009, 35001, 89671}},
	{6886, 17, 9864, {1, 1, 7, 11, 13, 21, 33, 151, 195, 373, 181, 1631, 355, 7857, 12555, 7531, 50417}},
	{6887, 17, 9877, {1, 3, 1, 15, 19, 25, 79, 195, 237, 385, 1531, 2509, 4371, 16103, 3575, 62265, 124251}},
	{6888, 17, 9884, {1, 3, 1, 11, 5, 61, 21, 159, 51, 37, 845, 3075, 8039, 14269, 10505, 36369, 73793}},
	{6889, 17, 9888, {1, 3, 5, 9, 11, 43, 67, 57, 271, 451, 989, 3705, 2481, 10717, 10861, 63785, 10183}},
	{6890, 17, 9897, {1, 3, 3, 5, 13, 29, 119, 171, 439, 459, 479, 3173, 3781, 11131, 6827, 53925, 119939}},
	{6891, 17, 9915, {1, 3, 7, 3, 27, 21, 1, 167, 79, 305, 1283, 1903, 5483, 5727, 17911, 16075, 97629}},
	{6892, 17, 9925, {1, 3, 1, 3, 23, 21, 29, 185, 227, 295, 915, 2033, 6269, 2089, 20785, 15207, 115675}},
	{6893, 17, 9949, {1, 3, 7, 15, 11, 15, 65, 103, 249, 27, 1805, 2079, 4797, 2535, 16865, 61449, 90923}},
	{6894, 17, 9954, {1, 3, 7, 9, 27, 41, 77, 181, 457, 677, 633, 1601, 8085, 2431, 7957, 55913, 38677}},
	{6895, 17, 9960, {1, 1, 5, 7, 11, 37, 3, 221, 79, 895, 1023, 653, 3925, 12755, 19729, 18221, 91123}},
	{6896, 17, 9965, {1, 3, 1, 5, 23, 61, 119, 191, 425, 41, 853, 3497, 6915, 1927, 5513, 55303, 4895}},
	{6897, 17, 9978, {1, 3, 5, 3, 7, 35, 47, 243, 167, 821, 267, 2149, 5797, 6329, 32495, 51037, 18313}},
	{6898, 17, 9986, {1, 1, 7, 9, 23, 29, 79, 205, 115, 839, 1217, 479, 1601, 9681, 1, 35293, 28731}},
	{6899, 17, 9992, {1, 3, 3, 5, 31, 17, 31, 161, 35, 953, 377, 451, 7985, 11371, 15115, 60131, 27033}},
	{6900, 17, 9995, {1, 1, 3, 9, 15, 19, 43, 215, 327, 429, 145, 1837, 725, 14775, 10465, 7367, 21271}},
	{6901, 17, 10005, {1, 3, 7, 13, 31, 17, 85, 49, 487, 795, 1679, 599, 3783, 3195, 2683, 53475, 38603}},
	{6902, 17, 10026, {1, 1, 1, 7, 19, 11, 71, 143, 443, 199, 1117, 3445, 6429, 12037, 13751, 43609, 101563}},
	{6903, 17, 10031, {1, 3, 5, 7, 29, 63, 65, 87, 305, 721, 851, 2235, 4987, 3051, 23015, 1281, 15755}},
	{6904, 17, 10040, {1, 1, 3, 9, 17, 3, 57, 47, 223, 305, 1409, 235, 4379, 5779, 27695, 22535, 9387}},
	{6905, 17, 10051, {1, 1, 3, 11, 25, 33, 75, 141, 155, 699, 85, 1729, 2551, 7101, 7739, 18025, 100819}},
	{6906, 17, 10057, {1, 3, 3, 13, 5, 45, 63, 83, 141, 383, 1931, 3343, 7397, 4823, 28893, 41279, 67805}},
	{6907, 17, 10072, {1, 3, 5, 7, 19, 29, 97, 67, 177, 583, 1783, 4007, 5087, 805, 30999, 23197, 117553}},
	{6908, 17, 10096, {1, 3, 5, 1, 25, 41, 33, 109, 511, 449, 653, 995, 5881, 2163, 13689, 54385, 97419}},
	{6909, 17, 10102, {1, 3, 3, 13, 25, 17, 49, 77, 497, 659, 783, 3513, 3735, 3541, 573, 50237, 99247}},
	{6910, 17, 10105, {1, 3, 1, 7, 17, 13, 37, 169, 19, 965, 289, 455, 6855, 11233, 7553, 7007, 57389}},
	{6911, 17, 10115, {1, 1, 7, 11, 5, 15, 11, 177, 75, 243, 453, 3861, 3091, 4625, 12489, 11537, 74199}},
	{6912, 17, 10124, {1, 1, 5, 13, 17, 21, 23, 57, 343, 985, 1755, 3947, 3899, 11847, 19321, 62295, 51265}},
	{6913, 17, 10139, {1, 1, 3, 9, 19, 37, 31, 243, 433, 725, 535, 3733, 33, 7885, 1425, 41919, 66507}},
	{6914, 17, 10145, {1, 3, 5, 11, 15, 11, 25, 255, 93, 33, 71, 2389, 1855, 317, 12773, 13311, 81927}},
	{6915, 17, 10148, {1, 3, 1, 3, 7, 55, 21, 175, 357, 235, 1679, 931, 2051, 14213, 20539, 38049, 122513}},
	{6916, 17, 10157, {1, 1, 5, 15, 5, 51, 127, 79, 297, 135, 1423, 2783, 7229, 14451, 27619, 7299, 49189}},
	{6917, 17, 10158, {1, 1, 1, 3, 5, 13, 9, 209, 455, 483, 1745, 323, 789, 7645, 26373, 61659, 23671}},
	{6918, 17, 10163, {1, 1, 1, 9, 23, 63, 99, 91, 377, 275, 275, 3005, 1563, 5945, 23825, 33211, 52753}},
	{6919, 17, 10180, {1, 1, 1, 1, 31, 55, 31, 109, 481, 581, 771, 197, 6155, 3465, 8451, 25925, 41159}},
	{6920, 17, 10187, {1, 3, 7, 13, 5, 33, 113, 161, 265, 493, 1723, 513, 5111, 10177, 21755, 5321, 58831}},
	{6921, 17, 10198, {1, 1, 7, 1, 21, 33, 117, 183, 89, 689, 1253, 2215, 6565, 3079, 16343, 22427, 96447}},
	{6922, 17, 10208, {1, 1, 1, 5, 15, 61, 5, 139, 111, 463, 573, 1907, 4615, 14975, 5715, 51017, 69827}},
	{6923, 17, 10214, {1, 1, 1, 13, 3, 3, 117, 249, 25, 361, 1177, 2901, 1601, 11381, 18981, 44811, 47117}},
	{6924, 17, 10220, {1, 1, 5, 3, 29, 5, 49, 221, 247, 57, 553, 1889, 479, 15581, 7035, 7293, 53065}},
	{6925, 17, 10237, {1, 3, 3, 3, 15, 49, 91, 187, 213, 981, 1417, 211, 3719, 13693, 17671, 16691, 57147}},
	{6926, 17, 10238, {1, 1, 7, 9, 7, 17, 109, 185, 459, 769, 1783, 899, 885, 2291, 30023, 26315, 7337}},
	{6927, 17, 10241, {1, 1, 5, 11, 11, 31, 73, 191, 95, 25, 1953, 1387, 1077, 7547, 9661, 57739, 76799}},
	{6928, 17, 10244, {1, 1, 7, 13, 23, 41, 69, 177, 407, 699, 1055, 3653, 1239, 8113, 12823, 1803, 117815}},
	{6929, 17, 10251, {1, 1, 1, 15, 1, 55, 71, 133, 401, 593, 605, 2855, 4569, 3533, 14141, 65457, 125655}},
	{6930, 17, 10253, {1, 1, 7, 9, 31, 55, 53, 11, 65, 17, 561, 925, 1561, 8929, 19859, 57111, 12777}},
	{6931, 17, 10256, {1, 3, 3, 11, 7, 59, 125, 205, 473, 655, 1429, 337, 6829, 7551, 27873, 11667, 39231}},
	{6932, 17, 10259, {1, 3, 3, 9, 13, 23, 25, 161, 443, 545, 1967, 1895, 6929, 5975, 17801, 41769, 30429}},
	{6933, 17, 10266, {1, 3, 7, 13, 15, 1, 99, 43, 45, 451, 21, 639, 7121, 4781, 2813, 419, 17761}},
	{6934, 17, 10284, {1, 1, 5, 13, 11, 9, 53, 83, 443, 441, 1601, 3177, 1913, 12211, 25835, 1733, 4793}},
	{6935, 17, 10290, {1, 3, 3, 1, 13, 15, 11, 187, 471, 699, 1751, 3279, 2305, 15259, 31541, 21357, 73763}},
	{6936, 17, 10331, {1, 3, 5, 9, 23, 11, 125, 57, 261, 479, 879, 719, 3221, 2943, 10593, 11521, 83979}},
	{6937, 17, 10334, {1, 3, 7, 13, 3, 39, 119, 135, 85, 417, 1675, 971, 7577, 12709, 20407, 26105, 97021}},
	{6938, 17, 10350, {1, 1, 5, 11, 15, 63, 83, 141, 281, 663, 1745, 2775, 5605, 9127, 553, 7177, 115969}},
	{6939, 17, 10355, {1, 1, 7, 1, 19, 47, 7, 165, 87, 95, 361, 1879, 6351, 2861, 9103, 37489, 24525}},
	{6940, 17, 10357, {1, 3, 3, 11, 9, 21, 51, 149, 375, 967, 1583, 1427, 1223, 11611, 7481, 36619, 128429}},
	{6941, 17, 10367, {1, 1, 5, 1, 3, 31, 7, 217, 453, 565, 1517, 2847, 6937, 1197, 24339, 44311, 66843}},
	{6942, 17, 10368, {1, 1, 5, 3, 3, 17, 127, 59, 3, 905, 531, 1179, 3559, 5175, 24627, 60941, 129457}},
	{6943, 17, 10377, {1, 1, 1, 7, 15, 15, 1, 31, 373, 643, 279, 3831, 4881, 9763, 17641, 43219, 83109}},
	{6944, 17, 10388, {1, 3, 3, 9, 5, 21, 41, 71, 371, 201, 573, 1481, 3631, 10783, 6679, 1089, 117347}},
	{6945, 17, 10407, {1, 1, 7, 7, 5, 25, 73, 63, 173, 197, 147, 981, 1491, 1597, 11733, 14285, 74021}},
	{6946, 17, 10421, {1, 1, 5, 11, 17, 15, 3, 175, 391, 503, 1745, 319, 791, 5607, 18173, 37319, 92025}},
	{6947, 17, 10434, {1, 3, 1, 1, 9, 37, 43, 81, 439, 951, 805, 251, 4625, 15617, 13715, 62263, 3827}},
	{6948, 17, 10439, {1, 3, 1, 1, 25, 21, 67, 191, 499, 205, 1355, 105, 1637, 563, 22291, 9045, 6545}},
	{6949, 17, 10440, {1, 1, 5, 5, 9, 3, 75, 75, 287, 303, 1767, 1789, 3437, 4637, 9605, 2537, 64935}},
	{6950, 17, 10443, {1, 1, 3, 3, 1, 51, 27, 155, 375, 149, 885, 187, 1551, 13109, 27011, 57301, 41047}},
	{6951, 17, 10446, {1, 1, 7, 5, 21, 23, 1, 81, 163, 231, 2039, 1519, 1279, 15379, 25549, 6711, 81499}},
	{6952, 17, 10457, {1, 1, 3, 5, 3, 37, 71, 243, 165, 365, 379, 351, 4649, 4287, 13395, 30329, 78383}},
	{6953, 17, 10469, {1, 3, 1, 1, 25, 63, 27, 215, 223, 699, 2029, 3737, 5947, 7287, 20813, 4931, 19345}},
	{6954, 17, 10476, {1, 1, 3, 15, 21, 7, 25, 187, 219, 53, 1749, 1797, 3533, 14307, 53, 11095, 75469}},
	{6955, 17, 10479, {1, 1, 3, 13, 27, 31, 91, 121, 481, 291, 915, 535, 4291, 5271, 12181, 55921, 125917}},
	{6956, 17, 10481, {1, 3, 1, 1, 3, 29, 21, 251, 361, 747, 997, 2989, 1809, 7235, 17855, 31027, 100689}},
	{6957, 17, 10494, {1, 3, 7, 1, 21, 13, 49, 93, 183, 673, 881, 1931, 7009, 2565, 26021, 53815, 19807}},
	{6958, 17, 10501, {1, 1, 7, 13, 9, 23, 47, 237, 487, 843, 1357, 919, 1753, 903, 2911, 31527, 73027}},
	{6959, 17, 10505, {1, 1, 1, 1, 25, 33, 97, 241, 421, 375, 73, 2541, 6231, 14659, 15335, 5915, 110791}},
	{6960, 17, 10516, {1, 3, 3, 7, 21, 17, 97, 125, 7, 271, 167, 475, 4887, 1847, 30173, 25913, 36659}},
	{6961, 17, 10532, {1, 1, 3, 15, 15, 37, 67, 5, 463, 423, 823, 941, 1551, 14175, 15377, 6017, 118297}},
	{6962, 17, 10541, {1, 1, 1, 7, 31, 51, 71, 127, 73, 517, 881, 3205, 6219, 11213, 14783, 64275, 70033}},
	{6963, 17, 10547, {1, 3, 1, 5, 17, 17, 57, 107, 359, 999, 1415, 757, 4743, 7775, 14111, 20075, 73269}},
	{6964, 17, 10550, {1, 3, 5, 3, 21, 57, 87, 43, 307, 777, 717, 3329, 4159, 12545, 31355, 31329, 41377}},
	{6965, 17, 10591, {1, 3, 7, 15, 25, 43, 19, 147, 487, 517, 977, 3625, 2311, 14173, 15167, 56563, 110417}},
	{6966, 17, 10597, {1, 3, 3, 11, 23, 1, 67, 157, 461, 169, 231, 1977, 5657, 865, 711, 24213, 76895}},
	{6967, 17, 10602, {1, 1, 7, 13, 5, 37, 51, 165, 331, 97, 431, 3819, 1379, 12083, 27521, 19689, 100119}},
	{6968, 17, 10610, {1, 1, 7, 15, 29, 21, 59, 193, 397, 467, 951, 3037, 2955, 13235, 20981, 63865, 30069}},
	{6969, 17, 10619, {1, 3, 3, 5, 7, 49, 41, 143, 319, 71, 353, 2159, 3043, 15317, 24095, 12017, 64393}},
	{6970, 17, 10631, {1, 1, 5, 13, 25, 45, 57, 153, 311, 805, 953, 1763, 5655, 3961, 12085, 58761, 76533}},
	{6971, 17, 10646, {1, 1, 3, 15, 29, 19, 71, 107, 203, 221, 1173, 1597, 1179, 9649, 21659, 10463, 8195}},
	{6972, 17, 10655, {1, 1, 3, 9, 31, 29, 53, 151, 247, 577, 543, 459, 8141, 5613, 12029, 24199, 118603}},
	{6973, 17, 10665, {1, 3, 1, 5, 1, 55, 103, 23, 405, 5, 181, 3805, 1103, 13389, 6725, 48733, 99639}},
	{6974, 17, 10673, {1, 1, 5, 9, 1, 47, 115, 231, 151, 885, 427, 2849, 361, 12969, 705, 41711, 53587}},
	{6975, 17, 10674, {1, 1, 3, 11, 9, 3, 11, 231, 77, 775, 657, 2721, 3431, 11919, 10425, 29405, 91561}},
	{6976, 17, 10680, {1, 1, 1, 5, 5, 7, 79, 41, 181, 333, 963, 3117, 7703, 2259, 16671, 51139, 27997}},
	{6977, 17, 10693, {1, 3, 7, 7, 13, 55, 59, 157, 377, 711, 1475, 1509, 1375, 6825, 13729, 28613, 109199}},
	{6978, 17, 10700, {1, 3, 3, 3, 13, 11, 51, 1, 67, 609, 467, 2161, 7693, 9019, 1847, 27969, 74863}},
	{6979, 17, 10721, {1, 1, 3, 3, 11, 33, 87, 217, 239, 505, 1451, 2801, 1417, 695, 29883, 15877, 99969}},
	{6980, 17, 10727, {1, 3, 3, 5, 3, 61, 9, 171, 57, 547, 2003, 2335, 2259, 3205, 5639, 21721, 25893}},
	{6981, 17, 10746, {1, 3, 1, 3, 19, 15, 83, 69, 47, 897, 627, 2839, 7123, 8567, 14707, 13159, 125139}},
	{6982, 17, 10748, {1, 3, 7, 11, 1, 59, 53, 33, 135, 1009, 1829, 3011, 1245, 421, 28909, 45517, 55071}},
	{6983, 17, 10757, {1, 1, 5, 9, 3, 27, 11, 243, 235, 683, 1329, 3145, 2141, 14027, 3707, 5933, 51965}},
	{6984, 17, 10761, {1, 1, 5, 7, 13, 63, 79, 105, 27, 195, 1657, 3107, 1245, 1681, 29619, 10589, 78197}},
	{6985, 17, 10770, {1, 3, 3, 7, 21, 1, 5, 79, 73, 125, 1587, 3053, 5977, 10745, 28343, 39023, 56201}},
	{6986, 17, 10776, {1, 1, 3, 15, 23, 21, 39, 41, 173, 913, 1267, 1323, 2967, 1979, 16763, 53753, 21905}},
	{6987, 17, 10782, {1, 1, 5, 7, 11, 11, 117, 151, 409, 345, 1461, 1703, 687, 557, 31651, 35507, 54909}},
	{6988, 17, 10791, {1, 1, 1, 15, 15, 49, 55, 223, 289, 765, 1737, 1117, 3717, 15465, 31949, 55061, 97091}},
	{6989, 17, 10792, {1, 1, 5, 9, 21, 29, 99, 13, 119, 35, 1461, 61, 5155, 6785, 15957, 11295, 52203}},
	{6990, 17, 10805, {1, 3, 5, 7, 23, 39, 73, 161, 465, 715, 153, 3529, 2243, 13773, 16573, 26233, 130263}},
	{6991, 17, 10810, {1, 3, 7, 9, 11, 51, 5, 149, 501, 119, 2047, 3417, 3955, 15055, 31633, 473, 127305}},
	{6992, 17, 10832, {1, 1, 1, 9, 31, 57, 91, 119, 215, 11, 1013, 3969, 1285, 11521, 8039, 36737, 86365}},
	{6993, 17, 10835, {1, 1, 5, 3, 7, 17, 9, 27, 59, 883, 541, 3027, 6219, 1091, 2453, 38247, 21323}},
	{6994, 17, 10841, {1, 1, 1, 1, 25, 39, 55, 249, 61, 313, 467, 1763, 4067, 8367, 32431, 44463, 66439}},
	{6995, 17, 10842, {1, 3, 3, 1, 13, 3, 37, 209, 21, 653, 1971, 3649, 6165, 3789, 12793, 56327, 60351}},
	{6996, 17, 10847, {1, 1, 7, 9, 31, 33, 21, 51, 313, 631, 515, 1761, 4149, 2601, 12481, 25975, 94061}},
	{6997, 17, 10853, {1, 1, 7, 15, 3, 7, 55, 129, 297, 735, 779, 633, 3265, 11713, 3893, 61197, 113991}},
	{6998, 17, 10860, {1, 3, 5, 13, 1, 15, 27, 253, 435, 595, 1163, 2753, 7399, 15225, 26215, 59753, 74933}},
	{6999, 17, 10871, {1, 1, 7, 7, 15, 23, 111, 43, 467, 957, 1687, 2893, 2315, 2025, 1475, 9061, 101611}},
	{7000, 17, 10878, {1, 1, 3, 3, 29, 41, 53, 169, 125, 415, 361, 869, 3399, 8821, 18193, 38575, 73979}},
	{7001, 17, 10881, {1, 1, 1, 15, 3, 5, 27, 5, 293, 765, 1809, 1961, 955, 12441, 10915, 2363, 49617}},
	{7002, 17, 10888, {1, 1, 5, 15, 19, 11, 3, 91, 59, 323, 545, 1177, 7967, 2729, 14085, 3283, 79859}},
	{7003, 17, 10894, {1, 1, 7, 13, 11, 17, 29, 163, 295, 951, 311, 3471, 1339, 10719, 701, 32377, 41685}},
	{7004, 17, 10901, {1, 3, 5, 7, 21, 19, 81, 247, 495, 767, 251, 3455, 6383, 7221, 19943, 64865, 33193}},
	{7005, 17, 10915, {1, 1, 7, 15, 23, 41, 63, 195, 311, 619, 211, 743, 889, 7627, 12527, 15865, 40103}},
	{7006, 17, 10918, {1, 1, 3, 1, 23, 23, 57, 221, 153, 27, 939, 3949, 411, 6357, 31985, 939, 91001}},
	{7007, 17, 10922, {1, 3, 5, 15, 7, 5, 35, 135, 245, 921, 307, 823, 775, 4891, 24575, 53503, 48147}},
	{7008, 17, 10936, {1, 1, 5, 7, 9, 31, 23, 139, 477, 495, 287, 807, 1855, 8321, 13963, 52197, 78509}},
	{7009, 17, 10954, {1, 3, 3, 3, 29, 59, 33, 83, 211, 65, 623, 1269, 1745, 16383, 10759, 57199, 14035}},
	{7010, 17, 10968, {1, 3, 3, 15, 25, 55, 69, 171, 411, 937, 731, 2275, 2597, 4133, 5089, 50507, 39989}},
	{7011, 17, 10971, {1, 3, 1, 9, 5, 47, 51, 21, 171, 913, 233, 43, 2673, 471, 27077, 57039, 32579}},
	{7012, 17, 10973, {1, 3, 5, 3, 29, 35, 5, 105, 233, 379, 77, 1775, 2409, 4597, 19879, 12691, 49739}},
	{7013, 17, 10978, {1, 3, 7, 13, 17, 29, 117, 177, 163, 927, 45, 3227, 7263, 5551, 9219, 32101, 122473}},
	{7014, 17, 10998, {1, 1, 7, 5, 31, 39, 75, 147, 311, 991, 1431, 3821, 6891, 9637, 17887, 661, 23067}},
	{7015, 17, 11009, {1, 3, 5, 13, 31, 53, 69, 79, 153, 329, 207, 479, 2395, 6505, 29553, 52023, 31531}},
	{7016, 17, 11021, {1, 3, 1, 7, 15, 7, 87, 233, 25, 275, 981, 1207, 3083, 16349, 30185, 60611, 120607}},
	{7017, 17, 11029, {1, 1, 5, 3, 21, 7, 47, 173, 291, 965, 65, 545, 7465, 4471, 2249, 34281, 107217}},
	{7018, 17, 11030, {1, 1, 3, 11, 19, 53, 17, 243, 193, 297, 1937, 1513, 4979, 14867, 15497, 10049, 9135}},
	{7019, 17, 11034, {1, 3, 1, 3, 25, 39, 29, 63, 231, 145, 247, 1745, 3439, 8635, 26687, 18595, 67123}},
	{7020, 17, 11050, {1, 1, 7, 9, 29, 33, 89, 175, 429, 675, 891, 1739, 3567, 5453, 30427, 33671, 83395}},
	{7021, 17, 11063, {1, 3, 1, 5, 31, 25, 69, 237, 235, 307, 1217, 3805, 153, 13387, 6209, 14179, 128725}},
	{7022, 17, 11064, {1, 1, 3, 3, 19, 45, 117, 135, 67, 601, 369, 3369, 5505, 2049, 24099, 22515, 96575}},
	{7023, 17, 11077, {1, 1, 1, 3, 3, 45, 29, 255, 327, 77, 1103, 4067, 2875, 6487, 5903, 26625, 19631}},
	{7024, 17, 11078, {1, 3, 5, 1, 31, 63, 115, 7, 255, 855, 913, 1779, 7001, 14387, 26765, 51987, 3191}},
	{7025, 17, 11105, {1, 1, 3, 11, 15, 43, 71, 247, 303, 231, 445, 3963, 3699, 11851, 18941, 43465, 63431}},
	{7026, 17, 11106, {1, 1, 3, 5, 31, 33, 93, 127, 267, 399, 653, 1997, 5005, 14535, 4813, 64065, 2159}},
	{7027, 17, 11126, {1, 3, 7, 13, 31, 39, 61, 155, 141, 515, 1217, 161, 4309, 3697, 22445, 43599, 43329}},
	{7028, 17, 11129, {1, 3, 3, 3, 7, 51, 103, 147, 511, 971, 195, 3731, 6629, 12125, 12053, 34951, 60059}},
	{7029, 17, 11135, {1, 1, 5, 11, 21, 49, 99, 31, 55, 309, 1805, 2253, 7095, 15265, 28445, 54813, 48615}},
	{7030, 17, 11151, {1, 3, 1, 15, 9, 41, 61, 125, 65, 143, 1567, 3259, 6757, 653, 31601, 63127, 52179}},
	{7031, 17, 11159, {1, 1, 5, 3, 29, 5, 19, 197, 153, 447, 7, 1713, 469, 6043, 1259, 63641, 29171}},
	{7032, 17, 11165, {1, 3, 3, 7, 3, 41, 95, 245, 445, 15, 607, 565, 2361, 2673, 21077, 20153, 6199}},
	{7033, 17, 11176, {1, 1, 5, 1, 5, 59, 93, 127, 485, 663, 683, 635, 1599, 16377, 31819, 6539, 27789}},
	{7034, 17, 11179, {1, 3, 1, 3, 31, 3, 11, 215, 441, 1005, 1815, 3945, 5109, 5539, 23935, 62671, 90731}},
	{7035, 17, 11182, {1, 3, 3, 1, 13, 47, 19, 229, 191, 427, 1141, 2321, 7105, 1587, 26347, 63265, 23377}},
	{7036, 17, 11189, {1, 1, 5, 15, 31, 55, 61, 93, 89, 945, 1203, 3631, 4457, 15097, 32019, 41747, 46009}},
	{7037, 17, 11204, {1, 1, 5, 13, 5, 33, 69, 59, 93, 247, 503, 421, 1923, 9855, 9825, 14257, 98663}},
	{7038, 17, 11216, {1, 3, 1, 13, 27, 21, 91, 39, 131, 571, 1527, 2715, 2061, 627, 19705, 47165, 84345}},
	{7039, 17, 11232, {1, 1, 1, 7, 3, 3, 7, 251, 225, 959, 1017, 2423, 6163, 1549, 7473, 3193, 104259}},
	{7040, 17, 11235, {1, 3, 3, 1, 5, 5, 115, 221, 505, 649, 1525, 2459, 167, 1899, 23939, 29253, 122835}},
	{7041, 17, 11242, {1, 3, 1, 5, 15, 9, 123, 221, 133, 43, 31, 1211, 4737, 5001, 20065, 6369, 93865}},
	{7042, 17, 11250, {1, 1, 5, 11, 11, 5, 23, 29, 333, 133, 1469, 1895, 5879, 15599, 2131, 25005, 96271}},
	{7043, 17, 11264, {1, 1, 3, 11, 25, 11, 19, 57, 397, 645, 1233, 2433, 6371, 10577, 15489, 60709, 3957}},
	{7044, 17, 11273, {1, 3, 1, 1, 19, 3, 33, 131, 429, 835, 1363, 2213, 3185, 14385, 8831, 43159, 32975}},
	{7045, 17, 11282, {1, 1, 5, 5, 23, 11, 127, 139, 213, 259, 897, 1913, 5737, 1287, 26617, 4885, 30193}},
	{7046, 17, 11288, {1, 3, 5, 13, 3, 27, 99, 31, 11, 27, 1003, 2473, 7055, 12923, 4269, 41433, 90637}},
	{7047, 17, 11291, {1, 3, 1, 7, 17, 25, 95, 151, 199, 237, 207, 1879, 2943, 9845, 3765, 53533, 111191}},
	{7048, 17, 11293, {1, 3, 1, 9, 19, 27, 5, 249, 85, 185, 1883, 1401, 2041, 12721, 20593, 30993, 2601}},
	{7049, 17, 11318, {1, 3, 3, 9, 23, 1, 15, 133, 387, 779, 707, 2723, 4485, 989, 27125, 37295, 125319}},
	{7050, 17, 11321, {1, 1, 7, 3, 9, 41, 81, 151, 349, 941, 357, 3817, 7123, 10079, 27519, 107, 102281}},
	{7051, 17, 11329, {1, 1, 1, 1, 13, 5, 111, 167, 73, 85, 1185, 1213, 333, 153, 13101, 38087, 39389}},
	{7052, 17, 11336, {1, 3, 3, 15, 11, 41, 99, 231, 377, 539, 1335, 1059, 5373, 9611, 27927, 29801, 85749}},
	{7053, 17, 11339, {1, 3, 1, 9, 19, 37, 125, 27, 15, 699, 1867, 2711, 1589, 1675, 32007, 61339, 96919}},
	{7054, 17, 11342, {1, 3, 3, 3, 3, 27, 21, 159, 249, 783, 1517, 2923, 2609, 1207, 13705, 57371, 43603}},
	{7055, 17, 11356, {1, 1, 5, 15, 17, 55, 77, 1, 401, 897, 987, 345, 5283, 5827, 17755, 44371, 13253}},
	{7056, 17, 11383, {1, 3, 1, 7, 3, 3, 99, 237, 487, 405, 771, 3503, 1199, 4779, 26893, 45821, 46383}},
	{7057, 17, 11389, {1, 1, 7, 3, 9, 47, 81, 27, 459, 989, 1891, 3997, 4081, 4075, 15079, 65081, 125185}},
	{7058, 17, 11405, {1, 3, 5, 9, 25, 23, 71, 251, 251, 197, 353, 3553, 2165, 2953, 3733, 52369, 100641}},
	{7059, 17, 11411, {1, 1, 1, 5, 25, 43, 63, 187, 495, 345, 1547, 2293, 7327, 7797, 14001, 61865, 40329}},
	{7060, 17, 11423, {1, 1, 5, 15, 25, 37, 67, 23, 315, 801, 71, 1235, 7293, 7207, 30929, 9417, 94735}},
	{7061, 17, 11424, {1, 1, 1, 3, 23, 29, 87, 171, 337, 457, 1597, 3933, 4151, 1237, 19563, 56997, 81497}},
	{7062, 17, 11430, {1, 3, 3, 11, 3, 33, 79, 239, 277, 611, 205, 2283, 7459, 425, 21999, 26491, 58681}},
	{7063, 17, 11444, {1, 1, 7, 1, 5, 37, 53, 93, 205, 97, 779, 3623, 7777, 521, 21915, 46539, 128811}},
	{7064, 17, 11447, {1, 1, 5, 7, 19, 7, 39, 183, 299, 193, 1351, 3867, 5709, 11655, 1231, 15555, 128023}},
	{7065, 17, 11459, {1, 3, 7, 11, 31, 13, 113, 57, 197, 841, 921, 2087, 2195, 8279, 8353, 1955, 22121}},
	{7066, 17, 11465, {1, 3, 3, 11, 21, 55, 61, 105, 357, 747, 363, 3511, 2547, 16283, 25747, 56041, 33695}},
	{7067, 17, 11473, {1, 3, 3, 13, 27, 13, 5, 27, 93, 691, 1869, 2331, 3131, 14411, 2355, 37195, 129273}},
	{7068, 17, 11479, {1, 3, 3, 7, 27, 9, 11, 165, 435, 811, 215, 1617, 347, 4289, 29373, 15749, 91445}},
	{7069, 17, 11501, {1, 1, 7, 13, 29, 3, 95, 53, 457, 633, 959, 3705, 7461, 9307, 21963, 51599, 6751}},
	{7070, 17, 11510, {1, 1, 1, 15, 29, 25, 95, 1, 125, 61, 683, 2067, 6485, 9095, 5571, 61281, 70865}},
	{7071, 17, 11514, {1, 1, 7, 7, 1, 35, 119, 107, 247, 991, 237, 1865, 3961, 12583, 11417, 14913, 90897}},
	{7072, 17, 11522, {1, 3, 7, 15, 11, 51, 73, 193, 289, 381, 1767, 3803, 3197, 3797, 15059, 19393, 98947}},
	{7073, 17, 11527, {1, 1, 5, 3, 13, 7, 91, 223, 347, 59, 1721, 1501, 6391, 4141, 14495, 47283, 47237}},
	{7074, 17, 11533, {1, 3, 7, 11, 17, 39, 43, 247, 35, 423, 1859, 3199, 5343, 7061, 8609, 6819, 88575}},
	{7075, 17, 11536, {1, 1, 5, 13, 31, 27, 57, 19, 499, 1007, 1965, 795, 1231, 12755, 24631, 53343, 82305}},
	{7076, 17, 11548, {1, 1, 1, 9, 13, 23, 127, 161, 245, 467, 2025, 2545, 3085, 13035, 27087, 14461, 35971}},
	{7077, 17, 11551, {1, 3, 5, 1, 7, 3, 99, 159, 75, 341, 1755, 2337, 5981, 5055, 19445, 30043, 61427}},
	{7078, 17, 11552, {1, 1, 1, 7, 13, 33, 41, 73, 267, 21, 961, 3509, 6839, 13215, 8471, 46735, 93071}},
	{7079, 17, 11555, {1, 3, 7, 7, 3, 25, 81, 239, 357, 445, 1483, 389, 3891, 5131, 21357, 34757, 111063}},
	{7080, 17, 11572, {1, 3, 7, 1, 1, 37, 119, 121, 195, 935, 1711, 2049, 7001, 7117, 9957, 7309, 102293}},
	{7081, 17, 11587, {1, 1, 7, 11, 1, 49, 107, 95, 149, 329, 289, 1121, 7217, 15091, 19071, 13801, 13}},
	{7082, 17, 11601, {1, 1, 1, 13, 17, 17, 7, 105, 81, 1017, 1867, 1567, 5133, 7325, 19797, 16301, 40471}},
	{7083, 17, 11613, {1, 3, 5, 5, 27, 45, 117, 135, 499, 53, 973, 121, 53, 8771, 11893, 35827, 57691}},
	{7084, 17, 11614, {1, 1, 1, 1, 7, 23, 11, 163, 17, 871, 129, 2959, 5583, 12253, 1419, 28367, 32539}},
	{7085, 17, 11618, {1, 1, 3, 5, 23, 31, 127, 33, 115, 799, 331, 1873, 1729, 1383, 23601, 51145, 72027}},
	{7086, 17, 11624, {1, 1, 1, 9, 15, 49, 105, 163, 51, 539, 451, 3983, 6509, 1073, 30757, 13971, 51371}},
	{7087, 17, 11630, {1, 1, 7, 1, 1, 57, 71, 135, 5, 171, 983, 951, 777, 9257, 3607, 3239, 76237}},
	{7088, 17, 11663, {1, 1, 7, 7, 21, 17, 49, 175, 9, 807, 289, 2777, 7309, 14911, 28349, 43871, 96019}},
	{7089, 17, 11682, {1, 3, 1, 13, 5, 7, 83, 215, 297, 319, 347, 633, 7285, 8293, 18811, 31065, 114077}},
	{7090, 17, 11684, {1, 3, 1, 11, 3, 29, 91, 231, 161, 601, 355, 2719, 2941, 6065, 21849, 58051, 46515}},
	{7091, 17, 11702, {1, 1, 3, 9, 25, 41, 111, 135, 71, 755, 29, 131, 1339, 5053, 15713, 14557, 106777}},
	{7092, 17, 11705, {1, 1, 7, 13, 21, 59, 13, 45, 503, 71, 1611, 4021, 2359, 11653, 7261, 14537, 33031}},
	{7093, 17, 11713, {1, 1, 1, 11, 5, 31, 1, 181, 37, 527, 1345, 1979, 4899, 3289, 25181, 49959, 44609}},
	{7094, 17, 11731, {1, 3, 3, 13, 21, 25, 33, 105, 57, 637, 841, 1595, 3881, 5053, 9441, 58717, 127255}},
	{7095, 17, 11734, {1, 3, 5, 7, 23, 57, 9, 117, 281, 769, 1573, 2857, 1139, 6413, 14001, 21097, 55215}},
	{7096, 17, 11740, {1, 1, 7, 7, 3, 5, 75, 149, 269, 353, 437, 61, 2451, 11987, 17243, 5649, 105107}},
	{7097, 17, 11747, {1, 1, 3, 3, 25, 61, 53, 21, 113, 57, 1415, 2825, 11, 14977, 6159, 4181, 96765}},
	{7098, 17, 11754, {1, 1, 7, 5, 15, 25, 121, 159, 71, 773, 601, 147, 6507, 16171, 16607, 32017, 77845}},
	{7099, 17, 11761, {1, 3, 1, 1, 27, 19, 59, 109, 347, 991, 165, 683, 6147, 493, 22017, 19069, 52857}},
	{7100, 17, 11762, {1, 1, 5, 5, 21, 1, 93, 115, 407, 15, 421, 1305, 3495, 14287, 31831, 65347, 35339}},
	{7101, 17, 11787, {1, 3, 5, 11, 29, 35, 87, 27, 453, 769, 1991, 2757, 2607, 9225, 293, 49441, 18185}},
	{7102, 17, 11792, {1, 1, 5, 3, 23, 41, 67, 195, 499, 903, 197, 1121, 4691, 9277, 29225, 34597, 37395}},
	{7103, 17, 11814, {1, 1, 7, 7, 21, 7, 65, 245, 241, 909, 1063, 2271, 1979, 10287, 1747, 61523, 72969}},
	{7104, 17, 11823, {1, 3, 1, 13, 23, 25, 3, 89, 385, 481, 1463, 3431, 6907, 1129, 3519, 35789, 82585}},
	{7105, 17, 11825, {1, 3, 5, 3, 31, 17, 11, 209, 77, 991, 885, 3341, 6895, 3429, 21611, 38555, 35475}},
	{7106, 17, 11837, {1, 1, 3, 1, 9, 61, 27, 219, 433, 787, 281, 1155, 2915, 4449, 30881, 34461, 15357}},
	{7107, 17, 11838, {1, 1, 3, 15, 27, 55, 51, 101, 117, 799, 1475, 4013, 5145, 14991, 27847, 49537, 57339}},
	{7108, 17, 11846, {1, 3, 7, 13, 13, 9, 13, 167, 283, 883, 1501, 2635, 1463, 3353, 14961, 30349, 62043}},
	{7109, 17, 11855, {1, 1, 7, 3, 3, 47, 119, 37, 389, 655, 701, 2471, 5749, 6645, 845, 27065, 82299}},
	{7110, 17, 11864, {1, 1, 7, 15, 27, 5, 95, 195, 227, 991, 1137, 3715, 4901, 9459, 1917, 43857, 126505}},
	{7111, 17, 11876, {1, 3, 7, 5, 29, 35, 45, 165, 361, 257, 641, 1265, 6533, 11333, 26081, 12621, 66909}},
	{7112, 17, 11885, {1, 1, 1, 11, 19, 55, 73, 137, 29, 355, 725, 1161, 6717, 2035, 19769, 43531, 72577}},
	{7113, 17, 11904, {1, 3, 7, 5, 19, 3, 99, 17, 387, 621, 137, 117, 6567, 7667, 14979, 17981, 68319}},
	{7114, 17, 11909, {1, 1, 5, 5, 7, 53, 31, 33, 245, 371, 691, 2763, 95, 16369, 7853, 29839, 34957}},
	{7115, 17, 11913, {1, 1, 3, 1, 9, 1, 83, 177, 17, 141, 1739, 1791, 3849, 3093, 22271, 53755, 7817}},
	{7116, 17, 11916, {1, 3, 3, 1, 3, 51, 71, 69, 439, 987, 807, 3353, 4747, 16031, 29591, 61091, 95675}},
	{7117, 17, 11940, {1, 3, 5, 1, 17, 47, 51, 211, 7, 5, 1751, 1735, 1647, 13389, 13861, 49427, 13577}},
	{7118, 17, 11943, {1, 3, 7, 5, 11, 23, 17, 55, 11, 61, 809, 927, 6533, 1509, 29261, 21555, 55075}},
	{7119, 17, 11972, {1, 3, 1, 1, 15, 51, 37, 47, 183, 117, 597, 3225, 1435, 13359, 19127, 17339, 17345}},
	{7120, 17, 11981, {1, 1, 5, 3, 5, 11, 33, 179, 295, 129, 29, 713, 1561, 27, 21087, 50305, 39253}},
	{7121, 17, 11990, {1, 1, 5, 7, 17, 25, 105, 241, 41, 915, 1223, 2625, 617, 10983, 10749, 2137, 101831}},
	{7122, 17, 11993, {1, 3, 5, 7, 15, 15, 85, 23, 193, 625, 1803, 2903, 1935, 523, 8377, 12165, 105851}},
	{7123, 17, 12000, {1, 3, 3, 7, 3, 35, 5, 107, 191, 855, 405, 1659, 5523, 5011, 6401, 45187, 31345}},
	{7124, 17, 12005, {1, 3, 3, 1, 9, 21, 103, 75, 501, 669, 547, 3685, 411, 2663, 14743, 13869, 124389}},
	{7125, 17, 12015, {1, 3, 5, 13, 15, 37, 39, 79, 19, 165, 1685, 1367, 5951, 12303, 13423, 51083, 119933}},
	{7126, 17, 12020, {1, 1, 3, 1, 7, 25, 1, 221, 415, 591, 859, 1457, 1789, 2269, 15947, 31913, 86397}},
	{7127, 17, 12038, {1, 3, 7, 15, 11, 49, 15, 171, 45, 925, 407, 1719, 4505, 5695, 17397, 28849, 77}},
	{7128, 17, 12042, {1, 1, 3, 11, 21, 33, 91, 115, 263, 141, 753, 3335, 7695, 1981, 6029, 22629, 2467}},
	{7129, 17, 12056, {1, 3, 5, 3, 25, 5, 21, 67, 429, 323, 223, 2395, 761, 14817, 12387, 37905, 19551}},
	{7130, 17, 12065, {1, 3, 1, 15, 31, 43, 35, 255, 73, 533, 1093, 965, 557, 607, 6913, 35283, 12261}},
	{7131, 17, 12066, {1, 3, 1, 15, 25, 13, 39, 83, 77, 269, 1205, 1577, 4095, 6669, 8643, 48807, 98227}},
	{7132, 17, 12072, {1, 3, 3, 7, 31, 57, 25, 177, 441, 973, 1255, 675, 5579, 4899, 27925, 52555, 70845}},
	{7133, 17, 12080, {1, 3, 1, 5, 13, 47, 15, 75, 387, 461, 1909, 841, 7, 9567, 913, 41411, 12565}},
	{7134, 17, 12083, {1, 1, 5, 7, 5, 21, 17, 189, 319, 645, 403, 2723, 6747, 15471, 26533, 12709, 49417}},
	{7135, 17, 12090, {1, 1, 5, 7, 7, 41, 99, 179, 137, 435, 1061, 3987, 4583, 4101, 23781, 54263, 36695}},
	{7136, 17, 12092, {1, 3, 1, 11, 19, 37, 125, 177, 111, 921, 1003, 1433, 1399, 3991, 28193, 40471, 97041}},
	{7137, 17, 12103, {1, 1, 7, 1, 5, 33, 7, 139, 127, 413, 1171, 2237, 265, 10145, 18793, 28957, 25037}},
	{7138, 17, 12109, {1, 3, 1, 1, 25, 37, 13, 17, 471, 195, 1645, 3165, 5635, 8433, 28507, 453, 107709}},
	{7139, 17, 12112, {1, 3, 3, 11, 1, 55, 119, 97, 243, 371, 95, 97, 7833, 777, 12177, 1861, 56323}},
	{7140, 17, 12117, {1, 1, 7, 5, 7, 29, 59, 219, 405, 411, 275, 111, 4899, 10367, 24331, 57295, 47065}},
	{7141, 17, 12121, {1, 1, 3, 3, 19, 23, 91, 111, 221, 195, 1013, 3001, 3227, 6359, 30383, 49699, 49157}},
	{7142, 17, 12137, {1, 1, 5, 7, 1, 21, 125, 23, 177, 291, 249, 861, 1899, 14101, 5079, 5211, 14373}},
	{7143, 17, 12143, {1, 1, 7, 11, 11, 59, 33, 41, 291, 919, 253, 609, 1657, 14633, 15189, 22245, 99815}},
	{7144, 17, 12145, {1, 3, 5, 3, 23, 49, 71, 137, 393, 343, 1845, 343, 5853, 6639, 17435, 62143, 76041}},
	{7145, 17, 12148, {1, 1, 5, 3, 9, 27, 55, 193, 25, 965, 1453, 2739, 3785, 12497, 29607, 11111, 25145}},
	{7146, 17, 12168, {1, 1, 1, 1, 29, 11, 111, 73, 491, 629, 405, 2779, 5313, 589, 1459, 47555, 67945}},
	{7147, 17, 12174, {1, 3, 1, 7, 13, 21, 99, 75, 79, 963, 207, 1725, 6875, 8359, 10573, 45219, 130463}},
	{7148, 17, 12188, {1, 3, 7, 13, 1, 17, 105, 227, 487, 891, 1053, 1333, 7651, 5415, 18661, 22085, 82055}},
	{7149, 17, 12191, {1, 1, 3, 3, 31, 27, 91, 93, 383, 331, 965, 3035, 4931, 13265, 9729, 28985, 118227}},
	{7150, 17, 12192, {1, 3, 1, 1, 11, 9, 59, 191, 253, 909, 301, 3811, 255, 14937, 28627, 54509, 95993}},
	{7151, 17, 12201, {1, 3, 3, 5, 11, 5, 105, 77, 323, 713, 637, 1857, 2697, 12473, 12261, 2933, 101287}},
	{7152, 17, 12224, {1, 3, 3, 11, 9, 63, 19, 19, 213, 859, 1479, 2849, 1067, 5749, 13511, 14933, 11125}},
	{7153, 17, 12230, {1, 1, 5, 9, 19, 19, 13, 49, 237, 511, 533, 543, 575, 8095, 27335, 18847, 18173}},
	{7154, 17, 12239, {1, 3, 5, 5, 9, 53, 47, 157, 35, 827, 637, 2327, 787, 5269, 5145, 10135, 111273}},
	{7155, 17, 12242, {1, 3, 3, 7, 27, 41, 69, 173, 53, 655, 809, 481, 6999, 3101, 20781, 2481, 94957}},
	{7156, 17, 12251, {1, 1, 5, 11, 17, 23, 95, 201, 363, 613, 863, 1365, 1131, 15417, 20705, 8283, 55235}},
	{7157, 17, 12258, {1, 1, 5, 13, 3, 15, 37, 219, 291, 595, 1665, 1861, 1953, 15385, 20569, 46085, 15163}},
	{7158, 17, 12264, {1, 3, 3, 11, 23, 43, 125, 133, 85, 45, 819, 243, 7325, 8723, 1499, 58139, 120353}},
	{7159, 17, 12310, {1, 1, 1, 11, 21, 49, 91, 145, 175, 619, 1817, 3533, 8155, 7521, 30361, 45431, 130175}},
	{7160, 17, 12319, {1, 1, 3, 1, 11, 59, 57, 51, 37, 903, 1221, 3813, 8043, 14165, 31503, 7905, 61515}},
	{7161, 17, 12323, {1, 1, 1, 1, 15, 9, 115, 175, 285, 839, 97, 3119, 719, 15283, 22947, 25417, 40665}},
	{7162, 17, 12325, {1, 3, 1, 7, 5, 49, 127, 111, 373, 747, 393, 2351, 4577, 15227, 23149, 16901, 80253}},
	{7163, 17, 12332, {1, 1, 5, 3, 15, 5, 95, 197, 251, 275, 831, 1389, 3907, 12343, 11599, 24369, 65361}},
	{7164, 17, 12343, {1, 3, 7, 5, 25, 37, 11, 75, 417, 789, 745, 811, 2189, 15381, 4785, 41657, 2897}},
	{7165, 17, 12344, {1, 3, 1, 13, 29, 55, 55, 33, 279, 383, 1645, 975, 4683, 1357, 1149, 30271, 90527}},
	{7166, 17, 12352, {1, 3, 5, 3, 5, 3, 79, 61, 371, 225, 141, 369, 1037, 12249, 29431, 37253, 9899}},
	{7167, 17, 12370, {1, 1, 3, 13, 13, 7, 127, 147, 507, 119, 1085, 1949, 6289, 10179, 10107, 55989, 74395}},
	{7168, 17, 12388, {1, 3, 1, 7, 21, 35, 53, 209, 103, 365, 683, 553, 4977, 14371, 24037, 11453, 45369}},
	{7169, 17, 12395, {1, 1, 5, 11, 27, 39, 41, 145, 437, 55, 893, 2375, 4977, 5451, 21225, 46815, 1423}},
	{7170, 17, 12403, {1, 3, 5, 1, 23, 53, 113, 75, 209, 323, 1975, 3809, 1829, 14625, 3821, 53773, 129173}},
	{7171, 17, 12409, {1, 1, 5, 3, 7, 51, 97, 73, 289, 481, 339, 1375, 3101, 4395, 13933, 33267, 68115}},
	{7172, 17, 12410, {1, 3, 5, 1, 5, 45, 83, 57, 3, 667, 109, 3979, 6447, 8603, 20147, 49291, 18023}},
	{7173, 17, 12415, {1, 3, 7, 1, 11, 7, 45, 233, 65, 745, 1009, 2979, 5965, 10681, 3499, 23077, 87479}},
	{7174, 17, 12419, {1, 3, 3, 3, 13, 25, 25, 189, 197, 83, 1429, 2857, 2877, 8577, 24811, 33049, 46009}},
	{7175, 17, 12426, {1, 1, 1, 7, 11, 47, 47, 255, 89, 625, 449, 3747, 2035, 3509, 4901, 2961, 14073}},
	{7176, 17, 12439, {1, 1, 1, 13, 9, 55, 35, 47, 389, 573, 847, 1037, 1345, 5487, 7575, 57435, 77303}},
	{7177, 17, 12445, {1, 1, 5, 11, 25, 51, 113, 109, 79, 339, 95, 2049, 5881, 13209, 20041, 26419, 110319}},
	{7178, 17, 12459, {1, 1, 7, 1, 27, 15, 93, 145, 253, 917, 1211, 2221, 1087, 14209, 32097, 20083, 67841}},
	{7179, 17, 12464, {1, 1, 3, 15, 13, 19, 67, 107, 75, 919, 2047, 3675, 6231, 1243, 14335, 35939, 17281}},
	{7180, 17, 12474, {1, 3, 7, 5, 27, 47, 53, 239, 475, 231, 1645, 825, 4039, 15985, 10853, 32951, 34985}},
	{7181, 17, 12484, {1, 1, 7, 5, 15, 61, 107, 93, 51, 221, 717, 2859, 7885, 9571, 11841, 45143, 33723}},
	{7182, 17, 12491, {1, 1, 7, 7, 9, 25, 63, 25, 47, 55, 2041, 3965, 215, 14857, 31669, 54775, 42157}},
	{7183, 17, 12501, {1, 3, 5, 1, 5, 45, 123, 109, 471, 599, 479, 475, 3499, 11963, 23709, 18851, 66861}},
	{7184, 17, 12505, {1, 3, 3, 3, 5, 29, 71, 81, 315, 329, 1471, 3995, 623, 5871, 11171, 15645, 97251}},
	{7185, 17, 12508, {1, 1, 7, 11, 15, 15, 101, 173, 445, 871, 765, 1121, 1937, 13055, 7309, 54175, 85559}},
	{7186, 17, 12511, {1, 3, 5, 7, 7, 13, 43, 237, 361, 981, 19, 3113, 4681, 3313, 19147, 35193, 87281}},
	{7187, 17, 12521, {1, 3, 5, 3, 27, 13, 37, 51, 233, 573, 1599, 2807, 7149, 12083, 28927, 7797, 130879}},
	{7188, 17, 12522, {1, 1, 1, 13, 31, 63, 127, 89, 209, 717, 1075, 3887, 1427, 87, 18565, 39973, 55025}},
	{7189, 17, 12530, {1, 3, 1, 5, 15, 11, 121, 247, 273, 613, 1857, 2059, 7399, 13951, 9025, 39523, 68121}},
	{7190, 17, 12544, {1, 3, 7, 13, 31, 9, 61, 143, 375, 433, 471, 1315, 5299, 1167, 10099, 11445, 51693}},
	{7191, 17, 12547, {1, 1, 7, 9, 25, 31, 125, 5, 13, 595, 621, 3551, 7959, 10643, 14345, 37683, 118377}},
	{7192, 17, 12561, {1, 1, 5, 11, 1, 33, 45, 31, 447, 229, 893, 3777, 4101, 2505, 4855, 14057, 20133}},
	{7193, 17, 12571, {1, 1, 1, 1, 7, 23, 89, 53, 483, 873, 521, 2115, 1461, 11241, 1003, 28749, 68227}},
	{7194, 17, 12580, {1, 3, 5, 5, 3, 17, 23, 219, 281, 975, 895, 4043, 6505, 5991, 27401, 38791, 89239}},
	{7195, 17, 12597, {1, 1, 1, 7, 29, 41, 63, 151, 195, 495, 469, 305, 7437, 1107, 31147, 30755, 116551}},
	{7196, 17, 12607, {1, 3, 7, 3, 13, 25, 33, 193, 23, 135, 3, 513, 4169, 15355, 2255, 32167, 68691}},
	{7197, 17, 12609, {1, 3, 3, 11, 29, 19, 125, 177, 83, 361, 393, 663, 1859, 1333, 17507, 10661, 72387}},
	{7198, 17, 12610, {1, 1, 5, 11, 23, 13, 61, 33, 149, 145, 995, 649, 7587, 6743, 25225, 54997, 10193}},
	{7199, 17, 12616, {1, 1, 7, 13, 29, 51, 107, 79, 467, 881, 1227, 1083, 3277, 2559, 26819, 57311, 48095}},
	{7200, 17, 12621, {1, 3, 1, 1, 1, 19, 23, 25, 239, 703, 119, 2525, 8079, 5433, 8989, 42517, 116755}},
	{7201, 17, 12624, {1, 1, 7, 11, 31, 9, 9, 113, 381, 363, 447, 3751, 7523, 15995, 3853, 42069, 81455}},
	{7202, 17, 12639, {1, 1, 5, 9, 29, 41, 103, 179, 477, 527, 1593, 3003, 1095, 6823, 6911, 44987, 32445}},
	{7203, 17, 12645, {1, 1, 7, 15, 5, 31, 55, 181, 149, 127, 1745, 2753, 801, 285, 20199, 33707, 118397}},
	{7204, 17, 12652, {1, 3, 7, 7, 11, 29, 89, 215, 351, 303, 1519, 2593, 2045, 14699, 1657, 40799, 39641}},
	{7205, 17, 12655, {1, 1, 7, 13, 3, 35, 73, 111, 15, 803, 1819, 3453, 3611, 8337, 14239, 14875, 83983}},
	{7206, 17, 12660, {1, 1, 5, 15, 15, 49, 27, 101, 149, 3, 717, 2229, 7397, 6579, 10965, 35997, 28823}},
	{7207, 17, 12667, {1, 1, 5, 7, 3, 17, 49, 245, 343, 657, 15, 749, 6413, 10811, 2909, 47309, 34613}},
	{7208, 17, 12686, {1, 3, 5, 15, 13, 35, 67, 99, 481, 379, 2003, 3367, 3065, 5845, 7799, 43931, 15263}},
	{7209, 17, 12688, {1, 1, 5, 13, 21, 49, 81, 77, 395, 919, 1931, 661, 123, 9965, 10487, 55131, 1567}},
	{7210, 17, 12697, {1, 3, 5, 11, 23, 39, 41, 121, 159, 473, 191, 1983, 6411, 10503, 10523, 40601, 64153}},
	{7211, 17, 12700, {1, 1, 5, 7, 9, 37, 73, 207, 497, 789, 1671, 325, 1697, 11281, 31185, 4961, 124431}},
	{7212, 17, 12707, {1, 1, 5, 15, 7, 51, 71, 91, 449, 707, 621, 2427, 627, 1747, 12779, 17569, 98289}},
	{7213, 17, 12710, {1, 1, 5, 5, 31, 3, 89, 163, 77, 647, 1747, 2965, 1669, 3311, 17651, 8111, 30739}},
	{7214, 17, 12719, {1, 1, 3, 11, 15, 31, 77, 173, 405, 913, 459, 2955, 6153, 13391, 20439, 64433, 12371}},
	{7215, 17, 12739, {1, 1, 3, 11, 13, 55, 29, 37, 379, 689, 407, 1373, 397, 5027, 15497, 25687, 48193}},
	{7216, 17, 12742, {1, 3, 3, 15, 13, 7, 81, 207, 395, 901, 779, 1683, 2491, 3807, 31979, 32403, 81113}},
	{7217, 17, 12746, {1, 3, 3, 13, 29, 31, 25, 81, 459, 991, 793, 3285, 2775, 16199, 11423, 52597, 86041}},
	{7218, 17, 12754, {1, 3, 3, 13, 17, 17, 101, 183, 19, 735, 671, 1097, 2461, 9863, 25985, 31915, 73047}},
	{7219, 17, 12765, {1, 3, 3, 3, 3, 11, 71, 63, 429, 899, 351, 1275, 3907, 14241, 19135, 14875, 43325}},
	{7220, 17, 12772, {1, 1, 7, 11, 11, 61, 15, 213, 411, 13, 1409, 1741, 5257, 8729, 28351, 6381, 77501}},
	{7221, 17, 12784, {1, 1, 7, 7, 29, 27, 51, 217, 411, 261, 599, 3027, 7871, 9133, 32423, 44275, 34701}},
	{7222, 17, 12789, {1, 3, 7, 7, 1, 1, 127, 209, 151, 845, 1421, 3115, 7775, 10133, 6163, 41165, 91187}},
	{7223, 17, 12800, {1, 3, 1, 9, 1, 35, 75, 3, 81, 477, 131, 1383, 1377, 6857, 3863, 12583, 7855}},
	{7224, 17, 12805, {1, 3, 1, 3, 3, 11, 1, 167, 347, 317, 557, 3763, 7175, 13341, 759, 23275, 78039}},
	{7225, 17, 12809, {1, 3, 5, 11, 19, 53, 85, 139, 67, 757, 487, 919, 6001, 16031, 24959, 28013, 65771}},
	{7226, 17, 12815, {1, 1, 1, 1, 23, 9, 83, 55, 249, 305, 1305, 109, 5559, 5129, 30973, 19889, 6691}},
	{7227, 17, 12827, {1, 1, 1, 3, 21, 19, 85, 89, 213, 847, 861, 1651, 6613, 6001, 8157, 2555, 98673}},
	{7228, 17, 12830, {1, 1, 1, 15, 25, 15, 125, 133, 177, 295, 549, 1763, 2811, 4381, 1079, 7813, 87909}},
	{7229, 17, 12833, {1, 1, 1, 5, 5, 17, 25, 225, 353, 997, 1565, 2225, 7265, 16227, 28209, 9011, 97193}},
	{7230, 17, 12840, {1, 3, 7, 15, 13, 13, 35, 239, 331, 965, 1547, 1627, 6409, 7745, 30899, 36915, 59293}},
	{7231, 17, 12851, {1, 1, 1, 13, 27, 45, 23, 179, 193, 801, 381, 3783, 3551, 11855, 11041, 49911, 62101}},
	{7232, 17, 12868, {1, 1, 7, 3, 3, 31, 61, 5, 421, 939, 1637, 217, 389, 1797, 32141, 28817, 6997}},
	{7233, 17, 12871, {1, 3, 3, 5, 21, 31, 83, 65, 421, 577, 1137, 2561, 2943, 4171, 2803, 23325, 92315}},
	{7234, 17, 12886, {1, 1, 3, 3, 27, 33, 75, 81, 477, 3, 1903, 773, 5551, 10069, 7285, 58103, 98311}},
	{7235, 17, 12899, {1, 3, 7, 15, 1, 17, 95, 209, 65, 747, 1633, 581, 7395, 1393, 21795, 15735, 129757}},
	{7236, 17, 12923, {1, 3, 5, 3, 17, 3, 9, 131, 51, 693, 1571, 1865, 8137, 915, 13345, 35137, 59005}},
	{7237, 17, 12926, {1, 1, 3, 7, 23, 27, 61, 163, 449, 87, 717, 1075, 4309, 4887, 11741, 24549, 96729}},
	{7238, 17, 12932, {1, 3, 7, 13, 21, 5, 3, 97, 191, 999, 1193, 1215, 5907, 10491, 2281, 6455, 68625}},
	{7239, 17, 12935, {1, 1, 5, 7, 9, 9, 101, 5, 375, 137, 1473, 1265, 5307, 259, 20699, 25367, 129393}},
	{7240, 17, 12939, {1, 3, 7, 7, 21, 1, 77, 65, 23, 139, 945, 491, 1069, 253, 12335, 26861, 129467}},
	{7241, 17, 12942, {1, 1, 3, 9, 15, 33, 85, 225, 45, 311, 281, 1601, 7325, 12265, 2591, 51955, 130681}},
	{7242, 17, 12953, {1, 1, 1, 3, 27, 33, 17, 89, 495, 91, 527, 3347, 7883, 9481, 28731, 54729, 15265}},
	{7243, 17, 12959, {1, 1, 3, 3, 9, 47, 115, 161, 299, 493, 1857, 3597, 7175, 15603, 11523, 33837, 57557}},
	{7244, 17, 12960, {1, 3, 7, 3, 15, 47, 127, 195, 391, 869, 99, 429, 7125, 10413, 5063, 61845, 71843}},
	{7245, 17, 13002, {1, 3, 3, 1, 7, 31, 27, 69, 7, 83, 315, 2749, 5693, 13377, 28091, 13065, 111029}},
	{7246, 17, 13004, {1, 1, 3, 15, 15, 45, 125, 229, 459, 611, 1167, 3375, 3587, 81, 9275, 45327, 39749}},
	{7247, 17, 13016, {1, 3, 7, 11, 9, 3, 43, 161, 221, 209, 51, 1475, 3577, 13973, 15285, 35553, 83935}},
	{7248, 17, 13021, {1, 1, 7, 9, 15, 55, 25, 119, 39, 537, 317, 1331, 2161, 1791, 19221, 63459, 124595}},
	{7249, 17, 13035, {1, 1, 1, 11, 9, 7, 113, 187, 295, 67, 1795, 113, 119, 9127, 32119, 7719, 67591}},
	{7250, 17, 13038, {1, 3, 7, 13, 1, 53, 17, 19, 331, 711, 359, 2945, 5847, 7237, 23617, 17411, 2203}},
	{7251, 17, 13052, {1, 3, 3, 7, 21, 63, 115, 159, 225, 161, 1255, 2381, 7411, 95, 1625, 30493, 56685}},
	{7252, 17, 13058, {1, 1, 5, 9, 13, 57, 5, 107, 195, 271, 677, 2081, 6027, 11091, 14171, 19007, 102119}},
	{7253, 17, 13069, {1, 3, 3, 3, 19, 13, 31, 155, 209, 89, 955, 523, 615, 5319, 16079, 9289, 49135}},
	{7254, 17, 13082, {1, 3, 1, 13, 1, 31, 69, 143, 329, 813, 635, 891, 2967, 5563, 19643, 35813, 14345}},
	{7255, 17, 13093, {1, 1, 5, 5, 17, 47, 97, 49, 123, 997, 15, 3685, 3925, 4973, 11195, 17115, 63709}},
	{7256, 17, 13094, {1, 1, 7, 13, 13, 17, 99, 149, 309, 281, 329, 905, 6487, 4495, 31831, 24413, 26431}},
	{7257, 17, 13100, {1, 1, 5, 5, 5, 47, 113, 115, 61, 157, 955, 2323, 4445, 229, 24049, 14753, 15189}},
	{7258, 17, 13115, {1, 3, 7, 15, 25, 21, 13, 137, 377, 45, 629, 1339, 8037, 5073, 24741, 48589, 28953}},
	{7259, 17, 13125, {1, 3, 3, 9, 3, 41, 7, 101, 333, 59, 1213, 1871, 3993, 11261, 4403, 42785, 58753}},
	{7260, 17, 13132, {1, 3, 1, 7, 5, 33, 87, 73, 317, 575, 1459, 905, 1033, 14179, 19595, 30269, 103853}},
	{7261, 17, 13143, {1, 1, 1, 1, 19, 49, 63, 181, 227, 401, 695, 1811, 2383, 3835, 14379, 30685, 114731}},
	{7262, 17, 13144, {1, 1, 5, 15, 9, 41, 35, 91, 357, 659, 155, 3725, 6509, 405, 25449, 37719, 6013}},
	{7263, 17, 13153, {1, 1, 7, 3, 11, 59, 33, 151, 291, 393, 741, 3961, 2787, 993, 10361, 11737, 42047}},
	{7264, 17, 13160, {1, 3, 7, 3, 15, 15, 55, 59, 419, 203, 55, 801, 2719, 15487, 13213, 58473, 50315}},
	{7265, 17, 13165, {1, 3, 7, 13, 17, 21, 113, 111, 159, 163, 711, 1135, 1133, 15519, 30515, 55777, 25025}},
	{7266, 17, 13173, {1, 1, 3, 5, 13, 25, 23, 3, 93, 873, 559, 1815, 3381, 5311, 14365, 34349, 17333}},
	{7267, 17, 13174, {1, 3, 5, 7, 15, 43, 85, 33, 23, 903, 1247, 3279, 1393, 12059, 19251, 19389, 5097}},
	{7268, 17, 13187, {1, 3, 1, 11, 21, 59, 3, 153, 403, 95, 1939, 2679, 419, 9035, 31219, 2897, 15727}},
	{7269, 17, 13190, {1, 3, 5, 1, 11, 21, 35, 169, 453, 15, 791, 3931, 1021, 16321, 6033, 10639, 16173}},
	{7270, 17, 13204, {1, 3, 1, 11, 7, 57, 39, 61, 381, 465, 451, 2863, 575, 5597, 31041, 8625, 82373}},
	{7271, 17, 13218, {1, 1, 3, 5, 13, 5, 63, 1, 75, 245, 1305, 285, 3367, 10107, 5853, 35275, 128255}},
	{7272, 17, 13247, {1, 1, 1, 3, 1, 57, 21, 91, 139, 669, 765, 1867, 2153, 10347, 26119, 35517, 4725}},
	{7273, 17, 13250, {1, 3, 5, 1, 21, 41, 59, 247, 473, 1015, 975, 485, 2161, 11941, 10341, 35245, 55587}},
	{7274, 17, 13262, {1, 1, 5, 9, 7, 59, 33, 149, 97, 619, 393, 3613, 6037, 10895, 19461, 15975, 47919}},
	{7275, 17, 13267, {1, 3, 3, 15, 7, 17, 95, 13, 147, 361, 915, 2585, 4483, 3159, 12255, 44685, 116163}},
	{7276, 17, 13274, {1, 3, 1, 15, 27, 31, 75, 31, 423, 233, 1453, 2815, 3633, 6531, 25721, 29649, 80645}},
	{7277, 17, 13304, {1, 3, 3, 3, 19, 7, 73, 33, 163, 495, 1483, 2277, 6455, 6523, 9331, 21869, 52175}},
	{7278, 17, 13309, {1, 3, 5, 13, 5, 1, 63, 35, 335, 189, 713, 2997, 3277, 10049, 4681, 16753, 17107}},
	{7279, 17, 13315, {1, 3, 5, 9, 3, 55, 29, 171, 395, 585, 671, 1875, 4449, 12895, 5455, 11023, 106189}},
	{7280, 17, 13317, {1, 3, 5, 3, 25, 53, 33, 169, 109, 285, 787, 861, 5549, 5171, 15293, 2977, 14559}},
	{7281, 17, 13324, {1, 3, 7, 1, 21, 25, 97, 115, 1, 999, 1033, 3471, 129, 16093, 495, 16859, 34615}},
	{7282, 17, 13332, {1, 3, 5, 9, 13, 5, 109, 41, 57, 957, 231, 3771, 2917, 15649, 8869, 14857, 64943}},
	{7283, 17, 13342, {1, 1, 7, 7, 21, 41, 101, 59, 167, 441, 997, 2951, 7891, 16325, 12669, 53829, 100705}},
	{7284, 17, 13346, {1, 1, 7, 9, 19, 59, 23, 141, 193, 237, 1067, 1823, 3351, 3239, 3135, 9275, 37069}},
	{7285, 17, 13355, {1, 3, 3, 9, 3, 17, 95, 73, 19, 231, 779, 3065, 2245, 2967, 24971, 62589, 16729}},
	{7286, 17, 13358, {1, 3, 3, 13, 25, 19, 117, 147, 443, 123, 157, 2037, 327, 14715, 5693, 54641, 33325}},
	{7287, 17, 13360, {1, 3, 1, 9, 21, 21, 21, 125, 49, 787, 767, 2831, 511, 2461, 31537, 27155, 44053}},
	{7288, 17, 13369, {1, 3, 7, 9, 31, 19, 125, 67, 119, 465, 287, 1869, 3979, 15723, 21069, 8581, 66939}},
	{7289, 17, 13372, {1, 1, 1, 11, 7, 37, 123, 237, 353, 499, 113, 3829, 217, 4751, 7385, 20343, 83699}},
	{7290, 17, 13398, {1, 3, 7, 13, 9, 3, 53, 27, 487, 87, 35, 2645, 3481, 14409, 27875, 31695, 78489}},
	{7291, 17, 13404, {1, 3, 3, 13, 9, 43, 67, 51, 153, 83, 591, 1991, 1787, 11973, 7273, 34801, 47199}},
	{7292, 17, 13407, {1, 3, 7, 1, 15, 53, 71, 11, 205, 853, 2011, 581, 1281, 7819, 23083, 33731, 74951}},
	{7293, 17, 13414, {1, 1, 5, 5, 17, 63, 109, 219, 225, 997, 1251, 3287, 1441, 13489, 22723, 45191, 50249}},
	{7294, 17, 13426, {1, 1, 5, 3, 13, 1, 43, 53, 293, 685, 1369, 1515, 7479, 3233, 20007, 65235, 102467}},
	{7295, 17, 13432, {1, 3, 5, 7, 29, 45, 63, 45, 219, 445, 2047, 317, 7553, 325, 1465, 949, 35163}},
	{7296, 17, 13466, {1, 1, 3, 9, 7, 31, 73, 211, 501, 233, 1495, 701, 5857, 10763, 9743, 10289, 23801}},
	{7297, 17, 13481, {1, 3, 7, 3, 23, 47, 99, 61, 179, 833, 1425, 1275, 4467, 4367, 5567, 23513, 68677}},
	{7298, 17, 13490, {1, 1, 5, 5, 27, 33, 119, 229, 329, 51, 1025, 3167, 3405, 4039, 4135, 6655, 43771}},
	{7299, 17, 13496, {1, 3, 7, 15, 5, 49, 91, 55, 425, 15, 2003, 1571, 3539, 10375, 29645, 5889, 51887}},
	{7300, 17, 13504, {1, 1, 7, 3, 13, 55, 85, 91, 181, 723, 1941, 75, 4443, 11507, 7027, 14189, 50685}},
	{7301, 17, 13516, {1, 3, 5, 5, 29, 49, 13, 3, 97, 165, 41, 3039, 3325, 2161, 775, 38501, 42381}},
	{7302, 17, 13527, {1, 1, 5, 11, 9, 57, 47, 109, 9, 585, 375, 1839, 937, 6877, 29847, 60163, 103081}},
	{7303, 17, 13533, {1, 1, 3, 13, 5, 47, 11, 195, 253, 235, 275, 2313, 163, 14683, 5681, 13381, 84553}},
	{7304, 17, 13537, {1, 3, 3, 15, 15, 1, 93, 157, 437, 557, 307, 1179, 6857, 3101, 16723, 50579, 69603}},
	{7305, 17, 13552, {1, 1, 5, 9, 11, 29, 23, 219, 337, 689, 1155, 2007, 6853, 6749, 20127, 13199, 48433}},
	{7306, 17, 13561, {1, 1, 1, 13, 1, 61, 73, 213, 335, 539, 903, 2719, 775, 2775, 29109, 33367, 3281}},
	{7307, 17, 13567, {1, 3, 1, 5, 15, 31, 65, 231, 439, 623, 1871, 2299, 5365, 10333, 9147, 2781, 63813}},
	{7308, 17, 13582, {1, 1, 3, 1, 1, 25, 23, 229, 173, 279, 181, 1299, 2893, 15475, 12473, 46097, 123387}},
	{7309, 17, 13587, {1, 1, 7, 13, 7, 43, 17, 187, 467, 113, 1293, 2013, 6091, 14621, 22195, 24079, 45379}},
	{7310, 17, 13589, {1, 3, 1, 7, 1, 7, 119, 159, 377, 11, 705, 2853, 3767, 13739, 23375, 25563, 73987}},
	{7311, 17, 13593, {1, 1, 3, 15, 21, 13, 111, 119, 401, 1005, 777, 1699, 2431, 15139, 27887, 28415, 71519}},
	{7312, 17, 13596, {1, 1, 7, 9, 1, 49, 19, 171, 297, 77, 1343, 1249, 5769, 13889, 21401, 24915, 17641}},
	{7313, 17, 13615, {1, 1, 7, 11, 31, 45, 51, 231, 123, 817, 13, 791, 6235, 2787, 475, 1717, 5071}},
	{7314, 17, 13617, {1, 3, 3, 13, 5, 9, 21, 129, 253, 731, 785, 2275, 7343, 7841, 5477, 8973, 101033}},
	{7315, 17, 13623, {1, 1, 7, 13, 23, 1, 119, 221, 293, 709, 2031, 3019, 1529, 2007, 10823, 43193, 82661}},
	{7316, 17, 13641, {1, 3, 1, 11, 29, 29, 87, 79, 415, 679, 1899, 3453, 7355, 8627, 28225, 41857, 106645}},
	{7317, 17, 13647, {1, 1, 5, 15, 9, 13, 21, 241, 491, 927, 999, 2131, 3501, 11063, 28595, 54691, 21297}},
	{7318, 17, 13650, {1, 1, 1, 3, 5, 41, 85, 89, 483, 309, 791, 825, 3043, 2715, 16573, 6551, 77875}},
	{7319, 17, 13659, {1, 3, 1, 1, 25, 21, 107, 123, 79, 1019, 821, 1251, 4943, 1429, 17843, 37013, 53285}},
	{7320, 17, 13671, {1, 1, 5, 3, 7, 5, 35, 123, 445, 315, 627, 2543, 1261, 13737, 15991, 36591, 18309}},
	{7321, 17, 13677, {1, 3, 3, 5, 25, 43, 65, 249, 309, 1023, 737, 1933, 4735, 7725, 12063, 53023, 126677}},
	{7322, 17, 13678, {1, 3, 1, 9, 13, 37, 77, 61, 179, 275, 277, 1431, 2869, 14563, 665, 60553, 7661}},
	{7323, 17, 13680, {1, 3, 5, 3, 29, 1, 127, 73, 363, 311, 1591, 3863, 6481, 4725, 8287, 61311, 39011}},
	{7324, 17, 13685, {1, 3, 1, 7, 13, 23, 115, 215, 385, 563, 1033, 2343, 5023, 11013, 12131, 26997, 48645}},
	{7325, 17, 13689, {1, 1, 5, 13, 3, 59, 41, 155, 263, 507, 1175, 2967, 7929, 8237, 11841, 15365, 51881}},
	{7326, 17, 13701, {1, 1, 3, 11, 19, 35, 89, 115, 121, 315, 1697, 2121, 1867, 6865, 23639, 26525, 44687}},
	{7327, 17, 13706, {1, 3, 5, 15, 9, 5, 125, 183, 149, 447, 309, 1743, 6089, 369, 16153, 63799, 57657}},
	{7328, 17, 13720, {1, 1, 1, 7, 7, 39, 89, 139, 457, 741, 1613, 2883, 5057, 12495, 18669, 55469, 97941}},
	{7329, 17, 13729, {1, 1, 5, 7, 29, 39, 97, 9, 481, 667, 1353, 3387, 2813, 16205, 8353, 22121, 92965}},
	{7330, 17, 13735, {1, 3, 7, 9, 11, 55, 79, 159, 349, 717, 829, 3157, 1457, 6199, 5861, 2553, 20387}},
	{7331, 17, 13736, {1, 1, 1, 11, 3, 51, 113, 53, 287, 109, 1717, 2405, 7207, 4473, 11145, 2549, 591}},
	{7332, 17, 13756, {1, 1, 3, 13, 3, 61, 31, 141, 217, 487, 299, 2755, 3389, 10053, 1105, 21129, 74203}},
	{7333, 17, 13759, {1, 1, 1, 3, 11, 55, 7, 113, 413, 449, 787, 3279, 5123, 16025, 15005, 12175, 6795}},
	{7334, 17, 13761, {1, 3, 1, 1, 25, 23, 107, 191, 3, 3, 49, 1083, 3275, 10385, 7989, 53739, 25505}},
	{7335, 17, 13771, {1, 1, 5, 13, 7, 17, 59, 13, 471, 147, 1627, 2119, 3555, 15555, 10333, 49363, 80959}},
	{7336, 17, 13782, {1, 1, 1, 15, 23, 33, 61, 191, 207, 939, 45, 2781, 71, 9661, 28433, 13089, 76419}},
	{7337, 17, 13786, {1, 3, 3, 7, 29, 47, 111, 19, 315, 381, 851, 1303, 2627, 6255, 30369, 37723, 12949}},
	{7338, 17, 13798, {1, 1, 1, 13, 31, 43, 3, 193, 5, 99, 769, 2523, 1949, 129, 9693, 60535, 67059}},
	{7339, 17, 13810, {1, 1, 3, 15, 5, 33, 73, 149, 253, 985, 863, 1551, 4369, 5911, 8269, 35463, 117055}},
	{7340, 17, 13819, {1, 3, 1, 5, 27, 57, 3, 105, 253, 731, 119, 3287, 613, 4627, 22003, 56027, 123005}},
	{7341, 17, 13826, {1, 1, 3, 3, 27, 47, 67, 147, 495, 865, 1233, 3707, 2511, 2951, 7367, 15625, 86417}},
	{7342, 17, 13831, {1, 1, 7, 1, 7, 7, 13, 255, 457, 529, 953, 1481, 5565, 12495, 4723, 41615, 121829}},
	{7343, 17, 13837, {1, 3, 5, 11, 1, 51, 91, 153, 323, 609, 1353, 2995, 4035, 13835, 28619, 46217, 4967}},
	{7344, 17, 13846, {1, 1, 5, 7, 25, 59, 81, 101, 185, 709, 1249, 2285, 6579, 8655, 17563, 9707, 63845}},
	{7345, 17, 13856, {1, 3, 3, 9, 31, 25, 17, 19, 111, 627, 1187, 2621, 6529, 9457, 25027, 18069, 47559}},
	{7346, 17, 13862, {1, 3, 7, 15, 7, 15, 103, 201, 391, 1023, 817, 535, 2713, 1317, 13469, 56043, 70847}},
	{7347, 17, 13866, {1, 1, 3, 7, 17, 57, 35, 99, 439, 367, 27, 2695, 3519, 8337, 14047, 58489, 69}},
	{7348, 17, 13885, {1, 1, 1, 3, 17, 23, 71, 189, 57, 39, 715, 1779, 3081, 14657, 21895, 59203, 31005}},
	{7349, 17, 13891, {1, 1, 7, 13, 1, 47, 69, 159, 353, 517, 271, 973, 5077, 15707, 11095, 19671, 3389}},
	{7350, 17, 13893, {1, 1, 7, 13, 25, 55, 115, 21, 43, 939, 1697, 101, 4751, 1993, 2389, 28353, 45251}},
	{7351, 17, 13905, {1, 3, 1, 15, 11, 57, 17, 49, 121, 419, 909, 121, 5047, 4235, 13051, 21529, 42097}},
	{7352, 17, 13908, {1, 1, 1, 3, 19, 37, 31, 233, 251, 175, 929, 1527, 7527, 3605, 17075, 61053, 56235}},
	{7353, 17, 13912, {1, 1, 1, 3, 9, 5, 117, 131, 251, 475, 1695, 1381, 2445, 5921, 14921, 937, 80791}},
	{7354, 17, 13917, {1, 1, 3, 9, 11, 5, 31, 215, 37, 567, 1537, 2183, 3291, 1601, 14025, 48807, 7243}},
	{7355, 17, 13918, {1, 3, 1, 9, 7, 13, 81, 249, 321, 473, 1419, 3977, 7037, 14191, 10865, 56131, 43225}},
	{7356, 17, 13946, {1, 1, 1, 13, 15, 23, 31, 69, 449, 491, 1461, 729, 7955, 4003, 16817, 37273, 72025}},
	{7357, 17, 13948, {1, 1, 5, 13, 7, 41, 93, 169, 347, 1013, 301, 2813, 1455, 13187, 10769, 60807, 46333}},
	{7358, 17, 13964, {1, 1, 5, 3, 23, 15, 1, 161, 29, 35, 415, 235, 93, 14543, 29585, 29657, 36489}},
	{7359, 17, 13970, {1, 3, 1, 3, 31, 63, 39, 235, 153, 549, 43, 147, 2317, 3537, 25561, 58287, 58725}},
	{7360, 17, 13975, {1, 1, 3, 5, 5, 11, 59, 97, 349, 307, 501, 1701, 4243, 13717, 17419, 23387, 29533}},
	{7361, 17, 13979, {1, 3, 1, 7, 7, 19, 33, 243, 67, 353, 2023, 3111, 7173, 10979, 28117, 40175, 45337}},
	{7362, 17, 13997, {1, 1, 1, 5, 15, 59, 55, 135, 107, 543, 1743, 2695, 3293, 111, 32629, 8249, 52273}},
	{7363, 17, 14000, {1, 3, 5, 13, 15, 57, 39, 79, 5, 451, 571, 1445, 1393, 2125, 31713, 59655, 20897}},
	{7364, 17, 14006, {1, 1, 3, 11, 29, 61, 1, 37, 173, 513, 1779, 2649, 3289, 4679, 2039, 47587, 28973}},
	{7365, 17, 14020, {1, 1, 5, 5, 15, 19, 17, 143, 387, 359, 275, 625, 7383, 15537, 10311, 40005, 20729}},
	{7366, 17, 14023, {1, 1, 3, 9, 7, 23, 71, 179, 85, 447, 345, 3459, 2857, 8331, 5489, 62207, 64933}},
	{7367, 17, 14024, {1, 1, 1, 1, 11, 61, 47, 131, 213, 611, 701, 713, 1269, 9563, 25223, 50697, 88679}},
	{7368, 17, 14029, {1, 1, 5, 15, 21, 5, 77, 59, 455, 243, 459, 2809, 13, 9325, 32047, 3939, 48389}},
	{7369, 17, 14035, {1, 3, 7, 1, 21, 53, 111, 225, 407, 119, 713, 3635, 1539, 15321, 29827, 36069, 74483}},
	{7370, 17, 14044, {1, 1, 5, 13, 7, 45, 75, 43, 191, 715, 169, 759, 33, 11329, 1069, 36103, 28055}},
	{7371, 17, 14047, {1, 3, 7, 5, 7, 13, 7, 35, 27, 391, 517, 1439, 5699, 1067, 23857, 7293, 66167}},
	{7372, 17, 14058, {1, 1, 7, 11, 3, 31, 1, 83, 299, 345, 65, 669, 1529, 7569, 28959, 50561, 69493}},
	{7373, 17, 14066, {1, 3, 1, 5, 25, 25, 43, 149, 83, 225, 1589, 1691, 7777, 773, 10421, 49523, 23533}},
	{7374, 17, 14075, {1, 1, 5, 11, 25, 29, 81, 11, 497, 43, 951, 2551, 821, 13805, 12315, 61299, 81397}},
	{7375, 17, 14080, {1, 3, 1, 9, 29, 23, 109, 123, 235, 255, 1519, 3289, 7761, 14575, 11851, 1719, 51655}},
	{7376, 17, 14095, {1, 3, 5, 15, 21, 49, 13, 43, 87, 517, 687, 1457, 1501, 15959, 31907, 13771, 69379}},
	{7377, 17, 14100, {1, 1, 5, 3, 21, 11, 87, 9, 343, 317, 845, 1663, 7933, 14063, 24915, 31487, 17445}},
	{7378, 17, 14114, {1, 1, 3, 13, 21, 31, 87, 99, 185, 333, 993, 3899, 971, 2851, 23643, 195, 66957}},
	{7379, 17, 14116, {1, 1, 1, 15, 19, 47, 23, 1, 67, 57, 165, 3903, 421, 10561, 11621, 13815, 10349}},
	{7380, 17, 14123, {1, 3, 5, 11, 9, 19, 73, 17, 229, 913, 459, 3809, 2667, 9775, 3693, 52945, 90837}},
	{7381, 17, 14134, {1, 1, 5, 15, 3, 25, 109, 131, 507, 637, 1615, 859, 6785, 14891, 24801, 39095, 79557}},
	{7382, 17, 14143, {1, 1, 5, 7, 1, 51, 71, 251, 19, 799, 835, 1119, 2349, 15083, 16509, 55621, 123501}},
	{7383, 17, 14151, {1, 3, 5, 9, 13, 39, 127, 1, 233, 37, 735, 3307, 5163, 4529, 5961, 12893, 103641}},
	{7384, 17, 14160, {1, 1, 7, 5, 23, 15, 49, 123, 511, 201, 2025, 289, 3847, 15755, 24279, 52543, 42017}},
	{7385, 17, 14163, {1, 1, 5, 3, 9, 61, 19, 37, 3, 361, 1065, 2971, 2517, 1259, 27359, 3823, 60181}},
	{7386, 17, 14179, {1, 3, 1, 7, 15, 17, 57, 249, 57, 979, 147, 2407, 2579, 3159, 8467, 8433, 72873}},
	{7387, 17, 14181, {1, 1, 3, 1, 25, 7, 47, 117, 449, 321, 143, 3867, 165, 7961, 27597, 10033, 2437}},
	{7388, 17, 14193, {1, 3, 5, 13, 19, 49, 1, 83, 477, 549, 509, 2911, 1559, 14017, 10469, 62171, 82829}},
	{7389, 17, 14209, {1, 3, 3, 7, 27, 21, 15, 63, 31, 45, 1223, 3903, 5469, 11983, 29627, 27453, 32019}},
	{7390, 17, 14210, {1, 1, 7, 7, 9, 9, 7, 77, 349, 467, 61, 3465, 6921, 15761, 15179, 38649, 2469}},
	{7391, 17, 14224, {1, 3, 1, 13, 9, 59, 55, 67, 271, 617, 643, 4071, 7963, 8153, 5121, 43917, 26219}},
	{7392, 17, 14245, {1, 1, 3, 7, 29, 21, 63, 103, 327, 623, 931, 1511, 3125, 229, 28949, 61315, 72667}},
	{7393, 17, 14249, {1, 3, 7, 1, 19, 37, 49, 63, 403, 885, 161, 121, 1447, 9227, 15019, 50049, 26939}},
	{7394, 17, 14255, {1, 3, 3, 3, 23, 57, 95, 79, 485, 173, 93, 835, 7161, 11247, 3485, 5759, 36393}},
	{7395, 17, 14267, {1, 3, 7, 13, 23, 33, 5, 97, 235, 531, 313, 2925, 2223, 847, 18591, 15477, 3129}},
	{7396, 17, 14270, {1, 1, 3, 13, 25, 25, 101, 183, 477, 947, 1251, 2631, 7987, 13417, 23759, 55305, 123817}},
	{7397, 17, 14277, {1, 1, 5, 9, 27, 63, 49, 137, 179, 861, 33, 2375, 3827, 6485, 19689, 7867, 124429}},
	{7398, 17, 14305, {1, 3, 7, 3, 15, 43, 63, 103, 45, 947, 1837, 833, 7055, 7487, 19669, 12045, 78377}},
	{7399, 17, 14308, {1, 3, 5, 3, 29, 35, 57, 19, 471, 985, 1147, 741, 5403, 10057, 25375, 50889, 82719}},
	{7400, 17, 14312, {1, 3, 3, 1, 17, 19, 111, 13, 121, 821, 1831, 4043, 123, 9529, 1511, 10917, 105961}},
	{7401, 17, 14325, {1, 1, 3, 11, 1, 43, 23, 75, 345, 9, 1379, 2157, 5887, 1197, 14849, 17103, 91925}},
	{7402, 17, 14332, {1, 1, 3, 3, 19, 11, 1, 179, 343, 1023, 1801, 915, 255, 519, 5787, 32913, 43471}},
	{7403, 17, 14345, {1, 3, 5, 5, 3, 3, 3, 211, 461, 55, 851, 3165, 2903, 15077, 8537, 2037, 109057}},
	{7404, 17, 14354, {1, 1, 7, 15, 7, 7, 43, 249, 27, 511, 1369, 735, 6093, 12575, 26675, 21745, 117053}},
	{7405, 17, 14372, {1, 1, 5, 7, 21, 53, 45, 83, 415, 645, 325, 4027, 5181, 8485, 1917, 55623, 45203}},
	{7406, 17, 14387, {1, 1, 3, 15, 7, 1, 121, 221, 387, 403, 1877, 1671, 2113, 2379, 5667, 39867, 8079}},
	{7407, 17, 14390, {1, 1, 1, 7, 5, 29, 35, 77, 197, 661, 1859, 2539, 4045, 13497, 305, 44987, 31215}},
	{7408, 17, 14402, {1, 1, 5, 5, 13, 37, 13, 85, 287, 347, 579, 2283, 7911, 5463, 21141, 9035, 105067}},
	{7409, 17, 14408, {1, 1, 1, 9, 17, 17, 63, 97, 57, 629, 1917, 1133, 779, 12365, 17127, 52549, 18755}},
	{7410, 17, 14413, {1, 1, 7, 11, 7, 17, 65, 137, 485, 841, 653, 2921, 4935, 16273, 23333, 7399, 43129}},
	{7411, 17, 14431, {1, 3, 1, 11, 31, 55, 93, 225, 319, 35, 947, 1909, 7733, 8303, 20739, 55713, 6633}},
	{7412, 17, 14438, {1, 1, 1, 3, 11, 25, 1, 165, 305, 275, 607, 3845, 5203, 1989, 13803, 597, 39751}},
	{7413, 17, 14447, {1, 1, 5, 11, 31, 43, 83, 237, 453, 59, 457, 741, 411, 15895, 18891, 30133, 66767}},
	{7414, 17, 14455, {1, 3, 5, 11, 3, 23, 65, 81, 299, 527, 1057, 2731, 3839, 6023, 28887, 64929, 41405}},
	{7415, 17, 14461, {1, 3, 1, 1, 3, 5, 11, 169, 123, 957, 1495, 1717, 4079, 13239, 28339, 33677, 30591}},
	{7416, 17, 14466, {1, 1, 7, 15, 3, 1, 37, 245, 169, 273, 2039, 415, 6555, 13131, 11181, 62179, 36885}},
	{7417, 17, 14480, {1, 1, 3, 11, 1, 55, 19, 19, 425, 113, 1367, 2101, 5581, 985, 2475, 53983, 68999}},
	{7418, 17, 14490, {1, 1, 5, 9, 5, 33, 101, 193, 303, 579, 1265, 2791, 479, 12083, 17609, 31801, 113089}},
	{7419, 17, 14492, {1, 1, 3, 3, 17, 61, 59, 249, 81, 821, 1, 431, 5327, 8675, 23469, 15349, 67711}},
	{7420, 17, 14508, {1, 1, 7, 9, 31, 51, 89, 19, 469, 843, 561, 559, 4823, 7803, 31699, 44537, 56835}},
	{7421, 17, 14513, {1, 3, 7, 9, 11, 57, 27, 43, 469, 655, 433, 3081, 6719, 6651, 30823, 61503, 110711}},
	{7422, 17, 14516, {1, 3, 5, 11, 9, 53, 25, 147, 61, 533, 1369, 879, 7935, 13829, 26655, 17327, 52983}},
	{7423, 17, 14519, {1, 3, 7, 11, 15, 27, 97, 175, 435, 53, 75, 807, 549, 5277, 1831, 19421, 55669}},
	{7424, 17, 14525, {1, 1, 7, 15, 23, 5, 99, 133, 485, 587, 65, 2585, 7667, 2783, 19437, 52769, 1587}},
	{7425, 17, 14534, {1, 1, 7, 7, 13, 39, 111, 165, 489, 355, 1963, 333, 2993, 5233, 9173, 18951, 93737}},
	{7426, 17, 14537, {1, 1, 5, 7, 1, 29, 67, 135, 427, 91, 53, 3109, 3745, 9529, 17567, 42361, 84577}},
	{7427, 17, 14543, {1, 3, 5, 1, 31, 35, 59, 181, 87, 345, 1975, 781, 603, 16365, 19453, 9933, 112739}},
	{7428, 17, 14545, {1, 3, 3, 1, 31, 41, 127, 35, 263, 403, 1811, 383, 1523, 8477, 5973, 41569, 99309}},
	{7429, 17, 14552, {1, 3, 7, 7, 5, 25, 11, 201, 231, 679, 519, 2481, 7415, 12397, 21265, 49419, 13903}},
	{7430, 17, 14562, {1, 1, 7, 5, 1, 11, 63, 221, 327, 509, 419, 871, 7891, 11835, 11099, 10669, 43853}},
	{7431, 17, 14571, {1, 1, 5, 11, 19, 11, 37, 105, 265, 513, 1013, 707, 6083, 14571, 17573, 7645, 5363}},
	{7432, 17, 14574, {1, 1, 1, 13, 19, 19, 67, 93, 113, 509, 1013, 4037, 1939, 7015, 24487, 57183, 123463}},
	{7433, 17, 14582, {1, 1, 1, 1, 21, 17, 95, 25, 261, 1005, 685, 691, 4467, 14723, 24043, 32287, 19651}},
	{7434, 17, 14611, {1, 3, 1, 15, 15, 15, 57, 191, 27, 719, 229, 1977, 241, 9021, 21335, 30967, 81207}},
	{7435, 17, 14614, {1, 3, 1, 9, 23, 61, 103, 67, 361, 925, 811, 1007, 5707, 11479, 5907, 3897, 65141}},
	{7436, 17, 14620, {1, 3, 5, 9, 17, 61, 11, 15, 351, 715, 939, 2141, 4857, 8397, 9693, 26845, 120007}},
	{7437, 17, 14633, {1, 3, 1, 5, 19, 55, 99, 19, 291, 309, 287, 1969, 4341, 7579, 30909, 37277, 54927}},
	{7438, 17, 14641, {1, 3, 7, 3, 19, 29, 43, 163, 367, 753, 1733, 1463, 7927, 10671, 16817, 41229, 113887}},
	{7439, 17, 14648, {1, 3, 7, 1, 11, 51, 39, 207, 283, 73, 1423, 2473, 7593, 3581, 30179, 6369, 112217}},
	{7440, 17, 14671, {1, 1, 3, 15, 15, 25, 43, 5, 271, 611, 959, 537, 303, 3659, 18073, 8147, 81531}},
	{7441, 17, 14674, {1, 3, 7, 1, 27, 55, 77, 11, 367, 209, 1967, 3409, 935, 5309, 18857, 46225, 8367}},
	{7442, 17, 14689, {1, 1, 5, 11, 11, 63, 75, 73, 43, 869, 2021, 3285, 269, 9113, 32699, 2091, 17327}},
	{7443, 17, 14690, {1, 1, 5, 11, 9, 25, 31, 245, 109, 805, 1645, 3607, 817, 9571, 12767, 65441, 129977}},
	{7444, 17, 14692, {1, 3, 7, 5, 11, 61, 67, 223, 433, 387, 935, 1615, 7915, 6133, 24087, 55323, 100619}},
	{7445, 17, 14699, {1, 1, 1, 15, 25, 61, 7, 39, 311, 353, 183, 33, 2591, 4951, 31377, 9081, 9707}},
	{7446, 17, 14710, {1, 1, 3, 3, 1, 9, 65, 229, 185, 47, 1255, 1365, 2231, 6843, 26927, 27195, 60651}},
	{7447, 17, 14719, {1, 1, 7, 5, 7, 25, 91, 133, 159, 737, 1767, 3117, 7321, 6159, 3361, 27793, 33473}},
	{7448, 17, 14730, {1, 3, 7, 3, 11, 7, 5, 125, 369, 951, 1277, 65, 7703, 1817, 11773, 25657, 67045}},
	{7449, 17, 14732, {1, 1, 3, 9, 21, 27, 21, 41, 131, 605, 1, 119, 1553, 1361, 31973, 43135, 119321}},
	{7450, 17, 14743, {1, 3, 7, 1, 25, 63, 55, 173, 323, 403, 1401, 1367, 3455, 15335, 13045, 20759, 8309}},
	{7451, 17, 14744, {1, 1, 3, 5, 3, 61, 59, 7, 39, 439, 721, 2829, 3035, 2293, 32015, 28509, 104831}},
	{7452, 17, 14750, {1, 3, 5, 1, 29, 35, 71, 87, 351, 917, 1661, 547, 4501, 7107, 5493, 17833, 130729}},
	{7453, 17, 14759, {1, 1, 5, 5, 7, 5, 69, 57, 319, 595, 1749, 3789, 1437, 6327, 24089, 7387, 125109}},
	{7454, 17, 14763, {1, 3, 5, 9, 15, 53, 95, 59, 217, 37, 1561, 401, 5259, 4361, 1049, 3437, 30559}},
	{7455, 17, 14768, {1, 3, 7, 13, 15, 15, 107, 167, 475, 157, 1565, 2219, 1891, 1433, 11829, 43433, 48111}},
	{7456, 17, 14773, {1, 1, 1, 3, 11, 41, 25, 211, 243, 355, 1831, 2093, 2747, 2523, 9885, 9503, 120089}},
	{7457, 17, 14777, {1, 3, 7, 5, 11, 3, 1, 231, 243, 541, 341, 887, 3567, 14759, 26763, 35705, 29417}},
	{7458, 17, 14786, {1, 1, 7, 13, 17, 35, 117, 177, 81, 361, 1425, 2437, 6821, 1061, 15019, 19135, 106007}},
	{7459, 17, 14795, {1, 3, 3, 11, 19, 5, 39, 23, 367, 9, 879, 3583, 2527, 14375, 28359, 27393, 55041}},
	{7460, 17, 14800, {1, 1, 7, 11, 9, 41, 63, 125, 33, 337, 587, 3939, 2635, 4559, 1007, 38991, 35651}},
	{7461, 17, 14812, {1, 1, 3, 1, 19, 11, 83, 13, 227, 649, 415, 1661, 3285, 55, 3683, 22319, 2127}},
	{7462, 17, 14816, {1, 3, 7, 13, 19, 49, 113, 129, 83, 5, 19, 1095, 6561, 11049, 3805, 11355, 84265}},
	{7463, 17, 14836, {1, 3, 1, 9, 19, 41, 111, 193, 429, 319, 67, 1717, 1819, 12959, 31449, 21035, 113161}},
	{7464, 17, 14840, {1, 1, 5, 11, 19, 19, 115, 237, 145, 681, 1525, 2215, 7915, 15529, 7533, 45981, 85461}},
	{7465, 17, 14856, {1, 1, 1, 1, 25, 3, 73, 207, 15, 69, 43, 1643, 7707, 12505, 27101, 40735, 6091}},
	{7466, 17, 14859, {1, 1, 5, 11, 21, 61, 119, 7, 37, 147, 1379, 3165, 6555, 3867, 24027, 45161, 93015}},
	{7467, 17, 14870, {1, 1, 3, 9, 9, 25, 51, 125, 511, 209, 75, 2849, 2299, 2901, 25157, 13079, 67733}},
	{7468, 17, 14873, {1, 3, 7, 9, 31, 49, 99, 21, 89, 1, 1391, 1741, 2733, 7283, 12087, 9287, 39713}},
	{7469, 17, 14879, {1, 1, 5, 5, 1, 5, 89, 109, 499, 343, 431, 401, 2023, 5541, 16615, 40059, 119195}},
	{7470, 17, 14880, {1, 3, 5, 15, 27, 27, 9, 159, 395, 31, 865, 2793, 55, 10961, 23123, 63731, 54385}},
	{7471, 17, 14889, {1, 3, 7, 1, 11, 47, 123, 239, 399, 383, 1497, 4075, 4659, 2911, 2101, 8295, 115717}},
	{7472, 17, 14892, {1, 1, 3, 1, 11, 63, 125, 171, 65, 15, 349, 753, 2981, 6713, 6219, 14093, 78797}},
	{7473, 17, 14895, {1, 1, 1, 13, 9, 15, 1, 113, 221, 867, 1907, 103, 1411, 27, 22743, 377, 116907}},
	{7474, 17, 14900, {1, 3, 1, 5, 27, 5, 27, 245, 221, 575, 2009, 1561, 4263, 11843, 28331, 12865, 10483}},
	{7475, 17, 14903, {1, 3, 7, 9, 1, 51, 119, 241, 439, 913, 1191, 2343, 2055, 10247, 18283, 40175, 63321}},
	{7476, 17, 14910, {1, 1, 3, 15, 21, 59, 45, 151, 485, 293, 981, 3523, 7689, 2789, 5003, 62383, 126221}},
	{7477, 17, 14912, {1, 1, 1, 1, 13, 15, 39, 201, 405, 513, 1721, 2077, 5995, 2433, 20421, 12695, 20393}},
	{7478, 17, 14942, {1, 3, 5, 15, 11, 35, 113, 133, 187, 583, 577, 291, 7563, 12959, 9383, 44255, 81763}},
	{7479, 17, 14948, {1, 3, 7, 15, 9, 55, 57, 227, 189, 595, 1311, 1131, 1323, 11347, 12777, 50963, 13827}},
	{7480, 17, 14957, {1, 3, 5, 3, 11, 49, 77, 157, 107, 959, 761, 1457, 7121, 3027, 9269, 26291, 125261}},
	{7481, 17, 14963, {1, 1, 5, 9, 23, 53, 125, 211, 303, 433, 1103, 41, 2643, 5325, 11885, 23825, 80415}},
	{7482, 17, 14975, {1, 1, 7, 1, 29, 25, 51, 107, 209, 165, 707, 1855, 7429, 1583, 5941, 47509, 90105}},
	{7483, 17, 14985, {1, 1, 3, 3, 1, 15, 121, 165, 181, 259, 1949, 3049, 3545, 3093, 5967, 49207, 37129}},
	{7484, 17, 14993, {1, 1, 5, 13, 9, 59, 93, 87, 57, 343, 389, 1995, 4001, 11495, 12909, 13491, 61759}},
	{7485, 17, 15003, {1, 3, 1, 5, 11, 27, 27, 133, 459, 733, 1845, 1795, 4613, 3397, 12313, 52839, 129583}},
	{7486, 17, 15010, {1, 3, 5, 3, 19, 1, 7, 145, 255, 337, 1649, 1473, 4113, 4425, 12233, 55477, 69157}},
	{7487, 17, 15022, {1, 1, 3, 1, 25, 27, 93, 59, 415, 437, 25, 1565, 319, 8981, 2453, 53579, 45033}},
	{7488, 17, 15039, {1, 3, 7, 1, 27, 49, 47, 233, 341, 101, 2017, 2827, 8085, 237, 6363, 61139, 88903}},
	{7489, 17, 15041, {1, 1, 1, 5, 23, 47, 65, 251, 423, 957, 1751, 3541, 5405, 1335, 22703, 12587, 60201}},
	{7490, 17, 15047, {1, 1, 3, 3, 5, 51, 85, 195, 423, 519, 1797, 3821, 5915, 12257, 5377, 62733, 41197}},
	{7491, 17, 15048, {1, 3, 7, 15, 3, 47, 97, 1, 5, 175, 1449, 1609, 6873, 12017, 5579, 2665, 58389}},
	{7492, 17, 15056, {1, 3, 3, 15, 19, 37, 35, 29, 79, 767, 21, 1279, 1997, 11611, 14381, 35607, 127701}},
	{7493, 17, 15066, {1, 3, 7, 7, 7, 43, 47, 33, 69, 155, 703, 1373, 1589, 6997, 8627, 50647, 16989}},
	{7494, 17, 15068, {1, 3, 1, 5, 13, 33, 69, 133, 399, 361, 1633, 321, 2077, 8857, 13419, 23227, 40003}},
	{7495, 17, 15075, {1, 1, 1, 15, 15, 9, 45, 181, 427, 1005, 341, 1697, 6423, 5727, 7163, 10401, 38957}},
	{7496, 17, 15077, {1, 1, 7, 1, 17, 5, 17, 95, 279, 171, 825, 2459, 5243, 10683, 1849, 32809, 8995}},
	{7497, 17, 15082, {1, 3, 5, 15, 27, 47, 103, 69, 69, 255, 961, 2173, 5297, 5987, 5863, 14311, 117569}},
	{7498, 17, 15096, {1, 1, 1, 11, 21, 27, 61, 239, 183, 1013, 1955, 3171, 4183, 965, 14885, 49605, 87851}},
	{7499, 17, 15102, {1, 1, 7, 9, 9, 53, 99, 211, 267, 803, 1545, 4011, 7613, 13889, 28277, 6817, 26515}},
	{7500, 17, 15116, {1, 1, 3, 9, 1, 19, 33, 227, 461, 679, 499, 1069, 837, 12129, 20779, 12937, 104367}},
	{7501, 17, 15122, {1, 3, 3, 15, 7, 3, 29, 245, 179, 1015, 1651, 3753, 4185, 15357, 17379, 52835, 51953}},
	{7502, 17, 15127, {1, 3, 3, 3, 3, 25, 95, 239, 263, 427, 1749, 183, 5251, 361, 32549, 24331, 30789}},
	{7503, 17, 15133, {1, 1, 7, 1, 5, 3, 79, 9, 403, 195, 1433, 385, 8105, 7893, 16415, 23253, 127837}},
	{7504, 17, 15137, {1, 3, 7, 3, 23, 45, 115, 27, 473, 241, 361, 1787, 4247, 13451, 5627, 32923, 29375}},
	{7505, 17, 15138, {1, 3, 7, 1, 5, 55, 43, 37, 481, 899, 51, 2459, 5005, 12365, 19261, 32797, 45843}},
	{7506, 17, 15149, {1, 3, 7, 5, 9, 41, 83, 163, 241, 899, 567, 231, 4897, 15175, 10329, 6625, 95927}},
	{7507, 17, 15152, {1, 3, 1, 1, 7, 51, 61, 55, 253, 315, 1893, 2635, 4061, 257, 14147, 36639, 24893}},
	{7508, 17, 15155, {1, 1, 5, 1, 13, 63, 115, 119, 205, 309, 277, 2191, 341, 4715, 13111, 58043, 51241}},
	{7509, 17, 15158, {1, 3, 1, 15, 17, 23, 89, 121, 205, 15, 295, 667, 421, 14071, 27719, 1335, 9887}},
	{7510, 17, 15187, {1, 3, 5, 5, 17, 49, 5, 93, 251, 613, 1029, 945, 1547, 10479, 20183, 26787, 120441}},
	{7511, 17, 15189, {1, 3, 3, 15, 17, 11, 63, 97, 499, 313, 881, 2233, 4287, 5141, 13841, 40725, 49285}},
	{7512, 17, 15190, {1, 3, 3, 11, 19, 33, 105, 203, 325, 337, 353, 1923, 7157, 8623, 23881, 4513, 71495}},
	{7513, 17, 15196, {1, 1, 5, 1, 3, 15, 119, 43, 85, 869, 1597, 2433, 845, 5065, 12813, 64849, 58491}},
	{7514, 17, 15199, {1, 3, 7, 7, 25, 63, 119, 93, 303, 665, 571, 1795, 5853, 13527, 12715, 36483, 57723}},
	{7515, 17, 15205, {1, 3, 7, 13, 19, 43, 55, 85, 189, 627, 1457, 3185, 3491, 1913, 13399, 30681, 69015}},
	{7516, 17, 15212, {1, 3, 5, 9, 5, 41, 51, 65, 147, 425, 569, 1317, 1557, 7631, 17243, 37847, 51161}},
	{7517, 17, 15236, {1, 1, 3, 7, 29, 39, 61, 127, 489, 89, 749, 2073, 195, 14367, 13533, 27403, 16365}},
	{7518, 17, 15243, {1, 3, 7, 15, 13, 35, 45, 157, 373, 415, 725, 779, 3559, 7489, 11369, 36501, 60761}},
	{7519, 17, 15246, {1, 3, 1, 3, 13, 45, 25, 215, 385, 709, 499, 3861, 761, 15597, 3335, 37013, 13173}},
	{7520, 17, 15260, {1, 1, 7, 1, 13, 49, 89, 135, 175, 1015, 67, 957, 4893, 9843, 13027, 14709, 59721}},
	{7521, 17, 15267, {1, 3, 3, 11, 19, 37, 109, 143, 135, 535, 1543, 3991, 189, 6739, 28087, 18845, 41819}},
	{7522, 17, 15274, {1, 1, 7, 5, 1, 7, 11, 5, 211, 251, 1593, 2527, 3539, 10471, 25595, 60119, 89213}},
	{7523, 17, 15279, {1, 1, 5, 7, 13, 51, 121, 167, 299, 403, 977, 521, 279, 15521, 15901, 935, 14065}},
	{7524, 17, 15281, {1, 1, 7, 13, 7, 21, 27, 205, 377, 801, 1365, 1567, 6651, 139, 14229, 30827, 50429}},
	{7525, 17, 15282, {1, 1, 1, 1, 17, 11, 75, 87, 217, 413, 1923, 1765, 2037, 14061, 12433, 30671, 24883}},
	{7526, 17, 15284, {1, 1, 5, 13, 17, 51, 91, 241, 95, 505, 349, 2689, 1117, 4435, 1713, 44501, 125619}},
	{7527, 17, 15291, {1, 1, 3, 15, 11, 21, 25, 59, 511, 353, 799, 91, 4517, 16005, 17061, 21841, 46311}},
	{7528, 17, 15293, {1, 1, 1, 5, 19, 53, 109, 177, 213, 373, 761, 453, 5753, 69, 3503, 49411, 111105}},
	{7529, 17, 15302, {1, 3, 1, 5, 21, 27, 103, 167, 109, 55, 1849, 3999, 7801, 4185, 9789, 7515, 124983}},
	{7530, 17, 15319, {1, 3, 7, 7, 25, 9, 65, 127, 141, 169, 1079, 3377, 691, 5119, 6629, 3517, 28963}},
	{7531, 17, 15329, {1, 1, 3, 11, 15, 61, 127, 35, 87, 891, 1459, 483, 6763, 16173, 5633, 6939, 63411}},
	{7532, 17, 15336, {1, 3, 5, 3, 7, 63, 111, 85, 415, 273, 1705, 4045, 5551, 2377, 29025, 16831, 90203}},
	{7533, 17, 15347, {1, 3, 5, 13, 7, 23, 103, 227, 477, 985, 1059, 1489, 7233, 1917, 10409, 38759, 86761}},
	{7534, 17, 15353, {1, 3, 5, 7, 31, 33, 75, 41, 355, 577, 225, 5, 897, 15653, 27415, 83, 14911}},
	{7535, 17, 15361, {1, 3, 5, 5, 23, 13, 5, 43, 165, 53, 149, 2005, 4545, 477, 17885, 21343, 35751}},
	{7536, 17, 15371, {1, 3, 3, 3, 25, 51, 33, 203, 291, 835, 241, 3255, 3709, 3573, 9859, 33027, 122801}},
	{7537, 17, 15397, {1, 3, 5, 7, 13, 7, 3, 141, 455, 67, 2003, 3411, 4717, 157, 29491, 14429, 44849}},
	{7538, 17, 15404, {1, 1, 1, 11, 3, 33, 101, 93, 219, 371, 1191, 1521, 1663, 8485, 24815, 38283, 120867}},
	{7539, 17, 15407, {1, 1, 3, 9, 25, 61, 71, 173, 69, 181, 1525, 2129, 2979, 19, 13489, 627, 72619}},
	{7540, 17, 15421, {1, 3, 1, 7, 25, 33, 39, 247, 221, 7, 683, 1837, 8037, 9125, 4259, 63049, 63021}},
	{7541, 17, 15433, {1, 3, 3, 9, 17, 15, 9, 189, 357, 707, 521, 711, 8189, 12945, 29675, 11851, 126813}},
	{7542, 17, 15441, {1, 3, 1, 1, 23, 3, 57, 133, 245, 301, 957, 239, 3139, 7949, 27133, 18229, 93015}},
	{7543, 17, 15442, {1, 1, 3, 1, 29, 23, 35, 87, 231, 257, 1997, 271, 3019, 3409, 10613, 42245, 111309}},
	{7544, 17, 15463, {1, 1, 3, 3, 1, 21, 17, 37, 393, 943, 791, 3101, 6715, 11907, 25369, 9061, 75381}},
	{7545, 17, 15472, {1, 1, 7, 7, 17, 31, 25, 7, 183, 819, 1265, 3343, 6845, 2039, 3779, 41861, 38309}},
	{7546, 17, 15478, {1, 1, 1, 5, 17, 25, 1, 41, 173, 995, 863, 3515, 1779, 2159, 28223, 64661, 40697}},
	{7547, 17, 15488, {1, 1, 3, 15, 29, 49, 81, 241, 511, 817, 1301, 3593, 6759, 7483, 8859, 30339, 106137}},
	{7548, 17, 15493, {1, 3, 1, 11, 17, 61, 95, 231, 3, 693, 37, 1091, 3111, 11941, 17475, 8073, 62373}},
	{7549, 17, 15498, {1, 1, 1, 3, 7, 25, 93, 7, 291, 957, 859, 2519, 241, 10963, 10403, 933, 50599}},
	{7550, 17, 15511, {1, 1, 7, 1, 7, 33, 121, 91, 369, 333, 229, 4073, 6063, 6491, 31711, 65061, 107843}},
	{7551, 17, 15521, {1, 1, 5, 15, 17, 1, 117, 195, 445, 547, 867, 2893, 4835, 6513, 29091, 60367, 33409}},
	{7552, 17, 15534, {1, 3, 3, 7, 15, 5, 125, 131, 165, 127, 207, 853, 5927, 3605, 17083, 44481, 111333}},
	{7553, 17, 15539, {1, 1, 1, 9, 3, 43, 75, 191, 319, 889, 1513, 3301, 1535, 4693, 10367, 12491, 43175}},
	{7554, 17, 15541, {1, 3, 5, 5, 29, 19, 75, 221, 393, 977, 1373, 1571, 7377, 1763, 18073, 11381, 101241}},
	{7555, 17, 15560, {1, 3, 1, 15, 3, 15, 73, 91, 165, 213, 1077, 1267, 2411, 15807, 3979, 12731, 86597}},
	{7556, 17, 15563, {1, 1, 3, 7, 3, 21, 5, 135, 95, 337, 1853, 1675, 2449, 12535, 18505, 60127, 76949}},
	{7557, 17, 15574, {1, 1, 7, 3, 15, 11, 63, 127, 329, 169, 1569, 675, 4801, 5859, 3243, 25811, 77841}},
	{7558, 17, 15578, {1, 1, 1, 9, 19, 13, 73, 119, 105, 537, 951, 1033, 5303, 5775, 815, 19277, 57607}},
	{7559, 17, 15584, {1, 3, 5, 5, 23, 21, 91, 231, 117, 1007, 1603, 841, 2595, 11223, 17171, 25963, 17049}},
	{7560, 17, 15593, {1, 1, 5, 11, 15, 43, 7, 229, 55, 129, 599, 993, 563, 15677, 16703, 36253, 17847}},
	{7561, 17, 15604, {1, 3, 1, 9, 25, 3, 109, 21, 87, 721, 1927, 3219, 3395, 3267, 9117, 13591, 89267}},
	{7562, 17, 15614, {1, 1, 1, 15, 11, 17, 47, 49, 125, 925, 333, 945, 2411, 10907, 12021, 47857, 84303}},
	{7563, 17, 15619, {1, 1, 1, 1, 23, 11, 99, 215, 105, 417, 823, 1289, 421, 12285, 17711, 35389, 1935}},
	{7564, 17, 15622, {1, 1, 1, 15, 27, 7, 23, 141, 7, 929, 147, 681, 5473, 4173, 28645, 42053, 83573}},
	{7565, 17, 15633, {1, 1, 1, 11, 5, 61, 71, 65, 287, 697, 1183, 3257, 7251, 14011, 21349, 42445, 4701}},
	{7566, 17, 15639, {1, 3, 5, 9, 5, 23, 45, 217, 369, 189, 1495, 107, 425, 10467, 4909, 64293, 17885}},
	{7567, 17, 15646, {1, 1, 3, 11, 21, 45, 75, 65, 57, 893, 783, 3429, 409, 13617, 483, 62489, 2919}},
	{7568, 17, 15673, {1, 1, 7, 3, 5, 61, 51, 255, 501, 839, 367, 1165, 7055, 8139, 23891, 18807, 20739}},
	{7569, 17, 15674, {1, 1, 3, 7, 23, 15, 97, 139, 323, 463, 921, 1529, 6655, 8697, 23577, 56761, 62023}},
	{7570, 17, 15684, {1, 1, 5, 11, 13, 11, 57, 225, 277, 713, 1427, 95, 1135, 7721, 30731, 32625, 107891}},
	{7571, 17, 15691, {1, 1, 5, 7, 23, 35, 39, 91, 291, 609, 919, 3325, 6843, 7659, 5603, 37471, 41495}},
	{7572, 17, 15694, {1, 1, 1, 1, 25, 11, 117, 15, 389, 589, 1345, 423, 6531, 9903, 20243, 9523, 22991}},
	{7573, 17, 15696, {1, 3, 5, 15, 29, 7, 57, 113, 387, 883, 1141, 3295, 2973, 4129, 16973, 33429, 109997}},
	{7574, 17, 15701, {1, 3, 5, 3, 25, 1, 73, 207, 353, 203, 1479, 985, 6373, 3079, 28403, 63675, 21787}},
	{7575, 17, 15705, {1, 3, 1, 5, 31, 39, 107, 197, 359, 45, 203, 559, 4721, 6579, 11305, 12957, 10061}},
	{7576, 17, 15715, {1, 3, 7, 15, 9, 3, 55, 153, 373, 981, 575, 827, 4757, 15743, 14295, 43875, 17847}},
	{7577, 17, 15729, {1, 3, 5, 1, 17, 1, 93, 87, 207, 997, 1695, 3643, 6973, 9507, 29309, 58531, 6849}},
	{7578, 17, 15730, {1, 3, 1, 11, 3, 39, 17, 241, 83, 931, 39, 3839, 6437, 5159, 28869, 61859, 96873}},
	{7579, 17, 15741, {1, 1, 5, 13, 29, 43, 71, 159, 261, 563, 695, 1205, 2273, 8077, 12569, 17187, 54369}},
	{7580, 17, 15745, {1, 3, 7, 5, 11, 57, 17, 31, 311, 1001, 1419, 3899, 6679, 15531, 28877, 28221, 105413}},
	{7581, 17, 15748, {1, 3, 3, 13, 23, 29, 127, 19, 345, 1003, 1571, 2219, 3199, 9903, 18701, 31865, 108879}},
	{7582, 17, 15757, {1, 3, 7, 13, 23, 51, 95, 43, 35, 439, 25, 323, 2365, 12407, 27525, 57795, 74495}},
	{7583, 17, 15765, {1, 1, 1, 1, 17, 43, 57, 185, 439, 929, 69, 813, 6205, 3139, 3853, 56967, 19073}},
	{7584, 17, 15766, {1, 3, 7, 1, 27, 5, 43, 211, 395, 113, 1675, 1505, 6171, 5169, 9991, 21641, 27101}},
	{7585, 17, 15775, {1, 3, 3, 1, 17, 41, 59, 131, 131, 339, 955, 1145, 5301, 4585, 20441, 43227, 23123}},
	{7586, 17, 15776, {1, 1, 7, 9, 9, 55, 61, 31, 71, 229, 963, 3247, 4677, 9595, 21715, 36391, 86997}},
	{7587, 17, 15779, {1, 1, 7, 5, 9, 17, 55, 179, 27, 229, 79, 1335, 5887, 1003, 22085, 34377, 51367}},
	{7588, 17, 15786, {1, 1, 1, 5, 11, 45, 15, 219, 411, 27, 1003, 1553, 303, 13571, 13985, 6801, 52407}},
	{7589, 17, 15788, {1, 3, 3, 7, 7, 55, 111, 255, 453, 409, 1863, 1449, 4103, 8725, 26923, 5017, 43657}},
	{7590, 17, 15813, {1, 1, 1, 15, 23, 3, 95, 57, 29, 727, 1111, 3309, 1089, 471, 16099, 11517, 51563}},
	{7591, 17, 15814, {1, 3, 1, 15, 17, 57, 83, 163, 251, 987, 1159, 2079, 3463, 13109, 7443, 8665, 123397}},
	{7592, 17, 15842, {1, 1, 7, 1, 27, 13, 35, 209, 471, 843, 1029, 1383, 5413, 2085, 13431, 26557, 47033}},
	{7593, 17, 15851, {1, 3, 1, 1, 21, 21, 83, 135, 303, 27, 1407, 1751, 331, 9207, 31891, 59287, 120687}},
	{7594, 17, 15862, {1, 1, 1, 9, 11, 35, 103, 157, 1, 855, 175, 3203, 4381, 3113, 27589, 4567, 31897}},
	{7595, 17, 15875, {1, 1, 3, 5, 21, 5, 123, 161, 301, 101, 909, 947, 6893, 15459, 29139, 49377, 94901}},
	{7596, 17, 15878, {1, 3, 7, 7, 21, 27, 5, 69, 427, 409, 1389, 3737, 847, 2775, 603, 1001, 87651}},
	{7597, 17, 15889, {1, 1, 3, 5, 1, 57, 109, 89, 99, 593, 581, 3527, 1557, 4971, 27523, 26909, 35787}},
	{7598, 17, 15896, {1, 1, 7, 3, 31, 19, 83, 65, 239, 919, 15, 2289, 4117, 9127, 6033, 49667, 89343}},
	{7599, 17, 15901, {1, 3, 7, 7, 9, 31, 87, 117, 195, 681, 1711, 1753, 2221, 10053, 1985, 6273, 21801}},
	{7600, 17, 15908, {1, 3, 1, 7, 21, 61, 53, 231, 309, 115, 1729, 3883, 6085, 4825, 31455, 50097, 59779}},
	{7601, 17, 15911, {1, 1, 1, 9, 29, 25, 45, 91, 145, 927, 147, 371, 2603, 12537, 17267, 59895, 128009}},
	{7602, 17, 15915, {1, 1, 1, 1, 15, 41, 63, 43, 167, 215, 15, 3387, 1811, 12391, 25721, 6961, 13701}},
	{7603, 17, 15938, {1, 1, 7, 1, 27, 63, 25, 85, 337, 799, 87, 2237, 4085, 14529, 11493, 60149, 86399}},
	{7604, 17, 15944, {1, 3, 1, 11, 1, 41, 103, 145, 279, 805, 1201, 823, 5411, 4227, 25999, 14373, 36295}},
	{7605, 17, 15950, {1, 1, 7, 3, 27, 51, 83, 105, 155, 657, 1879, 3869, 2559, 2939, 19785, 47167, 34503}},
	{7606, 17, 15955, {1, 3, 1, 5, 3, 31, 47, 241, 257, 15, 983, 4095, 3745, 3901, 1639, 5421, 81585}},
	{7607, 17, 15974, {1, 3, 3, 5, 31, 13, 127, 125, 175, 577, 1103, 3573, 6229, 13969, 6267, 19067, 3933}},
	{7608, 17, 15978, {1, 1, 7, 1, 31, 17, 15, 15, 411, 553, 1929, 3731, 1955, 11749, 21991, 39189, 124427}},
	{7609, 17, 15980, {1, 3, 5, 5, 19, 63, 93, 201, 491, 599, 1093, 767, 3411, 13087, 23569, 42981, 35757}},
	{7610, 17, 15983, {1, 1, 1, 15, 27, 7, 51, 101, 429, 939, 111, 781, 2055, 14227, 17821, 42097, 32485}},
	{7611, 17, 15991, {1, 3, 7, 13, 11, 21, 3, 161, 353, 389, 285, 2633, 6245, 7089, 21907, 40765, 88869}},
	{7612, 17, 16004, {1, 1, 5, 9, 7, 27, 101, 203, 243, 897, 1375, 1619, 5275, 12935, 22103, 38005, 65603}},
	{7613, 17, 16011, {1, 1, 5, 9, 13, 25, 15, 21, 447, 7, 947, 1613, 5055, 129, 18057, 58551, 6603}},
	{7614, 17, 16016, {1, 3, 7, 15, 17, 41, 11, 55, 103, 339, 349, 1813, 7423, 11837, 20641, 51951, 61615}},
	{7615, 17, 16019, {1, 3, 3, 15, 21, 59, 113, 3, 123, 689, 465, 3039, 4109, 3241, 30317, 65053, 117845}},
	{7616, 17, 16025, {1, 3, 3, 1, 31, 33, 73, 155, 245, 401, 473, 51, 1387, 489, 10573, 55401, 106733}},
	{7617, 17, 16041, {1, 3, 3, 1, 31, 37, 15, 139, 127, 201, 229, 1753, 7287, 9045, 18321, 63485, 26399}},
	{7618, 17, 16064, {1, 3, 5, 5, 5, 23, 93, 3, 125, 715, 1827, 419, 1213, 9031, 25139, 20771, 41345}},
	{7619, 17, 16067, {1, 3, 5, 15, 23, 15, 13, 145, 105, 477, 1131, 2699, 1929, 10447, 9655, 26791, 80101}},
	{7620, 17, 16074, {1, 1, 1, 13, 1, 35, 75, 73, 269, 851, 737, 1909, 6805, 11359, 28991, 52435, 83767}},
	{7621, 17, 16082, {1, 1, 7, 5, 11, 31, 31, 91, 111, 161, 1865, 2545, 133, 12215, 8957, 20671, 92975}},
	{7622, 17, 16103, {1, 1, 7, 5, 25, 53, 55, 121, 53, 457, 831, 2493, 339, 10955, 30783, 9095, 97921}},
	{7623, 17, 16109, {1, 1, 5, 3, 25, 33, 81, 51, 211, 737, 1865, 4039, 6931, 8473, 22459, 24885, 96355}},
	{7624, 17, 16122, {1, 3, 7, 13, 23, 5, 101, 171, 65, 793, 443, 411, 7629, 14791, 28633, 9055, 123763}},
	{7625, 17, 16149, {1, 3, 3, 1, 11, 7, 99, 79, 461, 481, 1689, 3777, 2125, 4783, 13061, 19537, 68109}},
	{7626, 17, 16170, {1, 1, 3, 11, 31, 53, 109, 7, 49, 925, 1017, 2371, 1537, 13557, 75, 40677, 49181}},
	{7627, 17, 16175, {1, 3, 3, 3, 9, 1, 95, 113, 189, 389, 377, 393, 6523, 3183, 6461, 30201, 66549}},
	{7628, 17, 16178, {1, 1, 7, 15, 13, 19, 41, 171, 475, 157, 949, 3245, 5581, 2783, 25263, 53023, 11155}},
	{7629, 17, 16189, {1, 3, 5, 7, 29, 63, 61, 65, 315, 595, 905, 899, 5059, 4243, 27287, 14023, 64213}},
	{7630, 17, 16202, {1, 3, 1, 3, 15, 37, 109, 161, 9, 867, 1023, 2513, 4593, 7747, 1505, 4801, 127091}},
	{7631, 17, 16204, {1, 3, 1, 7, 11, 59, 75, 129, 469, 695, 63, 2757, 6357, 8675, 6193, 23439, 66445}},
	{7632, 17, 16222, {1, 1, 3, 13, 17, 9, 47, 91, 161, 265, 139, 129, 6707, 9659, 8917, 54757, 77835}},
	{7633, 17, 16231, {1, 1, 3, 13, 19, 37, 113, 255, 99, 913, 1445, 487, 337, 1001, 16395, 37141, 66595}},
	{7634, 17, 16238, {1, 1, 1, 15, 3, 63, 69, 43, 185, 293, 1137, 2061, 2377, 8741, 26817, 5833, 7807}},
	{7635, 17, 16262, {1, 1, 1, 5, 3, 29, 39, 33, 263, 355, 597, 539, 5055, 13075, 8977, 19829, 88171}},
	{7636, 17, 16265, {1, 3, 7, 9, 17, 49, 125, 101, 447, 597, 1337, 559, 2807, 7925, 12421, 17427, 34815}},
	{7637, 17, 16276, {1, 3, 1, 9, 11, 57, 31, 163, 503, 925, 911, 3721, 2515, 8429, 25749, 55209, 90105}},
	{7638, 17, 16285, {1, 3, 5, 3, 21, 57, 119, 233, 319, 745, 563, 3057, 2683, 7063, 11513, 49157, 64561}},
	{7639, 17, 16313, {1, 1, 3, 9, 15, 21, 93, 99, 227, 479, 965, 51, 6941, 9887, 32409, 23171, 98387}},
	{7640, 17, 16314, {1, 3, 5, 5, 19, 1, 47, 49, 233, 931, 971, 2369, 2827, 1291, 18653, 725, 19791}},
	{7641, 17, 16321, {1, 1, 5, 15, 3, 7, 71, 251, 341, 861, 1203, 793, 7627, 10929, 10717, 10677, 49743}},
	{7642, 17, 16327, {1, 3, 1, 7, 3, 43, 9, 187, 247, 621, 1069, 2875, 1525, 4221, 18813, 35807, 117609}},
	{7643, 17, 16333, {1, 3, 3, 3, 29, 39, 83, 201, 205, 337, 231, 547, 2893, 2483, 6197, 26869, 18921}},
	{7644, 17, 16334, {1, 1, 7, 3, 23, 29, 33, 137, 491, 691, 979, 65, 5711, 11685, 5137, 37993, 37075}},
	{7645, 17, 16364, {1, 3, 3, 1, 11, 3, 99, 119, 203, 901, 1887, 879, 7547, 4613, 31233, 13279, 105089}},
	{7646, 17, 16369, {1, 1, 1, 13, 25, 23, 111, 167, 313, 141, 127, 1223, 5711, 4101, 10977, 34695, 128303}},
	{7647, 17, 16370, {1, 1, 7, 15, 5, 3, 89, 151, 289, 769, 539, 2883, 8121, 15403, 22345, 63765, 117015}},
	{7648, 17, 16375, {1, 1, 1, 13, 15, 9, 71, 95, 37, 705, 1575, 3735, 7445, 2027, 27523, 53321, 106085}},
	{7649, 17, 16376, {1, 3, 5, 7, 5, 29, 7, 25, 181, 491, 1173, 1947, 3321, 9233, 17265, 26999, 97783}},
	{7650, 17, 16379, {1, 1, 3, 15, 1, 63, 111, 113, 279, 123, 345, 1529, 2725, 8643, 8551, 30073, 26689}},
	{7651, 17, 16393, {1, 3, 7, 7, 5, 55, 117, 211, 293, 851, 1491, 3265, 4009, 14949, 10297, 16219, 69983}},
	{7652, 17, 16402, {1, 1, 3, 11, 23, 45, 35, 91, 97, 191, 417, 3545, 1733, 3955, 10763, 10229, 75027}},
	{7653, 17, 16408, {1, 1, 3, 13, 3, 61, 69, 205, 379, 627, 295, 3979, 85, 11305, 2493, 35583, 3133}},
	{7654, 17, 16418, {1, 3, 5, 9, 5, 63, 67, 201, 351, 367, 1009, 739, 5409, 8715, 28939, 31511, 34599}},
	{7655, 17, 16438, {1, 1, 1, 5, 3, 25, 21, 25, 477, 301, 623, 157, 563, 9457, 24515, 30135, 107165}},
	{7656, 17, 16441, {1, 1, 3, 15, 5, 41, 49, 171, 469, 427, 857, 2165, 1437, 2151, 24061, 63243, 105331}},
	{7657, 17, 16447, {1, 3, 5, 11, 21, 25, 59, 167, 29, 653, 1503, 2223, 3889, 4605, 28381, 36075, 74907}},
	{7658, 17, 16450, {1, 3, 7, 7, 17, 55, 73, 127, 33, 319, 1565, 2761, 6473, 2187, 19939, 56687, 112137}},
	{7659, 17, 16455, {1, 1, 1, 9, 7, 53, 105, 3, 299, 15, 1009, 607, 6885, 12875, 20719, 16841, 70471}},
	{7660, 17, 16459, {1, 3, 5, 9, 7, 33, 23, 163, 279, 739, 1541, 3017, 2309, 11827, 3875, 44337, 82063}},
	{7661, 17, 16483, {1, 1, 1, 5, 19, 53, 109, 193, 331, 339, 477, 4093, 5177, 13527, 25731, 64137, 81411}},
	{7662, 17, 16490, {1, 3, 7, 13, 15, 63, 101, 145, 127, 13, 1431, 3581, 4993, 14287, 12125, 60217, 102563}},
	{7663, 17, 16492, {1, 3, 1, 7, 17, 27, 127, 81, 223, 763, 761, 2061, 1031, 12251, 14141, 23587, 124813}},
	{7664, 17, 16495, {1, 3, 5, 13, 27, 21, 9, 249, 285, 875, 65, 4075, 6749, 13417, 3079, 29343, 87075}},
	{7665, 17, 16523, {1, 3, 5, 13, 1, 31, 61, 21, 169, 145, 1681, 1229, 5059, 13555, 21373, 35597, 70669}},
	{7666, 17, 16528, {1, 3, 7, 15, 23, 31, 43, 237, 139, 9, 1905, 3197, 801, 14205, 13323, 18717, 88523}},
	{7667, 17, 16543, {1, 1, 1, 11, 1, 7, 21, 83, 15, 459, 537, 4029, 6973, 4019, 1, 35147, 16329}},
	{7668, 17, 16553, {1, 3, 7, 15, 23, 11, 17, 101, 235, 683, 913, 3529, 4363, 13899, 3603, 27741, 74143}},
	{7669, 17, 16562, {1, 1, 7, 7, 3, 3, 91, 107, 499, 723, 315, 2805, 5909, 11041, 18281, 54981, 76041}},
	{7670, 17, 16564, {1, 3, 7, 9, 15, 7, 93, 171, 275, 647, 655, 3565, 2199, 14795, 21945, 9373, 122299}},
	{7671, 17, 16576, {1, 1, 1, 5, 27, 53, 73, 27, 431, 707, 53, 1281, 49, 13199, 1973, 18935, 114821}},
	{7672, 17, 16588, {1, 1, 3, 3, 25, 1, 17, 159, 217, 413, 1393, 2119, 5611, 7659, 6003, 19927, 22287}},
	{7673, 17, 16612, {1, 1, 7, 15, 29, 59, 77, 9, 205, 795, 627, 2167, 2477, 6841, 17663, 34871, 79823}},
	{7674, 17, 16630, {1, 3, 5, 9, 13, 35, 79, 237, 11, 335, 789, 2291, 13, 853, 20373, 39049, 407}},
	{7675, 17, 16654, {1, 1, 5, 7, 13, 27, 21, 173, 137, 659, 123, 2677, 2153, 14879, 26737, 56291, 47613}},
	{7676, 17, 16656, {1, 3, 5, 15, 23, 47, 15, 109, 311, 597, 261, 2407, 8139, 3215, 28169, 60731, 79937}},
	{7677, 17, 16668, {1, 3, 3, 5, 11, 61, 71, 29, 189, 741, 1171, 397, 2669, 10627, 20037, 51703, 6697}},
	{7678, 17, 16672, {1, 3, 3, 3, 9, 41, 125, 1, 381, 399, 349, 3265, 6337, 8113, 14869, 5305, 83409}},
	{7679, 17, 16675, {1, 1, 3, 13, 5, 19, 33, 225, 45, 55, 1809, 1037, 5443, 15719, 9963, 363, 15145}},
	{7680, 17, 16678, {1, 3, 7, 1, 31, 25, 103, 29, 207, 169, 305, 913, 7501, 15323, 10575, 13477, 65245}},
	{7681, 17, 16681, {1, 3, 3, 15, 13, 23, 69, 255, 333, 157, 279, 1989, 3439, 12955, 13649, 52431, 90009}},
	{7682, 17, 16689, {1, 3, 7, 5, 23, 61, 111, 121, 79, 469, 89, 1545, 3405, 12393, 2035, 15989, 84855}},
	{7683, 17, 16699, {1, 1, 7, 5, 17, 21, 127, 151, 283, 521, 5, 3023, 5365, 11633, 21177, 42207, 48925}},
	{7684, 17, 16719, {1, 3, 7, 5, 21, 21, 61, 17, 415, 879, 1485, 3727, 935, 9899, 23241, 651, 103701}},
	{7685, 17, 16734, {1, 3, 5, 15, 31, 47, 19, 245, 249, 467, 253, 1575, 337, 863, 19353, 13153, 125453}},
	{7686, 17, 16737, {1, 1, 7, 15, 9, 41, 39, 63, 139, 875, 1011, 1961, 1627, 7461, 28961, 47195, 16239}},
	{7687, 17, 16750, {1, 3, 3, 7, 27, 55, 51, 245, 231, 619, 43, 91, 2125, 2685, 23661, 10189, 43085}},
	{7688, 17, 16752, {1, 1, 7, 9, 27, 55, 35, 139, 187, 143, 1545, 2685, 3173, 12065, 21607, 42619, 105279}},
	{7689, 17, 16757, {1, 1, 5, 3, 29, 63, 15, 197, 49, 995, 389, 1959, 2441, 11509, 31753, 40539, 26989}},
	{7690, 17, 16761, {1, 3, 7, 15, 19, 37, 17, 37, 305, 469, 945, 2335, 1493, 13843, 19905, 49031, 107893}},
	{7691, 17, 16773, {1, 3, 1, 11, 3, 35, 113, 181, 223, 27, 485, 2435, 3423, 11321, 1687, 45755, 18017}},
	{7692, 17, 16774, {1, 3, 3, 13, 17, 47, 109, 145, 287, 769, 1373, 3423, 1251, 14357, 3209, 28363, 97987}},
	{7693, 17, 16801, {1, 1, 3, 13, 7, 25, 93, 11, 23, 331, 517, 1705, 1957, 291, 763, 10411, 120367}},
	{7694, 17, 16802, {1, 3, 7, 15, 25, 9, 1, 33, 83, 61, 97, 509, 5387, 8701, 14243, 31883, 7375}},
	{7695, 17, 16822, {1, 3, 1, 5, 19, 11, 59, 95, 265, 205, 533, 1857, 693, 12469, 24445, 19449, 130623}},
	{7696, 17, 16831, {1, 1, 7, 7, 1, 5, 15, 159, 333, 361, 391, 1889, 2645, 15115, 30709, 60515, 13315}},
	{7697, 17, 16840, {1, 3, 5, 15, 25, 61, 69, 213, 183, 575, 1573, 3147, 1753, 2387, 23063, 12853, 108507}},
	{7698, 17, 16854, {1, 1, 1, 15, 17, 31, 11, 177, 411, 23, 469, 3985, 2159, 2273, 14175, 20425, 107741}},
	{7699, 17, 16858, {1, 1, 3, 9, 5, 35, 55, 225, 263, 641, 1393, 1277, 595, 2671, 7039, 64999, 114387}},
	{7700, 17, 16863, {1, 1, 3, 3, 11, 23, 1, 161, 77, 755, 1325, 1773, 4291, 13119, 29677, 27295, 81713}},
	{7701, 17, 16867, {1, 1, 5, 13, 31, 45, 115, 141, 449, 171, 1413, 2411, 7937, 10859, 19453, 64403, 45169}},
	{7702, 17, 16876, {1, 3, 5, 7, 1, 27, 117, 157, 99, 119, 1281, 2633, 5117, 16009, 19545, 7421, 30807}},
	{7703, 17, 16891, {1, 1, 3, 13, 19, 11, 61, 239, 331, 731, 1723, 1773, 2623, 15255, 17197, 63793, 100433}},
	{7704, 17, 16894, {1, 3, 7, 11, 11, 7, 119, 33, 195, 521, 811, 2599, 3113, 5497, 16751, 2541, 21813}},
	{7705, 17, 16898, {1, 1, 1, 15, 23, 47, 25, 73, 429, 213, 557, 1613, 7055, 7211, 2225, 1345, 58033}},
	{7706, 17, 16907, {1, 1, 1, 13, 15, 39, 69, 71, 11, 543, 267, 2803, 4853, 9819, 603, 4629, 78343}},
	{7707, 17, 16915, {1, 1, 7, 1, 15, 55, 47, 223, 63, 679, 1135, 3225, 3845, 12031, 6761, 20337, 29021}},
	{7708, 17, 16917, {1, 1, 3, 3, 3, 51, 127, 103, 43, 379, 169, 2549, 7775, 2553, 27415, 30671, 34043}},
	{7709, 17, 16922, {1, 1, 3, 11, 1, 31, 89, 113, 475, 857, 499, 3901, 5343, 8819, 4503, 58757, 60513}},
	{7710, 17, 16924, {1, 3, 5, 11, 27, 49, 97, 217, 91, 971, 1835, 3447, 2021, 3747, 20533, 13659, 84007}},
	{7711, 17, 16933, {1, 1, 5, 1, 31, 39, 49, 21, 135, 983, 579, 3509, 3611, 15101, 29781, 49941, 14353}},
	{7712, 17, 16938, {1, 1, 1, 9, 7, 17, 55, 233, 295, 161, 823, 3823, 4771, 13531, 24197, 42629, 60269}},
	{7713, 17, 16952, {1, 1, 3, 15, 23, 5, 101, 167, 55, 297, 1733, 3819, 7041, 9915, 27803, 60359, 10249}},
	{7714, 17, 16960, {1, 1, 7, 9, 25, 47, 67, 253, 303, 313, 1389, 3785, 2729, 11471, 27267, 42783, 111595}},
	{7715, 17, 16963, {1, 1, 5, 13, 25, 63, 17, 195, 457, 793, 1553, 1673, 6799, 12171, 9003, 22195, 90229}},
	{7716, 17, 16969, {1, 1, 3, 15, 11, 43, 43, 221, 423, 985, 873, 599, 1753, 4875, 7149, 34625, 8941}},
	{7717, 17, 16978, {1, 3, 5, 11, 1, 7, 109, 163, 309, 477, 1291, 3019, 1933, 14055, 15005, 1141, 66867}},
	{7718, 17, 17014, {1, 3, 3, 15, 21, 35, 95, 131, 413, 1009, 147, 2165, 6333, 8313, 20873, 18377, 23579}},
	{7719, 17, 17020, {1, 3, 1, 5, 21, 49, 29, 187, 67, 419, 253, 2345, 3179, 12331, 23127, 8799, 102493}},
	{7720, 17, 17034, {1, 1, 7, 5, 29, 59, 13, 189, 377, 595, 1893, 527, 7993, 14867, 24671, 14585, 38645}},
	{7721, 17, 17036, {1, 3, 5, 13, 3, 11, 99, 69, 253, 833, 1961, 2719, 3953, 8143, 21277, 16257, 26929}},
	{7722, 17, 17042, {1, 3, 7, 3, 3, 19, 19, 57, 393, 187, 945, 2107, 669, 14785, 13895, 26907, 92439}},
	{7723, 17, 17047, {1, 3, 5, 15, 11, 5, 73, 167, 99, 887, 1213, 2019, 3781, 14345, 30249, 16215, 1893}},
	{7724, 17, 17051, {1, 1, 5, 1, 17, 11, 69, 145, 97, 393, 1587, 2513, 1011, 6933, 7945, 41387, 34361}},
	{7725, 17, 17054, {1, 1, 5, 1, 5, 59, 57, 1, 501, 855, 1485, 977, 4981, 7631, 31853, 30737, 103023}},
	{7726, 17, 17063, {1, 3, 1, 5, 3, 27, 55, 171, 317, 641, 1875, 2523, 1631, 4971, 18743, 25119, 118913}},
	{7727, 17, 17069, {1, 1, 3, 15, 7, 39, 73, 209, 125, 29, 1031, 1569, 1793, 5461, 985, 59441, 92997}},
	{7728, 17, 17075, {1, 3, 5, 11, 27, 23, 57, 13, 65, 555, 1309, 1149, 5125, 11573, 3835, 57913, 78699}},
	{7729, 17, 17077, {1, 3, 7, 5, 29, 7, 51, 131, 443, 623, 1491, 1067, 6647, 6277, 25799, 54843, 90869}},
	{7730, 17, 17089, {1, 1, 1, 11, 7, 33, 67, 113, 319, 665, 11, 1225, 3137, 16269, 20101, 40263, 31091}},
	{7731, 17, 17090, {1, 3, 5, 15, 7, 5, 101, 153, 165, 173, 97, 1651, 6633, 6071, 29079, 35641, 77305}},
	{7732, 17, 17107, {1, 3, 7, 13, 9, 45, 103, 55, 121, 1021, 1841, 315, 8127, 6547, 1093, 7181, 39575}},
	{7733, 17, 17126, {1, 3, 3, 11, 15, 17, 27, 55, 341, 443, 377, 681, 3635, 1091, 16719, 49403, 85507}},
	{7734, 17, 17135, {1, 3, 5, 5, 29, 53, 51, 213, 273, 475, 981, 549, 539, 14989, 4037, 23911, 45997}},
	{7735, 17, 17150, {1, 3, 5, 3, 27, 37, 73, 115, 331, 911, 991, 4049, 6299, 3919, 10231, 31507, 98651}},
	{7736, 17, 17162, {1, 1, 5, 13, 21, 13, 1, 175, 137, 837, 1067, 2845, 307, 4399, 15671, 1309, 107409}},
	{7737, 17, 17169, {1, 1, 3, 1, 5, 47, 111, 75, 193, 389, 157, 3731, 6237, 5053, 9933, 28413, 32939}},
	{7738, 17, 17172, {1, 1, 7, 5, 29, 1, 51, 85, 267, 935, 1021, 3135, 3135, 9263, 32597, 6779, 71473}},
	{7739, 17, 17175, {1, 3, 5, 9, 21, 59, 27, 99, 155, 507, 1911, 3501, 4307, 6755, 17127, 29815, 1577}},
	{7740, 17, 17176, {1, 1, 5, 1, 15, 63, 45, 105, 125, 299, 689, 3935, 7229, 5007, 25003, 30453, 27819}},
	{7741, 17, 17191, {1, 1, 7, 15, 19, 9, 67, 151, 45, 985, 2015, 833, 5435, 15383, 25881, 46735, 56717}},
	{7742, 17, 17209, {1, 1, 5, 15, 27, 59, 119, 163, 293, 63, 1251, 1309, 485, 4937, 27207, 47481, 114357}},
	{7743, 17, 17218, {1, 3, 5, 13, 23, 11, 111, 87, 329, 467, 1657, 3309, 3421, 12013, 23163, 14105, 88761}},
	{7744, 17, 17220, {1, 1, 5, 11, 17, 63, 9, 61, 299, 585, 341, 3375, 3213, 15953, 11455, 5333, 66889}},
	{7745, 17, 17227, {1, 3, 5, 5, 5, 35, 57, 235, 137, 543, 77, 2811, 857, 12793, 10791, 55711, 93353}},
	{7746, 17, 17229, {1, 3, 7, 3, 23, 37, 19, 81, 321, 23, 1625, 2359, 3569, 4685, 7385, 32677, 18073}},
	{7747, 17, 17238, {1, 3, 3, 7, 21, 35, 81, 229, 207, 547, 1397, 2709, 7159, 1265, 16823, 9921, 29159}},
	{7748, 17, 17251, {1, 3, 7, 13, 27, 13, 107, 241, 395, 317, 307, 3927, 1153, 15915, 25179, 25173, 21503}},
	{7749, 17, 17257, {1, 3, 1, 5, 1, 51, 25, 135, 381, 229, 1491, 2009, 3331, 16165, 8169, 65161, 9335}},
	{7750, 17, 17258, {1, 1, 5, 5, 17, 15, 57, 221, 183, 225, 1649, 3701, 299, 12349, 4691, 64479, 82237}},
	{7751, 17, 17272, {1, 3, 7, 7, 31, 39, 65, 183, 149, 67, 1697, 3933, 3709, 15501, 12583, 60117, 88691}},
	{7752, 17, 17277, {1, 1, 5, 15, 17, 49, 117, 233, 161, 891, 789, 1347, 4887, 10713, 10613, 4389, 42619}},
	{7753, 17, 17308, {1, 3, 5, 9, 13, 3, 83, 69, 381, 777, 743, 2843, 7233, 3285, 8931, 48667, 120777}},
	{7754, 17, 17311, {1, 3, 1, 3, 11, 7, 55, 107, 165, 533, 1897, 3385, 1069, 12805, 30125, 42729, 123977}},
	{7755, 17, 17321, {1, 1, 1, 5, 13, 17, 103, 237, 77, 537, 1843, 2817, 7467, 13647, 15259, 3525, 18313}},
	{7756, 17, 17329, {1, 1, 7, 7, 13, 59, 29, 197, 309, 917, 1173, 2605, 4313, 12007, 25611, 60409, 104931}},
	{7757, 17, 17342, {1, 3, 3, 3, 27, 57, 7, 207, 491, 467, 1973, 3075, 8043, 3977, 14517, 13179, 47111}},
	{7758, 17, 17344, {1, 1, 7, 5, 31, 33, 125, 235, 79, 847, 1893, 3875, 7513, 1435, 24959, 46813, 82053}},
	{7759, 17, 17350, {1, 3, 7, 5, 3, 53, 103, 1, 215, 71, 787, 223, 1399, 6793, 11281, 39201, 122119}},
	{7760, 17, 17356, {1, 3, 3, 3, 3, 57, 7, 151, 319, 463, 685, 2917, 4037, 14929, 11971, 41827, 57449}},
	{7761, 17, 17371, {1, 1, 7, 3, 5, 11, 15, 139, 379, 563, 135, 65, 5633, 7535, 1451, 18289, 62457}},
	{7762, 17, 17374, {1, 1, 1, 15, 11, 23, 37, 57, 205, 107, 995, 151, 3279, 2015, 28927, 40731, 95551}},
	{7763, 17, 17392, {1, 3, 5, 9, 15, 43, 95, 217, 203, 215, 203, 2207, 8189, 465, 2175, 29285, 25375}},
	{7764, 17, 17402, {1, 3, 3, 5, 19, 59, 51, 123, 285, 721, 1335, 1777, 1645, 15811, 16539, 14637, 123323}},
	{7765, 17, 17410, {1, 3, 5, 5, 11, 35, 23, 23, 259, 789, 567, 1921, 4743, 6635, 6965, 43025, 43175}},
	{7766, 17, 17421, {1, 3, 7, 3, 7, 27, 77, 121, 285, 65, 647, 591, 2553, 7163, 12057, 43675, 24227}},
	{7767, 17, 17424, {1, 1, 5, 9, 3, 25, 17, 85, 235, 715, 1913, 2391, 3719, 11029, 18359, 6323, 4703}},
	{7768, 17, 17427, {1, 1, 1, 3, 25, 31, 37, 31, 89, 311, 1797, 3409, 6785, 9627, 29721, 58591, 111429}},
	{7769, 17, 17434, {1, 3, 1, 5, 9, 37, 47, 45, 419, 115, 1009, 1359, 65, 1161, 15673, 16075, 63895}},
	{7770, 17, 17449, {1, 1, 3, 5, 25, 47, 27, 87, 441, 547, 1801, 3589, 3773, 12215, 14509, 12669, 99983}},
	{7771, 17, 17452, {1, 1, 1, 3, 19, 33, 51, 91, 447, 577, 491, 539, 3177, 7033, 21633, 51737, 47089}},
	{7772, 17, 17463, {1, 1, 3, 15, 23, 53, 93, 113, 143, 973, 999, 2355, 1489, 3451, 29821, 23769, 74633}},
	{7773, 17, 17470, {1, 3, 7, 11, 27, 1, 35, 109, 111, 51, 425, 3203, 2585, 11255, 20939, 281, 92133}},
	{7774, 17, 17477, {1, 1, 1, 11, 13, 37, 13, 149, 421, 655, 79, 3093, 6429, 1145, 27663, 52861, 81431}},
	{7775, 17, 17482, {1, 1, 5, 13, 19, 23, 105, 39, 97, 239, 469, 1047, 4727, 12009, 8055, 45557, 124219}},
	{7776, 17, 17490, {1, 1, 1, 7, 7, 7, 5, 53, 269, 391, 1893, 2263, 2109, 11531, 12109, 31437, 20445}},
	{7777, 17, 17496, {1, 1, 3, 11, 9, 35, 69, 209, 93, 455, 1117, 3297, 2597, 15035, 17943, 19955, 829}},
	{7778, 17, 17508, {1, 1, 5, 7, 23, 23, 101, 71, 339, 553, 1653, 2997, 1191, 3121, 4575, 49979, 17353}},
	{7779, 17, 17511, {1, 3, 3, 13, 13, 9, 51, 181, 33, 185, 111, 589, 3117, 10105, 28811, 31529, 79657}},
	{7780, 17, 17536, {1, 1, 5, 3, 9, 57, 103, 65, 211, 473, 519, 3815, 4087, 2767, 10213, 37829, 9523}},
	{7781, 17, 17563, {1, 1, 5, 7, 7, 31, 81, 161, 311, 617, 1689, 3133, 57, 14595, 22783, 18475, 85811}},
	{7782, 17, 17570, {1, 3, 5, 5, 21, 51, 99, 249, 7, 525, 885, 3981, 2851, 529, 947, 29885, 122901}},
	{7783, 17, 17581, {1, 3, 1, 5, 1, 23, 85, 91, 309, 747, 183, 1347, 2399, 15777, 16205, 15687, 117333}},
	{7784, 17, 17590, {1, 3, 7, 3, 29, 21, 99, 137, 297, 229, 1063, 2747, 6415, 7791, 4775, 62863, 50849}},
	{7785, 17, 17608, {1, 3, 1, 3, 11, 3, 53, 153, 103, 911, 1595, 1899, 1049, 11643, 21105, 61587, 92399}},
	{7786, 17, 17616, {1, 1, 5, 15, 29, 55, 99, 101, 181, 453, 1917, 2081, 4687, 4335, 2817, 11861, 103167}},
	{7787, 17, 17621, {1, 3, 7, 15, 11, 7, 9, 3, 477, 281, 1141, 453, 4993, 7843, 6169, 49785, 53827}},
	{7788, 17, 17628, {1, 3, 7, 11, 25, 61, 77, 159, 83, 95, 1223, 85, 6309, 16145, 18729, 133, 14193}},
	{7789, 17, 17644, {1, 3, 3, 1, 7, 27, 97, 183, 263, 59, 915, 3857, 3349, 8123, 11251, 55163, 125703}},
	{7790, 17, 17649, {1, 3, 5, 5, 17, 33, 57, 55, 503, 811, 953, 349, 1949, 9127, 31015, 14475, 116769}},
	{7791, 17, 17652, {1, 3, 1, 1, 3, 53, 59, 131, 421, 971, 89, 3047, 3513, 13599, 4569, 54525, 54779}},
	{7792, 17, 17679, {1, 1, 3, 11, 9, 45, 95, 123, 197, 257, 1073, 1461, 5, 12701, 12559, 34989, 71631}},
	{7793, 17, 17691, {1, 3, 3, 7, 1, 27, 55, 191, 447, 561, 1975, 2665, 1341, 8969, 18519, 47389, 70847}},
	{7794, 17, 17693, {1, 1, 5, 5, 3, 31, 15, 165, 95, 423, 233, 2309, 7777, 3503, 20105, 3085, 92349}},
	{7795, 17, 17697, {1, 3, 1, 13, 23, 61, 7, 55, 157, 1, 83, 515, 2169, 14397, 18149, 56855, 117265}},
	{7796, 17, 17698, {1, 3, 3, 3, 3, 59, 69, 95, 127, 241, 65, 3145, 491, 13809, 17529, 20223, 96579}},
	{7797, 17, 17700, {1, 1, 1, 5, 13, 43, 117, 163, 305, 955, 2007, 3337, 807, 16019, 5721, 5479, 90937}},
	{7798, 17, 17704, {1, 3, 3, 1, 19, 9, 127, 5, 113, 491, 1873, 2127, 7949, 5207, 32531, 775, 131065}},
	{7799, 17, 17707, {1, 1, 7, 3, 1, 3, 91, 187, 37, 873, 1039, 4075, 5645, 243, 15127, 45615, 3813}},
	{7800, 17, 17715, {1, 1, 3, 11, 3, 37, 67, 59, 439, 763, 213, 1099, 1659, 12783, 30297, 60713, 43497}},
	{7801, 17, 17718, {1, 3, 3, 13, 23, 49, 47, 191, 245, 985, 487, 3165, 7803, 2437, 19073, 30605, 119641}},
	{7802, 17, 17721, {1, 3, 7, 7, 19, 43, 7, 253, 93, 99, 145, 219, 699, 2433, 3009, 565, 99671}},
	{7803, 17, 17722, {1, 1, 3, 13, 7, 5, 9, 23, 219, 155, 925, 3427, 6631, 16353, 4115, 20831, 49525}},
	{7804, 17, 17727, {1, 1, 7, 11, 15, 45, 41, 35, 301, 273, 241, 4031, 5441, 8281, 9341, 8499, 73841}},
	{7805, 17, 17729, {1, 3, 7, 13, 9, 19, 79, 3, 163, 197, 509, 2301, 6971, 11525, 8805, 33805, 111595}},
	{7806, 17, 17747, {1, 3, 3, 1, 15, 45, 69, 253, 155, 639, 1045, 749, 3619, 14871, 18063, 49763, 66687}},
	{7807, 17, 17754, {1, 3, 3, 3, 29, 5, 41, 171, 265, 677, 1719, 2623, 1721, 12243, 18741, 39595, 92873}},
	{7808, 17, 17756, {1, 3, 5, 7, 27, 23, 69, 61, 453, 399, 1857, 3901, 6565, 15595, 1083, 15065, 91249}},
	{7809, 17, 17760, {1, 1, 5, 7, 1, 27, 9, 145, 95, 983, 685, 2079, 5117, 5037, 22695, 53135, 43569}},
	{7810, 17, 17765, {1, 1, 3, 5, 5, 7, 69, 59, 331, 409, 179, 333, 1293, 3863, 9473, 12537, 55605}},
	{7811, 17, 17778, {1, 3, 5, 1, 1, 19, 1, 49, 317, 769, 91, 2073, 1765, 1197, 15029, 52553, 57361}},
	{7812, 17, 17784, {1, 1, 5, 1, 23, 13, 11, 69, 345, 877, 41, 817, 617, 14415, 8337, 53973, 50007}},
	{7813, 17, 17794, {1, 1, 7, 3, 19, 27, 69, 103, 115, 893, 219, 2891, 2813, 9933, 26401, 63323, 30909}},
	{7814, 17, 17805, {1, 1, 5, 5, 27, 15, 119, 3, 11, 783, 541, 2431, 2395, 3921, 15471, 34657, 100295}},
	{7815, 17, 17806, {1, 1, 7, 11, 15, 25, 39, 191, 345, 1001, 1773, 715, 1627, 15957, 30085, 34097, 58747}},
	{7816, 17, 17817, {1, 1, 1, 5, 17, 43, 65, 81, 487, 387, 1359, 145, 2231, 6693, 15857, 59539, 79615}},
	{7817, 17, 17824, {1, 1, 3, 5, 15, 19, 17, 233, 247, 611, 587, 2587, 2321, 2835, 1477, 41991, 88143}},
	{7818, 17, 17839, {1, 3, 3, 15, 27, 15, 53, 61, 359, 989, 283, 3569, 5551, 11849, 20995, 34065, 69407}},
	{7819, 17, 17842, {1, 3, 3, 11, 13, 31, 41, 87, 379, 47, 1289, 3143, 4213, 8643, 17065, 10707, 62773}},
	{7820, 17, 17844, {1, 3, 7, 1, 9, 61, 59, 121, 453, 663, 27, 793, 4501, 7713, 285, 13279, 11633}},
	{7821, 17, 17851, {1, 1, 7, 5, 29, 51, 57, 15, 233, 743, 879, 2317, 3399, 15741, 605, 57529, 87427}},
	{7822, 17, 17862, {1, 1, 1, 1, 19, 59, 51, 119, 273, 403, 1649, 3877, 3561, 10931, 21139, 2599, 77957}},
	{7823, 17, 17868, {1, 3, 1, 3, 5, 1, 79, 131, 251, 585, 359, 2073, 7041, 13611, 22937, 24645, 72827}},
	{7824, 17, 17871, {1, 3, 7, 9, 19, 39, 93, 137, 359, 565, 1123, 1301, 4111, 13683, 1361, 25147, 38315}},
	{7825, 17, 17873, {1, 1, 5, 13, 27, 31, 11, 243, 111, 243, 1615, 1649, 2999, 15873, 19161, 57485, 35819}},
	{7826, 17, 17896, {1, 3, 3, 5, 25, 57, 61, 207, 219, 969, 303, 1165, 6753, 13473, 10789, 52883, 45205}},
	{7827, 17, 17904, {1, 1, 7, 11, 9, 53, 55, 175, 399, 981, 255, 2499, 373, 13131, 26803, 48017, 25599}},
	{7828, 17, 17909, {1, 1, 3, 3, 11, 23, 73, 25, 83, 39, 1813, 747, 3287, 795, 11917, 55509, 105057}},
	{7829, 17, 17920, {1, 3, 7, 15, 29, 59, 47, 171, 219, 875, 71, 123, 8131, 15595, 12471, 62439, 131}},
	{7830, 17, 17923, {1, 3, 5, 13, 9, 27, 119, 233, 323, 943, 375, 3647, 185, 1639, 431, 27225, 130175}},
	{7831, 17, 17947, {1, 3, 7, 3, 7, 17, 31, 155, 89, 835, 1015, 2019, 3973, 7087, 16899, 29591, 40797}},
	{7832, 17, 17950, {1, 3, 3, 1, 3, 11, 83, 231, 209, 537, 1227, 1519, 1059, 14027, 18591, 34031, 125755}},
	{7833, 17, 17956, {1, 3, 3, 1, 7, 39, 19, 99, 169, 961, 385, 1621, 7373, 7459, 8979, 23643, 17101}},
	{7834, 17, 17959, {1, 1, 3, 11, 11, 23, 61, 37, 359, 981, 209, 2555, 2673, 6501, 12731, 10735, 97321}},
	{7835, 17, 17966, {1, 1, 3, 13, 15, 61, 115, 119, 99, 755, 1933, 345, 3133, 12071, 26657, 7133, 18553}},
	{7836, 17, 17971, {1, 3, 1, 5, 17, 7, 29, 119, 445, 911, 89, 19, 6137, 8037, 19527, 22467, 29253}},
	{7837, 17, 17973, {1, 1, 3, 11, 31, 21, 119, 21, 249, 371, 343, 4037, 7539, 14473, 23829, 46415, 60911}},
	{7838, 17, 17992, {1, 1, 7, 9, 21, 53, 29, 149, 467, 893, 479, 359, 1007, 13955, 9667, 10245, 74761}},
	{7839, 17, 18006, {1, 3, 1, 7, 7, 45, 7, 77, 289, 271, 1329, 261, 5675, 8275, 7443, 57945, 117825}},
	{7840, 17, 18009, {1, 1, 1, 3, 21, 57, 117, 77, 287, 119, 1073, 915, 2521, 455, 7433, 56953, 84433}},
	{7841, 17, 18010, {1, 1, 1, 9, 27, 47, 1, 189, 303, 375, 215, 3117, 4541, 12877, 15523, 32317, 104213}},
	{7842, 17, 18022, {1, 1, 3, 1, 13, 39, 37, 249, 371, 159, 1073, 1351, 4703, 2715, 17463, 3945, 51523}},
	{7843, 17, 18039, {1, 3, 5, 5, 29, 15, 79, 25, 61, 995, 1081, 3377, 345, 13773, 4205, 20589, 83591}},
	{7844, 17, 18046, {1, 1, 3, 1, 9, 1, 41, 39, 389, 509, 561, 3273, 1911, 15271, 22655, 34713, 2045}},
	{7845, 17, 18069, {1, 3, 1, 15, 17, 1, 55, 55, 119, 707, 843, 2657, 3687, 11557, 18331, 4935, 110639}},
	{7846, 17, 18074, {1, 3, 5, 1, 23, 35, 119, 215, 471, 643, 1581, 1965, 2627, 2991, 3361, 6737, 111657}},
	{7847, 17, 18076, {1, 3, 5, 7, 9, 19, 33, 197, 33, 993, 1795, 595, 7113, 14721, 12647, 41035, 13669}},
	{7848, 17, 18085, {1, 1, 7, 15, 31, 39, 51, 157, 373, 473, 665, 3541, 6695, 11741, 5617, 17189, 129851}},
	{7849, 17, 18086, {1, 3, 3, 7, 15, 37, 33, 175, 391, 159, 717, 593, 113, 9331, 10685, 59003, 26975}},
	{7850, 17, 18095, {1, 1, 3, 5, 13, 41, 11, 109, 9, 899, 1503, 901, 6237, 7789, 9963, 14923, 63665}},
	{7851, 17, 18100, {1, 3, 7, 7, 25, 61, 3, 231, 235, 29, 1049, 1997, 5371, 9047, 29595, 49239, 108649}},
	{7852, 17, 18109, {1, 1, 3, 5, 27, 1, 53, 209, 315, 747, 1803, 11, 1815, 6539, 8839, 18385, 88681}},
	{7853, 17, 18121, {1, 1, 5, 13, 25, 55, 117, 197, 13, 689, 751, 1203, 2277, 1763, 23453, 54459, 118023}},
	{7854, 17, 18127, {1, 3, 3, 11, 21, 1, 51, 101, 429, 723, 273, 3021, 1491, 9923, 6629, 63741, 98813}},
	{7855, 17, 18129, {1, 1, 1, 15, 17, 25, 111, 251, 43, 403, 465, 17, 787, 6045, 32185, 22921, 115851}},
	{7856, 17, 18132, {1, 1, 5, 11, 11, 13, 13, 93, 489, 941, 1391, 383, 7735, 1921, 16199, 53099, 25181}},
	{7857, 17, 18141, {1, 3, 3, 7, 15, 1, 3, 159, 507, 867, 1589, 2111, 3839, 8989, 12589, 37657, 97055}},
	{7858, 17, 18146, {1, 3, 3, 13, 25, 23, 7, 95, 489, 1001, 105, 2737, 5013, 14465, 25383, 57551, 77425}},
	{7859, 17, 18151, {1, 3, 5, 3, 3, 7, 81, 15, 255, 297, 1183, 655, 741, 3081, 2141, 34493, 103707}},
	{7860, 17, 18157, {1, 1, 7, 9, 27, 57, 49, 121, 21, 239, 829, 2001, 613, 9569, 4419, 20007, 59109}},
	{7861, 17, 18160, {1, 3, 7, 1, 3, 21, 109, 255, 45, 333, 915, 1245, 5893, 765, 28289, 53927, 15335}},
	{7862, 17, 18183, {1, 3, 3, 7, 5, 35, 33, 79, 111, 509, 347, 3915, 2017, 6461, 11847, 17807, 48601}},
	{7863, 17, 18204, {1, 3, 5, 1, 13, 63, 87, 65, 507, 277, 339, 3637, 6289, 719, 9383, 38887, 55061}},
	{7864, 17, 18218, {1, 1, 5, 15, 17, 5, 59, 107, 355, 1021, 1849, 1807, 7679, 305, 31533, 1221, 98165}},
	{7865, 17, 18226, {1, 1, 1, 13, 19, 7, 37, 63, 267, 399, 1451, 2149, 1003, 13635, 18693, 215, 15887}},
	{7866, 17, 18238, {1, 1, 3, 7, 11, 63, 81, 251, 253, 963, 635, 1697, 6393, 9775, 24189, 9099, 106277}},
	{7867, 17, 18245, {1, 3, 3, 13, 17, 47, 63, 47, 279, 879, 271, 1655, 1897, 10743, 2607, 16695, 32779}},
	{7868, 17, 18249, {1, 3, 5, 15, 3, 1, 121, 181, 303, 973, 19, 3327, 3827, 2197, 31857, 22835, 122611}},
	{7869, 17, 18260, {1, 1, 5, 13, 25, 41, 105, 197, 195, 85, 1515, 2735, 7539, 7557, 24297, 38721, 46895}},
	{7870, 17, 18267, {1, 1, 1, 1, 15, 63, 33, 7, 43, 971, 781, 1461, 4483, 2113, 32459, 37653, 68017}},
	{7871, 17, 18270, {1, 3, 3, 9, 7, 1, 65, 183, 171, 695, 191, 3675, 6749, 6823, 3577, 45031, 81597}},
	{7872, 17, 18273, {1, 3, 3, 3, 9, 61, 13, 159, 295, 329, 943, 3293, 79, 14497, 21461, 4667, 96435}},
	{7873, 17, 18274, {1, 1, 7, 9, 29, 37, 117, 215, 295, 591, 1139, 3093, 7469, 7995, 13581, 48075, 5943}},
	{7874, 17, 18276, {1, 3, 1, 9, 11, 11, 117, 255, 419, 551, 1445, 1987, 1169, 14227, 31095, 36041, 63739}},
	{7875, 17, 18283, {1, 1, 7, 15, 17, 25, 81, 27, 87, 463, 1871, 551, 7449, 12231, 28645, 32663, 43037}},
	{7876, 17, 18307, {1, 3, 5, 11, 3, 49, 109, 123, 397, 113, 1269, 2433, 4463, 1257, 2127, 6677, 96009}},
	{7877, 17, 18314, {1, 1, 1, 11, 27, 19, 83, 123, 297, 867, 941, 3929, 3483, 4641, 32505, 48999, 66169}},
	{7878, 17, 18321, {1, 1, 5, 11, 5, 21, 11, 255, 369, 859, 657, 587, 5245, 12973, 22403, 47935, 121375}},
	{7879, 17, 18334, {1, 3, 1, 13, 17, 43, 83, 51, 339, 967, 499, 1485, 5203, 10053, 31707, 31443, 75033}},
	{7880, 17, 18355, {1, 1, 5, 13, 11, 5, 121, 121, 73, 101, 1751, 3805, 1333, 14043, 26957, 27557, 110899}},
	{7881, 17, 18364, {1, 3, 7, 11, 9, 7, 125, 237, 437, 177, 841, 175, 5509, 9157, 3129, 54119, 109315}},
	{7882, 17, 18372, {1, 3, 7, 15, 1, 59, 87, 121, 43, 475, 1589, 1267, 1501, 1585, 31705, 33959, 27247}},
	{7883, 17, 18390, {1, 1, 5, 3, 27, 63, 117, 205, 169, 701, 1081, 2835, 8049, 11875, 4143, 17663, 90043}},
	{7884, 17, 18399, {1, 3, 1, 9, 23, 27, 31, 141, 411, 145, 1177, 3681, 3403, 6943, 10729, 47219, 102713}},
	{7885, 17, 18415, {1, 1, 7, 11, 5, 49, 33, 27, 121, 865, 471, 1871, 6945, 10051, 4493, 7121, 65551}},
	{7886, 17, 18420, {1, 1, 5, 1, 17, 27, 53, 13, 31, 403, 1319, 1381, 1371, 11693, 18805, 54683, 30137}},
	{7887, 17, 18434, {1, 1, 7, 11, 9, 21, 71, 155, 79, 145, 943, 3891, 641, 3163, 28493, 14729, 83391}},
	{7888, 17, 18443, {1, 3, 3, 13, 3, 53, 21, 189, 245, 803, 1625, 4005, 1163, 16033, 5549, 14301, 115859}},
	{7889, 17, 18446, {1, 3, 1, 5, 17, 59, 61, 31, 293, 677, 1753, 1803, 1671, 14619, 22361, 61453, 78203}},
	{7890, 17, 18460, {1, 3, 3, 1, 5, 51, 99, 231, 175, 97, 1335, 689, 1913, 6157, 22757, 52395, 118347}},
	{7891, 17, 18467, {1, 3, 3, 7, 25, 11, 113, 19, 289, 507, 1143, 3437, 7965, 12551, 27603, 8423, 46713}},
	{7892, 17, 18482, {1, 1, 3, 9, 13, 1, 73, 9, 425, 407, 363, 2915, 4269, 2903, 9251, 17733, 80321}},
	{7893, 17, 18484, {1, 1, 3, 11, 31, 47, 37, 211, 433, 237, 1069, 1891, 6175, 9305, 30385, 2497, 94775}},
	{7894, 17, 18501, {1, 1, 3, 1, 23, 5, 113, 103, 427, 587, 1863, 1863, 679, 2575, 13059, 16163, 42289}},
	{7895, 17, 18506, {1, 1, 5, 3, 7, 17, 45, 33, 209, 609, 1897, 95, 5123, 2239, 5483, 60715, 126497}},
	{7896, 17, 18516, {1, 1, 5, 11, 1, 55, 67, 223, 41, 967, 337, 2511, 7879, 1157, 17635, 64081, 421}},
	{7897, 17, 18519, {1, 3, 3, 9, 27, 33, 87, 97, 231, 895, 1337, 829, 47, 8481, 14059, 57209, 109159}},
	{7898, 17, 18547, {1, 3, 7, 1, 25, 5, 41, 161, 393, 523, 1623, 3761, 1933, 8319, 17309, 46717, 97299}},
	{7899, 17, 18569, {1, 1, 1, 11, 5, 55, 19, 191, 41, 791, 1611, 59, 2633, 13873, 25859, 42879, 54807}},
	{7900, 17, 18575, {1, 3, 1, 11, 11, 33, 5, 13, 199, 411, 895, 759, 2735, 16225, 31469, 24081, 31857}},
	{7901, 17, 18589, {1, 1, 7, 13, 27, 57, 21, 191, 389, 977, 1013, 3493, 6401, 15957, 23181, 52461, 41853}},
	{7902, 17, 18590, {1, 3, 7, 5, 23, 19, 117, 117, 427, 923, 1347, 3099, 247, 8879, 5309, 53277, 100625}},
	{7903, 17, 18605, {1, 1, 5, 13, 11, 23, 69, 37, 119, 329, 1935, 2851, 5127, 6907, 24651, 11135, 118287}},
	{7904, 17, 18611, {1, 1, 3, 15, 23, 1, 69, 107, 253, 771, 1697, 4035, 3219, 15011, 6995, 19255, 39909}},
	{7905, 17, 18625, {1, 3, 1, 1, 5, 21, 35, 173, 407, 603, 27, 3589, 2879, 2755, 17679, 6145, 95989}},
	{7906, 17, 18652, {1, 1, 5, 13, 31, 23, 61, 139, 341, 593, 1673, 4031, 4809, 1107, 22657, 29073, 53401}},
	{7907, 17, 18665, {1, 3, 1, 15, 13, 37, 39, 61, 443, 417, 1125, 1529, 1943, 7317, 2985, 50285, 107069}},
	{7908, 17, 18673, {1, 1, 3, 9, 5, 51, 87, 91, 31, 491, 1455, 1685, 2579, 6023, 10989, 64635, 130147}},
	{7909, 17, 18674, {1, 3, 5, 5, 31, 23, 15, 163, 357, 161, 1597, 1571, 5039, 13213, 32221, 4405, 88079}},
	{7910, 17, 18683, {1, 1, 1, 15, 13, 43, 7, 109, 243, 389, 683, 2671, 8003, 4187, 6507, 11171, 116727}},
	{7911, 17, 18688, {1, 3, 7, 1, 17, 31, 53, 5, 227, 755, 1755, 2939, 1789, 8951, 16777, 30203, 79005}},
	{7912, 17, 18691, {1, 3, 3, 9, 27, 5, 111, 241, 89, 333, 371, 1035, 5719, 2433, 29343, 50829, 25131}},
	{7913, 17, 18698, {1, 1, 3, 13, 7, 37, 125, 69, 79, 397, 1595, 123, 255, 2257, 10881, 27085, 99411}},
	{7914, 17, 18717, {1, 1, 3, 15, 1, 35, 61, 73, 507, 775, 1631, 2005, 4277, 8421, 5669, 39221, 19053}},
	{7915, 17, 18733, {1, 1, 3, 7, 15, 17, 65, 157, 19, 997, 861, 1249, 4059, 7975, 955, 5833, 97733}},
	{7916, 17, 18754, {1, 1, 5, 5, 21, 43, 1, 181, 1, 17, 1337, 3333, 3761, 12283, 20941, 231, 30457}},
	{7917, 17, 18759, {1, 3, 3, 7, 7, 23, 105, 101, 101, 757, 1407, 565, 2187, 1529, 29385, 22847, 57675}},
	{7918, 17, 18760, {1, 3, 3, 1, 11, 3, 65, 93, 201, 773, 1037, 1325, 673, 6625, 2909, 63435, 114311}},
	{7919, 17, 18771, {1, 3, 5, 1, 21, 43, 87, 37, 491, 323, 1093, 2493, 4755, 7225, 12037, 9483, 70351}},
	{7920, 17, 18777, {1, 1, 7, 9, 23, 39, 59, 117, 103, 645, 1975, 1177, 55, 325, 23781, 64365, 94915}},
	{7921, 17, 18796, {1, 3, 7, 15, 21, 29, 109, 35, 307, 847, 777, 3457, 7899, 17, 24065, 10517, 35651}},
	{7922, 17, 18799, {1, 1, 7, 9, 25, 35, 49, 131, 377, 429, 1773, 2107, 6305, 15209, 9567, 17685, 5599}},
	{7923, 17, 18807, {1, 1, 3, 11, 13, 27, 47, 125, 483, 229, 921, 2733, 2217, 2615, 24135, 28759, 109411}},
	{7924, 17, 18813, {1, 3, 1, 7, 19, 45, 23, 195, 445, 955, 853, 2877, 6889, 9507, 2009, 18747, 50545}},
	{7925, 17, 18817, {1, 1, 1, 5, 15, 35, 75, 177, 145, 683, 309, 893, 4999, 827, 26563, 30819, 111115}},
	{7926, 17, 18820, {1, 3, 3, 11, 5, 45, 49, 39, 93, 653, 1053, 2553, 863, 12185, 30261, 16459, 121061}},
	{7927, 17, 18827, {1, 3, 7, 5, 11, 29, 57, 43, 409, 71, 67, 3537, 263, 13237, 8825, 58411, 44629}},
	{7928, 17, 18829, {1, 1, 5, 13, 1, 37, 23, 171, 13, 309, 239, 2023, 6233, 8751, 11371, 5825, 77355}},
	{7929, 17, 18838, {1, 3, 1, 13, 5, 1, 47, 217, 369, 609, 453, 879, 4337, 4441, 8785, 51963, 53819}},
	{7930, 17, 18842, {1, 3, 5, 5, 23, 1, 67, 147, 27, 121, 1369, 569, 1519, 11585, 18193, 30889, 78055}},
	{7931, 17, 18848, {1, 1, 1, 13, 11, 53, 33, 37, 419, 111, 1649, 2495, 6105, 12385, 30865, 3683, 63813}},
	{7932, 17, 18853, {1, 3, 3, 5, 17, 35, 107, 235, 471, 735, 1093, 1007, 567, 173, 9623, 39533, 56455}},
	{7933, 17, 18885, {1, 1, 7, 15, 27, 13, 123, 211, 407, 857, 801, 3951, 8153, 4427, 15333, 13217, 92675}},
	{7934, 17, 18890, {1, 1, 1, 7, 11, 61, 99, 131, 121, 119, 1483, 1485, 3521, 13163, 24899, 56849, 111943}},
	{7935, 17, 18898, {1, 3, 3, 1, 19, 1, 29, 139, 127, 557, 1913, 1487, 1381, 185, 11195, 52499, 45059}},
	{7936, 17, 18903, {1, 3, 7, 11, 5, 29, 95, 111, 235, 55, 1101, 2631, 1219, 9867, 22209, 3095, 56793}},
	{7937, 17, 18910, {1, 3, 7, 1, 31, 61, 37, 125, 241, 985, 1079, 1439, 433, 2779, 8463, 59985, 39667}},
	{7938, 17, 18913, {1, 3, 7, 15, 5, 7, 71, 7, 435, 727, 1611, 135, 1421, 8329, 29995, 64243, 58285}},
	{7939, 17, 18931, {1, 3, 3, 15, 27, 11, 121, 27, 281, 499, 267, 2651, 7575, 9499, 5051, 49475, 79573}},
	{7940, 17, 18934, {1, 3, 3, 15, 29, 47, 11, 183, 235, 537, 79, 2467, 3751, 831, 6725, 52173, 108645}},
	{7941, 17, 18956, {1, 3, 5, 13, 23, 31, 23, 19, 477, 511, 727, 183, 5955, 7613, 31979, 8441, 39835}},
	{7942, 17, 18961, {1, 1, 5, 7, 17, 31, 53, 133, 387, 787, 1573, 89, 1975, 1825, 19963, 27203, 124923}},
	{7943, 17, 18968, {1, 1, 1, 9, 3, 15, 125, 135, 89, 37, 517, 3931, 2013, 2143, 25413, 18421, 6097}},
	{7944, 17, 18978, {1, 1, 3, 11, 23, 29, 89, 45, 53, 135, 223, 3523, 7921, 3271, 1819, 40931, 65471}},
	{7945, 17, 18983, {1, 1, 1, 13, 17, 3, 121, 25, 509, 61, 1009, 2009, 7813, 8499, 5807, 63171, 75991}},
	{7946, 17, 18987, {1, 3, 5, 13, 15, 35, 37, 45, 161, 683, 1665, 59, 6297, 9595, 10193, 46745, 105411}},
	{7947, 17, 18992, {1, 3, 1, 7, 21, 19, 85, 107, 3, 845, 673, 1271, 7581, 15971, 27085, 35375, 29027}},
	{7948, 17, 18997, {1, 3, 3, 9, 5, 17, 79, 137, 123, 809, 583, 3507, 7559, 2857, 13911, 57083, 8595}},
	{7949, 17, 19002, {1, 1, 7, 5, 31, 29, 77, 33, 439, 787, 1223, 2471, 5851, 15891, 27925, 51661, 82645}},
	{7950, 17, 19004, {1, 1, 7, 15, 19, 35, 35, 37, 197, 245, 799, 3971, 277, 11289, 20111, 13857, 104571}},
	{7951, 17, 19010, {1, 3, 5, 15, 19, 3, 65, 17, 201, 1007, 1665, 107, 2409, 1469, 23265, 24547, 8189}},
	{7952, 17, 19012, {1, 3, 1, 13, 29, 45, 109, 243, 43, 383, 631, 265, 6671, 15333, 21931, 30675, 103077}},
	{7953, 17, 19030, {1, 1, 5, 1, 25, 25, 77, 189, 109, 777, 1485, 2265, 1403, 2627, 13843, 27263, 14263}},
	{7954, 17, 19043, {1, 3, 5, 1, 13, 49, 73, 107, 225, 243, 1253, 3503, 735, 2605, 27165, 45467, 93001}},
	{7955, 17, 19049, {1, 1, 5, 9, 15, 17, 1, 33, 69, 321, 1375, 3635, 8131, 6579, 1225, 38699, 17447}},
	{7956, 17, 19079, {1, 3, 5, 3, 25, 49, 15, 149, 483, 37, 1929, 1219, 5841, 11975, 805, 31339, 130971}},
	{7957, 17, 19086, {1, 3, 3, 3, 15, 29, 3, 143, 291, 593, 1769, 3603, 1693, 151, 27701, 9015, 25847}},
	{7958, 17, 19100, {1, 3, 1, 11, 5, 35, 55, 127, 137, 71, 967, 2501, 1023, 2061, 31387, 44011, 130121}},
	{7959, 17, 19103, {1, 1, 7, 1, 29, 13, 93, 41, 125, 263, 521, 2633, 4361, 12153, 30647, 55883, 65185}},
	{7960, 17, 19107, {1, 3, 7, 9, 23, 19, 61, 197, 139, 463, 1867, 3627, 5185, 8251, 26977, 48027, 66301}},
	{7961, 17, 19109, {1, 3, 3, 7, 27, 53, 25, 137, 175, 155, 1843, 2253, 4797, 4989, 32613, 55779, 91625}},
	{7962, 17, 19113, {1, 3, 3, 11, 1, 5, 21, 233, 295, 675, 47, 2995, 8075, 8201, 3845, 23925, 82559}},
	{7963, 17, 19116, {1, 1, 3, 7, 31, 53, 93, 21, 307, 709, 9, 3061, 3935, 11009, 13411, 3657, 30251}},
	{7964, 17, 19127, {1, 3, 7, 13, 13, 25, 51, 205, 391, 897, 275, 333, 6831, 9383, 16101, 14301, 99101}},
	{7965, 17, 19134, {1, 1, 5, 15, 17, 47, 119, 7, 189, 765, 753, 2909, 3373, 2379, 20331, 61535, 51345}},
	{7966, 17, 19141, {1, 1, 3, 1, 27, 43, 9, 185, 9, 197, 1179, 67, 7689, 9679, 29683, 29905, 29393}},
	{7967, 17, 19179, {1, 1, 5, 5, 31, 55, 69, 9, 477, 91, 1705, 1877, 5463, 15401, 13449, 27541, 125521}},
	{7968, 17, 19193, {1, 1, 7, 15, 15, 33, 11, 233, 69, 771, 845, 2715, 5293, 10351, 19557, 15319, 36857}},
	{7969, 17, 19194, {1, 3, 7, 7, 15, 9, 123, 47, 165, 739, 361, 1675, 2743, 8021, 10241, 48275, 51935}},
	{7970, 17, 19201, {1, 1, 5, 1, 9, 25, 99, 83, 487, 627, 343, 3233, 6697, 13197, 19771, 38337, 89139}},
	{7971, 17, 19208, {1, 3, 7, 13, 1, 31, 15, 63, 463, 621, 935, 1093, 6043, 14051, 13665, 43413, 104893}},
	{7972, 17, 19214, {1, 1, 1, 3, 27, 1, 47, 151, 127, 357, 689, 3263, 141, 4459, 9847, 205, 88493}},
	{7973, 17, 19225, {1, 3, 7, 15, 29, 13, 41, 113, 495, 421, 195, 197, 6857, 10555, 22861, 30229, 31707}},
	{7974, 17, 19226, {1, 3, 5, 11, 11, 1, 89, 227, 425, 623, 1605, 1901, 7933, 7211, 16301, 3403, 59651}},
	{7975, 17, 19235, {1, 1, 3, 3, 27, 41, 37, 89, 395, 903, 1641, 2739, 5599, 11371, 8683, 61125, 105231}},
	{7976, 17, 19242, {1, 3, 7, 9, 1, 17, 51, 233, 507, 783, 459, 1187, 7281, 15809, 27637, 6067, 125877}},
	{7977, 17, 19264, {1, 3, 1, 3, 13, 57, 5, 199, 261, 357, 1255, 1849, 8013, 10313, 9375, 1271, 64117}},
	{7978, 17, 19267, {1, 3, 1, 11, 9, 59, 55, 95, 401, 423, 1657, 513, 3565, 12957, 19243, 53027, 11323}},
	{7979, 17, 19293, {1, 3, 7, 13, 27, 35, 121, 215, 397, 991, 191, 3443, 3829, 6107, 5381, 48497, 107997}},
	{7980, 17, 19309, {1, 1, 5, 5, 19, 15, 21, 53, 165, 835, 1599, 3245, 5609, 5991, 18141, 28075, 102809}},
	{7981, 17, 19318, {1, 3, 5, 9, 25, 21, 71, 15, 453, 475, 915, 3097, 5869, 699, 13883, 34919, 127211}},
	{7982, 17, 19324, {1, 1, 3, 7, 21, 53, 27, 207, 373, 703, 593, 17, 6991, 15013, 10125, 34801, 129245}},
	{7983, 17, 19337, {1, 3, 3, 13, 17, 9, 89, 95, 291, 681, 1415, 2323, 2885, 11381, 7703, 3691, 51505}},
	{7984, 17, 19340, {1, 1, 1, 11, 15, 63, 55, 153, 373, 665, 983, 3987, 5997, 6873, 27031, 65449, 22021}},
	{7985, 17, 19345, {1, 1, 5, 5, 1, 55, 119, 61, 159, 889, 225, 709, 1879, 2521, 69, 7815, 18733}},
	{7986, 17, 19346, {1, 3, 5, 11, 23, 53, 23, 61, 71, 993, 633, 1829, 3465, 12465, 30205, 40723, 74499}},
	{7987, 17, 19352, {1, 3, 3, 1, 17, 37, 19, 43, 55, 133, 1171, 3101, 3963, 5177, 24791, 7255, 10263}},
	{7988, 17, 19364, {1, 3, 7, 1, 21, 13, 21, 87, 237, 629, 1167, 3699, 597, 6251, 11545, 34429, 104393}},
	{7989, 17, 19382, {1, 1, 7, 1, 11, 53, 105, 111, 463, 869, 549, 2423, 17, 917, 879, 49367, 72235}},
	{7990, 17, 19391, {1, 1, 5, 15, 17, 51, 69, 55, 309, 867, 257, 573, 4821, 5245, 28033, 61801, 18253}},
	{7991, 17, 19405, {1, 1, 5, 3, 1, 23, 103, 241, 13, 267, 21, 1751, 6637, 12537, 26741, 40651, 94493}},
	{7992, 17, 19411, {1, 3, 3, 13, 25, 35, 21, 83, 337, 363, 1111, 1865, 7889, 985, 465, 40287, 64469}},
	{7993, 17, 19439, {1, 1, 7, 5, 27, 1, 99, 95, 209, 211, 1445, 1577, 6097, 13813, 22463, 64395, 106383}},
	{7994, 17, 19442, {1, 3, 1, 15, 1, 45, 77, 247, 273, 1023, 1377, 1989, 5787, 15267, 24363, 42717, 125617}},
	{7995, 17, 19444, {1, 1, 1, 3, 9, 49, 79, 171, 427, 439, 1725, 3179, 6263, 12437, 31353, 22077, 94455}},
	{7996, 17, 19451, {1, 3, 5, 11, 11, 45, 57, 97, 409, 935, 967, 2509, 3431, 5707, 19473, 15853, 129059}},
	{7997, 17, 19465, {1, 1, 7, 5, 7, 21, 105, 29, 359, 145, 1691, 131, 6721, 10971, 16173, 38193, 37091}},
	{7998, 17, 19471, {1, 1, 1, 15, 15, 35, 5, 185, 455, 507, 681, 3355, 2091, 3437, 27231, 28527, 5383}},
	{7999, 17, 19474, {1, 3, 5, 3, 7, 29, 33, 127, 57, 495, 1069, 3635, 7461, 9861, 18757, 39039, 92421}},
	{8000, 17, 19476, {1, 3, 5, 5, 11, 31, 51, 59, 413, 417, 1577, 2837, 5229, 4501, 18645, 37613, 31325}},
	{8001, 17, 19479, {1, 1, 5, 13, 15, 61, 17, 247, 413, 817, 907, 2249, 3901, 11275, 7469, 33403, 30629}},
	{8002, 17, 19485, {1, 3, 5, 7, 31, 7, 109, 177, 277, 75, 449, 3029, 7135, 427, 26641, 43157, 47671}},
	{8003, 17, 19489, {1, 3, 7, 13, 29, 63, 21, 187, 471, 289, 835, 3885, 6111, 8721, 9841, 24017, 18673}},
	{8004, 17, 19496, {1, 1, 5, 13, 25, 37, 15, 35, 227, 623, 47, 189, 3443, 1911, 8579, 50911, 10895}},
	{8005, 17, 19507, {1, 3, 1, 1, 29, 53, 89, 101, 251, 203, 821, 2485, 633, 7943, 27563, 58735, 72057}},
	{8006, 17, 19513, {1, 3, 7, 9, 23, 61, 121, 199, 19, 165, 131, 1373, 637, 7307, 7109, 42475, 126669}},
	{8007, 17, 19514, {1, 3, 7, 13, 7, 5, 125, 173, 365, 65, 565, 751, 3343, 13421, 6177, 39095, 97375}},
	{8008, 17, 19521, {1, 1, 7, 3, 1, 59, 65, 39, 307, 793, 887, 3291, 3405, 4497, 19351, 1821, 67861}},
	{8009, 17, 19524, {1, 1, 1, 3, 19, 9, 101, 183, 163, 819, 769, 49, 5293, 3715, 4055, 32403, 90763}},
	{8010, 17, 19546, {1, 3, 3, 1, 27, 31, 21, 123, 457, 1021, 1791, 2217, 6171, 11445, 15605, 59945, 19663}},
	{8011, 17, 19552, {1, 1, 1, 9, 13, 53, 61, 201, 457, 111, 1217, 377, 5871, 4591, 16379, 42817, 129807}},
	{8012, 17, 19555, {1, 3, 1, 5, 23, 37, 25, 7, 125, 651, 349, 3727, 1487, 5103, 31407, 40215, 16065}},
	{8013, 17, 19557, {1, 1, 3, 11, 19, 29, 1, 193, 13, 287, 331, 985, 5391, 7307, 28075, 9939, 84999}},
	{8014, 17, 19575, {1, 1, 1, 11, 21, 37, 117, 241, 229, 957, 2019, 819, 459, 6185, 21533, 64725, 24709}},
	{8015, 17, 19579, {1, 3, 5, 13, 11, 25, 107, 245, 175, 519, 629, 537, 2227, 15123, 10619, 32611, 118697}},
	{8016, 17, 19591, {1, 3, 1, 11, 5, 53, 119, 253, 489, 181, 1365, 3465, 1323, 949, 3657, 2467, 38545}},
	{8017, 17, 19595, {1, 1, 3, 9, 27, 17, 109, 19, 297, 541, 89, 3021, 761, 5577, 907, 21405, 128029}},
	{8018, 17, 19605, {1, 1, 3, 9, 31, 9, 61, 149, 267, 707, 671, 2733, 1247, 14623, 6395, 42579, 30845}},
	{8019, 17, 19615, {1, 1, 7, 7, 25, 29, 63, 41, 309, 275, 2019, 1373, 1003, 13891, 16571, 16209, 30115}},
	{8020, 17, 19616, {1, 3, 7, 1, 5, 21, 53, 97, 475, 799, 1963, 1181, 4187, 8767, 24779, 10403, 98349}},
	{8021, 17, 19626, {1, 3, 3, 13, 19, 9, 125, 227, 347, 535, 353, 3087, 769, 9049, 20145, 27433, 23105}},
	{8022, 17, 19631, {1, 1, 1, 15, 7, 61, 51, 113, 403, 501, 1767, 2785, 7151, 14517, 17533, 24913, 7121}},
	{8023, 17, 19634, {1, 1, 1, 9, 7, 21, 27, 169, 425, 567, 31, 35, 7859, 929, 6545, 33983, 13227}},
	{8024, 17, 19640, {1, 1, 5, 15, 11, 15, 69, 33, 127, 1005, 1947, 989, 6333, 15587, 18523, 53547, 115613}},
	{8025, 17, 19645, {1, 3, 5, 3, 1, 55, 7, 99, 213, 737, 363, 3167, 3949, 3723, 15777, 23207, 22901}},
	{8026, 17, 19678, {1, 1, 1, 9, 9, 29, 121, 85, 467, 811, 1, 3543, 6259, 4477, 31371, 27633, 22995}},
	{8027, 17, 19681, {1, 3, 5, 3, 11, 21, 95, 19, 55, 71, 803, 3655, 3749, 5113, 13611, 38097, 20943}},
	{8028, 17, 19682, {1, 3, 3, 11, 19, 25, 127, 105, 447, 499, 485, 881, 2649, 10297, 22283, 18305, 128919}},
	{8029, 17, 19706, {1, 3, 7, 1, 11, 45, 21, 87, 481, 645, 815, 793, 5763, 3945, 14379, 19623, 103199}},
	{8030, 17, 19708, {1, 3, 5, 1, 3, 45, 39, 229, 359, 151, 1079, 3955, 2107, 9593, 6701, 2811, 55215}},
	{8031, 17, 19713, {1, 3, 7, 7, 27, 59, 69, 211, 373, 643, 977, 3545, 2647, 10473, 27919, 10719, 24823}},
	{8032, 17, 19714, {1, 1, 3, 11, 7, 27, 117, 21, 59, 679, 969, 3813, 2701, 7363, 17525, 31229, 100665}},
	{8033, 17, 19720, {1, 3, 3, 5, 29, 53, 113, 141, 197, 991, 81, 2701, 6831, 7949, 16569, 44185, 29631}},
	{8034, 17, 19725, {1, 1, 1, 3, 1, 31, 9, 101, 347, 585, 577, 2529, 7461, 11921, 29475, 34505, 74911}},
	{8035, 17, 19743, {1, 1, 1, 15, 25, 19, 95, 37, 93, 755, 1891, 2309, 623, 13503, 5811, 45863, 106501}},
	{8036, 17, 19753, {1, 1, 5, 3, 15, 23, 51, 225, 87, 679, 1225, 4015, 3971, 163, 3185, 12921, 51267}},
	{8037, 17, 19756, {1, 1, 5, 1, 1, 37, 105, 181, 379, 657, 571, 2775, 5905, 8391, 32069, 18713, 125833}},
	{8038, 17, 19759, {1, 1, 7, 11, 11, 19, 109, 125, 371, 321, 629, 2165, 2861, 7883, 15503, 37679, 33057}},
	{8039, 17, 19762, {1, 1, 3, 5, 7, 5, 21, 5, 169, 321, 1145, 2243, 6143, 2537, 4429, 56493, 39391}},
	{8040, 17, 19776, {1, 3, 5, 5, 31, 7, 15, 175, 441, 663, 921, 3113, 2261, 13033, 19135, 28657, 92225}},
	{8041, 17, 19786, {1, 3, 1, 11, 13, 9, 25, 57, 297, 3, 1561, 825, 2803, 11327, 2699, 28631, 57277}},
	{8042, 17, 19799, {1, 1, 3, 9, 15, 25, 81, 197, 345, 341, 1557, 1375, 2509, 11949, 30201, 6807, 95199}},
	{8043, 17, 19800, {1, 3, 5, 3, 15, 23, 91, 147, 277, 59, 495, 1423, 1775, 12065, 8401, 22639, 111199}},
	{8044, 17, 19803, {1, 1, 5, 1, 1, 59, 35, 255, 229, 293, 187, 2663, 3967, 6493, 20103, 36703, 108939}},
	{8045, 17, 19806, {1, 3, 7, 11, 15, 1, 23, 39, 27, 281, 11, 3119, 2791, 1691, 16521, 39715, 32145}},
	{8046, 17, 19815, {1, 3, 5, 5, 9, 53, 43, 49, 107, 1015, 431, 3017, 3317, 9655, 19193, 45621, 56861}},
	{8047, 17, 19816, {1, 3, 1, 15, 27, 63, 127, 31, 21, 245, 1503, 3339, 6375, 5411, 12369, 35973, 9473}},
	{8048, 17, 19857, {1, 1, 3, 13, 31, 61, 19, 99, 25, 593, 539, 1807, 673, 12339, 23567, 22005, 130341}},
	{8049, 17, 19860, {1, 1, 5, 3, 3, 3, 13, 183, 255, 41, 641, 581, 6509, 1891, 19195, 28277, 51725}},
	{8050, 17, 19874, {1, 1, 3, 5, 3, 59, 17, 227, 9, 345, 1303, 1535, 3089, 2653, 20647, 63159, 124677}},
	{8051, 17, 19883, {1, 3, 1, 11, 25, 19, 117, 29, 221, 461, 1285, 1427, 7183, 3051, 10839, 47491, 92613}},
	{8052, 17, 19885, {1, 1, 3, 5, 27, 19, 1, 235, 51, 215, 783, 2325, 1191, 4679, 14365, 35479, 65083}},
	{8053, 17, 19886, {1, 3, 3, 5, 27, 17, 79, 185, 259, 369, 1399, 1029, 2219, 10975, 30487, 15247, 39789}},
	{8054, 17, 19893, {1, 3, 3, 1, 13, 13, 59, 119, 249, 471, 1433, 1165, 5345, 4431, 375, 64999, 85577}},
	{8055, 17, 19932, {1, 1, 1, 3, 1, 19, 13, 243, 35, 675, 321, 3625, 7835, 6403, 651, 48283, 91819}},
	{8056, 17, 19960, {1, 3, 3, 1, 27, 13, 73, 159, 145, 59, 287, 2419, 8115, 7923, 18933, 36109, 123879}},
	{8057, 17, 19972, {1, 3, 1, 7, 21, 1, 83, 245, 477, 623, 391, 129, 6897, 3447, 11109, 17017, 68277}},
	{8058, 17, 19975, {1, 1, 3, 11, 13, 43, 119, 93, 99, 393, 1219, 995, 1881, 7929, 4337, 33579, 103211}},
	{8059, 17, 20005, {1, 1, 7, 7, 31, 5, 39, 25, 119, 819, 409, 2395, 151, 12763, 28265, 28909, 35685}},
	{8060, 17, 20009, {1, 1, 1, 1, 3, 13, 59, 205, 19, 843, 1691, 955, 1859, 1791, 22083, 37843, 63615}},
	{8061, 17, 20010, {1, 1, 1, 3, 11, 63, 41, 243, 103, 875, 1337, 3731, 6317, 12951, 31743, 56935, 55975}},
	{8062, 17, 20012, {1, 1, 3, 13, 19, 11, 51, 97, 469, 279, 1621, 3521, 853, 11849, 3331, 27907, 119081}},
	{8063, 17, 20023, {1, 1, 5, 1, 23, 55, 9, 141, 449, 41, 167, 2441, 6783, 2785, 3547, 35379, 74973}},
	{8064, 17, 20024, {1, 1, 5, 3, 15, 55, 13, 75, 107, 153, 1841, 3991, 3229, 6523, 18541, 21571, 31539}},
	{8065, 17, 20030, {1, 3, 1, 1, 27, 37, 35, 201, 401, 867, 1861, 1783, 6255, 14001, 29543, 25843, 39719}},
	{8066, 17, 20038, {1, 1, 7, 15, 3, 43, 113, 173, 297, 457, 1369, 4053, 5033, 5513, 27387, 14725, 79937}},
	{8067, 17, 20041, {1, 1, 5, 13, 5, 27, 109, 93, 187, 833, 1551, 2899, 1681, 6979, 1289, 3507, 66499}},
	{8068, 17, 20055, {1, 1, 3, 11, 11, 47, 121, 29, 129, 807, 2037, 1527, 6083, 14803, 10669, 46047, 70241}},
	{8069, 17, 20059, {1, 3, 1, 9, 29, 3, 19, 191, 461, 527, 1389, 3359, 81, 6773, 12185, 49207, 19091}},
	{8070, 17, 20061, {1, 3, 7, 7, 5, 47, 33, 27, 445, 1, 149, 343, 4827, 91, 29233, 37775, 89321}},
	{8071, 17, 20080, {1, 3, 5, 1, 13, 55, 107, 99, 259, 591, 983, 3863, 1231, 3457, 29645, 10709, 16543}},
	{8072, 17, 20086, {1, 3, 7, 9, 29, 5, 9, 165, 337, 187, 219, 97, 6511, 193, 23947, 36329, 35317}},
	{8073, 17, 20089, {1, 3, 7, 1, 31, 25, 5, 175, 409, 1021, 1873, 289, 7143, 15341, 31501, 25707, 106453}},
	{8074, 17, 20095, {1, 3, 7, 7, 27, 1, 15, 221, 341, 987, 1739, 1183, 8139, 11081, 29721, 42991, 72805}},
	{8075, 17, 20111, {1, 1, 1, 9, 11, 1, 13, 17, 501, 543, 1485, 987, 1861, 8527, 1621, 30461, 23057}},
	{8076, 17, 20116, {1, 3, 7, 7, 9, 41, 1, 253, 71, 1009, 427, 3347, 6987, 3303, 30535, 33345, 126459}},
	{8077, 17, 20130, {1, 1, 1, 7, 11, 11, 15, 11, 305, 559, 1805, 1111, 377, 1495, 13471, 34831, 123125}},
	{8078, 17, 20136, {1, 1, 5, 7, 9, 37, 27, 45, 61, 705, 263, 3181, 7077, 5227, 28121, 42865, 3809}},
	{8079, 17, 20147, {1, 1, 5, 1, 23, 25, 29, 199, 259, 959, 661, 43, 6157, 1547, 1497, 24077, 129939}},
	{8080, 17, 20153, {1, 3, 5, 3, 13, 49, 33, 19, 367, 891, 1777, 3119, 5673, 8383, 14487, 1763, 63495}},
	{8081, 17, 20156, {1, 1, 1, 13, 9, 57, 35, 181, 7, 225, 449, 3843, 6257, 4983, 31307, 16559, 27633}},
	{8082, 17, 20167, {1, 3, 1, 11, 7, 33, 55, 81, 41, 61, 1711, 3273, 7629, 11283, 9103, 24105, 107547}},
	{8083, 17, 20173, {1, 3, 5, 5, 13, 17, 13, 51, 235, 869, 1543, 1249, 7749, 14773, 21751, 53497, 108709}},
	{8084, 17, 20182, {1, 1, 3, 9, 3, 63, 89, 43, 23, 479, 115, 3917, 7943, 7323, 5659, 64507, 46941}},
	{8085, 17, 20185, {1, 1, 3, 1, 25, 63, 25, 169, 459, 633, 1785, 1059, 5113, 4799, 29281, 24561, 17017}},
	{8086, 17, 20202, {1, 1, 3, 3, 15, 3, 11, 173, 493, 5, 1575, 653, 7391, 7453, 8297, 28653, 6213}},
	{8087, 17, 20209, {1, 1, 7, 5, 29, 5, 1, 57, 75, 479, 787, 3417, 3349, 111, 17787, 41141, 97597}},
	{8088, 17, 20229, {1, 3, 7, 1, 11, 7, 107, 159, 435, 159, 1401, 803, 7065, 5923, 4005, 37271, 113791}},
	{8089, 17, 20233, {1, 1, 5, 1, 23, 55, 7, 59, 351, 801, 1279, 3231, 4561, 2857, 20563, 46115, 79489}},
	{8090, 17, 20236, {1, 3, 3, 15, 19, 13, 113, 33, 149, 175, 1127, 3815, 4357, 12645, 4449, 61355, 83023}},
	{8091, 17, 20241, {1, 1, 7, 15, 3, 17, 41, 57, 243, 319, 1631, 2751, 7853, 5977, 28367, 20023, 56049}},
	{8092, 17, 20242, {1, 3, 1, 7, 27, 59, 7, 13, 497, 241, 1827, 2861, 1331, 1759, 6037, 18967, 42849}},
	{8093, 17, 20248, {1, 3, 1, 1, 31, 43, 41, 183, 241, 219, 335, 2331, 755, 10589, 29431, 29007, 66667}},
	{8094, 17, 20278, {1, 1, 3, 1, 27, 37, 61, 117, 471, 39, 139, 3821, 2945, 7035, 23673, 20167, 56169}},
	{8095, 17, 20282, {1, 3, 1, 1, 9, 29, 123, 61, 171, 1015, 1029, 1695, 1039, 11883, 259, 10879, 97709}},
	{8096, 17, 20299, {1, 3, 3, 5, 29, 39, 65, 193, 285, 635, 999, 717, 5465, 1849, 4293, 19775, 79121}},
	{8097, 17, 20307, {1, 1, 7, 1, 3, 3, 103, 15, 451, 307, 1027, 263, 6585, 11651, 14457, 62695, 38407}},
	{8098, 17, 20313, {1, 3, 7, 11, 13, 13, 29, 83, 267, 561, 2041, 13, 3167, 3475, 14735, 34455, 117533}},
	{8099, 17, 20314, {1, 3, 1, 15, 5, 1, 35, 225, 151, 637, 1347, 833, 7077, 13145, 10285, 46583, 14351}},
	{8100, 17, 20320, {1, 1, 3, 15, 27, 63, 119, 159, 209, 421, 1413, 2727, 1607, 7175, 6133, 29051, 97387}},
	{8101, 17, 20326, {1, 1, 3, 5, 9, 29, 35, 93, 353, 903, 1037, 469, 5473, 7193, 21883, 14709, 89023}},
	{8102, 17, 20332, {1, 1, 1, 11, 9, 17, 51, 155, 145, 443, 1985, 423, 4721, 15721, 9747, 10303, 21909}},
	{8103, 17, 20350, {1, 3, 7, 15, 19, 49, 53, 153, 241, 739, 1585, 3945, 4869, 11, 15845, 17937, 69397}},
	{8104, 17, 20360, {1, 1, 5, 7, 19, 53, 43, 211, 327, 723, 1513, 1569, 919, 1771, 11309, 50787, 7459}},
	{8105, 17, 20371, {1, 1, 1, 9, 7, 29, 49, 89, 409, 685, 201, 1327, 2807, 13101, 2485, 62909, 102595}},
	{8106, 17, 20373, {1, 3, 1, 13, 21, 51, 37, 231, 271, 475, 855, 3703, 4447, 5161, 17937, 14471, 47173}},
	{8107, 17, 20377, {1, 1, 7, 3, 9, 7, 121, 197, 71, 147, 1669, 1745, 6589, 15029, 1529, 12625, 121925}},
	{8108, 17, 20390, {1, 1, 1, 3, 7, 47, 63, 61, 187, 341, 919, 307, 389, 14141, 12941, 17917, 104289}},
	{8109, 17, 20396, {1, 3, 5, 13, 19, 43, 57, 11, 383, 311, 1229, 3527, 3301, 12473, 24377, 16279, 92733}},
	{8110, 17, 20404, {1, 3, 3, 5, 25, 35, 63, 23, 131, 481, 809, 3627, 5685, 13695, 14121, 64751, 66181}},
	{8111, 17, 20413, {1, 3, 1, 5, 11, 43, 89, 55, 103, 219, 1861, 3223, 5111, 5879, 31399, 1395, 87419}},
	{8112, 17, 20434, {1, 3, 1, 11, 21, 27, 123, 205, 47, 923, 7, 1635, 4019, 8431, 28313, 24275, 129617}},
	{8113, 17, 20436, {1, 1, 1, 3, 1, 11, 91, 215, 393, 999, 1071, 3225, 4415, 759, 24499, 16109, 33791}},
	{8114, 17, 20440, {1, 1, 3, 13, 19, 45, 77, 103, 105, 395, 529, 3631, 8179, 4467, 30263, 39379, 70507}},
	{8115, 17, 20443, {1, 3, 3, 9, 17, 45, 101, 219, 433, 971, 471, 1243, 6955, 5941, 20641, 16119, 129383}},
	{8116, 17, 20445, {1, 1, 7, 7, 9, 9, 91, 95, 503, 171, 129, 1509, 7179, 5367, 2219, 50445, 112459}},
	{8117, 17, 20464, {1, 3, 7, 1, 17, 17, 19, 173, 229, 519, 147, 1835, 3797, 8091, 20335, 33177, 90197}},
	{8118, 17, 20476, {1, 3, 3, 5, 23, 29, 107, 205, 43, 969, 799, 1239, 1353, 5221, 15175, 42945, 34043}},
	{8119, 17, 20494, {1, 1, 5, 7, 31, 63, 67, 87, 189, 301, 1719, 3937, 965, 2505, 24781, 25133, 91675}},
	{8120, 17, 20496, {1, 3, 1, 7, 15, 25, 11, 39, 281, 35, 1149, 1445, 6451, 12069, 20959, 29895, 60059}},
	{8121, 17, 20501, {1, 1, 5, 1, 1, 17, 65, 213, 359, 561, 2015, 1629, 3521, 6877, 8099, 62483, 103903}},
	{8122, 17, 20518, {1, 1, 7, 9, 7, 49, 1, 227, 49, 823, 1141, 2419, 2697, 13293, 14143, 6323, 16081}},
	{8123, 17, 20527, {1, 3, 3, 1, 9, 13, 99, 235, 343, 601, 927, 183, 4545, 14529, 5521, 55571, 90675}},
	{8124, 17, 20529, {1, 1, 5, 1, 13, 49, 95, 153, 131, 955, 283, 2951, 3651, 7743, 4285, 42621, 110577}},
	{8125, 17, 20535, {1, 1, 1, 9, 19, 59, 97, 181, 67, 357, 497, 287, 3523, 3729, 28981, 59687, 39463}},
	{8126, 17, 20544, {1, 1, 3, 7, 5, 19, 107, 55, 393, 225, 1953, 669, 8063, 6537, 15983, 59891, 95079}},
	{8127, 17, 20568, {1, 3, 1, 5, 29, 21, 17, 169, 447, 697, 1613, 3483, 3139, 11175, 28865, 12065, 130241}},
	{8128, 17, 20589, {1, 3, 5, 7, 5, 49, 35, 181, 85, 505, 1537, 1345, 773, 3255, 27405, 3959, 126377}},
	{8129, 17, 20592, {1, 1, 7, 15, 9, 9, 17, 99, 409, 319, 807, 1721, 4023, 2171, 32657, 51663, 23253}},
	{8130, 17, 20601, {1, 3, 5, 1, 5, 3, 37, 219, 89, 263, 397, 573, 6147, 9525, 2521, 11153, 94319}},
	{8131, 17, 20617, {1, 1, 5, 5, 11, 39, 55, 205, 209, 239, 1443, 2477, 1941, 8337, 2883, 54593, 129859}},
	{8132, 17, 20625, {1, 1, 1, 7, 11, 13, 127, 65, 127, 413, 1553, 413, 3395, 9451, 7517, 34103, 57029}},
	{8133, 17, 20626, {1, 1, 1, 15, 5, 25, 109, 181, 399, 1023, 277, 3365, 6209, 827, 13933, 27483, 63835}},
	{8134, 17, 20632, {1, 1, 3, 3, 21, 57, 95, 127, 481, 365, 197, 3631, 7443, 4925, 31277, 35061, 35875}},
	{8135, 17, 20638, {1, 1, 7, 13, 3, 31, 59, 127, 441, 967, 1049, 1281, 3553, 375, 9683, 45755, 18889}},
	{8136, 17, 20644, {1, 1, 1, 13, 11, 39, 49, 43, 383, 607, 157, 1887, 3623, 13335, 31949, 49843, 96781}},
	{8137, 17, 20647, {1, 3, 7, 13, 19, 35, 21, 9, 299, 425, 1921, 3481, 6849, 4149, 5227, 56737, 27559}},
	{8138, 17, 20662, {1, 3, 7, 5, 21, 11, 79, 97, 1, 849, 819, 1133, 3393, 5429, 10621, 38787, 120785}},
	{8139, 17, 20671, {1, 1, 1, 13, 21, 29, 3, 239, 399, 619, 759, 2655, 3691, 655, 30979, 15507, 114791}},
	{8140, 17, 20674, {1, 3, 5, 3, 1, 61, 79, 43, 273, 965, 1759, 3901, 2437, 1703, 20205, 46291, 23679}},
	{8141, 17, 20683, {1, 1, 1, 9, 19, 57, 75, 245, 377, 261, 231, 3683, 6745, 7797, 28471, 56269, 109969}},
	{8142, 17, 20704, {1, 3, 1, 11, 9, 55, 53, 87, 33, 431, 1805, 2933, 455, 12479, 16235, 2667, 70105}},
	{8143, 17, 20724, {1, 3, 5, 1, 29, 1, 101, 17, 377, 499, 1365, 209, 4819, 15099, 8769, 37003, 53003}},
	{8144, 17, 20742, {1, 3, 5, 11, 11, 39, 109, 235, 337, 393, 35, 1071, 7085, 7165, 25143, 24223, 71493}},
	{8145, 17, 20748, {1, 3, 1, 5, 13, 49, 9, 205, 305, 943, 799, 2405, 319, 10755, 9785, 32023, 48015}},
	{8146, 17, 20765, {1, 3, 1, 9, 29, 35, 123, 101, 73, 747, 1257, 407, 5871, 4271, 14837, 52727, 13339}},
	{8147, 17, 20776, {1, 3, 3, 9, 31, 7, 113, 27, 89, 123, 1117, 531, 5531, 7897, 11209, 35267, 123457}},
	{8148, 17, 20789, {1, 1, 1, 1, 29, 19, 93, 11, 61, 743, 267, 13, 6561, 7667, 20537, 12675, 10481}},
	{8149, 17, 20796, {1, 1, 5, 13, 27, 47, 103, 171, 349, 139, 1709, 961, 783, 7147, 5569, 53395, 80797}},
	{8150, 17, 20802, {1, 3, 1, 9, 21, 49, 41, 175, 507, 861, 1157, 1033, 6795, 5795, 603, 12485, 75263}},
	{8151, 17, 20807, {1, 1, 5, 7, 23, 3, 21, 165, 123, 951, 785, 2065, 8001, 7009, 22981, 10011, 9807}},
	{8152, 17, 20814, {1, 3, 7, 15, 1, 53, 49, 197, 231, 351, 141, 3465, 7907, 10695, 30913, 26753, 71079}},
	{8153, 17, 20821, {1, 3, 5, 3, 29, 45, 23, 131, 65, 507, 75, 275, 7287, 187, 1093, 52657, 31533}},
	{8154, 17, 20832, {1, 3, 5, 9, 9, 7, 113, 125, 441, 75, 663, 4081, 3147, 1469, 28375, 35747, 122965}},
	{8155, 17, 20835, {1, 1, 7, 3, 3, 57, 5, 17, 183, 237, 965, 3709, 4161, 9513, 217, 58835, 78789}},
	{8156, 17, 20847, {1, 1, 3, 1, 7, 25, 13, 29, 173, 319, 1723, 57, 2401, 10405, 15541, 52915, 93747}},
	{8157, 17, 20859, {1, 1, 7, 5, 1, 31, 11, 61, 341, 97, 1199, 2585, 5909, 3707, 31507, 51233, 2389}},
	{8158, 17, 20871, {1, 1, 5, 15, 15, 21, 127, 155, 229, 203, 1303, 3215, 1965, 9481, 31909, 52307, 112207}},
	{8159, 17, 20883, {1, 3, 1, 13, 9, 45, 91, 39, 113, 587, 895, 637, 2475, 1695, 9347, 53255, 75797}},
	{8160, 17, 20886, {1, 3, 5, 13, 17, 11, 35, 83, 69, 255, 741, 3927, 153, 11001, 29145, 37107, 51873}},
	{8161, 17, 20892, {1, 1, 7, 5, 5, 37, 35, 219, 153, 1005, 973, 2555, 893, 5931, 23857, 34631, 74561}},
	{8162, 17, 20906, {1, 3, 1, 11, 21, 63, 113, 29, 115, 307, 957, 407, 879, 4819, 2865, 1773, 116825}},
	{8163, 17, 20908, {1, 3, 7, 3, 19, 55, 87, 21, 139, 571, 311, 2295, 2729, 6371, 11845, 30223, 19247}},
	{8164, 17, 20957, {1, 3, 7, 5, 23, 9, 59, 25, 357, 863, 435, 2509, 5599, 14039, 25731, 41645, 255}},
	{8165, 17, 20962, {1, 3, 7, 13, 9, 3, 63, 115, 503, 489, 1585, 813, 5419, 691, 23973, 18677, 59979}},
	{8166, 17, 20968, {1, 1, 1, 1, 13, 3, 55, 23, 185, 731, 459, 1497, 433, 16243, 29995, 20777, 59513}},
	{8167, 17, 20979, {1, 1, 7, 3, 27, 55, 77, 57, 349, 5, 617, 385, 6225, 7025, 23335, 877, 21973}},
	{8168, 17, 20991, {1, 3, 3, 5, 3, 37, 105, 197, 153, 639, 1643, 1093, 801, 4605, 4551, 46081, 7739}},
	{8169, 17, 20998, {1, 1, 5, 11, 29, 23, 5, 91, 39, 489, 2029, 887, 4451, 11463, 5641, 56299, 34027}},
	{8170, 17, 21007, {1, 1, 7, 3, 17, 11, 25, 161, 317, 701, 155, 1989, 7549, 11529, 9945, 18395, 61251}},
	{8171, 17, 21010, {1, 1, 7, 13, 23, 55, 113, 91, 17, 149, 1893, 2793, 8185, 81, 29487, 47925, 51837}},
	{8172, 17, 21026, {1, 3, 7, 7, 9, 29, 103, 161, 215, 129, 113, 1987, 919, 9639, 20715, 6541, 15041}},
	{8173, 17, 21037, {1, 1, 5, 9, 19, 19, 127, 43, 263, 997, 1687, 3801, 4249, 6309, 25889, 1787, 122771}},
	{8174, 17, 21038, {1, 3, 5, 13, 17, 3, 91, 183, 349, 467, 333, 3299, 3085, 12135, 16801, 31471, 37227}},
	{8175, 17, 21045, {1, 1, 5, 3, 7, 53, 11, 221, 407, 545, 237, 3631, 1791, 3729, 19737, 921, 57303}},
	{8176, 17, 21057, {1, 3, 7, 15, 3, 27, 71, 45, 219, 9, 1135, 2267, 6841, 8637, 30317, 9397, 115425}},
	{8177, 17, 21082, {1, 1, 3, 5, 29, 59, 121, 225, 419, 813, 1725, 3969, 3451, 8457, 31803, 57659, 33263}},
	{8178, 17, 21093, {1, 3, 3, 3, 17, 3, 65, 249, 423, 293, 1333, 3947, 1477, 4005, 30445, 28171, 95823}},
	{8179, 17, 21094, {1, 3, 3, 11, 29, 45, 67, 89, 75, 95, 555, 1823, 2795, 11929, 1995, 30013, 116241}},
	{8180, 17, 21105, {1, 3, 3, 3, 23, 35, 87, 221, 385, 275, 803, 387, 7765, 15637, 27953, 20913, 25279}},
	{8181, 17, 21117, {1, 3, 7, 15, 15, 43, 21, 179, 393, 95, 1913, 1715, 4467, 15093, 13613, 50775, 37133}},
	{8182, 17, 21121, {1, 1, 7, 7, 31, 27, 49, 71, 323, 123, 597, 2395, 4449, 7249, 20197, 19789, 92685}},
	{8183, 17, 21124, {1, 1, 5, 13, 31, 37, 89, 225, 357, 201, 1887, 3915, 2165, 10809, 21623, 34375, 110905}},
	{8184, 17, 21136, {1, 1, 5, 7, 11, 53, 37, 55, 61, 679, 1465, 1587, 2215, 16237, 14985, 50507, 128637}},
	{8185, 17, 21139, {1, 1, 7, 1, 15, 53, 115, 21, 279, 555, 43, 2429, 7251, 2141, 4813, 47047, 119551}},
	{8186, 17, 21142, {1, 1, 5, 13, 11, 41, 59, 245, 337, 117, 1125, 4007, 947, 10591, 17795, 48535, 72171}},
	{8187, 17, 21145, {1, 1, 5, 15, 27, 41, 71, 43, 505, 539, 975, 1567, 795, 4433, 11689, 53051, 98819}},
	{8188, 17, 21167, {1, 1, 7, 9, 1, 57, 57, 137, 323, 311, 759, 3027, 3713, 13, 24133, 21451, 1153}},
	{8189, 17, 21170, {1, 1, 5, 15, 31, 49, 23, 123, 271, 549, 1995, 5, 6065, 3797, 1085, 50137, 19741}},
	{8190, 17, 21175, {1, 3, 3, 13, 5, 15, 21, 117, 487, 43, 1759, 2899, 4239, 9729, 16711, 31559, 34553}},
	{8191, 17, 21179, {1, 1, 5, 13, 5, 23, 83, 49, 147, 267, 923, 1377, 1687, 1793, 30383, 19537, 66989}},
	{8192, 17, 21182, {1, 1, 1, 13, 9, 41, 105, 241, 499, 891, 885, 3349, 4703, 5609, 11999, 58025, 69089}},
	{8193, 17, 21193, {1, 1, 7, 9, 21, 11, 121, 69, 115, 895, 91, 3745, 41, 12787, 26181, 31399, 30463}},
	{8194, 17, 21194, {1, 1, 7, 13, 11, 3, 23, 173, 5, 907, 45, 3465, 2807, 3731, 14347, 27973, 8567}},
	{8195, 17, 21207, {1, 3, 7, 5, 27, 47, 43, 25, 499, 57, 649, 705, 6223, 4213, 4549, 23213, 13657}},
	{8196, 17, 21217, {1, 1, 7, 11, 21, 35, 5, 79, 295, 359, 1993, 99, 7917, 14917, 2131, 45527, 31451}},
	{8197, 17, 21224, {1, 1, 5, 15, 1, 39, 85, 93, 65, 991, 389, 585, 4835, 11671, 10913, 41787, 84953}},
	{8198, 17, 21244, {1, 1, 1, 5, 27, 5, 1, 15, 11, 83, 1191, 3945, 4697, 7703, 6929, 6057, 88721}},
	{8199, 17, 21247, {1, 1, 3, 7, 27, 39, 71, 181, 191, 997, 419, 1671, 7771, 15305, 18677, 45033, 64745}},
	{8200, 17, 21252, {1, 3, 7, 3, 15, 41, 33, 239, 93, 307, 153, 2701, 1549, 5011, 6913, 19257, 55829}},
	{8201, 17, 21264, {1, 3, 3, 11, 21, 47, 69, 223, 365, 877, 431, 1629, 4803, 11591, 13973, 56359, 11897}},
	{8202, 17, 21273, {1, 3, 7, 7, 1, 59, 63, 129, 251, 873, 603, 2707, 2847, 8739, 31343, 63291, 5607}},
	{8203, 17, 21289, {1, 3, 5, 5, 19, 13, 79, 235, 151, 571, 953, 2191, 5973, 4751, 11119, 14439, 97755}},
	{8204, 17, 21290, {1, 1, 5, 1, 27, 3, 105, 139, 13, 821, 221, 2025, 7303, 1891, 28193, 45537, 92703}},
	{8205, 17, 21295, {1, 3, 7, 9, 13, 63, 27, 149, 51, 121, 587, 3695, 4115, 3955, 22493, 34903, 51669}},
	{8206, 17, 21297, {1, 1, 5, 7, 19, 5, 89, 87, 269, 585, 421, 3827, 315, 14739, 109, 43009, 94969}},
	{8207, 17, 21309, {1, 1, 5, 15, 9, 53, 125, 83, 159, 917, 1583, 585, 6839, 14957, 20007, 60467, 96309}},
	{8208, 17, 21318, {1, 3, 5, 1, 23, 49, 109, 91, 17, 731, 1083, 3153, 1825, 14293, 25639, 44599, 47541}},
	{8209, 17, 21322, {1, 1, 3, 7, 21, 51, 45, 25, 367, 925, 799, 1673, 6977, 7155, 829, 25899, 104615}},
	{8210, 17, 21324, {1, 3, 3, 13, 13, 49, 95, 239, 195, 353, 1967, 1419, 929, 503, 11717, 9485, 62885}},
	{8211, 17, 21329, {1, 1, 1, 15, 3, 41, 109, 91, 327, 789, 1429, 1159, 2801, 4845, 19663, 47737, 11029}},
	{8212, 17, 21332, {1, 3, 5, 1, 21, 63, 57, 107, 229, 771, 1911, 647, 6989, 12615, 23191, 64941, 97595}},
	{8213, 17, 21336, {1, 1, 1, 15, 5, 13, 15, 109, 459, 447, 1625, 1269, 7629, 7929, 4405, 12143, 40481}},
	{8214, 17, 21342, {1, 3, 3, 1, 23, 3, 95, 229, 363, 379, 1149, 1615, 5125, 3645, 27535, 58791, 38091}},
	{8215, 17, 21351, {1, 1, 1, 1, 9, 45, 119, 85, 151, 171, 1123, 41, 6517, 8067, 17845, 23301, 95383}},
	{8216, 17, 21355, {1, 3, 1, 15, 17, 31, 103, 71, 35, 1019, 1687, 1175, 6119, 14075, 26601, 28873, 36617}},
	{8217, 17, 21363, {1, 3, 3, 9, 13, 39, 7, 17, 207, 219, 637, 3025, 1709, 4031, 563, 14865, 129389}},
	{8218, 17, 21372, {1, 3, 7, 1, 21, 11, 121, 85, 111, 641, 1163, 3173, 5189, 13261, 19471, 39635, 76545}},
	{8219, 17, 21382, {1, 3, 7, 15, 3, 45, 3, 37, 121, 967, 1861, 3257, 3699, 6881, 11905, 8413, 59397}},
	{8220, 17, 21388, {1, 3, 3, 13, 25, 53, 85, 181, 1, 979, 2045, 297, 1739, 8139, 17897, 35251, 7193}},
	{8221, 17, 21394, {1, 1, 1, 3, 5, 49, 77, 115, 377, 209, 1415, 3747, 485, 803, 2507, 27729, 52201}},
	{8222, 17, 21403, {1, 3, 1, 5, 31, 55, 85, 171, 135, 893, 1423, 3693, 6155, 5321, 8297, 39183, 88377}},
	{8223, 17, 21409, {1, 3, 3, 15, 1, 35, 73, 239, 181, 101, 509, 2449, 4955, 13049, 27631, 16871, 40151}},
	{8224, 17, 21416, {1, 1, 7, 13, 27, 7, 109, 71, 437, 835, 563, 1355, 3681, 7431, 32743, 59693, 125055}},
	{8225, 17, 21421, {1, 1, 7, 5, 19, 23, 29, 147, 291, 507, 1943, 2069, 3309, 11569, 1031, 49345, 86735}},
	{8226, 17, 21424, {1, 1, 3, 13, 17, 45, 91, 167, 19, 137, 527, 961, 4435, 2277, 6863, 57917, 129407}},
	{8227, 17, 21444, {1, 3, 5, 7, 11, 31, 79, 207, 43, 871, 1121, 2929, 6899, 4099, 29533, 45333, 33299}},
	{8228, 17, 21453, {1, 1, 7, 5, 5, 49, 13, 95, 475, 91, 337, 3531, 3157, 1331, 32655, 46169, 3549}},
	{8229, 17, 21466, {1, 3, 1, 5, 5, 9, 73, 177, 123, 251, 717, 541, 7083, 6907, 1417, 31203, 9755}},
	{8230, 17, 21471, {1, 3, 1, 11, 21, 11, 91, 155, 447, 165, 1525, 2073, 5103, 193, 2677, 43673, 70579}},
	{8231, 17, 21495, {1, 3, 7, 1, 7, 27, 115, 247, 211, 779, 1999, 209, 3215, 111, 25567, 34641, 130873}},
	{8232, 17, 21499, {1, 1, 5, 9, 25, 7, 15, 19, 217, 831, 1577, 2051, 3533, 2337, 7675, 2845, 69135}},
	{8233, 17, 21504, {1, 3, 5, 15, 29, 11, 91, 59, 221, 383, 1235, 1261, 2967, 14989, 11455, 6459, 58047}},
	{8234, 17, 21507, {1, 3, 5, 1, 3, 35, 5, 127, 99, 981, 493, 3001, 5651, 3125, 27789, 57389, 115631}},
	{8235, 17, 21521, {1, 3, 5, 5, 29, 63, 123, 161, 247, 177, 1653, 2665, 3903, 11129, 20961, 49429, 44075}},
	{8236, 17, 21527, {1, 3, 1, 1, 13, 9, 57, 167, 291, 765, 1929, 397, 5503, 5601, 6957, 62003, 104631}},
	{8237, 17, 21555, {1, 1, 7, 15, 9, 43, 57, 85, 157, 361, 1931, 2183, 8045, 14939, 2169, 25733, 29095}},
	{8238, 17, 21558, {1, 1, 3, 15, 13, 35, 47, 123, 13, 667, 1373, 4069, 6259, 13453, 13439, 25349, 99437}},
	{8239, 17, 21562, {1, 3, 7, 1, 31, 15, 69, 45, 355, 919, 415, 793, 3987, 8785, 4905, 8177, 123989}},
	{8240, 17, 21570, {1, 3, 7, 13, 29, 27, 69, 57, 385, 185, 171, 2499, 3983, 13437, 23585, 21501, 88079}},
	{8241, 17, 21576, {1, 1, 5, 11, 27, 3, 77, 99, 221, 997, 1653, 1963, 2251, 15505, 26347, 51933, 100679}},
	{8242, 17, 21579, {1, 1, 1, 7, 19, 39, 49, 69, 193, 235, 959, 2823, 2573, 8001, 4389, 13217, 60975}},
	{8243, 17, 21581, {1, 1, 7, 5, 1, 3, 3, 189, 199, 293, 1225, 1913, 7271, 2255, 661, 23293, 82069}},
	{8244, 17, 21587, {1, 1, 5, 5, 21, 31, 35, 113, 47, 479, 325, 1663, 7409, 8975, 14151, 56317, 79663}},
	{8245, 17, 21590, {1, 1, 5, 9, 15, 63, 99, 135, 277, 715, 667, 387, 6929, 12873, 12913, 2599, 84939}},
	{8246, 17, 21599, {1, 1, 7, 15, 23, 39, 67, 25, 179, 313, 459, 295, 1103, 1737, 7529, 29463, 104693}},
	{8247, 17, 21605, {1, 1, 3, 13, 23, 11, 111, 67, 105, 191, 1967, 3497, 7621, 487, 18545, 59521, 115315}},
	{8248, 17, 21612, {1, 1, 1, 7, 25, 45, 83, 61, 231, 569, 155, 2817, 6985, 5289, 6731, 3087, 89749}},
	{8249, 17, 21618, {1, 3, 7, 1, 1, 61, 103, 49, 135, 411, 659, 1735, 4461, 8077, 12885, 62791, 114769}},
	{8250, 17, 21630, {1, 1, 7, 13, 3, 53, 21, 81, 433, 563, 857, 891, 195, 11669, 24769, 56539, 47601}},
	{8251, 17, 21639, {1, 3, 1, 13, 3, 41, 59, 101, 67, 803, 1209, 3267, 1255, 5763, 5483, 36339, 38451}},
	{8252, 17, 21640, {1, 3, 5, 3, 25, 51, 53, 213, 329, 11, 483, 81, 2151, 7623, 26309, 15289, 85103}},
	{8253, 17, 21643, {1, 3, 3, 13, 23, 17, 9, 161, 417, 207, 39, 3615, 7567, 15207, 20383, 58885, 121765}},
	{8254, 17, 21648, {1, 3, 1, 7, 15, 59, 9, 195, 187, 583, 341, 2737, 3891, 331, 18055, 60455, 113271}},
	{8255, 17, 21669, {1, 1, 3, 3, 19, 25, 95, 37, 281, 59, 1613, 2733, 5715, 4067, 5509, 5851, 35189}},
	{8256, 17, 21679, {1, 3, 1, 3, 31, 43, 125, 107, 341, 109, 1991, 849, 7795, 13607, 20421, 3339, 78979}},
	{8257, 17, 21681, {1, 3, 7, 13, 15, 57, 87, 151, 479, 99, 479, 447, 7407, 303, 16397, 15699, 122273}},
	{8258, 17, 21687, {1, 1, 3, 1, 27, 61, 79, 195, 5, 839, 1411, 3451, 4627, 13715, 18401, 24325, 44027}},
	{8259, 17, 21688, {1, 1, 7, 15, 21, 39, 57, 207, 213, 367, 547, 1203, 6385, 2555, 31465, 15675, 7133}},
	{8260, 17, 21706, {1, 1, 5, 15, 27, 3, 75, 123, 337, 1019, 1525, 3065, 1919, 10779, 27409, 6291, 86291}},
	{8261, 17, 21713, {1, 1, 7, 11, 15, 27, 67, 145, 125, 521, 647, 2957, 6337, 14973, 24139, 29107, 27151}},
	{8262, 17, 21714, {1, 1, 1, 13, 25, 57, 103, 83, 321, 111, 131, 2051, 5267, 4723, 1939, 40389, 4803}},
	{8263, 17, 21716, {1, 3, 1, 7, 29, 7, 35, 133, 349, 855, 613, 2563, 2261, 2119, 13939, 24727, 26719}},
	{8264, 17, 21730, {1, 3, 3, 1, 11, 61, 25, 177, 427, 1005, 2027, 649, 7871, 7803, 4717, 59207, 57945}},
	{8265, 17, 21732, {1, 1, 7, 1, 15, 45, 75, 133, 193, 745, 485, 197, 6001, 13837, 615, 43629, 127321}},
	{8266, 17, 21749, {1, 3, 3, 13, 5, 7, 101, 183, 211, 283, 1327, 1395, 8175, 13359, 18361, 54243, 3555}},
	{8267, 17, 21756, {1, 1, 7, 13, 7, 43, 19, 41, 319, 701, 795, 1407, 7113, 9149, 31953, 17075, 53975}},
	{8268, 17, 21774, {1, 3, 5, 13, 11, 19, 101, 125, 327, 75, 1471, 3465, 2247, 5107, 11519, 45161, 71373}},
	{8269, 17, 21779, {1, 3, 7, 13, 23, 59, 53, 57, 137, 575, 59, 3829, 963, 11259, 25771, 29223, 79535}},
	{8270, 17, 21781, {1, 3, 3, 11, 17, 31, 111, 147, 499, 441, 1385, 769, 6901, 8349, 1427, 16561, 7485}},
	{8271, 17, 21786, {1, 1, 7, 9, 21, 7, 47, 83, 351, 867, 265, 1329, 7853, 6959, 11821, 44947, 42275}},
	{8272, 17, 21792, {1, 1, 7, 15, 3, 17, 79, 143, 449, 577, 1007, 1101, 3229, 6861, 23921, 37551, 117309}},
	{8273, 17, 21810, {1, 3, 5, 11, 27, 63, 107, 15, 289, 821, 1723, 1945, 1373, 14469, 30985, 55987, 21255}},
	{8274, 17, 21819, {1, 3, 5, 3, 21, 39, 79, 85, 485, 733, 2031, 1573, 6873, 12225, 30471, 54233, 26967}},
	{8275, 17, 21829, {1, 3, 5, 7, 17, 29, 93, 251, 437, 583, 825, 551, 6545, 10905, 27863, 37037, 52129}},
	{8276, 17, 21830, {1, 3, 7, 9, 23, 1, 23, 85, 195, 319, 1759, 3985, 2413, 16205, 22197, 48821, 94907}},
	{8277, 17, 21844, {1, 1, 3, 7, 17, 47, 3, 195, 167, 925, 11, 3431, 1767, 5917, 13915, 54565, 64625}},
	{8278, 17, 21853, {1, 3, 1, 13, 23, 43, 97, 93, 313, 773, 591, 127, 6005, 11497, 32573, 8173, 92053}},
	{8279, 17, 21867, {1, 1, 5, 9, 17, 47, 115, 237, 389, 619, 377, 561, 1333, 6433, 9743, 32673, 98039}},
	{8280, 17, 21869, {1, 3, 7, 15, 23, 17, 99, 225, 145, 191, 2041, 581, 841, 9377, 18123, 32773, 66849}},
	{8281, 17, 21882, {1, 3, 7, 15, 21, 49, 97, 41, 357, 527, 2019, 2083, 2611, 12449, 20037, 52503, 105991}},
	{8282, 17, 21891, {1, 1, 5, 13, 17, 53, 41, 75, 355, 207, 1675, 591, 5797, 9217, 16443, 3205, 81905}},
	{8283, 17, 21898, {1, 3, 7, 11, 1, 61, 29, 207, 449, 103, 1527, 2327, 7895, 10137, 25223, 51607, 60809}},
	{8284, 17, 21917, {1, 3, 3, 5, 15, 57, 87, 233, 301, 989, 485, 2143, 7411, 5475, 23377, 56005, 59721}},
	{8285, 17, 21934, {1, 3, 1, 15, 29, 7, 95, 141, 369, 231, 735, 1103, 1565, 11575, 571, 3257, 62961}},
	{8286, 17, 21946, {1, 1, 5, 15, 27, 19, 25, 35, 303, 555, 95, 1323, 6139, 5079, 21763, 59591, 103537}},
	{8287, 17, 21948, {1, 1, 1, 13, 25, 23, 85, 151, 135, 349, 1753, 1061, 7697, 1723, 5213, 12581, 103995}},
	{8288, 17, 21963, {1, 3, 1, 9, 29, 51, 101, 195, 59, 809, 1527, 2179, 63, 3681, 29823, 57537, 121371}},
	{8289, 17, 21966, {1, 1, 7, 11, 27, 61, 85, 213, 245, 261, 1649, 2423, 6127, 5687, 4247, 56061, 109793}},
	{8290, 17, 21968, {1, 3, 5, 15, 11, 33, 127, 31, 269, 857, 2027, 2611, 1729, 11783, 16459, 31083, 30671}},
	{8291, 17, 21973, {1, 1, 7, 9, 11, 29, 127, 177, 505, 227, 1499, 1309, 6855, 9999, 21815, 32987, 79109}},
	{8292, 17, 21977, {1, 3, 7, 11, 7, 21, 107, 1, 493, 459, 867, 3199, 7985, 12957, 28197, 41133, 105985}},
	{8293, 17, 21980, {1, 1, 3, 15, 1, 57, 113, 97, 213, 547, 1017, 2961, 461, 16125, 10621, 4243, 58277}},
	{8294, 17, 21984, {1, 1, 3, 5, 11, 57, 61, 47, 209, 961, 333, 795, 4491, 15115, 25745, 62633, 66269}},
	{8295, 17, 21994, {1, 1, 7, 3, 19, 13, 49, 167, 455, 863, 581, 1407, 4247, 15023, 2247, 19981, 125891}},
	{8296, 17, 21999, {1, 1, 7, 15, 17, 55, 27, 35, 33, 349, 879, 1781, 1075, 2475, 30689, 42043, 29423}},
	{8297, 17, 22018, {1, 1, 1, 11, 25, 53, 121, 117, 117, 845, 447, 3927, 1951, 8643, 24497, 44833, 99533}},
	{8298, 17, 22020, {1, 1, 7, 13, 3, 59, 117, 9, 359, 453, 327, 3419, 5957, 97, 20541, 49441, 5673}},
	{8299, 17, 22029, {1, 3, 5, 5, 31, 35, 95, 107, 435, 733, 827, 1361, 6627, 8905, 2681, 25473, 46093}},
	{8300, 17, 22032, {1, 3, 3, 5, 7, 23, 75, 137, 231, 915, 637, 2963, 4409, 12799, 31587, 65363, 69539}},
	{8301, 17, 22041, {1, 1, 1, 7, 15, 35, 19, 233, 189, 837, 243, 2525, 6185, 565, 8133, 4265, 3089}},
	{8302, 17, 22047, {1, 1, 5, 5, 19, 59, 103, 201, 287, 449, 21, 2331, 341, 13145, 18607, 46407, 2767}},
	{8303, 17, 22048, {1, 3, 3, 15, 19, 41, 49, 179, 109, 367, 1185, 1045, 1635, 9647, 16613, 25357, 34291}},
	{8304, 17, 22071, {1, 3, 5, 1, 13, 11, 89, 25, 159, 637, 1979, 549, 3553, 9163, 227, 50553, 46307}},
	{8305, 17, 22075, {1, 1, 3, 1, 17, 33, 73, 239, 261, 751, 1267, 2643, 2549, 8331, 25083, 9715, 67289}},
	{8306, 17, 22078, {1, 1, 1, 13, 3, 49, 7, 35, 367, 293, 903, 1045, 569, 6017, 27635, 51833, 32963}},
	{8307, 17, 22083, {1, 3, 5, 3, 31, 3, 69, 137, 57, 87, 719, 3977, 3031, 7675, 24605, 8757, 93173}},
	{8308, 17, 22089, {1, 3, 3, 1, 7, 45, 97, 35, 233, 69, 1525, 4047, 2599, 13679, 4389, 49079, 121465}},
	{8309, 17, 22097, {1, 1, 7, 13, 7, 25, 57, 211, 337, 189, 1825, 2451, 7651, 11277, 27763, 40671, 57223}},
	{8310, 17, 22110, {1, 1, 1, 1, 15, 59, 55, 169, 461, 907, 407, 803, 3349, 4727, 20983, 47717, 51647}},
	{8311, 17, 22113, {1, 3, 7, 1, 15, 51, 25, 119, 439, 593, 1289, 3959, 5489, 13283, 31837, 8441, 58373}},
	{8312, 17, 22119, {1, 3, 1, 9, 5, 1, 81, 45, 13, 537, 1091, 3861, 6781, 5679, 2807, 29757, 40917}},
	{8313, 17, 22120, {1, 3, 5, 3, 27, 41, 19, 235, 207, 697, 775, 837, 3431, 3175, 10807, 42775, 67987}},
	{8314, 17, 22126, {1, 3, 7, 3, 29, 33, 35, 119, 271, 609, 1747, 2839, 3415, 2275, 30979, 41293, 99341}},
	{8315, 17, 22133, {1, 3, 3, 3, 5, 17, 13, 169, 269, 709, 1449, 3169, 1545, 16075, 8937, 39705, 19609}},
	{8316, 17, 22134, {1, 3, 5, 15, 29, 13, 1, 199, 65, 385, 977, 797, 1181, 10979, 241, 40393, 73663}},
	{8317, 17, 22140, {1, 1, 3, 7, 17, 35, 47, 63, 193, 451, 151, 3415, 99, 14557, 26025, 31361, 112639}},
	{8318, 17, 22147, {1, 1, 3, 5, 19, 13, 29, 33, 365, 311, 1241, 217, 6205, 13067, 18585, 21693, 97127}},
	{8319, 17, 22161, {1, 1, 3, 15, 19, 7, 87, 25, 91, 13, 1839, 1445, 957, 9779, 25557, 37027, 38987}},
	{8320, 17, 22164, {1, 1, 5, 1, 21, 5, 79, 67, 481, 455, 37, 1321, 7723, 1413, 7919, 11035, 5739}},
	{8321, 17, 22173, {1, 1, 1, 15, 9, 55, 111, 1, 383, 439, 1037, 4055, 4243, 10443, 26737, 21039, 130847}},
	{8322, 17, 22197, {1, 1, 7, 9, 13, 25, 71, 137, 307, 717, 1009, 2477, 3861, 14145, 14549, 59589, 93401}},
	{8323, 17, 22207, {1, 1, 7, 5, 29, 63, 77, 49, 471, 267, 1457, 1743, 1915, 14793, 17899, 28011, 92183}},
	{8324, 17, 22210, {1, 3, 7, 7, 7, 41, 47, 251, 75, 749, 1915, 1015, 5869, 3211, 24097, 14349, 130571}},
	{8325, 17, 22216, {1, 1, 1, 1, 31, 63, 105, 83, 345, 147, 975, 135, 7299, 15801, 19311, 26143, 80293}},
	{8326, 17, 22234, {1, 1, 3, 1, 7, 1, 47, 45, 251, 635, 583, 3515, 5233, 6281, 7797, 37949, 75877}},
	{8327, 17, 22257, {1, 1, 3, 3, 5, 53, 99, 175, 155, 327, 1841, 211, 2811, 16099, 17255, 34253, 124141}},
	{8328, 17, 22264, {1, 3, 1, 3, 13, 27, 81, 217, 115, 245, 101, 1641, 29, 1441, 4829, 28399, 102303}},
	{8329, 17, 22278, {1, 3, 1, 5, 11, 55, 31, 91, 337, 203, 987, 977, 4929, 14933, 25149, 20493, 19783}},
	{8330, 17, 22284, {1, 1, 5, 9, 9, 37, 103, 211, 349, 165, 1421, 3015, 5133, 4615, 28173, 45787, 10711}},
	{8331, 17, 22287, {1, 1, 1, 1, 1, 17, 29, 117, 421, 651, 1617, 1677, 7841, 16303, 8843, 1321, 90701}},
	{8332, 17, 22299, {1, 1, 1, 15, 27, 23, 49, 195, 139, 319, 1277, 901, 63, 14677, 21815, 19497, 24883}},
	{8333, 17, 22301, {1, 3, 3, 13, 1, 23, 17, 189, 293, 765, 1503, 1485, 7427, 11807, 17629, 61739, 111365}},
	{8334, 17, 22308, {1, 1, 5, 5, 15, 41, 25, 53, 221, 449, 1597, 2763, 4119, 6319, 17509, 23493, 104707}},
	{8335, 17, 22337, {1, 3, 7, 11, 29, 21, 101, 197, 161, 457, 331, 3817, 5139, 14307, 23225, 55567, 62535}},
	{8336, 17, 22349, {1, 1, 7, 5, 9, 57, 39, 101, 5, 847, 1311, 313, 2877, 14811, 21969, 31741, 8075}},
	{8337, 17, 22350, {1, 3, 5, 3, 1, 11, 45, 163, 251, 775, 1031, 1957, 1631, 1691, 3191, 6255, 13037}},
	{8338, 17, 22357, {1, 1, 3, 13, 7, 11, 95, 97, 409, 835, 707, 1579, 2409, 9451, 15069, 62425, 106499}},
	{8339, 17, 22364, {1, 3, 3, 11, 5, 25, 23, 207, 429, 299, 537, 1467, 6309, 891, 15009, 56733, 60397}},
	{8340, 17, 22373, {1, 3, 5, 3, 29, 47, 95, 115, 207, 177, 543, 427, 145, 11169, 7441, 10911, 87413}},
	{8341, 17, 22377, {1, 3, 7, 11, 25, 53, 15, 225, 115, 295, 919, 39, 513, 9989, 11045, 24015, 102387}},
	{8342, 17, 22380, {1, 1, 7, 15, 13, 31, 103, 143, 357, 825, 183, 137, 2671, 9803, 14777, 48333, 79483}},
	{8343, 17, 22386, {1, 3, 5, 1, 25, 13, 65, 9, 461, 307, 1289, 1035, 7253, 14223, 16829, 23361, 84987}},
	{8344, 17, 22391, {1, 3, 5, 7, 5, 57, 47, 251, 5, 9, 965, 2883, 3105, 13931, 807, 31977, 119035}},
	{8345, 17, 22392, {1, 1, 3, 5, 3, 7, 55, 165, 3, 787, 1587, 989, 6049, 14021, 30789, 15283, 92851}},
	{8346, 17, 22411, {1, 1, 5, 5, 3, 17, 11, 167, 487, 885, 193, 3485, 8179, 9485, 24913, 40267, 70625}},
	{8347, 17, 22422, {1, 1, 7, 1, 27, 31, 9, 139, 73, 137, 783, 321, 691, 6157, 19905, 45525, 84877}},
	{8348, 17, 22425, {1, 3, 1, 9, 17, 39, 127, 177, 301, 579, 1065, 3899, 281, 9177, 16295, 51217, 120293}},
	{8349, 17, 22431, {1, 1, 7, 9, 31, 59, 17, 93, 247, 779, 847, 1183, 3453, 1073, 18597, 2655, 121633}},
	{8350, 17, 22438, {1, 1, 7, 1, 25, 43, 47, 253, 23, 999, 973, 1201, 1061, 5947, 5619, 36311, 1545}},
	{8351, 17, 22441, {1, 3, 5, 7, 11, 5, 103, 119, 229, 657, 1993, 1991, 1597, 13165, 19137, 7161, 83487}},
	{8352, 17, 22442, {1, 1, 1, 1, 11, 23, 105, 183, 467, 83, 899, 2447, 4949, 4171, 28943, 4829, 13033}},
	{8353, 17, 22449, {1, 3, 1, 7, 15, 7, 47, 215, 253, 109, 1975, 3337, 1553, 13575, 16835, 61525, 26423}},
	{8354, 17, 22452, {1, 1, 7, 1, 21, 17, 53, 79, 175, 267, 999, 249, 6177, 10453, 12475, 59801, 47351}},
	{8355, 17, 22461, {1, 3, 5, 11, 3, 57, 5, 193, 421, 799, 1833, 2635, 6537, 4669, 9597, 40661, 12113}},
	{8356, 17, 22467, {1, 1, 7, 11, 9, 11, 69, 103, 139, 167, 159, 2469, 703, 1519, 21553, 62875, 60449}},
	{8357, 17, 22479, {1, 1, 5, 3, 9, 11, 17, 183, 499, 301, 1275, 605, 7655, 12141, 7783, 39413, 116263}},
	{8358, 17, 22484, {1, 1, 1, 7, 31, 55, 23, 79, 49, 247, 761, 3573, 8187, 4879, 27379, 15725, 81415}},
	{8359, 17, 22487, {1, 3, 5, 5, 5, 49, 23, 205, 509, 383, 1165, 3839, 7663, 1539, 19967, 55901, 4351}},
	{8360, 17, 22493, {1, 1, 1, 11, 7, 15, 3, 159, 235, 735, 391, 2231, 5043, 9759, 4569, 35601, 71989}},
	{8361, 17, 22510, {1, 3, 3, 15, 23, 3, 49, 97, 99, 517, 1097, 3517, 1035, 2319, 27705, 25547, 101555}},
	{8362, 17, 22521, {1, 3, 7, 11, 27, 29, 33, 241, 205, 113, 291, 1993, 3277, 13155, 1039, 42367, 130477}},
	{8363, 17, 22533, {1, 1, 1, 3, 29, 19, 15, 159, 35, 153, 1177, 3011, 6271, 8203, 8971, 19183, 102871}},
	{8364, 17, 22534, {1, 1, 1, 5, 5, 51, 19, 175, 209, 895, 229, 2355, 499, 7877, 4935, 22737, 35587}},
	{8365, 17, 22543, {1, 3, 7, 11, 15, 9, 7, 113, 41, 835, 1593, 3933, 7165, 10959, 15487, 30019, 114505}},
	{8366, 17, 22548, {1, 1, 7, 5, 31, 21, 27, 11, 421, 165, 1605, 1859, 29, 13051, 3273, 3893, 56089}},
	{8367, 17, 22552, {1, 3, 5, 5, 17, 51, 55, 187, 401, 977, 95, 2617, 727, 9609, 5075, 48989, 120299}},
	{8368, 17, 22558, {1, 1, 5, 7, 21, 31, 127, 87, 379, 125, 247, 3607, 2555, 11873, 32535, 16677, 122273}},
	{8369, 17, 22561, {1, 1, 1, 5, 19, 21, 51, 185, 203, 145, 1073, 167, 235, 12753, 17315, 14683, 44101}},
	{8370, 17, 22562, {1, 3, 3, 1, 5, 61, 71, 17, 63, 151, 823, 17, 5315, 4861, 17279, 23049, 84971}},
	{8371, 17, 22568, {1, 3, 3, 5, 21, 63, 21, 235, 295, 467, 1661, 2487, 335, 6107, 28709, 55875, 129085}},
	{8372, 17, 22571, {1, 3, 3, 5, 1, 55, 35, 187, 5, 687, 1633, 2999, 4513, 10105, 15249, 22591, 102857}},
	{8373, 17, 22574, {1, 3, 1, 5, 19, 1, 113, 27, 261, 623, 831, 3011, 4091, 11967, 17191, 17433, 99925}},
	{8374, 17, 22581, {1, 3, 5, 5, 25, 59, 81, 249, 463, 523, 183, 3049, 3675, 2705, 28379, 1279, 25579}},
	{8375, 17, 22594, {1, 1, 3, 9, 19, 19, 71, 127, 189, 613, 647, 1449, 7755, 1415, 9067, 30683, 79703}},
	{8376, 17, 22603, {1, 1, 7, 1, 27, 33, 61, 135, 233, 633, 1969, 2245, 5841, 14069, 6497, 63617, 101483}},
	{8377, 17, 22605, {1, 3, 3, 9, 23, 3, 17, 163, 309, 741, 2023, 2647, 5847, 7871, 22311, 38377, 70663}},
	{8378, 17, 22606, {1, 3, 5, 15, 31, 33, 51, 243, 209, 273, 1305, 1599, 6115, 6249, 8639, 5903, 17215}},
	{8379, 17, 22623, {1, 1, 1, 1, 21, 11, 107, 185, 463, 435, 149, 3789, 6283, 1327, 20893, 10417, 78673}},
	{8380, 17, 22630, {1, 1, 1, 13, 5, 53, 121, 129, 493, 419, 1711, 2765, 7673, 8979, 25845, 62759, 9669}},
	{8381, 17, 22651, {1, 3, 5, 5, 1, 39, 123, 47, 449, 639, 625, 2355, 511, 1685, 1415, 32417, 76529}},
	{8382, 17, 22657, {1, 3, 1, 11, 1, 49, 67, 237, 203, 967, 1401, 2773, 4951, 13889, 14147, 41031, 71897}},
	{8383, 17, 22669, {1, 3, 5, 11, 13, 49, 17, 113, 315, 207, 1057, 3395, 6151, 2767, 16571, 1811, 66403}},
	{8384, 17, 22670, {1, 1, 7, 7, 29, 63, 49, 115, 327, 987, 1853, 3355, 8139, 2703, 30039, 51343, 86999}},
	{8385, 17, 22677, {1, 1, 3, 9, 1, 3, 45, 35, 509, 483, 159, 1795, 8023, 6989, 3755, 20887, 13587}},
	{8386, 17, 22682, {1, 1, 3, 7, 1, 27, 39, 159, 283, 843, 317, 3229, 2297, 15031, 22039, 21721, 70583}},
	{8387, 17, 22698, {1, 1, 3, 11, 9, 23, 1, 35, 79, 77, 1671, 2583, 647, 12313, 16271, 2959, 108389}},
	{8388, 17, 22712, {1, 1, 1, 7, 5, 55, 1, 233, 429, 231, 833, 1279, 7815, 1051, 30627, 4435, 25997}},
	{8389, 17, 22715, {1, 3, 1, 15, 19, 53, 9, 165, 307, 437, 551, 2477, 1841, 11799, 18477, 5871, 20065}},
	{8390, 17, 22729, {1, 1, 1, 1, 21, 5, 65, 41, 77, 909, 93, 751, 2973, 7341, 30427, 60075, 71457}},
	{8391, 17, 22732, {1, 1, 3, 11, 25, 51, 49, 63, 165, 263, 1915, 747, 8053, 6361, 4843, 20189, 110147}},
	{8392, 17, 22735, {1, 3, 1, 9, 29, 9, 45, 177, 415, 557, 1555, 2967, 1239, 8115, 12853, 19193, 73681}},
	{8393, 17, 22738, {1, 1, 5, 5, 11, 5, 51, 157, 325, 517, 1601, 3911, 1487, 13631, 7483, 61515, 48937}},
	{8394, 17, 22740, {1, 3, 7, 5, 29, 31, 107, 47, 437, 837, 1791, 477, 1717, 7, 25855, 48793, 16385}},
	{8395, 17, 22750, {1, 1, 1, 3, 29, 49, 31, 255, 233, 935, 1993, 125, 2255, 12785, 2807, 54697, 62591}},
	{8396, 17, 22753, {1, 3, 1, 7, 15, 13, 9, 245, 79, 289, 841, 253, 5259, 16123, 29189, 63837, 127915}},
	{8397, 17, 22760, {1, 3, 7, 15, 15, 55, 91, 103, 445, 289, 1471, 423, 3387, 15609, 19311, 28993, 23473}},
	{8398, 17, 22765, {1, 1, 3, 11, 31, 39, 69, 125, 115, 309, 397, 3417, 5693, 10301, 1489, 25955, 2699}},
	{8399, 17, 22768, {1, 3, 3, 5, 13, 21, 51, 207, 239, 311, 1601, 2925, 6285, 9597, 30579, 62957, 6153}},
	{8400, 17, 22778, {1, 1, 7, 1, 27, 21, 63, 143, 399, 971, 1385, 1875, 5143, 6423, 6223, 27009, 14237}},
	{8401, 17, 22785, {1, 3, 5, 1, 5, 59, 125, 133, 151, 997, 1315, 3007, 8173, 16289, 13409, 839, 103519}},
	{8402, 17, 22809, {1, 1, 1, 13, 7, 57, 83, 33, 191, 121, 939, 3927, 6089, 10083, 5903, 52229, 78325}},
	{8403, 17, 22810, {1, 1, 3, 5, 9, 61, 43, 107, 279, 135, 1109, 3779, 5305, 15333, 12217, 41257, 20265}},
	{8404, 17, 22812, {1, 3, 7, 1, 31, 59, 83, 43, 219, 119, 511, 2973, 4587, 10701, 30959, 21489, 124077}},
	{8405, 17, 22828, {1, 1, 7, 9, 17, 3, 59, 151, 281, 209, 1405, 173, 3589, 7679, 29803, 53947, 68291}},
	{8406, 17, 22840, {1, 1, 7, 7, 5, 19, 53, 91, 1, 513, 1495, 231, 3627, 1115, 16121, 12953, 108343}},
	{8407, 17, 22845, {1, 3, 1, 13, 17, 3, 35, 35, 211, 481, 2029, 1035, 3131, 5653, 18097, 10735, 102453}},
	{8408, 17, 22848, {1, 3, 1, 11, 29, 7, 121, 135, 51, 837, 681, 1497, 7435, 2215, 26527, 33029, 93241}},
	{8409, 17, 22857, {1, 3, 3, 15, 29, 43, 17, 243, 195, 315, 499, 3801, 5691, 12119, 4061, 51769, 80497}},
	{8410, 17, 22877, {1, 1, 3, 1, 11, 1, 113, 11, 387, 579, 275, 2995, 895, 11859, 4017, 1543, 11853}},
	{8411, 17, 22882, {1, 1, 7, 9, 31, 27, 63, 217, 97, 275, 435, 1355, 5205, 6587, 32589, 46485, 103587}},
	{8412, 17, 22887, {1, 3, 7, 3, 7, 19, 51, 41, 81, 261, 1909, 1475, 425, 3173, 5679, 34701, 34977}},
	{8413, 17, 22894, {1, 1, 7, 3, 27, 15, 35, 49, 387, 471, 1997, 3643, 2701, 11853, 21311, 36027, 104357}},
	{8414, 17, 22912, {1, 3, 1, 9, 5, 47, 73, 163, 309, 891, 229, 2433, 6715, 6721, 25233, 37043, 29367}},
	{8415, 17, 22930, {1, 1, 1, 7, 27, 15, 9, 185, 421, 597, 565, 143, 1531, 15585, 17057, 54309, 82915}},
	{8416, 17, 22936, {1, 1, 7, 1, 5, 43, 87, 61, 121, 341, 25, 3795, 7161, 11985, 32197, 789, 69543}},
	{8417, 17, 22939, {1, 3, 5, 13, 29, 39, 81, 39, 263, 729, 1833, 365, 1073, 9869, 1845, 52621, 5}},
	{8418, 17, 22957, {1, 1, 7, 7, 5, 33, 117, 11, 371, 161, 1303, 629, 2285, 5827, 32355, 43359, 115595}},
	{8419, 17, 22970, {1, 3, 5, 5, 13, 57, 63, 9, 243, 533, 173, 2197, 717, 13441, 22131, 17783, 3319}},
	{8420, 17, 22980, {1, 1, 7, 11, 15, 31, 87, 255, 183, 273, 805, 2347, 5881, 15401, 273, 17397, 41827}},
	{8421, 17, 22984, {1, 3, 1, 13, 7, 17, 121, 49, 47, 121, 333, 3629, 5337, 4117, 2735, 36581, 61345}},
	{8422, 17, 22992, {1, 3, 3, 11, 9, 7, 25, 223, 379, 119, 385, 1217, 4803, 2947, 30665, 7733, 77893}},
	{8423, 17, 22998, {1, 3, 3, 7, 31, 35, 127, 97, 5, 373, 7, 3035, 843, 5991, 9265, 34289, 42785}},
	{8424, 17, 23001, {1, 3, 7, 3, 27, 19, 95, 253, 349, 871, 807, 413, 5847, 10467, 4277, 12429, 75773}},
	{8425, 17, 23044, {1, 3, 3, 7, 21, 1, 79, 89, 219, 505, 41, 505, 5159, 12839, 3317, 49873, 73705}},
	{8426, 17, 23061, {1, 3, 1, 7, 21, 43, 121, 113, 477, 559, 1831, 3759, 3315, 6367, 7149, 16395, 44703}},
	{8427, 17, 23062, {1, 1, 1, 7, 13, 53, 35, 53, 489, 975, 631, 863, 3067, 1905, 21351, 14705, 80041}},
	{8428, 17, 23071, {1, 1, 1, 5, 13, 27, 121, 65, 351, 123, 1731, 367, 8061, 5229, 8537, 20897, 130373}},
	{8429, 17, 23075, {1, 1, 5, 11, 15, 63, 101, 107, 105, 619, 1771, 3549, 7191, 9083, 16827, 29639, 34219}},
	{8430, 17, 23089, {1, 3, 1, 9, 15, 13, 87, 157, 379, 433, 217, 815, 5079, 1797, 26707, 35165, 92305}},
	{8431, 17, 23090, {1, 1, 5, 13, 27, 35, 31, 65, 313, 629, 375, 1391, 5373, 3497, 7311, 23105, 45293}},
	{8432, 17, 23096, {1, 3, 1, 3, 5, 39, 91, 37, 401, 419, 949, 2431, 3685, 6671, 20789, 8597, 44215}},
	{8433, 17, 23101, {1, 1, 7, 11, 7, 15, 3, 181, 363, 913, 309, 2009, 3805, 6651, 27677, 37711, 40813}},
	{8434, 17, 23114, {1, 3, 5, 5, 17, 11, 47, 9, 27, 459, 773, 1403, 7069, 12651, 8163, 42425, 126697}},
	{8435, 17, 23121, {1, 3, 1, 3, 11, 21, 65, 103, 405, 843, 59, 3653, 1759, 5265, 401, 58019, 124999}},
	{8436, 17, 23124, {1, 1, 3, 7, 11, 25, 61, 211, 199, 849, 1835, 1181, 5003, 3873, 23743, 45451, 54901}},
	{8437, 17, 23127, {1, 3, 5, 3, 29, 25, 43, 199, 481, 991, 699, 3937, 7601, 1253, 24399, 6625, 93917}},
	{8438, 17, 23128, {1, 1, 7, 3, 29, 33, 33, 151, 3, 825, 743, 773, 7825, 8157, 22121, 50095, 16435}},
	{8439, 17, 23137, {1, 3, 1, 1, 27, 15, 81, 151, 271, 167, 1755, 1289, 7473, 8525, 12525, 63139, 48787}},
	{8440, 17, 23138, {1, 1, 7, 13, 27, 33, 87, 125, 211, 631, 149, 3451, 643, 6975, 2659, 12629, 33187}},
	{8441, 17, 23150, {1, 1, 3, 3, 5, 49, 99, 99, 85, 647, 351, 2829, 7005, 7283, 5857, 46157, 52061}},
	{8442, 17, 23155, {1, 1, 3, 5, 11, 37, 43, 129, 21, 639, 187, 2279, 8189, 12877, 28707, 7133, 93639}},
	{8443, 17, 23168, {1, 1, 3, 7, 19, 13, 35, 51, 77, 811, 1553, 2769, 763, 4965, 4643, 37639, 44229}},
	{8444, 17, 23173, {1, 3, 5, 15, 11, 29, 103, 203, 435, 1017, 531, 1453, 1407, 6569, 619, 52103, 45213}},
	{8445, 17, 23174, {1, 1, 7, 5, 25, 25, 47, 229, 201, 843, 473, 2637, 2265, 4627, 20013, 41217, 76095}},
	{8446, 17, 23195, {1, 3, 3, 15, 23, 61, 109, 31, 57, 595, 1303, 3915, 67, 8205, 3553, 9543, 103385}},
	{8447, 17, 23202, {1, 1, 3, 3, 21, 19, 21, 41, 137, 905, 2045, 491, 1783, 151, 20963, 38009, 735}},
	{8448, 17, 23225, {1, 1, 7, 11, 13, 33, 95, 251, 179, 211, 1687, 3189, 6213, 3905, 2117, 15153, 4855}},
	{8449, 17, 23226, {1, 1, 5, 3, 19, 9, 67, 243, 23, 611, 1007, 1317, 7303, 11065, 21157, 56677, 81683}},
	{8450, 17, 23239, {1, 1, 3, 5, 19, 41, 63, 129, 233, 15, 37, 1445, 1095, 11309, 30181, 49199, 85113}},
	{8451, 17, 23253, {1, 3, 7, 1, 21, 53, 83, 79, 155, 379, 773, 1823, 1003, 2787, 31107, 36115, 40987}},
	{8452, 17, 23263, {1, 3, 3, 5, 3, 19, 7, 247, 417, 573, 407, 3577, 6079, 10275, 29791, 35149, 102565}},
	{8453, 17, 23264, {1, 3, 3, 9, 21, 49, 57, 223, 125, 671, 655, 2995, 5849, 5355, 21171, 54857, 114841}},
	{8454, 17, 23281, {1, 3, 7, 3, 27, 23, 125, 103, 485, 955, 963, 1865, 2321, 2263, 32497, 47973, 122111}},
	{8455, 17, 23282, {1, 1, 3, 15, 3, 1, 37, 19, 287, 165, 1717, 851, 3619, 13623, 24295, 48253, 13143}},
	{8456, 17, 23288, {1, 1, 7, 9, 13, 59, 69, 97, 113, 163, 871, 1795, 2719, 13675, 11767, 23687, 65841}},
	{8457, 17, 23294, {1, 1, 5, 3, 21, 31, 41, 115, 469, 177, 137, 2129, 1385, 10835, 16471, 59411, 30795}},
	{8458, 17, 23302, {1, 1, 7, 7, 13, 45, 73, 119, 457, 673, 1481, 3735, 2675, 11413, 9069, 34741, 8757}},
	{8459, 17, 23311, {1, 3, 5, 3, 15, 9, 11, 191, 499, 51, 1963, 3957, 1341, 7129, 13491, 65369, 4339}},
	{8460, 17, 23320, {1, 3, 7, 1, 5, 45, 103, 209, 183, 205, 525, 2417, 847, 10801, 10699, 16723, 36421}},
	{8461, 17, 23325, {1, 3, 7, 13, 3, 57, 37, 75, 299, 359, 2017, 125, 6737, 4859, 18443, 20765, 40319}},
	{8462, 17, 23356, {1, 1, 3, 5, 5, 17, 43, 141, 31, 141, 1019, 1685, 6831, 9433, 31245, 29227, 64083}},
	{8463, 17, 23374, {1, 3, 1, 13, 25, 47, 107, 69, 459, 595, 1759, 3391, 1531, 15197, 25975, 16971, 70861}},
	{8464, 17, 23388, {1, 1, 3, 11, 3, 53, 63, 211, 69, 469, 1407, 1435, 2763, 917, 19943, 16591, 97101}},
	{8465, 17, 23402, {1, 3, 5, 13, 25, 41, 39, 61, 319, 809, 1109, 169, 3101, 8801, 21697, 50759, 130985}},
	{8466, 17, 23415, {1, 3, 1, 9, 23, 1, 11, 249, 243, 605, 1419, 269, 1601, 2063, 5365, 38077, 106161}},
	{8467, 17, 23421, {1, 1, 7, 7, 19, 55, 97, 155, 477, 845, 61, 263, 1337, 8857, 31611, 44417, 43111}},
	{8468, 17, 23426, {1, 3, 3, 15, 7, 63, 45, 239, 291, 279, 1875, 3769, 1571, 15857, 13335, 17209, 34399}},
	{8469, 17, 23443, {1, 1, 7, 11, 19, 19, 69, 111, 217, 927, 1643, 1077, 4763, 15893, 17491, 39737, 10705}},
	{8470, 17, 23446, {1, 1, 5, 11, 3, 3, 31, 199, 109, 403, 973, 3833, 2729, 7285, 26743, 53915, 96203}},
	{8471, 17, 23455, {1, 3, 3, 11, 9, 7, 19, 145, 495, 805, 381, 919, 1323, 4343, 15887, 5163, 68267}},
	{8472, 17, 23461, {1, 1, 3, 11, 15, 31, 27, 201, 251, 279, 1377, 1313, 7143, 9731, 10451, 63431, 31307}},
	{8473, 17, 23468, {1, 1, 7, 1, 1, 55, 35, 249, 133, 645, 425, 279, 6401, 11687, 751, 947, 21791}},
	{8474, 17, 23471, {1, 3, 5, 9, 5, 43, 89, 31, 419, 573, 1087, 2197, 3451, 2393, 6569, 4859, 36607}},
	{8475, 17, 23485, {1, 3, 5, 15, 25, 51, 11, 149, 483, 789, 661, 967, 3537, 15511, 26587, 29861, 120337}},
	{8476, 17, 23486, {1, 3, 5, 13, 21, 39, 75, 111, 57, 321, 1591, 381, 7399, 10807, 26651, 62489, 78341}},
	{8477, 17, 23488, {1, 3, 1, 13, 1, 1, 49, 137, 193, 967, 805, 221, 803, 11381, 27803, 51013, 10475}},
	{8478, 17, 23498, {1, 3, 7, 5, 3, 13, 47, 195, 123, 753, 397, 1203, 981, 12863, 20845, 36155, 19055}},
	{8479, 17, 23500, {1, 1, 1, 9, 9, 11, 53, 203, 3, 163, 1537, 2061, 941, 12629, 16053, 34881, 31489}},
	{8480, 17, 23515, {1, 1, 1, 15, 5, 23, 51, 197, 459, 21, 1989, 2529, 4267, 1505, 8951, 15777, 20493}},
	{8481, 17, 23521, {1, 1, 7, 3, 31, 55, 9, 55, 217, 695, 1563, 4077, 3207, 7029, 10881, 39581, 82511}},
	{8482, 17, 23527, {1, 3, 1, 5, 1, 11, 81, 1, 505, 631, 1093, 3655, 2085, 7349, 5009, 49381, 30527}},
	{8483, 17, 23534, {1, 1, 7, 1, 27, 51, 25, 235, 213, 59, 611, 3883, 2909, 6411, 19605, 49001, 114529}},
	{8484, 17, 23546, {1, 3, 5, 3, 25, 29, 19, 137, 199, 681, 1625, 2711, 4873, 14677, 9767, 30441, 54673}},
	{8485, 17, 23559, {1, 1, 1, 9, 27, 43, 109, 161, 139, 675, 741, 2839, 1425, 5701, 19897, 12787, 33069}},
	{8486, 17, 23560, {1, 3, 5, 11, 21, 19, 77, 107, 197, 591, 1899, 1311, 3347, 6369, 26891, 3771, 32455}},
	{8487, 17, 23566, {1, 1, 7, 15, 31, 13, 109, 69, 207, 349, 249, 971, 7891, 10919, 31579, 38453, 124601}},
	{8488, 17, 23584, {1, 3, 5, 5, 27, 61, 67, 193, 53, 259, 1729, 4033, 2637, 8217, 22351, 4001, 118527}},
	{8489, 17, 23587, {1, 1, 3, 5, 9, 45, 55, 73, 189, 131, 1947, 1889, 837, 4085, 10393, 64359, 1037}},
	{8490, 17, 23594, {1, 3, 7, 3, 13, 51, 55, 37, 335, 939, 35, 461, 5057, 2595, 3305, 58823, 3941}},
	{8491, 17, 23602, {1, 1, 7, 11, 7, 3, 121, 139, 241, 477, 615, 2707, 5391, 7611, 11563, 41083, 16719}},
	{8492, 17, 23607, {1, 3, 3, 15, 27, 55, 13, 221, 195, 543, 215, 4035, 1647, 8111, 26425, 43571, 79893}},
	{8493, 17, 23616, {1, 1, 1, 5, 31, 5, 35, 145, 481, 339, 1951, 2155, 1309, 9851, 31505, 37371, 21247}},
	{8494, 17, 23621, {1, 1, 7, 9, 7, 5, 73, 119, 3, 741, 1351, 2855, 2207, 1465, 12047, 13507, 129173}},
	{8495, 17, 23631, {1, 1, 7, 13, 5, 57, 119, 63, 367, 327, 1257, 3191, 6929, 9593, 16565, 54397, 100305}},
	{8496, 17, 23634, {1, 3, 1, 11, 9, 1, 85, 53, 65, 945, 17, 1963, 4819, 16173, 11669, 53579, 33701}},
	{8497, 17, 23636, {1, 1, 3, 15, 25, 27, 3, 25, 23, 429, 197, 2717, 6107, 6719, 12457, 31793, 78647}},
	{8498, 17, 23649, {1, 1, 3, 1, 7, 63, 111, 235, 299, 91, 369, 1423, 7083, 4229, 18535, 33793, 19943}},
	{8499, 17, 23652, {1, 1, 7, 13, 9, 11, 123, 9, 169, 895, 1989, 1047, 6139, 11773, 19381, 9593, 14809}},
	{8500, 17, 23679, {1, 3, 1, 3, 29, 31, 63, 91, 59, 391, 1695, 2459, 3301, 5615, 3425, 8029, 16069}},
	{8501, 17, 23686, {1, 1, 7, 1, 25, 25, 79, 49, 131, 695, 987, 2911, 1109, 8237, 18227, 37287, 22443}},
	{8502, 17, 23697, {1, 3, 3, 3, 25, 21, 33, 207, 187, 381, 129, 445, 2967, 5119, 18777, 14849, 97115}},
	{8503, 17, 23703, {1, 1, 7, 13, 19, 9, 93, 185, 391, 579, 1509, 3245, 3921, 9473, 4795, 6685, 49549}},
	{8504, 17, 23714, {1, 1, 5, 11, 1, 49, 57, 127, 363, 811, 1383, 2869, 7625, 15177, 2581, 64253, 53677}},
	{8505, 17, 23719, {1, 1, 7, 3, 7, 27, 73, 187, 31, 1011, 1013, 3269, 6625, 5001, 20805, 13331, 93725}},
	{8506, 17, 23723, {1, 3, 7, 1, 23, 61, 123, 9, 141, 113, 1009, 3713, 4947, 9929, 24125, 1101, 104249}},
	{8507, 17, 23726, {1, 3, 7, 3, 23, 17, 25, 187, 189, 875, 1435, 163, 4197, 6619, 29031, 23117, 45347}},
	{8508, 17, 23728, {1, 1, 5, 7, 11, 17, 9, 55, 117, 223, 417, 3993, 1843, 5817, 20435, 56705, 98337}},
	{8509, 17, 23733, {1, 1, 7, 3, 21, 59, 3, 77, 297, 61, 407, 1603, 3209, 1611, 30185, 50275, 56139}},
	{8510, 17, 23740, {1, 1, 1, 5, 31, 3, 101, 167, 367, 543, 339, 1885, 7855, 9989, 30969, 6735, 108123}},
	{8511, 17, 23751, {1, 1, 3, 9, 27, 63, 9, 79, 335, 351, 673, 3107, 3955, 1799, 16879, 57631, 109073}},
	{8512, 17, 23755, {1, 1, 1, 3, 27, 17, 107, 115, 155, 371, 379, 2837, 6213, 2663, 1101, 451, 69517}},
	{8513, 17, 23765, {1, 1, 3, 9, 13, 3, 55, 9, 449, 43, 1011, 3281, 5311, 223, 10715, 6639, 79949}},
	{8514, 17, 23766, {1, 3, 3, 11, 23, 9, 43, 185, 271, 1005, 1041, 2633, 377, 4247, 10417, 51903, 19239}},
	{8515, 17, 23769, {1, 3, 1, 9, 15, 39, 115, 233, 33, 425, 1979, 583, 1901, 8943, 1527, 56065, 50159}},
	{8516, 17, 23779, {1, 1, 3, 1, 13, 1, 105, 149, 13, 625, 671, 1811, 3701, 241, 27357, 25835, 127265}},
	{8517, 17, 23794, {1, 3, 1, 9, 11, 23, 107, 197, 21, 589, 1065, 2591, 1163, 15013, 8931, 6355, 87079}},
	{8518, 17, 23796, {1, 3, 5, 3, 17, 5, 121, 61, 99, 987, 2033, 2237, 2299, 14689, 19785, 9599, 101035}},
	{8519, 17, 23803, {1, 1, 1, 1, 17, 25, 5, 97, 55, 75, 1419, 2793, 7215, 3185, 7029, 23023, 89089}},
	{8520, 17, 23813, {1, 3, 3, 3, 11, 57, 103, 191, 405, 463, 1421, 253, 6069, 10905, 18193, 719, 17337}},
	{8521, 17, 23820, {1, 3, 5, 11, 23, 37, 39, 169, 295, 527, 1671, 3913, 6057, 689, 27719, 47245, 95895}},
	{8522, 17, 23841, {1, 3, 7, 5, 13, 9, 43, 189, 411, 155, 559, 3701, 1623, 2401, 10359, 22675, 41897}},
	{8523, 17, 23853, {1, 1, 1, 11, 17, 55, 47, 101, 357, 669, 857, 2745, 6425, 11839, 13095, 10757, 52383}},
	{8524, 17, 23861, {1, 1, 7, 5, 11, 13, 53, 151, 93, 455, 133, 3353, 1417, 7917, 12913, 2615, 34281}},
	{8525, 17, 23862, {1, 1, 3, 5, 29, 57, 43, 35, 203, 423, 311, 3133, 1757, 1291, 2019, 3115, 126939}},
	{8526, 17, 23873, {1, 1, 3, 11, 9, 43, 119, 95, 135, 351, 1865, 2821, 717, 6275, 19713, 42315, 97935}},
	{8527, 17, 23876, {1, 3, 7, 7, 31, 51, 7, 29, 405, 31, 1765, 3231, 1315, 1307, 26469, 62033, 35619}},
	{8528, 17, 23897, {1, 1, 5, 7, 5, 17, 49, 137, 501, 631, 1401, 2851, 6971, 14721, 4329, 26483, 120007}},
	{8529, 17, 23898, {1, 1, 5, 13, 21, 19, 95, 93, 125, 331, 1797, 1653, 1891, 11081, 30989, 24671, 95421}},
	{8530, 17, 23903, {1, 3, 3, 11, 13, 29, 61, 157, 165, 39, 661, 89, 637, 1397, 12561, 62399, 129107}},
	{8531, 17, 23904, {1, 3, 1, 15, 13, 19, 5, 115, 345, 903, 531, 4069, 6775, 7433, 569, 21779, 13271}},
	{8532, 17, 23910, {1, 3, 3, 9, 5, 53, 17, 115, 67, 939, 1907, 3979, 4311, 3573, 857, 34931, 112397}},
	{8533, 17, 23931, {1, 3, 7, 11, 9, 47, 83, 85, 277, 219, 1701, 3013, 3037, 3473, 3797, 40713, 118573}},
	{8534, 17, 23933, {1, 1, 3, 13, 25, 33, 117, 115, 179, 119, 487, 3213, 2873, 17, 20865, 20043, 64381}},
	{8535, 17, 23934, {1, 1, 1, 3, 1, 45, 73, 103, 75, 579, 981, 2449, 2141, 8697, 22995, 59693, 104461}},
	{8536, 17, 23943, {1, 3, 1, 1, 29, 9, 9, 201, 55, 389, 1069, 2057, 4149, 9217, 10753, 7889, 95849}},
	{8537, 17, 23952, {1, 3, 7, 9, 27, 39, 19, 223, 7, 253, 55, 503, 3339, 6049, 32603, 34807, 115403}},
	{8538, 17, 23955, {1, 1, 5, 3, 13, 21, 67, 87, 205, 309, 1371, 1579, 281, 16135, 28403, 25951, 24109}},
	{8539, 17, 23962, {1, 3, 1, 3, 17, 21, 49, 77, 393, 943, 1701, 2661, 5173, 12875, 2731, 40531, 19301}},
	{8540, 17, 23971, {1, 3, 1, 5, 23, 29, 61, 161, 373, 389, 1699, 359, 2513, 4717, 30397, 24395, 20881}},
	{8541, 17, 23978, {1, 3, 5, 5, 29, 3, 115, 251, 277, 487, 7, 3301, 7945, 14233, 20497, 62035, 21537}},
	{8542, 17, 23998, {1, 1, 1, 9, 7, 59, 23, 85, 367, 109, 1761, 4011, 6535, 8263, 2081, 63647, 69807}},
	{8543, 17, 24003, {1, 1, 7, 11, 21, 41, 29, 219, 271, 617, 929, 407, 2899, 14299, 7645, 44815, 58817}},
	{8544, 17, 24009, {1, 3, 5, 7, 11, 29, 119, 33, 261, 571, 2013, 3327, 2181, 12767, 93, 2437, 76533}},
	{8545, 17, 24017, {1, 1, 7, 13, 17, 39, 55, 203, 261, 917, 967, 3651, 7235, 13751, 14439, 7591, 96553}},
	{8546, 17, 24045, {1, 1, 1, 1, 11, 39, 19, 21, 125, 93, 1773, 1155, 6213, 7173, 9057, 6219, 4643}},
	{8547, 17, 24046, {1, 3, 1, 5, 1, 31, 55, 143, 425, 539, 61, 3377, 7647, 257, 15007, 24511, 8707}},
	{8548, 17, 24060, {1, 3, 3, 11, 27, 51, 103, 197, 427, 139, 181, 1169, 3123, 11803, 5285, 1321, 62267}},
	{8549, 17, 24064, {1, 3, 5, 9, 11, 3, 13, 149, 197, 37, 31, 927, 3313, 16149, 14209, 60177, 46525}},
	{8550, 17, 24076, {1, 1, 5, 13, 15, 29, 103, 49, 355, 797, 1253, 1833, 621, 3877, 9981, 49207, 91035}},
	{8551, 17, 24079, {1, 1, 3, 3, 13, 19, 27, 51, 151, 275, 35, 3755, 7511, 14197, 26141, 43765, 104327}},
	{8552, 17, 24087, {1, 3, 5, 15, 23, 47, 101, 213, 97, 957, 831, 1533, 7913, 15763, 29717, 60425, 38559}},
	{8553, 17, 24094, {1, 1, 7, 9, 29, 31, 49, 245, 361, 299, 151, 2969, 1487, 1761, 11697, 4043, 100909}},
	{8554, 17, 24100, {1, 1, 1, 3, 17, 49, 99, 159, 3, 525, 1527, 3435, 5113, 459, 13341, 54103, 85813}},
	{8555, 17, 24118, {1, 3, 1, 1, 5, 59, 35, 75, 107, 91, 1621, 3261, 619, 3271, 10813, 29857, 1547}},
	{8556, 17, 24121, {1, 1, 5, 9, 9, 33, 85, 245, 39, 879, 1621, 2587, 3825, 12939, 30113, 24611, 68491}},
	{8557, 17, 24132, {1, 3, 1, 3, 9, 39, 93, 241, 307, 237, 3, 1763, 7729, 9257, 31911, 32591, 77333}},
	{8558, 17, 24147, {1, 3, 1, 3, 27, 7, 51, 121, 317, 361, 1027, 95, 7035, 3097, 21007, 38311, 88287}},
	{8559, 17, 24165, {1, 3, 7, 3, 19, 3, 111, 115, 339, 793, 1571, 3101, 1911, 14929, 12841, 45871, 119905}},
	{8560, 17, 24172, {1, 1, 5, 7, 31, 61, 37, 143, 279, 941, 1215, 2411, 7617, 1657, 10189, 19139, 6307}},
	{8561, 17, 24177, {1, 1, 3, 9, 21, 35, 13, 157, 187, 79, 689, 1085, 37, 4549, 5901, 15321, 61627}},
	{8562, 17, 24184, {1, 3, 1, 13, 15, 39, 21, 231, 39, 327, 801, 2321, 587, 1877, 3489, 54467, 95773}},
	{8563, 17, 24187, {1, 1, 5, 7, 1, 9, 53, 1, 243, 365, 789, 3833, 317, 10697, 26567, 65187, 22507}},
	{8564, 17, 24213, {1, 3, 3, 7, 9, 41, 31, 135, 425, 939, 15, 2043, 6593, 7651, 25467, 62549, 35847}},
	{8565, 17, 24217, {1, 1, 1, 7, 15, 23, 19, 57, 421, 25, 1037, 3055, 6173, 12451, 485, 54567, 109561}},
	{8566, 17, 24223, {1, 3, 5, 1, 3, 29, 67, 233, 157, 677, 1711, 513, 4673, 2895, 1983, 31075, 1861}},
	{8567, 17, 24230, {1, 1, 1, 1, 7, 39, 115, 251, 275, 791, 15, 1685, 6835, 14685, 12607, 28213, 121475}},
	{8568, 17, 24234, {1, 1, 5, 5, 13, 11, 103, 93, 489, 709, 1339, 2407, 1663, 10195, 3135, 15531, 88427}},
	{8569, 17, 24241, {1, 1, 7, 7, 17, 1, 123, 143, 31, 721, 1739, 2273, 3785, 10261, 14741, 52573, 113677}},
	{8570, 17, 24248, {1, 3, 7, 3, 9, 21, 77, 13, 241, 429, 165, 3399, 7543, 2633, 21129, 13537, 84473}},
	{8571, 17, 24259, {1, 1, 1, 11, 21, 33, 125, 123, 189, 337, 163, 3727, 2101, 14113, 1719, 46017, 68601}},
	{8572, 17, 24262, {1, 1, 7, 9, 9, 53, 101, 111, 125, 605, 1419, 3901, 1769, 4585, 20063, 20857, 21901}},
	{8573, 17, 24271, {1, 3, 7, 11, 1, 19, 51, 7, 457, 119, 871, 3847, 57, 11437, 28763, 58831, 675}},
	{8574, 17, 24279, {1, 3, 1, 15, 25, 63, 69, 25, 405, 823, 1701, 2441, 7561, 8679, 31643, 29325, 25563}},
	{8575, 17, 24286, {1, 1, 3, 9, 15, 5, 89, 13, 73, 951, 959, 2693, 4565, 13095, 991, 12419, 8267}},
	{8576, 17, 24289, {1, 1, 7, 1, 15, 1, 119, 223, 213, 585, 1047, 2623, 4141, 2043, 1583, 59155, 5133}},
	{8577, 17, 24295, {1, 3, 3, 3, 17, 37, 81, 233, 87, 843, 1597, 1251, 4713, 10813, 24357, 48499, 84465}},
	{8578, 17, 24296, {1, 1, 1, 1, 11, 55, 125, 5, 255, 809, 543, 2351, 7079, 7801, 29247, 23937, 97405}},
	{8579, 17, 24299, {1, 3, 3, 5, 17, 55, 87, 245, 371, 679, 943, 655, 5857, 261, 28229, 22519, 35191}},
	{8580, 17, 24314, {1, 1, 7, 15, 9, 49, 25, 155, 13, 893, 1303, 2317, 2903, 15601, 1433, 20397, 70125}},
	{8581, 17, 24336, {1, 3, 5, 3, 11, 47, 99, 63, 253, 95, 1023, 397, 4307, 4771, 17027, 19833, 18269}},
	{8582, 17, 24342, {1, 3, 3, 7, 25, 17, 69, 119, 475, 575, 1637, 3785, 649, 11557, 22457, 38633, 96153}},
	{8583, 17, 24346, {1, 1, 1, 5, 31, 55, 85, 83, 307, 201, 1543, 727, 977, 15779, 21907, 31025, 38969}},
	{8584, 17, 24357, {1, 3, 5, 1, 7, 53, 107, 239, 341, 237, 1567, 2717, 3197, 12419, 23733, 42119, 86619}},
	{8585, 17, 24367, {1, 1, 5, 13, 3, 7, 105, 95, 201, 953, 781, 2043, 5263, 13427, 10303, 60027, 19297}},
	{8586, 17, 24370, {1, 1, 5, 15, 25, 51, 5, 77, 165, 297, 1281, 1635, 4139, 11569, 32325, 23135, 27013}},
	{8587, 17, 24372, {1, 1, 3, 9, 3, 59, 107, 137, 251, 715, 1477, 511, 5629, 12205, 7541, 62559, 4253}},
	{8588, 17, 24387, {1, 1, 7, 11, 31, 29, 7, 251, 119, 547, 1179, 3063, 1625, 8941, 30515, 13601, 72741}},
	{8589, 17, 24389, {1, 3, 7, 13, 27, 43, 31, 43, 465, 355, 1063, 2305, 1425, 11963, 27327, 53335, 127517}},
	{8590, 17, 24402, {1, 3, 1, 3, 21, 17, 53, 171, 269, 783, 349, 1879, 575, 13537, 16931, 61171, 23499}},
	{8591, 17, 24414, {1, 3, 5, 3, 11, 5, 121, 227, 237, 841, 431, 3209, 3241, 6071, 23465, 39533, 102391}},
	{8592, 17, 24420, {1, 3, 5, 11, 9, 1, 59, 143, 181, 869, 1859, 1543, 6419, 13305, 29075, 28051, 105799}},
	{8593, 17, 24435, {1, 1, 7, 13, 31, 1, 105, 169, 67, 693, 1667, 2181, 4127, 4605, 3701, 36467, 19631}},
	{8594, 17, 24437, {1, 1, 7, 5, 31, 15, 119, 161, 55, 549, 1195, 4051, 1923, 2497, 8289, 60393, 96181}},
	{8595, 17, 24442, {1, 1, 3, 3, 5, 43, 13, 123, 469, 603, 2047, 2347, 815, 3457, 7503, 25261, 71951}},
	{8596, 17, 24444, {1, 1, 7, 3, 13, 25, 85, 141, 497, 405, 957, 1407, 2075, 12445, 6675, 9993, 40429}},
	{8597, 17, 24447, {1, 1, 5, 13, 29, 43, 99, 113, 307, 1003, 859, 723, 7513, 12249, 12653, 57685, 89551}},
	{8598, 17, 24468, {1, 3, 7, 3, 11, 3, 9, 141, 501, 113, 69, 2285, 4525, 9049, 24765, 11585, 53787}},
	{8599, 17, 24475, {1, 1, 3, 1, 25, 41, 103, 159, 215, 871, 77, 1849, 609, 15877, 32515, 22931, 11933}},
	{8600, 17, 24484, {1, 1, 5, 11, 3, 27, 27, 111, 479, 861, 1041, 3777, 4443, 3095, 30379, 6293, 30823}},
	{8601, 17, 24493, {1, 3, 5, 5, 27, 45, 9, 25, 451, 845, 1153, 897, 325, 15679, 30151, 37695, 54593}},
	{8602, 17, 24494, {1, 3, 7, 1, 15, 47, 87, 135, 87, 567, 221, 3173, 769, 8173, 2957, 51287, 20961}},
	{8603, 17, 24496, {1, 3, 1, 9, 3, 33, 1, 71, 147, 983, 1485, 3531, 213, 2353, 28269, 49353, 88343}},
	{8604, 17, 24514, {1, 1, 3, 11, 11, 63, 109, 255, 35, 127, 1777, 791, 1379, 9539, 4915, 21593, 98901}},
	{8605, 17, 24519, {1, 1, 7, 5, 3, 47, 93, 219, 381, 963, 359, 2461, 7629, 2803, 17345, 54311, 79057}},
	{8606, 17, 24533, {1, 3, 5, 13, 13, 21, 1, 65, 455, 203, 29, 3717, 4495, 1285, 25289, 38597, 42431}},
	{8607, 17, 24538, {1, 1, 3, 3, 27, 57, 7, 171, 65, 469, 1921, 3855, 1637, 5517, 14907, 48239, 117573}},
	{8608, 17, 24559, {1, 3, 5, 1, 11, 35, 105, 251, 19, 219, 1191, 2177, 7885, 8399, 30527, 61415, 122215}},
	{8609, 17, 24568, {1, 3, 5, 5, 21, 25, 59, 193, 509, 147, 1805, 561, 3505, 9639, 14221, 31, 99261}},
	{8610, 17, 24573, {1, 1, 5, 13, 31, 23, 35, 143, 367, 385, 1335, 2497, 3573, 8113, 16661, 16147, 8763}},
	{8611, 17, 24577, {1, 1, 7, 13, 15, 27, 35, 15, 7, 539, 633, 1145, 2267, 11527, 20975, 16689, 58227}},
	{8612, 17, 24587, {1, 1, 1, 15, 9, 11, 51, 121, 381, 331, 1445, 187, 519, 15827, 27611, 32891, 113671}},
	{8613, 17, 24592, {1, 3, 1, 5, 19, 3, 77, 67, 107, 105, 1025, 3229, 6869, 5717, 4227, 28489, 59759}},
	{8614, 17, 24601, {1, 1, 5, 15, 25, 23, 7, 25, 103, 733, 525, 453, 6467, 2901, 7197, 33267, 68177}},
	{8615, 17, 24602, {1, 1, 5, 7, 27, 27, 41, 93, 449, 733, 571, 411, 1709, 9557, 549, 5925, 24123}},
	{8616, 17, 24608, {1, 1, 7, 5, 31, 57, 119, 227, 105, 533, 717, 3357, 2495, 6467, 7211, 38169, 44603}},
	{8617, 17, 24645, {1, 1, 5, 7, 29, 9, 125, 241, 471, 571, 1271, 2911, 8087, 5067, 31139, 39681, 28579}},
	{8618, 17, 24650, {1, 3, 5, 11, 25, 53, 109, 35, 183, 109, 1961, 1681, 7773, 6935, 28049, 37279, 96829}},
	{8619, 17, 24657, {1, 1, 1, 11, 1, 17, 47, 245, 231, 747, 1395, 1635, 5129, 3165, 627, 34463, 38967}},
	{8620, 17, 24664, {1, 3, 5, 1, 9, 41, 25, 215, 251, 525, 1399, 3405, 7399, 11041, 5599, 51167, 38697}},
	{8621, 17, 24670, {1, 3, 3, 13, 11, 15, 121, 95, 139, 611, 633, 3941, 2619, 15123, 28535, 64823, 17527}},
	{8622, 17, 24673, {1, 3, 7, 13, 21, 53, 65, 175, 81, 5, 699, 1525, 7397, 2465, 4479, 58225, 26387}},
	{8623, 17, 24676, {1, 1, 5, 7, 9, 31, 31, 149, 359, 613, 397, 153, 4861, 8195, 22969, 26003, 124423}},
	{8624, 17, 24680, {1, 3, 1, 13, 27, 17, 107, 27, 19, 13, 1481, 573, 7701, 6273, 30255, 16125, 11809}},
	{8625, 17, 24686, {1, 3, 1, 9, 15, 1, 45, 105, 287, 901, 667, 3197, 3493, 12259, 1511, 63361, 94257}},
	{8626, 17, 24700, {1, 3, 1, 3, 25, 53, 19, 87, 365, 585, 1569, 1731, 3747, 11985, 22673, 17767, 113779}},
	{8627, 17, 24704, {1, 3, 3, 9, 7, 21, 103, 201, 501, 149, 1939, 3111, 4739, 8389, 27127, 55889, 54487}},
	{8628, 17, 24714, {1, 3, 5, 7, 25, 53, 75, 57, 19, 505, 849, 2631, 6999, 11269, 24541, 17695, 97671}},
	{8629, 17, 24728, {1, 1, 7, 15, 5, 51, 123, 93, 445, 379, 1729, 2747, 5821, 10779, 29335, 57419, 109339}},
	{8630, 17, 24731, {1, 1, 7, 3, 7, 57, 117, 65, 297, 891, 487, 1535, 2361, 10457, 30759, 34571, 129949}},
	{8631, 17, 24733, {1, 3, 5, 5, 17, 51, 27, 103, 55, 925, 947, 1237, 1629, 12687, 14775, 49627, 100939}},
	{8632, 17, 24747, {1, 3, 3, 15, 1, 11, 75, 177, 399, 55, 1705, 1165, 7525, 8909, 13071, 60703, 11561}},
	{8633, 17, 24749, {1, 1, 1, 7, 13, 29, 23, 65, 279, 853, 637, 3947, 4099, 6465, 7061, 50417, 35015}},
	{8634, 17, 24750, {1, 1, 3, 3, 15, 11, 111, 169, 135, 279, 1941, 3035, 3027, 6813, 13363, 20387, 3257}},
	{8635, 17, 24764, {1, 3, 3, 11, 3, 5, 95, 181, 405, 313, 39, 1503, 2443, 3221, 17021, 23485, 43909}},
	{8636, 17, 24782, {1, 1, 1, 3, 17, 63, 27, 247, 441, 533, 449, 3845, 4021, 14269, 31477, 7013, 37473}},
	{8637, 17, 24784, {1, 1, 5, 13, 29, 39, 41, 95, 417, 21, 685, 609, 5787, 13145, 32677, 6121, 50919}},
	{8638, 17, 24793, {1, 1, 5, 3, 17, 5, 93, 143, 171, 681, 1143, 2875, 805, 15823, 29649, 63327, 12041}},
	{8639, 17, 24794, {1, 1, 1, 11, 3, 53, 123, 105, 59, 485, 1799, 2939, 657, 2485, 29563, 36221, 89095}},
	{8640, 17, 24810, {1, 1, 5, 5, 15, 13, 127, 87, 211, 579, 175, 793, 6895, 9051, 17681, 28831, 31783}},
	{8641, 17, 24817, {1, 3, 7, 5, 11, 37, 9, 219, 453, 697, 139, 335, 6411, 8495, 4203, 29065, 114837}},
	{8642, 17, 24820, {1, 1, 3, 5, 31, 25, 89, 215, 249, 271, 1731, 3133, 3947, 10227, 9679, 51303, 82833}},
	{8643, 17, 24823, {1, 3, 5, 1, 31, 15, 7, 131, 369, 757, 1963, 3223, 35, 13967, 31807, 5093, 113743}},
	{8644, 17, 24832, {1, 1, 7, 3, 15, 23, 21, 173, 295, 929, 1137, 3943, 1985, 13015, 8523, 59117, 127}},
	{8645, 17, 24855, {1, 3, 7, 1, 31, 1, 115, 229, 345, 859, 1757, 1835, 7491, 4545, 1483, 40149, 122321}},
	{8646, 17, 24859, {1, 1, 1, 3, 13, 5, 3, 133, 177, 47, 1515, 17, 5663, 3185, 2775, 31389, 37409}},
	{8647, 17, 24862, {1, 1, 3, 3, 31, 3, 43, 137, 185, 803, 709, 391, 3513, 8117, 32593, 46593, 61037}},
	{8648, 17, 24877, {1, 1, 1, 7, 29, 27, 13, 35, 61, 961, 777, 2725, 7379, 7053, 21781, 60285, 49221}},
	{8649, 17, 24890, {1, 3, 7, 15, 7, 7, 15, 123, 109, 97, 361, 791, 4773, 8111, 4319, 13981, 92505}},
	{8650, 17, 24900, {1, 1, 3, 11, 21, 33, 113, 221, 453, 981, 341, 4041, 5129, 5981, 11051, 17243, 19023}},
	{8651, 17, 24904, {1, 3, 1, 1, 19, 7, 75, 213, 467, 221, 1829, 1275, 5729, 6843, 23855, 44805, 89269}},
	{8652, 17, 24909, {1, 1, 3, 7, 5, 29, 39, 125, 147, 329, 1485, 2793, 2329, 14979, 18395, 37951, 58699}},
	{8653, 17, 24910, {1, 3, 1, 3, 11, 37, 117, 189, 103, 381, 39, 31, 5205, 5601, 17127, 49073, 121417}},
	{8654, 17, 24915, {1, 3, 3, 13, 23, 49, 57, 187, 441, 189, 349, 2559, 3313, 1321, 7731, 57309, 80195}},
	{8655, 17, 24922, {1, 3, 7, 1, 17, 9, 21, 15, 447, 333, 959, 3471, 5301, 8573, 9761, 23183, 57997}},
	{8656, 17, 24937, {1, 3, 1, 9, 19, 1, 101, 71, 325, 309, 85, 2097, 8003, 12249, 1887, 2097, 68375}},
	{8657, 17, 24945, {1, 1, 7, 7, 11, 39, 85, 241, 293, 205, 387, 1539, 6583, 1395, 8869, 48843, 49983}},
	{8658, 17, 24962, {1, 3, 7, 13, 11, 23, 83, 125, 55, 429, 169, 1893, 4657, 643, 3405, 9943, 128753}},
	{8659, 17, 24964, {1, 1, 3, 11, 19, 43, 13, 171, 495, 117, 437, 3697, 6723, 6199, 1859, 39637, 111499}},
	{8660, 17, 24991, {1, 1, 1, 5, 1, 31, 83, 199, 129, 941, 1637, 1997, 5011, 14957, 32427, 60797, 4989}},
	{8661, 17, 24992, {1, 3, 3, 3, 5, 61, 33, 225, 315, 157, 1709, 807, 7809, 11063, 319, 20901, 73599}},
	{8662, 17, 24995, {1, 1, 7, 3, 27, 3, 1, 173, 125, 769, 1203, 3357, 4899, 13115, 7081, 42459, 18525}},
	{8663, 17, 25001, {1, 1, 7, 9, 9, 27, 43, 115, 229, 867, 661, 1825, 2883, 4285, 22393, 65141, 24699}},
	{8664, 17, 25009, {1, 1, 3, 7, 5, 9, 93, 47, 33, 823, 309, 2977, 5791, 9177, 27645, 35683, 57455}},
	{8665, 17, 25019, {1, 1, 5, 7, 9, 53, 9, 77, 499, 1023, 917, 209, 7311, 249, 773, 18303, 41447}},
	{8666, 17, 25021, {1, 1, 3, 5, 7, 9, 33, 73, 325, 369, 1657, 2257, 2893, 13911, 10797, 21055, 103511}},
	{8667, 17, 25029, {1, 3, 1, 3, 21, 31, 125, 29, 149, 513, 979, 2271, 989, 9541, 4179, 13215, 71369}},
	{8668, 17, 25034, {1, 1, 7, 7, 19, 41, 39, 165, 59, 79, 137, 3479, 3389, 6635, 21467, 51073, 20765}},
	{8669, 17, 25036, {1, 3, 3, 5, 7, 13, 109, 53, 335, 627, 339, 3825, 287, 6077, 11319, 2377, 112693}},
	{8670, 17, 25039, {1, 3, 3, 1, 3, 57, 9, 47, 437, 717, 563, 1219, 6191, 9081, 21533, 2651, 17275}},
	{8671, 17, 25057, {1, 1, 1, 5, 21, 9, 109, 109, 339, 947, 1699, 1487, 6477, 12601, 12327, 39427, 80937}},
	{8672, 17, 25063, {1, 1, 7, 9, 1, 5, 55, 43, 95, 733, 1151, 3657, 2119, 11947, 21279, 21581, 22053}},
	{8673, 17, 25064, {1, 3, 5, 11, 7, 9, 97, 149, 55, 523, 1911, 1389, 5343, 5533, 15439, 65313, 73421}},
	{8674, 17, 25075, {1, 1, 3, 7, 19, 15, 119, 141, 57, 243, 423, 981, 1407, 12633, 20455, 53069, 98593}},
	{8675, 17, 25077, {1, 1, 3, 3, 15, 57, 71, 203, 15, 133, 601, 805, 2821, 11623, 147, 4333, 97681}},
	{8676, 17, 25084, {1, 1, 5, 7, 17, 61, 15, 251, 53, 699, 105, 1195, 3979, 41, 9077, 5145, 80057}},
	{8677, 17, 25088, {1, 1, 5, 15, 29, 33, 53, 19, 41, 471, 1143, 65, 5833, 8417, 17263, 35859, 45035}},
	{8678, 17, 25091, {1, 1, 1, 1, 15, 51, 73, 131, 181, 147, 1863, 3777, 1749, 10135, 11591, 12395, 85163}},
	{8679, 17, 25105, {1, 3, 1, 9, 23, 63, 83, 199, 87, 499, 2025, 863, 4665, 3941, 17647, 52463, 108615}},
	{8680, 17, 25134, {1, 3, 5, 7, 11, 39, 65, 161, 367, 593, 699, 1807, 7217, 5221, 22093, 44933, 6201}},
	{8681, 17, 25165, {1, 1, 7, 13, 9, 41, 35, 77, 353, 291, 1267, 3923, 5397, 15401, 30317, 14945, 8715}},
	{8682, 17, 25183, {1, 3, 1, 15, 11, 3, 29, 25, 505, 945, 1425, 2297, 1133, 4675, 8069, 55115, 114177}},
	{8683, 17, 25184, {1, 3, 1, 5, 27, 63, 25, 7, 5, 399, 473, 1325, 7391, 5953, 27755, 65407, 89435}},
	{8684, 17, 25202, {1, 3, 3, 13, 21, 61, 5, 119, 23, 999, 849, 1225, 3077, 821, 12059, 43223, 45427}},
	{8685, 17, 25204, {1, 3, 7, 13, 1, 5, 93, 173, 181, 453, 1449, 3823, 1713, 14737, 8891, 43643, 1983}},
	{8686, 17, 25211, {1, 1, 3, 3, 29, 53, 31, 163, 321, 539, 1283, 429, 3449, 15617, 4761, 21187, 120725}},
	{8687, 17, 25223, {1, 1, 1, 1, 13, 27, 49, 37, 33, 631, 375, 425, 2465, 8773, 2777, 2115, 35633}},
	{8688, 17, 25224, {1, 3, 5, 3, 25, 25, 27, 201, 63, 259, 1571, 1143, 2325, 6773, 11941, 28897, 19719}},
	{8689, 17, 25235, {1, 1, 3, 5, 11, 39, 59, 203, 37, 899, 559, 2599, 4397, 12159, 29579, 51251, 83213}},
	{8690, 17, 25241, {1, 1, 1, 7, 9, 19, 63, 169, 257, 921, 381, 3605, 3479, 1739, 26599, 20599, 29617}},
	{8691, 17, 25253, {1, 1, 1, 9, 7, 29, 123, 35, 419, 963, 855, 1903, 6199, 2727, 29811, 49279, 101673}},
	{8692, 17, 25258, {1, 3, 5, 11, 29, 23, 73, 13, 467, 935, 181, 3837, 8117, 11501, 18361, 26803, 99471}},
	{8693, 17, 25277, {1, 1, 1, 5, 31, 41, 109, 45, 115, 113, 1893, 727, 2453, 13463, 22339, 13495, 11473}},
	{8694, 17, 25278, {1, 1, 5, 9, 5, 31, 109, 145, 511, 243, 57, 2219, 1601, 1821, 12787, 48239, 89645}},
	{8695, 17, 25280, {1, 3, 1, 7, 19, 41, 25, 57, 45, 489, 1531, 3959, 2007, 14247, 13445, 1991, 114977}},
	{8696, 17, 25290, {1, 3, 7, 15, 7, 17, 107, 27, 249, 207, 183, 2483, 5817, 8927, 10715, 63631, 51947}},
	{8697, 17, 25295, {1, 3, 1, 3, 13, 21, 57, 113, 171, 885, 1335, 783, 7575, 4443, 19497, 13827, 130727}},
	{8698, 17, 25300, {1, 1, 5, 7, 19, 33, 95, 13, 387, 297, 1597, 767, 7543, 16063, 10367, 51683, 119811}},
	{8699, 17, 25307, {1, 3, 7, 9, 27, 57, 111, 209, 305, 139, 179, 25, 2295, 2593, 31361, 23677, 74501}},
	{8700, 17, 25319, {1, 3, 7, 3, 21, 63, 97, 189, 3, 693, 209, 2227, 7169, 9, 32575, 61521, 115155}},
	{8701, 17, 25323, {1, 1, 1, 11, 13, 21, 125, 249, 193, 895, 139, 1207, 5941, 5821, 6623, 7753, 80939}},
	{8702, 17, 25326, {1, 3, 5, 5, 11, 49, 17, 21, 423, 497, 835, 539, 6195, 12783, 1271, 20069, 2657}},
	{8703, 17, 25333, {1, 1, 7, 15, 13, 39, 83, 191, 77, 95, 661, 3627, 1853, 1349, 23109, 43583, 104121}},
	{8704, 17, 25334, {1, 3, 1, 15, 31, 15, 71, 255, 489, 91, 351, 367, 309, 6275, 18325, 51231, 52159}},
	{8705, 17, 25337, {1, 1, 7, 13, 21, 49, 37, 135, 355, 421, 507, 2563, 4955, 4095, 1933, 29517, 119699}},
	{8706, 17, 25348, {1, 1, 1, 1, 27, 41, 15, 161, 475, 635, 863, 3773, 6015, 6197, 24261, 26271, 42375}},
	{8707, 17, 25351, {1, 1, 7, 13, 25, 7, 23, 185, 129, 597, 1561, 3003, 2879, 15187, 4913, 24589, 12927}},
	{8708, 17, 25372, {1, 1, 3, 3, 9, 23, 49, 233, 345, 83, 823, 2627, 5019, 2365, 23755, 9855, 48515}},
	{8709, 17, 25381, {1, 3, 1, 1, 11, 7, 117, 213, 27, 923, 375, 2597, 8173, 8935, 16487, 49283, 104569}},
	{8710, 17, 25403, {1, 1, 7, 7, 23, 13, 61, 131, 313, 883, 495, 1105, 6207, 1473, 21655, 51883, 403}},
	{8711, 17, 25406, {1, 3, 3, 1, 25, 5, 5, 159, 243, 929, 1429, 1151, 5043, 11551, 21231, 38767, 105299}},
	{8712, 17, 25431, {1, 3, 7, 7, 15, 37, 49, 219, 67, 147, 873, 2391, 455, 9565, 8977, 64759, 40347}},
	{8713, 17, 25437, {1, 1, 1, 13, 21, 13, 13, 243, 303, 333, 187, 3591, 871, 2501, 30853, 5247, 48855}},
	{8714, 17, 25453, {1, 1, 1, 5, 1, 5, 127, 249, 23, 79, 789, 3507, 8119, 5025, 26545, 54009, 100633}},
	{8715, 17, 25459, {1, 3, 3, 11, 3, 31, 27, 115, 423, 309, 1805, 169, 789, 4081, 28139, 35355, 47991}},
	{8716, 17, 25462, {1, 3, 1, 5, 19, 13, 43, 165, 165, 241, 309, 1703, 7631, 5899, 12041, 21235, 97045}},
	{8717, 17, 25466, {1, 3, 1, 13, 15, 49, 29, 199, 93, 611, 77, 2681, 191, 10215, 8115, 11895, 108687}},
	{8718, 17, 25477, {1, 1, 3, 3, 13, 45, 15, 151, 345, 111, 1829, 1357, 6317, 5239, 26193, 46763, 73101}},
	{8719, 17, 25484, {1, 3, 7, 3, 1, 19, 119, 63, 23, 759, 173, 307, 967, 2731, 9353, 14479, 119}},
	{8720, 17, 25495, {1, 3, 5, 15, 5, 21, 127, 21, 419, 575, 991, 3465, 7365, 5711, 30657, 43513, 22447}},
	{8721, 17, 25501, {1, 3, 7, 1, 19, 5, 49, 7, 45, 963, 1755, 3745, 4061, 4619, 9089, 59953, 100265}},
	{8722, 17, 25506, {1, 1, 1, 3, 25, 53, 97, 97, 347, 749, 823, 1499, 8151, 9957, 731, 22317, 121623}},
	{8723, 17, 25511, {1, 3, 5, 5, 19, 3, 121, 127, 313, 457, 1737, 4065, 5295, 7957, 16373, 62085, 5711}},
	{8724, 17, 25515, {1, 1, 7, 13, 7, 37, 97, 43, 179, 837, 161, 477, 5095, 4985, 111, 58743, 24049}},
	{8725, 17, 25525, {1, 3, 1, 13, 27, 13, 91, 241, 339, 235, 111, 369, 3361, 15105, 11097, 23955, 53561}},
	{8726, 17, 25529, {1, 3, 5, 3, 9, 17, 103, 133, 309, 683, 71, 3329, 7229, 8763, 4165, 9649, 8529}},
	{8727, 17, 25532, {1, 3, 5, 13, 29, 55, 29, 205, 433, 1007, 1173, 731, 5653, 89, 18447, 37911, 65603}},
	{8728, 17, 25538, {1, 3, 5, 1, 15, 1, 7, 195, 397, 877, 1433, 3487, 1581, 1539, 3361, 7453, 46451}},
	{8729, 17, 25549, {1, 1, 5, 13, 23, 1, 47, 245, 19, 859, 681, 2971, 2531, 11393, 32765, 4595, 45213}},
	{8730, 17, 25552, {1, 3, 1, 3, 1, 11, 85, 185, 467, 413, 25, 3677, 881, 1791, 14655, 44811, 50819}},
	{8731, 17, 25564, {1, 3, 5, 9, 9, 21, 65, 99, 441, 215, 1453, 2873, 5883, 485, 20883, 1303, 11837}},
	{8732, 17, 25567, {1, 3, 3, 5, 9, 37, 87, 211, 247, 535, 1163, 1785, 4219, 12559, 17419, 48201, 21725}},
	{8733, 17, 25574, {1, 1, 1, 11, 29, 11, 9, 215, 375, 601, 627, 2641, 6961, 6175, 10995, 49299, 102891}},
	{8734, 17, 25577, {1, 3, 1, 3, 7, 7, 23, 139, 89, 1005, 1815, 947, 1507, 10349, 35, 43595, 104697}},
	{8735, 17, 25583, {1, 1, 5, 13, 29, 47, 77, 255, 341, 333, 1211, 3473, 1303, 11237, 28371, 43283, 77617}},
	{8736, 17, 25588, {1, 3, 3, 13, 27, 17, 73, 95, 227, 241, 1369, 833, 6683, 2193, 309, 64249, 6731}},
	{8737, 17, 25603, {1, 3, 3, 3, 15, 29, 45, 209, 401, 725, 1123, 1659, 6099, 15941, 5797, 30563, 119385}},
	{8738, 17, 25610, {1, 1, 1, 1, 7, 55, 95, 151, 351, 373, 1131, 2357, 7535, 3899, 19047, 17879, 34623}},
	{8739, 17, 25615, {1, 3, 1, 5, 31, 5, 33, 97, 477, 399, 1255, 1073, 1513, 11651, 2951, 31351, 102635}},
	{8740, 17, 25624, {1, 1, 3, 13, 17, 63, 51, 209, 57, 87, 977, 3663, 6717, 15441, 10709, 607, 48297}},
	{8741, 17, 25636, {1, 1, 5, 1, 9, 29, 1, 105, 343, 19, 977, 3401, 3873, 4259, 23057, 13071, 105771}},
	{8742, 17, 25639, {1, 1, 1, 5, 1, 33, 59, 17, 115, 225, 853, 3295, 965, 12547, 26971, 50097, 54999}},
	{8743, 17, 25643, {1, 3, 3, 13, 1, 51, 29, 19, 245, 781, 493, 1121, 2937, 4177, 3267, 47463, 101195}},
	{8744, 17, 25645, {1, 3, 7, 5, 3, 51, 25, 131, 451, 997, 1809, 1583, 355, 15383, 28159, 39141, 109379}},
	{8745, 17, 25648, {1, 1, 5, 7, 3, 19, 75, 103, 401, 115, 1627, 423, 2485, 7281, 6177, 54677, 31499}},
	{8746, 17, 25671, {1, 1, 1, 11, 23, 7, 57, 121, 5, 921, 1191, 1779, 1979, 3427, 25617, 19423, 73835}},
	{8747, 17, 25678, {1, 3, 3, 11, 17, 51, 15, 163, 265, 665, 1399, 1977, 3097, 7109, 14741, 24291, 79239}},
	{8748, 17, 25689, {1, 1, 7, 3, 25, 61, 69, 77, 341, 23, 713, 2879, 8075, 14855, 9691, 58241, 113277}},
	{8749, 17, 25708, {1, 3, 7, 9, 27, 43, 95, 11, 239, 445, 951, 3869, 1049, 6493, 9569, 9285, 29183}},
	{8750, 17, 25716, {1, 1, 3, 1, 1, 23, 27, 101, 337, 171, 1977, 3181, 2693, 8591, 32309, 24909, 106535}},
	{8751, 17, 25725, {1, 1, 1, 7, 23, 59, 79, 115, 49, 351, 871, 1209, 1045, 5985, 28427, 23047, 113571}},
	{8752, 17, 25730, {1, 1, 7, 13, 27, 3, 35, 7, 319, 503, 977, 3747, 4859, 16315, 30375, 25999, 24341}},
	{8753, 17, 25739, {1, 3, 3, 7, 23, 43, 67, 21, 399, 349, 1541, 2991, 5781, 14501, 5609, 65093, 12789}},
	{8754, 17, 25747, {1, 3, 1, 11, 5, 21, 17, 157, 311, 663, 469, 4033, 1557, 7569, 31163, 14079, 127771}},
	{8755, 17, 25753, {1, 1, 7, 15, 15, 31, 15, 183, 365, 35, 1433, 2793, 6685, 10565, 30409, 46815, 14173}},
	{8756, 17, 25790, {1, 1, 7, 7, 7, 45, 61, 163, 99, 353, 1535, 3185, 4023, 7999, 26173, 12675, 98073}},
	{8757, 17, 25797, {1, 1, 5, 13, 1, 11, 107, 41, 171, 773, 1513, 883, 2117, 14449, 32323, 58271, 97173}},
	{8758, 17, 25804, {1, 1, 3, 13, 27, 15, 123, 247, 281, 851, 233, 1173, 6863, 14805, 12401, 30729, 104127}},
	{8759, 17, 25809, {1, 1, 7, 11, 25, 9, 97, 215, 217, 51, 1865, 3897, 725, 4779, 21661, 11853, 72225}},
	{8760, 17, 25816, {1, 1, 5, 3, 5, 31, 125, 81, 367, 705, 325, 519, 3879, 5607, 3247, 7149, 33177}},
	{8761, 17, 25825, {1, 3, 3, 7, 17, 17, 19, 113, 331, 277, 317, 1893, 1287, 8965, 27523, 61355, 45331}},
	{8762, 17, 25831, {1, 3, 7, 9, 27, 15, 87, 21, 343, 479, 11, 2945, 1235, 1591, 28195, 40559, 42773}},
	{8763, 17, 25845, {1, 3, 3, 13, 1, 45, 115, 41, 263, 569, 71, 4051, 739, 1031, 19213, 23961, 110767}},
	{8764, 17, 25846, {1, 3, 7, 1, 9, 41, 21, 131, 3, 617, 191, 4051, 2445, 13451, 11889, 25075, 82631}},
	{8765, 17, 25867, {1, 3, 3, 15, 7, 55, 65, 67, 443, 1023, 1445, 1467, 3907, 11449, 2247, 65085, 102161}},
	{8766, 17, 25870, {1, 3, 5, 15, 19, 27, 97, 181, 51, 591, 99, 1443, 4927, 9809, 29693, 44293, 29369}},
	{8767, 17, 25897, {1, 1, 7, 7, 17, 59, 69, 163, 37, 171, 107, 2581, 3567, 9455, 19707, 6329, 27755}},
	{8768, 17, 25908, {1, 1, 1, 11, 15, 17, 83, 223, 183, 861, 1047, 3739, 3509, 5571, 28259, 42781, 130657}},
	{8769, 17, 25937, {1, 3, 3, 7, 15, 11, 33, 115, 297, 841, 1629, 1559, 2261, 11763, 22255, 63819, 55831}},
	{8770, 17, 25940, {1, 3, 3, 5, 19, 49, 17, 251, 507, 251, 805, 1041, 3947, 2219, 19977, 65449, 25031}},
	{8771, 17, 25944, {1, 1, 1, 11, 3, 7, 81, 17, 219, 729, 949, 3257, 6495, 4701, 2181, 7009, 106465}},
	{8772, 17, 25949, {1, 3, 7, 5, 27, 35, 15, 83, 43, 1013, 1427, 1943, 7555, 6613, 26879, 42685, 22071}},
	{8773, 17, 25954, {1, 1, 3, 13, 23, 55, 15, 87, 15, 579, 717, 777, 149, 11431, 26197, 17711, 7337}},
	{8774, 17, 25960, {1, 1, 5, 1, 31, 45, 113, 253, 211, 915, 1855, 4043, 2159, 1803, 5061, 40473, 3657}},
	{8775, 17, 25963, {1, 1, 3, 7, 25, 15, 37, 73, 467, 969, 1123, 4053, 4837, 10091, 25461, 40803, 91157}},
	{8776, 17, 25966, {1, 1, 5, 1, 7, 31, 77, 207, 21, 623, 577, 1195, 5839, 13013, 11189, 61691, 33327}},
	{8777, 17, 25978, {1, 3, 7, 7, 13, 3, 9, 55, 47, 779, 599, 3747, 1533, 14705, 23185, 4011, 36003}},
	{8778, 17, 25996, {1, 1, 5, 5, 31, 17, 99, 253, 103, 957, 241, 1893, 7435, 14907, 9089, 23205, 70639}},
	{8779, 17, 26007, {1, 3, 7, 15, 7, 55, 53, 101, 227, 541, 2017, 275, 577, 15621, 1799, 50373, 43197}},
	{8780, 17, 26011, {1, 3, 1, 15, 29, 23, 69, 193, 429, 359, 1045, 4091, 6551, 1673, 29113, 43247, 80993}},
	{8781, 17, 26027, {1, 3, 7, 11, 5, 37, 13, 27, 277, 65, 565, 2631, 6919, 5593, 8481, 14703, 9719}},
	{8782, 17, 26032, {1, 3, 1, 15, 5, 7, 83, 51, 77, 307, 1299, 1373, 5281, 15359, 15569, 50093, 59661}},
	{8783, 17, 26038, {1, 3, 5, 11, 13, 31, 99, 123, 263, 319, 2033, 4055, 2427, 103, 2009, 27517, 112467}},
	{8784, 17, 26049, {1, 1, 7, 3, 13, 1, 51, 131, 17, 861, 459, 3925, 5511, 5255, 28553, 36437, 54591}},
	{8785, 17, 26052, {1, 3, 7, 5, 9, 57, 49, 119, 291, 727, 1611, 4035, 4517, 10979, 28445, 26905, 57517}},
	{8786, 17, 26055, {1, 1, 5, 9, 9, 55, 43, 209, 411, 137, 1619, 3965, 5253, 8217, 7569, 42043, 104163}},
	{8787, 17, 26085, {1, 3, 3, 7, 21, 3, 107, 255, 353, 735, 71, 1789, 3351, 755, 22805, 53537, 126859}},
	{8788, 17, 26089, {1, 1, 7, 5, 15, 55, 13, 167, 165, 289, 1231, 2547, 8135, 5475, 2361, 49019, 110579}},
	{8789, 17, 26090, {1, 3, 1, 11, 17, 21, 59, 37, 177, 517, 499, 4035, 749, 14297, 22415, 54975, 29769}},
	{8790, 17, 26098, {1, 3, 7, 3, 3, 59, 55, 17, 483, 625, 875, 1465, 7583, 2969, 2741, 36965, 80367}},
	{8791, 17, 26104, {1, 1, 3, 13, 31, 5, 11, 149, 7, 297, 1485, 735, 4095, 10089, 5757, 64997, 56629}},
	{8792, 17, 26110, {1, 3, 1, 13, 19, 43, 77, 209, 309, 739, 1765, 3297, 8167, 6523, 27987, 25235, 80555}},
	{8793, 17, 26113, {1, 1, 3, 9, 31, 57, 125, 75, 3, 633, 85, 3339, 1691, 9721, 17465, 36801, 106147}},
	{8794, 17, 26126, {1, 3, 5, 15, 27, 7, 111, 7, 475, 523, 1825, 1367, 1549, 15533, 13827, 14471, 100271}},
	{8795, 17, 26133, {1, 1, 5, 3, 5, 61, 1, 221, 163, 183, 1701, 3549, 349, 10057, 26169, 20725, 55305}},
	{8796, 17, 26138, {1, 1, 3, 1, 15, 41, 13, 71, 269, 909, 1253, 2553, 83, 10055, 1057, 39841, 20437}},
	{8797, 17, 26140, {1, 3, 3, 5, 29, 39, 113, 23, 137, 601, 361, 1779, 279, 15803, 8993, 2633, 114847}},
	{8798, 17, 26156, {1, 1, 3, 7, 29, 45, 35, 27, 71, 253, 231, 3449, 1955, 9109, 9043, 50593, 15023}},
	{8799, 17, 26159, {1, 3, 1, 11, 17, 45, 85, 255, 341, 957, 769, 3009, 3997, 6435, 1161, 34219, 97077}},
	{8800, 17, 26176, {1, 1, 5, 1, 1, 19, 9, 51, 477, 129, 1411, 3223, 5069, 3237, 15947, 27215, 70401}},
	{8801, 17, 26186, {1, 1, 3, 1, 1, 1, 73, 31, 301, 227, 119, 607, 3379, 3907, 1263, 2651, 43769}},
	{8802, 17, 26203, {1, 1, 1, 15, 21, 59, 109, 155, 473, 361, 1871, 3085, 513, 12607, 12747, 41067, 44977}},
	{8803, 17, 26210, {1, 1, 7, 3, 27, 21, 89, 71, 437, 671, 1469, 2191, 4225, 6343, 1131, 29141, 25221}},
	{8804, 17, 26222, {1, 1, 7, 9, 7, 45, 95, 197, 391, 11, 1913, 3107, 5247, 959, 30395, 32809, 20893}},
	{8805, 17, 26224, {1, 3, 7, 15, 13, 49, 77, 245, 463, 769, 1807, 1311, 2715, 14819, 27595, 57521, 105221}},
	{8806, 17, 26229, {1, 1, 1, 5, 23, 45, 119, 77, 325, 405, 1631, 461, 6357, 7037, 31699, 46295, 118577}},
	{8807, 17, 26233, {1, 3, 5, 1, 21, 3, 31, 13, 7, 571, 633, 425, 6547, 3423, 19355, 49481, 76289}},
	{8808, 17, 26236, {1, 1, 3, 9, 7, 51, 113, 173, 169, 97, 1821, 979, 2553, 11505, 20047, 39277, 122905}},
	{8809, 17, 26239, {1, 1, 5, 13, 17, 9, 111, 205, 261, 671, 657, 507, 3903, 10767, 4387, 3045, 102617}},
	{8810, 17, 26267, {1, 1, 3, 3, 5, 11, 19, 61, 401, 605, 455, 2457, 4471, 7255, 18435, 49673, 97289}},
	{8811, 17, 26291, {1, 3, 1, 1, 31, 25, 77, 35, 65, 127, 929, 2325, 2315, 13819, 5509, 12515, 36991}},
	{8812, 17, 26293, {1, 1, 7, 5, 21, 49, 33, 119, 181, 645, 1425, 2411, 245, 13755, 18775, 50061, 47119}},
	{8813, 17, 26298, {1, 3, 7, 5, 27, 43, 81, 191, 233, 435, 829, 3881, 713, 11153, 4637, 37721, 115567}},
	{8814, 17, 26303, {1, 3, 7, 1, 27, 59, 51, 165, 59, 931, 1921, 3059, 6843, 813, 22063, 29445, 114765}},
	{8815, 17, 26308, {1, 1, 5, 11, 31, 53, 89, 69, 29, 893, 1241, 7, 1707, 16167, 8371, 14021, 103281}},
	{8816, 17, 26312, {1, 3, 1, 1, 23, 21, 3, 35, 73, 769, 1417, 4051, 3223, 813, 1141, 18823, 31951}},
	{8817, 17, 26315, {1, 3, 7, 11, 9, 17, 89, 85, 407, 137, 1865, 2881, 1495, 3757, 3711, 36651, 1797}},
	{8818, 17, 26342, {1, 3, 5, 1, 25, 29, 29, 217, 15, 173, 479, 2363, 2341, 6193, 16403, 64097, 1173}},
	{8819, 17, 26354, {1, 3, 3, 3, 11, 29, 113, 167, 333, 573, 1467, 2223, 5693, 1063, 20299, 40993, 69055}},
	{8820, 17, 26363, {1, 1, 3, 7, 3, 51, 45, 139, 79, 393, 1251, 1087, 1423, 1827, 23445, 41635, 78511}},
	{8821, 17, 26365, {1, 3, 3, 13, 29, 45, 85, 229, 33, 373, 113, 1205, 997, 11777, 7663, 18513, 5797}},
	{8822, 17, 26383, {1, 1, 5, 5, 15, 5, 127, 85, 49, 345, 901, 3215, 2347, 2329, 19597, 39555, 25031}},
	{8823, 17, 26391, {1, 1, 7, 11, 9, 25, 71, 183, 341, 1011, 439, 3649, 2859, 10183, 7635, 45297, 38581}},
	{8824, 17, 26398, {1, 1, 1, 11, 23, 13, 1, 69, 461, 77, 1641, 2851, 1889, 2413, 1131, 39009, 33773}},
	{8825, 17, 26416, {1, 3, 7, 7, 25, 19, 67, 233, 141, 207, 1501, 453, 4773, 7411, 22839, 25365, 53189}},
	{8826, 17, 26421, {1, 1, 7, 3, 3, 17, 13, 167, 73, 1005, 887, 2595, 4351, 3249, 5653, 36025, 33733}},
	{8827, 17, 26425, {1, 1, 7, 15, 11, 1, 105, 215, 329, 601, 1477, 723, 4597, 3525, 26235, 63957, 26677}},
	{8828, 17, 26434, {1, 1, 1, 11, 27, 15, 111, 133, 327, 567, 845, 2135, 7905, 7297, 29255, 14947, 104885}},
	{8829, 17, 26439, {1, 1, 7, 9, 21, 11, 67, 179, 459, 423, 295, 3445, 1681, 12907, 29281, 7445, 35539}},
	{8830, 17, 26453, {1, 1, 3, 11, 23, 61, 111, 123, 81, 439, 299, 3557, 2821, 705, 15393, 37175, 11533}},
	{8831, 17, 26454, {1, 1, 3, 1, 13, 15, 113, 227, 285, 313, 687, 2085, 6363, 8003, 32661, 36461, 68759}},
	{8832, 17, 26470, {1, 3, 3, 1, 27, 7, 101, 177, 363, 461, 1519, 2339, 473, 7469, 4335, 30951, 130987}},
	{8833, 17, 26491, {1, 1, 1, 3, 31, 39, 59, 159, 207, 93, 581, 1973, 945, 11343, 15901, 22001, 3515}},
	{8834, 17, 26493, {1, 3, 7, 5, 11, 53, 125, 57, 389, 985, 1055, 3803, 3879, 5537, 28221, 36805, 16025}},
	{8835, 17, 26500, {1, 1, 1, 9, 1, 63, 81, 57, 59, 813, 865, 3491, 3771, 6121, 6847, 14765, 68567}},
	{8836, 17, 26509, {1, 1, 7, 7, 13, 17, 23, 211, 435, 167, 933, 357, 3567, 3019, 28439, 17701, 119937}},
	{8837, 17, 26518, {1, 1, 1, 9, 7, 53, 103, 155, 211, 719, 413, 3673, 2795, 15687, 1737, 50855, 129133}},
	{8838, 17, 26531, {1, 1, 3, 13, 11, 23, 53, 121, 497, 383, 1655, 937, 5563, 2549, 23183, 46149, 78875}},
	{8839, 17, 26533, {1, 3, 5, 11, 25, 1, 45, 139, 437, 729, 2009, 3245, 4091, 551, 25993, 31655, 33641}},
	{8840, 17, 26540, {1, 3, 1, 13, 7, 23, 87, 111, 471, 501, 1767, 1051, 7037, 3199, 19609, 43227, 53667}},
	{8841, 17, 26552, {1, 3, 7, 1, 25, 1, 49, 189, 55, 375, 601, 2065, 2991, 7697, 25739, 14951, 43705}},
	{8842, 17, 26566, {1, 1, 7, 5, 29, 21, 51, 111, 81, 809, 1963, 2143, 5529, 15701, 4719, 11857, 88207}},
	{8843, 17, 26569, {1, 1, 5, 11, 27, 27, 7, 145, 281, 939, 537, 1255, 1475, 11383, 15081, 9105, 102775}},
	{8844, 17, 26583, {1, 1, 5, 7, 9, 45, 67, 23, 65, 139, 1871, 3975, 6357, 6515, 25423, 23915, 76289}},
	{8845, 17, 26590, {1, 1, 5, 11, 31, 31, 89, 65, 451, 341, 819, 2439, 6753, 15113, 9085, 32687, 5055}},
	{8846, 17, 26596, {1, 1, 5, 1, 15, 29, 123, 83, 495, 185, 303, 315, 6015, 5257, 2467, 4903, 78269}},
	{8847, 17, 26624, {1, 1, 3, 5, 31, 51, 49, 199, 501, 407, 733, 1181, 8023, 7321, 14765, 17535, 19893}},
	{8848, 17, 26636, {1, 1, 5, 5, 19, 15, 103, 183, 13, 969, 1537, 3053, 3173, 12983, 21761, 33733, 67799}},
	{8849, 17, 26667, {1, 3, 1, 1, 27, 55, 37, 149, 379, 11, 1655, 2317, 3135, 6459, 25941, 12679, 60245}},
	{8850, 17, 26669, {1, 1, 3, 9, 9, 13, 33, 243, 337, 741, 1685, 1147, 5465, 4865, 559, 23993, 47273}},
	{8851, 17, 26672, {1, 3, 5, 13, 21, 11, 39, 169, 135, 209, 1909, 3655, 3117, 1075, 8165, 54633, 28189}},
	{8852, 17, 26675, {1, 3, 1, 15, 9, 23, 11, 123, 63, 133, 947, 907, 3853, 10291, 22905, 4561, 92497}},
	{8853, 17, 26687, {1, 1, 3, 13, 17, 9, 5, 209, 429, 3, 2035, 1497, 6765, 5991, 24991, 8155, 103417}},
	{8854, 17, 26689, {1, 1, 5, 13, 27, 47, 79, 11, 41, 791, 1939, 3099, 4069, 4665, 20801, 18659, 72163}},
	{8855, 17, 26696, {1, 3, 1, 13, 25, 37, 79, 131, 233, 647, 291, 1419, 5157, 4261, 27715, 611, 83157}},
	{8856, 17, 26702, {1, 1, 5, 1, 17, 61, 45, 163, 137, 937, 91, 1695, 1553, 543, 28615, 6855, 75201}},
	{8857, 17, 26714, {1, 1, 3, 13, 7, 63, 109, 13, 351, 159, 1111, 2791, 4701, 5805, 9707, 18361, 98091}},
	{8858, 17, 26716, {1, 1, 3, 11, 11, 21, 55, 247, 111, 801, 93, 3091, 1043, 9761, 23679, 5555, 195}},
	{8859, 17, 26729, {1, 1, 1, 1, 13, 43, 123, 113, 265, 561, 659, 3755, 6605, 10949, 30511, 29495, 9075}},
	{8860, 17, 26732, {1, 3, 1, 7, 23, 63, 19, 73, 233, 1017, 851, 1971, 3999, 7407, 25309, 63991, 92867}},
	{8861, 17, 26737, {1, 3, 3, 3, 19, 63, 127, 107, 465, 463, 1507, 1323, 4729, 14717, 9129, 24859, 117565}},
	{8862, 17, 26747, {1, 1, 7, 1, 19, 11, 13, 85, 339, 939, 895, 887, 765, 15501, 8783, 65087, 77899}},
	{8863, 17, 26773, {1, 3, 7, 3, 7, 15, 43, 153, 365, 223, 1947, 2295, 787, 5549, 20089, 29203, 4807}},
	{8864, 17, 26780, {1, 3, 7, 3, 31, 27, 51, 217, 483, 623, 633, 2123, 1211, 9173, 17949, 54251, 89161}},
	{8865, 17, 26801, {1, 3, 3, 11, 3, 11, 111, 73, 283, 23, 1925, 253, 5141, 12545, 24913, 16847, 13067}},
	{8866, 17, 26811, {1, 3, 5, 5, 31, 39, 35, 235, 135, 85, 191, 999, 6633, 12527, 21401, 61339, 81239}},
	{8867, 17, 26813, {1, 1, 3, 15, 9, 13, 39, 125, 137, 639, 555, 813, 2821, 1199, 32075, 15079, 104609}},
	{8868, 17, 26831, {1, 3, 7, 7, 15, 43, 99, 51, 245, 25, 147, 89, 6841, 5523, 28493, 13967, 113109}},
	{8869, 17, 26836, {1, 1, 3, 13, 7, 5, 27, 121, 269, 231, 1011, 1365, 5055, 11619, 27393, 48033, 65725}},
	{8870, 17, 26839, {1, 1, 7, 5, 9, 23, 41, 71, 327, 339, 1681, 3303, 4143, 421, 15213, 48405, 98067}},
	{8871, 17, 26849, {1, 3, 7, 15, 15, 33, 73, 33, 351, 131, 1051, 3909, 7535, 7659, 9443, 35015, 329}},
	{8872, 17, 26862, {1, 1, 3, 9, 19, 55, 97, 223, 265, 877, 235, 867, 4961, 3137, 19885, 10955, 7655}},
	{8873, 17, 26869, {1, 3, 5, 13, 1, 11, 75, 215, 271, 793, 1691, 1437, 1317, 10977, 1311, 64865, 92361}},
	{8874, 17, 26873, {1, 1, 1, 5, 23, 23, 35, 53, 187, 345, 115, 929, 3919, 4523, 31709, 16771, 33399}},
	{8875, 17, 26874, {1, 3, 5, 11, 17, 7, 75, 57, 351, 359, 1737, 2665, 4259, 13905, 6999, 45359, 117891}},
	{8876, 17, 26888, {1, 1, 5, 3, 23, 29, 49, 209, 417, 843, 531, 1649, 7829, 6271, 3759, 39727, 47415}},
	{8877, 17, 26891, {1, 1, 7, 1, 21, 45, 101, 105, 385, 797, 985, 1447, 3757, 3287, 583, 29283, 96821}},
	{8878, 17, 26896, {1, 1, 7, 9, 1, 29, 15, 207, 289, 465, 815, 2289, 449, 9403, 19197, 13797, 102651}},
	{8879, 17, 26899, {1, 3, 7, 15, 5, 45, 81, 187, 21, 433, 679, 2759, 3375, 6935, 22595, 50149, 13557}},
	{8880, 17, 26912, {1, 3, 3, 5, 11, 55, 69, 219, 95, 21, 645, 1955, 7527, 6037, 29427, 36297, 62013}},
	{8881, 17, 26921, {1, 1, 7, 15, 13, 35, 25, 67, 383, 13, 539, 2399, 4611, 8065, 3815, 27771, 50411}},
	{8882, 17, 26935, {1, 3, 1, 3, 27, 47, 65, 33, 393, 895, 1663, 1289, 1057, 11887, 1259, 53611, 36811}},
	{8883, 17, 26941, {1, 3, 5, 3, 5, 1, 27, 65, 379, 15, 1643, 1461, 3009, 8177, 15589, 5889, 1103}},
	{8884, 17, 26949, {1, 1, 5, 15, 27, 53, 43, 173, 377, 665, 581, 1061, 1363, 15015, 26709, 29507, 28075}},
	{8885, 17, 26954, {1, 1, 7, 3, 9, 11, 45, 71, 23, 995, 1277, 855, 1001, 12927, 19753, 46639, 16949}},
	{8886, 17, 26967, {1, 1, 7, 5, 15, 33, 27, 27, 437, 415, 1785, 2091, 279, 8041, 2209, 15821, 122363}},
	{8887, 17, 26977, {1, 1, 7, 1, 21, 1, 47, 215, 463, 959, 849, 1703, 5175, 10043, 16991, 11023, 52201}},
	{8888, 17, 27011, {1, 1, 1, 11, 21, 7, 121, 31, 95, 631, 1717, 3017, 2083, 2047, 12051, 63117, 25949}},
	{8889, 17, 27018, {1, 1, 5, 5, 9, 5, 105, 121, 205, 643, 1721, 2601, 2991, 2381, 4873, 12049, 20043}},
	{8890, 17, 27026, {1, 3, 5, 11, 7, 11, 97, 187, 253, 571, 101, 3077, 1479, 9513, 15451, 37105, 34445}},
	{8891, 17, 27028, {1, 1, 7, 11, 19, 19, 39, 115, 221, 13, 217, 369, 6855, 14529, 143, 13461, 62927}},
	{8892, 17, 27032, {1, 3, 3, 7, 29, 27, 9, 171, 419, 571, 837, 3829, 1871, 12691, 30693, 4195, 38905}},
	{8893, 17, 27044, {1, 1, 1, 11, 5, 55, 17, 41, 241, 419, 337, 897, 4663, 14469, 18701, 18009, 44605}},
	{8894, 17, 27053, {1, 1, 7, 7, 1, 33, 63, 197, 257, 655, 1287, 2571, 57, 13275, 29669, 8501, 61389}},
	{8895, 17, 27062, {1, 3, 5, 3, 29, 39, 101, 215, 101, 271, 1373, 2171, 841, 9865, 28951, 20369, 42413}},
	{8896, 17, 27065, {1, 3, 5, 1, 31, 23, 119, 137, 263, 633, 1239, 281, 4965, 14913, 30229, 14147, 37183}},
	{8897, 17, 27068, {1, 3, 1, 3, 11, 55, 33, 45, 69, 913, 269, 1021, 4005, 15191, 11187, 45917, 76905}},
	{8898, 17, 27080, {1, 1, 1, 13, 27, 11, 75, 139, 243, 221, 1289, 2195, 7041, 10053, 5731, 35245, 41317}},
	{8899, 17, 27083, {1, 1, 7, 9, 25, 11, 81, 243, 233, 137, 831, 2825, 6007, 7305, 31733, 64343, 7047}},
	{8900, 17, 27093, {1, 3, 5, 9, 17, 61, 25, 245, 285, 969, 1397, 1331, 5393, 3653, 15533, 9121, 111115}},
	{8901, 17, 27098, {1, 1, 5, 9, 1, 9, 61, 205, 285, 849, 1071, 1013, 2655, 10121, 15165, 25189, 56207}},
	{8902, 17, 27107, {1, 3, 5, 7, 19, 45, 121, 19, 237, 711, 1523, 3251, 693, 13567, 31993, 11239, 64127}},
	{8903, 17, 27110, {1, 1, 1, 11, 23, 25, 33, 211, 321, 1021, 1855, 291, 2911, 11841, 21929, 64147, 63201}},
	{8904, 17, 27114, {1, 1, 7, 3, 27, 21, 119, 219, 431, 819, 83, 2487, 7533, 10697, 3129, 53301, 104999}},
	{8905, 17, 27121, {1, 3, 5, 15, 9, 25, 89, 65, 293, 411, 989, 3103, 5563, 15703, 8757, 32595, 43409}},
	{8906, 17, 27127, {1, 3, 3, 1, 31, 31, 45, 173, 231, 171, 1185, 1499, 1713, 9945, 11575, 37113, 103989}},
	{8907, 17, 27137, {1, 1, 5, 13, 27, 3, 93, 253, 23, 71, 1963, 2571, 6259, 15757, 9709, 42835, 42047}},
	{8908, 17, 27144, {1, 1, 7, 11, 5, 11, 123, 117, 39, 559, 111, 527, 6253, 781, 9177, 47189, 94031}},
	{8909, 17, 27155, {1, 1, 5, 15, 5, 49, 93, 185, 167, 787, 1553, 3291, 3723, 1651, 23225, 5643, 42967}},
	{8910, 17, 27171, {1, 3, 1, 13, 15, 35, 19, 193, 505, 127, 661, 1943, 1249, 5103, 8233, 64319, 76955}},
	{8911, 17, 27178, {1, 3, 7, 7, 17, 13, 97, 185, 415, 331, 283, 3341, 2903, 2927, 7729, 4095, 103083}},
	{8912, 17, 27180, {1, 1, 3, 15, 15, 25, 65, 45, 413, 521, 747, 2605, 5845, 12909, 7651, 45937, 99043}},
	{8913, 17, 27185, {1, 1, 5, 9, 9, 21, 3, 75, 335, 745, 1493, 1721, 1977, 11105, 22621, 49281, 107113}},
	{8914, 17, 27191, {1, 3, 1, 11, 25, 11, 99, 53, 239, 831, 655, 615, 7565, 14039, 29115, 26165, 127159}},
	{8915, 17, 27192, {1, 1, 7, 5, 31, 35, 75, 157, 441, 815, 119, 565, 2703, 14059, 7867, 42487, 93647}},
	{8916, 17, 27195, {1, 3, 7, 3, 3, 59, 101, 223, 257, 989, 363, 1059, 5157, 11129, 1481, 19287, 117623}},
	{8917, 17, 27212, {1, 1, 1, 1, 29, 27, 1, 129, 253, 673, 103, 1881, 7053, 1261, 32003, 23345, 102503}},
	{8918, 17, 27215, {1, 3, 1, 9, 11, 37, 3, 99, 303, 519, 1175, 3021, 2275, 9919, 25011, 45865, 71351}},
	{8919, 17, 27218, {1, 1, 7, 15, 27, 9, 107, 61, 385, 21, 861, 2119, 4643, 8379, 25455, 22425, 113387}},
	{8920, 17, 27230, {1, 1, 1, 7, 27, 15, 73, 211, 497, 527, 873, 329, 2213, 415, 13987, 56581, 27829}},
	{8921, 17, 27234, {1, 3, 5, 1, 31, 43, 107, 149, 209, 211, 2029, 2793, 2213, 12389, 27177, 51375, 51983}},
	{8922, 17, 27245, {1, 3, 3, 7, 25, 57, 67, 101, 127, 43, 1775, 801, 3343, 12203, 8667, 58387, 10735}},
	{8923, 17, 27248, {1, 1, 7, 5, 13, 47, 101, 123, 133, 593, 1409, 3525, 2545, 13009, 11873, 38463, 1075}},
	{8924, 17, 27263, {1, 3, 3, 1, 3, 19, 75, 221, 157, 67, 397, 1141, 5073, 10795, 9857, 35459, 62701}},
	{8925, 17, 27279, {1, 1, 7, 7, 23, 17, 41, 179, 83, 543, 1839, 3709, 131, 15681, 9147, 18685, 90389}},
	{8926, 17, 27288, {1, 1, 5, 7, 17, 15, 31, 217, 79, 687, 1927, 2889, 6127, 15095, 28437, 16403, 123275}},
	{8927, 17, 27294, {1, 3, 7, 15, 13, 17, 123, 75, 45, 635, 525, 3897, 6769, 13855, 16695, 18039, 37479}},
	{8928, 17, 27303, {1, 1, 5, 1, 23, 15, 69, 161, 503, 339, 1061, 839, 9, 10013, 24493, 32711, 50147}},
	{8929, 17, 27304, {1, 3, 3, 11, 5, 45, 9, 233, 131, 629, 1111, 3311, 2211, 9079, 19763, 23793, 85389}},
	{8930, 17, 27312, {1, 1, 7, 7, 7, 27, 15, 85, 291, 925, 1545, 3061, 4867, 1613, 13467, 53731, 92811}},
	{8931, 17, 27330, {1, 3, 5, 1, 21, 21, 71, 33, 141, 675, 1519, 3275, 1491, 10717, 28199, 14983, 18961}},
	{8932, 17, 27341, {1, 3, 5, 5, 31, 13, 109, 239, 369, 373, 257, 3765, 2531, 13281, 11877, 29439, 106939}},
	{8933, 17, 27342, {1, 3, 7, 15, 13, 39, 111, 203, 109, 179, 789, 3849, 433, 5745, 2343, 15795, 92603}},
	{8934, 17, 27344, {1, 3, 5, 5, 11, 57, 3, 245, 289, 249, 713, 315, 2261, 1249, 6963, 44507, 100829}},
	{8935, 17, 27353, {1, 3, 5, 11, 5, 49, 97, 245, 363, 315, 509, 17, 4485, 15393, 28491, 17945, 65663}},
	{8936, 17, 27356, {1, 3, 5, 1, 5, 13, 15, 17, 141, 119, 1393, 581, 2439, 2015, 11527, 8537, 103005}},
	{8937, 17, 27366, {1, 3, 5, 1, 25, 9, 117, 25, 99, 777, 985, 1159, 99, 3013, 21429, 19027, 61833}},
	{8938, 17, 27369, {1, 1, 1, 5, 1, 47, 37, 83, 159, 29, 281, 3789, 2525, 15999, 8965, 11145, 14453}},
	{8939, 17, 27377, {1, 1, 3, 1, 11, 63, 77, 207, 267, 473, 241, 629, 6969, 9093, 839, 3875, 18873}},
	{8940, 17, 27387, {1, 3, 5, 7, 31, 57, 103, 65, 349, 321, 717, 2403, 105, 643, 27999, 2509, 123061}},
	{8941, 17, 27409, {1, 1, 5, 7, 3, 31, 7, 113, 17, 995, 1211, 1749, 6757, 3391, 8011, 47715, 24301}},
	{8942, 17, 27437, {1, 3, 7, 11, 7, 55, 29, 155, 373, 81, 1255, 2205, 3233, 9537, 22129, 24505, 79021}},
	{8943, 17, 27443, {1, 1, 7, 5, 3, 49, 5, 51, 89, 107, 585, 3933, 745, 11685, 20663, 12521, 24357}},
	{8944, 17, 27446, {1, 1, 7, 1, 17, 17, 83, 215, 357, 581, 2025, 3411, 7287, 11925, 2253, 43513, 98655}},
	{8945, 17, 27450, {1, 3, 5, 3, 27, 27, 51, 147, 471, 509, 423, 3807, 677, 8429, 581, 47999, 35913}},
	{8946, 17, 27452, {1, 3, 3, 9, 15, 31, 1, 93, 207, 759, 1991, 473, 2273, 43, 12547, 58085, 20953}},
	{8947, 17, 27457, {1, 1, 3, 3, 1, 27, 39, 219, 381, 187, 159, 2333, 6141, 3775, 5693, 14853, 38765}},
	{8948, 17, 27472, {1, 3, 1, 5, 19, 1, 19, 237, 231, 975, 1609, 723, 241, 10105, 18817, 58373, 118889}},
	{8949, 17, 27475, {1, 1, 5, 7, 7, 43, 99, 181, 109, 529, 421, 1493, 1075, 12219, 24287, 33479, 29987}},
	{8950, 17, 27487, {1, 1, 7, 1, 17, 11, 79, 85, 157, 851, 1429, 3355, 139, 14327, 30025, 60303, 109015}},
	{8951, 17, 27488, {1, 3, 5, 9, 11, 15, 37, 79, 5, 169, 999, 815, 6255, 11763, 16299, 49891, 101917}},
	{8952, 17, 27493, {1, 3, 5, 9, 29, 45, 51, 211, 159, 771, 1631, 2871, 4877, 4941, 18127, 15669, 57515}},
	{8953, 17, 27506, {1, 1, 3, 3, 19, 29, 9, 205, 253, 399, 303, 2441, 3187, 641, 23341, 52951, 57559}},
	{8954, 17, 27508, {1, 3, 3, 15, 11, 29, 121, 227, 69, 935, 365, 217, 4617, 13193, 27663, 46903, 107843}},
	{8955, 17, 27521, {1, 1, 5, 11, 13, 31, 13, 243, 275, 685, 1613, 1915, 2775, 11225, 4729, 45549, 103875}},
	{8956, 17, 27522, {1, 1, 5, 1, 9, 61, 35, 143, 165, 441, 517, 1735, 5281, 10139, 21107, 11713, 130483}},
	{8957, 17, 27527, {1, 3, 5, 5, 13, 21, 7, 219, 97, 887, 1845, 469, 2523, 1569, 9959, 4397, 15823}},
	{8958, 17, 27536, {1, 3, 7, 11, 15, 27, 73, 223, 365, 939, 331, 145, 683, 6441, 23421, 59177, 31763}},
	{8959, 17, 27539, {1, 3, 1, 5, 9, 59, 85, 113, 343, 831, 121, 3157, 6059, 14923, 27401, 19759, 14223}},
	{8960, 17, 27541, {1, 3, 1, 7, 17, 25, 3, 39, 471, 759, 2041, 609, 4293, 7491, 8041, 50857, 25601}},
	{8961, 17, 27546, {1, 1, 5, 15, 19, 45, 21, 5, 269, 197, 527, 1839, 1719, 15105, 18671, 42167, 9617}},
	{8962, 17, 27557, {1, 3, 1, 3, 5, 35, 3, 105, 395, 113, 1945, 3945, 3951, 12207, 32135, 34121, 10237}},
	{8963, 17, 27564, {1, 1, 5, 13, 21, 43, 51, 255, 57, 399, 1937, 1573, 363, 11589, 26989, 54345, 94341}},
	{8964, 17, 27572, {1, 3, 3, 3, 5, 45, 83, 125, 179, 923, 39, 3617, 7683, 8191, 31469, 23633, 79179}},
	{8965, 17, 27576, {1, 3, 7, 9, 9, 37, 107, 65, 423, 77, 1779, 1375, 2085, 11779, 6535, 2973, 29425}},
	{8966, 17, 27584, {1, 1, 7, 3, 11, 39, 101, 137, 407, 855, 1767, 1717, 2821, 10447, 31187, 6329, 124111}},
	{8967, 17, 27608, {1, 1, 5, 11, 27, 27, 45, 103, 225, 681, 725, 1791, 2881, 2923, 14473, 12269, 58743}},
	{8968, 17, 27611, {1, 1, 3, 11, 15, 39, 113, 167, 143, 677, 1189, 1571, 5339, 6065, 30187, 19639, 42227}},
	{8969, 17, 27618, {1, 1, 1, 3, 13, 7, 113, 75, 129, 619, 1741, 1495, 4751, 11085, 22391, 199, 105463}},
	{8970, 17, 27620, {1, 1, 3, 3, 19, 47, 77, 195, 209, 453, 495, 1605, 5255, 15327, 8941, 18239, 54511}},
	{8971, 17, 27627, {1, 1, 7, 7, 21, 29, 95, 175, 3, 689, 611, 2467, 6919, 12399, 18869, 16171, 102753}},
	{8972, 17, 27630, {1, 1, 5, 1, 27, 43, 61, 133, 37, 603, 315, 1915, 813, 15769, 27447, 29589, 122281}},
	{8973, 17, 27637, {1, 1, 7, 3, 11, 1, 119, 235, 93, 481, 1811, 1643, 4853, 11313, 8991, 6153, 68985}},
	{8974, 17, 27638, {1, 1, 1, 7, 1, 13, 99, 83, 221, 775, 1345, 219, 4445, 11837, 10405, 43563, 122111}},
	{8975, 17, 27641, {1, 1, 5, 13, 21, 33, 105, 19, 343, 571, 703, 429, 2485, 15531, 9801, 24101, 88275}},
	{8976, 17, 27647, {1, 3, 5, 1, 27, 55, 73, 49, 33, 773, 1411, 1227, 6827, 1271, 28897, 24265, 32383}},
	{8977, 17, 27650, {1, 3, 7, 3, 9, 45, 59, 5, 157, 669, 261, 2077, 1425, 8221, 5849, 40857, 121029}},
	{8978, 17, 27676, {1, 3, 7, 11, 23, 5, 87, 113, 279, 611, 1195, 1775, 5813, 6737, 18051, 41341, 93331}},
	{8979, 17, 27683, {1, 1, 7, 7, 9, 55, 113, 3, 167, 295, 1579, 2833, 4003, 7583, 22833, 44427, 34781}},
	{8980, 17, 27690, {1, 1, 5, 13, 21, 33, 127, 175, 153, 961, 819, 143, 3969, 6159, 29437, 14123, 65317}},
	{8981, 17, 27695, {1, 1, 1, 15, 31, 27, 1, 17, 329, 963, 1907, 421, 535, 2323, 22749, 44375, 115531}},
	{8982, 17, 27700, {1, 1, 5, 15, 15, 23, 57, 171, 253, 401, 1577, 3855, 197, 7979, 17577, 25275, 88831}},
	{8983, 17, 27704, {1, 1, 7, 9, 27, 9, 7, 13, 381, 847, 533, 357, 6551, 13441, 5717, 20209, 64347}},
	{8984, 17, 27709, {1, 1, 7, 9, 1, 1, 31, 245, 315, 901, 1857, 497, 4285, 13227, 3937, 45025, 59627}},
	{8985, 17, 27715, {1, 1, 7, 3, 5, 23, 119, 147, 479, 71, 113, 3379, 863, 8285, 31259, 15863, 47267}},
	{8986, 17, 27727, {1, 3, 5, 1, 5, 7, 77, 163, 421, 353, 1757, 1335, 4975, 3011, 11703, 56075, 12045}},
	{8987, 17, 27745, {1, 1, 5, 3, 31, 25, 81, 59, 211, 463, 1693, 609, 3307, 3641, 19643, 29361, 8399}},
	{8988, 17, 27763, {1, 1, 7, 13, 19, 47, 43, 43, 275, 735, 535, 3689, 3987, 10695, 17243, 60565, 72299}},
	{8989, 17, 27769, {1, 3, 3, 5, 25, 35, 75, 63, 305, 127, 189, 1785, 731, 12089, 27811, 43259, 28191}},
	{8990, 17, 27770, {1, 3, 7, 11, 17, 17, 59, 107, 139, 355, 1055, 3181, 4743, 14785, 26323, 441, 35613}},
	{8991, 17, 27796, {1, 3, 1, 1, 17, 17, 39, 203, 373, 601, 449, 1837, 835, 7061, 14655, 61765, 80735}},
	{8992, 17, 27810, {1, 3, 5, 7, 27, 17, 25, 41, 125, 895, 1843, 3167, 1527, 4707, 6477, 33575, 97247}},
	{8993, 17, 27812, {1, 3, 3, 3, 13, 39, 25, 15, 279, 347, 1121, 3603, 3019, 9577, 16863, 61483, 15995}},
	{8994, 17, 27834, {1, 3, 5, 11, 15, 33, 15, 81, 185, 289, 909, 1237, 3623, 3983, 24211, 62719, 79685}},
	{8995, 17, 27847, {1, 1, 1, 7, 29, 1, 53, 17, 137, 269, 1209, 3937, 4167, 14057, 8061, 38863, 101477}},
	{8996, 17, 27848, {1, 1, 1, 9, 5, 45, 95, 127, 507, 159, 1763, 1527, 5689, 11007, 549, 22837, 99207}},
	{8997, 17, 27862, {1, 3, 3, 1, 15, 57, 127, 39, 73, 397, 67, 3159, 119, 8929, 29425, 57687, 68063}},
	{8998, 17, 27868, {1, 3, 1, 3, 27, 7, 111, 209, 291, 17, 1381, 1597, 5389, 4577, 20463, 28325, 23743}},
	{8999, 17, 27881, {1, 3, 3, 7, 23, 41, 83, 81, 213, 537, 1037, 2371, 1485, 8391, 12471, 58541, 27559}},
	{9000, 17, 27884, {1, 3, 1, 15, 21, 43, 87, 75, 451, 851, 1917, 2739, 2167, 12531, 29931, 8017, 15163}},
	{9001, 17, 27890, {1, 1, 3, 9, 27, 19, 41, 145, 401, 759, 527, 3085, 187, 10615, 4995, 22421, 69867}},
	{9002, 17, 27895, {1, 3, 3, 13, 29, 51, 65, 47, 157, 609, 1061, 1913, 6195, 12503, 10375, 55819, 122091}},
	{9003, 17, 27904, {1, 1, 3, 7, 1, 19, 3, 149, 453, 279, 569, 3429, 331, 711, 26773, 21163, 129339}},
	{9004, 17, 27914, {1, 1, 5, 3, 7, 47, 39, 181, 115, 771, 2037, 411, 2697, 7501, 6393, 19461, 74967}},
	{9005, 17, 27919, {1, 3, 3, 5, 5, 17, 89, 161, 409, 49, 1447, 3977, 4777, 15553, 3521, 32553, 126385}},
	{9006, 17, 27921, {1, 3, 1, 3, 25, 59, 73, 105, 505, 1009, 1147, 317, 3457, 13743, 8337, 38077, 7709}},
	{9007, 17, 27931, {1, 3, 3, 15, 23, 37, 25, 123, 413, 911, 637, 2345, 691, 15199, 22927, 52467, 126715}},
	{9008, 17, 27933, {1, 1, 5, 1, 9, 51, 93, 123, 269, 45, 1947, 179, 5091, 3743, 31491, 39771, 119175}},
	{9009, 17, 27938, {1, 3, 1, 5, 29, 23, 107, 183, 25, 115, 187, 857, 7337, 469, 8755, 17281, 45941}},
	{9010, 17, 27943, {1, 3, 1, 13, 25, 61, 85, 115, 181, 955, 1365, 837, 5941, 13209, 27009, 58865, 115149}},
	{9011, 17, 27949, {1, 1, 1, 11, 31, 63, 101, 29, 37, 185, 465, 2651, 6249, 6887, 25021, 60539, 50037}},
	{9012, 17, 28005, {1, 3, 1, 7, 7, 61, 57, 243, 143, 223, 1759, 4085, 6765, 13293, 31929, 29579, 35053}},
	{9013, 17, 28012, {1, 3, 1, 3, 29, 9, 121, 3, 285, 199, 1439, 3151, 5059, 8535, 17049, 38917, 46347}},
	{9014, 17, 28017, {1, 3, 1, 3, 17, 43, 63, 87, 427, 341, 1251, 3775, 7729, 3183, 10579, 917, 49035}},
	{9015, 17, 28024, {1, 1, 7, 3, 5, 59, 119, 227, 495, 345, 841, 2021, 2483, 15987, 24663, 9819, 33009}},
	{9016, 17, 28030, {1, 1, 5, 11, 19, 57, 23, 181, 63, 991, 1, 2927, 4785, 9645, 17435, 55627, 1069}},
	{9017, 17, 28034, {1, 1, 7, 1, 31, 11, 57, 123, 279, 815, 1407, 3509, 3963, 8503, 20345, 7777, 103701}},
	{9018, 17, 28045, {1, 1, 5, 5, 19, 51, 37, 15, 363, 939, 1863, 3485, 7073, 3035, 31279, 7289, 39811}},
	{9019, 17, 28048, {1, 1, 3, 3, 3, 41, 29, 37, 311, 535, 1993, 3937, 309, 13055, 22595, 59641, 95317}},
	{9020, 17, 28054, {1, 3, 7, 9, 19, 29, 23, 181, 503, 223, 1655, 997, 7861, 5867, 16979, 4559, 7447}},
	{9021, 17, 28063, {1, 3, 5, 3, 13, 13, 3, 137, 361, 101, 1005, 2339, 609, 11891, 15245, 9653, 63579}},
	{9022, 17, 28069, {1, 1, 1, 15, 31, 15, 117, 151, 51, 805, 1403, 3243, 4007, 11979, 3945, 61935, 43225}},
	{9023, 17, 28070, {1, 1, 7, 3, 1, 43, 93, 105, 9, 555, 731, 655, 2097, 8015, 27557, 27985, 11323}},
	{9024, 17, 28073, {1, 3, 3, 9, 23, 35, 59, 217, 437, 755, 685, 1431, 2965, 5269, 25621, 21735, 1397}},
	{9025, 17, 28084, {1, 1, 5, 1, 9, 61, 41, 53, 101, 111, 531, 3385, 4771, 9535, 15995, 29687, 99035}},
	{9026, 17, 28096, {1, 1, 7, 7, 1, 3, 25, 251, 463, 99, 677, 1889, 3697, 5579, 11429, 38301, 57917}},
	{9027, 17, 28114, {1, 1, 5, 9, 11, 15, 65, 31, 369, 825, 1229, 1595, 3891, 5235, 16973, 25307, 7805}},
	{9028, 17, 28126, {1, 3, 7, 15, 27, 37, 35, 103, 393, 781, 1713, 2009, 1973, 15461, 6801, 17557, 105139}},
	{9029, 17, 28141, {1, 3, 7, 7, 17, 51, 83, 29, 113, 173, 1733, 2137, 3041, 11361, 15999, 25779, 112493}},
	{9030, 17, 28149, {1, 3, 5, 11, 19, 3, 89, 103, 449, 375, 437, 4077, 889, 12163, 29323, 48845, 7783}},
	{9031, 17, 28165, {1, 3, 7, 1, 19, 25, 83, 35, 203, 27, 507, 25, 6629, 13941, 6187, 17533, 83349}},
	{9032, 17, 28177, {1, 3, 5, 9, 15, 59, 3, 87, 473, 733, 1263, 1733, 4275, 9283, 32535, 20807, 59487}},
	{9033, 17, 28189, {1, 3, 3, 9, 19, 11, 27, 83, 355, 949, 1339, 171, 921, 14171, 16271, 41485, 20405}},
	{9034, 17, 28190, {1, 1, 3, 11, 25, 51, 9, 241, 367, 519, 1895, 429, 7689, 9469, 32709, 46363, 75767}},
	{9035, 17, 28203, {1, 1, 7, 7, 27, 59, 85, 87, 467, 273, 1763, 391, 451, 16165, 7501, 44779, 68281}},
	{9036, 17, 28206, {1, 1, 7, 5, 1, 35, 5, 217, 31, 145, 1151, 2255, 3543, 401, 17141, 5981, 25183}},
	{9037, 17, 28208, {1, 1, 1, 13, 11, 11, 19, 93, 95, 751, 31, 1091, 2733, 10517, 2553, 5247, 42651}},
	{9038, 17, 28220, {1, 3, 7, 5, 15, 1, 67, 21, 303, 137, 355, 1989, 5211, 4985, 645, 6867, 126931}},
	{9039, 17, 28226, {1, 1, 3, 15, 21, 23, 59, 209, 121, 623, 575, 2447, 6149, 10481, 4959, 22913, 64963}},
	{9040, 17, 28231, {1, 3, 1, 1, 25, 55, 47, 95, 215, 609, 639, 1023, 1579, 5953, 3063, 13721, 17607}},
	{9041, 17, 28249, {1, 1, 1, 11, 7, 61, 127, 173, 307, 623, 453, 3827, 4847, 16085, 4407, 4043, 14881}},
	{9042, 17, 28256, {1, 1, 7, 11, 5, 15, 51, 125, 439, 795, 203, 91, 3543, 6925, 32055, 52277, 26841}},
	{9043, 17, 28259, {1, 1, 1, 13, 15, 9, 107, 183, 391, 751, 243, 1105, 8031, 7443, 137, 45695, 80163}},
	{9044, 17, 28265, {1, 3, 5, 9, 5, 61, 117, 113, 121, 291, 225, 1705, 4017, 13113, 11035, 28613, 25927}},
	{9045, 17, 28266, {1, 1, 3, 11, 27, 9, 45, 85, 309, 991, 1639, 1183, 8013, 14587, 7563, 21111, 48497}},
	{9046, 17, 28271, {1, 3, 1, 9, 21, 61, 123, 189, 265, 593, 163, 3681, 2271, 2795, 753, 48019, 129507}},
	{9047, 17, 28274, {1, 1, 1, 5, 31, 51, 127, 79, 333, 177, 1723, 1365, 2055, 3063, 10693, 61223, 60659}},
	{9048, 17, 28280, {1, 3, 7, 15, 9, 11, 11, 223, 31, 397, 319, 3283, 3765, 4729, 4711, 58323, 114063}},
	{9049, 17, 28296, {1, 1, 7, 11, 7, 63, 107, 215, 19, 491, 131, 2491, 6373, 11081, 2159, 1311, 109547}},
	{9050, 17, 28316, {1, 3, 5, 5, 21, 13, 105, 21, 193, 447, 1331, 2439, 4165, 15689, 21273, 4007, 55161}},
	{9051, 17, 28319, {1, 3, 5, 11, 19, 47, 25, 211, 335, 437, 1041, 2093, 7239, 2869, 18273, 40505, 13681}},
	{9052, 17, 28343, {1, 3, 3, 15, 7, 13, 127, 59, 439, 163, 1841, 1945, 4915, 16269, 18315, 15057, 43197}},
	{9053, 17, 28344, {1, 1, 3, 3, 15, 33, 101, 241, 131, 353, 1749, 3965, 1231, 8167, 9309, 40337, 4419}},
	{9054, 17, 28357, {1, 3, 1, 1, 1, 43, 33, 129, 137, 889, 799, 2937, 3633, 4769, 2411, 56059, 585}},
	{9055, 17, 28358, {1, 1, 3, 9, 25, 31, 45, 199, 229, 175, 1099, 1143, 1721, 11811, 22757, 59843, 117813}},
	{9056, 17, 28364, {1, 1, 7, 7, 19, 45, 43, 101, 219, 209, 1169, 599, 5523, 2463, 15161, 16675, 85111}},
	{9057, 17, 28370, {1, 1, 3, 15, 23, 27, 91, 51, 397, 705, 651, 2345, 3875, 10005, 29523, 42805, 76891}},
	{9058, 17, 28372, {1, 3, 3, 5, 29, 49, 17, 233, 149, 821, 1953, 1931, 7127, 957, 6477, 21259, 126657}},
	{9059, 17, 28379, {1, 3, 3, 3, 27, 49, 57, 145, 143, 1, 583, 3987, 651, 12285, 20139, 51063, 21449}},
	{9060, 17, 28388, {1, 1, 3, 5, 29, 61, 41, 93, 277, 111, 395, 2929, 5325, 15183, 3981, 23799, 72781}},
	{9061, 17, 28392, {1, 1, 7, 5, 25, 43, 85, 137, 401, 261, 1183, 3959, 1983, 16209, 30523, 429, 109181}},
	{9062, 17, 28395, {1, 1, 1, 5, 7, 19, 79, 237, 373, 929, 907, 1771, 6991, 211, 5269, 2135, 32051}},
	{9063, 17, 28406, {1, 1, 7, 5, 17, 15, 41, 49, 363, 15, 1483, 1017, 2439, 11713, 19983, 26275, 11945}},
	{9064, 17, 28418, {1, 1, 7, 7, 19, 5, 55, 15, 15, 573, 1075, 3225, 6815, 11893, 18417, 50833, 71903}},
	{9065, 17, 28423, {1, 1, 7, 9, 23, 37, 75, 3, 477, 291, 37, 1861, 2697, 13369, 24573, 27285, 96757}},
	{9066, 17, 28435, {1, 1, 1, 13, 5, 29, 65, 195, 365, 465, 865, 2705, 5249, 7051, 3795, 56611, 72317}},
	{9067, 17, 28444, {1, 1, 3, 9, 19, 9, 85, 239, 509, 313, 1137, 2221, 5475, 71, 11901, 1877, 68701}},
	{9068, 17, 28454, {1, 3, 3, 7, 3, 53, 55, 223, 441, 159, 933, 2573, 3441, 3295, 25005, 29021, 97145}},
	{9069, 17, 28468, {1, 3, 5, 5, 3, 11, 101, 181, 293, 319, 47, 2971, 387, 4697, 26613, 8531, 20461}},
	{9070, 17, 28477, {1, 3, 3, 13, 17, 11, 41, 233, 455, 353, 1817, 3065, 4657, 1717, 3039, 10937, 107085}},
	{9071, 17, 28478, {1, 1, 7, 1, 17, 23, 85, 5, 291, 725, 1791, 3525, 7705, 6561, 25311, 44679, 21419}},
	{9072, 17, 28480, {1, 1, 7, 1, 23, 41, 97, 117, 435, 261, 2007, 965, 6913, 12245, 25723, 8445, 30871}},
	{9073, 17, 28486, {1, 1, 1, 15, 29, 39, 101, 33, 55, 1019, 1431, 2601, 3837, 14873, 11785, 12449, 30815}},
	{9074, 17, 28489, {1, 1, 5, 3, 15, 35, 101, 7, 479, 535, 1875, 2435, 1461, 13967, 2755, 45879, 93561}},
	{9075, 17, 28500, {1, 3, 1, 5, 29, 57, 89, 209, 473, 289, 1843, 2051, 3997, 1753, 18179, 41355, 89301}},
	{9076, 17, 28507, {1, 3, 1, 11, 17, 45, 47, 57, 109, 309, 1009, 653, 5175, 15599, 21617, 35353, 55253}},
	{9077, 17, 28519, {1, 3, 5, 3, 1, 11, 57, 83, 385, 765, 1887, 785, 2115, 8689, 14783, 14841, 122221}},
	{9078, 17, 28523, {1, 3, 5, 11, 11, 5, 77, 115, 189, 371, 887, 3653, 8159, 15737, 6763, 52807, 128841}},
	{9079, 17, 28534, {1, 1, 1, 11, 11, 33, 9, 145, 439, 565, 171, 3867, 7149, 4369, 15073, 3277, 25873}},
	{9080, 17, 28547, {1, 1, 3, 1, 11, 9, 17, 255, 129, 835, 1705, 1551, 877, 4831, 12717, 2549, 62723}},
	{9081, 17, 28549, {1, 1, 7, 11, 17, 33, 21, 195, 143, 153, 1855, 1323, 1225, 16359, 16479, 8883, 76449}},
	{9082, 17, 28562, {1, 1, 1, 5, 31, 23, 61, 53, 77, 465, 1983, 4019, 4865, 14721, 18601, 48179, 100453}},
	{9083, 17, 28571, {1, 3, 1, 13, 19, 53, 83, 63, 165, 393, 469, 1465, 2669, 10155, 7029, 26185, 121223}},
	{9084, 17, 28583, {1, 1, 1, 3, 3, 3, 123, 23, 45, 359, 1063, 847, 3943, 6113, 23749, 30323, 10583}},
	{9085, 17, 28584, {1, 3, 5, 15, 1, 55, 65, 149, 139, 217, 141, 2425, 7019, 14127, 11725, 50821, 52643}},
	{9086, 17, 28595, {1, 3, 5, 15, 13, 13, 15, 93, 457, 869, 117, 585, 7159, 5957, 15073, 21861, 118119}},
	{9087, 17, 28597, {1, 3, 1, 15, 3, 31, 29, 245, 47, 895, 197, 1169, 945, 11503, 26615, 14079, 27175}},
	{9088, 17, 28604, {1, 3, 5, 5, 31, 1, 107, 109, 253, 999, 1451, 2243, 6675, 10779, 26181, 64597, 16443}},
	{9089, 17, 28607, {1, 3, 7, 15, 9, 53, 25, 41, 151, 197, 1955, 2365, 5305, 2901, 24825, 9595, 57377}},
	{9090, 17, 28609, {1, 3, 1, 3, 25, 37, 37, 193, 417, 373, 1127, 3239, 4583, 2861, 14501, 64163, 30055}},
	{9091, 17, 28612, {1, 3, 7, 9, 7, 21, 49, 231, 241, 95, 1757, 2281, 2679, 1611, 11115, 31743, 26851}},
	{9092, 17, 28630, {1, 3, 5, 5, 1, 1, 23, 173, 195, 593, 1639, 1449, 4733, 2451, 12677, 31959, 128217}},
	{9093, 17, 28640, {1, 1, 1, 7, 17, 49, 117, 253, 167, 721, 889, 3027, 7781, 13521, 15477, 17981, 95487}},
	{9094, 17, 28669, {1, 1, 1, 15, 13, 47, 125, 9, 33, 567, 1733, 1263, 307, 10285, 6325, 55827, 39823}},
	{9095, 17, 28677, {1, 1, 1, 15, 23, 3, 11, 169, 369, 667, 313, 2287, 6655, 16067, 5915, 8605, 92177}},
	{9096, 17, 28682, {1, 3, 3, 15, 13, 21, 125, 111, 171, 785, 79, 2281, 1247, 11321, 30397, 28555, 84863}},
	{9097, 17, 28690, {1, 1, 5, 13, 1, 31, 123, 97, 127, 245, 1213, 2381, 3545, 13545, 28657, 54087, 79039}},
	{9098, 17, 28695, {1, 1, 7, 9, 9, 21, 87, 111, 27, 33, 843, 199, 1465, 6555, 8019, 39521, 98883}},
	{9099, 17, 28696, {1, 3, 3, 5, 5, 55, 61, 219, 279, 207, 1811, 667, 2989, 3133, 25213, 51979, 87695}},
	{9100, 17, 28717, {1, 1, 7, 3, 17, 11, 31, 97, 277, 385, 229, 3045, 557, 13521, 32733, 36831, 43003}},
	{9101, 17, 28730, {1, 3, 5, 13, 27, 57, 31, 207, 147, 405, 1495, 2471, 4889, 14861, 4861, 28185, 62363}},
	{9102, 17, 28737, {1, 1, 5, 13, 23, 19, 5, 21, 509, 299, 1077, 1747, 6229, 2375, 17903, 58473, 72637}},
	{9103, 17, 28752, {1, 3, 7, 11, 15, 61, 63, 165, 27, 461, 1359, 3375, 3029, 9907, 17393, 11097, 43593}},
	{9104, 17, 28761, {1, 3, 5, 1, 17, 29, 15, 5, 419, 19, 1981, 3169, 2389, 9169, 31697, 26201, 6997}},
	{9105, 17, 28774, {1, 3, 3, 1, 29, 31, 89, 79, 347, 707, 505, 2087, 2163, 5465, 8677, 11421, 93217}},
	{9106, 17, 28783, {1, 3, 7, 9, 3, 23, 75, 215, 7, 971, 925, 3223, 7825, 12347, 19763, 10927, 41245}},
	{9107, 17, 28791, {1, 3, 5, 5, 3, 43, 119, 79, 373, 761, 709, 1897, 3953, 13895, 13421, 16939, 112449}},
	{9108, 17, 28801, {1, 1, 3, 15, 11, 25, 65, 101, 315, 1005, 1319, 1163, 161, 15331, 4845, 40375, 121361}},
	{9109, 17, 28802, {1, 1, 3, 7, 1, 57, 63, 131, 145, 1007, 549, 2327, 1513, 3591, 10839, 56297, 80613}},
	{9110, 17, 28814, {1, 1, 3, 5, 1, 19, 79, 81, 505, 945, 65, 319, 6105, 5491, 13257, 4651, 48247}},
	{9111, 17, 28837, {1, 3, 1, 9, 27, 41, 61, 41, 421, 707, 1279, 3699, 2403, 4075, 16947, 53435, 2917}},
	{9112, 17, 28838, {1, 1, 5, 13, 11, 29, 35, 141, 313, 769, 749, 4025, 2597, 12197, 32265, 32159, 37003}},
	{9113, 17, 28842, {1, 3, 7, 11, 25, 63, 121, 15, 273, 877, 637, 409, 5001, 4723, 27985, 55501, 43495}},
	{9114, 17, 28844, {1, 3, 1, 13, 27, 29, 127, 65, 275, 967, 1723, 4007, 6147, 13277, 8361, 12305, 95433}},
	{9115, 17, 28849, {1, 3, 3, 13, 11, 45, 7, 101, 169, 361, 517, 2897, 4283, 7587, 7495, 31549, 29113}},
	{9116, 17, 28864, {1, 3, 3, 1, 9, 27, 65, 15, 279, 127, 1039, 829, 5739, 1949, 24789, 30433, 54503}},
	{9117, 17, 28867, {1, 3, 7, 5, 13, 19, 95, 133, 25, 271, 1527, 3571, 101, 15987, 10985, 55761, 39833}},
	{9118, 17, 28881, {1, 3, 5, 9, 27, 5, 37, 219, 249, 947, 1063, 4081, 1763, 15003, 23753, 3975, 109803}},
	{9119, 17, 28882, {1, 3, 3, 5, 21, 37, 35, 145, 323, 573, 1939, 885, 4645, 4515, 16815, 28783, 76017}},
	{9120, 17, 28893, {1, 3, 7, 13, 27, 41, 39, 123, 423, 949, 1487, 2207, 8069, 15337, 20671, 20163, 70667}},
	{9121, 17, 28903, {1, 1, 3, 3, 15, 29, 69, 15, 151, 729, 35, 2067, 6715, 12759, 27611, 54133, 16561}},
	{9122, 17, 28907, {1, 3, 7, 13, 21, 13, 7, 161, 327, 339, 535, 2059, 413, 11161, 18415, 12415, 63713}},
	{9123, 17, 28909, {1, 3, 5, 5, 23, 49, 9, 181, 417, 339, 1013, 1707, 5097, 13319, 18951, 56415, 14397}},
	{9124, 17, 28921, {1, 1, 5, 7, 29, 23, 117, 159, 287, 695, 71, 2393, 2655, 6417, 24349, 20441, 77987}},
	{9125, 17, 28927, {1, 1, 5, 7, 31, 23, 81, 125, 145, 141, 1459, 4095, 713, 1817, 9263, 21025, 91983}},
	{9126, 17, 28929, {1, 1, 1, 9, 17, 23, 91, 39, 459, 299, 1951, 3229, 6229, 3267, 15883, 31719, 96573}},
	{9127, 17, 28935, {1, 1, 1, 15, 3, 51, 9, 7, 455, 653, 1447, 153, 8117, 723, 2177, 9107, 7757}},
	{9128, 17, 28941, {1, 3, 1, 15, 27, 47, 49, 245, 499, 807, 175, 1653, 1693, 3931, 27479, 30095, 62353}},
	{9129, 17, 28942, {1, 1, 5, 5, 23, 7, 15, 193, 499, 193, 201, 2771, 4153, 11533, 25883, 23337, 126685}},
	{9130, 17, 28954, {1, 1, 1, 7, 9, 43, 125, 181, 425, 557, 1401, 2593, 1933, 6803, 20021, 32687, 126465}},
	{9131, 17, 28959, {1, 3, 3, 13, 27, 19, 99, 29, 395, 765, 1137, 2963, 7397, 9695, 19259, 27965, 83157}},
	{9132, 17, 28960, {1, 3, 7, 7, 17, 29, 7, 241, 5, 281, 1489, 1599, 5567, 4579, 7739, 41413, 110571}},
	{9133, 17, 28990, {1, 3, 3, 9, 7, 5, 83, 1, 231, 1003, 631, 2557, 831, 6495, 30409, 53519, 79331}},
	{9134, 17, 28992, {1, 1, 5, 1, 7, 49, 45, 241, 201, 551, 1645, 2003, 1455, 3317, 23639, 7841, 100765}},
	{9135, 17, 29004, {1, 3, 5, 13, 25, 47, 103, 37, 81, 263, 1143, 801, 5863, 6871, 8615, 57363, 90161}},
	{9136, 17, 29010, {1, 3, 7, 11, 27, 23, 119, 211, 473, 207, 605, 637, 3369, 7287, 27827, 25003, 65925}},
	{9137, 17, 29012, {1, 3, 1, 15, 27, 31, 97, 247, 75, 893, 1099, 3609, 6807, 4393, 10253, 62759, 89971}},
	{9138, 17, 29021, {1, 1, 7, 15, 27, 19, 43, 59, 419, 263, 387, 3193, 5589, 4197, 19143, 64749, 103093}},
	{9139, 17, 29035, {1, 1, 7, 11, 21, 51, 97, 227, 251, 869, 1927, 2331, 7741, 8207, 12885, 13267, 17945}},
	{9140, 17, 29040, {1, 1, 7, 7, 5, 53, 41, 147, 75, 709, 607, 1073, 2853, 8081, 12797, 5279, 86083}},
	{9141, 17, 29059, {1, 1, 5, 5, 15, 21, 77, 189, 269, 595, 197, 3117, 5073, 14277, 26867, 49505, 75755}},
	{9142, 17, 29068, {1, 3, 5, 15, 13, 55, 1, 223, 135, 367, 735, 3139, 4851, 9773, 11699, 19081, 26011}},
	{9143, 17, 29080, {1, 3, 5, 7, 9, 3, 89, 103, 321, 727, 1809, 3527, 6881, 2399, 13593, 27867, 16219}},
	{9144, 17, 29086, {1, 3, 3, 1, 23, 5, 53, 51, 403, 753, 2037, 1261, 7575, 2725, 11341, 18533, 98767}},
	{9145, 17, 29089, {1, 1, 1, 11, 1, 13, 37, 141, 477, 689, 1789, 1913, 5753, 6069, 6965, 55209, 77329}},
	{9146, 17, 29090, {1, 3, 7, 3, 17, 59, 79, 249, 381, 163, 1773, 1645, 7295, 2359, 21889, 28429, 117073}},
	{9147, 17, 29095, {1, 3, 5, 15, 7, 45, 59, 3, 93, 259, 1257, 2967, 1175, 10171, 873, 51423, 67437}},
	{9148, 17, 29113, {1, 1, 7, 13, 17, 33, 53, 217, 159, 683, 1169, 3363, 4591, 3959, 10089, 35443, 99677}},
	{9149, 17, 29127, {1, 3, 7, 9, 3, 1, 5, 171, 17, 635, 369, 1529, 4861, 4977, 29303, 42357, 69309}},
	{9150, 17, 29131, {1, 3, 7, 9, 21, 17, 77, 127, 105, 427, 525, 1123, 2365, 7487, 6315, 64773, 122747}},
	{9151, 17, 29148, {1, 1, 1, 15, 19, 63, 65, 83, 219, 987, 169, 2589, 3809, 8807, 22473, 6479, 44617}},
	{9152, 17, 29161, {1, 3, 1, 7, 11, 51, 107, 19, 379, 191, 1013, 3145, 1501, 11871, 14111, 18269, 98247}},
	{9153, 17, 29172, {1, 1, 7, 5, 17, 63, 23, 231, 423, 855, 1955, 907, 4553, 16173, 7701, 40329, 42047}},
	{9154, 17, 29179, {1, 3, 7, 1, 7, 45, 103, 127, 185, 721, 1035, 839, 691, 6823, 23819, 50781, 92767}},
	{9155, 17, 29188, {1, 1, 1, 3, 9, 21, 57, 253, 285, 85, 1227, 365, 2347, 7491, 15183, 8619, 108819}},
	{9156, 17, 29192, {1, 1, 3, 15, 27, 13, 5, 85, 45, 1009, 1315, 1749, 2797, 3941, 19367, 50855, 60693}},
	{9157, 17, 29195, {1, 3, 5, 15, 29, 63, 115, 197, 317, 601, 711, 377, 7489, 4247, 4843, 56549, 108447}},
	{9158, 17, 29206, {1, 3, 7, 15, 11, 25, 7, 145, 371, 395, 1743, 267, 2609, 15707, 13909, 55031, 71115}},
	{9159, 17, 29210, {1, 3, 1, 1, 1, 53, 111, 245, 433, 309, 245, 15, 2091, 9051, 11095, 31821, 104535}},
	{9160, 17, 29215, {1, 3, 1, 15, 25, 31, 99, 89, 259, 595, 1095, 3681, 5105, 8671, 23663, 29717, 126429}},
	{9161, 17, 29221, {1, 3, 7, 7, 5, 31, 15, 59, 109, 527, 257, 1785, 6799, 1283, 11741, 34589, 102397}},
	{9162, 17, 29222, {1, 3, 3, 15, 9, 27, 55, 35, 291, 587, 1281, 779, 4615, 373, 24037, 64671, 124019}},
	{9163, 17, 29236, {1, 1, 5, 5, 13, 51, 49, 19, 37, 857, 539, 1291, 6021, 8645, 30351, 33839, 111515}},
	{9164, 17, 29246, {1, 1, 5, 13, 3, 47, 9, 197, 19, 337, 2025, 3003, 7179, 5755, 31187, 59317, 69753}},
	{9165, 17, 29248, {1, 1, 7, 3, 3, 43, 11, 3, 123, 29, 857, 3349, 791, 11157, 23967, 33729, 28445}},
	{9166, 17, 29253, {1, 1, 5, 1, 1, 11, 73, 231, 173, 925, 331, 161, 3303, 11015, 17507, 21271, 56865}},
	{9167, 17, 29272, {1, 1, 3, 9, 21, 21, 115, 145, 421, 981, 1789, 3343, 7591, 12043, 5795, 17737, 106501}},
	{9168, 17, 29288, {1, 1, 7, 13, 7, 15, 51, 75, 497, 637, 1073, 1505, 5613, 1415, 30861, 26159, 79573}},
	{9169, 17, 29293, {1, 1, 3, 15, 17, 35, 17, 129, 169, 283, 1383, 149, 211, 1381, 22205, 28367, 831}},
	{9170, 17, 29294, {1, 3, 5, 5, 17, 11, 127, 183, 503, 499, 2011, 2721, 2717, 3105, 4731, 60319, 9361}},
	{9171, 17, 29308, {1, 1, 1, 7, 27, 55, 77, 203, 255, 761, 1159, 2915, 4479, 13671, 19757, 65497, 4461}},
	{9172, 17, 29318, {1, 3, 1, 9, 9, 51, 67, 205, 445, 35, 371, 1837, 3623, 10365, 19463, 59005, 3185}},
	{9173, 17, 29321, {1, 3, 7, 3, 23, 5, 51, 141, 293, 489, 263, 2187, 1259, 2729, 1779, 61027, 53931}},
	{9174, 17, 29322, {1, 1, 1, 15, 27, 7, 15, 109, 475, 839, 175, 953, 4531, 437, 22475, 24167, 19051}},
	{9175, 17, 29339, {1, 1, 7, 9, 25, 23, 41, 107, 299, 115, 1713, 1559, 5701, 5427, 28813, 39913, 15941}},
	{9176, 17, 29365, {1, 3, 5, 7, 5, 25, 99, 9, 307, 591, 1303, 3501, 1589, 12095, 26629, 52127, 60635}},
	{9177, 17, 29369, {1, 3, 7, 3, 9, 23, 3, 29, 113, 49, 1601, 541, 1415, 11601, 19165, 46595, 111623}},
	{9178, 17, 29372, {1, 1, 3, 11, 5, 53, 37, 153, 51, 41, 1267, 545, 2055, 13137, 7749, 30721, 119591}},
	{9179, 17, 29377, {1, 3, 5, 11, 1, 15, 65, 17, 155, 65, 745, 2547, 6351, 2347, 13553, 5785, 129857}},
	{9180, 17, 29392, {1, 3, 7, 13, 5, 53, 11, 59, 453, 467, 1275, 3669, 4481, 5229, 2953, 23369, 100161}},
	{9181, 17, 29401, {1, 3, 1, 1, 13, 41, 91, 179, 331, 547, 1571, 1787, 6467, 12375, 4579, 45023, 119149}},
	{9182, 17, 29402, {1, 3, 3, 5, 17, 55, 105, 57, 227, 323, 1517, 1215, 3149, 13919, 18595, 5525, 82445}},
	{9183, 17, 29418, {1, 3, 1, 13, 19, 23, 17, 239, 81, 481, 1625, 2003, 7295, 2185, 7021, 19357, 37867}},
	{9184, 17, 29437, {1, 3, 5, 9, 11, 15, 61, 223, 153, 139, 2023, 2579, 495, 14319, 2835, 26541, 113115}},
	{9185, 17, 29445, {1, 1, 3, 13, 29, 9, 13, 149, 325, 87, 697, 2345, 2205, 5069, 9939, 61351, 127313}},
	{9186, 17, 29446, {1, 1, 7, 11, 13, 53, 45, 197, 167, 551, 439, 3715, 4587, 8549, 28193, 35827, 96721}},
	{9187, 17, 29460, {1, 3, 1, 1, 17, 7, 31, 205, 219, 739, 1165, 243, 3877, 15943, 15207, 43857, 120565}},
	{9188, 17, 29467, {1, 3, 7, 9, 7, 43, 81, 203, 295, 553, 279, 2717, 9, 751, 24715, 21591, 11485}},
	{9189, 17, 29473, {1, 1, 1, 11, 15, 17, 41, 121, 355, 157, 955, 3875, 7595, 235, 4937, 20607, 11401}},
	{9190, 17, 29476, {1, 1, 7, 7, 13, 49, 33, 161, 65, 251, 1895, 2665, 3017, 9725, 10797, 46313, 43407}},
	{9191, 17, 29488, {1, 1, 3, 9, 23, 17, 127, 69, 385, 875, 461, 3305, 3001, 5875, 13547, 61239, 113571}},
	{9192, 17, 29511, {1, 3, 7, 5, 15, 47, 113, 131, 465, 89, 733, 433, 799, 5689, 723, 63479, 106945}},
	{9193, 17, 29512, {1, 3, 3, 15, 29, 1, 51, 115, 317, 1021, 1219, 1797, 4005, 10435, 28935, 49467, 66833}},
	{9194, 17, 29517, {1, 1, 3, 11, 15, 9, 51, 209, 477, 479, 1099, 2781, 5525, 12715, 9067, 18317, 121671}},
	{9195, 17, 29523, {1, 3, 1, 7, 19, 35, 61, 27, 479, 815, 1639, 2825, 7449, 13697, 3079, 49833, 35119}},
	{9196, 17, 29526, {1, 3, 7, 3, 17, 53, 95, 155, 505, 185, 717, 3419, 3857, 2369, 14525, 22797, 38553}},
	{9197, 17, 29535, {1, 3, 1, 13, 27, 5, 11, 21, 507, 65, 39, 2841, 7887, 2783, 18767, 34171, 40527}},
	{9198, 17, 29539, {1, 3, 1, 7, 9, 47, 69, 217, 251, 775, 631, 1967, 5541, 10679, 16439, 33533, 57817}},
	{9199, 17, 29545, {1, 1, 5, 11, 27, 57, 103, 255, 359, 745, 63, 3725, 4113, 10943, 7833, 46857, 99239}},
	{9200, 17, 29559, {1, 3, 1, 5, 31, 41, 69, 245, 401, 451, 959, 351, 6999, 6187, 21437, 55067, 53547}},
	{9201, 17, 29560, {1, 1, 1, 13, 25, 49, 85, 181, 457, 731, 743, 1901, 7013, 12027, 14729, 24193, 89685}},
	{9202, 17, 29589, {1, 3, 3, 1, 31, 29, 101, 137, 117, 135, 345, 1419, 7133, 2695, 3631, 53049, 45875}},
	{9203, 17, 29593, {1, 1, 1, 13, 11, 51, 95, 221, 339, 655, 201, 3007, 8179, 8093, 22399, 59123, 127319}},
	{9204, 17, 29599, {1, 3, 7, 3, 31, 37, 23, 199, 191, 649, 817, 1959, 893, 2333, 21477, 29087, 115667}},
	{9205, 17, 29603, {1, 3, 3, 5, 9, 55, 123, 67, 39, 533, 797, 2575, 3955, 14585, 28587, 13079, 60053}},
	{9206, 17, 29610, {1, 3, 1, 1, 17, 19, 15, 219, 185, 21, 967, 667, 3361, 3883, 8059, 26199, 80913}},
	{9207, 17, 29629, {1, 3, 3, 11, 23, 5, 99, 57, 379, 151, 271, 3735, 7087, 12731, 2949, 54831, 37835}},
	{9208, 17, 29644, {1, 3, 1, 7, 21, 25, 9, 195, 497, 585, 901, 19, 7675, 13611, 31155, 14567, 20545}},
	{9209, 17, 29656, {1, 1, 3, 3, 27, 45, 51, 169, 397, 531, 673, 2935, 3779, 7169, 23479, 16157, 100237}},
	{9210, 17, 29662, {1, 1, 1, 1, 19, 49, 3, 75, 455, 805, 591, 1929, 2883, 2797, 31379, 12025, 120929}},
	{9211, 17, 29675, {1, 3, 5, 3, 17, 39, 115, 93, 341, 329, 1857, 2753, 4923, 12539, 25589, 19437, 29027}},
	{9212, 17, 29692, {1, 3, 5, 9, 27, 37, 21, 235, 499, 369, 1341, 3719, 6819, 3153, 30619, 50901, 52999}},
	{9213, 17, 29704, {1, 3, 1, 11, 3, 55, 43, 219, 83, 771, 783, 3569, 7879, 2363, 30605, 5965, 126855}},
	{9214, 17, 29707, {1, 3, 7, 13, 7, 25, 111, 63, 355, 317, 1579, 1523, 2733, 11963, 25205, 20545, 67389}},
	{9215, 17, 29715, {1, 3, 5, 7, 3, 17, 55, 99, 321, 633, 2013, 1991, 1019, 9223, 21117, 23337, 90589}},
	{9216, 17, 29717, {1, 3, 1, 1, 17, 25, 79, 171, 303, 403, 2037, 2595, 3951, 8021, 8669, 9363, 20345}},
	{9217, 17, 29722, {1, 1, 1, 7, 13, 7, 11, 7, 347, 53, 1763, 3097, 3353, 3769, 22947, 20919, 92247}},
	{9218, 17, 29731, {1, 3, 5, 11, 19, 29, 91, 191, 511, 705, 1317, 3367, 7, 4999, 30251, 18299, 66983}},
	{9219, 17, 29740, {1, 1, 3, 7, 19, 17, 71, 77, 285, 189, 853, 2305, 4205, 15603, 15501, 48073, 11411}},
	{9220, 17, 29743, {1, 3, 7, 5, 21, 15, 47, 13, 277, 969, 1861, 3493, 6723, 11521, 22145, 43779, 44713}},
	{9221, 17, 29752, {1, 1, 3, 5, 19, 47, 51, 207, 229, 957, 709, 267, 8081, 611, 26403, 14871, 111841}},
	{9222, 17, 29760, {1, 3, 7, 1, 19, 43, 71, 73, 405, 351, 1131, 3527, 5949, 14363, 20041, 48123, 68123}},
	{9223, 17, 29766, {1, 3, 3, 7, 23, 51, 81, 13, 161, 453, 365, 1089, 3505, 12359, 14277, 56113, 19717}},
	{9224, 17, 29770, {1, 1, 1, 7, 29, 35, 103, 137, 317, 417, 1465, 2787, 6935, 9885, 12943, 43937, 28353}},
	{9225, 17, 29794, {1, 1, 3, 13, 19, 37, 97, 5, 115, 89, 1005, 3033, 2011, 2633, 10615, 6641, 73385}},
	{9226, 17, 29796, {1, 3, 7, 7, 31, 39, 107, 165, 61, 1009, 1159, 865, 3469, 11093, 10425, 43959, 37395}},
	{9227, 17, 29814, {1, 3, 5, 9, 7, 51, 99, 91, 37, 457, 39, 2455, 7481, 4929, 29755, 50603, 48943}},
	{9228, 17, 29817, {1, 1, 5, 13, 5, 39, 47, 149, 341, 303, 843, 3619, 7527, 8739, 5893, 42623, 99899}},
	{9229, 17, 29829, {1, 1, 1, 1, 1, 41, 73, 71, 409, 351, 131, 515, 6657, 337, 23913, 583, 21665}},
	{9230, 17, 29841, {1, 1, 3, 3, 11, 45, 39, 113, 315, 965, 1605, 2779, 501, 7429, 2567, 7011, 71445}},
	{9231, 17, 29851, {1, 3, 7, 13, 21, 13, 45, 105, 385, 281, 1683, 3997, 6391, 10943, 22349, 37261, 57555}},
	{9232, 17, 29860, {1, 1, 3, 5, 17, 55, 109, 71, 393, 561, 433, 1091, 1923, 13861, 12981, 5523, 15467}},
	{9233, 17, 29864, {1, 3, 7, 5, 17, 11, 113, 119, 37, 989, 1609, 2191, 1511, 11835, 25423, 793, 15227}},
	{9234, 17, 29869, {1, 3, 1, 5, 5, 55, 105, 225, 349, 351, 1259, 1309, 821, 2733, 1357, 3665, 38863}},
	{9235, 17, 29870, {1, 3, 5, 1, 23, 61, 49, 113, 169, 319, 85, 1581, 97, 5271, 30625, 37693, 7297}},
	{9236, 17, 29889, {1, 3, 5, 1, 1, 25, 31, 125, 307, 731, 1815, 1047, 7251, 12481, 20781, 63275, 51985}},
	{9237, 17, 29896, {1, 3, 5, 9, 11, 9, 121, 111, 45, 751, 793, 2593, 6409, 4355, 30183, 36959, 105161}},
	{9238, 17, 29901, {1, 1, 3, 9, 25, 37, 95, 253, 401, 481, 1521, 3555, 231, 15459, 1581, 36661, 121727}},
	{9239, 17, 29913, {1, 3, 5, 3, 27, 11, 107, 115, 213, 813, 27, 1789, 603, 383, 1129, 63365, 51147}},
	{9240, 17, 29919, {1, 1, 3, 13, 25, 7, 33, 1, 97, 907, 35, 4069, 5001, 507, 911, 62037, 22019}},
	{9241, 17, 29920, {1, 1, 3, 5, 7, 55, 35, 95, 261, 217, 1565, 3473, 3475, 12181, 569, 27389, 81771}},
	{9242, 17, 29947, {1, 1, 3, 5, 11, 33, 95, 121, 453, 711, 361, 3927, 5231, 15685, 31143, 56915, 23707}},
	{9243, 17, 29958, {1, 1, 3, 11, 25, 15, 53, 155, 469, 647, 1547, 335, 3753, 12873, 13639, 25129, 79287}},
	{9244, 17, 29975, {1, 3, 3, 3, 1, 21, 21, 121, 105, 903, 83, 2287, 4295, 14369, 29183, 26841, 38115}},
	{9245, 17, 29981, {1, 1, 7, 11, 5, 29, 65, 191, 389, 419, 1315, 739, 3485, 10587, 2399, 36377, 28789}},
	{9246, 17, 29985, {1, 3, 3, 11, 3, 29, 71, 169, 265, 747, 395, 1211, 3487, 15705, 25611, 18183, 85771}},
	{9247, 17, 29998, {1, 1, 7, 3, 23, 5, 45, 47, 337, 571, 33, 1221, 5671, 1233, 28361, 36945, 911}},
	{9248, 17, 30010, {1, 1, 5, 11, 17, 15, 57, 97, 185, 999, 1277, 3371, 2785, 3341, 13395, 11925, 86777}},
	{9249, 17, 30012, {1, 1, 3, 1, 31, 37, 23, 105, 503, 869, 1309, 3733, 4629, 8263, 11763, 30669, 26179}},
	{9250, 17, 30044, {1, 1, 3, 15, 25, 61, 37, 27, 325, 413, 809, 2959, 8137, 3397, 21185, 27995, 84297}},
	{9251, 17, 30048, {1, 3, 5, 5, 1, 55, 95, 41, 493, 469, 331, 1789, 7037, 7947, 14239, 16109, 51795}},
	{9252, 17, 30051, {1, 1, 1, 1, 19, 33, 111, 237, 261, 331, 871, 3539, 1731, 6953, 11345, 37901, 5623}},
	{9253, 17, 30063, {1, 1, 5, 7, 21, 41, 49, 179, 49, 797, 231, 1299, 145, 7743, 725, 60595, 19581}},
	{9254, 17, 30065, {1, 3, 7, 15, 7, 43, 67, 219, 133, 641, 1657, 2301, 1591, 12309, 6395, 3999, 92961}},
	{9255, 17, 30072, {1, 1, 1, 5, 1, 49, 37, 81, 219, 323, 461, 1379, 1797, 14825, 21811, 7347, 35643}},
	{9256, 17, 30077, {1, 1, 1, 11, 1, 3, 83, 31, 307, 83, 1169, 3277, 1875, 13397, 20265, 46707, 15205}},
	{9257, 17, 30087, {1, 3, 7, 11, 29, 41, 69, 33, 405, 991, 1937, 1217, 2137, 8657, 4319, 41119, 43371}},
	{9258, 17, 30088, {1, 3, 3, 3, 25, 49, 107, 197, 347, 923, 1585, 3023, 4087, 13875, 22015, 35733, 33755}},
	{9259, 17, 30101, {1, 3, 3, 5, 15, 61, 89, 249, 141, 853, 1469, 999, 7425, 10015, 12341, 51535, 61619}},
	{9260, 17, 30112, {1, 1, 7, 13, 31, 61, 89, 113, 117, 429, 1011, 1589, 1419, 5083, 4843, 26759, 47401}},
	{9261, 17, 30121, {1, 1, 7, 9, 17, 37, 119, 39, 499, 93, 1155, 3069, 2033, 12483, 29849, 40077, 11103}},
	{9262, 17, 30122, {1, 3, 1, 15, 11, 5, 23, 121, 283, 717, 1573, 3911, 2031, 2617, 20387, 33157, 301}},
	{9263, 17, 30130, {1, 3, 5, 7, 17, 61, 109, 3, 205, 617, 1171, 223, 6609, 15027, 2629, 15801, 73749}},
	{9264, 17, 30135, {1, 1, 3, 7, 5, 49, 9, 73, 333, 401, 675, 2765, 993, 77, 19237, 60929, 88703}},
	{9265, 17, 30139, {1, 1, 1, 9, 21, 25, 53, 249, 241, 43, 1959, 545, 3729, 11395, 3027, 12661, 87729}},
	{9266, 17, 30142, {1, 3, 3, 15, 15, 61, 33, 59, 155, 773, 1043, 3111, 6549, 5397, 29099, 57851, 107671}},
	{9267, 17, 30164, {1, 3, 7, 1, 29, 29, 1, 161, 273, 883, 1913, 2389, 401, 9425, 17613, 50443, 84601}},
	{9268, 17, 30167, {1, 3, 7, 11, 15, 63, 41, 53, 371, 153, 1491, 3013, 6635, 4955, 30145, 20175, 16541}},
	{9269, 17, 30180, {1, 3, 3, 11, 31, 27, 127, 207, 11, 313, 1067, 3445, 3075, 4071, 28305, 58911, 85273}},
	{9270, 17, 30202, {1, 1, 7, 11, 17, 47, 77, 119, 209, 657, 1181, 459, 5821, 4267, 5757, 53703, 35621}},
	{9271, 17, 30211, {1, 3, 1, 15, 27, 41, 3, 217, 457, 531, 1749, 2847, 4715, 11451, 25071, 53999, 93503}},
	{9272, 17, 30214, {1, 3, 5, 3, 19, 29, 9, 177, 355, 265, 883, 1113, 2397, 1819, 20757, 50389, 95551}},
	{9273, 17, 30217, {1, 1, 3, 15, 3, 45, 85, 211, 377, 293, 1791, 1193, 1117, 9383, 28039, 27155, 129513}},
	{9274, 17, 30223, {1, 1, 7, 5, 1, 17, 59, 215, 161, 933, 1653, 2407, 2693, 3655, 7515, 2239, 88985}},
	{9275, 17, 30228, {1, 1, 3, 11, 31, 3, 1, 77, 459, 817, 651, 603, 1711, 9391, 20607, 48195, 127153}},
	{9276, 17, 30235, {1, 1, 5, 13, 11, 49, 25, 13, 51, 443, 1877, 1257, 163, 4673, 30313, 18841, 24547}},
	{9277, 17, 30256, {1, 1, 7, 7, 15, 33, 43, 79, 127, 625, 1991, 1311, 2095, 14659, 3477, 56023, 57955}},
	{9278, 17, 30265, {1, 3, 7, 15, 29, 7, 119, 183, 123, 323, 1723, 959, 2733, 2097, 2927, 57595, 86067}},
	{9279, 17, 30286, {1, 3, 5, 5, 29, 57, 93, 139, 495, 739, 1715, 713, 243, 2027, 11223, 44143, 119155}},
	{9280, 17, 30293, {1, 3, 1, 9, 3, 63, 113, 29, 19, 439, 869, 1101, 4947, 2191, 14737, 57049, 93505}},
	{9281, 17, 30298, {1, 1, 7, 1, 11, 39, 27, 29, 281, 829, 1979, 2185, 2207, 14969, 7447, 55541, 59593}},
	{9282, 17, 30310, {1, 1, 7, 13, 11, 15, 15, 143, 383, 469, 1439, 2823, 7489, 7675, 15433, 26203, 80433}},
	{9283, 17, 30314, {1, 1, 1, 7, 15, 45, 23, 93, 477, 1, 1431, 3173, 7879, 12211, 13051, 56971, 114289}},
	{9284, 17, 30327, {1, 3, 1, 1, 27, 55, 61, 185, 323, 569, 1063, 1357, 7373, 14947, 15967, 64517, 104625}},
	{9285, 17, 30333, {1, 1, 5, 11, 25, 1, 127, 163, 419, 657, 911, 361, 3675, 10765, 24565, 2661, 61979}},
	{9286, 17, 30334, {1, 3, 7, 15, 29, 53, 29, 149, 465, 535, 1865, 2243, 4783, 9327, 24843, 52313, 15683}},
	{9287, 17, 30337, {1, 1, 7, 7, 17, 7, 85, 187, 91, 1013, 1917, 2959, 3571, 12047, 25267, 34095, 9877}},
	{9288, 17, 30338, {1, 1, 1, 7, 5, 27, 111, 107, 313, 571, 1081, 3193, 1025, 2589, 1523, 40453, 77065}},
	{9289, 17, 30344, {1, 3, 7, 1, 19, 27, 1, 103, 479, 405, 583, 1737, 3495, 9093, 20397, 16429, 45609}},
	{9290, 17, 30349, {1, 1, 3, 11, 17, 1, 125, 97, 261, 651, 901, 1245, 1181, 14469, 16229, 31935, 100227}},
	{9291, 17, 30352, {1, 1, 7, 11, 15, 1, 19, 151, 453, 833, 1371, 1109, 5373, 25, 5619, 58351, 26349}},
	{9292, 17, 30362, {1, 1, 7, 15, 17, 55, 51, 67, 123, 13, 1873, 1035, 5871, 11943, 11543, 43261, 62587}},
	{9293, 17, 30374, {1, 3, 1, 13, 21, 15, 83, 205, 333, 379, 2021, 1389, 861, 10395, 20587, 38207, 49109}},
	{9294, 17, 30380, {1, 1, 5, 3, 13, 49, 89, 85, 463, 1005, 1367, 3487, 581, 12145, 22445, 35343, 65745}},
	{9295, 17, 30388, {1, 3, 3, 3, 29, 27, 99, 195, 89, 793, 1677, 3989, 4811, 8303, 9165, 50349, 96947}},
	{9296, 17, 30391, {1, 1, 3, 5, 29, 11, 91, 107, 13, 659, 213, 1613, 2245, 11567, 28157, 2937, 53275}},
	{9297, 17, 30395, {1, 3, 7, 5, 3, 41, 65, 27, 93, 747, 1143, 505, 3881, 2123, 2903, 54137, 96185}},
	{9298, 17, 30403, {1, 1, 3, 15, 9, 49, 3, 25, 77, 681, 1709, 915, 2243, 2127, 18243, 13915, 67399}},
	{9299, 17, 30409, {1, 3, 5, 7, 13, 49, 89, 67, 63, 271, 1651, 897, 4035, 1067, 13743, 56791, 44371}},
	{9300, 17, 30424, {1, 1, 7, 9, 25, 19, 125, 15, 125, 705, 1359, 817, 1241, 12447, 19097, 33209, 89091}},
	{9301, 17, 30427, {1, 3, 7, 3, 19, 43, 127, 197, 39, 709, 257, 3547, 3069, 1187, 21255, 6453, 40763}},
	{9302, 17, 30430, {1, 3, 3, 5, 3, 53, 37, 65, 415, 183, 991, 533, 7805, 9905, 18925, 52665, 100987}},
	{9303, 17, 30451, {1, 3, 3, 11, 17, 41, 17, 137, 143, 277, 945, 1531, 7427, 7287, 30869, 27651, 116507}},
	{9304, 17, 30460, {1, 1, 1, 3, 13, 3, 9, 163, 113, 373, 1909, 1051, 97, 10729, 28615, 40081, 129297}},
	{9305, 17, 30475, {1, 3, 3, 9, 11, 47, 113, 27, 307, 339, 1319, 3083, 7383, 1551, 26691, 28769, 114313}},
	{9306, 17, 30480, {1, 3, 7, 7, 25, 49, 31, 231, 485, 629, 59, 283, 7463, 6603, 23055, 63643, 10121}},
	{9307, 17, 30485, {1, 3, 7, 9, 5, 55, 53, 127, 147, 103, 1697, 485, 7051, 14153, 21631, 35561, 10393}},
	{9308, 17, 30486, {1, 3, 3, 7, 23, 7, 83, 17, 135, 487, 315, 719, 7003, 3919, 13255, 24031, 110493}},
	{9309, 17, 30489, {1, 3, 1, 15, 27, 55, 121, 177, 205, 733, 933, 1535, 2925, 4259, 22203, 59059, 89209}},
	{9310, 17, 30492, {1, 1, 1, 11, 9, 11, 127, 47, 493, 349, 1415, 3089, 4739, 14347, 31579, 20739, 72997}},
	{9311, 17, 30499, {1, 3, 1, 7, 9, 31, 121, 111, 163, 733, 1699, 1507, 5467, 13499, 25269, 6303, 70201}},
	{9312, 17, 30511, {1, 3, 1, 5, 7, 23, 123, 203, 329, 387, 577, 2331, 2283, 14029, 19409, 103, 2477}},
	{9313, 17, 30533, {1, 1, 7, 15, 11, 29, 29, 153, 289, 333, 1669, 2065, 5465, 8835, 28753, 21209, 34283}},
	{9314, 17, 30534, {1, 3, 5, 11, 15, 33, 45, 81, 241, 461, 1167, 1073, 5511, 795, 30955, 49121, 42805}},
	{9315, 17, 30540, {1, 1, 1, 13, 7, 33, 11, 201, 161, 475, 1359, 2329, 177, 9883, 8967, 21399, 73045}},
	{9316, 17, 30545, {1, 3, 3, 1, 25, 59, 85, 51, 481, 751, 1213, 3019, 421, 9903, 30071, 50661, 94715}},
	{9317, 17, 30552, {1, 3, 3, 1, 5, 61, 3, 179, 131, 233, 1627, 3577, 6323, 14161, 21595, 44963, 20215}},
	{9318, 17, 30557, {1, 1, 1, 5, 9, 53, 45, 105, 275, 769, 105, 2757, 6769, 14987, 19955, 18291, 81707}},
	{9319, 17, 30558, {1, 1, 7, 1, 1, 59, 33, 19, 385, 775, 423, 1799, 1325, 13545, 16027, 58347, 102397}},
	{9320, 17, 30567, {1, 1, 3, 11, 15, 61, 63, 59, 355, 659, 1483, 3781, 7383, 5563, 32537, 34175, 40303}},
	{9321, 17, 30568, {1, 3, 5, 7, 5, 37, 19, 223, 323, 129, 287, 2655, 3767, 16201, 4147, 315, 54885}},
	{9322, 17, 30581, {1, 1, 7, 13, 13, 23, 43, 129, 405, 205, 1691, 2189, 3313, 11789, 10263, 16367, 65547}},
	{9323, 17, 30597, {1, 3, 5, 1, 15, 21, 85, 233, 427, 71, 475, 3731, 3335, 11483, 28613, 4335, 35669}},
	{9324, 17, 30607, {1, 1, 3, 3, 31, 47, 27, 109, 373, 451, 1459, 3103, 1941, 10405, 20233, 30517, 122787}},
	{9325, 17, 30621, {1, 3, 1, 7, 3, 11, 113, 49, 355, 465, 1959, 1355, 6521, 10863, 11481, 13385, 31787}},
	{9326, 17, 30632, {1, 3, 1, 1, 9, 45, 125, 69, 267, 413, 717, 2767, 833, 317, 23019, 37753, 3081}},
	{9327, 17, 30650, {1, 1, 7, 7, 13, 55, 75, 105, 71, 505, 239, 3739, 4873, 4257, 18841, 41711, 24045}},
	{9328, 17, 30655, {1, 1, 5, 13, 21, 59, 107, 229, 421, 441, 1079, 3727, 7941, 8443, 30433, 56419, 105751}},
	{9329, 17, 30657, {1, 1, 1, 9, 15, 59, 29, 49, 117, 1009, 1971, 115, 2899, 1069, 29145, 11855, 35277}},
	{9330, 17, 30660, {1, 3, 7, 15, 19, 55, 111, 77, 169, 537, 1695, 2687, 3871, 14017, 15119, 27313, 112807}},
	{9331, 17, 30669, {1, 1, 3, 7, 29, 5, 41, 201, 211, 127, 1877, 643, 2441, 8743, 29393, 6077, 52597}},
	{9332, 17, 30675, {1, 3, 1, 11, 7, 1, 95, 15, 229, 339, 475, 3463, 7827, 9943, 30903, 65013, 1145}},
	{9333, 17, 30678, {1, 1, 1, 5, 23, 19, 81, 23, 475, 169, 373, 1147, 1805, 12779, 13173, 8945, 28175}},
	{9334, 17, 30682, {1, 3, 5, 9, 3, 53, 127, 33, 237, 803, 121, 307, 4981, 8765, 12761, 23601, 92921}},
	{9335, 17, 30687, {1, 1, 3, 7, 17, 63, 11, 37, 213, 619, 1095, 1693, 6747, 7373, 17355, 5987, 97923}},
	{9336, 17, 30688, {1, 1, 3, 15, 11, 37, 109, 175, 503, 339, 591, 2745, 2387, 7419, 13915, 4769, 48229}},
	{9337, 17, 30698, {1, 3, 5, 7, 19, 17, 5, 81, 471, 989, 249, 437, 7309, 5747, 25277, 31911, 87907}},
	{9338, 17, 30711, {1, 3, 1, 7, 15, 25, 49, 243, 423, 911, 1957, 911, 6331, 9831, 26021, 58877, 99257}},
	{9339, 17, 30728, {1, 3, 5, 11, 3, 55, 39, 129, 271, 145, 1973, 3391, 2905, 9229, 7989, 15641, 67933}},
	{9340, 17, 30733, {1, 3, 5, 9, 13, 13, 43, 135, 183, 319, 1391, 2953, 2207, 14205, 31203, 6329, 98907}},
	{9341, 17, 30741, {1, 3, 1, 9, 27, 41, 27, 155, 11, 191, 1747, 975, 7043, 13139, 30387, 47099, 120321}},
	{9342, 17, 30746, {1, 1, 5, 9, 25, 27, 53, 235, 437, 77, 371, 2413, 4867, 14245, 27199, 37387, 88217}},
	{9343, 17, 30752, {1, 1, 7, 7, 11, 59, 103, 15, 109, 65, 1987, 3695, 7737, 7341, 26963, 16075, 6301}},
	{9344, 17, 30764, {1, 3, 7, 5, 7, 59, 109, 159, 121, 377, 1851, 3983, 5421, 7633, 7321, 14869, 54633}},
	{9345, 17, 30769, {1, 1, 3, 15, 21, 51, 35, 243, 397, 411, 1107, 3689, 7913, 14715, 26349, 23361, 90665}},
	{9346, 17, 30784, {1, 1, 1, 3, 5, 11, 77, 205, 187, 981, 1969, 1749, 6295, 8267, 16073, 54451, 103603}},
	{9347, 17, 30796, {1, 3, 3, 9, 11, 13, 113, 83, 243, 1021, 2003, 2277, 6457, 10535, 13461, 52741, 9385}},
	{9348, 17, 30799, {1, 3, 3, 11, 19, 9, 103, 13, 219, 269, 1805, 2689, 5219, 11497, 4909, 57985, 40141}},
	{9349, 17, 30804, {1, 1, 1, 1, 29, 25, 15, 217, 69, 567, 839, 1515, 3627, 9837, 21433, 37177, 10487}},
	{9350, 17, 30811, {1, 1, 7, 15, 15, 23, 119, 217, 277, 447, 551, 825, 7571, 3083, 16573, 1189, 64959}},
	{9351, 17, 30814, {1, 1, 3, 11, 9, 13, 63, 77, 313, 195, 941, 1621, 165, 8905, 20265, 53761, 25091}},
	{9352, 17, 30818, {1, 3, 3, 9, 17, 5, 9, 183, 175, 1015, 253, 233, 2883, 15587, 27175, 38517, 22707}},
	{9353, 17, 30827, {1, 3, 3, 11, 23, 63, 83, 17, 49, 671, 749, 3249, 7821, 7189, 26735, 28995, 101737}},
	{9354, 17, 30838, {1, 1, 7, 5, 25, 15, 97, 247, 161, 585, 1307, 3803, 1105, 9093, 23523, 1383, 65671}},
	{9355, 17, 30842, {1, 1, 3, 15, 29, 51, 65, 237, 349, 709, 799, 1425, 6267, 6283, 4773, 18123, 74833}},
	{9356, 17, 30854, {1, 1, 5, 5, 11, 13, 9, 251, 373, 189, 467, 945, 7279, 11349, 10917, 6581, 83967}},
	{9357, 17, 30863, {1, 1, 7, 15, 23, 27, 1, 197, 41, 325, 757, 1229, 6295, 345, 26147, 40135, 123063}},
	{9358, 17, 30865, {1, 1, 7, 9, 23, 9, 93, 225, 191, 837, 103, 3367, 5411, 8289, 7057, 55391, 10669}},
	{9359, 17, 30877, {1, 1, 5, 15, 21, 17, 49, 221, 487, 23, 1943, 1563, 6157, 4035, 27769, 14933, 56913}},
	{9360, 17, 30881, {1, 1, 5, 13, 13, 43, 67, 245, 457, 365, 1115, 2205, 6229, 4173, 25167, 56333, 55605}},
	{9361, 17, 30887, {1, 1, 5, 11, 15, 59, 109, 83, 17, 913, 497, 1299, 5221, 321, 32139, 13717, 94311}},
	{9362, 17, 30908, {1, 3, 3, 3, 31, 11, 5, 203, 3, 843, 2039, 25, 6211, 14927, 6015, 46269, 90369}},
	{9363, 17, 30916, {1, 1, 3, 9, 21, 51, 91, 203, 149, 147, 197, 1771, 2921, 6609, 2343, 35249, 12963}},
	{9364, 17, 30919, {1, 3, 1, 7, 17, 43, 91, 229, 107, 521, 737, 2355, 5855, 6707, 21217, 47041, 81833}},
	{9365, 17, 30925, {1, 3, 3, 7, 7, 31, 97, 135, 503, 665, 1799, 2937, 6867, 4125, 7375, 34401, 18111}},
	{9366, 17, 30928, {1, 1, 7, 1, 11, 29, 89, 185, 495, 633, 507, 3727, 5999, 5871, 5911, 24877, 10259}},
	{9367, 17, 30944, {1, 1, 1, 13, 25, 3, 25, 65, 91, 411, 147, 3699, 7003, 3017, 25635, 56619, 101491}},
	{9368, 17, 30947, {1, 3, 5, 7, 31, 1, 63, 255, 115, 179, 87, 735, 1649, 4767, 31093, 60149, 49829}},
	{9369, 17, 30950, {1, 1, 3, 1, 21, 63, 67, 85, 399, 279, 1963, 1759, 4679, 15423, 11533, 54387, 36563}},
	{9370, 17, 30968, {1, 3, 5, 3, 31, 53, 73, 73, 481, 443, 1393, 2763, 1199, 5375, 8977, 5369, 114603}},
	{9371, 17, 30971, {1, 1, 1, 3, 29, 47, 73, 205, 469, 187, 815, 2787, 1431, 4705, 11455, 53643, 89269}},
	{9372, 17, 30973, {1, 3, 3, 9, 27, 57, 93, 55, 287, 539, 1259, 3279, 1563, 11399, 22975, 27077, 41031}},
	{9373, 17, 30976, {1, 1, 3, 15, 7, 27, 67, 25, 169, 263, 621, 1921, 509, 11715, 15363, 27447, 75535}},
	{9374, 17, 30986, {1, 1, 1, 9, 29, 63, 31, 99, 321, 361, 1693, 763, 5593, 10815, 741, 31257, 70843}},
	{9375, 17, 30993, {1, 1, 1, 9, 1, 17, 73, 141, 401, 549, 415, 1289, 1697, 1903, 8919, 59563, 60017}},
	{9376, 17, 30994, {1, 3, 7, 3, 5, 51, 127, 221, 9, 929, 153, 1045, 6587, 13653, 5343, 14043, 116125}},
	{9377, 17, 31012, {1, 3, 3, 13, 13, 17, 29, 93, 465, 59, 1207, 3121, 6331, 8647, 5047, 41869, 51969}},
	{9378, 17, 31016, {1, 1, 1, 15, 23, 29, 47, 149, 119, 855, 367, 1419, 7739, 1141, 19787, 38185, 84403}},
	{9379, 17, 31029, {1, 3, 1, 9, 29, 63, 5, 63, 435, 401, 1023, 1981, 6819, 7547, 30065, 33833, 7471}},
	{9380, 17, 31039, {1, 3, 1, 15, 1, 47, 35, 161, 243, 225, 935, 2179, 7737, 7841, 28523, 11505, 103741}},
	{9381, 17, 31041, {1, 1, 7, 3, 7, 57, 73, 55, 101, 905, 1705, 4057, 3781, 8213, 18997, 17185, 33265}},
	{9382, 17, 31042, {1, 1, 5, 1, 7, 57, 31, 77, 323, 395, 927, 1969, 6973, 9013, 30789, 757, 84075}},
	{9383, 17, 31044, {1, 1, 3, 7, 15, 53, 51, 205, 401, 679, 1295, 1955, 7739, 11423, 23207, 55449, 60419}},
	{9384, 17, 31053, {1, 3, 5, 11, 23, 21, 67, 141, 157, 767, 219, 3607, 1847, 11051, 31499, 8461, 106353}},
	{9385, 17, 31059, {1, 1, 3, 9, 17, 19, 123, 169, 1, 31, 1019, 1823, 6043, 1895, 17293, 62079, 16945}},
	{9386, 17, 31062, {1, 3, 5, 9, 3, 15, 49, 27, 183, 293, 989, 2161, 1845, 1103, 20149, 11121, 31935}},
	{9387, 17, 31077, {1, 3, 1, 3, 17, 39, 103, 45, 491, 91, 413, 487, 1381, 5457, 22503, 40477, 94297}},
	{9388, 17, 31095, {1, 1, 3, 7, 29, 11, 87, 79, 349, 437, 945, 3753, 6691, 4373, 24875, 54397, 33697}},
	{9389, 17, 31101, {1, 1, 7, 1, 9, 31, 105, 121, 97, 947, 129, 1909, 2371, 5493, 29523, 52685, 24325}},
	{9390, 17, 31105, {1, 1, 5, 9, 19, 21, 63, 115, 511, 525, 49, 1879, 6075, 8181, 10135, 56785, 53309}},
	{9391, 17, 31118, {1, 1, 5, 15, 3, 55, 75, 135, 451, 697, 1407, 2765, 2443, 11589, 24863, 47187, 98477}},
	{9392, 17, 31129, {1, 1, 1, 13, 27, 37, 77, 157, 121, 603, 491, 2201, 619, 9157, 32511, 19843, 49919}},
	{9393, 17, 31132, {1, 1, 3, 1, 23, 17, 23, 115, 119, 349, 987, 2587, 1847, 12099, 31955, 31685, 1989}},
	{9394, 17, 31141, {1, 3, 7, 7, 5, 47, 63, 209, 69, 921, 1041, 1391, 7485, 11121, 30993, 5691, 74551}},
	{9395, 17, 31159, {1, 3, 1, 3, 23, 61, 55, 253, 355, 299, 971, 279, 3543, 10073, 5199, 50539, 88303}},
	{9396, 17, 31183, {1, 1, 1, 11, 13, 19, 7, 255, 189, 267, 2021, 93, 219, 10537, 28627, 58141, 53675}},
	{9397, 17, 31185, {1, 3, 3, 7, 27, 61, 83, 95, 163, 777, 1533, 2485, 7211, 6979, 4013, 20797, 91411}},
	{9398, 17, 31195, {1, 1, 7, 13, 15, 37, 5, 109, 133, 225, 59, 3855, 3351, 659, 24321, 63531, 15573}},
	{9399, 17, 31202, {1, 1, 5, 1, 7, 55, 59, 213, 45, 77, 2003, 2921, 1105, 11089, 17197, 45459, 67681}},
	{9400, 17, 31213, {1, 1, 1, 5, 13, 21, 107, 245, 505, 409, 1453, 1201, 6945, 2103, 7377, 59413, 8785}},
	{9401, 17, 31238, {1, 1, 1, 13, 5, 37, 73, 55, 39, 219, 225, 3877, 6583, 3105, 25355, 14839, 23435}},
	{9402, 17, 31241, {1, 1, 7, 1, 21, 35, 87, 227, 487, 767, 609, 1685, 2731, 10135, 381, 24021, 122137}},
	{9403, 17, 31252, {1, 1, 1, 3, 29, 13, 19, 255, 355, 505, 1757, 3537, 3029, 11403, 22685, 61169, 397}},
	{9404, 17, 31262, {1, 1, 7, 1, 29, 43, 11, 207, 83, 97, 435, 1453, 4709, 4193, 18517, 47203, 3255}},
	{9405, 17, 31265, {1, 1, 1, 1, 21, 49, 39, 163, 459, 849, 561, 1207, 4109, 1435, 17519, 14839, 1331}},
	{9406, 17, 31295, {1, 1, 3, 11, 27, 35, 65, 219, 135, 559, 1111, 2959, 7835, 5165, 26641, 22765, 121829}},
	{9407, 17, 31300, {1, 3, 5, 3, 23, 31, 57, 149, 431, 451, 189, 1771, 5877, 3503, 7531, 46485, 129031}},
	{9408, 17, 31303, {1, 1, 3, 11, 1, 13, 47, 17, 331, 1003, 215, 2797, 689, 6289, 12719, 37139, 35827}},
	{9409, 17, 31317, {1, 3, 5, 9, 19, 13, 11, 29, 275, 165, 783, 313, 2153, 6009, 2247, 5699, 128753}},
	{9410, 17, 31318, {1, 1, 7, 13, 5, 43, 51, 75, 411, 743, 335, 217, 559, 15389, 6567, 41193, 127443}},
	{9411, 17, 31324, {1, 3, 5, 15, 5, 63, 7, 145, 445, 835, 825, 35, 5951, 5121, 16365, 36789, 2941}},
	{9412, 17, 31338, {1, 3, 5, 5, 29, 1, 61, 19, 427, 245, 445, 3505, 3647, 8817, 8031, 64577, 60745}},
	{9413, 17, 31340, {1, 1, 3, 9, 29, 9, 35, 225, 55, 535, 1537, 831, 6483, 16123, 26079, 32809, 62227}},
	{9414, 17, 31345, {1, 3, 7, 7, 25, 33, 15, 61, 343, 749, 1963, 2763, 3171, 6755, 6529, 49449, 88903}},
	{9415, 17, 31355, {1, 1, 7, 13, 17, 35, 91, 119, 87, 1023, 1101, 1785, 2005, 15947, 21679, 63179, 3389}},
	{9416, 17, 31362, {1, 3, 1, 1, 1, 1, 123, 195, 315, 681, 153, 1621, 5097, 3669, 20505, 39305, 127065}},
	{9417, 17, 31371, {1, 1, 5, 11, 1, 17, 73, 251, 185, 59, 1723, 2321, 2103, 6331, 29571, 63811, 66651}},
	{9418, 17, 31373, {1, 1, 7, 13, 15, 19, 111, 91, 211, 85, 711, 2197, 3107, 2717, 16725, 52995, 65791}},
	{9419, 17, 31381, {1, 3, 3, 9, 21, 41, 53, 145, 459, 155, 93, 2833, 6747, 737, 30625, 40581, 65825}},
	{9420, 17, 31391, {1, 3, 3, 3, 1, 45, 119, 81, 185, 431, 1221, 3043, 7277, 10537, 12355, 42261, 126117}},
	{9421, 17, 31409, {1, 3, 7, 7, 11, 47, 37, 41, 123, 643, 707, 2963, 6183, 15527, 10951, 24031, 38187}},
	{9422, 17, 31410, {1, 3, 1, 7, 13, 57, 1, 149, 117, 627, 1999, 2805, 4857, 12805, 31453, 25699, 109447}},
	{9423, 17, 31412, {1, 3, 5, 3, 9, 37, 83, 221, 77, 573, 661, 465, 1279, 7355, 24061, 36151, 96595}},
	{9424, 17, 31434, {1, 1, 7, 15, 29, 31, 125, 205, 449, 563, 1263, 3427, 8013, 14025, 15235, 11833, 25601}},
	{9425, 17, 31458, {1, 3, 7, 11, 31, 35, 99, 193, 163, 527, 1455, 395, 4853, 2561, 11909, 57311, 101007}},
	{9426, 17, 31467, {1, 1, 5, 3, 17, 39, 99, 173, 497, 245, 1671, 3457, 83, 11959, 2963, 3401, 102259}},
	{9427, 17, 31470, {1, 1, 1, 5, 25, 41, 119, 81, 301, 797, 661, 2543, 1195, 2111, 1785, 41533, 51947}},
	{9428, 17, 31475, {1, 3, 3, 13, 5, 59, 61, 153, 213, 541, 1849, 249, 3897, 3877, 17095, 6857, 76781}},
	{9429, 17, 31481, {1, 3, 7, 1, 19, 13, 57, 47, 359, 165, 1085, 2263, 3261, 12825, 17405, 25853, 20731}},
	{9430, 17, 31482, {1, 1, 1, 7, 7, 43, 7, 65, 51, 503, 173, 1023, 283, 14809, 1183, 33497, 110683}},
	{9431, 17, 31484, {1, 3, 5, 11, 19, 51, 29, 157, 159, 191, 1293, 2951, 6569, 12433, 14587, 30631, 30485}},
	{9432, 17, 31492, {1, 3, 7, 5, 1, 27, 25, 221, 255, 471, 779, 3991, 6985, 1803, 28451, 33403, 5567}},
	{9433, 17, 31507, {1, 1, 5, 5, 7, 29, 55, 241, 457, 863, 1715, 3393, 4127, 13985, 6313, 13683, 114837}},
	{9434, 17, 31514, {1, 3, 5, 5, 11, 27, 55, 109, 247, 199, 1593, 2881, 307, 97, 24751, 35921, 121931}},
	{9435, 17, 31538, {1, 3, 1, 13, 3, 59, 17, 161, 47, 467, 1019, 3629, 3017, 15645, 3983, 32393, 79213}},
	{9436, 17, 31547, {1, 1, 3, 11, 19, 57, 67, 199, 319, 107, 2043, 2045, 4025, 5733, 29979, 37721, 117031}},
	{9437, 17, 31549, {1, 3, 7, 11, 9, 23, 31, 81, 177, 801, 1177, 3451, 7777, 15351, 7579, 39033, 23847}},
	{9438, 17, 31555, {1, 1, 3, 5, 17, 61, 63, 7, 371, 905, 1147, 1383, 4075, 6721, 17503, 32015, 112547}},
	{9439, 17, 31557, {1, 1, 3, 13, 13, 25, 69, 159, 49, 133, 227, 2155, 1603, 10077, 3429, 39131, 18949}},
	{9440, 17, 31597, {1, 3, 5, 3, 29, 5, 115, 93, 243, 791, 1113, 2841, 4733, 3041, 31733, 28539, 84567}},
	{9441, 17, 31598, {1, 3, 3, 7, 21, 9, 5, 95, 489, 517, 1453, 2697, 7951, 12369, 19571, 29811, 51805}},
	{9442, 17, 31610, {1, 1, 5, 9, 1, 29, 97, 191, 73, 357, 745, 2787, 7815, 4565, 19761, 33729, 86849}},
	{9443, 17, 31625, {1, 3, 5, 13, 3, 5, 35, 79, 387, 813, 1673, 3187, 337, 5539, 6761, 46903, 122967}},
	{9444, 17, 31634, {1, 1, 7, 11, 1, 15, 125, 175, 255, 35, 145, 2391, 887, 10505, 11587, 53941, 5089}},
	{9445, 17, 31643, {1, 1, 7, 13, 9, 13, 15, 215, 361, 227, 1665, 3345, 3615, 14031, 16281, 4457, 52037}},
	{9446, 17, 31645, {1, 3, 5, 9, 31, 21, 3, 189, 211, 855, 1781, 2097, 1345, 6763, 27651, 54137, 52689}},
	{9447, 17, 31659, {1, 3, 1, 5, 29, 9, 99, 183, 183, 205, 149, 53, 7179, 3387, 9603, 4281, 47145}},
	{9448, 17, 31669, {1, 3, 1, 11, 13, 35, 97, 21, 29, 877, 191, 1621, 2501, 4283, 1707, 48957, 129029}},
	{9449, 17, 31670, {1, 3, 1, 9, 5, 19, 57, 219, 105, 467, 1179, 3155, 7743, 4835, 14845, 35671, 47655}},
	{9450, 17, 31682, {1, 3, 1, 7, 27, 41, 27, 185, 271, 611, 1173, 2875, 529, 11619, 20231, 18741, 41799}},
	{9451, 17, 31694, {1, 3, 7, 13, 9, 3, 35, 71, 467, 689, 1797, 319, 6657, 13193, 15861, 7567, 12891}},
	{9452, 17, 31717, {1, 1, 7, 13, 19, 57, 25, 141, 195, 995, 859, 811, 4685, 6711, 8963, 49657, 54751}},
	{9453, 17, 31718, {1, 1, 1, 11, 27, 25, 9, 91, 97, 251, 757, 2783, 5447, 3617, 26801, 32501, 55245}},
	{9454, 17, 31729, {1, 3, 7, 1, 5, 1, 103, 129, 127, 593, 857, 3957, 3665, 10279, 26211, 2095, 15869}},
	{9455, 17, 31736, {1, 1, 7, 1, 25, 49, 3, 139, 25, 545, 615, 1353, 4103, 1099, 21729, 45383, 110611}},
	{9456, 17, 31742, {1, 3, 5, 3, 7, 49, 83, 41, 209, 357, 939, 849, 5851, 3945, 831, 8131, 105897}},
	{9457, 17, 31749, {1, 1, 1, 3, 27, 19, 123, 71, 195, 1019, 1021, 1287, 5665, 5277, 8647, 27033, 89539}},
	{9458, 17, 31773, {1, 1, 1, 9, 27, 51, 49, 159, 401, 1013, 763, 653, 1449, 12441, 21191, 28871, 106181}},
	{9459, 17, 31777, {1, 1, 5, 7, 31, 7, 105, 137, 331, 367, 1305, 2761, 863, 3915, 12633, 32251, 82867}},
	{9460, 17, 31778, {1, 3, 7, 11, 9, 47, 35, 57, 137, 269, 443, 79, 11, 11817, 28995, 46681, 104263}},
	{9461, 17, 31784, {1, 3, 1, 5, 3, 25, 89, 179, 183, 835, 367, 2215, 295, 5365, 1899, 10785, 88979}},
	{9462, 17, 31801, {1, 3, 7, 13, 3, 5, 93, 43, 409, 363, 267, 2077, 3745, 445, 25957, 34103, 29475}},
	{9463, 17, 31812, {1, 1, 1, 7, 27, 21, 121, 29, 171, 783, 553, 265, 6835, 3929, 18127, 33463, 70999}},
	{9464, 17, 31821, {1, 3, 3, 15, 15, 55, 13, 1, 297, 935, 1307, 1779, 2239, 15471, 32453, 30649, 45973}},
	{9465, 17, 31822, {1, 3, 7, 5, 25, 41, 3, 171, 347, 607, 1873, 1087, 2433, 8377, 7959, 19941, 117319}},
	{9466, 17, 31836, {1, 1, 1, 3, 5, 47, 107, 69, 431, 63, 325, 1241, 3487, 11249, 28559, 30001, 93789}},
	{9467, 17, 31850, {1, 1, 1, 5, 15, 17, 9, 145, 335, 169, 1099, 3637, 5397, 6711, 16095, 27053, 124247}},
	{9468, 17, 31855, {1, 3, 3, 5, 3, 9, 65, 97, 421, 951, 2003, 2837, 7095, 15685, 5147, 56801, 98679}},
	{9469, 17, 31858, {1, 3, 7, 15, 1, 33, 115, 45, 215, 253, 361, 555, 787, 15483, 25531, 53273, 8933}},
	{9470, 17, 31860, {1, 3, 1, 9, 3, 63, 47, 205, 457, 977, 991, 3189, 1369, 14899, 10937, 56999, 11525}},
	{9471, 17, 31886, {1, 1, 7, 5, 11, 61, 53, 55, 231, 357, 1695, 2489, 2355, 7583, 14097, 50039, 96595}},
	{9472, 17, 31891, {1, 3, 7, 7, 3, 57, 115, 245, 259, 573, 1275, 2971, 1793, 13683, 8683, 51815, 26807}},
	{9473, 17, 31909, {1, 1, 5, 3, 17, 59, 55, 237, 491, 757, 1447, 2941, 2641, 14175, 4401, 4367, 36853}},
	{9474, 17, 31928, {1, 3, 1, 15, 3, 63, 67, 1, 403, 79, 1161, 2379, 3337, 14447, 5877, 40759, 12573}},
	{9475, 17, 31931, {1, 1, 7, 15, 17, 1, 91, 5, 173, 215, 1567, 1851, 3309, 9813, 21215, 19151, 96785}},
	{9476, 17, 31934, {1, 1, 1, 9, 31, 45, 123, 221, 397, 51, 1489, 3247, 923, 10423, 10461, 51231, 92909}},
	{9477, 17, 31941, {1, 1, 1, 13, 27, 17, 105, 163, 403, 193, 1487, 2421, 4415, 14303, 6419, 24105, 29997}},
	{9478, 17, 31942, {1, 1, 5, 13, 31, 55, 17, 125, 341, 219, 401, 1611, 891, 12909, 13949, 46245, 26769}},
	{9479, 17, 31945, {1, 3, 7, 3, 31, 41, 65, 207, 311, 643, 1617, 271, 3749, 14635, 26385, 55251, 50719}},
	{9480, 17, 31951, {1, 3, 3, 13, 7, 55, 69, 241, 413, 399, 137, 2255, 5395, 12625, 26583, 64603, 22571}},
	{9481, 17, 31959, {1, 3, 5, 3, 31, 15, 15, 161, 153, 445, 595, 273, 6631, 12845, 23331, 16963, 52099}},
	{9482, 17, 31963, {1, 3, 3, 1, 27, 39, 71, 41, 455, 841, 831, 1719, 3531, 5113, 29183, 1933, 42227}},
	{9483, 17, 31970, {1, 3, 7, 3, 1, 15, 31, 183, 429, 557, 1747, 1059, 2079, 16361, 29103, 43207, 921}},
	{9484, 17, 31984, {1, 3, 1, 1, 31, 39, 97, 73, 339, 405, 1423, 2215, 5435, 9205, 1889, 58249, 61517}},
	{9485, 17, 31987, {1, 3, 7, 1, 23, 59, 127, 245, 11, 627, 1555, 2497, 6427, 7205, 22675, 62847, 69691}},
	{9486, 17, 31990, {1, 1, 3, 5, 1, 13, 95, 9, 167, 481, 947, 3181, 8057, 5559, 7537, 33757, 72419}},
	{9487, 17, 32001, {1, 1, 7, 3, 15, 9, 105, 205, 287, 375, 115, 1731, 1063, 11551, 12077, 41013, 88853}},
	{9488, 17, 32007, {1, 3, 3, 9, 5, 63, 127, 33, 409, 279, 1379, 4069, 4091, 14703, 27435, 19525, 71261}},
	{9489, 17, 32008, {1, 3, 1, 13, 31, 31, 59, 205, 167, 131, 891, 1259, 6909, 211, 31517, 8085, 112065}},
	{9490, 17, 32025, {1, 1, 5, 11, 17, 25, 119, 77, 449, 569, 381, 825, 2459, 983, 2959, 51611, 90721}},
	{9491, 17, 32035, {1, 3, 1, 7, 17, 55, 91, 231, 133, 541, 499, 3609, 4237, 11627, 30007, 58911, 43443}},
	{9492, 17, 32038, {1, 3, 7, 7, 29, 5, 47, 187, 71, 695, 1389, 2855, 5815, 11605, 3643, 24961, 25793}},
	{9493, 17, 32047, {1, 3, 3, 5, 11, 31, 43, 31, 185, 1021, 795, 3585, 3981, 8627, 18117, 42351, 19513}},
	{9494, 17, 32049, {1, 1, 5, 13, 9, 3, 115, 45, 39, 577, 1847, 653, 2625, 9367, 27923, 35661, 113613}},
	{9495, 17, 32062, {1, 3, 7, 7, 17, 9, 69, 233, 367, 673, 11, 2215, 1177, 4501, 9693, 62013, 45647}},
	{9496, 17, 32067, {1, 3, 5, 7, 7, 53, 11, 227, 465, 843, 2017, 689, 6767, 10321, 25163, 56561, 6865}},
	{9497, 17, 32070, {1, 3, 3, 5, 13, 43, 119, 9, 185, 893, 133, 863, 7137, 6653, 7875, 23167, 13893}},
	{9498, 17, 32073, {1, 3, 5, 9, 1, 47, 17, 85, 273, 901, 493, 2411, 983, 15717, 25151, 21323, 57939}},
	{9499, 17, 32074, {1, 1, 7, 5, 19, 17, 49, 37, 425, 443, 781, 2593, 4929, 12313, 12727, 42285, 88451}},
	{9500, 17, 32079, {1, 3, 3, 11, 9, 53, 17, 67, 237, 463, 1509, 2153, 3715, 7909, 21151, 64517, 87695}},
	{9501, 17, 32081, {1, 3, 7, 1, 29, 39, 25, 83, 413, 1005, 2011, 3933, 2911, 7041, 10537, 23135, 22671}},
	{9502, 17, 32082, {1, 1, 3, 9, 23, 61, 117, 33, 431, 181, 1819, 683, 1809, 1723, 27041, 29113, 99347}},
	{9503, 17, 32107, {1, 1, 5, 11, 11, 7, 101, 181, 51, 857, 923, 3495, 7123, 7775, 30081, 48513, 116137}},
	{9504, 17, 32127, {1, 1, 3, 11, 15, 31, 97, 127, 365, 799, 715, 2101, 6081, 11607, 1055, 35027, 62967}},
	{9505, 17, 32145, {1, 3, 5, 7, 3, 31, 109, 247, 225, 221, 1093, 2633, 1847, 7427, 8767, 16581, 32145}},
	{9506, 17, 32151, {1, 3, 1, 7, 15, 23, 43, 109, 327, 417, 1895, 2333, 6265, 6599, 6623, 47375, 92731}},
	{9507, 17, 32152, {1, 3, 7, 1, 29, 29, 45, 217, 163, 941, 1327, 3685, 5481, 15783, 26281, 60339, 34277}},
	{9508, 17, 32173, {1, 1, 7, 11, 1, 7, 119, 201, 29, 193, 1805, 1395, 267, 2011, 637, 26765, 48883}},
	{9509, 17, 32174, {1, 1, 3, 7, 11, 63, 41, 89, 365, 729, 25, 3185, 2143, 1737, 29693, 7443, 78079}},
	{9510, 17, 32186, {1, 3, 1, 13, 25, 27, 63, 233, 79, 1007, 1357, 679, 7581, 8333, 2469, 31787, 128531}},
	{9511, 17, 32194, {1, 3, 1, 3, 23, 39, 53, 99, 219, 475, 931, 507, 3615, 10613, 14663, 1151, 123459}},
	{9512, 17, 32196, {1, 1, 1, 1, 13, 15, 67, 45, 393, 791, 415, 2731, 1151, 8935, 28983, 7239, 106247}},
	{9513, 17, 32200, {1, 3, 7, 7, 11, 35, 95, 153, 421, 193, 1997, 2587, 3183, 9229, 17663, 28221, 6759}},
	{9514, 17, 32208, {1, 3, 1, 7, 5, 5, 123, 55, 509, 973, 261, 463, 2723, 15225, 1925, 62283, 86329}},
	{9515, 17, 32218, {1, 1, 3, 13, 5, 47, 123, 239, 273, 407, 1725, 717, 1229, 1387, 11743, 13739, 104503}},
	{9516, 17, 32236, {1, 3, 3, 13, 23, 35, 43, 113, 299, 847, 1903, 3445, 3395, 641, 11271, 61517, 40747}},
	{9517, 17, 32260, {1, 3, 1, 15, 17, 49, 97, 9, 335, 731, 151, 167, 8129, 11845, 18285, 20113, 122397}},
	{9518, 17, 32263, {1, 1, 5, 11, 11, 63, 3, 153, 345, 511, 1939, 1815, 7231, 10555, 14293, 50753, 14681}},
	{9519, 17, 32288, {1, 3, 7, 5, 21, 31, 127, 223, 241, 783, 887, 3519, 4743, 3541, 4143, 57461, 27791}},
	{9520, 17, 32298, {1, 1, 5, 7, 13, 15, 83, 225, 201, 979, 145, 769, 1491, 12155, 21307, 64877, 113277}},
	{9521, 17, 32315, {1, 1, 7, 1, 27, 25, 105, 69, 239, 323, 1059, 573, 4913, 14215, 27007, 42351, 66315}},
	{9522, 17, 32332, {1, 1, 3, 11, 21, 33, 93, 23, 363, 633, 935, 637, 6171, 12695, 14077, 17505, 69681}},
	{9523, 17, 32340, {1, 3, 1, 5, 15, 11, 93, 211, 175, 377, 33, 1403, 5097, 1503, 8483, 2881, 85877}},
	{9524, 17, 32354, {1, 1, 5, 3, 5, 51, 5, 255, 429, 661, 625, 3015, 4813, 3573, 22917, 45967, 70559}},
	{9525, 17, 32359, {1, 1, 7, 3, 11, 41, 3, 197, 181, 897, 767, 1385, 7395, 15543, 4655, 40309, 73169}},
	{9526, 17, 32366, {1, 1, 5, 9, 15, 35, 71, 119, 509, 817, 1169, 75, 1337, 2959, 611, 38243, 46987}},
	{9527, 17, 32368, {1, 1, 1, 9, 1, 7, 43, 65, 479, 625, 1685, 1309, 5619, 14163, 13633, 18169, 8311}},
	{9528, 17, 32377, {1, 3, 5, 9, 19, 39, 95, 105, 273, 1023, 79, 229, 6895, 2931, 5717, 27911, 22139}},
	{9529, 17, 32384, {1, 3, 5, 7, 1, 55, 15, 15, 297, 731, 2029, 2789, 11, 1333, 26571, 62595, 15131}},
	{9530, 17, 32399, {1, 1, 5, 7, 29, 35, 3, 125, 381, 709, 2047, 2395, 6315, 2301, 7175, 19857, 75085}},
	{9531, 17, 32417, {1, 1, 5, 15, 23, 45, 95, 117, 49, 635, 1525, 1105, 7335, 4653, 18159, 29729, 62627}},
	{9532, 17, 32424, {1, 3, 3, 11, 29, 19, 29, 169, 141, 243, 1765, 1829, 4555, 16299, 3053, 58933, 44605}},
	{9533, 17, 32427, {1, 1, 3, 15, 5, 45, 35, 213, 385, 993, 1521, 9, 3561, 10497, 12601, 38163, 86501}},
	{9534, 17, 32429, {1, 3, 3, 13, 9, 23, 109, 95, 491, 1003, 473, 3325, 6577, 14617, 17765, 33391, 82927}},
	{9535, 17, 32438, {1, 3, 3, 11, 25, 31, 93, 111, 231, 71, 1233, 3581, 6789, 4569, 16741, 61967, 32249}},
	{9536, 17, 32442, {1, 3, 3, 15, 15, 63, 39, 247, 79, 923, 327, 2639, 2013, 12325, 18133, 60623, 2215}},
	{9537, 17, 32447, {1, 3, 5, 1, 5, 49, 121, 53, 283, 529, 37, 3233, 6285, 12447, 4355, 9343, 45631}},
	{9538, 17, 32469, {1, 1, 7, 11, 11, 11, 111, 139, 429, 279, 1019, 2139, 2033, 6809, 8847, 22535, 107005}},
	{9539, 17, 32479, {1, 3, 5, 1, 1, 21, 35, 97, 167, 57, 491, 511, 4065, 11699, 16851, 6847, 40929}},
	{9540, 17, 32483, {1, 3, 1, 15, 3, 55, 113, 33, 255, 537, 835, 1867, 3927, 839, 955, 29079, 93727}},
	{9541, 17, 32498, {1, 1, 7, 3, 5, 7, 35, 111, 165, 885, 115, 3051, 4541, 1701, 22827, 361, 91843}},
	{9542, 17, 32503, {1, 1, 7, 11, 7, 55, 81, 43, 237, 725, 1761, 1599, 639, 14189, 31241, 52827, 107943}},
	{9543, 17, 32507, {1, 3, 1, 3, 29, 35, 67, 119, 369, 877, 1861, 123, 8121, 13861, 31155, 60245, 79799}},
	{9544, 17, 32521, {1, 1, 3, 13, 7, 49, 63, 19, 253, 723, 639, 1677, 291, 13697, 22231, 46893, 90069}},
	{9545, 17, 32532, {1, 3, 5, 1, 7, 57, 29, 233, 35, 715, 515, 3221, 2715, 13839, 18321, 4445, 103843}},
	{9546, 17, 32539, {1, 3, 1, 7, 1, 63, 33, 7, 481, 461, 1923, 2679, 2441, 5449, 13233, 2245, 48667}},
	{9547, 17, 32551, {1, 1, 7, 11, 11, 9, 95, 151, 441, 333, 1871, 1181, 3027, 12887, 11923, 63847, 6953}},
	{9548, 17, 32572, {1, 3, 5, 5, 15, 33, 53, 47, 351, 387, 55, 393, 5475, 3027, 18565, 37997, 120877}},
	{9549, 17, 32577, {1, 3, 5, 9, 23, 43, 67, 97, 445, 783, 1499, 1977, 1441, 10159, 13479, 149, 4939}},
	{9550, 17, 32578, {1, 3, 7, 3, 15, 41, 119, 55, 139, 25, 849, 857, 53, 10421, 2683, 24839, 107797}},
	{9551, 17, 32587, {1, 1, 7, 13, 25, 51, 51, 13, 333, 93, 95, 1755, 3055, 12585, 3519, 44857, 11257}},
	{9552, 17, 32592, {1, 1, 5, 11, 29, 55, 13, 235, 419, 327, 823, 2675, 8031, 9303, 8749, 20215, 12111}},
	{9553, 17, 32602, {1, 1, 3, 5, 7, 31, 103, 19, 467, 255, 583, 419, 2845, 12179, 63, 51693, 9755}},
	{9554, 17, 32604, {1, 1, 1, 13, 15, 29, 109, 81, 381, 659, 601, 3867, 7663, 7307, 16445, 56327, 48559}},
	{9555, 17, 32613, {1, 3, 3, 15, 31, 35, 29, 153, 423, 247, 55, 3259, 6199, 4199, 13931, 14433, 52645}},
	{9556, 17, 32625, {1, 1, 5, 11, 9, 17, 17, 191, 231, 977, 721, 2817, 2485, 4965, 32341, 55131, 4547}},
	{9557, 17, 32631, {1, 1, 7, 7, 7, 7, 89, 69, 299, 503, 597, 311, 1321, 2335, 30193, 45347, 126631}},
	{9558, 17, 32641, {1, 1, 7, 11, 13, 43, 105, 153, 89, 229, 1573, 1549, 3699, 15981, 28911, 45011, 83759}},
	{9559, 17, 32642, {1, 3, 7, 3, 1, 3, 121, 137, 263, 325, 1449, 3793, 5795, 7715, 7449, 26453, 85081}},
	{9560, 17, 32644, {1, 3, 1, 7, 23, 15, 39, 217, 99, 873, 1641, 1411, 4627, 283, 20707, 41795, 62239}},
	{9561, 17, 32656, {1, 3, 5, 9, 15, 15, 35, 255, 501, 945, 79, 799, 2361, 4495, 27825, 27699, 129335}},
	{9562, 17, 32678, {1, 3, 1, 7, 9, 19, 89, 31, 65, 905, 1475, 1353, 7253, 12825, 20723, 47757, 12007}},
	{9563, 17, 32681, {1, 1, 3, 3, 15, 35, 83, 239, 463, 835, 1249, 2521, 3429, 14073, 13569, 6161, 71309}},
	{9564, 17, 32701, {1, 1, 7, 11, 31, 43, 15, 57, 461, 917, 339, 3787, 2925, 1879, 7217, 17091, 108819}},
	{9565, 17, 32713, {1, 3, 7, 3, 17, 51, 29, 105, 221, 941, 1291, 835, 1563, 15623, 2953, 62985, 63037}},
	{9566, 17, 32721, {1, 1, 7, 3, 1, 39, 83, 41, 399, 465, 587, 2011, 137, 6017, 5067, 52389, 71053}},
	{9567, 17, 32727, {1, 1, 7, 11, 17, 55, 103, 239, 173, 181, 1219, 2671, 5183, 3799, 19589, 31247, 68889}},
	{9568, 17, 32731, {1, 1, 3, 3, 21, 43, 123, 253, 281, 627, 353, 3077, 1685, 12143, 19723, 57775, 70761}},
	{9569, 17, 32734, {1, 1, 7, 15, 31, 13, 101, 159, 311, 305, 1783, 3523, 149, 9269, 7103, 40315, 30569}},
	{9570, 17, 32740, {1, 1, 5, 3, 29, 47, 11, 219, 301, 207, 1361, 563, 7831, 14469, 18983, 54535, 64647}},
	{9571, 17, 32773, {1, 1, 3, 15, 11, 37, 85, 237, 225, 1009, 1065, 985, 6849, 5395, 22853, 43965, 51363}},
	{9572, 17, 32774, {1, 3, 3, 1, 11, 61, 45, 131, 201, 609, 757, 2539, 3817, 9309, 24759, 26789, 41437}},
	{9573, 17, 32785, {1, 1, 7, 3, 21, 5, 19, 137, 75, 573, 583, 2499, 41, 3429, 24273, 36711, 110015}},
	{9574, 17, 32788, {1, 3, 7, 9, 1, 51, 39, 75, 115, 269, 1983, 2709, 6989, 6521, 5551, 43675, 1019}},
	{9575, 17, 32792, {1, 1, 3, 9, 27, 1, 125, 7, 67, 821, 275, 1253, 4635, 3557, 4155, 13831, 1523}},
	{9576, 17, 32797, {1, 1, 5, 15, 23, 15, 79, 43, 275, 791, 1867, 2495, 2933, 2167, 22819, 52913, 88871}},
	{9577, 17, 32801, {1, 1, 1, 5, 31, 59, 27, 153, 159, 919, 219, 3373, 3227, 6321, 27559, 33905, 126145}},
	{9578, 17, 32811, {1, 3, 3, 13, 23, 21, 119, 175, 119, 741, 1745, 3985, 3847, 5163, 13699, 32373, 75201}},
	{9579, 17, 32821, {1, 3, 7, 15, 1, 47, 101, 89, 425, 269, 713, 3587, 3373, 13315, 16481, 40031, 50353}},
	{9580, 17, 32828, {1, 3, 7, 3, 19, 29, 5, 69, 385, 979, 1893, 1849, 8007, 14415, 18343, 60555, 109117}},
	{9581, 17, 32839, {1, 1, 3, 13, 5, 35, 111, 239, 489, 395, 1565, 1607, 543, 89, 8971, 22311, 899}},
	{9582, 17, 32854, {1, 1, 7, 7, 11, 51, 105, 211, 341, 85, 991, 1275, 3995, 12611, 2363, 29501, 44217}},
	{9583, 17, 32867, {1, 1, 5, 13, 9, 17, 93, 69, 145, 917, 469, 1109, 7405, 12903, 8341, 50383, 20133}},
	{9584, 17, 32870, {1, 3, 1, 7, 27, 45, 45, 85, 101, 161, 1117, 2757, 7847, 359, 17155, 27073, 123535}},
	{9585, 17, 32873, {1, 3, 1, 3, 9, 11, 67, 205, 109, 257, 1635, 141, 3969, 11571, 211, 48683, 108671}},
	{9586, 17, 32881, {1, 1, 3, 7, 13, 9, 29, 251, 113, 851, 1549, 981, 5553, 6095, 28885, 32953, 112563}},
	{9587, 17, 32891, {1, 1, 5, 7, 11, 5, 13, 83, 343, 499, 587, 3887, 3859, 11459, 7361, 25665, 86151}},
	{9588, 17, 32900, {1, 1, 5, 1, 13, 43, 3, 37, 273, 749, 1707, 2069, 3083, 1095, 3081, 23919, 21939}},
	{9589, 17, 32903, {1, 3, 5, 13, 13, 49, 115, 99, 357, 95, 699, 2615, 1911, 12675, 8607, 12535, 118651}},
	{9590, 17, 32910, {1, 1, 7, 7, 29, 43, 17, 131, 271, 895, 1427, 3659, 1843, 8247, 1175, 48239, 54435}},
	{9591, 17, 32917, {1, 1, 1, 9, 1, 27, 85, 163, 353, 669, 745, 317, 2505, 7685, 14831, 31131, 106687}},
	{9592, 17, 32922, {1, 1, 7, 9, 1, 23, 121, 53, 289, 651, 303, 3049, 6819, 6733, 17485, 20023, 110009}},
	{9593, 17, 32928, {1, 3, 7, 3, 5, 47, 93, 75, 363, 479, 825, 1801, 6807, 3341, 6419, 9889, 5557}},
	{9594, 17, 32945, {1, 1, 3, 15, 23, 5, 7, 25, 73, 811, 1597, 2041, 6707, 6817, 20427, 50749, 46255}},
	{9595, 17, 32946, {1, 3, 7, 9, 1, 11, 61, 63, 435, 977, 1937, 93, 2685, 643, 20113, 25873, 63829}},
	{9596, 17, 32951, {1, 1, 3, 15, 5, 41, 31, 53, 143, 271, 27, 3899, 5045, 1063, 17229, 52715, 67689}},
	{9597, 17, 32958, {1, 1, 3, 11, 1, 57, 121, 13, 291, 861, 1547, 3899, 7949, 15401, 29807, 52307, 104359}},
	{9598, 17, 32965, {1, 3, 5, 15, 23, 3, 95, 43, 377, 437, 1687, 3075, 5131, 11791, 3637, 12621, 105575}},
	{9599, 17, 32978, {1, 3, 1, 3, 27, 1, 117, 11, 153, 401, 1971, 2097, 3227, 14603, 4757, 56281, 112263}},
	{9600, 17, 32980, {1, 3, 3, 5, 13, 25, 51, 209, 367, 327, 1941, 1943, 1347, 14393, 31997, 16001, 129047}},
	{9601, 17, 32983, {1, 1, 5, 11, 19, 51, 109, 229, 71, 923, 1741, 1193, 4657, 6043, 26703, 17757, 75009}},
	{9602, 17, 32987, {1, 1, 7, 3, 23, 3, 125, 165, 137, 999, 1583, 3493, 859, 15603, 7143, 28791, 28201}},
	{9603, 17, 33023, {1, 1, 5, 11, 29, 57, 65, 41, 295, 729, 635, 1871, 6347, 3509, 59, 40765, 42673}},
	{9604, 17, 33031, {1, 3, 3, 3, 15, 59, 53, 97, 15, 131, 891, 1105, 841, 6065, 14427, 4721, 106433}},
	{9605, 17, 33032, {1, 1, 1, 7, 19, 37, 101, 121, 141, 613, 1363, 691, 1731, 12477, 8339, 55669, 99379}},
	{9606, 17, 33035, {1, 3, 5, 13, 17, 49, 75, 25, 447, 113, 1853, 3465, 5225, 4531, 14287, 1039, 17399}},
	{9607, 17, 33038, {1, 3, 5, 3, 3, 49, 101, 79, 117, 939, 1161, 1991, 2343, 7183, 12599, 52877, 94337}},
	{9608, 17, 33040, {1, 3, 1, 1, 19, 47, 73, 195, 475, 435, 1807, 2723, 7885, 15469, 26057, 37325, 57005}},
	{9609, 17, 33043, {1, 1, 1, 11, 17, 7, 111, 143, 357, 977, 719, 553, 4559, 7225, 10405, 26895, 8385}},
	{9610, 17, 33050, {1, 3, 3, 9, 17, 5, 1, 73, 125, 913, 1275, 2387, 5153, 13611, 20585, 8465, 27545}},
	{9611, 17, 33059, {1, 1, 7, 5, 27, 51, 107, 147, 503, 699, 851, 1729, 2875, 16331, 28025, 26451, 92705}},
	{9612, 17, 33080, {1, 1, 5, 9, 3, 37, 21, 139, 13, 427, 225, 1345, 2491, 15495, 25847, 3095, 128879}},
	{9613, 17, 33098, {1, 1, 3, 11, 7, 47, 113, 133, 99, 871, 1151, 1953, 7931, 6389, 28715, 36861, 60017}},
	{9614, 17, 33108, {1, 1, 7, 1, 21, 47, 35, 83, 137, 945, 2047, 3491, 3719, 3001, 20563, 51243, 14491}},
	{9615, 17, 33115, {1, 1, 5, 15, 1, 13, 85, 61, 479, 853, 813, 805, 4931, 12651, 22757, 29531, 92861}},
	{9616, 17, 33117, {1, 3, 7, 7, 27, 63, 31, 169, 43, 185, 637, 729, 7231, 2381, 23539, 53885, 90215}},
	{9617, 17, 33133, {1, 1, 3, 13, 5, 51, 69, 111, 357, 277, 1889, 3809, 8031, 13341, 14261, 34001, 63317}},
	{9618, 17, 33134, {1, 1, 7, 3, 11, 59, 1, 43, 227, 503, 1407, 3917, 7077, 847, 4513, 53007, 66721}},
	{9619, 17, 33157, {1, 1, 5, 11, 15, 25, 109, 169, 25, 391, 597, 2997, 2377, 9045, 15239, 25291, 5451}},
	{9620, 17, 33169, {1, 3, 3, 11, 15, 11, 1, 59, 347, 707, 239, 2473, 8057, 4787, 32247, 17955, 79151}},
	{9621, 17, 33170, {1, 3, 7, 11, 9, 59, 9, 117, 137, 713, 451, 1105, 4485, 14979, 26271, 46017, 89211}},
	{9622, 17, 33176, {1, 3, 3, 3, 3, 19, 95, 131, 413, 291, 1179, 3265, 7107, 10419, 13527, 19905, 8059}},
	{9623, 17, 33182, {1, 3, 7, 9, 29, 43, 19, 243, 443, 27, 1401, 3469, 6925, 2833, 19715, 39667, 11983}},
	{9624, 17, 33192, {1, 3, 3, 7, 23, 33, 115, 59, 29, 61, 1085, 1115, 4007, 12673, 26479, 22397, 95609}},
	{9625, 17, 33205, {1, 3, 3, 5, 1, 47, 43, 83, 21, 621, 59, 1, 891, 12285, 31855, 48641, 52479}},
	{9626, 17, 33212, {1, 3, 3, 5, 3, 9, 17, 181, 15, 315, 1705, 2461, 1853, 14007, 17665, 40593, 126179}},
	{9627, 17, 33215, {1, 3, 5, 3, 3, 23, 83, 163, 29, 293, 1891, 2631, 2989, 7295, 2441, 21689, 8187}},
	{9628, 17, 33217, {1, 3, 1, 1, 1, 23, 53, 215, 185, 843, 1083, 2603, 3857, 4981, 25079, 20249, 93717}},
	{9629, 17, 33227, {1, 3, 5, 11, 7, 61, 127, 13, 449, 395, 1909, 3967, 2441, 3073, 8159, 33979, 26345}},
	{9630, 17, 33229, {1, 1, 5, 1, 15, 5, 93, 87, 319, 173, 1729, 1395, 1019, 5139, 10819, 29877, 81025}},
	{9631, 17, 33238, {1, 3, 3, 7, 17, 55, 61, 227, 299, 245, 849, 211, 895, 2999, 18215, 37069, 32821}},
	{9632, 17, 33241, {1, 1, 5, 3, 17, 49, 115, 55, 447, 533, 1463, 2983, 3245, 9345, 11955, 49145, 128035}},
	{9633, 17, 33260, {1, 3, 1, 7, 5, 17, 61, 71, 101, 529, 1761, 827, 7887, 5713, 31039, 18087, 82277}},
	{9634, 17, 33271, {1, 3, 1, 11, 27, 59, 1, 231, 303, 431, 1279, 3647, 1333, 3675, 29401, 55533, 65997}},
	{9635, 17, 33278, {1, 1, 5, 9, 7, 9, 111, 245, 269, 919, 1147, 1601, 6219, 4931, 3035, 12231, 4011}},
	{9636, 17, 33293, {1, 3, 5, 15, 3, 19, 83, 25, 129, 979, 79, 3027, 3983, 7703, 16859, 12085, 83115}},
	{9637, 17, 33294, {1, 1, 5, 11, 31, 41, 99, 3, 383, 943, 1579, 2435, 1209, 161, 31733, 11755, 95697}},
	{9638, 17, 33296, {1, 1, 1, 9, 9, 55, 115, 187, 499, 165, 1081, 813, 2545, 8065, 10501, 15475, 85107}},
	{9639, 17, 33302, {1, 1, 1, 3, 1, 31, 81, 213, 301, 575, 605, 543, 3347, 12759, 21645, 37173, 36127}},
	{9640, 17, 33305, {1, 3, 3, 9, 21, 29, 51, 91, 307, 617, 1839, 443, 1013, 4473, 3885, 57669, 123271}},
	{9641, 17, 33329, {1, 3, 1, 15, 31, 43, 83, 187, 51, 513, 1505, 3895, 3557, 9527, 27537, 6173, 99595}},
	{9642, 17, 33330, {1, 3, 3, 1, 3, 53, 113, 27, 431, 505, 219, 2143, 6691, 3219, 9589, 9885, 24037}},
	{9643, 17, 33332, {1, 1, 5, 9, 13, 3, 53, 145, 49, 411, 691, 289, 6443, 4963, 13815, 23663, 95497}},
	{9644, 17, 33354, {1, 3, 5, 9, 19, 7, 53, 101, 199, 69, 1821, 3233, 3267, 5947, 4869, 30095, 21255}},
	{9645, 17, 33383, {1, 1, 5, 11, 29, 7, 79, 11, 451, 585, 987, 2333, 1891, 1853, 14739, 34399, 62895}},
	{9646, 17, 33387, {1, 3, 1, 7, 29, 43, 103, 219, 139, 359, 1663, 3453, 7469, 1943, 11457, 19227, 62211}},
	{9647, 17, 33397, {1, 3, 3, 11, 9, 47, 17, 237, 87, 881, 583, 3473, 2579, 975, 1531, 50997, 76219}},
	{9648, 17, 33408, {1, 1, 7, 15, 31, 37, 79, 115, 95, 515, 2003, 2595, 4077, 4537, 9171, 31183, 41219}},
	{9649, 17, 33417, {1, 1, 1, 9, 21, 41, 93, 33, 211, 341, 233, 2217, 6657, 12913, 8329, 3881, 42563}},
	{9650, 17, 33420, {1, 3, 3, 11, 25, 3, 23, 197, 49, 339, 877, 1117, 7817, 14143, 1575, 50301, 92367}},
	{9651, 17, 33423, {1, 3, 5, 5, 19, 45, 69, 179, 447, 861, 1633, 1941, 5821, 1843, 4085, 23501, 109047}},
	{9652, 17, 33431, {1, 3, 1, 3, 31, 29, 49, 183, 311, 133, 345, 1541, 111, 5571, 1943, 11039, 127673}},
	{9653, 17, 33438, {1, 3, 1, 5, 3, 13, 63, 5, 59, 789, 71, 3271, 3871, 9105, 22525, 31, 117803}},
	{9654, 17, 33442, {1, 3, 1, 13, 31, 43, 97, 133, 313, 729, 287, 2971, 5623, 13183, 15179, 47271, 28853}},
	{9655, 17, 33444, {1, 1, 3, 13, 27, 15, 35, 37, 507, 139, 1933, 2847, 361, 10261, 21031, 3889, 56875}},
	{9656, 17, 33448, {1, 3, 1, 15, 31, 13, 45, 73, 279, 331, 471, 3881, 3295, 12035, 28329, 899, 47397}},
	{9657, 17, 33456, {1, 1, 3, 13, 1, 7, 81, 255, 315, 595, 43, 3919, 5229, 7953, 25711, 19509, 107181}},
	{9658, 17, 33459, {1, 1, 3, 15, 7, 33, 117, 169, 71, 577, 629, 3665, 7761, 13529, 26375, 17181, 22125}},
	{9659, 17, 33466, {1, 3, 5, 7, 5, 7, 1, 93, 489, 289, 329, 2273, 685, 14835, 11433, 26041, 112735}},
	{9660, 17, 33473, {1, 3, 3, 3, 9, 39, 45, 23, 171, 35, 571, 551, 7815, 6169, 24283, 61477, 71877}},
	{9661, 17, 33476, {1, 1, 5, 7, 23, 15, 81, 215, 297, 269, 655, 2059, 3643, 12741, 11955, 41085, 46047}},
	{9662, 17, 33491, {1, 1, 7, 5, 3, 35, 125, 141, 419, 137, 1031, 2053, 7925, 7267, 6267, 34323, 77495}},
	{9663, 17, 33494, {1, 1, 7, 11, 3, 57, 91, 43, 139, 691, 1569, 1825, 7855, 1093, 19263, 31601, 16019}},
	{9664, 17, 33507, {1, 3, 1, 5, 21, 7, 11, 225, 105, 757, 1493, 455, 4757, 12007, 5139, 3545, 79717}},
	{9665, 17, 33514, {1, 3, 1, 13, 17, 29, 125, 249, 475, 79, 1271, 341, 863, 853, 2105, 32897, 121261}},
	{9666, 17, 33521, {1, 3, 1, 11, 17, 59, 3, 29, 61, 399, 1465, 4029, 2103, 12481, 28495, 34363, 63781}},
	{9667, 17, 33528, {1, 3, 3, 15, 29, 13, 101, 191, 435, 215, 1355, 2263, 6059, 4545, 7535, 15041, 84091}},
	{9668, 17, 33534, {1, 1, 3, 9, 29, 23, 99, 55, 91, 145, 235, 2847, 725, 209, 24565, 16545, 103669}},
	{9669, 17, 33536, {1, 1, 1, 1, 31, 15, 93, 197, 207, 357, 667, 3511, 3865, 5329, 6491, 9027, 125979}},
	{9670, 17, 33551, {1, 3, 3, 13, 17, 35, 99, 187, 153, 589, 1633, 4053, 1023, 9541, 9841, 39585, 24885}},
	{9671, 17, 33554, {1, 3, 7, 11, 23, 5, 71, 89, 455, 665, 1221, 1821, 591, 11459, 503, 56777, 65691}},
	{9672, 17, 33563, {1, 3, 1, 1, 9, 33, 51, 203, 223, 709, 1263, 3535, 7753, 8279, 8673, 60259, 2671}},
	{9673, 17, 33575, {1, 1, 7, 9, 17, 63, 5, 229, 495, 435, 1711, 3359, 399, 15901, 28519, 56627, 8079}},
	{9674, 17, 33579, {1, 3, 5, 11, 9, 25, 49, 143, 275, 989, 461, 447, 1917, 9253, 28421, 1803, 119725}},
	{9675, 17, 33582, {1, 3, 3, 7, 25, 3, 39, 171, 303, 905, 1353, 2561, 7347, 7339, 15271, 61945, 26343}},
	{9676, 17, 33601, {1, 1, 1, 3, 5, 63, 9, 229, 107, 815, 1705, 3621, 2345, 3065, 16315, 17017, 33667}},
	{9677, 17, 33602, {1, 3, 5, 13, 29, 13, 91, 111, 475, 561, 443, 3825, 5331, 11211, 27639, 28305, 101831}},
	{9678, 17, 33614, {1, 3, 1, 9, 15, 33, 17, 47, 249, 89, 429, 3819, 1959, 14317, 10737, 28151, 40395}},
	{9679, 17, 33625, {1, 3, 7, 13, 19, 29, 83, 81, 511, 783, 823, 2865, 5823, 9459, 27413, 63297, 44181}},
	{9680, 17, 33628, {1, 3, 1, 1, 19, 53, 45, 227, 193, 631, 289, 1227, 6241, 6915, 16051, 31237, 50201}},
	{9681, 17, 33637, {1, 3, 7, 7, 15, 49, 77, 147, 421, 515, 927, 1561, 4391, 12943, 6807, 36889, 70249}},
	{9682, 17, 33656, {1, 3, 7, 7, 17, 15, 63, 123, 101, 283, 59, 977, 5185, 16161, 5007, 36255, 11537}},
	{9683, 17, 33665, {1, 1, 7, 1, 13, 17, 79, 35, 193, 947, 767, 1365, 2145, 13267, 30561, 51949, 37591}},
	{9684, 17, 33683, {1, 1, 1, 13, 11, 13, 91, 129, 355, 549, 295, 673, 209, 15953, 14703, 30857, 47967}},
	{9685, 17, 33695, {1, 3, 5, 9, 17, 17, 83, 161, 189, 585, 21, 1019, 4879, 15943, 17281, 46013, 94839}},
	{9686, 17, 33696, {1, 3, 5, 9, 23, 39, 65, 25, 181, 3, 2005, 635, 201, 9391, 8755, 38535, 88697}},
	{9687, 17, 33702, {1, 3, 1, 15, 13, 35, 47, 125, 429, 901, 895, 3495, 327, 397, 7847, 62157, 3489}},
	{9688, 17, 33708, {1, 3, 5, 3, 19, 21, 81, 39, 85, 169, 1981, 3323, 113, 2057, 16617, 58051, 55059}},
	{9689, 17, 33711, {1, 3, 1, 13, 9, 1, 101, 81, 129, 717, 1495, 4077, 5555, 93, 12957, 14805, 110219}},
	{9690, 17, 33716, {1, 3, 5, 5, 5, 47, 107, 111, 387, 987, 2009, 179, 1111, 3443, 25579, 12293, 123035}},
	{9691, 17, 33728, {1, 1, 7, 13, 21, 25, 33, 211, 9, 783, 1785, 2691, 6835, 2867, 22469, 17853, 90685}},
	{9692, 17, 33737, {1, 1, 3, 3, 19, 57, 59, 203, 197, 347, 553, 1361, 7593, 91, 15303, 30045, 86605}},
	{9693, 17, 33761, {1, 3, 5, 7, 29, 23, 1, 235, 159, 277, 1227, 1727, 1853, 9717, 2377, 13597, 18119}},
	{9694, 17, 33774, {1, 1, 1, 11, 15, 29, 5, 15, 349, 685, 197, 3127, 1075, 8847, 27873, 539, 57149}},
	{9695, 17, 33782, {1, 1, 7, 9, 23, 25, 121, 239, 219, 747, 1981, 2683, 5319, 75, 22569, 29697, 27627}},
	{9696, 17, 33788, {1, 3, 7, 5, 31, 43, 95, 131, 423, 547, 1437, 127, 1953, 861, 839, 54503, 20465}},
	{9697, 17, 33791, {1, 1, 5, 3, 29, 29, 71, 237, 275, 493, 513, 4067, 393, 9415, 20511, 29257, 86267}},
	{9698, 17, 33793, {1, 1, 1, 1, 25, 11, 59, 185, 211, 175, 37, 2999, 4919, 10225, 16727, 60447, 59985}},
	{9699, 17, 33811, {1, 1, 3, 3, 1, 9, 69, 195, 197, 677, 229, 599, 5613, 4537, 5495, 58801, 14297}},
	{9700, 17, 33813, {1, 3, 1, 15, 17, 23, 5, 101, 331, 943, 1433, 2199, 313, 469, 3651, 3281, 100119}},
	{9701, 17, 33818, {1, 1, 5, 15, 13, 25, 87, 45, 229, 821, 59, 761, 6259, 15159, 3197, 39763, 87301}},
	{9702, 17, 33829, {1, 3, 5, 7, 19, 21, 89, 15, 19, 623, 603, 4069, 3531, 13353, 21267, 6355, 53821}},
	{9703, 17, 33842, {1, 1, 5, 9, 13, 13, 111, 77, 439, 599, 1577, 959, 4567, 3117, 7127, 49265, 35667}},
	{9704, 17, 33854, {1, 3, 7, 9, 27, 61, 1, 19, 43, 475, 221, 655, 4351, 15827, 30489, 22245, 41077}},
	{9705, 17, 33856, {1, 1, 3, 13, 17, 17, 111, 85, 253, 11, 367, 2349, 4103, 12517, 27037, 42481, 84451}},
	{9706, 17, 33868, {1, 3, 5, 7, 7, 25, 53, 27, 429, 503, 893, 2923, 2539, 15849, 30157, 12111, 108893}},
	{9707, 17, 33879, {1, 1, 7, 9, 13, 29, 51, 113, 273, 745, 759, 263, 3031, 705, 23203, 64245, 127183}},
	{9708, 17, 33885, {1, 1, 1, 9, 29, 5, 25, 165, 261, 319, 645, 2199, 3135, 10263, 10711, 18713, 63337}},
	{9709, 17, 33886, {1, 1, 5, 1, 23, 41, 43, 71, 365, 683, 1107, 1671, 7079, 8933, 12815, 8095, 97955}},
	{9710, 17, 33892, {1, 3, 1, 15, 9, 43, 105, 217, 131, 299, 1459, 1087, 3493, 15297, 11741, 43383, 35021}},
	{9711, 17, 33907, {1, 3, 1, 3, 3, 57, 69, 7, 73, 977, 1163, 3591, 243, 13129, 23247, 20609, 22489}},
	{9712, 17, 33913, {1, 3, 7, 5, 1, 57, 65, 27, 121, 575, 903, 3527, 5601, 5597, 1941, 60079, 88121}},
	{9713, 17, 33923, {1, 3, 1, 3, 15, 3, 23, 87, 233, 389, 1671, 1557, 4825, 1017, 17697, 26735, 53421}},
	{9714, 17, 33925, {1, 3, 5, 3, 5, 43, 61, 249, 273, 251, 1383, 2415, 1061, 12363, 3071, 23785, 127909}},
	{9715, 17, 33935, {1, 3, 3, 13, 5, 63, 15, 165, 353, 603, 1627, 2037, 487, 11603, 719, 54693, 52645}},
	{9716, 17, 33937, {1, 3, 5, 11, 31, 41, 41, 83, 481, 251, 1903, 2655, 5237, 6073, 20201, 14069, 91649}},
	{9717, 17, 33954, {1, 3, 1, 15, 21, 41, 99, 61, 55, 63, 1595, 1805, 7625, 12261, 23275, 43471, 5147}},
	{9718, 17, 33963, {1, 3, 1, 5, 23, 21, 71, 169, 197, 51, 1653, 3053, 4663, 293, 12751, 15641, 83993}},
	{9719, 17, 33966, {1, 3, 5, 15, 29, 45, 55, 199, 275, 103, 1093, 3569, 5997, 9445, 2291, 30973, 68589}},
	{9720, 17, 33977, {1, 3, 5, 7, 15, 3, 15, 3, 287, 961, 1759, 1153, 7613, 9885, 8981, 5109, 112865}},
	{9721, 17, 33978, {1, 1, 1, 9, 1, 37, 111, 61, 309, 581, 875, 2121, 1035, 4345, 1351, 59743, 34955}},
	{9722, 17, 33991, {1, 3, 7, 7, 11, 23, 51, 235, 23, 697, 991, 1995, 3615, 6665, 15885, 18555, 11711}},
	{9723, 17, 33998, {1, 3, 7, 13, 3, 59, 87, 129, 405, 689, 1189, 2071, 877, 12347, 18381, 28367, 27247}},
	{9724, 17, 34012, {1, 1, 1, 9, 23, 29, 113, 71, 479, 421, 215, 1029, 6125, 13575, 10823, 45303, 3153}},
	{9725, 17, 34016, {1, 1, 3, 11, 13, 5, 31, 29, 279, 597, 791, 319, 1391, 14487, 3811, 36913, 11513}},
	{9726, 17, 34025, {1, 3, 7, 11, 9, 11, 55, 167, 69, 519, 1887, 145, 6133, 1307, 14465, 17419, 18319}},
	{9727, 17, 34033, {1, 1, 3, 1, 29, 25, 57, 75, 19, 187, 1591, 421, 959, 7499, 8377, 42811, 53423}},
	{9728, 17, 34036, {1, 3, 1, 3, 7, 9, 73, 217, 383, 755, 1561, 3923, 3891, 16129, 13195, 62097, 67493}},
	{9729, 17, 34045, {1, 3, 7, 9, 5, 7, 47, 29, 319, 243, 405, 2867, 5803, 2273, 4913, 54777, 88301}},
	{9730, 17, 34065, {1, 3, 7, 1, 25, 11, 51, 183, 387, 863, 39, 2119, 2395, 10175, 20833, 3235, 108197}},
	{9731, 17, 34078, {1, 1, 7, 13, 25, 43, 21, 67, 103, 709, 603, 1045, 7079, 8867, 29039, 61499, 39533}},
	{9732, 17, 34093, {1, 1, 7, 5, 7, 55, 77, 115, 409, 287, 1149, 1535, 7459, 5525, 27129, 43659, 86953}},
	{9733, 17, 34101, {1, 3, 5, 3, 21, 41, 47, 147, 267, 473, 1501, 2663, 5381, 41, 18265, 53845, 16039}},
	{9734, 17, 34108, {1, 1, 7, 15, 27, 63, 95, 103, 169, 1, 133, 3103, 7539, 5765, 11453, 4133, 95133}},
	{9735, 17, 34111, {1, 3, 3, 15, 3, 53, 121, 135, 385, 475, 889, 2557, 4937, 11129, 18461, 16757, 30339}},
	{9736, 17, 34120, {1, 3, 1, 13, 11, 39, 111, 13, 475, 201, 1973, 2151, 6973, 4083, 12593, 44093, 108037}},
	{9737, 17, 34123, {1, 3, 7, 9, 31, 31, 97, 235, 179, 689, 403, 1995, 7697, 7511, 29333, 11005, 50723}},
	{9738, 17, 34125, {1, 1, 7, 13, 23, 5, 7, 171, 441, 921, 1455, 3865, 7089, 5469, 10423, 53013, 80559}},
	{9739, 17, 34153, {1, 3, 5, 3, 25, 43, 105, 157, 507, 143, 297, 1111, 2761, 14103, 4965, 36733, 11741}},
	{9740, 17, 34171, {1, 3, 7, 9, 29, 61, 49, 239, 271, 697, 211, 1633, 2991, 14933, 12347, 44291, 12219}},
	{9741, 17, 34174, {1, 1, 7, 7, 17, 61, 29, 43, 87, 633, 937, 1931, 3541, 12259, 23045, 5923, 48479}},
	{9742, 17, 34178, {1, 3, 3, 3, 15, 25, 105, 17, 159, 863, 1377, 331, 1475, 10573, 28947, 8141, 26671}},
	{9743, 17, 34183, {1, 1, 7, 7, 31, 59, 81, 23, 467, 241, 1257, 1337, 7731, 9071, 3417, 51191, 78369}},
	{9744, 17, 34190, {1, 1, 5, 9, 11, 45, 49, 227, 319, 63, 1339, 885, 4571, 11649, 5607, 10509, 55055}},
	{9745, 17, 34201, {1, 3, 3, 9, 29, 17, 7, 235, 191, 927, 575, 1115, 4111, 14179, 2041, 13331, 29825}},
	{9746, 17, 34211, {1, 1, 5, 9, 27, 61, 71, 201, 341, 577, 221, 1371, 1135, 4347, 24211, 36171, 23435}},
	{9747, 17, 34220, {1, 3, 3, 1, 1, 29, 75, 121, 193, 647, 1429, 275, 5243, 783, 28533, 13941, 68035}},
	{9748, 17, 34225, {1, 3, 5, 15, 21, 27, 117, 183, 251, 991, 935, 3119, 5133, 2765, 7423, 28867, 120565}},
	{9749, 17, 34237, {1, 3, 5, 5, 13, 23, 29, 101, 299, 699, 1249, 1225, 1335, 6079, 17825, 60467, 87787}},
	{9750, 17, 34249, {1, 1, 1, 9, 15, 19, 11, 163, 433, 553, 1487, 813, 3293, 1195, 895, 28431, 62905}},
	{9751, 17, 34250, {1, 1, 1, 13, 25, 37, 111, 129, 391, 813, 1061, 4065, 7339, 10731, 23799, 41463, 99673}},
	{9752, 17, 34264, {1, 1, 7, 15, 3, 21, 45, 77, 471, 155, 967, 711, 4947, 13983, 27827, 28653, 117839}},
	{9753, 17, 34269, {1, 1, 5, 9, 13, 39, 107, 237, 233, 881, 297, 2189, 8085, 1221, 18659, 299, 90951}},
	{9754, 17, 34276, {1, 1, 1, 13, 21, 53, 83, 17, 487, 215, 1203, 3017, 7887, 3759, 10521, 31223, 87917}},
	{9755, 17, 34279, {1, 1, 7, 1, 13, 31, 123, 219, 127, 743, 1325, 3907, 129, 8901, 4855, 22509, 47331}},
	{9756, 17, 34293, {1, 1, 7, 11, 29, 37, 11, 157, 401, 35, 2037, 2873, 7409, 7837, 1247, 33911, 3979}},
	{9757, 17, 34303, {1, 1, 5, 15, 1, 13, 35, 253, 287, 1007, 1417, 1613, 6019, 11617, 6323, 56263, 45073}},
	{9758, 17, 34310, {1, 3, 1, 15, 1, 59, 41, 239, 373, 443, 897, 275, 5783, 8619, 18559, 16279, 92063}},
	{9759, 17, 34340, {1, 3, 1, 9, 23, 33, 83, 43, 231, 819, 1657, 1031, 5507, 12621, 8961, 23059, 63453}},
	{9760, 17, 34349, {1, 1, 7, 5, 29, 49, 21, 251, 267, 43, 729, 4013, 1497, 15489, 16761, 49689, 122755}},
	{9761, 17, 34352, {1, 3, 7, 1, 31, 21, 11, 149, 127, 711, 1249, 49, 5503, 677, 12313, 61301, 16279}},
	{9762, 17, 34355, {1, 1, 5, 11, 9, 15, 41, 61, 81, 991, 1387, 3567, 221, 15835, 8609, 28265, 98199}},
	{9763, 17, 34358, {1, 3, 1, 7, 21, 35, 13, 59, 173, 637, 107, 393, 4551, 6523, 27389, 33129, 45579}},
	{9764, 17, 34362, {1, 1, 1, 9, 29, 51, 65, 199, 417, 553, 1321, 2513, 4749, 8477, 19721, 24301, 16301}},
	{9765, 17, 34376, {1, 3, 5, 1, 25, 13, 7, 55, 163, 581, 1677, 2313, 6843, 15697, 3055, 53171, 59899}},
	{9766, 17, 34381, {1, 3, 1, 5, 31, 13, 101, 195, 235, 359, 911, 1017, 2575, 12801, 997, 7819, 73243}},
	{9767, 17, 34387, {1, 1, 7, 1, 9, 39, 59, 83, 57, 885, 317, 2689, 5741, 11833, 25563, 62581, 62239}},
	{9768, 17, 34389, {1, 1, 5, 15, 25, 25, 55, 207, 223, 907, 913, 387, 5599, 15567, 8859, 13703, 66071}},
	{9769, 17, 34394, {1, 1, 5, 15, 19, 39, 83, 177, 333, 531, 1257, 2687, 7793, 15967, 19175, 1381, 106629}},
	{9770, 17, 34410, {1, 3, 5, 13, 29, 29, 77, 1, 273, 483, 725, 3825, 5115, 4043, 11571, 8693, 49761}},
	{9771, 17, 34423, {1, 1, 7, 3, 5, 45, 37, 65, 267, 191, 301, 2863, 167, 9303, 14563, 41553, 119561}},
	{9772, 17, 34434, {1, 1, 7, 5, 21, 41, 107, 213, 267, 427, 699, 1485, 2125, 16011, 29243, 4691, 50545}},
	{9773, 17, 34436, {1, 3, 3, 9, 15, 29, 81, 53, 289, 689, 933, 2667, 5175, 10409, 28221, 56375, 49109}},
	{9774, 17, 34448, {1, 1, 1, 15, 3, 11, 77, 107, 353, 349, 219, 1961, 7559, 10081, 25119, 46041, 103827}},
	{9775, 17, 34453, {1, 3, 3, 1, 5, 27, 109, 17, 271, 543, 565, 397, 2649, 12037, 4525, 37835, 107071}},
	{9776, 17, 34454, {1, 1, 5, 15, 3, 37, 123, 157, 389, 619, 1379, 4093, 6107, 4419, 21011, 36189, 21269}},
	{9777, 17, 34460, {1, 3, 1, 7, 25, 17, 37, 133, 247, 113, 985, 815, 441, 7869, 25121, 49459, 429}},
	{9778, 17, 34464, {1, 3, 3, 11, 7, 23, 59, 51, 403, 685, 2019, 1167, 7973, 6915, 10819, 43807, 127793}},
	{9779, 17, 34479, {1, 1, 3, 1, 29, 3, 125, 107, 305, 101, 391, 2733, 6883, 5867, 5139, 16025, 112439}},
	{9780, 17, 34491, {1, 1, 5, 5, 23, 23, 89, 33, 275, 451, 1033, 649, 3761, 4735, 26021, 9627, 102747}},
	{9781, 17, 34501, {1, 1, 5, 13, 3, 17, 117, 251, 425, 917, 759, 3047, 8171, 14421, 27765, 11085, 64889}},
	{9782, 17, 34508, {1, 3, 1, 9, 7, 23, 107, 143, 123, 413, 2045, 655, 6283, 8783, 20263, 55463, 33271}},
	{9783, 17, 34516, {1, 3, 7, 11, 5, 49, 73, 55, 465, 43, 587, 3943, 521, 12357, 16273, 26603, 23219}},
	{9784, 17, 34529, {1, 3, 5, 13, 9, 3, 127, 171, 271, 227, 993, 1427, 2235, 6325, 13501, 1411, 44393}},
	{9785, 17, 34530, {1, 1, 1, 3, 13, 27, 19, 37, 175, 423, 5, 3403, 5427, 16345, 30297, 11909, 104647}},
	{9786, 17, 34553, {1, 3, 1, 3, 3, 39, 111, 179, 487, 923, 1945, 1609, 4689, 11807, 13725, 3081, 48163}},
	{9787, 17, 34564, {1, 3, 1, 1, 9, 35, 7, 151, 109, 925, 1249, 3171, 1207, 2053, 5135, 34821, 57291}},
	{9788, 17, 34568, {1, 1, 5, 13, 31, 35, 101, 199, 499, 725, 1229, 2857, 6437, 503, 14437, 35721, 24971}},
	{9789, 17, 34571, {1, 1, 1, 15, 3, 49, 75, 101, 373, 119, 875, 245, 15, 12937, 4731, 13037, 1555}},
	{9790, 17, 34582, {1, 1, 1, 7, 15, 5, 53, 5, 423, 69, 73, 2139, 383, 4035, 6723, 59941, 124503}},
	{9791, 17, 34586, {1, 1, 3, 13, 1, 23, 29, 47, 145, 785, 1013, 1579, 4579, 107, 17571, 46311, 27777}},
	{9792, 17, 34598, {1, 1, 1, 5, 23, 25, 97, 75, 105, 183, 827, 3871, 2005, 6453, 28729, 42583, 62979}},
	{9793, 17, 34604, {1, 3, 5, 9, 11, 49, 29, 201, 333, 441, 429, 1955, 5301, 11775, 22915, 58693, 111917}},
	{9794, 17, 34610, {1, 3, 3, 1, 15, 37, 117, 223, 319, 181, 61, 177, 507, 14871, 16419, 34261, 106937}},
	{9795, 17, 34619, {1, 3, 3, 9, 25, 27, 81, 253, 459, 5, 693, 1271, 485, 16171, 427, 17917, 4393}},
	{9796, 17, 34621, {1, 3, 3, 1, 27, 47, 11, 57, 269, 95, 569, 2733, 3275, 1599, 15073, 58071, 86805}},
	{9797, 17, 34633, {1, 3, 7, 13, 21, 57, 75, 63, 53, 487, 251, 3193, 4279, 2311, 6613, 38319, 93557}},
	{9798, 17, 34634, {1, 3, 5, 5, 31, 35, 39, 255, 11, 81, 605, 1457, 6367, 14121, 8069, 46653, 79945}},
	{9799, 17, 34657, {1, 1, 1, 7, 17, 19, 19, 247, 13, 757, 1069, 2811, 4969, 10943, 29399, 4153, 120817}},
	{9800, 17, 34682, {1, 1, 1, 15, 31, 13, 1, 247, 157, 785, 1565, 897, 4825, 8375, 4933, 60671, 88403}},
	{9801, 17, 34688, {1, 3, 3, 7, 31, 53, 117, 207, 243, 603, 625, 1039, 5725, 5021, 20227, 28613, 123759}},
	{9802, 17, 34691, {1, 1, 5, 1, 7, 29, 65, 153, 393, 821, 295, 2705, 5999, 15801, 31301, 15545, 52917}},
	{9803, 17, 34694, {1, 1, 1, 1, 11, 51, 97, 143, 27, 279, 1005, 1235, 5539, 1523, 26293, 35015, 47835}},
	{9804, 17, 34706, {1, 3, 3, 13, 27, 17, 123, 147, 39, 35, 567, 961, 5431, 5557, 17849, 46675, 102181}},
	{9805, 17, 34708, {1, 1, 7, 11, 7, 25, 73, 223, 459, 207, 1637, 647, 2057, 685, 24539, 48809, 26877}},
	{9806, 17, 34724, {1, 3, 1, 3, 21, 43, 121, 11, 431, 383, 1703, 1451, 2349, 11845, 13801, 20589, 43125}},
	{9807, 17, 34727, {1, 1, 5, 1, 27, 29, 89, 233, 437, 303, 853, 3425, 263, 2073, 14111, 39129, 59547}},
	{9808, 17, 34751, {1, 1, 1, 3, 3, 47, 99, 207, 261, 179, 1761, 2657, 4339, 6567, 25455, 18729, 51431}},
	{9809, 17, 34753, {1, 3, 3, 13, 5, 5, 109, 125, 123, 233, 1713, 1539, 4375, 12187, 18355, 49597, 109959}},
	{9810, 17, 34759, {1, 3, 7, 7, 9, 23, 45, 193, 363, 837, 855, 1413, 7587, 9091, 27907, 17809, 63249}},
	{9811, 17, 34763, {1, 3, 3, 9, 19, 23, 63, 85, 419, 1007, 1753, 539, 1471, 2171, 9239, 36289, 105503}},
	{9812, 17, 34777, {1, 3, 1, 11, 23, 5, 105, 79, 473, 879, 1623, 3155, 5157, 4699, 697, 41919, 15441}},
	{9813, 17, 34778, {1, 1, 7, 11, 5, 21, 43, 207, 491, 355, 857, 2325, 819, 15849, 24529, 5789, 110661}},
	{9814, 17, 34780, {1, 1, 5, 15, 19, 33, 81, 137, 473, 853, 1681, 3841, 5617, 13715, 1987, 52983, 66327}},
	{9815, 17, 34796, {1, 3, 5, 7, 11, 31, 69, 85, 33, 197, 1771, 1957, 1311, 169, 14159, 7327, 8577}},
	{9816, 17, 34799, {1, 1, 3, 9, 11, 23, 19, 143, 9, 579, 111, 2973, 3567, 8561, 10447, 55875, 64305}},
	{9817, 17, 34801, {1, 1, 5, 7, 1, 17, 93, 11, 423, 1007, 839, 719, 3965, 14531, 17301, 29577, 4083}},
	{9818, 17, 34817, {1, 3, 5, 13, 19, 17, 123, 61, 59, 115, 1165, 579, 2545, 633, 5597, 21865, 109167}},
	{9819, 17, 34824, {1, 1, 5, 3, 29, 29, 99, 163, 321, 367, 1523, 3719, 665, 15843, 28831, 63823, 113533}},
	{9820, 17, 34827, {1, 1, 1, 3, 15, 7, 85, 1, 181, 759, 537, 3315, 7159, 4363, 4183, 53775, 8801}},
	{9821, 17, 34837, {1, 3, 1, 1, 15, 53, 9, 35, 459, 417, 1169, 2055, 1175, 10923, 335, 24269, 93001}},
	{9822, 17, 34841, {1, 3, 1, 5, 31, 43, 51, 149, 175, 541, 629, 1147, 7585, 9725, 18623, 13345, 65391}},
	{9823, 17, 34853, {1, 3, 7, 1, 13, 39, 13, 217, 507, 765, 721, 1491, 5037, 6267, 2871, 19181, 123751}},
	{9824, 17, 34858, {1, 1, 3, 5, 21, 9, 123, 195, 63, 347, 7, 531, 3015, 9457, 29543, 51479, 26607}},
	{9825, 17, 34877, {1, 1, 1, 1, 21, 15, 81, 127, 429, 15, 901, 1503, 1919, 6515, 2477, 53571, 113447}},
	{9826, 17, 34886, {1, 3, 1, 13, 9, 33, 79, 169, 499, 767, 441, 2085, 2429, 10213, 4125, 2611, 26137}},
	{9827, 17, 34895, {1, 1, 3, 1, 19, 23, 83, 179, 447, 513, 913, 1201, 1861, 11595, 29037, 7775, 116417}},
	{9828, 17, 34897, {1, 3, 3, 7, 3, 57, 47, 183, 413, 319, 1375, 1401, 2231, 14331, 28625, 43839, 102717}},
	{9829, 17, 34898, {1, 1, 5, 11, 31, 27, 111, 85, 191, 155, 2025, 1501, 4991, 4655, 3451, 10219, 60391}},
	{9830, 17, 34916, {1, 3, 3, 7, 17, 19, 113, 37, 423, 479, 709, 3659, 6567, 1709, 13483, 61821, 77101}},
	{9831, 17, 34923, {1, 3, 1, 13, 3, 17, 73, 61, 275, 359, 1341, 449, 1373, 12047, 11207, 52651, 83305}},
	{9832, 17, 34928, {1, 1, 7, 9, 9, 45, 15, 121, 15, 51, 509, 2189, 5057, 6119, 11669, 14559, 108323}},
	{9833, 17, 34934, {1, 1, 7, 7, 25, 13, 13, 141, 157, 249, 823, 821, 1909, 5925, 3505, 13187, 19237}},
	{9834, 17, 34940, {1, 3, 3, 1, 9, 51, 79, 91, 5, 709, 787, 2427, 4613, 7307, 20141, 1675, 49779}},
	{9835, 17, 34944, {1, 1, 1, 11, 11, 13, 33, 81, 413, 981, 907, 2709, 4113, 10607, 2587, 12845, 11103}},
	{9836, 17, 34947, {1, 1, 7, 9, 13, 25, 37, 81, 375, 1013, 2027, 321, 3947, 2269, 10687, 7537, 67495}},
	{9837, 17, 34953, {1, 3, 5, 11, 9, 43, 53, 111, 339, 841, 503, 3209, 6437, 10893, 13627, 51809, 57229}},
	{9838, 17, 34956, {1, 3, 1, 1, 21, 15, 71, 93, 453, 405, 1099, 2979, 7471, 10173, 17875, 13179, 48615}},
	{9839, 17, 34967, {1, 3, 5, 9, 9, 1, 121, 117, 275, 157, 57, 3459, 4787, 15005, 24591, 23963, 45077}},
	{9840, 17, 34968, {1, 1, 5, 3, 21, 57, 113, 207, 169, 603, 637, 1455, 6281, 6527, 17219, 32307, 18617}},
	{9841, 17, 34971, {1, 3, 7, 5, 25, 15, 99, 91, 253, 267, 537, 713, 3929, 895, 7999, 47989, 118731}},
	{9842, 17, 34974, {1, 3, 7, 15, 23, 17, 5, 129, 121, 251, 219, 2547, 7291, 1079, 14577, 56229, 35253}},
	{9843, 17, 34977, {1, 3, 1, 15, 5, 61, 35, 135, 497, 681, 751, 2303, 6697, 11225, 30389, 61673, 87313}},
	{9844, 17, 34980, {1, 3, 1, 7, 7, 37, 9, 85, 257, 805, 1325, 3597, 6065, 727, 18203, 57077, 437}},
	{9845, 17, 34983, {1, 3, 5, 7, 5, 43, 29, 179, 73, 173, 1441, 1233, 1779, 7893, 10629, 27547, 7775}},
	{9846, 17, 34998, {1, 1, 7, 5, 31, 29, 21, 35, 289, 423, 449, 3331, 2929, 6827, 15569, 9873, 76889}},
	{9847, 17, 35004, {1, 1, 7, 13, 13, 37, 55, 99, 135, 797, 1263, 2539, 893, 4225, 16689, 38259, 50857}},
	{9848, 17, 35010, {1, 1, 3, 1, 5, 3, 95, 29, 15, 539, 825, 3931, 4809, 8299, 29891, 61357, 97523}},
	{9849, 17, 35012, {1, 3, 1, 9, 27, 25, 115, 239, 387, 163, 1153, 31, 2375, 7943, 31929, 1121, 33085}},
	{9850, 17, 35030, {1, 3, 5, 9, 3, 53, 121, 159, 165, 81, 317, 3051, 1991, 493, 2029, 43305, 130209}},
	{9851, 17, 35039, {1, 1, 1, 5, 9, 57, 39, 247, 73, 613, 1047, 3289, 2569, 5363, 18475, 32749, 39415}},
	{9852, 17, 35058, {1, 3, 1, 5, 19, 23, 39, 33, 151, 463, 153, 737, 2501, 7531, 2769, 35595, 71799}},
	{9853, 17, 35063, {1, 3, 5, 5, 29, 49, 105, 81, 67, 441, 1101, 2241, 6243, 6177, 7157, 51635, 81241}},
	{9854, 17, 35082, {1, 3, 3, 3, 29, 53, 13, 239, 487, 503, 97, 1323, 1817, 13021, 12881, 26943, 21011}},
	{9855, 17, 35095, {1, 1, 1, 15, 25, 9, 5, 205, 85, 635, 789, 2495, 5069, 4987, 847, 26857, 84225}},
	{9856, 17, 35096, {1, 1, 3, 15, 9, 51, 79, 13, 377, 637, 159, 3407, 2057, 13967, 31781, 40869, 52987}},
	{9857, 17, 35101, {1, 3, 1, 13, 11, 27, 103, 207, 383, 887, 749, 1119, 285, 4269, 31745, 57539, 5671}},
	{9858, 17, 35102, {1, 3, 1, 13, 23, 19, 41, 43, 455, 425, 1653, 4091, 4855, 16321, 169, 59289, 82397}},
	{9859, 17, 35105, {1, 3, 3, 15, 31, 39, 51, 127, 391, 989, 1831, 3327, 6487, 6077, 17277, 52093, 20389}},
	{9860, 17, 35112, {1, 3, 5, 15, 19, 1, 21, 241, 15, 543, 1529, 2355, 1503, 12795, 17321, 41219, 61115}},
	{9861, 17, 35118, {1, 1, 3, 11, 9, 33, 21, 197, 307, 141, 1663, 371, 1663, 8307, 3617, 56941, 62477}},
	{9862, 17, 35120, {1, 3, 7, 9, 19, 53, 123, 3, 29, 635, 1795, 2471, 2491, 15847, 9169, 2561, 101515}},
	{9863, 17, 35130, {1, 1, 5, 3, 19, 11, 117, 231, 475, 837, 1833, 3499, 4415, 9961, 28285, 37821, 81497}},
	{9864, 17, 35143, {1, 1, 3, 5, 7, 11, 57, 89, 345, 157, 1519, 3021, 7157, 2159, 32557, 31559, 128907}},
	{9865, 17, 35147, {1, 1, 7, 3, 27, 1, 15, 177, 489, 405, 811, 3597, 4939, 15595, 7279, 58097, 84703}},
	{9866, 17, 35152, {1, 3, 1, 9, 25, 61, 119, 219, 111, 339, 1091, 759, 6087, 16001, 6757, 15627, 1691}},
	{9867, 17, 35157, {1, 3, 7, 9, 1, 39, 107, 139, 143, 917, 421, 1623, 7135, 4851, 6687, 6177, 102425}},
	{9868, 17, 35164, {1, 1, 7, 13, 23, 17, 19, 167, 317, 331, 743, 3737, 2195, 545, 2185, 9125, 30503}},
	{9869, 17, 35178, {1, 1, 5, 13, 27, 33, 117, 141, 493, 129, 1553, 2335, 4161, 14205, 24177, 35163, 84869}},
	{9870, 17, 35195, {1, 3, 7, 1, 11, 9, 75, 133, 113, 507, 2007, 2473, 4769, 14655, 17967, 17709, 90653}},
	{9871, 17, 35197, {1, 1, 7, 11, 17, 11, 83, 23, 387, 61, 29, 3905, 4351, 15173, 28375, 9129, 111939}},
	{9872, 17, 35201, {1, 1, 5, 15, 15, 53, 81, 125, 189, 937, 1607, 2595, 2847, 7229, 22241, 26269, 64781}},
	{9873, 17, 35207, {1, 3, 1, 7, 5, 11, 61, 111, 13, 423, 885, 2329, 6003, 16331, 11207, 25743, 54619}},
	{9874, 17, 35231, {1, 3, 5, 9, 1, 13, 95, 241, 237, 629, 263, 1629, 1063, 12695, 14501, 5455, 121483}},
	{9875, 17, 35249, {1, 1, 7, 15, 5, 17, 45, 255, 143, 79, 87, 1755, 6215, 5095, 32411, 8695, 85511}},
	{9876, 17, 35250, {1, 3, 7, 7, 21, 11, 117, 135, 333, 73, 1471, 2749, 5801, 4209, 9353, 46171, 90645}},
	{9877, 17, 35256, {1, 1, 7, 13, 11, 35, 77, 149, 159, 783, 1527, 2881, 1409, 3455, 26991, 3225, 30693}},
	{9878, 17, 35259, {1, 1, 3, 15, 19, 55, 21, 245, 207, 103, 775, 2041, 4637, 7333, 11267, 60509, 43099}},
	{9879, 17, 35262, {1, 3, 3, 15, 17, 63, 23, 81, 183, 923, 75, 391, 615, 13343, 20839, 56529, 115747}},
	{9880, 17, 35273, {1, 3, 1, 13, 5, 5, 15, 27, 263, 497, 1365, 2733, 5395, 7461, 2725, 24735, 89251}},
	{9881, 17, 35282, {1, 1, 7, 7, 29, 17, 39, 117, 363, 915, 123, 283, 4575, 3497, 20995, 37883, 16645}},
	{9882, 17, 35298, {1, 3, 3, 9, 1, 25, 79, 181, 331, 617, 393, 1807, 5145, 8007, 9173, 45189, 37945}},
	{9883, 17, 35307, {1, 3, 1, 5, 1, 9, 127, 137, 379, 371, 367, 3237, 581, 15295, 18191, 37689, 103495}},
	{9884, 17, 35328, {1, 1, 7, 1, 29, 53, 103, 173, 171, 973, 933, 3847, 3185, 10107, 31701, 45021, 106251}},
	{9885, 17, 35334, {1, 1, 1, 7, 23, 9, 61, 25, 343, 471, 2041, 2179, 7647, 1885, 15353, 50379, 67681}},
	{9886, 17, 35343, {1, 1, 5, 11, 31, 13, 51, 185, 83, 917, 85, 1317, 8185, 14949, 32455, 57939, 1217}},
	{9887, 17, 35345, {1, 1, 7, 5, 23, 45, 101, 227, 497, 941, 985, 167, 6847, 9611, 20011, 40069, 83285}},
	{9888, 17, 35355, {1, 1, 5, 13, 17, 33, 61, 197, 433, 255, 67, 1479, 5663, 6501, 30695, 27235, 80141}},
	{9889, 17, 35388, {1, 1, 3, 5, 11, 45, 123, 49, 327, 893, 1963, 2225, 2611, 8925, 22811, 2313, 8411}},
	{9890, 17, 35399, {1, 3, 7, 7, 15, 39, 75, 235, 13, 847, 575, 3947, 6947, 2061, 13467, 103, 86285}},
	{9891, 17, 35403, {1, 1, 7, 3, 21, 43, 113, 197, 141, 873, 1139, 2707, 7235, 10683, 10831, 33695, 57063}},
	{9892, 17, 35408, {1, 3, 5, 1, 3, 27, 45, 43, 119, 979, 1933, 1851, 6497, 14937, 4965, 41285, 120221}},
	{9893, 17, 35413, {1, 1, 3, 1, 23, 59, 67, 7, 49, 351, 1053, 1837, 501, 7671, 26239, 51951, 95119}},
	{9894, 17, 35418, {1, 3, 5, 11, 3, 19, 33, 33, 219, 175, 1439, 197, 1841, 159, 11229, 20463, 81797}},
	{9895, 17, 35434, {1, 1, 7, 1, 13, 11, 79, 75, 53, 525, 91, 233, 5999, 2921, 21295, 56831, 116049}},
	{9896, 17, 35436, {1, 3, 3, 13, 29, 7, 71, 207, 193, 635, 1393, 3093, 3775, 12445, 23281, 29401, 103225}},
	{9897, 17, 35448, {1, 1, 7, 3, 29, 57, 111, 163, 63, 593, 881, 1587, 3027, 12599, 30977, 38891, 95495}},
	{9898, 17, 35460, {1, 1, 5, 15, 17, 57, 111, 169, 149, 767, 377, 765, 7533, 1539, 22979, 55489, 29799}},
	{9899, 17, 35475, {1, 3, 5, 15, 25, 7, 127, 71, 319, 389, 497, 1513, 1287, 7359, 12311, 45457, 45897}},
	{9900, 17, 35494, {1, 1, 5, 3, 3, 35, 45, 17, 49, 483, 197, 727, 5355, 7201, 3035, 14313, 40933}},
	{9901, 17, 35497, {1, 1, 7, 15, 1, 9, 27, 59, 455, 653, 1907, 281, 1435, 14593, 18909, 37655, 87603}},
	{9902, 17, 35503, {1, 1, 7, 11, 29, 9, 67, 17, 353, 709, 859, 3687, 7741, 4251, 12263, 41717, 79393}},
	{9903, 17, 35508, {1, 3, 3, 3, 1, 15, 113, 187, 255, 851, 503, 4089, 7923, 1701, 305, 8353, 16357}},
	{9904, 17, 35511, {1, 1, 5, 3, 17, 31, 29, 233, 377, 215, 1889, 3459, 2443, 3907, 4193, 16519, 49089}},
	{9905, 17, 35518, {1, 1, 3, 1, 17, 39, 11, 255, 247, 305, 669, 1769, 1355, 12055, 2275, 51681, 112337}},
	{9906, 17, 35520, {1, 3, 1, 1, 17, 17, 75, 95, 409, 21, 1513, 1443, 4931, 6491, 1587, 62979, 90395}},
	{9907, 17, 35530, {1, 1, 3, 5, 3, 19, 125, 175, 279, 911, 301, 407, 7773, 949, 32107, 13571, 58717}},
	{9908, 17, 35537, {1, 3, 3, 15, 31, 35, 11, 223, 125, 209, 1719, 1725, 3387, 14879, 32243, 7219, 126791}},
	{9909, 17, 35543, {1, 1, 3, 1, 31, 29, 67, 79, 93, 193, 1573, 2285, 3209, 8397, 17717, 5657, 61545}},
	{9910, 17, 35560, {1, 3, 1, 9, 11, 33, 85, 121, 193, 63, 461, 1835, 889, 10687, 19831, 49551, 59087}},
	{9911, 17, 35566, {1, 3, 3, 7, 11, 3, 9, 87, 91, 487, 289, 1113, 8135, 7971, 16693, 31009, 81197}},
	{9912, 17, 35571, {1, 3, 3, 1, 23, 23, 61, 209, 409, 845, 547, 1493, 465, 6399, 17633, 53647, 52425}},
	{9913, 17, 35598, {1, 1, 7, 7, 21, 31, 71, 249, 63, 895, 653, 93, 4429, 8951, 16873, 48089, 33947}},
	{9914, 17, 35609, {1, 3, 5, 11, 3, 35, 49, 15, 379, 645, 855, 3657, 8019, 2141, 11233, 60731, 80455}},
	{9915, 17, 35612, {1, 3, 1, 3, 1, 53, 101, 157, 255, 765, 1575, 1615, 7677, 9699, 13351, 2207, 90939}},
	{9916, 17, 35615, {1, 3, 7, 7, 5, 43, 123, 109, 119, 391, 1889, 1991, 3151, 1457, 16321, 65245, 75891}},
	{9917, 17, 35616, {1, 3, 1, 15, 9, 1, 113, 249, 1, 675, 501, 487, 2209, 4411, 6609, 29243, 100177}},
	{9918, 17, 35622, {1, 1, 1, 7, 9, 23, 9, 197, 341, 191, 453, 3733, 5475, 15515, 28979, 36077, 17801}},
	{9919, 17, 35626, {1, 1, 3, 13, 5, 35, 85, 121, 59, 429, 1251, 3437, 3121, 12411, 14713, 28125, 31921}},
	{9920, 17, 35633, {1, 3, 5, 3, 27, 17, 61, 255, 485, 709, 83, 3201, 2191, 3371, 2941, 10931, 22141}},
	{9921, 17, 35636, {1, 1, 1, 1, 19, 19, 25, 177, 397, 579, 529, 1619, 3887, 4537, 8123, 52481, 8305}},
	{9922, 17, 35645, {1, 1, 3, 15, 3, 15, 77, 51, 31, 881, 203, 2359, 4947, 6321, 14705, 16471, 84395}},
	{9923, 17, 35653, {1, 3, 7, 9, 13, 53, 67, 41, 289, 721, 1743, 2725, 435, 1327, 14953, 14283, 113211}},
	{9924, 17, 35663, {1, 3, 1, 5, 19, 23, 73, 181, 187, 675, 125, 1877, 6167, 7919, 3955, 25007, 28299}},
	{9925, 17, 35665, {1, 1, 3, 1, 5, 11, 123, 189, 173, 123, 499, 2175, 483, 13017, 14709, 5797, 36327}},
	{9926, 17, 35682, {1, 3, 7, 5, 21, 39, 79, 229, 19, 203, 375, 3901, 1053, 14209, 13535, 63155, 99727}},
	{9927, 17, 35687, {1, 1, 1, 13, 11, 29, 29, 173, 441, 271, 1147, 2891, 965, 10777, 16325, 37135, 101601}},
	{9928, 17, 35688, {1, 1, 3, 3, 25, 13, 79, 233, 75, 191, 987, 3231, 3667, 1525, 14193, 62027, 77441}},
	{9929, 17, 35691, {1, 3, 1, 1, 15, 53, 17, 45, 367, 263, 425, 1565, 6139, 13833, 12547, 61103, 75361}},
	{9930, 17, 35696, {1, 1, 5, 15, 5, 57, 123, 47, 407, 887, 375, 1181, 5367, 10283, 24799, 33121, 76373}},
	{9931, 17, 35727, {1, 1, 7, 3, 11, 17, 65, 133, 3, 609, 601, 3391, 7801, 4137, 32095, 55983, 23037}},
	{9932, 17, 35741, {1, 3, 1, 3, 25, 5, 125, 5, 297, 571, 145, 3601, 1929, 13457, 16977, 21049, 92169}},
	{9933, 17, 35742, {1, 3, 5, 13, 23, 29, 13, 143, 507, 187, 857, 427, 5125, 1377, 10947, 58473, 110541}},
	{9934, 17, 35746, {1, 3, 3, 15, 15, 49, 39, 103, 193, 507, 639, 2399, 3829, 12105, 15993, 52975, 115935}},
	{9935, 17, 35748, {1, 3, 7, 3, 7, 41, 95, 127, 193, 923, 1729, 3039, 7959, 3345, 7725, 35293, 34361}},
	{9936, 17, 35752, {1, 3, 5, 13, 17, 53, 111, 141, 151, 389, 1955, 3333, 4523, 6331, 21239, 57447, 113325}},
	{9937, 17, 35770, {1, 3, 7, 15, 31, 7, 11, 35, 105, 607, 1665, 3281, 487, 9417, 26205, 26963, 81537}},
	{9938, 17, 35811, {1, 3, 1, 1, 17, 15, 3, 55, 451, 691, 1525, 2009, 6443, 4629, 15091, 46961, 83361}},
	{9939, 17, 35817, {1, 3, 1, 15, 1, 29, 99, 79, 225, 665, 623, 2389, 3303, 7221, 20567, 15917, 24677}},
	{9940, 17, 35832, {1, 1, 3, 15, 3, 17, 125, 239, 485, 849, 327, 1459, 3911, 2145, 14475, 24337, 19695}},
	{9941, 17, 35838, {1, 3, 5, 7, 7, 37, 19, 51, 373, 587, 147, 563, 7623, 7781, 18289, 37239, 6803}},
	{9942, 17, 35850, {1, 3, 5, 1, 9, 63, 5, 87, 171, 5, 1553, 429, 5001, 7881, 1493, 20425, 57727}},
	{9943, 17, 35863, {1, 3, 5, 9, 25, 43, 17, 71, 87, 869, 1219, 2661, 4571, 9689, 18799, 62467, 128531}},
	{9944, 17, 35870, {1, 1, 3, 3, 19, 53, 61, 9, 55, 433, 1555, 2369, 1423, 9081, 19185, 8513, 111079}},
	{9945, 17, 35879, {1, 3, 5, 15, 11, 61, 1, 147, 17, 71, 1563, 1113, 4809, 16229, 23743, 59757, 64699}},
	{9946, 17, 35880, {1, 1, 5, 11, 29, 23, 61, 43, 203, 97, 1119, 237, 6445, 14507, 9799, 18447, 14745}},
	{9947, 17, 35891, {1, 3, 5, 15, 11, 17, 117, 139, 117, 537, 251, 149, 2731, 15863, 1381, 25435, 25501}},
	{9948, 17, 35893, {1, 3, 3, 15, 31, 57, 53, 43, 95, 445, 1423, 3833, 2485, 11789, 16011, 8101, 39165}},
	{9949, 17, 35903, {1, 1, 3, 11, 15, 37, 117, 3, 245, 57, 593, 2771, 7181, 11397, 5691, 3217, 44139}},
	{9950, 17, 35905, {1, 3, 5, 1, 11, 13, 121, 85, 85, 511, 1837, 611, 237, 4893, 24025, 28903, 102025}},
	{9951, 17, 35926, {1, 3, 1, 11, 5, 45, 43, 45, 393, 741, 1157, 1511, 1665, 2359, 19071, 24537, 122879}},
	{9952, 17, 35930, {1, 3, 3, 3, 9, 59, 27, 11, 257, 203, 1535, 2729, 2313, 3539, 1689, 31901, 42949}},
	{9953, 17, 35941, {1, 1, 1, 11, 17, 7, 21, 35, 479, 697, 107, 1317, 6585, 705, 3789, 20439, 33375}},
	{9954, 17, 35956, {1, 1, 3, 11, 19, 37, 123, 233, 253, 733, 901, 3047, 3595, 2357, 24533, 40519, 109171}},
	{9955, 17, 35963, {1, 3, 3, 13, 29, 51, 25, 149, 57, 253, 2001, 351, 7367, 15361, 4955, 60951, 19449}},
	{9956, 17, 35970, {1, 1, 3, 15, 21, 53, 25, 239, 257, 437, 711, 3599, 5441, 7405, 15039, 19207, 63841}},
	{9957, 17, 35984, {1, 3, 1, 9, 17, 41, 43, 231, 413, 747, 1447, 1407, 2615, 14529, 10781, 20001, 82713}},
	{9958, 17, 35996, {1, 3, 7, 7, 9, 29, 25, 55, 53, 423, 1711, 2871, 2675, 421, 31703, 57099, 2955}},
	{9959, 17, 36005, {1, 3, 1, 7, 31, 17, 113, 83, 387, 611, 1815, 2137, 3453, 4409, 20377, 60263, 81205}},
	{9960, 17, 36012, {1, 1, 5, 3, 11, 1, 7, 225, 367, 267, 95, 939, 3801, 2619, 1207, 62695, 116407}},
	{9961, 17, 36015, {1, 3, 3, 9, 5, 39, 85, 45, 247, 483, 491, 865, 3493, 8243, 8411, 26449, 50473}},
	{9962, 17, 36030, {1, 3, 3, 9, 1, 53, 23, 127, 13, 529, 1925, 2629, 3451, 15073, 16075, 29909, 34101}},
	{9963, 17, 36035, {1, 3, 1, 11, 1, 9, 125, 57, 79, 633, 979, 3843, 325, 883, 7769, 40155, 104057}},
	{9964, 17, 36042, {1, 1, 7, 13, 23, 53, 27, 157, 493, 901, 1077, 1079, 1327, 15903, 20603, 64377, 103335}},
	{9965, 17, 36047, {1, 3, 3, 3, 3, 35, 37, 167, 73, 301, 385, 1045, 6913, 2269, 22491, 19735, 70125}},
	{9966, 17, 36049, {1, 1, 1, 11, 5, 23, 23, 85, 267, 845, 207, 77, 1245, 16209, 25579, 12417, 48723}},
	{9967, 17, 36059, {1, 1, 5, 15, 11, 17, 43, 83, 373, 1005, 541, 115, 163, 2165, 8181, 35839, 44471}},
	{9968, 17, 36071, {1, 3, 5, 7, 27, 41, 101, 13, 213, 235, 2037, 2179, 2121, 4481, 8127, 20011, 3981}},
	{9969, 17, 36080, {1, 1, 5, 11, 7, 43, 59, 129, 127, 387, 489, 1985, 623, 13307, 19765, 62155, 93271}},
	{9970, 17, 36085, {1, 1, 7, 5, 23, 63, 23, 177, 211, 233, 101, 1809, 7411, 8003, 25101, 32601, 75071}},
	{9971, 17, 36097, {1, 1, 1, 11, 3, 25, 9, 91, 459, 611, 867, 3639, 5457, 9101, 15333, 40069, 67723}},
	{9972, 17, 36110, {1, 3, 7, 5, 3, 29, 111, 75, 459, 195, 1405, 2281, 6085, 4425, 29061, 57335, 87449}},
	{9973, 17, 36115, {1, 3, 7, 11, 21, 45, 53, 81, 77, 863, 1901, 3355, 5253, 10897, 26289, 48399, 26877}},
	{9974, 17, 36118, {1, 3, 3, 13, 21, 37, 69, 87, 259, 101, 1203, 167, 6229, 145, 9355, 15347, 68047}},
	{9975, 17, 36124, {1, 1, 3, 1, 31, 1, 15, 229, 429, 915, 929, 381, 1857, 8441, 22207, 47071, 127853}},
	{9976, 17, 36137, {1, 3, 7, 3, 15, 9, 13, 161, 173, 573, 405, 3253, 7331, 13965, 3061, 40687, 130185}},
	{9977, 17, 36138, {1, 3, 5, 5, 29, 29, 9, 115, 393, 377, 909, 321, 2861, 9881, 17863, 52033, 55133}},
	{9978, 17, 36155, {1, 1, 7, 7, 27, 53, 101, 213, 199, 301, 1995, 2549, 5037, 13639, 18423, 23547, 79359}},
	{9979, 17, 36160, {1, 3, 1, 7, 21, 51, 29, 151, 301, 665, 571, 53, 2637, 7229, 12517, 33647, 49413}},
	{9980, 17, 36189, {1, 3, 3, 13, 13, 49, 49, 131, 325, 273, 1127, 2981, 2365, 14287, 23185, 26915, 81755}},
	{9981, 17, 36190, {1, 1, 5, 3, 17, 45, 25, 79, 37, 265, 1205, 1805, 6707, 11525, 16473, 39525, 9571}},
	{9982, 17, 36203, {1, 3, 3, 15, 9, 43, 55, 101, 469, 939, 365, 3443, 5759, 4751, 28893, 46727, 74569}},
	{9983, 17, 36211, {1, 3, 7, 9, 5, 33, 11, 201, 263, 227, 1475, 2795, 1489, 11129, 18053, 31009, 73105}},
	{9984, 17, 36218, {1, 3, 5, 5, 5, 25, 41, 151, 393, 237, 2017, 3811, 953, 13835, 28761, 22439, 76355}},
	{9985, 17, 36230, {1, 1, 5, 13, 21, 37, 29, 11, 289, 67, 1317, 511, 685, 15227, 8731, 15039, 79491}},
	{9986, 17, 36241, {1, 3, 1, 9, 31, 59, 123, 169, 473, 139, 575, 1057, 3213, 8213, 21845, 28123, 105335}},
	{9987, 17, 36244, {1, 1, 1, 5, 21, 47, 23, 121, 403, 5, 1457, 2137, 569, 9267, 6367, 6991, 3113}},
	{9988, 17, 36253, {1, 3, 3, 7, 13, 7, 25, 215, 81, 1003, 2041, 1317, 3913, 14705, 30551, 50889, 83441}},
	{9989, 17, 36257, {1, 3, 3, 3, 13, 17, 63, 229, 83, 901, 953, 2603, 4685, 6961, 7519, 52441, 33223}},
	{9990, 17, 36264, {1, 3, 7, 5, 7, 57, 65, 73, 243, 531, 261, 2517, 4083, 5889, 22913, 49603, 67135}},
	{9991, 17, 36272, {1, 3, 5, 11, 15, 47, 81, 83, 35, 1021, 1313, 1109, 5103, 5469, 18149, 15307, 34939}},
	{9992, 17, 36290, {1, 3, 7, 5, 21, 13, 105, 157, 435, 23, 931, 3565, 1, 4987, 8829, 7327, 51049}},
	{9993, 17, 36292, {1, 1, 3, 11, 29, 9, 59, 49, 261, 1009, 1953, 2683, 8125, 10937, 16683, 36013, 5967}},
	{9994, 17, 36301, {1, 1, 1, 1, 19, 29, 57, 9, 307, 457, 675, 3023, 495, 15257, 7945, 10449, 30155}},
	{9995, 17, 36309, {1, 1, 7, 13, 25, 9, 51, 135, 491, 205, 1715, 3253, 1031, 4137, 14885, 39925, 6061}},
	{9996, 17, 36313, {1, 1, 7, 7, 3, 13, 111, 91, 469, 133, 1221, 1035, 919, 3697, 26387, 41675, 487}},
	{9997, 17, 36316, {1, 1, 3, 1, 19, 53, 11, 113, 245, 747, 189, 4051, 87, 1767, 3595, 10259, 100097}},
	{9998, 17, 36319, {1, 1, 5, 3, 23, 49, 31, 47, 341, 1019, 723, 2353, 6191, 3809, 3297, 39443, 73529}},
	{9999, 17, 36330, {1, 3, 3, 9, 25, 27, 123, 49, 51, 85, 1063, 2633, 6549, 14493, 7367, 3557, 60651}},
	{10000, 17, 36335, {1, 3, 7, 5, 13, 27, 127, 65, 115, 731, 1147, 283, 91, 14205, 2457, 57083, 35815}},
	{10001, 17, 36347, {1, 3, 3, 3, 25, 63, 99, 249, 25, 951, 733, 3621, 7139, 14223, 23641, 20287, 30743}},
	{10002, 17, 36353, {1, 3, 3, 7, 21, 23, 83, 207, 235, 467, 1857, 2661, 1391, 10097, 12297, 54825, 5035}},
	{10003, 17, 36356, {1, 1, 5, 3, 31, 17, 77, 9, 215, 553, 989, 3643, 729, 2057, 32053, 50305, 5499}},
	{10004, 17, 36368, {1, 1, 7, 1, 23, 5, 111, 195, 431, 947, 403, 1781, 943, 15073, 67, 52225, 98987}},
	{10005, 17, 36374, {1, 1, 5, 11, 23, 1, 41, 33, 457, 767, 275, 801, 5119, 3781, 14805, 52789, 41775}},
	{10006, 17, 36377, {1, 1, 5, 3, 9, 53, 15, 183, 281, 691, 165, 3277, 7673, 1509, 16605, 53799, 100185}},
	{10007, 17, 36384, {1, 3, 5, 11, 19, 45, 29, 159, 167, 67, 1259, 879, 7787, 8855, 24153, 42667, 102855}},
	{10008, 17, 36407, {1, 1, 7, 13, 31, 19, 43, 133, 295, 287, 1985, 2451, 2297, 3853, 22401, 27659, 11149}},
	{10009, 17, 36413, {1, 1, 7, 13, 31, 39, 125, 21, 173, 103, 1119, 3739, 6467, 2113, 4465, 26537, 129949}},
	{10010, 17, 36419, {1, 1, 5, 15, 21, 47, 35, 125, 199, 335, 421, 31, 185, 12769, 30659, 33427, 106981}},
	{10011, 17, 36425, {1, 3, 5, 13, 25, 35, 53, 253, 325, 921, 1705, 2735, 6437, 2287, 20479, 61107, 91453}},
	{10012, 17, 36426, {1, 3, 7, 13, 25, 63, 83, 183, 5, 401, 329, 525, 3141, 393, 30469, 16529, 9605}},
	{10013, 17, 36446, {1, 3, 3, 13, 19, 23, 15, 85, 323, 545, 149, 3645, 6269, 15595, 18453, 39, 128169}},
	{10014, 17, 36461, {1, 3, 7, 15, 17, 5, 61, 61, 91, 353, 1039, 2959, 4147, 13205, 12599, 53281, 39509}},
	{10015, 17, 36467, {1, 1, 3, 7, 21, 9, 97, 111, 249, 775, 845, 1789, 667, 489, 6689, 29217, 56527}},
	{10016, 17, 36474, {1, 3, 5, 7, 11, 5, 59, 219, 29, 803, 923, 3861, 7953, 8969, 1819, 43501, 20513}},
	{10017, 17, 36480, {1, 1, 5, 11, 7, 53, 63, 231, 193, 293, 1467, 1409, 6397, 13237, 15903, 19271, 66257}},
	{10018, 17, 36486, {1, 3, 1, 15, 23, 15, 37, 123, 189, 63, 1121, 751, 6711, 10095, 6493, 40709, 47641}},
	{10019, 17, 36489, {1, 3, 7, 3, 23, 59, 99, 183, 249, 479, 771, 1087, 7979, 409, 4819, 4337, 33345}},
	{10020, 17, 36495, {1, 1, 5, 1, 17, 7, 15, 167, 305, 411, 1429, 3127, 23, 9123, 7185, 44405, 114841}},
	{10021, 17, 36525, {1, 1, 5, 11, 3, 29, 29, 31, 399, 777, 251, 1841, 3607, 211, 23543, 29111, 54565}},
	{10022, 17, 36526, {1, 3, 3, 9, 27, 33, 79, 27, 469, 67, 1327, 183, 5783, 10039, 13165, 20443, 4913}},
	{10023, 17, 36533, {1, 3, 7, 15, 21, 23, 5, 227, 141, 1021, 69, 3347, 7221, 13837, 20921, 20525, 32567}},
	{10024, 17, 36534, {1, 1, 5, 5, 25, 53, 73, 111, 319, 311, 1597, 1809, 5343, 13963, 6613, 14471, 53871}},
	{10025, 17, 36540, {1, 3, 3, 1, 15, 57, 47, 205, 53, 471, 185, 273, 8077, 5031, 31195, 30859, 15979}},
	{10026, 17, 36555, {1, 1, 3, 5, 23, 15, 87, 211, 83, 265, 1629, 2979, 69, 12559, 30455, 36363, 61461}},
	{10027, 17, 36563, {1, 1, 7, 7, 1, 47, 5, 199, 95, 17, 57, 1887, 6847, 9501, 21361, 57763, 77069}},
	{10028, 17, 36565, {1, 1, 3, 5, 9, 15, 15, 149, 141, 605, 639, 2197, 7237, 5753, 9415, 4677, 129947}},
	{10029, 17, 36588, {1, 3, 7, 1, 7, 9, 29, 249, 275, 461, 1667, 4093, 5763, 3205, 24079, 11883, 86455}},
	{10030, 17, 36593, {1, 1, 3, 5, 15, 39, 117, 145, 153, 671, 1819, 111, 3607, 12279, 4927, 63759, 42905}},
	{10031, 17, 36596, {1, 1, 1, 5, 31, 5, 35, 183, 189, 839, 1811, 1877, 6545, 11373, 27947, 27183, 29857}},
	{10032, 17, 36606, {1, 3, 5, 7, 29, 47, 3, 183, 511, 145, 1953, 3419, 6385, 7745, 12823, 59783, 69399}},
	{10033, 17, 36614, {1, 3, 5, 9, 5, 39, 85, 145, 33, 899, 1009, 2035, 6145, 3855, 20583, 4329, 95231}},
	{10034, 17, 36626, {1, 1, 3, 3, 15, 61, 85, 181, 247, 705, 413, 1633, 7489, 1785, 30397, 42851, 80197}},
	{10035, 17, 36628, {1, 3, 3, 13, 23, 11, 3, 97, 307, 183, 113, 3881, 7455, 8327, 6749, 23977, 101629}},
	{10036, 17, 36641, {1, 1, 7, 13, 1, 23, 59, 219, 125, 789, 1401, 707, 6915, 6275, 25813, 46595, 54119}},
	{10037, 17, 36642, {1, 3, 7, 9, 5, 7, 37, 33, 165, 181, 833, 1993, 4541, 5799, 23323, 39825, 44575}},
	{10038, 17, 36651, {1, 3, 1, 13, 13, 43, 69, 219, 437, 521, 503, 2293, 3607, 6845, 22583, 291, 65645}},
	{10039, 17, 36653, {1, 1, 7, 9, 29, 13, 123, 67, 191, 933, 1875, 1223, 5525, 13797, 29771, 58191, 84469}},
	{10040, 17, 36673, {1, 1, 7, 7, 3, 57, 101, 69, 23, 239, 1023, 3289, 1541, 6245, 23379, 161, 61155}},
	{10041, 17, 36676, {1, 3, 7, 13, 25, 33, 49, 145, 487, 681, 451, 1719, 109, 16273, 20009, 3003, 115815}},
	{10042, 17, 36679, {1, 1, 5, 11, 11, 59, 41, 133, 303, 469, 1975, 847, 5291, 13947, 8759, 8533, 25099}},
	{10043, 17, 36694, {1, 1, 1, 1, 29, 31, 53, 11, 239, 57, 1627, 1247, 1577, 3269, 20751, 4627, 40499}},
	{10044, 17, 36698, {1, 3, 7, 15, 1, 1, 51, 39, 383, 203, 1841, 3867, 4975, 9937, 1863, 52611, 83189}},
	{10045, 17, 36704, {1, 3, 7, 7, 13, 59, 15, 217, 355, 945, 1317, 815, 2413, 10985, 30647, 37745, 126553}},
	{10046, 17, 36714, {1, 1, 3, 11, 7, 29, 101, 137, 97, 119, 927, 3269, 6977, 4253, 10741, 61907, 122815}},
	{10047, 17, 36721, {1, 3, 3, 1, 29, 5, 49, 137, 411, 349, 905, 2481, 4961, 4513, 29409, 19503, 77915}},
	{10048, 17, 36722, {1, 1, 7, 13, 29, 59, 93, 61, 393, 29, 257, 3601, 6281, 5105, 17339, 53827, 83137}},
	{10049, 17, 36727, {1, 1, 1, 13, 5, 23, 61, 7, 51, 161, 737, 1549, 6021, 3385, 5539, 21261, 69995}},
	{10050, 17, 36749, {1, 1, 1, 15, 31, 1, 21, 113, 481, 7, 175, 717, 1593, 5937, 12347, 51835, 66649}},
	{10051, 17, 36758, {1, 1, 3, 7, 9, 51, 9, 199, 39, 607, 1157, 3913, 7767, 14195, 28721, 27655, 34709}},
	{10052, 17, 36761, {1, 3, 5, 5, 1, 15, 49, 33, 441, 721, 1749, 1497, 2023, 8351, 12641, 11861, 78545}},
	{10053, 17, 36771, {1, 3, 1, 7, 7, 17, 103, 113, 243, 25, 889, 1419, 3163, 12401, 22459, 39037, 101719}},
	{10054, 17, 36788, {1, 1, 7, 11, 17, 45, 121, 215, 3, 409, 1871, 2149, 4249, 5071, 14277, 55869, 91233}},
	{10055, 17, 36797, {1, 1, 3, 7, 19, 31, 47, 241, 175, 749, 1709, 355, 6037, 10555, 24107, 64683, 42673}},
	{10056, 17, 36805, {1, 3, 7, 11, 5, 21, 105, 137, 307, 101, 417, 1903, 1027, 10257, 27767, 9755, 92105}},
	{10057, 17, 36830, {1, 1, 3, 13, 9, 59, 11, 63, 295, 923, 401, 1471, 3517, 7761, 28855, 11525, 72455}},
	{10058, 17, 36833, {1, 1, 7, 15, 31, 51, 77, 29, 323, 579, 1313, 3441, 2903, 1683, 20605, 8185, 29753}},
	{10059, 17, 36839, {1, 1, 5, 15, 11, 59, 119, 109, 233, 1001, 1527, 2709, 73, 5311, 18313, 27155, 85999}},
	{10060, 17, 36843, {1, 3, 1, 5, 9, 59, 105, 93, 213, 401, 839, 3225, 3263, 13501, 2413, 60367, 121281}},
	{10061, 17, 36860, {1, 1, 7, 3, 19, 25, 75, 27, 325, 435, 527, 1465, 3601, 5785, 6135, 32841, 60129}},
	{10062, 17, 36866, {1, 1, 3, 7, 31, 19, 37, 157, 189, 51, 869, 2963, 5269, 9151, 14845, 30441, 89685}},
	{10063, 17, 36871, {1, 3, 3, 9, 17, 51, 23, 177, 417, 255, 1739, 3085, 7811, 15177, 25433, 38487, 51021}},
	{10064, 17, 36875, {1, 1, 3, 7, 27, 1, 45, 235, 59, 491, 1327, 3967, 7585, 4313, 29669, 47193, 89427}},
	{10065, 17, 36877, {1, 1, 3, 9, 19, 5, 27, 63, 263, 593, 1599, 1311, 1029, 603, 25291, 51391, 98915}},
	{10066, 17, 36880, {1, 3, 3, 15, 11, 7, 97, 99, 263, 155, 437, 3849, 2665, 3371, 8179, 51883, 3601}},
	{10067, 17, 36892, {1, 1, 3, 15, 7, 35, 37, 149, 251, 619, 1423, 553, 4453, 16365, 22543, 6951, 34655}},
	{10068, 17, 36911, {1, 3, 3, 11, 15, 21, 95, 143, 31, 425, 179, 2383, 4799, 7655, 26945, 9273, 103469}},
	{10069, 17, 36914, {1, 3, 1, 9, 13, 49, 3, 117, 361, 459, 227, 2067, 4909, 13461, 22505, 10259, 59697}},
	{10070, 17, 36916, {1, 1, 7, 7, 7, 23, 67, 217, 313, 965, 1747, 995, 579, 6217, 8915, 49329, 851}},
	{10071, 17, 36923, {1, 1, 3, 1, 17, 19, 7, 99, 281, 207, 1685, 2401, 967, 9399, 28741, 28839, 6003}},
	{10072, 17, 36940, {1, 3, 3, 5, 31, 61, 105, 251, 499, 319, 1167, 2203, 1195, 2663, 11797, 12981, 125523}},
	{10073, 17, 36943, {1, 3, 1, 5, 23, 19, 99, 101, 85, 837, 501, 2737, 4051, 2413, 9275, 38995, 21633}},
	{10074, 17, 36948, {1, 3, 7, 13, 17, 17, 119, 75, 281, 527, 1477, 1515, 7765, 5573, 10143, 6219, 57817}},
	{10075, 17, 36957, {1, 1, 5, 11, 19, 35, 85, 171, 107, 905, 1395, 1199, 7345, 15719, 14021, 47425, 36081}},
	{10076, 17, 36958, {1, 1, 3, 9, 9, 63, 109, 15, 323, 73, 1541, 2227, 5197, 12617, 23379, 53415, 105291}},
	{10077, 17, 36967, {1, 3, 3, 5, 5, 41, 85, 99, 3, 895, 1383, 3627, 3897, 1893, 23673, 56501, 78411}},
	{10078, 17, 36974, {1, 1, 7, 1, 25, 27, 45, 185, 475, 577, 1619, 727, 1407, 2383, 9215, 55295, 27349}},
	{10079, 17, 36981, {1, 3, 7, 11, 3, 51, 53, 53, 399, 711, 1075, 511, 5369, 10777, 14419, 63217, 130181}},
	{10080, 17, 37001, {1, 1, 7, 13, 25, 19, 107, 71, 151, 73, 735, 3837, 5307, 10229, 10529, 9989, 111925}},
	{10081, 17, 37012, {1, 1, 1, 15, 19, 59, 65, 77, 465, 957, 1085, 1359, 3959, 15823, 6273, 12565, 126167}},
	{10082, 17, 37015, {1, 1, 5, 5, 31, 53, 23, 173, 407, 795, 41, 3275, 1953, 13673, 26625, 33477, 14149}},
	{10083, 17, 37019, {1, 1, 7, 7, 1, 11, 121, 139, 77, 321, 1939, 2597, 621, 9579, 11629, 13119, 30505}},
	{10084, 17, 37035, {1, 1, 1, 5, 3, 33, 45, 127, 169, 581, 1521, 1019, 6489, 1069, 2469, 40255, 66619}},
	{10085, 17, 37040, {1, 3, 7, 5, 29, 47, 7, 245, 459, 417, 1027, 857, 4905, 11255, 3267, 9491, 78013}},
	{10086, 17, 37063, {1, 3, 5, 9, 25, 49, 61, 215, 19, 731, 303, 1001, 6031, 3705, 7797, 31957, 119383}},
	{10087, 17, 37064, {1, 3, 5, 5, 1, 9, 37, 187, 235, 453, 963, 2833, 3501, 605, 2763, 41215, 93547}},
	{10088, 17, 37069, {1, 3, 1, 1, 21, 3, 41, 53, 425, 687, 1051, 2365, 7835, 3981, 5557, 61993, 127417}},
	{10089, 17, 37077, {1, 3, 3, 7, 13, 61, 41, 189, 261, 163, 1931, 1803, 2379, 16379, 25453, 17911, 123431}},
	{10090, 17, 37093, {1, 1, 7, 15, 23, 21, 95, 7, 27, 897, 721, 3917, 7971, 4643, 5223, 46583, 32453}},
	{10091, 17, 37097, {1, 1, 7, 7, 1, 25, 83, 109, 223, 573, 533, 449, 6477, 10719, 28705, 8283, 94963}},
	{10092, 17, 37106, {1, 1, 5, 13, 21, 45, 63, 31, 21, 223, 31, 1249, 425, 7199, 11539, 7731, 44333}},
	{10093, 17, 37115, {1, 1, 5, 15, 29, 5, 87, 215, 287, 567, 297, 451, 5867, 15511, 1005, 57469, 87257}},
	{10094, 17, 37118, {1, 3, 5, 11, 13, 51, 117, 139, 377, 1015, 1237, 2053, 7625, 1003, 22673, 64345, 16203}},
	{10095, 17, 37123, {1, 1, 3, 15, 19, 39, 73, 205, 185, 331, 869, 857, 5043, 7247, 25253, 5799, 64857}},
	{10096, 17, 37129, {1, 3, 7, 1, 25, 63, 125, 47, 161, 289, 373, 1603, 1663, 1123, 28907, 37855, 47935}},
	{10097, 17, 37130, {1, 1, 7, 15, 9, 17, 97, 63, 79, 123, 1357, 3055, 2323, 16083, 21861, 38743, 81291}},
	{10098, 17, 37135, {1, 1, 3, 15, 5, 23, 7, 159, 127, 511, 55, 2691, 6823, 16151, 8059, 43021, 18911}},
	{10099, 17, 37137, {1, 1, 3, 9, 27, 19, 41, 75, 375, 921, 1745, 35, 1189, 5857, 29869, 43827, 16899}},
	{10100, 17, 37138, {1, 1, 1, 5, 3, 21, 13, 235, 51, 529, 291, 2619, 5419, 12573, 10907, 8865, 54987}},
	{10101, 17, 37140, {1, 3, 1, 13, 7, 9, 85, 131, 159, 743, 1671, 3001, 4559, 12343, 27563, 49941, 68447}},
	{10102, 17, 37144, {1, 1, 7, 5, 17, 61, 99, 63, 199, 383, 485, 2569, 5329, 645, 18805, 20421, 101229}},
	{10103, 17, 37149, {1, 1, 1, 15, 3, 59, 41, 247, 213, 843, 2003, 125, 7755, 4203, 20277, 47195, 48249}},
	{10104, 17, 37156, {1, 1, 5, 15, 15, 17, 113, 101, 27, 811, 1791, 1777, 749, 14317, 17267, 54467, 118369}},
	{10105, 17, 37159, {1, 3, 3, 3, 19, 37, 23, 117, 275, 733, 1259, 567, 1769, 12071, 5413, 49411, 99259}},
	{10106, 17, 37163, {1, 3, 1, 11, 3, 27, 103, 113, 251, 731, 481, 2771, 3205, 14151, 19403, 30307, 114691}},
	{10107, 17, 37165, {1, 1, 5, 15, 19, 15, 103, 25, 357, 197, 1437, 3621, 4747, 773, 5769, 33465, 28307}},
	{10108, 17, 37183, {1, 1, 5, 15, 5, 17, 89, 87, 423, 611, 549, 2549, 1275, 14545, 2931, 3853, 24577}},
	{10109, 17, 37185, {1, 3, 5, 1, 15, 13, 29, 49, 279, 495, 697, 1015, 4899, 15977, 10765, 47979, 40237}},
	{10110, 17, 37195, {1, 3, 3, 9, 31, 51, 21, 5, 279, 947, 1871, 3075, 5433, 1631, 30075, 30517, 99609}},
	{10111, 17, 37198, {1, 1, 1, 15, 19, 63, 79, 81, 19, 629, 617, 1887, 4015, 15501, 10551, 56419, 108739}},
	{10112, 17, 37203, {1, 1, 3, 9, 31, 15, 45, 37, 43, 349, 1357, 189, 4551, 9363, 15683, 48445, 89279}},
	{10113, 17, 37212, {1, 1, 1, 1, 17, 19, 121, 119, 397, 947, 1797, 613, 1627, 9591, 15779, 62295, 118843}},
	{10114, 17, 37233, {1, 1, 1, 7, 25, 55, 71, 227, 507, 497, 1209, 2919, 5733, 15785, 21437, 40043, 2325}},
	{10115, 17, 37236, {1, 1, 1, 15, 11, 1, 59, 93, 69, 859, 67, 1831, 6345, 5643, 29515, 20337, 77281}},
	{10116, 17, 37240, {1, 3, 5, 9, 19, 53, 59, 63, 161, 853, 697, 1441, 3457, 951, 29659, 15337, 38443}},
	{10117, 17, 37256, {1, 3, 1, 9, 7, 21, 73, 81, 89, 291, 411, 3793, 4639, 2829, 6855, 38113, 32875}},
	{10118, 17, 37264, {1, 1, 7, 1, 15, 3, 79, 35, 363, 459, 907, 1157, 5165, 8021, 10135, 36367, 111991}},
	{10119, 17, 37267, {1, 3, 5, 13, 21, 23, 63, 155, 393, 869, 1553, 3345, 2711, 8249, 24907, 28111, 36667}},
	{10120, 17, 37273, {1, 1, 7, 11, 15, 25, 29, 93, 45, 637, 1473, 2053, 313, 8047, 23411, 8643, 2925}},
	{10121, 17, 37295, {1, 3, 7, 9, 11, 5, 73, 69, 311, 949, 2017, 259, 2861, 10547, 12017, 34125, 74101}},
	{10122, 17, 37315, {1, 3, 1, 13, 19, 61, 115, 59, 447, 787, 1621, 2221, 7841, 5329, 18137, 13857, 51889}},
	{10123, 17, 37330, {1, 3, 7, 13, 1, 23, 117, 49, 449, 541, 7, 3269, 1725, 6677, 15979, 4319, 40919}},
	{10124, 17, 37336, {1, 3, 5, 5, 17, 29, 35, 123, 3, 481, 305, 1589, 4319, 5183, 31907, 53019, 49375}},
	{10125, 17, 37339, {1, 3, 1, 7, 11, 59, 79, 89, 479, 821, 763, 3597, 7457, 13775, 11213, 22777, 80379}},
	{10126, 17, 37342, {1, 1, 3, 7, 13, 17, 65, 155, 335, 671, 331, 895, 7459, 1719, 10675, 60109, 63143}},
	{10127, 17, 37357, {1, 3, 5, 1, 29, 33, 105, 249, 61, 469, 1629, 3777, 4393, 14457, 11701, 6065, 2635}},
	{10128, 17, 37365, {1, 3, 7, 3, 13, 13, 21, 15, 363, 63, 1263, 1479, 1459, 6577, 7481, 30393, 19831}},
	{10129, 17, 37372, {1, 1, 3, 7, 29, 25, 71, 247, 501, 815, 1697, 2457, 4975, 3821, 25759, 24901, 120603}},
	{10130, 17, 37381, {1, 1, 1, 5, 19, 3, 59, 163, 367, 779, 47, 905, 897, 3293, 13951, 25497, 99151}},
	{10131, 17, 37399, {1, 3, 1, 5, 11, 47, 21, 171, 123, 215, 1797, 3741, 4921, 7213, 4847, 3239, 114839}},
	{10132, 17, 37416, {1, 3, 3, 5, 23, 63, 57, 31, 409, 431, 1337, 3301, 4695, 7401, 9383, 12639, 34347}},
	{10133, 17, 37429, {1, 3, 3, 5, 27, 57, 29, 147, 111, 1015, 815, 1509, 3967, 7255, 15109, 26001, 90775}},
	{10134, 17, 37436, {1, 1, 7, 13, 31, 45, 21, 99, 377, 399, 255, 459, 6043, 11055, 5675, 3333, 32813}},
	{10135, 17, 37442, {1, 3, 1, 7, 1, 55, 121, 77, 429, 433, 297, 3181, 3029, 6777, 22795, 61515, 58553}},
	{10136, 17, 37451, {1, 3, 5, 9, 1, 19, 121, 1, 499, 589, 1597, 2219, 1029, 4223, 31613, 45685, 53517}},
	{10137, 17, 37453, {1, 3, 1, 9, 29, 39, 83, 193, 43, 41, 467, 1711, 2761, 10635, 15503, 38043, 120615}},
	{10138, 17, 37475, {1, 1, 7, 13, 27, 61, 1, 181, 163, 613, 221, 63, 6147, 8215, 15093, 2417, 71489}},
	{10139, 17, 37482, {1, 1, 7, 15, 31, 63, 47, 139, 427, 847, 53, 1275, 1019, 9455, 12537, 22467, 129947}},
	{10140, 17, 37489, {1, 1, 5, 3, 7, 1, 67, 189, 501, 319, 37, 2849, 2535, 10917, 11115, 48083, 67255}},
	{10141, 17, 37490, {1, 1, 3, 13, 7, 31, 69, 137, 19, 73, 1553, 3945, 2381, 8761, 3977, 24291, 128189}},
	{10142, 17, 37523, {1, 3, 5, 11, 1, 59, 43, 229, 301, 771, 559, 195, 1675, 12605, 22211, 2915, 90351}},
	{10143, 17, 37525, {1, 3, 3, 9, 13, 27, 97, 33, 273, 229, 1537, 1179, 6985, 11679, 17889, 44673, 126641}},
	{10144, 17, 37530, {1, 1, 7, 3, 31, 29, 41, 123, 491, 639, 269, 45, 2155, 14103, 6725, 50781, 42785}},
	{10145, 17, 37535, {1, 3, 5, 9, 9, 11, 89, 249, 475, 701, 1029, 985, 8167, 439, 31897, 24529, 45759}},
	{10146, 17, 37539, {1, 1, 5, 11, 9, 39, 127, 179, 15, 135, 1437, 3331, 5553, 939, 15319, 64937, 110783}},
	{10147, 17, 37548, {1, 3, 1, 5, 7, 61, 1, 219, 111, 801, 85, 3427, 2533, 12861, 5395, 28969, 48091}},
	{10148, 17, 37559, {1, 1, 1, 9, 23, 57, 77, 41, 61, 635, 457, 231, 8121, 5349, 27021, 64807, 87563}},
	{10149, 17, 37560, {1, 3, 5, 7, 31, 31, 101, 155, 255, 199, 1973, 903, 7681, 15379, 12845, 47943, 60663}},
	{10150, 17, 37566, {1, 1, 5, 7, 1, 7, 71, 121, 323, 669, 193, 1209, 267, 9, 21223, 22037, 121567}},
	{10151, 17, 37597, {1, 3, 1, 5, 17, 29, 97, 189, 219, 813, 187, 1763, 5817, 13185, 467, 40159, 18037}},
	{10152, 17, 37601, {1, 1, 7, 9, 7, 59, 3, 189, 379, 843, 631, 3945, 2909, 4191, 30343, 11223, 105629}},
	{10153, 17, 37602, {1, 3, 1, 3, 15, 17, 23, 73, 439, 699, 657, 451, 6139, 15869, 4101, 32327, 55485}},
	{10154, 17, 37604, {1, 3, 3, 5, 21, 37, 87, 157, 205, 493, 705, 1539, 2193, 13539, 2797, 49063, 55595}},
	{10155, 17, 37616, {1, 1, 5, 11, 11, 41, 5, 131, 445, 781, 1153, 1371, 6763, 3101, 32449, 16065, 86579}},
	{10156, 17, 37622, {1, 3, 5, 1, 23, 51, 97, 87, 161, 261, 269, 2035, 2139, 3049, 32217, 25189, 93571}},
	{10157, 17, 37658, {1, 3, 1, 11, 23, 1, 111, 45, 19, 19, 1767, 3571, 6027, 3593, 17453, 53821, 28121}},
	{10158, 17, 37660, {1, 3, 3, 5, 17, 5, 17, 247, 5, 73, 29, 443, 7713, 15803, 22311, 56755, 100119}},
	{10159, 17, 37667, {1, 3, 1, 13, 7, 1, 41, 139, 317, 977, 1529, 1217, 529, 3231, 21491, 28461, 96699}},
	{10160, 17, 37684, {1, 3, 1, 13, 11, 41, 103, 99, 81, 849, 231, 1729, 761, 711, 11499, 25581, 59433}},
	{10161, 17, 37693, {1, 1, 5, 5, 13, 33, 79, 175, 89, 29, 295, 2867, 1197, 6137, 32063, 23471, 21721}},
	{10162, 17, 37694, {1, 1, 5, 15, 17, 29, 123, 249, 273, 437, 443, 2601, 3957, 11955, 261, 54863, 85727}},
	{10163, 17, 37696, {1, 1, 5, 13, 3, 31, 57, 205, 3, 903, 905, 3851, 757, 13761, 28615, 48185, 33227}},
	{10164, 17, 37706, {1, 3, 7, 1, 1, 1, 107, 15, 307, 405, 45, 297, 4365, 1569, 9263, 13685, 36027}},
	{10165, 17, 37711, {1, 3, 1, 9, 17, 61, 113, 121, 249, 743, 191, 2523, 6621, 5395, 23797, 57975, 51675}},
	{10166, 17, 37730, {1, 3, 5, 3, 27, 21, 49, 113, 59, 989, 501, 2651, 3827, 5121, 29667, 32903, 84199}},
	{10167, 17, 37735, {1, 1, 7, 7, 19, 43, 11, 191, 143, 93, 1167, 2521, 2569, 12187, 28575, 13073, 113545}},
	{10168, 17, 37742, {1, 1, 7, 3, 27, 39, 11, 85, 61, 979, 49, 2191, 2607, 13967, 28123, 48903, 16327}},
	{10169, 17, 37744, {1, 1, 5, 3, 17, 17, 1, 149, 189, 1017, 705, 3119, 6441, 1595, 30533, 18795, 34265}},
	{10170, 17, 37759, {1, 1, 3, 11, 31, 11, 125, 109, 39, 41, 191, 2615, 1377, 16089, 8793, 31425, 90507}},
	{10171, 17, 37777, {1, 1, 1, 1, 23, 15, 21, 245, 337, 649, 585, 2893, 927, 883, 15119, 2595, 127963}},
	{10172, 17, 37783, {1, 3, 5, 3, 13, 17, 123, 167, 471, 5, 1671, 2787, 5081, 12903, 4257, 19213, 2503}},
	{10173, 17, 37784, {1, 1, 5, 7, 21, 57, 75, 171, 509, 591, 85, 407, 1747, 6375, 19641, 55683, 111289}},
	{10174, 17, 37793, {1, 1, 7, 7, 3, 31, 121, 111, 19, 361, 1033, 4033, 2359, 13451, 15095, 61817, 69683}},
	{10175, 17, 37800, {1, 1, 5, 9, 21, 33, 83, 179, 387, 69, 1085, 2147, 2751, 10899, 16971, 40623, 110891}},
	{10176, 17, 37818, {1, 1, 7, 9, 11, 45, 81, 71, 73, 551, 145, 159, 7519, 3459, 5197, 48913, 59045}},
	{10177, 17, 37823, {1, 3, 1, 3, 7, 35, 17, 249, 207, 767, 1189, 1451, 4351, 3673, 28807, 671, 69271}},
	{10178, 17, 37825, {1, 3, 1, 15, 21, 27, 81, 243, 55, 191, 1497, 3205, 1601, 705, 14891, 14403, 130729}},
	{10179, 17, 37828, {1, 3, 1, 7, 17, 43, 41, 123, 507, 201, 1873, 3547, 5681, 1819, 4251, 39661, 57923}},
	{10180, 17, 37832, {1, 3, 1, 9, 3, 59, 57, 235, 445, 479, 961, 1937, 2229, 2511, 15235, 59707, 72261}},
	{10181, 17, 37849, {1, 1, 5, 13, 9, 63, 67, 217, 63, 259, 175, 2469, 3075, 12365, 7727, 42215, 12635}},
	{10182, 17, 37856, {1, 1, 3, 13, 11, 9, 125, 131, 17, 399, 675, 767, 7349, 10433, 21615, 46823, 3955}},
	{10183, 17, 37861, {1, 1, 3, 15, 19, 53, 73, 171, 125, 531, 1093, 1449, 2931, 10897, 12263, 9799, 98251}},
	{10184, 17, 37862, {1, 1, 5, 5, 11, 27, 33, 9, 503, 545, 339, 1099, 1973, 13261, 26871, 14569, 22755}},
	{10185, 17, 37886, {1, 3, 7, 1, 19, 5, 79, 133, 247, 1021, 1431, 3707, 4603, 3285, 5469, 46963, 98203}},
	{10186, 17, 37893, {1, 3, 5, 7, 15, 11, 87, 169, 495, 763, 1295, 2533, 4213, 8671, 21683, 12521, 90071}},
	{10187, 17, 37915, {1, 1, 3, 1, 17, 55, 7, 165, 313, 659, 49, 377, 6675, 15255, 9881, 11751, 87789}},
	{10188, 17, 37922, {1, 3, 1, 15, 3, 49, 27, 109, 145, 1011, 1939, 3201, 6141, 7229, 20741, 59285, 129365}},
	{10189, 17, 37936, {1, 1, 5, 5, 5, 51, 117, 17, 363, 795, 1343, 2637, 6209, 1045, 22515, 10687, 48487}},
	{10190, 17, 37939, {1, 1, 5, 15, 1, 37, 91, 241, 245, 703, 505, 3369, 6163, 10265, 12497, 46301, 109523}},
	{10191, 17, 37945, {1, 1, 7, 11, 3, 37, 67, 35, 229, 823, 193, 913, 3331, 4475, 9271, 11859, 52709}},
	{10192, 17, 37956, {1, 1, 3, 3, 7, 25, 61, 159, 81, 1011, 1491, 1439, 1031, 765, 9839, 61891, 20969}},
	{10193, 17, 37960, {1, 3, 3, 7, 25, 39, 73, 59, 101, 101, 225, 1105, 5943, 5223, 12585, 16411, 62699}},
	{10194, 17, 37971, {1, 1, 7, 5, 25, 19, 27, 113, 465, 319, 2035, 2127, 1319, 11793, 26821, 44805, 28217}},
	{10195, 17, 37980, {1, 3, 5, 11, 7, 9, 81, 107, 67, 31, 1503, 3303, 4451, 11417, 32681, 26861, 54845}},
	{10196, 17, 38002, {1, 3, 1, 1, 3, 51, 93, 235, 93, 247, 2027, 1517, 6797, 1703, 10233, 45313, 60771}},
	{10197, 17, 38007, {1, 3, 3, 15, 25, 11, 83, 77, 413, 189, 119, 597, 2199, 12347, 7935, 40191, 125569}},
	{10198, 17, 38018, {1, 3, 7, 9, 11, 3, 77, 31, 89, 163, 1993, 3017, 3973, 10943, 22247, 45565, 7261}},
	{10199, 17, 38020, {1, 3, 7, 7, 15, 13, 7, 155, 373, 893, 607, 3521, 7455, 13809, 6145, 31743, 86329}},
	{10200, 17, 38038, {1, 1, 1, 15, 25, 41, 111, 65, 11, 627, 59, 2725, 995, 3761, 25361, 45189, 48355}},
	{10201, 17, 38041, {1, 1, 7, 5, 29, 43, 91, 139, 323, 503, 679, 4079, 403, 1899, 1425, 26989, 117057}},
	{10202, 17, 38044, {1, 3, 5, 13, 1, 17, 19, 61, 205, 833, 345, 1031, 7995, 999, 27469, 15943, 88011}},
	{10203, 17, 38048, {1, 3, 1, 7, 23, 49, 123, 9, 11, 761, 1163, 669, 3837, 15225, 23393, 19513, 9457}},
	{10204, 17, 38051, {1, 1, 5, 9, 9, 33, 29, 123, 277, 433, 1799, 1583, 3133, 13461, 26443, 15807, 80173}},
	{10205, 17, 38054, {1, 3, 7, 3, 31, 29, 77, 105, 297, 617, 491, 647, 6541, 5033, 31841, 48405, 126985}},
	{10206, 17, 38065, {1, 1, 5, 5, 31, 39, 17, 25, 3, 279, 89, 3985, 3333, 5681, 3701, 36319, 12585}},
	{10207, 17, 38066, {1, 1, 7, 13, 13, 19, 93, 129, 51, 875, 1083, 1739, 5193, 6217, 10033, 51839, 66071}},
	{10208, 17, 38077, {1, 1, 7, 9, 15, 23, 93, 121, 507, 115, 707, 3181, 1521, 9609, 4577, 54389, 19167}},
	{10209, 17, 38090, {1, 3, 3, 7, 17, 51, 19, 29, 387, 711, 1105, 1627, 4421, 15183, 14149, 26485, 106425}},
	{10210, 17, 38109, {1, 3, 3, 15, 25, 59, 11, 45, 259, 1019, 1997, 3373, 5083, 5701, 30217, 44845, 67559}},
	{10211, 17, 38119, {1, 3, 3, 9, 17, 47, 5, 103, 477, 785, 235, 1523, 1505, 8811, 15255, 53493, 4383}},
	{10212, 17, 38120, {1, 3, 7, 3, 7, 37, 73, 247, 397, 409, 1065, 525, 5665, 8533, 30627, 19035, 22937}},
	{10213, 17, 38123, {1, 3, 3, 15, 15, 47, 123, 215, 413, 249, 55, 2563, 8033, 8743, 18659, 7947, 56057}},
	{10214, 17, 38146, {1, 1, 7, 3, 11, 61, 103, 109, 313, 293, 149, 999, 901, 13387, 15351, 52973, 68385}},
	{10215, 17, 38155, {1, 1, 1, 5, 31, 13, 57, 43, 263, 141, 335, 2777, 3435, 4231, 20623, 2597, 33481}},
	{10216, 17, 38160, {1, 1, 7, 13, 21, 53, 101, 75, 237, 275, 1903, 3501, 8023, 3651, 19609, 44417, 60287}},
	{10217, 17, 38181, {1, 1, 1, 1, 13, 43, 83, 255, 355, 567, 1781, 2943, 1061, 2701, 24861, 58381, 60255}},
	{10218, 17, 38188, {1, 1, 3, 9, 25, 5, 81, 85, 445, 857, 517, 3687, 2641, 6699, 19273, 4481, 8715}},
	{10219, 17, 38199, {1, 1, 3, 7, 17, 39, 33, 31, 29, 269, 379, 3149, 4731, 10387, 7941, 49199, 18423}},
	{10220, 17, 38203, {1, 1, 7, 15, 19, 37, 105, 157, 185, 1023, 1865, 53, 6765, 3, 22897, 17019, 109521}},
	{10221, 17, 38225, {1, 3, 7, 5, 5, 7, 117, 211, 19, 149, 1091, 3721, 201, 4455, 18965, 51401, 67225}},
	{10222, 17, 38226, {1, 3, 5, 11, 1, 55, 101, 41, 469, 271, 1251, 949, 861, 11903, 14773, 25675, 114161}},
	{10223, 17, 38231, {1, 1, 7, 7, 23, 13, 103, 185, 137, 575, 797, 1195, 5301, 13307, 12043, 26003, 31719}},
	{10224, 17, 38244, {1, 1, 5, 7, 11, 51, 17, 71, 321, 559, 1461, 3571, 1033, 15575, 7097, 14703, 52359}},
	{10225, 17, 38248, {1, 1, 1, 5, 21, 9, 123, 211, 233, 81, 111, 1433, 7825, 11771, 30743, 23993, 48717}},
	{10226, 17, 38259, {1, 3, 5, 15, 7, 3, 109, 33, 99, 135, 393, 3463, 7271, 14387, 30723, 19079, 83073}},
	{10227, 17, 38268, {1, 3, 3, 15, 3, 51, 77, 219, 409, 11, 67, 3787, 5155, 9259, 7185, 21611, 32577}},
	{10228, 17, 38271, {1, 3, 7, 1, 5, 49, 125, 85, 151, 301, 887, 1765, 5, 12849, 11775, 11319, 29547}},
	{10229, 17, 38272, {1, 1, 7, 11, 17, 15, 105, 29, 327, 637, 1493, 3361, 1823, 14709, 18355, 741, 57807}},
	{10230, 17, 38278, {1, 1, 7, 7, 3, 27, 15, 113, 227, 617, 1543, 1719, 8065, 13627, 23525, 20511, 64759}},
	{10231, 17, 38292, {1, 1, 3, 3, 21, 47, 89, 177, 381, 711, 1367, 2405, 887, 2351, 22957, 49679, 5963}},
	{10232, 17, 38296, {1, 3, 7, 9, 7, 1, 39, 71, 9, 275, 875, 1385, 6215, 10419, 25921, 63427, 18031}},
	{10233, 17, 38318, {1, 1, 7, 5, 23, 57, 27, 7, 445, 111, 953, 37, 2769, 1967, 8165, 35417, 36471}},
	{10234, 17, 38326, {1, 3, 3, 11, 23, 17, 119, 113, 275, 625, 1957, 2795, 3815, 7937, 11049, 31939, 128053}},
	{10235, 17, 38338, {1, 3, 3, 5, 7, 35, 45, 41, 251, 491, 1953, 3201, 751, 5705, 595, 27003, 77917}},
	{10236, 17, 38347, {1, 3, 5, 3, 25, 17, 55, 137, 299, 541, 289, 2225, 4667, 3569, 13687, 36193, 75705}},
	{10237, 17, 38373, {1, 1, 7, 15, 21, 9, 27, 157, 469, 441, 193, 2097, 4863, 2147, 31197, 24283, 102039}},
	{10238, 17, 38378, {1, 1, 1, 11, 17, 39, 73, 37, 91, 121, 1283, 367, 1875, 14365, 28349, 60993, 10791}},
	{10239, 17, 38380, {1, 3, 5, 9, 13, 63, 89, 53, 459, 347, 343, 2321, 5237, 2497, 279, 63833, 10709}},
	{10240, 17, 38388, {1, 1, 1, 15, 11, 23, 41, 79, 45, 567, 1217, 1669, 1679, 2561, 16191, 49041, 4081}},
	{10241, 17, 38392, {1, 3, 1, 5, 3, 9, 103, 245, 47, 561, 229, 2945, 6563, 797, 21571, 25769, 12995}},
	{10242, 17, 38397, {1, 3, 5, 7, 5, 47, 99, 55, 49, 951, 765, 2095, 8021, 4389, 20501, 26047, 119967}},
	{10243, 17, 38398, {1, 1, 5, 3, 15, 47, 81, 121, 379, 527, 419, 1093, 367, 10939, 17181, 13905, 49859}},
	{10244, 17, 38401, {1, 3, 3, 5, 7, 59, 53, 255, 131, 685, 1677, 3757, 3601, 89, 6225, 32705, 28287}},
	{10245, 17, 38404, {1, 3, 7, 1, 7, 55, 85, 47, 425, 793, 605, 2313, 791, 4247, 9693, 10633, 82915}},
	{10246, 17, 38422, {1, 3, 5, 13, 13, 49, 127, 213, 27, 657, 419, 55, 6289, 295, 4211, 8899, 120237}},
	{10247, 17, 38432, {1, 1, 7, 11, 11, 7, 75, 35, 315, 125, 517, 3677, 2323, 6897, 11535, 36789, 20179}},
	{10248, 17, 38469, {1, 1, 5, 9, 11, 3, 77, 43, 323, 647, 383, 485, 3937, 9081, 27745, 59149, 103433}},
	{10249, 17, 38482, {1, 3, 3, 13, 3, 47, 91, 81, 115, 625, 2003, 3601, 531, 113, 20719, 47391, 111039}},
	{10250, 17, 38493, {1, 1, 3, 13, 5, 49, 123, 189, 109, 325, 497, 923, 3861, 14029, 22651, 19857, 104801}},
	{10251, 17, 38507, {1, 1, 5, 3, 29, 23, 25, 23, 501, 371, 1983, 1303, 2261, 11865, 13281, 2587, 75741}},
	{10252, 17, 38518, {1, 3, 5, 13, 27, 61, 45, 11, 157, 615, 897, 1529, 2213, 757, 30105, 2011, 27267}},
	{10253, 17, 38521, {1, 3, 5, 13, 29, 31, 95, 159, 449, 307, 1575, 1897, 2301, 14023, 6921, 30543, 31843}},
	{10254, 17, 38527, {1, 3, 3, 5, 1, 1, 79, 147, 437, 623, 1161, 3407, 3073, 15425, 4329, 19651, 90597}},
	{10255, 17, 38533, {1, 1, 1, 11, 17, 7, 43, 171, 447, 841, 573, 3775, 5517, 3629, 18241, 31907, 51423}},
	{10256, 17, 38534, {1, 1, 3, 7, 15, 53, 111, 203, 171, 963, 1983, 2017, 6067, 12281, 3417, 7431, 33803}},
	{10257, 17, 38552, {1, 3, 1, 1, 31, 49, 125, 65, 7, 579, 1709, 1815, 2643, 11537, 30093, 11813, 65157}},
	{10258, 17, 38561, {1, 3, 7, 15, 1, 3, 61, 21, 163, 809, 1263, 2577, 7811, 12611, 6921, 18529, 25709}},
	{10259, 17, 38576, {1, 1, 3, 5, 17, 43, 13, 81, 29, 905, 1975, 589, 5875, 15683, 29447, 46453, 127911}},
	{10260, 17, 38581, {1, 1, 5, 3, 19, 29, 11, 67, 375, 771, 755, 3939, 1465, 3275, 1119, 24695, 105105}},
	{10261, 17, 38585, {1, 1, 3, 11, 23, 37, 33, 135, 329, 733, 245, 2353, 2547, 7823, 16265, 5975, 37877}},
	{10262, 17, 38594, {1, 3, 7, 13, 15, 9, 47, 181, 239, 723, 1219, 409, 1685, 5493, 14189, 18107, 26231}},
	{10263, 17, 38606, {1, 1, 5, 1, 9, 1, 65, 125, 439, 591, 1499, 3797, 5879, 4231, 18655, 9643, 42265}},
	{10264, 17, 38613, {1, 1, 7, 7, 11, 51, 111, 47, 169, 39, 45, 2211, 6729, 10987, 22367, 27257, 112711}},
	{10265, 17, 38617, {1, 3, 5, 3, 19, 61, 89, 185, 23, 793, 1457, 1743, 3743, 15063, 14053, 50201, 109175}},
	{10266, 17, 38634, {1, 1, 5, 13, 31, 51, 69, 135, 427, 527, 1673, 2393, 5829, 683, 1509, 56617, 105735}},
	{10267, 17, 38644, {1, 1, 1, 15, 31, 51, 3, 105, 125, 593, 1589, 3217, 7449, 525, 30599, 11689, 14781}},
	{10268, 17, 38651, {1, 1, 1, 11, 9, 37, 113, 45, 487, 961, 87, 1461, 3521, 8645, 19771, 43817, 43277}},
	{10269, 17, 38661, {1, 1, 7, 3, 7, 11, 45, 97, 11, 593, 319, 2597, 37, 4157, 6669, 29929, 17213}},
	{10270, 17, 38665, {1, 3, 7, 3, 29, 21, 101, 93, 289, 975, 1937, 3423, 757, 7075, 12575, 26801, 90989}},
	{10271, 17, 38668, {1, 1, 7, 15, 25, 49, 49, 149, 503, 365, 1359, 2155, 7977, 14955, 18439, 44269, 88995}},
	{10272, 17, 38674, {1, 3, 7, 13, 25, 27, 15, 67, 157, 873, 339, 2143, 1405, 12209, 30939, 36109, 107699}},
	{10273, 17, 38689, {1, 3, 5, 5, 21, 33, 121, 95, 61, 159, 1423, 2899, 3811, 263, 9139, 54481, 107375}},
	{10274, 17, 38695, {1, 1, 7, 7, 13, 49, 83, 53, 267, 131, 673, 3945, 5255, 2009, 21017, 41749, 44817}},
	{10275, 17, 38696, {1, 3, 1, 13, 25, 57, 125, 5, 323, 653, 221, 2013, 7225, 8719, 28049, 41953, 14725}},
	{10276, 17, 38707, {1, 3, 7, 7, 5, 13, 35, 161, 221, 951, 769, 717, 267, 2233, 23387, 47411, 95739}},
	{10277, 17, 38710, {1, 3, 7, 11, 23, 47, 37, 67, 501, 159, 763, 4045, 5125, 5667, 9895, 33041, 101171}},
	{10278, 17, 38716, {1, 1, 7, 1, 31, 31, 111, 183, 33, 895, 1819, 3593, 1285, 10145, 3679, 36615, 82863}},
	{10279, 17, 38733, {1, 3, 5, 7, 21, 59, 55, 163, 139, 855, 1903, 3229, 3745, 10289, 28831, 46895, 12647}},
	{10280, 17, 38736, {1, 3, 7, 9, 25, 31, 113, 177, 459, 201, 565, 2089, 725, 9273, 26249, 5987, 49573}},
	{10281, 17, 38742, {1, 1, 7, 15, 3, 37, 49, 145, 121, 803, 1197, 2797, 21, 833, 2277, 28189, 93171}},
	{10282, 17, 38751, {1, 1, 7, 7, 13, 31, 93, 153, 345, 363, 1825, 1481, 3347, 13277, 26119, 63153, 118231}},
	{10283, 17, 38752, {1, 3, 1, 11, 31, 9, 33, 95, 433, 595, 1131, 465, 1797, 15453, 32527, 40789, 37799}},
	{10284, 17, 38775, {1, 1, 5, 11, 31, 29, 83, 33, 243, 633, 1325, 3843, 7717, 851, 29789, 48827, 89209}},
	{10285, 17, 38779, {1, 1, 3, 7, 25, 31, 127, 219, 281, 51, 1843, 3363, 5985, 1697, 3083, 18967, 117421}},
	{10286, 17, 38792, {1, 3, 7, 9, 1, 19, 125, 199, 41, 117, 903, 1131, 7731, 14431, 24753, 62841, 50251}},
	{10287, 17, 38798, {1, 3, 5, 5, 11, 37, 19, 249, 97, 59, 1849, 1151, 2171, 1217, 31277, 26547, 86601}},
	{10288, 17, 38819, {1, 3, 3, 7, 29, 35, 21, 7, 93, 971, 1155, 1847, 89, 6759, 29611, 40793, 88327}},
	{10289, 17, 38822, {1, 3, 5, 5, 29, 23, 91, 71, 479, 943, 1839, 3699, 2491, 9603, 15061, 43221, 20435}},
	{10290, 17, 38839, {1, 3, 3, 7, 29, 11, 15, 83, 21, 585, 501, 161, 4797, 11243, 14879, 12519, 19069}},
	{10291, 17, 38851, {1, 1, 5, 15, 13, 37, 9, 215, 433, 925, 987, 3253, 8027, 7013, 20801, 12891, 36497}},
	{10292, 17, 38854, {1, 3, 7, 1, 15, 31, 95, 85, 355, 1013, 1963, 2943, 1925, 13691, 15237, 28943, 63873}},
	{10293, 17, 38863, {1, 3, 3, 1, 17, 21, 99, 201, 465, 819, 665, 901, 2671, 2457, 29603, 35275, 28339}},
	{10294, 17, 38865, {1, 1, 1, 9, 5, 23, 111, 107, 27, 433, 1341, 91, 6879, 1933, 25433, 37435, 99061}},
	{10295, 17, 38887, {1, 3, 5, 13, 11, 55, 27, 151, 397, 591, 89, 1221, 5581, 2701, 15033, 41879, 71415}},
	{10296, 17, 38896, {1, 3, 3, 5, 17, 35, 15, 119, 487, 903, 875, 3391, 7731, 12181, 27823, 32561, 59133}},
	{10297, 17, 38901, {1, 1, 5, 7, 15, 33, 67, 53, 307, 947, 857, 2713, 803, 765, 4175, 15513, 23985}},
	{10298, 17, 38912, {1, 3, 1, 15, 23, 11, 15, 101, 467, 429, 153, 3205, 5627, 7555, 22515, 12721, 7905}},
	{10299, 17, 38918, {1, 1, 3, 7, 19, 61, 55, 187, 413, 49, 1031, 3415, 3903, 6933, 20017, 50429, 116407}},
	{10300, 17, 38929, {1, 3, 1, 13, 13, 15, 33, 1, 403, 441, 1969, 775, 2209, 15061, 15657, 28819, 62705}},
	{10301, 17, 38941, {1, 1, 5, 13, 13, 47, 67, 97, 87, 677, 1639, 3281, 1395, 8499, 18449, 49935, 25775}},
	{10302, 17, 38948, {1, 3, 3, 13, 7, 13, 77, 45, 405, 881, 293, 2263, 6517, 15415, 25809, 5681, 31327}},
	{10303, 17, 38965, {1, 3, 7, 5, 5, 51, 63, 171, 401, 671, 1631, 1735, 7361, 8741, 31933, 44761, 12209}},
	{10304, 17, 38969, {1, 1, 7, 3, 13, 3, 39, 223, 105, 781, 241, 2895, 5165, 12135, 5683, 63009, 58399}},
	{10305, 17, 38977, {1, 1, 5, 11, 29, 11, 37, 1, 445, 157, 219, 2269, 3025, 5721, 24539, 41879, 128445}},
	{10306, 17, 38987, {1, 3, 7, 15, 23, 21, 125, 105, 141, 101, 1981, 3765, 5349, 13781, 10055, 7069, 77721}},
	{10307, 17, 38992, {1, 1, 7, 9, 3, 37, 111, 95, 33, 53, 1021, 1629, 6945, 4657, 19977, 36715, 101401}},
	{10308, 17, 38995, {1, 3, 3, 1, 9, 5, 65, 77, 459, 471, 1045, 2351, 2787, 13001, 16211, 22585, 116205}},
	{10309, 17, 39001, {1, 1, 7, 9, 25, 41, 21, 187, 471, 997, 583, 2243, 6537, 11837, 21089, 51051, 98517}},
	{10310, 17, 39004, {1, 1, 7, 5, 13, 15, 81, 39, 223, 935, 951, 5, 4315, 11789, 18365, 49647, 92461}},
	{10311, 17, 39011, {1, 3, 3, 5, 11, 15, 97, 245, 43, 819, 1415, 3287, 2051, 15879, 21727, 54467, 53875}},
	{10312, 17, 39018, {1, 3, 1, 11, 7, 47, 125, 155, 301, 469, 805, 3789, 6967, 12117, 30369, 55183, 12671}},
	{10313, 17, 39025, {1, 1, 3, 13, 17, 25, 45, 199, 69, 1015, 581, 3891, 7743, 9273, 7639, 59055, 93923}},
	{10314, 17, 39031, {1, 1, 5, 11, 9, 47, 39, 251, 489, 47, 83, 2147, 943, 1959, 21361, 5325, 106079}},
	{10315, 17, 39032, {1, 3, 5, 13, 27, 59, 35, 1, 155, 367, 165, 2665, 3021, 1127, 28585, 45347, 66763}},
	{10316, 17, 39044, {1, 3, 1, 5, 31, 31, 15, 125, 485, 581, 1987, 2293, 4573, 11431, 20773, 58661, 79869}},
	{10317, 17, 39056, {1, 3, 5, 15, 31, 11, 109, 229, 11, 831, 1545, 919, 6125, 9337, 4169, 58041, 67577}},
	{10318, 17, 39065, {1, 1, 1, 3, 1, 43, 13, 89, 89, 863, 1607, 1897, 4831, 5239, 24503, 51853, 126983}},
	{10319, 17, 39066, {1, 1, 5, 11, 11, 37, 11, 253, 495, 83, 941, 3665, 5187, 13679, 11811, 29563, 80571}},
	{10320, 17, 39084, {1, 3, 7, 15, 3, 45, 45, 157, 477, 321, 1401, 1133, 271, 12455, 31543, 18223, 116701}},
	{10321, 17, 39095, {1, 1, 5, 3, 7, 5, 17, 127, 195, 583, 715, 3975, 6865, 7617, 6749, 15687, 42375}},
	{10322, 17, 39099, {1, 1, 1, 7, 5, 7, 21, 163, 303, 45, 1435, 1345, 2489, 15333, 18459, 60837, 104339}},
	{10323, 17, 39101, {1, 3, 1, 1, 7, 23, 39, 93, 347, 947, 345, 1713, 6383, 15411, 10849, 32559, 126431}},
	{10324, 17, 39102, {1, 3, 1, 5, 19, 41, 119, 213, 3, 991, 1745, 3989, 155, 7761, 28179, 59805, 106759}},
	{10325, 17, 39104, {1, 1, 7, 5, 25, 11, 105, 89, 505, 711, 1213, 2831, 8087, 8855, 31171, 49749, 921}},
	{10326, 17, 39109, {1, 1, 5, 5, 23, 61, 49, 101, 209, 805, 123, 17, 805, 9033, 25753, 33261, 33753}},
	{10327, 17, 39114, {1, 1, 3, 5, 5, 17, 93, 223, 179, 307, 869, 1851, 4313, 477, 12925, 21365, 103999}},
	{10328, 17, 39121, {1, 1, 7, 13, 21, 23, 105, 53, 393, 939, 291, 2407, 4815, 4961, 30305, 8613, 62599}},
	{10329, 17, 39122, {1, 1, 1, 11, 9, 55, 55, 135, 411, 225, 205, 3357, 4553, 10139, 17649, 51209, 94037}},
	{10330, 17, 39127, {1, 3, 5, 15, 17, 17, 119, 15, 121, 581, 169, 2495, 3673, 7173, 13099, 7683, 53397}},
	{10331, 17, 39150, {1, 3, 1, 11, 29, 29, 119, 255, 447, 85, 845, 1015, 2793, 2471, 12639, 32155, 99193}},
	{10332, 17, 39172, {1, 3, 3, 3, 9, 33, 77, 251, 425, 1007, 1003, 2697, 4989, 7799, 26581, 15963, 50443}},
	{10333, 17, 39181, {1, 3, 1, 5, 13, 47, 13, 203, 473, 529, 147, 2061, 343, 4029, 14615, 51355, 27863}},
	{10334, 17, 39184, {1, 1, 3, 15, 19, 63, 39, 25, 241, 487, 461, 3021, 3545, 4537, 8991, 17689, 77131}},
	{10335, 17, 39193, {1, 3, 5, 5, 7, 1, 61, 89, 495, 943, 1061, 405, 449, 12785, 25151, 24497, 65709}},
	{10336, 17, 39199, {1, 1, 5, 3, 7, 43, 51, 55, 193, 615, 37, 1377, 2541, 3861, 29447, 32269, 106335}},
	{10337, 17, 39200, {1, 1, 5, 11, 21, 55, 103, 43, 421, 673, 175, 979, 5175, 1301, 8617, 55875, 111095}},
	{10338, 17, 39206, {1, 3, 5, 13, 29, 31, 33, 241, 129, 473, 201, 2015, 447, 99, 23781, 33517, 107851}},
	{10339, 17, 39209, {1, 3, 5, 3, 13, 27, 125, 205, 287, 957, 1609, 2907, 5481, 14239, 19719, 22459, 75365}},
	{10340, 17, 39235, {1, 3, 3, 5, 27, 23, 53, 39, 329, 381, 745, 517, 7853, 5333, 2773, 29119, 7049}},
	{10341, 17, 39238, {1, 3, 5, 15, 29, 11, 89, 3, 503, 755, 485, 2669, 6737, 16241, 7345, 50991, 107291}},
	{10342, 17, 39242, {1, 1, 7, 3, 17, 45, 11, 3, 157, 715, 577, 1309, 3323, 9401, 10573, 55135, 100067}},
	{10343, 17, 39244, {1, 1, 5, 9, 19, 21, 49, 103, 349, 503, 1447, 675, 4273, 7673, 27507, 57697, 80875}},
	{10344, 17, 39280, {1, 1, 5, 1, 9, 53, 19, 99, 225, 915, 431, 781, 3291, 4383, 26505, 50339, 99799}},
	{10345, 17, 39290, {1, 3, 1, 5, 7, 37, 87, 201, 481, 991, 1553, 1867, 7893, 13147, 18647, 10373, 51951}},
	{10346, 17, 39292, {1, 1, 3, 13, 17, 37, 19, 199, 253, 901, 759, 3545, 3565, 10461, 11867, 57605, 75555}},
	{10347, 17, 39296, {1, 1, 1, 5, 15, 23, 115, 69, 363, 673, 201, 2451, 3197, 10059, 1667, 47145, 89}},
	{10348, 17, 39305, {1, 1, 7, 13, 19, 31, 63, 35, 93, 939, 1057, 3221, 951, 3575, 9659, 7005, 2087}},
	{10349, 17, 39308, {1, 1, 7, 15, 15, 21, 79, 7, 23, 595, 1123, 1909, 6863, 7383, 28067, 30113, 107497}},
	{10350, 17, 39311, {1, 3, 5, 7, 11, 47, 41, 177, 163, 855, 1853, 3837, 6995, 9727, 27285, 62667, 77531}},
	{10351, 17, 39326, {1, 3, 7, 3, 3, 29, 99, 163, 95, 893, 1049, 2001, 2961, 601, 4613, 59745, 61203}},
	{10352, 17, 39329, {1, 3, 5, 1, 27, 5, 5, 47, 119, 631, 1171, 3467, 2115, 8581, 24863, 64193, 52093}},
	{10353, 17, 39330, {1, 1, 1, 5, 9, 51, 49, 251, 97, 177, 311, 993, 1103, 7875, 25273, 51587, 9081}},
	{10354, 17, 39336, {1, 3, 7, 5, 31, 21, 43, 137, 143, 779, 691, 2331, 5073, 5409, 13335, 999, 127765}},
	{10355, 17, 39339, {1, 1, 7, 1, 31, 33, 27, 193, 175, 755, 1559, 659, 5663, 10491, 29209, 50979, 116683}},
	{10356, 17, 39353, {1, 3, 1, 7, 23, 49, 1, 39, 117, 45, 1767, 3503, 4901, 5699, 23613, 44195, 17867}},
	{10357, 17, 39359, {1, 3, 7, 13, 3, 5, 105, 89, 343, 627, 1117, 3419, 2081, 5747, 7919, 44329, 125133}},
	{10358, 17, 39374, {1, 3, 1, 9, 13, 33, 53, 203, 17, 927, 127, 2195, 415, 11301, 15115, 54467, 128777}},
	{10359, 17, 39391, {1, 3, 7, 1, 9, 41, 15, 89, 403, 333, 57, 1159, 1325, 2335, 10609, 20485, 110317}},
	{10360, 17, 39392, {1, 3, 3, 5, 3, 61, 25, 155, 477, 907, 359, 359, 5119, 8157, 29945, 38955, 106485}},
	{10361, 17, 39402, {1, 1, 7, 5, 19, 47, 91, 89, 367, 703, 761, 431, 6813, 2983, 29739, 52453, 125935}},
	{10362, 17, 39409, {1, 1, 1, 7, 7, 61, 127, 239, 277, 649, 1111, 2319, 1737, 5071, 13469, 52119, 48899}},
	{10363, 17, 39410, {1, 3, 5, 15, 7, 17, 21, 209, 265, 895, 719, 263, 6871, 5835, 28483, 49859, 67619}},
	{10364, 17, 39415, {1, 3, 3, 15, 3, 7, 113, 109, 333, 545, 597, 1193, 7593, 3961, 25169, 64673, 47839}},
	{10365, 17, 39419, {1, 1, 1, 3, 15, 45, 55, 41, 317, 719, 1587, 2953, 2441, 1127, 9183, 21637, 69075}},
	{10366, 17, 39431, {1, 3, 1, 9, 25, 57, 59, 29, 89, 833, 379, 1085, 763, 14747, 30797, 24089, 83367}},
	{10367, 17, 39432, {1, 1, 5, 13, 29, 3, 117, 239, 453, 595, 243, 3103, 6047, 631, 22739, 41669, 11683}},
	{10368, 17, 39438, {1, 1, 3, 1, 9, 53, 81, 21, 67, 735, 827, 3519, 7991, 16249, 4183, 61295, 4531}},
	{10369, 17, 39443, {1, 3, 5, 3, 1, 57, 47, 99, 91, 71, 1421, 2949, 5951, 15439, 25239, 26453, 50199}},
	{10370, 17, 39445, {1, 1, 3, 15, 21, 3, 93, 21, 41, 809, 855, 727, 7797, 6957, 15835, 27175, 60617}},
	{10371, 17, 39449, {1, 1, 1, 13, 1, 3, 83, 135, 197, 171, 1459, 2841, 5021, 6961, 30675, 38295, 39555}},
	{10372, 17, 39461, {1, 1, 7, 5, 7, 49, 83, 83, 117, 73, 639, 2717, 651, 3253, 31635, 14427, 116509}},
	{10373, 17, 39462, {1, 3, 1, 3, 23, 63, 15, 23, 433, 539, 903, 2655, 1787, 12901, 12013, 41315, 128217}},
	{10374, 17, 39485, {1, 3, 1, 7, 5, 19, 3, 137, 493, 681, 775, 3725, 4855, 10817, 25277, 3631, 60779}},
	{10375, 17, 39508, {1, 3, 3, 5, 1, 7, 67, 39, 309, 77, 1679, 2853, 3803, 2065, 7461, 1555, 88219}},
	{10376, 17, 39515, {1, 1, 3, 5, 3, 47, 17, 193, 429, 789, 1525, 969, 7905, 6523, 10149, 64689, 40037}},
	{10377, 17, 39522, {1, 3, 5, 7, 21, 17, 65, 61, 255, 517, 1765, 2603, 4929, 11073, 7871, 29313, 84739}},
	{10378, 17, 39533, {1, 1, 5, 7, 13, 55, 111, 63, 499, 9, 1715, 957, 6951, 15839, 13531, 45483, 17923}},
	{10379, 17, 39541, {1, 1, 1, 1, 27, 7, 13, 135, 27, 259, 1735, 3847, 7931, 14777, 15249, 62367, 45773}},
	{10380, 17, 39542, {1, 1, 5, 3, 5, 7, 99, 171, 491, 1007, 195, 2223, 2657, 13557, 6549, 47125, 25117}},
	{10381, 17, 39557, {1, 1, 1, 9, 13, 21, 59, 205, 205, 951, 1707, 3387, 2901, 5463, 13403, 1917, 90591}},
	{10382, 17, 39570, {1, 3, 3, 5, 21, 37, 71, 91, 297, 885, 1415, 355, 2877, 9261, 6485, 45855, 90081}},
	{10383, 17, 39575, {1, 1, 5, 9, 23, 51, 107, 75, 93, 1015, 439, 3589, 3307, 337, 17247, 42285, 85417}},
	{10384, 17, 39576, {1, 1, 7, 15, 29, 33, 51, 23, 269, 35, 1241, 1137, 729, 14531, 14603, 47547, 17151}},
	{10385, 17, 39591, {1, 3, 3, 1, 25, 21, 51, 229, 55, 561, 653, 3289, 7629, 15445, 21115, 35941, 113669}},
	{10386, 17, 39597, {1, 1, 5, 15, 1, 33, 119, 171, 75, 621, 2025, 3235, 1895, 8279, 13205, 61085, 105401}},
	{10387, 17, 39603, {1, 3, 1, 7, 25, 33, 73, 25, 1, 531, 603, 77, 4939, 5957, 28065, 59467, 66659}},
	{10388, 17, 39610, {1, 1, 7, 3, 17, 61, 103, 47, 289, 39, 917, 2515, 6607, 1129, 23361, 46321, 81929}},
	{10389, 17, 39612, {1, 1, 7, 5, 29, 53, 5, 191, 151, 19, 895, 1215, 5401, 9861, 24751, 15481, 34179}},
	{10390, 17, 39617, {1, 3, 7, 9, 5, 3, 29, 205, 173, 547, 727, 947, 5619, 4181, 30621, 5553, 37587}},
	{10391, 17, 39620, {1, 1, 3, 9, 13, 59, 95, 145, 287, 849, 1483, 3375, 3531, 6585, 29565, 4243, 88333}},
	{10392, 17, 39624, {1, 1, 7, 11, 21, 23, 59, 223, 71, 743, 443, 697, 7789, 10371, 28565, 45127, 37967}},
	{10393, 17, 39627, {1, 3, 5, 11, 15, 25, 79, 71, 21, 817, 751, 189, 1769, 3835, 21465, 17991, 102043}},
	{10394, 17, 39635, {1, 1, 7, 11, 3, 19, 25, 5, 261, 181, 49, 261, 7715, 2195, 19771, 43463, 36533}},
	{10395, 17, 39641, {1, 1, 7, 13, 21, 21, 15, 235, 191, 197, 1305, 1351, 4511, 625, 6613, 37053, 59491}},
	{10396, 17, 39666, {1, 1, 1, 13, 15, 13, 93, 239, 251, 1009, 527, 1347, 4173, 14753, 27389, 20397, 13101}},
	{10397, 17, 39668, {1, 1, 3, 1, 15, 11, 127, 141, 277, 775, 1419, 2353, 6929, 2265, 7253, 19807, 74853}},
	{10398, 17, 39686, {1, 3, 3, 15, 15, 49, 9, 183, 407, 377, 675, 871, 347, 3417, 4409, 7805, 40507}},
	{10399, 17, 39692, {1, 3, 5, 3, 23, 11, 81, 53, 343, 681, 1777, 3411, 757, 10875, 3581, 56105, 79907}},
	{10400, 17, 39700, {1, 3, 5, 1, 25, 9, 109, 55, 283, 311, 1607, 2479, 5675, 8819, 10853, 38719, 44471}},
	{10401, 17, 39709, {1, 1, 7, 7, 9, 53, 33, 195, 503, 167, 993, 3203, 3885, 1921, 1039, 25785, 47411}},
	{10402, 17, 39726, {1, 3, 3, 3, 31, 3, 11, 85, 475, 743, 1825, 2649, 2373, 12177, 21481, 35579, 85803}},
	{10403, 17, 39728, {1, 3, 7, 3, 23, 45, 45, 207, 369, 773, 1579, 783, 2491, 7441, 21203, 57091, 107413}},
	{10404, 17, 39737, {1, 1, 1, 5, 19, 7, 97, 213, 431, 533, 637, 1767, 4945, 4693, 977, 64781, 111811}},
	{10405, 17, 39738, {1, 3, 7, 7, 1, 55, 101, 251, 153, 95, 1043, 3219, 3499, 6297, 11571, 9131, 61899}},
	{10406, 17, 39755, {1, 3, 5, 5, 25, 53, 121, 255, 69, 661, 799, 3559, 2029, 11701, 14151, 37897, 18333}},
	{10407, 17, 39757, {1, 1, 1, 9, 21, 19, 97, 21, 321, 957, 1115, 251, 5131, 8465, 24215, 34423, 12747}},
	{10408, 17, 39758, {1, 3, 5, 7, 17, 19, 99, 135, 429, 625, 1401, 1025, 4193, 2911, 7349, 34135, 9341}},
	{10409, 17, 39765, {1, 3, 5, 1, 5, 63, 97, 121, 307, 547, 1967, 641, 487, 4627, 30899, 62049, 94343}},
	{10410, 17, 39766, {1, 3, 5, 13, 1, 1, 109, 23, 267, 843, 271, 2277, 855, 4245, 2177, 33633, 113835}},
	{10411, 17, 39769, {1, 1, 3, 7, 3, 27, 91, 79, 27, 855, 2025, 443, 4797, 9005, 27533, 20497, 100431}},
	{10412, 17, 39779, {1, 3, 3, 5, 23, 7, 73, 35, 395, 649, 881, 2923, 4065, 853, 10829, 19461, 82383}},
	{10413, 17, 39785, {1, 3, 1, 5, 25, 13, 85, 93, 369, 259, 393, 3233, 799, 12409, 26631, 64291, 110133}},
	{10414, 17, 39794, {1, 3, 5, 5, 31, 35, 25, 239, 455, 893, 573, 1449, 3359, 12077, 17149, 12921, 66931}},
	{10415, 17, 39796, {1, 1, 7, 3, 25, 39, 67, 87, 215, 325, 627, 3609, 4417, 10021, 12047, 64593, 116525}},
	{10416, 17, 39809, {1, 1, 5, 5, 23, 51, 125, 247, 83, 419, 655, 635, 7053, 4907, 12887, 18083, 49481}},
	{10417, 17, 39815, {1, 1, 7, 11, 5, 25, 65, 139, 235, 331, 1885, 1851, 1061, 13265, 14371, 23067, 56757}},
	{10418, 17, 39829, {1, 1, 7, 9, 11, 15, 29, 255, 509, 869, 561, 2201, 487, 2989, 14943, 65373, 35789}},
	{10419, 17, 39834, {1, 1, 1, 3, 3, 33, 23, 121, 129, 351, 1481, 65, 321, 15927, 23849, 2813, 98547}},
	{10420, 17, 39839, {1, 1, 1, 3, 23, 55, 121, 35, 339, 87, 1147, 401, 1477, 10617, 15943, 20535, 89321}},
	{10421, 17, 39850, {1, 3, 5, 15, 25, 59, 111, 185, 305, 47, 523, 2801, 5485, 625, 30191, 58153, 9019}},
	{10422, 17, 39857, {1, 1, 7, 13, 15, 51, 105, 55, 77, 419, 1011, 1117, 2705, 15093, 15629, 51429, 58487}},
	{10423, 17, 39881, {1, 3, 7, 5, 15, 27, 19, 7, 401, 295, 1841, 1167, 2133, 1967, 6941, 13571, 29467}},
	{10424, 17, 39905, {1, 1, 5, 15, 25, 43, 23, 253, 173, 927, 1299, 2779, 5489, 16135, 1503, 51097, 105751}},
	{10425, 17, 39912, {1, 3, 3, 5, 9, 13, 5, 13, 411, 639, 1323, 1495, 2539, 15087, 21489, 49653, 76229}},
	{10426, 17, 39938, {1, 1, 1, 11, 7, 51, 47, 99, 247, 541, 1355, 2373, 4121, 13621, 7715, 16763, 127985}},
	{10427, 17, 39940, {1, 1, 3, 9, 7, 1, 85, 45, 269, 769, 581, 2229, 7143, 5203, 22483, 18511, 30997}},
	{10428, 17, 39944, {1, 3, 5, 7, 21, 41, 97, 225, 109, 195, 1197, 3417, 7613, 13225, 29157, 18969, 82045}},
	{10429, 17, 39955, {1, 3, 3, 3, 17, 41, 13, 77, 129, 679, 1659, 1299, 4809, 8537, 19081, 1281, 70793}},
	{10430, 17, 39961, {1, 1, 5, 5, 5, 49, 5, 15, 313, 941, 775, 259, 6579, 7745, 20531, 51669, 35257}},
	{10431, 17, 39977, {1, 1, 5, 5, 17, 35, 13, 235, 169, 699, 1365, 3907, 4231, 10965, 7737, 6735, 4253}},
	{10432, 17, 39980, {1, 1, 5, 3, 29, 3, 1, 197, 133, 935, 571, 3977, 2467, 2029, 12803, 64559, 6427}},
	{10433, 17, 39986, {1, 3, 5, 5, 27, 5, 69, 57, 439, 925, 1695, 827, 4685, 10971, 3011, 56821, 92187}},
	{10434, 17, 39988, {1, 1, 1, 3, 9, 45, 77, 179, 173, 1023, 907, 1999, 3913, 6973, 26987, 30237, 62987}},
	{10435, 17, 39991, {1, 3, 7, 3, 5, 21, 17, 97, 433, 277, 1515, 2923, 8025, 14119, 11243, 3983, 33943}},
	{10436, 17, 39998, {1, 1, 5, 7, 15, 13, 119, 169, 21, 927, 439, 361, 2655, 2237, 19775, 4157, 84245}},
	{10437, 17, 40003, {1, 3, 5, 5, 31, 41, 117, 159, 421, 505, 1617, 3855, 7835, 8105, 29525, 56735, 82335}},
	{10438, 17, 40005, {1, 1, 5, 5, 1, 33, 51, 3, 79, 933, 389, 493, 5969, 12493, 26723, 61159, 116951}},
	{10439, 17, 40023, {1, 3, 7, 13, 17, 23, 75, 13, 355, 111, 675, 3191, 3931, 5651, 17495, 4595, 49869}},
	{10440, 17, 40024, {1, 1, 7, 7, 15, 21, 35, 125, 89, 903, 697, 3493, 4043, 6631, 4793, 45655, 86969}},
	{10441, 17, 40045, {1, 3, 1, 15, 13, 43, 113, 213, 451, 473, 191, 2913, 6391, 1321, 29615, 24791, 26979}},
	{10442, 17, 40046, {1, 3, 3, 13, 17, 25, 9, 163, 163, 161, 1647, 3949, 1343, 12881, 10931, 31365, 70013}},
	{10443, 17, 40058, {1, 3, 7, 3, 3, 19, 1, 121, 387, 543, 1655, 1797, 6727, 2951, 21925, 21595, 73207}},
	{10444, 17, 40088, {1, 1, 5, 9, 7, 19, 91, 9, 83, 893, 1393, 163, 2219, 7763, 32395, 29569, 98645}},
	{10445, 17, 40091, {1, 1, 5, 7, 13, 63, 91, 115, 247, 387, 87, 3239, 7561, 297, 32615, 48817, 41761}},
	{10446, 17, 40098, {1, 3, 5, 3, 21, 23, 27, 141, 257, 377, 1745, 443, 897, 9033, 1715, 9225, 110181}},
	{10447, 17, 40109, {1, 1, 7, 9, 23, 49, 125, 131, 225, 253, 139, 2057, 3273, 4049, 6861, 4463, 11659}},
	{10448, 17, 40112, {1, 1, 5, 11, 5, 41, 97, 213, 133, 481, 2009, 2039, 1533, 10765, 22427, 23297, 80661}},
	{10449, 17, 40124, {1, 1, 5, 15, 9, 11, 77, 129, 421, 219, 1623, 703, 1611, 13377, 9859, 42869, 101943}},
	{10450, 17, 40130, {1, 3, 3, 3, 17, 63, 55, 29, 317, 973, 1159, 11, 1733, 14551, 25911, 39151, 45861}},
	{10451, 17, 40153, {1, 3, 7, 11, 29, 63, 107, 193, 263, 799, 1171, 543, 553, 12591, 21965, 8165, 64347}},
	{10452, 17, 40166, {1, 1, 7, 15, 23, 49, 65, 65, 401, 897, 681, 1113, 6737, 9157, 1557, 55891, 129175}},
	{10453, 17, 40175, {1, 3, 3, 1, 15, 23, 107, 123, 313, 633, 1009, 2615, 1155, 11701, 21945, 7939, 28111}},
	{10454, 17, 40183, {1, 3, 1, 11, 15, 11, 47, 137, 299, 393, 877, 1989, 5903, 6505, 9599, 4129, 23073}},
	{10455, 17, 40184, {1, 1, 7, 15, 9, 49, 67, 15, 79, 125, 505, 17, 8071, 12957, 13855, 23611, 66465}},
	{10456, 17, 40207, {1, 1, 5, 13, 31, 49, 1, 161, 121, 145, 711, 1347, 5297, 11309, 9871, 43075, 95541}},
	{10457, 17, 40215, {1, 3, 3, 13, 19, 7, 55, 199, 469, 471, 1269, 3779, 6251, 3513, 1775, 19501, 94055}},
	{10458, 17, 40225, {1, 3, 3, 13, 9, 41, 109, 211, 197, 227, 1211, 3327, 1247, 12253, 4493, 31507, 38677}},
	{10459, 17, 40235, {1, 1, 7, 3, 11, 45, 11, 103, 325, 849, 1817, 3971, 1059, 9047, 27237, 32211, 121165}},
	{10460, 17, 40240, {1, 3, 3, 3, 13, 43, 7, 35, 293, 3, 679, 1441, 5189, 7585, 32009, 6151, 89803}},
	{10461, 17, 40255, {1, 1, 7, 9, 29, 41, 127, 255, 363, 913, 2027, 3891, 5187, 10233, 8871, 48085, 125609}},
	{10462, 17, 40263, {1, 3, 1, 5, 21, 23, 59, 145, 171, 775, 535, 3803, 6981, 15901, 20255, 63199, 92905}},
	{10463, 17, 40270, {1, 3, 5, 9, 7, 63, 53, 7, 145, 547, 1753, 3351, 1273, 8175, 24103, 42133, 87459}},
	{10464, 17, 40277, {1, 3, 7, 7, 25, 33, 5, 217, 469, 473, 1573, 2525, 7345, 5261, 7023, 50893, 124129}},
	{10465, 17, 40282, {1, 3, 5, 13, 5, 51, 23, 61, 429, 775, 519, 2671, 1979, 9005, 21617, 33611, 120487}},
	{10466, 17, 40297, {1, 3, 3, 15, 23, 1, 73, 187, 47, 369, 943, 99, 2529, 5569, 13649, 51481, 128949}},
	{10467, 17, 40306, {1, 3, 1, 5, 25, 55, 35, 191, 327, 845, 1353, 261, 6297, 6067, 22241, 32381, 17749}},
	{10468, 17, 40315, {1, 1, 5, 15, 31, 5, 29, 129, 15, 47, 739, 755, 7595, 14743, 14705, 34347, 11805}},
	{10469, 17, 40333, {1, 3, 1, 3, 15, 49, 119, 47, 185, 63, 2003, 2847, 5393, 855, 7699, 29521, 67011}},
	{10470, 17, 40334, {1, 3, 7, 15, 11, 41, 37, 149, 173, 1015, 29, 1805, 1269, 16199, 32337, 11023, 60065}},
	{10471, 17, 40336, {1, 1, 1, 7, 31, 19, 65, 81, 255, 875, 1379, 2347, 1873, 14427, 29523, 38413, 65583}},
	{10472, 17, 40342, {1, 1, 1, 15, 13, 59, 3, 219, 127, 479, 1029, 3385, 563, 11825, 10081, 17423, 26431}},
	{10473, 17, 40345, {1, 1, 1, 1, 25, 27, 79, 87, 489, 281, 457, 3527, 5117, 4705, 21167, 46211, 90383}},
	{10474, 17, 40348, {1, 3, 7, 13, 7, 5, 67, 111, 53, 439, 1483, 3639, 7781, 9471, 10957, 60711, 64957}},
	{10475, 17, 40355, {1, 3, 7, 9, 7, 7, 41, 137, 159, 245, 551, 4007, 1277, 4743, 4863, 48689, 123289}},
	{10476, 17, 40372, {1, 3, 7, 9, 15, 49, 55, 77, 41, 475, 1563, 3569, 5993, 301, 14831, 44095, 22641}},
	{10477, 17, 40381, {1, 1, 1, 1, 15, 33, 39, 135, 81, 533, 869, 305, 1125, 6399, 14321, 37217, 121081}},
	{10478, 17, 40390, {1, 1, 7, 15, 21, 59, 43, 7, 225, 1, 115, 1531, 2931, 2593, 15935, 61973, 106899}},
	{10479, 17, 40407, {1, 1, 1, 1, 13, 13, 99, 191, 437, 367, 641, 1933, 5807, 11677, 13557, 46475, 34875}},
	{10480, 17, 40435, {1, 3, 7, 9, 21, 7, 119, 209, 31, 919, 901, 1229, 5823, 11439, 18151, 18991, 114743}},
	{10481, 17, 40437, {1, 3, 3, 3, 19, 37, 109, 53, 411, 617, 1841, 2769, 1271, 5719, 22359, 1199, 72405}},
	{10482, 17, 40441, {1, 1, 1, 5, 29, 3, 51, 59, 141, 897, 1907, 3799, 1463, 5661, 181, 50565, 95085}},
	{10483, 17, 40444, {1, 1, 1, 7, 1, 35, 77, 225, 341, 587, 137, 35, 2177, 15177, 12869, 35013, 39471}},
	{10484, 17, 40458, {1, 1, 3, 13, 15, 63, 45, 33, 337, 1, 1133, 263, 4985, 11591, 1085, 31197, 67897}},
	{10485, 17, 40460, {1, 1, 5, 13, 23, 11, 123, 21, 185, 639, 145, 3865, 2999, 6261, 23247, 23055, 32755}},
	{10486, 17, 40481, {1, 1, 5, 9, 19, 21, 47, 133, 281, 431, 1661, 3719, 3637, 973, 9727, 52627, 60035}},
	{10487, 17, 40484, {1, 1, 3, 5, 3, 19, 19, 89, 63, 549, 551, 3357, 5665, 4781, 22437, 1149, 10825}},
	{10488, 17, 40487, {1, 3, 5, 15, 3, 25, 81, 193, 11, 711, 1481, 1767, 1159, 4967, 16915, 3387, 26245}},
	{10489, 17, 40493, {1, 1, 1, 3, 29, 39, 23, 131, 473, 107, 765, 2249, 6087, 9145, 20751, 21085, 42989}},
	{10490, 17, 40494, {1, 3, 1, 9, 7, 39, 13, 199, 475, 333, 269, 1041, 5927, 14039, 19081, 9045, 119645}},
	{10491, 17, 40501, {1, 1, 5, 13, 11, 61, 99, 71, 151, 175, 1327, 3397, 5063, 10683, 7895, 62255, 85749}},
	{10492, 17, 40502, {1, 3, 7, 9, 1, 57, 21, 217, 423, 467, 1435, 2887, 1567, 8819, 19961, 36507, 110309}},
	{10493, 17, 40525, {1, 3, 3, 11, 11, 35, 77, 127, 153, 357, 865, 1943, 1947, 10995, 13617, 44347, 26851}},
	{10494, 17, 40550, {1, 3, 1, 11, 9, 43, 31, 81, 123, 813, 995, 169, 6593, 13621, 32195