/****************************************************************************
** Dooble - The Secure Internet Web Browser
**
** Copyright (c) 2008, 2009 Alexis Megas, 
** Gunther van Dooble, and the Dooble Team. 
** All rights reserved.
**
** License: GPL2 only: 
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; version 2 of the License only.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
** or see here: http://www.gnu.org/licenses/gpl.html
**
** For the WebKit library, please see: http://webkit.org.
**
** THE CODE IS PROVIDED BY THE AUTHORS ''AS IS'' AND ANY
** EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
** PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
** ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
** GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
** INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
** IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
** ARISING IN ANY WAY OUT OF THE USE OF THIS APPLICATION, EVEN IF ADVISED
** OF THE POSSIBILITY OF SUCH DAMAGE.
**
** Please report all praise, requests, bugs, and problems to the project
** team and administrators: http://sf.net/projects/dooble.
**
** You can find us listed at our project page. New team members are welcome.
** The name of the authors should not be used to endorse or promote products
** derived from Dooble without specific prior written permission.
** If you use this code for other projects, please let us know.
**
** Web sites:
**   http://dooble.net/projects/dooble
**   http://dooble.sf.net
**   http://www.getadooble.com
****************************************************************************/

#include <QDir>
#include <QHttp>
#include <QDebug>
#include <QPrinter>
#include <QProcess>
#include <QLineEdit>
#include <QSettings>
#include <QSizeGrip>
#include <QSplitter>
#include <QWebFrame>
#include <QClipboard>
#include <QFileDialog>
#include <QMessageBox>
#include <QTranslator>
#include <QWebHistory>
#include <QApplication>
#include <QPrintDialog>
#include <QNetworkReply>
#include <QDesktopServices>
#include <QPrintPreviewDialog>

#include "dooble.h"
#include "dsettings.h"
#include "ddesktopwidget.h"
#include "pagesourcewindow.h"

static int doobleCounter = 0;
static dsettings *settingsWindow = 0;
static ddesktopwidget *desktopWidget = 0;
static downloadwindow *downloadWindow = 0;

int main(int argc, char *argv[])
{
  QApplication qapp(argc, argv);

  /*
  ** Configure translations.
  */

  QTranslator qtTranslator;

  qtTranslator.load("qt_" + QLocale::system().name(), "translations");
  qapp.installTranslator(&qtTranslator);

  QTranslator myappTranslator;

  myappTranslator.load("dooble_" + QLocale::system().name(),
		       "translations");
  qapp.installTranslator(&myappTranslator);

#if defined(Q_WS_WIN)
  QProcess::startDetached("startYACY.bat");
  QProcess::startDetached("RetroLocal.exe");
  QProcess::startDetached("offsystem.exe");
#else
  QProcess::startDetached("./startYACY.sh");
#endif

  QDir homeDir(QDir::home());

  homeDir.mkdir(".dooble");
  QWebSettings::globalSettings()->setIconDatabasePath
    (QDir::homePath() + "/.dooble");

  /*
  ** Set application attributes.
  */

  QCoreApplication::setApplicationName("Dooble");
  QCoreApplication::setApplicationVersion("0.07");
  QCoreApplication::setOrganizationName("Dooble");
  QCoreApplication::setOrganizationDomain("dooble.sf.net");

  /*
  ** Remove old configuration settings.
  */

  QSettings settings;

  settings.remove("mainWindow/pos");
  settings.remove("mainWindow/size");
  settings.remove("downloadWindow/pos");
  settings.remove("downloadWindow/size");

  /*
  ** We need to set this before creating some of the support windows.
  */

  if(!settings.contains("iconSet"))
    settings.setValue("iconSet",
		      QString("%1/%2").arg(QDir::currentPath()).
		      arg("Icons/nuvola/configuration.cfg"));

  if(!settings.contains("settingsWindow/iconSet1"))
    settings.setValue("settingsWindow/iconSet1", settings.value("iconSet"));

  QWebSettings::globalSettings()->setAttribute
    (QWebSettings::JavascriptEnabled,
     settings.value("settingsWindow/javascriptEnabled", false).toBool());
  QWebSettings::globalSettings()->setAttribute
    (QWebSettings::JavascriptCanOpenWindows,
     settings.value("settingsWindow/javascriptAllowNewWindows",
		    false).toBool());
  QWebSettings::globalSettings()->setAttribute
    (QWebSettings::AutoLoadImages, true);
  desktopWidget = new ddesktopwidget();
  downloadWindow = new downloadwindow();
  settingsWindow = new dsettings();
  QObject::connect(settingsWindow,
		   SIGNAL(backgroundImageChangedSignal(void)),
		   desktopWidget,
		   SLOT(slotBackgroundImageChanged(void)));

  dooble *dbl = 0;
  QString urlText = QString("%1/Tab/Default/search.html").arg
    (QDir::currentPath());

  if(argc > 1)
    dbl = new dooble(QList<QAction *> (), QUrl(argv[1]));
  else
    dbl = new dooble(QList<QAction *> (), QUrl::fromLocalFile(urlText));

  qapp.connect(&qapp, SIGNAL(lastWindowClosed()), dbl, SLOT(slotQuit(void)));
  return qapp.exec();
}

void dooble::init(QList<QAction *> actionsList)
{
  doobleCounter += 1;
  ui.setupUi(this);
  sbWidget = new QWidget();
  sb.setupUi(sbWidget);
  sb.progressBar->setVisible(false);
  sb.statusLabel->setVisible(true);
  ui.findFrame->setVisible(false);
  ui.backToolButton->setMenu(new QMenu());
  ui.forwardToolButton->setMenu(new QMenu());
  ui.tabWidget->setTabData(0, QVariant("Desktop"));
  connect(this,
	  SIGNAL(iconsChangedSignal(void)),
	  downloadWindow,
	  SLOT(slotSetIcons(void)));
  connect(settingsWindow,
	  SIGNAL(iconsChangedSignal(void)),
	  this,
	  SLOT(slotSetIcons(void)));
  connect(this,
	  SIGNAL(iconsChangedSignal(void)),
	  ui.searchLineEdit,
	  SLOT(slotSetIcons(void)));
  connect(this,
	  SIGNAL(iconsChangedSignal(void)),
	  ui.locationLineEdit,
	  SLOT(slotSetIcons(void)));
  connect(ui.locationLineEdit,
	  SIGNAL(loadPageSignal(const QUrl &)),
	  this,
	  SLOT(slotLoadPage(const QUrl &)));
  connect(downloadWindow, SIGNAL(saveUrlSignal(const QUrl &)), this,
	  SLOT(slotSaveUrl(const QUrl &)));
  connect(ui.tabWidget, SIGNAL(currentChanged(int)), this,
	  SLOT(slotTabSelected(int)));
  connect(ui.actionNew_Tab, SIGNAL(triggered(void)), this,
	  SLOT(slotNewTab(void)));
  connect(ui.tabWidget, SIGNAL(createTabSignal(void)), this,
	  SLOT(slotNewTab(void)));
  connect(ui.tabWidget, SIGNAL(openInNewWindowSignal(const int)), this,
	  SLOT(slotOpenPageInNewWindow(const int)));
  connect(ui.actionNew_Window, SIGNAL(triggered(void)), this,
	  SLOT(slotNewWindow(void)));
  connect(ui.actionOpen_URL, SIGNAL(triggered(void)), this,
	  SLOT(slotOpenUrl(void)));
  connect(ui.actionClose_Tab, SIGNAL(triggered(void)), this,
	  SLOT(slotCloseTab(void)));
  connect(ui.tabWidget, SIGNAL(closeTabSignal(const int)), this,
	  SLOT(slotCloseTab(const int)));
  connect(ui.actionClose_Window, SIGNAL(triggered(void)), this,
	  SLOT(slotClose(void)));
  connect(ui.actionQuit, SIGNAL(triggered(void)), this,
	  SLOT(slotQuit(void)));
  connect(ui.locationLineEdit->menu(), SIGNAL(triggered(QAction *)),
	  this, SLOT(slotLoadPage(QAction *)));
  connect(ui.locationLineEdit->goButton(), SIGNAL(clicked(void)),
	  this, SLOT(slotLoadPage(void)));
  connect(ui.locationLineEdit, SIGNAL(textChanged(const QString &)),
	  this, SLOT(slotTextChanged(const QString &)));
  connect(ui.locationLineEdit, SIGNAL(returnPressed(void)), this,
	  SLOT(slotLoadPage(void)));
  connect(ui.backToolButton, SIGNAL(clicked(void)), this,
	  SLOT(slotBack(void)));
  connect(ui.forwardToolButton, SIGNAL(clicked(void)), this,
	  SLOT(slotForward(void)));
  connect(ui.reloadToolButton, SIGNAL(clicked(void)), this,
	  SLOT(slotReload(void)));
  connect(ui.actionReload, SIGNAL(triggered(void)), this,
	  SLOT(slotReload(void)));
  connect(ui.stopToolButton, SIGNAL(clicked(void)), this,
	  SLOT(slotStop(void)));
  connect(ui.actionStop, SIGNAL(triggered(void)), this,
	  SLOT(slotStop(void)));
  connect(ui.homeToolButton, SIGNAL(clicked(void)), this,
	  SLOT(slotGoHome(void)));
  connect(ui.searchLineEdit, SIGNAL(returnPressed(void)), this,
	  SLOT(slotSearch(void)));
  connect(ui.searchLineEdit->findButton(), SIGNAL(clicked(void)), this,
	  SLOT(slotSearch(void)));
  connect(ui.actionAbout_Dooble, SIGNAL(triggered(void)), this,
	  SLOT(slotAbout(void)));
  connect(ui.actionSave_Page, SIGNAL(triggered(void)), this,
	  SLOT(slotSavePage(void)));
  connect(ui.actionDownloads, SIGNAL(triggered(void)), this,
	  SLOT(slotDisplayDownloadWindow(void)));
  connect(ui.actionFind, SIGNAL(triggered(void)), this,
	  SLOT(slotShowFind(void)));
  connect(ui.actionPrint, SIGNAL(triggered(void)), this,
	  SLOT(slotPrint(void)));
  connect(ui.actionPrint_Preview, SIGNAL(triggered(void)), this,
	  SLOT(slotPrintPreview(void)));
  connect(ui.actionDesktop, SIGNAL(triggered(void)), this,
	  SLOT(slotShowDesktopTab(void)));
  connect(ui.actionPage_Source, SIGNAL(triggered(void)), this,
	  SLOT(slotShowPageSource(void)));
  connect(ui.actionSettings, SIGNAL(triggered(void)), this,
	  SLOT(slotShowSettingsWindow(void)));
  connect(ui.hideFindToolButton, SIGNAL(clicked(void)), this,
	  SLOT(slotHideFind(void)));
  connect(ui.findLineEdit, SIGNAL(returnPressed(void)), this,
	  SLOT(slotFind(void)));
  connect(ui.findLineEdit, SIGNAL(textChanged(const QString &)), this,
	  SLOT(slotFind(const QString &)));
  connect(ui.desktopToolButton, SIGNAL(clicked(void)), this,
	  SLOT(slotShowDesktopTab(void)));
  connect(ui.actionStatusbar, SIGNAL(triggered(void)), this,
	  SLOT(slotStatusBarDisplay(void)));
#if QT_VERSION >= 0x040500
  connect(ui.actionZoom_In, SIGNAL(triggered(void)), this,
	  SLOT(slotViewZoomIn(void)));
  connect(ui.actionZoom_Out, SIGNAL(triggered(void)), this,
	  SLOT(slotViewZoomOut(void)));
  connect(ui.actionReset_Zoom, SIGNAL(triggered(void)), this,
	  SLOT(slotViewResetZoom(void)));
  connect(ui.actionZoom_Text_Only, SIGNAL(toggled(bool)), this,
	  SLOT(slotViewZoomTextOnly(bool)));
  ui.actionZoom_Text_Only->setCheckable(true);
  ui.actionZoom_Text_Only->setChecked(false);
  ui.viewMenu->insertSeparator(ui.zoomMenu->menuAction());
#else
  ui.zoomMenu->setVisible(false);

  foreach(QAction *action, ui.zoomMenu->findChildren<QAction *> ())
    action->setVisible(false);
#endif
  ui.graphicsView->setScene(desktopWidget);

  for(int i = 0; i < actionsList.size(); i++)
    {
      QAction *action = 0;

      if(i == 0)
	{
	  action = ui.historyMenu->addAction(actionsList.at(i)->icon(),
					     actionsList.at(i)->text());

	  connect(action, SIGNAL(triggered(bool)), this,
		  SLOT(slotClearHistory(bool)));
	  ui.historyMenu->addSeparator();
	}
      else if(i > 1) // Neglect the separator.
	{
	  action = ui.historyMenu->addAction(actionsList.at(i)->icon(),
					     actionsList.at(i)->text());

	  action->setData(actionsList.at(i)->data());
	  connect(action, SIGNAL(triggered(bool)), this,
		  SLOT(slotHistoryActionTriggered(bool)));
	}
    }

  ui.forwardToolButton->setVisible(false);
  ui.backToolButton->setVisible(false);
  ui.stopToolButton->setVisible(false);
  ui.reloadToolButton->setVisible(false);
  ui.homeToolButton->setVisible(false);
  ui.searchLineEdit->setVisible(false);
  ui.locationLineEdit->setVisible(false);
  ui.editMenu->setEnabled(false);
  ui.viewMenu->setEnabled(false);
  ui.historyMenu->setEnabled(false);
  ui.actionPrint->setEnabled(false);
  ui.actionPrint_Preview->setEnabled(false);
  ui.actionOpen_URL->setEnabled(false);
  ui.actionSave_Page->setEnabled(false);
  ui.desktopToolButton->setVisible(false);

  /*
  ** According to Qt documentation, resize() followed by move() should
  ** be used to restore a window's geometry. Actual results may vary.
  */

  QSettings settings;

  if(settings.contains("mainWindow/geometry"))
    setGeometry(settings.value("mainWindow/geometry").toRect());

  statusBar()->addPermanentWidget(sbWidget, 100);
  statusBar()->setStyleSheet("QStatusBar::item {"
			     "border: none; "
			     "}");
  statusBar()->setMaximumHeight(sbWidget->height());
  statusBar()->setVisible
    (settings.value("mainWindow/statusbarDisplay", true).toBool());
  ui.actionStatusbar->setChecked(!statusBar()->isHidden());
  slotSetIcons();
}

dooble::dooble(QList<QAction *> actionsList):QMainWindow()
{
  init(actionsList);

  QSettings settings;

  if(!settings.value("settingsWindow/displayDesktopCheckBox", false).toBool())
    {
      closeDesktop(); // Close the Desktop tab.
      QApplication::sendPostedEvents(this, 0);
    }

  show();
}

dooble::dooble(QList<QAction *> actionsList, const QUrl &url):
  QMainWindow(), dest_url_(url.toString())
{
  init(actionsList);
  newTab(url);

  QSettings settings;

  if(!settings.value("settingsWindow/displayDesktopCheckBox", false).toBool())
    {
      closeDesktop(); // Close the Desktop tab.
      QApplication::sendPostedEvents(this, 0);
    }

  show();
}

dooble::dooble(QList<QAction *> actionsList, dview *p,
	       const QString &tabTitle):
  QMainWindow(), dest_url_(p->url().toString())
{
  init(actionsList);
  newTab(p, tabTitle);

  QSettings settings;

  if(!settings.value("settingsWindow/displayDesktopCheckBox", false).toBool())
    {
      closeDesktop(); // Close the Desktop tab.
      QApplication::sendPostedEvents(this, 0);
    }

  show();
}

dooble::~dooble()
{
  doobleCounter -= 1;
}

void dooble::slotSetIcons(void)
{
  QSettings settings;
  QSettings cfgSettings(settings.value("iconSet").toString(),
			QSettings::IniFormat);

  cfgSettings.beginGroup("mainWindow");
  ui.backToolButton->setIcon
    (QIcon(cfgSettings.value("backToolButton").toString()));
  ui.forwardToolButton->setIcon
    (QIcon(cfgSettings.value("forwardToolButton").toString()));
  ui.reloadToolButton->setIcon
    (QIcon(cfgSettings.value("reloadToolButton").toString()));
  ui.stopToolButton->setIcon
    (QIcon(cfgSettings.value("stopToolButton").toString()));
  ui.homeToolButton->setIcon
    (QIcon(cfgSettings.value("homeToolButton").toString()));
  ui.hideFindToolButton->setIcon
    (QIcon(cfgSettings.value("hideFindToolButton").toString()));
  ui.actionNew_Tab->setIcon
    (QIcon(cfgSettings.value("actionNew_Tab").toString()));
  ui.actionNew_Window->setIcon
    (QIcon(cfgSettings.value("actionNew_Window").toString()));
  ui.actionOpen_URL->setIcon
    (QIcon(cfgSettings.value("actionOpen_URL").toString()));
  ui.actionClose_Tab->setIcon
    (QIcon(cfgSettings.value("actionClose_Tab").toString()));
  ui.actionClose_Window->setIcon
    (QIcon(cfgSettings.value("actionClose_Window").toString()));
  ui.actionSave_Page->setIcon
    (QIcon(cfgSettings.value("actionSave_Page").toString()));
  ui.actionPrint->setIcon
    (QIcon(cfgSettings.value("actionPrint").toString()));
  ui.actionPrint_Preview->setIcon
    (QIcon(cfgSettings.value("actionPrint_Preview").toString()));
  ui.actionQuit->setIcon
    (QIcon(cfgSettings.value("actionQuit").toString()));
  ui.actionFind->setIcon
    (QIcon(cfgSettings.value("actionFind").toString()));
  ui.actionReload->setIcon
    (QIcon(cfgSettings.value("actionReload").toString()));
  ui.actionStop->setIcon
    (QIcon(cfgSettings.value("actionStop").toString()));
  ui.actionDesktop->setIcon
    (QIcon(cfgSettings.value("actionDesktop").toString()));
  ui.actionPage_Source->setIcon
    (QIcon(cfgSettings.value("actionPage_Source").toString()));
  ui.actionDownloads->
    setIcon(QIcon(cfgSettings.value("actionDownloads").toString()));
  ui.actionSettings->setIcon
    (QIcon(cfgSettings.value("actionSettings").toString()));
  ui.actionAbout_Dooble->
    setIcon(QIcon(cfgSettings.value("actionAbout_Dooble").toString()));
  ui.desktopToolButton->
    setIcon(QIcon(cfgSettings.value("desktopToolButton").toString()));
  ui.tabWidget->setTabIcon
    (0, QIcon(cfgSettings.value("tabWidget").toString()));
  setWindowIcon(QIcon(cfgSettings.value("windowIcon").toString()));
  emit iconsChangedSignal();
}

void dooble::newTabInit(dview *p, const QString &tabTitle)
{
  connect(p, SIGNAL(urlChanged(const QUrl &)), this,
	  SLOT(slotUrlChanged(const QUrl &)));
  connect(p, SIGNAL(titleChanged(const QString &)), this,
	  SLOT(slotTitleChanged(const QString &)));
  connect(p, SIGNAL(loadFinished(bool)), this,
	  SLOT(slotLoadFinished(bool)));
  connect(p, SIGNAL(loadProgress(int)), this,
	  SLOT(slotLoadProgress(int)));
  connect(p, SIGNAL(loadStarted(void)), this,
	  SLOT(slotLoadStarted(void)));
  connect(p, SIGNAL(openLinkInNewTabSignal(const QUrl &)), this,
	  SLOT(slotOpenLinkInNewTab(const QUrl &)));
  connect(p, SIGNAL(openLinkInNewWindowSignal(const QUrl &)), this,
	  SLOT(slotOpenLinkInNewWindow(const QUrl &)));
  connect(p, SIGNAL(copyLinkSignal(const QUrl &)), this,
	  SLOT(slotCopyLink(const QUrl &)));
  connect(p, SIGNAL(saveUrlSignal(const QUrl &)), this,
	  SLOT(slotSaveUrl(const QUrl &)));
  connect(p, SIGNAL(viewImageSignal(const QUrl &)), this,
	  SLOT(slotLoadPage(const QUrl &)));
  connect(p->page(),
	  SIGNAL(linkHovered(const QString &, const QString &,
			     const QString &)),
	  this,
	  SLOT(slotLinkHovered(const QString &, const QString &,
			       const QString &)));
  connect(p->page(),
	  SIGNAL(unsupportedContent(QNetworkReply *)),
	  this,
	  SLOT(slotUnsupportedContent(QNetworkReply *)));
  connect(p->page(),
	  SIGNAL(linkClicked(const QUrl &)),
	  this,
	  SLOT(slotLoadPage(const QUrl &)));
  connect(this,
	  SIGNAL(clearHistorySignal(void)),
	  p,
	  SLOT(slotClearHistory(void)));
  ui.tabWidget->addTab(p, p->icon(), tabTitle);

  QSettings settings;

  if(settings.value("settingsWindow/proceedToNewTab", true).toBool())
    ui.tabWidget->setCurrentWidget(p);

  sb.statusLabel->clear();
#if QT_VERSION >= 0x040500
  ui.tabWidget->setTabsClosable(ui.tabWidget->count() > 1);
#endif
  ui.actionClose_Tab->setEnabled(ui.tabWidget->count() > 1);
}

dview *dooble::newTab(dview *p, const QString &tabTitle)
{
  p->setParent(this);
  newTabInit(p, tabTitle);
  return p;
}

dview *dooble::newTab(const QUrl &url)
{
  dview *p = new dview(this);

  newTabInit(p, tr("(Untitled)"));
  p->load(url);
  return p;
}

void dooble::slotQuit(void)
{
#if defined(Q_WS_WIN)
  QProcess process;

  process.start("taskkill /F /IM offsystem.exe");
  QProcess::startDetached("stopYACY.bat");
#else
  QProcess::startDetached("./stopYACY.sh");
#endif
  downloadWindow->deleteLater();
  saveSettings();
  exit(0);
}

void dooble::saveSettings(void)
{
  QSettings settings;

  settings.setValue("mainWindow/geometry", geometry());
}

void dooble::loadPage(const QUrl &url)
{
  dview *p = qobject_cast<dview *> (ui.tabWidget->currentWidget());

  if(p)
    {
      int index = -1;
      QString str(url.toString().trimmed());
      QString scheme(QUrl(str).scheme().toLower());

      if(str.contains(" ", Qt::CaseInsensitive) &&
	 (!(scheme == "file" || scheme == "ftp" ||
	    scheme == "http" || scheme == "https" ||
	    str.contains("/", Qt::CaseInsensitive))))
	str.prepend("http://141.52.175.50:8080/yacysearch.html?search=");
      else if(!(scheme == "file" || scheme == "ftp" ||
		scheme == "http" || scheme == "https"))
	str.prepend("http://");

      /*
      ** Replace the user's text with an improved version.
      */

      if((index = ui.locationLineEdit->findText(url.toString())) > -1)
	ui.locationLineEdit->setItemText(index, str);

      ui.locationLineEdit->setIcon(p->icon());
      ui.locationLineEdit->blockSignals(true);
      ui.locationLineEdit->setText(str);
      ui.locationLineEdit->blockSignals(false);
      p->load(QUrl(str));
      sb.progressBar->setVisible(true);
      ui.stopToolButton->setEnabled(true);
      ui.actionStop->setEnabled(true);
    }
}

void dooble::slotLoadPage(void)
{
  loadPage(QUrl(ui.locationLineEdit->text().trimmed()));
}

void dooble::slotLoadPage(bool)
{
  loadPage(QUrl(dest_url_));
}

void dooble::slotLoadPage(const QUrl &url)
{
  loadPage(url);
}

void dooble::slotLoadPage(QAction *action)
{
  loadPage(QUrl(action->text()));
}

void dooble::slotNewTab(void)
{
  QUrl url = QUrl::fromLocalFile
    (QString("%1/Tab/Default/search.html").arg(QDir::currentPath()));

  newTab(url);
}

void dooble::slotNewWindow(void)
{
  QUrl url = QUrl::fromLocalFile
    (QString("%1/Tab/Default/search.html").arg(QDir::currentPath()));

  Q_UNUSED(new dooble(ui.historyMenu->actions(), url));
}

void dooble::slotClose(void)
{
  close();
}

void dooble::closeEvent(QCloseEvent *event)
{
  event->accept();
  deleteLater();

  if(doobleCounter <= 1)
    slotQuit();
}

void dooble::slotStop(void)
{
  dview *p = qobject_cast<dview *> (ui.tabWidget->currentWidget());

  if(p)
    p->stop();

  sb.progressBar->setVisible(false);
  ui.stopToolButton->setEnabled(false);
  ui.actionStop->setEnabled(false);
}

void dooble::slotReload(void)
{
  dview *p = qobject_cast<dview *> (ui.tabWidget->currentWidget());

  if(p)
    {
      p->reload();
      sb.progressBar->setVisible(true);
      ui.stopToolButton->setEnabled(true);
      ui.actionStop->setEnabled(true);
    }
}

void dooble::slotBack(void)
{
  dview *p = qobject_cast<dview *> (ui.tabWidget->currentWidget());

  if(p)
    {
      p->back();
      ui.forwardToolButton->setEnabled(p->page()->history()->canGoForward());
      sb.progressBar->setVisible(true);
      ui.stopToolButton->setEnabled(true);
      ui.actionStop->setEnabled(true);
    }
}

void dooble::slotForward(void)
{
  dview *p = qobject_cast<dview *> (ui.tabWidget->currentWidget());

  if(p)
    {
      p->forward();
      ui.backToolButton->setEnabled(p->page()->history()->canGoBack());
      sb.progressBar->setVisible(true);
      ui.stopToolButton->setEnabled(true);
      ui.actionStop->setEnabled(true);
    }
}

void dooble::slotTabSelected(const int index)
{
  dview *p = qobject_cast<dview *> (ui.tabWidget->widget(index));

  if(p)
    {
      ui.forwardToolButton->setVisible(true);
      ui.backToolButton->setVisible(true);
      ui.stopToolButton->setVisible(true);
      ui.reloadToolButton->setVisible(true);
      ui.homeToolButton->setVisible(true);
      ui.searchLineEdit->setVisible(true);
      ui.locationLineEdit->setVisible(true);
      ui.reloadToolButton->setEnabled(true);
      ui.homeToolButton->setEnabled(true);
      ui.searchLineEdit->setEnabled(true);
      ui.editMenu->setEnabled(true);
      ui.viewMenu->setEnabled(true);
      ui.historyMenu->setEnabled(true);
      ui.actionPrint->setEnabled(true);
      ui.actionPrint_Preview->setEnabled(true);
      ui.actionOpen_URL->setEnabled(true);
      ui.actionSave_Page->setEnabled(true);

      if(p->title().isEmpty())
	setWindowTitle(tr("Dooble Web Browser"));
      else
	setWindowTitle(p->title() + tr(" - Dooble Web Browser"));

      sb.progressBar->setVisible(!p->isLoaded());
      sb.progressBar->setValue(p->progress());
      ui.stopToolButton->setEnabled(!p->isLoaded());
      ui.actionStop->setEnabled(!p->isLoaded());
      ui.backToolButton->setEnabled(p->page()->history()->canGoBack());
      ui.forwardToolButton->setEnabled(p->page()->history()->canGoForward());
      ui.locationLineEdit->setIcon(p->icon());
      ui.locationLineEdit->blockSignals(true);
      ui.locationLineEdit->setText(p->url().toString());
      ui.locationLineEdit->blockSignals(false);
      prepareNavigationButtonMenus(p);
      p->setFocus();
      statusBar()->setVisible(ui.actionStatusbar->isChecked());
    }
  else
    {
      setWindowTitle(tr("Dooble Web Browser"));
      ui.forwardToolButton->setVisible(false);
      ui.backToolButton->setVisible(false);
      ui.stopToolButton->setVisible(false);
      ui.reloadToolButton->setVisible(false);
      ui.homeToolButton->setVisible(false);
      ui.searchLineEdit->setVisible(false);
      ui.locationLineEdit->setVisible(false);
      ui.editMenu->setEnabled(false);
      ui.viewMenu->setEnabled(false);
      ui.historyMenu->setEnabled(false);
      ui.actionPrint->setEnabled(false);
      ui.actionPrint_Preview->setEnabled(false);
      ui.actionOpen_URL->setEnabled(false);
      ui.actionSave_Page->setEnabled(false);
      ui.searchLineEdit->clear();
      ui.findFrame->setVisible(false);
      sb.statusLabel->clear();
      sb.progressBar->setValue(0);
      sb.progressBar->setVisible(false);
      statusBar()->setVisible(false);
    }
}

void dooble::slotTextChanged(const QString &text)
{
  dest_url_ = text;
}

void dooble::slotUrlChanged(const QUrl &url)
{
  QString urlText(url.toString());

  if(ui.locationLineEdit->findText(urlText) == -1)
    ui.locationLineEdit->addItem(urlText);

  dview *p = qobject_cast<dview *> (sender());

  if(p == qobject_cast<dview *> (ui.tabWidget->currentWidget()))
    {
      ui.locationLineEdit->blockSignals(true);
      ui.locationLineEdit->setText(urlText);
      ui.locationLineEdit->blockSignals(false);
    }
}

void dooble::slotTitleChanged(const QString &title_)
{
  dview *p = qobject_cast<dview *> (sender());
  QString title(title_.trimmed());

  if(p == qobject_cast<dview *> (ui.tabWidget->currentWidget()))
    {
      if(title_.isEmpty())
	setWindowTitle(tr("Dooble Web Browser"));
      else
	setWindowTitle(title + tr(" - Dooble Web Browser"));
    }

  if(p)
    {
      if(title.isEmpty())
	/*
	** The tab's title will be set to the URL.
	*/

	title = p->url().toString();

      /*
      ** Restrict the size of the tab.
      */

      if(title.length() > MAX_NUMBER_OF_TAB_TITLE_CHARACTERS)
	title = title.mid(0, MAX_NUMBER_OF_TAB_TITLE_CHARACTERS - 3) + "...";
      else if(title.length() < MAX_NUMBER_OF_TAB_TITLE_CHARACTERS)
	title = title.leftJustified(MAX_NUMBER_OF_TAB_TITLE_CHARACTERS, ' ');

      ui.tabWidget->setTabText(ui.tabWidget->indexOf(p),
			       title.replace("&", "&&"));
      ui.tabWidget->setTabToolTip(ui.tabWidget->indexOf(p),
				  p->title());
    }
}

void dooble::slotLoadProgress(int progress)
{
  dview *p = qobject_cast<dview *> (sender());

  if(p == qobject_cast<dview *> (ui.tabWidget->currentWidget()))
    {
      sb.progressBar->setValue(progress);
      ui.locationLineEdit->setIcon(p->icon());
    }

  if(p)
    {
      int index = ui.locationLineEdit->findText(p->url().toString());

      if(index > -1)
	ui.locationLineEdit->setItemIcon(index, p->icon());

      ui.tabWidget->setTabIcon(ui.tabWidget->indexOf(p), p->icon());
    }
}

void dooble::slotUnsupportedContent(QNetworkReply *reply)
{
  Q_UNUSED(reply);
}

void dooble::slotLoadFinished(bool ok)
{
  Q_UNUSED(ok);
  dview *p = qobject_cast<dview *> (sender());

  if(p == qobject_cast<dview *> (ui.tabWidget->currentWidget()))
    {
      sb.progressBar->setVisible(false);
      ui.stopToolButton->setEnabled(false);
      ui.actionStop->setEnabled(false);
      ui.backToolButton->setEnabled(p->page()->history()->canGoBack());
      ui.forwardToolButton->setEnabled(p->page()->history()->canGoForward());
    }

  if(p && !p->title().isNull())
    {
      bool actionExists = false;
      QList<QAction *> actions = ui.historyMenu->actions();

      for(int i = 0; i < actions.size(); i++)
 	if(actions.at(i)->data().toString() == p->url().toString())
 	  {
 	    actionExists = true;
 	    break;
 	  }

      actions.clear();

      if(!actionExists)
 	{
	  if(ui.historyMenu->actions().isEmpty())
	    {
	      QSettings settings;
	      QSettings cfgSettings
		(settings.value("iconSet").toString(),
		 QSettings::IniFormat);

	      cfgSettings.beginGroup("mainWindow");

	      QAction *action = ui.historyMenu->addAction
		(QIcon(cfgSettings.value("historyMenu").toString()),
		 tr("&Clear History"));

	      connect(action, SIGNAL(triggered(bool)), this,
		      SLOT(slotClearHistory(bool)));
	      ui.historyMenu->addSeparator();
	    }
	  else if(ui.historyMenu->actions().size() == 1)
	    {
	      ui.historyMenu->actions().at(0)->setEnabled(true);
	      ui.historyMenu->addSeparator();
	    }

 	  QString title(p->title());

	  if(title.isEmpty())
	    title = p->url().toString();

 	  if(title.length() > MAX_NUMBER_OF_MENU_TITLE_CHARACTERS)
 	    title = title.mid(0, MAX_NUMBER_OF_MENU_TITLE_CHARACTERS - 3) +
	      "...";

 	  if(title.contains("&"))
 	    title = title.replace("&", "&&");

 	  QAction *action = ui.historyMenu->addAction
	    (p->icon(), title, this, SLOT(slotHistoryActionTriggered(bool)));

	  if(action)
	    action->setData
	      (QVariant(p->url().toString()));
 	}
    }

  prepareNavigationButtonMenus(p);
}

void dooble::prepareNavigationButtonMenus(dview *p)
{
  if(!p)
    return;

  QList<QWebHistoryItem> list(p->history()->backItems(15));

  ui.backToolButton->menu()->clear();

  for(int i = 0; i < list.size(); i++)
    {
      QString title(list.at(i).title());

      if(title.length() > MAX_NUMBER_OF_MENU_TITLE_CHARACTERS)
	title = title.mid(0, MAX_NUMBER_OF_MENU_TITLE_CHARACTERS - 3) + "...";

      if(title.contains("&"))
	title = title.replace("&", "&&");

      QIcon icon(QWebSettings::iconForUrl(list.at(i).url()));

      if(icon.isNull())
	{
	  QSettings settings;
	  QSettings cfgSettings
	    (settings.value("iconSet").toString(), QSettings::IniFormat);

	  cfgSettings.beginGroup("mainWindow");
	  icon = QIcon(cfgSettings.value("emptyIcon").toString());
	}

      QAction *action = ui.backToolButton->menu()->addAction
	(icon, title, this, SLOT(slotHistoryActionTriggered(bool)));

      if(action)
	action->setData
	  (QVariant(list.at(i).url().toString()));
    }

  ui.forwardToolButton->menu()->clear();
  list = p->history()->forwardItems(15);

  for(int i = 0; i < list.size(); i++)
    {
      QString title(list.at(i).title());

      if(title.length() > MAX_NUMBER_OF_MENU_TITLE_CHARACTERS)
	title = title.mid(0, MAX_NUMBER_OF_MENU_TITLE_CHARACTERS - 3) + "...";

      if(title.contains("&"))
	title = title.replace("&", "&&");

      QIcon icon(QWebSettings::iconForUrl(list.at(i).url()));

      if(icon.isNull())
	{
	  QSettings settings;
	  QSettings cfgSettings
	    (settings.value("iconSet").toString(), QSettings::IniFormat);

	  cfgSettings.beginGroup("mainWindow");
	  icon = QIcon(cfgSettings.value("emptyIcon").toString());
	}

      QAction *action = ui.forwardToolButton->menu()->addAction
	(icon, title, this, SLOT(slotHistoryActionTriggered(bool)));

      if(action)
	action->setData
	  (QVariant(list.at(i).url().toString()));
    }
}

void dooble::slotGoHome(void)
{
  QSettings settings;

  settings.beginGroup("settingsWindow");

  QStringList allKeys = settings.allKeys();

  for(int i = 0; i < allKeys.size(); i++)
    {
      if(!allKeys[i].startsWith("url"))
	continue;

      QString value(settings.value(allKeys[i], "").toString().trimmed());

      if(value.endsWith("/"))
	value.remove(value.size() - 1);

      if(value.startsWith("file://"))
	value.remove("file://");
      else if(value.startsWith("http://"))
	value.remove("http://");
      else if(value.startsWith("https://"))
	value.remove("https://");
      else if(value.startsWith("http://www."))
	value.remove("http://www.");
      else if(value.startsWith("https://www."))
	value.remove("https://www.");

      if(value.startsWith("www."))
	value.remove(0, 4);

      if(!value.isEmpty())
	{
	  bool urlFound = false;

	  foreach(dview *p, findChildren<dview *> ())
	    {
	      QString url(p->url().toString(QUrl::StripTrailingSlash));

	      if(url.startsWith("file://"))
		url.remove("file://");
	      else if(url.startsWith("http://"))
		url.remove("http://");
	      else if(url.startsWith("https://"))
		url.remove("https://");
	      else if(url.startsWith("http://www."))
		url.remove("http://www.");
	      else if(url.startsWith("https://www."))
		url.remove("https://www.");

	      if(url.startsWith("www."))
		url.remove(0, 4);

	      if(url == value)
		{
		  urlFound = true;
		  break;
		}
	    }

	  if(!urlFound)
	    {
	      value = settings.value(allKeys[i], "").toString().trimmed();

	      if(!(value.startsWith("ftp") || value.startsWith("file") ||
		   value.startsWith("http") || value.startsWith("https")))
		value.prepend("http://");

	      newTab(QUrl(value));
	    }
	}
    }

  allKeys.clear();
}

void dooble::closeTab(const int index)
{
  if(index > -1)
    {
      int count = 0;

      if((count = ui.tabWidget->count()) > 1)
	{
	  dview *p = qobject_cast<dview *> (ui.tabWidget->widget(index));

	  if(p)
	    {
	      p->stop();
	      p->deleteLater();
	    }
	  else
	    closeDesktop();

	  count -= 1;
	}

#if QT_VERSION >= 0x040500
      ui.tabWidget->setTabsClosable(count > 1);
#endif
      ui.actionClose_Tab->setEnabled(count > 1);
    }
}

void dooble::slotCloseTab(void)
{
  closeTab(ui.tabWidget->currentIndex());
}

void dooble::slotCloseTab(const int index)
{
  closeTab(index);
}

void dooble::slotOpenUrl(void)
{
  ui.locationLineEdit->setFocus();
  ui.locationLineEdit->selectAll();
  update();
}

void dooble::slotHistoryActionTriggered(bool checked)
{
  Q_UNUSED(checked);
  QAction *action = qobject_cast<QAction *> (sender());

  if(action)
    slotLoadPage(action->data().toString());
}

void dooble::slotLoadStarted(void)
{
  dview *p = qobject_cast<dview *> (sender());

  if(p == qobject_cast<dview *> (ui.tabWidget->currentWidget()))
    {
      sb.statusLabel->clear();
      sb.progressBar->setVisible(true);
      ui.stopToolButton->setEnabled(true);
      ui.actionStop->setEnabled(true);
      p->setFocus();
    }
}

void dooble::slotSearch(void)
{
  dview *p = qobject_cast<dview *> (ui.tabWidget->currentWidget());

  if(p)
    {
      QString urlText("");
      QString searchName(ui.searchLineEdit->type());

      if(searchName == "dooble search")
	urlText = QString("http://www.metager.de/meta/cgi-bin/meta.ger1?"
			  "start=1&mm=and&maxtreffer=200&time=3"
			  "&hitsPerServer=5&textmenge=1&wikiboost=on"
			  "&QuickTips=beschleuniger&linkTest=no"
			  "&check_time=3&wikipedia=on"
			  "&yacy=on&eingabe=%1").arg
	  (ui.searchLineEdit->text().trimmed());
      else if(searchName == "my dooble search")
	urlText = QString("http://localhost:8080/yacy/user/"
			  "ysearch.html?search=%1").arg
	  (ui.searchLineEdit->text().trimmed());
      else if(searchName == "kvk booksearch")
	urlText = QString("http://kvk.ubka.uni-karlsruhe.de/"
			  "hylib-bin/kvk/nph-kvk2.cgi?maske="
			  "kvk-last&title=UB+Karlsruhe:+Karlsruhe+"
			  "Virtual+Catalog+KVK+:+Search+"
			  "results&header=http://www.ubka.uni-karlsruhe.de/"
			  "kvk/kvk/kvk-header_en_2007_07.html&spacer="
			  "http://www.ubka.uni-karlsruhe.de/kvk/kvk/"
			  "kvk-spacer_en_2007_07.html&footer="
			  "http://www.ubka.uni-karlsruhe.de/kvk/kvk/"
			  "kvk-footer_en_2008_01.html&lang=en&css="
			  "http://www.ubka.uni-karlsruhe.de/kvk/kvk/"
			  "kvk-neu2.css&input-charset=utf-8&TI=&PY="
			  "&AU=&SB=&CI=&SS=&ST=&PU=&VERBUENDE=&kataloge="
			  "SWB&kataloge=BVB&kataloge=NRW&kataloge="
			  "HEBIS&kataloge=HEBIS_RETRO&kataloge="
			  "KOBV&kataloge=GBV&kataloge=DDB&kataloge="
			  "DFG_EBOOKS&kataloge=STABI_BERLIN&kataloge="
			  "TIB&kataloge=OEVK_GBV&kataloge=ZDB&OESTERREICH="
			  "&kataloge=BIBOPAC&kataloge=OENB_1992&SCHWEIZ="
			  "&kataloge=HELVETICAT&kataloge=BASEL&kataloge="
			  "ZUERICH&kataloge=ETH&kataloge=NLAU&kataloge="
			  "CISTI&kataloge=NLCA&kataloge=NB_TSCHECHIEN&"
			  "kataloge=DAENEMARK_REX&kataloge=EROMM&kataloge="
			  "UBHS&kataloge=BNF_PARIS&kataloge=ABES&kataloge="
			  "NB_UNGARN&kataloge=VERBUND_ISRAEL&kataloge="
			  "ITALIEN_VERBUND&kataloge=ITALIEN_SERIALS&kataloge="
			  "LUXEMBURG&kataloge=NB_NIEDERLANDE&kataloge="
			  "VERBUND_NORWEGEN&kataloge=NB_POLEN&kataloge="
			  "PORTUGAL&kataloge=STAATSBIB_RUSSLAND&kataloge="
			  "BNE&kataloge=REBIUN&kataloge=VERBUND_SCHWEDEN&"
			  "kataloge=COPAC&kataloge=BL&kataloge=LOC_DIREKT&"
			  "kataloge=NLM&target=_blank&Timeout=120&ALL=%1").arg
	  (ui.searchLineEdit->text().trimmed());
      else if(searchName == "metager")
	urlText = QString("http://www.metager.de/meta/cgi-bin/meta.ger1?"
			  "start=1&mm=and&maxtreffer=100&time="
			  "3&hitsPerServer=5&textmenge=1&wikiboost="
			  "on&QuickTips=beschleuniger&linkTest=no&"
			  "check_time=3&exalead=on&wikipedia=on&"
			  "yacy=on&msn=on&yahoo=on&bildersuche="
			  "on&audioclipping=on&onlinks=on&eingabe=%1").arg
	  (ui.searchLineEdit->text().trimmed());
      else if(searchName == "lilirasa")
	urlText = QString("http://85.25.147.200:8080/"
			  "yacysearch.html?search=%1").arg
	  (ui.searchLineEdit->text().trimmed());
      else if(searchName == "opendesktop apps")
        urlText = QString("http://www.opendesktop.org/content/"
                          "search.php?search=1&name=%1&text=%1").arg
          (ui.searchLineEdit->text().trimmed());
      else if(searchName == "openstreetmap")
	urlText = QString("http://gazetteer.openstreetmap.org/"
			  "namefinder/?find=%1").arg
	  (ui.searchLineEdit->text().trimmed());
      else if(searchName == "sciencenet")
	{
	  urlText = "http://sciencenet.fzk.de:8080/yacysearch.html?query=";
 
	  QString searchText(ui.searchLineEdit->text().trimmed());
  
	  searchText = searchText.replace(" ", "+");
	  urlText += searchText + "&Enter=query";
	}
      else if(searchName == "wikipedia")
	{
	  QLocale locale;

	  urlText = QString
	    ("http://%1.wikipedia.org/wiki/Special:"
	     "Search?search=%2").
	    arg(locale.name().left(2)).
	    arg(ui.searchLineEdit->text().trimmed());
	}
      else if(searchName == "yacy suche")
	urlText = QString("http://www.yacy-suche.de/yacysearch.html?"
			  "display=0&query=%1").arg(ui.searchLineEdit->
						    text().trimmed());

      p->load(QUrl(urlText));
    }
}

void dooble::slotCopyLink(const QUrl &url)
{
  static_cast<QApplication *> (QCoreApplication::instance())->
    clipboard()->setText(url.toString());
}

void dooble::slotOpenLinkInNewTab(const QUrl &url)
{
  newTab(url);
}

void dooble::slotOpenLinkInNewWindow(const QUrl &url)
{
  Q_UNUSED(new dooble(ui.historyMenu->actions(), url));
}

void dooble::slotOpenPageInNewWindow(const int index)
{
  dview *p = qobject_cast<dview *> (ui.tabWidget->widget(index));

  if(p)
    Q_UNUSED(new dooble(ui.historyMenu->actions(), p,
			ui.tabWidget->tabText(index)));
}

void dooble::slotLinkHovered(const QString &link, const QString &title,
			     const QString &textContent)
{
  Q_UNUSED(title);
  Q_UNUSED(textContent);
  sb.statusLabel->setText(link.trimmed().mid(0, 100));
}

void dooble::slotAbout(void)
{
  QMessageBox mb(this);

  mb.setWindowTitle(tr("Dooble Web Browser: About"));
  mb.setTextFormat(Qt::RichText);
  mb.setText("<html>"
             "Version 0.07.<br><br>"
             "Open Source software for Linux, Mac, and Windows.<br>"
             "Copyright (c) 2008, 2009<br>"
             "The Dooble Team."
	     "<hr>"
             "Please visit <a href='http://dooble.sf.net'>"
             "http://dooble.sf.net</a><br>for more information."
             "</html>");
  mb.setStandardButtons(QMessageBox::Ok);
  mb.setIconPixmap(QPixmap("Icons/AxB/dooble.png"));
  mb.exec();
}

void dooble::slotSavePage(void)
{
  dview *p = qobject_cast<dview *> (ui.tabWidget->currentWidget());

  if(p)
    saveUrl(p->url());
}

void dooble::slotSaveUrl(const QUrl &url)
{
  saveUrl(url);
}

void dooble::saveUrl(const QUrl &url)
{
  if(url.isEmpty())
    return;

  QString currentPath = QDesktopServices::storageLocation
    (QDesktopServices::DesktopLocation);

  if(currentPath.isEmpty())
    currentPath = QDir::homePath();

  QWidget *parent = qobject_cast<QWidget *> (sender());

  if(!parent)
    parent = this;

  QFileDialog fileDialog(parent);

  fileDialog.setFileMode(QFileDialog::AnyFile);
  fileDialog.setDirectory(currentPath);
  fileDialog.setLabelText(QFileDialog::Accept, tr("&Save"));
  fileDialog.setAcceptMode(QFileDialog::AcceptSave);

#ifdef Q_WS_MAC
  /*
  ** Attempt to find the QLineEdit widget.
  */

  QLineEdit *lineEdit = fileDialog.findChild<QLineEdit *> ("fileNameEdit");

  if(!lineEdit)
    lineEdit = fileDialog.findChild<QLineEdit *> ();
#endif

  if(url.path().isEmpty() || url.path() == "/")
#ifdef Q_WS_MAC
    {
      if(lineEdit)
	lineEdit->setText("index.html");
    }
#else
    fileDialog.selectFile("index.html");
#endif
  else if(url.path().contains("/"))
    {
      QString filePath = url.path();

      filePath = filePath.mid(filePath.lastIndexOf("/") + 1);

#ifdef Q_WS_MAC
      if(lineEdit)
	lineEdit->setText(filePath);
#else
      fileDialog.selectFile(filePath);
#endif
    }

  if(fileDialog.exec() == QDialog::Accepted)
    {
      downloadWindow->show();

      if(!url.scheme().toLower().startsWith("file"))
	downloadWindow->addItem(url, fileDialog.selectedFiles()[0]);
      else
	downloadWindow->addItem(url.toLocalFile(),
				fileDialog.selectedFiles()[0]);
    }
}

void dooble::slotDisplayDownloadWindow(void)
{
  downloadWindow->show();
}

void dooble::slotShowFind(void)
{
  if(ui.tabWidget->
     tabData(ui.tabWidget->currentIndex()).toString() != "Desktop")
    ui.findFrame->setVisible(true);

  if(ui.tabWidget->isVisible())
    {
      ui.findLineEdit->setFocus();
      ui.findLineEdit->selectAll();
    }
}

void dooble::slotHideFind(void)
{
  ui.findFrame->setVisible(false);
}

void dooble::slotFind(void)
{
  slotFind(ui.findLineEdit->text());
}

void dooble::slotFind(const QString &text)
{
  dview *p = qobject_cast<dview *> (ui.tabWidget->currentWidget());
  static QPalette lineEditPalette(ui.findLineEdit->palette());

  if(p)
    {
      QWebPage::FindFlags findFlags = QWebPage::FindWrapsAroundDocument;

      if(ui.matchCaseCheckBox->isChecked())
	findFlags |= QWebPage::FindCaseSensitively;

      if(p->findText(text, findFlags) || text.isEmpty())
	ui.findLineEdit->setPalette(lineEditPalette);
      else
	{
	  QColor color(240, 128, 128); // Light Coral
	  QPalette palette(ui.findLineEdit->palette());

	  palette.setColor(ui.findLineEdit->backgroundRole(), color);
	  ui.findLineEdit->setPalette(palette);
	}
    }
}

void dooble::keyPressEvent(QKeyEvent *event)
{
  if(event->key() == Qt::Key_Escape && ui.findFrame->isVisible())
    ui.findFrame->setVisible(false);
  else if(event->key() == Qt::Key_Escape)
    slotStop();
  else if(event->key() == Qt::Key_X && event->modifiers() == Qt::AltModifier)
    showMinimized();
}

void dooble::slotPrint(void)
{
  QPrinter printer;
  QPrintDialog printDialog(&printer, this);

  if(printDialog.exec() == QDialog::Accepted)
    {
      dview *p = qobject_cast<dview *> (ui.tabWidget->currentWidget());

      if(p)
	p->page()->currentFrame()->print(&printer);
    }
}

void dooble::slotPrintPreview(void)
{
  dview *p = qobject_cast<dview *> (ui.tabWidget->currentWidget());

  if(p)
    {
      QPrinter printer;
      QPrintPreviewDialog printDialog(&printer, this);

      connect(&printDialog,
	      SIGNAL(paintRequested(QPrinter *)),
	      p->page()->currentFrame(),
	      SLOT(print(QPrinter *)));

      if(printDialog.exec() == QDialog::Accepted)
	p->page()->currentFrame()->print(&printer);
    }
}

void dooble::slotShowDesktopTab(void)
{
  bool desktopFound = false;

  for(int i = 0; i < ui.tabWidget->count(); i++)
    if(ui.tabWidget->tabData(i).toString() == "Desktop")
      {
	desktopFound = true;
	ui.tabWidget->setCurrentIndex(i);
	break;
      }

  if(!desktopFound)
    {
      QSettings settings;
      QSettings cfgSettings
	(settings.value("iconSet").toString(), QSettings::IniFormat);
      QGraphicsView *graphicsView = new QGraphicsView(this);

      cfgSettings.beginGroup("mainWindow");
      graphicsView->setScene(desktopWidget);
      ui.tabWidget->insertTab(0,
			      graphicsView,
			      QIcon(cfgSettings.value("tabWidget").toString()),
			      tr("Dooble Desktop"));
      ui.tabWidget->setTabData(0, QVariant("Desktop"));
      ui.tabWidget->setTabToolTip(0, tr("Dooble Desktop"));
      ui.tabWidget->setCurrentWidget(graphicsView);
      ui.desktopToolButton->setVisible(false);
#if QT_VERSION >= 0x040500
      ui.tabWidget->setTabsClosable(ui.tabWidget->count() > 1);
#endif
      ui.actionClose_Tab->setEnabled(ui.tabWidget->count() > 1);
    }
}

void dooble::slotShowPageSource(void)
{
  dview *p = qobject_cast<dview *> (ui.tabWidget->currentWidget());

  if(p)
    {
      pagesourcewindow *ps = new pagesourcewindow
	(p->url(),
	 p->page()->currentFrame()->toHtml());

      connect(this,
	      SIGNAL(iconsChangedSignal(void)),
	      ps,
	      SLOT(slotSetIcons(void)));
    }
}

void dooble::slotClearHistory(bool checked)
{
  Q_UNUSED(checked);

  foreach(QWidget *widget, QApplication::allWidgets())
    if(qobject_cast<dooble *> (widget))
      {
	dooble *d = qobject_cast<dooble *> (widget);

	/*
	** Clear every page's history.
	*/

	emit d->clearHistorySignal();

	/*
	** Remove all but one of the history menu's actions.
	*/

	while(d->ui.historyMenu->actions().size() > 1)
	  d->ui.historyMenu->removeAction(d->ui.historyMenu->actions().last());

	d->ui.historyMenu->actions().at(0)->setEnabled(false);

	/*
	** Clear the history of the back and forward buttons' menus.
	*/

	d->ui.backToolButton->menu()->clear();
	d->ui.forwardToolButton->menu()->clear();
	d->ui.backToolButton->setEnabled(false);
	d->ui.forwardToolButton->setEnabled(false);
      }
}

void dooble::slotShowSettingsWindow(void)
{
  settingsWindow->exec(this);
}

#if QT_VERSION >= 0x040500

void dooble::slotViewZoomIn(void)
{
  dview *p = qobject_cast<dview *> (ui.tabWidget->currentWidget());

  if(p && p->page() && p->page()->currentFrame())
    {
      p->page()->currentFrame()->setZoomFactor
	(p->page()->currentFrame()->zoomFactor() + 0.1);
      update();
    }
}

void dooble::slotViewZoomOut(void)
{
  dview *p = qobject_cast<dview *> (ui.tabWidget->currentWidget());
	
  if(p && p->page() && p->page()->currentFrame())
    {
      p->page()->currentFrame()->setZoomFactor
	(p->page()->currentFrame()->zoomFactor() - 0.1);
      update();
    }
}

void dooble::slotViewResetZoom(void)
{
  dview *p = qobject_cast<dview *> (ui.tabWidget->currentWidget());

  if(p && p->page()->currentFrame())
    {
      p->page()->currentFrame()->setZoomFactor(1.0);
      update();
    }
}

void dooble::slotViewZoomTextOnly(bool enable)
{
  dview *p = qobject_cast<dview *> (ui.tabWidget->currentWidget());

  if(p && p->page()->currentFrame())
    {
      p->page()->currentFrame()->page()->settings()->setAttribute
	(QWebSettings::ZoomTextOnly, enable);
      update();
    }
}

#endif

void dooble::slotStatusBarDisplay(void)
{
  statusBar()->setVisible(ui.actionStatusbar->isChecked());

  QSettings settings;

  settings.setValue("mainWindow/statusbarDisplay", statusBar()->isVisible());
}

void dooble::closeDesktop(void)
{
  /*
  ** Destroy the widget that is housing the Desktop.
  */

  int count = ui.tabWidget->count();

  for(int i = 0; i < count; i++)
    if(ui.tabWidget->tabData(i).toString() == "Desktop")
      {
	QWidget *widget = qobject_cast<QWidget *> (ui.tabWidget->widget(i));

	if(widget)
	  {
	    count -= 1;
	    widget->deleteLater();
	    ui.desktopToolButton->setVisible(true);
	  }

	break;
      }

#if QT_VERSION >= 0x040500
  ui.tabWidget->setTabsClosable(count > 1);
#endif
  ui.actionClose_Tab->setEnabled(count > 1);
}
