// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifndef DYN_EXPLICIT_AUTOMATON_CC
#define DYN_EXPLICIT_AUTOMATON_CC

#include <awali/dyn/core/automaton.hh>
#include <awali/dyn/bridge-sttc/explicit-context.cc>
#include <awali/sttc/core/mutable-automaton.hh>
#include <awali/sttc/core/transpose-view.hh>
#include <awali/sttc/ctx/lan.hh>
#include <awali/sttc/algos/js_print.hh>
#include <awali/sttc/misc/add-epsilon-trans.hh>
#include<sstream>
#include<stdexcept>

namespace awali { namespace dyn {

  template <typename Context>
  struct explicit_automaton_t : public dyn::abstract_automaton_t {

      sttc::mutable_automaton<Context> aut_;

      explicit_automaton_t(sttc::mutable_automaton<Context> aut) :
        aut_(aut) {}

  #define GEN_METH(TYPERET, NAME,CONST)		\
      TYPERET NAME() CONST {			\
        return aut_->NAME();			\
      }
      
      state_t add_state(std::string s) {
        //auto aut_->history();
        state_t i= aut_->add_state();
        aut_->set_state_name(i,s);
        return i;
      }

      GEN_METH(size_t, num_states, const)
      GEN_METH(size_t, num_initials, const)
      GEN_METH(size_t, num_finals, const)
      GEN_METH(size_t, num_transitions, const)
      GEN_METH(unsigned, add_state, )
      GEN_METH(unsigned, pre, const)
      GEN_METH(unsigned, post, const)
  #undef GEN_METH

      bool is_final(unsigned s) const {
        return !outin(s,post()).empty();
      }

      void unset_final(unsigned s) {
        for(auto tr : outin(s,post()))
          aut_->del_transition(tr);
      }

  #define GEN_METH(TYPERET, NAME, T1, CONST)	\
      TYPERET NAME(T1 x) CONST {			\
        return aut_->NAME(x);			\
      }
      GEN_METH(bool, has_state, unsigned, const)
      GEN_METH(bool, is_initial, unsigned, const)
      GEN_METH(dyn::any_t, get_initial_weight, unsigned, const)
      GEN_METH(dyn::any_t, get_final_weight, unsigned, const)
      GEN_METH(bool, has_transition, unsigned, const)
      GEN_METH(unsigned, src_of, unsigned, const)
      GEN_METH(unsigned, dst_of, unsigned, const)
      GEN_METH(dyn::any_t, label_of, unsigned, const)
      GEN_METH(dyn::any_t, weight_of, unsigned, const)
      GEN_METH(unsigned, get_state_by_name, std::string, const)
  #undef GEN_METH

  #define GEN_METH(NAME, T1, CONST)		\
      void NAME(T1 x) CONST {			\
        aut_->NAME(x);				\
      }
      GEN_METH(del_state, unsigned, )
      GEN_METH(set_initial, unsigned, )
      GEN_METH(unset_initial, unsigned, )
      GEN_METH(set_final, unsigned, )
      GEN_METH(del_transition, unsigned, )
  #undef GEN_METH

      unsigned get_transition(unsigned src, unsigned dst, dyn::any_t label) const {
        return aut_->get_transition(src, dst, label);
      }

      bool has_transition(unsigned src, unsigned dst, dyn::any_t label) const {
        return aut_->has_transition(src, dst, label);
      }

      void strip_history() {
        aut_->strip_history();
      }

    bool is_eps_transition(unsigned tr) const {
    return sttc::is_epsilon<typename Context::labelset_t>(aut_->label_of(tr));
    }

      std::string get_state_name(unsigned s) const {
        return aut_->get_state_name(s);
      }

      void set_state_name(unsigned s, std::string name) {
	aut_->set_state_name(s, name);
      }

      void set_initial(unsigned s, dyn::any_t w) {
        aut_->set_initial(s, w);
      }

      dyn::any_t add_initial(unsigned s, dyn::any_t w) {
        return aut_->add_initial(s, w);
      }
      void set_final(unsigned s, dyn::any_t w) {
        aut_->set_final(s, w);
      }
      dyn::any_t add_final(unsigned s, dyn::any_t w) {
        return aut_->add_final(s, w);
      }

      transition_t del_transition(unsigned src, unsigned dst, dyn::any_t label) {
        return aut_->del_transition(src, dst, label);
      }

      void del_transition(unsigned src, unsigned dst) {
        aut_->del_transition(src, dst);
      }

      void del_eps_transition(unsigned src, unsigned dst) {
        sttc::del_epsilon_trans(aut_, src, dst);
      }

      unsigned set_transition(unsigned src, unsigned dst, dyn::any_t label, dyn::any_t weight) {
        return aut_->set_transition(src, dst, label, weight);
      }

      unsigned set_transition(unsigned src, unsigned dst, dyn::any_t label) {
        return aut_->set_transition(src, dst, label);
      }

      unsigned set_eps_transition(unsigned src, unsigned dst, dyn::any_t weight) {
        return sttc::set_epsilon_trans(aut_, src, dst, weight) ;
      }

      unsigned set_eps_transition(unsigned src, unsigned dst) {
        return sttc::set_epsilon_trans(aut_, src, dst);
      }

      dyn::any_t add_transition(unsigned src, unsigned dst, dyn::any_t label, dyn::any_t weight) {
        return aut_->add_transition(src, dst, label, weight);
      }
      dyn::any_t add_transition(unsigned src, unsigned dst, dyn::any_t label) {
        return aut_->add_transition(src, dst, label);
      }

      dyn::any_t add_eps_transition(unsigned src, unsigned dst, dyn::any_t weight) {
        return sttc::add_epsilon_trans(aut_,src, dst, weight);
      }
      dyn::any_t add_eps_transition(unsigned src, unsigned dst) {
        return sttc::add_epsilon_trans(aut_,src, dst);
      }
      dyn::any_t set_weight(unsigned tr, dyn::any_t weight) {
        return aut_->set_weight(tr, weight);
      }
      dyn::any_t add_weight(unsigned tr, dyn::any_t weight) {
        return aut_->add_weight(tr, weight);
      }
      dyn::any_t lmul_weight(unsigned tr, dyn::any_t weight) {
        return aut_->lmul_weight(tr, weight);
      }
      dyn::any_t rmul_weight(unsigned tr, dyn::any_t weight) {
        return aut_->rmul_weight(tr, weight);
      }
  #define GEN_METH(NAME)				\
      std::vector<unsigned> NAME() const {	\
        std::vector<unsigned> res;		\
        for(unsigned i : aut_->NAME())		\
          res.emplace_back(i);			\
        return res;				\
      }
      GEN_METH(states)
      GEN_METH(all_states)
      GEN_METH(transitions)
      GEN_METH(all_transitions)
      GEN_METH(initial_transitions)
      GEN_METH(final_transitions)
  #undef GEN_METH
  #define GEN_METH(NAME)					\
      std::vector<unsigned> NAME(unsigned s) const {	\
        std::vector<unsigned> res;			\
        for(unsigned i : aut_->NAME(s))			\
          res.emplace_back(i);				\
        return res;					\
      }
      GEN_METH(out)
      GEN_METH(all_out)
      GEN_METH(in)
      GEN_METH(all_in)
  #undef GEN_METH

      std::vector<unsigned> outgoing(unsigned s) const {
        std::vector<unsigned> res;
        for(unsigned i : aut_->out(s))
          res.emplace_back(i);
        return res;
      }

      std::vector<unsigned> incoming(unsigned s) const {
        std::vector<unsigned> res;
        for(unsigned i : aut_->in(s))
          res.emplace_back(i);
        return res;
      }

      std::vector<unsigned> initial_states() const {
        std::vector<unsigned> res;
        for(unsigned i : aut_->initial_transitions())
          res.emplace_back(dst_of(i));
        return res;
      }

      std::vector<unsigned> final_states() const {
        std::vector<unsigned> res;
        for(unsigned i : aut_->final_transitions())
          res.emplace_back(src_of(i));
        return res;
      }

      std::vector<unsigned> successors(unsigned s) const {
        std::vector<unsigned> res;
        for(unsigned i : aut_->out(s))
          res.emplace_back(aut_->dst_of(i));
        return res;
      }

      std::vector<unsigned> successors(unsigned s, dyn::any_t label) const {
        std::vector<unsigned> res;
        for(unsigned i : aut_->out(s,label))
          res.emplace_back(aut_->dst_of(i));
        return res;
      }

      std::vector<unsigned> predecessors(unsigned s) const {
        std::vector<unsigned> res;
        for(unsigned i : aut_->in(s))
          res.emplace_back(aut_->src_of(i));
        return res;
      }

      std::vector<unsigned> predecessors(unsigned s, dyn::any_t label) const {
        std::vector<unsigned> res;
        for(unsigned i : aut_->in(s,label))
          res.emplace_back(aut_->src_of(i));
        return res;
      }

      std::vector<unsigned> out(unsigned s, dyn::any_t label) const {
        std::vector<unsigned> res;
        for(unsigned i : aut_->out(s, label))
          res.emplace_back(i);
        return res;
      }

      std::vector<unsigned> outgoing(unsigned s, dyn::any_t label) const {
        std::vector<unsigned> res;
        for(unsigned i : aut_->out(s, label))
          res.emplace_back(i);
        return res;
      }

      std::vector<unsigned> in(unsigned s, dyn::any_t label) const {
        std::vector<unsigned> res;
        for(unsigned i : aut_->in(s, label))
          res.emplace_back(i);
        return res;
      }

      std::vector<unsigned> incoming(unsigned s, dyn::any_t label) const {
        std::vector<unsigned> res;
        for(unsigned i : aut_->in(s, label))
          res.emplace_back(i);
        return res;
      }

      std::vector<unsigned> outin(unsigned s, unsigned d) const {
        std::vector<unsigned> res;
        for(unsigned i : aut_->outin(s, d))
          res.emplace_back(i);
        return res;
      }

      std::vector<dyn::any_t> alphabet() const {
        std::vector<dyn::any_t> res;
        for(auto l : aut_->labelset()->genset())
          res.emplace_back(l);
        return res;
      }

    private:
      template<typename T>
      bool is_transducer(const T&) const {
        return false;
      }

      template<typename... T>
      bool is_transducer(const sttc::tupleset<T...>& lab) const {
        return true;
      }

    public:
      bool is_transducer() const {
        return is_transducer(*(aut_->context().labelset()));
      }

      bool allow_eps_transition() const {
    return Context::labelset_t::has_one();
      }

      std::ostream& json(std::ostream& o) const {
        return sttc::js_print(aut_, o, true);
      }

      dyn::context_t get_context() const {
        return std::make_shared<dyn::explicit_context_t<Context>>(aut_->context());
      }

      sttc::mutable_automaton<Context> get_automaton() {
        return aut_;
      }
    };


  template<typename Context>
  sttc::mutable_automaton<Context>
  get_stc_automaton(dyn::automaton_t aut) {
    return dynamic_cast<explicit_automaton_t<Context>&>(*aut).get_automaton();
  }

  template<typename Context>
  dyn::automaton_t make_automaton(sttc::mutable_automaton<Context> a) {
    return dyn::automaton_t(std::make_shared<explicit_automaton_t<Context>>(a));
  }

  template<typename Aut>
  dyn::automaton_t make_automaton(std::shared_ptr<sttc::internal::transpose_view_impl<Aut>> a) {
    throw std::runtime_error("No dynamic encapsulation of transpose view");
  }

}}//end of ns awali::dyn


#endif
