// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

/** @file accessible.hh
 *
 * This files contains dynamic functions computing accessible states.
 */
#ifndef DYN_MODULES_ACCESSIBLE_HH
#define DYN_MODULES_ACCESSIBLE_HH

#include <set>

#include <awali/dyn/core/automaton.hh>
#include <awali/dyn/loading/handler.hh>

namespace awali {
  namespace dyn {

  /** @brief List of accessible states
   *
   * Computes the list of accessible states in aut.
   *
   * @param aut  A dynamic Awali automaton
   *
   * @return a standard set
   **/
    std::set<state_t> accessible_states(automaton_t aut);

  /** @brief List of coaccessible states
   *
   * Computes the list of coaccessible states in aut.
   *
   * @param aut  A dynamic Awali automaton
   *
   * @return a standard set
   **/    
    std::set<state_t> coaccessible_states(automaton_t aut);

    /** @brief List of useful states
     *
     * Computes the list of useful states in aut.
     *
     * @param aut  A dynamic Awali automaton
     *
     * @return a standard set
     **/
    std::set<state_t> useful_states(automaton_t aut);

    /** @brief Number of accessible states
     *
     * Computes the number of accessible states.
     * @param aut  A dynamic Awali automaton
     *
     * @return a size_t number
     **/
    size_t num_accessible_states(automaton_t aut);

    /** @brief Number of coaccessible states
     *
     * Computes the number of coaccessible states.
     * @param aut  A dynamic Awali automaton
     *
     * @return a size_t number
     **/
    size_t num_coaccessible_states(automaton_t aut);

    /** @brief Accessible subautomaton
     * 
     * Computes a fresh automaton with a copy of the accessible part of aut.
     * If the keep_history flag is true, the result is endowed with
     * a SINGLE history.
     *
     * @param aut  A dynamic Awali automaton
     * @param keep_history A Boolean that tells whether the history must be registered.
     *
     * @return A dynamic Awali automaton
     */
    automaton_t accessible(automaton_t aut, bool keep_history=true);

    /** @brief Coaccessible subautomaton
     * 
     * Computes a fresh automaton with a copy of the coaccessible part of aut.
     * If the keep_history flag is true, the result is endowed with
     * a SINGLE history.
     *
     * @param aut  A dynamic Awali automaton
     * @param keep_history A Boolean that tells whether the history must be registered.
     *
     * @return A dynamic Awali automaton
     */
    automaton_t coaccessible(automaton_t aut, bool keep_history=true);

    /** @brief Trim subautomaton
     * 
     * Computes a fresh automaton with a copy of the useful part of aut.
     * If the keep_history flag is true, the result is endowed with
     * a SINGLE history.
     *
     * @param aut  A dynamic Awali automaton
     * @param keep_history A Boolean that tells whether the history must be registered.
     *
     * @return A dynamic Awali automaton
     */
    automaton_t trim(automaton_t aut, bool keep_history=true);

    /** @brief In-place accessible subautomaton
     * 
     * Remove every non accessible state of aut.
     *
     * @param aut  A dynamic Awali automaton
     */
    void accessible_here(automaton_t aut);

    /** @brief In-place coaccessible subautomaton
     * 
     * Remove every non coaccessible state of aut.
     *
     * @param aut  A dynamic Awali automaton
     */
    void coaccessible_here(automaton_t aut);

    /** @brief In-place trim subautomaton
     * 
     * Remove every useless state of aut.
     *
     * @param aut  A dynamic Awali automaton
     */
    void trim_here(automaton_t aut);


    /** @brief Test whether the automaton is trim
     * 
     * @param aut  A dynamic Awali automaton
     *
     * @return true if the automaton is trim; false otherwise.
     */
    bool is_trim(automaton_t aut);

    /** @brief Test whether the automaton has useful states
     * 
     * @param aut  A dynamic Awali automaton
     *
     * @return true if the automaton has no useful state; false otherwise.
     */
    bool is_useless(automaton_t aut);

    /** @brief Test whether every state of the automaton is accessible
     * 
     * @param aut  A dynamic Awali automaton
     *
     * @return true if every state is accessible; false otherwise.
     */
    bool is_accessible(automaton_t aut);

    /** @brief Test whether every state of the automaton is coaccessible
     * 
     * @param aut  A dynamic Awali automaton
     *
     * @return true if every state is coaccessible; false otherwise.
     */
    bool is_coaccessible(automaton_t aut);

    /** @brief Test whether the automaton has no state
     * 
     * @param aut  A dynamic Awali automaton
     *
     * @return true if the automaton has no state; false otherwise.
     */
    bool is_empty(automaton_t aut);

  }
}//end of ns awali::dyn

#endif
