// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifndef DYN_MODULES_CONTEXT_CC
#define DYN_MODULES_CONTEXT_CC

#include <awali/dyn/modules/context.hh>


namespace awali { namespace dyn {
  automaton_t make_automaton_from_context(context_t ctx) {
    std::string stat_ctx = ctx->sname();
    typedef automaton_t (*bridge_t)(context_t);
    static std::unordered_map<std::string, bridge_t> bridges;
    auto it = bridges.find(stat_ctx);
    if(it == bridges.end()) {
      auto bridge = (bridge_t) loading::get_handler("make_automaton_from_context", "context", stat_ctx);
      bridges.emplace(stat_ctx, bridge);
      return bridge(ctx);
    }
    else
      return it->second(ctx);
  }

  automaton_t make_automaton_from_context(context_description cd) {
    std::string stat_ctx = tostring(cd, false);
    typedef automaton_t (*bridge_t)(context_description);
    static std::unordered_map<std::string, bridge_t> bridges;
    auto it = bridges.find(stat_ctx);
    if(it == bridges.end()) {
      auto bridge = (bridge_t) loading::get_handler("make_automaton_from_desc", "context", stat_ctx);
      bridges.emplace(stat_ctx, bridge);
      return bridge(cd);
    }
    else
      return it->second(cd);
  }

  automaton_t make_automaton_from_context(labelset_description ls, weightset_description ws) {
    return make_automaton_from_context(c_desc(ls, ws));
  }

  context_t make_context(context_description cd) {
    std::string stat_ctx = tostring(cd, false);
    typedef context_t (*bridge_t)(context_description);
    static std::unordered_map<std::string, bridge_t> bridges;
    auto it = bridges.find(stat_ctx);
    if(it == bridges.end()) {
      auto bridge = (bridge_t) loading::get_handler("make_context_from_desc", "context", stat_ctx);
      bridges.emplace(stat_ctx, bridge);
      return bridge(cd);
    }
    else
      return it->second(cd);
  }

  automaton_t make_automaton(std::string alphabet, std::string semiring) {
    return make_automaton_from_context(letterset(alphabet),weightset(semiring));
  }

    automaton_t make_int_automaton(int a, int b, std::string semiring) {
      return make_automaton_from_context(intletterset(a,b),weightset(semiring));
    }

    automaton_t make_int_automaton(int n, std::string semiring) {
      return make_automaton_from_context(intletterset(n),weightset(semiring));
    }

  automaton_t make_automaton(std::string alphabet) {
    return make_automaton(alphabet, "B");
  }

  automaton_t make_automaton_with_eps(std::string alphabet, std::string semiring) {
    return make_automaton_from_context(nullableset(letterset(alphabet)),weightset(semiring));
  }

    automaton_t make_int_automaton_with_eps(int a, int b, std::string semiring) {
      return make_automaton_from_context(nullableset(intletterset(a,b)),weightset(semiring));
    }

    automaton_t make_int_automaton_with_eps(int n, std::string semiring) {
      return make_automaton_from_context(nullableset(intletterset(n)),weightset(semiring));
    }

  automaton_t make_automaton_with_eps(std::string alphabet) {
    return make_automaton_with_eps(alphabet, "B");
  }

    automaton_t parse_automaton(std::istream& i) {
    std::string aut = sttc::get_first_attr(i);
    while(aut != "Automaton") {
      sttc::parsestring(i);
      aut=sttc::parsestring(i);
      sttc::check(i, ':');
    }
    sttc::check(i, '[');
    dyn::context_description ct= dyn::parse_context(i);
    sttc::check(i, ',');
    std::string stat_ctx = tostring(ct,false);
    typedef automaton_t (*bridge_t)(std::istream&, dyn::context_description);
    static std::unordered_map<std::string, bridge_t> bridges;
    auto it = bridges.find(stat_ctx);
    automaton_t res;
    if(it == bridges.end()) {
      auto bridge = (bridge_t) loading::get_handler("parse_automaton", "context", stat_ctx);
      bridges.emplace(stat_ctx, bridge);
      res= bridge(i, ct);
    }
    else
      res = it->second(i, ct);
    sttc::check(i,']');
    sttc::check(i,'}');
    return res;
  }



  ratexp_t make_ratexp_with_context(const std::string& exp, context_description cd) {
    std::string stat_ctx = tostring(cd, false);
    typedef ratexp_t (*bridge_t)(const char*, context_description);
    static std::unordered_map<std::string, bridge_t> bridges;
    auto it = bridges.find(stat_ctx);
    if(it == bridges.end()) {
      auto bridge = (bridge_t) loading::get_handler("make_ratexp_with_desc", "context", stat_ctx);
      bridges.emplace(stat_ctx, bridge);
      return bridge(exp.c_str(), cd);
    }
    else
      return it->second(exp.c_str(), cd);
  }

  ratexp_t make_ratexp_with_context(const std::string& exp, labelset_description ls, weightset_description ws) {
    return make_ratexp_with_context(exp, c_desc(ls, ws));
  }

  ratexp_t make_ratexp_with_alphabet(const std::string& exp, const std::string& alph, const std::string& sr) {
    return make_ratexp_with_context(exp, letterset(alph),weightset(sr));
  }

  ratexp_t make_ratexp_with_alphabet(const std::string& exp, const std::string& alph) {
    return make_ratexp_with_alphabet(exp, alph, "B");
  }

    ratexp_t make_int_ratexp_with_range(const std::string& exp, int a, int b, const std::string& sr) {
    return make_ratexp_with_context(exp, intletterset(a,b),weightset(sr));
  }

    ratexp_t make_int_ratexp_with_range(const std::string& exp, int a, int b) {
      return make_int_ratexp_with_range(exp, a, b, "B");
    }

  ratexp_t make_ratexp(const std::string& exp, const std::string& sr) {
    std::set<char> alph;
    unsigned inib=0;
    for(size_t i=0; i<exp.length(); ++i) {
      if(exp[i]=='<' || exp[i]=='{')
        ++inib;
        else if(exp[i]=='>' || exp[i]=='}')
          --inib;
        else if(inib==0)
          if(exp[i]!='(' && exp[i]!=')' && exp[i]!='.' && exp[i]!='+' && exp[i]!='*' && exp[i]!='?') {
            if ((exp[i] != '\\') || (exp[i+1] != 'e'))
              alph.emplace(exp[i]);
            else
              i++;
          }
    }
    std::string a="";
    for(char c : alph)
      a=a+c;
    return make_ratexp_with_alphabet(exp, a, sr);
  }

  ratexp_t make_ratexp(const std::string& exp) {
    return make_ratexp(exp, "B");
  }



  ratexp_t parse_ratexp(std::istream& i) {
    std::string aut = sttc::get_first_attr(i);
    if(aut != "Rational Expression")
      throw std::runtime_error("json: Rational Expression");
    sttc::check(i, '[');
    context_description ct= dyn::parse_context(i);
    sttc::check(i, ',');
    std::string stat_ctx = tostring(ct,false);
    typedef ratexp_t (*bridge_t)(context_description, std::istream&);
    static std::unordered_map<std::string, bridge_t> bridges;
    auto it = bridges.find(stat_ctx);
    ratexp_t res;
    if(it == bridges.end()) {
      auto bridge = (bridge_t) loading::get_handler("parse_ratexp", "context", stat_ctx);
      bridges.emplace(stat_ctx, bridge);
      res= bridge(ct, i);
    }
    else
      res = it->second(ct, i);
    sttc::check(i,']');
    sttc::check(i,'}');
    return res;
  }
}}//end of ns awali::dyn

#endif
