// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifndef DYN_MODULES_OUTPUT_CC
#define DYN_MODULES_OUTPUT_CC

#include <map>
#include <fstream>
#include <awali/dyn/core/automaton.hh>
#include <awali/dyn/modules/automaton.hh>
#include <awali/dyn/loading/handler.hh>

#include <awali/dyn/modules/context.hh>

#include <awali/dyn/modules/output.hh>

namespace awali { namespace dyn {

  const std::vector<const io*>& io::instances() {
    static std::vector<const io*> v { 
        &io::dot, &io::fado, &io::grail, &io::json, &io::pdf };
    return v;
  }

  std::ostream&
  daut(automaton_t aut, std::ostream& out) {
    return loading::call2<std::ostream&, std::ostream&, &daut>("daut", "output", aut, out);
  }

  std::ostream&
  dot(automaton_t aut, std::ostream& out, bool history, bool horizontal) {
    return loading::call4<std::ostream&, std::ostream&, bool, bool, &dot>(
									  "dot", "output", aut, out, history, horizontal);
  }

  std::ostream&
  fado(automaton_t aut, std::ostream& out) {
    return loading::call2<std::ostream&, std::ostream&, &fado>("fado", "output", aut, out);
  }

  std::ostream&
  grail(automaton_t aut, std::ostream& out) {
    return loading::call2<std::ostream&, std::ostream&, &grail>("grail", "output", aut, out);
  }

  std::ostream&
  efsm(automaton_t aut, std::ostream& out) {
    return loading::call2<std::ostream&, std::ostream&, &efsm>("efsm", "output", aut, out);
  }

  std::ostream&
  json(automaton_t aut, std::ostream& out) {
    return loading::call2<std::ostream&, std::ostream&, &json>("json", "output", aut, out);
  }


  std::ostream& pdf(automaton_t aut, std::ostream& o, bool history, bool horizontal) {
    std::string filename= "/tmp/tmp.gv";
    std::ofstream of(filename);
    dot(aut, of, history, horizontal) << std::endl;
    size_t buffersize = 1024;
    char buffer[buffersize];
    std::shared_ptr<FILE> pipe(popen("dot -Tpdf /tmp/tmp.gv", "r"), pclose);
    if (!pipe) throw std::runtime_error("popen() failed!");
    while (true) { 
      size_t n = fread(buffer, sizeof(char), buffersize, pipe.get());
      if (n == 0)
        break;
      o.write(buffer, n);
    }
    return o;
  }

  std::ostream& svg(automaton_t aut, std::ostream& o, bool history, bool horizontal) {
    std::string filename= "/tmp/tmp.gv";
    std::ofstream of(filename);
    dot(aut, of, history, horizontal) << std::endl;
    size_t buffersize = 1024;
    char buffer[buffersize];
    std::shared_ptr<FILE> pipe(popen("dot -Tsvg /tmp/tmp.gv", "r"), pclose);
    if (!pipe) throw std::runtime_error("popen() failed!");
    while (true) { 
      size_t n = fread(buffer, sizeof(char), buffersize, pipe.get());
      if (n == 0)
        break;
      o.write(buffer, n);
    }
    return o;
  }

  std::ostream& operator<<(std::ostream& o, automaton_t aut) {
    return json(aut, o);
  }

  autostream operator<<(std::ostream& o, const io& m) {
    return autostream(m.v, o);
  }

  automaton_t fado(std::istream& in) {
    char buffer[512];
    dyn::automaton_t aut = dyn::make_automaton("");
    std::unordered_map<std::string, unsigned> states;

    auto get_state = [&](const std::string& s) -> unsigned {
      unsigned i;
      auto it = states.find(s);
      if(it == states.end()) {
	i = aut->add_state(s);
	states.emplace(std::make_pair(s, i));
	return i;
      }
      return it->second;
    };

    in.getline(buffer,512);
    std::istringstream is(buffer);
    std::string token;
    std::vector<std::string> final_states;
    std::vector<std::string> initial_states;
    is >> token;
    if(token != "@NFA" && token != "@DFA" && token != "@NDFA")
      throw std::runtime_error("Fado format");
    while(!is.eof()) {
      is >> token;
      if(token =="*")
	break;
      final_states.emplace_back(token);
    }
    while(!is.eof()) {
      is >> token;
      initial_states.emplace_back(token);
    }
    bool first=true;
    while(!in.eof()) {
      in.getline(buffer,512);
      if(!buffer[0])
	continue;
      is.str(buffer);
      is.seekg(std::ios_base::beg);
      std::string st1, st2, label;
      is >> st1;

      if(first) {
	first=false;
	if(initial_states.empty())
	  aut->set_initial(get_state(st1));
      }

      is >> label;
      aut->get_context()->add_letter(label[0]);

      is >> st2;

      aut->set_transition(get_state(st1), get_state(st2), label[0]);
    }
    for(auto s : initial_states)
      aut->set_initial(get_state(s));
    for(auto s : final_states)
      aut->set_final(get_state(s));
    return aut;
  }


  automaton_t grail(std::istream& in) {
    char buffer[512];
    dyn::automaton_t aut = dyn::make_automaton("");
    std::unordered_map<std::string, unsigned> states;

    auto get_state = [&](const std::string& s) -> unsigned {
      unsigned i;
      auto it = states.find(s);
      if(it == states.end()) {
	i = aut->add_state(s);
	states.emplace(std::make_pair(s, i));
	return i;
      }
      return it->second;
    };

    std::string s1, s2, label;
    std::istringstream is;

    while(!in.eof()) {
      in.getline(buffer,512);
      if(!buffer[0])
	continue;
      is.str(buffer);
      is.seekg(std::ios_base::beg);
      is >> s1;
      is >> label;
      is >> s2;
      if(label == "|-")
	aut->set_initial(get_state(s2));
      else if(label == "-|")
	aut->set_final(get_state(s1));
      else {
	aut->get_context()->add_letter(label[0]);
	aut->set_transition(get_state(s1), get_state(s2), label[0]);
      }
    }
    return aut;
  }




  std::istream& operator>>(std::istream& i, automaton_t& aut) {
    aut=parse_automaton(i);
    return i;
  }

  autistream operator>>(std::istream& i, const io& m) {
    return autistream(m.v, i);
  }


  std::ostream& autostream::operator<<(automaton_t aut) {
    switch(v) {
      case 0: return dot(aut, o);
      case 1: return fado(aut, o);
      case 2: return grail(aut, o);
      case 3: return json(aut, o);
      case 4: return pdf(aut, o);
      case 5: return svg(aut, o);
      default : return o;
    }
  }

  std::istream& autistream::operator>>(automaton_t& aut) {
    switch(v) {
      case 1: aut=fado(i); return i;
      case 2: aut=grail(i); return i;
      case 3: aut=parse_automaton(i); return i;
      default : throw std::runtime_error("Format does not allow input.");
    }
  }

}}//end of ns awali::dyn

#endif
