// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifndef AWALI_ALGOS_PREFIX_HH
# define AWALI_ALGOS_PREFIX_HH

#include <awali/sttc/algos/accessible.hh>

namespace awali { namespace sttc {

  /*---------.
  | suffix.  |
  `---------*/

  /// Make all accessible states initial.
  template <typename Aut>
  Aut&
  suffix_here(Aut& aut)
  {
    for (auto s : accessible_states(aut))
      if (s != aut->pre()
          && s != aut->post()
          && !aut->is_initial(s))
        aut->set_initial(s);
    return aut;
  }

  template <typename Aut>
  auto
  suffix(const Aut& aut)
    ->decltype(::sttc::copy(aut))
  {
    auto res = sttc::copy(aut);
    suffix_here(res);
    return res;
  }

  /*---------.
  | prefix.  |
  `---------*/

  /// Make all coaccessible states final.
  template <typename Aut>
  Aut&
  prefix_here(Aut& aut)
  {
    auto t = transpose(aut);
    suffix_here(t);
    return aut;
  }

  template <typename Aut>
  auto
  prefix(const Aut& aut)
    ->decltype(::sttc::copy(aut))
  {
    auto res = sttc::copy(aut);
    prefix_here(res);
    return res;
  }

  /*---------.
  | factor.  |
  `---------*/

  /// Make each useful state both initial and final.
  template <typename Aut>
  Aut&
  factor_here(Aut& aut)
  {
    for (auto s : useful_states(aut))
      if (s != aut->pre() && s != aut->post())
	{
	  if (!aut->is_initial(s))
	    aut->set_initial(s);
	  if (!aut->is_final(s))
	    aut->set_final(s);
	}
    return aut;
  }

  template <typename Aut>
  auto
  factor(const Aut& aut)
    ->decltype(::sttc::copy(aut))
  {
    auto res = sttc::copy(aut);
    factor_here(res);
    return res;
  }

  /*----------.
  | subword.  |
  `----------*/

  /// Add spontaneous transitions for each non spontaneous transition,
  /// with same source, destination and weight.
  template <typename Aut>
  Aut&
  subword_here(Aut& aut)
  {
    const auto one = aut->labelset()->one();

    std::vector<transition_t> ts;
    for (auto s : aut->states())
      {
        ts.clear();
        for (auto t : aut->out(s))
          if (!aut->labelset()->is_one(aut->label_of(t)))
            ts.emplace_back(t);
        for (auto t : ts)
          aut->add_transition(s, aut->dst_of(t), one, aut->weight_of(t));
      }

    return aut;
  }

  /// Apply subword_here() to a copy of \a aut.
  template <typename Aut>
  auto
  subword(const Aut& aut)
    ->decltype(::sttc::copy(aut))
  {
    auto res = sttc::copy(aut);
    subword_here(res);
    return res;
  }

}}//end of ns awali::stc

#endif // !AWALI_ALGOS_PREFIX_HH
