// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifndef AWALI_ALGOS_U_HH
# define AWALI_ALGOS_U_HH

#include <awali/sttc/ctx/traits.hh>
#include <awali/sttc/alphabets/char.hh>
#include <awali/sttc/alphabets/setalpha.hh>
#include <awali/sttc/core/mutable-automaton.hh>
#include <awali/sttc/misc/raise.hh>

namespace awali { namespace sttc {

  /// The Brzozowski universal witness.
  template <typename Context>
  mutable_automaton<Context>
  u(const Context& ctx, unsigned n)
  {
    using context_t = Context;
    using automaton_t = mutable_automaton<context_t>;
    require(2 <= n, "u: n must be at least 3");
    const auto& gens = ctx.labelset()->genset();
    std::vector<typename context_t::labelset_t::letter_t> letters
      {std::begin(gens), std::end(gens)};
    require(3 <= letters.size(), "u: the alphabet needs at least 3 letters");
    automaton_t res = make_shared_ptr<automaton_t>(ctx);

    // The states.
    std::vector<state_t> states;
    for (unsigned i = 0; i < n; ++i)
      states.push_back(res->add_state());
    res->set_initial(states[0]);
    res->set_final(states[n-1]);

    // The 'a' transitions.
    auto a = letters[0];
    for (unsigned i = 0; i < n; ++i)
      res->new_transition(states[i], states[(i+1) % n], a);

    // The 'b' transitions.
    auto b = letters[1];
    res->new_transition(states[0], states[1], b);
    res->new_transition(states[1], states[0], b);
    for (unsigned i = 2; i < n; ++i)
      res->new_transition(states[i], states[i], b);

    // The 'c' transitions.
    auto c = letters[2];
    for (unsigned i = 0; i < n - 1; ++i)
      res->new_transition(states[i], states[i], c);
    res->new_transition(states[n - 1], states[0], c);

    return res;
  }

}}//end of ns awali::stc

#endif // !AWALI_ALGOS_U_HH
