// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifndef AWALI_LABELSET_WORDSET_HH
# define AWALI_LABELSET_WORDSET_HH

# include <memory>
# include <set>

#include <awali/sttc/core/kind.hh>
#include <awali/sttc/labelset/genset-labelset.hh>
#include <awali/sttc/labelset/labelset.hh>
#include <awali/sttc/misc/attributes.hh>
#include <awali/utils/hash.hh>
#include <awali/sttc/misc/raise.hh>

namespace awali { namespace sttc {

  /// Implementation of labels are words.
  template <typename GenSet>
  class wordset: public internal::genset_labelset<GenSet>
  {
  public:
    using genset_t = GenSet;
    using super_type = internal::genset_labelset<genset_t>;
    using self_type = wordset;
    using genset_ptr = std::shared_ptr<const genset_t>;

    using letter_t = typename genset_t::letter_t;
    using word_t = typename genset_t::word_t;
    using letters_t = std::set<letter_t>;

    using value_t = word_t;

    using kind_t = labels_are_words;

    wordset(const genset_ptr& gs)
      : super_type{gs}
    {}

    wordset(const genset_t& gs = {})
      : wordset{std::make_shared<const genset_t>(gs)}
    {}

    const super_type&
    super() const
    {
      return static_cast<const super_type&>(*this);
    }

    static std::string sname()
    {
      return "law_" + super_type::sname();
    }

    std::string vname(bool full = true) const
    {
      return "law_" + super().vname(full);
    }

    /// Build from the description in \a is.
    static wordset make(std::istream& is)
    {
      // name: law_char(abc).
      //       ^^^ ^^^^^^^^^
      //      kind   genset
      kind_t::make(is);
      eat(is, '_');
      auto gs = genset_t::make(is);
      return gs;
    }

    /// Whether unknown letters should be added, or rejected.
    /// \param o   whether to accept
    /// \returns   the previous status.
    bool open(bool o) const
    {
      return this->genset().open(o);
    }

    static constexpr bool is_free()
    {
      return false;
    }

    /// Value constructor.
    template <typename... Args>
    value_t value(Args&&... args) const
    {
      return value_t{std::forward<Args>(args)...};
    }

    /// Convert to a word.
    word_t word(const value_t& v) const
    {
      return v;
    }

    /// Prepare to iterate over the letters of v.
    static word_t
    letters_of(word_t v)
    {
      return v;
    }

    /// Whether \a l == \a r.
    static bool
    equals(const value_t l, const value_t r)
    {
      return l == r;
    }

    /// Whether \a l < \a r.
    static bool less_than(const value_t l, const value_t r)
    {
      return (std::forward_as_tuple(l.size(), l)
              < std::forward_as_tuple(r.size(), r));
    }

    static value_t
    special()
    {
      return genset_t::template special<value_t>();
    }

    static bool
    is_special(const value_t& v)
    {
      return v == special();
    }

    bool
    is_valid(const value_t& v) const
    {
      for (auto l: v)
        if (!this->has(l))
          return false;
      return true;
    }

    static constexpr bool
    is_ratexpset()
    {
      return false;
    }

    static constexpr bool
    has_one()
    {
      return true;
    }

    static constexpr bool
    is_letterized()
    {
      return false;
    }

    static value_t
    one()
    {
      return genset_t::empty_word();
    }

    static bool
    is_one(const value_t& l) ATTRIBUTE_PURE
    {
      return genset_t::is_empty_word(l);
    }

    static size_t size(const value_t& v)
    {
      return v.length();
    }

    static size_t hash(const value_t& v)
    {
      return utils::hash_value(v);
    }

    static value_t
    conv(self_type, value_t v)
    {
      return v;
    }

    // FIXME: Why do I need to repeat this?
    // It should be inherited from genset-labelset.
    value_t
    conv(std::istream& i) const
    {
      return this->genset().conv(i);
    }

    value_t
    parse(const std::string& s, size_t& p) const {
      size_t q=p;
      while(p<=q && this->has(s[--p]))
	;
      if(p>q)
	p=0u;
      return s.substr(p, q-p);
    }

    std::set<value_t>
    convs(std::istream& i) const
    {
      std::set<value_t> res;
      for (auto r : this->convs_(i))
        res.insert(value_t{r});
      return res;
    }

    std::ostream&
    print(const value_t& l, std::ostream& o,
          const std::string& format = "text") const
    {
      if (is_one(l))
        o << (format == "latex" ? "\\varepsilon" : "\\e");
      else if (!is_special(l))
        o << str_escape(l);
      return o;
    }

    std::ostream&
    print_set(std::ostream& o, const std::string& format = "text") const
    {
      if (format == "latex")
        {
          this->genset().print_set(o, format);
          o << "^*";
        }
      else if (format == "text")
        o << vname(true);
      else
        raise("invalid format: ", format);
      return o;
    }

    std::ostream&
    js_print(std::ostream& o) const {
      o << "{\"Label Type\":\"Words\" \"Alphabet\":[";
      bool first=true;
      for(auto l: this->genset()) {
	if(first)
	  first=false;
	else
	  o<< ',';
	this->genset().js_print(l, o);
      }
      o << "]}";
      
      return o;
    }

    std::ostream&
    js_print(std::ostream& o, value_t v) const {
      return o << '"' << v << '"';
    }

    value_t
    js_parse(std::istream& i) const {
      if(i.peek()==',')
	return special();
    	char c;
    	value_t r="";
        i >> c;
        if(c != '"')
        	throw std::runtime_error("json parser Letterset");
	while(true) {
	    i >> c;
	    if(this->has(c))
	      r=r+c;
	    else break;
	}
        if(c != '"')
      	  throw std::runtime_error("json parser Letterset");
       return r;
    }

  };

  namespace internal
  {
    template <typename GenSet>
    struct law_traits<wordset<GenSet>>
    {
      using type = wordset<GenSet>;
      static type value(const wordset<GenSet>& ls)
      {
        return ls;
      }
    };
  }

  /// Compute the meet with another alphabet.
  // FIXME: Factor in genset_labelset?
  template <typename GenSet>
  wordset<GenSet>
  meet(const wordset<GenSet>& lhs, const wordset<GenSet>& rhs)
  {
    return {intersection(lhs.genset(), rhs.genset())};
  }

  /// Compute the union with another alphabet.
  template <typename GenSet>
  wordset<GenSet>
  join(const wordset<GenSet>& lhs, const wordset<GenSet>& rhs)
  {
    return {get_union(lhs.genset(), rhs.genset())};
  }

}}//end of ns awali::stc

#endif // !AWALI_LABELSET_WORDSET_HH
