// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifndef AWALI_WEIGHTSET_F2_HH
# define AWALI_WEIGHTSET_F2_HH

# include <cassert>
# include <ostream>

#include <awali/sttc/misc/escape.hh>
#include <awali/utils/hash.hh>
#include <awali/sttc/misc/raise.hh>
#include <awali/common/star_status.hh>
#include <awali/sttc/misc/stream.hh>
#include <awali/sttc/weightset/weightset.hh>

namespace awali { namespace sttc {

  namespace internal
  {
  class f2_impl
  {
  public:
    using self_type = f2;

    static std::string sname()
    {
      return "f2";
    }

    std::string vname(bool = true) const
    {
      return sname();
    }

    /// Build from the description in \a is.
    static f2 make(std::istream& is)
    {
      eat(is, sname());
      return {};
    }

    using value_t = bool;

    static value_t
    zero()
    {
      return false;
    }

    static value_t
    one()
    {
      return true;
    }

    static value_t
    add(const value_t l, const value_t r)
    {
      return l ^ r;
    }

    static value_t
    sub(const value_t l, const value_t r)
    {
      return l ^ r;
    }

    static value_t
    mul(const value_t l, const value_t r)
    {
      return l && r;
    }

    static value_t
    rdiv(const value_t l, const value_t r)
    {
      require(!is_zero(r), "div: division by zero");
      return l;
    }

    static value_t
    ldiv(const value_t l, const value_t r)
    {
      return rdiv(r, l);
    }

    static value_t
    star(const value_t v)
    {
      if (!v)
        return true;
      else
        throw std::domain_error("f2: star: invalid value: 1");
    }

    static bool
    equals(const value_t l, const value_t r)
    {
      return l == r;
    }

    /// Whether \a lhs < \a rhs.
    static bool less_than(value_t lhs, value_t rhs)
    {
      return lhs < rhs;
    }

    constexpr static bool is_special(value_t)
    {
      return false;
    }

    static bool
    is_zero(const value_t v)
    {
      return !v;
    }

    static bool
    is_one(const value_t v)
    {
      return v;
    }

    static constexpr bool is_commutative_semiring() { return true; }

    static constexpr bool show_one() { return false; }

    static constexpr
    star_status_t star_status() { return star_status_t::NON_STARRABLE; }

    static value_t
    transpose(const value_t v)
    {
      return v;
    }

    static size_t hash(value_t v)
    {
      return utils::hash_value(v);
    }

    static value_t
    conv(self_type, value_t v)
    {
      return v;
    }

    static value_t
    conv(std::istream& stream)
    {
      int i;
      if ((stream >> i) && (i == 0 || i == 1))
        return i;
      else
        sttc::fail_reading(stream, sname() + ": invalid value");
    }

    static value_t
    parse(const std::string & s, size_t& p) {
      if(s[p--] == '0')
	return false;
      if(s[p+1] == '1')
	return true;
      throw parse_exception("Wrong F2 value");
    }

    static std::ostream&
    print(const value_t v, std::ostream& o,
          const std::string& = "text")
    {
      return o << (v ? '1' : '0');
    }

    std::ostream&
    print_set(std::ostream& o, const std::string& format = "text") const
    {
      if (format == "latex")
        o << "\\mathbb{F}_2";
      else if (format == "text")
        o << "F2";
      else
        raise("invalid format: ", format);
      return o;
    }

    std::ostream&
    js_print(std::ostream& o) const
    {
      o << "{\"Semiring\":\"F2\"}";
      return o;
    }

    value_t
    js_parse(std::istream& i) const
    {
      char c;
      i >> c;
      if(c != '"')
      	throw std::runtime_error("json parser F2");
	  value_t r;
	  i >> r;
      i >> c;
      if(c != '"')
      	throw std::runtime_error("json parser F2");
      return r;
    }

  };
  }// internal::

  AWALI_WEIGHTS_BINARY(f2, f2, f2);


}}//end of ns awali::stc

#endif // !AWALI_WEIGHTSET_F2_HH
