// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifndef AWALI_WEIGHTSET_ZMAX_HH
# define AWALI_WEIGHTSET_ZMAX_HH

# include <limits>
# include <ostream>
# include <string>
# include <sstream>
# include <utility>

#include <awali/sttc/misc/raise.hh>
#include <awali/common/star_status.hh>
#include <awali/sttc/misc/stream.hh> // eat
#include <awali/sttc/weightset/fwd.hh>
#include <awali/sttc/weightset/b.hh>
#include <awali/sttc/weightset/weightset.hh>

namespace awali { namespace sttc {

  namespace internal
  {
  class zmax_impl
  {
  public:
    using self_type = zmax;

    static std::string sname()
    {
      return "zmax";
    }

    std::string vname(bool = true) const
    {
      return sname();
    }

    /// Build from the description in \a is.
    static zmax make(std::istream& is)
    {
      eat(is, sname());
      return {};
    }

    using value_t = int;

    static value_t
    add(const value_t l, const value_t r)
    {
      return std::max(l, r);
    }

    static value_t
    mul(const value_t l, const value_t r)
    {
      return (is_zero(l) || is_zero(r) ? zero()
              : l + r);
    }

    static value_t
    rdiv(const value_t l, const value_t r)
    {
      require(!is_zero(r), "div: division by zero");
      return l - r;
    }

    static value_t
    ldiv(const value_t l, const value_t r)
    {
      return rdiv(r, l);
    }

    value_t
    star(const value_t v) const
    {
      if (0 >= v)
        return one();
      else
        raise(sname(), ": star: invalid value: ", format(*this, v));
    }

    static value_t
    one()
    {
      return 0;
    }

    static value_t
    zero()
    {
      return std::numeric_limits<value_t>::min();
    }

    static bool
    equals(const value_t l, const value_t r)
    {
      return l == r;
    }

    /// Whether \a lhs < \a rhs.
    static bool less_than(value_t lhs, value_t rhs)
    {
      return lhs < rhs;
    }

    constexpr static bool is_special(value_t)
    {
      return false;
    }

    static bool
    is_zero(const value_t v)
    {
      return v == zero();
    }

    static bool
    is_one(const value_t v)
    {
      return v == one();
    }

    static constexpr bool is_commutative_semiring() { return true; }

    static constexpr bool show_one() { return true; }
    static constexpr star_status_t star_status() { return star_status_t::TOPS; }

    static value_t
    transpose(const value_t v)
    {
      return v;
    }

    static size_t hash(value_t v)
    {
      return utils::hash_value(v);
    }

    static value_t
    conv(self_type, value_t v)
    {
      return v;
    }

    static value_t
    conv(b, b::value_t v)
    {
      return v ? one() : zero();
    }

    static value_t
    conv(std::istream& stream)
    {
      switch (int i = stream.peek())
        {
        case 'o':
          stream.ignore();
          if ((i = stream.get()) == 'o')
            return zero();
          else
            throw std::domain_error(sname() + ": invalid value: o" + str_escape(i));
        default:
          if (stream >> i)
            return i;
          else
            sttc::fail_reading(stream, sname() + ": invalid value");
        }
    }

    static value_t
    parse(const std::string & s, size_t& p) {
      if(s[p-1]=='o'){
	--p;
	return zero();
      }
      size_t i=p;
      for(; i>0 && s[i-1]>='0' && s[i-1]<='9'; --i)
	;
      if(i==p)
	throw parse_exception("Wrong Z-max value");
      if(i>0 && (s[i-1]=='-' || s[i-1]=='+'))
	--i;
      std::istringstream st(s.substr(i, p-i));
      value_t x;
      st >> x;
      p=i;
      return x;
    }

    static std::ostream&
    print(const value_t v, std::ostream& o,
          const std::string& format = "text")
    {
      if(format == "json")
      	o<< '"';
      if (is_zero(v))
        o << (format == "latex" ? "-\\infty" : "oo");
      else
        o << v;
      if(format == "json")
      	o<< '"';
      return o;
    }

    std::ostream&
    print_set(std::ostream& o, const std::string& format = "text") const
    {
      if (format == "latex")
        o << "\\mathbb{Z}_{max}";
      else if (format == "text")
        o << "Z-max-plus";
      else
        raise("invalid format: ", format);
      return o;
    }

    std::ostream&
    js_print(std::ostream& o) const
    {
      o << "{\"Semiring\":\"Z-max-plus\"}";
      return o;
    }

    value_t
    js_parse(std::istream& i) const
    {
      char c;
      i >> c;
      if(c != '"')
      	throw std::runtime_error("json parser zmax");
      if (i.peek()=='o') {
      	i >> c;
      	i >> c;
      	if(c=='o')
      		return zero();
      	throw std::runtime_error("json parser zmax");
	  }
	  value_t r;
	  i >> r;
      i >> c;
      if(c != '"')
      	throw std::runtime_error("json parser zmax");
      return r;
    }

  };
  }

  AWALI_WEIGHTS_BINARY(zmax, zmax, zmax);

  AWALI_WEIGHTS_BINARY(b, zmax, zmax);
  AWALI_WEIGHTS_BINARY(zmax, b, zmax);

}}//end of ns awali::stc

#endif // !AWALI_WEIGHTSET_ZMAX_HH
