// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifndef DYN_EDIT_HH
#define DYN_EDIT_HH

#include <ncurses.h>
#include <cstring>
#include <fstream>
#include <awali/dyn.hh>

namespace awali { 
  namespace dyn {

    /*
      Up Down -> Navigation in List
      Right -> From state mode to transition mode
      Left  -> From transition mode to state mode

      NOEDIT  --[s]ave->  SAVE
      --[e]dit->  TRANSEDIT (transition mode only)
      --[t]ransition-> ADDFROM (state mode only)
      --[d]elete-> NOEDIT
      --[a]dd-> NOEDIT (state mode) or ADDTRANS (transition mode)
      --[i]nitial-> NOEDIT (nfa) or INITIAL
      --[f]inal  -> NOEDIT (nfa) or FINAL
      --[w]indow -> NOEDIT
    */
    void gen_edit(automaton_t a, bool is_transducer=false) {
      auto context=a->get_context();
      // editmode tells whether a user's answer is required
      enum {NOEDIT, TRANSEDIT, ADDTRANS, ADDFROM,SAVE,NAME, INITIAL,FINAL,WEIGHT} editmode=NOEDIT;
      // statemode is true if a state is selected, false if a transition is selected
      bool statemode=true;
      bool nfa=(context->weightname().compare("B")==0);
      // array of available actions; depends on the mode
      unsigned int actions[10]={};
      // array of caption of actions
      const char* expl[10]={};
      // Start curses mode
      initscr();
      // w: width, h: height of the screen
      int w,h;
      getmaxyx(stdscr,h,w);
      //name of the automaton files
      char name[30];
      name[0]='\0';
      // enable arrow keys
      keypad(stdscr, TRUE);
      //number of tapes:
      int n;
      int posedit;
      if(is_transducer)
	n=num_tapes(a);
      else
	n=1;
      // index of the selected state or transition
      int j=0;
      // shift for the printing of states in case of many states
      int decs=0;
      // idem for transitions
      int dect=0;
      // main loop; if c='q' at the end of the loop, exit
      int c=0;
      do {
	clear();noecho();curs_set(0);
	//header
	attrset(A_BOLD);
	mvprintw(0,w/4,"%d states:",a->num_states());
	mvprintw(0,w/2,"%d transitions:",a->num_transitions());
	if(is_transducer) {
	  mvprintw(0,0,"Edit transducer");
	  mvprintw(2,2,"Alphabets:");
	  attrset(A_NORMAL);
	  mvprintw(1,2,"%d tapes",n);
	  unsigned arow=3;
	  for(auto v : alphabets(a)) {
	    move(arow++,2);
	    char sep=' ';
	    for(auto l : v){
	      addch(sep);
	      addch(l);
	      sep=',';
	    }
	  }
	}
	else {
	  if(nfa)
	    mvprintw(0,0,"Edit automaton");
	  else
	    mvprintw(0,0,"Edit %s-automaton", context->weightname().c_str());
	  mvprintw(2,2,"Alphabet:");
	  attrset(A_NORMAL);
	  move(3,2);
	  char sep=' ';
	  for(auto l : a->alphabet()) {
	    addch(sep);
	    addstr(context->label_to_string(l).c_str());
	    sep=',';
	  }
	}
	//vertical separation between left(states) and right(transitions)
	for(int t=1; t<h-2;++t)
	  mvaddch(t,w/2-2,ACS_VLINE);
	//vector of states
	std::vector<unsigned> states(a->states());
	//number of states
	int ns=states.size();
	//compute shift for printing
	if(statemode && ns>h-3) {
	  if(j<decs+2) --decs;
	  if(decs<0) decs=0;
	  if(j>decs+h-6) ++decs;
	  if(decs+h-3>ns) --decs;
	}
	//print states
	for(int s=decs;s<ns && s<decs+h-3;++s) {
	  if(s==j && statemode)
	    attrset(A_STANDOUT);
	  else
	    attrset(A_NORMAL);
	  if(a->is_initial(states[s])) {
	    if(!nfa && a->get_initial_weight(states[s])!=context->weight_one()) {
	      mvprintw(s+1-decs,w/4,"-%s-> ",context->weight_to_string(a->get_initial_weight(states[s])).c_str());
	    }
	    else
	      mvprintw(s+1-decs,w/4," --> ");
	  }
	  else
	    mvprintw(s+1-decs,w/4,"     ");
	  printw("%s",a->get_state_name(states[s]).c_str());
	  if(a->is_final(states[s])) {
	    if(!nfa && a->get_final_weight(states[s])!=context->weight_one()) {
	      printw(" -%s->",context->weight_to_string(a->get_final_weight(states[s])).c_str());
	    }
	    else
	      printw(" --> ");
	  }
	  else
	    printw("     ");
	}
	std::vector<unsigned> trans(a->transitions());
	// the transitions are sorted w.r.t. 1. source state
	trans.clear();
	for(int s=0;s<ns ;++s)
	  for(int t : a->out(states[s]))
	    trans.emplace_back(t);
	//number of transitions
	int nt=trans.size();
	//compute shift for printing
	if(!statemode && nt>h-3) {
	  if(j<dect+2) --dect;
	  if(dect<0) dect=0;
	  if(j>dect+h-6) ++dect;
	  if(dect+h-3>ns) --dect;
	}
	refresh();
	//print transitions
	for(int t=dect;t<nt && t<dect+h-3;++t) {
	  if(t==j && !statemode)
	    attrset(A_STANDOUT);
	  else
	    attrset(A_NORMAL);
	  if(is_transducer) {
	    auto v=get_tdc_label(a,trans[t]);
	    char ci[n];
	    for(int k=0; k<n; ++k)
	      if(v[k].empty())
		ci[k]='_';
	      else
		ci[k]=v[k][0];
	    mvprintw(1+t-dect,w/2,"%s -",a->get_state_name(a->src_of(trans[t])).c_str());
	    char sep='-';
	    for(int k=0; k<n; ++k) {
	      printw("%c %c ",sep,ci[k]);
	      sep='|';
	    }
	    printw("--> %s", a->get_state_name(a->dst_of(trans[t])).c_str());
	  }
	  else
	    if(nfa) {
	      mvprintw(1+t-dect,w/2,"%s -- %s --> %s",
		       a->get_state_name(a->src_of(trans[t])).c_str(),
		       (a->is_eps_transition(trans[t]))?"_":context->label_to_string(a->label_of(trans[t])).c_str(),
		       a->get_state_name(a->dst_of(trans[t])).c_str());
	    }
	    else {
	      mvprintw(1+t-dect,w/2,"%s -- <%s> %s --> %s",
		       a->get_state_name(a->src_of(trans[t])).c_str(),
		       context->weight_to_string(a->weight_of(trans[t])).c_str(),
		       context->label_to_string(a->label_of(trans[t])).c_str(),
		       a->get_state_name(a->dst_of(trans[t])).c_str());
	    }

	}
	//***Menu***
	int size=0;
	if(editmode==NOEDIT) {
	  actions[size]='q';
	  expl[size++]="quit";
	  actions[size]='s';
	  expl[size++]="save";
	  actions[size]='w';
	  expl[size++]="display";
	  // "state" mode
	  if(statemode) {
	    actions[size]='a';
	    expl[size++]="add state";
	    if(ns>0) {
	      actions[size]='d';
	      expl[size++]="delete state";
	      actions[size]='n';
	      expl[size++]="rename state";
	      actions[size]='i';
	      if (nfa) {
		if(a->is_initial(states[j]))
		  expl[size++]="unset initial";
		else
		  expl[size++]="set initial";
	      }
	      else
		expl[size++]="chg initial";
	      actions[size]='f';
	      if(nfa) {
		if(a->is_final(states[j]))
		  expl[size++]="unset final";
		else
		  expl[size++]="set final";
	      }
	      else
		expl[size++]="chg final";
	      actions[size]='t';
	      expl[size++]="trans. from";
	    }
	  }
	  // "transitions" mode
	  else {
	    actions[size]='a';
	    expl[size++]="add trans.";
	    if(nt>0) {
	      actions[size]='d';
	      expl[size++]="delete trans.";
	      actions[size]='e';
	      expl[size++]="edit trans.";
	    }
	  }
	}
	//Print actions
	attrset(A_STANDOUT);
	for(int i=0;i<size;i++)
	  mvaddch(h-2+(i%2),w/5*(i/2), actions[i]);
	attrset(A_NORMAL);
	for(int i=0;i<size;i++)
	  mvprintw(h-2+(i%2),w/5*(i/2)+2, expl[i]);
	//if there is no scanf, a character is waited
	if(editmode==NOEDIT) {
	  refresh();
	  c=getch();
	  switch(c) {
	  case KEY_UP : case '^':  if(j>0) j--; break;
	  case KEY_DOWN : case 'v': if((statemode && j<ns-1)||(!statemode && j<nt-1)) j++; break;
	  case KEY_LEFT : case '<' : if(!statemode) j+=decs-dect; statemode=true; if(j>ns-1) j=ns-1; if(j<0) j=0; break;
	  case KEY_RIGHT : case '>' : if(statemode && a->num_states()>0) j+=dect-decs; statemode=false; if(j>nt-1) j=nt-1; if(j<0) j=0; break;
	  case 'e' : if(!statemode) { editmode=TRANSEDIT; posedit=0;} break;
	  case 't' : if(statemode && a->num_states()>0) { editmode=ADDFROM; posedit=nfa?2:1;} break;
	  case 's' : editmode=SAVE;
	    break;
	  case 'd' :
	    if(statemode) {
	      if(a->num_states()>0) {
		a->del_state(states[j]);
		if(j==ns-1) --j; //if the last state is deleted
	      }
	      if(j<0) j=0;
	    }
	    else {
	      a->del_transition(trans[j]);
	      if(j==nt-1) --j; //if the last transition is deleted
	      if(j<0) j=0;
	    } break;
	  case 'a' : if (statemode)
	      a->add_state();
	    else {
	      editmode=ADDTRANS;
	      posedit=0;
	    }
	    break;
	  case 'i' :
	    if(nfa) {
	      if(statemode && a->num_states()>0) {
		if(a->is_initial(states[j]))
		  a->unset_initial(states[j]);
		else
		  a->set_initial(states[j]);
	      }
	    }
	    else
	      if(a->num_states()>0)
		editmode=INITIAL;
	    break;
	  case 'f' :
	    if(nfa) {
	      if(statemode && a->num_states()>0) {
		if(a->is_final(states[j]))
		  a->unset_final(states[j]);
		else
		  a->set_final(states[j]);
	      }
	    }
	    else
	      if(a->num_states()>0)
		editmode=FINAL;
	    break;
	  case 'w' :
	    { std::string s = "/tmp/tmp.gv";
	      std::ofstream o(s);
	      dot(a, o) << std::endl;
	      o.close();
	      system(("dotty "+s).c_str());}
	    break;
	  case 'n' :
	    if (statemode && a->num_states()>0)
	      editmode=NAME;
	    break;
	  }
	}
	else if(editmode==SAVE) {
	  attrset(A_STANDOUT);
	  echo(); curs_set(1);
	  mvprintw(h-3,0, "                                        ");
	  if(name[0]=='\0') {
	    mvprintw(h-3,0, "Name:");
	    scanw((char*)"%s",name);
	  } else {
	    char tmpname[30];
	    tmpname[0]='\0';
	    mvprintw(h-3,0, "Name [%s]:",name);
	    scanw((char*)"%s",tmpname);
	    if(tmpname[0]!='\0')
	      std::strcpy(name,tmpname);
	  }
	  std::ofstream out(name);
	  a->json(out);
	  out.close();
	  editmode=NOEDIT;
	}
	else if(editmode==NAME) {
	  attrset(A_STANDOUT);
	  echo(); curs_set(1);
	  mvprintw(h-3,0, "                                        ");
	  char tmpname[30];
	  tmpname[0]='\0';
	  mvprintw(h-3,0, "Name [%s]:",a->get_state_name(states[j]).c_str());
	  scanw((char*)"%s",tmpname);
	  if(tmpname[0]!='\0')
	    a->set_state_name(states[j], tmpname);
	  editmode=NOEDIT;
	}
	else if(editmode==INITIAL || editmode==FINAL) {
	  attrset(A_STANDOUT);
	  unsigned s;
	  weight_t we=context->weight_one();
	  s= states[j];
	  //we want to print the input and to see the cursor
	  echo(); curs_set(1);
	  // Source state
	  if(editmode==INITIAL) {
	    if(a->is_initial(s))
	      we=context->weight_zero();
	    bool verif;
	    do {
	      verif=true;
	      mvprintw(h-3,0, "                                        ");
	      if(a->is_initial(s))
		mvprintw(h-3,0, "Initial weight [unset initial] : ", context->weight_to_string(we).c_str());
	      else
		mvprintw(h-3,0, "Initial weight [%s] : ", context->weight_to_string(we).c_str());
	      char weight[20];
	      weight[0]='\0';
	      scanw((char*)"%s",weight);
	      if(weight[0]!='\0')
		try {
		  we=context->get_weight(weight);
		} catch(...) {
		  verif=false;
		}
	    } while (!verif);
	    a->set_initial(s,we);
	  }
	  if(editmode==FINAL) {
	    if(a->is_final(s))
	      we=context->weight_zero();
	    bool verif;
	    do {
	      verif=true;
	      mvprintw(h-3,0, "                                        ");
	      if(a->is_final(s))
		mvprintw(h-3,0, "Final weight [unset final] : ", context->weight_to_string(we).c_str());
	      else
		mvprintw(h-3,0, "Final weight [%s] : ", context->weight_to_string(we).c_str());
	      char weight[20];
	      weight[0]='\0';
	      scanw((char*)"%s",weight);
	      if(weight[0]!='\0')
		try {
		  we=context->get_weight(weight);
		} catch(...) {
		  verif=false;
		}
	    } while (!verif);
	    a->set_final(s,we);
	  }
	  editmode=NOEDIT;
	}
	else {//edition of transitions
	  //some scanf are required to retrieve informations concerning a transition
	  unsigned s,d;
	  char ci[n];
	  for(int i=0;i<n;++i) ci[i]='?';
	  std::string l{'?'};
	  weight_t we=context->weight_one();
	  char source[30]="?";
	  if(editmode==ADDFROM) {// in this case we are creating a transition from the current state
	    s= states[j];
	    std::strcpy(source, a->get_state_name(s).c_str());
	  }
	  if(editmode==TRANSEDIT) {
	    s= a->src_of(trans[j]);
	    std::strcpy(source, a->get_state_name(s).c_str());
	  }
	  char dest[30]="?";
	  if(editmode==TRANSEDIT) {
	    d= a->dst_of(trans[j]);
	    std::strcpy(dest, a->get_state_name(d).c_str());
	  }
	  if(editmode==TRANSEDIT) {
	    // characteristics of the current transition
	    we=a->weight_of(trans[j]);
	    if(is_transducer) {
	      auto v=get_tdc_label(a,trans[j]);
	      for(int ac=0;ac<n; ++ac)
		if(v[ac].empty())
		  ci[ac]='_';
		else
		  ci[ac]=v[ac][0];
	    }
	    else
	      l=(a->is_eps_transition(trans[j]))?"_":context->label_to_string(a->label_of(trans[j])).c_str();
	  }
	  char sweight[30];
	  if(!nfa) {
	    if(editmode==TRANSEDIT)
	      we=a->weight_of(trans[j]);
	    std::strcpy(sweight, context->weight_to_string(we).c_str());
	  }
	  char rep[30]={};
	  int nchar=0;
	  bool repeat = true;
	  do {
	    unsigned pos[n+3];
	    attrset(A_STANDOUT);
	    mvprintw(h-1,0, "<-");
	    attrset(A_NORMAL);
	    printw(" prev.");
	    attrset(A_STANDOUT);
	    mvprintw(h-2,0, "->");
	    attrset(A_NORMAL);
	    printw(" or ");
	    attrset(A_STANDOUT);
	    printw("TAB");
	    attrset(A_NORMAL);
	    printw(" next.");
	    attrset(A_STANDOUT);
	    mvprintw(h-2,w/5, "DEL");
	    attrset(A_NORMAL);
	    printw(" corr. ");
	    attrset(A_STANDOUT);
	    mvprintw(h-1,w/5, "RET");
	    attrset(A_NORMAL);
	    printw(" valid");
	    mvprintw(h-3,0, "Transition : ");
	    pos[0]=getcurx(stdscr);
	    attrset(A_STANDOUT);
	    if(nchar>0 && posedit==0)
	      printw("%s",rep);
	    else
	      printw("%s",source);
	    attrset(A_NORMAL);
	    printw(" --");
	    if(!nfa) {
	      printw(" <");
	      pos[1]=getcurx(stdscr);
	      attrset(A_STANDOUT);
	      if(nchar>0 && posedit==1)
		printw("%s",rep);
	      else
		printw("%s",sweight);
	      attrset(A_NORMAL);
	      printw(">");
	    }
	    if(is_transducer) {
	      char sep=' ';
	      for(int ac=0;ac<n;++ac){
		printw("%c ",sep);
		sep='|';
		pos[2+ac]=getcurx(stdscr);
		attrset(A_STANDOUT);
		if(nchar>0 && posedit==2+ac)
		  printw("%s",rep);
		else
		  printw("%c",ci[ac]);
		attrset(A_NORMAL);
		addch(' ');
	      }
	    }
	    else {
	      addch(' ');
	      pos[2]=getcurx(stdscr);
	      attrset(A_STANDOUT);
	      if(nchar>0 && posedit==2)
		printw("%s",rep);
	      else
		printw("%s",l.c_str());
	      attrset(A_NORMAL);
	      addch(' ');
	    }
	    printw("--> ");
	    pos[n+2]=getcurx(stdscr);
	    attrset(A_STANDOUT);
	    if(nchar>0 && posedit==n+2)
	      printw("%s",rep);
	    else
	      printw("%s",dest);
	    attrset(A_NORMAL);
	    printw("  ");
	    move(h-3,pos[posedit]+nchar);
	    refresh();
	    curs_set(1);

	    int ch=getch();

	    switch(ch) {
	    case KEY_BACKSPACE : case KEY_DC : case KEY_CANCEL : case KEY_UNDO : case 127 :
	      if(nchar>0)
		rep[--nchar]='\0';
	      break;
	    case KEY_LEFT : case '<' :
	    case KEY_RIGHT : case '\t' : case '>' :
	    case KEY_ENTER : case '\n' :
	      {
		bool valuecorrect=false;
		if(posedit==0) {
		  if(nchar==0 && source[0]!='?')
		    valuecorrect=true;
		  else {
		    unsigned p= a->get_state_by_name(rep);
		    if(a->has_state(p)){
		      valuecorrect=true;
		      s=p;
		      std::strcpy(source,rep);
		    }
		  }
		}
		else if(posedit==1) {
		  if(nchar==0 && sweight[0]!='?')
		    valuecorrect=true;
		  else
		    try{
		      we=context->get_weight(rep);
		      valuecorrect=true;
		      std::strcpy(sweight,rep);
		    }
		    catch(...) {}
		}
		else if(posedit==n+2) {
		  if(nchar==0 && dest[0]!='?')
		    valuecorrect=true;
		  else {
		    unsigned p= a->get_state_by_name(rep);
		    if(a->has_state(p)){
		      valuecorrect=true;
		      d=p;
		      std::strcpy(dest,rep);
		    }
		  }
		}
		else if(is_transducer) {
		  if(nchar==0 && ci[posedit-2]!='?')
		    valuecorrect=true;
		}
		else
		  if(nchar==0 && l[0]!='?')
		    valuecorrect=true;
		  else
		    try{
          context->get_label(strcmp(rep,"_")?rep:"\\e");
			    valuecorrect=true;
			    l= rep;
		    }
		    catch(...) {}
		if(valuecorrect) {
		  for(int i=0; i<nchar; i++)
		    rep[i]='\0';
		  nchar=0;
		  if(ch == KEY_LEFT || ch == '<') {
		    if(posedit>0)
		      --posedit;
		    if(nfa && posedit==1)
		      posedit=0;
		  }
		  if(ch == KEY_RIGHT || ch =='\t' || ch =='>') {
		    if(posedit<n+2)
		      ++posedit;
		    if(nfa && posedit==1)
		      posedit=2;
		  }
		  if(ch == KEY_ENTER || ch == '\n') {
		    repeat= source[0]=='?' || dest[0]=='?' || sweight[0]=='?';
		    if(is_transducer)
		      for(int i=0;i<n;++i)
			repeat |= ci[i]=='?';
		    else
		      repeat |= l=="?";
		    std::cerr << "Entree ---> " << repeat << std::endl;
		  }
		}
	      }
	      break;
	    default:
	      if(ch>31 && ch<128)
		rep[nchar++]=ch;
	      //
	      if(is_transducer && nchar==1 && posedit>1 && posedit <n+2) {
		if (rep[0]=='_') {
		  ci[posedit-2]='_';
		  ++posedit;
		}
		else if (has_label(a, posedit-2, rep)) {
		  ci[posedit-2]=rep[0];
		  ++posedit;
		}
		rep[--nchar]='\0';
	      }
	    }
	  }while(repeat);


	  // if it is an edition; the new transition replaces the former one
	  if(editmode==TRANSEDIT)
	    a->del_transition(trans[j]);
	  //creation of the new transition
	  if(is_transducer) {
	    std::vector<std::string> label;
	    for(int ac=0; ac<n;++ac)
	      if(ci[ac]=='_') {
		label.emplace_back("");
	      }
	      else{
		std::string tmp("a");
		tmp[0]=ci[ac];
		label.emplace_back(tmp);
	      }
	    set_tdc_transition(a, s, d, label);
	  }
	  else
	    if(l[0]=='_' && a->allow_eps_transition())
	      a->set_eps_transition(s, d, we);
	    else
	      a->set_transition(s, d, context->get_label(l), we);
	  editmode=NOEDIT;
	}

      } while(c!='q');
      attrset(A_STANDOUT);
      echo(); curs_set(1);
      do {
	mvprintw(h-3,0, "Save ? [y/n]                                      ");
	refresh();
	c=getch();
      } while(c!='y' && c!='n');
      if(c=='y') {
	mvprintw(h-3,0, "                                                  ");
	if(name[0]=='\0') {
	  mvprintw(h-3,0, "Name:");
	  scanw((char*)"%s",name);
	} else {
	  char tmpname[30];
	  tmpname[0]='\0';
	  mvprintw(h-3,0, "Name [%s]:",name);
	  scanw((char*)"%s",tmpname);
	  if(tmpname[0]!='\0')
	    std::strcpy(name,tmpname);
	}
	if(name[0]!='\0') {
	  std::ofstream out(name);
	  a->json(out);
	  out.close();
	}
	editmode=NOEDIT;
      }
      endwin();
      //end of curses
    }

    void edit(automaton_t a) {
      gen_edit(a, a->is_transducer());
    }
  } //end of awali::dyn
} //end of awali

#endif
