/* the evaluation functions */

#include "includes.h"
#include "knightcap.h"

#define EVAL_ALL_DEPTH 1

#define USE_EVAL_SHORTCUT 0
#define EVAL_SHORTCUT_THRESHOLD (2*PAWN_VALUE)
#define EVAL_SHORTCUT_OFFSET (2*PAWN_VALUE)

#define HUNG_PINS 1

#define POSITIONAL_FACTOR 0.08
#define BOARD_CONTROL_FACTOR 1
#define TRAPPED_FACTOR 1
#define PIECE_VALUE_FACTOR 1
#define SQUARE_CONTROL_FACTOR 1
#define TACTICAL_FACTOR 1
#define MOBILITY_FACTOR 1

#define KING_DEFENCE_FACTOR 2
#define KING_ATTACK_FACTOR 7

#define PAWN_FACTOR 1
#define KING_FACTOR 1
#define QUEEN_FACTOR 1
#define BISHOP_FACTOR 1
#define KNIGHT_FACTOR 1
#define ROOK_FACTOR 1

#define DRAW_SLOPE 3

#define BISHOP_PAIR 100
#define ATTACK_VALUE (64*TACTICAL_FACTOR)
#define WEAK_PAWN_ATTACK_VALUE (64*TACTICAL_FACTOR)
#define UNSTOPPABLE_PAWN 3000

#define QUEEN_ADVANCE 100
#define DOUBLED_PAWN 80
#define CASTLE_BONUS 350
#define ROOK_ON_OPEN_FILE 100
#define ROOK_ON_HALF_OPEN_FILE 25
#define KNIGHT_OUTPOST 100
#define BISHOP_OUTPOST 50
#define WEAK_PAWN  125
#define PAWN_ADVANCE 10
#define CONNECTED_ROOKS 70
#define ODD_BISHOPS_PAWN_POS 20
#define NO_PAWNS 900
#define OPPOSITE_BISHOPS 400
#define PINNED_HUNG_PIECE 1100
#define BLOCKED_PASSED_PAWN 250
#define KING_PASSED_PAWN_SUPPORT 200
#define PASSED_PAWN_ROOK_ATTACK 350
#define PASSED_PAWN_ROOK_SUPPORT 300
#define CONNECTED_PASSED_PAWNS 400
#define BAD_TRAP 800

#define PAWN_DEFENCE 150

#define OPENING_KING_ADVANCE 400
#define MID_KING_ADVANCE 200

#define BLOCKED_DPAWN 150
#define BLOCKED_EPAWN 150
#define BLOCKED_KNIGHT 95

#define USELESS_PIECE 80

#define DRAW_VALUE -10


extern struct state *state;
static int debug;

/* this is the value of pawns on open files as they move up the board
   - it is doubled for passed pawns */
static const int pawn_advance[7] = {0,  1700,1100, 800, 450, 300, 100}; 

static unsigned char black_pawn_loc[8];
static unsigned char white_pawn_loc[8];

#define PAWN_LOC(player) (player>0?white_pawn_loc:black_pawn_loc)

static int pop_count[256];

static void init_eval_tables(void)
{
	static int initialised;
	int i;

	if (initialised) return;
	initialised = 1;
	
	for (i=0;i<256;i++) {
		int j, ret=0;
		for (j=0;j<8;j++)
			if (i & (1<<j))
				ret++;
		pop_count[i] = ret;
	}
}

/* this basically tries to answer the question "can one of the players
   profitably capture on that square, assuming there was something there
   to capture */
static int get_control(Position *b, uint32 topieces, int piece) 
{
	int nw, nb;
	uint32 mask;
	/* the following masks select pieces that are of lower than
	   value than a piece */
	static const uint32 masks[2*KING+1] = 
	{0xFFFE, 0xFFFC, 0xFFF0, 0xFF00, 0xFF00, 0, 
	 0, 
	 0, 0xFF000000, 0xFF000000, 0xFFF00000, 0xFFFC0000, 0xFFFE0000};

	if (!topieces) return 0;

	mask = masks[piece+KING];
	if (topieces & mask) 
		return -piece;

	nw = pop_count[(topieces >> 8) & 0xFF];
	nb = pop_count[(topieces >> 24) & 0xFF];
	if (nw != nb) return nw - nb;

	nw = pop_count[topieces & 0xFF];
	nb = pop_count[(topieces >> 16) & 0xFF];

	return (nw - nb);
}




static const int pawn_pos_value[NUM_SQUARES] = {
		0, 5, 0, 0, 0, 0, 0, 0,
		0, 5, 2, 0, 0, 0, 0, 0,
		0, 5, 0, 0, 0, 0, 0, 0,
		0, 2,10,70,45, 0, 0, 0,
		0, 2,10,75,45, 0, 0, 0,
		0,15,-9, 0, 0, 0, 0, 0,
		0,15,-2,-9, 0, 0, 0, 0,
		0,15, 0,-9, 0, 0, 0, 0,
};


/* this evaluates a pawn - its more valuable if its a passed
   pawn or on an open file. In those cases it gains value
   as it moves up the board */
static int eval_white_pawn(Position *b, int pi)
{
	PieceStruct *piece = b->pieces + pi;
	int ret=0;
	int x = XPOS(piece->pos);
	int y = YPOS(piece->pos);
	int advance = 0;
	int weak = 0;

	if (b->stage != ENDING)
		ret = pawn_pos_value[piece->pos];

	/* a pawn is weak if it isn't supported by a pawn and can't
	   move to a place where it is supported by a pawn */
	if ((b->topieces[piece->pos] & WPAWN_MASK) == 0 &&
	    ((b->topieces[piece->pos+NORTH] & WPAWN_MASK) == 0 ||
	     get_control(b, b->topieces[piece->pos+NORTH], PAWN) < 0) &&
	    (y != 1 || (b->topieces[piece->pos+NORTH] & BPAWN_MASK) ||
	     (b->topieces[piece->pos+2*NORTH] & WPAWN_MASK) == 0 ||
	     get_control(b, b->topieces[piece->pos+2*NORTH], PAWN) < 0)) {
		if (debug)
			lprintf(0,"weak %s\n", posstr(piece->pos));
		weak = 1;

		ret -= WEAK_PAWN;

		/* if its on a half open file then its even weaker */
		if (black_pawn_loc[x] == 0)
			ret -= WEAK_PAWN;

		/* attacks on weak pawns are worth something */
		if ((b->topieces[piece->pos]>>16) & 0xFF)
			ret -= pop_count[(b->topieces[piece->pos]>>16) & 0xFF] * WEAK_PAWN_ATTACK_VALUE;
		
	}

	if (black_pawn_loc[x] < y) {
		int n = 7-y;
		if (!weak) {
			advance = pawn_advance[n] / 4;
		}
		if ((x==0 || black_pawn_loc[x-1] <= y) &&
		    (x==7 || black_pawn_loc[x+1] <= y)) {
			Square wkpos = WHITEPIECES(b)[IKING].pos;
			Square bkpos = BLACKPIECES(b)[IKING].pos;

			advance += pawn_advance[n];

			if (YPOS(bkpos) >= y && abs(XPOS(bkpos) - x) <= 1)
				advance >>= 1;

			if (YPOS(wkpos) >= y && abs(XPOS(wkpos) - x) <= 1)
				advance += KING_PASSED_PAWN_SUPPORT;

			if (b->board[piece->pos+NORTH] < 0)
				advance -= BLOCKED_PASSED_PAWN;

			if (x != 0 && white_pawn_loc[x-1] &&
			    abs(y - white_pawn_loc[x-1]) <= 1) {
				advance += CONNECTED_PASSED_PAWNS;
			}

			if (x != 7 && white_pawn_loc[x+1] &&
			    abs(y - white_pawn_loc[x+1]) <= 1) {
				advance += CONNECTED_PASSED_PAWNS;
			}

			if ((b->topieces[piece->pos] & BKROOK_MASK) &&
			    XPOS(BLACKPIECES(b)[IKROOK].pos) == x &&
			    YPOS(BLACKPIECES(b)[IKROOK].pos) < y) {
				/* there is a black king rook 
				   behind our passed pawn */
				advance -= PASSED_PAWN_ROOK_ATTACK;
			}

			if ((b->topieces[piece->pos] & BQROOK_MASK) &&
			    XPOS(BLACKPIECES(b)[IQROOK].pos) == x &&
			    YPOS(BLACKPIECES(b)[IQROOK].pos) < y) {
				/* there is a black queen rook 
				   behind our passed pawn */
				advance -= PASSED_PAWN_ROOK_ATTACK;
			}

			if ((b->topieces[piece->pos] & WKROOK_MASK) &&
			    XPOS(WHITEPIECES(b)[IKROOK].pos) == x &&
			    YPOS(WHITEPIECES(b)[IKROOK].pos) < y) {
				/* there is a white king rook 
				   behind our passed pawn */
				advance += PASSED_PAWN_ROOK_SUPPORT;
			}

			if ((b->topieces[piece->pos] & WQROOK_MASK) &&
			    XPOS(WHITEPIECES(b)[IQROOK].pos) == x &&
			    YPOS(WHITEPIECES(b)[IQROOK].pos) < y) {
				/* there is a white queen rook 
				   behind our passed pawn */
				advance += PASSED_PAWN_ROOK_SUPPORT;
			}


			/* a special case - is it unstoppable? */
			if ((b->piece_mask & BLACK_MASK) == BKING_MASK) {
				int kpos = BLACKPIECES(b)[IKING].pos;
				int kmoves = imax(abs(XPOS(kpos) - x), 7-YPOS(kpos));
				int pmoves = 7-y;
				if (blacks_move(b)) kmoves--;
				if (kmoves > pmoves) {
 					ret += UNSTOPPABLE_PAWN;
					ret += pawn_advance[n] * 2;
					if (debug)
						lprintf(0,"unstoppable %s\n",posstr(piece->pos));
				}
			}
		}
		ret += advance;
	}

	if (white_pawn_loc[x] != y)
		ret -= DOUBLED_PAWN;

	if (b->stage != OPENING)
		ret += y * PAWN_ADVANCE;

	if ((b->piece_mask & WBISHOP_MASK) == WKBISHOP_MASK) {
		if (!white_square(piece->pos)) ret += ODD_BISHOPS_PAWN_POS;
	} else if ((b->piece_mask & WBISHOP_MASK) == WQBISHOP_MASK) {
		if (white_square(piece->pos)) ret += ODD_BISHOPS_PAWN_POS;
	}

	/* pawns are worth more the less material the opponent has */
	ret += (INITIAL_MATERIAL - b->b_material) >> 4;
	

	return ret;
}

static int eval_black_pawn(Position *b, int pi)
{
	PieceStruct *piece = b->pieces + pi;
	int ret=0;
	int x = XPOS(piece->pos);
	int y = YPOS(piece->pos);
	int advance = 0;
	int weak = 0;

	ret = pawn_pos_value[mirror_square(piece->pos)];

	if (b->stage == ENDING)
		ret >>= 1;

	/* a pawn is weak if it isn't supported by a pawn and can't
	   move to a place where it is supported by a pawn */
	if ((b->topieces[piece->pos] & BPAWN_MASK) == 0 &&
	    ((b->topieces[piece->pos+SOUTH] & BPAWN_MASK) == 0 ||
	     get_control(b, b->topieces[piece->pos+SOUTH], -PAWN) > 0) &&
	    (y != 6 || (b->topieces[piece->pos+SOUTH] & WPAWN_MASK) ||
	     (b->topieces[piece->pos+2*SOUTH] & BPAWN_MASK) == 0 ||
	     get_control(b, b->topieces[piece->pos+2*SOUTH], -PAWN) > 0)) {
		if (debug)
			lprintf(0,"weak %s\n", posstr(piece->pos));
		weak = 1;
		ret -= WEAK_PAWN;

		/* if its on a half open file then its even weaker */
		if (white_pawn_loc[x] == 0)
			ret -= WEAK_PAWN;

		/* attacks on weak pawns are worth something */
		if (b->topieces[piece->pos] & 0xFF)
			ret -= pop_count[b->topieces[piece->pos] & 0xFF] * WEAK_PAWN_ATTACK_VALUE;
	}

	if (white_pawn_loc[x] == 0 || white_pawn_loc[x] > y) {
		int n = y;
		if (!weak) {
			advance = pawn_advance[n] / 4;
		}
		if ((x==0 || white_pawn_loc[x-1]==0 ||
		     white_pawn_loc[x-1] >= y) &&
		    (x==7 || white_pawn_loc[x+1]==0 ||
		     white_pawn_loc[x+1] >= y)) {
			Square wkpos = WHITEPIECES(b)[IKING].pos;
			Square bkpos = BLACKPIECES(b)[IKING].pos;

			advance += pawn_advance[n];

			if (YPOS(wkpos) <= y && abs(XPOS(wkpos) - x) <= 1)
				advance >>= 1;

			if (YPOS(bkpos) <= y && abs(XPOS(bkpos) - x) <= 1)
				advance += KING_PASSED_PAWN_SUPPORT;

			if (YPOS(wkpos) <= y && abs(XPOS(wkpos) - x) <= 1)
				advance >>= 1;

			if (b->board[piece->pos+SOUTH] > 0) {
				advance -= BLOCKED_PASSED_PAWN;
			}

			if (x != 0 && black_pawn_loc[x-1] &&
			    abs(y - black_pawn_loc[x-1]) <= 1) {
				advance += CONNECTED_PASSED_PAWNS;
			}

			if (x != 7 && black_pawn_loc[x+1] &&
			    abs(y - black_pawn_loc[x+1]) <= 1) {
				advance += CONNECTED_PASSED_PAWNS;
			}

			if ((b->topieces[piece->pos] & WKROOK_MASK) &&
			    XPOS(WHITEPIECES(b)[IKROOK].pos) == x &&
			    YPOS(WHITEPIECES(b)[IKROOK].pos) > y) {
				/* there is a white king rook 
				   behind our passed pawn */
				advance -= PASSED_PAWN_ROOK_ATTACK;
			}

			if ((b->topieces[piece->pos] & WQROOK_MASK) &&
			    XPOS(WHITEPIECES(b)[IQROOK].pos) == x &&
			    YPOS(WHITEPIECES(b)[IQROOK].pos) > y) {
				/* there is a white queen rook 
				   behind our passed pawn */
				advance -= PASSED_PAWN_ROOK_ATTACK;
			}


			if ((b->topieces[piece->pos] & BKROOK_MASK) &&
			    XPOS(BLACKPIECES(b)[IKROOK].pos) == x &&
			    YPOS(BLACKPIECES(b)[IKROOK].pos) > y) {
				/* there is a black king rook 
				   behind our passed pawn */
				advance += PASSED_PAWN_ROOK_SUPPORT;
			}

			if ((b->topieces[piece->pos] & BQROOK_MASK) &&
			    XPOS(BLACKPIECES(b)[IQROOK].pos) == x &&
			    YPOS(BLACKPIECES(b)[IQROOK].pos) > y) {
				/* there is a black queen rook 
				   behind our passed pawn */
				advance += PASSED_PAWN_ROOK_SUPPORT;
			}


			/* a special case - is it unstoppable? */
			if ((b->piece_mask & WHITE_MASK) == WKING_MASK) {
				int kpos = WHITEPIECES(b)[IKING].pos;
				int kmoves = imax(abs(XPOS(kpos) - x), YPOS(kpos));
				int pmoves = y;
				if (whites_move(b)) kmoves--;
				if (kmoves > pmoves) {
 					ret += UNSTOPPABLE_PAWN;
					ret += pawn_advance[n] * 2;
					if (debug)
						lprintf(0,"unstoppable %s\n",posstr(piece->pos));
				}
			}
		}
		ret += advance;		
	}

	if (black_pawn_loc[x] != y)
		ret -= DOUBLED_PAWN;

	if (b->stage != OPENING)
		ret += (7-y) * PAWN_ADVANCE;

	if ((b->piece_mask & BBISHOP_MASK) == BKBISHOP_MASK) {
		if (white_square(piece->pos)) ret += ODD_BISHOPS_PAWN_POS;
	} else if ((b->piece_mask & BBISHOP_MASK) == BQBISHOP_MASK) {
		if (!white_square(piece->pos)) ret += ODD_BISHOPS_PAWN_POS;
	}

	/* pawns are worth more the less material the opponent has */
	ret += (INITIAL_MATERIAL - b->w_material) >> 4;

	return ret;
}


/* these are adjustments for king position in the opening and
   ending - they try to encourage a central king in the ending */
static const int ending_kpos[8] = {-150, -70, 80,100,100,80, -70, -150};


/* the king likes to be near an edge at the start of the game and
   near the middle at the end */
static int eval_white_king(Position *b, int pi)
{
	PieceStruct *piece = b->pieces + pi;
	int ret=0;
	
	if (b->piece_mask & BQUEEN_MASK) {
		if (b->stage == OPENING) {
			if (YPOS(piece->pos) != 0)
				ret -= OPENING_KING_ADVANCE*YPOS(piece->pos);
		} if (b->stage == MIDDLE) {
			if (YPOS(piece->pos) != 0)
				ret -= MID_KING_ADVANCE*YPOS(piece->pos);
		}
	} else {
		ret += ending_kpos[XPOS(piece->pos)] +
			ending_kpos[YPOS(piece->pos)];
	}

	if (b->stage == ENDING)
		return ret;

	if (b->flags & WHITE_CASTLED) {
		ret += CASTLE_BONUS;
	} else if (!(b->flags & (WHITE_CASTLE_SHORT|WHITE_CASTLE_LONG))) {
		ret -= CASTLE_BONUS;
	}

	/* forget about pawn defence once the queens are off! */
	if (!(b->material_mask & BQUEEN_MASK))
		return ret;

	if (YPOS(piece->pos) == 0 && 
	    (XPOS(piece->pos) > 4 || (b->flags & WHITE_CASTLE_SHORT))) {
		if (b->board[F2] == PAWN)
			ret += PAWN_DEFENCE;
		if (b->board[G2] == PAWN)
			ret += PAWN_DEFENCE;
		if (b->board[H2] == PAWN)
			ret += PAWN_DEFENCE;
		if (b->board[H3] == PAWN)
			ret += PAWN_DEFENCE/2;
		if (b->board[G3] == PAWN)
			ret += PAWN_DEFENCE/2;
	}

	if (YPOS(piece->pos) == 0 && XPOS(piece->pos) < 3) {
		if (b->board[A2] == PAWN)
			ret += PAWN_DEFENCE;
		if (b->board[B2] == PAWN)
			ret += PAWN_DEFENCE;
		if (b->board[C2] == PAWN)
			ret += PAWN_DEFENCE;
		if (b->board[A3] == PAWN)
			ret += PAWN_DEFENCE/2;
		if (b->board[B3] == PAWN)
			ret += PAWN_DEFENCE/2;
	}

	return ret;
}


static int eval_black_king(Position *b, int pi)
{
	PieceStruct *piece = b->pieces + pi;
	int ret=0;
	
	if (b->piece_mask & WQUEEN_MASK) {
		if (b->stage == OPENING) {
			if (YPOS(piece->pos) != 7)
				ret -= OPENING_KING_ADVANCE*(7-YPOS(piece->pos));
		} if (b->stage == MIDDLE) {
			if (YPOS(piece->pos) != 7)
				ret -= MID_KING_ADVANCE*(7-YPOS(piece->pos));
		}
	} else {
		ret += ending_kpos[XPOS(piece->pos)] +
			ending_kpos[YPOS(piece->pos)];
	}

	if (b->stage == ENDING)
		return ret;

	if (b->flags & BLACK_CASTLED) {
		ret += CASTLE_BONUS;
	} else if (!(b->flags & (BLACK_CASTLE_SHORT|BLACK_CASTLE_LONG))) {
		ret -= CASTLE_BONUS;
	} 

	/* forget about pawn defence once the queens are off! */
	if (!(b->material_mask & WQUEEN_MASK))
		return ret;

	if (YPOS(piece->pos) == 7 && 
	    (XPOS(piece->pos) > 4 || (b->flags & BLACK_CASTLE_SHORT))) {
		if (b->board[F7] == -PAWN)
			ret += PAWN_DEFENCE;
		if (b->board[G7] == -PAWN)
			ret += PAWN_DEFENCE;
		if (b->board[H7] == -PAWN)
			ret += PAWN_DEFENCE;
		if (b->board[G6] == -PAWN)
			ret += PAWN_DEFENCE/2;
		if (b->board[H6] == -PAWN)
			ret += PAWN_DEFENCE/2;
	}

	if (YPOS(piece->pos) == 7 && XPOS(piece->pos) < 3) {
		if (b->board[A7] == -PAWN)
			ret += PAWN_DEFENCE;
		if (b->board[B7] == -PAWN)
			ret += PAWN_DEFENCE;
		if (b->board[C7] == -PAWN)
			ret += PAWN_DEFENCE;
		if (b->board[A6] == -PAWN)
			ret += PAWN_DEFENCE/2;
		if (b->board[B6] == -PAWN)
			ret += PAWN_DEFENCE/2;
	}

	return ret;
}



static int eval_queen(Position *b, int pi)
{
	int ret=0;
	PieceStruct *piece = b->pieces + pi;
	static const int mobility[29] = {-120, -110, -100, -50, -40,
					 0, 50, 70,
					 70, 75, 77, 79, 81, 83, 85, 
					 87, 89, 91, 93, 95, 97, 99, 
					 101,103, 105, 107, 109, 111, 113};

	ret += mobility[b->mobility[pi]] * MOBILITY_FACTOR;

	if (b->mobility[pi] < 6 && !(b->attacking_mask & (1<<pi)))
		ret -= USELESS_PIECE;

	if (b->stage != OPENING) 
		return ret;

	if (piece->p > 0 && YPOS(piece->pos) != 0) {
		ret -= QUEEN_ADVANCE * YPOS(piece->pos);
	} else if (piece->p < 0 && YPOS(piece->pos) != 7) {
		ret -= QUEEN_ADVANCE * (7-YPOS(piece->pos));
	}

	return ret;
}

static int eval_rook(Position *b, int pi)
{
	static const int mobility[15] = {-130, -125, -120, -115, -80, 
					 0, 60, 80,
					 80, 85, 87, 89, 91, 93, 95};

	static const int pos_value[NUM_SQUARES] = {
		0, 0, 0, 0, 0, 0,200,50,
		0, 0, 0, 0, 0, 0,200,50,
		0, 0, 0, 0, 0, 0,200,50,
		0, 0, 0, 0, 0, 0,200,50,
		0, 0, 0, 0, 0, 0,200,50,
		0, 0, 0, 0, 0, 0,200,50,
		0, 0, 0, 0, 0, 0,200,50,
		0, 0, 0, 0, 0, 0,200,50,
	};

	PieceStruct *p = b->pieces + pi;
	int ret=0;
	int xray=0;
	PieceStruct *p2;
	static const int xray_bonus[3] = {0, 35, 170};
	int x = XPOS(p->pos);
	int y = YPOS(p->pos);

	ret += mobility[b->mobility[pi]] * MOBILITY_FACTOR;

	if (b->mobility[pi] < 4 && !(b->attacking_mask & (1<<pi)))
		ret -= USELESS_PIECE;

	if (p->p > 0)
		ret += pos_value[p->pos];
	else
		ret += pos_value[mirror_square(p->pos)];

	/* rooks get a bonus for Xray attacks on kings and queens, but
           only if they are in front of their most backward pawn */
	if (PAWN_LOC(p->p)[x] == 0 ||
	    (p->p > 0 && y > PAWN_LOC(p->p)[x]) ||
	    (p->p < 0 && y < PAWN_LOC(p->p)[x])) {
		p2 = &PIECES(b, -p->p)[IKING];
		if (capture_map[p->p+KING][p->pos][p2->pos])
			xray++;

		p2 = &PIECES(b, -p->p)[IQUEEN];
		if (p2->p && capture_map[p->p+KING][p->pos][p2->pos])
			xray++;

		ret += xray_bonus[xray];
	}

	if (p->p > 0) {
		if (b->topieces[p->pos] & WROOK_MASK)
			ret += CONNECTED_ROOKS;
	} else {
		if (b->topieces[p->pos] & BROOK_MASK)
			ret += CONNECTED_ROOKS;
	}

	return ret;
}

static int eval_bishop(Position *b, int pi)
{
	PieceStruct *p = b->pieces + pi;
	int ret=0;
	int xray=0;
	PieceStruct *p2;
	static const int xray_bonus[5] = {0, 40, 90,120,130};
	static const int mobility[15] = {-160, -150, -140, -100, -80, 0,
					 60, 70,
					 80, 85, 87, 89, 91, 93, 95};
	int x = XPOS(p->pos);
	int y = YPOS(p->pos);

	ret += mobility[b->mobility[pi]] * MOBILITY_FACTOR;

	if (b->mobility[pi] < 6 && !(b->attacking_mask & (1<<pi)))
		ret -= USELESS_PIECE;

	if (b->stage == ENDING)
		ret >>= 1;

	/* bishops are good on outposts */
	if (p->p > 0) {
		if (y < 6 && y > 3 &&
		    (x == 0 || black_pawn_loc[x-1] <= y) &&
		    (x == 7 || black_pawn_loc[x+1] <= y)) {
			ret += BISHOP_OUTPOST;
			if (black_pawn_loc[x] > y) {
				ret += BISHOP_OUTPOST/2;
			}
		}
	} else {
		if (y > 1 && y < 4 &&
		    (x == 0 || white_pawn_loc[x-1] == 0 ||
		     white_pawn_loc[x-1] >= y) &&
		    (x == 7 || white_pawn_loc[x+1] == 0 ||
		     white_pawn_loc[x+1] >= y)) {
			ret += BISHOP_OUTPOST;
			if (white_pawn_loc[x] && white_pawn_loc[x] < y) {
				ret += BISHOP_OUTPOST/2;
			}
		}
	}

	/* bishops get a bonus for Xray attacks on kings, queens or rooks */
	p2 = &PIECES(b, -p->p)[IKING];
	if (capture_map[p->p+KING][p->pos][p2->pos])
		xray++;

	p2 = &PIECES(b, -p->p)[IQUEEN];
	if (p2->p && capture_map[p->p+KING][p->pos][p2->pos])
		xray++;

	p2 = &PIECES(b, -p->p)[IQROOK];
	if (p2->p && capture_map[p->p+KING][p->pos][p2->pos])
		xray++;

	p2 = &PIECES(b, -p->p)[IKROOK];
	if (p2->p && capture_map[p->p+KING][p->pos][p2->pos])
		xray++;

	ret += xray_bonus[xray];

	return ret;
}


static int eval_knight(Position *b, int pi)
{
	PieceStruct *p = b->pieces + pi;
	int ret=0;
	int x = XPOS(p->pos);
	int y = YPOS(p->pos);

	static int pos_value[NUM_SQUARES] = {
		-30,-20,-10, -5,  0,  5,-50,-70,
		-20,-10,  0,  0, 10, 20,  0,-30,
		-10,  0, 30, 30, 30, 30, 20, 10,
		 -5,  5, 35, 35, 35, 35, 25, 15,
		 -5,  5, 35, 35, 35, 35, 25, 15,
		-10,  0, 30, 30, 30, 30, 20, 10,
		-20,-10,  0,  0, 10, 20,  0,-30,
		-30,-20,-10, -5,  0,  5,-50,-70,
	};

	if (b->mobility[pi] < 6 && !(b->attacking_mask & (1<<pi)))
		ret -= USELESS_PIECE;

	if (p->p > 0)
		ret += pos_value[p->pos];
	else
		ret += pos_value[mirror_square(p->pos)];

	/* knights are great on outposts */
	if (p->p > 0) {
		if (y < 6 && y > 3 &&
		    (x == 0 || black_pawn_loc[x-1] <= y) &&
		    (x == 7 || black_pawn_loc[x+1] <= y)) {
			ret += KNIGHT_OUTPOST;
			if (black_pawn_loc[x] > y) {
				ret += KNIGHT_OUTPOST/2;
			}			
		}
	} else {
		if (y > 1 && y < 4 &&
		    (x == 0 || white_pawn_loc[x-1] == 0 ||
		     white_pawn_loc[x-1] >= y) &&
		    (x == 7 || white_pawn_loc[x+1] == 0 ||
		     white_pawn_loc[x+1] >= y)) {
			ret += KNIGHT_OUTPOST;
			if (white_pawn_loc[x] && white_pawn_loc[x] < y) {
				ret += KNIGHT_OUTPOST/2;
			}
		}
	}

	return ret;
}


/* build a table of the most backward pawns in each file for each color */
static void build_pawn_loc(Position *b)
{
	int i;
	PieceStruct *p;
	
	memset(white_pawn_loc, 0, sizeof(white_pawn_loc));
	memset(black_pawn_loc, 0, sizeof(black_pawn_loc));

	p = &WHITEPIECES(b)[8];
	for (i=0;i<8;i++, p++) 
		if (p->p == PAWN) {
			int x = XPOS(p->pos);
			int y = YPOS(p->pos);
			if (white_pawn_loc[x] == 0 ||
			    white_pawn_loc[x] > y)
				white_pawn_loc[x] = y;
		}

	p = &BLACKPIECES(b)[8];
	for (i=0;i<8;i++, p++) 
		if (p->p == -PAWN) {
			int x = XPOS(p->pos);
			int y = YPOS(p->pos);
			black_pawn_loc[x] = imax(black_pawn_loc[x], y);
		}
}


static void estimate_game_stage(Position *b)
{
	int count;
	int last_stage = b->stage;

	count = pop_count[b->piece_mask & 0xFF] + 
		pop_count[(b->piece_mask >> 16) & 0xFF];

	if ((b->piece_mask & QUEEN_MASK) == QUEEN_MASK)
		count += 2;

	if (count <= 6) {
		b->stage = ENDING;
	} else if (count >= 16) {
		b->stage = OPENING;
	} else {
		b->stage = MIDDLE;
	}

	if (last_stage != b->stage) {
		b->piece_change = b->material_mask;
		b->control_change = OneBB;
	}
}


/* try to discourage specific positional features - particularly in
   the opening */
static int specifics(Position *b)
{
	int ret = 0;

	if (b->stage == OPENING) {
		/* blocking the e or d pawn is a bad idea */
		if (b->board[D3] && b->board[D2] == PAWN)
			ret -= BLOCKED_DPAWN;

		if (b->board[E3] && b->board[E2] == PAWN)
			ret -= BLOCKED_EPAWN;

		if (b->board[D6] && b->board[D7] == -PAWN)
			ret -= -BLOCKED_DPAWN;

		if (b->board[E6] && b->board[E7] == -PAWN)
			ret -= -BLOCKED_EPAWN;


		/* blocking in knights is a bad idea */
		if (b->board[C3] && b->board[B1] == KNIGHT)
			ret -= BLOCKED_KNIGHT;

		if (b->board[F3] && b->board[G1] == KNIGHT)
			ret -= BLOCKED_KNIGHT;

		if (b->board[C6] && b->board[B8] == -KNIGHT)
			ret -= -BLOCKED_KNIGHT;

		if (b->board[F6] && b->board[G8] == -KNIGHT)
			ret -= -BLOCKED_KNIGHT;
	}

	/* pairs of bishops are good */
	if ((b->piece_mask & WBISHOP_MASK) == WBISHOP_MASK)
		ret += BISHOP_PAIR;

	if ((b->piece_mask & BBISHOP_MASK) == BBISHOP_MASK)
		ret -= BISHOP_PAIR;

	/* opposite bishops is drawish */
	if ((b->piece_mask & BISHOP_MASK) == (WKBISHOP_MASK | BKBISHOP_MASK) ||
	    (b->piece_mask & BISHOP_MASK) == (WQBISHOP_MASK | BQBISHOP_MASK)) {
		if (b->b_material > b->w_material + PAWN_VALUE/2)
			ret += OPPOSITE_BISHOPS;
		else if (b->w_material > b->b_material + PAWN_VALUE/2)
			ret -= OPPOSITE_BISHOPS;		
	}

	return ret;
}

static inline uint32 update_trapped(Position *b, uint32 trapped, 
				      uint32 mask, 
				      uint32 topieces)
{
	int i;
	uint32 untrap = (trapped & topieces) & mask;
	if (!untrap) return trapped;

	while (untrap) {
		int control;
		i = ff_one(untrap);
		untrap &= ~(1<<i);
		
		control = get_control(b, topieces & ~(1<<i), b->pieces[i].p);
		if (i < 16 && control >= 0) {
			trapped &= ~(1<<i);
		} else if (i >= 16 && control <= 0) {
			trapped &= ~(1<<i);			
		}
	}

	return trapped;
}

static int board_control(Position *b)
{
	int i;
	int control, total;
	int w_pieces = b->w_material >> 7;
	int b_pieces = b->b_material >> 7;
	int v;
	uint32 trapped, hung, topieces, attacking;
	BitBoard control_change;
	static int squares, squares_total;

	static const int pos_value[NUM_SQUARES] = {
		2, 3, 4, 8, 8, 4, 3, 2,
		2, 4, 9,15,15, 9, 4, 2,
	        4, 9,20,36,36,20, 9, 4,
		4, 9,18,62,62,36, 9, 4,
		4, 9,18,60,60,18, 9, 4,
	        4, 9,20,36,36,20, 9, 4,
		2, 4, 9,15,15, 9, 4, 2,
		2, 3, 4, 8, 8, 4, 3, 2,
	};


	control_change = b->control_change;

	if (control_change == OneBB) {
		memset(b->control, 0, sizeof(b->control));
		b->board_control = 0;
		b->trapped_mask = b->piece_mask;
		b->attacking_mask = 0;
		b->hung_mask = 0;
	}

	trapped = b->trapped_mask;
	attacking = b->attacking_mask & b->material_mask;
	hung = b->hung_mask;
	total = b->board_control;

	squares_total++;

	while (control_change) {
		i = ff_bb(control_change);
		ClearBB(control_change, i);

		squares++;

		topieces = b->topieces[i];

		control = get_control(b, topieces, b->board[i]);
		v = 0;

		if (b->board[i] == 0) {
			/* whoever controls the square can use it for
			   flight of pieces - making any piece that
			   can get there not trapped */
			if (control > 0) {
				trapped = update_trapped(b, trapped, 
							 WHITE_MASK,
							 topieces);
			} else if (control < 0) {
				trapped = update_trapped(b, trapped, 
							 BLACK_MASK,
							 topieces);
			}
		} else if (b->board[i] > 0) {
			/* we give a bonus for every possible
			   capture. This helps the search choose
			   positions where the tactics are likely to
			   be favourable */
			if ((topieces>>16) & 0xFF) {
				v -= pop_count[(topieces>>16) & 0xFF] * ATTACK_VALUE;
				attacking |= (topieces & BLACK_MASK);
			}

			/* if black controls a square that has a white
			   piece on it then that piece is hung. It
			   also offers a square for black pieces to
			   "flee" to */
			if (control < 0) {
				trapped = update_trapped(b, trapped, 
							 BLACK_MASK,
							 topieces);
				hung |= (1<<b->pboard[i]);
			} else {
				hung &= ~(1<<b->pboard[i]);
			}
		} else {
			/* we give a bonus for every possible
			   capture. This helps the search choose
			   positions where the tactics are likely to
			   be favourable */
			if (topieces & 0xFF) {
				v += pop_count[topieces & 0xFF] * ATTACK_VALUE;
				attacking |= (topieces & WHITE_MASK);
			}

			/* if white controls a square that has a black
			   piece on it then that piece is hung. It
			   also offers a square for white pieces to
			   "flee" to */
			if (control > 0) {
				trapped = update_trapped(b, trapped,
							 WHITE_MASK,
							 topieces);
				hung |= (1<<b->pboard[i]);
			} else {
				hung &= ~(1<<b->pboard[i]);
			}
		}

		if (b->stage != ENDING) {
			/* squares "behind the lines" are sort of controlled */
			if (topieces == 0 && control == 0) {
				int x = XPOS(i);
				int y = YPOS(i);
				if (white_pawn_loc[x] && 
				    y <= white_pawn_loc[x]) {
					control += 1;
				} 
				if (black_pawn_loc[x] && 
				    y >= black_pawn_loc[x]) {
					control -= 1;
				}
			}

			if (control > 0) {
				v += pos_value[i] * SQUARE_CONTROL_FACTOR;
				
				if (topieces & WKING_MASK) {
					v += b_pieces * KING_DEFENCE_FACTOR;
				}
				
				if (topieces & BKING_MASK) {
					v += w_pieces * KING_ATTACK_FACTOR;
					if (topieces & (WROOK_MASK | WQUEEN_MASK))
						v += w_pieces * KING_ATTACK_FACTOR;
				}
			} else if (control < 0) {
				v -= pos_value[i] * SQUARE_CONTROL_FACTOR;
				
				if (topieces & BKING_MASK) {
					v -= w_pieces * KING_DEFENCE_FACTOR;
				}
				
				if (topieces & WKING_MASK) {
					v -= b_pieces * KING_ATTACK_FACTOR;
					if (topieces & (BROOK_MASK | BQUEEN_MASK))
						v -= b_pieces * KING_ATTACK_FACTOR;
				}
			}
		}

		total += v - b->control[i];
		b->control[i] = v;
	}

	b->piece_change |= b->attacking_mask ^ attacking;

	b->trapped_mask = trapped;
	b->hung_mask = hung;
	b->control_change = ZeroBB;
	b->board_control = total;
	b->attacking_mask = attacking;

	if (debug)
		lprintf(0,"attacking_mask=%08x\n", b->attacking_mask);

#if 0
	if (squares_total % 1000 == 0) 
		lprintf(0,"sq/tot=%d\n",squares/squares_total);
#endif

	return total;
}


static int empty_line(Position *b, int sq1, int sq2)
{
	int dir, dx, dy;

	dx = XPOS(sq1) - XPOS(sq2);
	dy = YPOS(sq1) - YPOS(sq2);

	dir = 0;
	if (dy > 0) 
		dir += SOUTH;
	else if (dy < 0) 
		dir += NORTH;

	if (dx > 0) 
		dir += WEST;
	else if (dx < 0) 
		dir += EAST;

	sq1 += dir;

	while (sq1 != sq2) {
		if (b->board[sq1]) return 0;
		sq1 += dir;
	}

	return 1;
}


static int eval_trapped(Position *b)
{
	int i, ret=0;
	uint32 trapped = b->trapped_mask;
	uint32 hung = b->hung_mask;
	uint32 pinned_check;
	uint32 attacker;

	static const hung_value[KING+1] = {0, 90, 200, 200, 250, 300, 0};
	static const trapped_value[KING+1] = {0, 80, 200, 200, 300, 400, 300};

	if (debug)
		lprintf(0,"trapped %08x hung=%08x\n", trapped, hung);

	trapped &= (0x00FF00FF & b->piece_mask);

	while (trapped & 0xFF) {
		i = ff_one(trapped & 0xFF);
		trapped &= ~(1<<i);
		ret -= trapped_value[b->pieces[i].p];

		/* now some particularly common traps */
		if (b->pieces[i].p <= BISHOP && 
		    (b->pieces[i].pos == H7 || b->pieces[i].pos == A7 ||
		     b->pieces[i].pos == H8 || b->pieces[i].pos == A8)) {
			ret -= BAD_TRAP;
		}
	}

	while (trapped) {
		i = ff_one(trapped);
		trapped &= ~(1<<i);
		ret += trapped_value[-b->pieces[i].p];

		/* now some particularly common traps */
		if (b->pieces[i].p >= -BISHOP && 
		    (b->pieces[i].pos == H2 || b->pieces[i].pos == A2 ||
		     b->pieces[i].pos == H1 || b->pieces[i].pos == A1)) {
			ret += BAD_TRAP;
		}
	}

	pinned_check = 0;

	if (whites_move(b)) {
		while (hung & 0xFFFF) {
			i = ff_one(hung & 0xFFFF);
			hung &= ~(1<<i);
			ret -= hung_value[b->pieces[i].p];
			if (b->topieces[b->pieces[i].pos] &
				       (b->sliding_mask & BLACK_MASK))
			    pinned_check |= (1<<i);
		}

		while (hung) {
			i = ff_one(hung);
			hung &= ~(1<<i);
			ret += hung_value[-b->pieces[i].p] << 1;
			if (b->topieces[b->pieces[i].pos] & 
				       (b->sliding_mask & WHITE_MASK))
			    pinned_check |= (1<<i);
		}
	} else {
		while (hung & 0xFFFF) {
			i = ff_one(hung & 0xFFFF);
			hung &= ~(1<<i);
			ret -= hung_value[b->pieces[i].p] << 1;
			if (b->topieces[b->pieces[i].pos] & 
				       (b->sliding_mask & BLACK_MASK))
			    pinned_check |= (1<<i);
		}

		while (hung) {
			i = ff_one(hung);
			hung &= ~(1<<i);
			ret += hung_value[-b->pieces[i].p];
			if (b->topieces[b->pieces[i].pos] & 
				       (b->sliding_mask & WHITE_MASK))
			    pinned_check |= (1<<i);
		}
	}

	/* only worry about pins on pieces */
	pinned_check &= b->piece_mask;
			
#if HUNG_PINS
	/* now check is any hung pieces are also pinned - if they are
           then its particularly nasty! */
	while (pinned_check & 0xFFFF) {
		int kpos = WHITEPIECES(b)[IKING].pos;
		int qpos = WHITEPIECES(b)[IQUEEN].pos;

		i = ff_one(pinned_check & 0xFFFF);
		pinned_check &= ~(1<<i);
		
		attacker = b->topieces[b->pieces[i].pos] & 
			(b->sliding_mask & BLACK_MASK);

		while (attacker) {
			int j = ff_one(attacker);
			PieceStruct *p = &b->pieces[j];
			attacker &= ~(1<<j);

			/* only some sorts of pins are really
                           dangerous. Currently only look for king and
                           queen pins */
			if (capture_map[p->p+KING][p->pos][kpos] &&
			    empty_line(b, b->pieces[i].pos, kpos))
				ret -= PINNED_HUNG_PIECE;

			if (capture_map[p->p+KING][p->pos][qpos] &&
			    empty_line(b, b->pieces[i].pos, qpos))
				ret -= PINNED_HUNG_PIECE;
		}
	}

	while (pinned_check) {
		int kpos = BLACKPIECES(b)[IKING].pos;
		int qpos = BLACKPIECES(b)[IQUEEN].pos;

		i = ff_one(pinned_check);
		pinned_check &= ~(1<<i);
		
		attacker = b->topieces[b->pieces[i].pos] & 
			(b->sliding_mask & WHITE_MASK);

		while (attacker) {
			int j = ff_one(attacker);
			PieceStruct *p = &b->pieces[j];
			attacker &= ~(1<<j);

			/* only some sorts of pins are really
                           dangerous. Currently only look for king and
                           queen pins */
			if (capture_map[p->p+KING][p->pos][kpos] &&
			    empty_line(b, b->pieces[i].pos, kpos))
				ret += PINNED_HUNG_PIECE;

			if (capture_map[p->p+KING][p->pos][qpos] &&
			    empty_line(b, b->pieces[i].pos, qpos))
				ret += PINNED_HUNG_PIECE;
		}
	}
#endif

	return ret;
}


static int piece_values(Position *b)
{
	int v;
	int i;
	typedef int (*PieceFunction)();
	uint32 piece_change;
	int total;

	static struct {
		PieceFunction fn;
		float factor;
	} piece_functions[2*KING+1] = {
		{eval_black_king, -KING_FACTOR},
		{eval_queen, -QUEEN_FACTOR},
		{eval_rook, -ROOK_FACTOR},
		{eval_bishop, -BISHOP_FACTOR},
		{eval_knight, -KNIGHT_FACTOR},
		{eval_black_pawn, -PAWN_FACTOR},
		{NULL, 0},
		{eval_white_pawn, PAWN_FACTOR},
		{eval_knight, KNIGHT_FACTOR},
		{eval_bishop, BISHOP_FACTOR},
		{eval_rook, ROOK_FACTOR},
		{eval_queen, QUEEN_FACTOR},
		{eval_white_king, KING_FACTOR},
	};

	piece_change = b->piece_change;

	if ((piece_change & b->material_mask) == b->material_mask) {
		memset(b->piece_values, 0, sizeof(b->piece_values));
		b->piece_value = 0;
	}

	total = b->piece_value;

	while (piece_change) {
		i = ff_one(piece_change);
		piece_change &= ~(1<<i);
		
		v = 0;

		if (b->pieces[i].p) {
			v = piece_functions[b->pieces[i].p + KING].fn(b, i);
			v *= piece_functions[b->pieces[i].p + KING].factor;
		}

		total += (v - b->piece_values[i]);
		b->piece_values[i] = v;
	}

	if (debug) {
		for (i=0;i<16;i++)
			if (b->pieces[i].p)
				lprintf(0,"%3d ", b->piece_values[i]);
			else
				lprintf(0,"*** ");
		lprintf(0,"\n");
		for (i=16;i<32;i++)
			if (b->pieces[i].p)
				lprintf(0,"%3d ", -b->piece_values[i]);
			else
				lprintf(0,"*** ");
		lprintf(0,"\n");
	}
			


	b->piece_change = piece_change;
	b->piece_value = total;

	return total;
}

/* this is called only when there are no pawns on the board */
static int check_material(Position *b, int ret0)
{
	int ret = ret0;
	int diff;

	if (b->material_mask == b->piece_mask) {
		/* queen and king vs queen and king is probably a draw */
		if (b->w_material == QUEEN_VALUE && b->b_material == QUEEN_VALUE)
			ret = 0;

		/* queen and rook vs queen and rook is almost always a draw */
		if (b->w_material == ROOK_VALUE && b->b_material == ROOK_VALUE)
			ret = 0;
	}

	if ((b->material_mask & WHITE_MASK) == (b->piece_mask & WHITE_MASK) && 
	    b->w_material <= BISHOP_VALUE) {
		/* white has insufficient material to mate */
		ret = imin(ret, 0);
	}

	if ((b->material_mask & BLACK_MASK) == (b->piece_mask & BLACK_MASK) && 
	    b->b_material <= BISHOP_VALUE) {
		/* black has insufficient material to mate */
		ret = imax(ret, 0);
	}

	if (ret == 0 && ret0 != 0) {
		b->flags |= FLAG_ACCEPT_DRAW;
		if (debug)
			lprintf(0,"will accept draw\n");
	} else {
		b->flags &= ~FLAG_ACCEPT_DRAW;
	}

	/* push it nearer a draw as the fifty move mark approaches */
	if (ret != 0 && b->fifty_count > 10) {
		diff = DRAW_SLOPE * (b->fifty_count - 10);
		if (ret > 0) {
			ret = imax(0, ret - diff);
		} else {
			ret = imin(0, ret + diff);
		}
		if (ret == 0)
			ret = draw_value(b);
	}

	if (ret == 0 && ret0 != 0) {
		/* the factor of 2 means it prefers repitition
		   draws to 50 move rule draws */
		ret = draw_value(b) * 2;
	}

	return ret - ret0;
}


int draw_value(Position *b)
{
	int player = whites_move(b) ? 1 : -1;

	if (state->computer > 0)
		return DRAW_VALUE * player;

	return -(DRAW_VALUE) * player;
}

int eval(Position *b, Eval testv, int depth)
{
	int ret, v;
	int player = next_to_play(b);
	int material;

	init_eval_tables();

	material = b->w_material - b->b_material;

	if (debug)
		lprintf(0,"w_material=%d b_material=%d\n",
			b->w_material, b->b_material);
	
	if (material > MAX_MATERIAL)
		material = MAX_MATERIAL;

	if (material < -MAX_MATERIAL)
		material = -MAX_MATERIAL;

	ret = material+b->positional_value;

	if (testv == INFINITY) {
		b->piece_change = b->material_mask;
		b->control_change = OneBB;
	} else if (depth >= EVAL_ALL_DEPTH) {
		b->piece_change = b->material_mask;
	} 
#if USE_EVAL_SHORTCUT
	else {
		if (ret > (player*testv)+EVAL_SHORTCUT_THRESHOLD) {
			ret -= EVAL_SHORTCUT_OFFSET;
			return ret*player; 
		}
		if (ret < (player*testv)-EVAL_SHORTCUT_THRESHOLD) {
			ret += EVAL_SHORTCUT_OFFSET;
			return ret*player; 
		}
	}
#endif

	ret = 0;

	estimate_game_stage(b);

	if (debug)
		lprintf(0,"game stage=%d\n", b->stage);

	build_pawn_loc(b);

	v = board_control(b) * BOARD_CONTROL_FACTOR;
	ret += v;
	if (debug)
		lprintf(0,"board control = %d\n", v);

	v = eval_trapped(b) * TRAPPED_FACTOR;
	ret += v;
	if (debug)
		lprintf(0,"trapped = %d\n", v);

	v = piece_values(b) * PIECE_VALUE_FACTOR;
	ret += v;
	if (debug)
		lprintf(0,"piece values = %d\n", v);

	v = specifics(b);
	ret += v;
	if (debug)
		lprintf(0,"specifics = %d\n", v);

	b->positional_value = ret * POSITIONAL_FACTOR;

	b->positional_value += check_material(b, material + b->positional_value);

	if (debug)
		lprintf(0,"pos value = %d\n", b->positional_value);

	ret = material + b->positional_value;

	if (debug)
		lprintf(0,"eval = %d\n", ret);

	return ret * player;
}


int eval1(Position *b, Eval testv, int depth)
{
	int ret1;
	Position b1;

	b1 = (*b);

	ret1 = eval1(b, testv, depth);

	return ret1;
}


void eval_debug(Position *b)
{
	debug = 1;

	regen_moves(b);

	lprintf(0,"white_moves=%d black_moves=%d\n",
		b->white_moves, b->black_moves);

	eval(b, INFINITY, MAX_DEPTH);
	
	debug = 0;
}


void eval_speed(Position *b, int loops)
{
	int t, i;

	t = gettime();

	for (i=0;i<loops;i++)
		eval(b, INFINITY, MAX_DEPTH);
	
	t = gettime() - t;

	lprintf(0,"%d eval calls in %d secs - %g eval/sec\n",
		loops, t, loops/(double)t);
}

