#include "includes.h"
#include "knightcap.h"

#define ADD_GEN(to) \
{ \
	b->topieces[to] |= mask; \
	SetBB(b->control_change,to); \
	n++; \
}

static int pawn_generate(Position *b, int pi)
{
	int n = 0;
	uint32 mask = (1<<pi);
	PieceStruct *p = &b->pieces[pi];
	Square from = p->pos;
	int x = XPOS(from);

	if (p->p > 0) {
		/* white pawns */
		if (x != 0) {
			ADD_GEN(from + NORTH_WEST);
		}
		if (x != 7) {
			ADD_GEN(from + NORTH_EAST);
		}
	} else {
		/* black panws */
		if (x != 0) {
			ADD_GEN(from + SOUTH_WEST);
		}
		if (x != 7) {
			ADD_GEN(from + SOUTH_EAST);
		}
	}

	return n;
}


static int bishop_generate(Position *b, int pi)
{
	int n = 0;
	uint32 mask = (1<<pi);
	PieceStruct *p = &b->pieces[pi];
	Square from = p->pos;
	int x = XPOS(from);
	int y = YPOS(from);       
	int l;
	Square to;

	l = imin(7-x, 7-y);
	to = from + NORTH_EAST;

	while (l--) {
		ADD_GEN(to);
		if (b->board[to]) break;
		to += NORTH_EAST;
	}

	l = imin(x, 7-y);
	to = from + NORTH_WEST;

	while (l--) {
		ADD_GEN(to);
		if (b->board[to]) break;
		to += NORTH_WEST;
	}

	l = imin(x, y);
	to = from + SOUTH_WEST;

	while (l--) {
		ADD_GEN(to);
		if (b->board[to]) break;
		to += SOUTH_WEST;
	}

	l = imin(7-x, y);
	to = from + SOUTH_EAST;

	while (l--) {
		ADD_GEN(to);
		if (b->board[to]) break;
		to += SOUTH_EAST;
	}

	return n;
}


static int rook_generate(Position *b, int pi)
{
	int n = 0;
	uint32 mask = (1<<pi);
	PieceStruct *p = &b->pieces[pi];
	Square from = p->pos;
	int x = XPOS(from);
	int y = YPOS(from);       
	int l;
	Square to;

	l = 7-y;
	to = from + NORTH;

	while (l--) {
		ADD_GEN(to);
		if (b->board[to]) break;
		to += NORTH;
	}

	l = x;
	to = from + WEST;

	while (l--) {
		ADD_GEN(to);
		if (b->board[to]) break;
		to += WEST;
	}

	l = y;
	to = from + SOUTH;

	while (l--) {
		ADD_GEN(to);
		if (b->board[to]) break;
		to += SOUTH;
	}

	l = 7-x;
	to = from + EAST;

	while (l--) {
		ADD_GEN(to);
		if (b->board[to]) break;
		to += EAST;
	}

	return n;
}


static int knight_generate(Position *b, int pi)
{
	int n = 0;
	uint32 mask = (1<<pi);
	PieceStruct *p = &b->pieces[pi];
	Square from = p->pos;
	int x = XPOS(from);
	int y = YPOS(from);       
	Square to;

	if (x == 7) goto west;

	if (y<6) {
		to = from + 2*NORTH + EAST;
		ADD_GEN(to);
	}

	if (y>1) {
		to = from + EAST + 2*SOUTH;
		ADD_GEN(to);
	}

	if (x == 6) goto west;

	if (y<7) {
		to = from + NORTH + 2*EAST;
		ADD_GEN(to);
	}

	if (y>0) {
		to = from + SOUTH + 2*EAST;
		ADD_GEN(to);
	}

west:	
	if (x == 0) return n;

	if (y>1) {
		to = from + WEST + 2*SOUTH;
		ADD_GEN(to);
	}

	if (y<6) {
		to = from + WEST + 2*NORTH;
		ADD_GEN(to);
	}

	if (x == 1) return n;

	if (y>0) {
		to = from + 2*WEST + SOUTH;
		ADD_GEN(to);
	}
	
	if (y<7) {
		to = from + 2*WEST + NORTH;
		ADD_GEN(to);
	}

	return n;
}


static int king_generate(Position *b, int pi)
{
	int n = 0;
	uint32 mask = (1<<pi);
	PieceStruct *p = &b->pieces[pi];
	Square from = p->pos;
	int x = XPOS(from);
	int y = YPOS(from);       
	Square to;

	if (x != 0) {
		to = from + WEST;
		ADD_GEN(to);
	}

	if (x != 7) {
		to = from + EAST;
		ADD_GEN(to);
	}

	if (y == 7) goto south;
	
	if (x != 0) {
		to = from + NORTH_WEST;
		ADD_GEN(to);
	}

	if (x != 7) {
		to = from + NORTH_EAST;
		ADD_GEN(to);
	}

	to = from + NORTH;
	ADD_GEN(to);

south:
	if (y == 0) return n;

	if (x != 0) {
		to = from + SOUTH_WEST;
		ADD_GEN(to);
	}

	if (x != 7) {
		to = from + SOUTH_EAST;
		ADD_GEN(to);
	}

	to = from + SOUTH;
	ADD_GEN(to);

	return n;
}


static int queen_generate(Position *b, int pi)
{
	return rook_generate(b, pi) + bishop_generate(b, pi);
}


static int piece_generate(Position *b, int pi)
{
	int n = 0;
	b->piece_change |= (1<<pi);
	b->attacking_mask &= ~(1<<pi);
	b->hung_mask &= ~(1<<pi);

	SetBB(b->control_change,b->pieces[pi].pos);

	switch (abs(b->pieces[pi].p)) {
	case PAWN:
		n = pawn_generate(b, pi);
		break;
		
	case BISHOP:
		n = bishop_generate(b, pi);
		break;
		
	case KNIGHT:
		n = knight_generate(b, pi);
		break;
		
	case ROOK:
		n = rook_generate(b, pi);
		break;
		
	case QUEEN:
		n = queen_generate(b, pi);
		break;
		
	case KING:
		n = king_generate(b, pi);
		break;

	default:
		lprintf(0, "unknown pi %d in piece_generate\n", pi);
		break;
	}

	if (pi >= 16) {
		b->black_moves += (n - b->mobility[pi]);
	} else {
		b->white_moves += (n - b->mobility[pi]);
	}
	b->mobility[pi] = n;

	return n;
}


/* this routine is the core of the move generator. It is incremental,
   the topieces[] array always contains a bitmap of what pieces can move
   to that square. update_moves() is called to update the topieces[]
   array after each move. It is only called for "simple moves", ie not
   enpassent captures or castling */
void update_moves(Position *b, Position *oldb, Move *move)
{
	unsigned i, pi;
	uint32 redo1, redo2, redo3, undo, mask;
	uint32 control_mask;

	control_mask = 0;

	/* pieces that can move to the from square - these moves don't
	   get wiped but they may be augmented if the piece is a
	   sliding piece */
	control_mask |= oldb->topieces[move->from];
	redo1 = oldb->topieces[move->from] & b->sliding_mask; 

	/* pieces that can move to the to square - these need to be
	   wiped and regenerated for sliding pieces, unless the move
	   is a capture */
	control_mask |= oldb->topieces[move->to];
	if (b->capture) {
		redo2 = 0;
	} else {
		redo2 = oldb->topieces[move->to] & b->sliding_mask; 
	}

	/* the piece that is moving - always wipe and regen */
	control_mask |= (1<<b->pboard[move->to]); 
	redo3 = (1<<b->pboard[move->to]); 
	undo = 0;

	/* a piece being captured - always wipe */
	if (oldb->pboard[move->to] != -1) {
		control_mask |= (1<<oldb->pboard[move->to]); 		
		undo = (1<<oldb->pboard[move->to]); 
		b->hung_mask &= ~(1<<oldb->pboard[move->to]);
	}

	mask = ~(redo2 | redo3 | undo);

	SetBB(b->control_change, move->from);
	SetBB(b->control_change, move->to);

	for (i=0;i<NUM_SQUARES;i++) {
		if (oldb->topieces[i] & control_mask) {
			SetBB(b->control_change, i);
		}
		b->topieces[i] = oldb->topieces[i] & mask;
	}

	mask = (redo1 | redo2 | redo3) & ~undo;

	/* now regenerate all moves for any pieces that are affected */
	while (mask) {
		pi = ff_one(mask);
		mask &= ~(1<<pi);
		piece_generate(b, pi);
	} 

	/* ugly. we need to recalc pawns because of changes in doubled,
	   isolated or passed pawns */
	if (abs(b->capture) == PAWN || 
	    abs(b->board[move->to]) == PAWN) {
		b->piece_change |= (~b->piece_mask) & b->material_mask;
	}

	b->trapped_mask |= control_mask;
	b->piece_change |= control_mask;
	b->trapped_mask &= b->piece_mask;
}


/* this is called when the topieces[] array needs to be completely redone */
void regen_moves(Position *b)
{
	int pi;
	memset(b->topieces, 0, sizeof(b->topieces));

	b->control_change = OneBB;
	b->piece_change = ~0;
	b->white_moves = 0;
	b->black_moves = 0;

	for (pi=0;pi<32;pi++) {
		if (!b->pieces[pi].p) continue;
		b->mobility[pi] = 0;
		piece_generate(b, pi);
	}
}

static int generate_captures(Position *b, Move *moves)
{
	int i, n=0;
	Square to;
	uint32 pieces;

	if (whites_move(b)) {
		PieceStruct *p = WHITEPIECES(b);

		for (to=A1;to<=H8;to++) {
			if (b->board[to] >= 0) continue;
			pieces = b->topieces[to] & WHITE_MASK;
			while (pieces) {
				i = ff_one(pieces);
				moves->from = p[i].pos;
				moves->to = to;
				moves++;
				n++;
				pieces &= ~(1<<i);
			}
		}
		
		if ((to=b->enpassent) && (b->topieces[to] & WPAWN_MASK)) {
			if (XPOS(to) != 0 && b->board[to+SOUTH_WEST] == PAWN) {
				moves->from = to+SOUTH_WEST;
				moves->to = to;
				moves++;
				n++;
			}
			if (XPOS(to) != 7 && b->board[to+SOUTH_EAST] == PAWN) {
				moves->from = to+SOUTH_EAST;
				moves->to = to;
				moves++;
				n++;
			}
		}

		pieces = b->pawns7th & WPAWN_MASK;
		p = WHITEPIECES(b);
		while (pieces) {
			i = ff_one(pieces);
			if (!b->board[p[i].pos]) {
				moves->from = p[i].pos;
				moves->to = p[i].pos + NORTH;
				moves++;
				n++;
			}
			pieces &= ~(1<<i);
		}
	} else {
		PieceStruct *p = WHITEPIECES(b);

		for (to=A1;to<=H8;to++) {
			if (b->board[to] <= 0) continue;
			pieces = b->topieces[to] & BLACK_MASK;
			while (pieces) {
				i = ff_one(pieces);
				moves->from = p[i].pos;
				moves->to = to;
				moves++;
				n++;
				pieces &= ~(1<<i);
			}
		}

		if ((to=b->enpassent) && (b->topieces[to] & BPAWN_MASK)) {
			if (XPOS(to) != 0 && b->board[to+NORTH_WEST] == -PAWN) {
				moves->from = to+NORTH_WEST;
				moves->to = to;
				moves++;
				n++;
			}
			if (XPOS(to) != 7 && b->board[to+NORTH_EAST] == -PAWN) {
				moves->from = to+NORTH_EAST;
				moves->to = to;
				moves++;
				n++;
			}
		}


		pieces = b->pawns7th & BPAWN_MASK;
		p = WHITEPIECES(b);
		while (pieces) {
			i = ff_one(pieces);
			if (!b->board[p[i].pos]) {
				moves->from = p[i].pos;
				moves->to = p[i].pos + SOUTH;
				moves++;
				n++;
			}
			pieces &= ~(1<<i);
		}
	}

	return n;
}

#if 0
static int generate_captures_new(Position *b, Move *moves)
{
	int i, j, n=0;
	Square to;
	uint32 pieces, hung;
	PieceStruct *p;

	if (whites_move(b)) {
		hung = b->hung_mask & BLACK_MASK;
		while (hung) {
			i = ff_one(hung);
			hung &= ~(1<<i);
			to = b->pieces[i].pos;

			pieces = b->topieces[to] & WHITE_MASK;
			while (pieces) {
				j = ff_one(pieces);
				pieces &= ~(1<<j);
				moves->from = b->pieces[j].pos;
				moves->to = to;
				moves++;
				n++;
			}
		}
		
		if ((to=b->enpassent) && (b->topieces[to] & WPAWN_MASK)) {
			if (XPOS(to) != 0 && b->board[to+SOUTH_WEST] == PAWN) {
				moves->from = to+SOUTH_WEST;
				moves->to = to;
				moves++;
				n++;
			}
			if (XPOS(to) != 7 && b->board[to+SOUTH_EAST] == PAWN) {
				moves->from = to+SOUTH_EAST;
				moves->to = to;
				moves++;
				n++;
			}
		}

		pieces = b->pawns7th & WPAWN_MASK;
		p = WHITEPIECES(b);
		while (pieces) {
			i = ff_one(pieces);
			if (!b->board[p[i].pos]) {
				moves->from = p[i].pos;
				moves->to = p[i].pos + NORTH;
				moves++;
				n++;
			}
			pieces &= ~(1<<i);
		}
	} else {
		hung = b->hung_mask & WHITE_MASK;
		while (hung) {
			i = ff_one(hung);
			hung &= ~(1<<i);
			to = b->pieces[i].pos;

			pieces = b->topieces[to] & BLACK_MASK;
			while (pieces) {
				j = ff_one(pieces);
				pieces &= ~(1<<j);
				moves->from = b->pieces[j].pos;
				moves->to = to;
				moves++;
				n++;
			}
		}

		if ((to=b->enpassent) && (b->topieces[to] & BPAWN_MASK)) {
			if (XPOS(to) != 0 && b->board[to+NORTH_WEST] == -PAWN) {
				moves->from = to+NORTH_WEST;
				moves->to = to;
				moves++;
				n++;
			}
			if (XPOS(to) != 7 && b->board[to+NORTH_EAST] == -PAWN) {
				moves->from = to+NORTH_EAST;
				moves->to = to;
				moves++;
				n++;
			}
		}


		pieces = b->pawns7th & BPAWN_MASK;
		p = WHITEPIECES(b);
		while (pieces) {
			i = ff_one(pieces);
			if (!b->board[p[i].pos]) {
				moves->from = p[i].pos;
				moves->to = p[i].pos + SOUTH;
				moves++;
				n++;
			}
			pieces &= ~(1<<i);
		}
	}

	return n;
}
#endif


/* return the number of moves, and put all the moves in moves[].
   This assumes that moves[] is big enough to hold all the moves */
int generate_moves(Position *b, Move *moves)
{
	int n, i;
	PieceStruct *p;
	uint32 pieces;
	Square to;

	n = 0;

	if (whites_move(b)) {
		p = WHITEPIECES(b);

		for (to=A1;to<=H8;to++) {
			if (b->board[to]>0) continue;
			pieces = b->topieces[to] & WHITE_MASK;
			if (b->board[to]==0)
				pieces &= b->piece_mask;
			while (pieces) {
				i = ff_one(pieces);
				moves->from = p[i].pos;
				moves->to = to;
				moves++;
				n++;
				pieces &= ~(1<<i);
			}
		}

		if ((to=b->enpassent) && (b->topieces[to] & WPAWN_MASK)) {
			if (XPOS(to) != 0 && b->board[to+SOUTH_WEST] == PAWN) {
				moves->from = to+SOUTH_WEST;
				moves->to = to;
				moves++;
				n++;
			}
			if (XPOS(to) != 7 && b->board[to+SOUTH_EAST] == PAWN) {
				moves->from = to+SOUTH_EAST;
				moves->to = to;
				moves++;
				n++;
			}
		}


		if (b->flags & WHITE_CASTLE_SHORT) {
			if (b->board[F1] == 0 && b->board[G1] == 0 &&
			    !(b->topieces[E1] & BLACK_MASK) &&
			    !(b->topieces[F1] & BLACK_MASK) &&
			    !(b->topieces[G1] & BLACK_MASK)) {
				moves->from = E1;
				moves->to = G1;
				moves++;
				n++;
			}
		}
		if (b->flags & WHITE_CASTLE_LONG) {
			if (b->board[D1] == 0 && b->board[C1] == 0 &&
			    b->board[B1] == 0 &&
			    !(b->topieces[E1] & BLACK_MASK) &&
			    !(b->topieces[D1] & BLACK_MASK) &&
			    !(b->topieces[C1] & BLACK_MASK)) {
				moves->from = E1;
				moves->to = C1;
				moves++;
				n++;
			}
		}

		p = &WHITEPIECES(b)[8];
		for (i=0;i<8;i++, p++) {
			if (p->p != PAWN) continue;
			if (b->board[p->pos+NORTH]) continue;
			moves->from = p->pos;
			moves->to = p->pos+NORTH;
			moves++;
			n++;
			if (YPOS(p->pos) != 1 || b->board[p->pos+2*NORTH]) continue;
			moves->from = p->pos;
			moves->to = p->pos+2*NORTH;
			moves++;
			n++;			
		}
	} else {
		p = WHITEPIECES(b);

		for (to=A1;to<=H8;to++) {
			if (b->board[to]<0) continue;
			pieces = b->topieces[to] & BLACK_MASK;
			if (b->board[to]==0)
				pieces &= b->piece_mask;
			while (pieces) {
				i = ff_one(pieces);
				moves->from = p[i].pos;
				moves->to = to;
				moves++;
				n++;
				pieces &= ~(1<<i);
			}
		}

		if ((to=b->enpassent) && (b->topieces[to] & BPAWN_MASK)) {
			if (XPOS(to) != 0 && b->board[to+NORTH_WEST] == -PAWN) {
				moves->from = to+NORTH_WEST;
				moves->to = to;
				moves++;
				n++;
			}
			if (XPOS(to) != 7 && b->board[to+NORTH_EAST] == -PAWN) {
				moves->from = to+NORTH_EAST;
				moves->to = to;
				moves++;
				n++;
			}
		}


		if (b->flags & BLACK_CASTLE_SHORT) {
			if (b->board[F8] == 0 && b->board[G8] == 0 &&
			    !(b->topieces[E8] & WHITE_MASK) &&
			    !(b->topieces[F8] & WHITE_MASK) &&
			    !(b->topieces[G8] & WHITE_MASK)) {
				moves->from = E8;
				moves->to = G8;
				moves++;
				n++;
			}
		}
		if (b->flags & BLACK_CASTLE_LONG) {
			if (b->board[D8] == 0 && b->board[C8] == 0 &&
			    b->board[B8] == 0 &&
			    !(b->topieces[E8] & WHITE_MASK) &&
			    !(b->topieces[D8] & WHITE_MASK) &&
			    !(b->topieces[C8] & WHITE_MASK)) {
				moves->from = E8;
				moves->to = C8;
				moves++;
				n++;
			}
		}

		p = &BLACKPIECES(b)[8];
		for (i=0;i<8;i++, p++) {
			if (p->p != -PAWN) continue;
			if (b->board[p->pos+SOUTH]) continue;
			moves->from = p->pos;
			moves->to = p->pos+SOUTH;
			moves++;
			n++;
			if (YPOS(p->pos) != 6 || b->board[p->pos+2*SOUTH]) continue;
			moves->from = p->pos;
			moves->to = p->pos+2*SOUTH;
			moves++;
			n++;			
		}
	}

	return n;
}


int generate_check_avoidance(Position *b, Move *moves)
{
	int n=0;
	int kpos, pos, pi, pi2, dx, dy, to, dir;
	int x1, x2, y1, y2, x, y;
	uint32 giving_check, pieces;

	if (whites_move(b)) {
		kpos = WHITEPIECES(b)[IKING].pos;
		giving_check = b->topieces[kpos] & BLACK_MASK;
	} else {
		kpos = BLACKPIECES(b)[IKING].pos;
		giving_check = b->topieces[kpos] & WHITE_MASK;
	}

	pos = kpos;

	/* which piece is giving check? */
	pi = ff_one(giving_check);
	if (giving_check & ~(1<<pi))
		goto king_moves; /* its double check */

	pos = b->pieces[pi].pos;

	/* try to capture it */
	pieces = b->topieces[pos];

	/* a special case - if its a pawn and enpassent is possible then
	   take enpassent */
	if (b->enpassent && !(giving_check & b->piece_mask)) {
		pieces |= b->topieces[b->enpassent] & ~b->piece_mask;
	}

	if (whites_move(b)) {
		pieces &= WHITE_MASK;
	} else {
		pieces &= BLACK_MASK;
	}

	while (pieces) {
		pi2 = ff_one(pieces);
		moves->from = b->pieces[pi2].pos;
		moves->to = pos;
		moves++;
		n++;
		pieces &= ~(1<<pi2);
	}


	/* now try to block it if its a sliding piece */
	if (!(giving_check & b->sliding_mask))
		goto king_moves;

	dx = XPOS(kpos) - XPOS(pos);
	dy = YPOS(kpos) - YPOS(pos);

	dir = 0;
	if (dy > 0) 
		dir += SOUTH;
	else if (dy < 0) 
		dir += NORTH;

	if (dx > 0) 
		dir += WEST;
	else if (dx < 0) 
		dir += EAST;
	
	to = kpos + dir;
	while (to != pos) {
		if (whites_move(b)) {
			pieces = (b->topieces[to] & WHITE_MASK) & ~WKING_MASK;
			pieces &= b->piece_mask;
			if (b->board[to+SOUTH] == PAWN) {
				pieces |= (1<<b->pboard[to+SOUTH]);
			} else if (YPOS(to) == 3 && b->board[to+SOUTH] == 0 &&
				   b->board[to+2*SOUTH] == PAWN) {
				pieces |= (1<<b->pboard[to+2*SOUTH]);
			}
		} else {
			pieces = (b->topieces[to] & BLACK_MASK) & ~BKING_MASK;
			pieces &= b->piece_mask;
			if (b->board[to+NORTH] == -PAWN) {
				pieces |= (1<<b->pboard[to+NORTH]);
			} else if (YPOS(to) == 4 && b->board[to+NORTH] == 0 &&
				   b->board[to+2*NORTH] == -PAWN) {
				pieces |= (1<<b->pboard[to+2*NORTH]);
			}
		}
		while (pieces) {
			pi2 = ff_one(pieces);
			moves->from = b->pieces[pi2].pos;
			moves->to = to;
			moves++;
			n++;
			pieces &= ~(1<<pi2);
		}
		to += dir;
	}


king_moves:
	/* try to move the king */
	x1 = XPOS(kpos) - 1;
	x2 = x1 + 2;
	y1 = YPOS(kpos) - 1;
	y2 = y1 + 2;

	if (x1 < 0) x1 = 0;
	if (y1 < 0) y1 = 0;
	if (x2 == 8) x2 = 7;
	if (y2 == 8) y2 = 7;

	for (x=x1;x<=x2;x++)
		for (y=y1;y<=y2;y++) {
			int pos2 = POSN(x,y);
			if (pos2 == kpos || pos2 == pos) continue;
			pieces = b->topieces[pos2];
			if (whites_move(b)) {
				if (pieces & BLACK_MASK) continue;
				if (b->board[pos2] > 0) continue;
			} else {
				if (pieces & WHITE_MASK) continue;
				if (b->board[pos2] < 0) continue;
			}
			moves->from = kpos;
			moves->to = pos2;
			moves++;
			n++;
		}
			
	return n;
}





/* this is used to make sure the player didn't make an illegal move -
   it is not speed critical */
int legal_move(Position *b, Move *move)
{
	Move moves[MAX_MOVES];
	int num_moves;
	int m;
	Position b1;

	num_moves = generate_moves(b, moves);

	for (m=0;m<num_moves;m++)
		if (moves[m].from == move->from &&
		    moves[m].to == move->to) break;

	if (m == num_moves) return 0;

	if (!do_move(&b1, b, move))
		return 0;

	return 1;
}


static Move *get_move_list(int ply)
{
	static int num_move_lists;
	static Move **move_lists;
	int j;

	if (num_move_lists > ply) {
		return move_lists[ply];
	}

	move_lists = (Move **)Realloc(move_lists,
				      (num_move_lists+30)*sizeof(Move *));
	for (j=num_move_lists;j<num_move_lists+30;j++) {
		move_lists[j] = (Move *)Realloc(NULL, MAX_MOVES*sizeof(Move));
	}

	num_move_lists += 30;
	
	return move_lists[ply];
}


void gen_move_list(Position *b, int ply, int only_captures, Move *m1)
{
	int n;
	Move *moves;

	b->moves = get_move_list(ply);
	moves = b->moves;

	if (b->flags & FLAG_CHECK) {
		n = generate_check_avoidance(b, moves);
	} else if (only_captures) {
		n = generate_captures(b, moves);
	} else {
		n = generate_moves(b, moves);
	}

	b->num_moves = n;
	order_moves(b, moves, n, ply, m1);
}


		

