/* a 3D chess set

   Andrew.Tridgell@anu.edu.au January 1997 */

#include "includes.h"
#include "trackball.h"
#include "knightcap.h"

static int window_size = 500;

#define SLEEP_TIME 100000 /* microseconds */
#define REFRESH_TIME 20 /* in seconds */

struct state *state;

int demo_mode=0;
static int always_think;
static int no_display;
static int ascii_display;
static int ics_mode;
static int move_pid=0;
static int match_remaining;
static int match_total;
static int def_hash_table_size = DEFAULT_HASH_TABLE_SIZE;
static float match_points;
int color_display=1;

extern int need_redraw;

static int default_move_time = DEFAULT_MOVE_TIME;

char *ics_name = "KnightCap";

static char play_target[100];

#ifndef SHM_R
#define SHM_R 0400
#endif

#ifndef SHM_W
#define SHM_W 0200
#endif

/* setup a standard chess starting position */
void reset_board(void)
{
	setup_board(state);
	state->computer = 0;
	state->moved = 0;
	state->always_think = always_think;
	if (state->always_think)
		lprintf(0,"Thinking on opponents time\n");
	timer_reset();
}

static void display_move(Move *move)
{
	print_move(&state->position, move);
	do_move(&state->position, &state->position, move);
	redraw();
}

int player_moved(Square from, Square to)
{
	Move move;

	move.from = from;
	move.to = to;

	if (!legal_move(&state->position, &move)) {
		lprintf(0, "Illegal move %s\n",
			short_movestr(&state->position, &move));
		return 0;
	}

	state->game_record[state->position.move_num] = move;
	display_move(&move);
	state->stop_search = 1;

	return 1;
}


static void about(void)
{
	lprintf(0,"\nThe author of KnightCap is Andrew Tridgell. He can be contacted\n" \
"via email Andrew.Tridgell@anu.edu.au.\n" \
"\n" \
"The ftp site for KnightCap is ftp://samba.anu.edu.au/pub/KnightCap/\n" \
"\n" \
"KnightCap is free software distributed under the GNU Public License\n" \
"\n" \
);
}


static void help(void)
{
	lprintf(0,"\n" \
"go                              start searching\n" \
"black                           computer plays black\n" \
"white                           computer plays white\n" \
"new                             reset the board\n" \
"time <hsecs>                    update computers time in hundredths of a second\n" \
"otim <hsecs>                    update oppenents time in hundredths of a second\n" \
"level x <mins> <inc>            xboard compat command\n" \
"load <file>                     load a saved game\n" \
"save <file>                     save a game\n" \
"quit                            quit!\n" \
"demo <1|0>                      enable/disable demo mode\n" \
"ponder <1|0>                    enable/disable thinking on oppenents time\n" \
"robot <1|0>                     enable/disable ICS robot\n" \
"mtime <move time>               set time per move in seconds\n" \
"undo <ply>                      take back <ply> half moves\n" \
"redo <ply>                      replay <ply> half moves\n" \
"eval                            static evaluation debug\n" \
"espeed  <loops>                 measure eval fn speed\n" \
"testfin <file>                  load a .fin test file and test each position\n" \
"print                           print the current board\n" \
"about                           show some info on KnightCap\n" \
"target <opponent>               try challenging <opponent> when bored\n" \
"mm  <n>                         play n games against another computer\n" \
"color <1|0>                     enable/disable color ascii board display\n" \
"");

}


int parse_stdin(char *line)
{
	char *p;
	Move move;
	char tok[1000];
	int hsecs, level, mins;
	static int increment;
	char load_file[100];
	int on, i;
	int done = 0;

	p = line;

	while (next_token(&p, tok, "\n\r")) {
		/* is it a move? */
		if (parse_move(tok, &state->position, &move)) {
			if (player_moved(move.from, move.to))
				done = 1;
		}

		if (strcmp(tok,"go") == 0) {
			state->computer = next_to_play(&state->position);
			done = 1;
		}

		if (strcmp(tok,"black") == 0) {
			state->computer = -1;
			done = 1;
		}

		if (strcmp(tok,"white") == 0) {
			state->computer = 1;
			done = 1;
		}

		if (strcmp(tok,"new") == 0) {
			reset_board();
			redraw();
			done = 1;
		}

		if (sscanf(tok, "time %d", &hsecs) == 1) {
			timer_estimate(hsecs/100, increment);
			done = 1;
		}


		if (sscanf(tok, "otim %d", &hsecs) == 1) {
			if (hsecs <= 0)
				lprintf(0, "win on time!\n");
			done = 1;
		}

		if (sscanf(tok, "level %d %d %d", &level, &mins, &increment) == 3) {
			lprintf(0,"setting increment to %d\n", increment);
			done = 1;
		}

		if (sscanf(tok, "testfin %s", load_file) == 1) {
			test_fin(load_file);
			done = 1;
		}

		if (sscanf(tok, "load %s", load_file) == 1) {
			restore_game(load_file);
			done = 1;
		}

		if (sscanf(tok, "save %s", load_file) == 1) {
			save_game(load_file);
			done = 1;
		}

		if (sscanf(tok, "target %s", play_target) == 1) {
			done = 1;
		}

		if (strcmp(tok, "quit") == 0) {
			state->quit = 1;
			state->stop_search = 2;
			prog_printf("quit\n");
			prog_exit();
			exit(0);			
		}

		if (strcmp(tok, "print") == 0) {
			print_board(&state->position);
			done = 1;
		}

		if (sscanf(tok,"demo %d", &on) == 1) {
			demo_mode = on;
			done = 1;
		}

		if (sscanf(tok,"color %d", &on) == 1) {
			color_display = on;
			done = 1;
		}

		if (sscanf(tok,"mm %d", &match_remaining) == 1) {
			match_points = 0;
			match_total = 0;
			done = 1;
		}

		if (sscanf(tok,"robot %d", &on) == 1) {
			state->ics_robot = on;
			done = 1;
		}

		if (sscanf(tok,"display %d", &on) == 1) {
			no_display = !on;
			done = 1;
		}

		if (sscanf(tok,"ponder %d", &on) == 1) {
			always_think = state->always_think = on;
			done = 1;
		}

		if (sscanf(tok,"mtime %d", &i) == 1) {
			state->move_time = i;
			done = 1;
		}

		if (sscanf(tok,"undo %d", &i) == 1) {
			undo_menu(i);
			done = 1;
		}

		if (sscanf(tok,"redo %d", &i) == 1) {
			undo_menu(-i);
			done = 1;
		}

		if (strcmp(tok, "eval") == 0) {
			eval_debug(&state->position);
			done = 1;
		}

		if (sscanf(tok,"espeed %d", &i) == 1) {
			eval_speed(&state->position, i);
			done = 1;
		}

		if (strcmp(tok, "help") == 0) {
			help();
			done = 1;
		}

		if (strcmp(tok, "about") == 0) {
			about();
			done = 1;
		}
	}

	return done;
}


/* users can send us moves and commands on stdin */
static void check_stdin(void)
{
	char line[200];
	int n;
	fd_set set;
	struct timeval tval;
	static int nostdin;

	if (nostdin)
		return;

	FD_ZERO(&set);
	FD_SET(0, &set);

	tval.tv_sec = 0;
	tval.tv_usec = 0;

	if (select(1, &set, NULL, NULL, &tval) != 1)
		return;

	n = read(0, line, sizeof(line)-1);

	if (n <= 0) {
		lprintf(0,"disabling stdin\n");
		nostdin = 1;
		close(0);
		return;
	}

	line[n] = 0;

	if (!parse_stdin(line)) {
		if (prog_running()) {
			prog_printf("%s", line);
		}
	}
}


void update_display(void)
{
	if (!no_display) {
		if (ascii_display)
			print_board(&state->position);
		else
			draw_all();
	}
}


void match_hook(void)
{
	static int player=1;
	
	if (match_remaining <= 0) return;

	/* perhaps we are already playing */
	if (state->computer && !state->position.winner) return;

	if (state->position.winner) {
		if (state->position.winner == STALEMATE)
			match_points += 0.5;
		if (state->position.winner == player)
			match_points += 1;

		state->position.winner = 0;

		match_total++;
		match_remaining--;

		lprintf(0,"match %1.1f points out of %d games\n", 
			match_points, match_total);

		player = -player;
	}

	if (match_remaining <= 0) return;
	
	state->computer = 0;

	reset_board();
	prog_printf("new\n");
	if (player == 1) {
		prog_printf("black\n");
		state->computer = 1;
	} else {
		prog_printf("white\n");
		state->computer = -1;
	}
}

void idle_func(void)
{
	Move m1;
	static uint32 hash;
	static int loops;

	check_stdin();

	if (prog_check_move(&m1, next_to_play(&state->position))) {
		player_moved(m1.from, m1.to);
	}

	if (state->moved) {
		if (next_to_play(&state->position) == state->computer) {
			m1 = state->game_record[state->position.move_num];
			if (legal_move(&state->position, &m1)) {
				prog_tell_move(&state->position, &m1);
				display_move(&m1);
				if (state->position.winner) {
					state->computer = 0;
					if (state->position.winner == STALEMATE)
						prog_printf("draw\n");
					if (state->ics_robot) {
						/* make sure we are right! */
						prog_printf("refresh\n");
					}
				}
				if (!state->ics_robot)
					save_game("current.save");
			}
			state->stop_search = 0;
		}
		state->moved = 0;
	}

	if (need_redraw || hash != state->position.hash1) {
		need_redraw = 0;
		hash = state->position.hash1;
		update_display();
		loops=0;
	}

	if (demo_mode && !state->position.winner &&
	    next_to_play(&state->position) != state->computer) {
		state->stop_search = 1;
		state->computer = next_to_play(&state->position);
	}

	if (!process_exists(move_pid)) {
			state->quit = 1;
			state->stop_search = 2;
			prog_printf("quit\n");
			prog_exit();
			exit(0);					
	}

	if (++loops > REFRESH_TIME*(1000*1000/SLEEP_TIME)) {
		if (state->ics_robot && 
		    state->computer &&
		    state->computer != next_to_play(&state->position))
			prog_printf("refresh\n");
		loops=0;
		if (state->ics_robot &&
		    state->computer == 0 && *play_target && 
		    (random() % 5) == 0) {
			prog_printf("match %s\n", play_target);
		}
	}

	match_hook();

	usleep(SLEEP_TIME);
}


void save_game(char *fname)
{
	int fd = open(fname, O_WRONLY | O_CREAT | O_TRUNC, 0666);
	if (fd == -1) {
		perror(fname);
		return;
	}
	write(fd, state, sizeof(*state));
	close(fd);
}

void restore_game(char *fname)
{
	int fd = open(fname, O_RDONLY);
	if (fd == -1) {
		perror(fname);
		return;
	}
	read(fd, state, sizeof(*state));
	close(fd);
}


/* undo the last n ply of the game 
 Passing a negative number gives redo */
void undo_menu(int n)
{
	int i;

	state->computer = 0;
	state->stop_search = 1;
	demo_mode = 0;

	n = state->position.move_num - n;
	if (n < 0) n = 0;

	setup_board(state);

	for (i=0;i<n;i++) {
		Move *move = &state->game_record[i];
		if (!legal_move(&state->position, move))
			break;
		do_move(&state->position, &state->position, move);
	}

	state->stop_search = 1;

	redraw();
}

static void move_thread(void)
{
	int pondered = 0;
	chdir("prof");

#if APLINUX
	if (getcid() != 0) {
		move_slave();
		exit();
	}
#endif

	while (!state->quit) {
		Move move;

		if (next_to_play(&state->position) != state->computer ||
		    state->moved || state->position.winner) {
			if (state->always_think && 
			    !state->position.winner &&
			    !pondered &&
			    state->computer == -next_to_play(&state->position)) {
				ponder_move(&state->position, &move);
				pondered = 1;
			}
			usleep(SLEEP_TIME);
			continue;
		}

		usleep(SLEEP_TIME);

		pondered = 0;

		if (make_move(&state->position, &move)) {
			if (next_to_play(&state->position) == state->computer &&
			    legal_move(&state->position, &move)) {
				state->game_record[state->position.move_num] = move;
				state->moved = 1;
			}
		} else {
			state->computer = 0;
		}
	}

	sleep(1);

	exit(0);
}


static void create_threads(void)
{
	int page_size;
	int len, shmid;

	page_size = getpagesize();
	len = sizeof(*state);
	len = (len + page_size) & ~(page_size-1);

	shmid = shmget(IPC_PRIVATE, len, SHM_R | SHM_W);
	if (shmid == -1) {
		printf("can't get shared memory\n");
		exit(1);
	}
	state = (struct state *)shmat(shmid, 0, 0);
	if (!state || state == (struct state *)-1) {
		printf("can't attach to shared memory\n");
		exit(1);
	}
	shmctl(shmid, IPC_RMID, 0);
	memset(state, 0, len);
	state->move_time = default_move_time;

	reset_board();
	init_movements();

	signal(SIGCLD, SIG_IGN);

#ifdef APLINUX
	move_pid = getpid();
	if (getcid() != 0 || fork() != 0) {
		log_close();
		move_thread();
		exit(0);
	}
#else
	if ((move_pid=fork()) != 0) {
		move_thread();
		exit(0);
	}
#endif
}


static void usage(void)
{
	printf("\n" \
"-w <window size>\n" \
"-c <external chess program>\n" \
"-s <seed>\n" \
"-n                  no display\n" \
"-a                  ascii display\n" \
"-I <icsname>        ICS mode\n" \
"-t <time>           default move time\n" \
"-X                  catch sigint for xboard\n" \
"-A                  always think\n" \
"-H <size>           set hash table size in MB\n" \
"\n");
}


static void parse_options(int argc,char *argv[])
{
#ifdef SUNOS4
	extern char *optarg;
#endif
	char *opt = "c:s:w:nI:t:hXAD:l:aH:";
	int c;
	int seed = getpid();

	while ((c = getopt(argc, argv, opt)) != EOF) {
		switch (c) {
		case 'D':
			{
				char disp[100];
				sprintf(disp,"DISPLAY=%s", optarg);
				putenv(disp);
				printf("display set to %s\n", getenv("DISPLAY"));
			}
			break;

		case 'w':
#if APLINUX
			if (getcid()) break;
#endif
			window_size = atoi(optarg);
			break;

		case 'c':
#if APLINUX
			if (getcid()) break;
#endif
			prog_start(optarg);
			break;

		case 'H':
			def_hash_table_size = atoi(optarg);
			break;

		case 's':
			seed = atoi(optarg);
			break;

		case 'n':
			no_display = 1;
			break;

		case 'a':
			ascii_display = 1;
			break;

		case 'I':
			ics_mode = 1;
			ics_name = optarg;
			break;

		case 't':
			default_move_time = atoi(optarg);
			break;

		case 'h':
			usage();
			exit(0);
			break;

		case 'X':
			signal(SIGINT, SIG_IGN);
			break;

		case 'A':
			always_think = 1;
			break;
		}
	}

	srandom(seed); 
}


static void nodisp_loop(void)
{
	while (1) {
		idle_func();
		usleep(100000);
	}
}

int main(int argc,char *argv[])
{
#ifdef APLINUX
	ap_init();
#endif

	lprintf(0,"KnightCap starting\ntype \"help\" for commands or use the right button\n");

	parse_options(argc, argv);

	create_threads();

	state->ics_robot = ics_mode;
	state->hash_table_size = def_hash_table_size;

	reset_board();

#if RENDERED_DISPLAY
	if (!no_display && !ascii_display) {
		start_display(argc, argv);
		return 0;
	}
#endif

	nodisp_loop();

	return 0;
}

