#include "includes.h"
#include "knightcap.h"

static int nodes, quiesce_nodes, semi_quiesce_nodes;
static int maxply;
static int move_order_hits, move_order_misses;
static int pondering, ponder_stop;

extern struct state *state;


static int search_expired(void)
{
	if (timer_expired() || state->quit || state->stop_search) 
		return 1;

	if (pondering) {
		if (next_to_play(&state->position) == state->computer ||
		    ponder_stop)
			return 1;
	}

	return 0;
}

static inline int depth_extensions(Position *b, int depth, 
				   Eval v1, Eval testv)
{
#if USE_PAWN_DEEPENING
	if (b->board[b->last_dest] == PAWN && YPOS(b->last_dest) > 4)
		return depth+1;
	if (b->board[b->last_dest] == -PAWN && YPOS(b->last_dest) < 3)
		return depth+1;
#endif

#if USE_CHECK_DEEPENING
	if (b->flags & FLAG_CHECK)
		return depth+1;		
#endif

#if USE_RAZORING
	if (depth == 1 && v1 < (testv - RAZOR_THRESHOLD))
		return 0;
#endif

#if 0
	if (v1 >= testv && b->oldb && 
	    b->attacking_mask == b->oldb->attacking_mask &&
	    !(b->hung_mask & player_mask(b)) &&
	    depth < 9 &&
	    b->white_moves > NULL_MOVE_THRESHOLD &&
	    b->black_moves > NULL_MOVE_THRESHOLD)
		return imax(0, (depth/2) - 1);
#endif

	return depth;
}



#if APLINUX
struct slave {
	unsigned index;
	int depth;
	int testv, v;
	int nodes;
	int mo_hits, mo_misses;
	int hash_hits, hash_misses;
	int maxply;
	int busy;
	short x;
	Move move;
	int pad;
};

struct slave slave_in;
struct slave slave_out;
#endif



static Eval quiesce(Position *b, Eval testv, int ply)
{
	Move m1;
	Eval g, v1, v;
	int i, j;
	uint32 pieces, topieces;
	Position b1;

	if (ply > maxply)
		maxply = ply;

#if USE_HASH_TABLES
	if (check_hash_quiesce(b, testv, &g)) {
		if (g < (ILLEGAL+20))
			g += ply;
		if (g > (WIN-20))
			g -= ply;
		return g;
	}
#endif

	nodes++;
	quiesce_nodes++;

	/* check is a special case. All moves are generated
	   and the player can't elect to just "sit" */
	if (b->flags & FLAG_CHECK) {
		int m, num_moves;
		Move *moves;

		gen_move_list(b, ply, 0, NULL);
		moves = b->moves;
		num_moves = b->num_moves;

		g = ILLEGAL;

		for (m=0;m<num_moves;m++) {
			if (!do_move(&b1, b, &moves[m])) continue;

			v = -quiesce(&b1, -(testv-1), ply+1);

			if (v > g) {
				g = v;
				if (g >= testv) 
					break;
			}
		}

		insert_hash(b, 0, testv, g, NULL);
		return g;
	}


	v1 = eval(b, testv, 0);
	g = v1;

	/* in the quiesce search the player has the option to "sit" and
	   take the current evaluation */
	if (g >= testv || ply >= MAX_DEPTH) {
		goto evaluated;
	}

	if (whites_move(b)) {
		/* first see if we can promote */
		pieces = b->pawns7th & WHITE_MASK;

		while (pieces) {
			i = ff_one(pieces);
			pieces &= ~(1<<i);
			if (b->board[b->pieces[i].pos+NORTH]) 
				continue;

			m1.from = b->pieces[i].pos;
			m1.to = b->pieces[i].pos + NORTH;

			if (!do_move(&b1, b, &m1)) continue;

			v = -quiesce(&b1, -(testv-1), ply+1);
			if (v > g) {
				g = v;
				if (g >= testv) 
					goto evaluated;
			}
		}

		/* now see if we can capture - big pieces first */
		pieces = b->material_mask & BLACK_MASK;
		while (pieces) {
			i = ff_one(pieces);
			pieces &= ~(1<<i);

			topieces = b->topieces[b->pieces[i].pos] & WHITE_MASK;
			if (!topieces) continue;

			/* loop over all the capture possabilities */
			while (topieces) {
				j = fl_one(topieces);
				topieces &= ~(1<<j);
				m1.from = b->pieces[j].pos;
				m1.to = b->pieces[i].pos;

#if 0
				if (v1 + mat_value[-b->board[m1.to]] -
				    b->piece_values[i] < 
				    testv - FUTILE_THRESHOLD)
					continue;
#endif

				if (!do_move(&b1, b, &m1)) continue;

				v = -quiesce(&b1, -(testv-1), ply+1);
				if (v > g) {
					g = v;
					if (g >= testv) 
						goto evaluated;
				}
			}
		}
		
		/* one more chance - an enpassent capture */
		if (!b->enpassent) 
			goto evaluated;

		topieces = b->topieces[b->enpassent] & WHITE_MASK & ~b->piece_mask;
		while (topieces) {
			j = ff_one(topieces);
			topieces &= ~(1<<j);
			
			m1.from = b->pieces[j].pos;
			m1.to = b->enpassent;
			
			if (!do_move(&b1, b, &m1)) continue;
			
			v = -quiesce(&b1, -(testv-1), ply+1);
			if (v > g) {
				g = v;
				if (g >= testv) 
					goto evaluated;
			}
		}
	} else {
		/* first see if we can promote */
		pieces = b->pawns7th & BLACK_MASK;

		while (pieces) {
			i = ff_one(pieces);
			pieces &= ~(1<<i);
			if (b->board[b->pieces[i].pos+SOUTH]) 
				continue;

			m1.from = b->pieces[i].pos;
			m1.to = b->pieces[i].pos + SOUTH;

			if (!do_move(&b1, b, &m1)) continue;

			v = -quiesce(&b1, -(testv-1), ply+1);
			if (v > g) {
				g = v;
				if (g >= testv) 
					goto evaluated;
			}
		}

		/* now see if we can capture - big pieces first */
		pieces = b->material_mask & WHITE_MASK;
		while (pieces) {
			i = ff_one(pieces);
			pieces &= ~(1<<i);

			topieces = b->topieces[b->pieces[i].pos] & BLACK_MASK;
			if (!topieces) continue;

			/* loop over all the capture possabilities */
			while (topieces) {
				j = fl_one(topieces);
				topieces &= ~(1<<j);
				m1.from = b->pieces[j].pos;
				m1.to = b->pieces[i].pos;

#if 0
				if (v1 + mat_value[b->board[m1.to]] +
				    b->piece_values[i] < 
				    testv - FUTILE_THRESHOLD)
					continue;
#endif

				if (!do_move(&b1, b, &m1)) continue;

				v = -quiesce(&b1, -(testv-1), ply+1);
				if (v > g) {
					g = v;
					if (g >= testv) 
						goto evaluated;
				}
			}
		}
		
		/* one more chance - an enpassent capture */
		if (!b->enpassent) 
			goto evaluated;

		topieces = b->topieces[b->enpassent] & BLACK_MASK & ~b->piece_mask;
		while (topieces) {
			j = ff_one(topieces);
			topieces &= ~(1<<j);
			
			m1.from = b->pieces[j].pos;
			m1.to = b->enpassent;
			
			if (!do_move(&b1, b, &m1)) continue;
			
			v = -quiesce(&b1, -(testv-1), ply+1);
			if (v > g) {
				g = v;
				if (g >= testv) 
					goto evaluated;
			}
		}
	}


evaluated:
	insert_hash_quiesce(b, testv, g);

	return g;
}


static Eval semi_quiesce(Position *b, int depth, Eval testv, int ply)
{
	Move m1;
	Eval g, v1, v;
	Position b1;
	int m, num_moves;
	Move *moves=NULL;
	int bm = -1;

	if (depth <= 0)
		return quiesce(b, testv, ply);

	if (ply > maxply)
		maxply = ply;

	zero_move(&m1);

#if USE_HASH_TABLES
	if (check_hash(b, depth, testv, &g, &m1)) {
		if (g < (ILLEGAL+20))
			g += ply;
		if (g > (WIN-20))
			g -= ply;
		return g;
	}
#endif

	nodes++;
	semi_quiesce_nodes++;

	/* check is a special case. All moves are generated
	   and the player can't elect to just "sit" */
	if (b->flags & FLAG_CHECK) {
		gen_move_list(b, ply, 0, NULL);
		moves = b->moves;
		num_moves = b->num_moves;

		g = ILLEGAL;

		for (m=0;m<num_moves;m++) {
			if (!do_move(&b1, b, &moves[m])) continue;

			v = -semi_quiesce(&b1, depth-1, -(testv-1), ply+1);

			if (v > g) {
				g = v;
				if (g >= testv) 
					goto evaluated;
			}
		}
		goto evaluated;
	}


	v1 = eval(b, testv, 0);
	g = v1;

	/* in the semi-quiesce search the player has the option to "sit" and
	   take the current evaluation */
	if (g >= testv || ply >= MAX_DEPTH) {
		goto evaluated;
	}


	gen_move_list(b, ply, 0, &m1);
	moves = b->moves;
	num_moves = b->num_moves;

	for (m=0;m<num_moves;m++) {
		if (!do_move(&b1, b, &moves[m])) continue;

		v = -eval(&b1, testv, 0);

		if (!(b1.flags & FLAG_CHECK) &&
		     (v < g + PAWN_VALUE/4) && !b1.capture) continue;

		v = -semi_quiesce(&b1, depth-1, -(testv-1), ply+1);

		if (v > g) {
			bm = m;
			g = v;
			if (g >= testv) {
				cutoff_hint(b, &moves[m], depth, ply);
				goto evaluated;
			}
		}
	}


evaluated:
	insert_hash(b, depth, testv, g, bm>=0?&moves[bm]:NULL);

	return g;
}



/* this is like alpha-beta but only does a single null window search.
   It assumes that beta=testv and alpha=testv-1. */
static Eval abtest(Position *b, int depth0, Eval testv, int ply)
{
	static int null_ply[MAX_DEPTH];
	Move *moves;
	int num_moves, m;
	int bm = -1;
	Eval g;
	Eval v1, v;
	int depth = depth0;
	Move m1;

	if (depth <= 0) {
#if USE_SEMI_QUIESCE
		return semi_quiesce(b, ply/3, testv, ply);
#else
		return quiesce(b, testv, ply);
#endif
	}

	if (ply > maxply)
		maxply = ply;

	v1 = INFINITY;

	zero_move(&m1);

#if USE_HASH_TABLES
	if (check_hash(b, depth+HASH_OFFSET, testv, &v1, &m1)) {
		if (v1 < (ILLEGAL+20))
			v1 += ply;
		if (v1 > (WIN-20))
			v1 -= ply;
		return v1;
	}
#endif

	if (v1 == INFINITY) {
		v1 = eval(b, testv, depth);
	}

	m = -1;

	if (!null_ply[ply-1])
		depth = depth_extensions(b, depth, v1, testv);

	if (depth <= 0) {
#if USE_SEMI_QUIESCE
		return semi_quiesce(b, ply/3, testv, ply);
#else
		return quiesce(b, testv, ply);
#endif
	}

	nodes++;

	g = ILLEGAL;

#if USE_NULL_MOVE
	if (ply < MAX_DEPTH && 
	    !null_ply[ply-1] &&	    
	    !(b->flags & FLAG_CHECK) &&
	    v1 >= testv &&
	    (whites_move(b)?b->white_moves:b->black_moves) > NULL_MOVE_THRESHOLD &&
	    (!check_hash(b, HASH_OFFSET+depth-3, 
			 testv, &v, NULL) || 
	     v >= testv)) {
		int enpassent_saved = b->enpassent;
		
		null_ply[ply] = 1;			

		b->enpassent = 0;
		b->move_num++;
		
		moves = b->moves;
		num_moves = b->num_moves;
		b->moves = NULL;
		b->num_moves = 0;
		b->hash2 ^= 1;
		
		v = -abtest(b, depth-3,
			    -(testv-1), ply+1);
		
		b->hash2 ^= 1;
		b->move_num--;
		b->enpassent = enpassent_saved;
		b->moves = moves;
		b->num_moves = num_moves;
		
		null_ply[ply] = 0;
		
		if (v >= testv) {
			g = v;
			m = 0;
			goto end_loop;
		}
	}
#endif

	if (!b->moves) {
		gen_move_list(b, ply, 0, &m1);
	}

	moves = b->moves;
	num_moves = b->num_moves;

	if (num_moves == 0) {
		if (b->flags & FLAG_CHECK) 
			g = ILLEGAL;
		else 
			g = 0;
		goto evaluated;
	}

	if (num_moves == 1 && ply < DOUBLE_EXTENSION_PLY)
		depth++;

	for (m=0;m<num_moves;m++) {
		Position b1;

#if APLINUX
		if (slave_in.index != slave_out.index)
			return g;
#endif

		if (!do_move(&b1, b, &moves[m])) {
			continue;
		}

		if (check_repitition(&b1, 1)) {
			v = draw_value(b);
		} else {
			v = -abtest(&b1, depth-1, -(testv-1), ply+1);
		}

		if (search_expired()) 
			return g;

		if (v > g) {
			g = v;
			bm = m;
		} 

		if (ply < TREE_PRINT_PLY) {
			lindent(2, ply);
			lprintf(2, "%s %d %s -> %d (%d) testv=%d depth=%d\n",
				whites_move(b)?"W":"B", 
				b->move_num,
				short_movestr(b, &moves[m]), 
				v, g, testv, depth0);
		}

		if (g >= testv) {
			if (m == 0)
				move_order_hits++;
			else
				move_order_misses++;
			cutoff_hint(b, &moves[m], depth, ply);
			goto end_loop;
		}
	}

end_loop:

#if APLINUX
	if (slave_in.index != slave_out.index)
		return 0;
#endif

evaluated:

	if (search_expired()) 
		return g;

	if (g == ILLEGAL) {
		if (b->flags & FLAG_CHECK) {
			g = ILLEGAL + ply;
		} else {
			g = 0;
		}
	}

	insert_hash(b, HASH_OFFSET+depth0, testv, g, bm>=0?&moves[bm]:NULL);

	return g;
}


/* this is the abtest routine for the root node */
static Eval abroot(Position *b, int depth, Eval testv, Move *move)
{
	Move *moves;
	int num_moves, m;
	Eval g;
	Eval v;
	float range;
	int lower, upper;

	g = ILLEGAL;

	if (b->moves)
		goto got_moves;

	/* generate the root node move list */
	gen_move_list(b, 0, 0, NULL);
	
	moves = b->moves;
	num_moves = b->num_moves;

	/* now we compute some bonus values for each move by doing just
	   a quiesence search */
	for (m=0;m<num_moves;m++) {
		Position b1;

		if (!do_move(&b1, b, &moves[m])) {
			moves[m].v = ILLEGAL;
			b->num_moves--;
			continue;
		}

		moves[m].v = -quiesce(&b1, testv, 1);
	}

	sort_moves(moves, num_moves);
	num_moves = b->num_moves;

	if (num_moves == 0) return ILLEGAL;

	/* and we assign the bonus in a linear fashion */
	upper = moves[0].v;
	lower = imax(upper - PAWN_VALUE/4, moves[num_moves-1].v);
	range = upper - lower;
	
	for (m=0;m<num_moves;m++) {
		moves[m].bonus = (moves[m].v - lower) * BONUS_RANGE / range;
		if (b->move_num < 5)
			moves[m].bonus += random() % (RANDOM_EVAL_RANGE * 10);
		else
			moves[m].bonus += random() % RANDOM_EVAL_RANGE;
			
		moves[m].bonus = imax(0, moves[m].bonus);
		if (b->move_num > 10)
			moves[m].bonus = 0;
	}	

got_moves:
	moves = b->moves;
	num_moves = b->num_moves;

	for (m=0;m<num_moves;m++) {
		Position b1;

		if (!do_move(&b1, b, &moves[m])) {
			moves[m].v = ILLEGAL;
			continue;
		}

		if (check_repitition(&b1, 1)) {
			v = draw_value(b);
		} else {
			v = -abtest(&b1,depth-1,-(testv-(moves[m].bonus+1)),1);
			v += moves[m].bonus;
		}

		if (search_expired()) 
			return g;

		moves[m].v = v;

		if (v > g) {
			g = v;
		} 

#if TREE_PRINT_PLY
		lindent(2, 0);
		lprintf(2, "%s %d %s -> %d (%d) testv=%d depth=%d\n",
			whites_move(b)?"W":"B", 
			b->move_num,
			short_movestr(b, &moves[m]), 
			v, g, testv, depth);
#endif

		if (g >= testv) {
			(*move) = moves[m];
			goto end_loop;
		}
	}

end_loop:

	if (g == ILLEGAL) {
		if (b->flags & FLAG_CHECK) {
			g = ILLEGAL;
		} else {
			g = 0;
		}
	}

	if (m > 0)
		sort_moves(moves, imin(num_moves, m+1));

	return g;
}


#ifdef APLINUX
void move_slave(void)
{
	Eval v;
	Move m1;
	extern int hash_hits, hash_misses;
	Position b;

	init_hash_table();

	while (1) {
		while (slave_out.index == slave_in.index || 
		       slave_in.index == 0)
			usleep(1000);

		slave_out = slave_in;

		b = state->position;

		order_clear(b.move_num);
		hash_change_tag(b.move_num);
		
		v = abtest(&b, slave_out.depth, slave_out.testv, &m1);

		if (slave_in.index != slave_out.index) 
			continue;

		slave_out.v = v;
		slave_out.nodes = nodes;
		slave_out.mo_hits = move_order_hits;
		slave_out.mo_misses = move_order_misses;
		slave_out.hash_hits = hash_hits;
		slave_out.hash_misses = hash_misses;
		slave_out.maxply = maxply;
		slave_out.move = m1;
		l_asend(0, 0, 0, &slave_out, sizeof(slave_out));
		nodes = 0;
		move_order_hits = move_order_misses = 0;
		hash_hits = hash_misses = 0;
		maxply = 0;
	}
}
#endif

#if APLINUX
static Eval next_to_search(Eval v,struct slave *slaves, 
			   Eval alpha, Eval beta, int n)
{
	int i;
	int j;
	Eval y;
	int maxwidth = 5;

	if (v == alpha) {
		v += MTD_THRESHOLD + imin((beta - alpha)/10, maxwidth);
	} else if (v == beta) {
		v -= (MTD_THRESHOLD-1) + imin((beta - alpha)/10, maxwidth);
	}
	
	i = 0;

	while (v+i <= beta || v-i > alpha) {
		y = v+i;
			
		if (y <= beta) {
			for (j=1;j<n;j++)
				if (slaves[j].busy && slaves[j].testv == y) break;
			if (j == n) return y;
		}
		
		y = v-i;
		
		if (y >= alpha) {
			for (j=1;j<n;j++)
				if (slaves[j].busy && slaves[j].testv == y) break;
			if (j == n) return y;
		}
		
		i++;
	}

	return v;
}


static Eval par_search(Position *b, Move *move, int depth, Eval guess)
{
	int n;
	int num_cells = getncel();
	Eval v, g, alpha, beta;
	static unsigned par_index;
	extern int hash_hits, hash_misses;
	struct slave slaves[MAX_CELLS];

	hash_change_tag(b->move_num);

	slave_in.index = 0;

	for (n=1;n<num_cells;n++) {
		put(n, state, sizeof(*state), state, 0, 0, 0);	
		put(n, &slave_in, sizeof(slave_in), 
		    &slave_in, 0, 0, 0);	
	}

	usleep(1000);

	while (l_aprobe(ANY_CELL, 0, 0))
		l_aqrecv(ANY_CELL, 0, 0);


	alpha = -INFINITY;
	beta = INFINITY;
	
	nodes = 0;
	hash_hits = hash_misses = 0;

	memset(slaves, 0, sizeof(slaves));

	v = guess;

#if 0
	lprintf(0,"initial depth=%d v=%d (%d nodes)\n",init_depth, v, nodes);
#endif

	while (alpha < beta-MTD_THRESHOLD) {
		if (search_expired())
			return v;

		/* find a spare processor */
		for (n=1;n<num_cells;n++)
			if (!slaves[n].busy) break;

		if (n < num_cells) {
			/* assign it to the closest value to v that is not
			   being searched */
			slaves[n].testv = next_to_search(v, slaves, 
							 alpha, beta,
							 num_cells);
			slaves[n].index = ++par_index;
			slaves[n].depth = depth;
			slaves[n].busy = 1;
			put(n, &slaves[n], sizeof(slave_in), 
			    &slave_in, 0, 0, 0);	
			continue;
		}

		/* they are all busy - wait for one to
		   finish */
		while (!l_aprobe(ANY_CELL, 0, 0))
			if (search_expired()) return v;

		l_aqrecv(ANY_CELL, 0, 0);

		n = getmcid();
		readmsg(&slave_in, sizeof(slave_in));
		
		if (slaves[n].index != slave_in.index) {
			slaves[n].busy = 0;
			continue;
		}

		nodes += slave_in.nodes;
		move_order_hits += slave_in.mo_hits;
		move_order_misses += slave_in.mo_misses;
		hash_hits += slave_in.hash_hits;
		hash_misses += slave_in.hash_misses;
		maxply = imax(maxply, slave_in.maxply);

		g = slave_in.v;

		if (g < slave_in.testv) {
			if (g < beta) {
				v = beta = g;
			}
		} else {
			if (g > alpha) {
				v = alpha = g;
				(*move) = slave_in.move;
			}
		}

		slaves[n].busy = 0;


		/* abort any processors that are searching out of window */
		for (n=1;n<num_cells;n++)
			if (slaves[n].busy==1 && 
			    (slaves[n].testv < alpha || slaves[n].testv > beta)) {
				slaves[n].busy = 0;
			}
	}

	return v;
}
#endif

#if !APLINUX
static Eval mtd_search(Position *b, Move *move, int depth, 
		       Eval guess)
{
	Eval alpha, beta, g, y;
	int loops=0;
	Move m1;
	int delta=0;

	alpha = -INFINITY;
	beta = INFINITY;
	g = guess;

	while (alpha < (beta-MTD_THRESHOLD)) {
		loops++;
		if (g == alpha) {
			y = g+(MTD_THRESHOLD/2)+imin((beta - alpha)/8, delta);
		} else {
			y = (g-((MTD_THRESHOLD/2)-1)) + imax((alpha - beta)/8, delta);
		}

		zero_move(&m1);

		g = abroot(b, depth, y, &m1);
		if (search_expired())
			return y;
#if 0
		lprintf(0,"mtd g=%d y=%d alpha=%d beta=%d\n",
			g, y, alpha, beta);
#endif
		if (g < y) {
			beta = g;
			delta = imin(delta-5, 0);
		} else {
			alpha = g;
			(*move) = m1;
			delta = imax(delta+5, 0);
		}

		if (alpha > -INFINITY && b->num_moves == 1) {
			break;
		}
	}

	return g;
}
#endif

static Eval search_one_depth(Position *b, Move *move, 
			     int depth, Eval guess)
{
	maxply = 0;
	nodes = 0;
	quiesce_nodes = 0;
	semi_quiesce_nodes = 0;

#if APLINUX
	return par_search(b, move, depth, guess);
#else
	return mtd_search(b, move, depth, guess);
#endif
}

static struct hash_entry pv_chain[MAX_DEPTH];

static void show_pv(Position *b)
{
	Position b1;
	int i;
	Move m1;
	Eval v1, v2, v;

	m1.from = pv_chain[0].from;
	m1.to = pv_chain[0].to;

	do_move(&b1, b, &m1);

	lprintf(0,"PV: %s ", short_movestr(b, &m1));

	for (i=1;i<MAX_DEPTH;i++) {
		m1.from = pv_chain[i].from;
		m1.to = pv_chain[i].to;
		if (is_zero_move(&m1)) break;
		lprintf(0,"%s(%d:%d) ", 
			short_movestr(&b1, &m1),
			pv_chain[i].low, pv_chain[i].high);
		do_move(&b1, &b1, &m1);
	}
	lprintf(0,"\n");
#if 0
	print_board(&b1);
	v = eval(&b1, INFINITY, 0);
	v1 = next_to_play(&b1) * v;
	v2 = next_to_play(&b1) * quiesce(&b1, v-5, i);
	lprintf(0," -> %d  quiesce=%d\n", v1, v2);
#endif
}

static void store_pv(Position *b, Move *move)
{
	Move m1;
	Position b1;
	int i=0;
	struct hash_entry *t;

	memset(pv_chain, 0, sizeof(pv_chain));

	do_move(&b1, b, move);

	pv_chain[0].from = move->from;	
	pv_chain[0].to = move->to;

	for (i=1;i<MAX_DEPTH;i++) {
		t = fetch_hash(&b1);
		if (!t) break;
		m1.from = t->from;
		m1.to = t->to;
		if (is_zero_move(&m1)) break;
		if (!do_move(&b1, &b1, &m1)) break;
		pv_chain[i] = (*t);
	}
}

static int terrible_move(Position *b, Move *move, Eval v, int depth)
{
	struct hash_entry *t;
	Position b1;
	Move m1;

	do_move(&b1, b, move);
	t = fetch_hash(&b1);
	if (!t) return 0;

	if (t->depth < depth - 2) return 0;

	m1.from = t->from;
	m1.to = t->to;

	if (t->high > 30-v) {
		lprintf(0,"unbounded: %s %d low=%d high=%d %s\n",
			short_movestr(b, move), v, t->low, t->high,
			short_movestr(&b1, &m1));
		return 1;
	}

	return 0;
}


static Eval search(Position *b, Move *move)
{
	static Eval v=0, lastv=0;
	static int last_move_num, edepth;
	int depth;
	int terminate=0;
	Eval guess, v1, lastv1;
	int expired, force;

	order_clear(b->move_num);

	if (b->move_num == last_move_num+1) {
		depth = edepth-3;
	} else {
		depth = 3;
	}

	depth = imax(depth, 3);
	depth = imin(depth, 6);

	edepth = 0;

	if ((b->move_num & 1) != (last_move_num & 1)) {
		lastv = -lastv;
		v = -v;
	}

	last_move_num = b->move_num;

	while (!terminate && depth < MAX_DEPTH) {
		if (abs(lastv - v) < 50)
			guess = lastv;
		else
			guess = v;

		lastv = v;
	again:
		v = search_one_depth(b, move, depth, guess);
		
		expired = timer_expired();

		if (!pondering &&
		    expired && terrible_move(b, move, v, depth) &&
		    timer_extend()) {
			lprintf(0,"overtime\n");
			guess = v-20;
			goto again;
		}

		force = 0;
		if (b->num_moves == 1) {
			force = 1;
		}

		terminate = timer_terminate(depth, next_to_play(b), force);

		if (!search_expired()) {
			edepth = depth;
			store_pv(b, move);
		} else {
			terminate = 1;
		}

		if (terminate || !pondering) {
			lprintf(0, "%s: %d n/s %d/%d/%d/%d %d:%d mo=%d%% %s t=%d %s\n", 
				colorstr(next_to_play(b)), 
				v * next_to_play(b),
				nodes, semi_quiesce_nodes, quiesce_nodes, 
				timer_elapsed(),
				edepth, 
				maxply, 
				(100*move_order_hits)/(move_order_misses+move_order_hits+1),
				hashstats(),
				current_bonus(next_to_play(b)),
				short_movestr(b, move));
		}

		if (pondering) {
			v1 = -v;
			lastv1 = -lastv;
		} else {
			v1 = v;
			lastv1 = lastv;
		}

		if (state->ics_robot && edepth > 3) {
			if (v1 > lastv1 + 400 && lastv1 < 300) {
				prog_printf("whisper Eval=%d (+%d) yeah!\n", 
					    v1, v1-lastv1);
			} else if (lastv1 < 90 && v1 > 130) {
				prog_printf("whisper Eval=%d (+%d) looking good\n", 
					    v1, v1-lastv1);
			} else if (v1 < lastv1 - 90 && lastv1 < 0) {
				prog_printf("whisper Eval=%d (-%d) uh oh\n", 
					    v1, lastv1-v1);
			} else if (v1 < lastv1 - 400 && lastv1 < 400) {
				prog_printf("whisper Eval=%d (-%d) damn\n", 
					    v1, lastv-v1);
			}
		}

		if (v <= -(WIN-50) || v >= (WIN-50)) 
			terminate = 1;

		if (terminate && !pondering && state->ics_robot &&
		    state->move_time > 2) {
			prog_printf("whisper Eval=%d nodes/sec=%d/%d depth=%d maxply=%d\n",
				    v1, nodes, timer_elapsed(), edepth, maxply);
		}

		if (terminate && !pondering) {
			show_pv(b);
		}

		depth++;
	}

	return v;
}

static int find_winner(Position *b, Move *move)
{
	static Move moves[MAX_MOVES];
	int n, i;
	int bkpos = BLACKPIECES(b)[IKING].pos;
	int wkpos = WHITEPIECES(b)[IKING].pos;
	Position b1;

	b1 = (*b);

	if (whites_move(b) && (b1.topieces[bkpos] & WHITE_MASK)) {
		lprintf(0,"black is mated\n");
		b->winner = 1;
		return 1;
	}

	if (blacks_move(b) && (b1.topieces[wkpos] & BLACK_MASK)) {
		lprintf(0,"white is mated\n");
		b->winner = -1;
		return 1;
	}

	if (check_repitition(b, 3)) {
		lprintf(0,"claim a draw\n");
		prog_printf("draw\n");
		b->winner = STALEMATE;
		return 1;
	}

	if (move && legal_move(b, move)) {
		do_move(&b1, b, move);
	}

	if (check_repitition(&b1, 3)) {
		lprintf(0,"claim a draw\n");
		b->winner = STALEMATE;
		return 1;
	}

	n = generate_moves(&b1, moves);
	for (i=0;i<n;i++)
		if (legal_move(&b1, &moves[i]))
			return 0;

	if (whites_move(&b1) && (b1.topieces[wkpos] & BLACK_MASK)) {
		lprintf(0,"white is mated\n");
		b->winner = -1;
		return 1;
	}

	if (blacks_move(&b1) && (b1.topieces[bkpos] & WHITE_MASK)) {
		lprintf(0,"black is mated\n");
		b->winner = 1;
		return 1;
	}

	b->winner = STALEMATE;
	return 1;
}


/* this is the entry point for the chess program proper */
int make_move(Position *b, Move *move)
{
	Position b1;
	Eval bestv, v1;

	ponder_stop = 0;
	state->stop_search = 0;
	b->winner = 0;

	init_hash_table();
	hash_change_tag(b->move_num);

	if (find_winner(b, NULL))
		return 0;

	regen_moves(b);
	v1 = eval(b, INFINITY, MAX_DEPTH);

	b1 = (*b);

	timer_start(next_to_play(&b1));
	
	nodes = maxply = 0;
	quiesce_nodes = 0;
	semi_quiesce_nodes = 0;
	move_order_misses = move_order_hits = 0;

	regen_moves(&b1);
	
	bestv = search(&b1, move);

	if (b1.hash1 == b->hash1) {
		b->evaluation = bestv * next_to_play(&b1);

		/* don't carry on when its hopeless */
		if (bestv < -RESIGN_VALUE && v1 < -RESIGN_VALUE) {
			lprintf(0,"time to resign\n");
			b->winner = -next_to_play(b);
			prog_printf("resign\n");
		}
	}

	timer_off();

	find_winner(b, move);

	return 1;
}




static void ponder_alarm(void)
{
	ponder_stop = 1;
}


/* this is used to make KnightCap think on the oppenents time */
void ponder_move(Position *b, Move *move)
{
#define PONDER_MULTIPLE 5
	Position b1 = (*b);
	Eval bestv;

	if (ponder_stop)
		return;

	state->stop_search = 0;

	timer_off();
	
	pondering = 1;

	hash_change_tag(b1.move_num);

	signal(SIGALRM, ponder_alarm);
	alarm(state->move_time * PONDER_MULTIPLE + 60);

	init_hash_table();

	nodes = maxply = 0;
	quiesce_nodes = 0;
	semi_quiesce_nodes = 0;
	move_order_misses = move_order_hits = 0;

	regen_moves(&b1);
	eval(&b1, INFINITY, MAX_DEPTH);
	
	bestv = search(&b1, move);

	pondering = 0;

	alarm(0);
}




