#include "includes.h"
#include "knightcap.h"


#define GAME_MOVES 60
#define MORE_MOVES 30
#define SAFETY_TIME 80
#define MULT_HIGH 6
#define MULT_LOW 3
#define MULT_MAX 10
#define ALARM_LIMIT 1

extern struct state *state;

static time_t move_start_time;

static enum {IDLE, STARTED, EXPIRED} timer_state;

static int depth_estimates[MAX_DEPTH];
static int current_depth;

static void sig_alarm(void)
{
	if (timer_state == STARTED) {
		timer_state = EXPIRED;
	}
}

int timer_elapsed(void)
{
	if (timer_state == IDLE)
		return 0;

	return (int)(gettime() - move_start_time);
}

void timer_start(int player)
{
	timer_state = STARTED;
	move_start_time = gettime();
	current_depth = 0;
	alarm(0);
	signal(SIGALRM, sig_alarm);
	timer_state = STARTED;
	alarm(state->bonus_time[player+1] + state->move_time + ALARM_LIMIT);
}


int timer_extend(void)
{
	int extra;

	extra = (state->time_limit/2) - timer_elapsed();
	if (extra < 20) return 0;

	timer_state = STARTED;
	alarm(0);
	signal(SIGALRM, sig_alarm);
	timer_state = STARTED;
	
	extra = imin(extra, state->move_time);
	extra = imax(extra, 1);
	alarm(extra);

	return 1;
}

void timer_estimate(int secs, int increment)
{
	int n;
	int safety = imax(SAFETY_TIME, increment+10);

	state->time_limit = secs;

	if (secs <= 0) {
		state->move_time = 0;
		return;
	}

	/* guess how long the game will be */
	n = GAME_MOVES - state->position.move_num;
	if (n < MORE_MOVES) n = MORE_MOVES;

	state->move_time = increment-1;

	if (secs > safety) {
		state->move_time++;
		state->move_time += (secs-safety)/n;
	} else {
		state->move_time -= 3;
	}

	state->move_time = imin(state->move_time, secs - 4);

	if (state->move_time < 1)
		state->move_time = 1;

	state->bonus_time[state->computer+1] = 
		imin(state->bonus_time[state->computer+1],
		     secs/20);

	state->bonus_time[state->computer+1] = 
		imax(state->bonus_time[state->computer+1],
		     secs/50);

	if (!state->ics_robot)
		lprintf(0,"move_time=%d\n", state->move_time);
}


int timer_expired(void)
{
	if (timer_state == IDLE)
		return 0;

	return (timer_state == EXPIRED);
}

void timer_reset(void)
{
	memset(depth_estimates, 0, sizeof(depth_estimates));
	memset(state->bonus_time, 0, sizeof(state->bonus_time));
}

void timer_off(void)
{
	alarm(0);
	timer_state = IDLE;
}

int timer_terminate(int depth, int player, int force)
{
	int estimate;
	int elapsed = timer_elapsed();
	int available = state->move_time + state->bonus_time[1+player];

	if (timer_state == IDLE)
		return 0;

	if (timer_expired()) {
		if (timer_state != IDLE)
			state->bonus_time[1+player] += (state->move_time - elapsed);
		return 1;
	}

	if (depth >= MAX_DEPTH)
		depth = MAX_DEPTH-1;

	if (depth > 0) {
		depth_estimates[depth-1] = elapsed;
		depth_estimates[depth] = imin(depth_estimates[depth],
					      (depth_estimates[depth-1]+1)*MULT_HIGH);
		depth_estimates[depth] = imax(depth_estimates[depth],
					      depth_estimates[depth-1]*MULT_LOW);
	}

	if (force) goto terminate;

	estimate = depth_estimates[depth];

	if (estimate == 0 && depth > 3)
	    estimate = depth_estimates[depth-1]*10;

	if (estimate <= 1 || estimate < state->move_time/5)
		return 0;

	if (elapsed < available/2)
		return 0;

	if (estimate < state->move_time + state->bonus_time[1+player] &&
	    estimate < MULT_MAX*state->move_time)
		return 0;

terminate:

	state->bonus_time[1+player] += (state->move_time - elapsed);
	return 1;
}

int current_bonus(int player)
{
	if (timer_state == IDLE)
		return 0;

	return state->bonus_time[1+player];
}


int gettime(void)
{
#if WALL_CLOCK
	return time(NULL);
#else
	struct rusage ru;

	getrusage(RUSAGE_SELF, &ru);
	return ru.ru_utime.tv_sec;
#endif
}
