/* -*- c-file-style: "java"; indent-tabs-mode: nil -*-
 * 
 * distcc -- A simple distributed compiler system
 * $Header: /data/cvs/distcc/src/exec.c,v 1.22 2002/04/30 08:17:26 mbp Exp $ 
 *
 * Copyright (C) 2002 by Martin Pool <mbp@samba.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


			/* 18 Their bows also shall dash the young men
			 * to pieces; and they shall have no pity on
			 * the fruit of the womb; their eyes shall not
			 * spare children.
			 *		-- Isaiah 13 */

/**
 * @file
 *
 * Run compilers.
 *
 * @todo Use pipes to talk to the compiler rather than temporary
 * files.  In fact, we might simply just use named pipes directly in
 * place of the temporary files.  We need to make sure to start the
 * compiler *before* we start writing, in that case.  It seems that
 * this works for gcc's input, but it needs to seek on the output.
 * This makes sense -- object files are likely to be assembled in
 * random order, but cc1 needs to be able to read from a pipe.  This
 * will allow at least a bit more overlap between network IO and CPU
 * load on the server, which ought to be good.  The .o file is not so
 * important because it's much smaller.
 **/


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <assert.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/sendfile.h>

#include "distcc.h"
#include "trace.h"
#include "io.h"
#include "util.h"


void dcc_note_execution(const char *hostname, char **argv)
{
    char *astr;

    astr = dcc_argv_tostr(argv);
    rs_log(RS_LOG_INFO|RS_LOG_NONAME, "exec on %s: %s", hostname, astr);
    free(astr);
}



/**
 * Run the compiler locally rather than across the network.
 *
 * Does not return, either execs the compiler in place, or exits with
 * a message.
 **/
void dcc_run_here(char **argv,
                  const char *stdout_file,
                  const char *stderr_file)
{
    redirect_fd(STDIN_FILENO, "/dev/null", O_RDONLY);
    if (stdout_file)
        redirect_fd(STDOUT_FILENO, stdout_file, O_WRONLY|O_CREAT);
    if (stderr_file)
        redirect_fd(STDERR_FILENO, stderr_file, O_WRONLY|O_CREAT);    
    
    execvp(argv[0], argv);
    
    /* shouldn't be reached */
    rs_log_error("failed to exec %s: %s", argv[0], strerror(errno));
    
    exit(1);
}



/**
 * Run @p argv in a child asynchronously.
 *
 * The child's stdin comes from /dev/null, and stdout and stderr are
 * redirected as shown, unless those filenames are NULL.
 **/
int dcc_spawn_child(char **argv, pid_t *pidptr,
                    const char *stdout_file,
                    const char *stderr_file)
{
    pid_t pid;

    rs_trace("entered");
    rs_trace("forking to execute %s", dcc_argv_tostr(argv));
    
    pid = fork();
    if (pid == -1) {
        rs_log_error("failed to fork: %s", strerror(errno));
        return -1;
    } else if (pid == 0) {
        dcc_run_here(argv, stdout_file, stderr_file);
    } else {
        *pidptr = pid;
        rs_trace("child started as pid%d", (int) pid);
        return 0;
    }
}


int dcc_wait_child(pid_t pid, int *wait_status)
{
    /* TODO: Need to parse command result; command failing does
     * not mean we want to give up straight away, because the
     * client needs to see errors, warning messages, etc. */
    if (waitpid(pid, wait_status, 0) == -1) {
        rs_log_error("waitpid borked: %s", strerror(errno));
        return -1;
    }
    rs_trace("child %d terminated with status %#x", pid,
             *wait_status);
    return 0;
}


int dcc_critique_status(int s, const char *command, const char *host)
{
    if (WIFSIGNALED(s)) {
        rs_log_error("%s on %s died with signal %d%s",
                     command, host,
                     WTERMSIG(s),
                     WCOREDUMP(s) ? " (core dumped)" : "");
        return -1;
    } else if (WEXITSTATUS(s)) {
        rs_log_error("%s on %s failed with exit code %d",
                     command, host,
                     WEXITSTATUS(s));
        return -1;
    } else {
        rs_log(RS_LOG_INFO|RS_LOG_NONAME,
               "%s on %s completed ok", command, host);
        return 0;
    }
}

