/* -*- c-file-style: "java"; indent-tabs-mode: nil -*-
 * 
 * distcc -- A simple distributed compiler system
 * $Header: /data/cvs/distcc/src/serve.c,v 1.9 2002/05/15 06:15:01 mbp Exp $ 
 *
 * Copyright (C) 2002 by Martin Pool <mbp@samba.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

                /* He who waits until circumstances completely favour *
                 * his undertaking will never accomplish anything.    *
                 *              -- Martin Luther                      */
   

/**
 * @file
 *
 * Actually serve remote requests.  Called from daemon.c.
 **/


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <assert.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>

#include <sys/stat.h>
#include <sys/sendfile.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "distcc.h"
#include "trace.h"
#include "io.h"
#include "util.h"
#include "rpc.h"


static int dcc_r_request_header(int ifd)
{
    unsigned vers;
    
    if (dcc_expect_token(ifd, "DIST") == -1) {
        rs_log_error("client did not provide distcc magic fairy dust");
        return -1;
    }

    if (dcc_read_int(ifd, &vers) == -1) {
        rs_log_error("didn't get remote version");
        return -1;
    }
    
    if (vers != 1) {
        rs_log_error("client version is %d, i am %d", vers, 1);
        return -1;
    }
    
    return 0;
}


static int dcc_r_argv(int ifd, /*@out@*/ char ***argv)
{
    unsigned i;
    unsigned argc;
    char **a;

    *argv = NULL;
     
    if (dcc_expect_token(ifd, "ARGC"))
        return -1;

    if (dcc_read_int(ifd, &argc))
         return -1;
    
    rs_trace("reading %d arguments from job submission", argc);
    
    /* Have to make the argv one element too long, so that it can be
     * terminated by a null element. */
    *argv = a = (char **) calloc((size_t) argc+1, sizeof a[0]);
    if (a == NULL) {
        rs_log_error("alloc failed");
        return -1;
    }
    a[argc] = NULL;

    for (i = 0; i < argc; i++) {
        if (dcc_expect_token(ifd, "ARGV")
            || dcc_r_str_alloc(ifd, &a[i]))
            return -1;
        
        rs_trace("argv[%d] = \"%s\"", i, a[i]);
    }
    
    return 0;
}


/**
 * Read and execute a job to/from @p ifd
 *
 * We assume the input and output fds are the same, which is true if
 * you're on a socket or tty.
 **/
int dcc_accept_job(int fd)
{
    char **argv;
    int status;
    char *temp_i, *temp_o, *temp_err, *temp_out;
    size_t i_size = 0, o_size = 0;
    pid_t pid;
    int ret;

    /* We log this here, rather than from the accept() call, because
     * we want to also work when running from inetd. */
    dcc_log_clientname(fd);

    /* TODO: Add a command to return the daemon's version and perhaps
     * some information about the machine.  Should include hostname,
     * distcc version, perhaps the system's uptime as well. */

    temp_i = dcc_make_tmpnam("server", ".i");
    temp_o = dcc_make_tmpnam("server", ".o");
    temp_err = dcc_make_tmpnam("cc", ".err");
    temp_out = dcc_make_tmpnam("cc", ".out");

    tcp_cork_sock(fd, 1);

    ret = (dcc_r_request_header(fd)
           || dcc_r_argv(fd, &argv)
           || dcc_r_file(fd, temp_i, "DOTI", &i_size)
           || dcc_set_input(argv, temp_i)
           || dcc_set_output(argv, temp_o)
           || dcc_spawn_child(argv, &pid, temp_out, temp_err)
           || dcc_wait_child(pid, &status)
           || dcc_x_result_header(fd)
           || dcc_x_cc_status(fd, status)
           || dcc_x_file(fd, temp_err, "SERR", NULL)
           || dcc_x_file(fd, temp_out, "SOUT", NULL));

    if (ret == 0 && WIFEXITED(status) && WEXITSTATUS(status) == 0) {
        ret = dcc_x_file(fd, temp_o, "DOTO", &o_size);
    }

    dcc_critique_status(status, "compile", myhostname());
    rs_log(RS_LOG_INFO|RS_LOG_NONAME,
           "input file: %ld bytes; output file: %ld bytes",
           (long) i_size, (long) o_size);
    
    tcp_cork_sock(fd, 0);
    return ret;
}

