/* -*- c-file-style: "java"; indent-tabs-mode: nil -*-
 * 
 * distcc -- A simple distributed compiler system
 * $Header: /data/cvs/distcc/src/tempfile.c,v 1.11 2002/05/15 08:19:49 mbp Exp $ 
 *
 * Copyright (C) 2002 by Martin Pool <mbp@samba.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


                /* "More computing sins are committed in the name of
                 * efficiency (without necessarily achieving it) than
                 * for any other single reason - including blind
                 * stupidity."  -- W.A. Wulf
                 */


#define _GNU_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <assert.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>

#include <sys/stat.h>
#include <sys/sendfile.h>
#include <sys/types.h>

#include "distcc.h"
#include "trace.h"

/**
 * @file
 *
 * Routines for naming, generating and removing temporary files and
 * fifos.
 *
 * All our temporary files are held in a single directory owned by the
 * user running the program.  I think this avoids any security or
 * symlink problems, and it keeps things kind of tidy.  I'm not
 * completely sure, though.
 *
 * It's fine for the files in that directory to be removed by the
 * tmpreaper, or indeed by hand if you're sure they're no longer
 * active.
 *
 * We also support generation of fifos (aka named pipes), which are
 * used for feeding data from the preprocessor and to the compiler.
 **/


/** A subdirectory of tmp to hold all our stuff. **/
static char *tempdir;

/**
 * A list of files that need to be cleaned up on exit.  The fixed-size
 * array is kind of cheap and nasty, but we're never going to use that
 * many.
 **/
#define N_CLEANUPS 20
char *cleanups[N_CLEANUPS];


/**
 * Create a temporary directory used for all our files.  May be shared
 * by multiple instances.
 **/
int dcc_setup_tempdir(void)
{
    struct stat buf;
    
    asprintf(&tempdir, "/tmp/distcc_%d", getuid());
    if (mkdir(tempdir, 0755) == 0) {
        return 0;               /* great */
    } else if (errno == EEXIST) {
        /* If there's already a symbolic link of this name, then we
         * will have just failed with EEXIST.  We need to make sure
         * that if there was an existing file, then it is actually a
         * directory. */
        if (lstat(tempdir, &buf) == -1) {
            rs_log_error("lstat %s failed: %s", tempdir, strerror(errno));
            return -1;            
        } else if (!S_ISDIR(buf.st_mode)) {
            rs_log_error("%s is not a directory", tempdir);
            return -1;
        } else if (buf.st_mode & (S_IWGRP | S_IWOTH)) {
            rs_log_error("permissions on %s (%#o) seem insecure", tempdir,
                         buf.st_mode);
            return -1;
        } else {
            return 0;
        }
    } else {
        rs_log_error("mkdir %s failed: %s", tempdir, strerror(errno));
        return -1;
    }
}


char *dcc_make_tmpnam(const char *prefix, const char *suffix)
{
    char *s;
    int i;

    if (!tempdir)
        dcc_setup_tempdir();

    for (i = 0; cleanups[i]; i++)
        ;

    assert(i < N_CLEANUPS);
    
    asprintf(&s, "%s/%s_%07d%s", tempdir, prefix, getpid(), suffix);

    cleanups[i] = s;
    
    return s;
}


const char *dcc_get_tempdir(void)
{
    if (!tempdir)
        dcc_setup_tempdir();
    return tempdir;
}



void dcc_cleanup_tempfiles(void)
{
    int i;

    for (i = 0; i < N_CLEANUPS && cleanups[i]; i++) {
        if (unlink(cleanups[i]) == -1 && (errno != ENOENT)) {
            rs_log_notice("cleanup %s failed: %s", cleanups[i],
                          strerror(errno));
        }
        free(cleanups[i]);
        cleanups[i] = NULL;
    }
}
        
                         
