#! /usr/bin/env python

'''
Test suite for distcc based on PyUnit.

distcc itself does not use Python.  But PyUnit makes a nice system for
writing the tests for it, because some of them are a bit complex.

Copyright (C) 2002 by Martin Pool

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or (at
your option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
USA.

This file requires at least Python 2.0.
'''

__author__ = 'Martin Pool'
__version__ = '$Revision: 1.17 $'[11:-2] + ' for distcc 0.3'

# TODO: Try compiling a file locally and remotely; check that they are
# the same.

# TODO: Compile locally to .s, and assemble remotely; check that
# works.

# TODO: Check that successful compilations produce nothing on stdout
# or stderr.

# TODO: Add a base class that launches a daemon on some port and shuts
# it down again when done.  The Bitkeeper approach is to give the
# daemon a distinct exit code for "unable to bind socket", and to then
# just keep bumping that up until we find one that's not in use.  That
# way the shell script knows where the server is and it can point the
# client at the same port.

# Also we want some tests that use whatever is already set up in the
# user's environment, because they may have some remote machines we
# can use.  Also check that we correctly handle the case where the
# daemon's already gone and so can't take connections.

# TODO: Some kind of direct test of the host selection algorithm.

# TODO: Optionally run all discc tests under Valgrind

# TODO: Test that ccache correctly caches compilations through distcc:
# make up a random file so it won't hit, then compile once and compile
# twice and examine the log file to make sure we got a hit.  Also
# check that the binary works properly.

# TODO: Test cpp on non-C file (like .Xresources)

# TODO: Test cpp from stdin

# TODO: Do all this with malloc debugging on.

# TODO: Redirect daemon output to a file so that we can more easily
# check it.  Is there a straightforward way to test that it's also OK
# when send through syslogd?

# TODO: Check behaviour when children are killed off.

import time, sys, string, os, types, unittest, re, popen2

EXIT_BAD_ARGUMENTS = 101
EXIT_BIND_FAILED = 102


class SimpleDistCC_Case(unittest.TestCase):
    '''Any kind of test case for distcc'''
    def assertNoFile(self, filename):
        try:
            open(filename, 'r')
        except IOError:
            return
        self.fail("file exists but should not: %s" % filename)


class RunCmd_Case(SimpleDistCC_Case):
    '''A test case that involves running distcc and looking at the output.'''
    def runCmdUnchecked(self, cmd, mcheck=1):
        '''Invoke a distcc command; return (exitcode, stdout)'''
        if mcheck:
            cmd = "MALLOC_CHECK_=2 " + cmd
        pobj = popen2.Popen4(cmd)
        output = pobj.fromchild.read()
        waitstatus = pobj.wait()
        assert not os.WIFSIGNALED(waitstatus), ("%s terminated with signal %d",
                                                cmd, os.WTERMSIG(waitstatus))
        rc = os.WEXITSTATUS(waitstatus)
        return rc, output

    def runCmd(self, cmd, expectedResult = 0):
        rc, output = self.runCmdUnchecked(cmd)
        if rc != expectedResult:
            self.fail("command returned %d; expected %s: \"%s\"" %
                      (rc, expectedResult, cmd))
            
        return output


class Options_Case(RunCmd_Case):
    '''Check basic command-line options'''

    # We specifically don't set the environment variables because they shouldn't
    # be needed for local options.
    
    def subtestVersion(self, prog):
        o = self.runCmd("%s --version" % prog)
        self.assertEquals(o[-1], '\n')
        o = o[:-1]
        self.assert_(re.match(r'^%s [0-9.]+ [\w-]+ \(protocol version 1\)$' % prog, o),
                     "%s version string doesn't match pattern: \"%s\"" % (prog, o))

    def subtestHelp(self, prog):
        o = self.runCmd('%s --help' % prog)
        l = string.split(o, '\n')
        self.assert_(len(l) > 6, "Help text too short: \"%s\"" % o)

    def subtestInvalid(self, prog):
        o = self.runCmd("%s --bogus-option" % prog, EXIT_BAD_ARGUMENTS)

    def runTest(self):
        for prog in ('distcc', 'distccd'):
            for fn in self.subtestHelp, self.subtestVersion, self.subtestInvalid:
                fn(prog)
                

class GccHelp_Case(RunCmd_Case):
    """Check that options following the compiler name are passed
    to the compiler."""
    def runTest(self):
        out = self.runCmd("DISTCC_HOSTS=localhost distcc gcc --help")
        if re.search('distcc', out):
            self.fail("gcc help contains \"distcc\": \"%s\"" % out)
        if not re.match("^Usage: gcc", out):
            self.fail("gcc help doesn't seem correct: \"%s\"" % out)


class Filenames_Case(RunCmd_Case):
    '''Test distcc handling of filenames'''
    def checkExten(self, filename, extension):
        '''Check that distcc extracts the right extension from a filename'''
        o = self.runCmd("h_exten '%s'" % filename)
        self.assertEquals(o, extension)

    def runTest(self):
        for f, e in (("hello.c", ".c"),
                     ("hello.cpp", ".cpp"),
                     ("hello.2.4.4.4.c", ".c"),
                     (".foo", ".foo"),
                     ("gcc", "(NULL)")):
            self.checkExten(f, e)


class IsSource_Case(RunCmd_Case):
    '''Test distcc detection of source files'''
    
    def checkIsSource(self, f, issrc, iscpp):
        o = self.runCmd("h_issource '%s'" % f)
        if o[-1] == '\n':
            o = o[:-1]
        expected = ("%s %s" % (issrc, iscpp))
        if o != expected:
            self.fail("issource %s gave (%s), expected (%s)" %
                      (f, o, expected))
            
    def runTest(self):
        cases = (( "hello.c",          "source",       "not-preprocessed" ),
                 ( "hello.cpp",        "source",       "not-preprocessed" ),
                 ( "hello.2.4.4.i",    "source",       "preprocessed" ),
                 ( ".foo",             "not-source",   "not-preprocessed" ),
                 ( "gcc",              "not-source",   "not-preprocessed" ),
                 ( "hello.ii",         "source",       "preprocessed" ),
                 ( "hello.c++",        "source",       "not-preprocessed" ),
                 ( "boot.s",           "source",       "preprocessed" ),
                 ( "boot.S",           "source",       "not-preprocessed" ))
        for f, issrc, iscpp in cases:
            self.checkIsSource(f, issrc, iscpp)


class NoCompiler_Case(RunCmd_Case):
    """Invocation with no compiler name"""
    def runTest(self):
        self.runCmd("distcc -c hello.c -o hello.o", EXIT_BAD_ARGUMENTS)


class ScanArgs_Case(RunCmd_Case):
    '''Test understanding of gcc command lines'''
    def runTest(self):
        cases = [("gcc -c hello.c", "distribute", "hello.c", "hello.o"),
                 ("gcc hello.c", "local"),
                 ("gcc -o /tmp/hello.o -c ../src/hello.c", "distribute", "../src/hello.c", "/tmp/hello.o"),
                 ("gcc -DMYNAME=quasibar.c bar.c -c -o bar.o", "distribute", "bar.c", "bar.o"),
                 ("gcc -ohello.o -c hello.c", "distribute", "hello.c", "hello.o"),
                 ("ccache gcc -c hello.c", "distribute", "hello.c", "hello.o"),
                 ("gcc hello.o", "local"),
                 ("gcc -o hello.o hello.c", "local"),
                 ("gcc -o hello.o -c hello.s", "distribute", "hello.s", "hello.o"),
                 ("gcc -fprofile-arcs -ftest-coverage -c hello.c", "local", "hello.c", "hello.o"),
                 ]
        for tup in cases:
            self.checkScanArgs(*tup)

    def checkScanArgs(self, ccmd, mode, input=None, output=None):
        o = self.runCmd("h_scanargs %s" % ccmd)
        o = o[:-1]                      # trim \n
        os = string.split(o)
        if mode != os[0]:
            self.fail("h_scanargs %s gave %s mode, expected %s" % (ccmd, os[0], mode))
        if mode == 'distribute':
            if os[1] <> input:
                self.fail("h_scanargs %s gave %s input, expected %s" % (ccmd, os[1], input))
            if os[2] <> output:
                self.fail("h_scanargs %s gave %s output, expected %s" % (ccmd, os[2], output))

            
class WithDaemon_Case(RunCmd_Case):
    """Start the daemon, and then run a command locally against it.
"""
    
    pidfile = "daemonpid.tmp"
    
    def setUp(self):
        self.startDaemon()
        self.setupEnv()

    def tearDown(self):
        self.killDaemon()
        self.delPidFile()

    def setupEnv(self):
        os.environ['DISTCC_HOSTS'] = '127.0.0.1'

    def startDaemon(self):
        # eventually we ought to try different ports until we succeed
        port = 4200
        self.runCmd("distccd --pid-file %s --port %d" % (self.pidfile, port))

    def delPidFile(self):
        self.runCmd("rm %s" % (self.pidfile))

    def killDaemon(self):
        self.runCmd("kill `cat %s` && sleep 1" % (self.pidfile))
        # XXX: There's a slight race here -- the daemon might not exit straight away


class StartStopDaemon_Case(WithDaemon_Case):
    def runTest(self):
        pass


class BadPort_Case(RunCmd_Case):
    """Daemon invoked with invalid port number"""
    def runTest(self):
        self.runCmd("rm -f daemonpid.tmp")
        self.runCmd("distccd --pid-file daemonpid.tmp --port 80000", EXIT_BIND_FAILED)
        self.assertNoFile("daemonpid.tmp")
    
        
class Compilation_Case(WithDaemon_Case):
    '''Test distcc by actually compiling a file'''
    def setUp(self):
        WithDaemon_Case.setUp(self)
        self.createSource(self.sourceFilename(), self.source())

    def runTest(self):
        self.compile()
        self.checkBuiltProgram()

    def createSource(self, filename, contents):
        f = open(filename, 'w')
        f.write(contents)
        f.close()

    def sourceFilename(self):
        return "testtmp.c"              # default
    
    def compile(self):
        msgs = (self.runCmd(self.compileCmd())
                + self.runCmd(self.linkCmd()))
        self.checkCompileMsgs(msgs)

    def compileCmd(self):
        """Return command to compile source and run tests"""
        return "distcc cc -o testtmp.o -c %s" % (self.sourceFilename())

    def linkCmd(self):
        return "distcc cc -o testtmp testtmp.o"

    def checkCompileMsgs(self, msgs):
        if msgs <> '':
            self.fail("expected no compiler messages, got \"%s\""
                      % msgs)

    def checkBuiltProgram(self):
        '''Check compile results.  By default, just try to execute.'''
        msgs = self.runCmd("./testtmp")
        self.checkBuiltProgramMsgs(msgs)

    def checkBuiltProgramMsgs(self, msgs):
        pass


class CompileHello_Case(Compilation_Case):
    """Test the simple case of building a program that works properly"""
    def source(self):
        return """
#include <stdio.h>

int main(void) {
    puts("hello world");
    return 0;
}
"""

    def checkBuiltProgramMsgs(self, msgs):
        self.assertEquals(msgs, "hello world\n")


class ImpliedOutput_Case(CompileHello_Case):
    """Test handling absence of -o"""
    def compileCmd(self):
        return "distcc cc -c testtmp.c"


class SyntaxError_Case(Compilation_Case):
    """Test building a program containing syntax errors, so it won't build
    properly."""
    def source(self):
        return """not C source at all
"""

    def compile(self):
        rc, msgs = self.runCmdUnchecked(self.compileCmd())
        self.assertNotEqual(rc, 0)
        if not re.match(r'testtmp.c:1: parse error', msgs):
            self.fail("compiler error not as expected: \"%s\"" %
                      msgs)


class NoHosts_Case(CompileHello_Case):
    """Test running with no hosts defined.

    We expect compilation to succeed, but with a warning that it was
    run locally."""
    def compileCmd(self):
        return "DISTCC_HOSTS='' distcc cc -c %s" % self.sourceFilename()

    def checkCompileMsgs(self, msgs):
        "We expect only one message, a warning from distcc"
        if not re.search(r"Warning.*\$DISTCC_HOSTS is empty or undefined; can't distribute work",
                         msgs):
            self.fail("did not find expected warning about HOSTS in \"%s\""
                      % msgs)
        # XXX: Doesn't work yet, because parsing of an empty string is broken


class RemoteAssemble_Case(CompileHello_Case):
    """Test remote assembly of a .s file."""

    # We have a rather tricky method for testing assembly code when we
    # don't know what platform we're on.  I think this one will work
    # everywhere, though perhaps not.
    asm_source = """
        .file	"foo.c"
	.version	"01.01"
gcc2_compiled.:
.globl msg
.section	.rodata
.LC0:
	.string	"hello world"
.data
	.align 4
	.type	 msg,@object
	.size	 msg,4
msg:
	.long .LC0
	.ident	"GCC: (GNU) 2.95.4 20011002 (Debian prerelease)"
"""

    def source(self):
            return """
#include <stdio.h>

extern const char *msg;

int main(void) {
    puts(msg);
    return 0;
}
"""
    asm_filename = 'test2.s'

    def setUp(self):
        CompileHello_Case.setUp(self)
        self.createSource(self.asm_filename, self.asm_source)
        
        # TODO: Delete temp files

    def compile(self):
        msgs = self.runCmd(self.compileCmd()) \
               + self.runCmd(self.asmCmd()) \
               + self.runCmd(self.linkCmd())
        self.checkCompileMsgs(msgs)

    def compileCmd(self):
        return "distcc cc -o testtmp.o -c testtmp.c" 

    def asmCmd(self):
        return "distcc cc -o test2.o -c %s" % (self.asm_filename)

    def linkCmd(self):
        return "distcc cc -o testtmp testtmp.o test2.o"


class PreprocessAsm_Case(RemoteAssemble_Case):
    """Run preprocessor locally on assembly, then compile locally."""
    asm_source = """
#define MSG "hello world"
gcc2_compiled.:
.globl msg
.section	.rodata
.LC0:
	.string	 MSG
.data
	.align 4
	.type	 msg,@object
	.size	 msg,4
msg:
	.long .LC0
"""
    asm_filename = 'test2.S'
    

def asmCmd(self):
    return "distcc cc -o test2.o -c test2.S" 


class UnresolvableHost_Case(CompileHello_Case):
    """Test handling of an unresolvable host"""


class UnreachableHost_Case(CompileHello_Case):
    """Test handling of a host that (probably) can't be reached"""

    
def suite():
    from unittest import makeSuite
    s = unittest.TestSuite()
    s.addTest(Options_Case())
    s.addTest(GccHelp_Case())
    s.addTest(Filenames_Case())
    s.addTest(ScanArgs_Case())
    s.addTest(IsSource_Case())
    s.addTest(NoCompiler_Case())
    s.addTest(CompileHello_Case())
    s.addTest(ImpliedOutput_Case())
    s.addTest(SyntaxError_Case())
    s.addTest(BadPort_Case())
    s.addTest(StartStopDaemon_Case())
    s.addTest(RemoteAssemble_Case())
    s.addTest(PreprocessAsm_Case())
    ## s.addTest(NoHosts_Case())
    return s


if __name__ == '__main__':
    unittest.main()
    
